PAGE	58,132
NAME	PEEK
TITLE	PEEK - Peek at a server password
;
IF2
	%OUT	<Starting Pass Two>
ENDIF
;
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;									      *
; Filename:	PEEK.ASM   (written for the Microsoft MACRO Assembler v3.00)  *
;									      *
; Purpose:	This program will read block 8 of an OmnDrive and report back *
;		what the server name, server password, drive name, and drive  *
;		password is.						      *
;									      *
; Author:	Norman O. Doyle  nod					      *
;									      *
; Created:	9/13/85 						      *
; Modified:	9/13/85 						      *
; Version:	1.00a							      *
; Change Log:								      *
;									      *
;      9/13/85	1.00a - Original version - nod				      *
;									      *
;	  (c) Copyright 1985 Corvus Systems, Inc.  All rights reserved.       *
;									      *
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
;
CSEG	SEGMENT PARA PUBLIC 'CODE'

	ASSUME	CS:CSEG,DS:CSEG,ES:CSEG

	ORG	100H			;
START:	JMP	MAIN			;

;**************************************************************************
;
; EQUATES
;
;**************************************************************************
;
VER	EQU	1			; Version number
REVNBR	EQU	0			; Revision number
SUBREV	EQU	0			; Sub-revision number
REVLTR	EQU	'a'                     ; Revision letter (lower case)
;
; Version of the Corvus driver (CORDRV) required to run this program
;
CORVER	EQU	5			;CORDRV version and revision needed
CORREV	EQU	8			;
;
; ASCII equates
;
BACKSPC EQU	08H			;ASCII backspace
BELL	EQU	07H			;ASCII bell
BLANK	EQU	20H			;ASCII blank
CR	EQU	0DH			;ASCII carriage return
DOT	EQU	2EH			;ASCII period
ESCAPE	EQU	1BH			;ASCII ESCAPE character
LF	EQU	0AH			;ASCII line feed
TAB	EQU	09H			;ASCII horizontal tab
NULL	EQU	00H			;ASCII null
;
; Screen character equates
;
ERRLINE 	EQU	1800H		;Cursor position for error line
CMDWND		EQU	1500H		;
CONTPOS 	EQU	1800H		;
SRVRNPOS	EQU	070EH
SRVRPPOS	EQU	0812H
DRVNPOS 	EQU	090DH
DRVPPOS 	EQU	0A11H
;
; Drive type (OmniDrive or generic)
;
DSKSERV 	EQU	1		;generic disk server device type
ODRIVE		EQU	6		;Omni drive device type
;
; Server table Equates
;
NODENUM 	EQU	0		;node number offset
SRVRNAM 	EQU	1		;server name offset
SRVRSIZ 	EQU	11		;server size offset
;					;server sizes are stored using the same
;					;method as for sending an OmniDrive
;					;a sector number. (ie. d, lsb, msb)
;					; d  =the most significant byte, add
;					;     10H to it and swap the nibbles.
;					; lsb=the least significant byte
;					; msb=the middle byte
ENTRYSZ 	EQU	14		;size of one entry
TBLSZ		EQU	64		;number of entries
;
; Drive type table Equates
;
DT_TBLSZ	EQU	6		;number of entries
DT_ENTSIZ	EQU	5		;size of one entry (in bytes)
DT_OUTPUT	EQU	1		;offset for ascii string to output
;
; Misc Equates
;
COLOR		EQU	0		;
MONOCHROME	EQU	1		;
%ODLLEN 	EQU	38		;length of string "%ODLINE"
INBUF_SIZ	EQU	8		;size of %INBUFF
OUTSTR_SIZ	EQU	6		;max number of digits in %OUTSTR
TEN		EQU	10
ONEHUNDRED	EQU	100
ONETHOUSAND	EQU	1000
TENTHOUSAND	EQU	10000		;
TWOK		EQU	2048		;decimal value for 2k bytes
WAIT		EQU	000H		;# of .86 sec ticks to wait on disk
;					; server
RDBUF_SIZ	EQU	20		;Size of the read/write buffer
RETRIES 	EQU	00AH		;# of retransmissions before aborting
Strtblk 	EQU	10		;Offset into DRIVE VOL TABLE for the
					;... address of the 1st block of volume
Endblk		EQU	14		;Offset into DRIVE VOL TABLE for the
					;... address of the last block of vol.
DVblks		EQU	60		;Offset into BLK 8 for DRV VOL TBL size
DVaddr		EQU	74		;Offset into BLK 8 for DRV VOL TBL addr
TRUE		EQU	0FFFFH		;Logical true
FALSE		EQU	0		;Logical false
;
;**************************************************************************
;
; STORAGE
;
;**************************************************************************

EVEN
;
DRVVOLTBL_SIZ	DW	0		;Size of the Drive volume table in blks
msb_lsb 	DW	0		;
d		DB	0		;
msb_lsb2	DW	0		;
d2		DB	0		;
FRSTBLK 	DB	0		;
VIDEO:		DD	0		;address of video driver
CORPTR: 	DD	0		;address of CORTAB
PARMTAB:	DD	0		;offset & segment of parameter table
HANDLE: 	DW	0		;file handle
DRTN:		DW	0		;address of current display routine
TIMMULT:	DW	0		;time multiplier
%NUMOD: 	DW	0		;# of OmniDrives on the net (default=1)
%SRVR_TBL_PTR	DW	0		;pointer to the correct entry in the
;					;... %SRVTBL
SCTNBR: 	DW	0		;
SCREEN_TYPE	DB	MONOCHROME	;Default to a monochrome screen
ROWCOL		LABEL	WORD
COL		DB	0
ROW		DB	0
XMITDESC	LABEL	WORD
		DB	WAIT		;# of .86 sec ticks to wait on disk srvr
		DB	RETRIES 	;# of retransmissions before aborting

DRVNAME 	DB     10 DUP (" ")     ;Drive name
DRVPSW		DB	8 DUP (" ")     ;Drive password
SRVRNAME	DB     10 DUP (" ")     ;Server name
SRVRPSW 	DB	8 DUP (" ")     ;Server password
%NUMBER:	DB	0		;1 = only allow numbers for input
%ECHO:		DB	0		;0 = echo character
%MAXCHR:	DB	0		;maximum number of characters wanted
%CHRTYP:	DB	0		;number of characters entered
%INBUFF:	DB	INBUF_SIZ DUP(0) ;input buffer from console
%OUTSTR:	DB	15,OUTSTR_SIZ DUP (" "),"$"
RDWRBUF:	DB	RDBUF_SIZ DUP (0)      ;Read/write buffer

UTILSTR:	DB	"UTILHOOK",0    ;device 'UTILHOOK'
MAXCHR: 	DB	0		;maximum # of chars to input
CMD:		DB	0		;number chars read & cmd char
XPORTER:	DB	0FFH		;transporter # (gets patched in later)
NODE1		DB	0		;node number for the master
;
; CONSTELLATION II server commands
;
EVEN
WHOAREYOU LABEL NEAR
	DB	01,0FEH 		;protocol ID
	DB	02,0			;message type
	DB	0			;
WAYSRC: DB	0			;source
WAYDEV: DB	0,0FFH			;responding device type wanted

ID	LABEL	NEAR
	DW	0			;protocol ID
	DW	0			;message type
	DB	0			;
IDSRC:	DB	0			;sending station's address
	DB	0			;
IDDEV:	DB	0			;sending station's device type
IDNM:	DB	10 DUP (0)		;sending station's name

GETDRVPARMS	LABEL	NEAR
	DB	16			;protocol ID
DRVNBR: DB	1			;which drive do you want to respond

DRVPARMS LABEL	NEAR
	DB	37 DUP (0)		;1st 37 bytes returned are not needed
CAPACTY:DB	 3 DUP (0)		;capacity of drive in 512 byte sectors
	DB	88 DUP (0)		;last 88 bytes returned are not needed
;
;**************************************************************************
;
; Drive compare and copy command area (Read/Write sector commands)
;
READCMD 	LABEL	NEAR
		DB	  32H
RSECTOR:	DB	  3 DUP (0)

D1DATA: 	DB	512 DUP (0)		;space for a 512 byte sectors
DRV_SRVR_NAMES: DB	32 DUP (0)		;


;**************************************************************************
;
; Echo Table
;
%ECHOTBL:DB	(TBLSZ) DUP (0FFH)	;fill with FF's
;
; Server table
;
%SRVTBL:DB	(ENTRYSZ*TBLSZ) DUP (0) ;
;

;**************************************************************************
;
; SYSTEM ID STRINGS
;
;**************************************************************************

IBMID:	DB	3,'IBM'                 ;

;**************************************************************************
;
; LITERALS
;
;**************************************************************************

BANNER: DB	15,"OMNIDRIVE PEEK  - ["
	DB	VER+30H,".",REVNBR+30H,SUBREV+30H,REVLTR,"]",CR,LF
	DB	"Copyright (c) 1985, Corvus Systems, Inc.",CR,LF,"$"
LINE:	DB	7,79 DUP (0CDH)
CRLF:	DB	CR,LF,"$"
SRVRST: DB	14,"Current OmniDrive:",CR,LF,LF,"$"
ODLINE: DB	10,%ODLLEN DUP (" "),"$"
OUTLIN: DB	14,"Server Name: ",CR,LF
	DB	   "Server Password: ",CR,LF
	DB	   "Drive Name: ",CR,LF
	DB	   "Drive Password: $"
OUTINFO: DB	10,10 DUP(" "),"$"
;
;**************************************************************************
;
CMDPRMT:DB	14,"Which Omnidrive do you want to PEEK at: ","$"

CONT:	DB	7," <Press any key to continue>","$"

;**************************************************************************

ERRHDR: DB	4,CR,LF,BELL,"PEEK [",VER+30H,".",REVNBR+30H,SUBREV+30H
	DB	REVLTR,"] - ","$"

WRCORMG:DB	12,"CORDRV version [",CORVER+30H,".",CORREV+30H,"]"
	DB	" or above needed.",CR,LF,"$"

NRMLERR:DB	4,"          Unable to talk to the drive.  ","$"

NOCORMG:DB	"Cannot find CORTAb.",CR,LF,"$"

UNKID:	DB	"PEEK only supports the IBM PC."
	DB	CR,LF,"$"

VERERR: DB	12,16 DUP(" "),"Invalid node number, try again.",BELL,"$"

PAGE
;**************************************************************************
;
; MAIN -	main program loop
;
;**************************************************************************

MAIN	PROC	NEAR
	MOV	AX,CS			;
	MOV	DS,AX			; set up segments for ease of use
	MOV	ES,AX			;

	CALL	FINDVID 		;find video jump
	JC	SHORT IERROR1		;

	CALL	FINDID			;find system ID
	JC	SHORT IERROR1		;

	MOV	AH,15
	INT	10H			;Get the current video mode
	CMP	AL,0			;40x25 BW
	JZ	GTSRVRS
	CMP	AL,2			;80x25 BW
	JZ	GTSRVRS
	CMP	AL,7			;CRT MODE, 80x25 B&W card
	JZ	GTSRVRS
	MOV	SCREEN_TYPE,COLOR	;Set screen type to color
GTSRVRS:
	CALL	FINDSRV 		;find all disk servers

	CALL	DRVSIZ			;get the drive size for all OmniDrives

	CALL	SCRN			;display entire screen
;
;Main command loop
;
MAIN0:	CALL	CMDLINE 		;display command prompt
	CALL	GETDEC			;get character from keyboard
	JC	SHORT EXIT		;exit if ESCAPE
	CMP	AL,0DH			;exit if CR
	JZ	SHORT EXIT		;
	CALL	VERIFY			;Make sure that the node exists
	JC	MAIN0
	MOV	NODE1,AL		;Save the node number

	CALL	GETDATA 		;Go and get the information requested
	JMP	SHORT MAIN0		;go set up for next command
;
;End main command loop

RSTERR: CALL	CLRCMD			;
RSTERR1:				;problem reseting driver
	JMP	SHORT IERROR1		;
					;
IERROR1:PUSH	DX			;save error message
	MOV	DX,OFFSET ERRHDR	;print error header
	CALL	PRNTCLR
	POP	DX			;restore error message and print
	CALL	PRNTCLR 		;exit
	JMP	EXIT2

EXIT:	MOV	AH,1			;clear the screen
	CALL	DWORD PTR [VIDEO]	;
EXIT2:	MOV	AH,4CH			;exit system call
	XOR	AL,AL			;show no error
	INT	21H			;bye

MAIN	ENDP

PAGE
;
;
;************************************************************************
;*									*
;* Subroutine:	GETDATA 						*
;*									*
;* Function:	Get the server name and password, and the drive name	*
;*		and password.						*
;*									*
;* Input:	NODE1 = Node to look at.				*
;*		   AL =  "   "   "   "                                  *
;*									*
;* Output:	None							*
;*									*
;* Updates:	Nothing 						*
;*									*
;* Destroys:								*
;*									*
;************************************************************************
;
GETDATA PROC	NEAR
	CALL	SAVBLK8 		;Save block 8 which contains
	MOV	CX,36			;Copy the server/drive names and pswrds
	MOV	SI,OFFSET D1DATA
	MOV	DI,OFFSET DRVNAME
	CLD
	REP	MOVSB
	MOV	SI,OFFSET DRVPSW	;Decrypt the drive password
	MOV	DI,SI
	CALL	PSWDECRYPT
	MOV	SI,OFFSET SRVRPSW	;Decrypt the server password
	MOV	DI,SI
	CALL	PSWDECRYPT
;
;  Output the servername
;
	CALL	CLRINFO
	MOV	SI,OFFSET SRVRNAME
	MOV	DI,OFFSET OUTINFO
	INC	DI
	MOV	CX,10
	CLD
	REP	MOVSB
	MOV	DX,SRVRNPOS
	CALL	POSCUR
	MOV	DX,OFFSET OUTINFO
	CALL	PRNTCLR
;
;  Output the server password
;
	CALL	CLRINFO
	MOV	SI,OFFSET SRVRPSW
	MOV	DI,OFFSET OUTINFO
	INC	DI
	MOV	CX,8
	CLD
	REP	MOVSB
	MOV	DX,SRVRPPOS
	CALL	POSCUR
	MOV	DX,OFFSET OUTINFO
	CALL	PRNTCLR
;
;  Output the drive name
;
	CALL	CLRINFO
	MOV	SI,OFFSET DRVNAME
	MOV	DI,OFFSET OUTINFO
	INC	DI
	MOV	CX,10
	CLD
	REP	MOVSB
	MOV	DX,DRVNPOS
	CALL	POSCUR
	MOV	DX,OFFSET OUTINFO
	CALL	PRNTCLR
;
;  Output the drive password
;
	CALL	CLRINFO
	MOV	SI,OFFSET DRVPSW
	MOV	DI,OFFSET OUTINFO
	INC	DI
	MOV	CX,8
	CLD
	REP	MOVSB
	MOV	DX,DRVPPOS
	CALL	POSCUR
	MOV	DX,OFFSET OUTINFO
	CALL	PRNTCLR
;
	RET
GETDATA ENDP
;
;
;************************************************************************
;*									*
;* Subroutine:	CLRINFO 						*
;*									*
;* Function:	Set the string OUTINFO to all blanks.			*
;*									*
;* Input:	None							*
;*									*
;* Output:								*
;*									*
;* Updates:	OUTINFO is set to blanks				*
;*									*
;* Destroys:	Nothing 						*
;*									*
;************************************************************************
;
CLRINFO PROC	NEAR
	PUSH	CX
	PUSH	DI
	MOV	DI,OFFSET OUTINFO
	INC	DI				;Point past the color byte
	MOV	CX,10
	MOV	AL," "
	CLD
	REP	STOSB
	POP	DI
	POP	CX
	RET
CLRINFO ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	SAVBLK8 						*
;*									*
;* Function:	Save block 8 of the drive.  Block 8 contains the	*
;*		SERVERNAME, SERVERPASSWORD, DRIVENAME, DRIVEPASSWORD	*
;*									*
;* Input:	AL = Server # to read block 8 from			*
;*									*
;* Output:	AL = status of the command				*
;*									*
;* Updates:								*
;*									*
;* Destroys:	AX, BX, CX, DX, SI, DI					*
;*									*
;************************************************************************
;
SAVBLK8 PROC	NEAR
	MOV	BX,OFFSET RSECTOR
	MOV	BYTE PTR [BX],1 	;Drive 1
	MOV	WORD PTR [BX+1],8	;Block 8
	MOV	SI,OFFSET READCMD	;address of data to send to the drive
	MOV	DI,OFFSET D1DATA	;address of buffer for data from drive1
	MOV	AH,1			;cmd = Xmit/Recv data to network srvr
	MOV	CX,4			;number of bytes to transmit
	MOV	BX,WORD PTR XMITDESC	;BL = wait, BH = retries
	MOV	DX,512			;Number of bytes wanted back
	PUSH	CS			;fake a "FAR" call
	CALL	CRVIO			;execute the command
	RET
SAVBLK8 ENDP
;
;
;
;
;************************************************************************
;*									*
;* Subroutine:	VERIFY							*
;*									*
;* Function:	Verify that the node passed in AL exists.		*
;*									*
;* Input:	AL = node number					*
;*									*
;* Output:	Sets the carry flag if node not found.			*
;*									*
;* Updates:	Nothing 						*
;*									*
;* Destroys:	All registers preserved 				*
;*									*
;************************************************************************
;
VERIFY	PROC	NEAR
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
;
	MOV	BX,OFFSET %SRVTBL	;point to table
	MOV	CX,WORD PTR %NUMOD	;get the # of OmniDrives on the network
NXTONE: CMP	AL,BYTE PTR [BX]	;does it match?
	JE	SHORT FOUNDIT		;yes so leave loop
	ADD	BX,ENTRYSZ		;point to next entry
	LOOP	NXTONE			;try to match with the next one
	JMP	NOT_FOUND
FOUNDIT:
	JMP	VEROK
;
;  Could not find a match.  Clear answer and put up error message
;
NOT_FOUND:
	POP	DX			;get passed DX
	PUSH	DX
	CALL	CLREOL			;clear the invalid answer
	MOV	DX,ERRLINE
	CALL	CLREOL			;clear the error line before displaying
	MOV	DX,OFFSET VERERR	;display the verify error message
	CALL	PRNTCLR 		;
	STC				;set the carr flag for error return
	JMP	VERRET
;
VEROK:	CLC
VERRET: POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
VERIFY	ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	FINDSRVR						*
;*									*
;* Function:	Find all the active servers on the NET.  Then		*
;*		send a "WHO ARE YOU" to them to find all the            *
;*		OmniDrives.						*
;*									*
;* Input:	None							*
;*									*
;* Output:	None							*
;*									*
;* Updates:	%ECHOTBL, %SRVTBL, %NUMOD				*
;*									*
;* Destroys:	BX, CX, DX, SI						*
;*									*
;************************************************************************
;
FINDSRV PROC   NEAR

	MOV	BX,OFFSET %ECHOTBL	;build table of active nodes
	MOV	CX,64			;
	MOV	DX,600H 		;DH = ECHO; DL = NODE
FSRV0:	CALL	ECHONET 		;
	INC	DL			;
	LOOP	SHORT FSRV0		;

	MOV	BX,OFFSET %SRVTBL	;send WHO ARE YOUs to active nodes
	MOV	SI,OFFSET %ECHOTBL	;
	MOV	CX,64			;
	MOV	DX,300H 		;DH = SND/RCV TO SERVER; DL = NODE
FSRV1:	CMP	BYTE PTR [SI],0FFH	;if active node do the WHO ARE YOU
	JE	SHORT FSRV2		;
	MOV	DL,BYTE PTR [SI]	;
	CALL	CHKSRV			;
FSRV2:	INC	SI			;
	LOOP	SHORT FSRV1		;

	RET				;
FINDSRV ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	DRVSIZ							*
;*									*
;* Function:	Get the size of each OmniDrive				*
;*									*
;* Input:	None							*
;*									*
;* Output:	None							*
;*									*
;* Updates:	%SRVTBL 						*
;*									*
;* Destroys:	AX, BX, CX, DI, SI					*
;*									*
;************************************************************************
;
DRVSIZ	PROC	NEAR

	MOV	CX,0			;initialize counter
	MOV	BX,OFFSET %SRVTBL	;get pointer to the server table
NXTDRV: PUSH	CX
	PUSH	BX
	MOV	AH,01			;cmd = Xmit/Recv data to drive
	MOV	CX,2			;number of bytes to transmit
	MOV	DX,128			;number of bytes to receive
	MOV	AL,BYTE PTR [BX]	;get the network address of the server
	MOV	SI,OFFSET GETDRVPARMS	;cmd to get the drive parameters
	MOV	DI,OFFSET DRVPARMS	;this is for the returned data
	MOV	BX,0301H		;three attempts, about a second wait
	PUSH	CS			;fake a "FAR" call
	CALL	CRVIO
	POP	BX
	POP	CX
	ADD	BX,ENTRYSZ		;point to next server in server tbl
;
ITSOK:	MOV	SI,OFFSET CAPACTY	;point to capacity of the drive
	MOV	AX,ENTRYSZ		;
	MUL	CX			;
	MOV	DI,AX			;get index into drive size table
	ADD	DI,OFFSET %SRVTBL	;add the index to the starting address
	MOV	AL,BYTE PTR [SI+2]	;get d
	ADD	AL,10H			;convert d
	PUSH	CX
	MOV	CL,4
	ROL	AL,CL
	POP	CX
	MOV	BYTE PTR [DI+SRVRSIZ],AL   ;store d
	MOV	AX,WORD PTR [SI]	   ;get lsb
	MOV	WORD PTR [DI+SRVRSIZ+1],AX
;
	INC	CX
	CMP	CX,WORD PTR %NUMOD
	JLE	NXTDRV
	RET
DRVSIZ	ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	ECHONET 						*
;*									*
;* Function:	Send a ECHO command to a node on the net.		*
;*									*
;* Input:	DH = ECHO command, DL = Node, BX = pointer to %ECHOTBL	*
;*									*
;* Output:	None							*
;*									*
;* Updates:	%ECHOTBL, BX updated to point to the next entry 	*
;*		in %ECHOTBL						*
;*									*
;* Destroys:	AX, DX							*
;*									*
;************************************************************************
;
ECHONET PROC	NEAR
	MOV	AX,DX			;AX has function code and node number
	PUSH	CX			;Fix Companion bug that destroys CX
	PUSH	CS			;FAKE "FAR" CALL
	CALL	CRVIO			;do echo
	POP	CX			;Fix Companion bug that destroys CX
	CMP	AL,0C0H 		;
	JNE	SHORT ECHO1		;
	MOV	BYTE PTR [BX],DL	;mark node as active
ECHO1:	INC	BX			;
	RET				;
ECHONET ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	CHKSRV							*
;*									*
;* Function:	Send a "WHO ARE YOU" cmd to a server then check         *
;*		to see if it is an OmniDrive.  If it is then call	*
;*		a routine to save the node number and server name	*
;*		in %SRVTBL.						*
;*									*
;* Input:	DH = SND/RCV to server cmd, DL = node number		*
;*		BX = pointer to entry in %SRVTBL			*
;*									*
;* Output:	AL = status of ROM call 				*
;*									*
;* Updates:	BX is updated to point to the next entry in %SRVTBL	*
;*		%NUMOD is incremented if the server is an OmniDrive	*
;*									*
;* Destroys:	AH							*
;*									*
;************************************************************************
;
CHKSRV	PROC	NEAR
	PUSH	SI			;
	PUSH	CX			;
	PUSH	BX			;
	PUSH	DX			;

CSRVR0: MOV	SI,OFFSET WHOAREYOU	;check if we really found a server
	MOV	DI,OFFSET ID		;.and save name and node number if so
	MOV	AL,BYTE PTR XPORTER	;
	MOV	BYTE PTR [SI+5],AL	;
	MOV	AX,DX			;function code and node number
	MOV	BX,WORD PTR XMITDESC	; BL = wait, BH = retries
	MOV	CX,8			;
	MOV	DX,18			;
	PUSH	CS			;FAKE "FAR" CALL
	CALL	CRVIO			;
	OR	AL,AL			;
	JZ	SHORT CSRVR2		;
	POP	DX			;
	POP	BX			;
	POP	CX			;
	POP	SI			;
	RET				;

CSRVR2: POP	DX			;
	POP	BX			;
	CMP	BYTE PTR IDDEV,ODRIVE	;check if OmniDrive
	JNE	SHORT CSRVR4		;
CSRVR3: CALL	SAVESRV 		;save server info
CSRVR4: POP	CX			;
	POP	SI			;
	RET				;
CHKSRV	ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	SAVESRV 						*
;*									*
;* Function:	Save a servers node number and name in the		*
;*		server table (%SRVTBL). 				*
;*									*
;* Input:	DH = server number					*
;*		BX = pointer to entry in %SRVTBL			*
;*									*
;* Output:	None							*
;*									*
;* Updates:	BX is updated to point to the next entry in %SRVTBL	*
;*		%NUMOD is incremented if the server is an OmniDrive	*
;*									*
;* Destroys:	CX, DI, SI						*
;*									*
;************************************************************************
;
SAVESRV PROC	NEAR
	MOV	BYTE PTR [BX],DL	;save responding server's
	MOV	CX,10			;.address and name in server table
	MOV	SI,OFFSET IDNM		;
	MOV	DI,BX			;
	ADD	DI,SRVRNAM		;
	CLD
	REP	MOVSB			;
	ADD	BX,ENTRYSZ		;point at next entry in server table
	INC	WORD PTR %NUMOD 	;count server
	RET				;
SAVESRV ENDP

PAGE
;************************************************************************
;									*
; FILE ROUTINES 							*
;									*
;************************************************************************
;
;
;
;
;
;
;************************************************************************
;*									*
;* Subroutine:	OPENUTL 						*
;*									*
;* Function:	Open the character device driver named 'UTILHOOK'       *
;*									*
;* Input:	None							*
;*									*
;* Output:	Carry flag set means error occured			*
;*									*
;* Updates:	HANDLE							*
;*									*
;* Destroys:	AX, DX							*
;*									*
;************************************************************************
;
OPENUTL PROC	NEAR
	MOV	DX,OFFSET UTILSTR	;open 'UTILHOOK' character device
	MOV	AH,3DH			;
	MOV	AL,2			;
	INT	21H			;
	MOV	WORD PTR HANDLE,AX	;save file handle
	RET				;carry set means error
OPENUTL ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	CLSFIL							*
;*									*
;* Function:	Close a file handle					*
;*									*
;* Input:	BX = file handle to close				*
;*									*
;* Output:	Carry flag set means error occured			*
;*									*
;* Updates:	None							*
;*									*
;* Destroys:	AX							*
;*									*
;************************************************************************
;
CLSFIL	PROC	NEAR
	MOV	AH,3EH			;close
	INT	21H			;
	RET				;carry set means error
CLSFIL	ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	FNDCOR							*
;*									*
;* Function:	Get the segment & offset to CORTAb			*
;*									*
;* Input:	None							*
;*									*
;* Output:	Carry flag set means error occured			*
;*									*
;* Updates:	CORPTR to contain the segment & offset of CORTAb	*
;*									*
;* Destroys:	AX, BX, CX, DX						*
;*									*
;************************************************************************
;
FNDCOR	PROC	NEAR
	MOV	BX,WORD PTR HANDLE	;get handle
	MOV	AH,40H			;write 2 characters to 'UTILHOOK'
	MOV	CX,2			;.this will reset pointer to
	MOV	DX,OFFSET UTILSTR	;.CORTAb, get characters from name
	INT	21H			;
	MOV	AH,3FH			;now read 4 bytes from 'UTILHOOK'
	MOV	CX,4			;.the segment & offset of CORTAb
	MOV	DX,OFFSET CORPTR	;
	INT	21H			;
	RET				;carry set means error
FNDCOR	ENDP
;
;
;
;************************************************************************
;*									*
;* Subroutine:	IOCNTRL 						*
;*									*
;* Function:	Read to or write from a device control channel. 	*
;*									*
;* Input:								*
;*									*
;* Output:	Carry flag set means error occured			*
;*									*
;* Updates:	PARMTAB 						*
;*									*
;* Destroys:	AX, BX, CX, DX						*
;*									*
;************************************************************************
;
IOCNTRL PROC	NEAR			;
READ:	MOV	DX,OFFSET PARMTAB	;where to save address
	MOV	CX,4			;read 4 bytes (segment & offset)
	MOV	AL,2			;
	JMP	SHORT IO1		;
WRITE:	MOV	AL,3			;write CX bytes, DX points to data
IO1:	MOV	AH,44H			;I/O control
	MOV	BX,WORD PTR HANDLE	;get handle
	INT	21H			;
	RET				;carry set means error
IOCNTRL ENDP

;**************************************************************************

FINDVID PROC	NEAR
	CALL	OPENUTL 		;open 'UTILHOOK'
	JC	SHORT FVIDERR		;
	CALL	FNDCOR			;get address of CORTAb
	JC	SHORT FVIDERR		;
	CALL	CLSFIL			;
	JC	SHORT FVIDERR		;
	PUSH	DS			;
	LDS	SI,DWORD PTR CORPTR	;get pointer to CORTAb

	MOV	AL,BYTE PTR [SI+6]	;check CORDRV version, if equal then
	CMP	AL,CORVER		;.check revision; if version is not
	JNE	SHORT FINDVD0		;.equal then check if version is
	MOV	AL,BYTE PTR [SI+7]	;.greater
	CMP	AL,CORREV		;
	JL	SHORT WRGCOR		;
	JMP	FINDVD1

WRGCOR: MOV	DX,OFFSET WRCORMG	;wrong CORDRV version message
	POP	DS			;
	STC				;
	RET				;

FVIDERR:MOV	DX,OFFSET NOCORMG	;cannot find CORTAb message
	RET				;

FINDVD0:CMP	AL,CORVER		;if version is less than give error
	JL	SHORT WRGCOR		;

FINDVD1:MOV	AX,WORD PTR [SI+17H]	;the VIDEO routine jump is a FAR
	SUB	AX,5			;.jump right before ROM jumps in
	MOV	CS:WORD PTR VIDEO,AX	;.CORDRV, (see CORDRV for more info)
	MOV	CS:WORD PTR VIDEO+2,DS	;.save address of VIDEO jump
	CLC				;show no error
	POP	DS			;
	RET				;
FINDVID ENDP

;**************************************************************************

CMPID	PROC	NEAR
	MOV	SI,WORD PTR [BX+27H]	;get pointer to ID string in
	MOV	AL,BYTE PTR [SI]	;.CORDRV; first check length, if
	CMP	AL,CS:BYTE PTR [DI]	;.thats OK then compare
	JNE	SHORT NOGOOD		;
	XOR	AH,AH			;
	MOV	CX,AX			;
	INC	SI			;
	INC	DI			;
	CLD
  REPZ	CMPSB				;
	JNZ	SHORT NOGOOD		;
	CLC				;say no error
	RET				;
NOGOOD: STC				;show error
	RET				;
CMPID	ENDP

;**************************************************************************

FINDID	PROC	NEAR
	CALL	OPENUTL 		;open 'UTILHOOK'
	JC	SHORT IDERR		;
	CALL	FNDCOR			;get address of CORTAb
	JC	SHORT IDERR		;
	CALL	CLSFIL			;close 'UTILHOOK'
	JC	SHORT IDERR		;
	PUSH	DS			;
	LDS	BX,DWORD PTR CORPTR	;get pointer to CORTAb
;
;  Copy the CORDRV ROM jump table into local area
;
	MOV	SI,[BX+23]		;get the ROM function table pointer
	MOV	DI,OFFSET LNKTAB	;local area to copy the table to
	MOV	CX,20			;number of bytes in table
	CLD
	REP	MOVSB			;copy the table into this program
;
;  Copy the transporter number out of CORTAb
;
	MOV	AL,[BX+27]		;point to "XPORTER" in CORTAb
	MOV	CS:BYTE PTR XPORTER,AL
;
	MOV	CS:WORD PTR TIMMULT,18	;see if IBM
	MOV	DI,OFFSET IBMID 	;
	CALL	CMPID			;
	JNC	SHORT IDOK		;
;
IDERR:	MOV	DX,OFFSET UNKID 	;say unknown ID and return error
	STC				;
IDOK:	POP	DS			;
	RET				;
FINDID	ENDP

PAGE
;**************************************************************************
;
; VIDEO ROUTINES
;
;**************************************************************************

CLRCMD	PROC	NEAR
	MOV	DX,CMDWND		;
CLREOL	LABEL	NEAR
CLRCMD1:CALL	POSCUR			;
	MOV	AH,2			;
	CALL	CS:DWORD PTR [VIDEO]	;
	RET				;
CLRCMD	ENDP

;**************************************************************************

CLRCONT PROC	NEAR
	MOV	DX,CONTPOS		;
	JMP	SHORT CLRCMD1		;
CLRCONT ENDP

;**************************************************************************

CLRWNDO PROC	NEAR
	MOV	DX,CMDWND		;
CLRAGN: CALL	SHORT CLRCMD1		;
	INC	DH			;
	CMP	DX,CONTPOS		;
	JLE	SHORT CLRAGN		;
	RET
CLRWNDO ENDP

;**************************************************************************

CMDLINE PROC	NEAR
	CALL	CLRCMD			;
	MOV	DX,OFFSET CMDPRMT	;
	CALL	PRNTCLR 		;
	RET				;
CMDLINE ENDP

;**************************************************************************

POSCUR	PROC	NEAR
	MOV	AH,3			;expects position to be in DX in
	CALL	CS:DWORD PTR [VIDEO]	;.the form (DH=ROW,DL=COLUMN)
	RET				;
POSCUR	ENDP

;**************************************************************************

PRNTSTR PROC	NEAR
	MOV	AH,09H			;MSDOS print string function
	INT	21H			;
	RET				;
PRNTSTR ENDP

;**************************************************************************

PRNTCHR PROC	NEAR
	MOV	AH,02H			;MSDOS print character function
	INT	21H			;
	RET				;
PRNTCHR ENDP

;**************************************************************************

PRNTCLR PROC	NEAR			;Similure to DOS INT 21 function 9H
;					;...except that the characters are
;					;...printed in color.  The first byte
;					;...of the string contains the color.
;
	PUSH	AX			;Save the registers used
	PUSH	BX
	PUSH	CX
	PUSH	DI
	PUSH	SI
;
	CMP	SCREEN_TYPE,COLOR	;Is there a color screen
	JZ	DO_COLOR
	INC	DX			;Skip over the color byte
	CALL	PRNTSTR
	JMP	END_OF_STRING
;
DO_COLOR:
	MOV	AH,3			;Find the current cursor position
	XOR	BX,BX			;... for page 0
	PUSH	DX
	INT	10H
	MOV	WORD PTR ROWCOL,DX	;Save it.
	POP	DX
	MOV	SI,DX			;Point to the beginning of the string
	MOV	DI,DX			;...to print on the screen
	INC	DI
NXTCHR_TO_PRINT:
	CMP	BYTE PTR [DI],"$"       ;End of the string?
	JZ	END_OF_STRING		;Yes, then exit
	CMP	BYTE PTR [DI],CR	;Character = carriage return?
	JNZ	TRYLF			;No, then see if it's a line feed
	MOV	COL,-1			;Set the col to left edge of the screen
	INC	DI			;Point to the next character
	JMP	MOVCSR
TRYLF:	CMP	BYTE PTR [DI],LF	;Charcter = line feed?
	JNZ	TRYBS			;No, then see if it's a backspace char
	INC	ROW			;Put in the lf by advancing the row
	DEC	COL			;
	INC	DI			;Point to the next character
	JMP	MOVCSR
TRYBS:	CMP	BYTE PTR [DI],BACKSPC	;Character = backspace?
	JNZ	TRYBEL			;No, then see if it's a bell
	SUB	COL,2			;Put in the backspace
	INC	DI			;Point to the next character
	JMP	MOVCSR
TRYBEL: CMP	BYTE PTR [DI],BELL	;Character = bell?
	JNZ	NRML_CHR		;No, then process normal characters
	MOV	AX,0E07H		;Output the bell (AH=cmd, AL=bell)
	INT	10H
	INC	DI			;Point to the next character
	JMP	NXTCHR_TO_PRINT 	;Don't update the cursor position
NRML_CHR:
	MOV	AH,9
	MOV	BH,0
	MOV	CX,1
	MOV	AL,BYTE PTR [DI]	;Character to display
	MOV	BL,BYTE PTR [SI]	;Color of the character
	INT	10H
	INC	DI			;Point to the next character
;
MOVCSR: INC	COL			;Move the cursor one position to
	CMP	COL,80			;... the right.  If we are going past
	JNZ	MVCSR5			;... the right edge then wrap to col 1
	MOV	COL,0			;... and increment the row.  (if
	CMP	ROW,24			;... already at the bottom of the
	JZ	MVCSR5			;... screen then don't increment the
	INC	ROW			;... row)
MVCSR5: MOV	AH,2			;
	XOR	BX,BX
	MOV	DX,WORD PTR ROWCOL
	INT	10H
	JMP	NXTCHR_TO_PRINT
;
END_OF_STRING:
	POP	SI			;Restore all destroyed registers
	POP	DI
	POP	CX
	POP	BX
	POP	AX
	RET
PRNTCLR ENDP

PAGE
;**************************************************************************
;
; KEYBOARD ROUTINES
;
;**************************************************************************
;
;
;
;
;**************************************************************************
;
; BUFIN -	buffered input
;		requires a data space of the following form
;		BYTE1:	   # of chars wanted
;		BYTE2:	   # of chars typed
;		BYTE3 - N: buffer for chars, as least as big as BYTE 1
;
;		also requires a one byte flag %ECHO, 0 = echo character
;		also requires a one byte flag %NUMBER, 1 = numbers only
;
;		ON ENTRY - DX contains address of data space
;		ON EXIT  - carry set if ESCAPE	was typed or carriage
;			   return was first character typed; BYTE2 set
;			   to number of characters typed
;
;		All registers preserved; Uses routine 'GETKYB'
;
;**************************************************************************

BUFIN	PROC   NEAR
	PUSH	BX			;save regs used
	PUSH	DI			;
	PUSH	CX			;
	PUSH	AX			;
	PUSH	DX			;
	MOV	BX,DX			;address of data space
	MOV	DI,DX			;
	ADD	DI,2			;DI now points at buffer
	XOR	CX,CX			;clear count

BUFIN1: CALL	GETKYB			;get char from keyboard
	MOV	AH,0
	JC	SHORT ESCEXT		;carry set means ESCAPE
	CMP	AX,CR			;check for a carriage return,
	JE	SHORT GOTCR		;.backspace, control chars., etc.
	CMP	AX,BACKSPC		;
	JE	SHORT GOTBS		;
	CMP	AX,BLANK		;
	JL	SHORT BUFIN1		;
	CMP	CL,BYTE PTR [BX]	;see if we got all the characters
	JE	SHORT GOTALL		;.we want
;
;  Should we only allow numbers to be input?
;
	CMP	BYTE PTR %NUMBER,1	;allow numbers only?
	JNE	SHORT GETANY		;NO -- so continue
	CMP	AL,'0'                  ;is character between 0 and 9 ?
	JL	SHORT NUMERR
	CMP	AL,'9'
	JLE	SHORT GETANY
NUMERR: MOV	AL,BELL
	CALL	PRNT
	JMP	SHORT BUFIN1
GETANY: MOV	BYTE PTR [DI],AL	;save character
	INC	DI			;
	INC	CX			;
	CMP	BYTE PTR %ECHO,0	;check if we should echo the character
	JE	SHORT BUFIN2		;... if not print a dot
	MOV	AL,DOT			;
BUFIN2: CALL	PRNT			;display it
	JMP	SHORT BUFIN1		;get the next

GOTALL: MOV	AL,BELL 		;got all charcters we wanted so
	CALL	PRNT			;.bitch about any more
	JMP	SHORT BUFIN1		;

GOTBS:	JCXZ	SHORT BUFIN1		;if no chars yet ignore BS
	CALL	PRNT			;else back up cursor
	MOV	AL,BLANK		;blank out char
	CALL	PRNT			;
	MOV	AL,BACKSPC		;back up cursor again
	CALL	PRNT			;
	DEC	DI			;dec char count and buffer pointer
	DEC	CX			;
	MOV	AL,BLANK		;blank out char in buffer
	MOV	BYTE PTR [DI],AL	;
	JMP	SHORT BUFIN1		;go for more

ESCEXT: STC				;show ESCAPE
	MOV	BYTE PTR [BX+1],0	;no chars. returned
	JMP	SHORT BUFEXT		;
GOTCR:	JCXZ	SHORT ESCEXT		;if carriage return is first char
	CLC				;.return with carry set
	MOV	BYTE PTR [BX+1],CL	;
BUFEXT: POP	DX			;restore regs and return
	POP	AX			;
	POP	CX			;
	POP	DI			;
	POP	BX			;
	RET				;

;**************************************************************************

PRNT:	MOV	DL,AL			;
	CALL	PRNTCHR 		;
	RET				;
BUFIN	ENDP

;**************************************************************************

GETKYB	PROC	NEAR
	PUSH	BX
	XOR	BX,BX
	MOV	AH,08H			;get character from keyboard
	JMP	GKCONT
GETKYBW LABEL	NEAR			;entry point to clear kybrd buffer 1st
	PUSH	BX
	XOR	BX,BX
	MOV	AX,0C08H		;
GKCONT: INC	BX
	INT	21H			;
	CMP	AL,NULL 		;
	JE	GKCONT			;Get extended ASCII code
	CMP	AL,ESCAPE		;set carry and return if ESCAPE
	JE	SHORT GETK2		;
	CMP	AL,'a'                  ;
	JL	SHORT GETK1		;
	CMP	AL,'z'                  ;
	JG	SHORT GETK1		;
	CMP	BX,1
	JNE	GETK1			;don't convert extended ASCII codes
	SUB	AL,20H			;convert to upper case
GETK1:	CLC				;show that ESCAPE was pressed
	POP	BX
	RET				;
GETK2:	STC				;
	POP	BX
	RET				;
GETKYB	ENDP

;**************************************************************************
;	RETURNS VALUE IN AL

GETDEC	PROC	NEAR
	PUSH	SI			;
	PUSH	BX			;
	PUSH	CX			;
	PUSH	DX			;
	MOV	BYTE PTR %ECHO,0	;echo characters
	MOV	BYTE PTR %MAXCHR,02	;two digits maximum
	MOV	DX,OFFSET %MAXCHR	;
	XOR	AX,AX			;return zero if no characters typed
	MOV	BYTE PTR %NUMBER,1	;tell BUFIN to allow only numbers
	CALL	BUFIN			;
	JC	SHORT GETDEC3		;
	XOR	CH,CH			;
	MOV	CL,BYTE PTR %CHRTYP	;loop for number of characters typed
	MOV	SI,OFFSET %INBUFF	;
GETDEC1:MOV	BL,BYTE PTR [SI]	;
	INC	SI			;
	SUB	BL,"0"                  ;
	JL	SHORT GETDEC4		;
	CMP	BL,9			;
	JG	SHORT GETDEC4		;
	CBW				;
	MOV	DX,10			;
	MUL	DX			;
	ADD	AX,BX			;
	LOOP	GETDEC1 		;

GETDEC3:POP	DX			;
	POP	CX			;
	POP	BX			;
	POP	SI			;
	RET				;

GETDEC4:STC				;show error
	JMP	SHORT GETDEC3		;
GETDEC	ENDP

PAGE
;**************************************************************************
;
; MISCELLANEOUS ROUTINES
;
;**************************************************************************
;
;

;**************************************************************************

CONTINUE PROC	NEAR
	CALL	CLRCONT 		;
	MOV	DX,OFFSET CONT		;continue message
	CALL	PRNTCLR 		;
	CALL	GETKYB			;get a character from the keyboard
	CALL	CLRCONT 		;
	RET				;
CONTINUE ENDP

;**************************************************************************

SCRN	PROC	NEAR
	MOV	AH,1			;
	CALL	CS:DWORD PTR [VIDEO]	;
SCRN_NO_CLR	LABEL	NEAR		;entry point to display screen without
;					;... clearing the screen first
	MOV	DX,OFFSET BANNER	;get screen display
	CALL	PRNTCLR 		;show it
	MOV	DX,OFFSET LINE		;
	CALL	PRNTCLR 		;
	MOV	DX,OFFSET SRVRST	;
	CALL	PRNTCLR 		;
	CALL	SHOW_ODRIVES		;
	MOV	DX,OFFSET LINE		;print line
	CALL	PRNTCLR 		;
	MOV	DX,OFFSET OUTLIN	;
	CALL	PRNTCLR 		;
	RET				;
SCRN	ENDP

;**************************************************************************

SHOW_ODRIVES	PROC	NEAR
	MOV	CX,%ODLLEN
	MOV	DI,OFFSET ODLINE
	INC	DI			;Skip over the color byte
	MOV	AL,BLANK
	REP	STOSB			;init the line to blanks
	MOV	CX,1			;initialize counter
	MOV	SI,OFFSET %SRVTBL
	MOV	DI,OFFSET ODLINE
	ADD	DI,2			;Skip over the color byte
GETNXT:
	XOR	AH,AH
	MOV	AL,BYTE PTR [SI+NODENUM];get the node number
	CALL	CNVRT_SRVR		;convert decimal to ascii (0-64)
	MOV	[DI],AH 		;get tens digit
	INC	DI
	MOV	[DI],AL 		;get ones digit
;
	ADD	SI,ENTRYSZ		;point to next entry in %SRVTBL
	INC	CX
DONEYT: CMP	CX,WORD PTR %NUMOD	;have we done them all yet?
	JG	ADDLIN			;yes -- so lets get out of here
	INC	DI
	MOV	BYTE PTR [DI],","
	ADD	DI,2
	JMP	GETNXT			;no -- so do it again
ADDLIN:
;
;  Output the status line to the screen now
;
	MOV	DX,OFFSET ODLINE	;point to the output line again
	CALL	PRNTCLR 		;output the status line
	MOV	DX,OFFSET CRLF		;add a blank line
	CALL	PRNTSTR 		;
	RET
SHOW_ODRIVES	ENDP
;
;
;
;************************************************************************
;*									*
;* procedure:	CNVRT_SRVR						*
;*									*
;* input:	AX = Server number to convert to ASCII			*
;*									*
;* output:	AX = ASCII representation of server number.		*
;*		     AH = tens digit, AL = ones digit			*
;*									*
;* updates:								*
;*									*
;* destroys:	AX							*
;*									*
;************************************************************************
;
CNVRT_SRVR	PROC	NEAR
;
	PUSH	BX
	PUSH	CX
;
;  Convert the server number in AX to ASCII with AH holding the tens
;  digit and AL holding the ones digit.
;
	MOV	BX,3030H			; set up default to "00"
	CMP	AX,0				; do range testing on the srvr
	JLE	CS_END				; ... number to make sure it
	CMP	AX,63				; ... falls between 0-63.
	JG	CS_END
;
	MOV	CL,10				; divide the number by 10
	DIV	CL				; AL = quotient, AH = remainder
	XCHG	AH,AL				; put tens in AH, ones in AL
	XCHG	BX,AX				; put value in BX to work with
	ADD	BX,3030H			; convert tens and ones to ASCII
;
CS_END: MOV	AX,BX				; set up return value
	POP	CX
	POP	BX
	RET
;
CNVRT_SRVR	ENDP
;
;
;
;**************************************************************************
;
; BLANKBUF -	fills %INBUFF with blanks
;
;**************************************************************************

BLANKBUF PROC	NEAR
	MOV	DI,OFFSET %INBUFF	;fill buffer with blanks
	MOV	AL,BLANK		;
	MOV	CX,INBUF_SIZ		;
	CLD
	REP	STOSB			;
	RET				;
BLANKBUF ENDP


;**************************************************************************

ERRCON	PROC	NEAR
	PUSH	DX				;save error message
	MOV	DL,BELL 			;...beep...
	CALL	PRNTCHR 			;
	MOV	DX,ERRLINE			;
	CALL	POSCUR				;
	POP	DX				;print error message
	CALL	PRNTCLR 			;
	MOV	DX,OFFSET CONT			;wait for a key press
	CALL	PRNTCLR 			;
	CALL	GETKYB				;
	MOV	DX,ERRLINE			;
	CALL	POSCUR				;
	CALL	CLREOL				;clear line and continue
	RET					;
ERRCON	ENDP


PAGE
;
;**************************************************************************
;
;  PSWDECRYPT
;
;**************************************************************************

%MASK	DB	0				;
MASKMOD DB	9				;

NAMBASE DB	31,146,135,26,99,0,7,78,125,98	;
NAMEINC DB	 2,  1,  1, 1, 1,1,2, 1,  1, 1	;

PSWBASE DB	25,36,122,150,43,18,96,112	;
PSWINC	DB	 2, 1, -1,  1, 2, 1,-1,  1	;

;**************************************************************************
;**************************************************************************

PSWDECRYPT PROC NEAR

	PUSH	AX			;
	PUSH	BX			;
	PUSH	CX			;
	PUSH	DX			;

	XOR	BX,BX			;
	MOV	CX,8			;
	MOV	BYTE PTR %MASK,0	;

PSWDEC1:MOV	AL,BYTE PTR [SI+BX]	;get characater to decrypt,
	SUB	AL,BYTE PTR %MASK	;.subtract off mask and base
	SUB	AL,BYTE PTR PSWBASE[BX] ;
	XOR	AH,AH
	XOR	DX,DX
	PUSH	BX
	PUSH	AX
	MOV	AL,BYTE PTR PSWINC[BX]
	CBW
	MOV	BX,AX
	POP	AX
	IDIV	BX			;divide by increment
	POP	BX
	MOV	DL,AL			;

	ADD	DL,20H			;add in ASCII offset and store
	MOV	BYTE PTR [DI+BX],DL	;

	CMP	BX,0			;
	JNE	SHORT PSWDEC2		;

	CBW				;
	IDIV	BYTE PTR MASKMOD	;make a new mask value if first
	MOV	BYTE PTR %MASK,AH	;.time through

PSWDEC2:INC	BX			;
	LOOP	SHORT PSWDEC1		;do all characters

	POP	DX			;
	POP	CX			;
	POP	BX			;
	POP	AX			;
	RET				;

PSWDECRYPT ENDP
;
;
;
PAGE
;**************************************************************************
;
;  Space to load the call table.  This table gets patched to "ROM" calls
;  See CORDRV "ROMJMPS" for more detail
;
LNKTAB	PROC	NEAR
;	JMP	FAR LABEL	;5 BYTES FOR FAR JUMP (DOES A RET)
	DB	5 DUP (?)
;	JMP	FAR LABEL	;5 BYTES FOR FAR JUMP (DOES A RET)
	DB	5 DUP (?)
;
CRVIO	LABEL	NEAR
;	JMP	FAR LABEL	;5 BYTES FOR FAR JUMP (DOES A RET)
	DB	5 DUP (?)
;
;	JMP	FAR LABEL	;5 BYTES FOR FAR JUMP (DOES AN IRET)
	DB	5 DUP (?)
LNKTAB	ENDP

CSEG	ENDS
	END	START
