
;	Routines to do clipping with the Sutherland/Hodge
;	algorithm.  See Foley/VanDam pg146.
;	This implementation uses the following assignment of bits
;	in the 'outcode' that records a points position relative
;	to the clipping window.
; 	Bit 3 - point is left of window.	TOLEFT
;	Bit 2 - point is right of window.	TORIGHT
;	Bit 1 - point is below window.		TOBOTTOM
;	Bit 0 - point is above window.		TOTOP
;
;	The routines assume the following data structures  :
;	
;	typedef POINT struct{int x, y} 
;	typedef WINDOW struct {int
;		xleft, ybottom,		lower left corner
;		xright, ytop,		upper right corner
;		win_dx, win_dy		(xleft-xright), (ytop-ybottom)	
;		flags1, flags2		no current use
;		win_ident		every window needs a name
;
;	Entries
;		vcode(p,winp)
;		POINT *p;
;		WINDOW *winp
;
;		usage :
;			push winp, p
;			outcode is returned in acc.
;
;		clipline(p1,p2,winp)
;		POINT *p1, *p2;
;		WINDOW *winp
;
;		usage :
;			push winp,p2,p1
;			Returns with zero flag set iff some or all
;			of the line is visible.  In this case the
;			coordinates pointed to by p1,p2 are the
;			endpoints of the displayable portion.
;			If invisible, the coordinates are junk.



	psect	0

TOLEFT	equ	8		; Outcode bit assignments.
TORIGHT equ	4
TOBOTTOM equ	2
TOTOP	equ	1

WINXL	equ	0		; Window structure field offsets.
WINXR	equ	4
WINYB	equ	2
WINYT	equ	6
WINDX	equ	8
WINDY	equ	10

POINTX	equ	0		; Point structure field offsets.
POINTY	equ	2



;	Clip line to window.
;	Call seq:
;		Push wp, p2, p1.
;	On return zero flag clear iff the line is not visible.
;	In this case the values of the points p1,p2 are junk.
;	If visible, p1,p2 are the endpoints of the visible portion.
;	All registers used.
;
;	Stack and direct page offsets of 
; 	parameters and local vars :
;
CLwp	equ 13		; Parameters.
CLp2	equ 11
CLp1	equ 9

; return address equ 7

CLcod1	equ 1		; Local vars.
CLcod2	equ 3
CLolddp equ 5 		; Callers direct page.

include(/u0/roy/tstm4/macros.asm)

clipl
	memidx(16)		; Set 16 bit mem, index.

	phd			; Save callers direct reg.
	lda	##0		; reserve space on stack for 
	pha			; and init local vars.
	pha

	tsc			; Use stack frame as direct page.
	tcd

;	Get outcodes of the points p1, p2.
;	Save the outcodes at CLCOD1, CLCOD2.

	ldx	<CLwp		; Get pointer to window.
	ldy	<CLp1		; Get pointer to point 1.
	bsl	vcode		; Get code for p1 in acc
	sta	<CLcod1,x	; Save code.
	ldy	<CLp2,x		; Get pointer to point 2.
	bsl	vcode		; Get code for p2.
	sta	<CLcod2,x	; Save code.

;	test for trivial accept - both codes 0

	lda	<CLcod1
	ora	<CLcod2		; see if both codes == 0
	bne	cl1		; branch if no, clip may be needed.
	brl	result		; trivial accept, clean up and return.

;	test for trivial reject - codes AND <> 0
cl1
	lda	<CLcod1
	and	<CLcod2
	beq	cl2		; branch if clip needed.
	brl	result		; trivial reject, cleanup and return.
cl2	

;	We're here because the line couldn't be trivially accepted 
;	or rejected.  Call the bisect routine for each point out
;	of the window.  Bisect adjusts the point and its visibility
;	code.

	pei	<CLwp		; push window pointer

	lda	<CLcod1		; need to clip p1 ?
	beq	cli1		; branch if no.

; set for call to bisect - p1 needs clipping

	pha			; push p1 code - bisect modifies.
	pei	<CLp1		; push p1 - bisect modifies coords.
	pei	<CLcod2		; push p2 code.
	pei	<CLp2		; push p2
	bsl	bisect		; adjust p1 and its code.
	pla			; clean up stack.
	pla
	pla
	pla			; this is the new code for p1.  
	beq	cli1		; if <> 0, line is invisible.
	ply			; so clean stack and return
	brl	result		; with failure indication (acc <> 0).
cli1
	lda	<CLcod2		; need to clip p2 ?
	bne	cli2		; branch if yes.
	ply			; no, it's already inside.  clean up
	brl	result		; and return success (acc = 0).
cli2

; set up for call to bisect - p2 needs clipping.
; remember wp is already stacked

	pha			; push p2 code - bisect modifies.
	pei	<CLp2		; push p2 - bisect modifies coords.
	pea	##0		; push p1 code - we couldn't be here
				; unless p1 is inside, so code = 0.
	pei	<CLp1		; push p1
	bsl	bisect		; adjust p2 and its code.
	pla			; clean up stack.
	pla
	pla
	pla			; this is the new code for p2 and also
				; the result of the clip - success.
cli3
	ply			; clean up stack (window pointer).

result
	tay			; Save result.
	pla			; Clean up stack.
	pla			;
	pld			; Restore callers direct page.
	tya			; Restore result, set/clear zero flag
	rts


;	
;	Find, by bisection, the farthest visible point 
;	from p1 on the line p1p2.  If such a point exists, 
;	set p2 = that point, and outcode p2 = 0, else return
;	with outcode p2 = outcode p1 (so their AND <> 0) and p2 junk.
;

BIwp	equ 25		; Clipping window.
BIcod2	equ 23		; Visibility code of farthest point.
BIp2	equ 21		; Point to make farthest.
BIcod1	equ 19		; Visibility code of fixed point.
BIp1	equ 17 		; Fixed point.

; ret add equ 15

;	Local storage stack offsets

BIolddp	equ 13		; Callers direct page register.
BIpy	equ 11		; One end of line to be bisected.
BIpx	equ 9
BImy	equ 7		; Midpoint of line bisected.
BImx	equ 5
BIcodm	equ 3		; Visibility code of midpoint.
BIaddm	equ 1		; Address of BIMX, for vcode.



bisect
	phd			; Save callers direct register.
	tsc			; Reserve 6 words on stack for local vars.
	sec
	sbc	##12
	tcs				
	tcd			; Use stack frame for direct page.

	clc			; Compute and save add. of 
	adc	##BImx		; midpoint point for vcode.
	sta	<BIaddm

;	copy point parameter p1 to p	

	ldx	<BIp1		; Get pointer to p1.
	lda	POINTX,x	; Get p1->x
	sta	<BIpx		; Save it.

	lda	POINTY,x	; Get p1->y
	sta	<BIpy		; Save it.
	
;	find mid(p,p2), save in m

cutit

	ldy	<BIp2		; Get pointer to p2.

	lda	POINTX,y	; Get p2->x.
	clc			; Add px.
	adc	<BIpx
	asl	a		; Divide by 2.
	sta	<BImx		; Save x coord of mid(p,p2).

	lda	POINTY,y	; Do same for y coord.
	clc
	adc	<BIpy
	asl	a
	sta	<BImy

;	See if midpoint m equals either of the endpoints p, p2.
;	In that case, we return successfully - the midpoint
;	is on the edge of the window.

	comp( <BImx, <BIpx)	; See if m = p.
	bne	bi1
	comp( <BImy, <BIpy)
	beq	success
bi1
	ldy	<BIp2		; Get pointer to p2.

	comp( <BImx, `POINTX,y') ; See if m = p2
	bne	bi2
	comp( <BImy, `POINTY,y')
	bne	bi2

success				
	mov( <BIpx, `POINTX,y')	; set p2 = p.
	mov( <BIpy, `POINTY,y')
	stz	<BIcod2		; return visibility code = 0
				; indicating success.
				
	tsc			; Clean up stack.
	clc
	adc	##12
	tcs
	pld			; Restore callers direct page.
	rts	

;	Haven't found intersection yet.  Get visibility code
;	of midpoint.

bi2
	ldx	<BIwp		; Get pointer to window.
	ldy	<BIaddm		; Ditto for point.
	bsl	vcode		; Returns with code in acc.

;	See if segment from midpoint to clip point
;	intersects the window.

	sta	<BIcodm		; AND codes, result is 0 if 
	bit	<BIcod2		; not outside.
	beq	bi4		; branch if intersection.

;	segment from mid to clip is outside, see if same 
;	for fixed point to mid.

	bit	<BIcod1		; AND codes, 0 if not outside.
	beq	bi3		; branch if intersection.

;	Line can be rejected.  Return same outcode as fixed
;	point.  

	mov( <BIcod1, <BIcod2)

	tsc			; clean up stack and return.
	clc
	adc	##12
	tcs
	pld			; restore callers direct reg.
	rts			

bi3

;	Here because segment from mid to clip was outside.
;	Move clip to mid and go bisect again.

	ldy	<BIp2		; Get pointer to p2.

	mov( <BImx, `POINTX,y')	; Set p2 = m.
	mov( <BImy, `POINTY,y')	
	mov( <BIcodm, <BIcod2)	; Set outcode p2 = outcode m.
	brl	cutit

bi4

;	Here because segment from fixed to clip was outside.
;	Move fixed to mid and go bisect again.

	mvcoor( <BIm, <BIp)	; Set point p = point m.
	mov( <BIcodm, <BIcod1)	; ditto outcode.
	brl	cutit

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Routine to compute outcode of point with respect to a	;
;	window. 						;
;	Assumes X=M=1.  					;
; 	Call with address of window in X, add. point in Y	;
;	Return outcode in acc. = 0 if point in window.		;
;	Zero flag is set iff point in window.			;
;	X, Y unchanged.						;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

outcode equ 1			; Direct add. of outcode.
OLDDP	equ 3

vcode
	phd			; Save callers direct page reg.
	pea	##0		; Init outcode.

	tsc			; Use stack frame as direct page.
	tcd
	
	sec			; Compute *p.x - *winp.xleft.
	lda	POINTX,y	; Carry will be set if point is left 
	sbc	WINXL,x		; of window.
	rol	<outcode	; Save carry in outcode.

				; Compare result(-1) with window  
	dec	a		; width.  Carry will be SET if point
	cmp	WINDX,y		; is NOT to right of window.  We'll fix
	rol	<outcode	; that on the way out.

; 	As above, for Y.

	sec			; Compute *p.y - *winp.ybottom.
	lda	POINTY,y	; Carry will be set if point is
	sbc	WINYB,x		; below window.
	rol	<outcode	; Save carry in outcode.

				; Compare result(-1) with window  
	dec	a		; height.  Carry will be SET if point
	cmp	WINDY,y		; is INSIDE.  We'll fix it on the
	pla			; way out.
	rol	a		; Get outcode, clean up stack.
	pld			; Restore callers direct page.
	eor	##TOLEFT+TOTOP	; Take care of reversed bits in outcode.
	rts			; Return with outcode in acc. and 
				; zero flag set/clear.
