;;; regenerator.el --- Statistics regenerator

;; Copyright (C) 2014 Steven Rémot

;;; Author: Steven Rémot

;;; License:
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:
;; The statistics' regenerator preiodically improve
;; current statistics when they are under their maximal value.

;;; Code:

(require 'cl-generic)
(require 'eieio)
(require 'roguel-ike/stats)

(defclass rlk--stats-regenerator ()
  ((stats :initarg :stats
          :type rlk--stats
          :protection :private
          :documentation "Statistics to regenerate.")
   (slots :initarg :slots
          :type list
          :protection :private
          :documentation "Slots to regenrate.")
   (period :initarg :period
           :initform 10
           :type integer
           :protection :private
           :documentation "The number of turns before the regeneration.")
   (count :initform 0
          :type integer
          :protection :private
          :documentation "When count reaches period, a regenration occurs.")
   (rate :initform 0.05
         :type number
         :protection :private
         :documentation "Each stat is regenerated by max(1, base-value * rate)"))
  "Periodically normalize statistics.")

(cl-defmethod regenerate ((self rlk--stats-regenerator))
  "Regenerate the statistics one time."
  (let ((stats (oref self stats))
        (slot nil)
        (base-value 0)
        (current-value 0)
        (rate (oref self rate)))
    (dolist (slot-name (oref self slots))
      (setq slot (get-slot stats slot-name))
      (setq current-value (get-current-value slot))
      (setq base-value (get-base-value slot))
      (cond ((< current-value base-value)
             (set-current-value slot (+ current-value (max 1 (floor (* rate base-value))))))
            ((> current-value base-value)
             (set-current-value slot (- current-value (max 1 (floor (* rate base-value))))))))))

(cl-defmethod add-turns ((self rlk--stats-regenerator) nb-turns)
  "Add NB-TURNS in the turns count, and regenerate the period is reached."
  (let ((period (oref self period)))
    (oset self count (+ (oref self count) nb-turns))
    (while (>= (oref self count) period)
      (regenerate self)
      (oset self count (- (oref self count) period)))))

(provide 'roguel-ike/stats/regenerator)

;;; regenerator.el ends here
