;;; opensub.el --- Search and download from open-subtitles  -*- lexical-binding: t; -*-

;; Copyright (C) 2023  Daniel Fleischer

;; Author: Daniel Fleischer <danflscr@gmail.com>
;; Keywords: multimedia
;; Homepage: https://github.com/danielfleischer/opensub
;; Package-Version: 0.4.0
;; Package-Revision: 0.4.0-0-gd8b11e979897
;; Package-Requires: ((emacs "25.1") (plz))

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Package to query and download subtitles from opensubtitles.com.
;;
;; Set up an account in opensubtitles.com; get an API key and set it
;; in `opensub-api-key'.
;;
;; To run, call `opensub', provide a query string such as "matrix" or
;; "office s04e13". Select the relevant file among the options, and it
;; will be downloaded to `opensub-download-directory'.

;;; Code:
(require 'cl-lib)
(require 'json)
(require 'webjump)
(require 'plz)


(defgroup opensub nil
  "Search and fetch subtitles from opensubtitles.com."
  :group 'multimedia)

(defcustom opensub-api-key ""
  "API key for opensubtitles.com."
  :type 'string)

(defcustom opensub-app-name ""
  "User agent for opensub requests."
  :type 'string)

(defcustom opensub-download-directory "~/Downloads/"
  "Directory where subtitles will be downloaded to."
  :type 'directory)

(defcustom opensub-languages '("en")
  "Languages to search for; 2 letter codes."
  :type '(repeat string))

(defcustom opensub-column-width 72
  "Width of the column in the completion list."
  :type 'integer)

(defun opensub--curl-retrieve (query)
  "Run a QUERY on opensubtitles.com. Return parsed json."
  (let* ((clean-query (webjump-url-encode query))
         (url (format (concat "https://api.opensubtitles.com/api/v1/subtitles?query=%s"
                              (format "&languages=%s"
                                      (string-join (sort opensub-languages #'string<) ",")))
                      clean-query)))
    (plz 'get url
      :headers `(("Content-Type" . "application/json")
                 ("Accept" . "application/json")
                 ("Api-key" . ,opensub-api-key)
                 ("User-agent" . ,opensub-app-name))
      :as #'json-read)))


(defun opensub--annotation (cand)
  (let* ((rest (cdr (assoc cand minibuffer-completion-table))))
    (concat "  "
            (propertize (alist-get 'language rest) 'face 'success) "  "
            (propertize (string-pad (alist-get 'rating rest) 4) 'face 'warning) "  "
            (propertize (alist-get 'full_name rest)))))

(defun opensub--get-file-id (results)
  "Given query RESULTS, return the file id of user-selected item."
  (let* ((items (alist-get 'data results))
         (completion-extra-properties (list :annotation-function 'opensub--annotation))
         (options (cl-mapcar
                   (lambda (item)
                     `(,(substring (string-pad (let-alist item .attributes.release)
                                               opensub-column-width ? )
                                   0 opensub-column-width)
                       . ((language . ,(let-alist item .attributes.language))
                          (rating . ,(number-to-string (let-alist item .attributes.ratings)))
                          (year . ,(number-to-string (let-alist item .attributes.feature_details.year)))
                          (full_name . ,(let-alist item .attributes.feature_details.movie_name)))))
                   items))
         (option (string-trim-right (completing-read "Select: " options)))
         (item  (cl-find-if (lambda (x)
                              (string-search option (let-alist x .attributes.release)))
                            items)))
    (alist-get 'file_id (aref (let-alist item .attributes.files) 0))))

(defun opensub--curl-get-download-info (media-id)
  "Given a MEDIA-ID, generate a downloadable link.
Returns alist with link and file name.
Uses a POST call to generate a time-limited link.
May fail if exceeds daily usage limits."
  (let ((response
         (plz 'post "https://api.opensubtitles.com/api/v1/download"
           :headers `(("Content-Type" . "application/json")
                      ("Accept" . "application/json")
                      ("Api-key" . ,opensub-api-key)
                      ("User-agent" . ,opensub-app-name))
           :body (json-encode `(("file_id" . ,media-id)))
           :as #'json-read)))
    (if (alist-get 'link response) response
      (user-error (alist-get 'message response)))))

(defun opensub--download-subtitle (item)
  "Given ITEM information, downloads subtitle to downloads dir."
  (let* ((link (alist-get 'link item))
         (filename (concat (file-name-as-directory
                            opensub-download-directory)
                           (alist-get 'file_name item))))
    (unless (url-copy-file link filename)
      (error "Couldn't download the subtitle. Try again"))
    filename))

;;;###autoload
(defun opensub ()
  "Run opensub."
  (interactive)
  (let* ((query (read-from-minibuffer "Search a show/movie: "))
         (results (opensub--curl-retrieve query))
         (item-id (opensub--get-file-id results))
         (item (opensub--curl-get-download-info item-id))
         (filename (opensub--download-subtitle item)))
    (message "Subtitle downloaded: `%s'" filename)))

(provide 'opensub)
;;; opensub.el ends here
