"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// codespell:ignore isPlay
const chai_1 = require("chai");
const vscode_languageserver_1 = require("vscode-languageserver");
const yaml_1 = require("yaml");
const yaml_2 = require("../../src/utils/yaml");
const helper_1 = require("../helper");
function getPathInFile(yamlFile, line, character) {
    const textDoc = (0, helper_1.getDoc)(`yaml/${yamlFile}`);
    const parsedDocs = (0, yaml_2.parseAllDocuments)(textDoc.getText());
    return (0, yaml_2.getPathAt)(textDoc, { line: line - 1, character: character - 1 }, parsedDocs);
}
describe("yaml", () => {
    beforeEach(function () {
        const brokenTests = new Map([
        // ['<testName>', '<url-of-tracking-issue>'],
        ]);
        const reason = brokenTests.get(this.currentTest?.title);
        if ((0, helper_1.isWindows)() && reason && this.currentTest) {
            const msg = `Marked ${this.currentTest.title} as pending due to ${reason}`;
            if (process.env.GITHUB_ACTIONS) {
                console.log(`::warning file=${this.currentTest.file}:: ${msg}`);
            }
            else {
                console.log(`🚩 ${msg}`);
            }
            this.currentTest.pending = true;
        }
    });
    describe("ancestryBuilder", () => {
        it("canGetParent", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const node = new yaml_2.AncestryBuilder(path).parent().get();
            (0, chai_1.expect)(node).to.be.an.instanceOf(yaml_1.YAMLMap);
        });
        it("canGetAssertedParent", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const node = new yaml_2.AncestryBuilder(path).parent(yaml_1.YAMLMap).get();
            (0, chai_1.expect)(node).to.be.an.instanceOf(yaml_1.YAMLMap);
        });
        it("canAssertParent", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const node = new yaml_2.AncestryBuilder(path).parent(yaml_1.YAMLSeq).get();
            (0, chai_1.expect)(node).to.be.null;
        });
        it("canGetAncestor", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const node = new yaml_2.AncestryBuilder(path).parent().parent().get();
            (0, chai_1.expect)(node).to.be.an.instanceOf(yaml_1.YAMLSeq);
        });
        it("canGetParentPath", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const subPath = new yaml_2.AncestryBuilder(path).parent().getPath();
            (0, chai_1.expect)(subPath)
                .to.be.an.instanceOf(Array)
                .to.have.lengthOf((path?.length || 0) - 2);
        });
        it("canGetKey", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const key = new yaml_2.AncestryBuilder(path).parent(yaml_1.YAMLMap).getStringKey();
            (0, chai_1.expect)(key).to.be.equal("name");
        });
        it("canGetKeyForValue", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 13);
            const key = new yaml_2.AncestryBuilder(path).parent(yaml_1.YAMLMap).getStringKey();
            (0, chai_1.expect)(key).to.be.equal("name");
        });
        it("canGetKeyPath", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const subPath = new yaml_2.AncestryBuilder(path).parent(yaml_1.YAMLMap).getKeyPath();
            (0, chai_1.expect)(subPath)
                .to.be.an.instanceOf(Array)
                .to.have.lengthOf(path?.length || 0);
            if (subPath)
                (0, chai_1.expect)(subPath[subPath.length - 1])
                    .to.be.an.instanceOf(yaml_1.Scalar)
                    .to.have.property("value", "name");
        });
        it("canGetAssertedParentOfKey", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 7);
            const node = new yaml_2.AncestryBuilder(path).parentOfKey().get();
            (0, chai_1.expect)(node).to.be.an.instanceOf(yaml_1.YAMLMap);
            (0, chai_1.expect)(node).to.have.nested.property("items[0].key.value", "name");
        });
        it("canAssertParentOfKey", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 4, 13);
            const node = new yaml_2.AncestryBuilder(path).parentOfKey().get();
            (0, chai_1.expect)(node).to.be.null;
        });
        it("canGetIndentationParent", async () => {
            const path = await getPathInFile("ancestryBuilder.yml", 7, 9);
            const node = new yaml_2.AncestryBuilder(path)
                .parent(yaml_1.YAMLMap)
                .parent(yaml_1.YAMLMap)
                .getStringKey();
            (0, chai_1.expect)(node).to.be.equal("lineinfile");
        });
        it.skip("canGetIndentationParentAtEndOfMap", async () => {
            // skipped -> the YAML parser doesn't correctly interpret indentation in
            // otherwise empty lines; a workaround is implemented for completion
            // provider
            const path = await getPathInFile("ancestryBuilder.yml", 9, 9);
            if (path) {
                const node = new yaml_2.AncestryBuilder(path)
                    .parent(yaml_1.YAMLMap)
                    .parent(yaml_1.YAMLMap)
                    .getStringKey();
                (0, chai_1.expect)(node).to.be.equal("lineinfile");
            }
        });
        it.skip("canGetIndentationParentAtEOF", async () => {
            // skipped -> the YAML parser doesn't correctly interpret indentation in
            // otherwise empty lines; a workaround is implemented for completion
            // provider
            const path = await getPathInFile("ancestryBuilder.yml", 15, 9);
            const node = new yaml_2.AncestryBuilder(path)
                .parent(yaml_1.YAMLMap)
                .parent(yaml_1.YAMLMap)
                .getStringKey();
            (0, chai_1.expect)(node).to.be.equal("lineinfile");
        });
    });
    describe("getDeclaredCollections", () => {
        it("canGetCollections", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 13, 7);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([
                "mynamespace.mycollection",
                "mynamespace2.mycollection2",
            ]);
        });
        it("canGetCollectionsFromPreTasks", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 9, 7);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([
                "mynamespace.mycollection",
                "mynamespace2.mycollection2",
            ]);
        });
        it("canGetCollectionsFromBlock", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 12, 11);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([
                "mynamespace.mycollection",
                "mynamespace2.mycollection2",
            ]);
        });
        it("canGetCollectionsFromNestedBlock", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 23, 15);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([
                "mynamespace.mycollection",
                "mynamespace2.mycollection2",
            ]);
        });
        it("canGetCollectionsFromRescue", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 27, 11);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([
                "mynamespace.mycollection",
                "mynamespace2.mycollection2",
            ]);
        });
        it("canGetCollectionsFromAlways", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 31, 11);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([
                "mynamespace.mycollection",
                "mynamespace2.mycollection2",
            ]);
        });
        it("canWorkWithoutCollections", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 38, 7);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([]);
        });
        it("canWorkWithEmptyCollections", async () => {
            const path = await getPathInFile("getDeclaredCollections.yml", 46, 7);
            const collections = (0, yaml_2.getDeclaredCollections)(path);
            (0, chai_1.expect)(collections).to.have.members([]);
        });
    });
    describe("isTaskParam", () => {
        it("canCorrectlyConfirmTaskParam", async () => {
            const path = (await getPathInFile("isTaskParamInTaskFile.yml", 2, 3));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyNegateTaskParam", async () => {
            const path = (await getPathInFile("isTaskParamInvalid.yml", 1, 1));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegateTaskParamForValue", async () => {
            const path = (await getPathInFile("isTaskParamInTaskFile.yml", 2, 9));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegateTaskParamForPlay", async () => {
            const path = (await getPathInFile("isTaskParamInPlaybook.yml", 4, 3));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegateTaskParamForBlock", async () => {
            const path = (await getPathInFile("isTaskParamInPlaybook.yml", 14, 7));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegateTaskParamForRole", async () => {
            const path = (await getPathInFile("isTaskParamInPlaybook.yml", 17, 7));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyConfirmTaskParamInPreTasks", async () => {
            const path = (await getPathInFile("isTaskParamInPlaybook.yml", 6, 7));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyConfirmTaskParamInTasks", async () => {
            const path = (await getPathInFile("isTaskParamInPlaybook.yml", 9, 7));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyConfirmTaskParamInBlock", async () => {
            const path = (await getPathInFile("isTaskParamInPlaybook.yml", 13, 11));
            const test = (0, yaml_2.isTaskParam)(path);
            (0, chai_1.expect)(test).to.be.eq(true);
        });
    });
    describe("isPlayParam", () => {
        it("canCorrectlyConfirmPlayParam", async () => {
            const path = (await getPathInFile("isPlayParam.yml", 1, 3));
            const test = (0, yaml_2.isPlayParam)(path, "file://test/isPlay.yml");
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyConfirmPlayParamWithoutPath", async () => {
            const path = (await getPathInFile("isPlayParam.yml", 1, 3));
            const test = (0, yaml_2.isPlayParam)(path);
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyConfirmPlayParamInStrangePath", async () => {
            const path = (await getPathInFile("isPlayParam.yml", 1, 3));
            const test = (0, yaml_2.isPlayParam)(path, "file:///roles/test/tasks/isPlay.yml");
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyNegatePlayParamInRolePathWithoutPlayKeywords", async () => {
            const path = (await getPathInFile("isPlayParam.yml", 7, 3));
            const test = (0, yaml_2.isPlayParam)(path, "file:///roles/test/tasks/isPlay.yml");
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegatePlayParamForNonRootSequence", async () => {
            const path = (await getPathInFile("isPlayParam.yml", 14, 7));
            const test = (0, yaml_2.isPlayParam)(path, "file://test/isPlay.yml");
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegatePlayParamForNonRootSequenceWithoutPath", async () => {
            const path = (await getPathInFile("isPlayParam.yml", 14, 7));
            const test = (0, yaml_2.isPlayParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegatePlayParamForValue", async () => {
            const path = (await getPathInFile("isPlayParam.yml", 1, 9));
            const test = (0, yaml_2.isPlayParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
    });
    describe("isBlockParam", () => {
        it("canCorrectlyConfirmBlockParam", async () => {
            const path = (await getPathInFile("isBlockParam.yml", 2, 3));
            const test = (0, yaml_2.isBlockParam)(path);
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyNegateBlockParam", async () => {
            const path = (await getPathInFile("isBlockParam.yml", 5, 3));
            const test = (0, yaml_2.isBlockParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegateBlockParamOnValue", async () => {
            const path = (await getPathInFile("isBlockParam.yml", 2, 11));
            const test = (0, yaml_2.isBlockParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
    });
    describe("isRoleParam", () => {
        it("canCorrectlyConfirmRoleParam", async () => {
            const path = (await getPathInFile("isRoleParam.yml", 5, 7));
            const test = (0, yaml_2.isRoleParam)(path);
            (0, chai_1.expect)(test).to.be.eq(true);
        });
        it("canCorrectlyNegateRoleParam", async () => {
            const path = (await getPathInFile("isRoleParam.yml", 4, 3));
            const test = (0, yaml_2.isRoleParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("canCorrectlyNegateRoleParamOnValue", async () => {
            const path = (await getPathInFile("isRoleParam.yml", 5, 13));
            const test = (0, yaml_2.isRoleParam)(path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
    });
    describe("isCursorInsideJinjaBrackets", () => {
        const file = "isCursorInsideJinjaBrackets.yml";
        const document = (0, helper_1.getDoc)(`yaml/${file}`);
        it("can confirm cursor within normal jinja bracket", async () => {
            const line = 5;
            const character = 26;
            const position = vscode_languageserver_1.Position.create(line - 1, character - 1);
            const path = getPathInFile(file, line, character);
            if (path) {
                const test = (0, yaml_2.isCursorInsideJinjaBrackets)(document, position, path);
                (0, chai_1.expect)(test).to.be.eq(true);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
        it("can confirm cursor within jinja bracket in correct syntax", async () => {
            const line = 7;
            const character = 20;
            const position = vscode_languageserver_1.Position.create(line - 1, character - 1);
            const path = getPathInFile(file, line, character);
            if (path) {
                const test = (0, yaml_2.isCursorInsideJinjaBrackets)(document, position, path);
                (0, chai_1.expect)(test).to.be.eq(true);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
        it("can confirm cursor within jinja bracket in case of multiple bracket pairs", async () => {
            const line = 9;
            const character = 48;
            const position = vscode_languageserver_1.Position.create(line - 1, character - 1);
            const path = getPathInFile(file, line, character);
            if (path) {
                const test = (0, yaml_2.isCursorInsideJinjaBrackets)(document, position, path);
                (0, chai_1.expect)(test).to.be.eq(true);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
        it("can confirm cursor within jinja bracket even if text already present inside it", async () => {
            const line = 9;
            const character = 36;
            const position = vscode_languageserver_1.Position.create(line - 1, character - 1);
            const path = getPathInFile(file, line, character);
            if (path) {
                const test = (0, yaml_2.isCursorInsideJinjaBrackets)(document, position, path);
                (0, chai_1.expect)(test).to.be.eq(true);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
        it("can negate cursor outside jinja bracket", async () => {
            const line = 9;
            const character = 21;
            const position = vscode_languageserver_1.Position.create(line - 1, character - 1);
            const path = getPathInFile(file, line, character);
            const test = (0, yaml_2.isCursorInsideJinjaBrackets)(document, position, path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
        it("can negate cursor within jinja bracket in case of invalid yaml syntax", async () => {
            const line = 11;
            const character = 25;
            const position = vscode_languageserver_1.Position.create(line - 1, character - 1);
            const path = getPathInFile(file, line, character);
            const test = (0, yaml_2.isCursorInsideJinjaBrackets)(document, position, path);
            (0, chai_1.expect)(test).to.be.eq(false);
        });
    });
});
