/* Copyright (C) 2007-2021 Free Software Foundation, Inc.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 3, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

Under Section 7 of GPL version 3, you are granted additional
permissions described in the GCC Runtime Library Exception, version
3.1, as published by the Free Software Foundation.

You should have received a copy of the GNU General Public License and
a copy of the GCC Runtime Library Exception along with this program;
see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see
<http://www.gnu.org/licenses/>.  */

#include "bid_internal.h"


#if DECIMAL_GLOBAL_ROUNDING_ACCESS_FUNCTIONS

#include <fenv.h>

#define bid_set_excepts(except) __set_status_flags(pfpsf, except)

// Rounding to binary should be done according to destination
// format rounding mode.
static void __bid_set_to_bfp_rnd(void)
{
  switch (fegetround ()) {
    case FE_DOWNWARD: _IDEC_glbround = ROUNDING_DOWN; break;
    case FE_UPWARD: _IDEC_glbround = ROUNDING_UP; break;
    case FE_TOWARDZERO: _IDEC_glbround = ROUNDING_TO_ZERO; break;
    default: _IDEC_glbround = ROUNDING_TO_NEAREST; break;
  }
}

static void __restore_rnd(_IDEC_round *old)
{
  _IDEC_glbround = *old;
}

#define SET_RESTORE_RND_TO_BFP() \
  _IDEC_round __old_rnd __attribute__ ((__cleanup__ (__restore_rnd))) =  _IDEC_glbround; \
  __bid_set_to_bfp_rnd();

#else
#define bid_set_excepts (except) *pfpsf |= except
#define SET_RESTORE_RND_TO_BFP()
#endif

// Counting leading zeros in an unsigned 32-bit word
// The "_nz" version will return the wrong answer (31) for zero inputs

#define CLZ32_MASK16 0xFFFF0000ul
#define CLZ32_MASK8  0xFF00FF00ul
#define CLZ32_MASK4  0xF0F0F0F0ul
#define CLZ32_MASK2  0xCCCCCCCCul
#define CLZ32_MASK1  0xAAAAAAAAul

#define clz32_nz(n)                                             \
 (((((n) & CLZ32_MASK16) <= ((n) & ~CLZ32_MASK16)) ? 16 : 0) +  \
  ((((n) & CLZ32_MASK8) <= ((n) & ~CLZ32_MASK8)) ? 8 : 0) +     \
  ((((n) & CLZ32_MASK4) <= ((n) & ~CLZ32_MASK4)) ? 4 : 0) +     \
  ((((n) & CLZ32_MASK2) <= ((n) & ~CLZ32_MASK2)) ? 2 : 0) +     \
  ((((n) & CLZ32_MASK1) <= ((n) & ~CLZ32_MASK1)) ? 1 : 0))

#define clz32(n) (((n)==0) ? 32 : clz32_nz(n))

// Counting trailing zeros in an unsigned 32-bit word
// The ctz32_1bit version is for a single bit

#define ctz32_1bit(n)                                           \
 ((((n) & ~CLZ32_MASK16) ? 0 : 16) +                            \
  (((n) & ~CLZ32_MASK8) ? 0 : 8) +                              \
  (((n) & ~CLZ32_MASK4) ? 0 : 4) +                              \
  (((n) & ~CLZ32_MASK2) ? 0 : 2) +                              \
  (((n) & ~CLZ32_MASK1) ? 0 : 1))

#define ctz32(n) (((n) == 0) ? 32 : ctz32_1bit((n) & -(n)))

// Counting leading zeros in an unsigned 64-bit word
// The "_nz" version will return the wrong answer (63) for zero inputs

#define CLZ64_MASK32 0xFFFFFFFF00000000ull
#define CLZ64_MASK16 0xFFFF0000FFFF0000ull
#define CLZ64_MASK8  0xFF00FF00FF00FF00ull
#define CLZ64_MASK4  0xF0F0F0F0F0F0F0F0ull
#define CLZ64_MASK2  0xCCCCCCCCCCCCCCCCull
#define CLZ64_MASK1  0xAAAAAAAAAAAAAAAAull

#define clz64_nz(n)                                             \
 (((((n) & CLZ64_MASK32) <= ((n) & ~CLZ64_MASK32)) ? 32 : 0) +  \
  ((((n) & CLZ64_MASK16) <= ((n) & ~CLZ64_MASK16)) ? 16 : 0) +  \
  ((((n) & CLZ64_MASK8) <= ((n) & ~CLZ64_MASK8)) ? 8 : 0) +     \
  ((((n) & CLZ64_MASK4) <= ((n) & ~CLZ64_MASK4)) ? 4 : 0) +     \
  ((((n) & CLZ64_MASK2) <= ((n) & ~CLZ64_MASK2)) ? 2 : 0) +     \
  ((((n) & CLZ64_MASK1) <= ((n) & ~CLZ64_MASK1)) ? 1 : 0))      \

#define clz64(n) (((n)==0) ? 64 : clz64_nz(n))

// Counting trailing zeros in an unsigned 64-bit word
// The ctz64_1bit version is for a single bit

#define ctz64_1bit(n)                                           \
 ((((n) & ~CLZ64_MASK32) ? 0 : 32) +                            \
  (((n) & ~CLZ64_MASK16) ? 0 : 16) +                            \
  (((n) & ~CLZ64_MASK8) ? 0 : 8) +                              \
  (((n) & ~CLZ64_MASK4) ? 0 : 4) +                              \
  (((n) & ~CLZ64_MASK2) ? 0 : 2) +                              \
  (((n) & ~CLZ64_MASK1) ? 0 : 1))

#define ctz64(n) (((n) == 0) ? 64 : ctz64_1bit((n) & -(n)))

// Counting leading zeros in an unsigned 2-part 128-bit word

#define clz128(n_hi,n_lo) (((n_hi) == 0) ? 64 + clz64(n_lo) : clz64_nz(n_hi))

// Counting trailing zeros in a 2-part 128-bit word

#define ctz128(hi,lo) (((lo) == 0) ? 64 + ctz64(hi) : ctz64(lo))

// Shift 2-part 2^64 * hi + lo left by "c" bits
// The "short" form requires a shift 0 < c < 64 and will be faster
// Note that shifts of 64 can't be relied on as ANSI

#define sll128_short(hi,lo,c)                                   \
  ((hi) = ((hi) << (c)) + ((lo)>>(64-(c))),                     \
   (lo) = (lo) << (c)                                           \
  )

#define sll128(hi,lo,c)                                         \
  (((c) == 0) ? hi = hi, lo = lo :                              \
  (((c) >= 64) ? hi = lo << ((c) - 64), lo = 0 : sll128_short(hi,lo,c)))

// Shift 2-part 2^64 * hi + lo right by "c" bits
// The "short" form requires a shift 0 < c < 64 and will be faster
// Note that shifts of 64 can't be relied on as ANSI

#define srl128_short(hi,lo,c)                                   \
  ((lo) = ((hi) << (64 - (c))) + ((lo) >> (c)),                 \
   (hi) = (hi) >> (c)                                           \
  )

#define srl128(hi,lo,c)                                         \
  (((c) == 0) ? hi = hi, lo = lo :                              \
  (((c) >= 64) ? lo = hi >> ((c) - 64), hi = 0 : srl128_short(hi,lo,c)))

// Shift 4-part 2^196 * x3 + 2^128 * x2 + 2^64 * x1 + x0
// right by "c" bits (must have c < 64)

#define srl256(x3,x2,x1,x0,c)                                   \
  ((x0) = ((x1) << (64 - (c))) + ((x0) >> (c)),                 \
   (x1) = ((x2) << (64 - (c))) + ((x1) >> (c)),                 \
   (x2) = ((x3) << (64 - (c))) + ((x2) >> (c)),                 \
   (x3) = (x3) >> (c)                                           \
  )

// Compare "<" two 2-part unsigned integers

#define lt128(x_hi,x_lo,y_hi,y_lo)                              \
  (((x_hi) < (y_hi)) || (((x_hi) == (y_hi)) && ((x_lo) < (y_lo))))

// Likewise "<="

#define le128(x_hi,x_lo,y_hi,y_lo)                              \
  (((x_hi) < (y_hi)) || (((x_hi) == (y_hi)) && ((x_lo) <= (y_lo))))

// 128x256->384 bit multiplication (missing from existing macros)
// I derived this by propagating (A).w[2] = 0 in __mul_192x256_to_448

#define __mul_128x256_to_384(P, A, B)                           \
{                                                               \
UINT512 P0,P1;                                                  \
UINT64 CY;                                                      \
        __mul_64x256_to_320(P0, (A).w[0], B);                   \
        __mul_64x256_to_320(P1, (A).w[1], B);                   \
        (P).w[0] = P0.w[0];                                     \
        __add_carry_out((P).w[1],CY,P1.w[0],P0.w[1]);           \
        __add_carry_in_out((P).w[2],CY,P1.w[1],P0.w[2],CY);     \
        __add_carry_in_out((P).w[3],CY,P1.w[2],P0.w[3],CY);     \
        __add_carry_in_out((P).w[4],CY,P1.w[3],P0.w[4],CY);     \
        (P).w[5] = P1.w[4] + CY;                                \
}

// Multiply a 64-bit number by 10, getting "carry" and "sum"

#define __mul_10x64(sum,carryout,input,carryin)                         \
{ unsigned long long s3 = (input) + ((input) >> 2);                     \
  (carryout) = ((s3 < (unsigned long long)(input))<<3) + (s3>>61);      \
  s3 = (s3<<3) + ((input&3)<<1);                                        \
  (sum) = s3 + (carryin);                                               \
  if ((unsigned long long)(sum) < s3) ++(carryout);                     \
}

// Multiply a 256-bit number by 10, assuming no overflow

#define __mul_10x256_to_256(p3,p2,p1,p0,a3,a2,a1,a0)            \
{ unsigned long long c0,c1,c2,c3;                               \
  __mul_10x64(p0,c0,a0,0ull);                                   \
  __mul_10x64(p1,c1,a1,c0);                                     \
  __mul_10x64(p2,c2,a2,c1);                                     \
  __mul_10x64(p3,c3,a3,c2);                                     \
}

// Set up indices for low and high parts, depending on the endian-ness.
// Note that this only affects 128-bit input and output operands, not any
// of the internal workings, where w[0] is always the low-order part.

#if BID_BIG_ENDIAN
typedef union {
  struct {
    unsigned short hi;
    unsigned short lo1;
    unsigned short lo2;
    unsigned short lo3;
    unsigned short lo4;
    unsigned short pad;
    unsigned pad128;
  } i;
  BINARY80 f;
}
BID_BINARY80LDOUBLE;
#else
typedef union {
  struct {
    unsigned short lo4;
    unsigned short lo3;
    unsigned short lo2;
    unsigned short lo1;
    unsigned short hi;
    unsigned short pad;
    unsigned pad128;
  } i;
  BINARY80 f;
}
BID_BINARY80LDOUBLE;
#endif

// Pack and return binary floating-point numbers from raw fields

#if !DECIMAL_CALL_BY_REFERENCE
#define return_binary32(s,e,c)                  \
 { union {UINT32 i; float f; } x_out;           \
   x_out.i = (((UINT32)(s)) << 31) +            \
           (((UINT32)(e)) << 23) +              \
           (c);                                 \
   return x_out.f;                              \
}
#else
#define return_binary32(s,e,c)                  \
 { union {UINT32 i; float f; } x_out;           \
   x_out.i = (((UINT32)(s)) << 31) +            \
           (((UINT32)(e)) << 23) +              \
           (c);                                 \
   *pres = x_out.f;                             \
   return;                                      \
}
#endif

#if !DECIMAL_CALL_BY_REFERENCE
#define return_binary64(s,e,c)                  \
 { union {UINT64 i; double f; } x_out;          \
   x_out.i = (((UINT64)(s)) << 63) +            \
           (((UINT64)(e)) << 52) +              \
           (c);                                 \
   return x_out.f;                              \
 }
#else
#define return_binary64(s,e,c)                  \
 { union {UINT64 i; double f; } x_out;          \
   x_out.i = (((UINT64)(s)) << 63) +            \
           (((UINT64)(e)) << 52) +              \
           (c);                                 \
   *pres = x_out.f;                             \
   return;                                      \
 }
#endif

#if !DECIMAL_CALL_BY_REFERENCE
#define return_binary80(s,e,c)                      \
 { BID_BINARY80LDOUBLE x_out;                       \
   x_out.i.pad128 = 0;                              \
   x_out.i.pad = 0;                                 \
   x_out.i.lo4 =  (c)&0xffff;                       \
   x_out.i.lo3 = ((c)&0xffff0000) >> 16;            \
   x_out.i.lo2 = ((c)&0xffff00000000ull) >> 32;     \
   x_out.i.lo1 = ((c)&0xffff000000000000ull) >> 48; \
   x_out.i.hi = (((UINT64)(s)) << 15) +             \
                (e);                                \
   return x_out.f;                                  \
 }
#else
#define return_binary80(s,e,c)                      \
 { BID_BINARY80LDOUBLE x_out;                       \
   x_out.i.pad128 = 0;                              \
   x_out.i.pad = 0;                                 \
   x_out.i.lo4 =  (c)&0xffff;                       \
   x_out.i.lo3 = ((c)&0xffff0000) >> 16;            \
   x_out.i.lo2 = ((c)&0xffff00000000ull) >> 32;     \
   x_out.i.lo1 = ((c)&0xffff000000000000ull) >> 48; \
   x_out.i.hi = ((s) << 15) +                       \
                (e);                                \
   *pres = x_out.f;                                 \
   return;                                          \
 }
#endif

#if !DECIMAL_CALL_BY_REFERENCE
#define return_binary128(s,e,c_hi,c_lo)                 \
 { union {UINT128 i; BINARY128 f; } x_out;             \
   x_out.i.w[LOW_128W] = (c_lo);                   \
   x_out.i.w[HIGH_128W] = (((UINT64)(s)) << 63) +   \
                (((UINT64)(e)) << 48) +                 \
                (c_hi);                                 \
   return x_out.f;                                      \
 }
#else
#define return_binary128(s,e,c_hi,c_lo)                 \
 { union {UINT128 i; BINARY128 f; } x_out;             \
   x_out.i.w[LOW_128W] = (c_lo);                   \
   x_out.i.w[HIGH_128W] = (((UINT64)(s)) << 63) +   \
                (((UINT64)(e)) << 48) +                 \
                (c_hi);                                 \
   *pres = x_out.f;                                     \
   return;                                              \
 }
#endif

// Special cases of returning zero, infinity, NaN as binary FP
// Take parameters for the sign, and for NaN the significand

#define return_binary32_zero(s) return_binary32(s,0,0)
#define return_binary32_inf(s) return_binary32(s,255,0)
#define return_binary32_nan(s,c_hi,c_lo) \
  return_binary32(s,255,(c_hi>>42)+(1ul<<22))

#define return_binary64_zero(s) return_binary64(s,0,0)
#define return_binary64_inf(s) return_binary64(s,2047,0)
#define return_binary64_nan(s,c_hi,c_lo) \
  return_binary64(s,2047,(c_hi>>13)+(1ull<<51))

#define return_binary80_zero(s) return_binary80(s,0,0)
#define return_binary80_inf(s) return_binary80(s,32767,(1ull<<63))
#define return_binary80_nan(s,c_hi,c_lo) \
  return_binary80(s,32767,(c_hi>>2)+(3ull<<62))

#define return_binary128_zero(s) return_binary128(s,0,0,0)
#define return_binary128_inf(s) return_binary128(s,32767,0,0)
#define return_binary128_nan(s,c_hi,c_lo) \
  return_binary128(s,32767,(c_hi>>17)+(1ull<<47),((c_lo>>17)+(c_hi<<47)))

// Return finite values of maximal magnitude in the various formats

#define return_binary32_max(s) return_binary32(s,254,((1ul<<23)-1ul))
#define return_binary64_max(s) return_binary64(s,2046,((1ull<<52)-1ull))
#define return_binary80_max(s) return_binary80(s,32766,0xFFFFFFFFFFFFFFFFull)
#define return_binary128_max(s) \
  return_binary128(s,32766,((1ull<<48)-1ull),0xFFFFFFFFFFFFFFFFull)

#define return_bid32_max(s) return_bid32(s,191,9999999ul)
#define return_bid64_max(s) return_bid64(s,767,9999999999999999ull)
#define return_bid128_max(s)    \
  return_bid128(s,12287,542101086242752ull,4003012203950112767ull)

// Handle overflow by either infinity or maximal value as appropriate

#define return_binary32_ovf(s)                                               \
{ if ((rnd_mode==ROUNDING_TO_ZERO) ||                                        \
      (rnd_mode==((s!=0) ? ROUNDING_UP : ROUNDING_DOWN)))                    \
    return_binary32_max(s)                                                   \
  else return_binary32_inf(s)                                                \
}

#define return_binary64_ovf(s)                                               \
{ if ((rnd_mode==ROUNDING_TO_ZERO) ||                                        \
      (rnd_mode==((s!=0) ? ROUNDING_UP : ROUNDING_DOWN)))                    \
    return_binary64_max(s)                                                   \
  else return_binary64_inf(s)                                                \
}

#define return_binary80_ovf(s)                                               \
{ if ((rnd_mode==ROUNDING_TO_ZERO) ||                                        \
      (rnd_mode==((s!=0) ? ROUNDING_UP : ROUNDING_DOWN)))                    \
    return_binary80_max(s)                                                   \
  else return_binary80_inf(s)                                                \
}

#define return_binary128_ovf(s)                                              \
{ if ((rnd_mode==ROUNDING_TO_ZERO) ||                                        \
      (rnd_mode==((s!=0) ? ROUNDING_UP : ROUNDING_DOWN)))                    \
    return_binary128_max(s)                                                  \
  else return_binary128_inf(s)                                               \
}

#define return_bid32_ovf(s)                                                  \
{ if ((rnd_mode==ROUNDING_TO_ZERO) ||                                        \
      (rnd_mode==((s!=0) ? ROUNDING_UP : ROUNDING_DOWN)))                    \
    return_bid32_max(s)                                                      \
  else return_bid32_inf(s)                                                   \
}

#define return_bid64_ovf(s)                                                  \
{ if ((rnd_mode==ROUNDING_TO_ZERO) ||                                        \
      (rnd_mode==((s!=0) ? ROUNDING_UP : ROUNDING_DOWN)))                    \
    return_bid64_max(s)                                                      \
  else return_bid64_inf(s)                                                   \
}

#define return_bid128_ovf(s)                                                 \
{ if ((rnd_mode==ROUNDING_TO_ZERO) ||                                        \
      (rnd_mode==((s!=0) ? ROUNDING_UP : ROUNDING_DOWN)))                    \
    return_bid128_max(s)                                                     \
  else return_bid128_inf(s)                                                  \
}

// Unpack binary floating-point number x into
//
// int    s (sign in the LSB)
// int    e (true "integer" exponent)
//        c (normalized coefficient with explicit 1 bit)
//        t (trailing zero count, valid in normalized case only)
//        [c_hi,c_lo in the case of quad]
//
// Call the given zero, infinity or nan macros if appropriate

#define unpack_binary32(x,s,e,c,t,zero,inf,nan)                             \
{ union { UINT32 i; float f; } x_in;                                        \
  x_in.f = x;                                                               \
  c = x_in.i;                                                               \
  e = (c >> 23) & ((1ull<<8)-1);                                            \
  s = c >> 31;                                                              \
  c = c & ((1ull<<23)-1);                                                   \
  if (e == 0)                                                               \
   { int l;                                                                 \
     if (c == 0) zero;                                                      \
     l = clz32(c) - (32 - 24);                                              \
     c = c << l;                                                            \
     e = -(l + 149);                                                        \
     t = 0;                                                                 \
     bid_set_excepts (DENORMAL_EXCEPTION);                                  \
   }                                                                        \
  else if (e == ((1ull<<8)-1))                                              \
   { if (c == 0) inf;                                                       \
     if ((c&(1ul<<22))==0) bid_set_excepts (INVALID_EXCEPTION);             \
     nan(s,(((unsigned long long) c)) << 42,0ull)                           \
   }                                                                        \
  else                                                                      \
   { c += 1ull<<23;                                                         \
     t = ctz32(c);                                                          \
     e -= 150;                                                              \
   }                                                                        \
}

#define unpack_binary64(x,s,e,c,t,zero,inf,nan)                             \
{ union { UINT64 i; double f; } x_in;                                       \
  x_in.f = x;                                                               \
  c = x_in.i;                                                               \
  e = (c >> 52) & ((1ull<<11)-1);                                           \
  s = c >> 63;                                                              \
  c = c & ((1ull<<52)-1);                                                   \
  if (e == 0)                                                               \
   { int l;                                                                 \
     if (c == 0) zero;                                                      \
     l = clz64(c) - (64 - 53);                                              \
     c = c << l;                                                            \
     e = -(l + 1074);                                                       \
     t = 0;                                                                 \
     bid_set_excepts (DENORMAL_EXCEPTION);                                  \
   }                                                                        \
  else if (e == ((1ull<<11)-1))                                             \
   { if (c == 0) inf;                                                       \
     if ((c&(1ull<<51))==0) bid_set_excepts (INVALID_EXCEPTION);            \
     nan(s,(((unsigned long long) c) << 13),0ull)                           \
   }                                                                        \
  else                                                                      \
   { c += 1ull<<52;                                                         \
     t = ctz64(c);                                                          \
     e -= 1075;                                                             \
   }                                                                        \
}

#define unpack_binary80(x,s,e,c,t,zero,inf,nan)                             \
{ BID_BINARY80LDOUBLE x_in;                                                 \
  x_in.f = x;                                                               \
  c = x_in.i.lo4 + ((UINT64)x_in.i.lo3 << 16) +                             \
      ((UINT64)x_in.i.lo2 << 32) + ((UINT64)x_in.i.lo1 << 48);              \
  e = x_in.i.hi;                                                            \
  s = e >> 15;                                                              \
  e = (e & ((1<<15)-1));                                                    \
  if (e == 0)                                                               \
   { int l;                                                                 \
     if (c == 0) zero;                                                      \
     l = clz64(c);                                                          \
     c = c << l;                                                            \
     e -= (l + 16445);                                                      \
     t = 0;                                                                 \
     bid_set_excepts (DENORMAL_EXCEPTION);                                  \
    }                                                                       \
  else if (e == ((1ull<<15)-1))                                             \
   { if ((c & ((1ull<<63)-1)) == 0) inf;                                    \
     if ((c&(1ull<<62))==0) bid_set_excepts (INVALID_EXCEPTION);            \
     nan(s,(((unsigned long long) c) << 2),0ull)                            \
   }                                                                        \
  else                                                                      \
   { t = ctz64(c);                                                          \
     e -= 16446;                                                            \
   }                                                                        \
}

#define unpack_binary128(x,s,e,c_hi,c_lo,t,zero,inf,nan)                    \
{ union { UINT128 i; BINARY128 f; } x_in;                                  \
  x_in.f = x;                                                               \
  c_lo = x_in.i.w[LOW_128W];                                           \
  c_hi = x_in.i.w[HIGH_128W];                                           \
  e = (c_hi >> 48) & ((1ull<<15)-1);                                        \
  s = c_hi >> 63;                                                           \
  c_hi = c_hi & ((1ull<<48)-1);                                             \
  if (e == 0)                                                               \
   { int l;                                                                 \
     if ((c_hi == 0) && (c_lo == 0)) zero;                                  \
     l = clz128(c_hi,c_lo) - (128 - 113);                                   \
     sll128(c_hi,c_lo,l);                                                   \
     e = -(l + 16494);                                                      \
     t = 0;                                                                 \
     bid_set_excepts (DENORMAL_EXCEPTION);                                  \
   }                                                                        \
  else if (e == ((1ull<<15)-1))                                             \
   { if ((c_hi == 0) && (c_lo == 0)) inf;                                   \
     if ((c_hi&(1ull<<47))==0) bid_set_excepts (INVALID_EXCEPTION);         \
     nan(s,((((unsigned long long) c_hi) << 17) +                           \
            (((unsigned long long) c_lo) >> 47)),                           \
           (((unsigned long long) c_lo) << 17))                             \
   }                                                                        \
  else                                                                      \
   { c_hi += 1ull<<48;                                                      \
     t = ctz128(c_hi,c_lo);                                                 \
     e -= 16495;                                                            \
   }                                                                        \
}

// Pack and return decimal number from raw fields

#if !DECIMAL_CALL_BY_REFERENCE
#define return_bid32(s,e,c)                                                 \
  { if ((UINT32) (c) < (1ul<<23))                                           \
      return (((UINT32) (s) << 31) + ((UINT32) (e) << 23) + (UINT32) (c));  \
    else                                                                    \
      return (((UINT32) (s) << 31) + ((0x3ull<<29) - (1ull<<23)) +          \
             ((UINT32) (e) << 21) + (UINT32) (c));                          \
  }
#else
#define return_bid32(s,e,c)                                                 \
  { if ((UINT32) (c) < (1ul<<23))                                           \
      *pres = (((UINT32) (s) << 31) + ((UINT32) (e) << 23) + (UINT32) (c)); \
    else                                                                    \
      *pres = (((UINT32) (s) << 31) + ((0x3ull<<29) - (1ull<<23)) +         \
             ((UINT32) (e) << 21) + (UINT32) (c));                          \
    return;                                                                 \
  }
#endif

#if !DECIMAL_CALL_BY_REFERENCE
#define return_bid64(s,e,c)                                                 \
  { if ((c) < (1ull<<53))                                                   \
      return (((UINT64) (s) << 63) + ((UINT64) (e) << 53) + (c));           \
    else                                                                    \
      return (((UINT64) (s) << 63) + ((0x3ull<<61) - (1ull<<53)) +          \
             ((UINT64) (e) << 51) + (c));                                   \
  }
#else
#define return_bid64(s,e,c)                                                 \
  { if ((c) < (1ull<<53))                                                   \
      *pres = (((UINT64) (s) << 63) + ((UINT64) (e) << 53) + (c));          \
    else                                                                    \
      *pres = (((UINT64) (s) << 63) + ((0x3ull<<61) - (1ull<<53)) +         \
             ((UINT64) (e) << 51) + (c));                                   \
    return;                                                                 \
  }
#endif

#if !DECIMAL_CALL_BY_REFERENCE
#define return_bid128(s,e,c_hi,c_lo)                                        \
  { UINT128 x_out;                                                          \
    x_out.w[LOW_128W] = c_lo;                                          \
    x_out.w[HIGH_128W] = ((UINT64) (s) << 63) + ((UINT64) (e) << 49) +  \
                             (c_hi);                                        \
    return x_out;                                                           \
  }
#else
#define return_bid128(s,e,c_hi,c_lo)                                                    \
  { UINT128 x_out;                                                                      \
    x_out.w[LOW_128W] = c_lo;                                                      \
    x_out.w[HIGH_128W] = ((UINT64) (s) << 63) + ((UINT64) (e) << 49) + (c_hi);      \
    *pres = x_out;                                                                      \
    return;                                                                             \
  }
#endif

// Special cases of returning zero, infinity, NaN as decimal FP
// Take parameters for the sign, and for NaN the significand

#define return_bid32_zero(s) return_bid32(s,101,0)
#define return_bid32_inf(s) return_bid32(s,(0xF<<4),0)
#define return_bid32_nan(s,c_hi,c_lo)                                       \
  return_bid32(s,(0x1F<<3),(((c_hi>>44) > 999999ul) ? 0 : (c_hi>>44)));

#define return_bid64_zero(s) return_bid64(s,398,0)
#define return_bid64_inf(s) return_bid64(s,(0xF<<6),0)
#define return_bid64_nan(s,c_hi,c_lo)                                       \
  return_bid64(s,(0x1F<<5),                                                 \
               (((c_hi>>14) > 999999999999999ull) ? 0 : (c_hi>>14)));

#define return_bid128_zero(s) return_bid128(s,6176,0,0)
#define return_bid128_inf(s) return_bid128(s,(0xF<<10),0,0)
#define return_bid128_nan(s,c_hi,c_lo)                                      \
 { if (lt128(54210108624275ull,4089650035136921599ull,                      \
             (c_hi>>18),((c_lo>>18)+(c_hi<<46))))                           \
   return_bid128(s,(0x1F<<9),0ull,0ull)                                     \
   else return_bid128(s,(0x1F<<9),(c_hi>>18),((c_lo>>18)+(c_hi<<46)))       \
 }

// Unpack decimal floating-point number x into sign,exponent,coefficient
// In special cases, call the macros provided
// Coefficient is normalized in the binary sense with postcorrection k,
// so that x = 10^e * c / 2^k and the range of c is:
//
// 2^23 <= c < 2^24   (decimal32)
// 2^53 <= c < 2^54   (decimal64)
// 2^112 <= c < 2^113 (decimal128)

#define unpack_bid32(x,s,e,k,c,zero,inf,nan)                                \
{ s = x >> 31;                                                              \
  if ((x & (3ull<<29)) == (3ull<<29))                                       \
   { if ((x & (0xFull<<27)) == (0xFull<<27))                                \
      { if ((x & (0x1Full<<26)) != (0x1Full<<26)) inf;                      \
        if ((x & (1ul<<25))!=0) bid_set_excepts(INVALID_EXCEPTION);         \
        nan(s,((((x) & 0xFFFFul) > 999999ul) ? 0 :                          \
               (((unsigned long long) x) << 44)),0ull);                     \
      }                                                                     \
     e = ((x >> 21) & ((1ull<<8)-1)) - 101;                                 \
     c = (1ull<<23) + (x & ((1ull<<21)-1));                                 \
     if ((unsigned long)(c) > 9999999ul) c = 0;                             \
     k = 0;                                                                 \
   }                                                                        \
  else                                                                      \
   { e = ((x >> 23) & ((1ull<<8)-1)) - 101;                                 \
     c = x & ((1ull<<23)-1);                                                \
     if (c == 0) zero;                                                      \
     k = clz32(c) - 8;                                                      \
     c = c << k;                                                            \
   }                                                                        \
}

#define unpack_bid64(x,s,e,k,c,zero,inf,nan)                                \
{ s = x >> 63;                                                              \
  if ((x & (3ull<<61)) == (3ull<<61))                                       \
   { if ((x & (0xFull<<59)) == (0xFull<<59))                                \
      { if ((x & (0x1Full<<58)) != (0x1Full<<58)) inf;                      \
        if ((x & (1ull<<57))!=0) bid_set_excepts (INVALID_EXCEPTION);       \
        nan(s,((((x) & 0x3FFFFFFFFFFFFull) > 999999999999999ull) ? 0 :      \
               (((unsigned long long) x) << 14)),0ull);                     \
      }                                                                     \
     e = ((x >> 51) & ((1ull<<10)-1)) - 398;                                \
     c = (1ull<<53) + (x & ((1ull<<51)-1));                                 \
     if ((unsigned long long)(c) > 9999999999999999ull) c = 0;              \
     k = 0;                                                                 \
   }                                                                        \
  else                                                                      \
   { e = ((x >> 53) & ((1ull<<10)-1)) - 398;                                \
     c = x & ((1ull<<53)-1);                                                \
     if (c == 0) zero;                                                      \
     k = clz64(c) - 10;                                                     \
     c = c << k;                                                            \
   }                                                                        \
}

#define unpack_bid128(x,s,e,k,c,zero,inf,nan)                               \
{ s = x.w[HIGH_128W] >> 63;                                             \
  if ((x.w[HIGH_128W] & (3ull<<61)) == (3ull<<61))                      \
   { if ((x.w[HIGH_128W] & (0xFull<<59)) == (0xFull<<59))               \
      { if ((x.w[HIGH_128W] & (0x1Full<<58)) != (0x1Full<<58)) inf;     \
        if ((x.w[HIGH_128W] & (1ull<<57))!=0)                           \
           bid_set_excepts (INVALID_EXCEPTION);                             \
        if (lt128(54210108624275ull,4089650035136921599ull,                 \
            (x.w[HIGH_128W] & 0x3FFFFFFFFFFFull),x.w[LOW_128W]))   \
          nan(s,0ull,0ull);                                                 \
        nan(s,((((unsigned long long) x.w[HIGH_128W]) << 18) +          \
               (((unsigned long long) x.w[LOW_128W]) >> 46)),          \
              (((unsigned long long) x.w[LOW_128W]) << 18));           \
      }                                                                     \
     zero;                                                                  \
   }                                                                        \
  else                                                                      \
   { e = ((x.w[HIGH_128W] >> 49) & ((1ull<<14)-1)) - 6176;              \
     c.w[1] = x.w[HIGH_128W] & ((1ull<<49)-1);                          \
     c.w[0] = x.w[LOW_128W];                                           \
     if (lt128(542101086242752ull,4003012203950112767ull,                   \
               c.w[1],c.w[0]))                                              \
      { c.w[1] = 0ull; c.w[0] = 0ull; }                                     \
     if ((c.w[1] == 0) && (c.w[0] == 0)) zero;                              \
     k = clz128(c.w[1],c.w[0]) - 15;                                        \
     sll128(c.w[1],c.w[0],k);                                               \
   }                                                                        \
}

// Rounding boundaries table, indexed by
// 4 * rounding_mode + 2 * sign + lsb of truncation
// We round up if the round/sticky data is strictly > this boundary
//
// NB: This depends on the particular values of the rounding mode
// numbers, which are supposed to be defined as here:
//
// #define ROUNDING_TO_NEAREST     0x00000
// #define ROUNDING_DOWN           0x00001
// #define ROUNDING_UP             0x00002
// #define ROUNDING_TO_ZERO        0x00003
// #define ROUNDING_TIES_AWAY      0x00004
//
// Some of the shortcuts below in "underflow after rounding" also use
// the concrete values.
//
// So we add a directive here to double-check that this is the case

#if ((ROUNDING_TO_NEAREST!=0) || (ROUNDING_DOWN!=1) ||          \
     (ROUNDING_UP!=2) || (ROUNDING_TO_ZERO!=3) ||               \
     (ROUNDING_TIES_AWAY!=4))
#error "Rounding mode numbers don't match tables for binary/decimal conversion"
#endif

static const UINT128 roundbound_128[] = { {{0ull, (1ull << 63)}},       // ROUNDING_TO_NEAREST | positive | even
{{~0ull, (1ull << 63) - 1}},    // ROUNDING_TO_NEAREST | positive | odd
{{0ull, (1ull << 63)}}, // ROUNDING_TO_NEAREST | negative | even
{{~0ull, (1ull << 63) - 1}},    // ROUNDING_TO_NEAREST | negative | odd

{{~0ull, ~0ull}},       // ROUNDING_DOWN       | positive | even
{{~0ull, ~0ull}},       // ROUNDING_DOWN       | positive | odd
{{0ull, 0ull}}, // ROUNDING_DOWN       | negative | even
{{0ull, 0ull}}, // ROUNDING_DOWN       | negative | odd

{{0ull, 0ull}}, // ROUNDING_UP         | positive | even
{{0ull, 0ull}}, // ROUNDING_UP         | positive | odd
{{~0ull, ~0ull}},       // ROUNDING_UP         | negative | even
{{~0ull, ~0ull}},       // ROUNDING_UP         | negative | odd

{{~0ull, ~0ull}},       // ROUNDING_TO_ZERO    | positive | even
{{~0ull, ~0ull}},       // ROUNDING_TO_ZERO    | positive | odd
{{~0ull, ~0ull}},       // ROUNDING_TO_ZERO    | negative | even
{{~0ull, ~0ull}},       // ROUNDING_TO_ZERO    | negative | odd

{{~0ull, (1ull << 63) - 1}},    // ROUNDING_TIES_AWAY  | positive | even
{{~0ull, (1ull << 63) - 1}},    // ROUNDING_TIES_AWAY  | positive | odd
{{~0ull, (1ull << 63) - 1}},    // ROUNDING_TIES_AWAY  | negative | even
{{~0ull, (1ull << 63) - 1}}     // ROUNDING_TIES_AWAY  | negative | odd
};

// Table of powers of 5

static const UINT128 power_five[] = { {{1ull, 0ull}},
{{5ull, 0ull}},
{{25ull, 0ull}},
{{125ull, 0ull}},
{{625ull, 0ull}},
{{3125ull, 0ull}},
{{15625ull, 0ull}},
{{78125ull, 0ull}},
{{390625ull, 0ull}},
{{1953125ull, 0ull}},
{{9765625ull, 0ull}},
{{48828125ull, 0ull}},
{{244140625ull, 0ull}},
{{1220703125ull, 0ull}},
{{6103515625ull, 0ull}},
{{30517578125ull, 0ull}},
{{152587890625ull, 0ull}},
{{762939453125ull, 0ull}},
{{3814697265625ull, 0ull}},
{{19073486328125ull, 0ull}},
{{95367431640625ull, 0ull}},
{{476837158203125ull, 0ull}},
{{2384185791015625ull, 0ull}},
{{11920928955078125ull, 0ull}},
{{59604644775390625ull, 0ull}},
{{298023223876953125ull, 0ull}},
{{1490116119384765625ull, 0ull}},
{{7450580596923828125ull, 0ull}},
{{359414837200037393ull, 2ull}},
{{1797074186000186965ull, 10ull}},
{{8985370930000934825ull, 50ull}},
{{8033366502585570893ull, 252ull}},
{{3273344365508751233ull, 1262ull}},
{{16366721827543756165ull, 6310ull}},
{{8046632842880574361ull, 31554ull}},
{{3339676066983768573ull, 157772ull}},
{{16698380334918842865ull, 788860ull}},
{{9704925379756007861ull, 3944304ull}},
{{11631138751360936073ull, 19721522ull}},
{{2815461535676025517ull, 98607613ull}},
{{14077307678380127585ull, 493038065ull}},
{{15046306170771983077ull, 2465190328ull}},
{{1444554559021708921ull, 12325951644ull}},
{{7222772795108544605ull, 61629758220ull}},
{{17667119901833171409ull, 308148791101ull}},
{{14548623214327650581ull, 1540743955509ull}},
{{17402883850509598057ull, 7703719777548ull}},
{{13227442957709783821ull, 38518598887744ull}},
{{10796982567420264257ull, 192592994438723ull}}
};


// Tables of values for the various conversions:
//
// exponents: table of output exponents
// breakpoints: test values to decide between two possible exponents
// multipliers1/multipliers2: corresponding reciprocal multipliers
// coefflimits: used in exactness checks
//

static const UINT128 breakpoints_binary32[] =
  { {{17291492046443221751ull, 474778387287989ull}},
{{17522542451896487724ull, 379822709830391ull}},
{{10328685146775279856ull, 303858167864313ull}},
{{12836547420098537447ull, 486173068582901ull}},
{{6579889121336919634ull, 388938454866321ull}},
{{1574562482327625384ull, 311150763893057ull}},
{{6208648786466110938ull, 497841222228891ull}},
{{1277570214430978427ull, 398272977783113ull}},
{{8400753801028603388ull, 318618382226490ull}},
{{13441206081645765421ull, 509789411562384ull}},
{{14442313680058522660ull, 407831529249907ull}},
{{4175153314562997481ull, 326265223399926ull}},
{{17748291747526526940ull, 522024357439881ull}},
{{10509284583279311229ull, 417619485951905ull}},
{{8407427666623448983ull, 334095588761524ull}},
{{2383837822371787403ull, 534552942018439ull}},
{{5596419072639340246ull, 427642353614751ull}},
{{787786443369561873ull, 342113882891801ull}},
{{12328504753617029967ull, 547382212626881ull}},
{{6173454988151713650ull, 437905770101505ull}},
{{4938763990521370920ull, 350324616081204ull}},
{{15280720014318014119ull, 560519385729926ull}},
{{8535227196712500972ull, 448415508583941ull}},
{{3138832942628090454ull, 358732406867153ull}},
{{9889763983586293010ull, 286985925493722ull}},
{{1066227114770427523ull, 459177480789956ull}},
{{15610376950783983311ull, 367341984631964ull}},
{{16177650375369096972ull, 293873587705571ull}},
{{58798897397182893ull, 470197740328915ull}},
{{47039117917746314ull, 376158192263132ull}},
{{11105677738559928021ull, 300926553810505ull}},
{{17769084381695884834ull, 481482486096808ull}},
{{3147221061130976897ull, 385185988877447ull}},
{{13585823293130512487ull, 308148791101957ull}},
{{6979922010041178687ull, 493038065763132ull}},
{{16651984052258673919ull, 394430452610505ull}},
{{13321587241806939135ull, 315544362088404ull}},
{{10246493142665371647ull, 504870979341447ull}},
{{818496884648476671ull, 403896783473158ull}},
{{8033495137202601983ull, 323117426778526ull}},
{{5474894590040342527ull, 516987882845642ull}},
{{15447962116258004991ull, 413590306276513ull}},
{{1290323248780673023ull, 330872245021211ull}},
{{13132563642274807807ull, 529395592033937ull}},
{{3127353284336025599ull, 423516473627150ull}},
{{2501882627468820479ull, 338813178901720ull}},
{{4003012203950112767ull, 542101086242752ull}},
{{14270456207385821183ull, 433680868994201ull}},
{{7727016151166746623ull, 346944695195361ull}},
{{4984528212382973951ull, 555111512312578ull}},
{{11366320199390199807ull, 444089209850062ull}},
{{1714358530028339199ull, 355271367880050ull}},
{{1371486824022671359ull, 284217094304040ull}},
{{2194378918436274175ull, 454747350886464ull}},
{{5444851949490929663ull, 363797880709171ull}},
{{666532744850833407ull, 291038304567337ull}},
{{4755801206503243775ull, 465661287307739ull}},
{{7493989779944505343ull, 372529029846191ull}},
{{2305843009213693951ull, 298023223876953ull}},
{{18446744073709551615ull, 476837158203124ull}},
{{18446744073709551615ull, 381469726562499ull}},
{{18446744073709551615ull, 305175781249999ull}},
{{18446744073709551615ull, 488281249999999ull}},
{{18446744073709551615ull, 390624999999999ull}},
{{18446744073709551615ull, 312499999999999ull}},
{{18446744073709551615ull, 499999999999999ull}},
{{18446744073709551615ull, 399999999999999ull}},
{{18446744073709551615ull, 319999999999999ull}},
{{18446744073709551615ull, 511999999999999ull}},
{{18446744073709551615ull, 409599999999999ull}},
{{18446744073709551615ull, 327679999999999ull}},
{{18446744073709551615ull, 524287999999999ull}},
{{18446744073709551615ull, 419430399999999ull}},
{{18446744073709551615ull, 335544319999999ull}},
{{18446744073709551615ull, 536870911999999ull}},
{{18446744073709551615ull, 429496729599999ull}},
{{18446744073709551615ull, 343597383679999ull}},
{{18446744073709551615ull, 549755813887999ull}},
{{18446744073709551615ull, 439804651110399ull}},
{{18446744073709551615ull, 351843720888319ull}},
{{18446744073709551615ull, 281474976710655ull}},
{{11068046444225730969ull, 450359962737049ull}},
{{12543785970122495098ull, 360287970189639ull}},
{{13724377590839906402ull, 288230376151711ull}},
{{14580306515860029597ull, 461168601842738ull}},
{{596198768462292708ull, 368934881474191ull}},
{{15234354273737475459ull, 295147905179352ull}},
{{9617571579012319442ull, 472236648286964ull}},
{{11383406077951765876ull, 377789318629571ull}},
{{5417376047619502378ull, 302231454903657ull}},
{{12357150490933114128ull, 483570327845851ull}},
{{6196371578004580979ull, 386856262276681ull}},
{{1267748447661754460ull, 309485009821345ull}},
{{2028397516258807136ull, 495176015714152ull}},
{{12690764457232776679ull, 396140812571321ull}},
{{6463262751044311020ull, 316912650057057ull}},
{{14030569216412807955ull, 507060240091291ull}},
{{7535106558388336041ull, 405648192073033ull}},
{{13406782876194489479ull, 324518553658426ull}},
{{14072154972427362520ull, 519229685853482ull}},
{{3879026348458069369ull, 415383748682786ull}},
{{17860616337734096788ull, 332306998946228ull}},
{{6440893251923092922ull, 531691198313966ull}},
{{1463365786796564015ull, 425352958651173ull}},
{{8549390258921071858ull, 340282366920938ull}},
{{9989675599531804650ull, 544451787073501ull}},
{{4302391664883533397ull, 435561429658801ull}},
{{18199308590874468010ull, 348449143727040ull}},
{{10672149671689597200ull, 557518629963265ull}},
{{8537719737351677760ull, 446014903970612ull}},
{{17898222234107073178ull, 356811923176489ull}},
{{18007926602027568865ull, 285449538541191ull}},
{{2987240860050737922ull, 456719261665907ull}},
{{13457839132266321307ull, 365375409332725ull}},
{{10766271305813057046ull, 292300327466180ull}},
{{17226034089300891273ull, 467680523945888ull}},
{{2712780827214982049ull, 374144419156711ull}},
{{16927619920739626932ull, 299315535325368ull}},
{{4948098984731941152ull, 478904856520590ull}},
{{3958479187785552922ull, 383123885216472ull}}
};

static const int exponents_binary32[] = { -27,
  -24,
  -21,
  -17,
  -14,
  -11,
  -7,
  -4,
  -1,
  3,
  6,
  9,
  13,
  16,
  19,
  23,
  26,
  29,
  33,
  36,
  39,
  43,
  46,
  49,
  52,
  56,
  59,
  62,
  66,
  69,
  72,
  76,
  79,
  82,
  86,
  89,
  92,
  96,
  99,
  102,
  106,
  109,
  112,
  116,
  119,
  122,
  126,
  129,
  132,
  136,
  139,
  142,
  145,
  149,
  152,
  155,
  159,
  162,
  165,
  169,
  172,
  175,
  179,
  182,
  185,
  189,
  192,
  195,
  199,
  202,
  205,
  209,
  212,
  215,
  219,
  222,
  225,
  229,
  232,
  235,
  238,
  242,
  245,
  248,
  252,
  255,
  258,
  262,
  265,
  268,
  272,
  275,
  278,
  282,
  285,
  288,
  292,
  295,
  298,
  302,
  305,
  308,
  312,
  315,
  318,
  322,
  325,
  328,
  332,
  335,
  338,
  341,
  345,
  348,
  351,
  355,
  358,
  361,
  365,
  368,
};

static const UINT256 multipliers1_binary32[] =
  { {{6013890151484785128ull, 7481633477359093489ull,
      655737588518723529ull, 651851512427ull}},
{{12129048707783369314ull, 13963727865126254765ull,
  14654730040930568123ull, 814814390533ull}},
{{1326252829447047930ull, 12842973812980430553ull,
  4483354495881046442ull, 1018517988167ull}},
{{12358123064472874716ull, 12638544651540156999ull,
  9719625587566735882ull, 636573742604ull}},
{{10835967812163705491ull, 6574808777570420441ull,
  12149531984458419853ull, 795717178255ull}},
{{18156645783632019768ull, 12830196990390413455ull,
  10575228962145636912ull, 994646472819ull}},
{{18265432642411094211ull, 8018873118994008409ull,
  4303675092127329118ull, 621654045512ull}},
{{8996732747731704052ull, 800219361887734704ull, 5379593865159161398ull,
  777067556890ull}},
{{11245915934664630065ull, 10223646239214444188ull,
  15947864368303727555ull, 971334446112ull}},
{{16252069496020169599ull, 4083935890295333665ull,
  9967415230189829722ull, 607084028820ull}},
{{6480028814743048286ull, 14328291899723942890ull,
  12459269037737287152ull, 758855036025ull}},
{{17323408055283586166ull, 17910364874654928612ull,
  1739028241889445228ull, 948568795032ull}},
{{1603757997697465546ull, 1970606009804554575ull,
  1086892651180903268ull, 592855496895ull}},
{{15839755552403995644ull, 2463257512255693218ull,
  15193673869258292797ull, 741069371118ull}},
{{10576322403650218747ull, 7690757908747004427ull,
  9768720299718090188ull, 926336713898ull}},
{{4304358493067692765ull, 14030095729821653575ull,
  10717136205751194271ull, 578960446186ull}},
{{768762097907228052ull, 12925933643849679065ull,
  4173048220334217031ull, 723700557733ull}},
{{5572638640811422969ull, 11545731036384710927ull,
  9827996293845159193ull, 904625697166ull}},
{{10400428178148221212ull, 298552870099362473ull,
  1530811665225836592ull, 565391060729ull}},
{{17612221241112664419ull, 373191087624203091ull,
  6525200599959683644ull, 706738825911ull}},
{{17403590532963442619ull, 466488859530253864ull,
  3544814731522216651ull, 883423532389ull}},
{{10877244083102151637ull, 16432456601702266329ull,
  4521352216415079358ull, 552139707743ull}},
{{18208241122305077450ull, 11317198715273057103ull,
  1040004252091461294ull, 690174634679ull}},
{{18148615384453958909ull, 4923126357236545571ull,
  15135063370396490330ull, 862718293348ull}},
{{18074083212140060732ull, 15377279983400457772ull,
  472085139286061296ull, 1078397866686ull}},
{{2072929970732762150ull, 9610799989625286108ull,
  14130111267335952022ull, 673998666678ull}},
{{2591162463415952687ull, 2790127950176831827ull,
  8439267047315164220ull, 842498333348ull}},
{{17074011134552104570ull, 3487659937721039783ull,
  10549083809143955275ull, 1053122916685ull}},
{{17588785986736147213ull, 18320688525571507528ull,
  8899020389928665998ull, 658201822928ull}},
{{3539238409710632400ull, 13677488620109608603ull,
  11123775487410832498ull, 822752278660ull}},
{{18259106067420454212ull, 7873488738282234945ull,
  13904719359263540623ull, 1028440348325ull}},
{{4494412264496702026ull, 11838459489067478697ull,
  10996292608753406841ull, 642775217703ull}},
{{10229701349048265437ull, 963016306052184659ull,
  9133679742514370648ull, 803469022129ull}},
{{8175440667882943892ull, 1203770382565230824ull,
  16028785696570351214ull, 1004336277661ull}},
{{5109650417426839933ull, 14587414544385432977ull,
  12323834069570163460ull, 627710173538ull}},
{{10998749040210937820ull, 18234268180481791221ull,
  6181420550107928517ull, 784637716923ull}},
{{18360122318691060179ull, 8957777170320075314ull,
  3115089669207522743ull, 980797146154ull}},
{{16086762467609300516ull, 12516139759091128927ull,
  6558617061682089618ull, 612998216346ull}},
{{15496767066084237741ull, 6421802662009135351ull,
  17421643363957387831ull, 766247770432ull}},
{{14759272814177909272ull, 3415567309084031285ull,
  3330310131237183173ull, 957809713041ull}},
{{11530388518074887247ull, 4440572577391213505ull,
  13610658878091709243ull, 598631070650ull}},
{{577927592311445347ull, 939029703311628978ull, 7789951560759860746ull,
  748288838313ull}},
{{9945781527244082491ull, 10397159165994312030ull,
  14349125469377213836ull, 935361047891ull}},
{{1604427436100163653ull, 15721596515601220827ull,
  6662360409147064695ull, 584600654932ull}},
{{15840592350407368278ull, 15040309626074138129ull,
  8327950511433830869ull, 730750818665ull}},
{{5965682382727046636ull, 4965328977310508950ull,
  15021624157719676491ull, 913438523331ull}},
{{17563609544486567859ull, 797487601605374141ull,
  7082672089361103855ull, 570899077082ull}},
{{8119453875326046112ull, 14831917557288881389ull,
  18076712148556155626ull, 713623846352ull}},
{{14761003362584945544ull, 9316524909756325928ull,
  4149146111985642917ull, 892029807941ull}},
{{9225627101615590965ull, 8128671077811397657ull,
  4899059329204720775ull, 557518629963ull}},
{{16143719895446876610ull, 5549152828836859167ull,
  1512138143078513065ull, 696898287454ull}},
{{15567963850881207859ull, 11548127054473461863ull,
  11113544715702917139ull, 871122859317ull}},
{{14848268795174121920ull, 9823472799664439425ull, 56872839346482712ull,
  1088903574147ull}},
{{2362638969342744344ull, 6139670499790274641ull,
  16176446589087409359ull, 680564733841ull}},
{{7564984730105818334ull, 3062902106310455397ull,
  6385500181077097987ull, 850705917302ull}},
{{14067916931059660821ull, 17663685688170232958ull,
  17205247263201148291ull, 1063382396627ull}},
{{4180762063484900109ull, 8733960545892701647ull,
  8447436530287023730ull, 664613997892ull}},
{{614266560928737233ull, 1694078645511101251ull,
  10559295662858779663ull, 830767497365ull}},
{{14602891256443085253ull, 15952656362171040275ull,
  17810805597000862482ull, 1038459371706ull}},
{{6820964026063234331ull, 14582096244784288076ull,
  15743439516552926955ull, 649037107316ull}},
{{8526205032579042914ull, 13615934287552972191ull,
  1232555321981607078ull, 811296384146ull}},
{{6046070272296415738ull, 7796545822586439431ull,
  10764066189331784656ull, 1014120480182ull}},
{{10696322947826341692ull, 4872841139116524644ull,
  2115855349904977506ull, 633825300114ull}},
{{13370403684782927115ull, 15314423460750431613ull,
  11868191224235997690ull, 792281625142ull}},
{{2877946550696495182ull, 9919657289083263709ull,
  5611866993440221305ull, 990352031428ull}},
{{4104559603399003441ull, 17729000851745509578ull,
  12730788907754914123ull, 618970019642ull}},
{{14354071541103530109ull, 17549565046254499068ull,
  6690114097838866846ull, 773712524553ull}},
{{17942589426379412636ull, 12713584270963348027ull,
  12974328640725971462ull, 967140655691ull}},
{{8908275382273438946ull, 3334304150924704613ull,
  5803112391240038212ull, 604462909807ull}},
{{15747030246269186586ull, 4167880188655880766ull,
  2642204470622659861ull, 755578637259ull}},
{{10460415770981707425ull, 9821536254247238862ull,
  17137813643560488538ull, 944473296573ull}},
{{1926073838436179237ull, 10750146177331912193ull,
  13016976536438999288ull, 590295810358ull}},
{{7019278316472611950ull, 13437682721664890241ull,
  7047848633693973302ull, 737869762948ull}},
{{13385783914018152841ull, 7573731365226336993ull,
  8809810792117466628ull, 922337203685ull}},
{{1448585918620263670ull, 13956954140121236429ull,
  7811974754287110594ull, 576460752303ull}},
{{6422418416702717491ull, 8222820638296769728ull,
  5153282424431500339ull, 720575940379ull}},
{{8028023020878396864ull, 5666839779443574256ull,
  1829917012111987520ull, 900719925474ull}},
{{5017514388048998040ull, 3541774862152233910ull,
  5755384150997380104ull, 562949953421ull}},
{{15495265021916023358ull, 4427218577690292387ull,
  11805916207174113034ull, 703687441776ull}},
{{14757395258967641293ull, 14757395258967641292ull,
  14757395258967641292ull, 879609302220ull}},
{{0ull, 0ull, 0ull, 1099511627776ull}},
{{0ull, 0ull, 0ull, 687194767360ull}},
{{0ull, 0ull, 0ull, 858993459200ull}},
{{0ull, 0ull, 0ull, 1073741824000ull}},
{{0ull, 0ull, 0ull, 671088640000ull}},
{{0ull, 0ull, 0ull, 838860800000ull}},
{{0ull, 0ull, 0ull, 1048576000000ull}},
{{0ull, 0ull, 0ull, 655360000000ull}},
{{0ull, 0ull, 0ull, 819200000000ull}},
{{0ull, 0ull, 0ull, 1024000000000ull}},
{{0ull, 0ull, 0ull, 640000000000ull}},
{{0ull, 0ull, 0ull, 800000000000ull}},
{{0ull, 0ull, 0ull, 1000000000000ull}},
{{0ull, 0ull, 0ull, 625000000000ull}},
{{0ull, 0ull, 0ull, 781250000000ull}},
{{0ull, 0ull, 0ull, 976562500000ull}},
{{0ull, 0ull, 0ull, 610351562500ull}},
{{0ull, 0ull, 0ull, 762939453125ull}},
{{0ull, 0ull, 4611686018427387904ull, 953674316406ull}},
{{0ull, 0ull, 16717361816799281152ull, 596046447753ull}},
{{0ull, 0ull, 7061644215716937728ull, 745058059692ull}},
{{0ull, 0ull, 8827055269646172160ull, 931322574615ull}},
{{0ull, 0ull, 12434438571169939456ull, 582076609134ull}},
{{0ull, 0ull, 6319676177107648512ull, 727595761418ull}},
{{0ull, 0ull, 17122967258239336448ull, 909494701772ull}},
{{0ull, 0ull, 1478482499544809472ull, 568434188608ull}},
{{0ull, 0ull, 1848103124431011840ull, 710542735760ull}},
{{0ull, 0ull, 2310128905538764800ull, 888178419700ull}},
{{0ull, 0ull, 10667202602816503808ull, 555111512312ull}},
{{0ull, 0ull, 13334003253520629760ull, 693889390390ull}},
{{0ull, 0ull, 7444132030046011392ull, 867361737988ull}},
{{0ull, 0ull, 9305165037557514240ull, 1084202172485ull}},
{{0ull, 0ull, 8121571157687140352ull, 677626357803ull}},
{{0ull, 0ull, 5540277928681537536ull, 847032947254ull}},
{{0ull, 0ull, 16148719447706697728ull, 1058791184067ull}},
{{0ull, 0ull, 7787106645602992128ull, 661744490042ull}},
{{0ull, 0ull, 510511270148964352ull, 827180612553ull}},
{{0ull, 0ull, 5249825106113593344ull, 1033975765691ull}},
{{0ull, 0ull, 975297682107301888ull, 646234853557ull}},
{{0ull, 0ull, 5830808121061515264ull, 807793566946ull}}
};

static const UINT256 multipliers2_binary32[] =
  { {{12230317112597168372ull, 12964188775534322552ull,
      9551240831114137572ull, 325925756213ull}},
{{15287896390746460465ull, 16205235969417903190ull,
  16550737057320059869ull, 407407195266ull}},
{{9886498451578299773ull, 6421486906490215276ull,
  11465049284795299029ull, 509258994083ull}},
{{15402433569091213166ull, 6319272325770078499ull,
  4859812793783367941ull, 318286871302ull}},
{{14641355942936628554ull, 12510776425639986028ull,
  15298138029083985734ull, 397858589127ull}},
{{18301694928670785692ull, 6415098495195206727ull,
  14510986517927594264ull, 497323236409ull}},
{{18356088358060322914ull, 4009436559497004204ull,
  2151837546063664559ull, 310827022756ull}},
{{4498366373865852026ull, 400109680943867352ull, 2689796932579580699ull,
  388533778445ull}},
{{5622957967332315033ull, 14335195156461997902ull,
  7973932184151863777ull, 485667223056ull}},
{{17349406784864860608ull, 2041967945147666832ull,
  4983707615094914861ull, 303542014410ull}},
{{3240014407371524143ull, 7164145949861971445ull,
  15453006555723419384ull, 379427518012ull}},
{{8661704027641793083ull, 8955182437327464306ull, 869514120944722614ull,
  474284397516ull}},
{{10025251035703508581ull, 985303004902277287ull,
  9766818362445227442ull, 296427748447ull}},
{{7919877776201997822ull, 10455000792982622417ull,
  7596836934629146398ull, 370534685559ull}},
{{14511533238679885182ull, 3845378954373502213ull,
  4884360149859045094ull, 463168356949ull}},
{{11375551283388622191ull, 16238419901765602595ull,
  5358568102875597135ull, 289480223093ull}},
{{9607753085808389834ull, 15686338858779615340ull,
  11309896147021884323ull, 361850278866ull}},
{{12009691357260487293ull, 14996237555047131271ull,
  4913998146922579596ull, 452312848583ull}},
{{14423586125928886414ull, 149276435049681236ull,
  9988777869467694104ull, 282695530364ull}},
{{18029482657411108018ull, 186595543812101545ull,
  12485972336834617630ull, 353369412955ull}},
{{8701795266481721310ull, 9456616466619902740ull,
  10995779402615884133ull, 441711766194ull}},
{{14661994078405851627ull, 8216228300851133164ull,
  11484048145062315487ull, 276069853871ull}},
{{18327492598007314533ull, 5658599357636528551ull,
  9743374162900506455ull, 345087317339ull}},
{{18297679729081755263ull, 2461563178618272785ull,
  7567531685198245165ull, 431359146674ull}},
{{9037041606070030366ull, 7688639991700228886ull, 236042569643030648ull,
  539198933343ull}},
{{1036464985366381075ull, 4805399994812643054ull,
  7065055633667976011ull, 336999333339ull}},
{{10518953268562752152ull, 1395063975088415913ull,
  4219633523657582110ull, 421249166674ull}},
{{17760377604130828093ull, 10967202005715295699ull,
  14497913941426753445ull, 526561458342ull}},
{{8794392993368073607ull, 9160344262785753764ull,
  4449510194964332999ull, 329100911464ull}},
{{10992991241710092008ull, 6838744310054804301ull,
  5561887743705416249ull, 411376139330ull}},
{{18352925070565002914ull, 13160116405995893280ull,
  16175731716486546119ull, 514220174162ull}},
{{11470578169103126821ull, 15142601781388515156ull,
  14721518341231479228ull, 321387608851ull}},
{{14338222711378908527ull, 481508153026092329ull,
  13790211908111961132ull, 401734511064ull}},
{{4087720333941471946ull, 601885191282615412ull,
  17237764885139951415ull, 502168138830ull}},
{{11778197245568195775ull, 7293707272192716488ull,
  6161917034785081730ull, 313855086769ull}},
{{14722746556960244718ull, 18340506127095671418ull,
  12314082311908740066ull, 392318858461ull}},
{{9180061159345530090ull, 13702260622014813465ull,
  1557544834603761371ull, 490398573077ull}},
{{17266753270659426066ull, 6258069879545564463ull,
  3279308530841044809ull, 306499108173ull}},
{{16971755569896894679ull, 12434273367859343483ull,
  8710821681978693915ull, 383123885216ull}},
{{16603008443943730444ull, 10931155691396791450ull,
  10888527102473367394ull, 478904856520ull}},
{{14988566295892219432ull, 11443658325550382560ull,
  6805329439045854621ull, 299315535325ull}},
{{288963796155722674ull, 469514851655814489ull, 13118347817234706181ull,
  374144419156ull}},
{{4972890763622041246ull, 5198579582997156015ull,
  16397934771543382726ull, 467680523945ull}},
{{10025585754904857635ull, 17084170294655386221ull,
  3331180204573532347ull, 292300327466ull}},
{{17143668212058459947ull, 16743526849891844872ull,
  13387347292571691242ull, 365375409332ull}},
{{2982841191363523318ull, 11706036525510030283ull,
  16734184115714614053ull, 456719261665ull}},
{{18005176809098059738ull, 9622115837657462878ull,
  3541336044680551927ull, 285449538541ull}},
{{13283098974517798864ull, 7415958778644440694ull,
  9038356074278077813ull, 356811923176ull}},
{{7380501681292472772ull, 13881634491732938772ull,
  11297945092847597266ull, 446014903970ull}},
{{13836185587662571291ull, 13287707575760474636ull,
  11672901701457136195ull, 278759314981ull}},
{{17295231984578214113ull, 11997948451273205391ull,
  756069071539256532ull, 348449143727ull}},
{{17007353962295379738ull, 14997435564091506739ull,
  14780144394706234377ull, 435561429658ull}},
{{16647506434441836768ull, 4911736399832219712ull,
  9251808456528017164ull, 544451787073ull}},
{{10404691521526147980ull, 12293207286749913128ull,
  17311595331398480487ull, 340282366920ull}},
{{13005864401907684975ull, 10754823090010003506ull,
  3192750090538548993ull, 425352958651ull}},
{{7033958465529830411ull, 18055214880939892287ull,
  17825995668455349953ull, 531691198313ull}},
{{11313753068597225863ull, 4366980272946350823ull,
  4223718265143511865ull, 332306998946ull}},
{{9530505317319144425ull, 10070411359610326433ull,
  14503019868284165639ull, 415383748682ull}},
{{16524817665076318435ull, 7976328181085520137ull,
  8905402798500431241ull, 519229685853ull}},
{{3410482013031617166ull, 16514420159246919846ull,
  7871719758276463477ull, 324518553658ull}},
{{13486474553144297265ull, 6807967143776486095ull,
  616277660990803539ull, 405648192073ull}},
{{12246407173002983677ull, 3898272911293219715ull,
  5382033094665892328ull, 507060240091ull}},
{{5348161473913170846ull, 2436420569558262322ull,
  1057927674952488753ull, 316912650057ull}},
{{15908573879246239366ull, 7657211730375215806ull,
  5934095612117998845ull, 396140812571ull}},
{{10662345312203023399ull, 14183200681396407662ull,
  2805933496720110652ull, 495176015714ull}},
{{2052279801699501721ull, 18087872462727530597ull,
  6365394453877457061ull, 309485009821ull}},
{{7177035770551765055ull, 8774782523127249534ull,
  12568429085774209231ull, 386856262276ull}},
{{18194666750044482126ull, 6356792135481674013ull,
  15710536357217761539ull, 483570327845ull}},
{{13677509727991495281ull, 1667152075462352306ull,
  12124928232474794914ull, 302231454903ull}},
{{7873515123134593293ull, 11307312131182716191ull,
  10544474272166105738ull, 377789318629ull}},
{{5230207885490853713ull, 4910768127123619431ull,
  17792278858635020077ull, 472236648286ull}},
{{10186408956072865427ull, 5375073088665956096ull,
  6508488268219499644ull, 295147905179ull}},
{{12733011195091081783ull, 6718841360832445120ull,
  3523924316846986651ull, 368934881474ull}},
{{15916263993863852229ull, 3786865682613168496ull,
  13628277432913509122ull, 461168601842ull}},
{{9947664996164907643ull, 6978477070060618214ull,
  13129359413998331105ull, 288230376151ull}},
{{3211209208351358746ull, 13334782356003160672ull,
  11800013249070525977ull, 360287970189ull}},
{{4014011510439198432ull, 2833419889721787128ull, 914958506055993760ull,
  450359962737ull}},
{{2508757194024499020ull, 1770887431076116955ull,
  12101064112353465860ull, 281474976710ull}},
{{16971004547812787487ull, 2213609288845146193ull,
  5902958103587056517ull, 351843720888ull}},
{{7378697629483820647ull, 7378697629483820646ull,
  7378697629483820646ull, 439804651110ull}},
{{0ull, 0ull, 0ull, 549755813888ull}},
{{0ull, 0ull, 0ull, 343597383680ull}},
{{0ull, 0ull, 0ull, 429496729600ull}},
{{0ull, 0ull, 0ull, 536870912000ull}},
{{0ull, 0ull, 0ull, 335544320000ull}},
{{0ull, 0ull, 0ull, 419430400000ull}},
{{0ull, 0ull, 0ull, 524288000000ull}},
{{0ull, 0ull, 0ull, 327680000000ull}},
{{0ull, 0ull, 0ull, 409600000000ull}},
{{0ull, 0ull, 0ull, 512000000000ull}},
{{0ull, 0ull, 0ull, 320000000000ull}},
{{0ull, 0ull, 0ull, 400000000000ull}},
{{0ull, 0ull, 0ull, 500000000000ull}},
{{0ull, 0ull, 0ull, 312500000000ull}},
{{0ull, 0ull, 0ull, 390625000000ull}},
{{0ull, 0ull, 0ull, 488281250000ull}},
{{0ull, 0ull, 0ull, 305175781250ull}},
{{0ull, 0ull, 9223372036854775808ull, 381469726562ull}},
{{0ull, 0ull, 2305843009213693952ull, 476837158203ull}},
{{0ull, 0ull, 17582052945254416384ull, 298023223876ull}},
{{0ull, 0ull, 3530822107858468864ull, 372529029846ull}},
{{0ull, 0ull, 13636899671677861888ull, 465661287307ull}},
{{0ull, 0ull, 6217219285584969728ull, 291038304567ull}},
{{0ull, 0ull, 3159838088553824256ull, 363797880709ull}},
{{0ull, 0ull, 8561483629119668224ull, 454747350886ull}},
{{0ull, 0ull, 739241249772404736ull, 284217094304ull}},
{{0ull, 0ull, 924051562215505920ull, 355271367880ull}},
{{0ull, 0ull, 1155064452769382400ull, 444089209850ull}},
{{0ull, 0ull, 5333601301408251904ull, 277555756156ull}},
{{0ull, 0ull, 6667001626760314880ull, 346944695195ull}},
{{0ull, 0ull, 3722066015023005696ull, 433680868994ull}},
{{0ull, 0ull, 13875954555633532928ull, 542101086242ull}},
{{0ull, 0ull, 13284157615698345984ull, 338813178901ull}},
{{0ull, 0ull, 2770138964340768768ull, 423516473627ull}},
{{0ull, 0ull, 17297731760708124672ull, 529395592033ull}},
{{0ull, 0ull, 3893553322801496064ull, 330872245021ull}},
{{0ull, 0ull, 9478627671929257984ull, 413590306276ull}},
{{0ull, 0ull, 11848284589911572480ull, 516987882845ull}},
{{0ull, 0ull, 9711020877908426752ull, 323117426778ull}},
{{0ull, 0ull, 2915404060530757632ull, 403896783473ull}}
};

// **********************************************************************

static const UINT128 breakpoints_binary64[] =
  { {{5261314576080512960ull, 21426681862861333ull}},
{{4728754506986910400ull, 34282690980578133ull}},
{{11161701235073348928ull, 27426152784462506ull}},
{{5240012173316768832ull, 21940922227570005ull}},
{{8384019477306830144ull, 35105475564112008ull}},
{{14085913211329284736ull, 28084380451289606ull}},
{{7579381754321517504ull, 22467504361031685ull}},
{{12127010806914427968ull, 35948006977650696ull}},
{{6012259830789632064ull, 28758405582120557ull}},
{{15877854308857436608ull, 23006724465696445ull}},
{{12702283447085949312ull, 18405379572557156ull}},
{{12944955885853698240ull, 29448607316091450ull}},
{{10355964708682958592ull, 23558885852873160ull}},
{{8284771766946366848ull, 18847108682298528ull}},
{{9566286012372276672ull, 30155373891677645ull}},
{{7653028809897821312ull, 24124299113342116ull}},
{{2433074233176346752ull, 19299439290673693ull}},
{{203569958340244480ull, 30879102865077909ull}},
{{3852204781414105920ull, 24703282292062327ull}},
{{14149810269357015680ull, 19762625833649861ull}},
{{15260998801487404480ull, 31620201333839778ull}},
{{1140752596964192576ull, 25296161067071823ull}},
{{8291299707055174720ull, 20236928853657458ull}},
{{9576730716546369216ull, 32379086165851933ull}},
{{15040082202720916032ull, 25903268932681546ull}},
{{8342716947434822464ull, 20722615146145237ull}},
{{17037695930637626304ull, 33156184233832379ull}},
{{17319505559252011392ull, 26524947387065903ull}},
{{2787558003175878144ull, 21219957909652723ull}},
{{770743990339494720ull, 33951932655444357ull}},
{{11684641636497326720ull, 27161546124355485ull}},
{{9347713309197861376ull, 21729236899484388ull}},
{{11266992479974667904ull, 34766779039175021ull}},
{{5324245169237824000ull, 27813423231340017ull}},
{{15327442579615990144ull, 22250738585072013ull}},
{{2387815238934122304ull, 35601181736115222ull}},
{{12978298635373028800ull, 28480945388892177ull}},
{{3003941278814602368ull, 22784756311113742ull}},
{{13471199467277412864ull, 18227805048890993ull}},
{{17864570332901950336ull, 29164488078225589ull}},
{{17981005081063470592ull, 23331590462580471ull}},
{{10695455250108866112ull, 18665272370064377ull}},
{{2355333141206544512ull, 29864435792103004ull}},
{{5573615327707145920ull, 23891548633682403ull}},
{{11837589891649537408ull, 19113238906945922ull}},
{{4182748567671618560ull, 30581182251113476ull}},
{{18103594113104936128ull, 24464945800890780ull}},
{{14482875290483948864ull, 19571956640712624ull}},
{{12104554020548587264ull, 31315130625140199ull}},
{{13372992031180780160ull, 25052104500112159ull}},
{{14387742439686534400ull, 20041683600089727ull}},
{{8262992644530813824ull, 32066693760143564ull}},
{{10299742930366561344ull, 25653355008114851ull}},
{{4550445529551338752ull, 20522684006491881ull}},
{{18348759291507873024ull, 32836294410387009ull}},
{{18368356247948208704ull, 26269035528309607ull}},
{{7315987368874746304ull, 21015228422647686ull}},
{{4326882160715773504ull, 33624365476236298ull}},
{{10840203358056439424ull, 26899492380989038ull}},
{{16050860315928972160ull, 21519593904791230ull}},
{{7234632431776803904ull, 34431350247665969ull}},
{{9477054760163353472ull, 27545080198132775ull}},
{{7581643808130682752ull, 22036064158506220ull}},
{{12130630093009092416ull, 35257702653609952ull}},
{{2325806444923453248ull, 28206162122887962ull}},
{{12928691600164493568ull, 22564929698310369ull}},
{{14032302094873505216ull, 18051943758648295ull}},
{{4004939278088056704ull, 28883110013837273ull}},
{{10582649051954265984ull, 23106488011069818ull}},
{{15844816871047233408ull, 18485190408855854ull}},
{{14283660549449842560ull, 29576304654169367ull}},
{{4048230810076053376ull, 23661043723335494ull}},
{{6927933462802753024ull, 18928834978668395ull}},
{{11084693540484404864ull, 30286135965869432ull}},
{{1489057202903703232ull, 24228908772695546ull}},
{{15948641021290603904ull, 19383127018156436ull}},
{{18139128004581145600ull, 31013003229050298ull}},
{{3443255959439185472ull, 24810402583240239ull}},
{{6443953582293258688ull, 19848322066592191ull}},
{{2931628102185393280ull, 31757315306547506ull}},
{{17102697740715955904ull, 25405852245238004ull}},
{{17371507007314675072ull, 20324681796190403ull}},
{{5658318323252018176ull, 32519490873904646ull}},
{{837305843859704192ull, 26015592699123717ull}},
{{11737891119313494336ull, 20812474159298973ull}},
{{15091276976159680640ull, 33299958654878357ull}},
{{4694323951443923840ull, 26639966923902686ull}},
{{66110346413228736ull, 21311973539122149ull}},
{{7484474183744986688ull, 34099157662595438ull}},
{{13366276976479809984ull, 27279326130076350ull}},
{{10693021581183848000ull, 21823460904061080ull}},
{{17108834529894156800ull, 34917537446497728ull}},
{{2619021179689594432ull, 27934029957198183ull}},
{{9473914573235496192ull, 22347223965758546ull}},
{{7779565687692973312ull, 35755558345213674ull}},
{{9913001364896288960ull, 28604446676170939ull}},
{{11619749906658941440ull, 22883557340936751ull}},
{{5606451110585242816ull, 18306845872749401ull}},
{{1591624147452567936ull, 29290953396399042ull}},
{{12341345762187785280ull, 23432762717119233ull}},
{{17251774239234048896ull, 18746210173695386ull}},
{{1777397079581105984ull, 29993936277912619ull}},
{{5111266478406795072ull, 23995149022330095ull}},
{{4089013182725436096ull, 19196119217864076ull}},
{{17610467536586428672ull, 30713790748582521ull}},
{{10399025214527232640ull, 24571032598866017ull}},
{{940522542137965440ull, 19656826079092814ull}},
{{8883533696904565376ull, 31450921726548502ull}},
{{18174873401749383296ull, 25160737381238801ull}},
{{10850549906657596288ull, 20128589904991041ull}},
{{9982182221168333440ull, 32205743847985666ull}},
{{4296396962192756416ull, 25764595078388533ull}},
{{10815815199238025792ull, 20611676062710826ull}},
{{9926606689297020608ull, 32978681700337322ull}},
{{562587721953795840ull, 26382945360269858ull}},
{{7828767807046857280ull, 21106356288215886ull}},
{{5147330861791151040ull, 33770170061145418ull}},
{{11496562318916741504ull, 27016136048916334ull}},
{{12886598669875303488ull, 21612908839133067ull}},
{{5861162612832844352ull, 34580654142612908ull}},
{{12067627719750096128ull, 27664523314090326ull}},
{{5964753361058166592ull, 22131618651272261ull}},
{{2164907748209245888ull, 35410589842035618ull}},
{{9110623828051217344ull, 28328471873628494ull}},
{{10977847877182884160ull, 22662777498902795ull}},
{{8782278301746307328ull, 18130221999122236ull}},
{{6672947653310271104ull, 29008355198595578ull}},
{{12717055752132037568ull, 23206684158876462ull}},
{{2794946972221809408ull, 18565347327101170ull}},
{{4471915155554895040ull, 29704555723361872ull}},
{{14645578568669646976ull, 23763644578689497ull}},
{{4337765225451896960ull, 19010915662951598ull}},
{{3251075545981124800ull, 30417465060722557ull}},
{{13668906881010630784ull, 24333972048578045ull}},
{{10935125504808504640ull, 19467177638862436ull}},
{{10117503178209786752ull, 31147484222179898ull}},
{{15472700172051650048ull, 24917987377743918ull}},
{{1310113693415589056ull, 19934389902195135ull}},
{{2096181909464942528ull, 31895023843512216ull}},
{{16434340786539595328ull, 25516019074809772ull}},
{{5768774999747855616ull, 20412815259847818ull}},
{{5540691184854658688ull, 32660504415756509ull}},
{{8121901762625637248ull, 26128403532605207ull}},
{{17565567854326240768ull, 20902722826084165ull}},
{{9658164493212433600ull, 33444356521734665ull}},
{{7726531594569946880ull, 26755485217387732ull}},
{{17249271719881688448ull, 21404388173910185ull}},
{{9152090678101149952ull, 34247021078256297ull}},
{{18389718986706650944ull, 27397616862605037ull}},
{{7333077559881500096ull, 21918093490084030ull}},
{{11732924095810400192ull, 35068949584134448ull}},
{{16765036906132140800ull, 28055159667307558ull}},
{{2343983080679981632ull, 22444127733846047ull}},
{{7439721743829880960ull, 35910604374153675ull}},
{{5951777395063904768ull, 28728483499322940ull}},
{{4761421916051123840ull, 22982786799458352ull}},
{{14877183977066630016ull, 18386229439566681ull}},
{{16424796733822787392ull, 29417967103306690ull}},
{{13139837387058229888ull, 23534373682645352ull}},
{{3133172280162763264ull, 18827498946116282ull}},
{{8702424463002331584ull, 30123998313786051ull}},
{{3272590755659954944ull, 24099198651028841ull}},
{{17375467863495605248ull, 19279358920823072ull}},
{{13043353322625327104ull, 30846974273316916ull}},
{{6745333843358351360ull, 24677579418653533ull}},
{{12774964704170501696ull, 19742063534922826ull}},
{{13061245897188982144ull, 31587301655876522ull}},
{{3070299088267365056ull, 25269841324701218ull}},
{{9834936900097712704ull, 20215873059760974ull}},
{{4667852595930609344ull, 32345396895617559ull}},
{{7423630891486397760ull, 25876317516494047ull}},
{{17006951157414849216ull, 20701054013195237ull}},
{{12453726592896117440ull, 33121686421112380ull}},
{{9962981274316893952ull, 26497349136889904ull}},
{{11659733834195425472ull, 21197879309511923ull}},
{{14966225319970770432ull, 33916606895219077ull}},
{{4594282626492795712ull, 27133285516175262ull}},
{{14743472545419967552ull, 21706628412940209ull}},
{{12521509628446217088ull, 34730605460704335ull}},
{{10017207702756973632ull, 27784484368563468ull}},
{{15392463791689399552ull, 22227587494850774ull}},
{{13559895622477308352ull, 35564139991761239ull}},
{{14537265312723756992ull, 28451311993408991ull}},
{{7940463435437095296ull, 22761049594727193ull}},
{{13731068377833496832ull, 18208839675781754ull}},
{{10901662960307864000ull, 29134143481250807ull}},
{{1342632738762470592ull, 23307314785000646ull}},
{{15831501449977617728ull, 18645851828000516ull}},
{{17951704690480367744ull, 29833362924800826ull}},
{{10672014937642383872ull, 23866690339840661ull}},
{{4848263135371996800ull, 19093352271872529ull}},
{{15135918646079015488ull, 30549363634996046ull}},
{{8419386102121302080ull, 24439490907996837ull}},
{{17803555325922772608ull, 19551592726397469ull}},
{{17417642077250705216ull, 31282548362235951ull}},
{{10244764847058653888ull, 25026038689788761ull}},
{{4506463062905012736ull, 20020830951831009ull}},
{{14589038530131841088ull, 32033329522929614ull}},
{{15360579638847383168ull, 25626663618343691ull}},
{{8599114896335996224ull, 20501330894674953ull}},
{{10069235019395683648ull, 32802129431479925ull}},
{{8055388015516546880ull, 26241703545183940ull}},
{{6444310412413237504ull, 20993362836147152ull}},
{{14000245474603090368ull, 33589380537835443ull}},
{{132149935456741312ull, 26871504430268355ull}},
{{105719948365393024ull, 21497203544214684ull}},
{{7547849546868449536ull, 34395525670743494ull}},
{{9727628452236669952ull, 27516420536594795ull}},
{{7782102761789335936ull, 22013136429275836ull}},
{{5072666789379116928ull, 35221018286841338ull}},
{{11436831060987114176ull, 28176814629473070ull}},
{{9149464848789691328ull, 22541451703578456ull}},
{{3630223064289842752ull, 18033161362862765ull}},
{{5808356902863748416ull, 28853058180580424ull}},
{{8336034337032909056ull, 23082446544464339ull}},
{{10358176284368237568ull, 18465957235571471ull}},
{{9194384425505359424ull, 29545531576914354ull}},
{{11044856355146197888ull, 23636425261531483ull}},
{{16214582713600778944ull, 18909140209225186ull}},
{{117890638567874048ull, 30254624334760299ull}},
{{3783661325596209536ull, 24203699467808239ull}},
{{6716277875218877952ull, 19362959574246591ull}},
{{3367346970866384128ull, 30980735318794546ull}},
{{17451272835660748544ull, 24784588255035636ull}},
{{10271669453786688512ull, 19827670604028509ull}},
{{5366624681832970688ull, 31724272966445615ull}},
{{4293299745466376576ull, 25379418373156492ull}},
{{14502686240598832192ull, 20303534698525193ull}},
{{1068205096506669632ull, 32485655517640310ull}},
{{854564077205335680ull, 25988524414112248ull}},
{{8062348891248089216ull, 20790819531289798ull}},
{{9210409411255032384ull, 33265311250063677ull}},
{{18436373973229756864ull, 26612249000050941ull}},
{{11059750363841895168ull, 21289799200040753ull}},
{{14006251767405121984ull, 34063678720065205ull}},
{{11205001413924097600ull, 27250942976052164ull}},
{{12653349945881188352ull, 21800754380841731ull}},
{{12866662283926080768ull, 34881207009346770ull}},
{{10293329827140864640ull, 27904965607477416ull}},
{{4545315046970781376ull, 22323972485981933ull}},
{{3583155260411339840ull, 35718355977571093ull}},
{{10245221837812892544ull, 28574684782056874ull}},
{{11885526284992224320ull, 22859747825645499ull}},
{{13197769842735689792ull, 18287798260516399ull}},
{{10048385304151372736ull, 29260477216826239ull}},
{{11728057058063008512ull, 23408381773460991ull}},
{{5693096831708496448ull, 18726705418768793ull}},
{{5419606115991684032ull, 29962728670030069ull}},
{{8025033707535257536ull, 23970182936024055ull}},
{{6420026966028206016ull, 19176146348819244ull}},
{{17650740775128950336ull, 30681834158110790ull}},
{{14120592620103160256ull, 24545467326488632ull}},
{{3917776466598707520ull, 19636373861190906ull}},
{{17336488790783663040ull, 31418198177905449ull}},
{{17558539847368840768ull, 25134558542324359ull}},
{{17736180692636982912ull, 20107646833859487ull}},
{{13620493849251531392ull, 32172234934175180ull}},
{{10896395079401225152ull, 25737787947340144ull}},
{{12406464878262890432ull, 20590230357872115ull}},
{{1403599731511073088ull, 32944368572595385ull}},
{{1122879785208858432ull, 26355494858076308ull}},
{{8277001457650907392ull, 21084395886461046ull}},
{{5864504702757631232ull, 33735033418337674ull}},
{{8380952576948015296ull, 26988026734670139ull}},
{{10394110876300322560ull, 21590421387736111ull}},
{{9251879772596695424ull, 34544674220377778ull}},
{{14780201447561177024ull, 27635739376302222ull}},
{{4445463528565120960ull, 22108591501041778ull}},
{{3423392830962283200ull, 35373746401666845ull}},
{{2738714264769826560ull, 28298997121333476ull}},
{{16948366670783502528ull, 22639197697066780ull}},
{{13558693336626802048ull, 18111358157653424ull}},
{{10625862894377152256ull, 28978173052245479ull}},
{{12190039130243632128ull, 23182538441796383ull}},
{{17130728933678726336ull, 18546030753437106ull}},
{{1583724590692589952ull, 29673649205499371ull}},
{{16024374931521713216ull, 23738919364399496ull}},
{{9130151130475460224ull, 18991135491519597ull}},
{{18297590623502646720ull, 30385816786431355ull}},
{{14638072498802117376ull, 24308653429145084ull}},
{{15399806813783604224ull, 19446922743316067ull}},
{{9882295643086125504ull, 31115076389305708ull}},
{{15284534143952721024ull, 24892061111444566ull}},
{{8538278500420266496ull, 19913648889155653ull}},
{{9971896785930516096ull, 31861838222649045ull}},
{{7977517428744412864ull, 25489470578119236ull}},
{{2692665128253619968ull, 20391576462495389ull}},
{{11686961834689612608ull, 32626522339992622ull}},
{{1970871838267869440ull, 26101217871994098ull}},
{{8955395100098116160ull, 20880974297595278ull}},
{{10639283345415075584ull, 33409558876152445ull}},
{{8511426676332060480ull, 26727647100921956ull}},
{{3119792526323738048ull, 21382117680737565ull}},
{{4991668042117980864ull, 34211388289180104ull}},
{{7682683248436295040ull, 27369110631344083ull}},
{{13524844228232856640ull, 21895288505075266ull}},
{{14261053135688750016ull, 35032461608120426ull}},
{{7719493693809089664ull, 28025969286496341ull}},
{{2486246140305361408ull, 22420775429197073ull}},
{{288645009746667968ull, 35873240686715317ull}},
{{11298962452023065344ull, 28698592549372253ull}},
{{16417867591102272896ull, 22958874039497802ull}},
{{5755596443397997696ull, 18367099231598242ull}},
{{12898303124178706624ull, 29387358770557187ull}},
{{2939944869859144640ull, 23509887016445750ull}},
{{2351955895887315712ull, 18807909613156600ull}},
{{3763129433419705152ull, 30092655381050560ull}},
{{3010503546735764096ull, 24074124304840448ull}},
{{9787100466872431936ull, 19259299443872358ull}},
{{11970011932253980800ull, 30814879110195773ull}},
{{16954707175287005248ull, 24651903288156618ull}},
{{2495719296003873216ull, 19721522630525295ull}},
{{3993150873606197184ull, 31554436208840472ull}},
{{14262567143110688704ull, 25243548967072377ull}},
{{4031356085004730304ull, 20194839173657902ull}},
{{10139518550749478848ull, 32311742677852643ull}},
{{15490312470083403712ull, 25849394142282114ull}},
{{16081598790808633280ull, 20679515313825691ull}},
{{18351860435809992640ull, 33087224502121106ull}},
{{10992139533906083776ull, 26469779601696885ull}},
{{8793711627124867008ull, 21175823681357508ull}},
{{10380589788657876928ull, 33881317890172013ull}},
{{15683169460410122176ull, 27105054312137610ull}},
{{12546535568328097728ull, 21684043449710088ull}},
{{16385108094583046080ull, 34694469519536141ull}},
{{9418737660924526528ull, 27755575615628913ull}},
{{14913687758223441856ull, 22204460492503130ull}},
{{5415156339447955392ull, 35527136788005009ull}},
{{8021473886300274624ull, 28421709430404007ull}},
{{17485225553265950656ull, 22737367544323205ull}},
{{13988180442612760512ull, 18189894035458564ull}},
{{11313042263954685888ull, 29103830456733703ull}},
{{16429131440647569344ull, 23283064365386962ull}},
{{5764607523034234816ull, 18626451492309570ull}},
{{9223372036854775744ull, 29802322387695312ull}},
{{18446744073709551552ull, 23841857910156249ull}},
{{18446744073709551552ull, 19073486328124999ull}},
{{18446744073709551552ull, 30517578124999999ull}},
{{18446744073709551552ull, 24414062499999999ull}},
{{18446744073709551552ull, 19531249999999999ull}},
{{18446744073709551552ull, 31249999999999999ull}},
{{18446744073709551552ull, 24999999999999999ull}},
{{18446744073709551552ull, 19999999999999999ull}},
{{18446744073709551552ull, 31999999999999999ull}},
{{18446744073709551552ull, 25599999999999999ull}},
{{18446744073709551552ull, 20479999999999999ull}},
{{18446744073709551552ull, 32767999999999999ull}},
{{18446744073709551552ull, 26214399999999999ull}},
{{18446744073709551552ull, 20971519999999999ull}},
{{18446744073709551552ull, 33554431999999999ull}},
{{18446744073709551552ull, 26843545599999999ull}},
{{18446744073709551552ull, 21474836479999999ull}},
{{18446744073709551552ull, 34359738367999999ull}},
{{18446744073709551552ull, 27487790694399999ull}},
{{18446744073709551552ull, 21990232555519999ull}},
{{18446744073709551552ull, 35184372088831999ull}},
{{18446744073709551552ull, 28147497671065599ull}},
{{18446744073709551552ull, 22517998136852479ull}},
{{18446744073709551552ull, 18014398509481983ull}},
{{7378697629483820608ull, 28823037615171174ull}},
{{9592306918328966784ull, 23058430092136939ull}},
{{11363194349405083776ull, 18446744073709551ull}},
{{10802413329564313408ull, 29514790517935282ull}},
{{1263233034167630080ull, 23611832414348226ull}},
{{15767981686301745344ull, 18889465931478580ull}},
{{6782026624373240960ull, 30223145490365729ull}},
{{9114970114240503040ull, 24178516392292583ull}},
{{14670673720876223104ull, 19342813113834066ull}},
{{16094380323918136320ull, 30948500982134506ull}},
{{9186155444392598720ull, 24758800785707605ull}},
{{7348924355514078976ull, 19807040628566084ull}},
{{690232524596795392ull, 31691265005705735ull}},
{{552186019677436352ull, 25353012004564588ull}},
{{7820446445225769728ull, 20282409603651670ull}},
{{12512714312361231552ull, 32451855365842672ull}},
{{2631473820405164608ull, 25961484292674138ull}},
{{9483876685807952320ull, 20769187434139310ull}},
{{15174202697292723712ull, 33230699894622896ull}},
{{8450013343092268608ull, 26584559915698317ull}},
{{17828057118699545856ull, 21267647932558653ull}},
{{6388798501467811456ull, 34028236692093846ull}},
{{1421689986432338880ull, 27222589353675077ull}},
{{12205398433371602048ull, 21778071482940061ull}},
{{12149939863910742656ull, 34844914372704098ull}},
{{17098649520612414784ull, 27875931498163278ull}},
{{2610873172264200832ull, 22300745198530623ull}},
{{488048260880811008ull, 35681192317648997ull}},
{{11458485052930379776ull, 28544953854119197ull}},
{{1788090412860483200ull, 22835963083295358ull}},
{{8809169959772207168ull, 18268770466636286ull}},
{{6715974306151710848ull, 29230032746618058ull}},
{{12751477074405189312ull, 23384026197294446ull}},
{{6511832844782241152ull, 18707220957835557ull}},
{{14108281366393496128ull, 29931553532536891ull}},
{{7597276278372886592ull, 23945242826029513ull}},
{{13456518652182129920ull, 19156194260823610ull}},
{{3083685769781856256ull, 30649910817317777ull}},
{{13534995060051216000ull, 24519928653854221ull}},
{{7138647233299062464ull, 19615942923083377ull}},
{{15111184388020410240ull, 31385508676933403ull}},
{{1020901066190597248ull, 25108406941546723ull}},
{{8195418482436298432ull, 20086725553237378ull}},
{{9423320757156167168ull, 32138760885179805ull}},
{{7538656605724933760ull, 25711008708143844ull}},
{{9720274099321857280ull, 20568806966515075ull}},
{{15552438558914971712ull, 32910091146424120ull}},
{{12441950847131977344ull, 26328072917139296ull}},
{{6264211862963671552ull, 21062458333711437ull}},
{{13712087795483784832ull, 33699933333938299ull}},
{{14659019051128938176ull, 26959946667150639ull}},
{{15416564055645060864ull, 21567957333720511ull}},
{{17287804859548276736ull, 34508731733952818ull}},
{{2762197443412890432ull, 27606985387162255ull}},
{{2209757954730312320ull, 22085588309729804ull}},
{{10914310357052320384ull, 35336941295567686ull}},
{{5042099470899945984ull, 28269553036454149ull}},
{{7723028391461867136ull, 22615642429163319ull}},
{{9867771527911404032ull, 18092513943330655ull}},
{{15788434444658246400ull, 28948022309329048ull}},
{{1562701111500866176ull, 23158417847463239ull}},
{{4939509703942603264ull, 18526734277970591ull}},
{{524517896824344576ull, 29642774844752946ull}},
{{15177009576427116928ull, 23714219875802356ull}},
{{8452258846399783232ull, 18971375900641885ull}},
{{13523614154239653184ull, 30354201441027016ull}},
{{7129542508649812224ull, 24283361152821613ull}},
{{13082331636403670400ull, 19426688922257290ull}},
{{2484986544536321088ull, 31082702275611665ull}},
{{1987989235629056832ull, 24866161820489332ull}},
{{12658437832728976448ull, 19892929456391465ull}},
{{1806756458656810688ull, 31828687130226345ull}},
{{1445405166925448576ull, 25462949704181076ull}},
{{15913719392508000128ull, 20370359763344860ull}},
{{7015206954303248640ull, 32592575621351777ull}},
{{16680212007668329856ull, 26074060497081421ull}},
{{9654820791392753536ull, 20859248397665137ull}},
{{690318007260764416ull, 33374797436264220ull}},
{{552254405808611520ull, 26699837949011376ull}},
{{15199198783614530496ull, 21359870359209100ull}},
{{5871973980073697216ull, 34175792574734561ull}},
{{1008230369317047424ull, 27340634059787649ull}},
{{4495933110195548288ull, 21872507247830119ull}},
{{14572190605796697920ull, 34996011596528190ull}},
{{11657752484637358336ull, 27996809277222552ull}},
{{1947504358226065984ull, 22397447421778042ull}},
{{6805355787903615936ull, 35835915874844867ull}},
{{16512331074548623680ull, 28668732699875893ull}},
{{2141818415413168000ull, 22934986159900715ull}},
{{1713454732330534400ull, 18347988927920572ull}},
{{6430876386470765376ull, 29356782284672915ull}},
{{5144701109176612288ull, 23485425827738332ull}},
{{15183807331567020800ull, 18788340662190665ull}},
{{5847347656797681664ull, 30061345059505065ull}},
{{4677878125438145344ull, 24049076047604052ull}},
{{14810348944576247232ull, 19239260838083241ull}},
{{16317860681838174912ull, 30782817340933186ull}},
{{9364939730728629632ull, 24626253872746549ull}},
{{11181300599324814016ull, 19701003098197239ull}},
{{6822034514693971456ull, 31521604957115583ull}},
{{12836325241238997824ull, 25217283965692466ull}},
{{6579711378249287936ull, 20173827172553973ull}},
{{6838189390456950400ull, 32278123476086357ull}},
{{16538597956591291264ull, 25822498780869085ull}},
{{13230878365273033024ull, 20657999024695268ull}},
{{17480056569694942528ull, 33052798439512429ull}},
{{17673394070497864320ull, 26442238751609943ull}},
{{3070668812172560448ull, 21153791001287955ull}},
{{4913070099476096768ull, 33846065602060728ull}},
{{11309153709064698048ull, 27076852481648582ull}},
{{1668625337767937792ull, 21661481985318866ull}},
{{13737846984654431488ull, 34658371176510185ull}},
{{10990277587723545152ull, 27726696941208148ull}},
{{16170919699662656768ull, 22181357552966518ull}},
{{3737378631008788928ull, 35490172084746430ull}},
{{2989902904807031104ull, 28392137667797144ull}},
{{6081271138587535232ull, 22713710134237715ull}},
{{4865016910870028160ull, 18170968107390172ull}},
{{11473375872133955392ull, 29073548971824275ull}},
{{9178700697707164352ull, 23258839177459420ull}},
{{7342960558165731456ull, 18607071341967536ull}},
{{4370039263581349696ull, 29771314147148058ull}},
{{10874729040348900416ull, 23817051317718446ull}},
{{5010434417537209984ull, 19053641054174757ull}},
{{11706043882801446336ull, 30485825686679611ull}},
{{5675486291499246720ull, 24388660549343689ull}},
{{8229737847941307712ull, 19510928439474951ull}},
{{5788882927222271680ull, 31217485503159922ull}},
{{15699152786003548288ull, 24973988402527937ull}},
{{5180624599319017984ull, 19979190722022350ull}},
{{8288999358910428800ull, 31966705155235760ull}},
{{6631199487128343040ull, 25573364124188608ull}},
{{12683657219186495104ull, 20458691299350886ull}},
{{12915153921214571520ull, 32733906078961418ull}},
{{17710820766455477824ull, 26187124863169134ull}},
{{17858005427906292608ull, 20949699890535307ull}},
{{13815413425682426880ull, 33519519824856492ull}},
{{3673633111062120832ull, 26815615859885194ull}},
{{6628255303591606976ull, 21452492687908155ull}},
{{10605208485746571200ull, 34323988300653048ull}},
{{15862864418081077632ull, 27459190640522438ull}},
{{1622245090239131136ull, 21967352512417951ull}},
{{13663638588608340736ull, 35147764019868721ull}},
{{7241562056144762304ull, 28118211215894977ull}},
{{16861296089141540800ull, 22494568972715981ull}},
{{1152632039433092992ull, 35991310356345571ull}},
{{15679500890514115712ull, 28793048285076456ull}},
{{8854251897669382208ull, 23034438628061165ull}},
{{7083401518135505792ull, 18427550902448932ull}},
{{15022791243758719616ull, 29484081443918291ull}},
{{8328884180265065344ull, 23587265155134633ull}},
{{14041804973695872896ull, 18869812124107706ull}},
{{15088190328429576000ull, 30191699398572330ull}},
{{12070552262743660800ull, 24153359518857864ull}},
{{13345790624936838976ull, 19322687615086291ull}},
{{13974567370415121728ull, 30916300184138066ull}},
{{7490305081590187072ull, 24733040147310453ull}},
{{13370941694755970304ull, 19786432117848362ull}},
{{6636111452641911168ull, 31658291388557380ull}},
{{5308889162113528960ull, 25326633110845904ull}},
{{7936460144432733440ull, 20261306488676723ull}},
{{9008987416350463232ull, 32418090381882757ull}},
{{18275236377306101568ull, 25934472305506205ull}},
{{14620189101844881216ull, 20747577844404964ull}},
{{12324256118726079040ull, 33196124551047943ull}},
{{17238102524464683840ull, 26556899640838354ull}},
{{17479830834313657408ull, 21245519712670683ull}},
{{5831636446450389952ull, 33992831540273094ull}},
{{8354657971902222272ull, 27194265232218475ull}},
{{6683726377521777792ull, 21755412185774780ull}},
{{10693962204034844480ull, 34808659497239648ull}},
{{15933867392711696256ull, 27846927597791718ull}},
{{1679047469943626048ull, 22277542078233375ull}},
{{2686475951909801664ull, 35644067325173400ull}},
{{2149180761527841344ull, 28515253860138720ull}},
{{1719344609222273024ull, 22812203088110976ull}},
{{16132870946345459712ull, 18249762470488780ull}},
{{7365849440443183936ull, 29199619952782049ull}},
{{9582028367096457472ull, 23359695962225639ull}},
{{11354971508419076288ull, 18687756769780511ull}},
{{10789256783986701440ull, 29900410831648818ull}},
{{16010103056673181824ull, 23920328665319054ull}},
{{16497431260080455744ull, 19136262932255243ull}},
{{4259797127677267328ull, 30618020691608390ull}},
{{3407837702141813824ull, 24494416553286712ull}},
{{13794316605939182016ull, 19595533242629369ull}},
{{11002860125276960320ull, 31352853188206991ull}},
{{5112939285479657920ull, 25082282550565593ull}},
{{11469049057867546944ull, 20065826040452474ull}},
{{7282432048362344192ull, 32105321664723959ull}},
{{9515294453431785664ull, 25684257331779167ull}},
{{233537933261607872ull, 20547405865423334ull}},
{{7752358322702393280ull, 32875849384677334ull}},
{{9891235472903824960ull, 26300679507741867ull}},
{{534290748839239296ull, 21040543606193494ull}},
{{8233562827626603520ull, 33664869769909590ull}},
{{6586850262101282816ull, 26931895815927672ull}},
{{16337526653906757248ull, 21545516652742137ull}},
{{11382647387283170304ull, 34472826644387420ull}},
{{9106117909826536256ull, 27578261315509936ull}},
{{3595545513119318656ull, 22062609052407949ull}},
{{13131570450474730496ull, 35300174483852718ull}},
{{17883953989863605056ull, 28240139587082174ull}},
{{17996512006632794368ull, 22592111669665739ull}},
{{18086558420048145792ull, 18073689335732591ull}},
{{3113051768883661056ull, 28917902937172147ull}},
{{13558487859332659776ull, 23134322349737717ull}},
{{3468092657982307200ull, 18507457879790174ull}},
{{12927645882255512128ull, 29611932607664278ull}},
{{17720814335288230336ull, 23689546086131422ull}},
{{6797953838746763648ull, 18951636868905138ull}},
{{7187377327252911552ull, 30322618990248221ull}},
{{2060553047060418880ull, 24258095192198577ull}},
{{12716488881874066048ull, 19406476153758861ull}},
{{12967684581514685120ull, 31050361846014178ull}},
{{17752845294695568704ull, 24840289476811342ull}},
{{6823578606272634304ull, 19872231581449074ull}},
{{18296423399520035584ull, 31795570530318518ull}},
{{3569092275390297472ull, 25436456424254815ull}},
{{2855273820312237952ull, 20349165139403852ull}},
{{8257786927241491136ull, 32558664223046163ull}},
{{13984927171277013504ull, 26046931378436930ull}},
{{11187941737021610816ull, 20837545102749544ull}},
{{6832660335008846336ull, 33340072164399271ull}},
{{1776779453265166784ull, 26672057731519417ull}},
{{12489470006837864384ull, 21337646185215533ull}},
{{16293803196198672704ull, 34140233896344853ull}},
{{1966996112733207168ull, 27312187117075883ull}},
{{8952294519670386368ull, 21849749693660706ull}},
{{6944973601988797568ull, 34959599509857130ull}},
{{5555978881591038080ull, 27967679607885704ull}},
{{8134131920014740736ull, 22374143686308563ull}},
{{9325262257281674880ull, 35798629898093701ull}},
{{3770860991083429568ull, 28638903918474961ull}},
{{17774084051834384960ull, 22911123134779968ull}},
{{3151220797241777024ull, 18328898507823975ull}},
{{5041953275586843200ull, 29326237612518360ull}},
{{4033562620469474560ull, 23460990090014688ull}},
{{10605547725859400320ull, 18768792072011750ull}},
{{16968876361375040512ull, 30030067315218800ull}},
{{13575101089100032384ull, 24024053852175040ull}},
{{10860080871280025920ull, 19219243081740032ull}},
{{2618734135080400192ull, 30750788930784052ull}},
{{13163033752290051072ull, 24600631144627241ull}},
{{6841078187090130560ull, 19680504915701793ull}},
{{7256376284602298560ull, 31488807865122869ull}},
{{9494449842423749184ull, 25191046292098295ull}},
{{7595559873938999360ull, 20152837033678636ull}},
{{4774198168818578304ull, 32244539253885818ull}},
{{11198056164538683264ull, 25795631403108654ull}},
{{12647793746372856960ull, 20636505122486923ull}},
{{16547121179454660800ull, 33018408195979077ull}},
{{5858999314079907968ull, 26414726556783262ull}},
{{15755245895489657344ull, 21131781245426609ull}},
{{14140346988557720832ull, 33810849992682575ull}},
{{11312277590846176640ull, 27048679994146060ull}},
{{9049822072676941312ull, 21638943995316848ull}},
{{10790366501541195776ull, 34622310392506957ull}},
{{1253595571749136000ull, 27697848314005566ull}},
{{15760271716366950080ull, 22158278651204452ull}},
{{10459039487219478848ull, 35453245841927124ull}},
{{12056580404517493376ull, 28362596673541699ull}},
{{13334613138355905024ull, 22690077338833359ull}},
{{14357039325426634368ull, 18152061871066687ull}},
{{8213867661714973696ull, 29043298993706700ull}},
{{6571094129371978944ull, 23234639194965360ull}},
{{5256875303497583168ull, 18587711355972288ull}},
{{4721651670854222720ull, 29740338169555661ull}},
{{87972521941467840ull, 23792270535644529ull}},
{{3759726832295084608ull, 19033816428515623ull}},
{{2326214116930225024ull, 30454106285624997ull}},
{{12929017737769910976ull, 24363285028499997ull}},
{{2964516560732108160ull, 19490628022799998ull}},
{{1053877682429462720ull, 31185004836479997ull}},
{{11911148590169301120ull, 24948003869183997ull}},
{{2150221242651620288ull, 19958403095347198ull}},
{{18197749247210233728ull, 31933444952555516ull}},
{{10868850583026276672ull, 25546755962044413ull}},
{{16073778095904841984ull, 20437404769635530ull}},
{{7271300879738195520ull, 32699847631416849ull}},
{{9506389518532466752ull, 26159878105133479ull}},
{{11294460429567883712ull, 20927902484106783ull}},
{{14381787872566703680ull, 33484643974570853ull}},
{{437383853827631936ull, 26787715179656683ull}},
{{7728604712545926208ull, 21430172143725346ull}},
{{4987069910589661312ull, 34288275429960554ull}},
{{7679004743213639360ull, 27430620343968443ull}},
{{13521901424054732096ull, 21944496275174754ull}},
{{10566995834261840448ull, 35111194040279607ull}},
{{1074899037925651712ull, 28088955232223686ull}},
{{15617314489308162624ull, 22471164185778948ull}},
{{2851610294441598336ull, 35953862697246318ull}},
{{9659985865037099264ull, 28763090157797054ull}},
{{11417337506771589760ull, 23010472126237643ull}},
{{16512567634901092416ull, 18408377700990114ull}},
{{15352061771616016960ull, 29453404321584183ull}},
{{1213602973067082560ull, 23562723457267347ull}},
{{12038928822679397056ull, 18850178765813877ull}},
{{4504890857319393984ull, 30160286025302204ull}},
{{7293261500597425472ull, 24128228820241763ull}},
{{13213306829961761024ull, 19302583056193410ull}},
{{2694546854229266048ull, 30884132889909457ull}},
{{13223683927609143808ull, 24707306311927565ull}},
{{10578947142087315072ull, 19765845049542052ull}},
{{2168920168372062784ull, 31625352079267284ull}},
{{5424484949439560576ull, 25300281663413827ull}},
{{15407634403777379392ull, 20240225330731061ull}},
{{17273517416559986432ull, 32384360529169698ull}},
{{2750767489022258176ull, 25907488423335759ull}}
};

static const int exponents_binary64[] = { -55,
  -51,
  -48,
  -45,
  -41,
  -38,
  -35,
  -31,
  -28,
  -25,
  -22,
  -18,
  -15,
  -12,
  -8,
  -5,
  -2,
  2,
  5,
  8,
  12,
  15,
  18,
  22,
  25,
  28,
  32,
  35,
  38,
  42,
  45,
  48,
  52,
  55,
  58,
  62,
  65,
  68,
  71,
  75,
  78,
  81,
  85,
  88,
  91,
  95,
  98,
  101,
  105,
  108,
  111,
  115,
  118,
  121,
  125,
  128,
  131,
  135,
  138,
  141,
  145,
  148,
  151,
  155,
  158,
  161,
  164,
  168,
  171,
  174,
  178,
  181,
  184,
  188,
  191,
  194,
  198,
  201,
  204,
  208,
  211,
  214,
  218,
  221,
  224,
  228,
  231,
  234,
  238,
  241,
  244,
  248,
  251,
  254,
  258,
  261,
  264,
  267,
  271,
  274,
  277,
  281,
  284,
  287,
  291,
  294,
  297,
  301,
  304,
  307,
  311,
  314,
  317,
  321,
  324,
  327,
  331,
  334,
  337,
  341,
  344,
  347,
  351,
  354,
  357,
  360,
  364,
  367,
  370,
  374,
  377,
  380,
  384,
  387,
  390,
  394,
  397,
  400,
  404,
  407,
  410,
  414,
  417,
  420,
  424,
  427,
  430,
  434,
  437,
  440,
  444,
  447,
  450,
  454,
  457,
  460,
  463,
  467,
  470,
  473,
  477,
  480,
  483,
  487,
  490,
  493,
  497,
  500,
  503,
  507,
  510,
  513,
  517,
  520,
  523,
  527,
  530,
  533,
  537,
  540,
  543,
  547,
  550,
  553,
  556,
  560,
  563,
  566,
  570,
  573,
  576,
  580,
  583,
  586,
  590,
  593,
  596,
  600,
  603,
  606,
  610,
  613,
  616,
  620,
  623,
  626,
  630,
  633,
  636,
  640,
  643,
  646,
  649,
  653,
  656,
  659,
  663,
  666,
  669,
  673,
  676,
  679,
  683,
  686,
  689,
  693,
  696,
  699,
  703,
  706,
  709,
  713,
  716,
  719,
  723,
  726,
  729,
  733,
  736,
  739,
  743,
  746,
  749,
  752,
  756,
  759,
  762,
  766,
  769,
  772,
  776,
  779,
  782,
  786,
  789,
  792,
  796,
  799,
  802,
  806,
  809,
  812,
  816,
  819,
  822,
  826,
  829,
  832,
  836,
  839,
  842,
  845,
  849,
  852,
  855,
  859,
  862,
  865,
  869,
  872,
  875,
  879,
  882,
  885,
  889,
  892,
  895,
  899,
  902,
  905,
  909,
  912,
  915,
  919,
  922,
  925,
  929,
  932,
  935,
  939,
  942,
  945,
  948,
  952,
  955,
  958,
  962,
  965,
  968,
  972,
  975,
  978,
  982,
  985,
  988,
  992,
  995,
  998,
  1002,
  1005,
  1008,
  1012,
  1015,
  1018,
  1022,
  1025,
  1028,
  1032,
  1035,
  1038,
  1041,
  1045,
  1048,
  1051,
  1055,
  1058,
  1061,
  1065,
  1068,
  1071,
  1075,
  1078,
  1081,
  1085,
  1088,
  1091,
  1095,
  1098,
  1101,
  1105,
  1108,
  1111,
  1115,
  1118,
  1121,
  1125,
  1128,
  1131,
  1134,
  1138,
  1141,
  1144,
  1148,
  1151,
  1154,
  1158,
  1161,
  1164,
  1168,
  1171,
  1174,
  1178,
  1181,
  1184,
  1188,
  1191,
  1194,
  1198,
  1201,
  1204,
  1208,
  1211,
  1214,
  1218,
  1221,
  1224,
  1228,
  1231,
  1234,
  1237,
  1241,
  1244,
  1247,
  1251,
  1254,
  1257,
  1261,
  1264,
  1267,
  1271,
  1274,
  1277,
  1281,
  1284,
  1287,
  1291,
  1294,
  1297,
  1301,
  1304,
  1307,
  1311,
  1314,
  1317,
  1321,
  1324,
  1327,
  1330,
  1334,
  1337,
  1340,
  1344,
  1347,
  1350,
  1354,
  1357,
  1360,
  1364,
  1367,
  1370,
  1374,
  1377,
  1380,
  1384,
  1387,
  1390,
  1394,
  1397,
  1400,
  1404,
  1407,
  1410,
  1414,
  1417,
  1420,
  1424,
  1427,
  1430,
  1433,
  1437,
  1440,
  1443,
  1447,
  1450,
  1453,
  1457,
  1460,
  1463,
  1467,
  1470,
  1473,
  1477,
  1480,
  1483,
  1487,
  1490,
  1493,
  1497,
  1500,
  1503,
  1507,
  1510,
  1513,
  1517,
  1520,
  1523,
  1526,
  1530,
  1533,
  1536,
  1540,
  1543,
  1546,
  1550,
  1553,
  1556,
  1560,
  1563,
  1566,
  1570,
  1573,
  1576,
  1580,
  1583,
  1586,
  1590,
  1593,
  1596,
  1600,
  1603,
  1606,
  1610,
  1613,
  1616,
  1620,
  1623,
  1626,
  1629,
  1633,
  1636,
  1639,
  1643,
  1646,
  1649,
  1653,
  1656,
  1659,
  1663,
  1666,
  1669,
  1673,
  1676,
  1679,
  1683,
  1686,
  1689,
  1693,
  1696,
  1699,
  1703,
  1706,
  1709,
  1713,
  1716,
  1719,
  1722,
  1726,
  1729,
  1732,
  1736,
  1739,
  1742,
  1746,
  1749,
  1752,
  1756,
  1759,
  1762,
  1766,
  1769,
  1772,
  1776,
  1779,
  1782,
  1786,
  1789,
  1792,
  1796,
  1799,
  1802,
  1806,
  1809,
  1812,
  1815,
  1819,
  1822,
  1825,
  1829,
  1832,
  1835,
  1839,
  1842,
  1845,
  1849,
  1852,
  1855,
  1859,
  1862,
  1865,
  1869,
  1872,
  1875,
  1879,
  1882,
  1885,
  1889,
  1892,
  1895,
  1899,
  1902,
  1905,
  1909,
  1912,
  1915,
  1918,
  1922,
  1925,
  1928,
  1932,
  1935,
  1938,
  1942,
  1945,
  1948,
  1952,
  1955,
  1958,
  1962,
  1965,
  1968,
  1972,
  1975,
  1978,
  1982,
  1985,
  1988,
  1992,
  1995,
  1998,
  2002,
  2005,
  2008,
  2011,
  2015,
  2018,
  2021,
  2025,
  2028,
  2031,
  2035,
  2038,
  2041,
  2045,
  2048,
  2051,
  2055,
  2058,
  2061,
  2065,
  2068,
  2071,
  2075,
  2078,
  2081,
  2085,
  2088,
  2091,
  2095,
  2098,
  2101,
  2105,
  2108,
  2111,
  2114,
  2118,
  2121,
  2124,
  2128,
  2131,
  2134,
  2138,
  2141,
  2144,
  2148,
  2151,
  2154,
  2158,
  2161,
};

static const UINT256 multipliers1_binary64[] =
  { {{1837554224478941466ull, 10276842184138466546ull,
      11651621577776737258ull, 7754513766366540701ull}},
{{5760157408726726321ull, 11034712383513929495ull,
  9588106495324154738ull, 4846571103979087938ull}},
{{2588510742481019997ull, 4570018442537636061ull,
  2761761082300417615ull, 6058213879973859923ull}},
{{7847324446528662900ull, 1100837034744657172ull,
  17287259408157685731ull, 7572767349967324903ull}},
{{14127949815935190120ull, 16828924211211268396ull,
  17722066157739635437ull, 4732979593729578064ull}},
{{17659937269918987650ull, 7201097208731921783ull,
  3705838623464992681ull, 5916224492161972581ull}},
{{17463235568971346659ull, 13613057529342290133ull,
  9243984297758628755ull, 7395280615202465726ull}},
{{13220365239820785614ull, 6202317946625237381ull,
  1165804167671755068ull, 4622050384501541079ull}},
{{2690398494493818305ull, 7752897433281546727ull,
  15292313264871857547ull, 5777562980626926348ull}},
{{17198056173399436594ull, 5079435773174545504ull,
  668647507380270318ull, 7221953725783657936ull}},
{{3050826143039744126ull, 15572666753322957689ull,
  835809384225337897ull, 9027442157229572420ull}},
{{13435981385468309839ull, 2815387693185766699ull,
  9745752901995611994ull, 5642151348268482762ull}},
{{12183290713407999394ull, 12742606653336984182ull,
  2958819090639739184ull, 7052689185335603453ull}},
{{6005741354905223435ull, 15928258316671230228ull,
  8310209881727061884ull, 8815861481669504316ull}},
{{12976960383670540455ull, 731789411064743084ull,
  14417253212934189486ull, 5509913426043440197ull}},
{{16221200479588175569ull, 10138108800685704663ull,
  4186508460885573145ull, 6887391782554300247ull}},
{{15664814581057831557ull, 17284322019284518733ull,
  621449557679578527ull, 8609239728192875309ull}},
{{12096352122374838675ull, 17720230289693906064ull,
  2694248982763430531ull, 5380774830120547068ull}},
{{15120440152968548344ull, 17538601843689994676ull,
  3367811228454288164ull, 6725968537650683835ull}},
{{453806117501133814ull, 3476508230902941730ull,
  18044822090850023918ull, 8407460672063354793ull}},
{{4895314841865596538ull, 16007875699596502293ull,
  4360484779140183092ull, 5254662920039596746ull}},
{{10730829570759383576ull, 1563100550786076250ull,
  14673978010780004674ull, 6568328650049495932ull}},
{{4190164926594453662ull, 11177247725337371121ull,
  18342472513475005842ull, 8210410812561869915ull}},
{{14148068125190003299ull, 11597465846763244854ull,
  9158202311708184699ull, 5131506757851168697ull}},
{{8461713119632728315ull, 9885146290026668164ull,
  16059438908062618778ull, 6414383447313960871ull}},
{{10577141399540910394ull, 3133060825678559397ull,
  15462612616650885569ull, 8017979309142451089ull}},
{{8916556383926762949ull, 13487378062117569383ull,
  2746603857765721624ull, 5011237068214031931ull}},
{{6534009461481065782ull, 16859222577646961729ull,
  17268312877489315742ull, 6264046335267539913ull}},
{{12779197845278720131ull, 11850656185203926353ull,
  7750333041579480966ull, 7830057919084424892ull}},
{{1069469625658118226ull, 2794974097325066067ull,
  14067330187841951412ull, 4893786199427765557ull}},
{{15171895087354811494ull, 3493717621656332583ull,
  3749104679520275553ull, 6117232749284706947ull}},
{{14353182840766126464ull, 8978833045497803633ull, 74694830972956537ull,
  7646540936605883684ull}},
{{2053210247837747184ull, 17140985699504597031ull,
  9270056306212873643ull, 4779088085378677302ull}},
{{16401570865079347692ull, 16814546105953358384ull,
  2364198345911316246ull, 5973860106723346628ull}},
{{2055219507639632999ull, 11794810595586922173ull,
  2955247932389145308ull, 7467325133404183285ull}},
{{3590355201488464576ull, 16595128659096602166ull,
  4152872966956909769ull, 4667078208377614553ull}},
{{13711316038715356528ull, 6908852768588588995ull,
  9802777227123525116ull, 5833847760472018191ull}},
{{12527459029966807756ull, 8636065960735736244ull,
  7641785515477018491ull, 7292309700590022739ull}},
{{15659323787458509695ull, 6183396432492282401ull,
  4940545875918885210ull, 9115387125737528424ull}},
{{2869548339520486704ull, 8476308788735064405ull,
  3087841172449303256ull, 5697116953585955265ull}},
{{8198621442827996284ull, 10595385985918830506ull,
  8471487483989016974ull, 7121396191982444081ull}},
{{1024904766680219546ull, 4020860445543762325ull,
  15201045373413659122ull, 8901745239978055101ull}},
{{2946408488388831169ull, 7124723796892239357ull,
  11806496367597230903ull, 5563590774986284438ull}},
{{8294696628913426865ull, 4294218727687911292ull,
  5534748422641762821ull, 6954488468732855548ull}},
{{10368370786141783581ull, 9979459428037277019ull,
  6918435528302203526ull, 8693110585916069435ull}},
{{4174388732124920786ull, 1625476124095910233ull,
  2018179195975183252ull, 5433194116197543397ull}},
{{9829671933583538887ull, 2031845155119887791ull,
  7134410013396366969ull, 6791492645246929246ull}},
{{7675403898552035704ull, 7151492462327247643ull,
  18141384553600234519ull, 8489365806558661557ull}},
{{2491284427381328363ull, 11387211816595611633ull,
  13644208355213840526ull, 5305853629099163473ull}},
{{7725791552654048358ull, 5010642733889738733ull,
  3220202388735136946ull, 6632317036373954342ull}},
{{14268925459244948351ull, 15486675454216949224ull,
  13248625022773696990ull, 8290396295467442927ull}},
{{8918078412028092720ull, 5067486140458205361ull,
  15197919666874642475ull, 5181497684667151829ull}},
{{15759284033462503804ull, 1722671657145368797ull,
  5162341528311139382ull, 6476872105833939787ull}},
{{5864046986545966042ull, 11376711608286486805ull,
  1841240891961536323ull, 8096090132292424734ull}},
{{5970872375804922729ull, 4804601745965360301ull,
  14985833612758123914ull, 5060056332682765458ull}},
{{12075276488183541315ull, 15229124219311476184ull,
  9508919979092879084ull, 6325070415853456823ull}},
{{15094095610229426643ull, 589661200429793614ull,
  7274463955438710952ull, 7906338019816821029ull}},
{{4822123737966003748ull, 368538250268621009ull, 6852382981362888297ull,
  4941461262385513143ull}},
{{10639340690884892589ull, 5072358831263164165ull,
  3953792708276222467ull, 6176826577981891429ull}},
{{17910861882033503640ull, 1728762520651567302ull,
  9553926903772665988ull, 7721033222477364286ull}},
{{6582602657843551871ull, 10303848612262005372ull,
  1359518296430528338ull, 4825645764048352679ull}},
{{8228253322304439839ull, 3656438728472730907ull,
  15534455925820324135ull, 6032057205060440848ull}},
{{5673630634453161895ull, 18405606465873077346ull,
  971325833565853552ull, 7540071506325551061ull}},
{{8157705164960614088ull, 11503504041170673341ull,
  2912921655192352422ull, 4712544691453469413ull}},
{{14808817474628155514ull, 5156008014608565868ull,
  8252838087417828432ull, 5890680864316836766ull}},
{{64277769575642777ull, 6445010018260707336ull, 1092675572417509732ull,
  7363351080396045958ull}},
{{80347211969553471ull, 8056262522825884170ull, 10589216502376662973ull,
  9204188850495057447ull}},
{{4661903025908358824ull, 7341007085979871558ull,
  13535789341626496214ull, 5752618031559410904ull}},
{{15050750819240224337ull, 18399630894329615255ull,
  16919736677033120267ull, 7190772539449263630ull}},
{{14201752505622892517ull, 18387852599484631165ull,
  11926298809436624526ull, 8988465674311579538ull}},
{{11181938325228001776ull, 6880721856250506574ull,
  12065622774325278233ull, 5617791046444737211ull}},
{{4754050869680226411ull, 13212588338740521122ull,
  10470342449479209887ull, 7022238808055921514ull}},
{{15165935623955058822ull, 11904049404998263498ull,
  3864556024994236551ull, 8777798510069901893ull}},
{{14090395783399299668ull, 14357559905764996542ull,
  4721190524835091796ull, 5486124068793688683ull}},
{{8389622692394348777ull, 17946949882206245678ull,
  1289802137616476841ull, 6857655085992110854ull}},
{{1263656328638160163ull, 8598629297475643386ull,
  10835624708875371860ull, 8572068857490138567ull}},
{{5401471223826238006ull, 14597515347777052924ull,
  13689794470688189268ull, 5357543035931336604ull}},
{{6751839029782797507ull, 18246894184721316155ull,
  17112243088360236585ull, 6696928794914170755ull}},
{{3828112768801108980ull, 8973559675619481482ull,
  16778617842022907828ull, 8371160993642713444ull}},
{{7004256498928081017ull, 14831846834116951734ull,
  1263264114409541584ull, 5231975621026695903ull}},
{{17978692660514877079ull, 93064468936638051ull,
  15414138198294090693ull, 6539969526283369878ull}},
{{17861679807216208444ull, 4728016604598185468ull,
  10044300711012837558ull, 8174961907854212348ull}},
{{1940177842655354470ull, 16790068433156029630ull,
  15501059981237799281ull, 5109351192408882717ull}},
{{11648594340173968895ull, 7152527486162873325ull,
  5541266921265085390ull, 6386688990511103397ull}},
{{725684869935297407ull, 18164031394558367465ull,
  11538269670008744641ull, 7983361238138879246ull}},
{{11982768089778030640ull, 4434990593957897809ull,
  2599732525328077497ull, 4989600773836799529ull}},
{{1143402056940374587ull, 10155424260874760166ull,
  7861351675087484775ull, 6237000967295999411ull}},
{{10652624608030244042ull, 8082594307666062303ull,
  5215003575431968065ull, 7796251209119999264ull}},
{{13575419407659984382ull, 16580836488359758699ull,
  3259377234644980040ull, 4872657005699999540ull}},
{{12357588241147592574ull, 2279301536740146758ull,
  4074221543306225051ull, 6090821257124999425ull}},
{{6223613264579714909ull, 16684184976207347160ull,
  9704462947560169217ull, 7613526571406249281ull}},
{{3889758290362321819ull, 3510086582488510119ull,
  17594504388293575521ull, 4758454107128905800ull}},
{{250511844525514369ull, 8999294246538025553ull, 3546386411657417785ull,
  5948067633911132251ull}},
{{4924825824084280865ull, 15860803826599919845ull,
  18268041069853935943ull, 7435084542388915313ull}},
{{5383859149266369493ull, 16830531419266031759ull,
  4499996641017628108ull, 4646927838993072071ull}},
{{2118137918155573962ull, 2591420200372988083ull,
  1013309782844647232ull, 5808659798741340089ull}},
{{16482730452976631164ull, 3239275250466235103ull,
  5878323246983196944ull, 7260824748426675111ull}},
{{15991727047793401051ull, 4049094063082793879ull,
  2736218040301608276ull, 9076030935533343889ull}},
{{16912358432511957513ull, 11754055826281521982ull,
  13239351321256974932ull, 5672519334708339930ull}},
{{11917076003785171083ull, 14692569782851902478ull,
  7325817114716442857ull, 7090649168385424913ull}},
{{5672972967876688046ull, 4530654173282714386ull,
  13768957411822941476ull, 8863311460481781141ull}},
{{8157294123350317933ull, 12055030895156472299ull,
  10911441391603032374ull, 5539569662801113213ull}},
{{5584931635760509512ull, 5845416582090814566ull,
  18250987757931178372ull, 6924462078501391516ull}},
{{16204536581555412698ull, 7306770727613518207ull,
  4366990623704421349ull, 8655577598126739396ull}},
{{17045364391113214793ull, 6872574713972142831ull,
  11952741176670039151ull, 5409735998829212122ull}},
{{16695019470464130587ull, 3979032374037790635ull,
  5717554433982773131ull, 6762169998536515153ull}},
{{16257088319652775329ull, 362104449119850390ull,
  11758629060905854318ull, 8452712498170643941ull}},
{{5548994181355596677ull, 14061373335982070206ull,
  9654986172279852900ull, 5282945311356652463ull}},
{{16159614763549271654ull, 17576716669977587757ull,
  7457046696922428221ull, 6603681639195815579ull}},
{{6364460399154425855ull, 8135837782189820985ull,
  4709622352725647373ull, 8254602048994769474ull}},
{{15507002795539985920ull, 7390741623082332067ull,
  7555199988880917512ull, 5159126280621730921ull}},
{{14772067475997594496ull, 9238427028852915084ull,
  14055686004528534794ull, 6448907850777163651ull}},
{{18340271287441503ull, 2324661749211368048ull, 12957921487233280589ull,
  8061134813471454564ull}},
{{11462669554650940ull, 3758756602470798982ull, 17322072966375576176ull,
  5038209258419659102ull}},
{{9237700373798089483ull, 4698445753088498727ull,
  12429219171114694412ull, 6297761573024573878ull}},
{{6935439448820223949ull, 5873057191360623409ull,
  6313151927038592207ull, 7872201966280717348ull}},
{{15863864701581109728ull, 10588189772241471486ull,
  13169091991253895937ull, 4920126228925448342ull}},
{{10606458840121611352ull, 17846923233729227262ull,
  7237992952212594113ull, 6150157786156810428ull}},
{{4034701513297238382ull, 8473595986879370366ull,
  9047491190265742642ull, 7687697232696013035ull}},
{{16356746501092937701ull, 9907683510226994382ull,
  3348838984702395199ull, 4804810770435008147ull}},
{{11222561089511396318ull, 7772918369356355074ull,
  18021106786160157711ull, 6006013463043760183ull}},
{{4804829325034469590ull, 5104461943268055939ull,
  17914697464272809235ull, 7507516828804700229ull}},
{{697175318932849542ull, 884445705328841010ull, 13502528924384199724ull,
  4692198018002937643ull}},
{{10094841185520837735ull, 1105557131661051262ull,
  12266475137052861751ull, 5865247522503672054ull}},
{{3395179445046271361ull, 15217004469858477790ull,
  6109721884461301380ull, 7331559403129590068ull}},
{{13467346343162615009ull, 574511513613545621ull,
  7637152355576626726ull, 9164449253911987585ull}},
{{10722934473690328333ull, 14194127751290629725ull,
  16302435268303861463ull, 5727780783694992240ull}},
{{18015354110540298320ull, 13130973670685899252ull,
  1931300011670275213ull, 7159725979618740301ull}},
{{4072448564465821284ull, 2578659033075210354ull,
  7025811033015231921ull, 8949657474523425376ull}},
{{7156966371218526206ull, 13140876941740476231ull,
  4391131895634519950ull, 5593535921577140860ull}},
{{4334521945595769854ull, 7202724140320819481ull,
  5488914869543149938ull, 6991919901971426075ull}},
{{10029838450422100221ull, 18226777212255800159ull,
  2249457568501549518ull, 8739899877464282594ull}},
{{13186178059154894494ull, 6780049739232487195ull,
  6017596998740856353ull, 5462437423415176621ull}},
{{11871036555516230214ull, 13086748192467996898ull,
  12133682266853458345ull, 6828046779268970776ull}},
{{5615423657540511959ull, 2523377185302832411ull,
  15167102833566822932ull, 8535058474086213470ull}},
{{1203796776749126023ull, 10800482777669046065ull,
  4867753252551876428ull, 5334411546303883419ull}},
{{6116431989363795432ull, 13500603472086307581ull,
  1473005547262457631ull, 6668014432879854274ull}},
{{12257226005132132194ull, 12264068321680496572ull,
  11064628970932847847ull, 8335018041099817842ull}},
{{16884138290062358430ull, 14582571728691392213ull,
  11527079125260417808ull, 5209386275687386151ull}},
{{7270114807295784325ull, 18228214660864240267ull,
  9797162888148134356ull, 6511732844609232689ull}},
{{4475957490692342502ull, 4338524252370748718ull,
  16858139628612555850ull, 8139666055761540861ull}},
{{16632531486964877776ull, 7323263676159105852ull,
  12842180277096541358ull, 5087291284850963038ull}},
{{2343920284996545604ull, 18377451632053658124ull,
  6829353309515900889ull, 6359114106063703798ull}},
{{2929900356245682005ull, 9136756484784908943ull,
  17760063673749651920ull, 7948892632579629747ull}},
{{8748716750294633109ull, 5710472802990568089ull,
  8794196786879838498ull, 4968057895362268592ull}},
{{15547581956295679290ull, 16361463040592985919ull,
  10992745983599798122ull, 6210072369202835740ull}},
{{14822791426942211209ull, 11228456763886456591ull,
  13740932479499747653ull, 7762590461503544675ull}},
{{16181773669479963862ull, 9323628486642729321ull,
  6282239790473648331ull, 4851619038439715422ull}},
{{6392159031567791115ull, 7042849589876023748ull,
  17076171774946836222ull, 6064523798049644277ull}},
{{7990198789459738893ull, 18026934024199805493ull,
  7510156663401381565ull, 7580654747562055347ull}},
{{7299717252626030761ull, 13572676774338572385ull,
  2388004905412169526ull, 4737909217226284592ull}},
{{13736332584209926355ull, 7742473931068439673ull,
  2985006131765211908ull, 5922386521532855740ull}},
{{3335357674980244231ull, 9678092413835549592ull,
  3731257664706514885ull, 7402983151916069675ull}},
{{2084598546862652645ull, 8354650767860912447ull, 26193031227877851ull,
  4626864469947543547ull}},
{{16440806238860479518ull, 5831627441398752654ull,
  13867799344317011026ull, 5783580587434429433ull}},
{{11327635761720823589ull, 16512906338603216626ull,
  3499691125114100070ull, 7229475734293036792ull}},
{{4936172665296253678ull, 11417760886399244975ull,
  4374613906392625088ull, 9036844667866295990ull}},
{{10002636943451240405ull, 7136100553999528109ull,
  16569191746777554392ull, 5648027917416434993ull}},
{{17114982197741438410ull, 8920125692499410136ull,
  6876431628189779278ull, 7060034896770543742ull}},
{{2946983673467246397ull, 1926785078769486863ull,
  17818911572091999906ull, 8825043620963179677ull}},
{{8759393823558110854ull, 5815926692658317193ull,
  13442662741771193893ull, 5515652263101987298ull}},
{{15560928297875026471ull, 11881594384250284395ull,
  7579956390359216558ull, 6894565328877484123ull}},
{{14839474353916395185ull, 5628620943458079686ull,
  4863259469521632794ull, 8618206661096855154ull}},
{{4662985452770359087ull, 8129574108088687708ull,
  7651223186878408400ull, 5386379163185534471ull}},
{{5828731815962948858ull, 10161967635110859635ull,
  4952342965170622596ull, 6732973953981918089ull}},
{{2674228751526298169ull, 12702459543888574544ull,
  10802114724890666149ull, 8416217442477397611ull}},
{{1671392969703936356ull, 10244880224144053042ull,
  4445478693842972391ull, 5260135901548373507ull}},
{{11312613248984696253ull, 8194414261752678398ull,
  945162348876327585ull, 6575169876935466884ull}},
{{4917394524376094508ull, 14854703845618235902ull,
  1181452936095409481ull, 8218962346169333605ull}},
{{16908429633017222779ull, 2366660875870315582ull,
  3044251094273324878ull, 5136851466355833503ull}},
{{11912165004416752666ull, 12181698131692670286ull,
  17640371923123819809ull, 6421064332944791878ull}},
{{5666834218666165025ull, 1392064609333674146ull,
  12827092867049998954ull, 8026330416180989848ull}},
{{8153457405093741045ull, 5481726399260934245ull,
  8016933041906249346ull, 5016456510113118655ull}},
{{14803507774794564210ull, 16075530035930943614ull,
  5409480283955423778ull, 6270570637641398319ull}},
{{9281012681638429454ull, 10871040508058903710ull,
  2150164336516891819ull, 7838213297051747899ull}},
{{1188946907596630505ull, 4488557308323120867ull,
  17484753774818915051ull, 4898883310657342436ull}},
{{15321241689777951843ull, 999010616976513179ull,
  3409198144814092198ull, 6123604138321678046ull}},
{{14539866093795051900ull, 10472135308075417282ull,
  13484869717872391055ull, 7654505172902097557ull}},
{{13699102327049295341ull, 13462613595188217657ull,
  10733886582883938361ull, 4784065733063810973ull}},
{{3288819853529455465ull, 2993208938703108360ull,
  18029044247032310856ull, 5980082166329763716ull}},
{{4111024816911819331ull, 3741511173378885450ull,
  4089561235080836954ull, 7475102707912204646ull}},
{{7181076528997274986ull, 6950130501789191310ull,
  16391033827207686808ull, 4671939192445127903ull}},
{{18199717698101369540ull, 8687663127236489137ull,
  15877106265582220606ull, 5839923990556409879ull}},
{{8914589067344548213ull, 1636206872190835614ull,
  15234696813550387854ull, 7299904988195512349ull}},
{{1919864297325909458ull, 11268630627093320326ull,
  5208312961655821105ull, 9124881235244390437ull}},
{{15034973241110857124ull, 125365114292243347ull,
  5561038610248582143ull, 5703050772027744023ull}},
{{14182030532961183500ull, 13991764448147467896ull,
  2339612244383339774ull, 7128813465034680029ull}},
{{17727538166201479375ull, 8266333523329559062ull,
  7536201323906562622ull, 8911016831293350036ull}},
{{6468025335448536706ull, 554772433653586510ull,
  13933497864296377447ull, 5569385519558343772ull}},
{{17308403706165446690ull, 14528523597349146849ull,
  17416872330370471808ull, 6961731899447929715ull}},
{{7800446577424644651ull, 18160654496686433562ull,
  17159404394535701856ull, 8702164874309912144ull}},
{{9486965129317790811ull, 11350409060429020976ull,
  10724627746584813660ull, 5438853046443695090ull}},
{{11858706411647238513ull, 14188011325536276220ull,
  4182412646376241267ull, 6798566308054618863ull}},
{{14823383014559048142ull, 13123328138492957371ull,
  616329789542913680ull, 8498207885068273579ull}},
{{6958771374885711137ull, 8202080086558098357ull,
  16526107182960178714ull, 5311379928167670986ull}},
{{13310150237034526825ull, 1029228071342847138ull,
  11434261941845447585ull, 6639224910209588733ull}},
{{7414315759438382723ull, 5898221107605946827ull, 457769372024645769ull,
  8299031137761985917ull}},
{{2328104340435295250ull, 15215603238322186527ull,
  2591948866729097557ull, 5186894461101241198ull}},
{{16745188480826282774ull, 5184445992620569446ull,
  12463308120266147755ull, 6483618076376551497ull}},
{{11708113564178077660ull, 1868871472348323904ull,
  1744077095050520982ull, 8104522595470689372ull}},
{{7317570977611298537ull, 15003102725499866152ull,
  10313420221261351421ull, 5065326622169180857ull}},
{{9146963722014123172ull, 4918820351592668978ull,
  17503461295004077181ull, 6331658277711476071ull}},
{{2210332615662878157ull, 10760211457918224127ull,
  17267640600327708572ull, 7914572847139345089ull}},
{{8298986912430380704ull, 15948504198053665887ull,
  3874746347563736001ull, 4946608029462090681ull}},
{{5762047622110587976ull, 6100572192284918647ull,
  9455118952882057906ull, 6183260036827613351ull}},
{{2590873509210847066ull, 16849087277210924117ull,
  7207212672675184478ull, 7729075046034516689ull}},
{{3925138952470473368ull, 5918993529829439669ull,
  16033722966490460059ull, 4830671903771572930ull}},
{{9518109709015479614ull, 2787055893859411682ull,
  10818781671258299266ull, 6038339879714466163ull}},
{{2674265099414573710ull, 12707191904179040411ull,
  8911791070645486178ull, 7547924849643082704ull}},
{{17812316751629966233ull, 12553680958539288160ull,
  5569869419153428861ull, 4717453031026926690ull}},
{{3818651865827906175ull, 1857043142891946489ull,
  16185708810796561885ull, 5896816288783658362ull}},
{{9385000850712270622ull, 6932989947042321015ull,
  11008763976640926548ull, 7371020360979572953ull}},
{{7119565044962950374ull, 8666237433802901269ull,
  18372640989228546089ull, 9213775451224466191ull}},
{{6755571162315537936ull, 16945613442195283053ull,
  18400429645908923161ull, 5758609657015291369ull}},
{{13056149971321810324ull, 7346958747461940104ull,
  9165479002103990240ull, 7198262071269114212ull}},
{{16320187464152262904ull, 9183698434327425130ull,
  11456848752629987800ull, 8997827589086392765ull}},
{{14811803183522552219ull, 5739811521454640706ull,
  9466373479607436327ull, 5623642243178995478ull}},
{{9291381942548414466ull, 2563078383390912979ull,
  2609594812654519601ull, 7029552803973744348ull}},
{{7002541409758130179ull, 7815533997666029128ull,
  3261993515818149501ull, 8786941004967180435ull}},
{{4376588381098831362ull, 7190551757754962157ull,
  18179647011882201102ull, 5491838128104487771ull}},
{{10082421494800927106ull, 18211561734048478504ull,
  18112872746425363473ull, 6864797660130609714ull}},
{{12603026868501158883ull, 8929394112278434418ull,
  13417718896176928534ull, 8580997075163262143ull}},
{{12488577811240612206ull, 969185301746633607ull,
  15303603337751662190ull, 5363123171977038839ull}},
{{10999036245623377353ull, 10434853664038067817ull,
  14517818153762189833ull, 6703903964971298549ull}},
{{18360481325456609595ull, 17655253098474972675ull,
  4312214636920573579ull, 8379879956214123187ull}},
{{9169457819196687045ull, 8728690177333163970ull, 389291138861664535ull,
  5237424972633826992ull}},
{{2238450237141082998ull, 6299176703239067059ull, 486613923577080669ull,
  6546781215792283740ull}},
{{16633120851708517460ull, 12485656897476221727ull,
  608267404471350836ull, 8183476519740354675ull}},
{{17313229559958905269ull, 17026907597777414387ull,
  16521068192290451936ull, 5114672824837721671ull}},
{{17029850931521243682ull, 2836890423512216368ull,
  16039649221935677017ull, 6393341031047152089ull}},
{{2840569590692002986ull, 8157799047817658365ull,
  6214503472137432559ull, 7991676288808940112ull}},
{{4081199003396195818ull, 12016153432527118334ull,
  3884064670085895349ull, 4994797680505587570ull}},
{{14324870791100020581ull, 1185133735376734205ull,
  14078452874462144995ull, 6243497100631984462ull}},
{{4071030433592862014ull, 15316475224503081469ull,
  8374694056222905435ull, 7804371375789980578ull}},
{{4850237030209232711ull, 7266954006100731966ull,
  9845869803566703801ull, 4877732109868737861ull}},
{{15286168324616316697ull, 13695378526053302861ull,
  16919023272885767655ull, 6097165137335922326ull}},
{{5272652350488232159ull, 12507537139139240673ull,
  11925407054252433761ull, 7621456421669902908ull}},
{{14824622765123614859ull, 899681684320943564ull,
  16676751445762546909ull, 4763410263543689317ull}},
{{84034382694966958ull, 5736288123828567360ull, 7010881251921019924ull,
  5954262829429611647ull}},
{{105042978368708697ull, 7170360154785709200ull, 4151915546473887001ull,
  7442828536787014559ull}},
{{65651861480442936ull, 16010690142809538010ull, 9512476244187261231ull,
  4651767835491884099ull}},
{{9305436863705329478ull, 15401676660084534608ull,
  7278909286806688635ull, 5814709794364855124ull}},
{{11631796079631661847ull, 14640409806678280356ull,
  9098636608508360794ull, 7268387242956068905ull}},
{{14539745099539577309ull, 9077140221493074637ull,
  15984981779062838897ull, 9085484053695086131ull}},
{{11393183696425929770ull, 17202427684501641408ull,
  7684770602700580358ull, 5678427533559428832ull}},
{{14241479620532412213ull, 12279662568772275952ull,
  9605963253375725448ull, 7098034416949286040ull}},
{{17801849525665515266ull, 15349578210965344940ull,
  12007454066719656810ull, 8872543021186607550ull}},
{{1902783916686171233ull, 14205172400280728492ull,
  2892972773272397602ull, 5545339388241629719ull}},
{{2378479895857714042ull, 8533093463496134807ull,
  17451274021872660715ull, 6931674235302037148ull}},
{{16808157925104306264ull, 6054680810942780604ull,
  3367348453631274278ull, 8664592794127546436ull}},
{{1281726666335415607ull, 17619233562121401590ull,
  11327964820374322231ull, 5415370496329716522ull}},
{{10825530369774045317ull, 17412355934224364083ull,
  4936583988613126981ull, 6769213120412145653ull}},
{{8920226943790168742ull, 7930386862498291392ull,
  10782416004193796631ull, 8461516400515182066ull}},
{{5575141839868855464ull, 11874020816702513976ull,
  11350696021048510798ull, 5288447750321988791ull}},
{{6968927299836069330ull, 5619153984023366662ull,
  9576684007883250594ull, 6610559687902485989ull}},
{{17934531161649862470ull, 16247314516883984135ull,
  16582541028281451146ull, 8263199609878107486ull}},
{{18126611003672245900ull, 14766257591479877988ull,
  5752402124248519062ull, 5164499756173817179ull}},
{{4211519680880755759ull, 9234449952495071678ull,
  2578816636883260924ull, 6455624695217271474ull}},
{{14487771637955720506ull, 11543062440618839597ull,
  12446892832958851963ull, 8069530869021589342ull}},
{{11360700282936019269ull, 4908571016173080796ull,
  3167622002171894573ull, 5043456793138493339ull}},
{{14200875353670024086ull, 10747399788643738899ull,
  17794585557997031928ull, 6304320991423116673ull}},
{{13139408173660142203ull, 13434249735804673624ull,
  8408173892214126198ull, 7880401239278895842ull}},
{{8212130108537588877ull, 3784720066450533111ull,
  9866794701061216778ull, 4925250774549309901ull}},
{{5653476617244598192ull, 13954272119917942197ull,
  16945179394753908876ull, 6156563468186637376ull}},
{{11678531789983135644ull, 17442840149897427746ull,
  2734730169732834479ull, 7695704335233296721ull}},
{{11910768387166847682ull, 17819304121326974197ull,
  13238421402151491309ull, 4809815209520810450ull}},
{{1053402428676395890ull, 8439072096376554035ull,
  7324654715834588329ull, 6012269011901013063ull}},
{{15151811091127658574ull, 15160526138898080447ull,
  4544132376365847507ull, 7515336264876266329ull}},
{{16387410959595868465ull, 7169485827597606327ull,
  14369297781297124452ull, 4697085165547666455ull}},
{{15872577681067447677ull, 8961857284497007909ull,
  13349936208194017661ull, 5871356456934583069ull}},
{{6005664046052145885ull, 15814007624048647791ull,
  2852362204960358364ull, 7339195571168228837ull}},
{{2895394039137794452ull, 1320765456351258123ull,
  8177138774627835860ull, 9173994463960286046ull}},
{{17950522338956979196ull, 10048850447074312134ull,
  499025715715009508ull, 5733746539975178779ull}},
{{13214780886841448187ull, 12561063058842890168ull,
  14458840199925925597ull, 7167183174968973473ull}},
{{16518476108551810234ull, 1866270768271448998ull,
  4238492194625243285ull, 8958978968711216842ull}},
{{5712361549417493492ull, 3472262239383349576ull,
  7260743640068164957ull, 5599361855444510526ull}},
{{7140451936771866865ull, 8952013817656574874ull,
  18299301586939982004ull, 6999202319305638157ull}},
{{18148936957819609390ull, 11190017272070718592ull,
  9039068928392813793ull, 8749002899132047697ull}},
{{11343085598637255869ull, 76231767403117264ull,
  17178633126313978381ull, 5468126811957529810ull}},
{{14178856998296569836ull, 4706975727681284484ull,
  12249919371037697168ull, 6835158514946912263ull}},
{{17723571247870712295ull, 5883719659601605605ull,
  10700713195369733556ull, 8543948143683640329ull}},
{{13383075039132889136ull, 12900696824105779311ull,
  18217160793174553232ull, 5339967589802275205ull}},
{{12117157780488723516ull, 16125871030132224139ull,
  8936392936186027828ull, 6674959487252844007ull}},
{{10534761207183516491ull, 1710594713955728558ull,
  6558805151805146882ull, 8343699359066055009ull}},
{{1972539736062309903ull, 5680807714649718253ull,
  15628468265946686561ull, 5214812099416284380ull}},
{{7077360688505275283ull, 11712695661739535720ull,
  1088841258723806585ull, 6518515124270355476ull}},
{{8846700860631594104ull, 805811521892255938ull, 1361051573404758232ull,
  8148143905337944345ull}},
{{10140874056322134219ull, 503632201182659961ull,
  12379872279446443655ull, 5092589940836215215ull}},
{{17287778588830055677ull, 14464598306760488663ull,
  10863154330880666664ull, 6365737426045269019ull}},
{{16998037217610181693ull, 18080747883450610829ull,
  8967256895173445426ull, 7957171782556586274ull}},
{{12929616270220057510ull, 15912153445584019672ull,
  10216221577910791295ull, 4973232364097866421ull}},
{{16162020337775071888ull, 15278505788552636686ull,
  17381962990815877023ull, 6216540455122333026ull}},
{{10979153385364064051ull, 14486446217263407954ull,
  12504081701665070471ull, 7770675568902916283ull}},
{{11473656884279927936ull, 15971557913430711827ull,
  5509208054326975092ull, 4856672230564322677ull}},
{{9730385086922522016ull, 1517703318078838168ull,
  11498196086336106770ull, 6070840288205403346ull}},
{{12162981358653152520ull, 11120501184453323518ull,
  5149373071065357654ull, 7588550360256754183ull}},
{{2990177330730832421ull, 2338627221855939295ull,
  10135887197056930390ull, 4742843975160471364ull}},
{{17572779718695704238ull, 12146656064174699926ull,
  12669858996321162987ull, 5928554968950589205ull}},
{{12742602611514854490ull, 10571634061790987004ull,
  2002265690119290022ull, 7410693711188236507ull}},
{{17187498669051559864ull, 1995585270191978973ull,
  17392317120820413928ull, 4631683569492647816ull}},
{{7649315281032286118ull, 2494481587739973717ull,
  3293652327315965794ull, 5789604461865809771ull}},
{{14173330119717745552ull, 12341474021529742954ull,
  17952123464427120954ull, 7237005577332262213ull}},
{{8493290612792406132ull, 6203470490057402885ull,
  8605096275251737481ull, 9046256971665327767ull}},
{{7614149642208947785ull, 15406384102354346563ull,
  12295714199673417781ull, 5653910607290829854ull}},
{{4906001034333796827ull, 5422922072660769492ull,
  6146270712736996419ull, 7067388259113537318ull}},
{{6132501292917246033ull, 2166966572398573961ull,
  16906210427776021332ull, 8834235323891921647ull}},
{{15362028354141748531ull, 10577726144603884533ull,
  17483910545001095188ull, 5521397077432451029ull}},
{{5367477387395021951ull, 13222157680754855667ull,
  8019830125969205273ull, 6901746346790563787ull}},
{{2097660715816389535ull, 2692639045661405872ull,
  5413101639034118688ull, 8627182933488204734ull}},
{{1311037947385243460ull, 1682899403538378670ull,
  17218246579678487892ull, 5391989333430127958ull}},
{{10862169471086330132ull, 2103624254422973337ull,
  12299436187743334057ull, 6739986666787659948ull}},
{{18189397857285300569ull, 7241216336456104575ull,
  15374295234679167571ull, 8424983333484574935ull}},
{{18285902688444394712ull, 2219917201071371407ull,
  16526463549315561588ull, 5265614583427859334ull}},
{{18245692342128105486ull, 2774896501339214259ull,
  11434707399789676177ull, 6582018229284824168ull}},
{{18195429409232743953ull, 8080306645101405728ull,
  14293384249737095221ull, 8227522786606030210ull}},
{{11372143380770464971ull, 7356034662402072532ull,
  13545051174513072417ull, 5142201741628768881ull}},
{{14215179225963081214ull, 13806729346429978569ull,
  3096255912859176809ull, 6427752177035961102ull}},
{{3933915977171687805ull, 3423353627755309500ull,
  13093691927928746820ull, 8034690221294951377ull}},
{{11682069522587080686ull, 11362968054201844245ull,
  1266028427314384906ull, 5021681388309344611ull}},
{{767528847951687146ull, 4980338030897529499ull,
  15417593589425144845ull, 6277101735386680763ull}},
{{14794469115221772644ull, 10837108557049299777ull,
  14660305968354043152ull, 7846377169233350954ull}},
{{2329014169372526047ull, 6773192848155812361ull,
  13774377248648664874ull, 4903985730770844346ull}},
{{7522953730143045462ull, 17689863097049541259ull,
  7994599523956055284ull, 6129982163463555433ull}},
{{4792006144251418924ull, 3665584797602374958ull,
  14604935423372457010ull, 7662477704329444291ull}},
{{16830061895439300539ull, 6902676516928872252ull,
  6822241630394091679ull, 4789048565205902682ull}},
{{2590833295589574058ull, 4016659627733702412ull,
  17751174074847390407ull, 5986310706507378352ull}},
{{3238541619486967573ull, 409138516239740111ull, 3742223519849686393ull,
  7482888383134222941ull}},
{{8941617539820436589ull, 11784926618718307329ull,
  4644732709119747947ull, 4676805239458889338ull}},
{{15788707943202933640ull, 10119472254970496257ull,
  15029287923254460742ull, 5846006549323611672ull}},
{{5900826873721503338ull, 3425968281858344514ull, 339865830358524312ull,
  7307508186654514591ull}},
{{16599405629006654981ull, 4282460352322930642ull,
  14259890343230319102ull, 9134385233318143238ull}},
{{14986314536556547267ull, 16511595775483995363ull,
  4300745446091561534ull, 5708990770823839524ull}},
{{14121207152268296180ull, 11416122682500218396ull,
  5375931807614451918ull, 7136238463529799405ull}},
{{17651508940335370225ull, 5046781316270497187ull,
  11331600777945452802ull, 8920298079412249256ull}},
{{8726350078495912439ull, 7765924341096448646ull,
  7082250486215908001ull, 5575186299632655785ull}},
{{1684565561265114740ull, 14319091444797948712ull,
  13464499126197272905ull, 6968982874540819731ull}},
{{2105706951581393425ull, 4063806250715272178ull,
  12218937889319203228ull, 8711228593176024664ull}},
{{5927752863165758795ull, 11763250943551820919ull,
  7636836180824502017ull, 5444517870735015415ull}},
{{2798005060529810589ull, 869005624157612437ull, 4934359207603239618ull,
  6805647338418769269ull}},
{{8109192344089651141ull, 10309629067051791354ull,
  10779635027931437426ull, 8507059173023461586ull}},
{{9679931233483419867ull, 11055204185334757500ull,
  11348957910884536295ull, 5316911983139663491ull}},
{{12099914041854274834ull, 9207319213241058971ull,
  9574511370178282465ull, 6646139978924579364ull}},
{{10513206533890455638ull, 16120835034978711618ull,
  11968139212722853081ull, 8307674973655724205ull}},
{{11182440102108922678ull, 3157992869220612905ull,
  9785930017165477128ull, 5192296858534827628ull}},
{{142992072353989635ull, 3947491086525766132ull,
  12232412521456846410ull, 6490371073168534535ull}},
{{178740090442487044ull, 14157735895011983473ull,
  10678829633393670108ull, 8112963841460668169ull}},
{{11640927602595024163ull, 18071956971237265478ull,
  18203483566939513577ull, 5070602400912917605ull}},
{{5327787466389004395ull, 8754888158764418136ull,
  8919296403392228260ull, 6338253001141147007ull}},
{{6659734332986255494ull, 10943610198455522670ull,
  6537434485812897421ull, 7922816251426433759ull}},
{{17997392013398573396ull, 9145599383248395620ull,
  11003425581274142744ull, 4951760157141521099ull}},
{{4049995943038665129ull, 11431999229060494526ull,
  9142595958165290526ull, 6189700196426901374ull}},
{{14285866965653107219ull, 5066626999470842349ull,
  2204872910851837350ull, 7737125245533626718ull}},
{{11234509862746885964ull, 17001699929951440180ull,
  15213103624564562055ull, 4835703278458516698ull}},
{{14043137328433607455ull, 16640438894011912321ull,
  9793007493850926761ull, 6044629098073145873ull}},
{{3718863605259845606ull, 6965490562232726690ull,
  16852945385741046356ull, 7555786372591432341ull}},
{{6935975771714791408ull, 13576803638250229989ull,
  12838933875301847924ull, 4722366482869645213ull}},
{{13281655733070877164ull, 16971004547812787486ull,
  2213609288845146193ull, 5902958103587056517ull}},
{{7378697629483820647ull, 7378697629483820646ull,
  7378697629483820646ull, 7378697629483820646ull}},
{{0ull, 0ull, 0ull, 9223372036854775808ull}},
{{0ull, 0ull, 0ull, 5764607523034234880ull}},
{{0ull, 0ull, 0ull, 7205759403792793600ull}},
{{0ull, 0ull, 0ull, 9007199254740992000ull}},
{{0ull, 0ull, 0ull, 5629499534213120000ull}},
{{0ull, 0ull, 0ull, 7036874417766400000ull}},
{{0ull, 0ull, 0ull, 8796093022208000000ull}},
{{0ull, 0ull, 0ull, 5497558138880000000ull}},
{{0ull, 0ull, 0ull, 6871947673600000000ull}},
{{0ull, 0ull, 0ull, 8589934592000000000ull}},
{{0ull, 0ull, 0ull, 5368709120000000000ull}},
{{0ull, 0ull, 0ull, 6710886400000000000ull}},
{{0ull, 0ull, 0ull, 8388608000000000000ull}},
{{0ull, 0ull, 0ull, 5242880000000000000ull}},
{{0ull, 0ull, 0ull, 6553600000000000000ull}},
{{0ull, 0ull, 0ull, 8192000000000000000ull}},
{{0ull, 0ull, 0ull, 5120000000000000000ull}},
{{0ull, 0ull, 0ull, 6400000000000000000ull}},
{{0ull, 0ull, 0ull, 8000000000000000000ull}},
{{0ull, 0ull, 0ull, 5000000000000000000ull}},
{{0ull, 0ull, 0ull, 6250000000000000000ull}},
{{0ull, 0ull, 0ull, 7812500000000000000ull}},
{{0ull, 0ull, 0ull, 4882812500000000000ull}},
{{0ull, 0ull, 0ull, 6103515625000000000ull}},
{{0ull, 0ull, 0ull, 7629394531250000000ull}},
{{0ull, 0ull, 0ull, 4768371582031250000ull}},
{{0ull, 0ull, 0ull, 5960464477539062500ull}},
{{0ull, 0ull, 0ull, 7450580596923828125ull}},
{{0ull, 0ull, 2305843009213693952ull, 4656612873077392578ull}},
{{0ull, 0ull, 12105675798371893248ull, 5820766091346740722ull}},
{{0ull, 0ull, 5908722711110090752ull, 7275957614183425903ull}},
{{0ull, 0ull, 2774217370460225536ull, 9094947017729282379ull}},
{{0ull, 0ull, 17874786921033498624ull, 5684341886080801486ull}},
{{0ull, 0ull, 13120111614437097472ull, 7105427357601001858ull}},
{{0ull, 0ull, 7176767481191596032ull, 8881784197001252323ull}},
{{0ull, 0ull, 2179636666531053568ull, 5551115123125782702ull}},
{{0ull, 0ull, 11947917870018592768ull, 6938893903907228377ull}},
{{0ull, 0ull, 1099839282241077248ull, 8673617379884035472ull}},
{{0ull, 0ull, 687399551400673280ull, 5421010862427522170ull}},
{{0ull, 0ull, 10082621476105617408ull, 6776263578034402712ull}},
{{0ull, 0ull, 12603276845132021760ull, 8470329472543003390ull}},
{{0ull, 0ull, 3265362009780125696ull, 5293955920339377119ull}},
{{0ull, 0ull, 17916760567507320832ull, 6617444900424221398ull}},
{{0ull, 0ull, 13172578672529375232ull, 8271806125530276748ull}},
{{0ull, 0ull, 17456233707185635328ull, 5169878828456422967ull}},
{{0ull, 0ull, 17208606115554656256ull, 6462348535570528709ull}},
{{0ull, 0ull, 7675699589161156608ull, 8077935669463160887ull}},
{{0ull, 0ull, 11714841270866804736ull, 5048709793414475554ull}},
{{0ull, 0ull, 5420179551728730112ull, 6310887241768094443ull}},
{{0ull, 0ull, 2163538421233524736ull, 7888609052210118054ull}},
{{0ull, 0ull, 15187269568553116672ull, 4930380657631323783ull}},
{{0ull, 0ull, 14372400942264007936ull, 6162975822039154729ull}},
{{0ull, 0ull, 4130443122547846208ull, 7703719777548943412ull}},
{{0ull, 0ull, 11804898988447179688ull, 4814824860968089632ull}},
{{0ull, 0ull, 14756123735558974610ull, 6018531076210112040ull}},
{{0ull, 9223372036854775808ull, 18445154669448718262ull,
  7523163845262640050ull}},
{{0ull, 1152921504606846976ull, 16139907686832836818ull,
  4701977403289150031ull}},
{{0ull, 10664523917613334528ull, 15563198590113658118ull,
  5877471754111437539ull}},
{{0ull, 4107282860161892352ull, 14842312219214684744ull,
  7346839692639296924ull}},
{{0ull, 5134103575202365440ull, 106146200308804314ull,
  9183549615799121156ull}},
{{0ull, 7820500752928866304ull, 9289713412047778504ull,
  5739718509874450722ull}},
{{0ull, 9775625941161082880ull, 2388769728204947322ull,
  7174648137343063403ull}},
{{0ull, 2996160389596577792ull, 16821020215538347865ull,
  8968310171678829253ull}},
{{0ull, 13401815289566330880ull, 12818980643925161367ull,
  5605193857299268283ull}},
{{0ull, 12140583093530525696ull, 11412039786479063805ull,
  7006492321624085354ull}},
{{0ull, 1340670811630993408ull, 5041677696244053949ull,
  8758115402030106693ull}},
{{0ull, 3143762266483064832ull, 5456891569366227670ull,
  5473822126268816683ull}},
{{0ull, 13153074869958606848ull, 2209428443280396683ull,
  6842277657836020854ull}},
{{0ull, 11829657569020870656ull, 11985157590955271662ull,
  8552847072295026067ull}},
{{0ull, 2781849962210656256ull, 5184880485133350837ull,
  5345529420184391292ull}},
{{0ull, 8088998471190708224ull, 6481100606416688546ull,
  6681911775230489115ull}},
{{0ull, 887876052133609472ull, 3489689739593472779ull,
  8352389719038111394ull}},
{{0ull, 16695823597079363584ull, 6792742105673308390ull,
  5220243574398819621ull}},
{{0ull, 11646407459494428672ull, 13102613650519023392ull,
  6525304467998524526ull}},
{{0ull, 14558009324368035840ull, 7154895026294003432ull,
  8156630584998155658ull}},
{{0ull, 9098755827730022400ull, 9083495409861140049ull,
  5097894115623847286ull}},
{{0ull, 15985130803089915904ull, 2130997225471649253ull,
  6372367644529809108ull}},
{{0ull, 6146355448580231168ull, 2663746531839561567ull,
  7965459555662261385ull}},
{{0ull, 10759001183003726336ull, 13194056628468195739ull,
  4978412222288913365ull}},
{{0ull, 8837065460327270016ull, 2657512730303080962ull,
  6223015277861141707ull}},
{{0ull, 1822959788554311712ull, 17156948968161014915ull,
  7778769097326427133ull}},
{{0ull, 17280250932342302484ull, 13028936114314328273ull,
  4861730685829016958ull}},
{{0ull, 7765255610145714393ull, 7062798106038134534ull,
  6077163357286271198ull}},
{{4611686018427387904ull, 483197475827367183ull,
  18051869669402443976ull, 7596454196607838997ull}},
{{9799832789158199296ull, 301998422392104489ull,
  13588261552590221437ull, 4747783872879899373ull}},
{{16861477004875137024ull, 4989184046417518515ull,
  3150268885455613084ull, 5934729841099874217ull}},
{{16465160237666533376ull, 6236480058021898144ull,
  8549522125246904259ull, 7418412301374842771ull}},
{{10290725148541583360ull, 1591957027049992388ull,
  3037608319065621210ull, 4636507688359276732ull}},
{{12863406435676979200ull, 11213318320667266293ull,
  3797010398832026512ull, 5795634610449095915ull}},
{{2244199989314060288ull, 14016647900834082867ull,
  134576980112645236ull, 7244543263061369894ull}},
{{16640308041924739072ull, 17520809876042603583ull,
  9391593261995582353ull, 9055679078826712367ull}},
{{17317721553844043776ull, 4032977144885545383ull,
  12787274816388320827ull, 5659799424266695229ull}},
{{17035465923877666816ull, 429535412679543825ull,
  2149035465203237322ull, 7074749280333369037ull}},
{{7459274349564919808ull, 9760291302704205590ull,
  7297980349931434556ull, 8843436600416711296ull}},
{{50360450050686976ull, 15323554101044904302ull, 4561237718707146597ull,
  5527147875260444560ull}},
{{9286322599418134528ull, 5319384571023966665ull,
  5701547148383933247ull, 6908934844075555700ull}},
{{16219589267700056064ull, 2037544695352570427ull,
  7126933935479916559ull, 8636168555094444625ull}},
{{7831400283098841088ull, 8190994462236438373ull,
  15983548755743417609ull, 5397605346934027890ull}},
{{14400936372300939264ull, 14850429096222935870ull,
  10756063907824496203ull, 6747006683667534863ull}},
{{8777798428521398272ull, 13951350351851281934ull,
  8833393866353232350ull, 8433758354584418579ull}},
{{874437999398486016ull, 4107907951479663305ull, 3215028157257076267ull,
  5271098971615261612ull}},
{{5704733517675495424ull, 523198920922191227ull, 4018785196571345334ull,
  6588873714519077015ull}},
{{2519230878666981376ull, 9877370688007514842ull, 411795477286793763ull,
  8236092143148846269ull}},
{{6186205317594251264ull, 3867513670791002824ull,
  2563215182517940054ull, 5147557589468028918ull}},
{{7732756646992814080ull, 14057764125343529338ull,
  12427391015002200875ull, 6434446986835036147ull}},
{{442573771886241792ull, 12960519138252023769ull,
  10922552750325363190ull, 8043058733543795184ull}},
{{11805823653497370880ull, 3488638442980126951ull,
  6826595468953351994ull, 5026911708464871990ull}},
{{10145593548444325696ull, 13584170090579934497ull,
  17756616373046465800ull, 6283639635581089987ull}},
{{17293677953982795024ull, 16980212613224918121ull,
  17584084447880694346ull, 7854549544476362484ull}},
{{3891019693598165034ull, 15224318901692961730ull,
  1766680743070658158ull, 4909093465297726553ull}},
{{14087146653852482101ull, 9807026590261426354ull,
  6820036947265710602ull, 6136366831622158191ull}},
{{8385561280460826818ull, 3035411200972007135ull,
  3913360165654750349ull, 7670458539527697739ull}},
{{12158504827929098618ull, 4202975009821198411ull,
  140007094320525016ull, 4794036587204811087ull}},
{{10586445016483985368ull, 5253718762276498014ull,
  14010066923182819982ull, 5992545734006013858ull}},
{{4009684233750205902ull, 15790520489700398326ull,
  8289211617123749169ull, 7490682167507517323ull}},
{{16341110701376042401ull, 2951546278421667097ull,
  2874914251488649279ull, 4681676354692198327ull}},
{{6591330321437889289ull, 17524490903309247584ull,
  17428700869642975310ull, 5852095443365247908ull}},
{{8239162901797361611ull, 12682241592281783672ull,
  3339132013344167522ull, 7315119304206559886ull}},
{{10298953627246702013ull, 6629429953497453782ull,
  13397287053534985211ull, 9143899130258199857ull}},
{{1825159998601800855ull, 1837550711722214662ull,
  1455775380818283901ull, 5714936956411374911ull}},
{{11504822035107026876ull, 6908624408080156231ull,
  15654777281305018588ull, 7143671195514218638ull}},
{{9769341525456395691ull, 8635780510100195289ull,
  10345099564776497427ull, 8929588994392773298ull}},
{{17635053499478717067ull, 3091519809598928103ull,
  11077373246412698796ull, 5580993121495483311ull}},
{{17432130855921008430ull, 3864399761998660129ull,
  9235030539588485591ull, 6976241401869354139ull}},
{{7955105514619096825ull, 218813684070937258ull, 6932102156058219085ull,
  8720301752336692674ull}},
{{9583626965064323420ull, 2442601561758029738ull,
  8944249865963774832ull, 5450188595210432921ull}},
{{2756161669475628467ull, 3053251952197537173ull,
  15791998350882106444ull, 6812735744013041151ull}},
{{8056888105271923487ull, 3816564940246921466ull,
  15128311920175245151ull, 8515919680016301439ull}},
{{9647241084222340084ull, 9302882115295407772ull,
  16372723977750610075ull, 5322449800010188399ull}},
{{12059051355277925104ull, 7016916625691871811ull,
  15854218953760874690ull, 6653062250012735499ull}},
{{10462128175670018476ull, 17994517818969615572ull,
  15206087673773705458ull, 8316327812515919374ull}},
{{15762202146648537356ull, 15858259655283397636ull,
  4892118777681178007ull, 5197704882822449609ull}},
{{1256008609601120079ull, 15211138550676859142ull,
  10726834490528860413ull, 6497131103528062011ull}},
{{10793382798856175906ull, 5178865133063910215ull,
  8796857094733687613ull, 8121413879410077514ull}},
{{13663393276926191798ull, 5542633717378637836ull,
  10109721702635942662ull, 5075883674631298446ull}},
{{17079241596157739747ull, 16151664183578073103ull,
  3413780091440152519ull, 6344854593289123058ull}},
{{16737365976769786780ull, 15577894211045203475ull,
  13490597151154966457ull, 7931068241611403822ull}},
{{8155010726267422785ull, 2818654854262170316ull,
  3819937201044466132ull, 4956917651007127389ull}},
{{10193763407834278482ull, 3523318567827712895ull,
  9386607519732970569ull, 6196147063758909236ull}},
{{8130518241365460198ull, 9015834228212029023ull,
  11733259399666213211ull, 7745183829698636545ull}},
{{11999102928494494480ull, 3329053383418824187ull,
  415758097150301401ull, 4840739893561647841ull}},
{{10387192642190730196ull, 8773002747700918138ull,
  5131383639865264655ull, 6050924866952059801ull}},
{{3760618765883636937ull, 6354567416198759769ull,
  11025915568258968723ull, 7563656083690074751ull}},
{{13879601774745742846ull, 1665761625910530903ull,
  13808726257802937308ull, 4727285052306296719ull}},
{{12737816200004790653ull, 2082202032388163629ull,
  12649221803826283731ull, 5909106315382870899ull}},
{{2087212194723824604ull, 16437810595767368249ull,
  11199841236355466759ull, 7386382894228588624ull}},
{{12833722667770860138ull, 17191160649995687011ull,
  6999900772722166724ull, 4616489308892867890ull}},
{{11430467316286187268ull, 3042206738785057148ull,
  17973248002757484214ull, 5770611636116084862ull}},
{{14288084145357734085ull, 13026130460336097243ull,
  13243187966592079459ull, 7213264545145106078ull}},
{{13248419163269779702ull, 11670977056992733650ull,
  7330612921385323516ull, 9016580681431382598ull}},
{{12891947995471000218ull, 16517732697475234339ull,
  18416691131147990909ull, 5635362925894614123ull}},
{{11503248975911362368ull, 6812107816561879212ull,
  18409177895507600733ull, 7044203657368267654ull}},
{{14379061219889202960ull, 13126820789129736919ull,
  13788100332529725108ull, 8805254571710334568ull}},
{{15904442290071833706ull, 17427635030060861382ull,
  8617562707831078192ull, 5503284107318959105ull}},
{{10657180825735016325ull, 3337799713866525112ull,
  15383639403216235645ull, 6879105134148698881ull}},
{{13321476032168770406ull, 8783935660760544294ull,
  5394491198738130844ull, 8598881417685873602ull}},
{{3714236501678093600ull, 14713331824830115992ull,
  7983243017638719681ull, 5374300886053671001ull}},
{{4642795627097617000ull, 4556606725755481278ull,
  14590739790475787506ull, 6717876107567088751ull}},
{{15026866570726797057ull, 14919130444049127405ull,
  13626738719667346478ull, 8397345134458860939ull}},
{{11697634615917942113ull, 4712770509103316724ull,
  6210868690578397597ull, 5248340709036788087ull}},
{{14622043269897427641ull, 10502649154806533809ull,
  3151899844795609092ull, 6560425886295985109ull}},
{{4442496032089620839ull, 13128311443508167262ull,
  8551560824421899269ull, 8200532357869981386ull}},
{{16611618075338176737ull, 10511037661406298490ull,
  9956411533691074947ull, 5125332723668738366ull}},
{{11541150557317945113ull, 8527111058330485209ull,
  3222142380259067876ull, 6406665904585922958ull}},
{{591380141365267679ull, 10658888822913106512ull,
  13251050012178610653ull, 8008332380732403697ull}},
{{369612588353292299ull, 8967648523534385522ull, 1364377229970549802ull,
  5005207737957752311ull}},
{{9685387772296391182ull, 1986188617563206094ull,
  15540529592745350965ull, 6256509672447190388ull}},
{{2883362678515713170ull, 7094421790381395522ull, 978917917222137090ull,
  7820637090558987986ull}},
{{6413787692499708635ull, 9045699637415760105ull,
  5223509716691223585ull, 4887898181599367491ull}},
{{12628920634052023698ull, 15918810565197088035ull,
  1917701127436641577ull, 6109872726999209364ull}},
{{11174464774137641718ull, 6063455151214196332ull,
  2397126409295801972ull, 7637340908749011705ull}},
{{16207412520690801882ull, 13013031506363648515ull,
  13027419051878345992ull, 4773338067968132315ull}},
{{15647579632436114448ull, 16266289382954560644ull,
  11672587796420544586ull, 5966672584960165394ull}},
{{1112730466835591444ull, 11109489691838424998ull,
  5367362708670904925ull, 7458340731200206743ull}},
{{14530514597054408365ull, 9249274066612709575ull,
  10272130720560397434ull, 4661462957000129214ull}},
{{13551457227890622552ull, 2338220546411111161ull,
  3616791363845720985ull, 5826828696250161518ull}},
{{3104263479581114478ull, 7534461701441276856ull,
  13744361241661927039ull, 7283535870312701897ull}},
{{3880329349476393097ull, 4806391108374208166ull,
  3345393496795245087ull, 9104419837890877372ull}},
{{16260263898704909398ull, 9921523470374961959ull,
  11314242972351803987ull, 5690262398681798357ull}},
{{15713643854953748843ull, 7790218319541314545ull,
  307745660157591272ull, 7112827998352247947ull}},
{{5806996763410022342ull, 9737772899426643182ull,
  14219740130479152802ull, 8891034997940309933ull}},
{{17464431032413427676ull, 10697794080569039892ull,
  11193180590763164453ull, 5556896873712693708ull}},
{{3383794716807232979ull, 17983928619138687770ull,
  13991475738453955566ull, 6946121092140867135ull}},
{{13453115432863817032ull, 13256538737068583904ull,
  12877658654640056554ull, 8682651365176083919ull}},
{{8408197145539885645ull, 12897022729095252844ull,
  14966065686791117202ull, 5426657103235052449ull}},
{{10510246431924857056ull, 6897906374514290247ull,
  4872524053206732791ull, 6783321379043815562ull}},
{{8526122021478683416ull, 4010696949715474905ull,
  15314027103363191797ull, 8479151723804769452ull}},
{{16858041309492646895ull, 4812528602785865767ull,
  347894902747219065ull, 5299469827377980908ull}},
{{16460865618438420715ull, 10627346771909720113ull,
  434868628434023831ull, 6624337284222476135ull}},
{{6741023967765862181ull, 8672497446459762238ull,
  14378643840824693501ull, 8280421605278095168ull}},
{{18048198035135827576ull, 7726153913251045350ull,
  8986652400515433438ull, 5175263503298809480ull}},
{{13336875507065008661ull, 434320354709030880ull,
  11233315500644291798ull, 6469079379123511850ull}},
{{16671094383831260826ull, 9766272480241064408ull,
  4818272338950588939ull, 8086349223904389813ull}},
{{10419433989894538017ull, 3798077290936971303ull,
  5317263221057812039ull, 5053968264940243633ull}},
{{8412606468940784617ull, 135910595243826225ull,
  11258265044749652953ull, 6317460331175304541ull}},
{{15127444104603368675ull, 4781574262482170685ull,
  237773250654902479ull, 7896825413969130677ull}},
{{11760495574590799374ull, 9906012941692438534ull,
  2454451290873008001ull, 4935515883730706673ull}},
{{5477247431383723409ull, 16994202195542936072ull,
  7679750132018647905ull, 6169394854663383341ull}},
{{6846559289229654262ull, 7407694689146506378ull,
  14211373683450697786ull, 7711743568329229176ull}},
{{8890785574195921818ull, 9241495199143954390ull,
  8882108552156686116ull, 4819839730205768235ull}},
{{1890109930890126464ull, 11551868998929942988ull,
  6490949671768469741ull, 6024799662757210294ull}},
{{2362637413612658080ull, 604778193380265023ull,
  17337059126565362985ull, 7530999578446512867ull}},
{{8394177411148993156ull, 11907201416931135399ull,
  8529818944889657913ull, 4706874736529070542ull}},
{{5881035745508853541ull, 1048943715881755537ull,
  1438901644257296584ull, 5883593420661338178ull}},
{{11962980700313454830ull, 1311179644852194421ull,
  11021999092176396538ull, 7354491775826672722ull}},
{{1118667820109654825ull, 10862346592920018835ull,
  4554126828365719864ull, 9193114719783340903ull}},
{{16840068452064391930ull, 6788966620575011771ull,
  9763858295369656771ull, 5745696699864588064ull}},
{{16438399546653102009ull, 3874522257291376810ull,
  12204822869212070964ull, 7182120874830735080ull}},
{{11324627396461601703ull, 4843152821614221013ull,
  15256028586515088705ull, 8977651093538418850ull}},
{{9383735132002195016ull, 14556185559577357893ull,
  14146703884999318344ull, 5611031933461511781ull}},
{{16341354933430131674ull, 18195231949471697366ull,
  3848321800966984218ull, 7013789916826889727ull}},
{{11203321629932888785ull, 13520667899984845900ull,
  198716232781342369ull, 8767237396033612159ull}},
{{16225448055562831299ull, 1532888409849446831ull,
  7041726673129420837ull, 5479523372521007599ull}},
{{15670124051026151219ull, 6527796530739196443ull,
  4190472322984388142ull, 6849404215651259499ull}},
{{14975969045355301120ull, 17383117700278771362ull,
  626404385303097273ull, 8561755269564074374ull}},
{{13971666671774451104ull, 3946919535033150245ull,
  14226560796096599508ull, 5351097043477546483ull}},
{{3629525284435900168ull, 4933649418791437807ull,
  13171514976693361481ull, 6688871304346933104ull}},
{{18371964660827038922ull, 10778747791916685162ull,
  16464393720866701851ull, 8361089130433666380ull}},
{{16094163931444287230ull, 4430874360734234274ull,
  1066874038686912849ull, 5225680706521041488ull}},
{{10894332877450583230ull, 10150278969345180747ull,
  1333592548358641061ull, 6532100883151301860ull}},
{{9006230078385841133ull, 17299534730108863838ull,
  1666990685448301326ull, 8165126103939127325ull}},
{{1017207780563762804ull, 6200523187890651995ull,
  3347712187618882281ull, 5103203814961954578ull}},
{{15106567780986867217ull, 12362340003290702897ull,
  13408012271378378659ull, 6379004768702443222ull}},
{{5048151670951420310ull, 10841238985685990718ull,
  7536643302368197516ull, 7973755960878054028ull}},
{{16990152849626801406ull, 15999146402908520006ull,
  13933774100834899255ull, 4983597475548783767ull}},
{{12014319025178725949ull, 15387246985208262104ull,
  12805531607616236165ull, 6229496844435979709ull}},
{{15017898781473407436ull, 5399000676228163918ull,
  2171856454238131495ull, 7786871055544974637ull}},
{{4774500719993491744ull, 10291904450283684305ull,
  3663253293112526136ull, 4866794409715609148ull}},
{{10579811918419252584ull, 12864880562854605381ull,
  4579066616390657670ull, 6083493012144511435ull}},
{{17836450916451453633ull, 6857728666713480918ull,
  1112147252060934184ull, 7604366265180639294ull}},
{{6536095804354770617ull, 4286080416695925574ull,
  14530150087820247577ull, 4752728915737899558ull}},
{{17393491792298239079ull, 9969286539297294871ull,
  8939315572920533663ull, 5940911144672374448ull}},
{{17130178721945410945ull, 7849922155694230685ull,
  11174144466150667079ull, 7426138930840468060ull}},
{{13012204710429575793ull, 11823730374949976034ull,
  16207212328198942732ull, 4641336831775292537ull}},
{{7041883851182193933ull, 14779662968687470043ull,
  6423957354966514703ull, 5801671039719115672ull}},
{{4190668795550354512ull, 13862892692431949650ull,
  8029946693708143379ull, 7252088799648894590ull}},
{{14461708031292718948ull, 12716929847112549158ull,
  814061330280403416ull, 9065110999561118238ull}},
{{4426881501130561438ull, 7948081154445343224ull,
  14343846386707415847ull, 5665694374725698898ull}},
{{5533601876413201798ull, 5323415424629291126ull,
  8706435946529494001ull, 7082117968407123623ull}},
{{16140374382371278055ull, 11265955299214001811ull,
  6271358914734479597ull, 8852647460508904529ull}},
{{7781890979768354833ull, 9347065071222445084ull,
  15448814367777519508ull, 5532904662818065330ull}},
{{9727363724710443541ull, 11683831339028056355ull,
  10087645922867123577ull, 6916130828522581663ull}},
{{7547518637460666522ull, 769731118502906732ull, 7997871385156516568ull,
  8645163535653227079ull}},
{{13940571185267692384ull, 481081949064316707ull,
  11916198643363904711ull, 5403227209783266924ull}},
{{12814027963157227576ull, 14436410491612559596ull,
  14895248304204880888ull, 6754034012229083655ull}},
{{16017534953946534470ull, 18045513114515699495ull,
  14007374361828713206ull, 8442542515286354569ull}},
{{16928488373857665900ull, 6666759678144924280ull,
  1837079948501863898ull, 5276589072053971606ull}},
{{2713866393612530759ull, 17556821634535931159ull,
  11519721972482105680ull, 6595736340067464507ull}},
{{17227391047297827161ull, 3499282969460362332ull,
  9787966447175244197ull, 8244670425084330634ull}},
{{1543747367706366168ull, 4492894865126420410ull,
  10729165047911915527ull, 5152919015677706646ull}},
{{11153056246487733517ull, 1004432562980637608ull,
  4188084273035118601ull, 6441148769597133308ull}},
{{13941320308109666897ull, 5867226722153184914ull,
  5235105341293898251ull, 8051435961996416635ull}},
{{13325011210995929715ull, 1361173692132046619ull,
  966097829094992455ull, 5032147476247760397ull}},
{{12044577995317524239ull, 15536525170447221986ull,
  5819308304796128472ull, 6290184345309700496ull}},
{{5832350457292129491ull, 973912389349475867ull, 7274135380995160591ull,
  7862730431637125620ull}},
{{1339376026593886980ull, 7526224270984504273ull,
  13769706649976751177ull, 4914206519773203512ull}},
{{6285906051669746629ull, 14019466357158018245ull,
  17212133312470938971ull, 6142758149716504390ull}},
{{12469068583014571190ull, 12912646928020134902ull,
  12291794603733897906ull, 7678447687145630488ull}},
{{3181481845956719090ull, 12682090348439972218ull,
  7682371627333686191ull, 4799029804466019055ull}},
{{13200224344300674670ull, 11240926917122577368ull,
  4991278515739719835ull, 5998787255582523819ull}},
{{16500280430375843337ull, 9439472627975833806ull,
  1627412126247261890ull, 7498484069478154774ull}},
{{5700989250557514182ull, 10511356410912284033ull,
  14852190634186702393ull, 4686552543423846733ull}},
{{11737922581624280632ull, 17750881532067742945ull,
  4730180237451214279ull, 5858190679279808417ull}},
{{837345171748187077ull, 17576915896657290778ull,
  10524411315241405753ull, 7322738349099760521ull}},
{{10270053501540009654ull, 8136086815539449760ull,
  17767200162479145096ull, 9153422936374700651ull}},
{{6418783438462506034ull, 5085054259712156100ull,
  8798657092335771733ull, 5720889335234187907ull}},
{{8023479298078132543ull, 10968003843067583029ull,
  6386635346992326762ull, 7151111669042734884ull}},
{{14641035141025053582ull, 4486632766979702978ull,
  7983294183740408453ull, 8938889586303418605ull}},
{{13762332981568046393ull, 5109988488576008313ull,
  7295401874051449235ull, 5586805991439636628ull}},
{{3367858171677894279ull, 1775799592292622488ull,
  9119252342564311544ull, 6983507489299545785ull}},
{{4209822714597367849ull, 2219749490365778110ull,
  16010751446632777334ull, 8729384361624432231ull}},
{{16466197251905518618ull, 15222401486760775030ull,
  16924248681786567689ull, 5455865226015270144ull}},
{{11359374528027122464ull, 5192943803168805076ull,
  2708566778523657996ull, 6819831532519087681ull}},
{{14199218160033903080ull, 6491179753961006345ull,
  7997394491581960399ull, 8524789415648859601ull}},
{{1956982322380107569ull, 10974516373866710822ull,
  16527586603307195009ull, 5327993384780537250ull}},
{{11669599939829910269ull, 18329831485760776431ull,
  11436111217279217953ull, 6659991730975671563ull}},
{{9975313906359999932ull, 9077231301918806827ull,
  9683453003171634538ull, 8324989663719589454ull}},
{{3928728182261306006ull, 10284955582126642171ull,
  1440472108554883682ull, 5203118539824743409ull}},
{{299224209399244603ull, 3632822440803526906ull, 6412276154120992507ull,
  6503898174780929261ull}},
{{9597402298603831562ull, 18376086106286572344ull,
  12627031211078628537ull, 8129872718476161576ull}},
{{5998376436627394726ull, 4567524788788025859ull,
  7891894506924142836ull, 5081170449047600985ull}},
{{2886284527356855504ull, 5709405985985032324ull,
  14476554152082566449ull, 6351463061309501231ull}},
{{3607855659196069380ull, 11748443500908678309ull,
  13484006671675820157ull, 7939328826636876539ull}},
{{4560752796211237315ull, 9648620197281617895ull,
  6121661160583693646ull, 4962080516648047837ull}},
{{1089254976836658739ull, 2837403209747246561ull,
  12263762469157004962ull, 6202600645810059796ull}},
{{5973254739473211328ull, 12770126049038834009ull,
  15329703086446256202ull, 7753250807262574745ull}},
{{15262499258239226840ull, 12593014799076659159ull,
  2663535401387828270ull, 4845781754539109216ull}},
{{14466438054371645646ull, 6517896461991048141ull,
  3329419251734785338ull, 6057227193173886520ull}},
{{4247989512682393345ull, 17370742614343585985ull,
  4161774064668481672ull, 7571533991467358150ull}},
{{14184208491494965601ull, 10856714133964741240ull,
  16436166845699964757ull, 4732208744667098843ull}},
{{17730260614368707001ull, 18182578685883314454ull,
  15933522538697568042ull, 5915260930833873554ull}},
{{12939453731106107943ull, 13504851320499367260ull,
  10693531136517184245ull, 7394076163542341943ull}},
{{17310530618796093273ull, 10746375084525798489ull,
  13600985987964322009ull, 4621297602213963714ull}},
{{7803105218212952879ull, 18044654874084636016ull,
  7777860448100626703ull, 5776622002767454643ull}},
{{9753881522766191098ull, 17944132574178407116ull,
  5110639541698395475ull, 7220777503459318304ull}},
{{12192351903457738873ull, 17818479699295620991ull,
  6388299427122994344ull, 9025971879324147880ull}},
{{14537748967302168652ull, 11136549812059763119ull,
  3992687141951871465ull, 5641232424577592425ull}},
{{13560500190700322911ull, 85629209792540187ull, 9602544945867227236ull,
  7051540530721990531ull}},
{{12338939219948015734ull, 107036512240675234ull,
  7391495163906646141ull, 8814425663402488164ull}},
{{12323523030894897738ull, 2372740829364115973ull,
  13843056514296429646ull, 5509016039626555102ull}},
{{1569345733336458460ull, 12189298073559920775ull,
  8080448606015761249ull, 6886270049533193878ull}},
{{15796740221952736787ull, 1401564536667737256ull,
  877188720664925754ull, 8607837561916492348ull}},
{{9872962638720460492ull, 5487663853844723689ull,
  9771614987270354404ull, 5379898476197807717ull}},
{{16952889316827963519ull, 6859579817305904611ull,
  16826204752515330909ull, 6724873095247259646ull}},
{{16579425627607566495ull, 13186160790059768668ull,
  11809383903789387828ull, 8406091369059074558ull}},
{{1138768980399953251ull, 17464722530642131226ull,
  2769178921440979488ull, 5253807105661921599ull}},
{{10646833262354717372ull, 3384159089593112416ull,
  17296531707083388073ull, 6567258882077401998ull}},
{{13308541577943396715ull, 8841884880418778424ull,
  12397292596999459283ull, 8209073602596752498ull}},
{{8317838486214622947ull, 3220335041048042563ull,
  12359993891552049956ull, 5130671001622970311ull}},
{{5785612089340890780ull, 4025418801310053204ull,
  10838306346012674541ull, 6413338752028712889ull}}
};

static const UINT256 multipliers2_binary64[] =
  { {{918777112239470733ull, 5138421092069233273ull,
      15049182825743144437ull, 3877256883183270350ull}},
{{12103450741218138969ull, 5517356191756964747ull,
  4794053247662077369ull, 2423285551989543969ull}},
{{10517627408095285807ull, 11508381258123593838ull,
  10604252578004984615ull, 3029106939986929961ull}},
{{3923662223264331450ull, 9773790554227104394ull,
  17867001740933618673ull, 3786383674983662451ull}},
{{7063974907967595060ull, 17637834142460410006ull,
  8861033078869817718ull, 2366489796864789032ull}},
{{18053340671814269633ull, 12823920641220736699ull,
  11076291348587272148ull, 2958112246080986290ull}},
{{17954989821340449138ull, 16029900801525920874ull,
  4621992148879314377ull, 3697640307601232863ull}},
{{15833554656765168615ull, 3101158973312618690ull,
  9806274120690653342ull, 2311025192250770539ull}},
{{10568571284101684961ull, 13099820753495549171ull,
  7646156632435928773ull, 2888781490313463174ull}},
{{8599028086699718297ull, 2539717886587272752ull, 334323753690135159ull,
  3610976862891828968ull}},
{{10748785108374647871ull, 17009705413516254652ull,
  417904692112668948ull, 4513721078614786210ull}},
{{15941362729588930728ull, 1407693846592883349ull,
  4872876450997805997ull, 2821075674134241381ull}},
{{6091645356703999697ull, 6371303326668492091ull,
  10702781582174645400ull, 3526344592667801726ull}},
{{3002870677452611718ull, 7964129158335615114ull,
  4155104940863530942ull, 4407930740834752158ull}},
{{6488480191835270228ull, 365894705532371542ull,
  16431998643321870551ull, 2754956713021720098ull}},
{{17333972276648863593ull, 14292426437197628139ull,
  11316626267297562380ull, 3443695891277150123ull}},
{{17055779327383691587ull, 17865533046497035174ull,
  9534096815694565071ull, 4304619864096437654ull}},
{{6048176061187419338ull, 18083487181701728840ull,
  1347124491381715265ull, 2690387415060273534ull}},
{{7560220076484274172ull, 8769300921844997338ull,
  10907277651081919890ull, 3362984268825341917ull}},
{{226903058750566907ull, 1738254115451470865ull,
  18245783082279787767ull, 4203730336031677396ull}},
{{11671029457787574077ull, 8003937849798251146ull,
  2180242389570091546ull, 2627331460019798373ull}},
{{5365414785379691788ull, 781550275393038125ull, 7336989005390002337ull,
  3284164325024747966ull}},
{{11318454500152002639ull, 5588623862668685560ull,
  18394608293592278729ull, 4105205406280934957ull}},
{{7074034062595001650ull, 15022104960236398235ull,
  13802473192708868157ull, 2565753378925584348ull}},
{{4230856559816364158ull, 4942573145013334082ull,
  17253091490886085197ull, 3207191723656980435ull}},
{{14511942736625231005ull, 10789902449694055506ull,
  16954678345180218592ull, 4008989654571225544ull}},
{{13681650228818157283ull, 6743689031058784691ull,
  10596673965737636620ull, 2505618534107015965ull}},
{{12490376767595308699ull, 8429611288823480864ull,
  17857528475599433679ull, 3132023167633769956ull}},
{{15612970959494135874ull, 5925328092601963176ull,
  3875166520789740483ull, 3915028959542212446ull}},
{{9758106849683834921ull, 1397487048662533033ull,
  16257037130775751514ull, 2446893099713882778ull}},
{{16809319580532181555ull, 10970230847682942099ull,
  11097924376614913584ull, 3058616374642353473ull}},
{{16399963457237839040ull, 13712788559603677624ull,
  37347415486478268ull, 3823270468302941842ull}},
{{10249977160773649400ull, 17793864886607074323ull,
  4635028153106436821ull, 2389544042689338651ull}},
{{8200785432539673846ull, 8407273052976679192ull,
  1182099172955658123ull, 2986930053361673314ull}},
{{10250981790674592308ull, 5897405297793461086ull,
  10700996003049348462ull, 3733662566702091642ull}},
{{1795177600744232288ull, 17520936366403076891ull,
  11299808520333230692ull, 2333539104188807276ull}},
{{16079030056212454072ull, 3454426384294294497ull,
  14124760650416538366ull, 2916923880236009095ull}},
{{6263729514983403878ull, 13541405017222643930ull,
  13044264794593285053ull, 3646154850295011369ull}},
{{17053033930584030656ull, 3091698216246141200ull,
  2470272937959442605ull, 4557693562868764212ull}},
{{10658146206615019160ull, 4238154394367532202ull,
  10767292623079427436ull, 2848558476792977632ull}},
{{4099310721413998142ull, 5297692992959415253ull,
  13459115778849284295ull, 3560698095991222040ull}},
{{9735824420194885581ull, 2010430222771881162ull,
  16823894723561605369ull, 4450872619989027550ull}},
{{10696576281049191393ull, 12785733935300895486ull,
  5903248183798615451ull, 2781795387493142219ull}},
{{4147348314456713433ull, 11370481400698731454ull,
  2767374211320881410ull, 3477244234366427774ull}},
{{14407557429925667599ull, 4989729714018638509ull,
  12682589801005877571ull, 4346555292958034717ull}},
{{11310566402917236201ull, 812738062047955116ull,
  10232461634842367434ull, 2716597058098771698ull}},
{{14138208003646545252ull, 10239294614414719703ull,
  3567205006698183484ull, 3395746322623464623ull}},
{{13061073986130793660ull, 12799118268018399629ull,
  18294064313654893067ull, 4244682903279330778ull}},
{{10469014250545439990ull, 5693605908297805816ull,
  16045476214461696071ull, 2652926814549581736ull}},
{{13086267813181799987ull, 2505321366944869366ull,
  1610101194367568473ull, 3316158518186977171ull}},
{{7134462729622474176ull, 7743337727108474612ull,
  15847684548241624303ull, 4145198147733721463ull}},
{{13682411242868822168ull, 11757115107083878488ull,
  16822331870292097045ull, 2590748842333575914ull}},
{{17103014053586027710ull, 861335828572684398ull,
  11804542801010345499ull, 3238436052916969893ull}},
{{12155395530127758829ull, 14911727840998019210ull,
  920620445980768161ull, 4048045066146212367ull}},
{{12208808224757237173ull, 2402300872982680150ull,
  7492916806379061957ull, 2530028166341382729ull}},
{{6037638244091770658ull, 7614562109655738092ull,
  13977832026401215350ull, 3162535207926728411ull}},
{{7547047805114713322ull, 294830600214896807ull,
  12860604014574131284ull, 3953169009908410514ull}},
{{11634433905837777682ull, 9407641161989086312ull,
  12649563527536219956ull, 2470730631192756571ull}},
{{14543042382297222103ull, 11759551452486357890ull,
  11200268390992887041ull, 3088413288990945714ull}},
{{8955430941016751820ull, 864381260325783651ull, 4776963451886332994ull,
  3860516611238682143ull}},
{{3291301328921775936ull, 5151924306131002686ull,
  9903131185070039977ull, 2412822882024176339ull}},
{{13337498698006995728ull, 11051591401091141261ull,
  7767227962910162067ull, 3016028602530220424ull}},
{{2836815317226580948ull, 9202803232936538673ull,
  9709034953637702584ull, 3770035753162775530ull}},
{{13302224619335082852ull, 5751752020585336670ull,
  10679832864450952019ull, 2356272345726734706ull}},
{{7404408737314077757ull, 2578004007304282934ull,
  4126419043708914216ull, 2945340432158418383ull}},
{{32138884787821389ull, 3222505009130353668ull, 546337786208754866ull,
  3681675540198022979ull}},
{{40173605984776736ull, 13251503298267717893ull,
  14517980288043107294ull, 4602094425247528723ull}},
{{2330951512954179412ull, 3670503542989935779ull,
  6767894670813248107ull, 2876309015779705452ull}},
{{16748747446474887977ull, 18423187484019583435ull,
  8459868338516560133ull, 3595386269724631815ull}},
{{16324248289666222067ull, 9193926299742315582ull,
  5963149404718312263ull, 4494232837155789769ull}},
{{5590969162614000888ull, 12663732964980029095ull,
  15256183424017414924ull, 2808895523222368605ull}},
{{2377025434840113206ull, 15829666206225036369ull,
  5235171224739604943ull, 3511119404027960757ull}},
{{7582967811977529411ull, 15175396739353907557ull,
  11155650049351894083ull, 4388899255034950946ull}},
{{7045197891699649834ull, 7178779952882498271ull,
  11583967299272321706ull, 2743062034396844341ull}},
{{4194811346197174389ull, 18196846977957898647ull,
  644901068808238420ull, 3428827542996055427ull}},
{{631828164319080082ull, 4299314648737821693ull,
  14641184391292461738ull, 4286034428745069283ull}},
{{2700735611913119003ull, 7298757673888526462ull,
  6844897235344094634ull, 2678771517965668302ull}},
{{12599291551746174562ull, 18346819129215433885ull,
  17779493581034894100ull, 3348464397457085377ull}},
{{1914056384400554490ull, 4486779837809740741ull,
  8389308921011453914ull, 4185580496821356722ull}},
{{3502128249464040509ull, 7415923417058475867ull,
  9855004094059546600ull, 2615987810513347951ull}},
{{18212718367112214348ull, 9269904271323094833ull,
  7707069099147045346ull, 3269984763141684939ull}},
{{8930839903608104222ull, 2364008302299092734ull,
  5022150355506418779ull, 4087480953927106174ull}},
{{970088921327677235ull, 17618406253432790623ull,
  16973902027473675448ull, 2554675596204441358ull}},
{{15047669206941760256ull, 3576263743081436662ull,
  11994005497487318503ull, 3193344495255551698ull}},
{{9586214471822424512ull, 18305387734133959540ull,
  5769134835004372320ull, 3991680619069439623ull}},
{{15214756081743791128ull, 11440867333833724712ull,
  10523238299518814556ull, 2494800386918399764ull}},
{{571701028470187294ull, 14301084167292155891ull,
  13154047874398518195ull, 3118500483647999705ull}},
{{14549684340869897829ull, 13264669190687806959ull,
  2607501787715984032ull, 3898125604559999632ull}},
{{16011081740684767999ull, 8290418244179879349ull,
  1629688617322490020ull, 2436328502849999770ull}},
{{6178794120573796287ull, 10363022805224849187ull,
  11260482808507888333ull, 3045410628562499712ull}},
{{3111806632289857455ull, 17565464524958449388ull,
  14075603510634860416ull, 3806763285703124640ull}},
{{11168251182035936718ull, 10978415328099030867ull,
  8797252194146787760ull, 2379227053564452900ull}},
{{9348627959117532993ull, 13723019160123788584ull,
  10996565242683484700ull, 2974033816955566125ull}},
{{11685784948896916241ull, 17153773950154735730ull,
  18357392571781743779ull, 3717542271194457656ull}},
{{11915301611487960555ull, 8415265709633015879ull,
  11473370357363589862ull, 2323463919496536035ull}},
{{10282440995932562789ull, 1295710100186494041ull,
  9730026928277099424ull, 2904329899370670044ull}},
{{17464737263343091390ull, 1619637625233117551ull,
  12162533660346374280ull, 3630412374213337555ull}},
{{17219235560751476334ull, 2024547031541396939ull,
  10591481057005579946ull, 4538015467766671944ull}},
{{8456179216255978757ull, 5877027913140760991ull,
  6619675660628487466ull, 2836259667354169965ull}},
{{5958538001892585542ull, 16569656928280727047ull,
  12886280594212997236ull, 3545324584192712456ull}},
{{2836486483938344023ull, 2265327086641357193ull,
  16107850742766246546ull, 4431655730240890570ull}},
{{13302019098529934775ull, 6027515447578236149ull,
  14679092732656291995ull, 2769784831400556606ull}},
{{2792465817880254756ull, 2922708291045407283ull,
  9125493878965589186ull, 3462231039250695758ull}},
{{17325640327632482157ull, 12876757400661534911ull,
  2183495311852210674ull, 4327788799063369698ull}},
{{17746054232411383205ull, 12659659393840847223ull,
  5976370588335019575ull, 2704867999414606061ull}},
{{17570881772086841102ull, 11212888223873671125ull,
  12082149253846162373ull, 3381084999268257576ull}},
{{8128544159826387665ull, 181052224559925195ull,
  15102686567307702967ull, 4226356249085321970ull}},
{{2774497090677798339ull, 7030686667991035103ull,
  14050865122994702258ull, 2641472655678326231ull}},
{{17303179418629411635ull, 18011730371843569686ull,
  12951895385315989918ull, 3301840819597907789ull}},
{{12405602236431988736ull, 13291290927949686300ull,
  2354811176362823686ull, 4127301024497384737ull}},
{{16976873434624768768ull, 3695370811541166033ull,
  13000972031295234564ull, 2579563140310865460ull}},
{{7386033737998797248ull, 4619213514426457542ull,
  16251215039119043205ull, 3224453925388581825ull}},
{{9170135643720752ull, 10385702911460459832ull, 6478960743616640294ull,
  4030567406735727282ull}},
{{5731334777325470ull, 1879378301235399491ull, 8661036483187788088ull,
  2519104629209829551ull}},
{{13842222223753820550ull, 2349222876544249363ull,
  6214609585557347206ull, 3148880786512286939ull}},
{{12691091761264887783ull, 12159900632535087512ull,
  3156575963519296103ull, 3936100983140358674ull}},
{{7931932350790554864ull, 14517466922975511551ull,
  6584545995626947968ull, 2460063114462724171ull}},
{{5303229420060805676ull, 18146833653719389439ull,
  3618996476106297056ull, 3075078893078405214ull}},
{{2017350756648619191ull, 4236797993439685183ull,
  13747117631987647129ull, 3843848616348006517ull}},
{{8178373250546468851ull, 14177213791968272999ull,
  10897791529205973407ull, 2402405385217504073ull}},
{{5611280544755698159ull, 13109831221532953345ull,
  18233925429934854663ull, 3003006731521880091ull}},
{{11625786699372010603ull, 11775603008488803777ull,
  18180720768991180425ull, 3753758414402350114ull}},
{{348587659466424771ull, 442222852664420505ull, 15974636499046875670ull,
  2346099009001468821ull}},
{{5047420592760418868ull, 9776150602685301439ull,
  6133237568526430875ull, 2932623761251836027ull}},
{{1697589722523135681ull, 7608502234929238895ull,
  3054860942230650690ull, 3665779701564795034ull}},
{{15957045208436083313ull, 287255756806772810ull,
  13041948214643089171ull, 4582224626955993792ull}},
{{14584839273699939975ull, 16320435912500090670ull,
  8151217634151930731ull, 2863890391847496120ull}},
{{9007677055270149160ull, 15788858872197725434ull,
  10189022042689913414ull, 3579862989809370150ull}},
{{2036224282232910642ull, 10512701553392380985ull,
  3512905516507615960ull, 4474828737261712688ull}},
{{12801855222464038911ull, 6570438470870238115ull,
  2195565947817259975ull, 2796767960788570430ull}},
{{11390633009652660735ull, 3601362070160409740ull,
  11967829471626350777ull, 3495959950985713037ull}},
{{14238291262065825919ull, 9113388606127900079ull,
  1124728784250774759ull, 4369949938732141297ull}},
{{15816461066432223055ull, 12613396906471019405ull,
  12232170536225203984ull, 2731218711707588310ull}},
{{5935518277758115107ull, 15766746133088774257ull,
  6066841133426729172ull, 3414023389634485388ull}},
{{12031083865625031788ull, 1261688592651416205ull,
  7583551416783411466ull, 4267529237043106735ull}},
{{9825270425229338820ull, 5400241388834523032ull,
  11657248663130714022ull, 2667205773151941709ull}},
{{12281588031536673524ull, 15973673772897929598ull,
  736502773631228815ull, 3334007216439927137ull}},
{{6128613002566066097ull, 15355406197695024094ull,
  5532314485466423923ull, 4167509020549908921ull}},
{{17665441181885955023ull, 7291285864345696106ull,
  14986911599484984712ull, 2604693137843693075ull}},
{{12858429440502667971ull, 9114107330432120133ull,
  14121953480928842986ull, 3255866422304616344ull}},
{{2237978745346171251ull, 2169262126185374359ull,
  17652441851161053733ull, 4069833027880770430ull}},
{{8316265743482438888ull, 3661631838079552926ull,
  6421090138548270679ull, 2543645642425481519ull}},
{{1171960142498272802ull, 18412097852881604870ull,
  3414676654757950444ull, 3179557053031851899ull}},
{{10688322214977616811ull, 4568378242392454471ull,
  18103403873729601768ull, 3974446316289814873ull}},
{{13597730412002092363ull, 2855236401495284044ull,
  4397098393439919249ull, 2484028947681134296ull}},
{{16997163015002615453ull, 8180731520296492959ull,
  5496372991799899061ull, 3105036184601417870ull}},
{{16634767750325881413ull, 14837600418798004103ull,
  16093838276604649634ull, 3881295230751772337ull}},
{{17314258871594757739ull, 13885186280176140468ull,
  3141119895236824165ull, 2425809519219857711ull}},
{{3196079515783895558ull, 3521424794938011874ull,
  17761457924328193919ull, 3032261899024822138ull}},
{{13218471431584645255ull, 18236839048954678554ull,
  12978450368555466590ull, 3790327373781027673ull}},
{{12873230663167791189ull, 6786338387169286192ull,
  1194002452706084763ull, 2368954608613142296ull}},
{{16091538328959738986ull, 3871236965534219836ull,
  1492503065882605954ull, 2961193260766427870ull}},
{{1667678837490122116ull, 14062418243772550604ull,
  11089000869208033250ull, 3701491575958034837ull}},
{{10265671310286102131ull, 13400697420785232031ull,
  9236468552468714733ull, 2313432234973771773ull}},
{{8220403119430239759ull, 2915813720699376327ull,
  16157271709013281321ull, 2891790293717214716ull}},
{{5663817880860411795ull, 8256453169301608313ull,
  1749845562557050035ull, 3614737867146518396ull}},
{{11691458369502902647ull, 5708880443199622487ull,
  2187306953196312544ull, 4518422333933147995ull}},
{{14224690508580396011ull, 3568050276999764054ull,
  17507967910243553004ull, 2824013958708217496ull}},
{{8557491098870719205ull, 4460062846249705068ull,
  3438215814094889639ull, 3530017448385271871ull}},
{{10696863873588399007ull, 963392539384743431ull,
  18132827822900775761ull, 4412521810481589838ull}},
{{13603068948633831235ull, 12131335383183934404ull,
  6721331370885596946ull, 2757826131550993649ull}},
{{17003836185792289044ull, 5940797192125142197ull,
  13013350232034384087ull, 3447282664438742061ull}},
{{7419737176958197593ull, 2814310471729039843ull,
  2431629734760816397ull, 4309103330548427577ull}},
{{2331492726385179544ull, 4064787054044343854ull,
  13048983630293980008ull, 2693189581592767235ull}},
{{12137737944836250237ull, 5080983817555429817ull,
  11699543519440087106ull, 3366486976990959044ull}},
{{1337114375763149085ull, 15574601808799063080ull,
  14624429399300108882ull, 4208108721238698805ull}},
{{835696484851968178ull, 14345812148926802329ull,
  11446111383776262003ull, 2630067950774186753ull}},
{{5656306624492348127ull, 13320579167731115007ull,
  472581174438163792ull, 3287584938467733442ull}},
{{2458697262188047254ull, 16650723959663893759ull,
  9814098504902480548ull, 4109481173084666802ull}},
{{8454214816508611390ull, 1183330437935157791ull,
  10745497583991438247ull, 2568425733177916751ull}},
{{5956082502208376333ull, 15314221102701110951ull,
  8820185961561909904ull, 3210532166472395939ull}},
{{2833417109333082513ull, 696032304666837073ull, 6413546433524999477ull,
  4013165208090494924ull}},
{{13300100739401646331ull, 2740863199630467122ull,
  13231838557807900481ull, 2508228255056559327ull}},
{{7401753887397282105ull, 8037765017965471807ull,
  11928112178832487697ull, 3135285318820699159ull}},
{{4640506340819214727ull, 14658892290884227663ull,
  10298454205113221717ull, 3919106648525873949ull}},
{{9817845490653091061ull, 11467650691016336241ull,
  8742376887409457525ull, 2449441655328671218ull}},
{{16883992881743751730ull, 499505308488256589ull,
  1704599072407046099ull, 3061802069160839023ull}},
{{7269933046897525950ull, 14459439690892484449ull,
  15965806895790971335ull, 3827252586451048778ull}},
{{16072923200379423479ull, 15954678834448884636ull,
  14590315328296744988ull, 2392032866531905486ull}},
{{1644409926764727733ull, 1496604469351554180ull,
  9014522123516155428ull, 2990041083164881858ull}},
{{2055512408455909666ull, 1870755586689442725ull,
  2044780617540418477ull, 3737551353956102323ull}},
{{3590538264498637493ull, 3475065250894595655ull,
  17418888950458619212ull, 2335969596222563951ull}},
{{18323230885905460578ull, 4343831563618244568ull,
  17161925169645886111ull, 2919961995278204939ull}},
{{4457294533672274107ull, 818103436095417807ull,
  16840720443629969735ull, 3649952494097756174ull}},
{{959932148662954729ull, 14857687350401435971ull,
  11827528517682686360ull, 4562440617622195218ull}},
{{16740858657410204370ull, 9286054594000897481ull,
  12003891341979066879ull, 2851525386013872011ull}},
{{7091015266480591750ull, 6995882224073733948ull,
  10393178159046445695ull, 3564406732517340014ull}},
{{8863769083100739688ull, 4133166761664779531ull,
  3768100661953281311ull, 4455508415646675018ull}},
{{3234012667724268353ull, 9500758253681569063ull,
  6966748932148188723ull, 2784692759779171886ull}},
{{17877573889937499153ull, 7264261798674573424ull,
  17931808202040011712ull, 3480865949723964857ull}},
{{3900223288712322326ull, 9080327248343216781ull,
  8579702197267850928ull, 4351082437154956072ull}},
{{4743482564658895406ull, 5675204530214510488ull,
  5362313873292406830ull, 2719426523221847545ull}},
{{5929353205823619257ull, 16317377699622913918ull,
  11314578360042896441ull, 3399283154027309431ull}},
{{16635063544134299879ull, 6561664069246478685ull,
  9531536931626232648ull, 4249103942534136789ull}},
{{12702757724297631377ull, 4101040043279049178ull,
  8263053591480089357ull, 2655689964083835493ull}},
{{6655075118517263413ull, 9737986072526199377ull,
  14940503007777499600ull, 3319612455104794366ull}},
{{12930529916573967170ull, 12172482590657749221ull,
  9452256722867098692ull, 4149515568880992958ull}},
{{10387424207072423433ull, 16831173656015869071ull,
  1295974433364548778ull, 2593447230550620599ull}},
{{8372594240413141387ull, 11815595033165060531ull,
  15455026096987849685ull, 3241809038188275748ull}},
{{5854056782089038830ull, 934435736174161952ull, 872038547525260491ull,
  4052261297735344686ull}},
{{3658785488805649269ull, 16724923399604708884ull,
  14380082147485451518ull, 2532663311084590428ull}},
{{4573481861007061586ull, 11682782212651110297ull,
  17975102684356814398ull, 3165829138855738035ull}},
{{10328538344686214887ull, 5380105728959112063ull,
  17857192337018630094ull, 3957286423569672544ull}},
{{13372865493069966160ull, 17197624135881608751ull,
  11160745210636643808ull, 2473304014731045340ull}},
{{12104395847910069796ull, 3050286096142459323ull,
  13950931513295804761ull, 3091630018413806675ull}},
{{10518808791460199341ull, 8424543638605462058ull,
  12826978373192368047ull, 3864537523017258344ull}},
{{11185941513090012492ull, 12182868801769495642ull,
  8016861483245230029ull, 2415335951885786465ull}},
{{4759054854507739807ull, 1393527946929705841ull,
  14632762872483925441ull, 3019169939857233081ull}},
{{10560504586562062663ull, 6353595952089520205ull,
  4455895535322743089ull, 3773962424821541352ull}},
{{8906158375814983117ull, 15500212516124419888ull,
  2784934709576714430ull, 2358726515513463345ull}},
{{11132697969768728896ull, 10151893608300749052ull,
  8092854405398280942ull, 2948408144391829181ull}},
{{13915872462210911119ull, 3466494973521160507ull,
  14727754025175239082ull, 3685510180489786476ull}},
{{12783154559336250995ull, 13556490753756226442ull,
  18409692531469048852ull, 4606887725612233095ull}},
{{12601157618012544776ull, 17696178757952417334ull,
  18423586859809237388ull, 2879304828507645684ull}},
{{6528074985660905162ull, 3673479373730970052ull,
  4582739501051995120ull, 3599131035634557106ull}},
{{8160093732076131452ull, 4591849217163712565ull,
  14951796413169769708ull, 4498913794543196382ull}},
{{7405901591761276110ull, 12093277797582096161ull,
  4733186739803718163ull, 2811821121589497739ull}},
{{13869063008128983041ull, 10504911228550232297ull,
  1304797406327259800ull, 3514776401986872174ull}},
{{3501270704879065090ull, 13131139035687790372ull,
  10854368794763850558ull, 4393470502483590217ull}},
{{11411666227404191489ull, 3595275878877481078ull,
  18313195542795876359ull, 2745919064052243885ull}},
{{5041210747400463553ull, 18329152903879015060ull,
  9056436373212681736ull, 3432398830065304857ull}},
{{6301513434250579442ull, 4464697056139217209ull,
  15932231484943240075ull, 4290498537581631071ull}},
{{15467660942475081911ull, 484592650873316803ull,
  16875173705730606903ull, 2681561585988519419ull}},
{{14722890159666464485ull, 14440798868873809716ull,
  16482281113735870724ull, 3351951982485649274ull}},
{{18403612699583080606ull, 18050998586092262145ull,
  11379479355315062597ull, 4189939978107061593ull}},
{{4584728909598343523ull, 13587717125521357793ull,
  194645569430832267ull, 2618712486316913496ull}},
{{10342597155425317307ull, 12372960388474309337ull,
  243306961788540334ull, 3273390607896141870ull}},
{{17539932462709034538ull, 6242828448738110863ull,
  9527505739090451226ull, 4091738259870177337ull}},
{{17879986816834228443ull, 8513453798888707193ull,
  17483906133000001776ull, 2557336412418860835ull}},
{{8514925465760621841ull, 10641817248610883992ull,
  17243196647822614316ull, 3196670515523576044ull}},
{{10643656832200777301ull, 13302271560763604990ull,
  3107251736068716279ull, 3995838144404470056ull}},
{{2040599501698097909ull, 15231448753118334975ull,
  1942032335042947674ull, 2497398840252793785ull}},
{{16385807432404786099ull, 9815938904543142910ull,
  7039226437231072497ull, 3121748550315992231ull}},
{{11258887253651206815ull, 16881609649106316542ull,
  4187347028111452717ull, 3902185687894990289ull}},
{{2425118515104616356ull, 12856849039905141791ull,
  14146306938638127708ull, 2438866054934368930ull}},
{{16866456199162934157ull, 16071061299881427238ull,
  8459511636442883827ull, 3048582568667961163ull}},
{{11859698212098891888ull, 15477140606424396144ull,
  5962703527126216880ull, 3810728210834951454ull}},
{{7412311382561807430ull, 9673212879015247590ull,
  17561747759736049262ull, 2381705131771844658ull}},
{{42017191347483479ull, 2868144061914283680ull, 12728812662815285770ull,
  2977131414714805823ull}},
{{52521489184354349ull, 12808552114247630408ull,
  11299329810091719308ull, 3721414268393507279ull}},
{{32825930740221468ull, 17228717108259544813ull,
  13979610158948406423ull, 2325883917745942049ull}},
{{4652718431852664739ull, 16924210366897043112ull,
  3639454643403344317ull, 2907354897182427562ull}},
{{5815898039815830924ull, 7320204903339140178ull,
  13772690341108956205ull, 3634193621478034452ull}},
{{16493244586624564463ull, 13761942147601313126ull,
  17215862926386195256ull, 4542742026847543065ull}},
{{5696591848212964885ull, 8601213842250820704ull,
  3842385301350290179ull, 2839213766779714416ull}},
{{7120739810266206107ull, 6139831284386137976ull,
  4802981626687862724ull, 3549017208474643020ull}},
{{8900924762832757633ull, 7674789105482672470ull,
  6003727033359828405ull, 4436271510593303775ull}},
{{951391958343085617ull, 7102586200140364246ull,
  10669858423490974609ull, 2772669694120814859ull}},
{{10412611984783632829ull, 13489918768602843211ull,
  8725637010936330357ull, 3465837117651018574ull}},
{{8404078962552153132ull, 3027340405471390302ull,
  1683674226815637139ull, 4332296397063773218ull}},
{{640863333167707804ull, 18032988817915476603ull,
  5663982410187161115ull, 2707685248164858261ull}},
{{14636137221741798467ull, 17929550003966957849ull,
  11691664031161339298ull, 3384606560206072826ull}},
{{4460113471895084371ull, 13188565468103921504ull,
  5391208002096898315ull, 4230758200257591033ull}},
{{2787570919934427732ull, 5937010408351256988ull,
  14898720047379031207ull, 2644223875160994395ull}},
{{3484463649918034665ull, 2809576992011683331ull,
  14011714040796401105ull, 3305279843951242994ull}},
{{18190637617679707043ull, 8123657258441992067ull,
  8291270514140725573ull, 4131599804939053743ull}},
{{9063305501836122950ull, 7383128795739938994ull,
  12099573098979035339ull, 2582249878086908589ull}},
{{2105759840440377880ull, 4617224976247535839ull,
  1289408318441630462ull, 3227812347608635737ull}},
{{16467257855832636061ull, 14994903257164195606ull,
  6223446416479425981ull, 4034765434510794671ull}},
{{5680350141468009635ull, 11677657544941316206ull,
  10807183037940723094ull, 2521728396569246669ull}},
{{16323809713689787851ull, 5373699894321869449ull,
  18120664815853291772ull, 3152160495711558336ull}},
{{6569704086830071102ull, 6717124867902336812ull,
  4204086946107063099ull, 3940200619639447921ull}},
{{13329437091123570247ull, 1892360033225266555ull,
  14156769387385384197ull, 2462625387274654950ull}},
{{12050110345477074904ull, 6977136059958971098ull,
  8472589697376954438ull, 3078281734093318688ull}},
{{5839265894991567822ull, 17944792111803489681ull,
  10590737121721193047ull, 3847852167616648360ull}},
{{15178756230438199649ull, 18133024097518262906ull,
  6619210701075745654ull, 2404907604760405225ull}},
{{9750073251192973753ull, 13442908085043052825ull,
  12885699394772069972ull, 3006134505950506531ull}},
{{16799277582418605095ull, 16803635106303816031ull,
  11495438225037699561ull, 3757668132438133164ull}},
{{17417077516652710041ull, 3584742913798803163ull,
  16408020927503338034ull, 2348542582773833227ull}},
{{17159660877388499647ull, 13704300679103279762ull,
  15898340140951784638ull, 2935678228467291534ull}},
{{12226204059880848751ull, 7907003812024323895ull,
  10649553139334954990ull, 3669597785584114418ull}},
{{10671069056423673034ull, 660382728175629061ull,
  4088569387313917930ull, 4586997231980143023ull}},
{{8975261169478489598ull, 5024425223537156067ull,
  9472884894712280562ull, 2866873269987589389ull}},
{{6607390443420724094ull, 15503903566276220892ull,
  16452792136817738606ull, 3583591587484486736ull}},
{{8259238054275905117ull, 10156507420990500307ull,
  2119246097312621642ull, 4479489484355608421ull}},
{{2856180774708746746ull, 10959503156546450596ull,
  3630371820034082478ull, 2799680927722255263ull}},
{{3570225968385933433ull, 4476006908828287437ull,
  18373022830324766810ull, 3499601159652819078ull}},
{{9074468478909804695ull, 14818380672890135104ull,
  13742906501051182704ull, 4374501449566023848ull}},
{{5671542799318627935ull, 9261487920556334440ull,
  8589316563156989190ull, 2734063405978764905ull}},
{{7089428499148284918ull, 2353487863840642242ull,
  15348331722373624392ull, 3417579257473456131ull}},
{{18085157660790131956ull, 2941859829800802802ull,
  14573728634539642586ull, 4271974071841820164ull}},
{{15914909556421220376ull, 6450348412052889655ull,
  18331952433442052424ull, 2669983794901137602ull}},
{{15281950927099137566ull, 8062935515066112069ull,
  13691568504947789722ull, 3337479743626422003ull}},
{{5267380603591758246ull, 855297356977864279ull,
  12502774612757349249ull, 4171849679533027504ull}},
{{10209641904885930760ull, 12063775894179634934ull,
  7814234132973343280ull, 2607406049708142190ull}},
{{3538680344252637642ull, 15079719867724543668ull,
  544420629361903292ull, 3259257562135177738ull}},
{{4423350430315797052ull, 402905760946127969ull, 9903897823557154924ull,
  4074071952668972172ull}},
{{14293809065015842918ull, 9475188137446105788ull,
  15413308176577997635ull, 2546294970418107607ull}},
{{17867261331269803647ull, 7232299153380244331ull,
  14654949202295109140ull, 3182868713022634509ull}},
{{17722390645659866655ull, 9040373941725305414ull,
  4483628447586722713ull, 3978585891278293137ull}},
{{6464808135110028755ull, 17179448759646785644ull,
  14331482825810171455ull, 2486616182048933210ull}},
{{8081010168887535944ull, 16862624931131094151ull,
  8690981495407938511ull, 3108270227561166513ull}},
{{5489576692682032026ull, 16466595145486479785ull,
  15475412887687311043ull, 3885337784451458141ull}},
{{14960200478994739776ull, 7985778956715355913ull,
  11977976064018263354ull, 2428336115282161338ull}},
{{4865192543461261008ull, 758851659039419084ull, 5749098043168053385ull,
  3035420144102701673ull}},
{{6081490679326576260ull, 5560250592226661759ull,
  11798058572387454635ull, 3794275180128377091ull}},
{{10718460702220192019ull, 1169313610927969647ull,
  5067943598528465195ull, 2371421987580235682ull}},
{{8786389859347852119ull, 15296700068942125771ull,
  15558301535015357301ull, 2964277484475294602ull}},
{{6371301305757427245ull, 5285817030895493502ull,
  10224504881914420819ull, 3705346855594118253ull}},
{{17817121371380555740ull, 997792635095989486ull,
  8696158560410206964ull, 2315841784746323908ull}},
{{13048029677370918867ull, 1247240793869986858ull,
  10870198200512758705ull, 2894802230932904885ull}},
{{7086665059858872776ull, 6170737010764871477ull,
  18199433769068336285ull, 3618502788666131106ull}},
{{13470017343250978874ull, 12325107281883477250ull,
  13525920174480644548ull, 4523128485832663883ull}},
{{13030446857959249701ull, 16926564088031949089ull,
  6147857099836708890ull, 2826955303645414927ull}},
{{2453000517166898414ull, 11934833073185160554ull,
  3073135356368498209ull, 3533694129556768659ull}},
{{12289622683313398825ull, 1083483286199286980ull,
  17676477250742786474ull, 4417117661945960823ull}},
{{16904386213925650074ull, 5288863072301942266ull,
  17965327309355323402ull, 2760698538716225514ull}},
{{11907110730552286784ull, 15834450877232203641ull,
  13233287099839378444ull, 3450873173395281893ull}},
{{1048830357908194768ull, 1346319522830702936ull,
  2706550819517059344ull, 4313591466744102367ull}},
{{655518973692621730ull, 841449701769189335ull, 8609123289839243946ull,
  2695994666715063979ull}},
{{14654456772397940874ull, 10275184164066262476ull,
  6149718093871667028ull, 3369993333393829974ull}},
{{18318070965497426093ull, 12843980205082828095ull,
  16910519654194359593ull, 4212491666742287467ull}},
{{18366323381076973164ull, 1109958600535685703ull,
  8263231774657780794ull, 2632807291713929667ull}},
{{18346218207918828551ull, 10610820287524382937ull,
  5717353699894838088ull, 3291009114642412084ull}},
{{9097714704616371977ull, 13263525359405478672ull,
  7146692124868547610ull, 4113761393303015105ull}},
{{5686071690385232486ull, 12901389368055812074ull,
  15995897624111312016ull, 2571100870814384440ull}},
{{16330961649836316415ull, 16126736710069765092ull,
  1548127956429588404ull, 3213876088517980551ull}},
{{1966957988585843903ull, 1711676813877654750ull,
  15770218000819149218ull, 4017345110647475688ull}},
{{15064406798148316151ull, 5681484027100922122ull,
  9856386250511968261ull, 2510840694154672305ull}},
{{9607136460830619381ull, 11713541052303540557ull,
  16932168831567348230ull, 3138550867693340381ull}},
{{16620606594465662130ull, 5418554278524649888ull,
  7330152984177021576ull, 3923188584616675477ull}},
{{10387879121541038832ull, 3386596424077906180ull,
  6887188624324332437ull, 2451992865385422173ull}},
{{12984848901926298539ull, 8844931548524770629ull,
  13220671798832803450ull, 3064991081731777716ull}},
{{2396003072125709462ull, 1832792398801187479ull,
  16525839748541004313ull, 3831238852164722145ull}},
{{8415030947719650270ull, 12674710295319211934ull,
  3411120815197045839ull, 2394524282602951341ull}},
{{1295416647794787029ull, 11231701850721627014ull,
  8875587037423695203ull, 2993155353253689176ull}},
{{10842642846598259595ull, 9427941294974645863ull,
  11094483796779619004ull, 3741444191567111470ull}},
{{13694180806764994103ull, 15115835346213929472ull,
  2322366354559873973ull, 2338402619729444669ull}},
{{17117726008456242628ull, 5059736127485248128ull,
  7514643961627230371ull, 2923003274661805836ull}},
{{2950413436860751669ull, 1712984140929172257ull,
  9393304952034037964ull, 3653754093327257295ull}},
{{8299702814503327491ull, 2141230176161465321ull,
  7129945171615159551ull, 4567192616659071619ull}},
{{16716529305133049442ull, 8255797887741997681ull,
  2150372723045780767ull, 2854495385411919762ull}},
{{7060603576134148090ull, 5708061341250109198ull,
  11911337940662001767ull, 3568119231764899702ull}},
{{18049126507022460921ull, 2523390658135248593ull,
  5665800388972726401ull, 4460149039706124628ull}},
{{4363175039247956220ull, 13106334207403000131ull,
  12764497279962729808ull, 2787593149816327892ull}},
{{842282780632557370ull, 16382917759253750164ull,
  15955621599953412260ull, 3484491437270409865ull}},
{{1052853475790696713ull, 2031903125357636089ull,
  6109468944659601614ull, 4355614296588012332ull}},
{{12187248468437655206ull, 15104997508630686267ull,
  13041790127267026816ull, 2722258935367507707ull}},
{{10622374567119681103ull, 434502812078806218ull,
  11690551640656395617ull, 3402823669209384634ull}},
{{4054596172044825571ull, 5154814533525895677ull,
  5389817513965718713ull, 4253529586511730793ull}},
{{4839965616741709934ull, 14750974129522154558ull,
  14897850992297043955ull, 2658455991569831745ull}},
{{15273329057781913225ull, 13827031643475305293ull,
  4787255685089141232ull, 3323069989462289682ull}},
{{5256603266945227819ull, 17283789554344131617ull,
  15207441643216202348ull, 4153837486827862102ull}},
{{14814592087909237147ull, 1578996434610306452ull,
  4892965008582738564ull, 2596148429267413814ull}},
{{71496036176994818ull, 1973745543262883066ull, 15339578297583199013ull,
  3245185536584267267ull}},
{{9312742082076019330ull, 7078867947505991736ull,
  14562786853551610862ull, 4056481920730334084ull}},
{{5820463801297512082ull, 18259350522473408547ull,
  18325113820324532596ull, 2535301200456458802ull}},
{{2663893733194502198ull, 4377444079382209068ull,
  13683020238550889938ull, 3169126500570573503ull}},
{{3329867166493127747ull, 14695177136082537143ull,
  12492089279761224518ull, 3961408125713216879ull}},
{{8998696006699286698ull, 4572799691624197810ull,
  14725084827491847180ull, 2475880078570760549ull}},
{{2024997971519332565ull, 5715999614530247263ull,
  4571297979082645263ull, 3094850098213450687ull}},
{{16366305519681329418ull, 2533313499735421174ull,
  1102436455425918675ull, 3868562622766813359ull}},
{{5617254931373442982ull, 17724222001830495898ull,
  7606551812282281027ull, 2417851639229258349ull}},
{{16244940701071579536ull, 17543591483860731968ull,
  14119875783780239188ull, 3022314549036572936ull}},
{{1859431802629922803ull, 3482745281116363345ull,
  17649844729725298986ull, 3777893186295716170ull}},
{{12691359922712171512ull, 6788401819125114994ull,
  15642838974505699770ull, 2361183241434822606ull}},
{{6640827866535438582ull, 17708874310761169551ull,
  10330176681277348904ull, 2951479051793528258ull}},
{{3689348814741910324ull, 3689348814741910323ull,
  3689348814741910323ull, 3689348814741910323ull}},
{{0ull, 0ull, 0ull, 4611686018427387904ull}},
{{0ull, 0ull, 0ull, 2882303761517117440ull}},
{{0ull, 0ull, 0ull, 3602879701896396800ull}},
{{0ull, 0ull, 0ull, 4503599627370496000ull}},
{{0ull, 0ull, 0ull, 2814749767106560000ull}},
{{0ull, 0ull, 0ull, 3518437208883200000ull}},
{{0ull, 0ull, 0ull, 4398046511104000000ull}},
{{0ull, 0ull, 0ull, 2748779069440000000ull}},
{{0ull, 0ull, 0ull, 3435973836800000000ull}},
{{0ull, 0ull, 0ull, 4294967296000000000ull}},
{{0ull, 0ull, 0ull, 2684354560000000000ull}},
{{0ull, 0ull, 0ull, 3355443200000000000ull}},
{{0ull, 0ull, 0ull, 4194304000000000000ull}},
{{0ull, 0ull, 0ull, 2621440000000000000ull}},
{{0ull, 0ull, 0ull, 3276800000000000000ull}},
{{0ull, 0ull, 0ull, 4096000000000000000ull}},
{{0ull, 0ull, 0ull, 2560000000000000000ull}},
{{0ull, 0ull, 0ull, 3200000000000000000ull}},
{{0ull, 0ull, 0ull, 4000000000000000000ull}},
{{0ull, 0ull, 0ull, 2500000000000000000ull}},
{{0ull, 0ull, 0ull, 3125000000000000000ull}},
{{0ull, 0ull, 0ull, 3906250000000000000ull}},
{{0ull, 0ull, 0ull, 2441406250000000000ull}},
{{0ull, 0ull, 0ull, 3051757812500000000ull}},
{{0ull, 0ull, 0ull, 3814697265625000000ull}},
{{0ull, 0ull, 0ull, 2384185791015625000ull}},
{{0ull, 0ull, 0ull, 2980232238769531250ull}},
{{0ull, 0ull, 9223372036854775808ull, 3725290298461914062ull}},
{{0ull, 0ull, 1152921504606846976ull, 2328306436538696289ull}},
{{0ull, 0ull, 6052837899185946624ull, 2910383045673370361ull}},
{{0ull, 0ull, 12177733392409821184ull, 3637978807091712951ull}},
{{0ull, 0ull, 10610480722084888576ull, 4547473508864641189ull}},
{{0ull, 0ull, 8937393460516749312ull, 2842170943040400743ull}},
{{0ull, 0ull, 6560055807218548736ull, 3552713678800500929ull}},
{{0ull, 0ull, 12811755777450573824ull, 4440892098500626161ull}},
{{0ull, 0ull, 1089818333265526784ull, 2775557561562891351ull}},
{{0ull, 0ull, 15197330971864072192ull, 3469446951953614188ull}},
{{0ull, 0ull, 549919641120538624ull, 4336808689942017736ull}},
{{0ull, 0ull, 343699775700336640ull, 2710505431213761085ull}},
{{0ull, 0ull, 5041310738052808704ull, 3388131789017201356ull}},
{{0ull, 0ull, 6301638422566010880ull, 4235164736271501695ull}},
{{0ull, 0ull, 10856053041744838656ull, 2646977960169688559ull}},
{{0ull, 0ull, 8958380283753660416ull, 3308722450212110699ull}},
{{0ull, 0ull, 6586289336264687616ull, 4135903062765138374ull}},
{{0ull, 0ull, 17951488890447593472ull, 2584939414228211483ull}},
{{0ull, 0ull, 17827675094632103936ull, 3231174267785264354ull}},
{{0ull, 0ull, 13061221831435354112ull, 4038967834731580443ull}},
{{0ull, 0ull, 5857420635433402368ull, 2524354896707237777ull}},
{{0ull, 0ull, 11933461812719140864ull, 3155443620884047221ull}},
{{0ull, 0ull, 1081769210616762368ull, 3944304526105059027ull}},
{{0ull, 0ull, 16817006821131334144ull, 2465190328815661891ull}},
{{0ull, 0ull, 16409572507986779776ull, 3081487911019577364ull}},
{{0ull, 0ull, 2065221561273923104ull, 3851859888774471706ull}},
{{0ull, 0ull, 5902449494223589844ull, 2407412430484044816ull}},
{{0ull, 0ull, 7378061867779487305ull, 3009265538105056020ull}},
{{0ull, 4611686018427387904ull, 9222577334724359131ull,
  3761581922631320025ull}},
{{0ull, 576460752303423488ull, 17293325880271194217ull,
  2350988701644575015ull}},
{{0ull, 5332261958806667264ull, 17004971331911604867ull,
  2938735877055718769ull}},
{{0ull, 2053641430080946176ull, 7421156109607342372ull,
  3673419846319648462ull}},
{{0ull, 2567051787601182720ull, 53073100154402157ull,
  4591774807899560578ull}},
{{0ull, 3910250376464433152ull, 4644856706023889252ull,
  2869859254937225361ull}},
{{0ull, 4887812970580541440ull, 10417756900957249469ull,
  3587324068671531701ull}},
{{0ull, 10721452231653064704ull, 17633882144623949740ull,
  4484155085839414626ull}},
{{0ull, 15924279681637941248ull, 15632862358817356491ull,
  2802596928649634141ull}},
{{0ull, 15293663583620038656ull, 5706019893239531902ull,
  3503246160812042677ull}},
{{0ull, 9893707442670272512ull, 11744210884976802782ull,
  4379057701015053346ull}},
{{0ull, 1571881133241532416ull, 11951817821537889643ull,
  2736911063134408341ull}},
{{0ull, 15799909471834079232ull, 1104714221640198341ull,
  3421138828918010427ull}},
{{0ull, 5914828784510435328ull, 15215950832332411639ull,
  4276423536147513033ull}},
{{0ull, 10614297017960103936ull, 2592440242566675418ull,
  2672764710092195646ull}},
{{0ull, 4044499235595354112ull, 12463922340063120081ull,
  3340955887615244557ull}},
{{0ull, 9667310062921580544ull, 1744844869796736389ull,
  4176194859519055697ull}},
{{0ull, 8347911798539681792ull, 12619743089691430003ull,
  2610121787199409810ull}},
{{0ull, 5823203729747214336ull, 6551306825259511696ull,
  3262652233999262263ull}},
{{0ull, 7279004662184017920ull, 3577447513147001716ull,
  4078315292499077829ull}},
{{0ull, 13772749950719787008ull, 4541747704930570024ull,
  2548947057811923643ull}},
{{0ull, 17215937438399733760ull, 1065498612735824626ull,
  3186183822264904554ull}},
{{0ull, 12296549761144891392ull, 10555245302774556591ull,
  3982729777831130692ull}},
{{0ull, 14602872628356638976ull, 15820400351088873677ull,
  2489206111144456682ull}},
{{0ull, 4418532730163635008ull, 10552128402006316289ull,
  3111507638930570853ull}},
{{0ull, 10134851931131931664ull, 17801846520935283265ull,
  3889384548663213566ull}},
{{0ull, 17863497503025927050ull, 6514468057157164136ull,
  2430865342914508479ull}},
{{9223372036854775808ull, 3882627805072857196ull,
  3531399053019067267ull, 3038581678643135599ull}},
{{11529215046068469760ull, 241598737913683591ull,
  18249306871555997796ull, 3798227098303919498ull}},
{{14123288431433875456ull, 9374371248050828052ull,
  16017502813149886526ull, 2373891936439949686ull}},
{{17654110539292344320ull, 2494592023208759257ull,
  10798506479582582350ull, 2967364920549937108ull}},
{{8232580118833266688ull, 12341612065865724880ull,
  13498133099478227937ull, 3709206150687421385ull}},
{{5145362574270791680ull, 795978513524996194ull, 1518804159532810605ull,
  2318253844179638366ull}},
{{15655075254693265408ull, 5606659160333633146ull,
  11121877236270789064ull, 2897817305224547957ull}},
{{10345472031511805952ull, 7008323950417041433ull, 67288490056322618ull,
  3622271631530684947ull}},
{{17543526057817145344ull, 17983776974876077599ull,
  13919168667852566984ull, 4527839539413356183ull}},
{{17882232813776797696ull, 11239860609297548499ull,
  15617009445048936221ull, 2829899712133347614ull}},
{{17741104998793609216ull, 214767706339771912ull,
  10297889769456394469ull, 3537374640166684518ull}},
{{3729637174782459904ull, 4880145651352102795ull,
  3648990174965717278ull, 4421718300208355648ull}},
{{25180225025343488ull, 16885149087377227959ull, 2280618859353573298ull,
  2763573937630222280ull}},
{{13866533336563843072ull, 11883064322366759140ull,
  2850773574191966623ull, 3454467422037777850ull}},
{{17333166670704803840ull, 10242144384531061021ull,
  12786839004594734087ull, 4318084277547222312ull}},
{{13139072178404196352ull, 13318869267972994994ull,
  7991774377871708804ull, 2698802673467013945ull}},
{{7200468186150469632ull, 16648586584966243743ull,
  14601403990767023909ull, 3373503341833767431ull}},
{{4388899214260699136ull, 6975675175925640967ull,
  13640068970031391983ull, 4216879177292209289ull}},
{{9660591036554018816ull, 11277326012594607460ull,
  1607514078628538133ull, 2635549485807630806ull}},
{{12075738795692523520ull, 261599460461095613ull,
  11232764635140448475ull, 3294436857259538507ull}},
{{1259615439333490688ull, 14162057380858533229ull,
  9429269775498172689ull, 4118046071574423134ull}},
{{3093102658797125632ull, 1933756835395501412ull,
  1281607591258970027ull, 2573778794734014459ull}},
{{3866378323496407040ull, 16252254099526540477ull,
  15437067544355876245ull, 3217223493417518073ull}},
{{9444658922797896704ull, 6480259569126011884ull,
  5461276375162681595ull, 4021529366771897592ull}},
{{15126283863603461248ull, 1744319221490063475ull,
  3413297734476675997ull, 2513455854232435995ull}},
{{14296168811076938656ull, 6792085045289967248ull,
  18101680223378008708ull, 3141819817790544993ull}},
{{17870211013846173320ull, 8490106306612459060ull,
  8792042223940347173ull, 3927274772238181242ull}},
{{1945509846799082517ull, 7612159450846480865ull,
  10106712408390104887ull, 2454546732648863276ull}},
{{7043573326926241051ull, 4903513295130713177ull,
  12633390510487631109ull, 3068183415811079095ull}},
{{13416152677085189217ull, 10741077637340779375ull,
  11180052119682150982ull, 3835229269763848869ull}},
{{15302624450819325117ull, 2101487504910599205ull,
  9293375584015038316ull, 2397018293602405543ull}},
{{5293222508241992684ull, 2626859381138249007ull,
  7005033461591409991ull, 2996272867003006929ull}},
{{2004842116875102951ull, 17118632281704974971ull,
  13367977845416650392ull, 3745341083753758661ull}},
{{17393927387542797009ull, 10699145176065609356ull,
  10660829162599100447ull, 2340838177346099163ull}},
{{3295665160718944645ull, 8762245451654623792ull,
  8714350434821487655ull, 2926047721682623954ull}},
{{4119581450898680806ull, 6341120796140891836ull,
  1669566006672083761ull, 3657559652103279943ull}},
{{5149476813623351007ull, 12538087013603502699ull,
  15922015563622268413ull, 4571949565129099928ull}},
{{912579999300900428ull, 10142147392715883139ull,
  9951259727263917758ull, 2857468478205687455ull}},
{{14975783054408289246ull, 3454312204040078115ull,
  7827388640652509294ull, 3571835597757109319ull}},
{{14108042799582973654ull, 13541262291904873452ull,
  5172549782388248713ull, 4464794497196386649ull}},
{{18040898786594134342ull, 1545759904799464051ull,
  14762058660061125206ull, 2790496560747741655ull}},
{{17939437464815280023ull, 11155571917854105872ull,
  13840887306649018603ull, 3488120700934677069ull}},
{{3977552757309548413ull, 9332778878890244437ull,
  3466051078029109542ull, 4360150876168346337ull}},
{{4791813482532161710ull, 1221300780879014869ull,
  13695496969836663224ull, 2725094297605216460ull}},
{{10601452871592590042ull, 1526625976098768586ull,
  17119371212295829030ull, 3406367872006520575ull}},
{{4028444052635961744ull, 11131654506978236541ull,
  16787527996942398383ull, 4257959840008150719ull}},
{{4823620542111170042ull, 13874813094502479694ull,
  17409734025730080845ull, 2661224900005094199ull}},
{{15252897714493738360ull, 3508458312845935905ull,
  17150481513735213153ull, 3326531125006367749ull}},
{{5231064087835009238ull, 8997258909484807786ull,
  7603043836886852729ull, 4158163906257959687ull}},
{{7881101073324268678ull, 17152501864496474626ull,
  11669431425695364811ull, 2598852441411224804ull}},
{{628004304800560040ull, 16828941312193205379ull,
  14586789282119206014ull, 3248565551764031005ull}},
{{14620063436282863761ull, 11812804603386730915ull,
  4398428547366843806ull, 4060706939705038757ull}},
{{6831696638463095899ull, 2771316858689318918ull,
  5054860851317971331ull, 2537941837315649223ull}},
{{17762992834933645682ull, 17299204128643812359ull,
  1706890045720076259ull, 3172427296644561529ull}},
{{17592055025239669198ull, 17012319142377377545ull,
  6745298575577483228ull, 3965534120805701911ull}},
{{4077505363133711393ull, 1409327427131085158ull,
  11133340637377008874ull, 2478458825503563694ull}},
{{14320253740771915049ull, 10985031320768632255ull,
  4693303759866485284ull, 3098073531879454618ull}},
{{13288631157537505907ull, 13731289150960790319ull,
  15090001736687882413ull, 3872591914849318272ull}},
{{15222923501102023048ull, 10887898728564187901ull,
  9431251085429926508ull, 2420369946780823920ull}},
{{5193596321095365098ull, 13609873410705234877ull,
  11789063856787408135ull, 3025462433476029900ull}},
{{11103681419796594277ull, 12400655744954155692ull,
  14736329820984260169ull, 3781828041845037375ull}},
{{16163172924227647231ull, 832880812955265451ull,
  16127735165756244462ull, 2363642526153148359ull}},
{{15592280136857171135ull, 10264473053048857622ull,
  15547982938767917673ull, 2954553157691435449ull}},
{{10266978134216688110ull, 17442277334738459932ull,
  5599920618177733379ull, 3693191447114294312ull}},
{{15640233370740205877ull, 8595580324997843505ull,
  3499950386361083362ull, 2308244654446433945ull}},
{{5715233658143093634ull, 1521103369392528574ull,
  8986624001378742107ull, 2885305818058042431ull}},
{{16367414109533642851ull, 15736437267022824429ull,
  6621593983296039729ull, 3606632272572553039ull}},
{{6624209581634889851ull, 5835488528496366825ull,
  3665306460692661758ull, 4508290340715691299ull}},
{{15669346034590275917ull, 17482238385592392977ull,
  18431717602428771262ull, 2817681462947307061ull}},
{{5751624487955681184ull, 12629425945135715414ull,
  9204588947753800366ull, 3522101828684133827ull}},
{{16412902646799377288ull, 6563410394564868459ull,
  6894050166264862554ull, 4402627285855167284ull}},
{{7952221145035916853ull, 8713817515030430691ull,
  13532153390770314904ull, 2751642053659479552ull}},
{{5328590412867508163ull, 10892271893788038364ull,
  16915191738462893630ull, 3439552567074349440ull}},
{{6660738016084385203ull, 4391967830380272147ull,
  2697245599369065422ull, 4299440708842936801ull}},
{{1857118250839046800ull, 16580037949269833804ull,
  13214993545674135648ull, 2687150443026835500ull}},
{{2321397813548808500ull, 2278303362877740639ull,
  16518741932092669561ull, 3358938053783544375ull}},
{{16736805322218174337ull, 7459565222024563702ull,
  16036741396688449047ull, 4198672567229430469ull}},
{{5848817307958971057ull, 11579757291406434170ull,
  12328806382143974606ull, 2624170354518394043ull}},
{{16534393671803489629ull, 5251324577403266904ull,
  10799321959252580354ull, 3280212943147992554ull}},
{{2221248016044810420ull, 15787527758608859439ull,
  4275780412210949634ull, 4100266178934990693ull}},
{{8305809037669088369ull, 14478890867557925053ull,
  4978205766845537473ull, 2562666361834369183ull}},
{{14993947315513748365ull, 4263555529165242604ull,
  1611071190129533938ull, 3203332952292961479ull}},
{{295690070682633840ull, 14552816448311329064ull,
  15848897042944081134ull, 4004166190366201848ull}},
{{184806294176646150ull, 4483824261767192761ull, 9905560651840050709ull,
  2502603868978876155ull}},
{{4842693886148195591ull, 10216466345636378855ull,
  7770264796372675482ull, 3128254836223595194ull}},
{{1441681339257856585ull, 3547210895190697761ull, 489458958611068545ull,
  3910318545279493993ull}},
{{12430265883104630126ull, 13746221855562655860ull,
  11835126895200387600ull, 2443949090799683745ull}},
{{15537832353880787657ull, 17182777319453319825ull,
  958850563718320788ull, 3054936363499604682ull}},
{{5587232387068820859ull, 3031727575607098166ull,
  10421935241502676794ull, 3818670454374505852ull}},
{{17327078297200176749ull, 6506515753181824257ull,
  15737081562793948804ull, 2386669033984066157ull}},
{{7823789816218057224ull, 8133144691477280322ull,
  5836293898210272293ull, 2983336292480082697ull}},
{{556365233417795722ull, 14778116882773988307ull,
  11907053391190228270ull, 3729170365600103371ull}},
{{16488629335381979991ull, 4624637033306354787ull,
  5136065360280198717ull, 2330731478500064607ull}},
{{15999100650800087084ull, 10392482310060331388ull,
  1808395681922860492ull, 2913414348125080759ull}},
{{1552131739790557239ull, 12990602887575414236ull,
  16095552657685739327ull, 3641767935156350948ull}},
{{1940164674738196549ull, 11626567591041879891ull,
  1672696748397622543ull, 4552209918945438686ull}},
{{17353503986207230507ull, 14184133772042256787ull,
  14880493523030677801ull, 2845131199340899178ull}},
{{17080193964331650230ull, 3895109159770657272ull,
  9377244866933571444ull, 3556413999176123973ull}},
{{2903498381705011171ull, 4868886449713321591ull,
  16333242102094352209ull, 4445517498970154966ull}},
{{8732215516206713838ull, 14572269077139295754ull,
  5596590295381582226ull, 2778448436856346854ull}},
{{1691897358403616490ull, 8991964309569343885ull,
  16219109906081753591ull, 3473060546070433567ull}},
{{6726557716431908516ull, 6628269368534291952ull,
  15662201364174804085ull, 4341325682588041959ull}},
{{4204098572769942823ull, 6448511364547626422ull,
  16706404880250334409ull, 2713328551617526224ull}},
{{14478495252817204336ull, 12672325224111920931ull,
  2436262026603366395ull, 3391660689521907781ull}},
{{13486433047594117516ull, 11228720511712513260ull,
  7657013551681595898ull, 4239575861902384726ull}},
{{17652392691601099256ull, 11629636338247708691ull,
  173947451373609532ull, 2649734913688990454ull}},
{{17453804846073986166ull, 14537045422809635864ull,
  9440806351071787723ull, 3312168642111238067ull}},
{{3370511983882931091ull, 13559620760084656927ull,
  7189321920412346750ull, 4140210802639047584ull}},
{{9024099017567913788ull, 3863076956625522675ull,
  4493326200257716719ull, 2587631751649404740ull}},
{{6668437753532504331ull, 217160177354515440ull, 5616657750322145899ull,
  3234539689561755925ull}},
{{8335547191915630413ull, 14106508276975308012ull,
  11632508206330070277ull, 4043174611952194906ull}},
{{14433089031802044817ull, 11122410682323261459ull,
  11882003647383681827ull, 2526984132470121816ull}},
{{13429675271325168117ull, 9291327334476688920ull,
  14852504559229602284ull, 3158730165587652270ull}},
{{16787094089156460146ull, 11614159168095861150ull,
  9342258662182227047ull, 3948412706984565338ull}},
{{5880247787295399687ull, 14176378507700995075ull,
  10450597682291279808ull, 2467757941865353336ull}},
{{2738623715691861705ull, 17720473134626243844ull,
  13063247102864099760ull, 3084697427331691670ull}},
{{3423279644614827131ull, 3703847344573253189ull,
  7105686841725348893ull, 3855871784164614588ull}},
{{4445392787097960909ull, 4620747599571977195ull,
  13664426312933118866ull, 2409919865102884117ull}},
{{945054965445063232ull, 14999306536319747302ull,
  3245474835884234870ull, 3012399831378605147ull}},
{{10404690743661104848ull, 9525761133544908319ull,
  17891901600137457300ull, 3765499789223256433ull}},
{{13420460742429272386ull, 15176972745320343507ull,
  4264909472444828956ull, 2353437368264535271ull}},
{{12163889909609202579ull, 524471857940877768ull, 719450822128648292ull,
  2941796710330669089ull}},
{{15204862387011503223ull, 655589822426097210ull,
  5510999546088198269ull, 3677245887913336361ull}},
{{9782705946909603221ull, 5431173296460009417ull,
  11500435451037635740ull, 4596557359891670451ull}},
{{17643406262886971773ull, 12617855347142281693ull,
  4881929147684828385ull, 2872848349932294032ull}},
{{8219199773326551005ull, 1937261128645688405ull,
  6102411434606035482ull, 3591060437415367540ull}},
{{14885685735085576660ull, 11644948447661886314ull,
  7628014293257544352ull, 4488825546769209425ull}},
{{13915239602855873316ull, 7278092779788678946ull,
  16296723979354434980ull, 2805515966730755890ull}},
{{8170677466715065837ull, 9097615974735848683ull,
  11147532937338267917ull, 3506894958413444863ull}},
{{5601660814966444393ull, 15983705986847198758ull,
  9322730153245446992ull, 4383618698016806079ull}},
{{17336096064636191458ull, 9989816241779499223ull,
  12744235373419486226ull, 2739761686260503799ull}},
{{17058434062367851418ull, 3263898265369598221ull,
  11318608198346969879ull, 3424702107825629749ull}},
{{7487984522677650560ull, 17914930886994161489ull,
  313202192651548636ull, 4280877634782037187ull}},
{{16209205372742001360ull, 1973459767516575122ull,
  16336652434903075562ull, 2675548521738773241ull}},
{{11038134679072725892ull, 11690196746250494711ull,
  6585757488346680740ull, 3344435652173466552ull}},
{{9185982330413519461ull, 14612745932813118389ull,
  8232196860433350925ull, 4180544565216833190ull}},
{{8047081965722143615ull, 11438809217221892945ull,
  533437019343456424ull, 2612840353260520744ull}},
{{14670538475580067423ull, 14298511521527366181ull,
  666796274179320530ull, 3266050441575650930ull}},
{{4503115039192920567ull, 8649767365054431919ull,
  10056867379578926471ull, 4082563051969563662ull}},
{{9731975927136657210ull, 12323633630800101805ull,
  1673856093809441140ull, 2551601907480977289ull}},
{{16776655927348209417ull, 15404542038500127256ull,
  6704006135689189329ull, 3189502384351221611ull}},
{{2524075835475710155ull, 5420619492842995359ull,
  3768321651184098758ull, 3986877980439027014ull}},
{{8495076424813400703ull, 17222945238309035811ull,
  16190259087272225435ull, 2491798737774391883ull}},
{{6007159512589362975ull, 16916995529458906860ull,
  15626137840662893890ull, 3114748422217989854ull}},
{{7508949390736703718ull, 11922872374968857767ull,
  10309300263973841555ull, 3893435527772487318ull}},
{{11610622396851521680ull, 5145952225141842152ull,
  1831626646556263068ull, 2433397204857804574ull}},
{{14513277996064402100ull, 6432440281427302690ull,
  11512905345050104643ull, 3041746506072255717ull}},
{{8918225458225726817ull, 3428864333356740459ull, 556073626030467092ull,
  3802183132590319647ull}},
{{3268047902177385309ull, 11366412245202738595ull,
  7265075043910123788ull, 2376364457868949779ull}},
{{17920117933003895348ull, 14208015306503423243ull,
  4469657786460266831ull, 2970455572336187224ull}},
{{17788461397827481281ull, 13148333114701891150ull,
  5587072233075333539ull, 3713069465420234030ull}},
{{6506102355214787897ull, 5911865187474988017ull,
  17326978200954247174ull, 2320668415887646268ull}},
{{12744313962445872775ull, 16613203521198510829ull,
  3211978677483257351ull, 2900835519859557836ull}},
{{2095334397775177256ull, 16154818383070750633ull,
  4014973346854071689ull, 3626044399824447295ull}},
{{7230854015646359474ull, 6358464923556274579ull, 407030665140201708ull,
  4532555499780559119ull}},
{{2213440750565280719ull, 13197412614077447420ull,
  7171923193353707923ull, 2832847187362849449ull}},
{{2766800938206600899ull, 11885079749169421371ull,
  13576590010119522808ull, 3541058984203561811ull}},
{{17293559228040414836ull, 14856349686461776713ull,
  12359051494222015606ull, 4426323730254452264ull}},
{{3890945489884177417ull, 4673532535611222542ull,
  7724407183888759754ull, 2766452331409032665ull}},
{{14087053899209997579ull, 15065287706368803985ull,
  14267194998288337596ull, 3458065414261290831ull}},
{{3773759318730333261ull, 384865559251453366ull,
  13222307729433034092ull, 4322581767826613539ull}},
{{16193657629488622000ull, 9463913011386934161ull,
  5958099321681952355ull, 2701613604891633462ull}},
{{6407013981578613788ull, 7218205245806279798ull,
  16670996188957216252ull, 3377017006114541827ull}},
{{17232139513828043043ull, 9022756557257849747ull,
  16227059217769132411ull, 4221271257643177284ull}},
{{8464244186928832950ull, 3333379839072462140ull, 918539974250931949ull,
  2638294536026985803ull}},
{{10580305233661041188ull, 8778410817267965579ull,
  14983233023095828648ull, 3297868170033732253ull}},
{{8613695523648913581ull, 10973013521584956974ull,
  4893983223587622098ull, 4122335212542165317ull}},
{{771873683853183084ull, 11469819469417986013ull,
  5364582523955957763ull, 2576459507838853323ull}},
{{5576528123243866759ull, 9725588318345094612ull,
  2094042136517559300ull, 3220574384798566654ull}},
{{6970660154054833449ull, 12156985397931368265ull,
  11840924707501724933ull, 4025717980998208317ull}},
{{15885877642352740666ull, 9903958882920799117ull,
  9706420951402272035ull, 2516073738123880198ull}},
{{6022288997658762120ull, 7768262585223610993ull,
  2909654152398064236ull, 3145092172654850248ull}},
{{12139547265500840554ull, 9710328231529513741ull,
  3637067690497580295ull, 3931365215818562810ull}},
{{9893060050151719298ull, 12986484172347027944ull,
  6884853324988375588ull, 2457103259886601756ull}},
{{12366325062689649123ull, 16233105215433784930ull,
  8606066656235469485ull, 3071379074858252195ull}},
{{6234534291507285595ull, 6456323464010067451ull,
  6145897301866948953ull, 3839223843572815244ull}},
{{1590740922978359545ull, 15564417211074761917ull,
  13064557850521618903ull, 2399514902233009527ull}},
{{6600112172150337335ull, 14843835495416064492ull,
  11719011294724635725ull, 2999393627791261909ull}},
{{8250140215187921669ull, 4719736313987916903ull, 813706063123630945ull,
  3749242034739077387ull}},
{{12073866662133532899ull, 14479050242310917824ull,
  16649467353948127004ull, 2343276271711923366ull}},
{{15092333327666916124ull, 18098812802888647280ull,
  11588462155580382947ull, 2929095339639904208ull}},
{{418672585874093539ull, 18011829985183421197ull,
  14485577694475478684ull, 3661369174549880260ull}},
{{5135026750770004827ull, 4068043407769724880ull,
  18106972118094348356ull, 4576711468187350325ull}},
{{3209391719231253017ull, 11765899166710853858ull,
  13622700583022661674ull, 2860444667617093953ull}},
{{13235111685893842080ull, 5484001921533791514ull,
  3193317673496163381ull, 3575555834521367442ull}},
{{7320517570512526791ull, 11466688420344627297ull,
  13215019128724980034ull, 4469444793151709302ull}},
{{16104538527638799005ull, 11778366281142779964ull,
  3647700937025724617ull, 2793402995719818314ull}},
{{1683929085838947140ull, 887899796146311244ull,
  13782998208136931580ull, 3491753744649772892ull}},
{{2104911357298683925ull, 1109874745182889055ull,
  17228747760171164475ull, 4364692180812216115ull}},
{{8233098625952759309ull, 16834572780235163323ull,
  8462124340893283844ull, 2727932613007635072ull}},
{{5679687264013561232ull, 2596471901584402538ull,
  10577655426116604806ull, 3409915766259543840ull}},
{{16322981116871727348ull, 12468961913835278980ull,
  13222069282645756007ull, 4262394707824429800ull}},
{{978491161190053785ull, 14710630223788131219ull,
  8263793301653597504ull, 2663996692390268625ull}},
{{15058172006769730943ull, 18388287779735164023ull,
  14941427645494384784ull, 3329995865487835781ull}},
{{14211028990034775774ull, 4538615650959403413ull,
  4841726501585817269ull, 4162494831859794727ull}},
{{11187736127985428811ull, 5142477791063321085ull,
  9943608091132217649ull, 2601559269912371704ull}},
{{149612104699622302ull, 11039783257256539261ull,
  12429510113915272061ull, 3251949087390464630ull}},
{{4798701149301915781ull, 18411415089998061980ull,
  6313515605539314268ull, 4064936359238080788ull}},
{{12222560255168473171ull, 2283762394394012929ull,
  13169319290316847226ull, 2540585224523800492ull}},
{{1443142263678427752ull, 12078075029847291970ull,
  16461649112896059032ull, 3175731530654750615ull}},
{{11027299866452810498ull, 15097593787309114962ull,
  15965375372692685886ull, 3969664413318438269ull}},
{{11503748434960394466ull, 4824310098640808947ull,
  12284202617146622631ull, 2481040258324023918ull}},
{{9767999525273105178ull, 1418701604873623280ull,
  6131881234578502481ull, 3101300322905029898ull}},
{{12209999406591381472ull, 6385063024519417004ull,
  16888223580077903909ull, 3876625403631287372ull}},
{{16854621665974389228ull, 6296507399538329579ull,
  1331767700693914135ull, 2422890877269554608ull}},
{{16456591064040598631ull, 3258948230995524070ull,
  1664709625867392669ull, 3028613596586943260ull}},
{{11347366793195972481ull, 8685371307171792992ull,
  2080887032334240836ull, 3785766995733679075ull}},
{{7092104245747482801ull, 14651729103837146428ull,
  17441455459704758186ull, 2366104372333549421ull}},
{{8865130307184353501ull, 9091289342941657227ull,
  7966761269348784021ull, 2957630465416936777ull}},
{{6469726865553053972ull, 15975797697104459438ull,
  14570137605113367930ull, 3697038081771170971ull}},
{{17878637346252822445ull, 14596559579117675052ull,
  6800492993982161004ull, 2310648801106981857ull}},
{{3901552609106476440ull, 18245699473897093816ull,
  13112302260905089159ull, 2888311001383727321ull}},
{{4876940761383095549ull, 18195438323943979366ull,
  2555319770849197737ull, 3610388751729659152ull}},
{{15319547988583645245ull, 8909239849647810495ull,
  3194149713561497172ull, 4512985939662073940ull}},
{{16492246520505860134ull, 14791646942884657367ull,
  11219715607830711540ull, 2820616212288796212ull}},
{{16003622132204937264ull, 42814604896270093ull,
  14024644509788389426ull, 3525770265360995265ull}},
{{6169469609974007867ull, 9276890292975113425ull,
  3695747581953323070ull, 4407212831701244082ull}},
{{15385133552302224677ull, 1186370414682057986ull,
  6921528257148214823ull, 2754508019813277551ull}},
{{10008044903523005038ull, 15318021073634736195ull,
  4040224303007880624ull, 3443135024766596939ull}},
{{7898370110976368394ull, 700782268333868628ull, 438594360332462877ull,
  4303918780958246174ull}},
{{14159853356215006054ull, 2743831926922361844ull,
  14109179530489953010ull, 2689949238098903858ull}},
{{17699816695268757568ull, 12653161945507728113ull,
  8413102376257665454ull, 3362436547623629823ull}},
{{8289712813803783248ull, 6593080395029884334ull,
  5904691951894693914ull, 4203045684529537279ull}},
{{569384490199976626ull, 8732361265321065613ull,
  10607961497575265552ull, 2626903552830960799ull}},
{{5323416631177358686ull, 10915451581651332016ull,
  8648265853541694036ull, 3283629441038700999ull}},
{{6654270788971698358ull, 13644314477064165020ull,
  6198646298499729641ull, 4104536801298376249ull}},
{{13382291279962087282ull, 1610167520524021281ull,
  15403368982630800786ull, 2565335500811485155ull}},
{{2892806044670445390ull, 11236081437509802410ull,
  14642525209861113078ull, 3206669376014356444ull}}
};

// **********************************************************************

#if __ENABLE_BINARY80__

static const UINT128 breakpoints_binary80[] =
  { {{6337302757928054309ull, 494016656451265ull}},
{{5069842206342443447ull, 395213325161012ull}},
{{15123920209299685727ull, 316170660128809ull}},
{{13130225890653766194ull, 505873056206095ull}},
{{10504180712523012955ull, 404698444964876ull}},
{{4713995755276500041ull, 323758755971901ull}},
{{163695578958579419ull, 518014009555042ull}},
{{11199002907392594505ull, 414411207644033ull}},
{{16337899955397896250ull, 331528966115226ull}},
{{315198225443261738ull, 530446345784363ull}},
{{7630856209838430037ull, 424357076627490ull}},
{{6104684967870744030ull, 339485661301992ull}},
{{13456844763335100771ull, 543177058083187ull}},
{{3386778181184259970ull, 434541646466550ull}},
{{2709422544947407976ull, 347633317173240ull}},
{{4335076071915852762ull, 556213307477184ull}},
{{7157409672274592533ull, 444970645981747ull}},
{{16793974182045404996ull, 355976516785397ull}},
{{6056481716152503350ull, 284781213428318ull}},
{{6001021931102095037ull, 455649941485309ull}},
{{8490166359623586353ull, 364519953188247ull}},
{{17860179531924600052ull, 291615962550597ull}},
{{13818891992111718790ull, 466585540080956ull}},
{{7365764778947464709ull, 373268432064765ull}},
{{5892611823157971767ull, 298614745651812ull}},
{{13117527731794665151ull, 477783593042899ull}},
{{14183371000177642444ull, 382226874434319ull}},
{{15036045614884024278ull, 305781499547455ull}},
{{5610928910104887229ull, 489250399275929ull}},
{{8178091942825820106ull, 391400319420743ull}},
{{13921171183744476731ull, 313120255536594ull}},
{{11205827449765431801ull, 500992408858551ull}},
{{5275313145070435117ull, 400793927086841ull}},
{{530901701314437771ull, 320635141669473ull}},
{{15606837981070741726ull, 513016226671156ull}},
{{8796121570114683058ull, 410412981336925ull}},
{{7036897256091746446ull, 328330385069540ull}},
{{11259035609746794314ull, 525328616111264ull}},
{{12696577302539345774ull, 420262892889011ull}},
{{6467913027289566296ull, 336210314311209ull}},
{{17727358473147126720ull, 537936502897934ull}},
{{17871235593259611699ull, 430349202318347ull}},
{{6918290845123868713ull, 344279361854678ull}},
{{7379916537456279618ull, 550846978967485ull}},
{{5903933229965023694ull, 440677583173988ull}},
{{12101844213455839602ull, 352542066539190ull}},
{{9681475370764671681ull, 282033653231352ull}},
{{732965334255833398ull, 451253845170164ull}},
{{4275721082146577041ull, 361003076136131ull}},
{{18177972124684902926ull, 288802460908904ull}},
{{18016708955270113712ull, 462083937454247ull}},
{{7034669534732270323ull, 369667149963398ull}},
{{13006433257269636905ull, 295733719970718ull}},
{{17120944396889508724ull, 473173951953149ull}},
{{17386104332253517303ull, 378539161562519ull}},
{{17598232280544724165ull, 302831329250015ull}},
{{9710427575162007049ull, 484530126800025ull}},
{{7768342060129605639ull, 387624101440020ull}},
{{6214673648103684511ull, 310099281152016ull}},
{{2564780207482074571ull, 496158849843226ull}},
{{16809219424953300950ull, 396927079874580ull}},
{{13447375539962640760ull, 317541663899664ull}},
{{10447754419714494246ull, 508066662239463ull}},
{{15736901165255416043ull, 406453329791570ull}},
{{12589520932204332835ull, 325162663833256ull}},
{{12764535862043111889ull, 520260262133210ull}},
{{10211628689634489511ull, 416208209706568ull}},
{{15548000581191412255ull, 332966567765254ull}},
{{13808754485680528639ull, 532746508424407ull}},
{{3668305959060602265ull, 426197206739526ull}},
{{17692040026216123105ull, 340957765391620ull}},
{{9860519968236245352ull, 545532424626593ull}},
{{15267113604072816928ull, 436425939701274ull}},
{{15903039698000163865ull, 349140751761019ull}},
{{14376817072574531215ull, 558625202817631ull}},
{{7812104843317714649ull, 446900162254105ull}},
{{6249683874654171719ull, 357520129803284ull}},
{{8689095914465247698ull, 286016103842627ull}},
{{17591902277886306641ull, 457625766148203ull}},
{{3005475378083314343ull, 366100612918563ull}},
{{9783077931950472121ull, 292880490334850ull}},
{{15652924691120755393ull, 468608784535760ull}},
{{12522339752896604314ull, 374887027628608ull}},
{{17396569431801104098ull, 299909622102886ull}},
{{2009069387688394294ull, 479855395364619ull}},
{{5296604324892625759ull, 383884316291695ull}},
{{4237283459914100607ull, 307107453033356ull}},
{{17847699980088291941ull, 491371924853369ull}},
{{17967508798812543876ull, 393097539882695ull}},
{{14374007039050035101ull, 314478031906156ull}},
{{15619713632996235515ull, 503164851049850ull}},
{{12495770906396988412ull, 402531880839880ull}},
{{9996616725117590729ull, 322025504671904ull}},
{{4926540315962414197ull, 515240807475047ull}},
{{15009278696995662327ull, 412192645980037ull}},
{{4628725328112709215ull, 329754116784030ull}},
{{7405960524980334745ull, 527606586854448ull}},
{{13303466049468088442ull, 422085269483558ull}},
{{18021470469058291400ull, 337668215586846ull}},
{{3008911047299893978ull, 540269144938955ull}},
{{2407128837839915182ull, 432215315951164ull}},
{{5615051885013842469ull, 345772252760931ull}},
{{1605385386538327304ull, 553235604417490ull}},
{{1284308309230661843ull, 442588483533992ull}},
{{12095493091610260444ull, 354070786827193ull}},
{{17055092102772029002ull, 283256629461754ull}},
{{16220100920209515433ull, 453210607138807ull}},
{{5597383106683791700ull, 362568485711046ull}},
{{788557670605123037ull, 290054788568837ull}},
{{4951041087710107183ull, 464087661710139ull}},
{{7650181684909996069ull, 371270129368111ull}},
{{2430796533186086532ull, 297016103494489ull}},
{{11267972082581559098ull, 475225765591182ull}},
{{1635680036581426632ull, 380180612472946ull}},
{{16065939288232782598ull, 304144489978356ull}},
{{18326805231688631511ull, 486631183965370ull}},
{{14661444185350905209ull, 389304947172296ull}},
{{8039806533538813844ull, 311443957737837ull}},
{{16553039268404012473ull, 498310332380539ull}},
{{16931780229465120302ull, 398648265904431ull}},
{{9856075368830185918ull, 318918612723545ull}},
{{15769720590128297469ull, 510269780357672ull}},
{{5237078842618817329ull, 408215824286138ull}},
{{11568360703578874509ull, 326572659428910ull}},
{{62633052016647599ull, 522516255086257ull}},
{{11118152885839049049ull, 418013004069005ull}},
{{8894522308671239239ull, 334410403255204ull}},
{{3163189249648251813ull, 535056645208327ull}},
{{13598597843944332420ull, 428045316166661ull}},
{{7189529460413555613ull, 342436252933329ull}},
{{435200692435958011ull, 547898004693327ull}},
{{11416206998174497378ull, 438318403754661ull}},
{{5443616783797687579ull, 350654723003729ull}},
{{16088484483560120774ull, 561047556805966ull}},
{{9181438772106186296ull, 448838045444773ull}},
{{14723848647168769683ull, 359070436355818ull}},
{{711032473509284777ull, 287256349084655ull}},
{{1137651957614855643ull, 459610158535448ull}},
{{8288819195575705161ull, 367688126828358ull}},
{{14009752985944384775ull, 294150501462686ull}},
{{15036907148027194994ull, 470640802340298ull}},
{{961479274196025025ull, 376512641872239ull}},
{{4458532234098730343ull, 301210113497791ull}},
{{18201698018783699519ull, 481936181596465ull}},
{{14561358415026959615ull, 385548945277172ull}},
{{4270389102537747046ull, 308439156221738ull}},
{{3143273749318484950ull, 493502649954781ull}},
{{17272014258422429253ull, 394802119963824ull}},
{{17506960221479853725ull, 315841695971059ull}},
{{16943089910142034991ull, 505346713553695ull}},
{{13554471928113627993ull, 404277370842956ull}},
{{7154228727748992071ull, 323421896674365ull}},
{{11446765964398387314ull, 517475034678984ull}},
{{12846761586260620174ull, 413980027743187ull}},
{{2898711639524675493ull, 331184022194550ull}},
{{4637938623239480789ull, 529894435511280ull}},
{{3710350898591584631ull, 423915548409024ull}},
{{6657629533615178028ull, 339132438727219ull}},
{{18030904883268105491ull, 542611901963550ull}},
{{14424723906614484393ull, 434089521570840ull}},
{{11539779125291587514ull, 347271617256672ull}},
{{3706251341498898730ull, 555634587610676ull}},
{{17722396332166760277ull, 444507670088540ull}},
{{14177917065733408221ull, 355606136070832ull}},
{{3963636023102905931ull, 284484908856666ull}},
{{17409864081190380459ull, 455175854170665ull}},
{{13927891264952304367ull, 364140683336532ull}},
{{3763615382478022847ull, 291312546669226ull}},
{{17089831056190567525ull, 466100074670761ull}},
{{9982516030210543697ull, 372880059736609ull}},
{{11675361638910345281ull, 298304047789287ull}},
{{3923183363288911157ull, 477286476462860ull}},
{{3138546690631128925ull, 381829181170288ull}},
{{9889534981988723786ull, 305463344936230ull}},
{{15823255971181958059ull, 488741351897968ull}},
{{1590558332719835477ull, 390993081518375ull}},
{{1272446666175868382ull, 312794465214700ull}},
{{2035914665881389411ull, 500471144343520ull}},
{{1628731732705111529ull, 400376915474816ull}},
{{16060380645131730516ull, 320301532379852ull}},
{{10939213773243127533ull, 512482451807764ull}},
{{12440719833336412349ull, 409985961446211ull}},
{{6263227051927219556ull, 327988769156969ull}},
{{17399860912567371936ull, 524782030651150ull}},
{{13919888730053897549ull, 419825624520920ull}},
{{11135910984043118039ull, 335860499616736ull}},
{{10438759944985168216ull, 537376799386778ull}},
{{15729705585471955219ull, 429901439509422ull}},
{{5205066838893743529ull, 343921151607538ull}},
{{4638758127488079324ull, 550273842572061ull}},
{{21657687248553136ull, 440219074057649ull}},
{{3706674964540752832ull, 352175259246119ull}},
{{6654688786374512588ull, 281740207396895ull}},
{{10647502058199220142ull, 450784331835032ull}},
{{1139304017075555467ull, 360627465468026ull}},
{{15668838472628085666ull, 288501972374420ull}},
{{6623397482495385450ull, 461603155799073ull}},
{{12677415615480129006ull, 369282524639258ull}},
{{17520630121867923851ull, 295426019711406ull}},
{{2207566491795305900ull, 472681631538251ull}},
{{16523448452403886013ull, 378145305230600ull}},
{{13218758761923108810ull, 302516244184480ull}},
{{2703269945367422481ull, 484025990695169ull}},
{{5851964771035848308ull, 387220792556135ull}},
{{4681571816828678646ull, 309776634044908ull}},
{{3801166092183975511ull, 495642614471853ull}},
{{10419630503231001055ull, 396514091577482ull}},
{{957006773100980197ull, 317211273261986ull}},
{{12599257281187299286ull, 507538037219177ull}},
{{2700708195466018782ull, 406030429775342ull}},
{{13228613000598545995ull, 324824343820273ull}},
{{17476431986215763269ull, 519718950112437ull}},
{{6602447959488789969ull, 415775160089950ull}},
{{5281958367591031975ull, 332620128071960ull}},
{{8451133388145651160ull, 532192204915136ull}},
{{3071557895774610605ull, 425753763932109ull}},
{{6146595131361598807ull, 340603011145687ull}},
{{13523901024920468415ull, 544964817833099ull}},
{{14508469634678285055ull, 435971854266479ull}},
{{15296124522484538367ull, 348777483413183ull}},
{{2337706347523799448ull, 558043973461094ull}},
{{5559513892760949882ull, 446435178768875ull}},
{{4447611114208759905ull, 357148143015100ull}},
{{3558088891367007924ull, 285718514412080ull}},
{{5692942226187212679ull, 457149623059328ull}},
{{11933051410433590789ull, 365719698447462ull}},
{{2167743498863051985ull, 292575758757970ull}},
{{3468389598180883176ull, 468121214012752ull}},
{{13842758122770437511ull, 374496971210201ull}},
{{7384857683474439685ull, 299597576968161ull}},
{{4437074664075282850ull, 479356123149058ull}},
{{10928357360744046927ull, 383484898519246ull}},
{{5053337073853327218ull, 306787918815397ull}},
{{11774688132907233872ull, 490860670104635ull}},
{{9419750506325787098ull, 392688536083708ull}},
{{14914498034544450324ull, 314150828866966ull}},
{{16484499225787299873ull, 502641326187146ull}},
{{9498250565887929575ull, 402113060949717ull}},
{{219902823226523014ull, 321690448759774ull}},
{{7730542146646257468ull, 514704718015638ull}},
{{13563131346800826621ull, 411763774412510ull}},
{{10850505077440661297ull, 329411019530008ull}},
{{13671459309163147752ull, 527057631248013ull}},
{{18315865076814338848ull, 421646104998410ull}},
{{14652692061451471078ull, 337316883998728ull}},
{{1308214409870891786ull, 539707014397966ull}},
{{15803966786864354722ull, 431765611518372ull}},
{{5264475800007663131ull, 345412489214698ull}},
{{4733812465270350686ull, 552659982743517ull}},
{{14855096416442011519ull, 442127986194813ull}},
{{816030688927878245ull, 353702388955851ull}},
{{15410219810109943889ull, 282961911164680ull}},
{{6209607622466358606ull, 452739057863489ull}},
{{8657034912714997208ull, 362191246290791ull}},
{{3236279115430087443ull, 289752997032633ull}},
{{1488697769946229586ull, 463604795252213ull}},
{{8569655845440804315ull, 370883836201770ull}},
{{6855724676352643452ull, 296707068961416ull}},
{{3590461852680408877ull, 474731310338266ull}},
{{17629764741111968395ull, 379785048270612ull}},
{{6725114163405754069ull, 303828038616490ull}},
{{10760182661449206511ull, 486124861786384ull}},
{{12297494943901275532ull, 388899889429107ull}},
{{2459298325637199779ull, 311119911543286ull}},
{{15002923765245250616ull, 497791858469257ull}},
{{4623641382712379847ull, 398233486775406ull}},
{{9564291427993554ull, 318586789420325ull}},
{{15302866284789687ull, 509738863072520ull}},
{{12242293027831749ull, 407791090458016ull}},
{{14767189093389906692ull, 326232872366412ull}},
{{8870107290456209415ull, 521972595786260ull}},
{{7096085832364967532ull, 417578076629008ull}},
{{13055566295375794672ull, 334062461303206ull}},
{{13510208443117450829ull, 534499938085130ull}},
{{10808166754493960663ull, 427599950468104ull}},
{{12335882218337078853ull, 342079960374483ull}},
{{16048062734597415842ull, 547327936599173ull}},
{{1770403743452201704ull, 437862349279339ull}},
{{5105671809503671686ull, 350289879423471ull}},
{{790377265722054052ull, 560463807077554ull}},
{{4321650627319553565ull, 448371045662043ull}},
{{10836018131339463498ull, 358696836529634ull}},
{{12358163319813481122ull, 286957469223707ull}},
{{5015666052733928502ull, 459131950757932ull}},
{{15080579286412873771ull, 367305560606345ull}},
{{12064463429130299017ull, 293844448485076ull}},
{{11924443857124657781ull, 470151117576122ull}},
{{2160857456215905578ull, 376120894060898ull}},
{{9107383594456545109ull, 300896715248718ull}},
{{10882464936388561851ull, 481434744397949ull}},
{{12395320763852759804ull, 385147795518359ull}},
{{13605605425824118166ull, 308118236414687ull}},
{{7011573422350947774ull, 492989178263500ull}},
{{5609258737880758219ull, 394391342610800ull}},
{{4487406990304606575ull, 315513074088640ull}},
{{7179851184487370520ull, 504820918541824ull}},
{{9433229762331806739ull, 403856734833459ull}},
{{11235932624607355715ull, 323085387866767ull}},
{{3220096940404127851ull, 516936620586828ull}},
{{9954775181807122927ull, 413549296469462ull}},
{{585122515961877695ull, 330839437175570ull}},
{{936196025539004313ull, 529343099480912ull}},
{{11817003264656934420ull, 423474479584729ull}},
{{13142951426467457859ull, 338779583667783ull}},
{{17339373467606022251ull, 542047333868453ull}},
{{2803452329859086831ull, 433637867094763ull}},
{{9621459493371090111ull, 346910293675810ull}},
{{15394335189393744178ull, 555056469881296ull}},
{{8626119336773085019ull, 444045175905037ull}},
{{17968941913644198985ull, 355236140724029ull}},
{{18064502345657269511ull, 284188912579223ull}},
{{6767110864600169279ull, 454702260126758ull}},
{{12792386321163956069ull, 363761808101406ull}},
{{6544560242189254532ull, 291009446481125ull}},
{{10471296387502807252ull, 465615114369800ull}},
{{8377037110002245801ull, 372492091495840ull}},
{{6701629688001796641ull, 297993673196672ull}},
{{14411956315544784949ull, 476789877114675ull}},
{{11529565052435827959ull, 381431901691740ull}},
{{9223652041948662367ull, 305145521353392ull}},
{{448008150218495ull, 488232834165428ull}},
{{7379056036003995442ull, 390586267332342ull}},
{{16971291273028927323ull, 312469013865873ull}},
{{5017973148394821778ull, 499950422185398ull}},
{{11393076148199678069ull, 399960337748318ull}},
{{16493158548043563102ull, 319968270198654ull}},
{{15321007232643969993ull, 511949232317847ull}},
{{4878108156631355348ull, 409559385854278ull}},
{{11281184154788904925ull, 327647508683422ull}},
{{3292499388694606587ull, 524236013893476ull}},
{{17391394769923326562ull, 419388811114780ull}},
{{13913115815938661250ull, 335511048891824ull}},
{{11192938861276127030ull, 536817678226919ull}},
{{12643699903762811947ull, 429454142581535ull}},
{{10114959923010249558ull, 343563314065228ull}},
{{12494587062074488970ull, 549701302504365ull}},
{{9995669649659591176ull, 439761042003492ull}},
{{617838090243852294ull, 351808833602794ull}},
{{8367238573873984317ull, 562894133764470ull}},
{{6693790859099187453ull, 450315307011576ull}},
{{1665683872537439639ull, 360252245609261ull}},
{{16089942356997593004ull, 288201796487408ull}},
{{3607814882744686868ull, 461122874379854ull}},
{{6575600720937659817ull, 368898299503883ull}},
{{12639178206233948500ull, 295118639603106ull}},
{{12843987500490496954ull, 472189823364970ull}},
{{10275190000392397563ull, 377751858691976ull}},
{{4530803185572007727ull, 302201486953581ull}},
{{18317331541140943334ull, 483522379125729ull}},
{{18343214047654664990ull, 386817903300583ull}},
{{3606524793898001022ull, 309454322640467ull}},
{{9459788484978711959ull, 495126916224747ull}},
{{189133158499148921ull, 396101532979798ull}},
{{7530004156283139783ull, 316881226383838ull}},
{{8358657835311113330ull, 507009962214141ull}},
{{2997577453506980341ull, 405607969771313ull}},
{{9776759592289404919ull, 324486375817050ull}},
{{15642815347663047870ull, 519178201307280ull}},
{{12514252278130438296ull, 415342561045824ull}},
{{13700750637246260960ull, 332274048836659ull}},
{{10853154575368286567ull, 531638478138655ull}},
{{8682523660294629253ull, 425310782510924ull}},
{{10635367742977613726ull, 340248626008739ull}},
{{5948541944538450992ull, 544397801613983ull}},
{{12137531185114581440ull, 435518241291186ull}},
{{6020676133349754829ull, 348414593032949ull}},
{{17011779442843428373ull, 557463348852718ull}},
{{2541377110049011728ull, 445970679082175ull}},
{{2033101688039209383ull, 356776543265740ull}},
{{1626481350431367506ull, 285421234612592ull}},
{{6291718975432098333ull, 456673975380147ull}},
{{16101421624571409636ull, 365339180304117ull}},
{{5502439670173307062ull, 292271344243294ull}},
{{16182601101761111946ull, 467634150789270ull}},
{{12946080881408889557ull, 374107320631416ull}},
{{6667515890385201322ull, 299285856505133ull}},
{{6978676609874411793ull, 478857370408213ull}},
{{12961638917383350080ull, 383085896326570ull}},
{{10369311133906680064ull, 306468717061256ull}},
{{9212200184766867457ull, 490349947298010ull}},
{{7369760147813493965ull, 392279957838408ull}},
{{13274505747734615819ull, 313823966270726ull}},
{{13860511566891564664ull, 502118346033162ull}},
{{3709711624029431084ull, 401694676826530ull}},
{{2967769299223544867ull, 321355741461224ull}},
{{12127128508241492434ull, 514169186337958ull}},
{{17080400436077014594ull, 411335349070366ull}},
{{9974971534119701352ull, 329068279256293ull}},
{{12270605639849611840ull, 526509246810069ull}},
{{13505833326621599795ull, 421207397448055ull}},
{{10804666661297279836ull, 336965917958444ull}},
{{6219420213849916768ull, 539145468733511ull}},
{{1286187356338023091ull, 431316374986809ull}},
{{4718298699812328796ull, 345053099989447ull}},
{{11238626734441636397ull, 552084959983115ull}},
{{8990901387553309118ull, 441667967986492ull}},
{{18260767554268378264ull, 353334374389193ull}},
{{3540567599188971641ull, 282667499511355ull}},
{{5664908158702354626ull, 452267999218168ull}},
{{11910624156445704347ull, 361814399374534ull}},
{{13217848139898473801ull, 289451519499627ull}},
{{6391161764869916789ull, 463122431199404ull}},
{{8802278226637843754ull, 370497944959523ull}},
{{14420520210794095650ull, 296398355967618ull}},
{{936739448819091101ull, 474237369548190ull}},
{{749391559055272880ull, 379389895638552ull}},
{{11667559691469949274ull, 303511916510841ull}},
{{11289397876868098192ull, 485619066417346ull}},
{{5342169486752568230ull, 388495253133877ull}},
{{15341782033627785554ull, 310796202507101ull}},
{{17168153624320636240ull, 497273924011362ull}},
{{6355825269972688346ull, 397819139209090ull}},
{{5084660215978150676ull, 318255311367272ull}},
{{11824805160306951406ull, 509208498187635ull}},
{{9459844128245561124ull, 407366798550108ull}},
{{14946572932080269546ull, 325893438840086ull}},
{{16535819061844610627ull, 521429502144138ull}},
{{2160608805249957532ull, 417143601715311ull}},
{{16485882303167607318ull, 333714881372248ull}},
{{4241318796616709770ull, 533943810195598ull}},
{{10771752666777188463ull, 427155048156478ull}},
{{15996099762905571416ull, 341724038525182ull}},
{{10836364361681272974ull, 546758461640292ull}},
{{1290393859861197733ull, 437406769312234ull}},
{{4721663902630868509ull, 349925415449787ull}},
{{11244011058951299938ull, 559880664719659ull}},
{{12684557661902950274ull, 447904531775727ull}},
{{2768948500038539572ull, 358323625420582ull}},
{{13283205244256562627ull, 286658900336465ull}},
{{2806384317100948588ull, 458654240538345ull}},
{{2245107453680758870ull, 366923392430676ull}},
{{16553481221912248389ull, 293538713944540ull}},
{{8038825881350045807ull, 469661942311265ull}},
{{6431060705080036645ull, 375729553849012ull}},
{{16212895008289760286ull, 300583643079209ull}},
{{14872585569037885488ull, 480933828926735ull}},
{{11898068455230308390ull, 384747063141388ull}},
{{16897152393668067358ull, 307797650513110ull}},
{{8588699756159356158ull, 492476240820977ull}},
{{17939006249153215896ull, 393980992656781ull}},
{{10661856184580662393ull, 315184794125425ull}},
{{17058969895329059830ull, 504295670600680ull}},
{{13647175916263247864ull, 403436536480544ull}},
{{14607089547752508614ull, 322749229184435ull}},
{{4924599202694462167ull, 516398766695097ull}},
{{15007725806381300703ull, 413119013356077ull}},
{{4627483015621219916ull, 330495210684862ull}},
{{11093321639735862189ull, 528792337095779ull}},
{{12564006126530600074ull, 423033869676623ull}},
{{17429902530708300706ull, 338427095741298ull}},
{{5751751160681819190ull, 541483353186078ull}},
{{11980098558029275998ull, 433186682548862ull}},
{{2205381216939600152ull, 346549346039090ull}},
{{3528609947103360244ull, 554478953662544ull}},
{{6512236772424598518ull, 443583162930035ull}},
{{5209789417939678814ull, 354866530344028ull}},
{{11546529163835563698ull, 283893224275222ull}},
{{3717051403169260624ull, 454229158840356ull}},
{{17731036381503049792ull, 363383327072284ull}},
{{17874177919944350157ull, 290706661657827ull}},
{{13841289412943318958ull, 465130658652524ull}},
{{14762380345096565490ull, 372104526922019ull}},
{{15499253090819162715ull, 297683621537615ull}},
{{6352060871601108728ull, 476293794460185ull}},
{{5081648697280886982ull, 381035035568148ull}},
{{11444016587308530232ull, 304828028454518ull}},
{{14621077724951738048ull, 487724845527229ull}},
{{15386210994703300762ull, 390179876421783ull}},
{{1240922351536909640ull, 312143901137427ull}},
{{5674824577200965747ull, 499430241819883ull}},
{{11918557291244593244ull, 399544193455906ull}},
{{5845497018253764272ull, 319635354764725ull}},
{{9352795229206022835ull, 511416567623560ull}},
{{7482236183364818268ull, 409133254098848ull}},
{{13364486576175675261ull, 327306603279078ull}},
{{17693829707139170094ull, 523690565246525ull}},
{{14155063765711336075ull, 418952452197220ull}},
{{11324051012569068860ull, 335161961757776ull}},
{{10739783990626689530ull, 536259138812442ull}},
{{1213129563017530978ull, 429007311049954ull}},
{{4659852465155935105ull, 343205848839963ull}},
{{3766415129507585846ull, 549129358143941ull}},
{{17770527362573709969ull, 439303486515152ull}},
{{6837724260575147329ull, 351442789212122ull}},
{{14629707631662146050ull, 562308462739395ull}},
{{11703766105329716840ull, 449846770191516ull}},
{{5673664069521863148ull, 359877416153213ull}},
{{11917628885101311165ull, 287901932922570ull}},
{{621462142452546248ull, 460643092676113ull}},
{{7875867343445857645ull, 368514474140890ull}},
{{6300693874756686116ull, 294811579312712ull}},
{{13770459014352608109ull, 471698526900339ull}},
{{14705716026223996810ull, 377358821520271ull}},
{{8075224006237287125ull, 301887057216217ull}},
{{16609707224721569723ull, 483019291545947ull}},
{{5909068150293435132ull, 386415433236758ull}},
{{12105952149718568752ull, 309132346589406ull}},
{{11990825810065889357ull, 494611754543050ull}},
{{9592660648052711486ull, 395689403634440ull}},
{{7674128518442169188ull, 316551522907552ull}},
{{15967954444249381025ull, 506482436652083ull}},
{{1706317111173773850ull, 405185949321667ull}},
{{12433100133164750050ull, 324148759457333ull}},
{{16203611398321689757ull, 518638015131733ull}},
{{1894842674431620836ull, 414910412105387ull}},
{{12583920583771027638ull, 331928329684309ull}},
{{9066226489807913251ull, 531085327494895ull}},
{{7252981191846330601ull, 424868261995916ull}},
{{2113036138735154157ull, 339894609596733ull}},
{{18138253080943887945ull, 543831375354772ull}},
{{7131904835271289710ull, 435065100283818ull}},
{{13084221497700852414ull, 348052080227054ull}},
{{9866707952095632893ull, 556883328363287ull}},
{{514668732192685668ull, 445506662690630ull}},
{{411734985754148534ull, 356405330152504ull}},
{{4018736803345229150ull, 285124264122003ull}},
{{2740630070610456318ull, 456198822595205ull}},
{{2192504056488365054ull, 364959058076164ull}},
{{5443352059932602366ull, 291967246460931ull}},
{{1330665666408343140ull, 467147594337490ull}},
{{1064532533126674512ull, 373718075469992ull}},
{{11919672470727070579ull, 298974460375993ull}},
{{15382127138421402604ull, 478359136601589ull}},
{{15995050525479032406ull, 382687309281271ull}},
{{9106691605641315602ull, 306149847425017ull}},
{{18260055383768015286ull, 489839755880027ull}},
{{7229346677530591582ull, 391871804704022ull}},
{{16851523786250204235ull, 313497443763217ull}},
{{12205042799032685484ull, 501595910021148ull}},
{{17142731868709969034ull, 401276728016918ull}},
{{2646139050742244257ull, 321021382413535ull}},
{{4233822481187590812ull, 513634211861656ull}},
{{18144453243917713942ull, 410907369489324ull}},
{{18204911409876081477ull, 328725895591459ull}},
{{18059811811575999394ull, 525961432946335ull}},
{{14447849449260799515ull, 420769146357068ull}},
{{490233115182908642ull, 336615317085655ull}},
{{784372984292653828ull, 538584507337048ull}},
{{8006196016917943708ull, 430867605869638ull}},
{{13783654443018175613ull, 344694084695710ull}},
{{3607103035119529365ull, 551510535513137ull}},
{{13953728872321354462ull, 441208428410509ull}},
{{14852331912598993892ull, 352966742728407ull}},
{{4503167900595374467ull, 282373394182726ull}},
{{18273115085178330118ull, 451797430692361ull}},
{{10929143253400753771ull, 361437944553889ull}},
{{12432663417462513340ull, 289150355643111ull}},
{{12513563838456200698ull, 462640569028978ull}},
{{17389548700248781204ull, 370112455223182ull}},
{{6532941330715204317ull, 296089964178546ull}},
{{3074008499660506261ull, 473743942685674ull}},
{{6148555614470315332ull, 378995154148539ull}},
{{8608193306318162589ull, 303196123318831ull}},
{{6394411660625239496ull, 485113797310130ull}},
{{5115529328500191597ull, 388091037848104ull}},
{{7781772277542063600ull, 310472830278483ull}},
{{8761486829325391438ull, 496756528445573ull}},
{{14387887092944133796ull, 397405222756458ull}},
{{442263230129576067ull, 317924178205167ull}},
{{4396969982949232031ull, 508678685128267ull}},
{{14585622430585116595ull, 406942948102613ull}},
{{600451500242362306ull, 325554358482091ull}},
{{12028768844613510660ull, 520886973571345ull}},
{{9623015075690808528ull, 416709578857076ull}},
{{4009063245810736499ull, 333367663085661ull}},
{{17482547637522909368ull, 533388260937057ull}},
{{6607340480534506848ull, 426710608749646ull}},
{{1596523569685695155ull, 341368486999717ull}},
{{6243786526239022571ull, 546189579199547ull}},
{{16063075665216949027ull, 436951663359637ull}},
{{5471762902689738575ull, 349561330687710ull}},
{{8754820644303581720ull, 559298129100336ull}},
{{3314507700700955053ull, 447438503280269ull}},
{{6340954975302674365ull, 357950802624215ull}},
{{5072763980242139492ull, 286360642099372ull}},
{{11805771183129333511ull, 458177027358995ull}},
{{9444616946503466809ull, 366541621887196ull}},
{{3866344742460863124ull, 293233297509757ull}},
{{9875500402679291321ull, 469173276015611ull}},
{{4211051507401522734ull, 375338620812489ull}},
{{7058190020663128510ull, 300270896649991ull}},
{{3914406403577184970ull, 480433434639986ull}},
{{17888920381829389269ull, 384346747711988ull}},
{{3243089861237780445ull, 307477398169591ull}},
{{16256990222206179682ull, 491963837071345ull}},
{{13005592177764943746ull, 393571069657076ull}},
{{6715124927470044673ull, 314856855725661ull}},
{{3365502254468250831ull, 503770969161058ull}},
{{10071099433058421311ull, 403016775328846ull}},
{{4367530731704826726ull, 322413420263077ull}},
{{10677397985469633085ull, 515861472420923ull}},
{{15920616017859527114ull, 412689177936738ull}},
{{1668446370061890721ull, 330151342349391ull}},
{{13737560636324756124ull, 528242147759025ull}},
{{10990048509059804899ull, 422593718207220ull}},
{{8792038807247843919ull, 338074974565776ull}},
{{6688564462112729625ull, 540919959305242ull}},
{{16418898013915914669ull, 432735967444193ull}},
{{2067071966907000766ull, 346188773955355ull}},
{{3307315147051201226ull, 553902038328568ull}},
{{10024549747124781627ull, 443121630662854ull}},
{{11708988612441735624ull, 354497304530283ull}},
{{16745888519437209146ull, 283597843624226ull}},
{{967979927906162371ull, 453756549798763ull}},
{{8153081571808750543ull, 363005239839010ull}},
{{6522465257447000435ull, 290404191871208ull}},
{{6746595597173290372ull, 464646706993933ull}},
{{12775974107222452944ull, 371717365595146ull}},
{{6531430471036052032ull, 297373892476117ull}},
{{14139637568399593575ull, 475798227961787ull}},
{{3933012425235854213ull, 380638582369430ull}},
{{3146409940188683371ull, 304510865895544ull}},
{{12412953533785714040ull, 487217385432870ull}},
{{9930362827028571232ull, 389773908346296ull}},
{{4254941446880946662ull, 311819126677037ull}},
{{10497255129751424983ull, 498910602683259ull}},
{{12087152918543050309ull, 399128482146607ull}},
{{2291024705350619601ull, 319302785717286ull}},
{{14733685972786722331ull, 510884457147657ull}},
{{4408251148745557219ull, 408707565718126ull}},
{{18283996177964087068ull, 326966052574500ull}},
{{10807649811032987692ull, 523145684119201ull}},
{{4956771034084479831ull, 418516547295361ull}},
{{276068012525673541ull, 334813237836289ull}},
{{7820406449524898313ull, 535701180538062ull}},
{{17324371603845649620ull, 428560944430449ull}},
{{17548846097818430019ull, 342848755544359ull}},
{{17010107312283757061ull, 548558008870975ull}},
{{13608085849827005649ull, 438846407096780ull}},
{{10886468679861604519ull, 351077125677424ull}},
{{6350303443552836261ull, 561723401083879ull}},
{{8769591569584179332ull, 449378720867103ull}},
{{14394370885151164112ull, 359502976693682ull}},
{{4136799078637110643ull, 287602381354946ull}},
{{17686924970045107998ull, 460163810167913ull}},
{{3081493531810355429ull, 368131048134331ull}},
{{17222590084415925636ull, 294504838507464ull}},
{{16488097690839750048ull, 471207741611943ull}},
{{2122431708446069069ull, 376966193289555ull}},
{{1697945366756855255ull, 301572954631644ull}},
{{10095410216294789055ull, 482516727410630ull}},
{{8076328173035831244ull, 386013381928504ull}},
{{10150411353170575318ull, 308810705542803ull}},
{{12551309350331010186ull, 494097128868485ull}},
{{10041047480264808149ull, 395277703094788ull}},
{{15411535613695667165ull, 316222162475830ull}},
{{6211712908203515849ull, 505955459961329ull}},
{{8658719141304723002ull, 404764367969063ull}},
{{14305672942527599048ull, 323811494375250ull}},
{{4442332634334606861ull, 518098391000401ull}},
{{18311261366435326781ull, 414478712800320ull}},
{{14649009093148261425ull, 331582970240256ull}},
{{16059716919553397634ull, 530532752384410ull}},
{{12847773535642718107ull, 424426201907528ull}},
{{17656916457997995132ull, 339540961526022ull}},
{{13493671073829150919ull, 543265538441636ull}},
{{7105588044321410412ull, 434612430753309ull}},
{{9373819250199038652ull, 347689944602647ull}},
{{240715541350820551ull, 556303911364236ull}},
{{14949967692048297734ull, 445043129091388ull}},
{{891927709412907217ull, 356034503273111ull}},
{{15470937426497967066ull, 284827602618488ull}},
{{2617406993945285367ull, 455724164189582ull}},
{{13161972039381959263ull, 364579331351665ull}},
{{10529577631505567411ull, 291663465081332ull}},
{{2089928951441266565ull, 466661544130132ull}},
{{12739989605378744221ull, 373329235304105ull}},
{{10191991684302995377ull, 298663388243284ull}},
{{5239140250659061634ull, 477861421189255ull}},
{{4191312200527249307ull, 382289136951404ull}},
{{7042398575163709769ull, 305831309561123ull}},
{{7578488905520025307ull, 489330095297797ull}},
{{17130837568641751215ull, 391464076238237ull}},
{{6325972425429580325ull, 313171260990590ull}},
{{10121555880687328521ull, 501074017584944ull}},
{{11786593519291773140ull, 400859214067955ull}},
{{9429274815433418512ull, 320687371254364ull}},
{{4018793260467738650ull, 513099794006983ull}},
{{10593732237858011566ull, 410479835205586ull}},
{{4785636975544498930ull, 328383868164469ull}},
{{15035716790355018934ull, 525414189063150ull}},
{{12028573432284015147ull, 420331351250520ull}},
{{9622858745827212118ull, 336265081000416ull}},
{{8017876363839718742ull, 538024129600666ull}},
{{2724952276329864670ull, 430419303680533ull}},
{{9558659450547712383ull, 344335442944426ull}},
{{7915157491392519166ull, 550936708711082ull}},
{{17400172437339746302ull, 440749366968865ull}},
{{13920137949871797042ull, 352599493575092ull}},
{{3757412730413616987ull, 282079594860074ull}},
{{13390557998145607826ull, 451327351776118ull}},
{{18091144028000306907ull, 361061881420894ull}},
{{18162264037142155849ull, 288849505136715ull}},
{{10612878385717897742ull, 462159208218745ull}},
{{8490302708574318193ull, 369727366574996ull}},
{{3102893352117544231ull, 295781893259997ull}},
{{8653978178129981094ull, 473251029215995ull}},
{{6923182542503984875ull, 378600823372796ull}},
{{1849197219261277577ull, 302880658698237ull}},
{{6648064365559954446ull, 484609053917179ull}},
{{9007800307189873880ull, 387687243133743ull}},
{{14584937875235719750ull, 310149794506994ull}},
{{12267854156151420631ull, 496239671211191ull}},
{{6124934510179226182ull, 396991736968953ull}},
{{12278645237627201592ull, 317593389575162ull}},
{{4888437121235881254ull, 508149423320260ull}},
{{3910749696988705003ull, 406519538656208ull}},
{{10507297387074784649ull, 325215630924966ull}},
{{9432978189835834792ull, 520345009479946ull}},
{{3857033737126757510ull, 416276007583957ull}},
{{14153673433927136978ull, 333020806067165ull}},
{{4199133420573867549ull, 532833289707465ull}},
{{3359306736459094039ull, 426266631765972ull}},
{{13755491833393006201ull, 341013305412777ull}},
{{7251391674461168629ull, 545621288660444ull}},
{{9490462154310845226ull, 436497030928355ull}},
{{7592369723448676181ull, 349197624742684ull}},
{{1079745113292150920ull, 558716199588295ull}},
{{863796090633720736ull, 446972959670636ull}},
{{15448432131474617881ull, 357578367736508ull}},
{{1290699260953963335ull, 286062694189207ull}},
{{5754467632268251660ull, 457700310702731ull}},
{{914225291072691005ull, 366160248562185ull}},
{{731380232858152804ull, 292928198849748ull}},
{{15927603631540685779ull, 468685118159596ull}},
{{9052734090490638300ull, 374948094527677ull}},
{{18310233716618241609ull, 299958475622141ull}},
{{3470932243395814313ull, 479933560995427ull}},
{{13844792238942382420ull, 383946848796341ull}},
{{7386484976411995613ull, 307157479037073ull}},
{{8129027147517282657ull, 491451966459317ull}},
{{17571268162239557095ull, 393161573167453ull}},
{{2988968085565914707ull, 314529258533963ull}},
{{1093000122163553208ull, 503246813654341ull}},
{{15631795356698483859ull, 402597450923472ull}},
{{5126738655874966440ull, 322077960738778ull}},
{{4513433034658035982ull, 515324737182045ull}},
{{3610746427726428785ull, 412259789745636ull}},
{{17645992401148784321ull, 329807831796508ull}},
{{6097494953386592975ull, 527692530874414ull}},
{{8567344777451184703ull, 422154024699531ull}},
{{3164527007219037439ull, 337723219759625ull}},
{{5063243211550459903ull, 540357151615400ull}},
{{4050594569240367922ull, 432285721292320ull}},
{{3240475655392294338ull, 345828577033856ull}},
{{16252807492853401910ull, 553325723254169ull}},
{{16691594809024631851ull, 442660578603335ull}},
{{13353275847219705481ull, 354128462882668ull}},
{{18061318307259585031ull, 283302770306134ull}},
{{17830062847389605080ull, 453284432489815ull}},
{{14264050277911684064ull, 362627545991852ull}},
{{4032542592845526605ull, 290102036793482ull}},
{{10141416963294752891ull, 464163258869571ull}},
{{4423784755893891990ull, 371330607095657ull}},
{{14607074248940844561ull, 297064485676525ull}},
{{4924574724595799682ull, 475303177082441ull}},
{{250310964934729422ull, 380242541665953ull}},
{{7578946401431604184ull, 304194033332762ull}},
{{15815663057032477018ull, 486710453332419ull}},
{{16341879260367891938ull, 389368362665935ull}},
{{13073503408294313550ull, 311494690132748ull}},
{{17228256638528991357ull, 498391504212397ull}},
{{6403907681339372439ull, 398713203369918ull}},
{{12501823774555318598ull, 318970562695934ull}},
{{8934871595062778787ull, 510352900313495ull}},
{{7147897276050223030ull, 408282320250796ull}},
{{2028969006098268100ull, 326625856200637ull}},
{{6935699224499139284ull, 522601369921019ull}},
{{9237908194341221750ull, 418081095936815ull}},
{{7390326555472977400ull, 334464876749452ull}},
{{15513871303498674164ull, 535143802799123ull}},
{{1343050598573208361ull, 428115042239299ull}},
{{4763789293600477012ull, 342492033791439ull}},
{{15000760499244583866ull, 547987254066302ull}},
{{4621910769911846446ull, 438389803253042ull}},
{{14765575060155208127ull, 350711842602433ull}},
{{1488827207796871064ull, 561138948163894ull}},
{{4880410580979407174ull, 448911158531115ull}},
{{3904328464783525739ull, 359128926824892ull}},
{{14191509216052551561ull, 287303141459913ull}},
{{570321857232620558ull, 459685026335862ull}},
{{11524303930011827416ull, 367748021068689ull}},
{{12908791958751372256ull, 294198416854951ull}},
{{13275369504518374964ull, 470717466967922ull}},
{{3241597974130879324ull, 376573973574338ull}},
{{9971976008788524106ull, 301259178859470ull}},
{{15955161614061638570ull, 482014686175152ull}},
{{5385431661765490209ull, 385611748940122ull}},
{{15376391773638123137ull, 308489399152097ull}},
{{9844831578853355726ull, 493583038643356ull}},
{{4186516448340774258ull, 394866430914685ull}},
{{3349213158672619406ull, 315893144731748ull}},
{{1669392239134280727ull, 505429031570797ull}},
{{12403560235533155551ull, 404343225256637ull}},
{{2544150558942703794ull, 323474580205310ull}},
{{4070640894308326071ull, 517559328328496ull}},
{{18013907974414302150ull, 414047462662796ull}},
{{10721777564789531397ull, 331237970130237ull}},
{{2397448844695608942ull, 529980752208380ull}},
{{1917959075756487153ull, 423984601766704ull}},
{{5223716075347100046ull, 339187681413363ull}},
{{4668596905813449750ull, 542700290261381ull}},
{{45528709908849477ull, 434160232209105ull}},
{{36422967927079582ull, 347328185767284ull}},
{{7436974378167147977ull, 555725097227654ull}},
{{9638928317275628705ull, 444580077782123ull}},
{{15089840283304323610ull, 355664062225698ull}},
{{1003825782417727918ull, 284531249780559ull}},
{{8984818881352185316ull, 455249999648894ull}},
{{10877203919823658576ull, 364199999719115ull}},
{{8701763135858926861ull, 291359999775292ull}},
{{17612169832116193301ull, 466175999640467ull}},
{{6711038236209133994ull, 372940799712374ull}},
{{9058179403709217518ull, 298352639769899ull}},
{{3425040601709017060ull, 477364223631839ull}},
{{6429381296109123971ull, 381891378905471ull}},
{{1454156222145388854ull, 305513103124377ull}},
{{6015998770174532489ull, 488820964999003ull}},
{{12191496645623446638ull, 391056771999202ull}},
{{2374499687014936664ull, 312845417599362ull}},
{{7488548313965808985ull, 500552668158979ull}},
{{9680187465914557511ull, 400442134527183ull}},
{{15122847602215466655ull, 320353707621746ull}},
{{16817858534060926003ull, 512565932194794ull}},
{{17143635641990651125ull, 410052745755835ull}},
{{13714908513592520900ull, 328042196604668ull}},
{{18254504807006123117ull, 524867514567469ull}},
{{18292952660346808817ull, 419894011653975ull}},
{{14634362128277447053ull, 335915209323180ull}},
{{4968235331534363670ull, 537464334917089ull}},
{{7663937079969401259ull, 429971467933671ull}},
{{2441800849233610684ull, 343977174346937ull}},
{{7596230173515687418ull, 550363478955099ull}},
{{9766332953554460257ull, 440290783164079ull}},
{{11502415177585478529ull, 352232626531263ull}},
{{16580629771552203469ull, 281786101225010ull}},
{{8082263560773973935ull, 450857761960017ull}},
{{17533857292844910118ull, 360686209568013ull}},
{{2959039390050197124ull, 288548967654411ull}},
{{15802509468306046369ull, 461678348247057ull}},
{{5263309945161016449ull, 369342678597646ull}},
{{521299141386902836ull, 295474142878117ull}},
{{4523427440960954861ull, 472758628604987ull}},
{{14686788396994494858ull, 378206902883989ull}},
{{15438779532337506209ull, 302565522307191ull}},
{{17323349622256189289ull, 484104835691506ull}},
{{10169330883063041108ull, 387283868553205ull}},
{{8135464706450432886ull, 309827094842564ull}},
{{1948697086094961649ull, 495723351748103ull}},
{{8937655298359789965ull, 396578681398482ull}},
{{18218170682913562942ull, 317262945118785ull}},
{{10702329018952149091ull, 507620712190057ull}},
{{1183165585677898626ull, 406096569752046ull}},
{{15703927727509960194ull, 324877255801636ull}},
{{17747586734532115664ull, 519803609282618ull}},
{{3130022943399961561ull, 415842887426095ull}},
{{2504018354719969249ull, 332674309940876ull}},
{{15074475811777681768ull, 532278895905401ull}},
{{8370231834680235091ull, 425823116724321ull}},
{{3006836653002277750ull, 340658493379457ull}},
{{8500287459545554723ull, 545053589407131ull}},
{{3110881152894533455ull, 436042871525705ull}},
{{2488704922315626764ull, 348834297220564ull}},
{{11360625505188823469ull, 558134875552902ull}},
{{1709802774667238129ull, 446507900442322ull}},
{{12435888663959521473ull, 357206320353857ull}},
{{2570013301683796532ull, 285765056283086ull}},
{{15180067726919805421ull, 457224090052937ull}},
{{4765356552052023690ull, 365779272042350ull}},
{{3812285241641618952ull, 292623417633880ull}},
{{6099656386626590323ull, 468197468214208ull}},
{{12258422738785092905ull, 374557974571366ull}},
{{6117389376286164001ull, 299646379657093ull}},
{{6098474187315952078ull, 479434207451349ull}},
{{8568128164594671986ull, 383547365961079ull}},
{{10543851346417647912ull, 306837892768863ull}},
{{13180813339526326336ull, 490940628430181ull}},
{{6855301856879150745ull, 392752502744145ull}},
{{5484241485503320596ull, 314202002195316ull}},
{{1396088747321492308ull, 502723203512506ull}},
{{15874266256824835139ull, 402178562810004ull}},
{{16388761820201778434ull, 321742850248003ull}},
{{4085926023871383556ull, 514788560396806ull}},
{{18026136078064748137ull, 411830848317444ull}},
{{18110257677193708833ull, 329464678653955ull}},
{{10529668209800382517ull, 527143485846329ull}},
{{12113083382582216337ull, 421714788677063ull}},
{{17069164335549593716ull, 337371830941650ull}},
{{8863918863169798329ull, 539794929506641ull}},
{{3401786275793928340ull, 431835943605313ull}},
{{10100126650118963318ull, 345468754884250ull}},
{{16160202640190341310ull, 552750007814800ull}},
{{12928162112152273048ull, 442200006251840ull}},
{{10342529689721818438ull, 353760005001472ull}},
{{895326122293634104ull, 283008004001178ull}},
{{16189917054637455859ull, 452812806401884ull}},
{{16641282458451875011ull, 362250245121507ull}},
{{5934328337277679362ull, 289800196097206ull}},
{{2116227710160466333ull, 463680313755530ull}},
{{1692982168128373066ull, 370944251004424ull}},
{{5043734549244608776ull, 296755400803539ull}},
{{15448672908275194689ull, 474808641285662ull}},
{{4980240697136335104ull, 379846913028530ull}},
{{3984192557709068083ull, 303877530422824ull}},
{{13753405721818329580ull, 486204048676518ull}},
{{18381422206938484310ull, 388963238941214ull}},
{{18394486580292697771ull, 311170591152971ull}},
{{3605736825274944172ull, 497872945844755ull}},
{{2884589460219955338ull, 398298356675804ull}},
{{5997020382917874593ull, 318638685340643ull}},
{{5905883797926689026ull, 509821896545029ull}},
{{8414055853083261544ull, 407857517236023ull}},
{{14109942311950429882ull, 326286013788818ull}},
{{439814810669225872ull, 522057622062110ull}},
{{351851848535380697ull, 417646097649688ull}},
{{7660179108312125204ull, 334116878119750ull}},
{{12256286573299400327ull, 534587004991600ull}},
{{9805029258639520261ull, 427669603993280ull}},
{{7844023406911616209ull, 342135683194624ull}},
{{1482391006832854965ull, 547417093111399ull}},
{{4875261620208194295ull, 437933674489119ull}},
{{7589558110908465759ull, 350346939591295ull}},
{{12143292977453545215ull, 560555103346072ull}},
{{2335936752479015526ull, 448444082676858ull}},
{{9247447031467033067ull, 358755266141486ull}},
{{3708608810431716130ull, 287004212913189ull}},
{{13312471726174566455ull, 459206740661102ull}},
{{3271279751455832517ull, 367365392528882ull}},
{{13685070245390396983ull, 293892314023105ull}},
{{3449368318915083558ull, 470227702436969ull}},
{{6448843469873977169ull, 376182161949575ull}},
{{5159074775899181735ull, 300945729559660ull}},
{{8254519641438690777ull, 481513167295456ull}},
{{2914266898409042298ull, 385210533836365ull}},
{{2331413518727233838ull, 308168427069092ull}},
{{7419610444705484465ull, 493069483310547ull}},
{{17003734799990118541ull, 394455586648437ull}},
{{6224290210508274187ull, 315564469318750ull}},
{{9958864336813238699ull, 504903150910000ull}},
{{7967091469450590959ull, 403922520728000ull}},
{{6373673175560472767ull, 323138016582400ull}},
{{10197877080896756428ull, 517020826531840ull}},
{{8158301664717405142ull, 413616661225472ull}},
{{17594687775999655083ull, 330893328980377ull}},
{{13394105182631806841ull, 529429326368604ull}},
{{14404632960847355796ull, 423543461094883ull}},
{{455659924452153667ull, 338834768875907ull}},
{{4418404693865356190ull, 542135630201451ull}},
{{18292119014059926245ull, 433708504161160ull}},
{{14633695211247940996ull, 346966803328928ull}},
{{1277819449545243655ull, 555146885326286ull}},
{{15779650818603836216ull, 444117508261028ull}},
{{1555674210657338003ull, 355294006608823ull}},
{{8623236998009691049ull, 284235205287058ull}},
{{10107830382073595355ull, 454776328459293ull}},
{{15464961935142696931ull, 363821062767434ull}},
{{16061318362856067868ull, 291056850213947ull}},
{{10940714121602067296ull, 465690960342316ull}},
{{5063222482539743513ull, 372552768273853ull}},
{{11429275615515615457ull, 298042214619082ull}},
{{3529445725857343439ull, 476867543390532ull}},
{{13891603024911605720ull, 381494034712425ull}},
{{11113282419929284576ull, 305195227769940ull}},
{{17781251871886855322ull, 488312364431904ull}},
{{17914350312251394581ull, 390649891545523ull}},
{{3263433805575384695ull, 312519913236419ull}},
{{12600191718404436159ull, 500031861178270ull}},
{{10080153374723548927ull, 400025488942616ull}},
{{4374773885036928818ull, 320020391154093ull}},
{{3310289401317175786ull, 512032625846549ull}},
{{6337580335795650952ull, 409626100677239ull}},
{{8759413083378431085ull, 327700880541791ull}},
{{6636363303921669089ull, 524321408866866ull}},
{{1619741828395424948ull, 419457127093493ull}},
{{8674491092200160605ull, 335565701674794ull}},
{{2811139303294525999ull, 536905122679671ull}},
{{17006306701603262092ull, 429524098143736ull}},
{{9915696546540699350ull, 343619278514989ull}},
{{4797068030239387991ull, 549790845623983ull}},
{{11216352053675331039ull, 439832676499186ull}},
{{5283732828198354508ull, 351866141199349ull}},
{{7916335077300593929ull, 281492912959479ull}},
{{1598089679455219318ull, 450388660735167ull}},
{{12346518187789906424ull, 360310928588133ull}},
{{17255912179715745785ull, 288248742870506ull}},
{{1784017784351820994ull, 461197988592811ull}},
{{16184609486449098088ull, 368958390874248ull}},
{{1879641144933547501ull, 295166712699399ull}},
{{10386123461377496648ull, 472266740319038ull}},
{{15687596398585817965ull, 377813392255230ull}},
{{12550077118868654372ull, 302250713804184ull}},
{{9012076945964116025ull, 483601142086695ull}},
{{7209661556771292820ull, 386880913669356ull}},
{{2078380430675123933ull, 309504730935485ull}},
{{3325408689080198293ull, 495207569496776ull}},
{{17417722210231799927ull, 396166055597420ull}},
{{13934177768185439941ull, 316932844477936ull}},
{{14915986799612883260ull, 507092551164698ull}},
{{864742995464575638ull, 405674040931759ull}},
{{4381143211113570834ull, 324539232745407ull}},
{{10699177952523623658ull, 519262772392651ull}},
{{4869993547276988603ull, 415410217914121ull}},
{{206646023079680559ull, 332328174331297ull}},
{{4019982451669399218ull, 531725078930075ull}},
{{3215985961335519374ull, 425380063144060ull}},
{{2572788769068415499ull, 340304050515248ull}},
{{427113215767554476ull, 544486480824397ull}},
{{11409737016839774550ull, 435589184659517ull}},
{{1749091983987998994ull, 348471347727614ull}},
{{10177244803864619036ull, 557554156364182ull}},
{{763098213607874583ull, 446043325091346ull}},
{{15367873829853940959ull, 356834660073076ull}},
{{8604950249141242444ull, 285467728058461ull}},
{{6389222769142167264ull, 456748364893538ull}},
{{12490075844797554457ull, 365398691914830ull}},
{{9992060675838043566ull, 292318953531864ull}},
{{4919250637115138736ull, 467710325650983ull}},
{{11314098139175931635ull, 374168260520786ull}},
{{5361929696598834985ull, 299334608416629ull}},
{{15957785144041956622ull, 478935373466606ull}},
{{9076879300491654975ull, 383148298773285ull}},
{{7261503440393323980ull, 306518639018628ull}},
{{7929056689887408044ull, 490429822429805ull}},
{{6343245351909926435ull, 392343857943844ull}},
{{8763945096269851471ull, 313875086355075ull}},
{{14022312154031762355ull, 502200138168120ull}},
{{11217849723225409884ull, 401760110534496ull}},
{{5284930963838417584ull, 321408088427597ull}},
{{12145238356883378457ull, 514252941484155ull}},
{{9716190685506702766ull, 411402353187324ull}},
{{11462301363147272536ull, 329121882549859ull}},
{{7271635736809905088ull, 526595012079775ull}},
{{5817308589447924070ull, 421276009663820ull}},
{{4653846871558339256ull, 337020807731056ull}},
{{67457365009522164ull, 539233292369690ull}},
{{53965892007617731ull, 431386633895752ull}},
{{11111219157831825154ull, 345109307116601ull}},
{{10399253023047099600ull, 552174891386562ull}},
{{940704788953859034ull, 441739913109250ull}},
{{752563831163087227ull, 353391930487400ull}},
{{602051064930469781ull, 282713544389920ull}},
{{963281703888751651ull, 452341671023872ull}},
{{11838671807336732290ull, 361873336819097ull}},
{{2092239816385565186ull, 289498669455278ull}},
{{18104978965184545590ull, 463197871128444ull}},
{{18173331986889546795ull, 370558296902755ull}},
{{14538665589511637436ull, 296446637522204ull}},
{{12193818498992888928ull, 474314620035527ull}},
{{2376357169710490496ull, 379451696028422ull}},
{{12969132179994123366ull, 303561356822737ull}},
{{5993216229022956094ull, 485698170916380ull}},
{{4794572983218364875ull, 388558536733104ull}},
{{7525007201316602223ull, 310846829386483ull}},
{{8350662707364653234ull, 497354927018373ull}},
{{14059227795375543233ull, 397883941614698ull}},
{{179335792074703617ull, 318307153291759ull}},
{{7665634896803346434ull, 509291445266814ull}},
{{9821856732184587470ull, 407433156213451ull}},
{{4168136571005759653ull, 325946524970761ull}},
{{17737064957834946414ull, 521514439953217ull}},
{{6810954336784136485ull, 417211551962574ull}},
{{9138112284169219511ull, 333769241570059ull}},
{{3552933210445020249ull, 534030786512095ull}},
{{2842346568356016199ull, 427224629209676ull}},
{{17031272513652454252ull, 341779703367740ull}},
{{8803291948134375187ull, 546847525388385ull}},
{{7042633558507500150ull, 437478020310708ull}},
{{13012804476289820766ull, 349982416248566ull}},
{{13441789532579892579ull, 559971865997706ull}},
{{7064082811322003740ull, 447977492798165ull}},
{{5651266249057602992ull, 358381994238532ull}},
{{15589059443471813363ull, 286705595390825ull}},
{{6495751035845349765ull, 458728952625321ull}},
{{1507252013934369489ull, 366983162100257ull}},
{{12273848055373226561ull, 293586529680205ull}},
{{1191412814887610881ull, 469738447488329ull}},
{{4642479066651999028ull, 375790757990663ull}},
{{11092680882805419869ull, 300632606392530ull}},
{{17748289412488671790ull, 481012170228048ull}},
{{3130585085765206463ull, 384809736182439ull}},
{{6193816883354075493ull, 307847788945951ull}},
{{2531409383882700143ull, 492556462313522ull}},
{{13093173951331891084ull, 394045169850817ull}},
{{3095841531581692221ull, 315236135880654ull}},
{{12332044080014528200ull, 504377817409046ull}},
{{6176286449269712237ull, 403502253927237ull}},
{{16009075603641500759ull, 322801803141789ull}},
{{14546474521600670245ull, 516482885026863ull}},
{{569133173054805226ull, 413186308021491ull}},
{{15212701797411485474ull, 330549046417192ull}},
{{9582927616890735465ull, 528878474267508ull}},
{{15045039722996409018ull, 423102779414006ull}},
{{8346682963655216891ull, 338482223531205ull}},
{{13354692741848347027ull, 541571557649928ull}},
{{18062451822962498268ull, 433257246119942ull}},
{{7071263828886177968ull, 346605796895954ull}},
{{245975681992153779ull, 554569275033527ull}},
{{11264826989819453993ull, 443655420026821ull}},
{{5322512777113652871ull, 354924336021457ull}},
{{15326056665916653266ull, 283939468817165ull}},
{{6074946591757093610ull, 454303150107465ull}},
{{4859957273405674888ull, 363442520085972ull}},
{{14956012262950270880ull, 290754016068777ull}},
{{9172224361752792115ull, 465206425710044ull}},
{{11027128304144144015ull, 372165140568035ull}},
{{8821702643315315212ull, 297732112454428ull}},
{{10425375414562594017ull, 476371379927085ull}},
{{8340300331650075213ull, 381097103941668ull}},
{{14050937894803880817ull, 304877683153334ull}},
{{11413454187460478338ull, 487804293045335ull}},
{{9130763349968382670ull, 390243434436268ull}},
{{14683308309458526782ull, 312194747549014ull}},
{{12425246850907911882ull, 499511596078423ull}},
{{17318895110210150152ull, 399609276862738ull}},
{{2787069643942389152ull, 319687421490191ull}},
{{15527357874533553613ull, 511499874384305ull}},
{{12421886299626842890ull, 409199899507444ull}},
{{13626857854443384635ull, 327359919605955ull}},
{{3356228493399863801ull, 523775871369529ull}},
{{6374331609461801364ull, 419020697095623ull}},
{{12478162917053261737ull, 335216557676498ull}},
{{16275711852543308457ull, 536346492282397ull}},
{{5641871852550826119ull, 429077193825918ull}},
{{11892195111524481542ull, 343261755060734ull}},
{{7959465734213439497ull, 549218808097175ull}},
{{6367572587370751598ull, 439375046477740ull}},
{{5094058069896601278ull, 351500037182192ull}},
{{11839841726576472368ull, 562400059491507ull}},
{{2093175751777357248ull, 449920047593206ull}},
{{16431935860389527091ull, 359936038074564ull}},
{{16834897503053531996ull, 287948830459651ull}},
{{1110394301692278932ull, 460718128735443ull}},
{{8267013070837643792ull, 368574502988354ull}},
{{10302959271412025356ull, 294859602390683ull}},
{{12795386019517330247ull, 471775363825093ull}},
{{17615006445097684844ull, 377420291060074ull}},
{{17781353970820058198ull, 301936232848059ull}},
{{17382119909086362148ull, 483097972556895ull}},
{{13905695927269089718ull, 386478378045516ull}},
{{7435207927073361451ull, 309182702436413ull}},
{{8206983868575467999ull, 494692323898261ull}},
{{2876238280118464076ull, 395753859118609ull}},
{{5990339438836681584ull, 316603087294887ull}},
{{13273891916880600858ull, 506564939671819ull}},
{{14308462348246391010ull, 405251951737455ull}},
{{11446769878597112808ull, 324201561389964ull}},
{{7246785361529649523ull, 518722498223943ull}},
{{13176125918707540264ull, 414977998579154ull}},
{{14230249549707942535ull, 331982398863323ull}},
{{632306391081246117ull, 531171838181318ull}},
{{7884542742348817540ull, 424937470545054ull}},
{{9996983008620964355ull, 339949976436043ull}},
{{12305823999051632645ull, 543919962297669ull}},
{{13534008013983216439ull, 435135969838135ull}},
{{10827206411186573151ull, 348108775870508ull}},
{{13634181443156606719ull, 556974041392813ull}},
{{18286042784009106021ull, 445579233114250ull}},
{{14628834227207284817ull, 356463386491400ull}},
{{11703067381765827853ull, 285170709193120ull}},
{{278163737115772950ull, 456273134708993ull}},
{{7601228619176439006ull, 365018507767194ull}},
{{9770331710083061528ull, 292014806213755ull}},
{{15632530736132898445ull, 467223689942008ull}},
{{1437978144680587786ull, 373778951953607ull}},
{{12218428959970201199ull, 299023161562885ull}},
{{1102742262242770302ull, 478437058500617ull}},
{{11950240254019947211ull, 382749646800493ull}},
{{16938889832699778415ull, 306199717440394ull}},
{{16034177288093914495ull, 489919547904631ull}},
{{9137993015733221273ull, 391935638323705ull}},
{{7310394412586577018ull, 313548510658964ull}},
{{628584615912792260ull, 501677617054343ull}},
{{7881565322214054454ull, 401342093643474ull}},
{{9994601072513153886ull, 321073674914779ull}},
{{4923315271795315249ull, 513717879863647ull}},
{{15006698661661983169ull, 410974303890917ull}},
{{4626661299845765888ull, 328779443112734ull}},
{{14781355709237046068ull, 526047108980374ull}},
{{15514433382131547178ull, 420837687184299ull}},
{{16100895520447148065ull, 336670149747439ull}},
{{14693386388489705935ull, 538672239595903ull}},
{{686662666566033778ull, 430937791676723ull}},
{{7928027762736647669ull, 344750233341378ull}},
{{8995495605636725947ull, 551600373346205ull}},
{{7196396484509380758ull, 441280298676964ull}},
{{9446466002349414929ull, 353024238941571ull}},
{{3867823987137621620ull, 282419391153257ull}},
{{9877867194162104916ull, 451871025845211ull}},
{{4212944940587773609ull, 361496820676169ull}},
{{7059704767212129211ull, 289197456540935ull}},
{{11295527627539406737ull, 462715930465496ull}},
{{5347073287289615066ull, 370172744372397ull}},
{{15345705074057423023ull, 296138195497917ull}},
{{9795732859524235544ull, 473821112796668ull}},
{{15215283917103209081ull, 379056890237334ull}},
{{15861575948424477588ull, 303245512189867ull}},
{{10621126258511522848ull, 485192819503788ull}},
{{15875598636293038925ull, 388154255603030ull}},
{{12700478909034431140ull, 310523404482424ull}},
{{9252719810229358855ull, 496837447171879ull}},
{{11091524662925397407ull, 397469957737503ull}},
{{16251917359824138572ull, 317975966190002ull}},
{{11245672516750980422ull, 508761545904004ull}},
{{12685886828142694661ull, 407009236723203ull}},
{{17527407091997976375ull, 325607389378562ull}},
{{13286456088229120908ull, 520971823005700ull}},
{{10629164870583296726ull, 416777458404560ull}},
{{8503331896466637381ull, 333421966723648ull}},
{{9915982219604709486ull, 533475146757837ull}},
{{554088146199946942ull, 426780117406270ull}},
{{443270516959957554ull, 341424093925016ull}},
{{11777279271361663056ull, 546278550280025ull}},
{{9421823417089330445ull, 437022840224020ull}},
{{7537458733671464356ull, 349618272179216ull}},
{{4681236344390522323ull, 559389235486746ull}},
{{55640260770507535ull, 447511388389397ull}},
{{11112558652842136998ull, 358009110711517ull}},
{{1511349292789888952ull, 286407288569214ull}},
{{9796856497947642969ull, 458251661710742ull}},
{{458787568874293729ull, 366601329368594ull}},
{{4056378869841345306ull, 293281063494875ull}},
{{6490206191746152490ull, 469249701591800ull}},
{{5192164953396921992ull, 375399761273440ull}},
{{4153731962717537594ull, 300319809018752ull}},
{{10335319955089970473ull, 480511694430003ull}},
{{15646953593555797025ull, 384409355544002ull}},
{{5138865245360816973ull, 307527484435202ull}},
{{11911533207319217481ull, 492043975096323ull}},
{{16907924195339194631ull, 393635180077058ull}},
{{2458292912045624735ull, 314908144061647ull}},
{{7622617474014909900ull, 503853030498635ull}},
{{6098093979211927920ull, 403082424398908ull}},
{{12257172812853362982ull, 322465939519126ull}},
{{12232778871081560125ull, 515945503230602ull}},
{{2407525467381427454ull, 412756402584482ull}},
{{12994066818130872932ull, 330205122067585ull}},
{{2343762835299845076ull, 528328195308137ull}},
{{12943056712465607030ull, 422662556246509ull}},
{{14043794184714395947ull, 338130044997207ull}},
{{7712675436575392223ull, 541008071995532ull}},
{{17238186793486044748ull, 432806457596425ull}},
{{13790549434788835798ull, 346245166077140ull}},
{{3618135021952585662ull, 553992265723425ull}},
{{2894508017562068529ull, 443193812578740ull}},
{{2315606414049654823ull, 354555050062992ull}},
{{12920531575465454828ull, 283644040050393ull}},
{{16983501706002817402ull, 453830464080629ull}},
{{17276150179544164245ull, 363064371264503ull}},
{{2752873699409600426ull, 290451497011603ull}},
{{715249104313450359ull, 464722395218565ull}},
{{572199283450760287ull, 371777916174852ull}},
{{11525805870986339199ull, 297422332939881ull}},
{{11062591764094322073ull, 475875732703810ull}},
{{8850073411275457658ull, 380700586163048ull}},
{{14458756358504186773ull, 304560468930438ull}},
{{997917285155236897ull, 487296750288702ull}},
{{11866380272349920487ull, 389837400230961ull}},
{{5803755403138026067ull, 311869920184769ull}},
{{16664706274504662353ull, 498991872295630ull}},
{{13331765019603729883ull, 399193497836504ull}},
{{14354760830424894229ull, 319354798269203ull}},
{{831524440228368828ull, 510967677230726ull}},
{{15422614811150336355ull, 408774141784580ull}},
{{12338091848920269084ull, 327019313427664ull}},
{{8672900514046699565ull, 523230901484263ull}},
{{14317018040721180298ull, 418584721187410ull}},
{{11453614432576944238ull, 334867776949928ull}},
{{14636434277381200459ull, 535788443119885ull}},
{{11709147421904960367ull, 428630754495908ull}},
{{16746015567007788940ull, 342904603596726ull}},
{{968183204019090041ull, 548647365754763ull}},
{{8153244192699092679ull, 438917892603810ull}},
{{6522595354159274143ull, 351134314083048ull}},
{{6746803751912928307ull, 561814902532877ull}},
{{16465489445756073615ull, 449451922026301ull}},
{{9483042741862948569ull, 359561537621041ull}},
{{3897085378748448532ull, 287649230096833ull}},
{{2545987791255607328ull, 460238768154933ull}},
{{9415487862488306508ull, 368191014523946ull}},
{{3843041475248734883ull, 294552811619157ull}},
{{9838215175139886137ull, 471284498590651ull}},
{{4181223325369998586ull, 377027598872521ull}},
{{18102373919263640162ull, 301622079098016ull}},
{{3138356567628451997ull, 482595326556827ull}},
{{13578731698328492567ull, 386076261245461ull}},
{{7173636543920883730ull, 308861008996369ull}},
{{409772026047682999ull, 494177614394191ull}},
{{15085212879805787692ull, 395342091515352ull}},
{{4689472674360809507ull, 316273673212282ull}},
{{11192505093719205535ull, 506037877139651ull}},
{{5264655260233454104ull, 404830301711721ull}},
{{522375393444852960ull, 323864241369377ull}},
{{4525149444253675060ull, 518182786191003ull}},
{{10998817184886760694ull, 414546228952802ull}},
{{1420356118425587909ull, 331636983162242ull}},
{{5961918604222850978ull, 530619173059587ull}},
{{15837581327604011752ull, 424495338447669ull}},
{{16359413876825119724ull, 339596270758135ull}},
{{7728318129210639943ull, 543354033213017ull}},
{{17250700947594242924ull, 434683226570413ull}},
{{2732514313849663370ull, 347746581256331ull}},
{{15440069346385192361ull, 556394530010129ull}},
{{16041404291850064212ull, 445115624008103ull}},
{{1765076989254320400ull, 356092499206483ull}},
{{8790759220887276966ull, 284873999365186ull}},
{{6686517123935822500ull, 455798398984298ull}},
{{12727911328632478646ull, 364638719187438ull}},
{{17561026692389803563ull, 291710975349950ull}},
{{9650898634114134086ull, 466737560559921ull}},
{{4031370092549396945ull, 373390048447937ull}},
{{14293142518265248526ull, 298712038758349ull}},
{{11800981584998666672ull, 477939262013359ull}},
{{13130134082740843661ull, 382351409610687ull}},
{{3125409636708854282ull, 305881127688550ull}},
{{5000655418734166851ull, 489409804301680ull}},
{{4000524334987333481ull, 391527843441344ull}},
{{6889768282731777108ull, 313222274753075ull}},
{{11023629252370843373ull, 501155639604920ull}},
{{8818903401896674698ull, 400924511683936ull}},
{{3365773906775429435ull, 320739609347149ull}},
{{12763935880324507743ull, 513183374955438ull}},
{{17589846333743426841ull, 410546699964350ull}},
{{14071877066994741473ull, 328437359971480ull}},
{{4068259233482034740ull, 525499775954369ull}},
{{6943956201527538115ull, 420399820763495ull}},
{{5555164961222030492ull, 336319856610796ull}},
{{1509566308471428142ull, 538111770577274ull}},
{{4897001861519052836ull, 430489416461819ull}},
{{7606950303957152592ull, 344391533169455ull}},
{{12171120486331444148ull, 551026453071128ull}},
{{17115594018548975964ull, 440821162456902ull}},
{{6313777585355360125ull, 352656929965522ull}},
{{16119068512510019070ull, 282125543972417ull}},
{{11033114361048389219ull, 451400870355868ull}},
{{16205189118322532021ull, 361120696284694ull}},
{{16653500109399935940ull, 288896557027755ull}},
{{8198856101330345889ull, 462234491244409ull}},
{{10248433695806187034ull, 369787592995527ull}},
{{820049327161128981ull, 295830074396422ull}},
{{5001427738199716693ull, 473328119034275ull}},
{{4001142190559773354ull, 378662495227420ull}},
{{3200913752447818683ull, 302929996181936ull}},
{{16189508448142240863ull, 484687993891097ull}},
{{5572909129029972044ull, 387750395112878ull}},
{{11837024932707798281ull, 310200316090302ull}},
{{4181844633364835958ull, 496320505744484ull}},
{{7034824521433779089ull, 397056404595587ull}},
{{16695906061372754241ull, 317645123676469ull}},
{{15645403253970675816ull, 508232197882351ull}},
{{8826973788434630330ull, 406585758305881ull}},
{{3372230216005793940ull, 325268606644705ull}},
{{5395568345609270305ull, 520429770631528ull}},
{{11695152305971236890ull, 416343816505222ull}},
{{1977424215293168866ull, 333075053204178ull}},
{{17921274003436711478ull, 532920085126684ull}},
{{18026368017491279506ull, 426336068101347ull}},
{{7042396784509202958ull, 341068854481078ull}},
{{7578486040472814410ull, 545710167169725ull}},
{{6062788832378251528ull, 436568133735780ull}},
{{4850231065902601222ull, 349254506988624ull}},
{{15139067334927982602ull, 558807211181798ull}},
{{1043207423716655112ull, 447045768945439ull}},
{{4523914753715234413ull, 357636615156351ull}},
{{18376527061939828823ull, 286109292125080ull}},
{{10955699225394174501ull, 457774867400129ull}},
{{12453908195057249924ull, 366219893920103ull}},
{{17341824185529620585ull, 292975915136082ull}},
{{12989523437879751644ull, 468761464217732ull}},
{{3012921120819980669ull, 375009171374186ull}},
{{17167732155623625828ull, 300007337099348ull}},
{{5332278560546339385ull, 480011739358958ull}},
{{11644520477920892155ull, 384009391487166ull}},
{{5626267567594803400ull, 307207513189733ull}},
{{5312679293409775118ull, 491532021103573ull}},
{{11628841064211640740ull, 393225616882858ull}},
{{16681770480853133239ull, 314580493506286ull}},
{{865391066171640920ull, 503328789610059ull}},
{{4381661667679223059ull, 402663031688047ull}},
{{14573375778369109417ull, 322130425350437ull}},
{{8560005986422933774ull, 515408680560700ull}},
{{6848004789138347019ull, 412326944448560ull}},
{{5478403831310677615ull, 329861555558848ull}},
{{5076097315355173862ull, 527778488894157ull}},
{{15128924296509870059ull, 422222791115325ull}},
{{12103139437207896047ull, 337778232892260ull}},
{{918279025823082059ull, 540445172627617ull}},
{{11802669664884196617ull, 432356138102093ull}},
{{16820833361391177940ull, 345884910481674ull}},
{{15845286934000153734ull, 553415856770679ull}},
{{16365578361942033311ull, 442732685416543ull}},
{{2024416245327895679ull, 354186148333235ull}},
{{1619532996262316543ull, 283348918666588ull}},
{{17348648052987347762ull, 453358269866540ull}},
{{13878918442389878209ull, 362686615893232ull}},
{{3724437124428081921ull, 290149292714586ull}},
{{17027145843310662043ull, 464238868343337ull}},
{{6243019045164708988ull, 371391094674670ull}},
{{4994415236131767191ull, 297112875739736ull}},
{{612366748327006859ull, 475380601183578ull}},
{{7868591028145426133ull, 380304480946862ull}},
{{17362919266742071876ull, 304243584757489ull}},
{{16712624382561584033ull, 486789735611983ull}},
{{2302053061823536256ull, 389431788489587ull}},
{{12909688893684559975ull, 311545430791669ull}},
{{9587455785669564990ull, 498472689266671ull}},
{{3980615813793741669ull, 398778151413337ull}},
{{14252539095260724304ull, 319022521130669ull}},
{{11736016108191427918ull, 510436033809071ull}},
{{5699464071811232011ull, 408348827047257ull}},
{{15627617701674716578ull, 326679061637805ull}},
{{6557444248969994910ull, 522686498620489ull}},
{{8935304213917906251ull, 418149198896391ull}},
{{3458894556392414677ull, 334519359117113ull}},
{{1844882475485953161ull, 535230974587381ull}},
{{16233301239356403821ull, 428184779669904ull}},
{{16675989806227033380ull, 342547823735923ull}},
{{4545490801511791469ull, 548076517977478ull}},
{{11015090270693253822ull, 438461214381982ull}},
{{1433374587070782411ull, 350768971505586ull}},
{{13361445783538982827ull, 561230354408937ull}},
{{3310458997347365615ull, 448984283527150ull}},
{{2648367197877892492ull, 359187426821720ull}},
{{2118693758302313994ull, 287349941457376ull}},
{{14457956457509433360ull, 459759906331801ull}},
{{7877016351265636365ull, 367807925065441ull}},
{{2612264266270598768ull, 294246340052353ull}},
{{490274011291047706ull, 470794144083765ull}},
{{392219209032838165ull, 376635315267012ull}},
{{11381821811452001502ull, 301308252213609ull}},
{{7142868454097471433ull, 482093203541775ull}},
{{5714294763277977146ull, 385674562833420ull}},
{{4571435810622381717ull, 308539650266736ull}},
{{18382343741221541717ull, 493663440426777ull}},
{{7327177363493412727ull, 394930752341422ull}},
{{16929788335020461151ull, 315944601873137ull}},
{{12330266077065096550ull, 505511362997020ull}},
{{9864212861652077240ull, 404409090397616ull}},
{{4202021474579751468ull, 323527272318093ull}},
{{3033885544585692026ull, 517643635708949ull}},
{{6116457250410463944ull, 414114908567159ull}},
{{8582514615070281478ull, 331291926853727ull}},
{{17421372198854360689ull, 530067082965963ull}},
{{2869051314857757582ull, 424053666372771ull}},
{{17052636310853847358ull, 339242933098216ull}},
{{1458776394172783511ull, 542788692957147ull}},
{{12235067559563957778ull, 434230954365717ull}},
{{2409356418167345576ull, 347384763492574ull}},
{{11233667898551573568ull, 555815621588118ull}},
{{16365631948325079501ull, 444652497270494ull}},
{{16781854373401973924ull, 355721997816395ull}},
{{13425483498721579139ull, 284577598253116ull}},
{{14102075968470705976ull, 455324157204986ull}},
{{7592311960034654458ull, 364259325763989ull}},
{{9763198382769633889ull, 291407460611191ull}},
{{8242419782947593577ull, 466251936977906ull}},
{{2904587011616164538ull, 373001549582325ull}},
{{2323669609292931630ull, 298401239665860ull}},
{{3717871374868690609ull, 477441983465376ull}},
{{17731692358862593780ull, 381953586772300ull}},
{{14185353887090075024ull, 305562869417840ull}},
{{4249822145634568422ull, 488900591068545ull}},
{{3399857716507654738ull, 391120472854836ull}},
{{17477281432173765083ull, 312896378283868ull}},
{{5827557403026562194ull, 500634205254190ull}},
{{4662045922421249755ull, 400507364203352ull}},
{{14797683182162730773ull, 320405891362681ull}},
{{16297595461976548591ull, 512649426180290ull}},
{{13038076369581238873ull, 410119540944232ull}},
{{3051763466181170452ull, 328095632755386ull}},
{{15950867990115603693ull, 524953012408617ull}},
{{5381996762608662308ull, 419962409926894ull}},
{{7994946224828840169ull, 335969927941515ull}},
{{12791913959726144271ull, 537551884706424ull}},
{{13922879982522825740ull, 430041507765139ull}},
{{14827652800760170915ull, 344033206212111ull}},
{{16345546851732452818ull, 550453129939378ull}},
{{2008391037160231285ull, 440362503951503ull}},
{{8985410459212005674ull, 352290003161202ull}},
{{18256374811595335509ull, 281832002528961ull}},
{{3384757995359164552ull, 450931204046339ull}},
{{6397155211029241965ull, 360744963237071ull}},
{{1428375354081483248ull, 288595970589657ull}},
{{5974749381272283521ull, 461753552943451ull}},
{{1090450690275916493ull, 369402842354761ull}},
{{15629755811188374487ull, 295522273883808ull}},
{{2871516409449937241ull, 472835638214094ull}},
{{5986561942301860116ull, 378268510571275ull}},
{{4789249553841488093ull, 302614808457020ull}},
{{7662799286146380949ull, 484183693531232ull}},
{{17198285873142835728ull, 387346954824985ull}},
{{13758628698514268583ull, 309877563859988ull}},
{{18324457102880919409ull, 495804102175981ull}},
{{10970216867562825204ull, 396643281740785ull}},
{{8776173494050260163ull, 317314625392628ull}},
{{10352528775738505938ull, 507703400628205ull}},
{{8282023020590804750ull, 406162720502564ull}},
{{10314967231214554123ull, 324930176402051ull}},
{{9125249940459465951ull, 519888282243282ull}},
{{18368246396593303731ull, 415910625794625ull}},
{{14694597117274642984ull, 332728500635700ull}},
{{5064611313929877159ull, 532365601017121ull}},
{{362340236401991404ull, 425892480813697ull}},
{{11357918633347324093ull, 340713984650957ull}},
{{3415274554388077256ull, 545142375441532ull}},
{{13800266087736192774ull, 436113900353225ull}},
{{11040212870188954219ull, 348891120282580ull}},
{{17664340592302326751ull, 558225792452128ull}},
{{3063426029616130431ull, 446580633961703ull}},
{{9829438453176724991ull, 357264507169362ull}},
{{484853133057559347ull, 285811605735490ull}},
{{775765012892094955ull, 457298569176784ull}},
{{4309960825055586287ull, 365838855341427ull}},
{{14516015104270199999ull, 292671084273141ull}},
{{15846926537348499352ull, 468273734837026ull}},
{{8988192415136889159ull, 374618987869621ull}},
{{3501205117367601004ull, 299695190295697ull}},
{{9291277002530071929ull, 479512304473115ull}},
{{7433021602024057543ull, 383609843578492ull}},
{{17014463725844977004ull, 306887874862793ull}},
{{5087049072900501268ull, 491020599780470ull}},
{{4069639258320401014ull, 392816479824376ull}},
{{18013106665623962104ull, 314253183859500ull}},
{{10374226591288787751ull, 502805094175201ull}},
{{4610032458289119877ull, 402244075340161ull}},
{{18445421225598937194ull, 321795260272128ull}},
{{7376581072506837572ull, 514872416435406ull}},
{{2211916043263559735ull, 411897933148325ull}},
{{1769532834610847788ull, 329518346518660ull}},
{{2831252535377356460ull, 527229354429856ull}},
{{17022397287269526461ull, 421783483543884ull}},
{{17307266644557531492ull, 337426786835107ull}},
{{12934231372324409095ull, 539882858936172ull}},
{{2968687468375706629ull, 431906287148938ull}},
{{9753647604184385950ull, 345525029719150ull}},
{{15605836166695017520ull, 552840047550640ull}},
{{12484668933356014016ull, 442272038040512ull}},
{{2609037517200990566ull, 353817630432410ull}},
{{2087230013760792453ull, 283054104345928ull}},
{{18096963280984909217ull, 452886566953484ull}},
{{18166919439529837697ull, 362309253562787ull}},
{{7154837922140049511ull, 289847402850230ull}},
{{11447740675424079218ull, 463755844560368ull}},
{{16536890169823084021ull, 371004675648294ull}},
{{16918860950600377540ull, 296803740518635ull}},
{{8623433447251052448ull, 474885984829817ull}},
{{17966793202026572928ull, 379908787863853ull}},
{{3305388117395527373ull, 303927030291083ull}},
{{1599272173090933473ull, 486283248465733ull}},
{{8658115367956567425ull, 389026598772586ull}},
{{3237143479623343617ull, 311221279018069ull}},
{{12558127196881170433ull, 497954046428910ull}},
{{10046501757504936346ull, 398363237143128ull}},
{{15415899035487769723ull, 318690589714502ull}},
{{9908043197812790265ull, 509904943543204ull}},
{{11615783372992142535ull, 407923954834563ull}},
{{16671324327877534674ull, 326339163867650ull}},
{{8227374850894503863ull, 522142662188241ull}},
{{2892551065973692767ull, 417714129750593ull}},
{{9692738482262774860ull, 334171303800474ull}},
{{4440335127394708807ull, 534674086080759ull}},
{{7241616916657677369ull, 427739268864607ull}},
{{16861339977551872864ull, 342191415091685ull}},
{{8531399890373444967ull, 547506264146697ull}},
{{17893166356524486943ull, 438005011317357ull}},
{{6935835455735768908ull, 350404009053886ull}},
{{3718639099693409607ull, 560646414486218ull}},
{{10353608909238548332ull, 448517131588974ull}},
{{11972235942132748989ull, 358813705271179ull}},
{{13267137568448109514ull, 287050964216943ull}},
{{17538071294775064900ull, 459281542747109ull}},
{{17719805850561962243ull, 367425234197687ull}},
{{6797147050965749148ull, 293940187358150ull}},
{{10875435281545198637ull, 470304299773040ull}},
{{8700348225236158909ull, 376243439818432ull}},
{{18028325024414658097ull, 300994751854745ull}},
{{10398575965353901339ull, 481591602967593ull}},
{{15697558401766941718ull, 385273282374074ull}},
{{16247395536155463697ull, 308218625899259ull}},
{{14927786413623010946ull, 493149801438815ull}},
{{11942229130898408757ull, 394519841151052ull}},
{{2175085675234906359ull, 315615872920842ull}},
{{7169485895117760498ull, 504985396673347ull}},
{{16803635160319939368ull, 403988317338677ull}},
{{6064210498772130848ull, 323190653870942ull}},
{{13392085612777319680ull, 517105046193507ull}},
{{3334970860738035098ull, 413684036954806ull}},
{{17425371947558069371ull, 330947229563844ull}},
{{16812548671867180024ull, 529515567302151ull}},
{{9760690122751833696ull, 423612453841721ull}},
{{4119203283459556633ull, 338889963073377ull}},
{{10280074068277200937ull, 542223940917403ull}},
{{15602756884105581396ull, 433779152733922ull}},
{{5103507877800644470ull, 347023322187138ull}},
{{4476263789739120829ull, 555237315499421ull}},
{{18338406290758937956ull, 444189852399536ull}},
{{10981376217865240042ull, 355351881919629ull}},
{{12474449789034102356ull, 284281505535703ull}},
{{16269770847712653447ull, 454850408857125ull}},
{{13015816678170122758ull, 363880327085700ull}},
{{10412653342536098206ull, 291104261668560ull}},
{{16660245348057757130ull, 465766818669696ull}},
{{9638847463704295381ull, 372613454935757ull}},
{{332380341479615658ull, 298090763948606ull}},
{{11599854990593116023ull, 476945222317769ull}},
{{12969232807216403141ull, 381556177854215ull}},
{{10375386245773122513ull, 305244942283372ull}},
{{1843222734269354728ull, 488391907653396ull}},
{{16231973446383125075ull, 390713526122716ull}},
{{9296229942364589737ull, 312570820898173ull}},
{{11184619093041433256ull, 500113313437077ull}},
{{1568997644949325959ull, 400090650749662ull}},
{{12323244560185191736ull, 320072520599729ull}},
{{8649144852070575809ull, 512116032959567ull}},
{{17987362325882191617ull, 409692826367653ull}},
{{3321843416480022324ull, 327754261094123ull}},
{{1625600651626125395ull, 524406817750597ull}},
{{12368526965526631285ull, 419525454200477ull}},
{{2516123942937484382ull, 335620363360382ull}},
{{7715147123441885334ull, 536992581376611ull}},
{{2482768884011597944ull, 429594065101289ull}},
{{5675563921951188678ull, 343675252081031ull}},
{{1702204645638081239ull, 549880403329650ull}},
{{1361763716510464991ull, 439904322663720ull}},
{{1089410973208371993ull, 351923458130976ull}},
{{15628924037534338887ull, 281538766504780ull}},
{{6559534386345390604ull, 450462026407649ull}},
{{8936976323818222806ull, 360369621126119ull}},
{{10838929873796488568ull, 288295696900895ull}},
{{17342287798074381709ull, 461273115041432ull}},
{{6495132608975684721ull, 369018492033146ull}},
{{1506757272438637453ull, 295214793626517ull}},
{{6100160450643730249ull, 472343669802427ull}},
{{15948174804740715168ull, 377874935841941ull}},
{{9069191029050661811ull, 302299948673553ull}},
{{10821356831739148575ull, 483679917877685ull}},
{{8657085465391318860ull, 386943934302148ull}},
{{14304366001796875735ull, 309555147441718ull}},
{{750892714423539236ull, 495288235906750ull}},
{{600714171538831389ull, 396230588725400ull}},
{{480571337231065111ull, 316984470980320ull}},
{{768914139569704178ull, 507175153568512ull}},
{{11683177755881494312ull, 405740122854809ull}},
{{13035891019447105773ull, 324592098283847ull}},
{{6100030372147727944ull, 519347357254156ull}},
{{1190675482976272032ull, 415477885803325ull}},
{{952540386381017625ull, 332382308642660ull}},
{{1524064618209628201ull, 531811693828256ull}},
{{15976646953535343853ull, 425449355062604ull}},
{{16470666377570185406ull, 340359484050083ull}},
{{4216973315660834710ull, 544575174480134ull}},
{{7062927467270578091ull, 435660139584107ull}},
{{16718388418042193443ull, 348528111667285ull}},
{{8302677395157957892ull, 557644978667657ull}},
{{17710188360352097283ull, 446115982934125ull}},
{{14168150688281677827ull, 356892786347300ull}},
{{11334520550625342261ull, 285514229077840ull}},
{{18135232881000547618ull, 456822766524544ull}},
{{18197535119542348418ull, 365458213219635ull}},
{{14558028095633878734ull, 292366570575708ull}},
{{1156752064562744035ull, 467786512921134ull}},
{{4614750466392105551ull, 374229210336907ull}},
{{14759846817339415411ull, 299383368269525ull}},
{{5169010834033513041ull, 479013389231241ull}},
{{445859852484900110ull, 383210711384993ull}},
{{7735385511471740734ull, 306568569107994ull}},
{{1308570374129054205ull, 490509710572791ull}},
{{15804251558270884657ull, 392407768458232ull}},
{{5264703617132887079ull, 313926214766586ull}},
{{1044828157928798680ull, 502281943626538ull}},
{{8214560155826859591ull, 401825554901230ull}},
{{6571648124661487672ull, 321460443920984ull}},
{{17893334628942200923ull, 514336710273574ull}},
{{18004016517895671061ull, 411469368218859ull}},
{{18092562029058447172ull, 329175494575087ull}},
{{14190703987525874183ull, 526680791320140ull}},
{{11352563190020699346ull, 421344633056112ull}},
{{1703352922532738830ull, 337075706444890ull}},
{{2725364676052382129ull, 539321130311824ull}},
{{5869640555583816026ull, 431456904249459ull}},
{{8385061259208963144ull, 345165523399567ull}},
{{17105446829476251354ull, 552264837439307ull}},
{{6305659834097180437ull, 441811869951446ull}},
{{1355179052535834026ull, 353449495961157ull}},
{{12152189686254398190ull, 282759596768925ull}},
{{996759424297485489ull, 452415354830281ull}},
{{15554802798405629684ull, 361932283864224ull}},
{{16133191053466414070ull, 289545827091379ull}},
{{14745059241320531543ull, 463273323346207ull}},
{{4417349763572604588ull, 370618658676966ull}},
{{18291275069825724963ull, 296494926941572ull}},
{{14508644852753518648ull, 474391883106516ull}},
{{7917567067460904595ull, 379513506485213ull}},
{{13712751283452544323ull, 303610805188170ull}},
{{3493657979814519300ull, 485777288301073ull}},
{{10173624013335436087ull, 388621830640858ull}},
{{15517596840152169516ull, 310897464512686ull}},
{{17449457314759650579ull, 497435943220298ull}},
{{2891519407581989493ull, 397948754576239ull}},
{{6002564340807501918ull, 318359003660991ull}},
{{2225405315808182422ull, 509374405857586ull}},
{{16537719511614187231ull, 407499524686068ull}},
{{2162129165065618815ull, 325999619748855ull}},
{{3459406664104990104ull, 521599391598168ull}},
{{10146222960767812729ull, 417279513278534ull}},
{{11806327183356160507ull, 333823610622827ull}},
{{4132728234402215518ull, 534117776996524ull}},
{{6995531402263682738ull, 427294221597219ull}},
{{9285773936552856513ull, 341835377277775ull}},
{{14857238298484570421ull, 546936603644440ull}},
{{11885790638787656337ull, 437549282915552ull}},
{{2129934881546304423ull, 350039426332442ull}},
{{7097244625215997401ull, 560063082131907ull}},
{{16745842144398528890ull, 448050465705525ull}},
{{13396673715518823112ull, 358440372564420ull}},
{{10717338972415058489ull, 286752298051536ull}},
{{9769044726380272937ull, 458803676882458ull}},
{{15193933410588038996ull, 367042941505966ull}},
{{8465797913728520873ull, 293634353204773ull}},
{{9855927847223723074ull, 469814965127637ull}},
{{506044648295157813ull, 375851972102110ull}},
{{404835718636126250ull, 300681577681688ull}},
{{15405132408785443294ull, 481090524290700ull}},
{{12324105927028354635ull, 384872419432560ull}},
{{9859284741622683708ull, 307897935546048ull}},
{{12085506771854383610ull, 492636696873677ull}},
{{2289707787999686241ull, 394109357498942ull}},
{{12899812674625479962ull, 315287485999153ull}},
{{16950351464658857617ull, 504459977598645ull}},
{{13560281171727086093ull, 403567982078916ull}},
{{7158876122639758551ull, 322854385663133ull}},
{{7764852981481703359ull, 516567017061013ull}},
{{13590580014669183334ull, 413253613648810ull}},
{{10872464011735346667ull, 330602890919048ull}},
{{13706593604034644344ull, 528964625470477ull}},
{{3586577253743894829ull, 423171700376382ull}},
{{13937308247220846833ull, 338537360301105ull}},
{{3852949121843803316ull, 541659776481769ull}},
{{6771708112216952976ull, 433327821185415ull}},
{{5417366489773562381ull, 346662256948332ull}},
{{12357135198379610133ull, 554659611117331ull}},
{{6196359343961777783ull, 443727688893865ull}},
{{4957087475169422226ull, 354982151115092ull}},
{{15033716424361268751ull, 283985720892073ull}},
{{1917853390526568062ull, 454377153427318ull}},
{{8912980341905075096ull, 363501722741854ull}},
{{10819733088265970400ull, 290801378193483ull}},
{{13622224126483642317ull, 465282205109573ull}},
{{18276476930670734500ull, 372225764087658ull}},
{{3553135100310856630ull, 297780611270127ull}},
{{9374364975239280932ull, 476448978032203ull}},
{{14878189609675245392ull, 381159182425762ull}},
{{4523854058256375667ull, 304927345940610ull}},
{{7238166493210201067ull, 487883753504976ull}},
{{2101184379826250531ull, 390307002803981ull}},
{{16438342762828641717ull, 312245602243184ull}},
{{15233301976300095778ull, 499592963589095ull}},
{{12186641581040076622ull, 399674370871276ull}},
{{6059964450090150975ull, 319739496697021ull}},
{{2317245490660420913ull, 511583194715234ull}},
{{5543145207270247054ull, 409266555772187ull}},
{{15502562610041928612ull, 327413244617749ull}},
{{13736053731841354811ull, 523861191388399ull}},
{{14678191800214994172ull, 419088953110719ull}},
{{15431902254913905660ull, 335271162488575ull}},
{{6244299534152697441ull, 536433859981721ull}},
{{1306090812580247629ull, 429147087985377ull}},
{{12112919094289929073ull, 343317670388301ull}},
{{12001972921380065871ull, 549308272621282ull}},
{{2222880707620232050ull, 439446618097026ull}},
{{16535699825063826933ull, 351557294477620ull}},
{{8010375646392571477ull, 562491671164193ull}},
{{13786998146597877828ull, 449993336931354ull}},
{{14718947332020212585ull, 359994669545083ull}},
{{707111421390439099ull, 287995735636067ull}},
{{4820727088966612881ull, 460793177017707ull}},
{{14924628115399021274ull, 368634541614165ull}},
{{11939702492319217019ull, 294907633291332ull}},
{{4346128728743105939ull, 471852213266132ull}},
{{14544949427220215720ull, 377481770612905ull}},
{{11635959541776172576ull, 301985416490324ull}},
{{7549488822616145153ull, 483176666384519ull}},
{{9728939872834826445ull, 386541333107615ull}},
{{7783151898267861156ull, 309233066486092ull}},
{{16142391851970488173ull, 494772906377747ull}},
{{5535215852092569892ull, 395818325102198ull}},
{{11806870311157876560ull, 316654660081758ull}},
{{15201643683110692173ull, 506647456130813ull}},
{{1093268502262822769ull, 405317964904651ull}},
{{15632010060777899508ull, 324254371923720ull}},
{{6564472023535087597ull, 518806995077953ull}},
{{12630275248311890724ull, 415045596062362ull}},
{{2725522569165691932ull, 332036476849890ull}},
{{4360836110665107092ull, 531258362959824ull}},
{{7178017703273995997ull, 425006690367859ull}},
{{9431762977361107120ull, 340005352294287ull}},
{{333425504810130100ull, 544008563670860ull}},
{{266740403848104080ull, 435206850936688ull}},
{{7592089952562303910ull, 348165480749350ull}},
{{12147343924099686257ull, 557064769198960ull}},
{{9717875139279749005ull, 445651815359168ull}},
{{15152997740907619851ull, 356521452287334ull}},
{{15811747007468006204ull, 285217161829867ull}},
{{10541399952981168633ull, 456347458927788ull}},
{{15811817591868755553ull, 365077967142230ull}},
{{12649454073495004442ull, 292062373713784ull}},
{{9171080073366276138ull, 467299797942055ull}},
{{7336864058693020911ull, 373839838353644ull}},
{{9558840061696327052ull, 299071870682915ull}},
{{15294144098714123283ull, 478514993092664ull}},
{{15924664093713208949ull, 382811994474131ull}},
{{9050382460228656836ull, 306249595579305ull}},
{{14480611936365850938ull, 489999352926888ull}},
{{516443104866949781ull, 391999482341511ull}},
{{15170549742861201117ull, 313599585873208ull}},
{{2136786700126459849ull, 501759337397134ull}},
{{5398778174843078202ull, 401407469917707ull}},
{{15387068984100193531ull, 321125975934165ull}},
{{6172566300850758034ull, 513801561494665ull}},
{{4938053040680606427ull, 411041249195732ull}},
{{15018488876770216111ull, 328832999356585ull}},
{{5582838129122794162ull, 526132798970537ull}},
{{15534316947523966299ull, 420906239176429ull}},
{{16116802372761083363ull, 336724991341143ull}},
{{3650790907966271441ull, 538759986145830ull}},
{{2920632726373017153ull, 431007988916664ull}},
{{6025854995840324045ull, 344806391133331ull}},
{{2262670363860697827ull, 551690225813330ull}},
{{1810136291088558261ull, 441352180650664ull}},
{{5137457847612756932ull, 353081744520531ull}},
{{420617463348295222ull, 282465395616425ull}},
{{672987941357272356ull, 451944632986280ull}},
{{538390353085817885ull, 361555706389024ull}},
{{4120061097210564631ull, 289244565111219ull}},
{{13970795385020724056ull, 462791304177950ull}},
{{11176636308016579245ull, 370233043342360ull}},
{{8941309046413263396ull, 296186434673888ull}},
{{10616745659519311110ull, 473898295478221ull}},
{{4804047712873538565ull, 379118636382577ull}},
{{14911284614524561821ull, 303294909106061ull}},
{{16479357753755478268ull, 485271854569698ull}},
{{2115439758778651645ull, 388217483655759ull}},
{{5381700621764831639ull, 310573986924607ull}},
{{12300069809565640946ull, 496918379079371ull}},
{{6150707032910602433ull, 397534703263497ull}},
{{15988612070554212916ull, 318027762610797ull}},
{{10824384053919099374ull, 508844420177276ull}},
{{4970158428393369176ull, 407075536141821ull}},
{{286777927972785017ull, 325660428913457ull}},
{{4148193499498366351ull, 521056686261531ull}},
{{18075950058566334373ull, 416845349009224ull}},
{{18150108861594977822ull, 333476279207379ull}},
{{17972127734326233546ull, 533562046731807ull}},
{{6999004557977166190ull, 426849637385446ull}},
{{1909854831639822629ull, 341479709908357ull}},
{{6745116545365626529ull, 546367535853371ull}},
{{1706744421550590900ull, 437094028682697ull}},
{{12433441981466203690ull, 349675222946157ull}},
{{5136111911378284611ull, 559480356713852ull}},
{{15176935973328358658ull, 447584285371081ull}},
{{8452199963920776603ull, 358067428296865ull}},
{{6761759971136621283ull, 286453942637492ull}},
{{14508164768560504376ull, 458326308219987ull}},
{{4227834185364582854ull, 366661046575990ull}},
{{3382267348291666283ull, 293328837260792ull}},
{{9100976572008576376ull, 469326139617267ull}},
{{18348827701832592071ull, 375460911693813ull}},
{{3611015717240342687ull, 300368729355051ull}},
{{16845671591810279269ull, 480589966968081ull}},
{{9787188458706313092ull, 384471973574465ull}},
{{7829750766965050473ull, 307577578859572ull}},
{{16216950041885991081ull, 492124126175315ull}},
{{12973560033508792865ull, 393699300940252ull}},
{{3000150397323213645ull, 314959440752202ull}},
{{8489589450459052156ull, 503935105203523ull}},
{{14170369189851062371ull, 403148084162818ull}},
{{268248907655118927ull, 322518467330255ull}},
{{429198252248190284ull, 516029547728408ull}},
{{7722056231282372873ull, 412823638182726ull}},
{{2488296170283987975ull, 330258910546181ull}},
{{15049320316680111730ull, 528414256873889ull}},
{{15728805068085999707ull, 422731405499111ull}},
{{8893695239726889443ull, 338185124399289ull}},
{{3161865939337292139ull, 541096199038863ull}},
{{9908190380953654357ull, 432876959231090ull}},
{{7926552304762923486ull, 346301567384872ull}},
{{16371832502362587901ull, 554082507815795ull}},
{{13097466001890070320ull, 443266006252636ull}},
{{6788623986770145933ull, 354612805002109ull}},
{{9120248004158027070ull, 283690244001687ull}},
{{18281745621394753635ull, 453904390402699ull}},
{{18314745311857713231ull, 363123512322159ull}},
{{18341145064228080908ull, 290498809857727ull}},
{{14588436843797288160ull, 464798095772364ull}},
{{15360098289779740851ull, 371838476617891ull}},
{{8598729817081882358ull, 297470781294313ull}},
{{10068618892589101449ull, 475953250070901ull}},
{{4365546299329370836ull, 380762600056721ull}},
{{18249832298431137962ull, 304610080045376ull}},
{{3374289974296448476ull, 487376128072603ull}},
{{10078129608920979427ull, 389900902458082ull}},
{{683806057652962895ull, 311920721966466ull}},
{{12162136136470471603ull, 499073155146345ull}},
{{9729708909176377282ull, 399258524117076ull}},
{{4094418312599191502ull, 319406819293661ull}},
{{17619115744384437374ull, 511050910869857ull}},
{{6716594966023729252ull, 408840728695886ull}},
{{1683927158077073079ull, 327072582956709ull}},
{{10072981082407137572ull, 523316132730734ull}},
{{11747733680667620381ull, 418652906184587ull}},
{{2019489315050275658ull, 334922324947670ull}},
{{3231182904080441054ull, 535875719916272ull}},
{{13652992767490083812ull, 428700575933017ull}},
{{3543696584508246403ull, 342960460746414ull}},
{{13048612164697014892ull, 548736737194262ull}},
{{3060192102273791267ull, 438989389755410ull}},
{{2448153681819033014ull, 351191511804328ull}},
{{227697076168542499ull, 561906418886925ull}},
{{182157660934833999ull, 449525135109540ull}},
{{145726128747867199ull, 359620108087632ull}},
{{11184627347224024729ull, 287696086470105ull}},
{{17895403755558439566ull, 460313738352168ull}},
{{3248276560221020683ull, 368250990681735ull}},
{{2598621248176816547ull, 294600792545388ull}},
{{468445182340996152ull, 471361268072621ull}},
{{15132151404840438214ull, 377089014458096ull}},
{{8416372309130440248ull, 301671211566477ull}},
{{17155544509350614720ull, 482673938506363ull}},
{{2656389163254760807ull, 386139150805091ull}},
{{16882506589571449938ull, 308911320644072ull}},
{{12254615284346678608ull, 494258113030516ull}},
{{6114343412735432563ull, 395406490424413ull}},
{{12270172359672166697ull, 316325192339530ull}},
{{1185531701765915099ull, 506120307743249ull}},
{{4637774176154642403ull, 404896246194599ull}},
{{7399568155665624245ull, 323916996955679ull}},
{{771262604839267823ull, 518267195129087ull}},
{{11685056528097145228ull, 414613756103269ull}},
{{13037394037219626505ull, 331691004882615ull}},
{{2413086385841850793ull, 530705607812185ull}},
{{1930469108673480634ull, 424564486249748ull}},
{{8923072916422605154ull, 339651588999798ull}},
{{10587567851534257923ull, 543442542399677ull}},
{{1091356651743585692ull, 434754033919742ull}},
{{11941131765620599523ull, 347803227135793ull}},
{{15416462010251048914ull, 556485163417269ull}},
{{16022518422942749454ull, 445188130733815ull}},
{{12818014738354199563ull, 356150504587052ull}},
{{2875714161199539004ull, 284920403669642ull}},
{{8290491472661172730ull, 455872645871427ull}},
{{17700439622354669154ull, 364698116697141ull}},
{{10471002883141825000ull, 291758493357713ull}},
{{13064255798285009676ull, 466813589372341ull}},
{{6762055823886097418ull, 373450871497873ull}},
{{12788342288592698581ull, 298760697198298ull}},
{{16771998847006407406ull, 478017115517277ull}},
{{6038901448121305278ull, 382413692413822ull}},
{{15899167602722775192ull, 305930953931057ull}},
{{10681272905388799015ull, 489489526289692ull}},
{{1166320694827218565ull, 391591621031754ull}},
{{4622405370603685175ull, 313273296825403ull}},
{{3706499778223985958ull, 501237274920645ull}},
{{2965199822579188766ull, 400989819936516ull}},
{{17129555117030992306ull, 320791855949212ull}},
{{12649892928281946397ull, 513266969518740ull}},
{{10119914342625557117ull, 410613575614992ull}},
{{717233844616625047ull, 328490860491994ull}},
{{8526271780870420722ull, 525585376787190ull}},
{{6821017424696336578ull, 420468301429752ull}},
{{16524860383982800232ull, 336374641143801ull}},
{{614334911179108109ull, 538199425830083ull}},
{{7870165558427107133ull, 430559540664066ull}},
{{2606783631999775383ull, 344447632531253ull}},
{{481504996457730290ull, 551116212050005ull}},
{{385203997166184232ull, 440892969640004ull}},
{{3997512012474857709ull, 352714375712003ull}},
{{10576707239463706813ull, 282171500569602ull}},
{{2165336324174289609ull, 451474400911364ull}},
{{5421617874081342010ull, 361179520729091ull}},
{{647945484523163285ull, 288943616583273ull}},
{{15794108034204702549ull, 462309786533236ull}},
{{8945937612621851716ull, 369847829226589ull}},
{{10846098904839391696ull, 295878263381271ull}},
{{9975060618259206067ull, 473405221410034ull}},
{{11669397309349275177ull, 378724177128027ull}},
{{1956820217995599495ull, 302979341702422ull}},
{{6820261163534869515ull, 484766946723875ull}},
{{5456208930827895612ull, 387813557379100ull}},
{{4364967144662316490ull, 310250845903280ull}},
{{6983947431459706384ull, 496401353445248ull}},
{{12965855574651585753ull, 397121082756198ull}},
{{17751382089205089249ull, 317696866204958ull}},
{{6266118454276680859ull, 508314985927934ull}},
{{8702243578163255011ull, 406651988742347ull}},
{{18029841306756334978ull, 325321590993877ull}},
{{14090350831842494672ull, 520514545590204ull}},
{{14961629480215906061ull, 416411636472163ull}},
{{901257139946993879ull, 333129309177731ull}},
{{12510057868140921176ull, 533006894684369ull}},
{{13697395109254647264ull, 426405515747495ull}},
{{10957916087403717811ull, 341124412597996ull}},
{{10153968110362127852ull, 545799060156794ull}},
{{11812523303031612605ull, 436639248125435ull}},
{{9450018642425290084ull, 349311398500348ull}},
{{11430681013138553811ull, 558898237600557ull}},
{{1765847181027022402ull, 447118590080446ull}},
{{16170073003789259214ull, 357694872064356ull}},
{{9246709588289497048ull, 286155897651485ull}},
{{14794735341263195277ull, 457849436242376ull}},
{{8146439458268645899ull, 366279548993901ull}},
{{2827802751873006396ull, 293023639195121ull}},
{{15592530847222541203ull, 468837822712193ull}},
{{1405978233552301993ull, 375070258169755ull}},
{{1124782586841841594ull, 300056206535804ull}},
{{9178349768430767197ull, 480089930457286ull}},
{{3653331000002703434ull, 384071944365829ull}},
{{6612013614744073071ull, 307257555492663ull}},
{{6889872968848606590ull, 491612088788261ull}},
{{1822549560336974949ull, 393289671030609ull}},
{{5147388463011490282ull, 314631736824487ull}},
{{11925170355560294775ull, 503410778919179ull}},
{{13229485099190146143ull, 402728623135343ull}},
{{17962285708835937561ull, 322182898508274ull}},
{{17671610689911769128ull, 515492637613239ull}},
{{17826637366671325625ull, 412394110090591ull}},
{{10571961078595150177ull, 329915288072473ull}},
{{13225788911010329960ull, 527864460915957ull}},
{{3201933499324443322ull, 422291568732766ull}},
{{17318942058427195950ull, 337833254986212ull}},
{{12952912034515872228ull, 540533207977940ull}},
{{10362329627612697782ull, 432426566382352ull}},
{{911166072606337579ull, 345941253105882ull}},
{{5147214530912050450ull, 553506004969411ull}},
{{428422809987730037ull, 442804803975529ull}},
{{4032087062732094352ull, 354243843180423ull}},
{{10604367279669496128ull, 283395074544338ull}},
{{13277638832729283482ull, 453432119270941ull}},
{{6932762251441516463ull, 362745695416753ull}},
{{12924907430637033816ull, 290196556333402ull}},
{{5922456630051612814ull, 464314490133444ull}},
{{8427314118783200574ull, 371451592106755ull}},
{{6741851295026560459ull, 297161273685404ull}},
{{18165659701526317381ull, 475458037896646ull}},
{{10843178946479143582ull, 380366430317317ull}},
{{1295845527699494219ull, 304293144253854ull}},
{{9452050473803011397ull, 486869030806166ull}},
{{3872291564300498794ull, 389495224644933ull}},
{{10476530880924219682ull, 311596179715946ull}},
{{9383751779994930845ull, 498553887545514ull}},
{{11196350238737854999ull, 398843110036411ull}},
{{5267731376248373676ull, 319074488029129ull}},
{{15807067831481218528ull, 510519180846606ull}},
{{8956305450443064499ull, 408415344677285ull}},
{{7165044360354451599ull, 326732275741828ull}},
{{7774722161825212236ull, 522771641186925ull}},
{{6219777729460169788ull, 418217312949540ull}},
{{4975822183568135831ull, 334573850359632ull}},
{{11650664308450927653ull, 535318160575411ull}},
{{5631182632018831799ull, 428254528460329ull}},
{{8194294920356975762ull, 342603622768263ull}},
{{9421523057829250896ull, 548165796429221ull}},
{{3847869631521490394ull, 438532637143377ull}},
{{14146342149442923285ull, 350826109714701ull}},
{{15255449809624856609ull, 561321775543522ull}},
{{4825662218216064641ull, 449057420434818ull}},
{{11239227404056672359ull, 359245936347854ull}},
{{12680730737987248210ull, 287396749078283ull}},
{{16599820366037686814ull, 459834798525253ull}},
{{2211809848604418481ull, 367867838820203ull}},
{{9148145508367355431ull, 294294271056162ull}},
{{18326381628129679013ull, 470870833689859ull}},
{{18350454117245653534ull, 376696666951887ull}},
{{7301665664312702181ull, 301357333561510ull}},
{{11682665062900323489ull, 482171733698416ull}},
{{5656783235578348468ull, 385737386958733ull}},
{{11904124217946499421ull, 308589909566986ull}},
{{11667901119230578427ull, 493743855307178ull}},
{{16713018524868283388ull, 394995084245742ull}},
{{5991717190410806064ull, 315996067396594ull}},
{{16965445134141110349ull, 505593707834550ull}},
{{13572356107312888279ull, 404474966267640ull}},
{{10857884885850310623ull, 323579973014112ull}},
{{2615220558392855705ull, 517727956822580ull}},
{{2092176446714284564ull, 414182365458064ull}},
{{5363089972113337974ull, 331345892366451ull}},
{{1202246325897520112ull, 530153427786322ull}},
{{12029843504943747059ull, 424122742229057ull}},
{{2245177174471177001ull, 339298193783246ull}},
{{14660329923379614171ull, 542877110053193ull}},
{{660217494477960367ull, 434301688042555ull}},
{{528173995582368294ull, 347441350434044ull}},
{{8223776022415609917ull, 555906160694470ull}},
{{6579020817932487933ull, 444724928555576ull}},
{{1573867839604080023ull, 355779942844461ull}},
{{16016489530650905311ull, 284623954275568ull}},
{{3490290360589986559ull, 455398326840910ull}},
{{2792232288471989247ull, 364318661472728ull}},
{{9612483460261412044ull, 291454929178182ull}},
{{622578277450617978ull, 466327886685092ull}},
{{11566109066186225352ull, 373062309348073ull}},
{{16631584882432800928ull, 298449847478458ull}},
{{4474442923441019546ull, 477519755965534ull}},
{{7268903153494725960ull, 382015804772427ull}},
{{16883168967021511737ull, 305612643817941ull}},
{{1187628644041046518ull, 488980230108707ull}},
{{12018149359458568184ull, 391184184086965ull}},
{{9614519487566854547ull, 312947347269572ull}},
{{625835921139325982ull, 500715755631316ull}},
{{15258063995879102079ull, 400572604505052ull}},
{{4827753567219461016ull, 320458083604042ull}},
{{11413754522293047950ull, 512732933766467ull}},
{{1752305988350617713ull, 410186347013174ull}},
{{5091193605422404494ull, 328149077610539ull}},
{{15524607398159667837ull, 525038524176862ull}},
{{5040988289043913623ull, 420030819341490ull}},
{{4032790631235130898ull, 336024655473192ull}},
{{10141813824718119760ull, 537639448757107ull}},
{{734753430290675162ull, 430111559005686ull}},
{{15345198003200181422ull, 344089247204548ull}},
{{2416223916668828337ull, 550542795527278ull}},
{{9311676762818883316ull, 440434236421822ull}},
{{70643780771286006ull, 352347389137458ull}},
{{7435212654100849451ull, 281877911309966ull}},
{{4517642617077538476ull, 451004658095946ull}},
{{18371509352629672073ull, 360803726476756ull}},
{{11007858667361827335ull, 288642981181405ull}},
{{17612573867778923737ull, 461828769890248ull}},
{{3022012649997408020ull, 369463015912199ull}},
{{6106958934739836739ull, 295570412729759ull}},
{{17149831925067559429ull, 472912660367614ull}},
{{17409214354795957866ull, 378330128294091ull}},
{{10238022669094855970ull, 302664102635273ull}},
{{12691487455809859229ull, 484262564216437ull}},
{{2774492335164066736ull, 387410051373150ull}},
{{2219593868131253389ull, 309928041098520ull}},
{{3551350189010005423ull, 495884865757632ull}},
{{13909126595433735308ull, 396707892606105ull}},
{{11127301276346988246ull, 317366314084884ull}},
{{6735635597929450224ull, 507786102535815ull}},
{{5388508478343560179ull, 406228882028652ull}},
{{15378853226900579113ull, 324983105622921ull}},
{{17227467533557105935ull, 519972968996674ull}},
{{17471322841587595071ull, 415978375197339ull}},
{{17666407088011986380ull, 332782700157871ull}},
{{2440809637625805945ull, 532452320252595ull}},
{{1952647710100644756ull, 425961856202076ull}},
{{16319513427048157098ull, 340769484961660ull}},
{{7664477409567499741ull, 545231175938657ull}},
{{17199628371879730762ull, 436184940750925ull}},
{{13759702697503784610ull, 348947952600740ull}},
{{3568780242296503760ull, 558316724161185ull}},
{{2855024193837203008ull, 446653379328948ull}},
{{9662716984553583052ull, 357322703463158ull}},
{{15108871217126687088ull, 285858162770526ull}},
{{16795496317918878695ull, 457373060432842ull}},
{{6057699424851282309ull, 365898448346274ull}},
{{8535508354622936171ull, 292718758677019ull}},
{{2588766923170966904ull, 468350013883231ull}},
{{16828408797504414816ull, 374680011106584ull}},
{{17152075852745442176ull, 299744008885267ull}},
{{12685926105425066189ull, 479590414216428ull}},
{{17527438513823873597ull, 383672331373142ull}},
{{6643253181575278231ull, 306937865098514ull}},
{{18007902720004265817ull, 491100584157622ull}},
{{7027624546519592007ull, 392880467326098ull}},
{{13000797266699494252ull, 314304373860878ull}},
{{17111926811977280480ull, 502886998177405ull}},
{{13689541449581824384ull, 402309598541924ull}},
{{14640981974407369830ull, 321847678833539ull}},
{{12357524714826060759ull, 514956286133663ull}},
{{17264717401344669253ull, 411965028906930ull}},
{{13811773921075735403ull, 329572023125544ull}},
{{11030791829495445675ull, 527315237000871ull}},
{{5135284648854446217ull, 421852189600697ull}},
{{15176274163309287943ull, 337481751680557ull}},
{{9524643402327219416ull, 539970802688892ull}},
{{241017092377954886ull, 431976642151114ull}},
{{3882162488644274232ull, 345581313720891ull}},
{{17279506426056569741ull, 552930101953425ull}},
{{13823605140845255793ull, 442344081562740ull}},
{{11058884112676204634ull, 353875265250192ull}},
{{1468409660657143061ull, 283100212200154ull}},
{{9728153086535249544ull, 452960339520246ull}},
{{4093173654486289312ull, 362368271616197ull}},
{{14342585367814762419ull, 289894617292957ull}},
{{8190741329535978578ull, 463831387668732ull}},
{{17620639507854513832ull, 371065110134985ull}},
{{14096511606283611066ull, 296852088107988ull}},
{{418325681602315766ull, 474963340972782ull}},
{{11402706989507583582ull, 379970672778225ull}},
{{9122165591606066866ull, 303976538222580ull}},
{{14595464946569706985ull, 486362461156128ull}},
{{608325513030034619ull, 389089968924903ull}},
{{7865358039907848341ull, 311271975139922ull}},
{{16273921678594467669ull, 498035160223875ull}},
{{13019137342875574135ull, 398428128179100ull}},
{{10415309874300459308ull, 318742502543280ull}},
{{16664495798880734894ull, 509988004069248ull}},
{{2263550194878856945ull, 407990403255399ull}},
{{5500188970644995879ull, 326392322604319ull}},
{{16178999982515814053ull, 522227716166910ull}},
{{12943199986012651243ull, 417782172933528ull}},
{{17733257618293941640ull, 334225738346822ull}},
{{13615816930302665332ull, 534761181354916ull}},
{{7203304729500221942ull, 427808945083933ull}},
{{13141341413083998200ull, 342247156067146ull}},
{{13647448631450576474ull, 547595449707434ull}},
{{14607307719902371503ull, 438076359765947ull}},
{{4307148546438076556ull, 350461087812758ull}},
{{3202088859559012166ull, 560737740500413ull}},
{{9940368717131030379ull, 448590192400330ull}},
{{7952294973704824303ull, 358872153920264ull}},
{{10051184793705769766ull, 287097723136211ull}},
{{8703198040445410979ull, 459356357017938ull}},
{{14341256061840149429ull, 367485085614350ull}},
{{11473004849472119543ull, 293988068491480ull}},
{{18356807759155391270ull, 470380909586368ull}},
{{3617399763098582046ull, 376304727669095ull}},
{{2893919810478865637ull, 301043782135276ull}},
{{15698318140991915989ull, 481670051416441ull}},
{{8869305698051622468ull, 385336041133153ull}},
{{14474142187925118620ull, 308268832906522ull}},
{{8401232241712548500ull, 493230132650436ull}},
{{3031636978628128477ull, 394584106120349ull}},
{{6114658397644413105ull, 315667284896279ull}},
{{17162151065714881614ull, 505067655834046ull}},
{{10040372037829994968ull, 404054124667237ull}},
{{653600000780175328ull, 323243299733790ull}},
{{1045760001248280525ull, 517189279574064ull}},
{{4525956815740534743ull, 413751423659251ull}},
{{18378160711560069087ull, 331001138927400ull}},
{{10958313064786558924ull, 529601822283841ull}},
{{5077301637087336816ull, 423681457827073ull}},
{{11440538939153690099ull, 338945166261658ull}},
{{14615513487903993835ull, 542312266018653ull}},
{{624364346097464098ull, 433849812814923ull}},
{{7878189106361791925ull, 347079850251938ull}},
{{8915753755436956757ull, 555327760403101ull}},
{{3443254189607655082ull, 444262208322481ull}},
{{17511998610653765359ull, 355409766657984ull}},
{{17698947703264922610ull, 284327813326387ull}},
{{13560921066256234884ull, 454924501322220ull}},
{{10848736853004987907ull, 363939601057776ull}},
{{4989640667662080002ull, 291151680846221ull}},
{{604727438775507357ull, 465842689353954ull}},
{{4173130765762316209ull, 372674151483163ull}},
{{10717202242093673613ull, 298139321186530ull}},
{{17147523587349877782ull, 477022913898448ull}},
{{2649972425654171256ull, 381618331118759ull}},
{{5809326755265247328ull, 305294664895007ull}},
{{12984271623166306048ull, 488471463832011ull}},
{{6698068483791134515ull, 390777171065609ull}},
{{9047803601774817935ull, 312621736852487ull}},
{{18165834577581619020ull, 500194778963979ull}},
{{18222016476807205539ull, 400155823171183ull}},
{{3509566737220033461ull, 320124658536947ull}},
{{9304655594293963862ull, 512199453659115ull}},
{{7443724475435171089ull, 409759562927292ull}},
{{17023026024573867841ull, 327807650341833ull}},
{{5100748750866726606ull, 524492240546934ull}},
{{7769947815435291608ull, 419593792437547ull}},
{{17284004696573964256ull, 335675033950037ull}},
{{12897012255550701517ull, 537080054320060ull}},
{{10317609804440561214ull, 429664043456048ull}},
{{15632785473036269617ull, 343731234764838ull}},
{{2876363868406569449ull, 549969975623742ull}},
{{13369137538950986528ull, 439975980498993ull}},
{{18074007660644609869ull, 351980784399194ull}},
{{18148554943257598218ull, 281584627519355ull}},
{{10590943835502605534ull, 450535404030969ull}},
{{12162103883143994750ull, 360428323224775ull}},
{{9729683106515195800ull, 288342658579820ull}},
{{15567492970424313280ull, 461348253727712ull}},
{{5075296746855629978ull, 369078602982170ull}},
{{4060237397484503982ull, 295262882385736ull}},
{{17564426280200937341ull, 472420611817177ull}},
{{6672843394676929226ull, 377936489453742ull}},
{{16406321159967274351ull, 302349191562993ull}},
{{4114020967496177022ull, 483758706500790ull}},
{{3291216773996941618ull, 387006965200632ull}},
{{13701019863423284264ull, 309605572160505ull}},
{{3474887707767703206ull, 495368915456809ull}},
{{6469258980956072888ull, 396295132365447ull}},
{{16243453628990589280ull, 317036105892357ull}},
{{11232130547417301555ull, 507257769427772ull}},
{{1607006808450020598ull, 405806215542218ull}},
{{8664303076243837124ull, 324644972433774ull}},
{{2794838477764408430ull, 519431955894039ull}},
{{5925219596953437067ull, 415545564715231ull}},
{{1050826862820839330ull, 332436451772185ull}},
{{1681322980513342929ull, 531898322835496ull}},
{{16102453643378315636ull, 425518658268396ull}},
{{9192614099960742185ull, 340414926614717ull}},
{{18397531374679097820ull, 544663882583547ull}},
{{7339327470259457609ull, 435731106066838ull}},
{{13250159605691386734ull, 348584884853470ull}},
{{2753511295396667158ull, 557735815765553ull}},
{{9581506665801154373ull, 446188652612442ull}},
{{286507703157102852ull, 356950922089954ull}},
{{3918554977267592605ull, 285560737671963ull}},
{{2580339148886237845ull, 456897180275141ull}},
{{16821666578076631568ull, 365517744220112ull}},
{{6078635632977484608ull, 292414195376090ull}},
{{9725817012763975373ull, 467862712601744ull}},
{{11470002424953090622ull, 374290170081395ull}},
{{9176001939962472497ull, 299432136065116ull}},
{{7302905474456135350ull, 479091417704186ull}},
{{2152975564822997956ull, 383273134163349ull}},
{{5411729266600308688ull, 306618507330679ull}},
{{16037464456044314548ull, 490589611729086ull}},
{{9140622750093541315ull, 392471689383269ull}},
{{11001847014816743375ull, 313977351506615ull}},
{{17602955223706789400ull, 502363762410584ull}},
{{17771712993707341843ull, 401891009928467ull}},
{{6838672765482052828ull, 321512807942774ull}},
{{18320574054255105172ull, 514420492708438ull}},
{{3588412799178353168ull, 411536394166751ull}},
{{17628125498310323827ull, 329229115333400ull}},
{{9758256723586966507ull, 526766584533441ull}},
{{4117256564127662883ull, 421413267626753ull}},
{{10672502880785950952ull, 337130614101402ull}},
{{2318609350289880231ull, 539408982562244ull}},
{{5544236294973814508ull, 431527186049795ull}},
{{4435389035979051606ull, 345221748839836ull}},
{{18164668901792213540ull, 552354798143737ull}},
{{7153037491949950186ull, 441883838514990ull}},
{{5722429993559960148ull, 353507070811992ull}},
{{15645990439073699088ull, 282805656649593ull}},
{{2897491814066456602ull, 452489050639350ull}},
{{2317993451253165282ull, 361991240511480ull}},
{{1854394761002532225ull, 289592992409184ull}},
{{10345729247087872207ull, 463348787854694ull}},
{{11965932212412208089ull, 370679030283755ull}},
{{9572745769929766471ull, 296543224227004ull}},
{{4248346787661895384ull, 474469158763207ull}},
{{14466723874355247277ull, 379575327010565ull}},
{{11573379099484197821ull, 303660261608452ull}},
{{3760011300207075222ull, 485856418573524ull}},
{{6697357854907570501ull, 388685134858819ull}},
{{9047235098667966724ull, 310948107887055ull}},
{{14475576157868746758ull, 497516972619288ull}},
{{512414482069266437ull, 398013578095431ull}},
{{15167326844623054442ull, 318410862476344ull}},
{{13199676507171156138ull, 509457379962151ull}},
{{6870392390995014587ull, 407565903969721ull}},
{{1806965098054101346ull, 326052723175777ull}},
{{6580492971628472478ull, 521684357081243ull}},
{{12643092006786598628ull, 417347485664994ull}},
{{13803822420171189226ull, 333877988531995ull}},
{{3639371798564351146ull, 534204781651193ull}},
{{10290195068335301563ull, 427363825320954ull}},
{{11921504869410151573ull, 341891060256763ull}},
{{15385058976314332194ull, 547025696410821ull}},
{{8618698366309555432ull, 437620557128657ull}},
{{17963005137273375315ull, 350096445702925ull}},
{{10294064145927848889ull, 560154313124681ull}},
{{4545902502000368788ull, 448123450499745ull}},
{{3636722001600295030ull, 358498760399796ull}},
{{17666772860247877317ull, 286799008319836ull}},
{{2441394873203231445ull, 458878413311739ull}},
{{5642464713304495479ull, 367102730649391ull}},
{{824622955901686060ull, 293682184519513ull}},
{{16076791988410338989ull, 469891495231220ull}},
{{12861433590728271191ull, 375913196184976ull}},
{{6599798057840706629ull, 300730556947981ull}},
{{3180979263061309961ull, 481168891116770ull}},
{{2544783410449047969ull, 384935112893416ull}},
{{16793221987326879668ull, 307948090314732ull}},
{{12111759920755366176ull, 492716944503572ull}},
{{2310710307120472294ull, 394173555602858ull}},
{{9227265875180198482ull, 315338844482286ull}},
{{7384927770804496924ull, 504542151171658ull}},
{{13286639846127418186ull, 403633720937326ull}},
{{6939963062160024225ull, 322906976749861ull}},
{{3725243269972218115ull, 516651162799778ull}},
{{10358892245461595138ull, 413320930239822ull}},
{{908416166885455464ull, 330656744191858ull}},
{{16210861125984370035ull, 529050790706972ull}},
{{5589991271303675382ull, 423240632565578ull}},
{{11850690646526760952ull, 338592506052462ull}},
{{4203709775475176230ull, 541748009683940ull}},
{{3362967820380140984ull, 433398407747152ull}},
{{13758420700529843757ull, 346718726197721ull}},
{{14634775491363929365ull, 554749961916354ull}},
{{15397169207833053815ull, 443799969533083ull}},
{{1249688922040712082ull, 355039975626467ull}},
{{12067797581858300635ull, 284031980501173ull}},
{{15619127316231370693ull, 454451168801877ull}},
{{5116604223501275908ull, 363560935041502ull}},
{{15161329823026751696ull, 290848748033201ull}},
{{16879430087358982068ull, 465357996853122ull}},
{{6124846440403365008ull, 372286397482498ull}},
{{12278574781806512652ull, 297829117985998ull}},
{{15956370836148509921ull, 476526588777597ull}},
{{5386399039434987290ull, 381221271022078ull}},
{{11687816861031810479ull, 304977016817662ull}},
{{3943111718683255473ull, 487963226908260ull}},
{{3154489374946604378ull, 390370581526608ull}},
{{9902289129441104149ull, 312296465221286ull}},
{{8464964977621945992ull, 499674344354058ull}},
{{14150669611581377440ull, 399739475483246ull}},
{{7631186874523191629ull, 319791580386597ull}},
{{15899247813979016930ull, 511666528618555ull}},
{{12719398251183213544ull, 409333222894844ull}},
{{13864867415688481158ull, 327466578315875ull}},
{{3737043791392018237ull, 523946525305401ull}},
{{17747030292081255882ull, 419157220244320ull}},
{{14197624233665004706ull, 335325776195456ull}},
{{15337501144380186883ull, 536521241912730ull}},
{{12270000915504149506ull, 429216993530184ull}},
{{13505349547145229928ull, 343373594824147ull}},
{{6851164016464726593ull, 549397751718636ull}},
{{1791582398429870951ull, 439518201374909ull}},
{{5122614733485807084ull, 351614561099927ull}},
{{11885532388319201658ull, 562583297759883ull}},
{{16887123540139181972ull, 450066638207906ull}},
{{9820350017369435255ull, 360053310566325ull}},
{{7856280013895548204ull, 288042648453060ull}},
{{12570048022232877126ull, 460868237524896ull}},
{{6366689603044391378ull, 368694590019917ull}},
{{16161398126661244072ull, 294955672015933ull}},
{{3722144114206528576ull, 471929075225494ull}},
{{6667064106107133184ull, 377543260180395ull}},
{{5333651284885706547ull, 302034608144316ull}},
{{1155144426333309829ull, 483255373030906ull}},
{{15681510800034289156ull, 386604298424724ull}},
{{16234557454769341648ull, 309283438739779ull}},
{{14907245483405215667ull, 494853501983647ull}},
{{4547098757240351887ull, 395882801586918ull}},
{{11016376635276102156ull, 316706241269534ull}},
{{6558156172216032480ull, 506729986031255ull}},
{{5246524937772825984ull, 405383988825004ull}},
{{7886568764960171110ull, 324307191060003ull}},
{{8929161209194363454ull, 518891505696005ull}},
{{7143328967355490763ull, 415113204556804ull}},
{{9404011988626302933ull, 332090563645443ull}},
{{11357070367060174370ull, 531344901832709ull}},
{{12775005108390049819ull, 425075921466167ull}},
{{2841306457228219209ull, 340060737172934ull}},
{{11924787961048971381ull, 544097179476694ull}},
{{13229179183581087428ull, 435277743581355ull}},
{{10583343346864869942ull, 348222194865084ull}},
{{5865302910758060938ull, 557155511784135ull}},
{{4692242328606448751ull, 445724409427308ull}},
{{11132491492368979647ull, 356579527541846ull}},
{{5216644379153273394ull, 285263622033477ull}},
{{12035979821387147754ull, 456421795253563ull}},
{{17007481486593538850ull, 365137436202850ull}},
{{13605985189274831080ull, 292109948962280ull}},
{{3322832229130178112ull, 467375918339649ull}},
{{6347614598046052812ull, 373900734671719ull}},
{{8767440493178752573ull, 299120587737375ull}},
{{14027904789086004117ull, 478592940379800ull}},
{{11222323831268803294ull, 382874352303840ull}},
{{8977859065015042635ull, 306299481843072ull}},
{{18053923318765978539ull, 490079170948915ull}},
{{14443138655012782831ull, 392063336759132ull}},
{{4175813294526405618ull, 313650669407306ull}},
{{17749347715467979959ull, 501841071051689ull}},
{{17888826987116294291ull, 401472856841351ull}},
{{10621712774951125109ull, 321178285473081ull}},
{{9616042810437979529ull, 513885256756930ull}},
{{7692834248350383623ull, 411108205405544ull}},
{{9843616213422217221ull, 328886564324435ull}},
{{15749785941475547554ull, 526218502919096ull}},
{{8910479938438527720ull, 420974802335277ull}},
{{18196430394976553146ull, 336779841868221ull}},
{{3288846928769112771ull, 538847746989155ull}},
{{2631077543015290217ull, 431078197591324ull}},
{{5794210849154142496ull, 344862558073059ull}},
{{16649434988130448641ull, 551780092916894ull}},
{{17008896805246269236ull, 441424074333515ull}},
{{13607117444197015389ull, 353139259466812ull}},
{{3506996325873791664ull, 282511407573450ull}},
{{5611194121398066663ull, 452018252117520ull}},
{{4488955297118453331ull, 361614601694016ull}},
{{18348559496662403957ull, 289291681355212ull}},
{{14600299935692205039ull, 462866690168340ull}},
{{11680239948553764031ull, 370293352134672ull}},
{{1965494329359190578ull, 296234681707738ull}},
{{17902186185942346219ull, 473975490732380ull}},
{{14321748948753876975ull, 379180392585904ull}},
{{15146747973745011903ull, 303344314068723ull}},
{{2098703869540557106ull, 485350902509958ull}},
{{9057660725116266331ull, 388280722007966ull}},
{{3556779765351102741ull, 310624577606373ull}},
{{2001498809819854063ull, 496999324170197ull}},
{{12669245492081614220ull, 397599459336157ull}},
{{2756698764181470730ull, 318079567468926ull}},
{{15478764466916084137ull, 508927307950281ull}},
{{8693662758790956987ull, 407141846360225ull}},
{{6954930207032765589ull, 325713477088180ull}},
{{11127888331252424943ull, 521141563341088ull}},
{{16281008294485760601ull, 416913250672870ull}},
{{13024806635588608480ull, 333530600538296ull}},
{{13460992987457952923ull, 533648960861274ull}},
{{14458143204708272661ull, 426919168689019ull}},
{{15255863378508528452ull, 341535334951215ull}},
{{5962637331904093908ull, 546456535921945ull}},
{{4770109865523275126ull, 437165228737556ull}},
{{126739077676709777ull, 349732182990045ull}},
{{202782524282735644ull, 559571492784072ull}},
{{11230272463651919485ull, 447657194227257ull}},
{{1605520341437714941ull, 358125755381806ull}},
{{16041811532117813246ull, 286500604305444ull}},
{{14598852007162770224ull, 458400966888711ull}},
{{7989732790988305856ull, 366720773510969ull}},
{{10081135047532555008ull, 293376618808775ull}},
{{16129816076052088013ull, 469402590094040ull}},
{{12903852860841670410ull, 375522072075232ull}},
{{2944384659189515682ull, 300417657660186ull}},
{{15779061898928956060ull, 480668252256297ull}},
{{5244551889659344202ull, 384534601805038ull}},
{{11574339141211296008ull, 307627681444030ull}},
{{72198552228521997ull, 492204290310449ull}},
{{3747107656524727921ull, 393763432248359ull}},
{{6687034939961692660ull, 315010745798687ull}},
{{14388604718680618579ull, 504017193277899ull}},
{{15200232589686405186ull, 403213754622319ull}},
{{15849534886491034472ull, 322571003697855ull}},
{{6912511744676103539ull, 516113605916569ull}},
{{9219358210482793155ull, 412890884733255ull}},
{{7375486568386234524ull, 330312707786604ull}},
{{732732065192244269ull, 528500332458567ull}},
{{11654232096379526384ull, 422800265966853ull}},
{{16702083306587441754ull, 338240212773482ull}},
{{11965938031572265514ull, 541184340437572ull}},
{{2194052795773991764ull, 432947472350058ull}},
{{9133939866103014058ull, 346357977880046ull}},
{{7235606156281001846ull, 554172764608074ull}},
{{9477833739766711800ull, 443338211686459ull}},
{{11271615806555279763ull, 354670569349167ull}},
{{1638595015760403164ull, 283736455479334ull}},
{{10000449654700465709ull, 453978328766934ull}},
{{11689708538502282890ull, 363182663013547ull}},
{{1973069201318005666ull, 290546130410838ull}},
{{17914305981076450358ull, 464873808657340ull}},
{{14331444784861160287ull, 371899046925872ull}},
{{4086458198405107583ull, 297519237540698ull}},
{{2848984302706261810ull, 476030780065117ull}},
{{13347233886390740417ull, 380824624052093ull}},
{{18056484738596412980ull, 304659699241674ull}},
{{17822329137528529799ull, 487455518786679ull}},
{{17947212124764734162ull, 389964415029343ull}},
{{3289723255586056360ull, 311971532023475ull}},
{{5263557208937690176ull, 499154451237560ull}},
{{4210845767150152141ull, 399323560990048ull}},
{{10747374243203942359ull, 319458848792038ull}},
{{13506449974384397452ull, 511134158067261ull}},
{{7115811164765607638ull, 408907326453809ull}},
{{9381997746554396434ull, 327125861163047ull}},
{{253801135519393001ull, 523401377860876ull}},
{{14960436167383155694ull, 418721102288700ull}},
{{11968348933906524555ull, 334976881830960ull}},
{{702614220540887672ull, 535963010929537ull}},
{{11630137820658441107ull, 428770408743629ull}},
{{12993459071268663209ull, 343016326994903ull}},
{{17100185699287950811ull, 548826123191845ull}},
{{13680148559430360649ull, 439060898553476ull}},
{{7254770032802378196ull, 351248718842781ull}},
{{4228934422999984467ull, 561997950148450ull}},
{{3383147538399987574ull, 449598360118760ull}},
{{2706518030719990059ull, 359678688095008ull}},
{{9543912054059812693ull, 287742950476006ull}},
{{7891561657011879663ull, 460388720761610ull}},
{{6313249325609503730ull, 368310976609288ull}},
{{12429297089971423631ull, 294648781287430ull}},
{{1440131270244726193ull, 471438050059889ull}},
{{4841453830937691278ull, 377150440047911ull}},
{{183814250008242699ull, 301720352038329ull}},
{{7672800429497008965ull, 482752563261326ull}},
{{2448891528855696849ull, 386202050609061ull}},
{{16716508482052198772ull, 308961640487248ull}},
{{4610320682832056096ull, 494338624779598ull}},
{{11066954175749465523ull, 395470899823678ull}},
{{16232260970083393065ull, 316376719858942ull}},
{{11214222293165787611ull, 506202751774308ull}},
{{16350075464016450735ull, 404962201419446ull}},
{{9390711556471250265ull, 323969761135557ull}},
{{267743231386359131ull, 518351617816892ull}},
{{11282241029334818274ull, 414681294253513ull}},
{{16404490452951675266ull, 331745035402810ull}},
{{7800440651013128809ull, 530792056644497ull}},
{{17308398965036234017ull, 424633645315597ull}},
{{6468021542545166567ull, 339706916252478ull}},
{{6659485653330356185ull, 543531066003965ull}},
{{5327588522664284948ull, 434824852803172ull}},
{{15330117262357158928ull, 347859882242537ull}},
{{9770792360803812992ull, 556575811588060ull}},
{{7816633888643050393ull, 445260649270448ull}},
{{13632004740398260961ull, 356208519416358ull}},
{{18284301421802429415ull, 284966815533086ull}},
{{3429440571690514802ull, 455946904852939ull}},
{{6432901272094322165ull, 364757523882351ull}},
{{1456972202933547408ull, 291806019105881ull}},
{{13399201968919406823ull, 466889630569409ull}},
{{14408710389877435782ull, 373511704455527ull}},
{{4148270682418127979ull, 298809363564422ull}},
{{10326581906610915090ull, 478094981703075ull}},
{{8261265525288732072ull, 382475985362460ull}},
{{6609012420230985657ull, 305980788289968ull}},
{{6885071057627666729ull, 489569261263949ull}},
{{9197405660844043706ull, 391655409011159ull}},
{{11047273343417145288ull, 313324327208927ull}},
{{2918242090499791168ull, 501318923534284ull}},
{{6023942487141743258ull, 401055138827427ull}},
{{15887200433939125576ull, 320844111061941ull}},
{{18040823064818780275ull, 513350577699106ull}},
{{10743309637113113897ull, 410680462159285ull}},
{{8594647709690491117ull, 328544369727428ull}},
{{10062087520762875465ull, 525670991563885ull}},
{{8049670016610300372ull, 420536793251108ull}},
{{13818433642772060944ull, 336429434600886ull}},
{{14730796198951476864ull, 538287095361418ull}},
{{716590514935450521ull, 430629676289135ull}},
{{573272411948360417ull, 344503741031308ull}},
{{15674631118085017960ull, 551205985650092ull}},
{{5161007264984193722ull, 440964788520074ull}},
{{7818154626729265300ull, 352771830816059ull}},
{{9943872516125322563ull, 282217464652847ull}},
{{1152800766832874809ull, 451547943444556ull}},
{{15679635872433941140ull, 361238354755644ull}},
{{16233057512689063235ull, 288990683804515ull}},
{{7526147946592949560ull, 462385094087225ull}},
{{6020918357274359648ull, 369908075269780ull}},
{{4816734685819487718ull, 295926460215824ull}},
{{15085473126795000996ull, 473482336345318ull}},
{{1000332057210269827ull, 378785869076255ull}},
{{800265645768215862ull, 303028695261004ull}},
{{8659122662712966025ull, 484845912417606ull}},
{{3237949315428462497ull, 387876729934085ull}},
{{2590359452342769997ull, 310301383947268ull}},
{{455226309006521673ull, 496482214315629ull}},
{{4053529861947127662ull, 397185771452503ull}},
{{10621521519041522776ull, 317748617162002ull}},
{{2237039171498795148ull, 508397787459204ull}},
{{5478980151940946442ull, 406718229967363ull}},
{{11761881751036577800ull, 325374583973890ull}},
{{372266727948972864ull, 520599334358225ull}},
{{297813382359178291ull, 416479467486580ull}},
{{238250705887342633ull, 333183573989264ull}},
{{7759898758903568859ull, 533093718382822ull}},
{{17275965451348586057ull, 426474974706257ull}},
{{6442074731595048199ull, 341179979765006ull}},
{{2928621941068256472ull, 545887967624010ull}},
{{2342897552854605178ull, 436710374099208ull}},
{{9253015671767504788ull, 349368299279366ull}},
{{7426127445344187015ull, 558989278846986ull}},
{{2251553141533439289ull, 447191423077589ull}},
{{5490591327968661754ull, 357753138462071ull}},
{{703124247633019080ull, 286202510769657ull}},
{{4814347610954740852ull, 457924017231451ull}},
{{162129274021882358ull, 366339213785161ull}},
{{14887098678185147179ull, 293071371028128ull}},
{{1683264996644773548ull, 468914193645006ull}},
{{16104007256283460131ull, 375131354916004ull}},
{{16572554619768678428ull, 300105083932803ull}},
{{4379994503178423545ull, 480168134292486ull}},
{{18261390861510380129ull, 384134507433988ull}},
{{3541066244982573134ull, 307307605947191ull}},
{{16733752436197847984ull, 491692169515505ull}},
{{13387001948958278387ull, 393353735612404ull}},
{{14398950373908533033ull, 314682988489923ull}},
{{902227709802190913ull, 503492781583878ull}},
{{8100479797325573377ull, 402794225267102ull}},
{{17548430282086189671ull, 322235380213681ull}},
{{2252046748144531211ull, 515576608341891ull}},
{{16559032657483266262ull, 412461286673512ull}},
{{5868528496502792363ull, 329969029338810ull}},
{{9389645594404467781ull, 527950446942096ull}},
{{3822367660781663902ull, 422360357553677ull}},
{{14125940572851062091ull, 337888286042941ull}},
{{15222807287077878699ull, 540621257668706ull}},
{{8488897014920392636ull, 432497006134965ull}},
{{6791117611936314109ull, 345997604907972ull}},
{{14555136993840012897ull, 553596167852755ull}},
{{11644109595072010318ull, 442876934282204ull}},
{{13004636490799518577ull, 354301547425763ull}},
{{17782406822123435508ull, 283441237940610ull}},
{{10005106841687945197ull, 453505980704977ull}},
{{625387843866535511ull, 362804784563982ull}},
{{11568356719318959379ull, 290243827651185ull}},
{{62626677200783390ull, 464390124241897ull}},
{{11118147785986357682ull, 371512099393517ull}},
{{1515820599305265499ull, 297209679514814ull}},
{{9804010588372245445ull, 475535487223702ull}},
{{464510841213975709ull, 380428389778962ull}},
{{11439655117196911537ull, 304342711823169ull}},
{{7235401743289327490ull, 486948338917071ull}},
{{2098972579889551669ull, 389558671133657ull}},
{{12747224508137372304ull, 311646936906925ull}},
{{1948815139310244071ull, 498635099051081ull}},
{{16316447370415836550ull, 398908079240864ull}},
{{16742506711074579563ull, 319126463392691ull}},
{{962569034525955038ull, 510602341428307ull}},
{{11838101671846495000ull, 408481873142645ull}},
{{9470481337477196000ull, 326785498514116ull}},
{{7774072510479692954ull, 522856797622586ull}},
{{2529909193641844040ull, 418285438098069ull}},
{{5713276169655385555ull, 334628350478455ull}},
{{9141241871448616888ull, 535405360765528ull}},
{{14691691126642714157ull, 428324288612422ull}},
{{4374655271830350679ull, 342659430889938ull}},
{{3310099620186650764ull, 548255089423901ull}},
{{17405474955116961904ull, 438604071539120ull}},
{{13924379964093569523ull, 350883257231296ull}},
{{14900310313065890590ull, 561413211570074ull}},
{{15609597065194622795ull, 449130569256059ull}},
{{16177026466897608559ull, 359304455404847ull}},
{{5562923544034266201ull, 287443564323878ull}},
{{5211328855712915599ull, 459909702918205ull}},
{{4169063084570332479ull, 367927762334564ull}},
{{7024599282398176306ull, 294342209867651ull}},
{{3860661222353261444ull, 470947535788242ull}},
{{14156575422108340125ull, 376758028630593ull}},
{{257213893460941130ull, 301406422904475ull}},
{{411542229537505808ull, 482250276647160ull}},
{{329233783630004647ull, 385800221317728ull}},
{{7642084656387824364ull, 308640177054182ull}},
{{15916684264962429305ull, 493824283286691ull}},
{{9043998597228033121ull, 395059426629353ull}},
{{14613896507266247143ull, 316047541303482ull}},
{{8624839152658354136ull, 505676066085572ull}},
{{17967917766352414278ull, 404540852868457ull}},
{{6995636583598110776ull, 323632682294766ull}},
{{3814320904273156596ull, 517812291671626ull}},
{{17808851982386166569ull, 414249833337300ull}},
{{14247081585908933255ull, 331399866669840ull}},
{{4348586463744741593ull, 530239786671745ull}},
{{3478869170995793274ull, 424191829337396ull}},
{{17540490595764275912ull, 339353463469916ull}},
{{2239343250029469197ull, 542965541551867ull}},
{{12859521044249306327ull, 434372433241493ull}},
{{17666314464883265708ull, 347497946593194ull}},
{{17198056699587494164ull, 555996714549111ull}},
{{10069096544928085008ull, 444797371639289ull}},
{{11744626050684378329ull, 355837897311431ull}},
{{5706352025805592340ull, 284670317849145ull}},
{{9130163241288947745ull, 455472508558632ull}},
{{18372177037256889165ull, 364378006846905ull}},
{{14697741629805511332ull, 291502405477524ull}},
{{12448340163463087162ull, 466403848764039ull}},
{{13648020945512380053ull, 373123079011231ull}},
{{7229067941667993719ull, 298498463208985ull}},
{{11566508706668789950ull, 477597541134376ull}},
{{5563858150593121637ull, 382078032907501ull}},
{{761737705732586986ull, 305662426326001ull}},
{{12286826773397870148ull, 489059882121601ull}},
{{6140112603976385795ull, 391247905697281ull}},
{{1222741268439198313ull, 312998324557825ull}},
{{1956386029502717301ull, 500797319292520ull}},
{{1565108823602173840ull, 400637855434016ull}},
{{16009482317849380365ull, 320510284347212ull}},
{{10857776449591367292ull, 512816454955540ull}},
{{8686221159673093833ull, 410253163964432ull}},
{{18017023371964206036ull, 328202531171545ull}},
{{10380493321433178042ull, 525124049874473ull}},
{{15683092286630363080ull, 420099239899578ull}},
{{1478427385078559494ull, 336079391919663ull}},
{{17122879075093336484ull, 537727027071460ull}},
{{13698303260074669187ull, 430181621657168ull}},
{{18337340237543555996ull, 344145297325734ull}},
{{18271697935843958624ull, 550632475721175ull}},
{{14617358348675166899ull, 440505980576940ull}},
{{11693886678940133519ull, 352404784461552ull}},
{{1976411713668286169ull, 281923827569242ull}},
{{6851607556611168194ull, 451078124110787ull}},
{{16549332489514665525ull, 360862499288629ull}},
{{16928814806353642743ull, 288689999430903ull}},
{{4950010801714366449ull, 461903999089446ull}},
{{270659826629582836ull, 369523199271557ull}},
{{11284574305529397238ull, 295618559417245ull}},
{{18055318888847035582ull, 472989695067592ull}},
{{7065557481593807819ull, 378391756054074ull}},
{{9341794800016956578ull, 302713404843259ull}},
{{3878825235801399556ull, 484341447749215ull}},
{{3103060188641119645ull, 387473158199372ull}},
{{13550494595138626685ull, 309978526559497ull}},
{{6923396093254161404ull, 495965642495196ull}},
{{1849368059861418800ull, 396772513996157ull}},
{{12547540892114866009ull, 317418011196925ull}},
{{1629321353674233999ull, 507868817915081ull}},
{{16060852341907028492ull, 406295054332064ull}},
{{16538030688267533117ull, 325036043465651ull}},
{{635407398034680725ull, 520057669545043ull}},
{{7887023547911565226ull, 416046135636034ull}},
{{9998967653071162504ull, 332836908508827ull}},
{{1240952985946218714ull, 532539053614124ull}},
{{4682111203498885294ull, 426031242891299ull}},
{{7435037777541018558ull, 340824994313039ull}},
{{828013999839898724ull, 545319990900863ull}},
{{8041108829355739626ull, 436255992720690ull}},
{{6432887063484591700ull, 349004794176552ull}},
{{13981968116317257044ull, 558407670682483ull}},
{{117528048828074666ull, 446726136545987ull}},
{{11162068883288190702ull, 357380909236789ull}},
{{12619003921372462885ull, 285904727389431ull}},
{{12811708644712119969ull, 457447563823090ull}},
{{10249366915769695975ull, 365958051058472ull}},
{{820795903131936134ull, 292766440846778ull}},
{{16070668703978739107ull, 468426305354844ull}},
{{16545883777924901609ull, 374741044283875ull}},
{{13236707022339921287ull, 299792835427100ull}},
{{2731987162034322444ull, 479668536683361ull}},
{{16942984988595099248ull, 383734829346688ull}},
{{2486341546650348428ull, 306987863477351ull}},
{{15046192918866288455ull, 491180581563761ull}},
{{8347605520351120441ull, 392944465251009ull}},
{{10367433231022806676ull, 314355572200807ull}},
{{1830497910668849389ull, 502968915521292ull}},
{{12532444772760810481ull, 402375132417033ull}},
{{17404653447692469031ull, 321900105933626ull}},
{{2022003813114578187ull, 515040169493803ull}},
{{8996300679975483196ull, 412032135595042ull}},
{{18265086988206117526ull, 329625708476033ull}},
{{7088046292678326103ull, 527401133561654ull}},
{{9359785848884571206ull, 421920906849323ull}},
{{14866526308591477611ull, 337536725479458ull}},
{{1650349205294902238ull, 540058760767134ull}},
{{5009628178977832114ull, 432047008613707ull}},
{{15075748987407996661ull, 345637606890965ull}},
{{5674454306143243041ull, 553020171025545ull}},
{{4539563444914594433ull, 442416136820436ull}},
{{18389046014899316839ull, 353932909456348ull}},
{{3643190367693722501ull, 283146327565079ull}},
{{13207802217793776649ull, 453034124104126ull}},
{{6876892959493110996ull, 362427299283301ull}},
{{1812165552852578473ull, 289941839426641ull}},
{{13967511328789856527ull, 463906943082625ull}},
{{11174009063031885222ull, 371125554466100ull}},
{{8939207250425508177ull, 296900443572880ull}},
{{14302731600680813084ull, 475040709716608ull}},
{{374138836318919498ull, 380032567773287ull}},
{{11367357513280866568ull, 304026054218629ull}},
{{7119725577023655539ull, 486441686749807ull}},
{{16763826905844655401ull, 389153349399845ull}},
{{13411061524675724320ull, 311322679519876ull}},
{{14079000809997338266ull, 498116287231802ull}},
{{3884503018514049967ull, 398493029785442ull}},
{{14175648859036970943ull, 318794423828353ull}},
{{544945286007691570ull, 510071078125366ull}},
{{15193351487773794548ull, 408056862500292ull}},
{{4775983560735214992ull, 326445490000234ull}},
{{15020271326660164634ull, 522312784000374ull}},
{{15705565876070042031ull, 417850227200299ull}},
{{16253801515597943948ull, 334280181760239ull}},
{{14938035980730979347ull, 534848290816383ull}},
{{882382340359052508ull, 427878632653107ull}},
{{11773952316512972976ull, 342302906122485ull}},
{{391579632711205145ull, 547684649795977ull}},
{{11381310150394695086ull, 438147719836781ull}},
{{5415699305573845745ull, 350518175869425ull}},
{{8665118888918153193ull, 560829081391080ull}},
{{6932095111134522554ull, 448663265112864ull}},
{{9235024903649528366ull, 358930612090291ull}},
{{3698671108177712370ull, 287144489672233ull}},
{{2228524958342429469ull, 459431183475573ull}},
{{9161517596157764221ull, 367544946780458ull}},
{{14707911706410032023ull, 294035957424366ull}},
{{16153961100772230591ull, 470457531878986ull}},
{{9233820065875874150ull, 376366025503189ull}},
{{11076404867442609643ull, 301092820402551ull}},
{{10343550158424354783ull, 481748512644082ull}},
{{896142497255663180ull, 385398810115266ull}},
{{15474309256772171836ull, 308319048092212ull}},
{{10001499551867833646ull, 493310476947540ull}},
{{8001199641494266916ull, 394648381558032ull}},
{{17469006157421144503ull, 315718705246425ull}},
{{9503665778164279588ull, 505149928394281ull}},
{{3913583807789513347ull, 404119942715425ull}},
{{3130867046231610678ull, 323295954172340ull}},
{{5009387273970577085ull, 517273526675744ull}},
{{7696858633918371991ull, 413818821340595ull}},
{{6157486907134697593ull, 331055057072476ull}},
{{2473281421931695502ull, 529688091315962ull}},
{{13046671581771087371ull, 423750473052769ull}},
{{14126686080158780220ull, 339000378442215ull}},
{{4155953654544496737ull, 542400605507545ull}},
{{3324762923635597389ull, 433920484406036ull}},
{{17417205597876119204ull, 347136387524828ull}},
{{5731436068150328788ull, 555418220039726ull}},
{{895800039778352707ull, 444334576031781ull}},
{{15474035290790323458ull, 355467660825424ull}},
{{16068577047374169090ull, 284374128660339ull}},
{{14641676831572939574ull, 454998605856543ull}},
{{645295021032620689ull, 363998884685235ull}},
{{516236016826096551ull, 291199107748188ull}},
{{15583372885889395775ull, 465918572397100ull}},
{{12466698308711516620ull, 372734857917680ull}},
{{9973358646969213296ull, 298187886334144ull}},
{{4889327390925010305ull, 477100618134631ull}},
{{222113097998097920ull, 381680494507705ull}},
{{177690478398478336ull, 305344395606164ull}},
{{7663002394921385985ull, 488551032969862ull}},
{{17198448360162839757ull, 390840826375889ull}},
{{17448107502872182129ull, 312672661100711ull}},
{{2091530301402119144ull, 500276257761139ull}},
{{5362573055863605638ull, 400221006208911ull}},
{{600709629948974187ull, 320176804967129ull}},
{{8339833037402179346ull, 512282887947406ull}},
{{2982517615179833154ull, 409826310357925ull}},
{{2386014092143866523ull, 327861048286340ull}},
{{3817622547430186437ull, 524577677258144ull}},
{{6743446852686059473ull, 419662141806515ull}},
{{5394757482148847578ull, 335729713445212ull}},
{{12320960786180066448ull, 537167541512339ull}},
{{13546117443685963482ull, 429734033209871ull}},
{{7147545140206860462ull, 343787226567897ull}},
{{15125421039072887063ull, 550059562508635ull}},
{{12100336831258309650ull, 440047650006908ull}},
{{17058967094490468367ull, 352038120005526ull}},
{{9957824860850464370ull, 281630496004421ull}},
{{8553822147876922346ull, 450608793607074ull}},
{{10532406533043448200ull, 360487034885659ull}},
{{12115274041176668883ull, 288389627908527ull}},
{{4627043206915028920ull, 461423404653644ull}},
{{7390983380273933459ull, 369138723722915ull}},
{{5912786704219146767ull, 295310978978332ull}},
{{13149807541492545151ull, 472497566365331ull}},
{{6830497218452125798ull, 377998053092265ull}},
{{5464397774761700638ull, 302398442473812ull}},
{{12432385254360631344ull, 483837507958099ull}},
{{13635257018230415399ull, 387070006366479ull}},
{{14597554429326242642ull, 309656005093183ull}},
{{1219994198470526288ull, 495449608149094ull}},
{{4665344173518331354ull, 396359686519275ull}},
{{3732275338814665083ull, 317087749215420ull}},
{{5971640542103464133ull, 507340398744672ull}},
{{15845358877908502276ull, 405872318995737ull}},
{{5297589472842981174ull, 324697855196590ull}},
{{8476143156548769879ull, 519516568314544ull}},
{{10470263339980926226ull, 415613254651635ull}},
{{8376210671984740981ull, 332490603721308ull}},
{{9712588260433675246ull, 531984965954093ull}},
{{15148768237830760843ull, 425587972763274ull}},
{{15808363405006518998ull, 340470378210619ull}},
{{14225335003784699427ull, 544752605136991ull}},
{{7690919188285849219ull, 435802084109593ull}},
{{13531432980112500021ull, 348641667287674ull}},
{{10582246323954269064ull, 557826667660279ull}},
{{12155145873905325575ull, 446261334128223ull}},
{{17102814328608081106ull, 357009067302578ull}},
{{2614205018660733915ull, 285607253842063ull}},
{{493379215115263941ull, 456971606147301ull}},
{{15152098631059852446ull, 365577284917840ull}},
{{12121678904847881957ull, 292461827934272ull}},
{{4637290988788969838ull, 467938924694836ull}},
{{20483976289265547ull, 374351139755869ull}},
{{3705735995773322761ull, 299480911804695ull}},
{{5929177593237316417ull, 479169458887512ull}},
{{15811388518815584103ull, 383335567110009ull}},
{{16338459629794377606ull, 306668453688007ull}},
{{11384140148703362877ull, 490669525900812ull}},
{{1728614489478869655ull, 392535620720650ull}},
{{1382891591583095724ull, 314028496576520ull}},
{{2212626546532953159ull, 502445594522432ull}},
{{12838147681452093496ull, 401956475617945ull}},
{{10270518145161674797ull, 321565180494356ull}},
{{9054131402774859029ull, 514504288790970ull}},
{{7243305122219887223ull, 411603431032776ull}},
{{2105295283033999455ull, 329282744826221ull}},
{{14436518897080130098ull, 526852391721953ull}},
{{481168673438373109ull, 421481913377563ull}},
{{7763632568234519133ull, 337185530702050ull}},
{{12421812109175230614ull, 539496849123280ull}},
{{9937449687340184491ull, 431597479298624ull}},
{{11639308564614057916ull, 345277983438899ull}},
{{7554847259156761696ull, 552444773502239ull}},
{{9733226622067319680ull, 441955818801791ull}},
{{4097232482911945421ull, 353564655041433ull}},
{{10656483615813376983ull, 282851724033146ull}},
{{9671676155817582526ull, 452562758453034ull}},
{{11426689739395976344ull, 362050206762427ull}},
{{1762654162032960429ull, 289640165409942ull}},
{{6509595473994647010ull, 463424264655907ull}},
{{16275722823421448577ull, 370739411724725ull}},
{{13020578258737158862ull, 296591529379780ull}},
{{2386181140269902563ull, 474546447007649ull}},
{{5598293726957832374ull, 379637157606119ull}},
{{8167983796308176222ull, 303709726084895ull}},
{{13068774074093081955ull, 485935561735832ull}},
{{3076321629790644918ull, 388748449388666ull}},
{{17218452562800157227ull, 310998759510932ull}},
{{12792128841512610271ull, 497598015217492ull}},
{{2855005443726267570ull, 398078412173994ull}},
{{5973353169722924379ull, 318462729739195ull}},
{{9557365071556679007ull, 509540367582712ull}},
{{267194427761522559ull, 407632294066170ull}},
{{213755542209218047ull, 326105835252936ull}},
{{11410055311760479845ull, 521769336404697ull}},
{{1749346619924563230ull, 417415469123758ull}},
{{8778174925423471230ull, 333932375299006ull}},
{{6666382251193733322ull, 534291800478410ull}},
{{5333105800954986657ull, 427433440382728ull}},
{{11645182270247809972ull, 341946752306182ull}},
{{3874896373428854663ull, 547114803689892ull}},
{{14167963542968814700ull, 437691842951913ull}},
{{266324390149320790ull, 350153474361531ull}},
{{11494165468464644234ull, 560245558978449ull}},
{{12884681189513625711ull, 448196447182759ull}},
{{13997093766352810892ull, 358557157746207ull}},
{{3818977383598428067ull, 286845726196966ull}},
{{17178410257983215877ull, 458953161915145ull}},
{{13742728206386572701ull, 367162529532116ull}},
{{7304833750367347838ull, 293730023625693ull}},
{{7998385185845846217ull, 469968037801109ull}},
{{10088056963418587297ull, 375974430240887ull}},
{{691747941251049191ull, 300779544192710ull}},
{{1106796706001678706ull, 481247270708336ull}},
{{15642832623768984258ull, 384997816566668ull}},
{{1446219654789456436ull, 307998253253335ull}},
{{2313951447663130299ull, 492797205205336ull}},
{{16608556417098145532ull, 394237764164268ull}},
{{2218798689452785456ull, 315390211331415ull}},
{{3550077903124456729ull, 504624338130264ull}},
{{6529411137241475706ull, 403699470504211ull}},
{{1534180095051270242ull, 322959576403369ull}},
{{9833385781565853034ull, 516735322245390ull}},
{{7866708625252682427ull, 413388257796312ull}},
{{17361413344427876911ull, 330710606237049ull}},
{{16710214906858872088ull, 529136969979279ull}},
{{17057520740229007994ull, 423309575983423ull}},
{{2577970147957475425ull, 338647660786739ull}},
{{11503449866215781327ull, 541836257258782ull}},
{{1824062263488804415ull, 433469005807026ull}},
{{16216645069758684825ull, 346775204645620ull}},
{{7499888037904344104ull, 554840327432993ull}},
{{13378608059807295930ull, 443872261946394ull}},
{{14392235262587747067ull, 355097809557115ull}},
{{11513788210070197653ull, 284078247645692ull}},
{{3664665877144674953ull, 454525196233108ull}},
{{10310430331199560608ull, 363620156986486ull}},
{{4558995450217738163ull, 290896125589189ull}},
{{14673090349832201708ull, 465433800942702ull}},
{{4359774650381940720ull, 372347040754162ull}},
{{14555866164531283546ull, 297877632603329ull}},
{{12221339419024322704ull, 476604212165327ull}},
{{2398373905735637516ull, 381283369732262ull}},
{{12986745568814240983ull, 305026695785809ull}},
{{9710746465877054603ull, 488042713257295ull}},
{{7768597172701643682ull, 390434170605836ull}},
{{2525528923419404623ull, 312347336484669ull}},
{{11419543906954868043ull, 499755738375470ull}},
{{9135635125563894434ull, 399804590700376ull}},
{{3619159285709205224ull, 319843672560301ull}},
{{16858701301360459328ull, 511749876096481ull}},
{{9797612226346457139ull, 409399900877185ull}},
{{7838089781077165711ull, 327519920701748ull}},
{{8851594834981554815ull, 524031873122797ull}},
{{18149322312210974822ull, 419225498498237ull}},
{{7140760220284959211ull, 335380398798590ull}},
{{11425216352455934738ull, 536608638077744ull}},
{{12829521896706658113ull, 429286910462195ull}},
{{10263617517365326490ull, 343429528369756ull}},
{{9043090398300701739ull, 549487245391610ull}},
{{7234472318640561391ull, 439589796313288ull}},
{{13166275484396269759ull, 351671837050630ull}},
{{2619296701324479999ull, 562674939281009ull}},
{{5784786175801494322ull, 450139951424807ull}},
{{15695875384866926427ull, 360111961139845ull}},
{{12556700307893541142ull, 288089568911876ull}},
{{12712022863145845180ull, 460943310259002ull}},
{{2790920661032855498ull, 368754648207202ull}},
{{13300782973052015368ull, 295003718565761ull}},
{{13902555127399403942ull, 472005949705218ull}},
{{53997657693792184ull, 377604759764175ull}},
{{43198126155033747ull, 302083807811340ull}},
{{69117001848053996ull, 483334092498144ull}},
{{3744642416220353520ull, 386667273998515ull}},
{{2995713932976282816ull, 309333819198812ull}},
{{8482491107503962829ull, 494934110718099ull}},
{{10475341700745080586ull, 395947288574479ull}},
{{12069622175337974792ull, 316757830859583ull}},
{{15622046665798849344ull, 506812529375333ull}},
{{1429590888413348506ull, 405450023500267ull}},
{{12211719154956409774ull, 324360018800213ull}},
{{15849401833188345316ull, 518976030080341ull}},
{{8990172651808765929ull, 415180824064273ull}},
{{14570835750930833390ull, 332144659251418ull}},
{{1177244313037871484ull, 531431454802270ull}},
{{941795450430297187ull, 425145163841816ull}},
{{15510831619311879043ull, 340116131073452ull}},
{{10059935331931365176ull, 544185809717524ull}},
{{11737297080287002464ull, 435348647774019ull}},
{{13079186478971512294ull, 348278918219215ull}},
{{2479954292644868055ull, 557246269150745ull}},
{{1983963434115894444ull, 445797015320596ull}},
{{16344566006260356848ull, 356637612256476ull}},
{{9386303990266375155ull, 285310089805181ull}},
{{7639388754942379602ull, 456496143688290ull}},
{{6111511003953903681ull, 365196914950632ull}},
{{15957255247388853915ull, 292157531960505ull}},
{{7084864322112614648ull, 467452051136809ull}},
{{9357240272432002041ull, 373961640909447ull}},
{{107094588461780986ull, 299169312727558ull}},
{{14928746600506490871ull, 478670900364092ull}},
{{4564299650921372051ull, 382936720291274ull}},
{{7340788535479007964ull, 306349376233019ull}},
{{677215212540681772ull, 490159001972831ull}},
{{15299167429000186711ull, 392127201578264ull}},
{{15928682757942059692ull, 313701761262611ull}},
{{18107194783223474860ull, 501922818020178ull}},
{{3417709382353048919ull, 401538254416143ull}},
{{10112865135366259781ull, 321230603532914ull}},
{{5112537772360284681ull, 513968965652663ull}},
{{11468727847372048391ull, 411175172522130ull}},
{{9174982277897638713ull, 328940138017704ull}},
{{3611925200410490971ull, 526304220828327ull}},
{{13957586604554123746ull, 421043376662661ull}},
{{7476720468901388674ull, 336834701330129ull}},
{{894706306016490908ull, 538935522128207ull}},
{{11783811489038923696ull, 431148417702565ull}},
{{9427049191231138957ull, 344918734162052ull}},
{{325883447002181039ull, 551869974659284ull}},
{{3950055572343655154ull, 441495979727427ull}},
{{14228090902100655093ull, 353196783781941ull}},
{{7693123906938613751ull, 282557427025553ull}},
{{8619649436359871678ull, 452091883240885ull}},
{{6895719549087897343ull, 361673506592708ull}},
{{12895273268754138520ull, 289338805274166ull}},
{{13253739600522800987ull, 462942088438666ull}},
{{6913642865676330466ull, 370353670750933ull}},
{{12909611922024885019ull, 296282936600746ull}},
{{13276681445755995384ull, 474052698561194ull}},
{{14310693971346706631ull, 379242158848955ull}},
{{11448555177077365304ull, 303393727079164ull}},
{{7249641839098053518ull, 485429963326663ull}},
{{13178411100762263460ull, 388343970661330ull}},
{{10542728880609810768ull, 310675176529064ull}},
{{5800319764749966260ull, 497080282446503ull}},
{{12018953441283793654ull, 397664225957202ull}},
{{2236465123543214277ull, 318131380765762ull}},
{{7267693012411053167ull, 509010209225219ull}},
{{9503503224670752856ull, 407208167380175ull}},
{{7602802579736602285ull, 325766533904140ull}},
{{12164484127578563656ull, 521226454246624ull}},
{{13420936116804761248ull, 416981163397299ull}},
{{14426097708185719322ull, 333584930717839ull}},
{{12013709888871419945ull, 533735889148543ull}},
{{16989665540580956603ull, 426988711318834ull}},
{{17281081247206675605ull, 341590969055067ull}},
{{12892334736563039676ull, 546545550488108ull}},
{{17692565418734252387ull, 437236440390486ull}},
{{10464703520245491586ull, 349789152312389ull}},
{{5675479188167055569ull, 559662643699823ull}},
{{11919080980017465101ull, 447730114959858ull}},
{{16913962413497792727ull, 358184091967886ull}},
{{9841821116056323858ull, 286547273574309ull}},
{{4678867341464387204ull, 458475637718895ull}},
{{3743093873171509763ull, 366780510175116ull}},
{{17751870357504849103ull, 293424408140092ull}},
{{13645597313040117273ull, 469479053024148ull}},
{{18295175479915914465ull, 375583242419318ull}},
{{3568093939707000602ull, 300466593935455ull}},
{{5708950303531200963ull, 480746550296728ull}},
{{11945857872308781417ull, 384597240237382ull}},
{{2177988668363204487ull, 307677792189906ull}},
{{14552828313606858149ull, 492284467503849ull}},
{{15331611465627396843ull, 393827574003079ull}},
{{15954637987243827797ull, 315062059202463ull}},
{{3391327891138662537ull, 504099294723942ull}},
{{13781108757136660999ull, 403279435779153ull}},
{{18403584635193149445ull, 322623548623322ull}},
{{14688340157341397820ull, 516197677797316ull}},
{{8061323311131207933ull, 412958142237853ull}},
{{13827756278388786992ull, 330366513790282ull}},
{{7367014786454417895ull, 528586422064452ull}},
{{16961658273389265286ull, 422869137651561ull}},
{{9879977803969501905ull, 338295310121249ull}},
{{4739918042125472079ull, 541272496193999ull}},
{{7481283248442287987ull, 433017996955199ull}},
{{9674375413495740712ull, 346414397564159ull}},
{{4410954217367454170ull, 554263036102655ull}},
{{3528763373893963336ull, 443410428882124ull}},
{{6512359513857080992ull, 354728343105699ull}},
{{8899236425827575117ull, 283782674484559ull}},
{{3170731837098389218ull, 454052279175295ull}},
{{2536585469678711374ull, 363241823340236ull}},
{{16786663634710610392ull, 290593458672188ull}},
{{4722568927085514688ull, 464949533875502ull}},
{{14846101585894142720ull, 371959627100401ull}},
{{8187532453973403853ull, 297567701680321ull}},
{{5721354296873625518ull, 476108322688514ull}},
{{8266432252240810738ull, 380886658150811ull}},
{{2923796987050738267ull, 304709326520649ull}},
{{12056772808765001874ull, 487534922433038ull}},
{{17024115876495822145ull, 390027937946430ull}},
{{13619292701196657716ull, 312022350357144ull}},
{{10722821877688921376ull, 499235760571431ull}},
{{4888908687409226778ull, 399388608457145ull}},
{{3911126949927381422ull, 319510886765716ull}},
{{17325849564109541245ull, 511217418825145ull}},
{{13860679651287632996ull, 408973935060116ull}},
{{7399194906288196074ull, 327179148048093ull}},
{{8149363035319203395ull, 523486636876949ull}},
{{10208839242997273039ull, 418789309501559ull}},
{{11856420209139728754ull, 335031447601247ull}},
{{4212877075655924714ull, 536050316161996ull}},
{{18127696919492381064ull, 428840252929596ull}},
{{10812808720851994528ull, 343072202343677ull}},
{{2543098694395549952ull, 548915523749884ull}},
{{5723827770258350285ull, 439132418999907ull}},
{{15647108660432411197ull, 351305935199925ull}},
{{6588629782982306300ull, 562089496319881ull}},
{{1581555011643934717ull, 449671597055905ull}},
{{1265244009315147773ull, 359737277644724ull}},
{{4701544022194028542ull, 287789822115779ull}},
{{14901168064994266314ull, 460463715385246ull}},
{{8231585637253502728ull, 368370972308197ull}},
{{17653314954028533152ull, 294696777846557ull}},
{{13487908667478011750ull, 471514844554492ull}},
{{3411629304498588754ull, 377211875643594ull}},
{{6418652258340781326ull, 301769500514875ull}},
{{10269843613345250122ull, 482831200823800ull}},
{{8215874890676200097ull, 386264960659040ull}},
{{6572699912540960078ull, 309011968527232ull}},
{{14205668674807446448ull, 494419149643571ull}},
{{7675186125104046835ull, 395535319714857ull}},
{{17208195344308968438ull, 316428255771885ull}},
{{9086368477184797884ull, 506285209235017ull}},
{{18337141225973569277ull, 405028167388013ull}},
{{3601666536553124452ull, 324022533910411ull}},
{{16830712902710730093ull, 518436054256657ull}},
{{6085872692684763428ull, 414748843405326ull}},
{{1179349339405900419ull, 331799074724261ull}},
{{12955005387275171640ull, 530878519558817ull}},
{{2985306680336316666ull, 424702815647054ull}},
{{6077594159010963656ull, 339762252517643ull}},
{{6034801839675631526ull, 543619604028229ull}},
{{8517190286482415544ull, 434895683222583ull}},
{{14192449858669753082ull, 347916546578066ull}},
{{15329222144387784284ull, 556666474524906ull}},
{{8574028900768317104ull, 445333179619925ull}},
{{6859223120614653683ull, 356266543695940ull}},
{{5487378496491722946ull, 285013234956752ull}},
{{12469154409128667038ull, 456021175930803ull}},
{{17354021156786754277ull, 364816940744642ull}},
{{6504519295945582775ull, 291853552595714ull}},
{{17785928502996753086ull, 466965684153142ull}},
{{6850045172913581823ull, 373572547322514ull}},
{{9169384953072775781ull, 298858037858011ull}},
{{7292318295432620604ull, 478172860572818ull}},
{{13212552265829917129ull, 382538288458254ull}},
{{14259390627405844027ull, 306030630766603ull}},
{{678932115397888504ull, 489649009226566ull}},
{{15300540951285952096ull, 391719207381252ull}},
{{4861735131544941030ull, 313375365905002ull}},
{{11468125025213815971ull, 501400585448003ull}},
{{16553197649654873423ull, 401120468358402ull}},
{{5863860490240078092ull, 320896374686722ull}},
{{13071525599126035271ull, 513434199498755ull}},
{{10457220479300828217ull, 410747359599004ull}},
{{12055125198182572897ull, 328597887679203ull}},
{{15598851502350206312ull, 525756620286725ull}},
{{12479081201880165049ull, 420605296229380ull}},
{{9983264961504132039ull, 336484236983504ull}},
{{4905177494180880293ull, 538374779173607ull}},
{{14992188439570435204ull, 430699823338885ull}},
{{11993750751656348163ull, 344559858671108ull}},
{{15500652387908246738ull, 551295773873773ull}},
{{1332475466100866421ull, 441036619099019ull}},
{{4755329187622603460ull, 352829295279215ull}},
{{3804263350098082768ull, 282263436223372ull}},
{{9776170174898842752ull, 451621497957395ull}},
{{7820936139919074202ull, 361297198365916ull}},
{{2567400097193349038ull, 289037758692733ull}},
{{418491340767448138ull, 462460413908373ull}},
{{7713490702097779157ull, 369968331126698ull}},
{{13549490191162043972ull, 295974664901358ull}},
{{17989835491117360032ull, 473559463842173ull}},
{{3323821948668157056ull, 378847571073739ull}},
{{6348406373676435968ull, 303078056858991ull}},
{{2778752568398476902ull, 484924890974386ull}},
{{16980397313686422814ull, 387939912779508ull}},
{{2516271406723407282ull, 310351930223607ull}},
{{7715383065499361974ull, 496563088357771ull}},
{{2482957637657579256ull, 397250470686217ull}},
{{13054412554351794374ull, 317800376548973ull}},
{{17197711272220960676ull, 508480602478357ull}},
{{6379471388292947894ull, 406784481982686ull}},
{{1414228295892447992ull, 325427585586149ull}},
{{9641462902911737434ull, 520684136937838ull}},
{{15091867951813210594ull, 416547309550270ull}},
{{12073494361450568475ull, 333237847640216ull}},
{{11938893348837088914ull, 533180556224346ull}},
{{5861765864327760808ull, 426544444979477ull}},
{{15757459135687939616ull, 341235555983581ull}},
{{17833236987616882739ull, 545976889573730ull}},
{{14266589590093506191ull, 436781511658984ull}},
{{15102620486816715276ull, 349425209327187ull}},
{{9406797519939103149ull, 559080334923500ull}},
{{7525438015951282519ull, 447264267938800ull}},
{{6020350412761026015ull, 357811414351040ull}},
{{4816280330208820812ull, 286249131480832ull}},
{{11395397343076023623ull, 457998610369331ull}},
{{5426969059718908575ull, 366398888295465ull}},
{{4341575247775126860ull, 293119110636372ull}},
{{10635869211182113299ull, 468990577018195ull}},
{{8508695368945690639ull, 375192461614556ull}},
{{3117607480414642188ull, 300153969291645ull}},
{{4988171968663427501ull, 480246350866632ull}},
{{15058584019156472970ull, 384197080693305ull}},
{{12046867215325178376ull, 307357664554644ull}},
{{8206941100294554433ull, 491772263287431ull}},
{{2876204065493733223ull, 393417810629945ull}},
{{2300963252394986578ull, 314734248503956ull}},
{{14749587648057709495ull, 503574797606329ull}},
{{15489018933188077919ull, 402859838085063ull}},
{{1323168702324731366ull, 322287870468051ull}},
{{13185116367945301155ull, 515660592748881ull}},
{{6858744279614330601ull, 412528474199105ull}},
{{5486995423691464480ull, 330022779359284ull}},
{{16157890307390163815ull, 528036446974854ull}},
{{16615661060654041375ull, 422429157579883ull}},
{{2224482404297502131ull, 337943326063907ull}},
{{7248520661617913732ull, 540709321702251ull}},
{{2109467714552420663ull, 432567457361801ull}},
{{16444969430609577823ull, 346053965889440ull}},
{{7865207015265772901ull, 553686345423105ull}},
{{6292165612212618321ull, 442949076338484ull}},
{{8723081304512004980ull, 354359261070787ull}},
{{18046511487835334953ull, 283487408856629ull}},
{{17806371936310804956ull, 453579854170607ull}},
{{6866399919564823318ull, 362863883336486ull}},
{{1803771120909948331ull, 290291106669189ull}},
{{10264731422939737977ull, 464465770670702ull}},
{{833087508867969735ull, 371572616536562ull}},
{{11734516451320106757ull, 297258093229249ull}},
{{7707179877886439842ull, 475612949166799ull}},
{{9855092717051062197ull, 380490359333439ull}},
{{11573422988382760081ull, 304392287466751ull}},
{{11138779151928595483ull, 487027659946802ull}},
{{1532325692059055740ull, 389622127957442ull}},
{{12293906997872975561ull, 311697702365953ull}},
{{15980902381854850575ull, 498716323785525ull}},
{{12784721905483880460ull, 398973059028420ull}},
{{10227777524387104368ull, 319178447222736ull}},
{{8985746409535546343ull, 510685515556378ull}},
{{14567294757112257721ull, 408548412445102ull}},
{{4275138176205985530ull, 326838729956082ull}},
{{10529569896671487171ull, 522941967929731ull}},
{{4734307102595279414ull, 418353574343785ull}},
{{3787445682076223531ull, 334682859475028ull}},
{{2370564276580047327ull, 535492575160045ull}},
{{1896451421264037861ull, 428394060128036ull}},
{{16274556395978871582ull, 342715248102428ull}},
{{3903197345114732592ull, 548344396963886ull}},
{{17879953135059427366ull, 438675517571108ull}},
{{3235916063821810923ull, 350940414056887ull}},
{{8866814516856807801ull, 561504662491019ull}},
{{10782800428227356564ull, 449203729992815ull}},
{{8626240342581885251ull, 359362983994252ull}},
{{17969038718291239170ull, 287490387195401ull}},
{{2925020246072610410ull, 459984619512643ull}},
{{9718713826341908974ull, 367987695610114ull}},
{{11464319875815437503ull, 294390156488091ull}},
{{10964214171820879358ull, 471024250380946ull}},
{{5082022522714793163ull, 376819400304757ull}},
{{15133664462397565500ull, 301455520243805ull}},
{{5767119066126553184ull, 482328832390089ull}},
{{8303044067643152871ull, 385863065912071ull}},
{{2953086439372611973ull, 308690452729657ull}},
{{8414287117738089480ull, 493904724367451ull}},
{{3042080879448561261ull, 395123779493961ull}},
{{17191059962526490302ull, 316099023595168ull}},
{{5369603051590922544ull, 505758437752270ull}},
{{4295682441272738035ull, 404606750201816ull}},
{{18193941211985831721ull, 323685400161452ull}},
{{14352910680209689460ull, 517896640258324ull}},
{{15171677358909661891ull, 414317312206659ull}},
{{15826690701869639836ull, 331453849765327ull}},
{{10565309864023782445ull, 530326159624524ull}},
{{12141596705960936279ull, 424260927699619ull}},
{{13402626179510659347ull, 339408742159695ull}},
{{2997457813507503339ull, 543053987455513ull}},
{{9776663880289823317ull, 434443189964410ull}},
{{7821331104231858654ull, 347554551971528ull}},
{{8824780952029063523ull, 556087283154445ull}},
{{7059824761623250819ull, 444869826523556ull}},
{{1958510994556690332ull, 355895861218845ull}},
{{1566808795645352265ull, 284716688975076ull}},
{{13574940517258294594ull, 455546702360121ull}},
{{7170603599064725352ull, 364437361888097ull}},
{{16804529323477511251ull, 291549889510477ull}},
{{12129851658596376709ull, 466479823216764ull}},
{{13393230141619011690ull, 373183858573411ull}},
{{7025235298553299029ull, 298547086858729ull}},
{{172330033459547477ull, 477675338973967ull}},
{{11205910470993368951ull, 382140271179173ull}},
{{16343426006278515807ull, 305712216943338ull}},
{{4013388721594163353ull, 489139547109342ull}},
{{14278757421501061652ull, 391311637687473ull}},
{{354959492975118352ull, 313049310149979ull}},
{{7946632818244010009ull, 500878896239966ull}},
{{2667957439853297684ull, 400703116991973ull}},
{{9513063581366458794ull, 320562493593578ull}},
{{11531552915444423747ull, 512899989749725ull}},
{{9225242332355538998ull, 410319991799780ull}},
{{7380193865884431198ull, 328255993439824ull}},
{{740263741189358947ull, 525209589503719ull}},
{{4281559807693397481ull, 420167671602975ull}},
{{3425247846154717985ull, 336134137282380ull}},
{{5480396553847548776ull, 537814619651808ull}},
{{11763014872561859667ull, 430251695721446ull}},
{{5721063083307577410ull, 344201356577157ull}},
{{12843049748034034180ull, 550722170523451ull}},
{{6585090983685317021ull, 440577736418761ull}},
{{1578723972206343293ull, 352462189135009ull}},
{{4952327992506984958ull, 281969751308007ull}},
{{11613073602753086256ull, 451151602092811ull}},
{{5601110067460558681ull, 360921281674249ull}},
{{8170236868710357268ull, 288737025339399ull}},
{{2004332545710840660ull, 461979240543039ull}},
{{5292814851310582851ull, 369583392434431ull}},
{{544903066306555957ull, 295666713947545ull}},
{{871844906090489532ull, 473066742316072ull}},
{{11765522369098122595ull, 378453393852857ull}},
{{2033720265794677430ull, 302762715082286ull}},
{{14321998869497214857ull, 484420344131657ull}},
{{4078901466113951239ull, 387536275305326ull}},
{{18020516431858802284ull, 310029020244260ull}},
{{10386082217264532039ull, 496046432390817ull}},
{{930168144327804985ull, 396837145912654ull}},
{{4433483330204154311ull, 317469716730123ull}},
{{3404224513584736575ull, 507951546768197ull}},
{{13791426055093520229ull, 406361237414557ull}},
{{3654443214590995537ull, 325088989931646ull}},
{{16915155587571323829ull, 520142383890633ull}},
{{2464078025831328093ull, 416113907112507ull}},
{{13039308864890793444ull, 332891125690005ull}},
{{2416150110115717895ull, 532625801104009ull}},
{{5622268902834484639ull, 426100640883207ull}},
{{15565861566493318681ull, 340880512706565ull}},
{{6458634432679758274ull, 545408820330505ull}},
{{5166907546143806619ull, 436327056264404ull}},
{{7822874851656955618ull, 349061645011523ull}},
{{8827250947909218666ull, 558498632018437ull}},
{{18129847202553105902ull, 446798905614749ull}},
{{18193226576784395045ull, 357439124491799ull}},
{{18243930076169426359ull, 285951299593439ull}},
{{18122241677645351205ull, 457522079349503ull}},
{{3429746897890549994ull, 366017663479603ull}},
{{10122495147796260642ull, 292814130783682ull}},
{{1438596977506375734ull, 468502609253892ull}},
{{12218924026230831557ull, 374802087403113ull}},
{{17153836850468485892ull, 299841669922490ull}},
{{8999394887040025811ull, 479746671875985ull}},
{{7199515909632020649ull, 383797337500788ull}},
{{13138310357189437166ull, 307037870000630ull}},
{{2574552497793547849ull, 491260592001009ull}},
{{5748990812976748602ull, 393008473600807ull}},
{{15667239094607129851ull, 314406778880645ull}},
{{6620838477661856147ull, 503050846209033ull}},
{{12675368411613305564ull, 402440676967226ull}},
{{6450945914548734128ull, 321952541573781ull}},
{{2942815833794153958ull, 515124066518050ull}},
{{2354252667035323166ull, 412099253214440ull}},
{{1883402133628258533ull, 329679402571552ull}},
{{6702792228547123976ull, 527487044114483ull}},
{{12740931412321519827ull, 421989635291586ull}},
{{6503396315115305539ull, 337591708233269ull}},
{{17784131733668309508ull, 540146733173230ull}},
{{14227305386934647607ull, 432117386538584ull}},
{{15071193124289628408ull, 345693909230867ull}},
{{9356513739895764161ull, 553110254769388ull}},
{{14863908621400431975ull, 442488203815510ull}},
{{11891126897120345580ull, 353990563052408ull}},
{{16891599147180097110ull, 283192450441926ull}},
{{1201116932294783114ull, 453107920707083ull}},
{{8339591175319647138ull, 362486336565666ull}},
{{2982324125513807387ull, 289989069252533ull}},
{{1082369786080181496ull, 463982510804053ull}},
{{8244593458347965843ull, 371186008643242ull}},
{{17663721210904103644ull, 296948806914593ull}},
{{6125861048995103892ull, 475118091063350ull}},
{{4900688839196083113ull, 380094472850680ull}},
{{3920551071356866490ull, 304075578280544ull}},
{{13651579343654807031ull, 486520925248870ull}},
{{10921263474923845625ull, 389216740199096ull}},
{{5047661965197166177ull, 311373392159277ull}},
{{11765607959057376206ull, 498197427454843ull}},
{{16791183996729721611ull, 398557941963874ull}},
{{17122296012125687612ull, 318846353571099ull}},
{{16327627175175369210ull, 510154165713759ull}},
{{16751450554882205691ull, 408123332571007ull}},
{{6022462814421943906ull, 326498666056806ull}},
{{2257242873591289604ull, 522397865690890ull}},
{{1805794298873031683ull, 417918292552712ull}},
{{12512681883324156316ull, 334334634042169ull}},
{{8952244569092919136ull, 534935414467471ull}},
{{3472446840532424986ull, 427948331573977ull}},
{{13846003916651670958ull, 342358665259181ull}},
{{14774908637158852887ull, 547773864414690ull}},
{{11819926909727082310ull, 438219091531752ull}},
{{2077243898297845201ull, 350575273225402ull}},
{{7012939052018462645ull, 560920437160643ull}},
{{12989048871098590763ull, 448736349728514ull}},
{{14080587911620782933ull, 358989079782811ull}},
{{7575121514554716023ull, 287191263826249ull}},
{{1052147979061814668ull, 459506022121999ull}},
{{4531067197991362057ull, 367604817697599ull}},
{{7314202573134999969ull, 294083854158079ull}},
{{634677672790268981ull, 470534166652927ull}},
{{11575788582457946154ull, 376427333322341ull}},
{{5571282051224446600ull, 301141866657873ull}},
{{5224702467217204237ull, 481826986652597ull}},
{{15247808417999494359ull, 385461589322077ull}},
{{4819549104915774841ull, 308369271457662ull}},
{{11400627382607150069ull, 493390834332259ull}},
{{12809850720827630378ull, 394712667465807ull}},
{{2869182947178283656ull, 315770133972646ull}},
{{15658739159710984820ull, 505232214356233ull}},
{{1458944883543056886ull, 404185771484987ull}},
{{12235202351060176478ull, 323348617187989ull}},
{{8508277317470551396ull, 517357787500783ull}},
{{14185319483460261763ull, 413886230000626ull}},
{{7658906772026299087ull, 331108984000501ull}},
{{4875553205758257894ull, 529774374400802ull}},
{{14968489008832337284ull, 423819499520641ull}},
{{8285442392323959504ull, 339055599616513ull}},
{{9567359012976424884ull, 542488959386421ull}},
{{3964538395639229584ull, 433991167509137ull}},
{{14239677160737114637ull, 347192934007309ull}},
{{11715437012953652449ull, 555508694411695ull}},
{{9372349610362921959ull, 444406955529356ull}},
{{3808530873548427244ull, 355525564423485ull}},
{{3046824698838741795ull, 284420451538788ull}},
{{1185570703400076549ull, 455072722462061ull}},
{{15705851821687702532ull, 364058177969648ull}},
{{1496635013124431056ull, 291246542375719ull}},
{{9773313650482910336ull, 465994467801150ull}},
{{7818650920386328269ull, 372795574240920ull}},
{{6254920736309062615ull, 298236459392736ull}},
{{2629175548610679538ull, 477178335028378ull}},
{{9482038068372364277ull, 381742668022702ull}},
{{206932825214070775ull, 305394134418162ull}},
{{4020441335084423563ull, 488630615069059ull}},
{{6905701882809449174ull, 390904492055247ull}},
{{16592607950473290309ull, 312723593644197ull}},
{{11790777461789623201ull, 500357749830716ull}},
{{5743273154689788238ull, 400286199864573ull}},
{{11973316153235651236ull, 320228959891658ull}},
{{15467957030435131655ull, 512366335826653ull}},
{{1306319180122374355ull, 409893068661323ull}},
{{8423752973581720130ull, 327914454929058ull}},
{{9788655942988841885ull, 524663127886493ull}},
{{15209622383874894154ull, 419730502309194ull}},
{{15857046721841825647ull, 335784401847355ull}},
{{6924530681237369419ull, 537255042955769ull}},
{{9228973359731805858ull, 429804034364615ull}},
{{7383178687785444686ull, 343843227491692ull}},
{{15502434715198621822ull, 550149163986707ull}},
{{5023250142675076811ull, 440119331189366ull}},
{{329251299398151125ull, 352095464951493ull}},
{{7642098669002341547ull, 281676371961194ull}},
{{1159311426178015505ull, 450682195137911ull}},
{{15684844399910053697ull, 360545756110328ull}},
{{1479829075702311988ull, 288436604888263ull}},
{{17125121780091340474ull, 461498567821220ull}},
{{13700097424073072379ull, 369198854256976ull}},
{{7270729124516547580ull, 295359083405581ull}},
{{4254468969742655482ull, 472574533448930ull}},
{{3403575175794124385ull, 378059626759144ull}},
{{6412208955377209831ull, 302447701407315ull}},
{{10259534328603535730ull, 483916322251704ull}},
{{11896976277624738907ull, 387133057801363ull}},
{{16896278651583611772ull, 309706446241090ull}},
{{8587301768824227220ull, 495530313985745ull}},
{{6869841415059381776ull, 396424251188596ull}},
{{1806524317305595097ull, 317139400950877ull}},
{{6579787722430862479ull, 507423041521403ull}},
{{12642527807428510630ull, 405938433217122ull}},
{{2735324616458987857ull, 324750746573698ull}},
{{687170571592470248ull, 519601194517917ull}},
{{11617782901499707168ull, 415680955614333ull}},
{{16672923950683586381ull, 332544764491466ull}},
{{851236617900365947ull, 532071623186347ull}},
{{11749035738546023727ull, 425657298549077ull}},
{{2020530961352998335ull, 340525838839262ull}},
{{6922198352906707660ull, 544841342142819ull}},
{{9227107497067276451ull, 435873073714255ull}},
{{7381685997653821161ull, 348698458971404ull}},
{{742651152020382888ull, 557917534354247ull}},
{{11662167365842037280ull, 446334027483397ull}},
{{1951036263189809178ull, 357067221986718ull}},
{{8939526640035667988ull, 285653777589374ull}},
{{3235196179831337812ull, 457046044142999ull}},
{{6277505758606980573ull, 365636835314399ull}},
{{8711353421627494781ull, 292509468251519ull}},
{{2870119030378260681ull, 468015149202431ull}},
{{17053490483270249837ull, 374412119361944ull}},
{{17332141201358110193ull, 299529695489555ull}},
{{9284681848463424693ull, 479247512783289ull}},
{{11117094293512650077ull, 383398010226631ull}},
{{5204326620068209739ull, 306718408181305ull}},
{{8326922592109135582ull, 490749453090088ull}},
{{14040235703171129112ull, 392599562472070ull}},
{{11232188562536903290ull, 314079649977656ull}},
{{10592804070575224617ull, 502527439964250ull}},
{{8474243256460179694ull, 402021951971400ull}},
{{6779394605168143755ull, 321617561577120ull}},
{{10847031368269030008ull, 514588098523392ull}},
{{1298927465131403360ull, 411670478818714ull}},
{{4728490786847033011ull, 329336383054971ull}},
{{186887629471432172ull, 526938212887954ull}},
{{3838858918319056060ull, 421550570310363ull}},
{{10449784764139065495ull, 337240456248290ull}},
{{16719655622622504792ull, 539584729997264ull}},
{{17065073312839914156ull, 431667783997811ull}},
{{9962709835530021002ull, 345334227198249ull}},
{{4872289292622302634ull, 552534763517199ull}},
{{7587180248839752430ull, 442027810813759ull}},
{{9759093013813712267ull, 353622248651007ull}},
{{428576781567149167ull, 282897798920806ull}},
{{11753769294733169637ull, 452636478273289ull}},
{{13092364250528446033ull, 362109182618631ull}},
{{6784542585680846503ull, 289687346094905ull}},
{{10855268137089354405ull, 463499753751848ull}},
{{16062912139155304170ull, 370799803001478ull}},
{{1782283267098512367ull, 296639842401183ull}},
{{17609048486325261080ull, 474623747841892ull}},
{{6708541159576388217ull, 379698998273514ull}},
{{9056181742403020897ull, 303759198618811ull}},
{{7111193158361012789ull, 486014717790098ull}},
{{13067652156172630878ull, 388811774232078ull}},
{{17832819354421925348ull, 311049419385662ull}},
{{13775115708107439265ull, 497679071017060ull}},
{{11020092566485951412ull, 398143256813648ull}},
{{16194771682672581776ull, 318514605450918ull}},
{{3775541803824668902ull, 509623368721470ull}},
{{3020433443059735122ull, 407698694977176ull}},
{{17173742013415429390ull, 326158955981740ull}},
{{9031243147755135408ull, 521854329570785ull}},
{{7224994518204108327ull, 417483463656628ull}},
{{13158693244047107308ull, 333986770925302ull}},
{{6296513931507730400ull, 534378833480484ull}},
{{8726559959948094643ull, 427503066784387ull}},
{{18049294412184206684ull, 342002453427509ull}},
{{17810824615268999725ull, 547203925484015ull}},
{{14248659692215199780ull, 437763140387212ull}},
{{4020230124288339177ull, 350210512309770ull}},
{{6432368198861342684ull, 560336819695632ull}},
{{16213941003314805117ull, 448269455756505ull}},
{{12971152802651844093ull, 358615564605204ull}},
{{14066271056863385598ull, 286892451684163ull}},
{{369940802529955017ull, 459027922694662ull}},
{{11363999086249694983ull, 367222338155729ull}},
{{12780548083741666310ull, 293777870524583ull}},
{{16759528119244755773ull, 470044592839333ull}},
{{2339576051170073649ull, 376035674271467ull}},
{{12939707285161789888ull, 300828539417173ull}},
{{17014182841516953498ull, 481325663067477ull}},
{{6232648643729742152ull, 385060530453982ull}},
{{16054165359209524691ull, 308048424363185ull}},
{{7239920501025687890ull, 492877478981097ull}},
{{16859982845046281282ull, 394301983184877ull}},
{{6109288646553204379ull, 315441586547902ull}},
{{13464210649227037330ull, 504706538476643ull}},
{{18150066148865450510ull, 403765230781314ull}},
{{18209401733834270731ull, 323012184625051ull}},
{{3309601070941460908ull, 516819495400083ull}},
{{10026378486236989373ull, 413455596320066ull}},
{{4331753974247681175ull, 330764477056053ull}},
{{3241457544054379557ull, 529223163289685ull}},
{{2593166035243503645ull, 423378530631748ull}},
{{9453230457678623562ull, 338702824505398ull}},
{{11435819917543887377ull, 541924519208637ull}},
{{1769958304551289255ull, 433539615366910ull}},
{{1415966643641031404ull, 346831692293528ull}},
{{17022941888793291540ull, 554930707669644ull}},
{{17307702325776543555ull, 443944566135715ull}},
{{13846161860621234844ull, 355155652908572ull}},
{{3698231859013167228ull, 284124522326858ull}},
{{2227822159679157243ull, 454599235722973ull}},
{{9160955357227146440ull, 363679388578378ull}},
{{14707461915265537799ull, 290943510862702ull}},
{{8774543805457219185ull, 465509617380324ull}},
{{10708983859107685671ull, 372407693904259ull}},
{{12256535902028058860ull, 297926155123407ull}},
{{4853062184277252884ull, 476681848197452ull}},
{{14950496191647533277ull, 381345478557961ull}},
{{8271048138576116298ull, 305076382846369ull}},
{{2165630577496055107ull, 488122212554191ull}},
{{16489899720964485379ull, 390497770043352ull}},
{{5813222147287767656ull, 312398216034682ull}},
{{12990504250402338574ull, 499837145655491ull}},
{{6703054585579960536ull, 399869716524393ull}},
{{12741141297947789075ull, 319895773219514ull}},
{{9317779632490731550ull, 511833237151223ull}},
{{14832921335476405887ull, 409466589720978ull}},
{{798290624155393740ull, 327573271776783ull}},
{{16034660257616271276ull, 524117234842852ull}},
{{5449030576609196375ull, 419293787874282ull}},
{{15427270905513088069ull, 335435030299425ull}},
{{6236889375111389295ull, 536696048479081ull}},
{{1300162685347201113ull, 429356838783265ull}},
{{1040130148277760890ull, 343485471026612ull}},
{{5353557051986327748ull, 549576753642579ull}},
{{7972194456330972521ull, 439661402914063ull}},
{{13756453194548598663ull, 351729122331250ull}},
{{3563581037568206246ull, 562766595730001ull}},
{{17608260089022206289ull, 450213276584000ull}},
{{14086608071217765031ull, 360170621267200ull}},
{{11269286456974212025ull, 288136497013760ull}},
{{18030858331158739240ull, 461018395222016ull}},
{{10735337850185081069ull, 368814716177613ull}},
{{15966967909631885501ull, 295051772942090ull}},
{{7100404581701465186ull, 472082836707345ull}},
{{5680323665361172149ull, 377666269365876ull}},
{{854910117547027396ull, 302133015492701ull}},
{{12435902632300974803ull, 483412824788321ull}},
{{6259373291098869519ull, 386730259830657ull}},
{{16075545077104826585ull, 309384207864525ull}},
{{7274128049658170920ull, 495014732583241ull}},
{{2129953624984626413ull, 396011786066593ull}},
{{9082660529471521777ull, 316809428853274ull}},
{{3464210402928703873ull, 506895086165239ull}},
{{6460717137084873422ull, 405516068932191ull}},
{{1479224894925988414ull, 324412855145753ull}},
{{17124155090849222756ull, 519060568233204ull}},
{{17388672887421288528ull, 415248454586563ull}},
{{2842891865711299852ull, 332198763669251ull}},
{{15616673429363810734ull, 531518021870801ull}},
{{8803989928749138264ull, 425214417496641ull}},
{{3353843128257400288ull, 340171533997313ull}},
{{1676800190469930137ull, 544274454395701ull}},
{{16098835411343585403ull, 435419563516560ull}},
{{12879068329074868322ull, 348335650813248ull}},
{{16917160511777878992ull, 557337041301197ull}},
{{6155030779938482547ull, 445869633040958ull}},
{{12302722253434606684ull, 356695706432766ull}},
{{6152828988005775024ull, 285356565146213ull}},
{{6155177566067329716ull, 456570504233941ull}},
{{1234793238111953449ull, 365256403387153ull}},
{{8366532219973383406ull, 292205122709722ull}},
{{17075800366699323773ull, 467528196335555ull}},
{{13660640293359459018ull, 374022557068444ull}},
{{14617861049429477537ull, 299218045654755ull}},
{{4941833605377612444ull, 478748873047609ull}},
{{7642815699044000278ull, 382999098438087ull}},
{{17182299003460931192ull, 306399278750469ull}},
{{16423631961311758938ull, 490238846000751ull}},
{{9449556754307496827ull, 392191076800601ull}},
{{3870296588704087139ull, 313752861440481ull}},
{{17260520986152270392ull, 502004578304769ull}},
{{17497765603663726636ull, 401603662643815ull}},
{{13998212482930981309ull, 321282930115052ull}},
{{7639744713721928802ull, 514052688184084ull}},
{{9801144585719453365ull, 411242150547267ull}},
{{462218039091742045ull, 328993720437814ull}},
{{8118246492030607919ull, 526389952700502ull}},
{{17562643637850217305ull, 421111962160401ull}},
{{10360766095538263520ull, 336889569728321ull}},
{{9198528123377400987ull, 539023311565314ull}},
{{11048171313443831112ull, 431218649252251ull}},
{{5149188236013154567ull, 344974919401801ull}},
{{860003548137226661ull, 551959871042882ull}},
{{11756049282735512298ull, 441567896834305ull}},
{{9404839426188409838ull, 353254317467444ull}},
{{11213220355692638194ull, 282603453973955ull}},
{{17941152569108221110ull, 452165526358328ull}},
{{3284875611060845918ull, 361732421086663ull}},
{{10006598118332497381ull, 289385936869330ull}},
{{16010556989331995810ull, 463017498990928ull}},
{{1740399147239865678ull, 370413999192743ull}},
{{8771016947275713189ull, 296331199354194ull}},
{{2965580671415410133ull, 474129918966711ull}},
{{17129859796099969399ull, 379303935173368ull}},
{{2635841392654244550ull, 303443148138695ull}},
{{4217346228246791280ull, 485509037021912ull}},
{{14441923426823163993ull, 388407229617529ull}},
{{15242887556200441518ull, 310725783694023ull}},
{{2252527201469244489ull, 497161253910438ull}},
{{9180719390659216238ull, 397729003128350ull}},
{{7344575512527372990ull, 318183202502680ull}},
{{11751320820043796785ull, 509093124004288ull}},
{{16779754285518858074ull, 407274499203430ull}},
{{13423803428415086459ull, 325819599362744ull}},
{{10410039041238407365ull, 521311358980391ull}},
{{4638682418248815569ull, 417049087184313ull}},
{{11089643564082873101ull, 333639269747450ull}},
{{17743429702532596963ull, 533822831595920ull}},
{{14194743762026077570ull, 427058265276736ull}},
{{7666446194878951733ull, 341646612221389ull}},
{{1198267467580591803ull, 546634579554223ull}},
{{8337311603548294089ull, 437307663643378ull}},
{{14048546912322455917ull, 349846130914702ull}},
{{7720279800748288175ull, 559753809463524ull}},
{{9865572655340540863ull, 447803047570819ull}},
{{11581806939014343014ull, 358242438056655ull}},
{{9265445551211474411ull, 286593950445324ull}},
{{3756666437712628088ull, 458550320712519ull}},
{{6694681964912012793ull, 366840256570015ull}},
{{5355745571929610235ull, 293472205256012ull}},
{{12258541729829286699ull, 469555528409619ull}},
{{13496182198605339682ull, 375644422727695ull}},
{{10796945758884271746ull, 300515538182156ull}},
{{9896415584731014147ull, 480824861091450ull}},
{{7917132467784811318ull, 384659888873160ull}},
{{6333705974227849054ull, 307727911098528ull}},
{{6444580744022648163ull, 492364657757645ull}},
{{5155664595218118531ull, 393891726206116ull}},
{{435182861432584501ull, 315113380964893ull}},
{{15453687837259776495ull, 504181409543828ull}},
{{1294903825582090226ull, 403345127635063ull}},
{{8414620689949492827ull, 322676102108050ull}},
{{13463393103919188524ull, 516281763372880ull}},
{{10770714483135350819ull, 413025410698304ull}},
{{12305920401250190978ull, 330420328558643ull}},
{{16000123827258395243ull, 528672525693829ull}},
{{16489447876548626517ull, 422938020555063ull}},
{{2123511857013170244ull, 338350416444051ull}},
{{14465665415446803360ull, 541360666310481ull}},
{{7883183517615532365ull, 433088533048385ull}},
{{6306546814092425892ull, 346470826438708ull}},
{{6401126087805971104ull, 554353322301933ull}},
{{12499598499728597530ull, 443482657841546ull}},
{{6310329985040967700ull, 354786126273237ull}},
{{16116310432258505130ull, 283828901018589ull}},
{{14718050247387877238ull, 454126241629743ull}},
{{706393753684570821ull, 363300993303795ull}},
{{565115002947656657ull, 290640794643036ull}},
{{11972230448941981621ull, 465025271428857ull}},
{{2199086729669764650ull, 372020217143086ull}},
{{16516664642703453013ull, 297616173714468ull}},
{{4290570539874062881ull, 476185877943150ull}},
{{3432456431899250305ull, 380948702354520ull}},
{{2745965145519400244ull, 304758961883616ull}},
{{15461590677056771360ull, 487614339013785ull}},
{{12369272541645417088ull, 390091471211028ull}},
{{17274115662800154317ull, 312073176968822ull}},
{{12881189801512605614ull, 499317083150116ull}},
{{6615603026468174168ull, 399453666520093ull}},
{{12671180050658359981ull, 319562933216074ull}},
{{9205841636827645000ull, 511300693145719ull}},
{{11054022124204026323ull, 409040554516575ull}},
{{8843217699363221058ull, 327232443613260ull}},
{{14149148318981153694ull, 523571909781216ull}},
{{7629969840443012632ull, 418857527824973ull}},
{{13482673501838230752ull, 335086022259978ull}},
{{17882928788199258880ull, 536137635615965ull}},
{{14306343030559407104ull, 428910108492772ull}},
{{4066376794963705036ull, 343128086794218ull}},
{{2816854057200017735ull, 549004938870749ull}},
{{5942832060501924511ull, 439203951096599ull}},
{{8443614463143449932ull, 351363160877279ull}},
{{2441736696803788922ull, 562181057403647ull}},
{{13021435801668762107ull, 449744845922917ull}},
{{3038451011851189039ull, 359795876738334ull}},
{{6120109624222861555ull, 287836701390667ull}},
{{13481524213498488811ull, 460538722225067ull}},
{{3406521741314970402ull, 368430977780054ull}},
{{6414566207793886645ull, 294744782224043ull}},
{{6573957117728308309ull, 471591651558469ull}},
{{8948514508924556970ull, 377273321246775ull}},
{{7158811607139645576ull, 301818656997420ull}},
{{11454098571423432922ull, 482909851195872ull}},
{{1784581227654925691ull, 386327880956698ull}},
{{8806362611607761199ull, 309062304765358ull}},
{{10400831363830507596ull, 494499687624573ull}},
{{15699362720548226723ull, 395599750099658ull}},
{{1491443732212850409ull, 316479800079727ull}},
{{6075658786282470977ull, 506367680127563ull}},
{{12239224658509797428ull, 405094144102050ull}},
{{9791379726807837942ull, 324075315281640ull}},
{{15666207562892540708ull, 518520504450624ull}},
{{16222314865055942890ull, 414816403560499ull}},
{{16667200706786664635ull, 331853122848399ull}},
{{15599474686632932446ull, 530964996557439ull}},
{{16168928564048256280ull, 424771997245951ull}},
{{9245794036496694701ull, 339817597796761ull}},
{{7414572828910890875ull, 543708156474818ull}},
{{13310355892612533346ull, 434966525179854ull}},
{{14337633528831937000ull, 347973220143883ull}},
{{804120757679637262ull, 556757152230214ull}},
{{4332645420885620132ull, 445405721784171ull}},
{{18223511595676137399ull, 356324577427336ull}},
{{10889460461798999596ull, 285059661941869ull}},
{{6355090294652668384ull, 456095459106991ull}},
{{1394723420980224384ull, 364876367285593ull}},
{{8494476366268000153ull, 291901093828474ull}},
{{2523115741803069276ull, 467041750125559ull}},
{{5707841408184365744ull, 373633400100447ull}},
{{15634319570773223564ull, 298906720080357ull}},
{{10257516054269516411ull, 478250752128572ull}},
{{827315213931792482ull, 382600601702858ull}},
{{8040549800629254632ull, 306080481362286ull}},
{{5486182051522986765ull, 489728770179658ull}},
{{11767643270702210058ull, 391783016143726ull}},
{{5724765801819857723ull, 313426412914981ull}},
{{1780927653427951711ull, 501482260663970ull}},
{{1424742122742361369ull, 401185808531176ull}},
{{15897188957161530388ull, 320948646824940ull}},
{{6988758257748897005ull, 513517834919905ull}},
{{5591006606199117604ull, 410814267935924ull}},
{{8162154099701204406ull, 328651414348739ull}},
{{1991400115296196080ull, 525842262957983ull}},
{{8971817721720777511ull, 420673810366386ull}},
{{3488105362634711685ull, 336539048293109ull}},
{{12959666209699359343ull, 538462477268974ull}},
{{14057081782501397798ull, 430769981815179ull}},
{{14935014240743028561ull, 344615985452143ull}},
{{1759929896737383759ull, 551385576723430ull}},
{{1407943917389907007ull, 441108461378744ull}},
{{4815703948653835929ull, 352886769102995ull}},
{{3852563158923068743ull, 282309415282396ull}},
{{17232147498502640958ull, 451695064451833ull}},
{{2717671554576381797ull, 361356051561467ull}},
{{13242183687886836407ull, 289084841249173ull}},
{{17498145085877027929ull, 462535745998677ull}},
{{6619818439217801696ull, 370028596798942ull}},
{{16363901195599972327ull, 296022877439153ull}},
{{4046149024508493784ull, 473636603902646ull}},
{{17994314478574436320ull, 378909283122116ull}},
{{10706102768117638732ull, 303127426497693ull}},
{{13440415614246311649ull, 485003882396309ull}},
{{14441681306138959642ull, 388003105917047ull}},
{{4174647415427347067ull, 310402484733638ull}},
{{2990087049941844985ull, 496643975573821ull}},
{{17149464898921117280ull, 397315180459056ull}},
{{10030223104394983501ull, 317852144367245ull}},
{{16048356967031973602ull, 508563430987592ull}},
{{5459987944141758235ull, 406850744790074ull}},
{{8057339170055316911ull, 325480595832059ull}},
{{1823696227862776089ull, 520768953331295ull}},
{{1458956982290220871ull, 416615162665036ull}},
{{15924560844799817989ull, 333292130132028ull}},
{{3343204463228246844ull, 533267408211246ull}},
{{17431958829550238768ull, 426613926568996ull}},
{{10256218248898280691ull, 341291141255197ull}},
{{1652553939269607813ull, 546065826008316ull}},
{{16079438410383327543ull, 436852660806652ull}},
{{5484853098822841388ull, 349482128645322ull}},
{{12465113772858456544ull, 559171405832515ull}},
{{9972091018286765235ull, 447337124666012ull}},
{{598975185145591542ull, 357869699732810ull}},
{{479180148116473233ull, 286295759786248ull}},
{{15524083495953998467ull, 458073215657996ull}},
{{8729917982021288450ull, 366458572526397ull}},
{{18051980829842761729ull, 293166858021117ull}},
{{14125774068780777475ull, 469066972833788ull}},
{{232572810798891010ull, 375253578267031ull}},
{{14943453507606754101ull, 300202862613624ull}},
{{12841479167945075592ull, 480324580181799ull}},
{{13962532149097970797ull, 384259664145439ull}},
{{14859374534020286960ull, 307407731316351ull}},
{{16396301624948638490ull, 491852370106162ull}},
{{5738343670475090146ull, 393481896084930ull}},
{{4590674936380072117ull, 314785516867944ull}},
{{14723777527691936033ull, 503656826988710ull}},
{{11779022022153548826ull, 402925461590968ull}},
{{16801915247206659707ull, 322340369272774ull}},
{{15815017951304924563ull, 515744590836439ull}},
{{16341363175785849973ull, 412595672669151ull}},
{{9383741725886769655ull, 330076538135321ull}},
{{7635289131935010802ull, 528122461016514ull}},
{{9797580120289918965ull, 422497968813211ull}},
{{4148715281490024849ull, 337998375050569ull}},
{{14016642079867860405ull, 540797400080910ull}},
{{11213313663894288324ull, 432637920064728ull}},
{{16349348560599251305ull, 346110336051782ull}},
{{11401562437991160796ull, 553776537682852ull}},
{{1742552320909107990ull, 443021230146282ull}},
{{12462088300953017362ull, 354416984117025ull}},
{{9969670640762413889ull, 283533587293620ull}},
{{15951473025219862223ull, 453653739669792ull}},
{{5382480790692069132ull, 362922991735834ull}},
{{7995333447295565629ull, 290338393388667ull}},
{{16481882330414815329ull, 464541429421867ull}},
{{5806808234848031617ull, 371633143537494ull}},
{{8334795402620335617ull, 297306514829995ull}},
{{13335672644192536987ull, 475690423727992ull}},
{{3289840485870208943ull, 380552338982394ull}},
{{6321221203438077478ull, 304441871185915ull}},
{{10113953925500923964ull, 487106993897464ull}},
{{11780511955142649495ull, 389685595117971ull}},
{{5735060749372209272ull, 311748476094377ull}},
{{12865446013737445159ull, 498797561751003ull}},
{{17671054440473776774ull, 399038049400802ull}},
{{6758145922895200772ull, 319230439520642ull}},
{{14502382291374231559ull, 510768703233027ull}},
{{4223208203615564601ull, 408614962586422ull}},
{{14446613007118182650ull, 326891970069137ull}},
{{8357185552421450948ull, 523027152110620ull}},
{{6685748441937160758ull, 418421721688496ull}},
{{1659249938807818283ull, 334737377350797ull}},
{{6344148716834419577ull, 535579803761275ull}},
{{5075318973467535661ull, 428463843009020ull}},
{{4060255178774028529ull, 342771074407216ull}},
{{17564454730264176616ull, 548433719051545ull}},
{{14051563784211341293ull, 438746975241236ull}},
{{7551902212627162711ull, 350997580192989ull}},
{{1014997095977729368ull, 561596128308783ull}},
{{8190695306266004141ull, 449276902647026ull}},
{{2863207430270892990ull, 359421522117621ull}},
{{17047961203184355684ull, 287537217694096ull}},
{{1451296221901596833ull, 460059548310555ull}},
{{1161036977521277466ull, 368047638648444ull}},
{{4618178396758932296ull, 294438110918755ull}},
{{7389085434814291674ull, 471100977470008ull}},
{{13289965977335253985ull, 376880781976006ull}},
{{6942623967126292865ull, 301504625580805ull}},
{{11108198347402068584ull, 482407400929288ull}},
{{16265256307405475514ull, 385925920743430ull}},
{{13012205045924380411ull, 308740736594744ull}},
{{9751481629253277688ull, 493985178551591ull}},
{{4111836488660711827ull, 395188142841273ull}},
{{10668166820412390108ull, 316150514273018ull}},
{{13379718097917913850ull, 505840822836829ull}},
{{14393123293076241403ull, 404672658269463ull}},
{{446452190235262153ull, 323738126615571ull}},
{{11782369948602150414ull, 517981002584913ull}},
{{16804593588365540978ull, 414384802067930ull}},
{{13443674870692432782ull, 331507841654344ull}},
{{10441833348882161482ull, 530412546646951ull}},
{{4664117864363818862ull, 424330037317561ull}},
{{41945476749144767ull, 339464029854049ull}},
{{7445810392282452273ull, 543142447766478ull}},
{{13335345943309782465ull, 434513958213182ull}},
{{3289579125164005326ull, 347611166570546ull}},
{{16331373044488139491ull, 556177866512873ull}},
{{1997051991364780623ull, 444942293210299ull}},
{{5286990407833734821ull, 355953834568239ull}},
{{7918941141008898180ull, 284763067654591ull}},
{{5291608196130416442ull, 455620908247346ull}},
{{543937742162422831ull, 364496726597877ull}},
{{11503196637955669234ull, 291597381278301ull}},
{{11026416991245250128ull, 466555810045282ull}},
{{1442435963512379456ull, 373244648036226ull}},
{{15911344029777544858ull, 298595718428980ull}},
{{7011406373934520156ull, 477753149486369ull}},
{{9298473913889526448ull, 382202519589095ull}},
{{7438779131111621158ull, 305762015671276ull}},
{{4523348980294773207ull, 489219225074042ull}},
{{14686725628461549535ull, 391375380059233ull}},
{{681334058543508659ull, 313100304047387ull}},
{{4779483308411524177ull, 500960486475819ull}},
{{7512935461471129665ull, 400768389180655ull}},
{{6010348369176903732ull, 320614711344524ull}},
{{16995255020166866618ull, 512983538151238ull}},
{{2528157571907762324ull, 410386830520991ull}},
{{16779921316493851152ull, 328309464416792ull}},
{{12090478847422520551ull, 525295143066868ull}},
{{17051080707421837087ull, 420236114453494ull}},
{{17330213380679379993ull, 336188891562795ull}},
{{9281597335377456373ull, 537902226500473ull}},
{{14803975497785785745ull, 430321781200378ull}},
{{775133954002897626ull, 344257424960303ull}},
{{15997609585372277495ull, 550811879936484ull}},
{{16487436483039732319ull, 440649503949187ull}},
{{5811251556947965209ull, 352519603159350ull}},
{{4649001245558372167ull, 282015682527480ull}},
{{7438401992893395467ull, 451225092043968ull}},
{{13329419223798537020ull, 360980073635174ull}},
{{14352884193780739939ull, 288784058908139ull}},
{{11896568265823452933ull, 462054494253023ull}},
{{16895952242142582993ull, 369643595402418ull}},
{{2448715349488335425ull, 295714876321935ull}},
{{3917944559181336680ull, 473143802115096ull}},
{{17891750906312710636ull, 378515041692076ull}},
{{10624051910308258186ull, 302812033353661ull}},
{{9619785427009392451ull, 484499253365858ull}},
{{15074525971091334607ull, 387599402692686ull}},
{{8370271962131157363ull, 310079522154149ull}},
{{2324388695184120811ull, 496127235446639ull}},
{{5548859770889206972ull, 396901788357311ull}},
{{749739001969455254ull, 317521430685849ull}},
{{8578280032634949053ull, 508034289097358ull}},
{{14241321655591779889ull, 406427431277886ull}},
{{7703708509731513588ull, 325141945022309ull}},
{{1257887171344690771ull, 520227112035695ull}},
{{1006309737075752617ull, 416181689628556ull}},
{{15562443048628243386ull, 332945351702844ull}},
{{13831862433579458449ull, 532712562724551ull}},
{{7376141132121656436ull, 426170050179641ull}},
{{2211564090955414825ull, 340936040143713ull}},
{{18295897804496305013ull, 545497664229940ull}},
{{14636718243597044011ull, 436398131383952ull}},
{{4330676965393814562ull, 349118505107162ull}},
{{10618431959372013623ull, 558589608171459ull}},
{{12184094382239521221ull, 446871686537167ull}},
{{2368577876307796330ull, 357497349229734ull}},
{{5584211115788147387ull, 285997879383787ull}},
{{12624086600002946143ull, 457596607014059ull}},
{{13788618094744267238ull, 366077285611247ull}},
{{3652196846311593144ull, 292861828488998ull}},
{{2154166139356638707ull, 468578925582397ull}},
{{12791379355711041935ull, 374863140465917ull}},
{{2854405855085012902ull, 299890512372734ull}},
{{11945746997619841289ull, 479824819796374ull}},
{{13245946412837783355ull, 383859855837099ull}},
{{14286105945012137007ull, 307087884669679ull}},
{{11789723067793688242ull, 491340615471487ull}},
{{2053080824751129947ull, 393072492377190ull}},
{{1642464659800903957ull, 314457993901752ull}},
{{6317292270423356655ull, 503132790242803ull}},
{{12432531445822505970ull, 402506232194242ull}},
{{2567327527174184130ull, 322004985755394ull}},
{{11486421672962515254ull, 515207977208630ull}},
{{9189137338370012203ull, 412166381766904ull}},
{{11040658685437920086ull, 329733105413523ull}},
{{13975705081958761815ull, 527572968661637ull}},
{{3801866436083188805ull, 422058374929310ull}},
{{3041493148866551044ull, 337646699943448ull}},
{{1177040223444571347ull, 540234719909517ull}},
{{12009678622981388047ull, 432187775927613ull}},
{{16986440527868931084ull, 345750220742090ull}},
{{8731560770880738119ull, 553200353187345ull}},
{{6985248616704590495ull, 442560282549876ull}},
{{1898850078621762073ull, 354048226039901ull}},
{{16276475321865050951ull, 283238580831920ull}},
{{7595616441274529906ull, 453181729331073ull}},
{{13455190782503444571ull, 362545383464858ull}},
{{18142850255486576303ull, 290036306771886ull}},
{{3203118705585149823ull, 464058090835019ull}},
{{6251843779210030181ull, 371246472668015ull}},
{{5001475023368024145ull, 296997178134412ull}},
{{11691708852130748956ull, 475195485015059ull}},
{{13042715896446509488ull, 380156388012047ull}},
{{3055475087673386944ull, 304125110409638ull}},
{{1199411325535508787ull, 486600176655421ull}},
{{15716924319396048322ull, 389280141324336ull}},
{{8884190640774928334ull, 311424113059469ull}},
{{3146658581014154366ull, 498278580895151ull}},
{{17274722123778964785ull, 398622864716120ull}},
{{13819777699023171828ull, 318898291772896ull}},
{{14732946688953254279ull, 510237266836634ull}},
{{15475706165904513746ull, 408189813469307ull}},
{{5001867303239790350ull, 326551850775446ull}},
{{624290055699843915ull, 522482961240714ull}},
{{4188780859301785455ull, 417986368992571ull}},
{{18108419946409069657ull, 334389095194056ull}},
{{3148030211061139188ull, 535022552310491ull}},
{{17275819427816552643ull, 428018041848392ull}},
{{6441957912769421468ull, 342414433478714ull}},
{{17685830289914894996ull, 547863093565942ull}},
{{6769966602448095350ull, 438290474852754ull}},
{{9105322096700386603ull, 350632379882203ull}},
{{10879166539978708242ull, 561011807811525ull}},
{{8703333231982966594ull, 448809446249220ull}},
{{6962666585586373275ull, 359047556999376ull}},
{{1880784453727188297ull, 287238045599501ull}},
{{14077301570189232244ull, 459580872959201ull}},
{{7572492441409475472ull, 367664698367361ull}},
{{2368645138385670055ull, 294131758693889ull}},
{{11168529850900892734ull, 470610813910222ull}},
{{1556126251236893541ull, 376488651128178ull}},
{{8623598630473335479ull, 301190920902542ull}},
{{17487106623499247090ull, 481905473444067ull}},
{{6610987669315577025ull, 385524378755254ull}},
{{8978138950194371943ull, 308419503004203ull}},
{{10675673505569084786ull, 493471204806725ull}},
{{8540538804455267829ull, 394776963845380ull}},
{{6832431043564214263ull, 315821571076304ull}},
{{18310587299186563468ull, 505314513722086ull}},
{{10959121024607340451ull, 404251610977669ull}},
{{12456645634427782684ull, 323401288782135ull}},
{{1483888941374900678ull, 517442062051417ull}},
{{12255157597325651512ull, 413953649641133ull}},
{{17182823707344341856ull, 331162919712906ull}},
{{1667076228557574707ull, 529860671540651ull}},
{{16091056241813701059ull, 423888537232520ull}},
{{12872844993450960847ull, 339110829786016ull}},
{{13217854360037716709ull, 542577327657626ull}},
{{6884934673288263044ull, 434061862126101ull}},
{{1818598923888700112ull, 347249489700881ull}},
{{13977804722447651149ull, 555599183521409ull}},
{{14871592592700031242ull, 444479346817127ull}},
{{4518576444676204347ull, 355583477453702ull}},
{{14682907599966694447ull, 284466781962961ull}},
{{16113954530462890469ull, 455146851140738ull}},
{{1823117180144581406ull, 364117480912591ull}},
{{16215889003083306417ull, 291293984730072ull}},
{{11188027145965648975ull, 466070375568116ull}},
{{5261072902030608857ull, 372856300454493ull}},
{{11587555951108307732ull, 298285040363594ull}},
{{7472043077547561402ull, 477256064581751ull}},
{{2288285647296138798ull, 381804851665401ull}},
{{16588023776804552331ull, 305443881332320ull}},
{{8094093969177732114ull, 488710210131713ull}},
{{13853972804826006338ull, 390968168105370ull}},
{{11083178243860805070ull, 312774534484296ull}},
{{10354387560693467466ull, 500439255174874ull}},
{{11972858863296684296ull, 400351404139899ull}},
{{13267635905379257760ull, 320281123311919ull}},
{{10160171004381081446ull, 512449797299071ull}},
{{4438787988762954834ull, 409959837839257ull}},
{{14619076835236094836ull, 327967870271405ull}},
{{4943778862668200123ull, 524748592434249ull}},
{{7644371904876470421ull, 419798873947399ull}},
{{9804846338643086660ull, 335839099157919ull}},
{{4619707697603207687ull, 537342558652671ull}},
{{6417343340655826ull, 429874046922137ull}},
{{11073180318898255630ull, 343899237537709ull}},
{{6649042066011478039ull, 550238780060335ull}},
{{5319233652809182431ull, 440191024048268ull}},
{{11634084551731166591ull, 352152819238614ull}},
{{12996616456126843596ull, 281722255390891ull}},
{{13415888700319129108ull, 450755608625426ull}},
{{7043362145513392963ull, 360604486900341ull}},
{{1945340901668804047ull, 288483589520273ull}},
{{17869940701637727768ull, 461573743232436ull}},
{{10606603746568271891ull, 369258994585949ull}},
{{12174631811996527836ull, 295407195668759ull}},
{{8411364454968713569ull, 472651513070015ull}},
{{6729091563974970855ull, 378121210456012ull}},
{{16451319695405707653ull, 302496968364809ull}},
{{15254065068423401276ull, 483995149383695ull}},
{{12203252054738721021ull, 387196119506956ull}},
{{6073252829049066493ull, 309756895605565ull}},
{{9717204526478506390ull, 495611032968904ull}},
{{11463112435924715435ull, 396488826375123ull}},
{{16549187578223592994ull, 317191061100098ull}},
{{4342607236706286852ull, 507505697760158ull}},
{{10852783418848850128ull, 406004558208126ull}},
{{4992877920337169779ull, 324803646566501ull}},
{{609907043055651000ull, 519685834506402ull}},
{{11555972078670251770ull, 415748667605121ull}},
{{5555428848194291092ull, 332598934084097ull}},
{{12578034971852776071ull, 532158294534555ull}},
{{10062427977482220857ull, 425726635627644ull}},
{{11739291196727687009ull, 340581308502115ull}},
{{336121841054747598ull, 544930093603385ull}},
{{268897472843798078ull, 435944074882708ull}},
{{7593815607758859109ull, 348755259906166ull}},
{{4771407342930353928ull, 558008415849866ull}},
{{127777059602372819ull, 446406732679893ull}},
{{7480919277165718902ull, 357125386143914ull}},
{{9674084236474485445ull, 285700308915131ull}},
{{8099837148875356065ull, 457120494264210ull}},
{{6479869719100284852ull, 365696395411368ull}},
{{12562593404764048528ull, 292557116329094ull}},
{{9032103003396746675ull, 468091386126551ull}},
{{3536333587975487017ull, 374473108901241ull}},
{{17586462129348030906ull, 299578487120992ull}},
{{13380944147989208158ull, 479325579393588ull}},
{{18083452947875187172ull, 383460463514870ull}},
{{14466762358300149738ull, 306768370811896ull}},
{{15768122143796418934ull, 490829393299034ull}},
{{16303846529779045471ull, 392663514639227ull}},
{{5664379594339415730ull, 314130811711382ull}},
{{12752356165684975491ull, 502609298738211ull}},
{{6512536117806070070ull, 402087438990569ull}},
{{8899377708986766379ull, 321669951192455ull}},
{{14239004334378826207ull, 514671921907928ull}},
{{323157023277329996ull, 411737537526343ull}},
{{7637223248105684643ull, 329390030021074ull}},
{{1151510752743364459ull, 527024048033719ull}},
{{4610557416936601890ull, 421619238426975ull}},
{{3688445933549281512ull, 337295390741580ull}},
{{5901513493678850420ull, 539672625186528ull}},
{{12099908424426900982ull, 431738100149222ull}},
{{2301229110057700139ull, 345390480119378ull}},
{{18439361835059961516ull, 552624768191004ull}},
{{18440838282789879536ull, 442099814552803ull}},
{{3684624182006172659ull, 353679851642243ull}},
{{10326396975088758774ull, 282943881313794ull}},
{{5454188715916283068ull, 452710210102071ull}},
{{674002157991116131ull, 362168168081657ull}},
{{11607248170618623875ull, 289734534465325ull}},
{{124852999280246584ull, 463575255144521ull}},
{{14857277658391838560ull, 370860204115616ull}},
{{8196473311971560524ull, 296688163292493ull}},
{{9425008484412586516ull, 474701061267989ull}},
{{11229355602271979536ull, 379760849014391ull}},
{{5294135667075673305ull, 303808679211513ull}},
{{4781268252579166966ull, 486093886738421ull}},
{{135665787321423249ull, 388875109390737ull}},
{{11176579074082869569ull, 311100087512589ull}},
{{6814480074306860341ull, 497760140020143ull}},
{{12830281688929308919ull, 398208112016114ull}},
{{13953574165885357458ull, 318566489612891ull}},
{{14947021035932751287ull, 509706383380626ull}},
{{8268268014004290707ull, 407765106704501ull}},
{{2925265596461522242ull, 326212085363601ull}},
{{15748471398564166557ull, 521939336581761ull}},
{{8909428304109422922ull, 417551469265409ull}},
{{10816891458029448661ull, 334041175412327ull}},
{{2549631073879476565ull, 534465880659724ull}},
{{5729053673845491575ull, 427572704527779ull}},
{{8272591753818303583ull, 342058163622223ull}},
{{9546797991367375410ull, 547293061795557ull}},
{{258740763610079682ull, 437834449436446ull}},
{{14964387869855705038ull, 350267559549156ull}},
{{16564322962285307415ull, 560428095278650ull}},
{{13251458369828245932ull, 448342476222920ull}},
{{10601166695862596745ull, 358673980978336ull}},
{{4791584541948167073ull, 286939184782669ull}},
{{15045232896600887963ull, 459102695652270ull}},
{{12036186317280710371ull, 367282156521816ull}},
{{5939600239082657973ull, 293825725217453ull}},
{{5814011567790342434ull, 470121160347925ull}},
{{4651209254232273947ull, 376096928278340ull}},
{{3720967403385819158ull, 300877542622672ull}},
{{9642896660159220976ull, 481404068196275ull}},
{{7714317328127376781ull, 385123254557020ull}},
{{6171453862501901424ull, 308098603645616ull}},
{{2495628550519221633ull, 492957765832986ull}},
{{16753898099383018599ull, 394366212666388ull}},
{{2335072035280683910ull, 315492970133111ull}},
{{14804161700674825225ull, 504788752212977ull}},
{{4464631731056039534ull, 403831001770382ull}},
{{14639751829070562596ull, 323064801416305ull}},
{{4976858852803348539ull, 516903682266089ull}},
{{7670835896984589154ull, 413522945812871ull}},
{{2447319902845761000ull, 330818356650297ull}},
{{7605060659295127923ull, 529309370640475ull}},
{{6084048527436102339ull, 423447496512380ull}},
{{4867238821948881871ull, 338757997209904ull}},
{{15166279744602031640ull, 542012795535846ull}},
{{8443674980939714989ull, 433610236428677ull}},
{{17822986428977502960ull, 346888189142941ull}},
{{2691336583170632474ull, 555021102628707ull}},
{{13221115710762236949ull, 444016882102965ull}},
{{10576892568609789559ull, 355213505682372ull}},
{{1082816425404011001ull, 284170804545898ull}},
{{16489901539614058894ull, 454673287273436ull}},
{{9502572416949336792ull, 363738629818749ull}},
{{11291406748301379757ull, 290990903854999ull}},
{{6998204353056476642ull, 465585446167999ull}},
{{9287912297187091636ull, 372468356934399ull}},
{{11119678652491583632ull, 297974685547519ull}},
{{6723439399760802842ull, 476759496876031ull}},
{{1689402705066731951ull, 381407597500825ull}},
{{1351522164053385560ull, 305126078000660ull}},
{{2162435462485416897ull, 488201724801056ull}},
{{16487343628955974810ull, 390561379840844ull}},
{{16879223717906690171ull, 312449103872675ull}},
{{8560013874941152658ull, 499918566196281ull}},
{{3158662285211011803ull, 399934852957025ull}},
{{2526929828168809443ull, 319947882365620ull}},
{{4043087725070095108ull, 511916611784992ull}},
{{14302516624281807056ull, 409533289427993ull}},
{{373966855199714675ull, 327626631542395ull}},
{{598346968319543481ull, 524202610467832ull}},
{{11546724018881365754ull, 419362088374265ull}},
{{9237379215105092603ull, 335489670699412ull}},
{{22411485200506873ull, 536783473119060ull}},
{{17929188160405498ull, 429426778495248ull}},
{{7393040980012145045ull, 343541422796198ull}},
{{8139516753277521749ull, 549666276473917ull}},
{{17579659846847748368ull, 439733021179133ull}},
{{2995681433252467725ull, 351786416943307ull}},
{{8482439107945858684ull, 562858267109291ull}},
{{3096602471614776624ull, 450286613687433ull}},
{{9855979606775641945ull, 360229290949946ull}},
{{4195434870678603233ull, 288183432759957ull}},
{{10402044607827675496ull, 461093492415931ull}},
{{4632286871520230073ull, 368874793932745ull}},
{{3705829497216184059ull, 295099835146196ull}},
{{16997373639771625464ull, 472159736233913ull}},
{{2529852467591569401ull, 377727788987131ull}},
{{16781277233040896814ull, 302182231189704ull}},
{{15781997128639703933ull, 483491569903527ull}},
{{5246900073427942500ull, 386793255922822ull}},
{{15265566502968084969ull, 309434604738257ull}},
{{9667511145781294658ull, 495095367581212ull}},
{{355311287141215080ull, 396076294064970ull}},
{{284249029712972064ull, 316861035251976ull}},
{{11522844891766486272ull, 506977656403161ull}},
{{5528927098671278695ull, 405582125122529ull}},
{{8112490493678933279ull, 324465700098023ull}},
{{9290635975144382923ull, 519145120156837ull}},
{{53811150631685692ull, 415316096125470ull}},
{{43048920505348553ull, 332252876900376ull}},
{{11136924717034288655ull, 531604603040601ull}},
{{5220190958885520601ull, 425283682432481ull}},
{{486803952366506158ull, 340226945945985ull}},
{{778886323786409852ull, 544363113513576ull}},
{{15380504317996769175ull, 435490490810860ull}},
{{12304403454397415340ull, 348392392648688ull}},
{{15997696712293954220ull, 557427828237901ull}},
{{9108808555093253053ull, 445942262590321ull}},
{{3597698029332692119ull, 356753810072257ull}},
{{13946204867691884665ull, 285403048057805ull}},
{{3867183714597463848ull, 456644876892489ull}},
{{6783095786419881402ull, 365315901513991ull}},
{{1737127814393994798ull, 292252721211193ull}},
{{17536799761998032970ull, 467604353937908ull}},
{{2961393365372695406ull, 374083483150327ull}},
{{13437161136523887294ull, 299266786520261ull}},
{{14120760188954399025ull, 478826858432418ull}},
{{228561706937788250ull, 383061486745935ull}},
{{182849365550230600ull, 306449189396748ull}},
{{15049954243848010253ull, 490318703034796ull}},
{{8350614580336497879ull, 392254962427837ull}},
{{17748538108494929273ull, 313803969942269ull}},
{{17329614529366155867ull, 502086351907631ull}},
{{10174342808751014371ull, 401669081526105ull}},
{{8139474247000811496ull, 321335265220884ull}},
{{1955112350975567425ull, 514136424353415ull}},
{{1564089880780453940ull, 411309139482732ull}},
{{12319318348850094121ull, 329047311586185ull}},
{{1264165284450598978ull, 526475698537897ull}},
{{12079378671786210152ull, 421180558830317ull}},
{{2284805307945147475ull, 336944447064254ull}},
{{11034386122196056607ull, 539111115302806ull}},
{{5138160083014934962ull, 431288892242245ull}},
{{4110528066411947970ull, 345031113793796ull}},
{{17644891350484847722ull, 552049782070073ull}},
{{3047866636162147208ull, 441639825656059ull}},
{{6127642123671628089ull, 353311860524847ull}},
{{15970160143163033441ull, 282649488419877ull}},
{{10794860970093212213ull, 452239181471804ull}},
{{12325237590816480093ull, 361791345177443ull}},
{{17238887702137004721ull, 289433076141954ull}},
{{16514173879193476584ull, 463092921827127ull}},
{{5832641473870960621ull, 370474337461702ull}},
{{15734159623322499466ull, 296379469969361ull}},
{{17795957767832178500ull, 474207151950978ull}},
{{3168719770040011830ull, 379365721560783ull}},
{{9913673445515830111ull, 303492577248626ull}},
{{8483179883341507531ull, 485588123597802ull}},
{{17854590350898936994ull, 388470498878241ull}},
{{10594323465977239272ull, 310776399102593ull}},
{{13261568730821672512ull, 497242238564149ull}},
{{14298603799399248333ull, 397793790851319ull}},
{{15128231854261308989ull, 318235032681055ull}},
{{5758426893108542767ull, 509176052289689ull}},
{{8296090329228744537ull, 407340841831751ull}},
{{2947523448641085306ull, 325872673465401ull}},
{{15784083962051467460ull, 521396277544641ull}},
{{8937918354899263645ull, 417117022035713ull}},
{{14529032313403231562ull, 333693617628570ull}},
{{4799707627735618884ull, 533909788205713ull}},
{{11218463731672315753ull, 427127830564570ull}},
{{8974770985337852602ull, 341702264451656ull}},
{{6980935947056743518ull, 546723623122650ull}},
{{5584748757645394814ull, 437378898498120ull}},
{{4467799006116315851ull, 349903118798496ull}},
{{18216524854011836332ull, 559844990077593ull}},
{{3505173438983738096ull, 447875992062075ull}},
{{2804138751186990477ull, 358300793649660ull}},
{{2243311000949592381ull, 286640634919728ull}},
{{18346692860486989103ull, 458625015871564ull}},
{{18366703103131501605ull, 366900012697251ull}},
{{11004013667763290961ull, 293520010157801ull}},
{{10227724238937444892ull, 469632016252482ull}},
{{803481761666135267ull, 375705613001986ull}},
{{15400180668300549506ull, 300564490401588ull}},
{{2504196180829417271ull, 480903184642542ull}},
{{13071403388889264786ull, 384722547714033ull}},
{{17835820340595232475ull, 307778038171226ull}},
{{2711870841758999698ull, 492444861073963ull}},
{{9548194302891020405ull, 393955888859170ull}},
{{7638555442312816324ull, 315164711087336ull}},
{{4842991078216685472ull, 504263537739738ull}},
{{11253090492057169024ull, 403410830191790ull}},
{{9002472393645735219ull, 322728664153432ull}},
{{18093304644575086674ull, 516365862645491ull}},
{{10785294900918159016ull, 413092690116393ull}},
{{16006933550218347859ull, 330474152093114ull}},
{{14543047236123625605ull, 528758643348983ull}},
{{566391344673169514ull, 423006914679187ull}},
{{11521159519964266581ull, 338405531743349ull}},
{{7365808787717095560ull, 541448850789359ull}},
{{9581995844915586771ull, 433159080631487ull}},
{{286899046448648771ull, 346527264505190ull}},
{{459038474317838033ull, 554443623208304ull}},
{{4056579594196180750ull, 443554898566643ull}},
{{10623961304840765246ull, 354843918853314ull}},
{{12188517858614522520ull, 283875135082651ull}},
{{12122930944299415386ull, 454200216132242ull}},
{{2319647125955711662ull, 363360172905794ull}},
{{5545066515506479653ull, 290688138324635ull}},
{{8872106424810367445ull, 465101021319416ull}},
{{3408336325106383633ull, 372080817055533ull}},
{{10105366689568927552ull, 297664653644426ull}},
{{8789889073826463438ull, 476263445831082ull}},
{{18099957703286901720ull, 381010756664865ull}},
{{14479966162629521376ull, 304808605331892ull}},
{{8410550601239592908ull, 487693768531028ull}},
{{14107138110475494973ull, 390155014824822ull}},
{{3907012858896575332ull, 312124011859858ull}},
{{2561871759492610208ull, 499398418975773ull}},
{{9428195037077908813ull, 399518735180618ull}},
{{14921253659146147697ull, 319614988144494ull}},
{{12805959410408105345ull, 511383981031191ull}},
{{6555418713584573953ull, 409107184824953ull}},
{{12623032600351479809ull, 327285747859962ull}},
{{5439456901594726401ull, 523657196575940ull}},
{{4351565521275781121ull, 418925757260752ull}},
{{14549298861246355866ull, 335140605808601ull}},
{{15900180548510348740ull, 536224969293762ull}},
{{5341446809324458345ull, 428979975435010ull}},
{{4273157447459566676ull, 343183980348008ull}},
{{3147703101193396359ull, 549094368556813ull}},
{{9896860110438537734ull, 439275494845450ull}},
{{7917488088350830187ull, 351420395876360ull}},
{{12667980941361328299ull, 562272633402176ull}},
{{6445035938347152316ull, 449818106721741ull}},
{{1466679935935811529ull, 359854485377393ull}},
{{8552041578232469870ull, 287883588301914ull}},
{{2615220080946220823ull, 460613741283063ull}},
{{9470873694240797304ull, 368490993026450ull}},
{{7576698955392637843ull, 294792794421160ull}},
{{12122718328628220550ull, 471668471073856ull}},
{{6008825848160666116ull, 377334776859085ull}},
{{4807060678528532893ull, 301867821487268ull}},
{{4001948270903742306ull, 482988514379629ull}},
{{6890907431464904168ull, 386390811503703ull}},
{{12891423574655743981ull, 309112649202962ull}},
{{5868882460481549076ull, 494580238724740ull}},
{{4695105968385239261ull, 395664190979792ull}},
{{14824131218933922378ull, 316531352783833ull}},
{{1582517061842813866ull, 506450164454134ull}},
{{4955362464216161416ull, 405160131563307ull}},
{{15032336415598660103ull, 324128105250645ull}},
{{5604994191248304548ull, 518604968401033ull}},
{{11862692982482464285ull, 414883974720826ull}},
{{5800805571244061105ull, 331907179776661ull}},
{{1902591284506677121ull, 531051487642658ull}},
{{8900770657089162343ull, 424841190114126ull}},
{{3431267710929419551ull, 339872952091301ull}},
{{16558074781712802252ull, 543796723346081ull}},
{{9557111010628331478ull, 435037378676865ull}},
{{7645688808502665183ull, 348029902941492ull}},
{{15922450908346174616ull, 556847844706387ull}},
{{5359263097193119046ull, 445478275765110ull}},
{{4287410477754495237ull, 356382620612088ull}},
{{10808626011687416836ull, 285106096489670ull}},
{{17293801618699866937ull, 456169754383472ull}},
{{6456343665476072903ull, 364935803506778ull}},
{{12543772561864678969ull, 291948642805422ull}},
{{5312640840015845058ull, 467117828488676ull}},
{{560763857270765723ull, 373694262790941ull}},
{{15206006344784253871ull, 298955410232752ull}},
{{9572214892687164901ull, 478328656372404ull}},
{{11347120728891642244ull, 382662925097923ull}},
{{16456394212597134442ull, 306130340078338ull}},
{{4194137851703953168ull, 489808544125342ull}},
{{14423356725588893504ull, 391846835300273ull}},
{{470638936245383833ull, 313477468240219ull}},
{{8131719927476434780ull, 501563949184350ull}},
{{6505375941981147824ull, 401251159347480ull}},
{{5204300753584918259ull, 321000927477984ull}},
{{15705578835219689861ull, 513601483964774ull}},
{{16253811882917662212ull, 410881187171819ull}},
{{16692398321076040093ull, 328704949737455ull}},
{{8261093240012112533ull, 525927919579929ull}},
{{10298223406751600349ull, 420742335663943ull}},
{{15617276354885100926ull, 336593868531154ull}},
{{13919595723590430512ull, 538550189649847ull}},
{{3756978949388523763ull, 430840151719878ull}},
{{10384280788994639657ull, 344672121375902ull}},
{{1857454003423782158ull, 551475394201444ull}},
{{5175312017480936050ull, 441180315361155ull}},
{{4140249613984748840ull, 352944252288924ull}},
{{7001548505929709395ull, 282355401831139ull}},
{{134431165261804062ull, 451768642929823ull}},
{{7486242561693263896ull, 361414914343858ull}},
{{13367691678838431763ull, 289131931475086ull}},
{{14009609056657670175ull, 462611090360138ull}},
{{139640801100405170ull, 370088872288111ull}},
{{14869107899847965429ull, 296071097830488ull}},
{{1654479751305282747ull, 473713756528782ull}},
{{12391630245269957167ull, 378971005223025ull}},
{{9913304196215965734ull, 303176804178420ull}},
{{15861286713945545174ull, 485082886685472ull}},
{{5310331741672615493ull, 388066309348378ull}},
{{11626963022821913041ull, 310453047478702ull}},
{{3845745577547419573ull, 496724875965924ull}},
{{6765945276779845981ull, 397379900772739ull}},
{{9102105036165787108ull, 317903920618191ull}},
{{7184670428381438727ull, 508646272989106ull}},
{{2058387527963240658ull, 406917018391285ull}},
{{1646710022370592526ull, 325533614713028ull}},
{{17392131294760589335ull, 520853783540844ull}},
{{17603053850550381791ull, 416683026832675ull}},
{{14082443080440305433ull, 333346421466140ull}},
{{4085164854994937077ull, 533354274345825ull}},
{{3268131883995949662ull, 426683419476660ull}},
{{2614505507196759729ull, 341346735581328ull}},
{{493859996772905244ull, 546154776930125ull}},
{{395087997418324195ull, 436923821544100ull}},
{{316070397934659356ull, 349539057235280ull}},
{{505712636695454970ull, 559262491576448ull}},
{{7783267738840184622ull, 447409993261158ull}},
{{13605311820555968344ull, 357927994608926ull}},
{{7194900641702864352ull, 286342395687141ull}},
{{4133143397240762317ull, 458147833099426ull}},
{{18063909976760251146ull, 366518266479540ull}},
{{14451127981408200917ull, 293214613183632ull}},
{{8364409511285480174ull, 469143381093812ull}},
{{17759574053254115109ull, 375314704875049ull}},
{{17897008057345202410ull, 300251763900039ull}},
{{17567166447526592887ull, 480402822240063ull}},
{{2985686713795543340ull, 384322257792051ull}},
{{17145944630004075965ull, 307457806233640ull}},
{{8986767334296969928ull, 491932489973825ull}},
{{7189413867437575942ull, 393545991979060ull}},
{{5751531093950060754ull, 314836793583248ull}},
{{5513100935578186883ull, 503738869733197ull}},
{{15478527192688280476ull, 402991095786557ull}},
{{5004124124666803734ull, 322392876629246ull}},
{{627900969983065329ull, 515828602606794ull}},
{{4191669590728362586ull, 412662882085435ull}},
{{3353335672582690069ull, 330130305668348ull}},
{{1675988261390393787ull, 528208489069357ull}},
{{12408837053338045999ull, 422566791255485ull}},
{{9927069642670436799ull, 338053433004388ull}},
{{12193962613530788556ull, 540885492807021ull}},
{{6065821276082720521ull, 432708394245617ull}},
{{15920703465091907387ull, 346166715396493ull}},
{{3337032655695589880ull, 553866744634390ull}},
{{2669626124556471904ull, 443093395707512ull}},
{{13203747343870908492ull, 354474716566009ull}},
{{14252346689838637117ull, 283579773252807ull}},
{{8046359444774178095ull, 453727637204492ull}},
{{17505134000045073445ull, 362982109763593ull}},
{{2936060755810327787ull, 290385687810875ull}},
{{4697697209296524459ull, 464617100497400ull}},
{{3758157767437219567ull, 371693680397920ull}},
{{3006526213949775653ull, 297354944318336ull}},
{{15878488386545372015ull, 475767910909337ull}},
{{5324093079752476966ull, 380614328727470ull}},
{{4259274463801981573ull, 304491462981976ull}},
{{17882885586308901486ull, 487186340771161ull}},
{{10616959654305210865ull, 389749072616929ull}},
{{12182916538186079015ull, 311799258093543ull}},
{{15803317646355816102ull, 498878812949669ull}},
{{16332002931826563205ull, 399103050359735ull}},
{{13065602345461250564ull, 319282440287788ull}},
{{17215614937996090579ull, 510851904460461ull}},
{{10083143135654962140ull, 408681523568369ull}},
{{11755863323265880035ull, 326945218854695ull}},
{{362637243515856440ull, 523112350167513ull}},
{{7668807424296505798ull, 418489880134010ull}},
{{6135045939437204639ull, 334791904107208ull}},
{{6126724688357617099ull, 535667046571533ull}},
{{12280077380169914325ull, 428533637257226ull}},
{{6134713089394021137ull, 342826909805781ull}},
{{2436843313546613173ull, 548523055689250ull}},
{{1949474650837290538ull, 438818444551400ull}},
{{1559579720669832431ull, 351054755641120ull}},
{{2495327553071731889ull, 561687609025792ull}},
{{13064308486683116481ull, 449350087220633ull}},
{{17830144418830313831ull, 359480069776506ull}},
{{10574766720322340742ull, 287584055821205ull}},
{{16919626752515745187ull, 460134489313928ull}},
{{2467654957786865180ull, 368107591451143ull}},
{{9352821595713312790ull, 294486073160914ull}},
{{3896468108915569495ull, 471177717057463ull}},
{{10495872116616276242ull, 376942173645970ull}},
{{8396697693293020994ull, 301553738916776ull}},
{{6056018679785012944ull, 482485982266842ull}},
{{15912861388053741325ull, 385988785813473ull}},
{{1662242666217262090ull, 308791028650779ull}},
{{10038285895431439991ull, 494065645841246ull}},
{{4341279901603241669ull, 395252516672997ull}},
{{14541070365508324305ull, 316202013338397ull}},
{{8508317325845677595ull, 505923221341436ull}},
{{3117305045934631753ull, 404738577073149ull}},
{{6183192851489615725ull, 323790861658519ull}},
{{17271806191867205807ull, 518065378653630ull}},
{{13817444953493764646ull, 414452302922904ull}},
{{14743304777536922040ull, 331561842338323ull}},
{{1453194755607613325ull, 530498947741318ull}},
{{8541253433969911306ull, 424399158193054ull}},
{{10522351561917839368ull, 339519326554443ull}},
{{13146413684326632666ull, 543230922487109ull}},
{{14206479762203216456ull, 434584737989687ull}},
{{3986486180278752518ull, 347667790391750ull}},
{{6378377888446004029ull, 556268464626800ull}},
{{5102702310756803223ull, 445014771701440ull}},
{{4082161848605442578ull, 356011817361152ull}},
{{14333775923110085032ull, 284809453888921ull}},
{{15555343847492315406ull, 455695126222274ull}},
{{16133623892735762648ull, 364556100977819ull}},
{{16596247928930520441ull, 291644880782255ull}},
{{8107252612579281090ull, 466631809251609ull}},
{{10175150904805335195ull, 373305447401287ull}},
{{761423094360447510ull, 298644357921030ull}},
{{1218276950976716016ull, 477830972673648ull}},
{{8353319190265193459ull, 382264778138918ull}},
{{14061352981695975413ull, 305811822511134ull}},
{{11430118326487829692ull, 489298916017815ull}},
{{9144094661190263754ull, 391439132814252ull}},
{{18383322173177941972ull, 313151306251401ull}},
{{3587873773891334894ull, 501042090002243ull}},
{{10248996648596888561ull, 400833672001794ull}},
{{11888546133619421172ull, 320666937601435ull}},
{{574929740081522260ull, 513067100162297ull}},
{{11527990236290948777ull, 410453680129837ull}},
{{1843694559548938375ull, 328362944103870ull}},
{{2949911295278301401ull, 525380710566192ull}},
{{13427975480448372090ull, 420304568452953ull}},
{{18121078013842518318ull, 336243654762362ull}},
{{14236329563180388017ull, 537989847619780ull}},
{{11389063650544310414ull, 430391878095824ull}},
{{12800599735177358654ull, 344313502476659ull}},
{{9412913132058042877ull, 550901603962655ull}},
{{7530330505646434301ull, 440721283170124ull}},
{{9713613219259057764ull, 352577026536099ull}},
{{11460239390149156535ull, 282061621228879ull}},
{{7268336580012919486ull, 451298593966207ull}},
{{16882715708236066558ull, 361038875172965ull}},
{{13506172566588853246ull, 288831100138372ull}},
{{6852480847574523902ull, 462129760221396ull}},
{{1792635863317708798ull, 369703808177117ull}},
{{12502155134879898008ull, 295763046541693ull}},
{{16314099401065926490ull, 473220874466709ull}},
{{16740628335594651515ull, 378576699573367ull}},
{{6013805038991900566ull, 302861359658694ull}},
{{17000785691870861552ull, 484578175453910ull}},
{{13600628553496689241ull, 387662540363128ull}},
{{18259200472281172039ull, 310130032290502ull}},
{{14457325496682233970ull, 496208051664804ull}},
{{15255209212087697499ull, 396966441331843ull}},
{{1136120925444427030ull, 317573153065475ull}},
{{1817793480711083248ull, 508117044904760ull}},
{{1454234784568866598ull, 406493635923808ull}},
{{8542085457138913925ull, 325194908739046ull}},
{{6288639101938441634ull, 520311853982474ull}},
{{8720260096292663630ull, 416249483185979ull}},
{{10665556891776041227ull, 332999586548783ull}},
{{13375542212099755641ull, 532799338478053ull}},
{{18079131399163625159ull, 426239470782442ull}},
{{7084607489847079481ull, 340991576625954ull}},
{{267325539529596200ull, 545586522601527ull}},
{{11281906875849407929ull, 436469218081221ull}},
{{5336176685937616020ull, 349175374464977ull}},
{{12227231512242095956ull, 558680599143963ull}},
{{17160482839277497411ull, 446944479315170ull}},
{{13728386271421997928ull, 357555583452136ull}},
{{7293360202395688019ull, 286044466761709ull}},
{{601329879607369862ull, 457671146818735ull}},
{{481063903685895889ull, 366136917454988ull}},
{{7763548752432537358ull, 292909533963990ull}},
{{12421678003892059773ull, 468655254342384ull}},
{{13626691217855558141ull, 374924203473907ull}},
{{3522655344800625867ull, 299939362779126ull}},
{{16704294995906732356ull, 479902980446601ull}},
{{9674087181983475562ull, 383922384357281ull}},
{{4049920930844870126ull, 307137907485825ull}},
{{6479873489351792202ull, 491420651977320ull}},
{{5183898791481433762ull, 393136521581856ull}},
{{457770218443236686ull, 314509217265485ull}},
{{732432349509178698ull, 503214747624776ull}},
{{15343341138574984251ull, 402571798099820ull}},
{{12274672910859987401ull, 322057438479856ull}},
{{12260779027892159195ull, 515291901567770ull}},
{{9808623222313727356ull, 412233521254216ull}},
{{4157549763109071561ull, 329786817003373ull}},
{{2962730806232604175ull, 527658907205397ull}},
{{13438231089211814310ull, 422127125764317ull}},
{{3371887241885630801ull, 337701700611454ull}},
{{12773717216500829929ull, 540322720978326ull}},
{{6529624958458753620ull, 432258176782661ull}},
{{1534351152025092573ull, 345806541426129ull}},
{{9833659472723968763ull, 553290466281806ull}},
{{4177578763437264687ull, 442632373025445ull}},
{{3342063010749811749ull, 354105898420356ull}},
{{17431045667567490692ull, 283284718736284ull}},
{{16821626623882254138ull, 453255549978055ull}},
{{13457301299105803311ull, 362604439982444ull}},
{{14455189854026552972ull, 290083551985955ull}},
{{4681559692732933139ull, 464133683177529ull}},
{{7434596568928256834ull, 371306946542023ull}},
{{13326374884626426114ull, 297045557233618ull}},
{{17632851000660371459ull, 475272891573789ull}},
{{17795629615270207490ull, 380218313259031ull}},
{{10547154877474255669ull, 304174650607225ull}},
{{16875447803958809070ull, 486679440971560ull}},
{{13500358243167047256ull, 389343552777248ull}},
{{18178984224017458451ull, 311474842221798ull}},
{{6950281869976471583ull, 498359747554878ull}},
{{12938923125464997913ull, 398687798043902ull}},
{{2972440870888177684ull, 318950238435122ull}},
{{8445254208162994618ull, 510320381496195ull}},
{{6756203366530395694ull, 408256305196956ull}},
{{1715613878482406232ull, 326605044157565ull}},
{{2744982205571849971ull, 522568070652104ull}},
{{5885334579199390300ull, 418054456521683ull}},
{{12086965292843332886ull, 334443565217346ull}},
{{11960446839065511972ull, 535109704347754ull}},
{{13257706285994319901ull, 428087763478203ull}},
{{17984862658279276567ull, 342470210782562ull}},
{{14018384994279201215ull, 547952337252100ull}},
{{11214707995423360972ull, 438361869801680ull}},
{{8971766396338688777ull, 350689495841344ull}},
{{3286779789916171074ull, 561103193346151ull}},
{{17386819090900578152ull, 448882554676920ull}},
{{13909455272720462522ull, 359106043741536ull}},
{{7438215403434459694ull, 287284834993229ull}},
{{833098201269404541ull, 459655735989167ull}},
{{11734525005241254602ull, 367724588791333ull}},
{{16766317633676824328ull, 294179671033066ull}},
{{1000666510689546663ull, 470687473652907ull}},
{{11868579652777368300ull, 376549978922325ull}},
{{9494863722221894640ull, 301239983137860ull}},
{{15191781955555031424ull, 481983973020576ull}},
{{8464076749702114816ull, 385587178416461ull}},
{{3081912585019781529ull, 308469742733169ull}},
{{12309757765515471094ull, 493551588373070ull}},
{{9847806212412376875ull, 394841270698456ull}},
{{4188896155187991177ull, 315873016558765ull}},
{{6702233848300785883ull, 505396826494024ull}},
{{9051135893382539029ull, 404317461195219ull}},
{{10930257529447941547ull, 323453968956175ull}},
{{17488412047116706475ull, 517526350329880ull}},
{{13990729637693365180ull, 414021080263904ull}},
{{14881932524896602467ull, 331216864211123ull}},
{{1674999151383102008ull, 529946982737798ull}},
{{8718696950590302253ull, 423957586190238ull}},
{{14353655189956062449ull, 339166068952190ull}},
{{4519104230220148302ull, 542665710323505ull}},
{{3615283384176118641ull, 434132568258804ull}},
{{6581575522082805236ull, 347306054607043ull}},
{{6841172020590578055ull, 555689687371269ull}},
{{9162286431214372767ull, 444551749897015ull}},
{{7329829144971498214ull, 355641399917612ull}},
{{16931909760202929540ull, 284513119934089ull}},
{{16023009172098956295ull, 455220991894543ull}},
{{1750360893453434067ull, 364176793515635ull}},
{{1400288714762747253ull, 291341434812508ull}},
{{16997857202588036898ull, 466146295700012ull}},
{{6219588132586608872ull, 372917036560010ull}},
{{4975670506069287098ull, 298333629248008ull}},
{{4271723994968949033ull, 477333806796813ull}},
{{10796076825458979873ull, 381867045437450ull}},
{{8636861460367183898ull, 305493636349960ull}},
{{13818978336587494237ull, 488789818159936ull}},
{{7365833854528085067ull, 391031854527949ull}},
{{9582015898364378376ull, 312825483622359ull}},
{{4263178993157274433ull, 500520773795775ull}},
{{3410543194525819546ull, 400416619036620ull}},
{{2728434555620655637ull, 320333295229296ull}},
{{15433541733218779989ull, 512533272366873ull}},
{{1278786942349293021ull, 410026617893499ull}},
{{4712378368621344740ull, 328021294314799ull}},
{{14918503019277972231ull, 524834070903678ull}},
{{866755971196646815ull, 419867256722943ull}},
{{8072102406441138098ull, 335893805378354ull}},
{{1847317406080089988ull, 537430088605367ull}},
{{12545900369089802960ull, 429944070884293ull}},
{{17415417924755663014ull, 343955256707434ull}},
{{16796622235383329854ull, 550328410731895ull}},
{{13437297788306663883ull, 440262728585516ull}},
{{7060489415903420783ull, 352210182868413ull}},
{{13027089162206557273ull, 281768146294730ull}},
{{2396598585820940021ull, 450829034071569ull}},
{{5606627683398662340ull, 360663227257255ull}},
{{4485302146718929872ull, 288530581805804ull}},
{{14555181064234108441ull, 461648930889286ull}},
{{7954796036645376430ull, 369319144711429ull}},
{{10053185644058211467ull, 295455315769143ull}},
{{12395748215751228024ull, 472728505230629ull}},
{{13605947387342892742ull, 378182804184503ull}},
{{18263455539358134840ull, 302546243347602ull}},
{{14464133604005374452ull, 484073989356164ull}},
{{15260655697946209884ull, 387259191484931ull}},
{{8519175743615057584ull, 309807353187945ull}},
{{13630681189784092135ull, 495691765100712ull}},
{{3525847322343453062ull, 396553412080570ull}},
{{2820677857874762449ull, 317242729664456ull}},
{{15581131016825350889ull, 507588367463129ull}},
{{16154253628202191034ull, 406070693970503ull}},
{{1855356458336021858ull, 324856555176403ull}},
{{17725965592305276265ull, 519770488282244ull}},
{{17870121288586131335ull, 415816390625795ull}},
{{14296097030868905068ull, 332653112500636ull}},
{{15495057619906427463ull, 532244980001018ull}},
{{1327999651699411001ull, 425795984000815ull}},
{{1062399721359528800ull, 340636787200652ull}},
{{5389188368917156404ull, 545018859521043ull}},
{{11690048324617545770ull, 436015087616834ull}},
{{13041387474435946939ull, 348812070093467ull}},
{{6108824700129873809ull, 558099312149548ull}},
{{12265757389587719694ull, 446479449719638ull}},
{{17191303541153996401ull, 357183559775710ull}},
{{13753042832923197121ull, 285746847820568ull}},
{{18315519717935205071ull, 457194956512909ull}},
{{18341764589090074380ull, 365755965210327ull}},
{{7294714041788238857ull, 292604772168262ull}},
{{15360891281603092495ull, 468167635469219ull}},
{{15978061840024384319ull, 374534108375375ull}},
{{12782449472019507455ull, 299627286700300ull}},
{{2005175081521660313ull, 479403658720481ull}},
{{16361535324184969543ull, 383522926976384ull}},
{{16778577074089885957ull, 306818341581107ull}},
{{12088328059576176239ull, 490909346529772ull}},
{{2291964818177120345ull, 392727477223818ull}},
{{9212269484025516922ull, 314181981779054ull}},
{{3671584730215096106ull, 502691170846487ull}},
{{14005314228397807855ull, 402152936677189ull}},
{{14893600197460156607ull, 321722349341751ull}},
{{16451062686452429925ull, 514755758946802ull}},
{{5782152519678123293ull, 411804607157442ull}},
{{15693768459968229604ull, 329443685725953ull}},
{{2973936647497705428ull, 527109897161526ull}},
{{17136544576965805635ull, 421687917729220ull}},
{{13709235661572644508ull, 337350334183376ull}},
{{14556079429032410566ull, 539760534693402ull}},
{{4266165913742107807ull, 431808427754722ull}},
{{14480979175219417215ull, 345446742203777ull}},
{{8412171421383426251ull, 552714787526044ull}},
{{10419085951848651324ull, 442171830020835ull}},
{{8335268761478921059ull, 353737464016668ull}},
{{14046912638666957494ull, 282989971213334ull}},
{{11407013777641401020ull, 452783953941335ull}},
{{9125611022113120816ull, 362227163153068ull}},
{{14679186447174317299ull, 289781730522454ull}},
{{12418651871253176710ull, 463650768835927ull}},
{{2556223867518720721ull, 370920615068742ull}},
{{13113025538240707546ull, 296736492054993ull}},
{{17291492046443221751ull, 474778387287989ull}},
{{17522542451896487724ull, 379822709830391ull}},
{{10328685146775279856ull, 303858167864313ull}},
{{12836547420098537447ull, 486173068582901ull}},
{{6579889121336919634ull, 388938454866321ull}},
{{1574562482327625384ull, 311150763893057ull}},
{{6208648786466110938ull, 497841222228891ull}},
{{1277570214430978427ull, 398272977783113ull}},
{{8400753801028603388ull, 318618382226490ull}},
{{13441206081645765421ull, 509789411562384ull}},
{{14442313680058522660ull, 407831529249907ull}},
{{4175153314562997481ull, 326265223399926ull}},
{{17748291747526526940ull, 522024357439881ull}},
{{10509284583279311229ull, 417619485951905ull}},
{{8407427666623448983ull, 334095588761524ull}},
{{2383837822371787403ull, 534552942018439ull}},
{{5596419072639340246ull, 427642353614751ull}},
{{787786443369561873ull, 342113882891801ull}},
{{12328504753617029967ull, 547382212626881ull}},
{{6173454988151713650ull, 437905770101505ull}},
{{4938763990521370920ull, 350324616081204ull}},
{{15280720014318014119ull, 560519385729926ull}},
{{8535227196712500972ull, 448415508583941ull}},
{{3138832942628090454ull, 358732406867153ull}},
{{9889763983586293010ull, 286985925493722ull}},
{{1066227114770427523ull, 459177480789956ull}},
{{15610376950783983311ull, 367341984631964ull}},
{{16177650375369096972ull, 293873587705571ull}},
{{58798897397182893ull, 470197740328915ull}},
{{47039117917746314ull, 376158192263132ull}},
{{11105677738559928021ull, 300926553810505ull}},
{{17769084381695884834ull, 481482486096808ull}},
{{3147221061130976897ull, 385185988877447ull}},
{{13585823293130512487ull, 308148791101957ull}},
{{6979922010041178687ull, 493038065763132ull}},
{{16651984052258673919ull, 394430452610505ull}},
{{13321587241806939135ull, 315544362088404ull}},
{{10246493142665371647ull, 504870979341447ull}},
{{818496884648476671ull, 403896783473158ull}},
{{8033495137202601983ull, 323117426778526ull}},
{{5474894590040342527ull, 516987882845642ull}},
{{15447962116258004991ull, 413590306276513ull}},
{{1290323248780673023ull, 330872245021211ull}},
{{13132563642274807807ull, 529395592033937ull}},
{{3127353284336025599ull, 423516473627150ull}},
{{2501882627468820479ull, 338813178901720ull}},
{{4003012203950112767ull, 542101086242752ull}},
{{14270456207385821183ull, 433680868994201ull}},
{{7727016151166746623ull, 346944695195361ull}},
{{4984528212382973951ull, 555111512312578ull}},
{{11366320199390199807ull, 444089209850062ull}},
{{1714358530028339199ull, 355271367880050ull}},
{{1371486824022671359ull, 284217094304040ull}},
{{2194378918436274175ull, 454747350886464ull}},
{{5444851949490929663ull, 363797880709171ull}},
{{666532744850833407ull, 291038304567337ull}},
{{4755801206503243775ull, 465661287307739ull}},
{{7493989779944505343ull, 372529029846191ull}},
{{2305843009213693951ull, 298023223876953ull}},
{{18446744073709551615ull, 476837158203124ull}},
{{18446744073709551615ull, 381469726562499ull}},
{{18446744073709551615ull, 305175781249999ull}},
{{18446744073709551615ull, 488281249999999ull}},
{{18446744073709551615ull, 390624999999999ull}},
{{18446744073709551615ull, 312499999999999ull}},
{{18446744073709551615ull, 499999999999999ull}},
{{18446744073709551615ull, 399999999999999ull}},
{{18446744073709551615ull, 319999999999999ull}},
{{18446744073709551615ull, 511999999999999ull}},
{{18446744073709551615ull, 409599999999999ull}},
{{18446744073709551615ull, 327679999999999ull}},
{{18446744073709551615ull, 524287999999999ull}},
{{18446744073709551615ull, 419430399999999ull}},
{{18446744073709551615ull, 335544319999999ull}},
{{18446744073709551615ull, 536870911999999ull}},
{{18446744073709551615ull, 429496729599999ull}},
{{18446744073709551615ull, 343597383679999ull}},
{{18446744073709551615ull, 549755813887999ull}},
{{18446744073709551615ull, 439804651110399ull}},
{{18446744073709551615ull, 351843720888319ull}},
{{18446744073709551615ull, 281474976710655ull}},
{{11068046444225730969ull, 450359962737049ull}},
{{12543785970122495098ull, 360287970189639ull}},
{{13724377590839906402ull, 288230376151711ull}},
{{14580306515860029597ull, 461168601842738ull}},
{{596198768462292708ull, 368934881474191ull}},
{{15234354273737475459ull, 295147905179352ull}},
{{9617571579012319442ull, 472236648286964ull}},
{{11383406077951765876ull, 377789318629571ull}},
{{5417376047619502378ull, 302231454903657ull}},
{{12357150490933114128ull, 483570327845851ull}},
{{6196371578004580979ull, 386856262276681ull}},
{{1267748447661754460ull, 309485009821345ull}},
{{2028397516258807136ull, 495176015714152ull}},
{{12690764457232776679ull, 396140812571321ull}},
{{6463262751044311020ull, 316912650057057ull}},
{{14030569216412807955ull, 507060240091291ull}},
{{7535106558388336041ull, 405648192073033ull}},
{{13406782876194489479ull, 324518553658426ull}},
{{14072154972427362520ull, 519229685853482ull}},
{{3879026348458069369ull, 415383748682786ull}},
{{17860616337734096788ull, 332306998946228ull}},
{{6440893251923092922ull, 531691198313966ull}},
{{1463365786796564015ull, 425352958651173ull}},
{{8549390258921071858ull, 340282366920938ull}},
{{9989675599531804650ull, 544451787073501ull}},
{{4302391664883533397ull, 435561429658801ull}},
{{18199308590874468010ull, 348449143727040ull}},
{{10672149671689597200ull, 557518629963265ull}},
{{8537719737351677760ull, 446014903970612ull}},
{{17898222234107073178ull, 356811923176489ull}},
{{18007926602027568865ull, 285449538541191ull}},
{{2987240860050737922ull, 456719261665907ull}},
{{13457839132266321307ull, 365375409332725ull}},
{{10766271305813057046ull, 292300327466180ull}},
{{17226034089300891273ull, 467680523945888ull}},
{{2712780827214982049ull, 374144419156711ull}},
{{16927619920739626932ull, 299315535325368ull}},
{{4948098984731941152ull, 478904856520590ull}},
{{3958479187785552922ull, 383123885216472ull}},
{{14234829794454173307ull, 306499108173177ull}},
{{8018332412159035998ull, 490398573077084ull}},
{{10104014744469139122ull, 392318858461667ull}},
{{704514166091490651ull, 313855086769334ull}},
{{8505920295230205688ull, 502168138830934ull}},
{{10494085050926074874ull, 401734511064747ull}},
{{1016570411257039252ull, 321387608851798ull}},
{{16383907916978904097ull, 514220174162876ull}},
{{9417777518841212954ull, 411376139330301ull}},
{{3844873200331060040ull, 329100911464241ull}},
{{17219843564755427034ull, 526561458342785ull}},
{{13775874851804341627ull, 421249166674228ull}},
{{18399397510927293948ull, 336999333339382ull}},
{{14681640758516029024ull, 539198933343012ull}},
{{4366614977329002573ull, 431359146674410ull}},
{{3493291981863202058ull, 345087317339528ull}},
{{1899918356239212970ull, 552139707743245ull}},
{{1519934684991370376ull, 441711766194596ull}},
{{15973343006960737594ull, 353369412955676ull}},
{{9089325590826679752ull, 282695530364541ull}},
{{7164223315838866956ull, 452312848583266ull}},
{{2042029837929183242ull, 361850278866613ull}},
{{9012321499827167240ull, 289480223093290ull}},
{{14419714399723467584ull, 463168356949264ull}},
{{15225120334520684390ull, 370534685559411ull}},
{{8490747452874637189ull, 296427748447529ull}},
{{2517149480373688533ull, 474284397516047ull}},
{{13081766028524681796ull, 379427518012837ull}},
{{3086715193335924790ull, 303542014410270ull}},
{{4938744309337479665ull, 485667223056432ull}},
{{15019041891695714701ull, 388533778445145ull}},
{{12015233513356571761ull, 310827022756116ull}},
{{11845675991886694171ull, 497323236409786ull}},
{{5787191978767445014ull, 397858589127829ull}},
{{8319102397755866334ull, 318286871302263ull}},
{{9621215021667475812ull, 509258994083621ull}},
{{4007623202592070326ull, 407407195266897ull}},
{{14274145006299387230ull, 325925756213517ull}},
{{8081236751111378276ull, 521481209941628ull}},
{{13843687030372923267ull, 417184967953302ull}},
{{3696251994814517967ull, 333747974362642ull}},
{{9603352006445139071ull, 533996758980227ull}},
{{303983975672290610ull, 427197407184182ull}},
{{11311233624763563458ull, 341757925747345ull}},
{{18097973799621701533ull, 546812681195752ull}},
{{7099681410213540580ull, 437450144956602ull}},
{{16747791572396563433ull, 349960115965281ull}},
{{971024812641129231ull, 559936185544451ull}},
{{15534215109080544677ull, 447948948435560ull}},
{{12427372087264435742ull, 358359158748448ull}},
{{17320595299295369240ull, 286687326998758ull}},
{{5576859590421128845ull, 458699723198014ull}},
{{8150836487078813399ull, 366959778558411ull}},
{{2831320374921140396ull, 293567822846729ull}},
{{11908810229357645280ull, 469708516554766ull}},
{{5837699368744205901ull, 375766813243813ull}},
{{12048857124479185367ull, 300613450595050ull}},
{{831427325457144971ull, 480981520952081ull}},
{{15422537119333357270ull, 384785216761664ull}},
{{16027378510208596139ull, 307828173409331ull}},
{{18265107986849933176ull, 492525077454930ull}},
{{14612086389479946541ull, 394020061963944ull}},
{{15379017926325867556ull, 315216049571155ull}},
{{6159684608411836474ull, 504345679313849ull}},
{{8617096501471379502ull, 403476543451079ull}},
{{10583026015919013925ull, 322781234760863ull}},
{{13243492810728511957ull, 516449975617381ull}},
{{6905445433840899242ull, 413159980493905ull}},
{{5524356347072719393ull, 330527984395124ull}},
{{16217667784800171676ull, 528844775032198ull}},
{{1906087783614406371ull, 423075820025759ull}},
{{5214219041633435420ull, 338460656020607ull}},
{{12032099281355406996ull, 541537049632971ull}},
{{5936330610342415273ull, 433229639706377ull}},
{{15817110932499663188ull, 346583711765101ull}},
{{17928679862515640455ull, 554533938824162ull}},
{{6964246260528691717ull, 443627151059330ull}},
{{5571397008422953374ull, 354901720847464ull}},
{{8146466421480273022ull, 283921376677971ull}},
{{5655648644884616189ull, 454274202684754ull}},
{{8213867730649603275ull, 363419362147803ull}},
{{13949791814003503266ull, 290735489718242ull}},
{{7562271643437963933ull, 465176783549188ull}},
{{13428514944234191793ull, 372141426839350ull}},
{{10742811955387353434ull, 297713141471480ull}},
{{17188499128619765495ull, 476341026354368ull}},
{{2682752858670081426ull, 381072821083495ull}},
{{2146202286936065141ull, 304858256866796ull}},
{{14501970103323435195ull, 487773210986873ull}},
{{533529638433017186ull, 390218568789499ull}},
{{4116172525488324072ull, 312174855031599ull}},
{{13964573670265139162ull, 499479768050558ull}},
{{103612491986380360ull, 399583814440447ull}},
{{11150936437814835258ull, 319667051552357ull}},
{{3084103041536095120ull, 511467282483772ull}},
{{13535328877454607065ull, 409173825987017ull}},
{{3449565472479865006ull, 327339060789614ull}},
{{12898002385451604656ull, 523742497263382ull}},
{{2939704278877463078ull, 418993997810706ull}},
{{17109158682069611755ull, 335195198248564ull}},
{{16306607447085647839ull, 536312317197703ull}},
{{1977239513442787302ull, 429049853758163ull}},
{{8960489240238050488ull, 343239883006530ull}},
{{14336782784380880780ull, 549183812810448ull}},
{{401379783278973655ull, 439347050248359ull}},
{{4010452641365089247ull, 351477640198687ull}},
{{10106073040926053118ull, 562364224317899ull}},
{{11774207247482752818ull, 449891379454319ull}},
{{13108714612728112577ull, 359913103563455ull}},
{{10486971690182490062ull, 287930482850764ull}},
{{5711108260066253130ull, 460688772561223ull}},
{{11947584237536823150ull, 368551018048978ull}},
{{16936765019513279166ull, 294840814439182ull}},
{{12341428772253605373ull, 471745303102692ull}},
{{2494445388319063652ull, 377396242482154ull}},
{{5684905125397161245ull, 301916993985723ull}},
{{5406499385893547669ull, 483067190377157ull}},
{{15393245952940569105ull, 386453752301725ull}},
{{12314596762352455284ull, 309163001841380ull}},
{{1256610746054376838ull, 494660802946209ull}},
{{4694637411585411794ull, 395728642356967ull}},
{{14823756373494060404ull, 316582913885573ull}},
{{1581917309139034708ull, 506532662216918ull}},
{{8644231476795048413ull, 405226129773534ull}},
{{10604733996177949053ull, 324180903818827ull}},
{{2210179134917077193ull, 518689446110124ull}},
{{5457492122675572077ull, 414951556888099ull}},
{{8055342512882367985ull, 331961245510479ull}},
{{1820501576386057807ull, 531137992816767ull}},
{{12524447705334577215ull, 424910394253413ull}},
{{17398255793751482418ull, 339928315402730ull}},
{{9390465196292820253ull, 543885304644369ull}},
{{11201720971776166526ull, 435108243715495ull}},
{{8961376777420933221ull, 348086594972396ull}},
{{6959505214389672507ull, 556938551955834ull}},
{{9256952986253648329ull, 445550841564667ull}},
{{26864759519098016ull, 356440673251734ull}},
{{3710840622357188736ull, 285152538601387ull}},
{{9626693810513412301ull, 456244061762219ull}},
{{11390703863152640164ull, 364995249409775ull}},
{{9112563090522112131ull, 291996199527820ull}},
{{14580100944835379410ull, 467193919244512ull}},
{{4285383126384482882ull, 373755135395610ull}},
{{3428306501107586305ull, 299004108316488ull}},
{{1795941587030227766ull, 478406573306381ull}},
{{16194148528591823505ull, 382725258645104ull}},
{{16644667637615369127ull, 306180206916083ull}},
{{4495375331733128665ull, 489888331065734ull}},
{{7285649080128413255ull, 391910664852587ull}},
{{16896565708328461573ull, 313528531882069ull}},
{{15966458689099807548ull, 501645651011311ull}},
{{9083818136537935715ull, 401316520809049ull}},
{{10956403323972258895ull, 321053216647239ull}},
{{6462198874129883263ull, 513685146635583ull}},
{{12548456728787727257ull, 410948117308466ull}},
{{6349416568288271482ull, 328758493846773ull}},
{{6469717694519324048ull, 526013590154837ull}},
{{16243820599841190208ull, 420810872123869ull}},
{{16684405294614862490ull, 336648697699095ull}},
{{8248304397674228368ull, 538637916318553ull}},
{{13977341147623203341ull, 430910333054842ull}},
{{3803175288614742026ull, 344728266443874ull}},
{{13463778091267407888ull, 551565226310198ull}},
{{18149720102497746957ull, 441252181048158ull}},
{{3451729637772466596ull, 353001744838527ull}},
{{13829430154443704246ull, 282401395870821ull}},
{{14748390617626106148ull, 451842233393314ull}},
{{15488061308842795241ull, 361473786714651ull}},
{{8701100232332325870ull, 289179029371721ull}},
{{6543062742247900745ull, 462686446994754ull}},
{{8923799008540230919ull, 370149157595803ull}},
{{14517736836316005382ull, 296119326076642ull}},
{{8470983679137967319ull, 473790921722628ull}},
{{14155484572794194501ull, 379032737378102ull}},
{{3945690028751534954ull, 303226189902482ull}},
{{10002452860744366251ull, 485161903843971ull}},
{{4312613473853582677ull, 388129523075177ull}},
{{14518137223308597111ull, 310503618460141ull}},
{{15850321927809934732ull, 496805789536226ull}},
{{8990908727506037462ull, 397444631628981ull}},
{{3503378167262919646ull, 317955705303185ull}},
{{5605405067620671435ull, 508729128485096ull}},
{{794975239354626824ull, 406983302788077ull}},
{{11704026635709432429ull, 325586642230461ull}},
{{11347744987651271240ull, 520938627568738ull}},
{{16456893619604837639ull, 416750902054990ull}},
{{13165514895683870111ull, 333400721643992ull}},
{{6307428574126550885ull, 533441154630388ull}},
{{12424640488785061354ull, 426752923704310ull}},
{{9939712391028049083ull, 341402338963448ull}},
{{12214191010902968210ull, 546243742341517ull}},
{{2392655179238553922ull, 436994993873214ull}},
{{5603472958132753460ull, 349595995098571ull}},
{{1586859103528584890ull, 559353592157714ull}},
{{4958836097564778235ull, 447482873726171ull}},
{{277720063309912265ull, 357986298980937ull}},
{{11290222494873660782ull, 286389039184749ull}},
{{6996309547572126281ull, 458222462695599ull}},
{{9286396452799611348ull, 366577970156479ull}},
{{11118465976981599402ull, 293262376125183ull}},
{{14100196748428648720ull, 469219801800293ull}},
{{212110954517188006ull, 375375841440235ull}},
{{169688763613750405ull, 300300673152188ull}},
{{15028897280749641941ull, 480481077043500ull}},
{{12023117824599713552ull, 384384861634800ull}},
{{9618494259679770842ull, 307507889307840ull}},
{{15389590815487633347ull, 492012622892544ull}},
{{16001021467132017001ull, 393610098314035ull}},
{{12800817173705613601ull, 314888078651228ull}},
{{16791958663187071438ull, 503820925841965ull}},
{{13433566930549657150ull, 403056740673572ull}},
{{3368155914955905074ull, 322445392538858ull}},
{{1699700649187537795ull, 515912628062173ull}},
{{8738458148833850882ull, 412730102449738ull}},
{{14369464148550901352ull, 330184081959790ull}},
{{4544398563971890548ull, 528294531135665ull}},
{{3635518851177512438ull, 422635624908532ull}},
{{13976461525167740920ull, 338108499926825ull}},
{{3915594366558833856ull, 540973599882921ull}},
{{17889870752214708378ull, 432778879906336ull}},
{{10622547787029856379ull, 346223103925069ull}},
{{5928030015022039237ull, 553956966280111ull}},
{{1053075197275721066ull, 443165573024089ull}},
{{4531808972562487176ull, 354532458419271ull}},
{{18382842437017631034ull, 283625966735416ull}},
{{3587106196034837392ull, 453801546776667ull}},
{{13937731401053600883ull, 363041237421333ull}},
{{82138676617149737ull, 290432989937067ull}},
{{3820770697329349902ull, 464692783899307ull}},
{{14124663002089210891ull, 371754227119445ull}},
{{11299730401671368713ull, 297403381695556ull}},
{{10700871013190369294ull, 475845410712890ull}},
{{8560696810552295435ull, 380676328570312ull}},
{{17916603892667567318ull, 304541062856249ull}},
{{17598519784042376739ull, 487265700569999ull}},
{{17768164641975811714ull, 389812560455999ull}},
{{17903880528322559695ull, 311850048364799ull}},
{{17578162401090364542ull, 498960077383679ull}},
{{17751878735614201957ull, 399168061906943ull}},
{{3133456544265630596ull, 319334449525555ull}},
{{5013530470825008953ull, 510935119240888ull}},
{{11389522006143827809ull, 408748095392710ull}},
{{9111617604915062247ull, 326998476314168ull}},
{{10889239353122189273ull, 523197562102669ull}},
{{12400740297239661741ull, 418558049682135ull}},
{{9920592237791729393ull, 334846439745708ull}},
{{12183598765724856706ull, 535754303593133ull}},
{{17125576642063706011ull, 428603442874506ull}},
{{10011112498909054485ull, 342882754299605ull}},
{{16017779998254487177ull, 548612406879368ull}},
{{1746177554377858772ull, 438889925503495ull}},
{{1396942043502287017ull, 351111940402796ull}},
{{13303153713829390198ull, 561779104644473ull}},
{{18021220600547332804ull, 449423283715578ull}},
{{3348930036212135274ull, 359538626972463ull}},
{{10057841658453528865ull, 287630901577970ull}},
{{16092546653525646185ull, 460209442524752ull}},
{{5495339693336696301ull, 368167554019802ull}},
{{15464318198895088011ull, 294534043215841ull}},
{{17364211488748320171ull, 471254469145346ull}},
{{10202020376256745813ull, 377003575316277ull}},
{{782918671521576004ull, 301602860253022ull}},
{{4942018689176431930ull, 482564576404835ull}},
{{3953614951341145544ull, 386051661123868ull}},
{{10541589590556737082ull, 308841328899094ull}},
{{5798496900665048361ull, 494146126238551ull}},
{{949448705790128366ull, 395316900990841ull}},
{{15516954223599743985ull, 316253520792672ull}},
{{10069731498791949084ull, 506005633268276ull}},
{{4366436384291648944ull, 404804506614621ull}},
{{18250544366400960448ull, 323843605291696ull}},
{{3375429283048164454ull, 518149768466715ull}},
{{2700343426438531563ull, 414519814773372ull}},
{{13228321185376556220ull, 331615851818697ull}},
{{6407918637634848660ull, 530585362909916ull}},
{{1436986095365968605ull, 424468290327933ull}},
{{8528286505776595530ull, 339574632262346ull}},
{{6266560779758732202ull, 543319411619754ull}},
{{8702597438548896085ull, 434655529295803ull}},
{{14340775580322937514ull, 347724423436642ull}},
{{8187845669549058730ull, 556359077498628ull}},
{{13928974165123067630ull, 445087261998902ull}},
{{3764481702614633458ull, 356069809599122ull}},
{{14079631806317437736ull, 284855847679297ull}},
{{7770015631140259084ull, 455769356286876ull}},
{{2526663690170296944ull, 364615485029501ull}},
{{16778726211103878848ull, 291692388023600ull}},
{{8399217864056654541ull, 466707820837761ull}},
{{3030025476503413310ull, 373366256670209ull}},
{{6113369195944640971ull, 298693005336167ull}},
{{13470739528253335877ull, 477908808537867ull}},
{{3397893993118848055ull, 382327046830294ull}},
{{6407664009236988767ull, 305861637464235ull}},
{{10252262414779182028ull, 489378619942776ull}},
{{4512461117081435299ull, 391502895954221ull}},
{{18367364152632789532ull, 313202316763376ull}},
{{3562340941019090989ull, 501123706821403ull}},
{{10228570382299093437ull, 400898965457122ull}},
{{804158676355454103ull, 320719172365698ull}},
{{16044049141136367858ull, 513150675785116ull}},
{{9145890498167183963ull, 410520540628093ull}},
{{14695410028017567817ull, 328416432502474ull}},
{{12444609600602377538ull, 525466292003959ull}},
{{13645036495223812353ull, 420373033603167ull}},
{{3537331566695229236ull, 336298426882534ull}},
{{13038428136196187425ull, 538077483012054ull}},
{{14120091323698860263ull, 430461986409643ull}},
{{228026614733357241ull, 344369589127715ull}},
{{364842583573371585ull, 550991342604344ull}},
{{3981222881600607591ull, 440793074083475ull}},
{{3184978305280486073ull, 352634459266780ull}},
{{2547982644224388858ull, 282107567413424ull}},
{{11455469860242842820ull, 451372107861478ull}},
{{16543073517678094902ull, 361097686289182ull}},
{{5855761184658655275ull, 288878149031346ull}},
{{1990520265970027794ull, 462205038450154ull}},
{{5281765027517932559ull, 369764030760123ull}},
{{11604109651498166693ull, 295811224608098ull}},
{{14877226627655156386ull, 473297959372957ull}},
{{4523083672640304462ull, 378638367498366ull}},
{{18375862197079884863ull, 302910693998692ull}},
{{14643984256360174488ull, 484657110397908ull}},
{{647140960862408621ull, 387725688318327ull}},
{{11585759212915657866ull, 310180550654661ull}},
{{11158517111181231939ull, 496288881047458ull}},
{{16305511318428806198ull, 397031104837966ull}},
{{9355060240001134635ull, 317624883870373ull}},
{{11278747569259905093ull, 508199814192597ull}},
{{1644300425924103428ull, 406559851354078ull}},
{{8694137970223103389ull, 325247881083262ull}},
{{17599969567098875745ull, 520396609733219ull}},
{{17769324468421010919ull, 416317287786575ull}},
{{14215459574736808735ull, 333053830229260ull}},
{{4297991245869342361ull, 532886128366817ull}},
{{14506439440921204858ull, 426308902693453ull}},
{{537105108511232917ull, 341047122154763ull}},
{{15616763432585613960ull, 545675395447620ull}},
{{12493410746068491168ull, 436540316358096ull}},
{{6305379782112882611ull, 349232253086477ull}},
{{13777956466122522501ull, 558771604938363ull}},
{{18401062802381838647ull, 447017283950690ull}},
{{14720850241905470918ull, 357613827160552ull}},
{{4397982564040556088ull, 286091061728442ull}},
{{10726120917206800064ull, 457745698765507ull}},
{{1202199104281619405ull, 366196559012406ull}},
{{15719154542392936816ull, 292957247209924ull}},
{{14082600823602967937ull, 468731595535879ull}},
{{14955429473624284673ull, 374985276428703ull}},
{{896297134673696768ull, 299988221142963ull}},
{{16191470674445556122ull, 479981153828740ull}},
{{12953176539556444898ull, 383984923062992ull}},
{{2983843602161335272ull, 307187938450394ull}},
{{12152847392941957082ull, 491500701520630ull}},
{{9722277914353565665ull, 393200561216504ull}},
{{11467171146224762855ull, 314560448973203ull}},
{{14658125019217710245ull, 503296718357125ull}},
{{11726500015374168196ull, 402637374685700ull}},
{{9381200012299334557ull, 322109899748560ull}},
{{15009920019678935291ull, 515375839597696ull}},
{{8318587201001237910ull, 412300671678157ull}},
{{17722916205026721297ull, 329840537342525ull}},
{{9909921854333202460ull, 527744859748041ull}},
{{4238588668724651645ull, 422195887798433ull}},
{{10769568564463541962ull, 337756710238746ull}},
{{9852612073657846493ull, 540410736381994ull}},
{{11571438473668187518ull, 432328589105595ull}},
{{9257150778934550014ull, 345862871284476ull}},
{{7432743616811459376ull, 553380594055162ull}},
{{17014241337674898471ull, 442704475244129ull}},
{{17300741884881829100ull, 354163580195303ull}},
{{2772547063679732310ull, 283330864156243ull}},
{{746726487145661373ull, 453329382649989ull}},
{{4286730004458439422ull, 362663506119991ull}},
{{18186779262534392830ull, 290130804895992ull}},
{{14341451561087387235ull, 464209287833588ull}},
{{405114804644178819ull, 371367430266871ull}},
{{15081487102682984348ull, 297093944213496ull}},
{{16751681734808954310ull, 475350310741594ull}},
{{17090694202589073771ull, 380280248593275ull}},
{{13672555362071259017ull, 304224198874620ull}},
{{3429344505604462811ull, 486758718199393ull}},
{{10122173233967390895ull, 389406974559514ull}},
{{11787087401915823039ull, 311525579647611ull}},
{{11480642213581496217ull, 498440927436178ull}},
{{16563211400349017620ull, 398752741948942ull}},
{{5871871490795393449ull, 319002193559154ull}},
{{16773692014756450166ull, 510403509694646ull}},
{{9729604797063249809ull, 408322807755717ull}},
{{404986208166779201ull, 326658246204574ull}},
{{8026675562550667368ull, 522653193927318ull}},
{{13800038079524354541ull, 418122555141854ull}},
{{14729379278361393956ull, 334498044113483ull}},
{{1430913956926768390ull, 535196870581574ull}},
{{4834079980283325035ull, 428157496465259ull}},
{{7556612798968570351ull, 342525997172207ull}},
{{15779929293091622885ull, 548041595475531ull}},
{{8934594619731387985ull, 438433276380425ull}},
{{7147675695785110388ull, 350746621104340ull}},
{{11436281113256176621ull, 561194593766944ull}},
{{12838373705346851620ull, 448955675013555ull}},
{{10270698964277481296ull, 359164540010844ull}},
{{11905907986163895360ull, 287331632008675ull}},
{{602708704152680960ull, 459730611213881ull}},
{{15239562222289786061ull, 367784488971104ull}},
{{15880998592573739172ull, 294227591176883ull}},
{{3273504859666520736ull, 470764145883014ull}},
{{6308152702475126912ull, 376611316706411ull}},
{{1357173347238191206ull, 301289053365129ull}},
{{9550174985064926576ull, 482062485384206ull}},
{{3950791173310030938ull, 385649988307365ull}},
{{3160632938648024750ull, 308519990645892ull}},
{{8746361516578749924ull, 493631985033427ull}},
{{18065135657488730908ull, 394905588026741ull}},
{{10762759711249074403ull, 315924470421393ull}},
{{13531066723256608723ull, 505479152674229ull}},
{{14514202193347197301ull, 404383322139383ull}},
{{543315310452026871ull, 323506657711507ull}},
{{4558653311465153318ull, 517610652338411ull}},
{{18404317908139763947ull, 414088521870728ull}},
{{3655407882286080188ull, 331270817496583ull}},
{{2159303796915817977ull, 530033307994533ull}},
{{9106140667016475028ull, 424026646395626ull}},
{{3595563718871269699ull, 339221317116501ull}},
{{16820948394419762489ull, 542754107386401ull}},
{{9767409900793899668ull, 434203285909121ull}},
{{4124579105893209411ull, 347362628727297ull}},
{{10288675384171045381ull, 555780205963675ull}},
{{8230940307336836305ull, 444624164770940ull}},
{{6584752245869469044ull, 355699331816752ull}},
{{16335848240921306204ull, 284559465453401ull}},
{{311915482280717665ull, 455295144725443ull}},
{{7628230015308394778ull, 364236115780354ull}},
{{9791932826988626146ull, 291388892624283ull}},
{{11977743708439891510ull, 466222228198853ull}},
{{16960892596235733854ull, 372977782559082ull}},
{{6190016447504766437ull, 298382226047266ull}},
{{2525328686523805653ull, 477411561675626ull}},
{{16777658208186685815ull, 381929249340500ull}},
{{13422126566549348652ull, 305543399472400ull}},
{{3028658432769406228ull, 488869439155841ull}},
{{17180322005183166275ull, 391095551324672ull}},
{{6365559974662712373ull, 312876441059738ull}},
{{6495547144718429474ull, 500602305695581ull}},
{{1507088901032833256ull, 400481844556465ull}},
{{1205671120826266605ull, 320385475645172ull}},
{{5618422608063936891ull, 512616761032275ull}},
{{4494738086451149513ull, 410093408825820ull}},
{{3595790469160919610ull, 328074727060656ull}},
{{16821311194883202346ull, 524919563297049ull}},
{{17146397770648472200ull, 419935650637639ull}},
{{17406467031260688083ull, 335948520510111ull}},
{{2024905546823728671ull, 537517632816179ull}},
{{5309273252200893260ull, 430014106252943ull}},
{{11626116231244535254ull, 344011285002354ull}},
{{7533739525765525438ull, 550418056003767ull}},
{{17095038064838151320ull, 440334444803013ull}},
{{2607984007644790086ull, 352267555842411ull}},
{{16843782465083473361ull, 281814044673928ull}},
{{4813959055682095439ull, 450902471478286ull}},
{{161818429803766028ull, 360721977182629ull}},
{{3818803558584923146ull, 288577581746103ull}},
{{2420736878993966710ull, 461724130793765ull}},
{{1936589503195173368ull, 369379304635012ull}},
{{12617318046781869664ull, 295503443708009ull}},
{{9119662430625260493ull, 472805509932815ull}},
{{7295729944500208394ull, 378244407946252ull}},
{{16904630399825897685ull, 302595526357001ull}},
{{1221966936528064034ull, 484152842171203ull}},
{{8356271178706271873ull, 387322273736962ull}},
{{17753063387190748468ull, 309857818989569ull}},
{{17336854975279466580ull, 495772510383311ull}},
{{10180135165481662941ull, 396618008306649ull}},
{{11833456947127240676ull, 317294406645319ull}},
{{7865484671177854112ull, 507671050632511ull}},
{{2603038922200372966ull, 406136840506009ull}},
{{5771779952502208696ull, 324909472404807ull}},
{{12924196738745444237ull, 519855155847691ull}},
{{6650008576254445066ull, 415884124678153ull}},
{{12698704490487376699ull, 332707299742522ull}},
{{5560531925812161426ull, 532331679588036ull}},
{{759076725907818818ull, 425865343670429ull}},
{{4296610195468165377ull, 340692274936343ull}},
{{3185227498007154281ull, 545107639898149ull}},
{{6237530813147633748ull, 436086111918519ull}},
{{8679373465260017321ull, 348868889534815ull}},
{{13886997544416027714ull, 558190223255704ull}},
{{14798946850274732495ull, 446552178604563ull}},
{{771111035994055026ull, 357241742883651ull}},
{{15374284087762885313ull, 285793394306920ull}},
{{6152110466711064886ull, 457269430891073ull}},
{{12300386002852672555ull, 365815544712858ull}},
{{17219006431765958690ull, 292652435770286ull}},
{{1724968587632161642ull, 468243897232459ull}},
{{5069323684847639637ull, 374595117785967ull}},
{{15123505392103842679ull, 299676094228773ull}},
{{2061515738914686348ull, 479481750766038ull}},
{{9027910220615569724ull, 383585400612830ull}},
{{7222328176492455779ull, 306868320490264ull}},
{{487678638162198278ull, 490989312784423ull}},
{{7768840540013579269ull, 392791450227538ull}},
{{13593770061494684061ull, 314233160182030ull}},
{{3303288024681942882ull, 502773056291249ull}},
{{6331979234487464629ull, 402218445032999ull}},
{{8754932202331882026ull, 321774756026399ull}},
{{2939845079505280272ull, 514839609642239ull}},
{{6041224878346134541ull, 411871687713791ull}},
{{1143631087934997310ull, 329497350171033ull}},
{{16587204999663636988ull, 527195760273652ull}},
{{5891066370247088944ull, 421756608218922ull}},
{{15780899540423402125ull, 337405286575137ull}},
{{10492044005709802107ull, 539848458520220ull}},
{{8393635204567841686ull, 431878766816176ull}},
{{3025559348912363025ull, 345503013452941ull}},
{{15908941402485511810ull, 552804821524705ull}},
{{12727153121988409448ull, 442243857219764ull}},
{{13871071312332637882ull, 353795085775811ull}},
{{7407508235124199982ull, 283036068620649ull}},
{{783966731972989002ull, 452857709793039ull}},
{{4316522200320301525ull, 362286167834431ull}},
{{18210613019223882512ull, 289828934267544ull}},
{{18068934386532481051ull, 463726294828071ull}},
{{10765798694484074517ull, 370981035862457ull}},
{{1233941326103438967ull, 296784828689966ull}},
{{13042352565991233317ull, 474855725903945ull}},
{{10433882052792986654ull, 379884580723156ull}},
{{4657756827492479000ull, 303907664578525ull}},
{{7452410923987966400ull, 486252263325640ull}},
{{5961928739190373120ull, 389001810660512ull}},
{{15837589435578029465ull, 311201448528409ull}},
{{14272096652699116175ull, 497922317645455ull}},
{{11417677322159292940ull, 398337854116364ull}},
{{12823490672469344675ull, 318670283293091ull}},
{{13138887446467130834ull, 509872453268946ull}},
{{6821761142431794344ull, 407897962615157ull}},
{{16525455358171166445ull, 326318370092125ull}},
{{7993984499364314696ull, 522109392147401ull}},
{{2705838784749541434ull, 417687513717921ull}},
{{16922066286767274440ull, 334150010974336ull}},
{{1249864355634266841ull, 534640017558939ull}},
{{4689240299249323796ull, 427712014047151ull}},
{{62043424657548713ull, 342169611237721ull}},
{{11167315923677808911ull, 547471377980353ull}},
{{16312550368426067775ull, 437977102384282ull}},
{{5671342665257033574ull, 350381681907426ull}},
{{1695450634927433072ull, 560610691051882ull}},
{{12424406952167677427ull, 448488552841505ull}},
{{9939525561734141942ull, 358790842273204ull}},
{{11640969264129223876ull, 287032673818563ull}},
{{14936202007864847879ull, 459252278109701ull}},
{{8259612791549967980ull, 367401822487761ull}},
{{2918341418498064061ull, 293921457990209ull}},
{{12048043899080723144ull, 470274332784334ull}},
{{13327783934006488838ull, 376219466227467ull}},
{{3283529517721370424ull, 300975572981974ull}},
{{12632344857838013325ull, 481560916771158ull}},
{{17484573515754231307ull, 385248733416926ull}},
{{10298309997861474722ull, 308198986733541ull}},
{{9098598367094538909ull, 493118378773666ull}},
{{3589529878933720804ull, 394494703018933ull}},
{{10250321532630797289ull, 315595762415146ull}},
{{9021816822725455017ull, 504953219864234ull}},
{{10906802272922274337ull, 403962575891387ull}},
{{1346744188853998823ull, 323170060713110ull}},
{{2154790702166398117ull, 517072097140976ull}},
{{16481227820700759786ull, 413657677712780ull}},
{{13184982256560607829ull, 330926142170224ull}},
{{10027925166271241557ull, 529481827472359ull}},
{{11711688947758903569ull, 423585461977887ull}},
{{1990653528723302208ull, 338868369582310ull}},
{{3185045645957283534ull, 542189391331696ull}},
{{17305431775733468120ull, 433751513065356ull}},
{{10154996605844864172ull, 347001210452285ull}},
{{16247994569351782676ull, 555201936723656ull}},
{{9309046840739515818ull, 444161549378925ull}},
{{7447237472591612654ull, 355329239503140ull}},
{{5957789978073290123ull, 284263391602512ull}},
{{13221812779659174521ull, 454821426564019ull}},
{{14266799038469249940ull, 363857141251215ull}},
{{11413439230775399952ull, 291085713000972ull}},
{{3504107510272998630ull, 465737140801556ull}},
{{17560681267186040197ull, 372589712641244ull}},
{{17737893828490742480ull, 298071770112995ull}},
{{9933886051875636353ull, 476914832180793ull}},
{{15325806470984329729ull, 381531865744634ull}},
{{15949993991529374106ull, 305225492595707ull}},
{{10762595127479357277ull, 488360788153132ull}},
{{1231378472499665175ull, 390688630522506ull}},
{{15742498036967373433ull, 312550904418004ull}},
{{14119950414922066523ull, 500081447068807ull}},
{{3917262702453832572ull, 400065157655046ull}},
{{17891205420930707350ull, 320052126124036ull}},
{{2800486970295759499ull, 512083401798459ull}},
{{5929738390978517922ull, 409666721438767ull}},
{{15811837157008545307ull, 327733377151013ull}},
{{3162846562762210552ull, 524373403441622ull}},
{{13598323694435499411ull, 419498722753297ull}},
{{3499961326064578883ull, 335598978202638ull}},
{{1910589306961415889ull, 536958365124221ull}},
{{16285866704536774004ull, 429566692099376ull}},
{{9339344548887508880ull, 343653353679501ull}},
{{7564253648736193562ull, 549845365887202ull}},
{{17119449363214685819ull, 439876292709761ull}},
{{10006210675829838332ull, 351901034167809ull}},
{{11694317355405780989ull, 281520827334247ull}},
{{3953512509681608289ull, 450433323734796ull}},
{{17920205266712927924ull, 360346658987836ull}},
{{10646815398628432016ull, 288277327190269ull}},
{{5966858193579760256ull, 461243723504431ull}},
{{1084137740121897882ull, 368994978803545ull}},
{{867310192097518305ull, 295195983042836ull}},
{{12455742751581760258ull, 472313572868537ull}},
{{2585896571781587560ull, 377850858294830ull}},
{{2068717257425270048ull, 302280686635864ull}},
{{10688645241364252724ull, 483649098617382ull}},
{{1172218563607581532ull, 386919278893906ull}},
{{15695170109853706519ull, 309535423115124ull}},
{{14044225731540199461ull, 495256676984199ull}},
{{14924729399974069892ull, 396205341587359ull}},
{{15629132334721166236ull, 316964273269887ull}},
{{10249216476586224686ull, 507142837231820ull}},
{{8199373181268979748ull, 405714269785456ull}},
{{2870149730273273475ull, 324571415828365ull}},
{{4592239568437237561ull, 519314265325384ull}},
{{7363140469491700372ull, 415451412260307ull}},
{{16958558819819091267ull, 332361129808245ull}},
{{8686950038000994412ull, 531777807693193ull}},
{{14328257659884616176ull, 425422246154554ull}},
{{15151954942649603264ull, 340337796923643ull}},
{{2107035019787903283ull, 544540475077830ull}},
{{1685628015830322626ull, 435632380062264ull}},
{{5037851227406168424ull, 348505904049811ull}},
{{681864334366048832ull, 557609446479698ull}},
{{7924189096976659712ull, 446087557183758ull}},
{{13718048907065148416ull, 356870045747006ull}},
{{7285090310910208409ull, 285496036597605ull}},
{{11656144497456333455ull, 456793658556168ull}},
{{16703613227448887411ull, 365434926844934ull}},
{{17052239396701020252ull, 292347941475947ull}},
{{12526187775753991110ull, 467756706361516ull}},
{{6331601405861282565ull, 374205365089213ull}},
{{12443978754172846698ull, 299364292071370ull}},
{{1463621932967003101ull, 478982867314193ull}},
{{8549595175857423127ull, 383186293851354ull}},
{{10529024955427848825ull, 306549035081083ull}},
{{13157091113942647797ull, 490478456129733ull}},
{{17904370520637938884ull, 392382764903786ull}},
{{10634147601768440784ull, 313906211923029ull}},
{{5946589718603774285ull, 502249939076847ull}},
{{15825318219108750397ull, 401799951261477ull}},
{{5281556945803179671ull, 321439961009182ull}},
{{12139839928026997798ull, 514303937614691ull}},
{{6022523127679687915ull, 411443150091753ull}},
{{12196716131627570978ull, 329154520073402ull}},
{{4757350551636472273ull, 526647232117444ull}},
{{7495229256051088141ull, 421317785693955ull}},
{{5996183404840870513ull, 337054228555164ull}},
{{16972591077229213467ull, 539286765688262ull}},
{{6199375232299550127ull, 431429412550610ull}},
{{4959500185839640102ull, 345143530040488ull}},
{{4245851482601513840ull, 552229648064781ull}},
{{18154076445048852365ull, 441783718451824ull}},
{{18212609970780992215ull, 353426974761459ull}},
{{18259436791366704095ull, 282741579809167ull}},
{{14457703607219085259ull, 452386527694668ull}},
{{498116441549537238ull, 361909222155735ull}},
{{398493153239629790ull, 289527377724588ull}},
{{15394984304151048957ull, 463243804359340ull}},
{{12315987443320839166ull, 370595043487472ull}},
{{2474092325172850686ull, 296476034789978ull}},
{{269198905534650775ull, 474361655663965ull}},
{{215359124427720620ull, 379489324531172ull}},
{{11240333743767907465ull, 303591459624937ull}},
{{3227138731061010652ull, 485746335399900ull}},
{{2581710984848808522ull, 388597068319920ull}},
{{2065368787879046817ull, 310877654655936ull}},
{{14372636504832205877ull, 497404247449497ull}},
{{4119411574381944055ull, 397923397959598ull}},
{{10674226888989375891ull, 318338718367678ull}},
{{13389414207641091102ull, 509341949388285ull}},
{{10711531366112872882ull, 407473559510628ull}},
{{15947922722374118952ull, 325978847608502ull}},
{{10759281096830949030ull, 521566156173604ull}},
{{12296773692206669547ull, 417252924938883ull}},
{{17216116583249156284ull, 333802339951106ull}},
{{1720344830005277792ull, 534083743921771ull}},
{{16133671122971863527ull, 427266995137416ull}},
{{9217588083635580498ull, 341813596109933ull}},
{{11058792119075018474ull, 546901753775893ull}},
{{16225731324743835425ull, 437521403020714ull}},
{{16669933874536978663ull, 350017122416571ull}},
{{846452496065793599ull, 560027395866515ull}},
{{677161996852634879ull, 448021916693212ull}},
{{11609776041707838873ull, 358417533354569ull}},
{{12977169648108181421ull, 286734026683655ull}},
{{2316727363263538659ull, 458774442693849ull}},
{{5542730705352741250ull, 367019554155079ull}},
{{8123533379024103323ull, 293615643324063ull}},
{{9308304591696654994ull, 469785029318501ull}},
{{3757294858615413672ull, 375828023454801ull}},
{{17763231145859972230ull, 300662418763840ull}},
{{9974425759666403953ull, 481059870022145ull}},
{{7979540607733123162ull, 384847896017716ull}},
{{2694283671444588206ull, 307878316814173ull}},
{{621505059569430807ull, 492605306902677ull}},
{{11565250491881275615ull, 394084245522141ull}},
{{5562851578763110169ull, 315267396417713ull}},
{{5211213711279065947ull, 504427834268341ull}},
{{479622154281342435ull, 403542267414673ull}},
{{7762395352908894594ull, 322833813931738ull}},
{{8730483749912321028ull, 516534102290781ull}},
{{3295038185187946499ull, 413227281832625ull}},
{{2636030548150357199ull, 330581825466100ull}},
{{4217648877040571519ull, 528930920745760ull}},
{{3374119101632457215ull, 423144736596608ull}},
{{10077992910789786418ull, 338515789277286ull}},
{{8746091027779837623ull, 541625262843658ull}},
{{14375570451707690745ull, 433300210274926ull}},
{{7811107546624242272ull, 346640168219941ull}},
{{5119074445114966990ull, 554624269151906ull}},
{{405910741350063268ull, 443699415321525ull}},
{{324728593080050615ull, 354959532257220ull}},
{{259782874464040492ull, 283967625805776ull}},
{{11483699043368195756ull, 454348201289241ull}},
{{5497610419952646282ull, 363478561031393ull}},
{{11776785965445937672ull, 290782848825114ull}},
{{7774811100487769306ull, 465252558120183ull}},
{{13598546509874036091ull, 372202046496146ull}},
{{7189488393157318549ull, 297761637196917ull}},
{{15192530243793620002ull, 476418619515067ull}},
{{4775326565551075355ull, 381134895612054ull}},
{{7509610067182770607ull, 304907916489643ull}},
{{8326027292750522649ull, 487852666383429ull}},
{{10350170648942328442ull, 390282133106743ull}},
{{15658834148637683400ull, 312225706485394ull}},
{{13986088193594562471ull, 499561130376631ull}},
{{7499521740133739653ull, 399648904301305ull}},
{{5999617392106991723ull, 319719123441044ull}},
{{16978085456855007403ull, 511550597505670ull}},
{{13582468365484005922ull, 409240478004536ull}},
{{7176625877645294415ull, 327392382403629ull}},
{{414554960006740094ull, 523827811845807ull}},
{{11399690412231123045ull, 419062249476645ull}},
{{9119752329784898436ull, 335249799581316ull}},
{{7212906098172016851ull, 536399679330106ull}},
{{2080976063795703157ull, 429119743464085ull}},
{{1664780851036562526ull, 343295794771268ull}},
{{17421044620626141334ull, 549273271634028ull}},
{{2868789252275182098ull, 439418617307223ull}},
{{9673729031303966325ull, 351534893845778ull}},
{{11788617635344435796ull, 562455830153245ull}},
{{9430894108275548637ull, 449964664122596ull}},
{{3855366471878528586ull, 359971731298077ull}},
{{14152339621728553839ull, 287977385038461ull}},
{{15265045765281865496ull, 460763816061538ull}},
{{1143990167999761427ull, 368611052849231ull}},
{{15672587393367450434ull, 294888842279384ull}},
{{14008093385162189725ull, 471822147647015ull}},
{{11206474708129751780ull, 377457718117612ull}},
{{1586482137019980778ull, 301966174494090ull}},
{{2538371419231969244ull, 483145879190544ull}},
{{5720045950127485719ull, 386516703352435ull}},
{{4576036760101988575ull, 309213362681948ull}},
{{3632310001421271397ull, 494741380291117ull}},
{{13973894445362748087ull, 395793104232893ull}},
{{111069112064467500ull, 316634483386315ull}},
{{177710579303148000ull, 506615173418104ull}},
{{3831517278184428723ull, 405292138734483ull}},
{{10443911452031363625ull, 324233710987586ull}},
{{9331560693766361154ull, 518773937580138ull}},
{{14843946184496909569ull, 415019150064110ull}},
{{11875156947597527655ull, 332015320051288ull}},
{{15310902301414133926ull, 531224512082061ull}},
{{8559373026389396817ull, 424979609665649ull}},
{{10536847235853427777ull, 339983687732519ull}},
{{5790909133139753474ull, 543973900372031ull}},
{{943378491769892456ull, 435179120297625ull}},
{{754702793415913964ull, 348143296238100ull}},
{{1207524469465462343ull, 557029273980960ull}},
{{966019575572369875ull, 445623419184768ull}},
{{8151513289941716546ull, 356498735347814ull}},
{{10210559446695283560ull, 285198988278251ull}},
{{8958197485228633050ull, 456318381245202ull}},
{{18234604432408637409ull, 365054704996161ull}},
{{10898334731184999604ull, 292043763996929ull}},
{{6369289125670268397ull, 467270022395087ull}},
{{16163477744761945687ull, 373816017916069ull}},
{{16620131010551466873ull, 299052814332855ull}},
{{8145465543172795381ull, 478484502932569ull}},
{{10205721249280146628ull, 382787602346055ull}},
{{8164576999424117302ull, 306230081876844ull}},
{{1995276754852856714ull, 489968131002951ull}},
{{16353616662849926664ull, 391974504802360ull}},
{{13082893330279941331ull, 313579603841888ull}},
{{17243280513705995807ull, 501727366147021ull}},
{{10105275596222886322ull, 401381892917617ull}},
{{705522847494488411ull, 321105514334094ull}},
{{8507534185475002105ull, 513768822934550ull}},
{{6806027348380001684ull, 411015058347640ull}},
{{5444821878704001347ull, 328812046678112ull}},
{{12401063820668312478ull, 526099274684979ull}},
{{13610199871276560306ull, 420879419747983ull}},
{{18266857526505068891ull, 336703535798386ull}},
{{3401530339214737964ull, 538725657277419ull}},
{{6410573086113700694ull, 430980525821935ull}},
{{5128458468890960555ull, 344784420657548ull}},
{{4516184735483626565ull, 551655073052077ull}},
{{14680994232612632222ull, 441324058441661ull}},
{{8055446571348195454ull, 353059246753329ull}},
{{10133706071820466686ull, 282447397402663ull}},
{{12524580900170836375ull, 451915835844261ull}},
{{6330315905394758777ull, 361532668675409ull}},
{{8753601539057717345ull, 289226134940327ull}},
{{17695111277234258075ull, 462761815904523ull}},
{{3088042577561675490ull, 370209452723619ull}},
{{6159782876791250715ull, 296167562178895ull}},
{{9855652602866001145ull, 473868099486232ull}},
{{505824452808980269ull, 379094479588986ull}},
{{15162054821214825508ull, 303275583671188ull}},
{{2123194825492258874ull, 485240933873902ull}},
{{12766602304619538069ull, 388192747099121ull}},
{{6523933028953720132ull, 310554197679297ull}},
{{14127641661067862534ull, 496886716286875ull}},
{{11302113328854290027ull, 397509373029500ull}},
{{9041690663083432022ull, 318007498423600ull}},
{{14466705060933491235ull, 508811997477760ull}},
{{11573364048746792988ull, 407049597982208ull}},
{{16637388868481255037ull, 325639678385766ull}},
{{794380486376635797ull, 521023485417227ull}},
{{11703550833327039607ull, 416818788333781ull}},
{{5673491851919721362ull, 333455030667025ull}},
{{9077586963071554180ull, 533528049067240ull}},
{{7262069570457243344ull, 426822439253792ull}},
{{16877702100591525644ull, 341457951403033ull}},
{{4868230472494979092ull, 546332722244854ull}},
{{7583933192737893597ull, 437066177795883ull}},
{{13445844183674135524ull, 349652942236706ull}},
{{14134653064394796192ull, 559444707578730ull}},
{{11307722451515836953ull, 447555766062984ull}},
{{12735526775954579886ull, 358044612850387ull}},
{{2809723791279843262ull, 286435690280310ull}},
{{4495558066047749220ull, 458297104448496ull}},
{{18353841711805840668ull, 366637683558796ull}},
{{10993724554702762211ull, 293310146847037ull}},
{{2832564028556778246ull, 469296234955260ull}},
{{2266051222845422597ull, 375436987964208ull}},
{{9191538607760158724ull, 300349590371366ull}},
{{7327764142932433312ull, 480559344594186ull}},
{{2172862499604036326ull, 384447475675349ull}},
{{5427638814425139384ull, 307557980540279ull}},
{{16062919732564043661ull, 492092768864446ull}},
{{9160986971309324605ull, 393674215091557ull}},
{{18396836021273190654ull, 314939372073245ull}},
{{10988193560327553430ull, 503902995317193ull}},
{{16169252477745863391ull, 403122396253754ull}},
{{16624750796938601036ull, 322497917003003ull}},
{{4463508386650299718ull, 515996667204806ull}},
{{18328201968287881067ull, 412797333763844ull}},
{{18351910389372215177ull, 330237867011075ull}},
{{10916312549285992667ull, 528380587217721ull}},
{{5043701224686883811ull, 422704469774177ull}},
{{15103007423975238018ull, 338163575819341ull}},
{{16786114248876560183ull, 541061721310946ull}},
{{9739542584359337823ull, 432849377048757ull}},
{{412936438003649612ull, 346279501639006ull}},
{{11728744745031570349ull, 554047202622409ull}},
{{13072344610767166602ull, 443237762097927ull}},
{{3079178059129912635ull, 354590209678342ull}},
{{13531388891529661078ull, 283672167742673ull}},
{{17960873411705547401ull, 453875468388277ull}},
{{6990001099880617274ull, 363100374710622ull}},
{{16660047324130224789ull, 290480299768497ull}},
{{11898680459640718370ull, 464768479629596ull}},
{{5829595552970664373ull, 371814783703677ull}},
{{15731722886602262468ull, 297451826962941ull}},
{{17792058989079799302ull, 475922923140706ull}},
{{10544298376521929118ull, 380738338512565ull}},
{{8435438701217543295ull, 304590670810052ull}},
{{17186050736689979595ull, 487345073296083ull}},
{{2680794145126252706ull, 389876058636867ull}},
{{13212681760326733135ull, 311900846909493ull}},
{{17450942001780862692ull, 499041355055189ull}},
{{17650102416166600477ull, 399233084044151ull}},
{{10430733118191370058ull, 319386467235321ull}},
{{9310475359622371447ull, 511018347576514ull}},
{{11137729102439807481ull, 408814678061211ull}},
{{5220834467209935661ull, 327051742448969ull}},
{{15732032777019717705ull, 523282787918350ull}},
{{12585626221615774164ull, 418626230334680ull}},
{{10068500977292619331ull, 334900984267744ull}},
{{5041555119442459960ull, 535841574828391ull}},
{{343895280812057645ull, 428673259862713ull}},
{{7653813854133466762ull, 342938607890170ull}},
{{12246102166613546820ull, 548701772624272ull}},
{{2418184103807016809ull, 438961418099418ull}},
{{9313244912529434094ull, 351169134479534ull}},
{{3833145415821363581ull, 561870615167255ull}},
{{3066516332657090864ull, 449496492133804ull}},
{{6142561880867583015ull, 359597193707043ull}},
{{12292747134177887058ull, 287677754965634ull}},
{{8600348970458888324ull, 460284407945015ull}},
{{6880279176367110659ull, 368227526356012ull}},
{{16572269785319419497ull, 294582021084809ull}},
{{15447585212285340225ull, 471331233735695ull}},
{{12358068169828272180ull, 377064986988556ull}},
{{6197105721120707421ull, 301651989590845ull}},
{{9915369153793131873ull, 482643183345352ull}},
{{553597693550684852ull, 386114546676282ull}},
{{11510924599066278851ull, 308891637341025ull}},
{{18417479358506046162ull, 494226619745640ull}},
{{14733983486804836930ull, 395381295796512ull}},
{{4408489159960048897ull, 316305036637210ull}},
{{7053582655936078236ull, 506088058619536ull}},
{{1953517310006952266ull, 404870446895629ull}},
{{5252162662747472136ull, 323896357516503ull}},
{{4714111445654045094ull, 518234172026405ull}},
{{3771289156523236075ull, 414587337621124ull}},
{{6706380139960499183ull, 331669870096899ull}},
{{18108905853420619340ull, 530671792155038ull}},
{{3419078238510764502ull, 424537433724031ull}},
{{17492657849776252894ull, 339629946979224ull}},
{{16920206115416273662ull, 543407915166759ull}},
{{17225513707074929252ull, 434726332133407ull}},
{{6401713336176122755ull, 347781065706726ull}},
{{2864043708397975763ull, 556449705130762ull}},
{{13359281410944111580ull, 445159764104609ull}},
{{14376773943497199587ull, 356127811283687ull}},
{{4122721525313939023ull, 284902249026950ull}},
{{6596354440502302437ull, 455843598443120ull}},
{{5277083552401841949ull, 364674878754496ull}},
{{532318027179563236ull, 291739903003597ull}},
{{4541057658229211502ull, 466783844805755ull}},
{{3632846126583369201ull, 373427075844604ull}},
{{6595625716008605684ull, 298741660675683ull}},
{{6863652330871858772ull, 477986657081093ull}},
{{12869619494181307664ull, 382389325664874ull}},
{{13985044410086956454ull, 305911460531899ull}},
{{11308024611913399357ull, 489458336851039ull}},
{{12735768504272629809ull, 391566669480831ull}},
{{6499265988676193524ull, 313253335584665ull}},
{{10398825581881909638ull, 501205336935464ull}},
{{12008409280247438034ull, 400964269548371ull}},
{{5917378609456040104ull, 320771415638697ull}},
{{13157154589871574489ull, 513234265021915ull}},
{{10525723671897259591ull, 410587412017532ull}},
{{1041881308033987027ull, 328469929614026ull}},
{{12735056537080110212ull, 525551887382441ull}},
{{6498696414922177847ull, 420441509905953ull}},
{{12577654761421562924ull, 336353207924762ull}},
{{5366852359306859385ull, 538165132679620ull}},
{{4293481887445487508ull, 430532106143696ull}},
{{18192180768924031299ull, 344425684914956ull}},
{{3282047527085077817ull, 551081095863931ull}},
{{17383033280635703546ull, 440864876691144ull}},
{{17595775439250473160ull, 352691901352915ull}},
{{14076620351400378528ull, 282153521082332ull}},
{{7765197303272964352ull, 451445633731732ull}},
{{17280204286844102451ull, 361156506985385ull}},
{{13824163429475281961ull, 288925205588308ull}},
{{18429312672418540814ull, 462280328941293ull}},
{{3675403693709101682ull, 369824263153035ull}},
{{2940322954967281345ull, 295859410522428ull}},
{{1015167913205739830ull, 473375056835885ull}},
{{812134330564591864ull, 378700045468708ull}},
{{8028405093935494137ull, 302960036374966ull}},
{{5466750520812969973ull, 484736058199946ull}},
{{684051601908465655ull, 387788846559957ull}},
{{11615287725752503494ull, 310231077247965ull}},
{{137716287494453974ull, 496369723596745ull}},
{{110173029995563179ull, 397095778877396ull}},
{{14845533682964091836ull, 317676623101916ull}},
{{16374156263258726292ull, 508282596963066ull}},
{{9409976195865070710ull, 406626077570453ull}},
{{14906678586175877214ull, 325300862056362ull}},
{{9093290478913762251ull, 520481379290180ull}},
{{7274632383131009800ull, 416385103432144ull}},
{{9509054721246718163ull, 333108082745715ull}},
{{15214487553994749062ull, 532972932393144ull}},
{{15860938857937709572ull, 426378345914515ull}},
{{12688751086350167658ull, 341102676731612ull}},
{{5544606479192626960ull, 545764282770580ull}},
{{4435685183354101568ull, 436611426216464ull}},
{{7237896961425191577ull, 349289140973171ull}},
{{4201937508796485878ull, 558862625557074ull}},
{{7050898821779099025ull, 447090100445659ull}},
{{9330067872165189543ull, 357672080356527ull}},
{{85356668248330988ull, 286137664285222ull}},
{{3825919483939239905ull, 457820262856355ull}},
{{3060735587151391924ull, 366256210285084ull}},
{{6137937284463023862ull, 293004968228067ull}},
{{13510048469882748503ull, 468807949164907ull}},
{{3429341146422378156ull, 375046359331926ull}},
{{17500868176105543817ull, 300037087465540ull}},
{{9554645008059318492ull, 480059339944865ull}},
{{7643716006447454793ull, 384047471955892ull}},
{{17183019249383694804ull, 307237977564713ull}},
{{5356737910562449748ull, 491580764103542ull}},
{{15353436772675690768ull, 393264611282833ull}},
{{1214702973914821645ull, 314611689026267ull}},
{{5632873573005624955ull, 503378702442027ull}},
{{15574345302630230933ull, 402702961953621ull}},
{{8770127427362274423ull, 322162369562897ull}},
{{17721552698521549401ull, 515459791300635ull}},
{{14177242158817239521ull, 412367833040508ull}},
{{273747282828060647ull, 329894266432407ull}},
{{4127344467266807358ull, 527830826291851ull}},
{{18059270832781087179ull, 422264661033480ull}},
{{14447416666224869743ull, 337811728826784ull}},
{{12047820221734060620ull, 540498766122855ull}},
{{9638256177387248496ull, 432399012898284ull}},
{{11399953756651709120ull, 345919210318627ull}},
{{3482530751675093299ull, 553470736509804ull}},
{{6475373416081984963ull, 442776589207843ull}},
{{12558996362349408616ull, 354221271366274ull}},
{{13736545904621437216ull, 283377017093019ull}},
{{10910427003168568577ull, 453403227348831ull}},
{{5038992787792944538ull, 362722581879065ull}},
{{4031194230234355630ull, 290178065503252ull}},
{{10139259583116879332ull, 464284904805203ull}},
{{15490105295977324112ull, 371427923844162ull}},
{{5013386607298038643ull, 297142339075330ull}},
{{8021418571676861829ull, 475427742520528ull}},
{{13795832486825310110ull, 380342194016422ull}},
{{3657968359976427441ull, 304273755213138ull}},
{{2163400561220373583ull, 486838008341021ull}},
{{16488115707943940159ull, 389470406672816ull}},
{{9501143751613241804ull, 311576325338253ull}},
{{11512481187839276563ull, 498522120541205ull}},
{{9209984950271421251ull, 398817696432964ull}},
{{11057336774959047324ull, 319054157146371ull}},
{{10313041210450655072ull, 510486651434194ull}},
{{11939781783102434380ull, 408389321147355ull}},
{{9551825426481947504ull, 326711456917884ull}},
{{4214874238145385037ull, 522738331068615ull}},
{{3371899390516308030ull, 418190664854892ull}},
{{13765565956638777393ull, 334552531883913ull}},
{{18335556715880133507ull, 535284051014261ull}},
{{10979096557962196482ull, 428227240811409ull}},
{{12472626061111667509ull, 342581792649127ull}},
{{5198806438811026721ull, 548130868238604ull}},
{{7848393965790731700ull, 438504694590883ull}},
{{13657412802116406006ull, 350803755672706ull}},
{{14473162853902428964ull, 561286009076330ull}},
{{11578530283121943171ull, 449028807261064ull}},
{{12952173041239464860ull, 359223045808851ull}},
{{6672389618249661565ull, 287378436647081ull}},
{{3297125759715637858ull, 459805498635330ull}},
{{2637700607772510286ull, 367844398908264ull}},
{{5799509300959918552ull, 294275519126611ull}},
{{1900517252052049037ull, 470840830602578ull}},
{{8899111431125459876ull, 376672664482062ull}},
{{18187335589126098870ull, 301338131585649ull}},
{{18031690498376027223ull, 482141010537039ull}},
{{18114701213442732101ull, 385712808429631ull}},
{{10802412156012275358ull, 308570246743705ull}},
{{17283859449619640573ull, 493712394789928ull}},
{{2759041115469981489ull, 394969915831943ull}},
{{9585930521859805837ull, 315975932665554ull}},
{{4269442390749958370ull, 505561492264887ull}},
{{14483600356825697666ull, 404449193811909ull}},
{{15276229100202468456ull, 323559355049527ull}},
{{9684571301356308236ull, 517694968079244ull}},
{{11437005855826956912ull, 414155974463395ull}},
{{9149604684661565530ull, 331324779570716ull}},
{{7260669865974684201ull, 530119647313146ull}},
{{2119187078037837038ull, 424095717850517ull}},
{{12763396106656000600ull, 339276574280413ull}},
{{16732084955907690637ull, 542842518848661ull}},
{{9696319149984242186ull, 434274015078929ull}},
{{11446404134729304072ull, 347419212063143ull}},
{{14624897800824976192ull, 555870739301029ull}},
{{15389267055401891277ull, 444696591440823ull}},
{{1243367200095782052ull, 355757273152659ull}},
{{4684042574818535965ull, 284605818522127ull}},
{{11183816934451567867ull, 455369309635403ull}},
{{16325751177045074940ull, 364295447708322ull}},
{{5681903312152239305ull, 291436358166658ull}},
{{5401696484701672566ull, 466298173066653ull}},
{{11700054817245158699ull, 373038538453322ull}},
{{1981346224312306313ull, 298430830762658ull}},
{{17927549217867331393ull, 477489329220252ull}},
{{6963341744810044468ull, 381991463376202ull}},
{{16638719840073766544ull, 305593170700961ull}},
{{796510040924654208ull, 488949073121539ull}},
{{4326556847481633690ull, 391159258497231ull}},
{{18218640736952948244ull, 312927406797784ull}},
{{18081778734898986222ull, 500683850876455ull}},
{{14465422987919188977ull, 400547080701164ull}},
{{15261687205077261505ull, 320437664560931ull}},
{{17040001898639797762ull, 512700263297490ull}},
{{13632001518911838209ull, 410160210637992ull}},
{{3526903585645649921ull, 328128168510394ull}},
{{13021743366516860520ull, 525005069616630ull}},
{{10417394693213488416ull, 420004055693304ull}},
{{12023264569312701056ull, 336003244554643ull}},
{{15547874496158411367ull, 537605191287429ull}},
{{16127648411668639416ull, 430084153029943ull}},
{{1834072285109180563ull, 344067322423955ull}},
{{2934515656174688902ull, 550507715878328ull}},
{{9726310154423571768ull, 440406172702662ull}},
{{402350494055036768ull, 352324938162130ull}},
{{321880395244029414ull, 281859950529704ull}},
{{7893706261874267709ull, 450975920847526ull}},
{{2625616194757503844ull, 360780736678021ull}},
{{16857888214773644368ull, 288624589342416ull}},
{{1147179440444458727ull, 461799342947867ull}},
{{11985789996581297951ull, 369439474358293ull}},
{{16967329626748859007ull, 295551579486634ull}},
{{16079680958572443442ull, 472882527178615ull}},
{{12863744766857954753ull, 378306021742892ull}},
{{2912298184002543156ull, 302644817394314ull}},
{{12038374723887889697ull, 484231707830902ull}},
{{2252002149626491111ull, 387385366264722ull}},
{{12869648163926923858ull, 309908293011777ull}},
{{5834041803315436880ull, 495853268818844ull}},
{{8356582257394259827ull, 396682615055075ull}},
{{6685265805915407862ull, 317346092044060ull}},
{{10696425289464652579ull, 507753747270496ull}},
{{4867791416829811740ull, 406202997816397ull}},
{{14962279577689580362ull, 324962398253117ull}},
{{9182252065335687286ull, 519939837204988ull}},
{{14724499281752370475ull, 415951869763990ull}},
{{11779599425401896380ull, 332761495811192ull}},
{{4089963821675392916ull, 532418393297908ull}},
{{10650668686824134979ull, 425934714638326ull}},
{{4831186134717397660ull, 340747771710661ull}},
{{351200186064015609ull, 545196434737058ull}},
{{7659657778335033134ull, 436157147789646ull}},
{{2438377407926116184ull, 348925718231717ull}},
{{7590752667423696218ull, 558281149170747ull}},
{{17140648578164687944ull, 446624919336597ull}},
{{6333821233047929708ull, 357299935469278ull}},
{{12445754615922164413ull, 285839948375422ull}},
{{5155812126507821768ull, 457343917400676ull}},
{{435300886464347091ull, 365875133920541ull}},
{{15105635968139118966ull, 292700107136432ull}},
{{9411622290054949053ull, 468320171418292ull}},
{{150600202560138596ull, 374656137134634ull}},
{{3809828976790021200ull, 299724909707707ull}},
{{9785075177605944243ull, 479559855532331ull}},
{{4138711327342845071ull, 383647884425865ull}},
{{3310969061874276057ull, 306918307540692ull}},
{{8986899313740752014ull, 491069292065107ull}},
{{18257565895218332581ull, 392855433652085ull}},
{{14606052716174666065ull, 314284346921668ull}},
{{1233591457428003764ull, 502854955074670ull}},
{{986873165942403011ull, 402283964059736ull}},
{{15546893791721563702ull, 321827171247788ull}},
{{2738937178303039984ull, 514923473996462ull}},
{{13259196186868162957ull, 411938779197169ull}},
{{14296705764236440689ull, 329551023357735ull}},
{{4427985149068753486ull, 527281637372377ull}},
{{14610434563480733758ull, 421825309897901ull}},
{{7998998836042676683ull, 337460247918321ull}},
{{5419700508184462047ull, 539936396669314ull}},
{{8025109221289479961ull, 431949117335451ull}},
{{2730738562289673645ull, 345559293868361ull}},
{{15437228143889208802ull, 552894870189377ull}},
{{4971084885627546395ull, 442315896151502ull}},
{{15044914352727768086ull, 353852716921201ull}},
{{8346582667440304145ull, 283082173536961ull}},
{{5975834638420665987ull, 452931477659138ull}},
{{12159365340220353436ull, 362345182127310ull}},
{{9727492272176282748ull, 289876145701848ull}},
{{11874638820740142074ull, 463801833122957ull}},
{{2121013427108293013ull, 371041466498366ull}},
{{16454206000654275703ull, 296833173198692ull}},
{{11569334342079199833ull, 474933077117908ull}},
{{16634165103147180512ull, 379946461694326ull}},
{{9617983267775834087ull, 303957169355461ull}},
{{8010075598957513892ull, 486331470968738ull}},
{{13786758108649831760ull, 389065176774990ull}},
{{11029406486919865408ull, 311252141419992ull}},
{{2889655120104143360ull, 498003426271988ull}},
{{9690421725567135335ull, 398402741017590ull}},
{{7752337380453708268ull, 318722192814072ull}},
{{16093088623467843552ull, 509955508502515ull}},
{{12874470898774274841ull, 407964406802012ull}},
{{2920879089535599226ull, 326371525441610ull}},
{{4673406543256958763ull, 522194440706576ull}},
{{49376419863656687ull, 417755552565261ull}},
{{14796896394858566642ull, 334204442052208ull}},
{{1538941343322244689ull, 534727107283534ull}},
{{4920501889399706074ull, 427781685826827ull}},
{{15004447955745495829ull, 342225348661461ull}},
{{16628419099708972680ull, 547560557858338ull}},
{{2234688835541447174ull, 438048446286671ull}},
{{16545146327400799032ull, 350438757029336ull}},
{{646792420647906189ull, 560702011246939ull}},
{{4206782751260235274ull, 448561608997551ull}},
{{18122821459975829512ull, 358849287198040ull}},
{{14498257167980663610ull, 287079429758432ull}},
{{8439816209801420483ull, 459327087613492ull}},
{{17819899412066867356ull, 367461670090793ull}},
{{3187873085427762915ull, 293969336072635ull}},
{{5100596936684420664ull, 470350937716216ull}},
{{391128734605626208ull, 376280750172973ull}},
{{7691600617168321613ull, 301024600138378ull}},
{{8617212172727404258ull, 481639360221405ull}},
{{6893769738181923406ull, 385311488177124ull}},
{{9204364605287449048ull, 308249190541699ull}},
{{3658936924234187507ull, 493198704866719ull}},
{{6616498354129260329ull, 394558963893375ull}},
{{5293198683303408263ull, 315647171114700ull}},
{{8469117893285453221ull, 505035473783520ull}},
{{6775294314628362577ull, 404028379026816ull}},
{{1730886636960779738ull, 323222703221453ull}},
{{17526813878104888874ull, 517156325154324ull}},
{{17710799917225821422ull, 413725060123459ull}},
{{17857988748522567461ull, 330980048098767ull}},
{{13815386738668466645ull, 529568076958028ull}},
{{18431007020418593962ull, 423654461566422ull}},
{{7366107986851054523ull, 338923569253138ull}},
{{8096423964219776915ull, 542277710805021ull}},
{{2787790356633911208ull, 433822168644017ull}},
{{13298278729532859936ull, 347057734915213ull}},
{{17587897152510665575ull, 555292375864341ull}},
{{10380968907266622137ull, 444233900691473ull}},
{{15683472755297118356ull, 355387120553178ull}},
{{1478731760011963715ull, 284309696442543ull}},
{{17123366074986783237ull, 454895514308068ull}},
{{2630646415763695620ull, 363916411446455ull}},
{{2104517132610956496ull, 291133129157164ull}},
{{10745925041661351040ull, 465813006651462ull}},
{{1218042403845260185ull, 372650405321170ull}},
{{974433923076208148ull, 298120324256936ull}},
{{12627140721147664007ull, 476992518811097ull}},
{{2723014947434310559ull, 381594015048878ull}},
{{9557109587431269094ull, 305275212039102ull}},
{{533980080922389257ull, 488440339262564ull}},
{{4116532879479821729ull, 390752271410051ull}},
{{18050621562551498676ull, 312601817128040ull}},
{{10434250426372846266ull, 500162907404865ull}},
{{8347400341098277012ull, 400130325923892ull}},
{{17745966717104352579ull, 320104260739113ull}},
{{6257453858915502188ull, 512166817182582ull}},
{{16074009531358132720ull, 409733453746065ull}},
{{12859207625086506176ull, 327786762996852ull}},
{{5817336941170768589ull, 524458820794964ull}},
{{8343218367678525194ull, 419567056635971ull}},
{{2985225879400909832ull, 335653645308777ull}},
{{8465710221783366055ull, 537045832494043ull}},
{{14151265806910513490ull, 429636665995234ull}},
{{15010361460270321115ull, 343709332796187ull}},
{{9259183077464872492ull, 549934932473900ull}},
{{7407346461971897993ull, 439947945979120ull}},
{{5925877169577518395ull, 351958356783296ull}},
{{1051352920920104392ull, 281566685426637ull}},
{{5371513488214077351ull, 450506696682619ull}},
{{7986559605313172204ull, 360405357346095ull}},
{{6389247684250537763ull, 288324285876876ull}},
{{2844098665317039775ull, 461318857403002ull}},
{{13343325376479362790ull, 369055085922401ull}},
{{6985311486441579908ull, 295244068737921ull}},
{{3797800748822707207ull, 472390509980674ull}},
{{6727589413800076089ull, 377912407984539ull}},
{{9071420345781971194ull, 302329926387631ull}},
{{7135574923767333265ull, 483727882220210ull}},
{{5708459939013866612ull, 386982305776168ull}},
{{11945465580694913936ull, 309585844620934ull}},
{{8044698484886131328ull, 495337351393495ull}},
{{6435758787908905062ull, 396269881114796ull}},
{{1459258215585213726ull, 317015904891837ull}},
{{6024161959678252286ull, 507225447826939ull}},
{{8508678382484512152ull, 405780358261551ull}},
{{3117593891245699398ull, 324624286609241ull}},
{{16056196670218850007ull, 519398858574785ull}},
{{12844957336175080005ull, 415519086859828ull}},
{{17654663498423884651ull, 332415269487862ull}},
{{13490066338510574148ull, 531864431180580ull}},
{{10792053070808459319ull, 425491544944464ull}},
{{12322991271388677778ull, 340393235955571ull}},
{{12338088404738063799ull, 544629177528914ull}},
{{13559819538532361362ull, 435703342023131ull}},
{{7158506816083978766ull, 348562673618505ull}},
{{11453610905734366026ull, 557700277789608ull}},
{{16541586354071313467ull, 446160222231686ull}},
{{9543920268515140451ull, 356928177785349ull}},
{{11324485029554022684ull, 285542542228279ull}},
{{7051129603060705325ull, 456868067565247ull}},
{{16708950126674295229ull, 365494454052197ull}},
{{5988462471855615537ull, 292395563241758ull}},
{{5892191140227074536ull, 467832901186813ull}},
{{12092450541665480275ull, 374266320949450ull}},
{{9673960433332384220ull, 299413056759560ull}},
{{15478336693331814752ull, 479060890815296ull}},
{{8693320539923541478ull, 383248712652237ull}},
{{18022702876164564152ull, 306598970121789ull}},
{{17768278157637571674ull, 490558352194863ull}},
{{3146576081884326370ull, 392446681755891ull}},
{{17274656124475102389ull, 313957345404712ull}},
{{12882054540192522529ull, 502331752647540ull}},
{{10305643632154018023ull, 401865402118032ull}},
{{865817276239393772ull, 321492321694426ull}},
{{12453354086208761005ull, 514387714711081ull}},
{{6273334454225098481ull, 411510171768865ull}},
{{5018667563380078785ull, 329208137415092ull}},
{{11719216916150036379ull, 526733019864147ull}},
{{1996675903436208457ull, 421386415891318ull}},
{{8976038352232787412ull, 337109132713054ull}},
{{3293614919346728889ull, 539374612340887ull}},
{{13702938379703114081ull, 431499689872709ull}},
{{14651699518504401588ull, 345199751898167ull}},
{{8685323970639401248ull, 552319603037068ull}},
{{14326956805995341645ull, 441855682429654ull}},
{{15150914259538183639ull, 353484545943723ull}},
{{1052684963404815941ull, 282787636754979ull}},
{{9062993570931526153ull, 452460218807966ull}},
{{3561046042003310599ull, 361968175046373ull}},
{{10227534463086469126ull, 289574540037098ull}},
{{12674706326196440278ull, 463319264059357ull}},
{{2761067431473331576ull, 370655411247486ull}},
{{16966249204146306553ull, 296524328997988ull}},
{{5009905838182628546ull, 474438926396782ull}},
{{15075971114771833807ull, 379551141117425ull}},
{{12060776891817467045ull, 303640912893940ull}},
{{850498953198395657ull, 485825460630305ull}},
{{680399162558716525ull, 388660368504244ull}},
{{4233668144788883543ull, 310928294803395ull}},
{{6773869031662213670ull, 497485271685432ull}},
{{16487141669555501905ull, 397988217348345ull}},
{{13189713335644401524ull, 318390573878676ull}},
{{13724843707547221792ull, 509424918205882ull}},
{{3601177336553956787ull, 407539934564706ull}},
{{17638337128210806723ull, 326031947651764ull}},
{{17153292960911559787ull, 521651116242823ull}},
{{2654587924503516860ull, 417320892994259ull}},
{{5813019154344723811ull, 333856714395407ull}},
{{12990179461693468421ull, 534170743032651ull}},
{{6702794754612864414ull, 427336594426121ull}},
{{1672886988948381208ull, 341869275540897ull}},
{{6365967997059320256ull, 546990840865435ull}},
{{5092774397647456204ull, 437592672692348ull}},
{{11452917147601785610ull, 350074138153878ull}},
{{14635318621420946653ull, 560118621046205ull}},
{{11708254897136757322ull, 448094896836964ull}},
{{13055952732451316181ull, 358475917469571ull}},
{{6755413371219142621ull, 286780733975657ull}},
{{14498010208692538518ull, 458849174361051ull}},
{{7909059352212120491ull, 367079339488841ull}},
{{2637898667027786069ull, 293663471591073ull}},
{{531289052502547388ull, 469861554545717ull}},
{{11493077686227768880ull, 375889243636573ull}},
{{16573159778466035750ull, 300711394909258ull}},
{{4380962757094195261ull, 481138231854814ull}},
{{7194119020417266532ull, 384910585483851ull}},
{{2065946401591902903ull, 307928468387081ull}},
{{14373560686772775614ull, 492685549419329ull}},
{{15188197364160130814ull, 394148439535463ull}},
{{1082511447102373682ull, 315318751628371ull}},
{{12800064759589528861ull, 504510002605393ull}},
{{17618749437155443735ull, 403608002084314ull}},
{{17784348364466265311ull, 322886401667451ull}},
{{2629515679952652235ull, 516618242667923ull}},
{{9482310173445942435ull, 413294594134338ull}},
{{14964545768240574594ull, 330635675307470ull}},
{{5496529155475367735ull, 529017080491953ull}},
{{11775920953864114834ull, 423213664393562ull}},
{{2042039133607471221ull, 338570931514850ull}},
{{3267262613771953953ull, 541713490423760ull}},
{{2613810091017563163ull, 433370792339008ull}},
{{9469745702297871176ull, 346696633871206ull}},
{{7772895494192773236ull, 554714614193930ull}},
{{6218316395354218589ull, 443771691355144ull}},
{{8664001931025285194ull, 355017353084115ull}},
{{6931201544820228155ull, 284013882467292ull}},
{{14779271286454275372ull, 454422211947667ull}},
{{4444719399679599651ull, 363537769558134ull}},
{{7245124334485590044ull, 290830215646507ull}},
{{15281547749918854394ull, 465328345034411ull}},
{{8535889385193173192ull, 372262676027529ull}},
{{10518060322896448876ull, 297810140822023ull}},
{{13139547701892407879ull, 476496225315237ull}},
{{3132940532030105657ull, 381196980252190ull}},
{{2506352425624084526ull, 304957584201752ull}},
{{7699512695740445564ull, 487932134722803ull}},
{{13538307786076177098ull, 390345707778242ull}},
{{3451948599377121032ull, 312276566222594ull}},
{{12901815388487214297ull, 499642505956150ull}},
{{10321452310789771438ull, 399714004764920ull}},
{{8257161848631817150ull, 319771203811936ull}},
{{5832761328327086794ull, 511633926099098ull}},
{{12044906692145490082ull, 409307140879278ull}},
{{17014622983200212712ull, 327445712703422ull}},
{{12466001514152699046ull, 523913140325476ull}},
{{6283452396580248914ull, 419130512260381ull}},
{{1337413102522288808ull, 335304409808305ull}},
{{2139860964035662092ull, 536487055693288ull}},
{{9090586400712350320ull, 429189644554630ull}},
{{7272469120569880256ull, 343351715643704ull}},
{{567904148686077440ull, 549362745029927ull}},
{{11522369763174592922ull, 439490196023941ull}},
{{5528546995797764014ull, 351592156819153ull}},
{{5156326378534512100ull, 562547450910645ull}},
{{4125061102827609680ull, 450037960728516ull}},
{{18057444141229729036ull, 360030368582812ull}},
{{7067257683499962583ull, 288024294866250ull}},
{{11307612293599940133ull, 460838871786000ull}},
{{9046089834879952106ull, 368671097428800ull}},
{{7236871867903961685ull, 294936877943040ull}},
{{11578994988646338696ull, 471899004708864ull}},
{{12952544805658981280ull, 377519203767091ull}},
{{6672687029785274700ull, 302015363013673ull}},
{{6986950432914529198ull, 483224580821877ull}},
{{16657606790557354328ull, 386579664657501ull}},
{{9636736617703973139ull, 309263731726001ull}},
{{8040080958842536376ull, 494821970761602ull}},
{{17500111211299760070ull, 395857576609281ull}},
{{10310740154297897733ull, 316686061287425ull}},
{{16497184246876636373ull, 506697698059880ull}},
{{13197747397501309098ull, 405358158447904ull}},
{{14247546732742957602ull, 324286526758323ull}},
{{659981883937270224ull, 518858442813318ull}},
{{7906683136633636826ull, 415086754250654ull}},
{{10014695324048819784ull, 332069403400523ull}},
{{12334163703736201331ull, 531311045440837ull}},
{{2488633333505140418ull, 425048836352670ull}},
{{1990906666804112334ull, 340039069082136ull}},
{{14253497111112310705ull, 544062510531417ull}},
{{4024100059406027917ull, 435250008425134ull}},
{{6908628862266732657ull, 348200006740107ull}},
{{14743154994368682575ull, 557120010784171ull}},
{{8105175180753035737ull, 445696008627337ull}},
{{17552186588828159559ull, 356556806901869ull}},
{{17731098085804437970ull, 285245445521495ull}},
{{9923012863577549136ull, 456392712834393ull}},
{{15317107920345859955ull, 365114170267514ull}},
{{15943035151018598287ull, 292091336214011ull}},
{{18130158612145936614ull, 467346137942418ull}},
{{3436080445491018321ull, 373876910353935ull}},
{{2748864356392814657ull, 299101528283148ull}},
{{708834155486593128ull, 478562445253037ull}},
{{11635113768615005472ull, 382849956202429ull}},
{{12997439829633914701ull, 306279964961943ull}},
{{17106554912672353198ull, 490047943939109ull}},
{{17374592744879792882ull, 392038355151287ull}},
{{6520976566420013659ull, 313630684121030ull}},
{{10433562506272021855ull, 501809094593648ull}},
{{15725547634501438130ull, 401447275674918ull}},
{{1512391663375419534ull, 321157820539935ull}},
{{2419826661400671255ull, 513852512863896ull}},
{{16693256588088178297ull, 411082010291116ull}},
{{9665256455728632314ull, 328865608232893ull}},
{{11775061514423901380ull, 526184973172629ull}},
{{13109398026281031427ull, 420947978538103ull}},
{{17866216050508645788ull, 336758382830482ull}},
{{13828550421846191968ull, 538813412528772ull}},
{{3684142707993132928ull, 431050730023018ull}},
{{10326011795878326989ull, 344840584018414ull}},
{{5453572429179592213ull, 551744934429463ull}},
{{11741555572827494416ull, 441395947543570ull}},
{{9393244458261995533ull, 353116758034856ull}},
{{3825246751867686103ull, 282493406427885ull}},
{{6120394802988297765ull, 451989450284616ull}},
{{1206967027648727889ull, 361591560227693ull}},
{{8344271251602802957ull, 289273248182154ull}},
{{2282787558338753763ull, 462837197091447ull}},
{{12894276490896733980ull, 370269757673157ull}},
{{2936723563233566537ull, 296215806138526ull}},
{{15766804145399437429ull, 473945289821641ull}},
{{8924094501577639620ull, 379156231857313ull}},
{{14517973230745932342ull, 303324985485850ull}},
{{4782013095483940132ull, 485319976777361ull}},
{{136261661645241782ull, 388255981421889ull}},
{{3798358144058103749ull, 310604785137511ull}},
{{17145419474718696968ull, 496967656220017ull}},
{{6337637950291136928ull, 397574124976014ull}},
{{8759459174974819866ull, 318059299980811ull}},
{{6636437050475891139ull, 508894879969298ull}},
{{12687847269864533557ull, 407115903975438ull}},
{{17528975445375447492ull, 325692723180350ull}},
{{9599616638891164372ull, 521108357088561ull}},
{{3990344496371021174ull, 416886685670849ull}},
{{6881624411838727263ull, 333509348536679ull}},
{{18389296688425784267ull, 533614957658686ull}},
{{11022088535998717090ull, 426891966126949ull}},
{{12507019643540883995ull, 341513572901559ull}},
{{8943184985439683423ull, 546421716642495ull}},
{{7154547988351746738ull, 437137373313996ull}},
{{2034289575939487067ull, 349709898651197ull}},
{{6944212136245089631ull, 559535837841915ull}},
{{5555369708996071705ull, 447628670273532ull}},
{{15512342211422588333ull, 358102936218825ull}},
{{12409873769138070667ull, 286482348975060ull}},
{{1409053956911361451ull, 458371758360097ull}},
{{12195289609754820130ull, 366697406688077ull}},
{{2377534058320035458ull, 293357925350462ull}},
{{7493403308053967056ull, 469372680560739ull}},
{{9684071461185083968ull, 375498144448591ull}},
{{4057908354206156851ull, 300398515558873ull}},
{{2803304551987940638ull, 480637624894197ull}},
{{13310690085816083480ull, 384510099915357ull}},
{{3269854439169046138ull, 307608079932286ull}},
{{16299813546896204790ull, 492172927891657ull}},
{{5661153208033143186ull, 393738342313326ull}},
{{839573751684604225ull, 314990673850661ull}},
{{12411364446921097730ull, 503985078161057ull}},
{{2550393928053057538ull, 403188062528846ull}},
{{16797710401410087323ull, 322550450023076ull}},
{{1050894939062767455ull, 516080720036923ull}},
{{8219413580734034610ull, 412864576029538ull}},
{{13954228494071048334ull, 330291660823630ull}},
{{3880021516804125719ull, 528466657317809ull}},
{{6793366028185210898ull, 422773325854247ull}},
{{16502739266773899688ull, 338218660683397ull}},
{{11646987567870598209ull, 541149857093436ull}},
{{5628241239554568244ull, 432919885674749ull}},
{{8191941806385564918ull, 346335908539799ull}},
{{2039060445991172899ull, 554137453663679ull}},
{{5320597171534848643ull, 443309962930943ull}},
{{11635175366711699560ull, 354647970344754ull}},
{{12997489108111269971ull, 283718376275803ull}},
{{17106633758236121631ull, 453949402041285ull}},
{{13685307006588897305ull, 363159521633028ull}},
{{18326943234754938490ull, 290527617306422ull}},
{{14565713916640260292ull, 464844187690276ull}},
{{7963222318570297910ull, 371875350152221ull}},
{{2681229040114328005ull, 297500280121777ull}},
{{7979315278924835131ull, 476000448194843ull}},
{{13762149852623688751ull, 380800358555874ull}},
{{14699068696840861324ull, 304640286844699ull}},
{{12450463470719647150ull, 487424458951519ull}},
{{13649719591317628043ull, 389939567161215ull}},
{{10919775673054102434ull, 311951653728972ull}},
{{2714245817918922602ull, 499122645966356ull}},
{{16928791913302779374ull, 399298116773084ull}},
{{17232382345384133823ull, 319438493418467ull}},
{{12814416493646972824ull, 511101589469548ull}},
{{17630230824401398905ull, 408881271575638ull}},
{{3036138215295388154ull, 327105017260511ull}},
{{15925867588698352017ull, 523368027616817ull}},
{{5361996441474860967ull, 418694422093454ull}},
{{7978945967921799097ull, 334955537674763ull}},
{{9076964733932968232ull, 535928860279621ull}},
{{3572222972404464262ull, 428743088223697ull}},
{{13925824822149302379ull, 342994470578957ull}},
{{7523924456471242514ull, 548791152926332ull}},
{{17087186009402724981ull, 439032922341065ull}},
{{13669748807522179985ull, 351226337872852ull}},
{{7114202833067846683ull, 561962140596564ull}},
{{9380711081196187670ull, 449569712477251ull}},
{{3815220050215039812ull, 359655769981801ull}},
{{17809571299139673143ull, 287724615985440ull}},
{{10048570004913925412ull, 460359385576705ull}},
{{8038856003931140330ull, 368287508461364ull}},
{{10120433617886822587ull, 294630006769091ull}},
{{8813996159135095493ull, 471408010830546ull}},
{{3361848112566166071ull, 377126408664437ull}},
{{13757524934278663826ull, 301701126931549ull}},
{{10943993450620131153ull, 482721803090479ull}},
{{12444543575238015245ull, 386177442472383ull}},
{{17334332489674232843ull, 308941953977906ull}},
{{1909490280285400286ull, 494307126364651ull}},
{{16284987483195961522ull, 395445701091720ull}},
{{13027989986556769217ull, 316356560873376ull}},
{{13466086349007010101ull, 506170497397402ull}},
{{3394171449721787435ull, 404936397917922ull}},
{{13783383604003160917ull, 323949118334337ull}},
{{7296018507437416175ull, 518318589334940ull}},
{{5836814805949932940ull, 414654871467952ull}},
{{15737498288985677321ull, 331723897174361ull}},
{{17801299632893263068ull, 530758235478978ull}},
{{3172993262088879485ull, 424606588383183ull}},
{{9917092239154924234ull, 339685270706546ull}},
{{8488649953164058129ull, 543496433130474ull}},
{{10480268777273156826ull, 434797146504379ull}},
{{12073563836560435784ull, 347837717203503ull}},
{{15628353323754786931ull, 556540347525605ull}},
{{12502682659003829545ull, 445232278020484ull}},
{{13691494941944973959ull, 356185822416387ull}},
{{3574498324072158521ull, 284948657933110ull}},
{{5719197318515453634ull, 455917852692976ull}},
{{886009040070452584ull, 364734282154381ull}},
{{15466202491024003360ull, 291787425723504ull}},
{{13677877541412674406ull, 466859881157607ull}},
{{3563604403646318878ull, 373487904926086ull}},
{{17608278781884696395ull, 298790323940868ull}},
{{6037153162564052294ull, 478064518305390ull}},
{{4829722530051241835ull, 382451614644312ull}},
{{14931824468266724437ull, 305961291715449ull}},
{{12822872705001028130ull, 489538066744719ull}},
{{13947646978742732827ull, 391630453395775ull}},
{{11158117582994186262ull, 313304362716620ull}},
{{17852988132790698019ull, 501286980346592ull}},
{{6903692876748737769ull, 401029584277274ull}},
{{9212303116140900538ull, 320823667421819ull}},
{{3671638541599709892ull, 513317867874911ull}},
{{17694706092247409206ull, 410654294299928ull}},
{{3087718429572196395ull, 328523435439943ull}},
{{1251000672573603909ull, 525637496703909ull}},
{{4690149352800793451ull, 420509997363127ull}},
{{14820165926466365730ull, 336407997890501ull}},
{{16333567852862364522ull, 538252796624802ull}},
{{5688156652806070971ull, 430602237299842ull}},
{{15618571766470587746ull, 344481789839873ull}},
{{2853621937901478455ull, 551170863743798ull}},
{{9661595179805003410ull, 440936690995038ull}},
{{15107973773327823375ull, 352749352796030ull}},
{{12086379018662258700ull, 282199482236824ull}},
{{8270159985633882950ull, 451519171578919ull}},
{{10305476803249016683ull, 361215337263135ull}},
{{8244381442599213346ull, 288972269810508ull}},
{{9501661493416831031ull, 462355631696813ull}},
{{14980026824217285471ull, 369884505357450ull}},
{{11984021459373828377ull, 295907604285960ull}},
{{727690261288573788ull, 473452166857537ull}},
{{11650198653256590000ull, 378761733486029ull}},
{{13009507737347182323ull, 303009386788823ull}},
{{17125863565013581394ull, 484815018862117ull}},
{{6321993222527044468ull, 387852015089694ull}},
{{8746943392763545898ull, 310281612071755ull}},
{{13995109428421673437ull, 496450579314808ull}},
{{128041098511607780ull, 397160463451847ull}},
{{11170479323035017193ull, 317728370761477ull}},
{{3115371657888386217ull, 508365393218364ull}},
{{6181646141052619296ull, 406692314574691ull}},
{{1255968098100185114ull, 325353851659753ull}},
{{16766944215927937475ull, 520566162655604ull}},
{{17102904187484260303ull, 416452930124483ull}},
{{2614276905761677273ull, 333162344099587ull}},
{{7872191863960593960ull, 533059750559339ull}},
{{9987102305910385491ull, 426447800447471ull}},
{{4300333029986398070ull, 341158240357977ull}},
{{10569881662720147235ull, 545853184572763ull}},
{{15834602959659938434ull, 436682547658210ull}},
{{12667682367727950747ull, 349346038126568ull}},
{{16578942973622810873ull, 558953661002509ull}},
{{16952503193640159021ull, 447162928802007ull}},
{{6183304925428306571ull, 357730343041606ull}},
{{1257295125600734933ull, 286184274433285ull}},
{{2011672200961175893ull, 457894839093256ull}},
{{16366733019736582007ull, 366315871274604ull}},
{{16782735230531175929ull, 293052697019683ull}},
{{4716283480398419548ull, 468884315231494ull}},
{{7462375599060645961ull, 375107452185195ull}},
{{5969900479248516769ull, 300085961748156ull}},
{{2173143137313806184ull, 480137538797050ull}},
{{1738514509851044947ull, 384110031037640ull}},
{{1390811607880835958ull, 307288024830112ull}},
{{5914647387351247856ull, 491660839728179ull}},
{{8421066724622908608ull, 393328671782543ull}},
{{14115551009182147532ull, 314662937426034ull}},
{{11516835170465705083ull, 503460699881655ull}},
{{9213468136372564066ull, 402768559905324ull}},
{{11060123323839961576ull, 322214847924259ull}},
{{6628150873918207552ull, 515543756678815ull}},
{{5302520699134566042ull, 412435005343052ull}},
{{15310063003533383803ull, 329948004274441ull}},
{{17117403176169593438ull, 527916806839106ull}},
{{10004573726193764427ull, 422333445471285ull}},
{{8003658980955011542ull, 337866756377028ull}},
{{9116505554786108144ull, 540586810203245ull}},
{{7293204443828886515ull, 432469448162596ull}},
{{2145214740321198889ull, 345975558530077ull}},
{{7121692399255828545ull, 553560893648123ull}},
{{13076051548888483483ull, 442848714918498ull}},
{{17839538868594607432ull, 354278971934798ull}},
{{3203584650649954976ull, 283423177547839ull}},
{{12504433070523748609ull, 453477084076542ull}},
{{2624848826935178240ull, 362781667261234ull}},
{{5789227876290052915ull, 290225333808987ull}},
{{12952113416805994988ull, 464360534094379ull}},
{{14051039548186706314ull, 371488427275503ull}},
{{172785194323634081ull, 297190741820403ull}},
{{15033851569885455823ull, 475505186912644ull}},
{{15716430070650274982ull, 380404149530115ull}},
{{12573144056520219985ull, 304323319624092ull}},
{{5359635231464710684ull, 486917311398548ull}},
{{11666405814655589193ull, 389533849118838ull}},
{{16711822281208292001ull, 311627079295070ull}},
{{8292171576223715586ull, 498603326872113ull}},
{{14012434890462793115ull, 398882661497690ull}},
{{11209947912370234492ull, 319106129198152ull}},
{{3178521400824733895ull, 510569806717044ull}},
{{6232165935401697439ull, 408455845373635ull}},
{{4985732748321357951ull, 326764676298908ull}},
{{4287823582572262398ull, 522823482078253ull}},
{{10808956495541630565ull, 418258785662602ull}},
{{1268467566949483806ull, 334607028530082ull}},
{{5718896921861084412ull, 535371245648131ull}},
{{885768722746957207ull, 428296996518505ull}},
{{708614978197565765ull, 342637597214804ull}},
{{8512481594599925871ull, 548220155543686ull}},
{{3120636460938030374ull, 438576124434949ull}},
{{6185857983492334622ull, 350860899547959ull}},
{{17276070403071556042ull, 561377439276734ull}},
{{17510205137199155157ull, 449101951421387ull}},
{{6629466480275503479ull, 359281561137110ull}},
{{5303573184220402783ull, 287425248909688ull}},
{{4796368280010734130ull, 459880398255501ull}},
{{147745809266676980ull, 367904318604401ull}},
{{14875591906380982877ull, 294323454883520ull}},
{{5354202976500020988ull, 470917527813633ull}},
{{11662060010683837436ull, 376734022250906ull}},
{{5640299193805159626ull, 301387217800725ull}},
{{9024478710088255402ull, 482219548481160ull}},
{{7219582968070604321ull, 385775638784928ull}},
{{13154364003940304103ull, 308620511027942ull}},
{{6289587147336845273ull, 493792817644708ull}},
{{12410367347353296864ull, 395034254115766ull}},
{{6238945063140727168ull, 316027403292613ull}},
{{6292963286283253146ull, 505643845268181ull}},
{{1345021814284692194ull, 404515076214545ull}},
{{1076017451427753755ull, 323612060971636ull}},
{{12789674366510136978ull, 517779297554617ull}},
{{2853041863724288936ull, 414223438043694ull}},
{{5971782305721341472ull, 331378750434955ull}},
{{9554851689154146355ull, 530206000695928ull}},
{{15022578980807137730ull, 424164800556742ull}},
{{4639365555161889538ull, 339331840445394ull}},
{{14801682517742843907ull, 542930944712630ull}},
{{11841346014194275126ull, 434344755770104ull}},
{{13162425626097330424ull, 347475804616083ull}},
{{17370532187013818355ull, 555961287385733ull}},
{{2828379305385323714ull, 444769029908587ull}},
{{13330749888533989941ull, 355815223926869ull}},
{{14353948725569102276ull, 284652179141495ull}},
{{4519573887201012025ull, 455443486626393ull}},
{{10994356739244630267ull, 364354789301114ull}},
{{12484834206137614536ull, 291483831440891ull}},
{{12597037100336362612ull, 466374130305426ull}},
{{6388280865527179766ull, 373099304244341ull}},
{{1421275877679833490ull, 298479443395473ull}},
{{17031436663255374877ull, 477567109432756ull}},
{{9935800515862389578ull, 382053687546205ull}},
{{7948640412689911662ull, 305642950036964ull}},
{{1649778216078127691ull, 489028720059143ull}},
{{8698520202346322799ull, 391222976047314ull}},
{{10648164976618968562ull, 312978380837851ull}},
{{9658366333106529053ull, 500765409340562ull}},
{{347995437001402596ull, 400612327472450ull}},
{{278396349601122077ull, 320489861977960ull}},
{{445434159361795323ull, 512783779164736ull}},
{{15113742586457077551ull, 410227023331788ull}},
{{1022947624939931071ull, 328181618665431ull}},
{{12704762644129620684ull, 525090589864689ull}},
{{13853158930045606870ull, 420072471891751ull}},
{{7393178329294575173ull, 336057977513401ull}},
{{4450387697387499631ull, 537692764021442ull}},
{{14628356602135730674ull, 430154211217153ull}},
{{634638837482853569ull, 344123368973723ull}},
{{15772817398940207004ull, 550597390357956ull}},
{{8928905104410255280ull, 440477912286365ull}},
{{7143124083528204224ull, 352382329829092ull}},
{{16782545711048294349ull, 281905863863273ull}},
{{4715980249225809019ull, 451049382181238ull}},
{{11151481828864467862ull, 360839505744990ull}},
{{8921185463091574289ull, 288671604595992ull}},
{{17963245555688429186ull, 461874567353587ull}},
{{6991898815066922702ull, 369499653882870ull}},
{{5593519052053538162ull, 295599723106296ull}},
{{1570932853801840413ull, 472959556970074ull}},
{{4946095097783382653ull, 378367645576059ull}},
{{7646224892968616446ull, 302694116460847ull}},
{{15923308643491696637ull, 484310586337355ull}},
{{12738646914793357309ull, 387448469069884ull}},
{{13880266346576596171ull, 309958775255907ull}},
{{7451030895554912580ull, 495934040409452ull}},
{{17028871160669661034ull, 396747232327561ull}},
{{9933748113793818504ull, 317397785862049ull}},
{{4825950537844378637ull, 507836457379279ull}},
{{7550109245017413232ull, 406269165903423ull}},
{{13418785025497751232ull, 325015332722738ull}},
{{17780707226054491649ull, 520024532356381ull}},
{{10535216966101682996ull, 416019625885105ull}},
{{8428173572881346396ull, 332815700708084ull}},
{{2417031272384423265ull, 532505121132935ull}},
{{1933625017907538612ull, 426004096906348ull}},
{{8925597643809851536ull, 340803277525078ull}},
{{10591607415353852134ull, 545285244040125ull}},
{{8473285932283081707ull, 436228195232100ull}},
{{6778628745826465366ull, 348982556185680ull}},
{{10845805993322344586ull, 558372089897088ull}},
{{16055342424141696315ull, 446697671917670ull}},
{{12844273939313357052ull, 357358137534136ull}},
{{6586070336708775318ull, 285886510027309ull}},
{{17916410168217861156ull, 457418416043694ull}},
{{18022476949316199248ull, 365934732834955ull}},
{{14417981559452959398ull, 292747786267964ull}},
{{12000724050899004067ull, 468396458028743ull}},
{{16979276870203023900ull, 374717166422994ull}},
{{17272770310904329443ull, 299773733138395ull}},
{{9189688423737375494ull, 479637973021433ull}},
{{14730448368473721041ull, 383710378417146ull}},
{{8095009880037066510ull, 306968302733717ull}},
{{16641364622801216739ull, 491149284373947ull}},
{{5934394068757152745ull, 392919427499158ull}},
{{12126212884489542842ull, 314335541999326ull}},
{{12023242985699447901ull, 502936867198922ull}},
{{2239896759075737674ull, 402349493759138ull}},
{{9170615036744410786ull, 321879595007310ull}},
{{14672984058791057258ull, 515007352011696ull}},
{{8049038432290935483ull, 412005881609357ull}},
{{17507277190058479356ull, 329604705287485ull}},
{{9564899430384015353ull, 527367528459977ull}},
{{273221914823391636ull, 421894022767982ull}},
{{11286623976084444279ull, 337515218214385ull}},
{{18058598361735110846ull, 540024349143016ull}},
{{10757529874646178353ull, 432019479314413ull}},
{{15984721529200763329ull, 345615583451530ull}},
{{7128810373011669711ull, 552984933522449ull}},
{{9392397113151246092ull, 442387946817959ull}},
{{11203266505262907196ull, 353910357454367ull}},
{{1583915574726505111ull, 283128285963494ull}},
{{9912962549046228824ull, 453005257541590ull}},
{{7930370039236983059ull, 362404206033272ull}},
{{17412342475615317417ull, 289923364826617ull}},
{{13102352702016866574ull, 463877383722588ull}},
{{17860579791097313906ull, 371101906978070ull}},
{{14288463832877851124ull, 296881525582456ull}},
{{15482844503120741153ull, 475010440931930ull}},
{{12386275602496592922ull, 380008352745544ull}},
{{13598369296739184661ull, 304006682196435ull}},
{{3310646801073143842ull, 486410691514297ull}},
{{13716563885084246043ull, 389128553211437ull}},
{{3594553478583576188ull, 311302842569150ull}},
{{5751285565733721901ull, 498084548110640ull}},
{{4601028452586977521ull, 398467638488512ull}},
{{14748869206295312986ull, 318774110790809ull}},
{{12530144285846769808ull, 510038577265295ull}},
{{10024115428677415846ull, 408030861812236ull}},
{{4329943528200022354ull, 326424689449789ull}},
{{14306607274603856413ull, 522279503119662ull}},
{{4066588190199264484ull, 417823602495730ull}},
{{3253270552159411587ull, 334258881996584ull}},
{{12583930512938879186ull, 534814211194534ull}},
{{13756493225093013672ull, 427851368955627ull}},
{{3626496950590590291ull, 342281095164502ull}},
{{9491743935686854789ull, 547649752263203ull}},
{{14972092778033304477ull, 438119801810562ull}},
{{4598976592942822935ull, 350495841448450ull}},
{{7358362548708516697ull, 560793346317520ull}},
{{5886690038966813358ull, 448634677054016ull}},
{{1020003216431540363ull, 358907741643213ull}},
{{8194700202629052936ull, 287126193314570ull}},
{{13111520324206484699ull, 459401909303312ull}},
{{3110518629881367112ull, 367521527442650ull}},
{{2488414903905093690ull, 294017221954120ull}},
{{3981463846248149904ull, 470427555126592ull}},
{{14253217521224250893ull, 376342044101273ull}},
{{334527572753669744ull, 301073635281019ull}},
{{7913941745889692238ull, 481717816449630ull}},
{{6331153396711753790ull, 385374253159704ull}},
{{8754271532111313355ull, 308299402527763ull}},
{{10317485636636191046ull, 493279044044421ull}},
{{4564639694567042513ull, 394623235235537ull}},
{{14719758199879364980ull, 315698588188429ull}},
{{12483566675581252999ull, 505117741101487ull}},
{{2608155710981181752ull, 404094192881190ull}},
{{2086524568784945402ull, 323275354304952ull}},
{{7027788124797822966ull, 517240566887923ull}},
{{13000928129322079019ull, 413792453510338ull}},
{{17779440132941483862ull, 331033962808270ull}},
{{10000360138996822563ull, 529654340493233ull}},
{{15378985740681278697ull, 423723472394586ull}},
{{8613839777803112634ull, 338978777915669ull}},
{{2714097200259249246ull, 542366044665071ull}},
{{16928673019175040689ull, 433892835732056ull}},
{{9853589600598122228ull, 347114268585645ull}},
{{15765743360956995565ull, 555382829737032ull}},
{{5233897059281775806ull, 444306263789626ull}},
{{497768832683510321ull, 355445011031701ull}},
{{15155610325114449550ull, 284356008825360ull}},
{{5802232446473567664ull, 454969614120577ull}},
{{15709832401404585100ull, 363975691296461ull}},
{{8878517106381757757ull, 291180553037169ull}},
{{3137580925985081442ull, 465888884859471ull}},
{{17267459999755706446ull, 372711107887576ull}},
{{10124619185062654834ull, 298168886310061ull}},
{{8820693066616427088ull, 477070218096098ull}},
{{14435252082776962317ull, 381656174476878ull}},
{{480155221995838884ull, 305324939581503ull}},
{{15525643614160983507ull, 488519903330404ull}},
{{16109863706070697129ull, 390815922664323ull}},
{{1819844520630826733ull, 312652738131459ull}},
{{10290448862493143420ull, 500244381010334ull}},
{{11921707904736425059ull, 400195504808267ull}},
{{2158668694305319401ull, 320156403846614ull}},
{{10832567540372331688ull, 512250246154582ull}},
{{1287356402814044704ull, 409800196923666ull}},
{{15787280381218877056ull, 327840157538932ull}},
{{10502253350982561997ull, 524544252062292ull}},
{{1023105051302228951ull, 419635401649834ull}},
{{4507832855783693484ull, 335708321319867ull}},
{{10901881383995819897ull, 537133314111787ull}},
{{1342807477712835271ull, 429706651289430ull}},
{{1074245982170268217ull, 343765321031544ull}},
{{9097491200956249794ull, 550024513650470ull}},
{{7277992960764999835ull, 440019610920376ull}},
{{2133045553870089545ull, 352015688736301ull}},
{{16463831702063712928ull, 281612550989040ull}},
{{7895386649592389070ull, 450580081582465ull}},
{{6316309319673911256ull, 360464065265972ull}},
{{16121093899964859974ull, 288371252212777ull}},
{{11036354980976134666ull, 461394003540444ull}},
{{12518432799522818056ull, 369115202832355ull}},
{{10014746239618254445ull, 295292162265884ull}},
{{4955547539163476142ull, 472467459625415ull}},
{{3964438031330780914ull, 377973967700332ull}},
{{14239596869290355700ull, 302379174160265ull}},
{{4336610917155017505ull, 483806678656425ull}},
{{3469288733724014004ull, 387045342925140ull}},
{{2775430986979211203ull, 309636274340112ull}},
{{8130038393908648248ull, 495418038944179ull}},
{{10193379529868828922ull, 396334431155343ull}},
{{15533401253378883784ull, 317067544924274ull}},
{{13785395561180483085ull, 507308071878839ull}},
{{14717665263686296791ull, 405846457503071ull}},
{{8084783396207127109ull, 324677166002457ull}},
{{16625002248673313698ull, 519483465603931ull}},
{{9610652984196740635ull, 415586772483145ull}},
{{7688522387357392508ull, 332469417986516ull}},
{{4922938190288007367ull, 531951068778426ull}},
{{249001737488495570ull, 425560855022741ull}},
{{14956596648958437749ull, 340448684018192ull}},
{{9173159379365859106ull, 544717894429108ull}},
{{14717225132976507931ull, 435774315543286ull}},
{{8084431291639296022ull, 348619452434629ull}},
{{1867043622397142665ull, 557791123895407ull}},
{{12561681342143445102ull, 446232899116325ull}},
{{10049345073714756081ull, 356986319293060ull}},
{{8039476058971804865ull, 285589055434448ull}},
{{9173812879612977461ull, 456942488695117ull}},
{{18407096747916112938ull, 365553990956093ull}},
{{3657630954107159381ull, 292443192764875ull}},
{{5852209526571455010ull, 467909108423800ull}},
{{4681767621257164008ull, 374327286739040ull}},
{{3745414097005731206ull, 299461829391232ull}},
{{9682011369951080253ull, 479138927025971ull}},
{{4056260281218953879ull, 383311141620777ull}},
{{14313054669200894073ull, 306648913296621ull}},
{{15522189841237609870ull, 490638261274594ull}},
{{16107100687731998219ull, 392510609019675ull}},
{{12885680550185598575ull, 314008487215740ull}},
{{2170344806587406105ull, 502413579545185ull}},
{{1736275845269924884ull, 401930863636148ull}},
{{8767718305699760553ull, 321544690908918ull}},
{{10339000474377706563ull, 514471505454269ull}},
{{11960549194244075573ull, 411577204363415ull}},
{{9568439355395260458ull, 329261763490732ull}},
{{552107709664775441ull, 526818821585172ull}},
{{11509732611957551322ull, 421455057268137ull}},
{{1829088460082220411ull, 337164045814510ull}},
{{2926541536131552659ull, 539462473303216ull}},
{{17098628487872883420ull, 431569978642572ull}},
{{6300205160814486089ull, 345255982914058ull}},
{{6390979442561267420ull, 552409572662493ull}},
{{12491481183532834582ull, 441927658129994ull}},
{{13682533761568177989ull, 353542126503995ull}},
{{10946027009254542391ull, 282833701203196ull}},
{{10134945585323447179ull, 452533921925114ull}},
{{11797305283000668067ull, 362027137540091ull}},
{{5748495411658624130ull, 289621710032073ull}},
{{5508243843911888285ull, 463394736051317ull}},
{{15474641519355241598ull, 370715788841053ull}},
{{1311666771258462308ull, 296572631072843ull}},
{{16856062092981180986ull, 474516209716548ull}},
{{2416803230159213819ull, 379612967773239ull}},
{{5622791398869281379ull, 303690374218591ull}},
{{1617768608707029560ull, 485904598749746ull}},
{{16051610145933264940ull, 388723678999796ull}},
{{9151939302004701629ull, 310978943199837ull}},
{{18332451697949432930ull, 497566309119739ull}},
{{18355310173101456667ull, 398053047295791ull}},
{{10994899323739255010ull, 318442437836633ull}},
{{13902490103240897694ull, 509507900538613ull}},
{{53945638366987185ull, 407606320430891ull}},
{{14800551769661231041ull, 326085056344712ull}},
{{8923487572490328373ull, 521736090151540ull}},
{{7138790057992262698ull, 417388872121232ull}},
{{16779078490619541128ull, 333911097696985ull}},
{{8399781511281714189ull, 534257756315177ull}},
{{17787871653251102321ull, 427406205052141ull}},
{{10540948507858971533ull, 341924964041713ull}},
{{13176168797832444131ull, 547079942466741ull}},
{{6851586223524044981ull, 437663953973393ull}},
{{12859966608303056631ull, 350131163178714ull}},
{{9507900129059159641ull, 560209861085943ull}},
{{14985017732731148359ull, 448167888868754ull}},
{{15677363000926829010ull, 358534311095003ull}},
{{1473843956515732238ull, 286827448876003ull}},
{{17115545589392812875ull, 458923918201604ull}},
{{17381785286256160623ull, 367139134561283ull}},
{{2837381784779197529ull, 293711307649027ull}},
{{8229159670388626369ull, 469938092238443ull}},
{{13962025365794721742ull, 375950473790754ull}},
{{14858969107377687716ull, 300760379032603ull}},
{{1638257683352838407ull, 481216606452166ull}},
{{16068001405649912019ull, 384973285161732ull}},
{{5475703495036108968ull, 307978628129386ull}},
{{1382427962573953703ull, 492765805007018ull}},
{{8484639999542983609ull, 394212644005614ull}},
{{10477060814376297210ull, 315370115204491ull}},
{{9384599673518254890ull, 504592184327186ull}},
{{3818330924072693589ull, 403673747461749ull}},
{{6744013554000065194ull, 322938997969399ull}},
{{18169119315883924958ull, 516702396751038ull}},
{{3467249008481408996ull, 413361917400831ull}},
{{17531194465752768490ull, 330689533920664ull}},
{{16981864700978698614ull, 529103254273063ull}},
{{2517445316557227922ull, 423282603418451ull}},
{{16771351512213423630ull, 338626082734760ull}},
{{8387418345831926192ull, 541801732375617ull}},
{{17777981120891271923ull, 433441385900493ull}},
{{3154338452487286569ull, 346753108720395ull}},
{{5046941523979658511ull, 554804973952632ull}},
{{15105599663409457778ull, 443843979162105ull}},
{{12084479730727566222ull, 355075183329684ull}},
{{13356932599323963301ull, 284060146663747ull}},
{{6613696899950699989ull, 454496234661996ull}},
{{1601608705218649668ull, 363596987729597ull}},
{{12349333408400650704ull, 290877590183677ull}},
{{5001538194473399834ull, 465404144293884ull}},
{{7690579370320630190ull, 372323315435107ull}},
{{17220509940482235122ull, 297858652348085ull}},
{{9106071831062024579ull, 476573843756937ull}},
{{18352903909075350633ull, 381259075005549ull}},
{{18371671942002190829ull, 305007260004439ull}},
{{18326628662977774357ull, 488011616007103ull}},
{{3593256486156488516ull, 390409292805683ull}},
{{10253302818409011459ull, 312327434244546ull}},
{{9026586879970597689ull, 499723894791274ull}},
{{10910618318718388474ull, 399779115833019ull}},
{{12417843469716621102ull, 319823292666415ull}},
{{1421805477837042148ull, 511717268266265ull}},
{{1137444382269633718ull, 409373814613012ull}},
{{11978001950041437944ull, 327499051690409ull}},
{{8096756675840569741ull, 523998482704655ull}},
{{6477405340672455793ull, 419198786163724ull}},
{{8871273087279874958ull, 335359028930979ull}},
{{3125990495422068963ull, 536574446289567ull}},
{{13568838840563386140ull, 429259557031653ull}},
{{18233768701934529558ull, 343407645625322ull}},
{{14416634664127606000ull, 549452233000516ull}},
{{7843958916560174477ull, 439561786400413ull}},
{{13653864762731960228ull, 351649429120330ull}},
{{3399439546661584749ull, 562639086592529ull}},
{{6408900452071178122ull, 450111269274023ull}},
{{12505817991140763144ull, 360089015419218ull}},
{{17383352022396431162ull, 288071212335374ull}},
{{16745316791608558889ull, 460913939736599ull}},
{{17085602248028757434ull, 368731151789279ull}},
{{17357830613164916271ull, 294984921431423ull}},
{{5636436092612404094ull, 471975874290278ull}},
{{11887846503573743922ull, 377580699432222ull}},
{{2131579573375174491ull, 302064559545778ull}},
{{18167922576367920478ull, 483303295273244ull}},
{{18223686875836246706ull, 386642636218595ull}},
{{14578949500668997365ull, 309314108974876ull}},
{{15947621571586575137ull, 494902574359802ull}},
{{5379399627785439463ull, 395922059487842ull}},
{{15371566146454082540ull, 316737647590273ull}},
{{2458412945875070125ull, 506780236144438ull}},
{{9345427986183876747ull, 405424188915550ull}},
{{7476342388947101397ull, 324339351132440ull}},
{{11962147822315362236ull, 518942961811904ull}},
{{13259067072594200112ull, 415154369449523ull}},
{{17985951287559180736ull, 332123495559618ull}},
{{6641429171643227238ull, 531397592895390ull}},
{{5313143337314581790ull, 425118074316312ull}},
{{15318561114077396402ull, 340094459453049ull}},
{{13441651338298103274ull, 544151135124879ull}},
{{14442669885380392942ull, 435320908099903ull}},
{{486089464078583384ull, 348256726479923ull}},
{{15535138401493374707ull, 557210762367876ull}},
{{8738761906452789443ull, 445768609894301ull}},
{{3301660710420321231ull, 356614887915441ull}},
{{17398723827303898277ull, 285291910332352ull}},
{{13080562864718595951ull, 456467056531764ull}},
{{14153799106516787084ull, 365173645225411ull}},
{{7633690470471519344ull, 292138916180329ull}},
{{1145858308528699981ull, 467422265888527ull}},
{{11984733091048690954ull, 373937812710821ull}},
{{5898437658097042440ull, 299150250168657ull}},
{{13126849067697178228ull, 478640400269851ull}},
{{6812130439415832259ull, 382912320215881ull}},
{{1760355536790755484ull, 306329856172705ull}},
{{2816568858865208774ull, 490127769876328ull}},
{{9631952716575987666ull, 392102215901062ull}},
{{326864543776969486ull, 313681772720850ull}},
{{522983270043151178ull, 501890836353360ull}},
{{418386616034520942ull, 401512669082688ull}},
{{7713406922311437400ull, 321210135266150ull}},
{{12341451075698299841ull, 513936216425840ull}},
{{9873160860558639872ull, 411148973140672ull}},
{{519831058963091251ull, 328919178512538ull}},
{{15589124953308587295ull, 526270685620060ull}},
{{12471299962646869836ull, 421016548496048ull}},
{{17355737599601316515ull, 336813238796838ull}},
{{5633087270910644485ull, 538901182074942ull}},
{{15574516260954246558ull, 431120945659953ull}},
{{1391566564537666277ull, 344896756527963ull}},
{{16983901762227907336ull, 551834810444740ull}},
{{13587121409782325868ull, 441467848355792ull}},
{{3490999498342040048ull, 353174278684634ull}},
{{6482148413415542362ull, 282539422947707ull}},
{{14060786276206778102ull, 452063076716331ull}},
{{7559280206223512158ull, 361650461373065ull}},
{{6047424164978809727ull, 289320369098452ull}},
{{13365227478708005886ull, 462912590557523ull}},
{{18070879612450225355ull, 370330072446018ull}},
{{3388657245734449315ull, 296264057956815ull}},
{{5421851593175118904ull, 474022492730904ull}},
{{8026830089282005446ull, 379217994184723ull}},
{{13800161700909425003ull, 303374395347778ull}},
{{18390909906713169682ull, 485399032556445ull}},
{{14712727925370535745ull, 388319226045156ull}},
{{8080833525554518273ull, 310655380836125ull}},
{{12929333640887229237ull, 497048609337800ull}},
{{10343466912709783390ull, 397638887470240ull}},
{{8274773530167826712ull, 318111109976192ull}},
{{16928986463010433062ull, 508977775961907ull}},
{{6164491540924525803ull, 407182220769526ull}},
{{1242244417997710319ull, 325745776615621ull}},
{{13055637513022067481ull, 521193242584993ull}},
{{17823207639901474631ull, 416954594067994ull}},
{{17947914926663090028ull, 333563675254395ull}},
{{10269919808951392429ull, 533701880407033ull}},
{{15594633476644934589ull, 426961504325626ull}},
{{8786357966574037348ull, 341569203460501ull}},
{{6679475117034639111ull, 546510725536802ull}},
{{16411626537853442258ull, 437208580429441ull}},
{{9439952415540843483ull, 349766864343553ull}},
{{11414575050123439250ull, 559626982949685ull}},
{{9131660040098751400ull, 447701586359748ull}},
{{14684025661562821766ull, 358161269087798ull}},
{{679174085024526443ull, 286529015270239ull}},
{{8465376165523062956ull, 458446424432382ull}},
{{17840347376644181335ull, 366757139545905ull}},
{{14272277901315345068ull, 293405711636724ull}},
{{11767598197878821139ull, 469449138618759ull}},
{{13103427373044967234ull, 375559310895007ull}},
{{3104044268952153141ull, 300447448716006ull}},
{{16034517274549175995ull, 480715917945609ull}},
{{16516962634381251119ull, 384572734356487ull}},
{{5834872478021180249ull, 307658187485190ull}},
{{9335795964833888398ull, 492253099976304ull}},
{{11157985586609021042ull, 393802479981043ull}},
{{16305086098771037480ull, 315041983984834ull}},
{{15020091313807928998ull, 504067174375735ull}},
{{12016073051046343199ull, 403253739500588ull}},
{{16991556070320895205ull, 322602991600470ull}},
{{8739745638803880713ull, 516164786560753ull}},
{{14370494140526925216ull, 412931829248602ull}},
{{4117697682937719527ull, 330345463398882ull}},
{{10277665107442261566ull, 528552741438211ull}},
{{4532783271211898930ull, 422842193150569ull}},
{{7315575431711429467ull, 338273754520455ull}},
{{11704920690738287147ull, 541238007232728ull}},
{{16742634182074450364ull, 432990405786182ull}},
{{6015409716175739645ull, 346392324628946ull}},
{{2245957916397362785ull, 554227719406314ull}},
{{5486115147859800551ull, 443382175525051ull}},
{{699543303545930118ull, 354705740420041ull}},
{{15317029901804385387ull, 283764592336032ull}},
{{9749852583919375327ull, 454023347737652ull}},
{{421184437651679615ull, 363218678190122ull}},
{{11404993994347074661ull, 290574942552097ull}},
{{3490595131987678166ull, 464919908083356ull}},
{{17549871364557783825ull, 371935926466684ull}},
{{17729245906388137383ull, 297548741173347ull}},
{{13609398191253378521ull, 476077985877356ull}},
{{7198169738260792493ull, 380862388701885ull}},
{{5758535790608633995ull, 304689910961508ull}},
{{5524308450231904068ull, 487503857538413ull}},
{{11798144389669343901ull, 390003086030730ull}},
{{9438515511735475121ull, 312002468824584ull}},
{{4033578374551029224ull, 499203950119335ull}},
{{3226862699640823379ull, 399363160095468ull}},
{{9960187789196479349ull, 319490528076374ull}},
{{4868254018488635990ull, 511184844922199ull}},
{{7583952029532819115ull, 408947875937759ull}},
{{9756510438368165615ull, 327158300750207ull}},
{{853021442421423692ull, 523453281200332ull}},
{{11750463598162869923ull, 418762624960265ull}},
{{9400370878530295938ull, 335010099968212ull}},
{{283198146680832208ull, 536016159949140ull}},
{{226558517344665767ull, 428812927959312ull}},
{{11249293258101463583ull, 343050342367449ull}},
{{6930822768736610763ull, 548880547787919ull}},
{{9234007029731198934ull, 439104438230335ull}},
{{7387205623784959147ull, 351283550584268ull}},
{{8130180183314024312ull, 562053680934829ull}},
{{10193492961393129773ull, 449642944747863ull}},
{{15533491998598324465ull, 359714355798290ull}},
{{12426793598878659572ull, 287771484638632ull}},
{{5125474499238214022ull, 460434375421812ull}},
{{15168426043616302187ull, 368347500337449ull}},
{{15824089649634952073ull, 294678000269959ull}},
{{14250496995190192347ull, 471484800431935ull}},
{{11400397596152153878ull, 377187840345548ull}},
{{16499015706405543748ull, 301750272276438ull}},
{{4262332241797408058ull, 482800435642302ull}},
{{14477912237663657416ull, 386240348513841ull}},
{{7892980975389015610ull, 308992278811073ull}},
{{8939420745880514653ull, 494387646097717ull}},
{{18219583040930142692ull, 395510116878173ull}},
{{3507619988518383184ull, 316408093502539ull}},
{{12990889611113233740ull, 506252949604062ull}},
{{3014014059406766346ull, 405002359683250ull}},
{{2411211247525413077ull, 324001887746600ull}},
{{3857937996040660923ull, 518403020394560ull}},
{{3086350396832528738ull, 414722416315648ull}},
{{9847777946949843637ull, 331777933052518ull}},
{{12067095900377839496ull, 530844692884029ull}},
{{13343025535044181920ull, 424675754307223ull}},
{{18053118057519166182ull, 339740603445778ull}},
{{6748896003579203953ull, 543584965513246ull}},
{{1709767988121452839ull, 434867972410597ull}},
{{12435860834722893241ull, 347894377928477ull}},
{{5139982076588987892ull, 556631004685564ull}},
{{7801334476013100637ull, 445304803748451ull}},
{{2551718766068570186ull, 356243842998761ull}},
{{16798770271822497442ull, 284995074399008ull}},
{{4741939546464533968ull, 455992119038414ull}},
{{7482900451913537497ull, 364793695230731ull}},
{{2296971546788919675ull, 291834956184585ull}},
{{3675154474862271480ull, 466935929895336ull}},
{{17697518838857458476ull, 373548743916268ull}},
{{3089968626860235811ull, 298838995133015ull}},
{{4943949802976377299ull, 478142392212824ull}},
{{7644508657123012162ull, 382513913770259ull}},
{{9804955740440320053ull, 306011131016207ull}},
{{930533925736870792ull, 489617809625932ull}},
{{11812473584815227603ull, 391694247700745ull}},
{{9449978867852182082ull, 313355398160596ull}},
{{7741268559079670686ull, 501368637056954ull}},
{{9882363662005646872ull, 401094909645563ull}},
{{15284588559088338144ull, 320875927716450ull}},
{{6008597620831789414ull, 513401484346321ull}},
{{1117529281923521208ull, 410721187477057ull}},
{{11962069869764547936ull, 328576949981645ull}},
{{692567717913725082ull, 525723119970633ull}},
{{7932751803814800712ull, 420578495976506ull}},
{{2656852628309930246ull, 336462796781205ull}},
{{4250964205295888394ull, 538340474849928ull}},
{{10779468993720531361ull, 430672379879942ull}},
{{1244877565492604443ull, 344537903903954ull}},
{{9370501734271987755ull, 551260646246326ull}},
{{3807052572675679881ull, 441008516997061ull}},
{{17803037317108185197ull, 352806813597648ull}},
{{3174383409460817188ull, 282245450878119ull}},
{{12457711084621128148ull, 451592721404990ull}},
{{9966168867696902518ull, 361274177123992ull}},
{{594237464673701368ull, 289019341699194ull}},
{{8329477572961742835ull, 462430946718710ull}},
{{6663582058369394268ull, 369944757374968ull}},
{{12709563276179336061ull, 295955805899974ull}},
{{9267254797661206728ull, 473529289439959ull}},
{{11103152652870875705ull, 378823431551967ull}},
{{1503824492812879918ull, 303058745241574ull}},
{{9784816817984428515ull, 484893992386518ull}},
{{15206551083871363459ull, 387915193909214ull}},
{{15854589681839001090ull, 310332155127371ull}},
{{17988645861458581098ull, 496531448203794ull}},
{{18080265503908775201ull, 397225158563035ull}},
{{14464212403127020161ull, 317780126850428ull}},
{{1006646956551770319ull, 508448202960686ull}},
{{15562712824209057548ull, 406758562368548ull}},
{{1382123815141515068ull, 325406849894839ull}},
{{9590095733710244756ull, 520650959831742ull}},
{{293378957484375158ull, 416520767865394ull}},
{{3924051980729410450ull, 333216614292315ull}},
{{6278483169167056720ull, 533146582867704ull}},
{{8712135350075555699ull, 426517266294163ull}},
{{14348405909544265206ull, 341213813035330ull}},
{{4510705381561272713ull, 545942100856529ull}},
{{7297913119990928494ull, 436753680685223ull}},
{{13217028125476563441ull, 349402944548178ull}},
{{17457896186020591183ull, 559044711277085ull}},
{{13966316948816472946ull, 447235769021668ull}},
{{105007114827447387ull, 357788615217335ull}},
{{84005691861957910ull, 286230892173868ull}},
{{14891804365946773949ull, 457969427478188ull}},
{{845397048531688189ull, 366375541982551ull}},
{{15433712897792991844ull, 293100433586040ull}},
{{6247196562759235335ull, 468960693737665ull}},
{{4997757250207388268ull, 375168554990132ull}},
{{15066252244391641584ull, 300134843992105ull}},
{{5659259517317074918ull, 480215750387369ull}},
{{8216756428595570258ull, 384172600309895ull}},
{{6573405142876456206ull, 307338080247916ull}},
{{3138750599118509284ull, 491740928396666ull}},
{{17268395738262448720ull, 393392742717332ull}},
{{6436018961126138329ull, 314714194173866ull}},
{{2918932708318000681ull, 503542710678186ull}},
{{17092541425622041837ull, 402834168542548ull}},
{{2605986696271902500ull, 322267334834039ull}},
{{11548276343518864647ull, 515627735734462ull}},
{{1859923445331271071ull, 412502188587570ull}},
{{1487938756265016857ull, 330001750870056ull}},
{{13448748454249757941ull, 528002801392089ull}},
{{14448347578141716676ull, 422402241113671ull}},
{{7869329247771463017ull, 337921792890937ull}},
{{16280275611176251151ull, 540674868625499ull}},
{{16713569303682911244ull, 432539894900399ull}},
{{17060204257688239318ull, 346031915920319ull}},
{{16228280368075451940ull, 553651065472511ull}},
{{9293275479718451229ull, 442920852378009ull}},
{{11123969198516671306ull, 354336681902407ull}},
{{1520477729329516398ull, 283469345521926ull}},
{{13500810811152957207ull, 453550952835081ull}},
{{7111299834180455442ull, 362840762268065ull}},
{{5689039867344364354ull, 290272609814452ull}},
{{12791812602492893290ull, 464436175703123ull}},
{{17612147711478135278ull, 371548940562498ull}},
{{3021671724956777253ull, 297239152449999ull}},
{{12213372389414664251ull, 475582643919998ull}},
{{17149395541015552047ull, 380466115135998ull}},
{{2651469988586710668ull, 304372892108799ull}},
{{11621049611222557716ull, 486996627374078ull}},
{{16675537318461866819ull, 389597301899262ull}},
{{5961732225285672808ull, 311677841519410ull}},
{{9538771560457076494ull, 498684546431056ull}},
{{3941668433623750872ull, 398947637144845ull}},
{{3153334746899000697ull, 319158109715876ull}},
{{16113382039264132086ull, 510652975545401ull}},
{{9201356816669395345ull, 408522380436321ull}},
{{3671736638593605953ull, 326817904349057ull}},
{{9564127436491679848ull, 522908646958491ull}},
{{3961953134451433555ull, 418326917566793ull}},
{{10548260137044967490ull, 334661534053434ull}},
{{5809169775046217015ull, 535458454485495ull}},
{{4647335820036973612ull, 428366763588396ull}},
{{28519841287668566ull, 342693410870717ull}},
{{3734980560802180030ull, 548309457393147ull}},
{{14056030892867474993ull, 438647565914517ull}},
{{3866127084810159348ull, 350918052731614ull}},
{{13564500965180075604ull, 561468884370582ull}},
{{3472903142660239837ull, 449175107496466ull}},
{{17535717773095833162ull, 359340085997172ull}},
{{6649876588992845883ull, 287472068797738ull}},
{{6950453727646643090ull, 459955310076381ull}},
{{1871014167375404149ull, 367964248061105ull}},
{{1496811333900323319ull, 294371398448884ull}},
{{9773595763724337957ull, 470994237518214ull}},
{{11508225425721380689ull, 376795390014571ull}},
{{5517231525835194228ull, 301436312011657ull}},
{{12516919256078221088ull, 482298099218651ull}},
{{6324186590120666547ull, 385838479374921ull}},
{{1370000457354622914ull, 308670783499937ull}},
{{5881349546509306986ull, 493873253599899ull}},
{{8394428451949355912ull, 395098602879919ull}},
{{10404891576301395053ull, 316078882303935ull}},
{{16647826522082232085ull, 505726211686296ull}},
{{9628912402923875344ull, 404580969349037ull}},
{{324432292855279629ull, 323664775479230ull}},
{{519091668568447407ull, 517863640766768ull}},
{{7793970964338578572ull, 414290912613414ull}},
{{9924525586212773180ull, 331432730090731ull}},
{{8500543308456616443ull, 530292368145170ull}},
{{6800434646765293154ull, 424233894516136ull}},
{{1750998902670324200ull, 339387115612909ull}},
{{10180295873756339367ull, 543019384980654ull}},
{{11833585513746981816ull, 434415507984523ull}},
{{16845566040481406099ull, 347532406387618ull}},
{{4816812776318787820ull, 556051850220190ull}},
{{3853450221055030256ull, 444841480176152ull}},
{{14150806621069755174ull, 355873184140921ull}},
{{7631296482113893816ull, 284698547312737ull}},
{{15899423186124140429ull, 455517675700379ull}},
{{16408887363641222667ull, 364414140560303ull}},
{{2059063446687247164ull, 291531312448243ull}},
{{18051896773667236755ull, 466450099917188ull}},
{{3373470974708058434ull, 373160079933751ull}},
{{17456172038734088040ull, 298528063947000ull}},
{{9483131188264989248ull, 477644902315201ull}},
{{3897156135870081075ull, 382115921852161ull}},
{{17875120167663706153ull, 305692737481728ull}},
{{6464099379810467906ull, 489108379970766ull}},
{{1481930689106464001ull, 391286703976613ull}},
{{8564242180768991847ull, 313029363181290ull}},
{{13702787489230386956ull, 500846981090064ull}},
{{14651578806126219888ull, 400677584872051ull}},
{{8031914230159065587ull, 320542067897641ull}},
{{5472365138770684293ull, 512867308636226ull}},
{{688543296274637111ull, 410293846908981ull}},
{{15308229895987350982ull, 328235077527184ull}},
{{13425121389354030601ull, 525176124043495ull}},
{{10740097111483224481ull, 420140899234796ull}},
{{4902728874444669262ull, 336112719387837ull}},
{{11533715013853381142ull, 537780351020539ull}},
{{12916320825824615237ull, 430224280816431ull}},
{{6643707845917781866ull, 344179424653145ull}},
{{10629932553468450986ull, 550687079445032ull}},
{{1125248413290940142ull, 440549663556026ull}},
{{15657593989600393407ull, 352439730844820ull}},
{{12526075191680314725ull, 281951784675856ull}},
{{12663022677204682914ull, 451122855481370ull}},
{{10130418141763746331ull, 360898284385096ull}},
{{4414985698669086742ull, 288718627508077ull}},
{{10753325932612449110ull, 461949804012923ull}},
{{15981358375573779934ull, 369559843210338ull}},
{{1717040256233292978ull, 295647874568271ull}},
{{13815310854198999734ull, 473036599309233ull}},
{{18430946312843020434ull, 378429279447386ull}},
{{11055408235532506024ull, 302743423557909ull}},
{{6620606732626278669ull, 484389477692655ull}},
{{5296485386101022935ull, 387511582154124ull}},
{{7926537123622728671ull, 310009265723299ull}},
{{1614412953570634904ull, 496014825157279ull}},
{{4980879177598418247ull, 396811860125823ull}},
{{11363400971562555244ull, 317449488100658ull}},
{{14492092739758178067ull, 507919180961053ull}},
{{525627747580811484ull, 406335344768843ull}},
{{7799199827548469833ull, 325068275815074ull}},
{{1410673279851820764ull, 520109241304119ull}},
{{4817887438623366934ull, 416087393043295ull}},
{{3854309950898693547ull, 332869914434636ull}},
{{17234942365663640646ull, 532591863095417ull}},
{{6409256263047091870ull, 426073490476334ull}},
{{8816753825179583819ull, 340858792381067ull}},
{{17796154935029244434ull, 545374067809707ull}},
{{6858226318539574901ull, 436299254247766ull}},
{{1797232240089749597ull, 349039403398213ull}},
{{17632966843111240649ull, 558463045437140ull}},
{{14106373474488992519ull, 446770436349712ull}},
{{3906401150107373369ull, 357416349079770ull}},
{{3125120920085898695ull, 285933079263816ull}},
{{16068239916363168882ull, 457492926822105ull}},
{{12854591933090535105ull, 365994341457684ull}},
{{13973022361214338407ull, 292795473166147ull}},
{{7599440518975300159ull, 468472757065836ull}},
{{2390203600438329804ull, 374778205652669ull}},
{{5601511695092574166ull, 299822564522135ull}},
{{8962418712148118666ull, 479716103235416ull}},
{{3480586154976584610ull, 383772882588333ull}},
{{10163166553465088334ull, 307018306070666ull}},
{{8882368856060320688ull, 491229289713066ull}},
{{3416546270106346227ull, 392983431770453ull}},
{{10111934645568897628ull, 314386745416362ull}},
{{1421700173942594913ull, 503018792666180ull}},
{{1137360139154075930ull, 402415034132944ull}},
{{4599236926065171067ull, 321932027306355ull}},
{{7358779081704273708ull, 515091243690168ull}},
{{13265720894847239612ull, 412072994952134ull}},
{{14301925530619702013ull, 329658395961707ull}},
{{8125685590023881928ull, 527453433538732ull}},
{{17568594916244836512ull, 421962746830985ull}},
{{14054875932995869210ull, 337570197464788ull}},
{{351708604341928797ull, 540112315943662ull}},
{{11349413327699274007ull, 432089852754929ull}},
{{12768879476901329529ull, 345671882203943ull}},
{{16740858348300216923ull, 553075011526309ull}},
{{17082035493382083861ull, 442460009221047ull}},
{{6286930765221846443ull, 353968007376838ull}},
{{12408242241661297800ull, 283174405901470ull}},
{{1406443512948524865ull, 453079049442353ull}},
{{8503852439842640538ull, 362463239553882ull}},
{{17871128396099843400ull, 289970591643105ull}},
{{10147061360050197824ull, 463952946628969ull}},
{{11806997902782068583ull, 371162357303175ull}},
{{9445598322225654866ull, 296929885842540ull}},
{{15112957315561047786ull, 475087817348064ull}},
{{15779714667190748552ull, 380070253878451ull}},
{{8934422919010688518ull, 304056203102761ull}},
{{6916379040933280983ull, 486489924964418ull}},
{{12911800862230445433ull, 389191939971534ull}},
{{14018789504526266669ull, 311353551977227ull}},
{{7672667948274385378ull, 498165683163564ull}},
{{9827483173361418626ull, 398532546530851ull}},
{{4172637723947224577ull, 318826037224681ull}},
{{17744266802541290293ull, 510121659559489ull}},
{{17884762256774942558ull, 408097327647591ull}},
{{10618460990678043723ull, 326477862118073ull}},
{{13300188770342959634ull, 522364579388917ull}},
{{3261453386790547061ull, 417891663511134ull}},
{{6298511524174347972ull, 334313330808907ull}},
{{13766967253420867078ull, 534901329294251ull}},
{{7324224987994783339ull, 427921063435401ull}},
{{2170031175653916348ull, 342336850748321ull}},
{{14540096325271997127ull, 547738961197313ull}},
{{564030615991866732ull, 438191168957851ull}},
{{15208619751761134678ull, 350552935166280ull}},
{{5887047529108263869ull, 560884696266049ull}},
{{8398986838028521418ull, 448707757012839ull}},
{{10408538285164727458ull, 358966205610271ull}},
{{4637481813389871643ull, 287172964488217ull}},
{{11109319716165704952ull, 459476743181147ull}},
{{1508758143448743315ull, 367581394544918ull}},
{{8585704144242815299ull, 294065115635934ull}},
{{2669080186562773508ull, 470504185017495ull}},
{{2135264149250218807ull, 376403348013996ull}},
{{16465606578367816338ull, 301122678411196ull}},
{{519528822195133879ull, 481796285457915ull}},
{{415623057756107103ull, 385437028366332ull}},
{{11400544890430616652ull, 308349622693065ull}},
{{18240871824688986643ull, 493359396308904ull}},
{{18282046274493099638ull, 394687517047123ull}},
{{3557590575368748740ull, 315750013637699ull}},
{{13070842550073818631ull, 505200021820318ull}},
{{17835371669542875551ull, 404160017456254ull}},
{{17957646150376210764ull, 323328013965003ull}},
{{6596140952150475284ull, 517324822344006ull}},
{{1587563946978469904ull, 413859857875205ull}},
{{1270051157582775923ull, 331087886300164ull}},
{{9410779481616262123ull, 529740618080262ull}},
{{149925955809189052ull, 423792494464210ull}},
{{119940764647351242ull, 339033995571368ull}},
{{14949300482403403280ull, 542454392914188ull}},
{{891393941696991654ull, 433963514331351ull}},
{{15470510412325234616ull, 347170811465080ull}},
{{6306072586010823770ull, 555473298344129ull}},
{{8734206883550569339ull, 444378638675303ull}},
{{14366063136324276117ull, 355502910940242ull}},
{{4114152879575600247ull, 284402328752194ull}},
{{13961342236804781043ull, 455043726003510ull}},
{{11169073789443824834ull, 364034980802808ull}},
{{16313956661038880514ull, 291227984642246ull}},
{{276888954468836560ull, 465964775427595ull}},
{{221511163575069248ull, 372771820342076ull}},
{{14934604189827696691ull, 298217456273660ull}},
{{5448622630014763089ull, 477147930037857ull}},
{{15426944548237541441ull, 381718344030285ull}},
{{12341555638590033153ull, 305374675224228ull}},
{{16057140207002142722ull, 488599480358765ull}},
{{12845712165601714177ull, 390879584287012ull}},
{{2897872102997550695ull, 312703667429610ull}},
{{4636595364796081113ull, 500325867887376ull}},
{{19927477094954567ull, 400260694309901ull}},
{{14773337240643604946ull, 320208555447920ull}},
{{5190595511320216298ull, 512333688716673ull}},
{{11531174038539993685ull, 409866950973338ull}},
{{16603636860315815594ull, 327893560778670ull}},
{{8119074902795753335ull, 524629697245873ull}},
{{13873957551720423314ull, 419703757796698ull}},
{{31119597150607682ull, 335763006237359ull}},
{{7428488984924792937ull, 537220809979774ull}},
{{9632140002681744673ull, 429776647983819ull}},
{{11395060816887306061ull, 343821318387055ull}},
{{18232097307019689699ull, 550114109419288ull}},
{{3517631401390020789ull, 440091287535431ull}},
{{17571500380079657924ull, 352073030028344ull}},
{{17746549118805636662ull, 281658424022675ull}},
{{9947734516379467044ull, 450653478436281ull}},
{{4268838798361663312ull, 360522782749025ull}},
{{3415071038689330649ull, 288418226199220ull}},
{{5464113661902929039ull, 461469161918752ull}},
{{15439337373748074201ull, 369175329535001ull}},
{{8662121084256549038ull, 295340263628001ull}},
{{6480696105326657814ull, 472544421804802ull}},
{{16252603328487057221ull, 378035537443841ull}},
{{9312733848047735453ull, 302428429955073ull}},
{{11211025342134466402ull, 483885487928117ull}},
{{1590122644223752475ull, 387108390342494ull}},
{{4961446930120912303ull, 309686712273995ull}},
{{7938315088193459686ull, 495498739638392ull}},
{{17418698514780498718ull, 396398991710713ull}},
{{2866912367598668005ull, 317119193368571ull}},
{{15655106232383599778ull, 507390709389713ull}},
{{1456038541681148852ull, 405912567511771ull}},
{{15922226092312560375ull, 324730054009416ull}},
{{18096864118216275953ull, 519568086415066ull}},
{{10788142479831110439ull, 415654469132053ull}},
{{16009211613348708998ull, 332523575305642ull}},
{{10857343322390293104ull, 532037720489028ull}},
{{16064572287396055129ull, 425630176391222ull}},
{{5472960200433023457ull, 340504141112978ull}},
{{5067387505950927208ull, 544806625780765ull}},
{{4053910004760741767ull, 435845300624612ull}},
{{14311174448034324383ull, 348676240499689ull}},
{{11829832672629188043ull, 557881984799503ull}},
{{16842563767587171081ull, 446305587839602ull}},
{{6095353384585916218ull, 357044470271682ull}},
{{15944329151894463944ull, 285635576217345ull}},
{{7064182569321590695ull, 457016921947753ull}},
{{13030043684941093202ull, 365613537558202ull}},
{{3045337318469053915ull, 292490830046562ull}},
{{8561888524292396588ull, 467985328074499ull}},
{{10538859634175827593ull, 374388262459599ull}},
{{12120436522082572398ull, 299510609967679ull}},
{{8324651991106384867ull, 479216975948287ull}},
{{17727768037110838863ull, 383373580758629ull}},
{{17871563244430581414ull, 306698864606903ull}},
{{6458408302637468323ull, 490718183371046ull}},
{{1477377827368064335ull, 392574546696837ull}},
{{12249948706120182437ull, 314059637357469ull}},
{{8531871485566560931ull, 502495419771951ull}},
{{3136148373711338421ull, 401996335817561ull}},
{{17266313957936712030ull, 321597068654048ull}},
{{5490009444247277309ull, 514555309846478ull}},
{{11770705184881642493ull, 411644247877182ull}},
{{2037866518421493348ull, 329315398301746ull}},
{{14328632873700120327ull, 526904637282793ull}},
{{394859854734365292ull, 421523709826235ull}},
{{315887883787492233ull, 337218967860988ull}},
{{15262815873027628866ull, 539550348577580ull}},
{{12210252698422103093ull, 431640278862064ull}},
{{13457550973479592797ull, 345312223089651ull}},
{{14153383928083527830ull, 552499556943442ull}},
{{3944009512983001617ull, 441999645554754ull}},
{{6844556425128311617ull, 353599716443803ull}},
{{12854342769586469940ull, 282879773155042ull}},
{{5809553172370710611ull, 452607637048068ull}},
{{12026340167380389135ull, 362086109638454ull}},
{{13310420948646221631ull, 289668887710763ull}},
{{17607324703092044287ull, 463470220337221ull}},
{{10396510947731725107ull, 370776176269777ull}},
{{938511128701559439ull, 296620941015822ull}},
{{5190966620664405426ull, 474593505625315ull}},
{{4152773296531524340ull, 379674804500252ull}},
{{14390265081450950442ull, 303739843600201ull}},
{{15645726500837700061ull, 485983749760322ull}},
{{5137883571186339402ull, 388786999808258ull}},
{{11489004486432892168ull, 311029599846606ull}},
{{11003709548808806823ull, 497647359754570ull}},
{{8802967639047045458ull, 398117887803656ull}},
{{3353025296495726043ull, 318494310242925ull}},
{{5364840474393161669ull, 509590896388680ull}},
{{4291872379514529335ull, 407672717110944ull}},
{{7122846718353533791ull, 326138173688755ull}},
{{11396554749365654066ull, 521821077902008ull}},
{{16495941428976343899ull, 417456862321606ull}},
{{9507404328439164796ull, 333965489857285ull}},
{{15211846925502663674ull, 534344783771656ull}},
{{8480128725660220616ull, 427475827017325ull}},
{{6784102980528176493ull, 341980661613860ull}},
{{10854564768845082389ull, 547169058582176ull}},
{{4994303000334155588ull, 437735246865741ull}},
{{306093585525414147ull, 350188197492593ull}},
{{15247144995808303928ull, 560301115988148ull}},
{{1129669552420912173ull, 448240892790519ull}},
{{4593084456678640061ull, 358592714232415ull}},
{{3674467565342912049ull, 286874171385932ull}},
{{9568496919290569602ull, 458998674217491ull}},
{{3965448720690545358ull, 367198939373993ull}},
{{10551056606036256933ull, 293759151499194ull}},
{{5813644125432280123ull, 470014642398711ull}},
{{961566485603913775ull, 376011713918969ull}},
{{4458602003225041343ull, 300809371135175ull}},
{{7133763205160066150ull, 481294993816280ull}},
{{5707010564128052920ull, 385035995053024ull}},
{{8254957266044352659ull, 308028796042419ull}},
{{2139885181445233285ull, 492846073667871ull}},
{{16469303404123827921ull, 394276858934296ull}},
{{9486093908557152013ull, 315421487147437ull}},
{{420354994723801929ull, 504674379435900ull}},
{{336283995779041543ull, 403739503548720ull}},
{{269027196623233234ull, 322991602838976ull}},
{{11498489958822904144ull, 516786564542361ull}},
{{5509443152316412992ull, 413429251633889ull}},
{{8096903336595040717ull, 330743401307111ull}},
{{5576347709068244501ull, 529189442091378ull}},
{{11839775796738416247ull, 423351553673102ull}},
{{2093123007906912351ull, 338681242938482ull}},
{{7038345627392970085ull, 541889988701571ull}},
{{1941327687172465745ull, 433511990961257ull}},
{{12621108593963703565ull, 346809592769005ull}},
{{1747029676632374089ull, 554895348430409ull}},
{{5086972556047809594ull, 443916278744327ull}},
{{15137624489063978645ull, 355133022995461ull}},
{{8420750776509272593ull, 284106418396369ull}},
{{2405154798189105179ull, 454570269434191ull}},
{{16681519097518925436ull, 363656215547352ull}},
{{5966517648531319702ull, 290924972437882ull}},
{{13235777052392021847ull, 465479955900611ull}},
{{6899272827171707154ull, 372383964720489ull}},
{{9208767076479276047ull, 297907171776391ull}},
{{7355329692883021028ull, 476651474842226ull}},
{{2194914939564506499ull, 381321179873781ull}},
{{16513327210619246492ull, 305056943899024ull}},
{{15353277092765063418ull, 488091110238439ull}},
{{15971970488953961058ull, 390472888190751ull}},
{{9088227576421258523ull, 312378310552601ull}},
{{7162466492790192990ull, 499805296884162ull}},
{{16798019638457885362ull, 399844237507329ull}},
{{17127764525508218613ull, 319875390005863ull}},
{{5268330352361687841ull, 511800624009382ull}},
{{15282710726115081243ull, 409440499207505ull}},
{{12226168580892064994ull, 327552399366004ull}},
{{8493823285201573021ull, 524083838985607ull}},
{{17863105072386989386ull, 419267071188485ull}},
{{14290484057909591509ull, 335413656950788ull}},
{{728681604203884475ull, 536661851121262ull}},
{{11650991727588838550ull, 429329480897009ull}},
{{13010142196812981163ull, 343463584717607ull}},
{{6058832255933128568ull, 549541735548172ull}},
{{15915112248972233824ull, 439633388438537ull}},
{{5353392169693966413ull, 351706710750830ull}},
{{8565427471510346261ull, 562730737201328ull}},
{{14231039606692097655ull, 450184589761062ull}},
{{4006134055869857477ull, 360147671808850ull}},
{{3204907244695885982ull, 288118137447080ull}},
{{5127851591513417571ull, 460989019915328ull}},
{{11480978902694554703ull, 368791215932262ull}},
{{1806085492671823116ull, 295032972745810ull}},
{{2889736788274916986ull, 472052756393296ull}},
{{17069184689587574882ull, 377642205114636ull}},
{{9965998936928149582ull, 302113764091709ull}},
{{4877551854859308362ull, 483382022546735ull}},
{{3902041483887446689ull, 386705618037388ull}},
{{10500330816593777998ull, 309364494429910ull}},
{{16800529306550044797ull, 494983191087856ull}},
{{9751074630498125514ull, 395986552870285ull}},
{{7800859704398500411ull, 316789242296228ull}},
{{8792026712295690335ull, 506862787673965ull}},
{{7033621369836552268ull, 405490230139172ull}},
{{16694943540094972784ull, 324392184111337ull}},
{{11954514405184315162ull, 519027494578140ull}},
{{9563611524147452129ull, 415221995662512ull}},
{{272191589834141057ull, 332177596530010ull}},
{{435506543734625691ull, 531484154448016ull}},
{{15105800493955341846ull, 425187323558412ull}},
{{4705942765680452830ull, 340149858846730ull}},
{{7529508425088724528ull, 544239774154768ull}},
{{13402304369554800269ull, 435391819323814ull}},
{{14411192310385750538ull, 348313455459051ull}},
{{15679210067133380215ull, 557301528734482ull}},
{{5164670424222883526ull, 445841222987586ull}},
{{442387524636396497ull, 356672978390069ull}},
{{4043258834451027521ull, 285338382712055ull}},
{{6469214135121644034ull, 456541412339288ull}},
{{12554068937581135873ull, 365233129871430ull}},
{{10043255150064908699ull, 292186503897144ull}},
{{5001161795878122948ull, 467498406235431ull}},
{{311580621960588035ull, 373998724988345ull}},
{{249264497568470428ull, 299198979990676ull}},
{{11466869640335283655ull, 478718367985081ull}},
{{5484146897526316601ull, 382974694388065ull}},
{{4387317518021053280ull, 306379755510452ull}},
{{10709056843575595572ull, 490207608816723ull}},
{{15945943104344297104ull, 392166087053378ull}},
{{1688708039249706714ull, 313732869642703ull}},
{{17459328121767172035ull, 501972591428324ull}},
{{17656811312155647951ull, 401578073142659ull}},
{{17814797864466428684ull, 321262458514127ull}},
{{13746281324178644602ull, 514019933622604ull}},
{{14686373874084826004ull, 411215946898083ull}},
{{681052655042129834ull, 328972757518467ull}},
{{4779033062809318058ull, 526356412029547ull}},
{{14891272894473185416ull, 421085129623637ull}},
{{4534320686094727686ull, 336868103698910ull}},
{{7254913097751564298ull, 538988965918256ull}},
{{2114581663459341115ull, 431191172734605ull}},
{{1691665330767472892ull, 344952938187684ull}},
{{10085362158711777274ull, 551924701100294ull}},
{{11757638541711332142ull, 441539760880235ull}},
{{9406110833369065714ull, 353231808704188ull}},
{{14903586296179073217ull, 282585446963350ull}},
{{5398994000176965532ull, 452136715141361ull}},
{{629846385399662102ull, 361709372113089ull}},
{{4193225923061640005ull, 289367497690471ull}},
{{17777207921124354978ull, 462987996304753ull}},
{{3153719892673753012ull, 370390397043803ull}},
{{9901673543622823056ull, 296312317635042ull}},
{{1085282410828875597ull, 474099708216068ull}},
{{8246923558146921124ull, 379279766572854ull}},
{{10286887661259447222ull, 303423813258283ull}},
{{12769671443273205233ull, 485478101213253ull}},
{{17594434784102384833ull, 388382480970602ull}},
{{6696850197798087220ull, 310705984776482ull}},
{{14404309131218849875ull, 497129575642371ull}},
{{7834098490233169577ull, 397703660513897ull}},
{{17335325236412266631ull, 318162928411117ull}},
{{12979125119291985317ull, 509060685457788ull}},
{{17761997724917408900ull, 407248548366230ull}},
{{14209598179933927120ull, 325798838692984ull}},
{{11667310643668552422ull, 521278141908775ull}},
{{9333848514934841938ull, 417022513527020ull}},
{{7467078811947873550ull, 333618010821616ull}},
{{4568628469632777034ull, 533788817314586ull}},
{{18412298034673862920ull, 427031053851668ull}},
{{3661791983513359366ull, 341624843081335ull}},
{{5858867173621374986ull, 546599748930136ull}},
{{997744924155189666ull, 437279799144109ull}},
{{4487544754066062056ull, 349823839315287ull}},
{{10869420421247609613ull, 559718142904459ull}},
{{12384885151739998013ull, 447774514323567ull}},
{{2529210491908177764ull, 358219611458854ull}},
{{5712717208268452534ull, 286575689167083ull}},
{{5450998718487613732ull, 458521102667333ull}},
{{11739496604273911632ull, 366816882133866ull}},
{{5702248468677218982ull, 293453505707093ull}},
{{5434248735141640049ull, 469525609131349ull}},
{{8036747802855222362ull, 375620487305079ull}},
{{10118747057026088213ull, 300496389844063ull}},
{{12500646476499830818ull, 480794223750501ull}},
{{6311168366457954331ull, 384635379000401ull}},
{{1359585878424453141ull, 307708303200321ull}},
{{13243383849704855996ull, 492333285120513ull}},
{{17973404709247705443ull, 393866628096410ull}},
{{14378723767398164354ull, 315093302477128ull}},
{{869865139385601028ull, 504149283963406ull}},
{{15453287370476122115ull, 403319427170724ull}},
{{16051978711122808015ull, 322655541736579ull}},
{{14615119493570761855ull, 516248866778527ull}},
{{4313397965372788838ull, 412999093422822ull}},
{{14518764816523962040ull, 330399274738257ull}},
{{8472628447470697971ull, 528638839581212ull}},
{{17846149202202289346ull, 422911071664969ull}},
{{17966268176503741800ull, 338328857331975ull}},
{{10299285008696435264ull, 541326171731161ull}},
{{4550079192215237888ull, 433060937384929ull}},
{{7329412168514100634ull, 346448749907943ull}},
{{8037710654880650691ull, 554317999852709ull}},
{{10119517338646430876ull, 443454399882167ull}},
{{716916241433324054ull, 354763519905734ull}},
{{4262881807888569566ull, 283810815924587ull}},
{{10509959707363621630ull, 454097305479339ull}},
{{12097316580632807627ull, 363277844383471ull}},
{{5988504449764335778ull, 290622275506777ull}},
{{13270955934364847569ull, 464995640810843ull}},
{{17995462376975698701ull, 371996512648674ull}},
{{18085718716322469284ull, 297597210118939ull}},
{{17869103501890219885ull, 476155536190303ull}},
{{3227236357286444938ull, 380924428952243ull}},
{{9960486715312976597ull, 304739543161794ull}},
{{4868732300275031586ull, 487583269058871ull}},
{{205637025478114945ull, 390066615247097ull}},
{{11232556064608222926ull, 312053292197677ull}},
{{3214694444405515389ull, 499285267516284ull}},
{{6261104370266322634ull, 399428214013027ull}},
{{16076929940438789077ull, 319542571210421ull}},
{{18344390275218241877ull, 511268113936674ull}},
{{18364861034916503825ull, 409014491149339ull}},
{{18381237642675113383ull, 327211592919471ull}},
{{3584538525086809150ull, 523538548671155ull}},
{{2867630820069447320ull, 418830838936924ull}},
{{5983453470797468179ull, 335064671149539ull}},
{{16952223182759769734ull, 536103473839262ull}},
{{6183080916723995140ull, 428882779071410ull}},
{{4946464733379196112ull, 343106223257128ull}},
{{4224994758664803457ull, 548969957211405ull}},
{{3379995806931842765ull, 439175965769124ull}},
{{6393345460287384535ull, 351340772615299ull}},
{{17608050365943635903ull, 562145236184478ull}},
{{3018393848529177753ull, 449716188947583ull}},
{{9793412708307162849ull, 359772951158066ull}},
{{4145381351903819956ull, 287818360926453ull}},
{{2943261348304201606ull, 460509377482325ull}},
{{2354609078643361285ull, 368407501985860ull}},
{{1883687262914689028ull, 294726001588688ull}},
{{17771294879631143737ull, 471561602541900ull}},
{{14217035903704914990ull, 377249282033520ull}},
{{11373628722963931992ull, 301799425626816ull}},
{{10819108327258470541ull, 482879081002906ull}},
{{4965937847064866109ull, 386303264802325ull}},
{{3972750277651892887ull, 309042611841860ull}},
{{6356400444243028620ull, 494468178946976ull}},
{{1395771540652512573ull, 395574543157581ull}},
{{15874012491489651351ull, 316459634526064ull}},
{{14330373542157711192ull, 506335415241703ull}},
{{396252389500437984ull, 405068332193363ull}},
{{7695699541084171033ull, 324054665754690ull}},
{{12313119265734673654ull, 518487465207504ull}},
{{13539844227329649246ull, 414789972166003ull}},
{{18210573011347540043ull, 331831977732802ull}},
{{14379521559188422777ull, 530931164372484ull}},
{{15192966062092648544ull, 424744931497987ull}},
{{4775675220190298189ull, 339795945198390ull}},
{{7641080352304477103ull, 543673512317424ull}},
{{9802213096585492005ull, 434938809853939ull}},
{{11531119292010303927ull, 347951047883151ull}},
{{11071093237732665638ull, 556721676613042ull}},
{{1478176960702311864ull, 445377341290434ull}},
{{4871890383303759814ull, 356301873032347ull}},
{{14965558750868738821ull, 285041498425877ull}},
{{9187498742422340821ull, 456066397481404ull}},
{{11039347808679782980ull, 364853117985123ull}},
{{16210175876427647030ull, 291882494388098ull}},
{{3800188513832773309ull, 467011991020958ull}},
{{10418848440550039293ull, 373609592816766ull}},
{{4645729937698121111ull, 298887674253413ull}},
{{3743819085575083455ull, 478220278805461ull}},
{{17752450527427708057ull, 382576223044368ull}},
{{3133913977716435476ull, 306060978435495ull}},
{{5014262364346296762ull, 489697565496792ull}},
{{15079456335702768379ull, 391758052397433ull}},
{{995518624336483733ull, 313406441917947ull}},
{{5282178613680284297ull, 501450307068715ull}},
{{4225742890944227438ull, 401160245654972ull}},
{{14448640756981112920ull, 320928196523977ull}},
{{8360429952202139379ull, 513485114438364ull}},
{{10377692776503621826ull, 410788091550691ull}},
{{4612805406460987138ull, 328630473240553ull}},
{{3691139835595669097ull, 525808757184885ull}},
{{2952911868476535278ull, 420647005747908ull}},
{{9741027124265048868ull, 336517604598326ull}},
{{8206945769340257543ull, 538428167357322ull}},
{{17633603059697937004ull, 430742533885857ull}},
{{6728184818274528957ull, 344594027108686ull}},
{{3386398079755425685ull, 551350443373898ull}},
{{10087816093288161194ull, 441080354699118ull}},
{{15448950504114349602ull, 352864283759294ull}},
{{16048509218033390004ull, 282291427007435ull}},
{{7230870675143872391ull, 451666283211897ull}},
{{16852742984340828883ull, 361333026569517ull}},
{{6103496757988842460ull, 289066421255614ull}},
{{17144292442265968582ull, 462506274008982ull}},
{{6336736324328954219ull, 370005019207186ull}},
{{1380040244721253052ull, 296004015365749ull}},
{{9586762021037825530ull, 473606424585198ull}},
{{15048107246314081070ull, 378885139668158ull}},
{{970439352825533886ull, 303108111734527ull}},
{{5242051779262764542ull, 484972978775243ull}},
{{11572339052894032280ull, 387978383020194ull}},
{{12947220057057136147ull, 310382706416155ull}},
{{2268808017581866219ull, 496612330265849ull}},
{{5504395228807403299ull, 397289864212679ull}},
{{8092864997787832962ull, 317831891370143ull}},
{{9259235181718622416ull, 508531026192229ull}},
{{11096736960116808256ull, 406824820953783ull}},
{{16256087197577267251ull, 325459856763026ull}},
{{184297812930255340ull, 520735770820843ull}},
{{7526135879828024918ull, 416588616656674ull}},
{{9710257518604330258ull, 333270893325339ull}},
{{4468365585541197443ull, 533233429320543ull}},
{{10953390097916778601ull, 426586743456434ull}},
{{12452060893075333204ull, 341269394765147ull}},
{{5165902169952891833ull, 546031031624236ull}},
{{443372921220403143ull, 436824825299389ull}},
{{4044047151718232838ull, 349459860239511ull}},
{{17538521886974903510ull, 559135776383217ull}},
{{6652119880096102162ull, 447308621106574ull}},
{{9011044718818792053ull, 357846896885259ull}},
{{10898184589796943965ull, 286277517508207ull}},
{{2679700084707469052ull, 458044028013132ull}},
{{13211806511991706211ull, 366435222410505ull}},
{{10569445209593364969ull, 293148177928404ull}},
{{5843065891123652980ull, 469037084685447ull}},
{{15742499157124653354ull, 375229667748357ull}},
{{5215301696215902037ull, 300183734198686ull}},
{{965785084461622612ull, 480293974717898ull}},
{{8151325697053118736ull, 384235179774318ull}},
{{13899758187126315635ull, 307388143819454ull}},
{{11171566655176374047ull, 491821030111127ull}},
{{1558555694657278591ull, 393456824088902ull}},
{{12314890999951553842ull, 314765459271121ull}},
{{12325127970438665502ull, 503624734833794ull}},
{{13549451191092842725ull, 402899787867035ull}},
{{10839560952874274180ull, 322319830293628ull}},
{{13653948709856928364ull, 515711728469805ull}},
{{10923158967885542691ull, 412569382775844ull}},
{{12427875989050344476ull, 330055506220675ull}},
{{1437857508770999546ull, 528088809953081ull}},
{{15907681265984440930ull, 422471047962464ull}},
{{16415493827529463067ull, 337976838369971ull}},
{{439348420853768645ull, 540762941391955ull}},
{{351478736683014916ull, 432610353113564ull}},
{{3970531804088322256ull, 346088282490851ull}},
{{17420897330767046579ull, 553741251985361ull}},
{{10247369049871726940ull, 442993001588289ull}},
{{11887244054639291875ull, 354394401270631ull}},
{{5820446428969523177ull, 283515521016505ull}},
{{9312714286351237083ull, 453624833626408ull}},
{{14828869058564810313ull, 362899866901126ull}},
{{8173746432109937927ull, 290319893520901ull}},
{{5699296661892080037ull, 464511829633442ull}},
{{15627483773739394999ull, 371609463706753ull}},
{{1433940574765785030ull, 297287570965403ull}},
{{17051700178592897340ull, 475660113544644ull}},
{{17330708957616228195ull, 380528090835715ull}},
{{13864567166092982556ull, 304422472668572ull}},
{{7425912206781130798ull, 487075956269716ull}},
{{2251380950682994315ull, 389660765015773ull}},
{{9179802390030216098ull, 311728612012618ull}},
{{10998335009306435434ull, 498765779220189ull}},
{{12488016822187058670ull, 399012623376151ull}},
{{6301064643007736613ull, 319210098700921ull}},
{{2703005799328557935ull, 510736157921474ull}},
{{5851753454204756671ull, 408588926337179ull}},
{{8370751578105715660ull, 326871141069743ull}},
{{9703853710227234733ull, 522993825711589ull}},
{{11452431782923698109ull, 418395060569271ull}},
{{5472596611597048164ull, 334716048455417ull}},
{{12445503393297187386ull, 535545677528667ull}},
{{2577705085153929262ull, 428436542022934ull}},
{{5751512882865053733ull, 342749233618347ull}},
{{12891769427325996296ull, 548398773789355ull}},
{{10313415541860797037ull, 438719019031484ull}},
{{11940081248230547953ull, 350975215225187ull}},
{{4346734738201235432ull, 561560344360300ull}},
{{3477387790560988345ull, 449248275488240ull}},
{{2781910232448790676ull, 359398620390592ull}},
{{13293574630184763510ull, 287518896312473ull}},
{{17580370593553711294ull, 460030234099957ull}},
{{6685598845359148388ull, 368024187279966ull}},
{{1659130261545408387ull, 294419349823973ull}},
{{17412003677440294713ull, 471070959718356ull}},
{{10240254127210325447ull, 376856767774685ull}},
{{8192203301768260358ull, 301485414219748ull}},
{{9418176468087306249ull, 482376662751597ull}},
{{155843544986024353ull, 385901330201278ull}},
{{7503372465472640129ull, 308721064161022ull}},
{{15694744759498134529ull, 493953702657635ull}},
{{12555795807598507623ull, 395162962126108ull}},
{{17423334275562626745ull, 316130369700886ull}},
{{2051893137706830530ull, 505808591521419ull}},
{{5330863324907374747ull, 404646873217135ull}},
{{4264690659925899798ull, 323717498573708ull}},
{{3134156241139529353ull, 517947997717933ull}},
{{9886022622395444129ull, 414358398174346ull}},
{{4219469283174444980ull, 331486718539477ull}},
{{10440499667821022291ull, 530378749663163ull}},
{{15731097363740638479ull, 424302999730530ull}},
{{12584877890992510783ull, 339442399784424ull}},
{{9067758181362286284ull, 543107839655079ull}},
{{10943555359831739350ull, 434486271724063ull}},
{{16133541917349212126ull, 347589017379250ull}},
{{7366922994049187787ull, 556142427806801ull}},
{{2204189580497439906ull, 444913942245441ull}},
{{16520746923365593218ull, 355931153796352ull}},
{{5837899909208653928ull, 284744923037082ull}},
{{13029988669475756608ull, 455591876859331ull}},
{{6734642120838694963ull, 364473501487465ull}},
{{5387713696670955970ull, 291578801189972ull}},
{{12309690729415439876ull, 466526081903955ull}},
{{9847752583532351900ull, 373220865523164ull}},
{{11567550881567791843ull, 298576692418531ull}},
{{11129383781024646303ull, 477722707869650ull}},
{{8903507024819717043ull, 382178166295720ull}},
{{7122805619855773634ull, 305742533036576ull}},
{{4017791362285417168ull, 489188052858522ull}},
{{14282279534054064704ull, 391350442286817ull}},
{{4047125997759431117ull, 313080353829454ull}},
{{13854099225898910434ull, 500928566127126ull}},
{{7393930565977218024ull, 400742852901701ull}},
{{2225795638039864096ull, 320594282321361ull}},
{{14629319465089513523ull, 512950851714177ull}},
{{4324757942587790172ull, 410360681371342ull}},
{{14527852798295963107ull, 328288545097073ull}},
{{1108471588822079032ull, 525261672155318ull}},
{{8265474900541483872ull, 420209337724254ull}},
{{10301728735175097421ull, 336167470179403ull}},
{{12793417161538245550ull, 537867952287045ull}},
{{10234733729230596440ull, 430294361829636ull}},
{{4498438168642566829ull, 344235489463709ull}},
{{14576198699311927573ull, 550776783141934ull}},
{{15350307774191452381ull, 440621426513547ull}},
{{4901548589869341258ull, 352497141210838ull}},
{{11299936501379293653ull, 281997712968670ull}},
{{18079898402206869845ull, 451196340749872ull}},
{{7085221092281675230ull, 360957072599898ull}},
{{13046874503309160830ull, 288765658079918ull}},
{{17185650390552747005ull, 462025052927869ull}},
{{17437869127184107927ull, 369620042342295ull}},
{{13950295301747286342ull, 295696033873836ull}},
{{14941774853311837501ull, 473113654198138ull}},
{{885373438423739031ull, 378490923358511ull}},
{{15465694009706632517ull, 302792738686808ull}},
{{2609017527079150089ull, 484468381898894ull}},
{{5776562836405230394ull, 387574705519115ull}},
{{4621250269124184315ull, 310059764415292ull}},
{{11083349245340605228ull, 496095623064467ull}},
{{1487981766788663536ull, 396876498451574ull}},
{{4879734228172841152ull, 317501198761259ull}},
{{15186272394560366489ull, 508001918018014ull}},
{{15838366730390203515ull, 406401534414411ull}},
{{8981344569570252488ull, 325121227531529ull}},
{{3302104867086673012ull, 520193964050447ull}},
{{13709730337895069379ull, 416155171240357ull}},
{{3589086640832234857ull, 332924136992286ull}},
{{16810585069557306741ull, 532678619187657ull}},
{{6069770426162024746ull, 426142895350126ull}},
{{1166467526187709474ull, 340914316280101ull}},
{{12934394486126066128ull, 545462906048161ull}},
{{6658166774158942579ull, 436370324838529ull}},
{{9015882234069064386ull, 349096259870823ull}},
{{10736062759768592695ull, 558554015793317ull}},
{{1210152578331053510ull, 446843212634654ull}},
{{4657470877406753131ull, 357474570107723ull}},
{{11104674331409223151ull, 285979656086178ull}},
{{14078130115512846719ull, 457567449737885ull}},
{{11262504092410277375ull, 366053959790308ull}},
{{16388700903412042546ull, 292843167832246ull}},
{{396479742265895812ull, 468549068531595ull}},
{{317183793812716649ull, 374839254825276ull}},
{{15011142294017814612ull, 299871403860220ull}},
{{5571083596718951764ull, 479794246176353ull}},
{{11835564506858982057ull, 383835396941082ull}},
{{2089753976003364999ull, 307068317552866ull}},
{{14411652805831114969ull, 491309308084585ull}},
{{11529322244664891975ull, 393047446467668ull}},
{{16602155425215734226ull, 314437957174134ull}},
{{15495402236119443793ull, 503100731478615ull}},
{{12396321788895555034ull, 402480585182892ull}},
{{2538359801632623381ull, 321984468146314ull}},
{{11440073312096018056ull, 515175149034102ull}},
{{1773361020192993798ull, 412140119227282ull}},
{{12486735260380126008ull, 329712095381825ull}},
{{1532032342898649997ull, 527539352610921ull}},
{{15983021133286561290ull, 422031482088736ull}},
{{9097068091887338709ull, 337625185670989ull}},
{{3487262502794010965ull, 540200297073583ull}},
{{10168507631719029418ull, 432160237658866ull}},
{{4445457290633313211ull, 345728190127093ull}},
{{3423382850271390815ull, 553165104203349ull}},
{{6428055094959022975ull, 442532083362679ull}},
{{8831792890709128703ull, 354025666690143ull}},
{{14444131942051123609ull, 283220533352114ull}},
{{12042564663056066805ull, 453152853363383ull}},
{{17012749359928674090ull, 362522282690706ull}},
{{9920850673201028949ull, 290017826152565ull}},
{{15873361077121646319ull, 464028521844104ull}},
{{16388037676439227378ull, 371222817475283ull}},
{{2042383696925650933ull, 296978253980227ull}},
{{6957162729822951816ull, 475165206368363ull}},
{{12944427813342182099ull, 380132165094690ull}},
{{10355542250673745679ull, 304105732075752ull}},
{{1811472342110351794ull, 486569171321204ull}},
{{5138526688430191758ull, 389255337056963ull}},
{{11489518980227974053ull, 311404269645570ull}},
{{18383230368364758485ull, 498246831432912ull}},
{{7327886665207986142ull, 398597465146330ull}},
{{5862309332166388913ull, 318877972117064ull}},
{{16758392560950042908ull, 510204755387302ull}},
{{6028016419276213680ull, 408163804309842ull}},
{{15890459579646701913ull, 326531043447873ull}},
{{3288642438983261122ull, 522449669516598ull}},
{{10009611580670429544ull, 417959735613278ull}},
{{15386386894020164282ull, 334367788490622ull}},
{{9860823771464621558ull, 534988461584996ull}},
{{4199310202429786923ull, 427990769267997ull}},
{{14427494606169560508ull, 342392615414397ull}},
{{8326596110903655521ull, 547828184663036ull}},
{{2971928073981014093ull, 438262547730429ull}},
{{6066891273926721598ull, 350610038184343ull}},
{{6017677223540844233ull, 560976061094949ull}},
{{8503490593574585710ull, 448780848875959ull}},
{{10492141289601578891ull, 359024679100767ull}},
{{1015015402197442466ull, 287219743280614ull}},
{{9002722272999728593ull, 459551589248982ull}},
{{18270224262625513844ull, 367641271399185ull}},
{{14616179410100411075ull, 294113017119348ull}},
{{1249794167709195781ull, 470580827390958ull}},
{{8378532963651177271ull, 376464661912766ull}},
{{3013477556179031493ull, 301171729530213ull}},
{{1132215275144540067ull, 481874767248341ull}},
{{15663167479083273346ull, 385499813798672ull}},
{{5151836353782798030ull, 308399851038938ull}},
{{4553589351310566526ull, 493439761662301ull}},
{{18400266740016094513ull, 394751809329840ull}},
{{14720213392012875610ull, 315801447463872ull}},
{{8794946168252959684ull, 505282315942196ull}},
{{3346608119860457424ull, 404225852753757ull}},
{{13745332940114096909ull, 323380682203005ull}},
{{3545788630473003438ull, 517409091524809ull}},
{{6525979719120313074ull, 413927273219847ull}},
{{16288830219521981428ull, 331141818575877ull}},
{{11304733092267528993ull, 529826909721404ull}},
{{12733135288555933517ull, 423861527777123ull}},
{{17565205860328567460ull, 339089222221698ull}},
{{5968236488074245998ull, 542542755554718ull}},
{{12153286819943217444ull, 434034204443774ull}},
{{13411978270696484279ull, 347227363555019ull}},
{{10391118788888643876ull, 555563781688031ull}},
{{4623546216369004778ull, 444451025350425ull}},
{{3698836973095203822ull, 355560820280340ull}},
{{2959069578476163058ull, 284448656224272ull}},
{{8423860140303771216ull, 455117849958835ull}},
{{6739088112243016972ull, 364094279967068ull}},
{{12769968119278234224ull, 291275423973654ull}},
{{9363902546619443790ull, 466040678357847ull}},
{{112424407811734385ull, 372832542686278ull}},
{{7468637155733208154ull, 298266034149022ull}},
{{15639168263915043371ull, 477225654638435ull}},
{{12511334611132034696ull, 381780523710748ull}},
{{17387765318389448403ull, 305424418968598ull}},
{{5684331620971655507ull, 488679070349758ull}},
{{11926162926261145052ull, 390943256279806ull}},
{{5851581526267005718ull, 312754605023845ull}},
{{9362530442027209149ull, 500407368038152ull}},
{{111326724137946673ull, 400325894430522ull}},
{{11157107823536088308ull, 320260715544417ull}},
{{3093977258690100000ull, 512417144871068ull}},
{{9853879436435900646ull, 409933715896854ull}},
{{11572452363890630840ull, 327946972717483ull}},
{{14826574967483099021ull, 524715156347973ull}},
{{793213529760748247ull, 419772125078379ull}},
{{4323919638550508921ull, 335817700062703ull}},
{{3228922606938903950ull, 537308320100325ull}},
{{2583138085551123160ull, 429846656080260ull}},
{{2066510468440898528ull, 343877324864208ull}},
{{18063812008473078938ull, 550203719782732ull}},
{{7072351977294642504ull, 440162975826186ull}},
{{1968532767093803680ull, 352130380660949ull}},
{{5264175028416953267ull, 281704304528759ull}},
{{15801377674950945874ull, 450726887246014ull}},
{{16330450954702667022ull, 360581509796811ull}},
{{9375011949020223294ull, 288465207837449ull}},
{{3931972674206626302ull, 461544332539919ull}},
{{6834926954107211365ull, 369235466031935ull}},
{{5467941563285769092ull, 295388372825548ull}},
{{5059357686515320224ull, 472621396520877ull}},
{{15115532593437987148ull, 378097117216701ull}},
{{8403077260008479395ull, 302477693773361ull}},
{{6066225986529746386ull, 483964310037378ull}},
{{12231678418707617755ull, 387171448029902ull}},
{{2406645105482273558ull, 309737158423922ull}},
{{7539980983513548016ull, 495579453478275ull}},
{{6031984786810838413ull, 396463562782620ull}},
{{4825587829448670730ull, 317170850226096ull}},
{{342242897634052522ull, 507473360361754ull}},
{{3963143132849152341ull, 405978688289403ull}},
{{10549212135763142519ull, 324782950631522ull}},
{{2121344158253386738ull, 519652721010436ull}},
{{16454470585570350683ull, 415722176808348ull}},
{{2095530024230549577ull, 332577741446679ull}},
{{10731545668252699969ull, 532124386314686ull}},
{{4895887719860249652ull, 425699509051749ull}},
{{7606058990630110045ull, 340559607241399ull}},
{{1101647940782445102ull, 544895371586239ull}},
{{4570667167367866405ull, 435916297268991ull}},
{{18413928992861934417ull, 348733037815192ull}},
{{14704891129611453774ull, 557972860504308ull}},
{{695866459463432050ull, 446378288403447ull}},
{{11624739611796476609ull, 357102630722757ull}},
{{1921094059953360641ull, 285682104578206ull}},
{{14141796940151107995ull, 457091367325129ull}},
{{15002786366862796719ull, 365673093860103ull}},
{{934182649264506406ull, 292538475088083ull}},
{{16252087497790851542ull, 468061560140932ull}},
{{5622972368748860587ull, 374449248112746ull}},
{{809029080257178147ull, 299559398490197ull}},
{{4983795343153395358ull, 479295037584315ull}},
{{3987036274522716286ull, 383436030067452ull}},
{{14257675463843903999ull, 306748824053961ull}},
{{15433583112666425752ull, 490798118486338ull}},
{{1278820045907409632ull, 392638494789071ull}},
{{15780451295693568998ull, 314110795831256ull}},
{{17870024443625889751ull, 502577273330010ull}},
{{14296019554900711801ull, 402061818664008ull}},
{{368769199694838471ull, 321649454931207ull}},
{{4279379534253651877ull, 514639127889931ull}},
{{18180898886370562794ull, 411711302311944ull}},
{{18234067923838360558ull, 329369041849555ull}},
{{10727764604431825278ull, 526990466959289ull}},
{{12271560498287370545ull, 421592373567431ull}},
{{6127899583887986113ull, 337273898853945ull}},
{{9804639334220777781ull, 539638238166312ull}},
{{465013837892801578ull, 431710590533050ull}},
{{372011070314241262ull, 345368472426440ull}},
{{595217712502786020ull, 552589555882304ull}},
{{4165522984744139139ull, 442071644705843ull}},
{{10711116017279131958ull, 353657315764674ull}},
{{12258241628565215889ull, 282925852611739ull}},
{{8545140161478614454ull, 452681364178783ull}},
{{14214809758666712209ull, 362145091343026ull}},
{{7682498992191459444ull, 289716073074421ull}},
{{4913300758022514464ull, 463545716919074ull}},
{{7619989421159921895ull, 370836573535259ull}},
{{9785340351669847839ull, 296669258828207ull}},
{{899149303704115249ull, 474670814125132ull}},
{{11787365887189023169ull, 379736651300105ull}},
{{9429892709751218535ull, 303789321040084ull}},
{{4019781891376218687ull, 486062913664135ull}},
{{3215825513100974949ull, 388850330931308ull}},
{{9951358039964600606ull, 311080264745046ull}},
{{8543475234459540323ull, 497728423592074ull}},
{{10524129002309542582ull, 398182738873659ull}},
{{12108652016589544388ull, 318546191098927ull}},
{{4616447967575629729ull, 509673905758284ull}},
{{7382507188802414106ull, 407739124606627ull}},
{{16974052195267662255ull, 326191299685301ull}},
{{1333041809234887345ull, 521906079496483ull}},
{{8445131076871730522ull, 417524863597186ull}},
{{3066756046755474095ull, 334019890877749ull}},
{{12285507304292579198ull, 534431825404398ull}},
{{17207103472917884005ull, 427545460323518ull}},
{{2697636334108576234ull, 342036368258815ull}},
{{4316218134573721975ull, 547258189214104ull}},
{{7142323322400887903ull, 437806551371283ull}},
{{13092556287404530969ull, 350245241097026ull}},
{{13569392430363428904ull, 560392385755242ull}},
{{3476816314806922477ull, 448313908604194ull}},
{{6470801866587448304ull, 358651126883355ull}},
{{5176641493269958643ull, 286920901506684ull}},
{{15661324018715754476ull, 459073442410694ull}},
{{16218408029714513904ull, 367258753928555ull}},
{{12974726423771611123ull, 293807003142844ull}},
{{9691515833808846828ull, 470091205028551ull}},
{{4063863852305167139ull, 376072964022841ull}},
{{18008486340811775004ull, 300858371218272ull}},
{{14056182886331198714ull, 481373393949236ull}},
{{7555597494323048648ull, 385098715159389ull}},
{{9733826810200349241ull, 308078972127511ull}},
{{8195425266836738140ull, 492926355404018ull}},
{{13935037842953211158ull, 394341084323214ull}},
{{14837379089104479250ull, 315472867458571ull}},
{{16361108913083346153ull, 504756587933714ull}},
{{16778235945208587246ull, 403805270346971ull}},
{{9733239941424959473ull, 323044216277577ull}},
{{815788647312293865ull, 516870746044124ull}},
{{4341979732591745415ull, 413496596835299ull}},
{{7162932600815306655ull, 330797277468239ull}},
{{392645717078759679ull, 529275643949183ull}},
{{7692814203146828389ull, 423420515159346ull}},
{{2464902547775552388ull, 338736412127477ull}},
{{7633192891182794144ull, 541978259403963ull}},
{{13485251942430055962ull, 433582607523170ull}},
{{10788201553944044769ull, 346866086018536ull}},
{{9882424856826650985ull, 554985737629658ull}},
{{15284637514945141434ull, 443988590103726ull}},
{{8538361197214202824ull, 355190872082981ull}},
{{3141340143029451936ull, 284152697666385ull}},
{{5026144228847123098ull, 454644316266216ull}},
{{331566568335788155ull, 363715453012973ull}},
{{7643950884152451170ull, 290972362410378ull}},
{{8540972599902011550ull, 465555779856605ull}},
{{6832778079921609240ull, 372444623885284ull}},
{{9155571278679197715ull, 297955699108227ull}},
{{18338262860628626667ull, 476729118573163ull}},
{{3602563844277170364ull, 381383294858531ull}},
{{17639446334389377584ull, 305106635886824ull}},
{{17155067690797273165ull, 488170617418919ull}},
{{17413402967379728855ull, 390536493935135ull}},
{{13930722373903783084ull, 312429195148108ull}},
{{153062909794590995ull, 499886712236974ull}},
{{3811799142577583119ull, 399909369789579ull}},
{{6738788128803976819ull, 319927495831663ull}},
{{7092712191344452587ull, 511883993330661ull}},
{{1984820938333651746ull, 409507194664529ull}},
{{5277205565408831720ull, 327605755731623ull}},
{{4754180089912220429ull, 524169209170597ull}},
{{14871390516155507313ull, 419335367336477ull}},
{{4518414783440585204ull, 335468293869182ull}},
{{10918812468246846650ull, 536749270190691ull}},
{{5045701159855566996ull, 429399416152553ull}},
{{11415258557368274243ull, 343519532922042ull}},
{{3507018432821597497ull, 549631252675268ull}},
{{10184312375741098644ull, 439705002140214ull}},
{{11836798715334789238ull, 351764001712171ull}},
{{11560180315051842135ull, 562822402739474ull}},
{{12937493066783384031ull, 450257922191579ull}},
{{14039343268168617548ull, 360206337753263ull}},
{{163428170309163069ull, 288165070202611ull}},
{{11329531516720391880ull, 461064112324177ull}},
{{1684927583892492857ull, 368851289859342ull}},
{{12415988511339725255ull, 295081031887473ull}},
{{16176232803401650086ull, 472129651019957ull}},
{{5562288613237499422ull, 377703720815966ull}},
{{760482075848089214ull, 302162976652773ull}},
{{15974166580324584036ull, 483460762644436ull}},
{{9089984449517756906ull, 386768610115549ull}},
{{10961336374356115848ull, 309414888092439ull}},
{{6470091754744054387ull, 495063820947903ull}},
{{12554771033279064156ull, 396051056758322ull}},
{{2665119197139430678ull, 316840845406658ull}},
{{574841900681178762ull, 506945352650653ull}},
{{7838571150028763656ull, 405556282120522ull}},
{{17338903364248741894ull, 324445025696417ull}},
{{12984850123830345738ull, 519112041114268ull}},
{{17766577728548097237ull, 415289632891414ull}},
{{17902610997580388113ull, 332231706313131ull}},
{{2818735892935248718ull, 531570730101011ull}},
{{17012383973315840267ull, 425256584080808ull}},
{{2541860734426941244ull, 340205267264647ull}},
{{7756325989825016314ull, 544328427623435ull}},
{{6205060791860013051ull, 435462742098748ull}},
{{12342746262971831087ull, 348370193678998ull}},
{{16059045206013019417ull, 557392309886397ull}},
{{5468538535326594887ull, 445913847909118ull}},
{{11753528457745096556ull, 356731078327294ull}},
{{13092171580937987568ull, 285384862661835ull}},
{{2500730455791228493ull, 456615780258937ull}},
{{13068630808858713764ull, 365292624207149ull}},
{{14144253461828881334ull, 292234099365719ull}},
{{11562759094700479165ull, 467574558985151ull}},
{{5560858461018473009ull, 374059647188121ull}},
{{759337954072868084ull, 299247717750497ull}},
{{4904289541258499257ull, 478796348400795ull}},
{{3923431633006799406ull, 383037078720636ull}},
{{17896140565373080817ull, 306429662976508ull}},
{{6497732016145467369ull, 490287460762414ull}},
{{8887534427658284218ull, 392229968609931ull}},
{{3420678727384717051ull, 313783974887945ull}},
{{5473085963815547282ull, 502054359820712ull}},
{{15446515215278168795ull, 401643487856569ull}},
{{16046560986964445359ull, 321314790285255ull}},
{{7227753505433560959ull, 514103664456409ull}},
{{9471551619088759090ull, 411282931565127ull}},
{{198543665787186626ull, 329026345252102ull}},
{{4007018680001408925ull, 526442152403363ull}},
{{10584312573484947786ull, 421153721922690ull}},
{{8467450058787958229ull, 336922977538152ull}},
{{17237268908802643490ull, 539076764061043ull}},
{{2721768682816383822ull, 431261411248835ull}},
{{2177414946253107058ull, 345009128999068ull}},
{{18241259172972612585ull, 552014606398508ull}},
{{3524960894152359098ull, 441611685118807ull}},
{{13888015159547618248ull, 353289348095045ull}},
{{11110412127638094599ull, 282631478476036ull}},
{{10397961774737130712ull, 452210365561658ull}},
{{15697067049273525216ull, 361768292449326ull}},
{{8868304824676909849ull, 289414633959461ull}},
{{6810590089999235112ull, 463063414335138ull}},
{{12827169701483208736ull, 370450731468110ull}},
{{10261735761186566989ull, 296360585174488ull}},
{{12729428403156596859ull, 474176936279181ull}},
{{6494193907783367164ull, 379341549023345ull}},
{{5195355126226693731ull, 303473239218676ull}},
{{933870572478889324ull, 485557182749882ull}},
{{11815142902208842429ull, 388445746199905ull}},
{{9452114321767073943ull, 310756596959924ull}},
{{4055336470601587339ull, 497210555135879ull}},
{{6933617991223180194ull, 397768444108703ull}},
{{12925592022462364802ull, 318214755286962ull}},
{{5923551976972142390ull, 509143608459140ull}},
{{4738841581577713912ull, 407314886767312ull}},
{{14859119709487902099ull, 325851909413849ull}},
{{12706545090954912390ull, 521363055062159ull}},
{{13854584887505840235ull, 417090444049727ull}},
{{3704970280520851541ull, 333672355239782ull}},
{{9617301263575272790ull, 533875768383651ull}},
{{4004492196118307908ull, 427100614706921ull}},
{{17960989015862287619ull, 341680491765536ull}},
{{2912140722186287929ull, 546688786824859ull}},
{{6019061392490940666ull, 437351029459887ull}},
{{15883295558218483503ull, 349880823567909ull}},
{{14345226448923842635ull, 559809317708655ull}},
{{11476181159139074108ull, 447847454166924ull}},
{{12870293742053169609ull, 358277963333539ull}},
{{13985583808384446011ull, 286622370666831ull}},
{{14998236463931292971ull, 458595793066930ull}},
{{11998589171145034376ull, 366876634453544ull}},
{{13288220151657937824ull, 293501307562835ull}},
{{2814408168943148903ull, 469602092100537ull}},
{{13319572979380250092ull, 375681673680429ull}},
{{14345007198246110397ull, 300545338944343ull}},
{{815918628742314696ull, 480872542310950ull}},
{{652734902993851757ull, 384698033848760ull}},
{{522187922395081405ull, 307758427079008ull}},
{{15592895934799771541ull, 492413483326412ull}},
{{5095619118355996587ull, 393930786661130ull}},
{{4076495294684797269ull, 315144629328904ull}},
{{13901090100979496277ull, 504231406926246ull}},
{{7431523266041686699ull, 403385125540997ull}},
{{17013265057059080328ull, 322708100432797ull}},
{{12463828832326887233ull, 516332960692476ull}},
{{6281714251119599463ull, 413066368553981ull}},
{{1336022586153769247ull, 330453094843185ull}},
{{2137636137846030796ull, 528724951749096ull}},
{{16467504169244465929ull, 422979961399276ull}},
{{9484654520653662420ull, 338383969119421ull}},
{{7796749603562039226ull, 541414350591074ull}},
{{9926748497591541704ull, 433131480472859ull}},
{{11630747612815143686ull, 346505184378287ull}},
{{3851800921536588606ull, 554408295005260ull}},
{{3081440737229270884ull, 443526636004208ull}},
{{9843850219267237354ull, 354821308803366ull}},
{{4185731360671879560ull, 283857047042693ull}},
{{3007821362333096973ull, 454171275268309ull}},
{{6095605904608387901ull, 363337020214647ull}},
{{15944531167912441290ull, 290669616171717ull}},
{{10753854609692264772ull, 465071385874748ull}},
{{15981781317237632464ull, 372057108699798ull}},
{{1717378609564375002ull, 297645686959839ull}},
{{10126503404786820649ull, 476233099135742ull}},
{{722505094345635873ull, 380986479308594ull}},
{{4267352890218419021ull, 304789183446875ull}},
{{6827764624349470435ull, 487662693515000ull}},
{{5462211699479576348ull, 390130154812000ull}},
{{4369769359583661078ull, 312104123849600ull}},
{{6991630975333857725ull, 499366598159360ull}},
{{5593304780267086180ull, 399493278527488ull}},
{{11853341453697489590ull, 319594622821990ull}},
{{518602252206431729ull, 511351396515185ull}},
{{414881801765145383ull, 409081117212148ull}},
{{7710603070895936953ull, 327264893769718ull}},
{{8647616098691588801ull, 523623830031549ull}},
{{10607441693695181364ull, 418899064025239ull}},
{{12175302169698055414ull, 335119251220191ull}},
{{12101785842033068017ull, 536190801952306ull}},
{{5992079858884544090ull, 428952641561845ull}},
{{4793663887107635272ull, 343162113249476ull}},
{{291164589888395789ull, 549059381199162ull}},
{{11300978116136447601ull, 439247504959329ull}},
{{12730131307651068404ull, 351398003967463ull}},
{{16678861277499799123ull, 562236806347941ull}},
{{9653740207257928975ull, 449789445078353ull}},
{{15101689795290163826ull, 359831556062682ull}},
{{4702654206748310415ull, 287865244850146ull}},
{{145549101313476017ull, 460584391760234ull}},
{{3805788095792691137ull, 368467513408187ull}},
{{14112676920859883879ull, 294774010726549ull}},
{{11512236629150083237ull, 471638417162479ull}},
{{12899138118061976913ull, 377310733729983ull}},
{{17698008123933402177ull, 301848586983986ull}},
{{2491371295100071220ull, 482957739174379ull}},
{{5682445850821967299ull, 386366191339503ull}},
{{11924654310141394486ull, 309092953071602ull}},
{{4322051637258589885ull, 494548724914564ull}},
{{7146990124548782231ull, 395638979931651ull}},
{{2028243284897115462ull, 316511183945321ull}},
{{14313235700061115708ull, 506417894312513ull}},
{{382542115823161597ull, 405134315450011ull}},
{{15063428951626170570ull, 324107452360008ull}},
{{1965393434150410974ull, 518571923776014ull}},
{{5261663562062239102ull, 414857539020811ull}},
{{519982034907880958ull, 331886031216649ull}},
{{8210668885336430180ull, 531017649946638ull}},
{{13947232737752964790ull, 424814119957310ull}},
{{11157786190202371832ull, 339851295965848ull}},
{{14163109089581884608ull, 543762073545357ull}},
{{3951789642181687040ull, 435009658836286ull}},
{{17918826972712990925ull, 348007727069028ull}},
{{6534030267889323541ull, 556812363310446ull}},
{{1537875399569548509ull, 445449890648357ull}},
{{12298346763881369777ull, 356359912518685ull}},
{{9838677411105095822ull, 285087930014948ull}},
{{12052535043026242992ull, 456140688023917ull}},
{{2263330404937173747ull, 364912550419134ull}},
{{5500013138691649320ull, 291930040335307ull}},
{{12489369836648549236ull, 467088064536491ull}},
{{6302147054576929066ull, 373670451629193ull}},
{{12420415273145363899ull, 298936361303354ull}},
{{8804617992806851269ull, 478298178085367ull}},
{{18111740838471211985ull, 382638542468293ull}},
{{3421346226551238618ull, 306110833974635ull}},
{{5474153962481981789ull, 489777334359416ull}},
{{689974355243675108ull, 391821867487533ull}},
{{7930677113678760733ull, 313457493990026ull}},
{{5310385752402196526ull, 501531990384042ull}},
{{15316355046147488190ull, 401225592307233ull}},
{{1185037592692259583ull, 320980473845787ull}},
{{5585408963049525656ull, 513568758153259ull}},
{{8157675985181530848ull, 410855006522607ull}},
{{17594187232370955648ull, 328684005218085ull}},
{{9703955498083977421ull, 525894408348937ull}},
{{384466768983361290ull, 420715526679150ull}},
{{307573415186689032ull, 336572421343320ull}},
{{492117464298702451ull, 538515874149312ull}},
{{11461740415664692931ull, 430812699319449ull}},
{{12858741147273664668ull, 344650159455559ull}},
{{9505939391412132499ull, 551440255128895ull}},
{{7604751513129705999ull, 441152204103116ull}},
{{2394452395761854476ull, 352921763282493ull}},
{{9294259546093304227ull, 282337410625994ull}},
{{3802768829523555794ull, 451739857001591ull}},
{{17799610322586485928ull, 361391885601272ull}},
{{6860990628585368096ull, 289113508481018ull}},
{{7288236190994678630ull, 462581613569629ull}},
{{9519937767537653227ull, 370065290855703ull}},
{{14994647843513943228ull, 296052232684562ull}},
{{9234041290654667873ull, 473683572295300ull}},
{{7387233032523734298ull, 378946857836240ull}},
{{5909786426018987438ull, 303157486268992ull}},
{{13145007096372290225ull, 485051978030387ull}},
{{3137308047614011533ull, 388041582424310ull}},
{{2509846438091209226ull, 310433265939448ull}},
{{326405486204024439ull, 496693225503117ull}},
{{11329170833188950521ull, 397354580402493ull}},
{{16442034296034981063ull, 317883664321994ull}},
{{15239208429430238732ull, 508613862915191ull}},
{{8502017928802280662ull, 406891090332153ull}},
{{14180311972525645176ull, 325512872265722ull}},
{{7931103897073390989ull, 520820595625156ull}},
{{2655534302916802468ull, 416656476500125ull}},
{{2124427442333441974ull, 333325181200100ull}},
{{3399083907733507159ull, 533320289920160ull}},
{{2719267126186805727ull, 426656231936128ull}},
{{9554111330433265228ull, 341324985548902ull}},
{{529182869725583072ull, 546119976878244ull}},
{{4112695110522376781ull, 436895981502595ull}},
{{3290156088417901425ull, 349516785202076ull}},
{{16332296185694373249ull, 559226856323321ull}},
{{9376488133813588276ull, 447381485058657ull}},
{{122492877567049975ull, 357905188046926ull}},
{{14855389561021281272ull, 286324150437540ull}},
{{5321879223924498420ull, 458118640700065ull}},
{{4257503379139598736ull, 366494912560052ull}},
{{14474049147537409958ull, 293195930048041ull}},
{{15779781006576035287ull, 469113488076866ull}},
{{8934475990518917906ull, 375290790461493ull}},
{{14526278421898954971ull, 300232632369194ull}},
{{12173999030812596985ull, 480372211790711ull}},
{{6049850409908167265ull, 384297769432569ull}},
{{8529229142668444135ull, 307438215546055ull}},
{{13646766628269510616ull, 491901144873688ull}},
{{18296110932099429139ull, 393520915898950ull}},
{{14636888745679543311ull, 314816732719160ull}},
{{4972277919377717682ull, 503706772350657ull}},
{{15045868779727905115ull, 402965417880525ull}},
{{12036695023782324092ull, 322372334304420ull}},
{{811967964342166932ull, 515795734887073ull}},
{{8028272000957554192ull, 412636587909658ull}},
{{13801315230249864000ull, 330109270327726ull}},
{{14703406738915961753ull, 528174832524362ull}},
{{4384027761648948756ull, 422539866019490ull}},
{{3507222209319159005ull, 338031892815592ull}},
{{9300904349652564731ull, 540851028504947ull}},
{{62025850238231138ull, 432680822803958ull}},
{{7428318309674405557ull, 346144658243166ull}},
{{4506611665995228245ull, 553831453189066ull}},
{{18362684591763823889ull, 443065162551252ull}},
{{7311450043927238465ull, 354452130041002ull}},
{{16917206479367521741ull, 283561704032801ull}},
{{1242088663794662524ull, 453698726452483ull}},
{{8372368560519550665ull, 362958981161986ull}},
{{3008546033673730209ull, 290367184929589ull}},
{{12192371283361788981ull, 464587495887342ull}},
{{2375199397205610538ull, 371669996709874ull}},
{{5589508332506398754ull, 297335997367899ull}},
{{16321910961494058653ull, 475737595788638ull}},
{{1989482324969515952ull, 380590076630911ull}},
{{16348981118943254055ull, 304472061304728ull}},
{{4022276901857744549ull, 487155298087566ull}},
{{17975216780453836932ull, 389724238470052ull}},
{{7001475794879248899ull, 311779390776042ull}},
{{14891710086548708561ull, 498847025241667ull}},
{{4534670439755146203ull, 399077620193334ull}},
{{7317085166546027285ull, 319262096154667ull}},
{{15396685081215553980ull, 510819353847467ull}},
{{4938650435488622537ull, 408655483077974ull}},
{{7640269163132808353ull, 326924386462379ull}},
{{1156384216786762396ull, 523079018339807ull}},
{{11993153817655140886ull, 418463214671845ull}},
{{9594523054124112709ull, 334770571737476ull}},
{{7972539257114759688ull, 535632914779962ull}},
{{17446077849917538720ull, 428506331823969ull}},
{{17646211094675941299ull, 342805065459175ull}},
{{9787193677771954462ull, 548488104734681ull}},
{{4140406127475653247ull, 438790483787745ull}},
{{3312324901980522597ull, 351032387030196ull}},
{{16367766287394567125ull, 561651819248313ull}},
{{2026166585689922731ull, 449321455398651ull}},
{{16378328527519579477ull, 359457164318920ull}},
{{13102662822015663582ull, 287565731455136ull}},
{{13585562885741241085ull, 460105170328218ull}},
{{18247147938076813514ull, 368084136262574ull}},
{{18287067165203361134ull, 294467309010059ull}},
{{18191261020099646846ull, 471147694416095ull}},
{{14553008816079717476ull, 376918155532876ull}},
{{7953058238121863658ull, 301534524426301ull}},
{{5346195551511161206ull, 482455239082082ull}},
{{15345002885434659935ull, 385964191265665ull}},
{{12276002308347727948ull, 308771353012532ull}},
{{4884208434388723424ull, 494034164820052ull}},
{{14975413191736709708ull, 395227331856041ull}},
{{8290981738647457443ull, 316181865484833ull}},
{{9576221967094021587ull, 505890984775733ull}},
{{15039675203159037916ull, 404712787820586ull}},
{{8342391347785320009ull, 323770230256469ull}},
{{2279779712230781045ull, 518032368410351ull}},
{{16581219028752266129ull, 414425894728280ull}},
{{13264975223001812903ull, 331540715782624ull}},
{{10155913912577169676ull, 530465145252199ull}},
{{11814079944803646064ull, 424372116201759ull}},
{{13140612770584827174ull, 339497692961407ull}},
{{6267585173968082186ull, 543196308738252ull}},
{{16082114583400196718ull, 434557046990601ull}},
{{9176342851978247051ull, 347645637592481ull}},
{{7303450933681374636ull, 556233020147970ull}},
{{5842760746945099709ull, 444986416118376ull}},
{{984859782814169444ull, 355989132894701ull}},
{{15545283085218976848ull, 284791306315760ull}},
{{6425708862640811340ull, 455666090105217ull}},
{{16208613534338380042ull, 364532872084173ull}},
{{1898844383244973064ull, 291626297667339ull}},
{{10416848642675777549ull, 466602076267742ull}},
{{954781284656801392ull, 373281661014194ull}},
{{4453173842467351437ull, 298625328811355ull}},
{{7125078147947762300ull, 477800526098168ull}},
{{13078760147842030486ull, 382240420878534ull}},
{{14152356933015534712ull, 305792336702827ull}},
{{7886375833857214247ull, 489267738724524ull}},
{{9998449481827681720ull, 391414190979619ull}},
{{11688108400204055699ull, 313131352783695ull}},
{{254229366616937503ull, 501010164453913ull}},
{{7582081122777370649ull, 400808131563130ull}},
{{6065664898221896519ull, 320646505250504ull}},
{{17083761466638855077ull, 513034408400806ull}},
{{9977660358569173738ull, 410427526720645ull}},
{{7982128286855338991ull, 328342021376516ull}},
{{5392707629484721739ull, 525347234202426ull}},
{{624817288845867068ull, 420277787361941ull}},
{{15257249090044334947ull, 336222229889552ull}},
{{9654203285103294623ull, 537955567823284ull}},
{{11412711442824546021ull, 430364454258627ull}},
{{1751471524775816170ull, 344291563406902ull}},
{{6491703254383216196ull, 550866501451043ull}},
{{12572060232990393603ull, 440693201160834ull}},
{{13746997001134225206ull, 352554560928667ull}},
{{3618899971423559518ull, 282043648742934ull}},
{{13168937583761515876ull, 451269837988694ull}},
{{14224498881751123024ull, 361015870390955ull}},
{{11379599105400898419ull, 288812696312764ull}},
{{7139312124415706501ull, 462100314100423ull}},
{{13090147329016385847ull, 369680251280338ull}},
{{17850815492696929324ull, 295744201024270ull}},
{{10114560714605535302ull, 473190721638833ull}},
{{15470346201168248888ull, 378552577311066ull}},
{{8686928146192688787ull, 302842061848853ull}},
{{10209736219166391737ull, 484547298958165ull}},
{{8167788975333113389ull, 387637839166532ull}},
{{17602277624492221681ull, 310110271333225ull}},
{{9716900125478003074ull, 496176434133161ull}},
{{4084171285640492136ull, 396941147306529ull}},
{{6956685843254304032ull, 317552917845223ull}},
{{7441348534464976128ull, 508084668552357ull}},
{{17021125271797711872ull, 406467734841885ull}},
{{13616900217438169497ull, 325174187873508ull}},
{{18097691533159160873ull, 520278700597613ull}},
{{3410106782301597729ull, 416222960478091ull}},
{{17485480684808919476ull, 332978368382472ull}},
{{13219373836726629868ull, 532765389411956ull}},
{{6886150254639393571ull, 426212311529565ull}},
{{5508920203711514857ull, 340969849223652ull}},
{{12503621140680334095ull, 545551758757843ull}},
{{17381594542028087922ull, 436441407006274ull}},
{{17594624448364380661ull, 349153125605019ull}},
{{17083352673157278088ull, 558645000968031ull}},
{{9977333323783912147ull, 446916000774425ull}},
{{7981866659027129718ull, 357532800619540ull}},
{{6385493327221703774ull, 286026240495632ull}},
{{13906138138296636362ull, 457641984793011ull}},
{{7435561695895398766ull, 366113587834409ull}},
{{9637798171458229336ull, 292890870267527ull}},
{{663081815365525645ull, 468625392428044ull}},
{{4219814267034330839ull, 374900313942435ull}},
{{3375851413627464671ull, 299920251153948ull}},
{{1712013447062033151ull, 479872401846317ull}},
{{12437657201875357490ull, 383897921477053ull}},
{{17328823390984106639ull, 307118337181642ull}},
{{12968722166606929329ull, 491389339490628ull}},
{{17753675362769364110ull, 393111471592502ull}},
{{6824242660731670641ull, 314489177274002ull}},
{{14608137071912583349ull, 503182683638403ull}},
{{618463213304335710ull, 402546146910723ull}},
{{7873468200127289214ull, 322036917528578ull}},
{{8908200305461752420ull, 515259068045725ull}},
{{7126560244369401936ull, 412207254436580ull}},
{{5701248195495521549ull, 329765803549264ull}},
{{16500694742276655124ull, 527625285678822ull}},
{{5821858164337503453ull, 422100228543058ull}},
{{12036184160953823409ull, 337680182834446ull}},
{{11879197028042296808ull, 540288292535114ull}},
{{13192706437175747769ull, 432230634028091ull}},
{{6864816334998687892ull, 345784507222473ull}},
{{7294357321255990305ull, 553255211555957ull}},
{{16903532301230523213ull, 442604169244765ull}},
{{13522825840984418570ull, 354083335395812ull}},
{{3439563043303714210ull, 283266668316650ull}},
{{5503300869285942736ull, 453226669306640ull}},
{{4402640695428754189ull, 362581335445312ull}},
{{14590159000568734320ull, 290065068356249ull}},
{{12276207956684243943ull, 464104109369999ull}},
{{13510315180089305478ull, 371283287495999ull}},
{{14497600958813354705ull, 297026629996799ull}},
{{12128115089875636559ull, 475242607994879ull}},
{{13391840886642419571ull, 380194086395903ull}},
{{18092170338797756303ull, 304155269116722ull}},
{{14190077283108768792ull, 486648430586756ull}},
{{7662713011745104710ull, 389318744469405ull}},
{{6130170409396083768ull, 311454995575524ull}},
{{17186970284517554676ull, 498327992920838ull}},
{{2681529783388312771ull, 398662394336671ull}},
{{16902619085678291509ull, 318929915469336ull}},
{{1218748833891894153ull, 510287864750939ull}},
{{4664347881855425645ull, 408230291800751ull}},
{{42129490742430193ull, 326584233440601ull}},
{{11135453629413619279ull, 522534773504961ull}},
{{5219014088788985100ull, 418027818803969ull}},
{{7864560085773098403ull, 334422255043175ull}},
{{12583296137236957445ull, 535075608069080ull}},
{{10066636909789565956ull, 428060486455264ull}},
{{11742658342573563088ull, 342448389164211ull}},
{{11409555718633880294ull, 547917422662738ull}},
{{16506342204390924882ull, 438333938130190ull}},
{{13205073763512739905ull, 350667150504152ull}},
{{6370722762652742556ull, 561067440806644ull}},
{{8785927024864104368ull, 448853952645315ull}},
{{7028741619891283494ull, 359083162116252ull}},
{{16691039740138757765ull, 287266529693001ull}},
{{880221881028640162ull, 459626447508803ull}},
{{8082875134306732776ull, 367701158007042ull}},
{{17534346551671117190ull, 294160926405633ull}},
{{5918861594222325565ull, 470657482249014ull}},
{{8424438090119770775ull, 376525985799211ull}},
{{3050201657353906297ull, 301220788639369ull}},
{{12259020281250070722ull, 481953261822990ull}},
{{9807216225000056577ull, 385562609458392ull}},
{{467075350516224615ull, 308450087566714ull}},
{{8126018190309780031ull, 493520140106742ull}},
{{17568860996473554994ull, 394816112085393ull}},
{{2987042352953113026ull, 315852889668315ull}},
{{4779267764724980842ull, 505364623469304ull}},
{{7512763026521894996ull, 404291698775443ull}},
{{13388908050701336643ull, 323433359020354ull}},
{{10354206436896407660ull, 517493374432567ull}},
{{904667520033305482ull, 413994699546054ull}},
{{4413082830768554708ull, 331195759636843ull}},
{{3371583714487777211ull, 529913215418949ull}},
{{6386615786332132092ull, 423930572335159ull}},
{{8798641443807615996ull, 339144457868127ull}},
{{17767175124834095918ull, 542631132589003ull}},
{{3145693655641545764ull, 434104906071203ull}},
{{9895252553997057258ull, 347283924856962ull}},
{{1075008827427650320ull, 555654279771140ull}},
{{860007061942120256ull, 444523423816912ull}},
{{11756052093779427174ull, 355618739053529ull}},
{{13094190489765452062ull, 284494991242823ull}},
{{17261355968882812977ull, 455191985988517ull}},
{{6430387145622429735ull, 364153588790814ull}},
{{8833658531239854111ull, 291322871032651ull}},
{{6755156020499945932ull, 466116593652242ull}},
{{16472171260625687715ull, 372893274921793ull}},
{{2109690564274819202ull, 298314619937435ull}},
{{3375504902839710724ull, 477303391899896ull}},
{{17457799181239409872ull, 381842713519916ull}},
{{10276890530249617574ull, 305474170815933ull}},
{{12753676033657477796ull, 488758673305493ull}},
{{17581638456409802883ull, 391006938644394ull}},
{{17754659579869752629ull, 312805550915515ull}},
{{9960711254082052591ull, 500488881464825ull}},
{{7968569003265642073ull, 400391105171860ull}},
{{6374855202612513658ull, 320312884137488ull}},
{{6510419509438111530ull, 512500614619981ull}},
{{1518986792808578901ull, 410000491695985ull}},
{{1215189434246863121ull, 328000393356788ull}},
{{16701698353762622286ull, 524800629370860ull}},
{{13361358683010097829ull, 419840503496688ull}},
{{18067784575891898909ull, 335872402797350ull}},
{{10461711247717486639ull, 537395844475761ull}},
{{4680020183432078988ull, 429916675580609ull}},
{{7433364961487573514ull, 343933340464487ull}},
{{15582732753122027945ull, 550293344743179ull}},
{{16155535017239532679ull, 440234675794543ull}},
{{1856381569565895174ull, 352187740635635ull}},
{{1485105255652716139ull, 281750192508508ull}},
{{17133563668011987115ull, 450800308013612ull}},
{{6328153304925769046ull, 360640246410890ull}},
{{5062522643940615237ull, 288512197128712ull}},
{{11789385045046894702ull, 461619515405939ull}},
{{13120856850779426085ull, 369295612324751ull}},
{{6807336665881630544ull, 295436489859801ull}},
{{3513041035926788225ull, 472698383775682ull}},
{{13878479272967161550ull, 378158707020545ull}},
{{11102783418373729240ull, 302526965616436ull}},
{{10385755839914146137ull, 484043144986298ull}},
{{15687302301415137556ull, 387234515989038ull}},
{{1481795396906379075ull, 309787612791231ull}},
{{13438919079275937490ull, 495660180465969ull}},
{{14440484078162660315ull, 396528144372775ull}},
{{11552387262530128252ull, 317222515498220ull}},
{{37075546338653588ull, 507556024797153ull}},
{{7408358066554743516ull, 406044819837722ull}},
{{16994732897469525783ull, 324835855870177ull}},
{{12434177376983599960ull, 519737369392284ull}},
{{13636690716328790291ull, 415789895513827ull}},
{{3530654943579211586ull, 332631916411062ull}},
{{9338396724468648861ull, 532211066257699ull}},
{{11160066194316829412ull, 425768853006159ull}},
{{12617401770195373853ull, 340615082404927ull}},
{{5430447573344956872ull, 544984131847884ull}},
{{8033706873417875821ull, 435987305478307ull}},
{{17495011942960031626ull, 348789844382645ull}},
{{9545275035026498986ull, 558063751012233ull}},
{{15014917657505019835ull, 446451000809786ull}},
{{8322585311262105545ull, 357160800647829ull}},
{{10347417063751594759ull, 285728640518263ull}},
{{12866518487260641291ull, 457165824829221ull}},
{{6603865975066602710ull, 365732659863377ull}},
{{16351139224279013137ull, 292586127890701ull}},
{{336381055653048758ull, 468137804625123ull}},
{{7647802474006259652ull, 374510243700098ull}},
{{13496939608688828368ull, 299608194960078ull}},
{{17905754559160215066ull, 479373111936125ull}},
{{14324603647328172053ull, 383498489548900ull}},
{{11459682917862537642ull, 306798791639120ull}},
{{18335492668580060228ull, 490878066622592ull}},
{{7289696505380227536ull, 392702453298074ull}},
{{9521106019046092352ull, 314161962638459ull}},
{{4165723186248016794ull, 502659140221535ull}},
{{3332578548998413435ull, 402127312177228ull}},
{{10044760468682551394ull, 321701849741782ull}},
{{1314221490924440938ull, 514722959586852ull}},
{{12119423636965283720ull, 411778367669481ull}},
{{6006190094830316653ull, 329422694135585ull}},
{{9609904151728506645ull, 527076310616936ull}},
{{3998574506640894992ull, 421661048493549ull}},
{{6888208420054626317ull, 337328838794839ull}},
{{18399831101571222754ull, 539726142071742ull}},
{{7341167251773157557ull, 431780913657394ull}},
{{9562282616160436368ull, 345424730925915ull}},
{{15299652185856698190ull, 552679569481464ull}},
{{15929070563427268875ull, 442143655585171ull}},
{{9053907635999904777ull, 353714924468137ull}},
{{18311172553025654791ull, 282971939574509ull}},
{{18229829640615316696ull, 452755103319215ull}},
{{14583863712492253357ull, 362204082655372ull}},
{{4288393340509982039ull, 289763266124298ull}},
{{3172080530074060939ull, 463621225798877ull}},
{{13605710868284979721ull, 370896980639101ull}},
{{7195219879886073453ull, 296717584511281ull}},
{{4133654178333896879ull, 474748135218050ull}},
{{3306923342667117503ull, 379798508174440ull}},
{{2645538674133694003ull, 303838806539552ull}},
{{7922210693355820728ull, 486142090463283ull}},
{{13716466184168477228ull, 388913672370626ull}},
{{7283824132592871459ull, 311130937896501ull}},
{{4275420982664773689ull, 497809500634402ull}},
{{14488383230357549921ull, 398247600507521ull}},
{{7901357769544129613ull, 318598080406017ull}},
{{16331521246012517705ull, 509756928649627ull}},
{{5686519367326193517ull, 407805542919702ull}},
{{15617261938086685783ull, 326244434335761ull}},
{{17608921471454876607ull, 521991094937218ull}},
{{3019090732938170316ull, 417592875949775ull}},
{{2415272586350536253ull, 334074300759820ull}},
{{3864436138160858005ull, 534518881215712ull}},
{{14159595354754417373ull, 427615104972569ull}},
{{15017025098545444222ull, 342092083978055ull}},
{{5580496083963159139ull, 547347334364889ull}},
{{8153745681912437634ull, 437877867491911ull}},
{{2833647730788039784ull, 350302293993529ull}},
{{11912533998744684301ull, 560483670389646ull}},
{{5840678384253837118ull, 448386936311717ull}},
{{15740589151628800664ull, 358709549049373ull}},
{{1524424877077309561ull, 286967639239499ull}},
{{9817777432807515945ull, 459148222783198ull}},
{{15232919575729833402ull, 367318578226558ull}},
{{1118289216358135752ull, 293854862581247ull}},
{{5478611560914927527ull, 470167780129995ull}},
{{4382889248731942021ull, 376134224103996ull}},
{{18263706657953194910ull, 300907379283196ull}},
{{3396488949531739593ull, 481451806853115ull}},
{{2717191159625391675ull, 385161445482492ull}},
{{13241799371926044309ull, 308129156385993ull}},
{{17497530180339760572ull, 493006650217589ull}},
{{17687372959013718781ull, 394405320174071ull}},
{{10460549552469064701ull, 315524256139257ull}},
{{1979484024982862229ull, 504838809822812ull}},
{{12651633664212020753ull, 403871047858249ull}},
{{13810655746111526925ull, 323096838286599ull}},
{{11029002749552712111ull, 516954941258559ull}},
{{12512551014384080012ull, 413563953006847ull}},
{{2631343182023443363ull, 330851162405478ull}},
{{520800276495599058ull, 529361859848765ull}},
{{416640221196479247ull, 423489487879012ull}},
{{11401358621182914367ull, 338791590303209ull}},
{{7174127349666932018ull, 542066544485135ull}},
{{5739301879733545614ull, 433653235588108ull}},
{{11970139133270657137ull, 346922588470486ull}},
{{11773524983749230774ull, 555076141552778ull}},
{{16797517616483205265ull, 444060913242222ull}},
{{6059316463702743566ull, 355248730593778ull}},
{{12226150800446015499ull, 284198984475022ull}},
{{4804446021745983506ull, 454718375160036ull}},
{{154208002654876481ull, 363774700128029ull}},
{{3812715216865811508ull, 291019760102423ull}},
{{2410995532243388090ull, 465631616163877ull}},
{{12996842870020441442ull, 372505292931101ull}},
{{6708125481274442830ull, 298004234344881ull}},
{{3354303140555287882ull, 476806774951810ull}},
{{2683442512444230305ull, 381445419961448ull}},
{{9525451639439204891ull, 305156335969158ull}},
{{11551373808360817502ull, 488250137550653ull}},
{{16619796676172474648ull, 390600110040522ull}},
{{5917139711454159072ull, 312480088032418ull}},
{{5778074723584744192ull, 499968140851869ull}},
{{8311808593609705677ull, 399974512681495ull}},
{{6649446874887764541ull, 319979610145196ull}},
{{3260417370336602620ull, 511967376232314ull}},
{{6297682711011192419ull, 409573900985851ull}},
{{1348797354067043612ull, 327659120788681ull}},
{{13226122210733000749ull, 524254593261889ull}},
{{14270246583328310922ull, 419403674609511ull}},
{{7726848451920738415ull, 335522939687609ull}},
{{1294911078847450494ull, 536836703500175ull}},
{{1035928863077960395ull, 429469362800140ull}},
{{828743090462368316ull, 343575490240112ull}},
{{5015337759481699629ull, 549720784384179ull}},
{{7701619022327270026ull, 439776627507343ull}},
{{13539992847345636667ull, 351821302005874ull}},
{{10595942111527287699ull, 562914083209399ull}},
{{12166102503963740482ull, 450331266567519ull}},
{{13422230817912902709ull, 360265013254015ull}},
{{10737784654330322167ull, 288212010603212ull}},
{{2423060187960874174ull, 461139216965140ull}},
{{1938448150368699339ull, 368911373572112ull}},
{{12618804964520690441ull, 295129098857689ull}},
{{9122041499007373736ull, 472206558172303ull}},
{{14676330828689719635ull, 377765246537842ull}},
{{4362367033467955062ull, 302212197230274ull}},
{{14358484883032548746ull, 483539515568438ull}},
{{418741462200308027ull, 386831612454751ull}},
{{15092388428727887714ull, 309465289963800ull}},
{{5701077412255068727ull, 495144463942081ull}},
{{871513115062144658ull, 396115571153665ull}},
{{697210492049715726ull, 316892456922932ull}},
{{4804885602021455486ull, 507027931076691ull}},
{{154559666875254065ull, 405622344861353ull}},
{{7502345362984023899ull, 324497875889082ull}},
{{15693101395516348561ull, 519196601422531ull}},
{{8865132301671168526ull, 415357281138025ull}},
{{7092105841336934820ull, 332285824910420ull}},
{{11347369346139095713ull, 531657319856672ull}},
{{1699197847427455924ull, 425325855885338ull}},
{{8738055907425785385ull, 340260684708270ull}},
{{13980889451881256617ull, 544417095533232ull}},
{{3806013932021184647ull, 435533676426586ull}},
{{17802206404584589010ull, 348426941141268ull}},
{{6347437358883880478ull, 557483105826030ull}},
{{5077949887107104382ull, 445986484660824ull}},
{{7751708724427593829ull, 356789187728659ull}},
{{9890715794283985386ull, 285431350182927ull}},
{{1067750011886735325ull, 456690160292684ull}},
{{4543548824251298583ull, 365352128234147ull}},
{{14702885503626769836ull, 292281702587317ull}},
{{8767221546835190445ull, 467650724139708ull}},
{{14392474866951973003ull, 374120579311766ull}},
{{7824631078819668079ull, 299296463449413ull}},
{{8830060911369558603ull, 478874341519061ull}},
{{3374699914353736559ull, 383099473215249ull}},
{{6389108746224899570ull, 306479578572199ull}},
{{17601271623443659959ull, 490367325715518ull}},
{{3012970854529196998ull, 392293860572415ull}},
{{2410376683623357598ull, 313835088457932ull}},
{{7545951508539282481ull, 502136141532691ull}},
{{2347412392089515661ull, 401708913226153ull}},
{{9256627543155433175ull, 321367130580922ull}},
{{53208810081051788ull, 514187408929476ull}},
{{14799962307032482723ull, 411349927143580ull}},
{{11839969845625986178ull, 329079941714864ull}},
{{7875905308775846916ull, 526527906743783ull}},
{{13679421876504498179ull, 421222325395026ull}},
{{7254188686461688220ull, 336977860316021ull}},
{{4228004268854880506ull, 539164576505634ull}},
{{7071752229825814728ull, 431331661204507ull}},
{{16725448228086382752ull, 345065328963605ull}},
{{8313973091228660787ull, 552104526341769ull}},
{{10340527287724838953ull, 441683621073415ull}},
{{8272421830179871162ull, 353346896858732ull}},
{{17685983908369627899ull, 282677517486985ull}},
{{9850830179681853023ull, 452284027979177ull}},
{{501966514261661772ull, 361827222383342ull}},
{{11469619655635060387ull, 289461777906673ull}},
{{14662042634274186296ull, 463138844650677ull}},
{{4350936477935528391ull, 370511075720542ull}},
{{14548795626574153682ull, 296408860576433ull}},
{{1141980114067183952ull, 474254176922294ull}},
{{4602932905995657485ull, 379403341537835ull}},
{{3682346324796525988ull, 303522673230268ull}},
{{2202405304932531258ull, 485636277168429ull}},
{{5451273058687935329ull, 388509021734743ull}},
{{11739716076434168910ull, 310807217387794ull}},
{{7715499278068939286ull, 497291547820471ull}},
{{2483050607713241106ull, 397833238256377ull}},
{{13054486930396323854ull, 318266590605101ull}},
{{13508481459150297520ull, 509226544968162ull}},
{{3428087537836417370ull, 407381235974530ull}},
{{2742470030269133896ull, 325904988779624ull}},
{{11766649677914434880ull, 521447982047398ull}},
{{16792017371815368550ull, 417158385637918ull}},
{{2365567453226563870ull, 333726708510335ull}},
{{3784907925162502193ull, 533962733616536ull}},
{{17785321599097643047ull, 427170186893228ull}},
{{3160210835052383468ull, 341736149514583ull}},
{{1366988521341903226ull, 546777839223333ull}},
{{8472288446557343227ull, 437422271378666ull}},
{{3088481942503964258ull, 349937817102933ull}},
{{1252222293264432490ull, 559900507364693ull}},
{{8380475464095366639ull, 447920405891754ull}},
{{10393729186018203634ull, 358336324713403ull}},
{{15693680978298383553ull, 286669059770722ull}},
{{10352494306309772393ull, 458670495633156ull}},
{{4592646630305907591ull, 366936396506525ull}},
{{3674117304244726073ull, 293549117205220ull}},
{{5878587686791561717ull, 469678587528352ull}},
{{15770916593658980343ull, 375742870022681ull}},
{{8927384460185273951ull, 300594296018145ull}},
{{14283815136296438322ull, 480950873629032ull}},
{{4048354479553330011ull, 384760698903226ull}},
{{17996078842610305302ull, 307808559122580ull}},
{{10346982074466936867ull, 492493694596129ull}},
{{11966934474315459817ull, 393994955676903ull}},
{{16952245208936188500ull, 315195964541522ull}},
{{12366197075330260307ull, 504313543266436ull}},
{{6203608845522297922ull, 403450834613149ull}},
{{8652235891159748661ull, 322760667690519ull}},
{{2775530981629866888ull, 516417068304831ull}},
{{16977820044271534803ull, 413133654643864ull}},
{{17271604850159138166ull, 330506923715091ull}},
{{1809126057061248803ull, 528811077944147ull}},
{{12515347289874730012ull, 423048862355317ull}},
{{2633580202415963363ull, 338439089884254ull}},
{{11592425953349362027ull, 541502543814806ull}},
{{5584591947937579299ull, 433202035051845ull}},
{{4467673558350063439ull, 346561628041476ull}},
{{18216324137585832472ull, 554498604866361ull}},
{{10883710495326755654ull, 443598883893089ull}},
{{12396317211003314846ull, 354879107114471ull}},
{{6227704954060741554ull, 283903285691577ull}},
{{13653676741239096810ull, 454245257106523ull}},
{{18301639022475098094ull, 363396205685218ull}},
{{3573264773754347506ull, 290716964548175ull}},
{{5717223638006956009ull, 465147143277080ull}},
{{4573778910405564807ull, 372117714621664ull}},
{{7348371943066362169ull, 297694171697331ull}},
{{4378697479422358824ull, 476310674715730ull}},
{{3502957983537887059ull, 381048539772584ull}},
{{6491715201572219970ull, 304838831818067ull}},
{{14076093137257462276ull, 487742130908907ull}},
{{3882176880322149174ull, 390193704727126ull}},
{{17863136763225360632ull, 312154963781700ull}},
{{10134274747451025396ull, 499447942050721ull}},
{{4418070983218909993ull, 399558353640577ull}},
{{14602503230800858964ull, 319646682912461ull}},
{{15985307539797553697ull, 511434692659938ull}},
{{1720199587612311988ull, 409147754127951ull}},
{{16133554929057490883ull, 327318203302360ull}},
{{7366943812782433797ull, 523709125283777ull}},
{{16961601494451678007ull, 418967300227021ull}},
{{9879932380819432082ull, 335173840181617ull}},
{{1050496550343450039ull, 536278144290588ull}},
{{8219094869758580678ull, 429022515432470ull}},
{{6575275895806864542ull, 343218012345976ull}},
{{3141743803807162621ull, 549148819753562ull}},
{{13581441487271461066ull, 439319055802849ull}},
{{14554502004559079176ull, 351455244642279ull}},
{{12219156763068795713ull, 562328391427647ull}},
{{2396627780971215924ull, 449862713142118ull}},
{{9295999854260793385ull, 359890170513694ull}},
{{11126148698150545031ull, 287912136410955ull}},
{{17801837917040872050ull, 460659418257528ull}},
{{3173423889406966671ull, 368527534606023ull}},
{{9917436741009393983ull, 294822027684818ull}},
{{12178549970873120049ull, 471715244295709ull}},
{{13432188791440406363ull, 377372195436567ull}},
{{3367053403668504444ull, 301897756349254ull}},
{{12765983075353427757ull, 483036410158806ull}},
{{6523437645540831882ull, 386429128127045ull}},
{{5218750116432665505ull, 309143302501636ull}},
{{971302556808444163ull, 494629284002618ull}},
{{8155739674930575976ull, 395703427202094ull}},
{{10213940554686371104ull, 316562741761675ull}},
{{16342304887498193767ull, 506500386818680ull}},
{{13073843909998555014ull, 405200309454944ull}},
{{14148423942740754334ull, 324160247563955ull}},
{{4190734234675655319ull, 518656396102329ull}},
{{7041936202482434578ull, 414925116881863ull}},
{{13012246591469768309ull, 331940093505490ull}},
{{2372850472642077678ull, 531104149608785ull}},
{{1898280378113662142ull, 424883319687028ull}},
{{8897321931974750360ull, 339906655749622ull}},
{{17925063905901510900ull, 543850649199395ull}},
{{14340051124721208720ull, 435080519359516ull}},
{{7782692085035056653ull, 348064415487613ull}},
{{8762958521314180321ull, 556903064780181ull}},
{{3321018002309433934ull, 445522451824145ull}},
{{2656814401847547147ull, 356417961459316ull}},
{{16882846780445679010ull, 285134369167452ull}},
{{12255159589745445124ull, 456214990667924ull}},
{{13493476486538266422ull, 364971992534339ull}},
{{14484130003972523461ull, 291977594027471ull}},
{{15795910376872216891ull, 467164150443954ull}},
{{16326077116239683836ull, 373731320355163ull}},
{{1992815248766016099ull, 298985056284131ull}},
{{14256550842251356728ull, 478376090054609ull}},
{{15094589488542995706ull, 382700872043687ull}},
{{4696973961350575918ull, 306160697634950ull}},
{{7515158338160921469ull, 489857116215920ull}},
{{6012126670528737175ull, 391885692972736ull}},
{{1120352521681079417ull, 313508554378189ull}},
{{9171261664173547714ull, 501613687005102ull}},
{{18405055775564569141ull, 401290949604081ull}},
{{11034695805709744989ull, 321032759683265ull}},
{{17655513289135591983ull, 513652415493224ull}},
{{17813759446050383910ull, 410921932394579ull}},
{{17940356371582217451ull, 328737545915663ull}},
{{6568477306080085982ull, 525980073465062ull}},
{{16322828289089799755ull, 420784058772049ull}},
{{16747611446013750127ull, 336627247017639ull}},
{{15728131869396269235ull, 538603595228223ull}},
{{1514459051291284418ull, 430882876182579ull}},
{{4900916055774937857ull, 344706300946063ull}},
{{4152116874497990249ull, 551530081513701ull}},
{{18079088758566033492ull, 441224065210960ull}},
{{14463271006852826793ull, 352979252168768ull}},
{{502570361256530465ull, 282383401735015ull}},
{{804112578010448744ull, 451813442776024ull}},
{{4332638877150269319ull, 361450754220819ull}},
{{7155459916462125778ull, 289160603376655ull}},
{{11448735866339401245ull, 462656965402648ull}},
{{16537686322555341642ull, 370125572322118ull}},
{{2162102613818542344ull, 296100457857695ull}},
{{3459364182109667751ull, 473760732572312ull}},
{{13835537789913465170ull, 379008586057849ull}},
{{14757779046672682459ull, 303206868846279ull}},
{{12544400030450560966ull, 485130990154047ull}},
{{2656822394876628126ull, 388104792123238ull}},
{{9504155545385123147ull, 310483833698590ull}},
{{15206648872616197036ull, 496774133917744ull}},
{{15854667912834867952ull, 397419307134195ull}},
{{12683734330267894361ull, 317935445707356ull}},
{{12915277298944810332ull, 508696713131770ull}},
{{10332221839155848265ull, 406957370505416ull}},
{{4576428656582768289ull, 325565896404333ull}},
{{3632937035790518939ull, 520905434246933ull}},
{{10285047258116235798ull, 416724347397546ull}},
{{4538688991751078315ull, 333379477918037ull}},
{{10951251201543635627ull, 533407164668859ull}},
{{12450349775976818825ull, 426725731735087ull}},
{{2581582191297634414ull, 341380585388070ull}},
{{4130531506076215062ull, 546208936620912ull}},
{{14372471649086703019ull, 436967149296729ull}},
{{15187326134011272738ull, 349573719437383ull}},
{{2163628925966574443ull, 559317951099814ull}},
{{5420251955515169877ull, 447454360879851ull}},
{{646852749670225578ull, 357963488703881ull}},
{{15274877458703821755ull, 286370790963104ull}},
{{13371757489700383839ull, 458193265540967ull}},
{{3318708362276486425ull, 366554612432774ull}},
{{6344315504563099463ull, 293243689946219ull}},
{{17529602436784779788ull, 469189903913950ull}},
{{14023681949427823830ull, 375351923131160ull}},
{{11218945559542259064ull, 300281538504928ull}},
{{14260964080525704179ull, 480450461607885ull}},
{{11408771264420563343ull, 384360369286308ull}},
{{16505714641020271321ull, 307488295429046ull}},
{{583701722439061852ull, 491981272686475ull}},
{{466961377951249481ull, 393585018149180ull}},
{{373569102360999585ull, 314868014519344ull}},
{{7976408193261419982ull, 503788823230950ull}},
{{6381126554609135986ull, 403031058584760ull}},
{{5104901243687308789ull, 322424846867808ull}},
{{4478493175157783739ull, 515879754988493ull}},
{{10961492169610047637ull, 412703803990794ull}},
{{12458542550429948433ull, 330163043192635ull}},
{{1486924006978365877ull, 528260869108217ull}},
{{12257585649808423671ull, 422608695286573ull}},
{{17184766149330559583ull, 338086956229258ull}},
{{5359532950477433394ull, 540939129966814ull}},
{{7976975175123857038ull, 432751303973451ull}},
{{2692231325357175307ull, 346201043178761ull}},
{{15375616564797211462ull, 553921669086017ull}},
{{4921795622353948523ull, 443137335268814ull}},
{{7626785312625069141ull, 354509868215051ull}},
{{2412079435358144990ull, 283607894572041ull}},
{{14927373540798762954ull, 453772631315265ull}},
{{11941898832639010363ull, 363018105052212ull}},
{{2174821436627387644ull, 290414484041770ull}},
{{3479714298603820230ull, 464663174466832ull}},
{{13851817883108787154ull, 371730539573465ull}},
{{11081454306487029723ull, 297384431658772ull}},
{{2972931631411606264ull, 475815090654036ull}},
{{17135740564096926304ull, 380652072523228ull}},
{{2640546007051810074ull, 304521658018583ull}},
{{535524796540985795ull, 487234652829733ull}},
{{7807117466716609282ull, 389787722263786ull}},
{{2556345158631377102ull, 311830177811029ull}},
{{11468849883294024011ull, 498928284497646ull}},
{{5485731091893308885ull, 399142627598117ull}},
{{15456631317740378078ull, 319314102078493ull}},
{{2594517219933142985ull, 510902563325590ull}},
{{2075613775946514388ull, 408722050660472ull}},
{{12728537464982942480ull, 326977640528377ull}},
{{5608264685005066675ull, 523164224845404ull}},
{{8175960562745963663ull, 418531379876323ull}},
{{13919466079680591577ull, 334825103901058ull}},
{{135052839037484584ull, 535720166241694ull}},
{{3797391085971897991ull, 428576132993355ull}},
{{3037912868777518392ull, 342860906394684ull}},
{{12239358219527850075ull, 548577450231494ull}},
{{13480835390364190383ull, 438861960185195ull}},
{{10784668312291352306ull, 351089568148156ull}},
{{9876771670182343044ull, 561743309037050ull}},
{{7901417336145874435ull, 449394647229640ull}},
{{6321133868916699548ull, 359515717783712ull}},
{{16124953539359090608ull, 287612574226969ull}},
{{14731879218748814003ull, 460180118763151ull}},
{{8096154560257140879ull, 368144095010521ull}},
{{2787574833463802380ull, 294515276008417ull}},
{{8149468548283994132ull, 471224441613467ull}},
{{17587621282852926275ull, 376979553290773ull}},
{{3002050582056610050ull, 301583642632619ull}},
{{12181978560774396727ull, 482533828212190ull}},
{{9745582848619517381ull, 386027062569752ull}},
{{417768649411793259ull, 308821650055802ull}},
{{4357778653800779537ull, 494114640089283ull}},
{{10864920552524444276ull, 395291712071426ull}},
{{5002587627277645098ull, 316233369657141ull}},
{{625442574160411510ull, 505973391451426ull}},
{{15257749318295970501ull, 404778713161140ull}},
{{12206199454636776400ull, 323822970528912ull}},
{{4772523868451200948ull, 518116752846260ull}},
{{3818019094760960759ull, 414493402277008ull}},
{{10433112905292589253ull, 331594721821606ull}},
{{9314283018984322159ull, 530551554914570ull}},
{{7451426415187457727ull, 424441243931656ull}},
{{2271792317408055858ull, 339552995145325ull}},
{{3634867707852889374ull, 543284792232520ull}},
{{2907894166282311499ull, 434627833786016ull}},
{{17083710591993490492ull, 347702267028812ull}},
{{12576541688221943494ull, 556323627246100ull}},
{{10061233350577554795ull, 445058901796880ull}},
{{8048986680462043836ull, 356047121437504ull}},
{{10128538159111545392ull, 284837697150003ull}},
{{12516312239836562304ull, 455740315440005ull}},
{{10013049791869249843ull, 364592252352004ull}},
{{11699788648237310198ull, 291673801881603ull}},
{{15030313022437785994ull, 466678083010565ull}},
{{12024250417950228795ull, 373342466408452ull}},
{{2240702704876362389ull, 298673973126762ull}},
{{7274473142544090146ull, 477878357002819ull}},
{{9508927328777182440ull, 382302685602255ull}},
{{7607141863021745952ull, 305842148481804ull}},
{{1103380536609062554ull, 489347437570887ull}},
{{11950750873512981013ull, 391477950056709ull}},
{{13249949513552295133ull, 313182360045367ull}},
{{6442523962716030921ull, 501091776072588ull}},
{{12532716799656645383ull, 400873420858070ull}},
{{10026173439725316306ull, 320698736686456ull}},
{{8663179874076685444ull, 513117978698330ull}},
{{6930543899261348355ull, 410494382958664ull}},
{{9233783934150989007ull, 328395506366931ull}},
{{7395356665157761765ull, 525432810187090ull}},
{{5916285332126209412ull, 420346248149672ull}},
{{15801074709926698499ull, 336276998519737ull}},
{{10524324276915076306ull, 538043197631580ull}},
{{8419459421532061045ull, 430434558105264ull}},
{{10424916351967559159ull, 344347646484211ull}},
{{9301168533664274008ull, 550956234374738ull}},
{{14819632456415239853ull, 440764987499790ull}},
{{11855705965132191882ull, 352611989999832ull}},
{{2105867142621932859ull, 282089591999866ull}},
{{14437433872420823545ull, 451343347199785ull}},
{{11549947097936658836ull, 361074677759828ull}},
{{16618655307833147715ull, 288859742207862ull}},
{{11832453233565395051ull, 462175587532580ull}},
{{9465962586852316041ull, 369740470026064ull}},
{{11262118884223763156ull, 295792376020851ull}},
{{10640692585274200403ull, 473267801633362ull}},
{{1133856438735539676ull, 378614241306690ull}},
{{907085150988431741ull, 302891393045352ull}},
{{5140685056323401109ull, 484626228872563ull}},
{{11491245674542541533ull, 387700983098050ull}},
{{9192996539634033226ull, 310160786478440ull}},
{{14708794463414453163ull, 496257258365504ull}},
{{15456384385473472853ull, 397005806692403ull}},
{{1297061064153047313ull, 317604645353923ull}},
{{16832692961612516994ull, 508167432566276ull}},
{{9776805554548103272ull, 406533946053021ull}},
{{4132095628896572294ull, 325227156842417ull}},
{{10300701820976425994ull, 520363450947867ull}},
{{861863827297320149ull, 416290760758294ull}},
{{4378839876579766442ull, 333032608606635ull}},
{{7006143802527626308ull, 532852173770616ull}},
{{1915566227280190723ull, 426281739016493ull}},
{{8911150611307973225ull, 341025391213194ull}},
{{3189794533867026190ull, 545640625941111ull}},
{{17309230886061262245ull, 436512500752888ull}},
{{2779338264623278826ull, 349210000602311ull}},
{{15514987667622977092ull, 558736000963697ull}},
{{5033292504614561027ull, 446988800770958ull}},
{{11405331633175469468ull, 357591040616766ull}},
{{5434916491798465251ull, 286072832493413ull}},
{{5006517572135634078ull, 457716531989461ull}},
{{315865242966596939ull, 366173225591569ull}},
{{3942041009115187875ull, 292938580473255ull}},
{{6307265614584300600ull, 468701728757208ull}},
{{12424510121151261126ull, 374961383005766ull}},
{{6250259282179098578ull, 299969106404613ull}},
{{6311066036744647401ull, 479950570247381ull}},
{{1359504014653807598ull, 383960456197905ull}},
{{1087603211723046078ull, 307168364958324ull}},
{{9118862768240694372ull, 491469383933318ull}},
{{14673787844076376144ull, 393175507146654ull}},
{{15428379090003011238ull, 314540405717323ull}},
{{2549313655553356042ull, 503264649147718ull}},
{{9418148553926505480ull, 402611719318174ull}},
{{11223867657883114707ull, 322089375454539ull}},
{{6890141808387252562ull, 515343000727263ull}},
{{12890811076193622696ull, 412274400581810ull}},
{{10312648860954898156ull, 329819520465448ull}},
{{12810889362785926727ull, 527711232744717ull}},
{{2870013860744920735ull, 422168986195774ull}},
{{5985359903337846911ull, 337735188956619ull}},
{{16955273474824375705ull, 540376302330590ull}},
{{13564218779859500564ull, 432301041864472ull}},
{{3472677394403779805ull, 345840833491578ull}},
{{1866935016304137365ull, 553345333586525ull}},
{{1493548013043309892ull, 442676266869220ull}},
{{1194838410434647913ull, 354141013495376ull}},
{{15713265987315359623ull, 283312810796300ull}},
{{6694481505995023781ull, 453300497274081ull}},
{{1666236390054108702ull, 362640397819265ull}},
{{1332989112043286961ull, 290112318255412ull}},
{{5822131394011169462ull, 464179709208659ull}},
{{8347053929950845892ull, 371343767366927ull}},
{{17745689588186407683ull, 297075013893541ull}},
{{2567661637904880031ull, 475320022229667ull}},
{{13122175754549634995ull, 380256017783733ull}},
{{17876438233123528642ull, 304204814226986ull}},
{{2776859469804273565ull, 486727702763179ull}},
{{5910836390585329175ull, 389382162210543ull}},
{{12107366741952083986ull, 311505729768434ull}},
{{8303740342897603409ull, 498409167629495ull}},
{{6642992274318082727ull, 398727334103596ull}},
{{1625045004712555858ull, 318981867282877ull}},
{{6289420822281999697ull, 510370987652603ull}},
{{12410234287309420404ull, 408296790122082ull}},
{{2549489800363715677ull, 326637432097666ull}},
{{15147230124807676052ull, 522619891356265ull}},
{{12117784099846140842ull, 418095913085012ull}},
{{2315529650393092027ull, 334476730468010ull}},
{{3704847440628947243ull, 535162768748816ull}},
{{17721273211470799087ull, 428130214999052ull}},
{{6798320939692818623ull, 342504171999242ull}},
{{14566662318250420121ull, 548006675198787ull}},
{{4274632225116515450ull, 438405340159030ull}},
{{3419705780093212360ull, 350724272127224ull}},
{{12850226877632960423ull, 561158835403558ull}},
{{17658879131590188985ull, 448927068322846ull}},
{{10437754490530240864ull, 359141654658277ull}},
{{971505962940372045ull, 287313323726622ull}},
{{5243758355446505596ull, 459701317962595ull}},
{{4195006684357204476ull, 367761054370076ull}},
{{18113400606453404874ull, 294208843496060ull}},
{{10534696896615896182ull, 470734149593697ull}},
{{1049059887808896299ull, 376587319674958ull}},
{{8217945539730937686ull, 301269855739966ull}},
{{5770015234085679651ull, 482031769183946ull}},
{{926663372526633398ull, 385625415347157ull}},
{{11809377142247037688ull, 308500332277725ull}},
{{448259353885708685ull, 493600531644361ull}},
{{15116002742076208240ull, 394880425315488ull}},
{{1024755749435235623ull, 315904340252391ull}},
{{12707655643322107966ull, 505446944403825ull}},
{{10166124514657686373ull, 404357555523060ull}},
{{8132899611726149098ull, 323486044418448ull}},
{{9323290564019928234ull, 517577671069517ull}},
{{79934821732121941ull, 414062136855614ull}},
{{3753296672127607876ull, 331249709484491ull}},
{{17073321119629903571ull, 529999535175185ull}},
{{13658656895703922857ull, 423999628140148ull}},
{{18305623146046958932ull, 339199702512118ull}},
{{7152904145223672352ull, 542719524019390ull}},
{{5722323316178937881ull, 434175619215512ull}},
{{15645905097168881275ull, 347340495372409ull}},
{{13965401711244479070ull, 555744792595855ull}},
{{11172321368995583256ull, 444595834076684ull}},
{{12627205909938376928ull, 355676667261347ull}},
{{2723067098466880896ull, 284541333809078ull}},
{{667558542805099110ull, 455266134094525ull}},
{{534046834244079288ull, 364212907275620ull}},
{{427237467395263430ull, 291370325820496ull}},
{{11751626392058152458ull, 466192521312793ull}},
{{16779998743130342613ull, 372954017050234ull}},
{{17113347809246184414ull, 298363213640187ull}},
{{12623961235826253769ull, 477381141824300ull}},
{{10099168988661003015ull, 381904913459440ull}},
{{8079335190928802412ull, 305523930767552ull}},
{{16616285120227994183ull, 488838289228083ull}},
{{2224981651956664377ull, 391070631382467ull}},
{{12848031765791062471ull, 312856505105973ull}},
{{16867502010523789630ull, 500570408169557ull}},
{{6115303978935211058ull, 400456326535646ull}},
{{1202894368406258523ull, 320365061228517ull}},
{{5613979804191923960ull, 512584097965627ull}},
{{15559230287579270138ull, 410067278372501ull}},
{{8758035415321505787ull, 328053822698001ull}},
{{6634159035030588613ull, 524886116316802ull}},
{{16375373672250201860ull, 419908893053441ull}},
{{9410950123058251165ull, 335927114442753ull}},
{{11368171382151291540ull, 537483383108405ull}},
{{9094537105721033232ull, 429986706486724ull}},
{{10964978499318736909ull, 343989365189379ull}},
{{6475919154684248085ull, 550382984303007ull}},
{{16248781767973129437ull, 440306387442405ull}},
{{12999025414378503550ull, 352245109953924ull}},
{{14088569146244713163ull, 281796087963139ull}},
{{11473664189765810091ull, 450873740741023ull}},
{{16557628981296468719ull, 360698992592818ull}},
{{2178056740811444006ull, 288559194074255ull}},
{{3484890785298310410ull, 461694710518808ull}},
{{10166610257722468974ull, 369355768415046ull}},
{{4443939391436064856ull, 295484614732037ull}},
{{10799651841039614093ull, 472775383571259ull}},
{{12329070287573601597ull, 378220306857007ull}},
{{2484558600575060631ull, 302576245485606ull}},
{{15043340205145827980ull, 484121992776969ull}},
{{15724020978858572707ull, 387297594221575ull}},
{{12579216783086858166ull, 309838075377260ull}},
{{1680002779229421449ull, 495740920603617ull}},
{{12412048667609268129ull, 396592736482893ull}},
{{17308336563571235150ull, 317274189186314ull}},
{{16625292057488245270ull, 507638702698103ull}},
{{2232187201764865246ull, 406110962158483ull}},
{{9164447390895712843ull, 324888769726786ull}},
{{7284418195949319903ull, 519822031562858ull}},
{{13206232186243276569ull, 415857625250286ull}},
{{6875636934252710932ull, 332686100200229ull}},
{{18379716724288158138ull, 532297760320366ull}},
{{11014424564688616187ull, 425838208256293ull}},
{{16190237281234713596ull, 340670566605034ull}},
{{14836333205749810784ull, 545072906568055ull}},
{{11869066564599848627ull, 436058325254444ull}},
{{13184602066421789225ull, 348846660203555ull}},
{{2648619232565311144ull, 558154656325689ull}},
{{5808244200794159238ull, 446523725060551ull}},
{{957246545893417067ull, 357218980048441ull}},
{{15523192495682374946ull, 285775184038752ull}},
{{10079712734124158622ull, 457240294462004ull}},
{{11753119002041237221ull, 365792235569603ull}},
{{16781192831116810423ull, 292633788455682ull}},
{{12092513270819255384ull, 468214061529092ull}},
{{2295312987171583661ull, 374571249223274ull}},
{{5525599204479177252ull, 299656999378619ull}},
{{16219656356650504249ull, 479451199005790ull}},
{{12975725085320403399ull, 383560959204632ull}},
{{3001882438772502073ull, 306848767363706ull}},
{{15871058346261734287ull, 490958027781929ull}},
{{16386195491751297752ull, 392766422225543ull}},
{{2040909949175307232ull, 314213137780435ull}},
{{3265455918680491572ull, 502741020448696ull}},
{{17369759993912034550ull, 402192816358956ull}},
{{10206459180387717317ull, 321754253087165ull}},
{{16330334688620347707ull, 514806804939464ull}},
{{16753616565638188489ull, 411845443951571ull}},
{{9713544437768640468ull, 329476355161257ull}},
{{784275841462183456ull, 527162168258012ull}},
{{11695467117395477734ull, 421729734606409ull}},
{{13045722508658292511ull, 337383787685127ull}},
{{6115760754885626724ull, 539814060296204ull}},
{{8581957418650411703ull, 431851248236963ull}},
{{14244263564404150008ull, 345480998589570ull}},
{{4344077629337088398ull, 552769597743313ull}},
{{10853959732953491364ull, 442215678194650ull}},
{{8683167786362793091ull, 353772542555720ull}},
{{6946534229090234473ull, 283018034044576ull}},
{{3735757137060554511ull, 452828854471322ull}},
{{14056652153874174578ull, 362263083577057ull}},
{{3866624093615519016ull, 289810466861646ull}},
{{17254644994010561396ull, 463696746978633ull}},
{{2735669550982718147ull, 370957397582907ull}},
{{13256582085011905487ull, 296765918066325ull}},
{{2763787262309497163ull, 474825468906121ull}},
{{16968425068815239023ull, 379860375124896ull}},
{{9885391240310280895ull, 303888300099917ull}},
{{1059230725528808140ull, 486221280159868ull}},
{{8226082209906867158ull, 388977024127894ull}},
{{10270214582667404050ull, 311181619302315ull}},
{{16432343332267846480ull, 497890590883704ull}},
{{16835223480556187507ull, 398312472706963ull}},
{{2400132340219219036ull, 318649978165571ull}},
{{14908258188576481427ull, 509839965064913ull}},
{{858560106635454172ull, 407871972051931ull}},
{{15444243344276004631ull, 326297577641544ull}},
{{13642742906615876440ull, 522076124226471ull}},
{{7224845510550790828ull, 417660899381177ull}},
{{16847922852666363632ull, 334128719504941ull}},
{{1131234861072809549ull, 534605951207907ull}},
{{11973034333083978609ull, 427684760966325ull}},
{{9578427466467182887ull, 342147808773060ull}},
{{15325483946347492620ull, 547436494036896ull}},
{{8571038342336083772ull, 437949195229517ull}},
{{17924877118094597987ull, 350359356183613ull}},
{{6543710500499894841ull, 560574969893782ull}},
{{16303014844625646842ull, 448459975915025ull}},
{{13042411875700517474ull, 358767980732020ull}},
{{10433929500560413979ull, 287014384585616ull}},
{{9315589571412841720ull, 459223015336986ull}},
{{3763122842388363053ull, 367378412269589ull}},
{{6699847088652600765ull, 293902729815671ull}},
{{3341057712360340578ull, 470244367705074ull}},
{{6362194984630182786ull, 376195494164059ull}},
{{8779104802446056552ull, 300956395331247ull}},
{{17735916498655600806ull, 481530232529995ull}},
{{14188733198924480645ull, 385224186023996ull}},
{{7661637744397674193ull, 308179348819197ull}},
{{15947969205778189032ull, 493086958110715ull}},
{{12758375364622551225ull, 394469566488572ull}},
{{2828002662214220334ull, 315575653190858ull}},
{{835455444800842211ull, 504921045105373ull}},
{{8047061985324494415ull, 403936836084298ull}},
{{13816347217743416178ull, 323149468867438ull}},
{{18416806733647555563ull, 517039150187901ull}},
{{11044096572176134127ull, 413631320150321ull}},
{{5145928442998996978ull, 330905056120257ull}},
{{11922834323540305489ull, 529448089792411ull}},
{{5848918644090334068ull, 423558471833929ull}},
{{8368483730014177577ull, 338846777467143ull}},
{{9700225153280773800ull, 542154843947429ull}},
{{11449528937366529363ull, 433723875157943ull}},
{{16538320779377044137ull, 346979100126354ull}},
{{15393266802777539650ull, 555166560202167ull}},
{{4935915812738211074ull, 444133248161734ull}},
{{7638081464932479182ull, 355306598529387ull}},
{{17178511616171714315ull, 284245278823509ull}},
{{16417572141649011935ull, 454792446117615ull}},
{{13134057713319209548ull, 363833956894092ull}},
{{3128548541171546992ull, 291067165515274ull}},
{{12384375295358295833ull, 465707464824438ull}},
{{17286197865770457313ull, 372565971859550ull}},
{{13828958292616365850ull, 298052777487640ull}},
{{3679589194476633745ull, 476884443980225ull}},
{{2943671355581306996ull, 381507555184180ull}},
{{2354937084465045596ull, 305206044147344ull}},
{{11146596964627893601ull, 488329670635750ull}},
{{8917277571702314881ull, 390663736508600ull}},
{{7133822057361851905ull, 312530989206880ull}},
{{11414115291778963048ull, 500049582731008ull}},
{{16509989862906991084ull, 400039666184806ull}},
{{9518643075583682544ull, 320031732947845ull}},
{{15229828920933892071ull, 512050772716552ull}},
{{4805165507263293010ull, 409640618173242ull}},
{{14912178850036365378ull, 327712494538593ull}},
{{1723393271606722665ull, 524339991261750ull}},
{{1378714617285378132ull, 419471993009400ull}},
{{1102971693828302506ull, 335577594407520ull}},
{{1764754710125284009ull, 536924151052032ull}},
{{12479850212325958177ull, 429539320841625ull}},
{{9983880169860766542ull, 343631456673300ull}},
{{15974208271777226467ull, 549810330677280ull}},
{{12779366617421781173ull, 439848264541824ull}},
{{13912842108679335262ull, 351878611633459ull}},
{{14819622501685378532ull, 281502889306767ull}},
{{8954000743728964359ull, 450404622890828ull}},
{{14541898224466992134ull, 360323698312662ull}},
{{4254820950089773061ull, 288258958650130ull}},
{{6807713520143636897ull, 461214333840208ull}},
{{12824868445598730164ull, 368971467072166ull}},
{{6570545941737073808ull, 295177173657733ull}},
{{6823524692037407770ull, 472283477852373ull}},
{{12837517383113746862ull, 377826782281898ull}},
{{17648711535974818136ull, 302261425825518ull}},
{{6101845569108247079ull, 483618281320830ull}},
{{4881476455286597663ull, 386894625056664ull}},
{{7594529978971188453ull, 309515700045331ull}},
{{4772550336870080879ull, 495225120072530ull}},
{{3818040269496064703ull, 396180096058024ull}},
{{6743781030338762086ull, 316944076846419ull}},
{{18168747278025839984ull, 507110522954270ull}},
{{14534997822420671987ull, 405688418363416ull}},
{{7938649443194627266ull, 324550734690733ull}},
{{9012490294369493303ull, 519281175505173ull}},
{{14588689864979415289ull, 415424940404138ull}},
{{602905447757801261ull, 332339952323311ull}},
{{12032695160638212988ull, 531743923717297ull}},
{{2247458499026749744ull, 425395138973838ull}},
{{9176664428705220442ull, 340316111179070ull}},
{{14682663085928352707ull, 544505777886512ull}},
{{4367432839258861519ull, 435604622309210ull}},
{{3493946271407089215ull, 348483697847368ull}},
{{1900965219509432421ull, 557573916555789ull}},
{{5210120990349456260ull, 446059133244631ull}},
{{478747977537654685ull, 356847306595705ull}},
{{382998382030123748ull, 285477845276564ull}},
{{7991495040732018643ull, 456764552442502ull}},
{{17461242476811345884ull, 365411641954001ull}},
{{10279645166707166384ull, 292329313563201ull}},
{{9068734637247645568ull, 467726901701122ull}},
{{18323034154023847424ull, 374181521360897ull}},
{{7279729693735257293ull, 299345217088718ull}},
{{7958218695234501345ull, 478952347341949ull}},
{{10055923770929511399ull, 383161877873559ull}},
{{11734087831485519443ull, 306529502298847ull}},
{{4017145271409189816ull, 490447203678156ull}},
{{17971111476094993145ull, 392357762942524ull}},
{{18066237995617904839ull, 313886210354019ull}},
{{17837934348762916774ull, 502217936566431ull}},
{{10580998664268423096ull, 401774349253145ull}},
{{8464798931414738476ull, 321419479402516ull}},
{{6164980660779760916ull, 514271167044026ull}},
{{1242635713881898410ull, 411416933635221ull}},
{{15751503830073160020ull, 329133546908176ull}},
{{17823708498633235386ull, 526613675053082ull}},
{{6880269169422767663ull, 421290940042466ull}},
{{1814866520796303807ull, 337032752033973ull}},
{{17661181692241727384ull, 539252403254356ull}},
{{10439596539051471584ull, 431401922603485ull}},
{{8351677231241177267ull, 345121538082788ull}},
{{9673334755243973304ull, 552194460932461ull}},
{{4049318989453268320ull, 441755568745969ull}},
{{6928804006304524979ull, 353404454996775ull}},
{{5543043205043619983ull, 282723563997420ull}},
{{8868869128069791974ull, 452357702395872ull}},
{{18163141746681564548ull, 361886161916697ull}},
{{7151815767861430992ull, 289508929533358ull}},
{{7753556413836379265ull, 463214287253373ull}},
{{13581542760552924058ull, 370571429802698ull}},
{{18243931837926159893ull, 296457143842158ull}},
{{7054198052230393889ull, 474331430147454ull}},
{{9332707256526225435ull, 379465144117963ull}},
{{14844863434704800994ull, 303572115294370ull}},
{{5305037421818129975ull, 485715384470993ull}},
{{11622727566938324626ull, 388572307576794ull}},
{{12987530868292570024ull, 310857846061435ull}},
{{2333305315558560423ull, 497372553698297ull}},
{{12934690696672579308ull, 397898042958637ull}},
{{2969054927854242800ull, 318318434366910ull}},
{{4750487884566788480ull, 509309494987056ull}},
{{111041492911520460ull, 407447595989645ull}},
{{88833194329216368ull, 325958076791716ull}},
{{11210179555152477159ull, 521532922866745ull}},
{{8968143644121981727ull, 417226338293396ull}},
{{3485166100555675058ull, 333781070634717ull}},
{{9265614575630990417ull, 534049713015547ull}},
{{33794031020971687ull, 427239770412438ull}},
{{7405732854300597996ull, 341791816329950ull}},
{{11849172566880956794ull, 546866906127920ull}},
{{9479338053504765435ull, 437493524902336ull}},
{{3894121628061902025ull, 349994819921869ull}},
{{13609292234382863886ull, 559991711874990ull}},
{{10887433787506291109ull, 447993369499992ull}},
{{1331249400521212241ull, 358394695599994ull}},
{{4754348335158880116ull, 286715756479995ull}},
{{7606957336254208185ull, 458745210367992ull}},
{{17153612313229097518ull, 366996168294393ull}},
{{2654843406357547044ull, 293596934635515ull}},
{{4247749450172075271ull, 469755095416824ull}},
{{7087548374879570540ull, 375804076333459ull}},
{{9359387514645566755ull, 300643261066767ull}},
{{217624764465265516ull, 481029217706828ull}},
{{7552797441056033059ull, 384823374165462ull}},
{{17110284397070557417ull, 307858699332369ull}},
{{16308408591087160897ull, 492573918931791ull}},
{{9357378058127818395ull, 394059135145433ull}},
{{14864600075986075362ull, 315247308116346ull}},
{{16404662492093899933ull, 504395692986154ull}},
{{16813078808417030270ull, 403516554388923ull}},
{{2382416602507893246ull, 322813243511139ull}},
{{11190564193496449840ull, 516501189617822ull}},
{{1573753725313339226ull, 413200951694258ull}},
{{8637700609734492027ull, 330560761355406ull}},
{{6441623346091366597ull, 528897218168650ull}},
{{5153298676873093277ull, 423117774534920ull}},
{{4122638941498474622ull, 338494219627936ull}},
{{17664268750623290365ull, 541590751404697ull}},
{{6752717371014811645ull, 433272601123758ull}},
{{12780871526295669963ull, 346618080899006ull}},
{{13070696812589251294ull, 554588929438410ull}},
{{10456557450071401035ull, 443671143550728ull}},
{{15743943589540941474ull, 354936914840582ull}},
{{5216457242148932533ull, 283949531872466ull}},
{{967633957954471407ull, 454319250995946ull}},
{{15531502425331218418ull, 363455400796756ull}},
{{8735853125523064411ull, 290764320637405ull}},
{{13977365000836903058ull, 465222913019848ull}},
{{113845556443791477ull, 372178330415879ull}},
{{3780425259896943505ull, 297742664332703ull}},
{{2359331601093199284ull, 476388262932325ull}},
{{1887465280874559427ull, 381110610345860ull}},
{{1509972224699647542ull, 304888488276688ull}},
{{17173350818487077360ull, 487821581242700ull}},
{{13738680654789661888ull, 390257264994160ull}},
{{10990944523831729510ull, 312205811995328ull}},
{{13896162423388856894ull, 499529299192525ull}},
{{11116929938711085515ull, 399623439354020ull}},
{{8893543950968868412ull, 319698751483216ull}},
{{6850972692066368813ull, 511518002373146ull}},
{{1791429338911184727ull, 409214401898517ull}},
{{12501189915354678751ull, 327371521518813ull}},
{{16312555049825575679ull, 523794434430101ull}},
{{9360695225118550220ull, 419035547544081ull}},
{{3799207365352929852ull, 335228438035265ull}},
{{6078731784564687764ull, 536365500856424ull}},
{{8552334242393660534ull, 429092400685139ull}},
{{10531216208656838751ull, 343273920548111ull}},
{{9471248304367121355ull, 549238272876978ull}},
{{14955696272977517730ull, 439390618301582ull}},
{{4585859388898193538ull, 351512494641266ull}},
{{18405421466462840630ull, 562419991426025ull}},
{{14724337173170272504ull, 449935993140820ull}},
{{11779469738536218003ull, 359948794512656ull}},
{{5734226976087064079ull, 287959035610125ull}},
{{9174763161739302527ull, 460734456976200ull}},
{{7339810529391442021ull, 368587565580960ull}},
{{5871848423513153617ull, 294870052464768ull}},
{{5705608662879135464ull, 471792083943629ull}},
{{8253835745045218695ull, 377433667154903ull}},
{{13981766225519995602ull, 301946933723922ull}},
{{7613430701864351671ull, 483115093958276ull}},
{{2401395746749571013ull, 386492075166621ull}},
{{16678511856367298103ull, 309193660133296ull}},
{{860177266994304703ull, 494709856213275ull}},
{{688141813595443762ull, 395767884970620ull}},
{{550513450876355010ull, 316614307976496ull}},
{{11948867965627898986ull, 506582892762393ull}},
{{16937792001986139835ull, 405266314209914ull}},
{{17239582416330822191ull, 324213051367931ull}},
{{1757890162935943243ull, 518740882188691ull}},
{{16163707389316395887ull, 414992705750952ull}},
{{5552268281969296063ull, 331994164600762ull}},
{{12572978065892784025ull, 531190663361219ull}},
{{13747731267456137543ull, 424952530688975ull}},
{{10998185013964910034ull, 339962024551180ull}},
{{17597096022343856055ull, 543939239281888ull}},
{{3009630373649353875ull, 435151391425511ull}},
{{17165099557887124392ull, 348121113140408ull}},
{{5328066404167937089ull, 556993781024654ull}},
{{7951801938076259994ull, 445595024819723ull}},
{{13740139179944828642ull, 356476019855778ull}},
{{18370808973439683560ull, 285180815884622ull}},
{{14635899098535852403ull, 456289305415396ull}},
{{8019370464086771599ull, 365031444332317ull}},
{{17483542815495148249ull, 292025155465853ull}},
{{5837575616340775259ull, 467240248745366ull}},
{{980711678330709884ull, 373792198996293ull}},
{{8163266972148388553ull, 299033759197034ull}},
{{1993180711211690716ull, 478454014715255ull}},
{{1594544568969352573ull, 382763211772204ull}},
{{4964984469917392381ull, 306210569417763ull}},
{{4254626337125917487ull, 489936911068421ull}},
{{18161096328668375283ull, 391949528854736ull}},
{{10839528248192789903ull, 313559623083789ull}},
{{6275198752882732875ull, 501695396934063ull}},
{{12398856631790006946ull, 401356317547250ull}},
{{9919085305432005557ull, 321085054037800ull}},
{{15870536488691208892ull, 513736086460480ull}},
{{12696429190952967113ull, 410988869168384ull}},
{{13846492167504284014ull, 328791095334707ull}},
{{7396992209039213129ull, 526065752535532ull}},
{{16985640211457101473ull, 420852602028425ull}},
{{13588512169165681178ull, 336682081622740ull}},
{{3294875396955538270ull, 538691330596385ull}},
{{2635900317564430616ull, 430953064477108ull}},
{{9487417883535365139ull, 344762451581686ull}},
{{7801170984172763576ull, 551619922530698ull}},
{{13619634416822031507ull, 441295938024558ull}},
{{18274405162941445852ull, 353036750419646ull}},
{{10930175315611246358ull, 282429400335717ull}},
{{2730885246010352881ull, 451887040537148ull}},
{{9563405826292102951ull, 361509632429718ull}},
{{15029422290517503007ull, 289207705943774ull}},
{{12979029220602273842ull, 462732329510039ull}},
{{14072572191223729397ull, 370185863608031ull}},
{{7568708938237073194ull, 296148690886425ull}},
{{12109934301179317111ull, 473837905418280ull}},
{{9687947440943453688ull, 379070324334624ull}},
{{11439706767496673274ull, 303256259467699ull}},
{{7235484383768946269ull, 485210015148319ull}},
{{9477736321757067338ull, 388168012118655ull}},
{{7582189057405653870ull, 310534409694924ull}},
{{1063456047623315223ull, 496855055511879ull}},
{{4540113652840562502ull, 397484044409503ull}},
{{11010788551756270648ull, 317987235527602ull}},
{{2859866423842391744ull, 508779576844164ull}},
{{5977241953815823718ull, 407023661475331ull}},
{{1092444748310748651ull, 325618929180265ull}},
{{1747911597297197842ull, 520990286688424ull}},
{{5087678092579668597ull, 416792229350739ull}},
{{7759491288805645201ull, 333433783480591ull}},
{{5036488432605211675ull, 533494053568946ull}},
{{339841931342259017ull, 426795242855157ull}},
{{11339919989299538183ull, 341436194284125ull}},
{{18143871982879261093ull, 546297910854600ull}},
{{14515097586303408874ull, 437038328683680ull}},
{{11612078069042727099ull, 349630662946944ull}},
{{7511278466242632389ull, 559409060715111ull}},
{{2319673958252195588ull, 447527248572089ull}},
{{5545087981343666794ull, 358021798857671ull}},
{{746721570333023112ull, 286417439086137ull}},
{{4884103327274747302ull, 458267902537819ull}},
{{7596631476561708165ull, 366614322030255ull}},
{{6077305181249366532ull, 293291457624204ull}},
{{17102385919482807098ull, 469266332198726ull}},
{{9992559920844335355ull, 375413065758981ull}},
{{4304699121933557960ull, 300330452607185ull}},
{{6887518595093692737ull, 480528724171496ull}},
{{1820666061333043866ull, 384422979337197ull}},
{{12524579293292166063ull, 307538383469757ull}},
{{5281931610299824408ull, 492061413551612ull}},
{{15293591732465590496ull, 393649130841289ull}},
{{15924222200714382720ull, 314919304673031ull}},
{{18100057891659191705ull, 503870887476850ull}},
{{14480046313327353364ull, 403096709981480ull}},
{{11584037050661882691ull, 322477367985184ull}},
{{7466412836833281337ull, 515963788776295ull}},
{{5973130269466625069ull, 412771031021036ull}},
{{1089155400831389732ull, 330216824816829ull}},
{{9121346270814044218ull, 528346919706926ull}},
{{3607728201909325051ull, 422677535765541ull}},
{{17643577820495101334ull, 338142028612432ull}},
{{13472329253824520841ull, 541027245779892ull}},
{{3399165773575796026ull, 432821796623914ull}},
{{6408681433602547144ull, 346257437299131ull}},
{{2875192664280254785ull, 554011899678610ull}},
{{2300154131424203828ull, 443209519742888ull}},
{{9218820934623183708ull, 354567615794310ull}},
{{7375056747698546967ull, 283654092635448ull}},
{{8110741981575764824ull, 453846548216717ull}},
{{17556640029486342828ull, 363077238573373ull}},
{{2977265579363343293ull, 290461790858699ull}},
{{12142322556465169916ull, 464738865373918ull}},
{{17092555674655956579ull, 371791092299134ull}},
{{17363393354466675586ull, 297432873839307ull}},
{{13024034108179039645ull, 475892598142892ull}},
{{3040529657059411070ull, 380714078514314ull}},
{{6121772540389439179ull, 304571262811451ull}},
{{2416138435139282040ull, 487314020498322ull}},
{{13000957192337156602ull, 389851216398657ull}},
{{3022068124385904635ull, 311880973118926ull}},
{{15903355443243178386ull, 499009556990281ull}},
{{9033335539852632385ull, 399207645592225ull}},
{{7226668431882105908ull, 319366116473780ull}},
{{11562669491011369453ull, 510985786358048ull}},
{{16628833222292916209ull, 408788629086438ull}},
{{2235020133608601997ull, 327030903269151ull}},
{{14644078657999494166ull, 523249445230641ull}},
{{8025914111657685009ull, 418599556184513ull}},
{{13799428918809968654ull, 334879644947610ull}},
{{3632342196386398230ull, 535807431916177ull}}
};

static const int exponents_binary80[] = { -65,
  -62,
  -59,
  -55,
  -52,
  -49,
  -45,
  -42,
  -39,
  -35,
  -32,
  -29,
  -25,
  -22,
  -19,
  -15,
  -12,
  -9,
  -6,
  -2,
  1,
  4,
  8,
  11,
  14,
  18,
  21,
  24,
  28,
  31,
  34,
  38,
  41,
  44,
  48,
  51,
  54,
  58,
  61,
  64,
  68,
  71,
  74,
  78,
  81,
  84,
  87,
  91,
  94,
  97,
  101,
  104,
  107,
  111,
  114,
  117,
  121,
  124,
  127,
  131,
  134,
  137,
  141,
  144,
  147,
  151,
  154,
  157,
  161,
  164,
  167,
  171,
  174,
  177,
  181,
  184,
  187,
  190,
  194,
  197,
  200,
  204,
  207,
  210,
  214,
  217,
  220,
  224,
  227,
  230,
  234,
  237,
  240,
  244,
  247,
  250,
  254,
  257,
  260,
  264,
  267,
  270,
  274,
  277,
  280,
  283,
  287,
  290,
  293,
  297,
  300,
  303,
  307,
  310,
  313,
  317,
  320,
  323,
  327,
  330,
  333,
  337,
  340,
  343,
  347,
  350,
  353,
  357,
  360,
  363,
  367,
  370,
  373,
  377,
  380,
  383,
  386,
  390,
  393,
  396,
  400,
  403,
  406,
  410,
  413,
  416,
  420,
  423,
  426,
  430,
  433,
  436,
  440,
  443,
  446,
  450,
  453,
  456,
  460,
  463,
  466,
  470,
  473,
  476,
  479,
  483,
  486,
  489,
  493,
  496,
  499,
  503,
  506,
  509,
  513,
  516,
  519,
  523,
  526,
  529,
  533,
  536,
  539,
  543,
  546,
  549,
  553,
  556,
  559,
  563,
  566,
  569,
  572,
  576,
  579,
  582,
  586,
  589,
  592,
  596,
  599,
  602,
  606,
  609,
  612,
  616,
  619,
  622,
  626,
  629,
  632,
  636,
  639,
  642,
  646,
  649,
  652,
  656,
  659,
  662,
  666,
  669,
  672,
  675,
  679,
  682,
  685,
  689,
  692,
  695,
  699,
  702,
  705,
  709,
  712,
  715,
  719,
  722,
  725,
  729,
  732,
  735,
  739,
  742,
  745,
  749,
  752,
  755,
  759,
  762,
  765,
  768,
  772,
  775,
  778,
  782,
  785,
  788,
  792,
  795,
  798,
  802,
  805,
  808,
  812,
  815,
  818,
  822,
  825,
  828,
  832,
  835,
  838,
  842,
  845,
  848,
  852,
  855,
  858,
  862,
  865,
  868,
  871,
  875,
  878,
  881,
  885,
  888,
  891,
  895,
  898,
  901,
  905,
  908,
  911,
  915,
  918,
  921,
  925,
  928,
  931,
  935,
  938,
  941,
  945,
  948,
  951,
  955,
  958,
  961,
  964,
  968,
  971,
  974,
  978,
  981,
  984,
  988,
  991,
  994,
  998,
  1001,
  1004,
  1008,
  1011,
  1014,
  1018,
  1021,
  1024,
  1028,
  1031,
  1034,
  1038,
  1041,
  1044,
  1048,
  1051,
  1054,
  1058,
  1061,
  1064,
  1067,
  1071,
  1074,
  1077,
  1081,
  1084,
  1087,
  1091,
  1094,
  1097,
  1101,
  1104,
  1107,
  1111,
  1114,
  1117,
  1121,
  1124,
  1127,
  1131,
  1134,
  1137,
  1141,
  1144,
  1147,
  1151,
  1154,
  1157,
  1160,
  1164,
  1167,
  1170,
  1174,
  1177,
  1180,
  1184,
  1187,
  1190,
  1194,
  1197,
  1200,
  1204,
  1207,
  1210,
  1214,
  1217,
  1220,
  1224,
  1227,
  1230,
  1234,
  1237,
  1240,
  1244,
  1247,
  1250,
  1253,
  1257,
  1260,
  1263,
  1267,
  1270,
  1273,
  1277,
  1280,
  1283,
  1287,
  1290,
  1293,
  1297,
  1300,
  1303,
  1307,
  1310,
  1313,
  1317,
  1320,
  1323,
  1327,
  1330,
  1333,
  1337,
  1340,
  1343,
  1347,
  1350,
  1353,
  1356,
  1360,
  1363,
  1366,
  1370,
  1373,
  1376,
  1380,
  1383,
  1386,
  1390,
  1393,
  1396,
  1400,
  1403,
  1406,
  1410,
  1413,
  1416,
  1420,
  1423,
  1426,
  1430,
  1433,
  1436,
  1440,
  1443,
  1446,
  1449,
  1453,
  1456,
  1459,
  1463,
  1466,
  1469,
  1473,
  1476,
  1479,
  1483,
  1486,
  1489,
  1493,
  1496,
  1499,
  1503,
  1506,
  1509,
  1513,
  1516,
  1519,
  1523,
  1526,
  1529,
  1533,
  1536,
  1539,
  1543,
  1546,
  1549,
  1552,
  1556,
  1559,
  1562,
  1566,
  1569,
  1572,
  1576,
  1579,
  1582,
  1586,
  1589,
  1592,
  1596,
  1599,
  1602,
  1606,
  1609,
  1612,
  1616,
  1619,
  1622,
  1626,
  1629,
  1632,
  1636,
  1639,
  1642,
  1645,
  1649,
  1652,
  1655,
  1659,
  1662,
  1665,
  1669,
  1672,
  1675,
  1679,
  1682,
  1685,
  1689,
  1692,
  1695,
  1699,
  1702,
  1705,
  1709,
  1712,
  1715,
  1719,
  1722,
  1725,
  1729,
  1732,
  1735,
  1738,
  1742,
  1745,
  1748,
  1752,
  1755,
  1758,
  1762,
  1765,
  1768,
  1772,
  1775,
  1778,
  1782,
  1785,
  1788,
  1792,
  1795,
  1798,
  1802,
  1805,
  1808,
  1812,
  1815,
  1818,
  1822,
  1825,
  1828,
  1832,
  1835,
  1838,
  1841,
  1845,
  1848,
  1851,
  1855,
  1858,
  1861,
  1865,
  1868,
  1871,
  1875,
  1878,
  1881,
  1885,
  1888,
  1891,
  1895,
  1898,
  1901,
  1905,
  1908,
  1911,
  1915,
  1918,
  1921,
  1925,
  1928,
  1931,
  1934,
  1938,
  1941,
  1944,
  1948,
  1951,
  1954,
  1958,
  1961,
  1964,
  1968,
  1971,
  1974,
  1978,
  1981,
  1984,
  1988,
  1991,
  1994,
  1998,
  2001,
  2004,
  2008,
  2011,
  2014,
  2018,
  2021,
  2024,
  2028,
  2031,
  2034,
  2037,
  2041,
  2044,
  2047,
  2051,
  2054,
  2057,
  2061,
  2064,
  2067,
  2071,
  2074,
  2077,
  2081,
  2084,
  2087,
  2091,
  2094,
  2097,
  2101,
  2104,
  2107,
  2111,
  2114,
  2117,
  2121,
  2124,
  2127,
  2130,
  2134,
  2137,
  2140,
  2144,
  2147,
  2150,
  2154,
  2157,
  2160,
  2164,
  2167,
  2170,
  2174,
  2177,
  2180,
  2184,
  2187,
  2190,
  2194,
  2197,
  2200,
  2204,
  2207,
  2210,
  2214,
  2217,
  2220,
  2223,
  2227,
  2230,
  2233,
  2237,
  2240,
  2243,
  2247,
  2250,
  2253,
  2257,
  2260,
  2263,
  2267,
  2270,
  2273,
  2277,
  2280,
  2283,
  2287,
  2290,
  2293,
  2297,
  2300,
  2303,
  2307,
  2310,
  2313,
  2317,
  2320,
  2323,
  2326,
  2330,
  2333,
  2336,
  2340,
  2343,
  2346,
  2350,
  2353,
  2356,
  2360,
  2363,
  2366,
  2370,
  2373,
  2376,
  2380,
  2383,
  2386,
  2390,
  2393,
  2396,
  2400,
  2403,
  2406,
  2410,
  2413,
  2416,
  2419,
  2423,
  2426,
  2429,
  2433,
  2436,
  2439,
  2443,
  2446,
  2449,
  2453,
  2456,
  2459,
  2463,
  2466,
  2469,
  2473,
  2476,
  2479,
  2483,
  2486,
  2489,
  2493,
  2496,
  2499,
  2503,
  2506,
  2509,
  2513,
  2516,
  2519,
  2522,
  2526,
  2529,
  2532,
  2536,
  2539,
  2542,
  2546,
  2549,
  2552,
  2556,
  2559,
  2562,
  2566,
  2569,
  2572,
  2576,
  2579,
  2582,
  2586,
  2589,
  2592,
  2596,
  2599,
  2602,
  2606,
  2609,
  2612,
  2615,
  2619,
  2622,
  2625,
  2629,
  2632,
  2635,
  2639,
  2642,
  2645,
  2649,
  2652,
  2655,
  2659,
  2662,
  2665,
  2669,
  2672,
  2675,
  2679,
  2682,
  2685,
  2689,
  2692,
  2695,
  2699,
  2702,
  2705,
  2708,
  2712,
  2715,
  2718,
  2722,
  2725,
  2728,
  2732,
  2735,
  2738,
  2742,
  2745,
  2748,
  2752,
  2755,
  2758,
  2762,
  2765,
  2768,
  2772,
  2775,
  2778,
  2782,
  2785,
  2788,
  2792,
  2795,
  2798,
  2802,
  2805,
  2808,
  2811,
  2815,
  2818,
  2821,
  2825,
  2828,
  2831,
  2835,
  2838,
  2841,
  2845,
  2848,
  2851,
  2855,
  2858,
  2861,
  2865,
  2868,
  2871,
  2875,
  2878,
  2881,
  2885,
  2888,
  2891,
  2895,
  2898,
  2901,
  2904,
  2908,
  2911,
  2914,
  2918,
  2921,
  2924,
  2928,
  2931,
  2934,
  2938,
  2941,
  2944,
  2948,
  2951,
  2954,
  2958,
  2961,
  2964,
  2968,
  2971,
  2974,
  2978,
  2981,
  2984,
  2988,
  2991,
  2994,
  2998,
  3001,
  3004,
  3007,
  3011,
  3014,
  3017,
  3021,
  3024,
  3027,
  3031,
  3034,
  3037,
  3041,
  3044,
  3047,
  3051,
  3054,
  3057,
  3061,
  3064,
  3067,
  3071,
  3074,
  3077,
  3081,
  3084,
  3087,
  3091,
  3094,
  3097,
  3100,
  3104,
  3107,
  3110,
  3114,
  3117,
  3120,
  3124,
  3127,
  3130,
  3134,
  3137,
  3140,
  3144,
  3147,
  3150,
  3154,
  3157,
  3160,
  3164,
  3167,
  3170,
  3174,
  3177,
  3180,
  3184,
  3187,
  3190,
  3193,
  3197,
  3200,
  3203,
  3207,
  3210,
  3213,
  3217,
  3220,
  3223,
  3227,
  3230,
  3233,
  3237,
  3240,
  3243,
  3247,
  3250,
  3253,
  3257,
  3260,
  3263,
  3267,
  3270,
  3273,
  3277,
  3280,
  3283,
  3287,
  3290,
  3293,
  3296,
  3300,
  3303,
  3306,
  3310,
  3313,
  3316,
  3320,
  3323,
  3326,
  3330,
  3333,
  3336,
  3340,
  3343,
  3346,
  3350,
  3353,
  3356,
  3360,
  3363,
  3366,
  3370,
  3373,
  3376,
  3380,
  3383,
  3386,
  3389,
  3393,
  3396,
  3399,
  3403,
  3406,
  3409,
  3413,
  3416,
  3419,
  3423,
  3426,
  3429,
  3433,
  3436,
  3439,
  3443,
  3446,
  3449,
  3453,
  3456,
  3459,
  3463,
  3466,
  3469,
  3473,
  3476,
  3479,
  3483,
  3486,
  3489,
  3492,
  3496,
  3499,
  3502,
  3506,
  3509,
  3512,
  3516,
  3519,
  3522,
  3526,
  3529,
  3532,
  3536,
  3539,
  3542,
  3546,
  3549,
  3552,
  3556,
  3559,
  3562,
  3566,
  3569,
  3572,
  3576,
  3579,
  3582,
  3585,
  3589,
  3592,
  3595,
  3599,
  3602,
  3605,
  3609,
  3612,
  3615,
  3619,
  3622,
  3625,
  3629,
  3632,
  3635,
  3639,
  3642,
  3645,
  3649,
  3652,
  3655,
  3659,
  3662,
  3665,
  3669,
  3672,
  3675,
  3679,
  3682,
  3685,
  3688,
  3692,
  3695,
  3698,
  3702,
  3705,
  3708,
  3712,
  3715,
  3718,
  3722,
  3725,
  3728,
  3732,
  3735,
  3738,
  3742,
  3745,
  3748,
  3752,
  3755,
  3758,
  3762,
  3765,
  3768,
  3772,
  3775,
  3778,
  3781,
  3785,
  3788,
  3791,
  3795,
  3798,
  3801,
  3805,
  3808,
  3811,
  3815,
  3818,
  3821,
  3825,
  3828,
  3831,
  3835,
  3838,
  3841,
  3845,
  3848,
  3851,
  3855,
  3858,
  3861,
  3865,
  3868,
  3871,
  3874,
  3878,
  3881,
  3884,
  3888,
  3891,
  3894,
  3898,
  3901,
  3904,
  3908,
  3911,
  3914,
  3918,
  3921,
  3924,
  3928,
  3931,
  3934,
  3938,
  3941,
  3944,
  3948,
  3951,
  3954,
  3958,
  3961,
  3964,
  3968,
  3971,
  3974,
  3977,
  3981,
  3984,
  3987,
  3991,
  3994,
  3997,
  4001,
  4004,
  4007,
  4011,
  4014,
  4017,
  4021,
  4024,
  4027,
  4031,
  4034,
  4037,
  4041,
  4044,
  4047,
  4051,
  4054,
  4057,
  4061,
  4064,
  4067,
  4070,
  4074,
  4077,
  4080,
  4084,
  4087,
  4090,
  4094,
  4097,
  4100,
  4104,
  4107,
  4110,
  4114,
  4117,
  4120,
  4124,
  4127,
  4130,
  4134,
  4137,
  4140,
  4144,
  4147,
  4150,
  4154,
  4157,
  4160,
  4164,
  4167,
  4170,
  4173,
  4177,
  4180,
  4183,
  4187,
  4190,
  4193,
  4197,
  4200,
  4203,
  4207,
  4210,
  4213,
  4217,
  4220,
  4223,
  4227,
  4230,
  4233,
  4237,
  4240,
  4243,
  4247,
  4250,
  4253,
  4257,
  4260,
  4263,
  4266,
  4270,
  4273,
  4276,
  4280,
  4283,
  4286,
  4290,
  4293,
  4296,
  4300,
  4303,
  4306,
  4310,
  4313,
  4316,
  4320,
  4323,
  4326,
  4330,
  4333,
  4336,
  4340,
  4343,
  4346,
  4350,
  4353,
  4356,
  4359,
  4363,
  4366,
  4369,
  4373,
  4376,
  4379,
  4383,
  4386,
  4389,
  4393,
  4396,
  4399,
  4403,
  4406,
  4409,
  4413,
  4416,
  4419,
  4423,
  4426,
  4429,
  4433,
  4436,
  4439,
  4443,
  4446,
  4449,
  4453,
  4456,
  4459,
  4462,
  4466,
  4469,
  4472,
  4476,
  4479,
  4482,
  4486,
  4489,
  4492,
  4496,
  4499,
  4502,
  4506,
  4509,
  4512,
  4516,
  4519,
  4522,
  4526,
  4529,
  4532,
  4536,
  4539,
  4542,
  4546,
  4549,
  4552,
  4555,
  4559,
  4562,
  4565,
  4569,
  4572,
  4575,
  4579,
  4582,
  4585,
  4589,
  4592,
  4595,
  4599,
  4602,
  4605,
  4609,
  4612,
  4615,
  4619,
  4622,
  4625,
  4629,
  4632,
  4635,
  4639,
  4642,
  4645,
  4649,
  4652,
  4655,
  4658,
  4662,
  4665,
  4668,
  4672,
  4675,
  4678,
  4682,
  4685,
  4688,
  4692,
  4695,
  4698,
  4702,
  4705,
  4708,
  4712,
  4715,
  4718,
  4722,
  4725,
  4728,
  4732,
  4735,
  4738,
  4742,
  4745,
  4748,
  4751,
  4755,
  4758,
  4761,
  4765,
  4768,
  4771,
  4775,
  4778,
  4781,
  4785,
  4788,
  4791,
  4795,
  4798,
  4801,
  4805,
  4808,
  4811,
  4815,
  4818,
  4821,
  4825,
  4828,
  4831,
  4835,
  4838,
  4841,
  4844,
  4848,
  4851,
  4854,
  4858,
  4861,
  4864,
  4868,
  4871,
  4874,
  4878,
  4881,
  4884,
  4888,
  4891,
  4894,
  4898,
  4901,
  4904,
  4908,
  4911,
  4914,
  4918,
  4921,
  4924,
  4928,
  4931,
  4934,
  4938,
  4941,
  4944,
  4947,
  4951,
  4954,
  4957,
  4961,
  4964,
  4967,
  4971,
  4974,
  4977,
  4981,
  4984,
  4987,
  4991,
  4994,
  4997,
  5001,
  5004,
  5007,
  5011,
  5014,
  5017,
  5021,
  5024,
  5027,
  5031,
  5034,
  5037,
  5040,
  5044,
  5047,
  5050,
  5054,
  5057,
  5060,
  5064,
  5067,
  5070,
  5074,
  5077,
  5080,
  5084,
  5087,
  5090,
  5094,
  5097,
  5100,
  5104,
  5107,
  5110,
  5114,
  5117,
  5120,
  5124,
  5127,
  5130,
  5134,
  5137,
  5140,
  5143,
  5147,
  5150,
  5153,
  5157,
  5160,
  5163,
  5167,
  5170,
  5173,
  5177,
  5180,
  5183,
  5187,
  5190,
  5193,
  5197,
  5200,
  5203,
  5207,
  5210,
  5213,
  5217,
  5220,
  5223,
  5227,
  5230,
  5233,
  5236,
  5240,
  5243,
  5246,
  5250,
  5253,
  5256,
  5260,
  5263,
  5266,
  5270,
  5273,
  5276,
  5280,
  5283,
  5286,
  5290,
  5293,
  5296,
  5300,
  5303,
  5306,
  5310,
  5313,
  5316,
  5320,
  5323,
  5326,
  5329,
  5333,
  5336,
  5339,
  5343,
  5346,
  5349,
  5353,
  5356,
  5359,
  5363,
  5366,
  5369,
  5373,
  5376,
  5379,
  5383,
  5386,
  5389,
  5393,
  5396,
  5399,
  5403,
  5406,
  5409,
  5413,
  5416,
  5419,
  5423,
  5426,
  5429,
  5432,
  5436,
  5439,
  5442,
  5446,
  5449,
  5452,
  5456,
  5459,
  5462,
  5466,
  5469,
  5472,
  5476,
  5479,
  5482,
  5486,
  5489,
  5492,
  5496,
  5499,
  5502,
  5506,
  5509,
  5512,
  5516,
  5519,
  5522,
  5525,
  5529,
  5532,
  5535,
  5539,
  5542,
  5545,
  5549,
  5552,
  5555,
  5559,
  5562,
  5565,
  5569,
  5572,
  5575,
  5579,
  5582,
  5585,
  5589,
  5592,
  5595,
  5599,
  5602,
  5605,
  5609,
  5612,
  5615,
  5619,
  5622,
  5625,
  5628,
  5632,
  5635,
  5638,
  5642,
  5645,
  5648,
  5652,
  5655,
  5658,
  5662,
  5665,
  5668,
  5672,
  5675,
  5678,
  5682,
  5685,
  5688,
  5692,
  5695,
  5698,
  5702,
  5705,
  5708,
  5712,
  5715,
  5718,
  5721,
  5725,
  5728,
  5731,
  5735,
  5738,
  5741,
  5745,
  5748,
  5751,
  5755,
  5758,
  5761,
  5765,
  5768,
  5771,
  5775,
  5778,
  5781,
  5785,
  5788,
  5791,
  5795,
  5798,
  5801,
  5805,
  5808,
  5811,
  5815,
  5818,
  5821,
  5824,
  5828,
  5831,
  5834,
  5838,
  5841,
  5844,
  5848,
  5851,
  5854,
  5858,
  5861,
  5864,
  5868,
  5871,
  5874,
  5878,
  5881,
  5884,
  5888,
  5891,
  5894,
  5898,
  5901,
  5904,
  5908,
  5911,
  5914,
  5917,
  5921,
  5924,
  5927,
  5931,
  5934,
  5937,
  5941,
  5944,
  5947,
  5951,
  5954,
  5957,
  5961,
  5964,
  5967,
  5971,
  5974,
  5977,
  5981,
  5984,
  5987,
  5991,
  5994,
  5997,
  6001,
  6004,
  6007,
  6010,
  6014,
  6017,
  6020,
  6024,
  6027,
  6030,
  6034,
  6037,
  6040,
  6044,
  6047,
  6050,
  6054,
  6057,
  6060,
  6064,
  6067,
  6070,
  6074,
  6077,
  6080,
  6084,
  6087,
  6090,
  6094,
  6097,
  6100,
  6104,
  6107,
  6110,
  6113,
  6117,
  6120,
  6123,
  6127,
  6130,
  6133,
  6137,
  6140,
  6143,
  6147,
  6150,
  6153,
  6157,
  6160,
  6163,
  6167,
  6170,
  6173,
  6177,
  6180,
  6183,
  6187,
  6190,
  6193,
  6197,
  6200,
  6203,
  6206,
  6210,
  6213,
  6216,
  6220,
  6223,
  6226,
  6230,
  6233,
  6236,
  6240,
  6243,
  6246,
  6250,
  6253,
  6256,
  6260,
  6263,
  6266,
  6270,
  6273,
  6276,
  6280,
  6283,
  6286,
  6290,
  6293,
  6296,
  6300,
  6303,
  6306,
  6309,
  6313,
  6316,
  6319,
  6323,
  6326,
  6329,
  6333,
  6336,
  6339,
  6343,
  6346,
  6349,
  6353,
  6356,
  6359,
  6363,
  6366,
  6369,
  6373,
  6376,
  6379,
  6383,
  6386,
  6389,
  6393,
  6396,
  6399,
  6402,
  6406,
  6409,
  6412,
  6416,
  6419,
  6422,
  6426,
  6429,
  6432,
  6436,
  6439,
  6442,
  6446,
  6449,
  6452,
  6456,
  6459,
  6462,
  6466,
  6469,
  6472,
  6476,
  6479,
  6482,
  6486,
  6489,
  6492,
  6495,
  6499,
  6502,
  6505,
  6509,
  6512,
  6515,
  6519,
  6522,
  6525,
  6529,
  6532,
  6535,
  6539,
  6542,
  6545,
  6549,
  6552,
  6555,
  6559,
  6562,
  6565,
  6569,
  6572,
  6575,
  6579,
  6582,
  6585,
  6589,
  6592,
  6595,
  6598,
  6602,
  6605,
  6608,
  6612,
  6615,
  6618,
  6622,
  6625,
  6628,
  6632,
  6635,
  6638,
  6642,
  6645,
  6648,
  6652,
  6655,
  6658,
  6662,
  6665,
  6668,
  6672,
  6675,
  6678,
  6682,
  6685,
  6688,
  6691,
  6695,
  6698,
  6701,
  6705,
  6708,
  6711,
  6715,
  6718,
  6721,
  6725,
  6728,
  6731,
  6735,
  6738,
  6741,
  6745,
  6748,
  6751,
  6755,
  6758,
  6761,
  6765,
  6768,
  6771,
  6775,
  6778,
  6781,
  6785,
  6788,
  6791,
  6794,
  6798,
  6801,
  6804,
  6808,
  6811,
  6814,
  6818,
  6821,
  6824,
  6828,
  6831,
  6834,
  6838,
  6841,
  6844,
  6848,
  6851,
  6854,
  6858,
  6861,
  6864,
  6868,
  6871,
  6874,
  6878,
  6881,
  6884,
  6887,
  6891,
  6894,
  6897,
  6901,
  6904,
  6907,
  6911,
  6914,
  6917,
  6921,
  6924,
  6927,
  6931,
  6934,
  6937,
  6941,
  6944,
  6947,
  6951,
  6954,
  6957,
  6961,
  6964,
  6967,
  6971,
  6974,
  6977,
  6980,
  6984,
  6987,
  6990,
  6994,
  6997,
  7000,
  7004,
  7007,
  7010,
  7014,
  7017,
  7020,
  7024,
  7027,
  7030,
  7034,
  7037,
  7040,
  7044,
  7047,
  7050,
  7054,
  7057,
  7060,
  7064,
  7067,
  7070,
  7074,
  7077,
  7080,
  7083,
  7087,
  7090,
  7093,
  7097,
  7100,
  7103,
  7107,
  7110,
  7113,
  7117,
  7120,
  7123,
  7127,
  7130,
  7133,
  7137,
  7140,
  7143,
  7147,
  7150,
  7153,
  7157,
  7160,
  7163,
  7167,
  7170,
  7173,
  7176,
  7180,
  7183,
  7186,
  7190,
  7193,
  7196,
  7200,
  7203,
  7206,
  7210,
  7213,
  7216,
  7220,
  7223,
  7226,
  7230,
  7233,
  7236,
  7240,
  7243,
  7246,
  7250,
  7253,
  7256,
  7260,
  7263,
  7266,
  7270,
  7273,
  7276,
  7279,
  7283,
  7286,
  7289,
  7293,
  7296,
  7299,
  7303,
  7306,
  7309,
  7313,
  7316,
  7319,
  7323,
  7326,
  7329,
  7333,
  7336,
  7339,
  7343,
  7346,
  7349,
  7353,
  7356,
  7359,
  7363,
  7366,
  7369,
  7372,
  7376,
  7379,
  7382,
  7386,
  7389,
  7392,
  7396,
  7399,
  7402,
  7406,
  7409,
  7412,
  7416,
  7419,
  7422,
  7426,
  7429,
  7432,
  7436,
  7439,
  7442,
  7446,
  7449,
  7452,
  7456,
  7459,
  7462,
  7465,
  7469,
  7472,
  7475,
  7479,
  7482,
  7485,
  7489,
  7492,
  7495,
  7499,
  7502,
  7505,
  7509,
  7512,
  7515,
  7519,
  7522,
  7525,
  7529,
  7532,
  7535,
  7539,
  7542,
  7545,
  7549,
  7552,
  7555,
  7559,
  7562,
  7565,
  7568,
  7572,
  7575,
  7578,
  7582,
  7585,
  7588,
  7592,
  7595,
  7598,
  7602,
  7605,
  7608,
  7612,
  7615,
  7618,
  7622,
  7625,
  7628,
  7632,
  7635,
  7638,
  7642,
  7645,
  7648,
  7652,
  7655,
  7658,
  7661,
  7665,
  7668,
  7671,
  7675,
  7678,
  7681,
  7685,
  7688,
  7691,
  7695,
  7698,
  7701,
  7705,
  7708,
  7711,
  7715,
  7718,
  7721,
  7725,
  7728,
  7731,
  7735,
  7738,
  7741,
  7745,
  7748,
  7751,
  7755,
  7758,
  7761,
  7764,
  7768,
  7771,
  7774,
  7778,
  7781,
  7784,
  7788,
  7791,
  7794,
  7798,
  7801,
  7804,
  7808,
  7811,
  7814,
  7818,
  7821,
  7824,
  7828,
  7831,
  7834,
  7838,
  7841,
  7844,
  7848,
  7851,
  7854,
  7857,
  7861,
  7864,
  7867,
  7871,
  7874,
  7877,
  7881,
  7884,
  7887,
  7891,
  7894,
  7897,
  7901,
  7904,
  7907,
  7911,
  7914,
  7917,
  7921,
  7924,
  7927,
  7931,
  7934,
  7937,
  7941,
  7944,
  7947,
  7951,
  7954,
  7957,
  7960,
  7964,
  7967,
  7970,
  7974,
  7977,
  7980,
  7984,
  7987,
  7990,
  7994,
  7997,
  8000,
  8004,
  8007,
  8010,
  8014,
  8017,
  8020,
  8024,
  8027,
  8030,
  8034,
  8037,
  8040,
  8044,
  8047,
  8050,
  8053,
  8057,
  8060,
  8063,
  8067,
  8070,
  8073,
  8077,
  8080,
  8083,
  8087,
  8090,
  8093,
  8097,
  8100,
  8103,
  8107,
  8110,
  8113,
  8117,
  8120,
  8123,
  8127,
  8130,
  8133,
  8137,
  8140,
  8143,
  8146,
  8150,
  8153,
  8156,
  8160,
  8163,
  8166,
  8170,
  8173,
  8176,
  8180,
  8183,
  8186,
  8190,
  8193,
  8196,
  8200,
  8203,
  8206,
  8210,
  8213,
  8216,
  8220,
  8223,
  8226,
  8230,
  8233,
  8236,
  8240,
  8243,
  8246,
  8249,
  8253,
  8256,
  8259,
  8263,
  8266,
  8269,
  8273,
  8276,
  8279,
  8283,
  8286,
  8289,
  8293,
  8296,
  8299,
  8303,
  8306,
  8309,
  8313,
  8316,
  8319,
  8323,
  8326,
  8329,
  8333,
  8336,
  8339,
  8342,
  8346,
  8349,
  8352,
  8356,
  8359,
  8362,
  8366,
  8369,
  8372,
  8376,
  8379,
  8382,
  8386,
  8389,
  8392,
  8396,
  8399,
  8402,
  8406,
  8409,
  8412,
  8416,
  8419,
  8422,
  8426,
  8429,
  8432,
  8436,
  8439,
  8442,
  8445,
  8449,
  8452,
  8455,
  8459,
  8462,
  8465,
  8469,
  8472,
  8475,
  8479,
  8482,
  8485,
  8489,
  8492,
  8495,
  8499,
  8502,
  8505,
  8509,
  8512,
  8515,
  8519,
  8522,
  8525,
  8529,
  8532,
  8535,
  8538,
  8542,
  8545,
  8548,
  8552,
  8555,
  8558,
  8562,
  8565,
  8568,
  8572,
  8575,
  8578,
  8582,
  8585,
  8588,
  8592,
  8595,
  8598,
  8602,
  8605,
  8608,
  8612,
  8615,
  8618,
  8622,
  8625,
  8628,
  8631,
  8635,
  8638,
  8641,
  8645,
  8648,
  8651,
  8655,
  8658,
  8661,
  8665,
  8668,
  8671,
  8675,
  8678,
  8681,
  8685,
  8688,
  8691,
  8695,
  8698,
  8701,
  8705,
  8708,
  8711,
  8715,
  8718,
  8721,
  8725,
  8728,
  8731,
  8734,
  8738,
  8741,
  8744,
  8748,
  8751,
  8754,
  8758,
  8761,
  8764,
  8768,
  8771,
  8774,
  8778,
  8781,
  8784,
  8788,
  8791,
  8794,
  8798,
  8801,
  8804,
  8808,
  8811,
  8814,
  8818,
  8821,
  8824,
  8827,
  8831,
  8834,
  8837,
  8841,
  8844,
  8847,
  8851,
  8854,
  8857,
  8861,
  8864,
  8867,
  8871,
  8874,
  8877,
  8881,
  8884,
  8887,
  8891,
  8894,
  8897,
  8901,
  8904,
  8907,
  8911,
  8914,
  8917,
  8921,
  8924,
  8927,
  8930,
  8934,
  8937,
  8940,
  8944,
  8947,
  8950,
  8954,
  8957,
  8960,
  8964,
  8967,
  8970,
  8974,
  8977,
  8980,
  8984,
  8987,
  8990,
  8994,
  8997,
  9000,
  9004,
  9007,
  9010,
  9014,
  9017,
  9020,
  9023,
  9027,
  9030,
  9033,
  9037,
  9040,
  9043,
  9047,
  9050,
  9053,
  9057,
  9060,
  9063,
  9067,
  9070,
  9073,
  9077,
  9080,
  9083,
  9087,
  9090,
  9093,
  9097,
  9100,
  9103,
  9107,
  9110,
  9113,
  9116,
  9120,
  9123,
  9126,
  9130,
  9133,
  9136,
  9140,
  9143,
  9146,
  9150,
  9153,
  9156,
  9160,
  9163,
  9166,
  9170,
  9173,
  9176,
  9180,
  9183,
  9186,
  9190,
  9193,
  9196,
  9200,
  9203,
  9206,
  9210,
  9213,
  9216,
  9219,
  9223,
  9226,
  9229,
  9233,
  9236,
  9239,
  9243,
  9246,
  9249,
  9253,
  9256,
  9259,
  9263,
  9266,
  9269,
  9273,
  9276,
  9279,
  9283,
  9286,
  9289,
  9293,
  9296,
  9299,
  9303,
  9306,
  9309,
  9312,
  9316,
  9319,
  9322,
  9326,
  9329,
  9332,
  9336,
  9339,
  9342,
  9346,
  9349,
  9352,
  9356,
  9359,
  9362,
  9366,
  9369,
  9372,
  9376,
  9379,
  9382,
  9386,
  9389,
  9392,
  9396,
  9399,
  9402,
  9406,
  9409,
  9412,
  9415,
  9419,
  9422,
  9425,
  9429,
  9432,
  9435,
  9439,
  9442,
  9445,
  9449,
  9452,
  9455,
  9459,
  9462,
  9465,
  9469,
  9472,
  9475,
  9479,
  9482,
  9485,
  9489,
  9492,
  9495,
  9499,
  9502,
  9505,
  9508,
  9512,
  9515,
  9518,
  9522,
  9525,
  9528,
  9532,
  9535,
  9538,
  9542,
  9545,
  9548,
  9552,
  9555,
  9558,
  9562,
  9565,
  9568,
  9572,
  9575,
  9578,
  9582,
  9585,
  9588,
  9592,
  9595,
  9598,
  9601,
  9605,
  9608,
  9611,
  9615,
  9618,
  9621,
  9625,
  9628,
  9631,
  9635,
  9638,
  9641,
  9645,
  9648,
  9651,
  9655,
  9658,
  9661,
  9665,
  9668,
  9671,
  9675,
  9678,
  9681,
  9685,
  9688,
  9691,
  9695,
  9698,
  9701,
  9704,
  9708,
  9711,
  9714,
  9718,
  9721,
  9724,
  9728,
  9731,
  9734,
  9738,
  9741,
  9744,
  9748,
  9751,
  9754,
  9758,
  9761,
  9764,
  9768,
  9771,
  9774,
  9778,
  9781,
  9784,
  9788,
  9791,
  9794,
  9797,
  9801,
  9804,
  9807,
  9811,
  9814,
  9817,
  9821,
  9824,
  9827,
  9831,
  9834,
  9837,
  9841,
  9844,
  9847,
  9851,
  9854,
  9857,
  9861,
  9864,
  9867,
  9871,
  9874,
  9877,
  9881,
  9884,
  9887,
  9891,
  9894,
  9897,
  9900,
  9904,
  9907,
  9910,
  9914,
  9917,
  9920,
  9924,
  9927,
  9930,
  9934,
  9937,
  9940,
  9944,
  9947,
  9950,
  9954,
  9957,
  9960,
  9964,
  9967,
  9970,
  9974,
  9977,
  9980,
  9984,
  9987,
  9990,
  9993,
  9997,
  10000,
  10003,
  10007,
  10010,
  10013,
  10017,
  10020,
  10023,
  10027,
  10030,
  10033,
  10037,
  10040,
  10043,
  10047,
  10050,
  10053,
  10057,
  10060,
  10063,
  10067,
  10070,
  10073,
  10077,
  10080,
  10083,
  10087,
  10090,
  10093,
  10096,
  10100,
  10103,
  10106,
  10110,
  10113,
  10116,
  10120,
  10123,
  10126,
  10130,
  10133,
  10136,
  10140,
  10143,
  10146,
  10150,
  10153,
  10156,
  10160,
  10163,
  10166,
  10170,
  10173,
  10176,
  10180,
  10183,
  10186,
  10189,
  10193,
  10196,
  10199,
  10203,
  10206,
  10209,
  10213,
  10216,
  10219,
  10223,
  10226,
  10229,
  10233,
  10236,
  10239,
  10243,
  10246,
  10249,
  10253,
  10256,
  10259,
  10263,
  10266,
  10269,
  10273,
  10276,
  10279,
  10282,
  10286,
  10289,
  10292,
  10296,
  10299,
  10302,
  10306,
  10309,
  10312,
  10316,
  10319,
  10322,
  10326,
  10329,
  10332,
  10336,
  10339,
  10342,
  10346,
  10349,
  10352,
  10356,
  10359,
  10362,
  10366,
  10369,
  10372,
  10376,
  10379,
  10382,
  10385,
  10389,
  10392,
  10395,
  10399,
  10402,
  10405,
  10409,
  10412,
  10415,
  10419,
  10422,
  10425,
  10429,
  10432,
  10435,
  10439,
  10442,
  10445,
  10449,
  10452,
  10455,
  10459,
  10462,
  10465,
  10469,
  10472,
  10475,
  10478,
  10482,
  10485,
  10488,
  10492,
  10495,
  10498,
  10502,
  10505,
  10508,
  10512,
  10515,
  10518,
  10522,
  10525,
  10528,
  10532,
  10535,
  10538,
  10542,
  10545,
  10548,
  10552,
  10555,
  10558,
  10562,
  10565,
  10568,
  10572,
  10575,
  10578,
  10581,
  10585,
  10588,
  10591,
  10595,
  10598,
  10601,
  10605,
  10608,
  10611,
  10615,
  10618,
  10621,
  10625,
  10628,
  10631,
  10635,
  10638,
  10641,
  10645,
  10648,
  10651,
  10655,
  10658,
  10661,
  10665,
  10668,
  10671,
  10674,
  10678,
  10681,
  10684,
  10688,
  10691,
  10694,
  10698,
  10701,
  10704,
  10708,
  10711,
  10714,
  10718,
  10721,
  10724,
  10728,
  10731,
  10734,
  10738,
  10741,
  10744,
  10748,
  10751,
  10754,
  10758,
  10761,
  10764,
  10767,
  10771,
  10774,
  10777,
  10781,
  10784,
  10787,
  10791,
  10794,
  10797,
  10801,
  10804,
  10807,
  10811,
  10814,
  10817,
  10821,
  10824,
  10827,
  10831,
  10834,
  10837,
  10841,
  10844,
  10847,
  10851,
  10854,
  10857,
  10861,
  10864,
  10867,
  10870,
  10874,
  10877,
  10880,
  10884,
  10887,
  10890,
  10894,
  10897,
  10900,
  10904,
  10907,
  10910,
  10914,
  10917,
  10920,
  10924,
  10927,
  10930,
  10934,
  10937,
  10940,
  10944,
  10947,
  10950,
  10954,
  10957,
  10960,
  10963,
  10967,
  10970,
  10973,
  10977,
  10980,
  10983,
  10987,
  10990,
  10993,
  10997,
  11000,
  11003,
  11007,
  11010,
  11013,
  11017,
  11020,
  11023,
  11027,
  11030,
  11033,
  11037,
  11040,
  11043,
  11047,
  11050,
  11053,
  11057,
  11060,
  11063,
  11066,
  11070,
  11073,
  11076,
  11080,
  11083,
  11086,
  11090,
  11093,
  11096,
  11100,
  11103,
  11106,
  11110,
  11113,
  11116,
  11120,
  11123,
  11126,
  11130,
  11133,
  11136,
  11140,
  11143,
  11146,
  11150,
  11153,
  11156,
  11159,
  11163,
  11166,
  11169,
  11173,
  11176,
  11179,
  11183,
  11186,
  11189,
  11193,
  11196,
  11199,
  11203,
  11206,
  11209,
  11213,
  11216,
  11219,
  11223,
  11226,
  11229,
  11233,
  11236,
  11239,
  11243,
  11246,
  11249,
  11252,
  11256,
  11259,
  11262,
  11266,
  11269,
  11272,
  11276,
  11279,
  11282,
  11286,
  11289,
  11292,
  11296,
  11299,
  11302,
  11306,
  11309,
  11312,
  11316,
  11319,
  11322,
  11326,
  11329,
  11332,
  11336,
  11339,
  11342,
  11346,
  11349,
  11352,
  11355,
  11359,
  11362,
  11365,
  11369,
  11372,
  11375,
  11379,
  11382,
  11385,
  11389,
  11392,
  11395,
  11399,
  11402,
  11405,
  11409,
  11412,
  11415,
  11419,
  11422,
  11425,
  11429,
  11432,
  11435,
  11439,
  11442,
  11445,
  11448,
  11452,
  11455,
  11458,
  11462,
  11465,
  11468,
  11472,
  11475,
  11478,
  11482,
  11485,
  11488,
  11492,
  11495,
  11498,
  11502,
  11505,
  11508,
  11512,
  11515,
  11518,
  11522,
  11525,
  11528,
  11532,
  11535,
  11538,
  11542,
  11545,
  11548,
  11551,
  11555,
  11558,
  11561,
  11565,
  11568,
  11571,
  11575,
  11578,
  11581,
  11585,
  11588,
  11591,
  11595,
  11598,
  11601,
  11605,
  11608,
  11611,
  11615,
  11618,
  11621,
  11625,
  11628,
  11631,
  11635,
  11638,
  11641,
  11644,
  11648,
  11651,
  11654,
  11658,
  11661,
  11664,
  11668,
  11671,
  11674,
  11678,
  11681,
  11684,
  11688,
  11691,
  11694,
  11698,
  11701,
  11704,
  11708,
  11711,
  11714,
  11718,
  11721,
  11724,
  11728,
  11731,
  11734,
  11737,
  11741,
  11744,
  11747,
  11751,
  11754,
  11757,
  11761,
  11764,
  11767,
  11771,
  11774,
  11777,
  11781,
  11784,
  11787,
  11791,
  11794,
  11797,
  11801,
  11804,
  11807,
  11811,
  11814,
  11817,
  11821,
  11824,
  11827,
  11831,
  11834,
  11837,
  11840,
  11844,
  11847,
  11850,
  11854,
  11857,
  11860,
  11864,
  11867,
  11870,
  11874,
  11877,
  11880,
  11884,
  11887,
  11890,
  11894,
  11897,
  11900,
  11904,
  11907,
  11910,
  11914,
  11917,
  11920,
  11924,
  11927,
  11930,
  11933,
  11937,
  11940,
  11943,
  11947,
  11950,
  11953,
  11957,
  11960,
  11963,
  11967,
  11970,
  11973,
  11977,
  11980,
  11983,
  11987,
  11990,
  11993,
  11997,
  12000,
  12003,
  12007,
  12010,
  12013,
  12017,
  12020,
  12023,
  12027,
  12030,
  12033,
  12036,
  12040,
  12043,
  12046,
  12050,
  12053,
  12056,
  12060,
  12063,
  12066,
  12070,
  12073,
  12076,
  12080,
  12083,
  12086,
  12090,
  12093,
  12096,
  12100,
  12103,
  12106,
  12110,
  12113,
  12116,
  12120,
  12123,
  12126,
  12129,
  12133,
  12136,
  12139,
  12143,
  12146,
  12149,
  12153,
  12156,
  12159,
  12163,
  12166,
  12169,
  12173,
  12176,
  12179,
  12183,
  12186,
  12189,
  12193,
  12196,
  12199,
  12203,
  12206,
  12209,
  12213,
  12216,
  12219,
  12223,
  12226,
  12229,
  12232,
  12236,
  12239,
  12242,
  12246,
  12249,
  12252,
  12256,
  12259,
  12262,
  12266,
  12269,
  12272,
  12276,
  12279,
  12282,
  12286,
  12289,
  12292,
  12296,
  12299,
  12302,
  12306,
  12309,
  12312,
  12316,
  12319,
  12322,
  12325,
  12329,
  12332,
  12335,
  12339,
  12342,
  12345,
  12349,
  12352,
  12355,
  12359,
  12362,
  12365,
  12369,
  12372,
  12375,
  12379,
  12382,
  12385,
  12389,
  12392,
  12395,
  12399,
  12402,
  12405,
  12409,
  12412,
  12415,
  12418,
  12422,
  12425,
  12428,
  12432,
  12435,
  12438,
  12442,
  12445,
  12448,
  12452,
  12455,
  12458,
  12462,
  12465,
  12468,
  12472,
  12475,
  12478,
  12482,
  12485,
  12488,
  12492,
  12495,
  12498,
  12502,
  12505,
  12508,
  12512,
  12515,
  12518,
  12521,
  12525,
  12528,
  12531,
  12535,
  12538,
  12541,
  12545,
  12548,
  12551,
  12555,
  12558,
  12561,
  12565,
  12568,
  12571,
  12575,
  12578,
  12581,
  12585,
  12588,
  12591,
  12595,
  12598,
  12601,
  12605,
  12608,
  12611,
  12614,
  12618,
  12621,
  12624,
  12628,
  12631,
  12634,
  12638,
  12641,
  12644,
  12648,
  12651,
  12654,
  12658,
  12661,
  12664,
  12668,
  12671,
  12674,
  12678,
  12681,
  12684,
  12688,
  12691,
  12694,
  12698,
  12701,
  12704,
  12708,
  12711,
  12714,
  12717,
  12721,
  12724,
  12727,
  12731,
  12734,
  12737,
  12741,
  12744,
  12747,
  12751,
  12754,
  12757,
  12761,
  12764,
  12767,
  12771,
  12774,
  12777,
  12781,
  12784,
  12787,
  12791,
  12794,
  12797,
  12801,
  12804,
  12807,
  12810,
  12814,
  12817,
  12820,
  12824,
  12827,
  12830,
  12834,
  12837,
  12840,
  12844,
  12847,
  12850,
  12854,
  12857,
  12860,
  12864,
  12867,
  12870,
  12874,
  12877,
  12880,
  12884,
  12887,
  12890,
  12894,
  12897,
  12900,
  12903,
  12907,
  12910,
  12913,
  12917,
  12920,
  12923,
  12927,
  12930,
  12933,
  12937,
  12940,
  12943,
  12947,
  12950,
  12953,
  12957,
  12960,
  12963,
  12967,
  12970,
  12973,
  12977,
  12980,
  12983,
  12987,
  12990,
  12993,
  12997,
  13000,
  13003,
  13006,
  13010,
  13013,
  13016,
  13020,
  13023,
  13026,
  13030,
  13033,
  13036,
  13040,
  13043,
  13046,
  13050,
  13053,
  13056,
  13060,
  13063,
  13066,
  13070,
  13073,
  13076,
  13080,
  13083,
  13086,
  13090,
  13093,
  13096,
  13099,
  13103,
  13106,
  13109,
  13113,
  13116,
  13119,
  13123,
  13126,
  13129,
  13133,
  13136,
  13139,
  13143,
  13146,
  13149,
  13153,
  13156,
  13159,
  13163,
  13166,
  13169,
  13173,
  13176,
  13179,
  13183,
  13186,
  13189,
  13193,
  13196,
  13199,
  13202,
  13206,
  13209,
  13212,
  13216,
  13219,
  13222,
  13226,
  13229,
  13232,
  13236,
  13239,
  13242,
  13246,
  13249,
  13252,
  13256,
  13259,
  13262,
  13266,
  13269,
  13272,
  13276,
  13279,
  13282,
  13286,
  13289,
  13292,
  13295,
  13299,
  13302,
  13305,
  13309,
  13312,
  13315,
  13319,
  13322,
  13325,
  13329,
  13332,
  13335,
  13339,
  13342,
  13345,
  13349,
  13352,
  13355,
  13359,
  13362,
  13365,
  13369,
  13372,
  13375,
  13379,
  13382,
  13385,
  13388,
  13392,
  13395,
  13398,
  13402,
  13405,
  13408,
  13412,
  13415,
  13418,
  13422,
  13425,
  13428,
  13432,
  13435,
  13438,
  13442,
  13445,
  13448,
  13452,
  13455,
  13458,
  13462,
  13465,
  13468,
  13472,
  13475,
  13478,
  13482,
  13485,
  13488,
  13491,
  13495,
  13498,
  13501,
  13505,
  13508,
  13511,
  13515,
  13518,
  13521,
  13525,
  13528,
  13531,
  13535,
  13538,
  13541,
  13545,
  13548,
  13551,
  13555,
  13558,
  13561,
  13565,
  13568,
  13571,
  13575,
  13578,
  13581,
  13584,
  13588,
  13591,
  13594,
  13598,
  13601,
  13604,
  13608,
  13611,
  13614,
  13618,
  13621,
  13624,
  13628,
  13631,
  13634,
  13638,
  13641,
  13644,
  13648,
  13651,
  13654,
  13658,
  13661,
  13664,
  13668,
  13671,
  13674,
  13678,
  13681,
  13684,
  13687,
  13691,
  13694,
  13697,
  13701,
  13704,
  13707,
  13711,
  13714,
  13717,
  13721,
  13724,
  13727,
  13731,
  13734,
  13737,
  13741,
  13744,
  13747,
  13751,
  13754,
  13757,
  13761,
  13764,
  13767,
  13771,
  13774,
  13777,
  13780,
  13784,
  13787,
  13790,
  13794,
  13797,
  13800,
  13804,
  13807,
  13810,
  13814,
  13817,
  13820,
  13824,
  13827,
  13830,
  13834,
  13837,
  13840,
  13844,
  13847,
  13850,
  13854,
  13857,
  13860,
  13864,
  13867,
  13870,
  13873,
  13877,
  13880,
  13883,
  13887,
  13890,
  13893,
  13897,
  13900,
  13903,
  13907,
  13910,
  13913,
  13917,
  13920,
  13923,
  13927,
  13930,
  13933,
  13937,
  13940,
  13943,
  13947,
  13950,
  13953,
  13957,
  13960,
  13963,
  13967,
  13970,
  13973,
  13976,
  13980,
  13983,
  13986,
  13990,
  13993,
  13996,
  14000,
  14003,
  14006,
  14010,
  14013,
  14016,
  14020,
  14023,
  14026,
  14030,
  14033,
  14036,
  14040,
  14043,
  14046,
  14050,
  14053,
  14056,
  14060,
  14063,
  14066,
  14069,
  14073,
  14076,
  14079,
  14083,
  14086,
  14089,
  14093,
  14096,
  14099,
  14103,
  14106,
  14109,
  14113,
  14116,
  14119,
  14123,
  14126,
  14129,
  14133,
  14136,
  14139,
  14143,
  14146,
  14149,
  14153,
  14156,
  14159,
  14163,
  14166,
  14169,
  14172,
  14176,
  14179,
  14182,
  14186,
  14189,
  14192,
  14196,
  14199,
  14202,
  14206,
  14209,
  14212,
  14216,
  14219,
  14222,
  14226,
  14229,
  14232,
  14236,
  14239,
  14242,
  14246,
  14249,
  14252,
  14256,
  14259,
  14262,
  14265,
  14269,
  14272,
  14275,
  14279,
  14282,
  14285,
  14289,
  14292,
  14295,
  14299,
  14302,
  14305,
  14309,
  14312,
  14315,
  14319,
  14322,
  14325,
  14329,
  14332,
  14335,
  14339,
  14342,
  14345,
  14349,
  14352,
  14355,
  14359,
  14362,
  14365,
  14368,
  14372,
  14375,
  14378,
  14382,
  14385,
  14388,
  14392,
  14395,
  14398,
  14402,
  14405,
  14408,
  14412,
  14415,
  14418,
  14422,
  14425,
  14428,
  14432,
  14435,
  14438,
  14442,
  14445,
  14448,
  14452,
  14455,
  14458,
  14461,
  14465,
  14468,
  14471,
  14475,
  14478,
  14481,
  14485,
  14488,
  14491,
  14495,
  14498,
  14501,
  14505,
  14508,
  14511,
  14515,
  14518,
  14521,
  14525,
  14528,
  14531,
  14535,
  14538,
  14541,
  14545,
  14548,
  14551,
  14554,
  14558,
  14561,
  14564,
  14568,
  14571,
  14574,
  14578,
  14581,
  14584,
  14588,
  14591,
  14594,
  14598,
  14601,
  14604,
  14608,
  14611,
  14614,
  14618,
  14621,
  14624,
  14628,
  14631,
  14634,
  14638,
  14641,
  14644,
  14648,
  14651,
  14654,
  14657,
  14661,
  14664,
  14667,
  14671,
  14674,
  14677,
  14681,
  14684,
  14687,
  14691,
  14694,
  14697,
  14701,
  14704,
  14707,
  14711,
  14714,
  14717,
  14721,
  14724,
  14727,
  14731,
  14734,
  14737,
  14741,
  14744,
  14747,
  14750,
  14754,
  14757,
  14760,
  14764,
  14767,
  14770,
  14774,
  14777,
  14780,
  14784,
  14787,
  14790,
  14794,
  14797,
  14800,
  14804,
  14807,
  14810,
  14814,
  14817,
  14820,
  14824,
  14827,
  14830,
  14834,
  14837,
  14840,
  14844,
  14847,
  14850,
  14853,
  14857,
  14860,
  14863,
  14867,
  14870,
  14873,
  14877,
  14880,
  14883,
  14887,
  14890,
  14893,
  14897,
  14900,
  14903,
  14907,
  14910,
  14913,
  14917,
  14920,
  14923,
  14927,
  14930,
  14933,
  14937,
  14940,
  14943,
  14946,
  14950,
  14953,
  14956,
  14960,
  14963,
  14966,
  14970,
  14973,
  14976,
  14980,
  14983,
  14986,
  14990,
  14993,
  14996,
  15000,
  15003,
  15006,
  15010,
  15013,
  15016,
  15020,
  15023,
  15026,
  15030,
  15033,
  15036,
  15039,
  15043,
  15046,
  15049,
  15053,
  15056,
  15059,
  15063,
  15066,
  15069,
  15073,
  15076,
  15079,
  15083,
  15086,
  15089,
  15093,
  15096,
  15099,
  15103,
  15106,
  15109,
  15113,
  15116,
  15119,
  15123,
  15126,
  15129,
  15133,
  15136,
  15139,
  15142,
  15146,
  15149,
  15152,
  15156,
  15159,
  15162,
  15166,
  15169,
  15172,
  15176,
  15179,
  15182,
  15186,
  15189,
  15192,
  15196,
  15199,
  15202,
  15206,
  15209,
  15212,
  15216,
  15219,
  15222,
  15226,
  15229,
  15232,
  15235,
  15239,
  15242,
  15245,
  15249,
  15252,
  15255,
  15259,
  15262,
  15265,
  15269,
  15272,
  15275,
  15279,
  15282,
  15285,
  15289,
  15292,
  15295,
  15299,
  15302,
  15305,
  15309,
  15312,
  15315,
  15319,
  15322,
  15325,
  15329,
  15332,
  15335,
  15338,
  15342,
  15345,
  15348,
  15352,
  15355,
  15358,
  15362,
  15365,
  15368,
  15372,
  15375,
  15378,
  15382,
  15385,
  15388,
  15392,
  15395,
  15398,
  15402,
  15405,
  15408,
  15412,
  15415,
  15418,
  15422,
  15425,
  15428,
  15431,
  15435,
  15438,
  15441,
  15445,
  15448,
  15451,
  15455,
  15458,
  15461,
  15465,
  15468,
  15471,
  15475,
  15478,
  15481,
  15485,
  15488,
  15491,
  15495,
  15498,
  15501,
  15505,
  15508,
  15511,
  15515,
  15518,
  15521,
  15524,
  15528,
  15531,
  15534,
  15538,
  15541,
  15544,
  15548,
  15551,
  15554,
  15558,
  15561,
  15564,
  15568,
  15571,
  15574,
  15578,
  15581,
  15584,
  15588,
  15591,
  15594,
  15598,
  15601,
  15604,
  15608,
  15611,
  15614,
  15618,
  15621,
  15624,
  15627,
  15631,
  15634,
  15637,
  15641,
  15644,
  15647,
  15651,
  15654,
  15657,
  15661,
  15664,
  15667,
  15671,
  15674,
  15677,
  15681,
  15684,
  15687,
  15691,
  15694,
  15697,
  15701,
  15704,
  15707,
  15711,
  15714,
  15717,
  15720,
  15724,
  15727,
  15730,
  15734,
  15737,
  15740,
  15744,
  15747,
  15750,
  15754,
  15757,
  15760,
  15764,
  15767,
  15770,
  15774,
  15777,
  15780,
  15784,
  15787,
  15790,
  15794,
  15797,
  15800,
  15804,
  15807,
  15810,
  15814,
  15817,
  15820,
  15823,
  15827,
  15830,
  15833,
  15837,
  15840,
  15843,
  15847,
  15850,
  15853,
  15857,
  15860,
  15863,
  15867,
  15870,
  15873,
  15877,
  15880,
  15883,
  15887,
  15890,
  15893,
  15897,
  15900,
  15903,
  15907,
  15910,
  15913,
  15916,
  15920,
  15923,
  15926,
  15930,
  15933,
  15936,
  15940,
  15943,
  15946,
  15950,
  15953,
  15956,
  15960,
  15963,
  15966,
  15970,
  15973,
  15976,
  15980,
  15983,
  15986,
  15990,
  15993,
  15996,
  16000,
  16003,
  16006,
  16009,
  16013,
  16016,
  16019,
  16023,
  16026,
  16029,
  16033,
  16036,
  16039,
  16043,
  16046,
  16049,
  16053,
  16056,
  16059,
  16063,
  16066,
  16069,
  16073,
  16076,
  16079,
  16083,
  16086,
  16089,
  16093,
  16096,
  16099,
  16103,
  16106,
  16109,
  16112,
  16116,
  16119,
  16122,
  16126,
  16129,
  16132,
  16136,
  16139,
  16142,
  16146,
  16149,
  16152,
  16156,
  16159,
  16162,
  16166,
  16169,
  16172,
  16176,
  16179,
  16182,
  16186,
  16189,
  16192,
  16196,
  16199,
  16202,
  16205,
  16209,
  16212,
  16215,
  16219,
  16222,
  16225,
  16229,
  16232,
  16235,
  16239,
  16242,
  16245,
  16249,
  16252,
  16255,
  16259,
  16262,
  16265,
  16269,
  16272,
  16275,
  16279,
  16282,
  16285,
  16289,
  16292,
  16295,
  16299,
  16302,
  16305,
  16308,
  16312,
  16315,
  16318,
  16322,
  16325,
  16328,
  16332,
  16335,
  16338,
  16342,
  16345,
  16348,
  16352,
  16355,
  16358,
  16362,
  16365,
  16368,
  16372,
  16375,
  16378,
  16382,
  16385,
  16388,
  16392,
  16395,
  16398,
  16401,
  16405,
  16408,
  16411,
  16415,
  16418,
  16421,
  16425,
  16428,
  16431,
  16435,
  16438,
  16441,
  16445,
  16448,
  16451,
  16455,
  16458,
  16461,
  16465,
  16468,
  16471,
  16475,
  16478,
  16481,
  16485,
  16488,
  16491,
  16494,
  16498,
  16501,
  16504,
  16508,
  16511,
  16514,
  16518,
  16521,
  16524,
  16528,
  16531,
  16534,
  16538,
  16541,
  16544,
  16548,
  16551,
  16554,
  16558,
  16561,
  16564,
  16568,
  16571,
  16574,
  16578,
  16581,
  16584,
  16588,
  16591,
  16594,
  16597,
  16601,
  16604,
  16607,
  16611,
  16614,
  16617,
  16621,
  16624,
  16627,
  16631,
  16634,
  16637,
  16641,
  16644,
  16647,
  16651,
  16654,
  16657,
  16661,
  16664,
  16667,
  16671,
  16674,
  16677,
  16681,
  16684,
  16687,
  16690,
  16694,
  16697,
  16700,
  16704,
  16707,
  16710,
  16714,
  16717,
  16720,
  16724,
  16727,
  16730,
  16734,
  16737,
  16740,
  16744,
  16747,
  16750,
  16754,
  16757,
  16760,
  16764,
  16767,
  16770,
  16774,
  16777,
  16780,
  16784,
  16787,
  16790,
  16793,
  16797,
  16800,
  16803,
  16807,
  16810,
  16813,
  16817,
  16820,
  16823,
  16827,
  16830,
  16833,
  16837,
  16840,
  16843,
  16847,
  16850,
  16853,
  16857,
  16860,
  16863,
  16867,
  16870,
  16873,
  16877,
  16880,
  16883,
  16886,
  16890,
  16893,
  16896,
  16900,
  16903,
  16906,
  16910,
  16913,
  16916,
  16920,
  16923,
  16926,
  16930,
  16933,
  16936,
  16940,
  16943,
  16946,
  16950,
  16953,
  16956,
  16960,
  16963,
  16966,
  16970,
  16973,
  16976,
  16980,
  16983,
  16986,
  16989,
  16993,
  16996,
  16999,
  17003,
  17006,
  17009,
  17013,
  17016,
  17019,
  17023,
  17026,
  17029,
  17033,
  17036,
  17039,
  17043,
  17046,
  17049,
  17053,
  17056,
  17059,
  17063,
  17066,
  17069,
  17073,
  17076,
  17079,
  17082,
  17086,
  17089,
  17092,
  17096,
  17099,
  17102,
  17106,
  17109,
  17112,
  17116,
  17119,
  17122,
  17126,
  17129,
  17132,
  17136,
  17139,
  17142,
  17146,
  17149,
  17152,
  17156,
  17159,
  17162,
  17166,
  17169,
  17172,
  17175,
  17179,
  17182,
  17185,
  17189,
  17192,
  17195,
  17199,
  17202,
  17205,
  17209,
  17212,
  17215,
  17219,
  17222,
  17225,
  17229,
  17232,
  17235,
  17239,
  17242,
  17245,
  17249,
  17252,
  17255,
  17259,
  17262,
  17265,
  17269,
  17272,
  17275,
  17278,
  17282,
  17285,
  17288,
  17292,
  17295,
  17298,
  17302,
  17305,
  17308,
  17312,
  17315,
  17318,
  17322,
  17325,
  17328,
  17332,
  17335,
  17338,
  17342,
  17345,
  17348,
  17352,
  17355,
  17358,
  17362,
  17365,
  17368,
  17371,
  17375,
  17378,
  17381,
  17385,
  17388,
  17391,
  17395,
  17398,
  17401,
  17405,
  17408,
  17411,
  17415,
  17418,
  17421,
  17425,
  17428,
  17431,
  17435,
  17438,
  17441,
  17445,
  17448,
  17451,
  17455,
  17458,
  17461,
  17465,
  17468,
  17471,
  17474,
  17478,
  17481,
  17484,
  17488,
  17491,
  17494,
  17498,
  17501,
  17504,
  17508,
  17511,
  17514,
  17518,
  17521,
  17524,
  17528,
  17531,
  17534,
  17538,
  17541,
  17544,
  17548,
  17551,
  17554,
  17558,
  17561,
  17564,
  17567,
  17571,
  17574,
  17577,
  17581,
  17584,
  17587,
  17591,
  17594,
  17597,
  17601,
  17604,
  17607,
  17611,
  17614,
  17617,
  17621,
  17624,
  17627,
  17631,
  17634,
  17637,
  17641,
  17644,
  17647,
  17651,
  17654,
  17657,
  17660,
  17664,
  17667,
  17670,
  17674,
  17677,
  17680,
  17684,
  17687,
  17690,
  17694,
  17697,
  17700,
  17704,
  17707,
  17710,
  17714,
  17717,
  17720,
  17724,
  17727,
  17730,
  17734,
  17737,
  17740,
  17744,
  17747,
  17750,
  17754,
  17757,
  17760,
  17763,
  17767,
  17770,
  17773,
  17777,
  17780,
  17783,
  17787,
  17790,
  17793,
  17797,
  17800,
  17803,
  17807,
  17810,
  17813,
  17817,
  17820,
  17823,
  17827,
  17830,
  17833,
  17837,
  17840,
  17843,
  17847,
  17850,
  17853,
  17856,
  17860,
  17863,
  17866,
  17870,
  17873,
  17876,
  17880,
  17883,
  17886,
  17890,
  17893,
  17896,
  17900,
  17903,
  17906,
  17910,
  17913,
  17916,
  17920,
  17923,
  17926,
  17930,
  17933,
  17936,
  17940,
  17943,
  17946,
  17950,
  17953,
  17956,
  17959,
  17963,
  17966,
  17969,
  17973,
  17976,
  17979,
  17983,
  17986,
  17989,
  17993,
  17996,
  17999,
  18003,
  18006,
  18009,
  18013,
  18016,
  18019,
  18023,
  18026,
  18029,
  18033,
  18036,
  18039,
  18043,
  18046,
  18049,
  18052,
  18056,
  18059,
  18062,
  18066,
  18069,
  18072,
  18076,
  18079,
  18082,
  18086,
  18089,
  18092,
  18096,
  18099,
  18102,
  18106,
  18109,
  18112,
  18116,
  18119,
  18122,
  18126,
  18129,
  18132,
  18136,
  18139,
  18142,
  18145,
  18149,
  18152,
  18155,
  18159,
  18162,
  18165,
  18169,
  18172,
  18175,
  18179,
  18182,
  18185,
  18189,
  18192,
  18195,
  18199,
  18202,
  18205,
  18209,
  18212,
  18215,
  18219,
  18222,
  18225,
  18229,
  18232,
  18235,
  18239,
  18242,
  18245,
  18248,
  18252,
  18255,
  18258,
  18262,
  18265,
  18268,
  18272,
  18275,
  18278,
  18282,
  18285,
  18288,
  18292,
  18295,
  18298,
  18302,
  18305,
  18308,
  18312,
  18315,
  18318,
  18322,
  18325,
  18328,
  18332,
  18335,
  18338,
  18341,
  18345,
  18348,
  18351,
  18355,
  18358,
  18361,
  18365,
  18368,
  18371,
  18375,
  18378,
  18381,
  18385,
  18388,
  18391,
  18395,
  18398,
  18401,
  18405,
  18408,
  18411,
  18415,
  18418,
  18421,
  18425,
  18428,
  18431,
  18435,
  18438,
  18441,
  18444,
  18448,
  18451,
  18454,
  18458,
  18461,
  18464,
  18468,
  18471,
  18474,
  18478,
  18481,
  18484,
  18488,
  18491,
  18494,
  18498,
  18501,
  18504,
  18508,
  18511,
  18514,
  18518,
  18521,
  18524,
  18528,
  18531,
  18534,
  18537,
  18541,
  18544,
  18547,
  18551,
  18554,
  18557,
  18561,
  18564,
  18567,
  18571,
  18574,
  18577,
  18581,
  18584,
  18587,
  18591,
  18594,
  18597,
  18601,
  18604,
  18607,
  18611,
  18614,
  18617,
  18621,
  18624,
  18627,
  18630,
  18634,
  18637,
  18640,
  18644,
  18647,
  18650,
  18654,
  18657,
  18660,
  18664,
  18667,
  18670,
  18674,
  18677,
  18680,
  18684,
  18687,
  18690,
  18694,
  18697,
  18700,
  18704,
  18707,
  18710,
  18714,
  18717,
  18720,
  18724,
  18727,
  18730,
  18733,
  18737,
  18740,
  18743,
  18747,
  18750,
  18753,
  18757,
  18760,
  18763,
  18767,
  18770,
  18773,
  18777,
  18780,
  18783,
  18787,
  18790,
  18793,
  18797,
  18800,
  18803,
  18807,
  18810,
  18813,
  18817,
  18820,
  18823,
  18826,
  18830,
  18833,
  18836,
  18840,
  18843,
  18846,
  18850,
  18853,
  18856,
  18860,
  18863,
  18866,
  18870,
  18873,
  18876,
  18880,
  18883,
  18886,
  18890,
  18893,
  18896,
  18900,
  18903,
  18906,
  18910,
  18913,
  18916,
  18920,
  18923,
  18926,
  18929,
  18933,
  18936,
  18939,
  18943,
  18946,
  18949,
  18953,
  18956,
  18959,
  18963,
  18966,
  18969,
  18973,
  18976,
  18979,
  18983,
  18986,
  18989,
  18993,
  18996,
  18999,
  19003,
  19006,
  19009,
  19013,
  19016,
  19019,
  19022,
  19026,
  19029,
  19032,
  19036,
  19039,
  19042,
  19046,
  19049,
  19052,
  19056,
  19059,
  19062,
  19066,
  19069,
  19072,
  19076,
  19079,
  19082,
  19086,
  19089,
  19092,
  19096,
  19099,
  19102,
  19106,
  19109,
  19112,
  19116,
  19119,
  19122,
  19125,
  19129,
  19132,
  19135,
  19139,
  19142,
  19145,
  19149,
  19152,
  19155,
  19159,
  19162,
  19165,
  19169,
  19172,
  19175,
  19179,
  19182,
  19185,
  19189,
  19192,
  19195,
  19199,
  19202,
  19205,
  19209,
  19212,
  19215,
  19218,
  19222,
  19225,
  19228,
  19232,
  19235,
  19238,
  19242,
  19245,
  19248,
  19252,
  19255,
  19258,
  19262,
  19265,
  19268,
  19272,
  19275,
  19278,
  19282,
  19285,
  19288,
  19292,
  19295,
  19298,
  19302,
  19305,
  19308,
  19311,
  19315,
  19318,
  19321,
  19325,
  19328,
  19331,
  19335,
  19338,
  19341,
  19345,
  19348,
  19351,
  19355,
  19358,
  19361,
  19365,
  19368,
  19371,
  19375,
  19378,
  19381,
  19385,
  19388,
  19391,
  19395,
  19398,
  19401,
  19405,
  19408,
  19411,
  19414,
  19418,
  19421,
  19424,
  19428,
  19431,
  19434,
  19438,
  19441,
  19444,
  19448,
  19451,
  19454,
  19458,
  19461,
  19464,
  19468,
  19471,
  19474,
  19478,
  19481,
  19484,
  19488,
  19491,
  19494,
  19498,
  19501,
  19504,
  19507,
  19511,
  19514,
  19517,
  19521,
  19524,
  19527,
  19531,
  19534,
  19537,
  19541,
  19544,
  19547,
  19551,
  19554,
  19557,
  19561,
  19564,
  19567,
  19571,
  19574,
  19577,
  19581,
  19584,
  19587,
  19591,
  19594,
  19597,
  19601,
  19604,
  19607,
  19610,
  19614,
  19617,
  19620,
  19624,
  19627,
  19630,
  19634,
  19637,
  19640,
  19644,
  19647,
  19650,
  19654,
  19657,
  19660,
  19664,
  19667,
  19670,
  19674,
  19677,
  19680,
  19684,
  19687,
  19690,
  19694,
  19697,
  19700,
  19703,
  19707,
  19710,
  19713,
  19717,
  19720,
  19723,
  19727,
  19730,
  19733,
  19737,
  19740,
  19743,
  19747,
  19750,
  19753,
  19757,
  19760,
  19763,
  19767,
  19770,
  19773,
  19777,
  19780,
  19783,
  19787,
  19790,
  19793,
  19796,
  19800,
  19803,
  19806,
  19810,
  19813,
  19816,
  19820,
  19823,
  19826,
  19830,
  19833,
  19836,
  19840,
  19843,
  19846,
  19850,
  19853,
  19856,
  19860,
  19863,
  19866,
  19870,
  19873,
  19876,
  19880,
  19883,
  19886,
  19890,
  19893,
  19896,
  19899,
  19903,
  19906,
  19909,
  19913,
  19916,
  19919,
  19923,
  19926,
  19929,
  19933,
  19936,
  19939,
  19943,
  19946,
  19949,
  19953,
  19956,
  19959,
  19963,
  19966,
  19969,
  19973,
  19976,
  19979,
  19983,
  19986,
  19989,
  19992,
  19996,
  19999,
  20002,
  20006,
  20009,
  20012,
  20016,
  20019,
  20022,
  20026,
  20029,
  20032,
  20036,
  20039,
  20042,
  20046,
  20049,
  20052,
  20056,
  20059,
  20062,
  20066,
  20069,
  20072,
  20076,
  20079,
  20082,
  20086,
  20089,
  20092,
  20095,
  20099,
  20102,
  20105,
  20109,
  20112,
  20115,
  20119,
  20122,
  20125,
  20129,
  20132,
  20135,
  20139,
  20142,
  20145,
  20149,
  20152,
  20155,
  20159,
  20162,
  20165,
  20169,
  20172,
  20175,
  20179,
  20182,
  20185,
  20188,
  20192,
  20195,
  20198,
  20202,
  20205,
  20208,
  20212,
  20215,
  20218,
  20222,
  20225,
  20228,
  20232,
  20235,
  20238,
  20242,
  20245,
  20248,
  20252,
  20255,
  20258,
  20262,
  20265,
  20268,
  20272,
  20275,
  20278,
  20281,
  20285,
  20288,
  20291,
  20295,
  20298,
  20301,
  20305,
  20308,
  20311,
  20315,
  20318,
  20321,
  20325,
  20328,
  20331,
  20335,
  20338,
  20341,
  20345,
  20348,
  20351,
  20355,
  20358,
  20361,
  20365,
  20368,
  20371,
  20375,
  20378,
  20381,
  20384,
  20388,
  20391,
  20394,
  20398,
  20401,
  20404,
  20408,
  20411,
  20414,
  20418,
  20421,
  20424,
  20428,
  20431,
  20434,
  20438,
  20441,
  20444,
  20448,
  20451,
  20454,
  20458,
  20461,
  20464,
  20468,
  20471,
  20474,
  20477,
  20481,
  20484,
  20487,
  20491,
  20494,
  20497,
  20501,
  20504,
  20507,
  20511,
  20514,
  20517,
  20521,
  20524,
  20527,
  20531,
  20534,
  20537,
  20541,
  20544,
  20547,
  20551,
  20554,
  20557,
  20561,
  20564,
  20567,
  20571,
  20574,
  20577,
  20580,
  20584,
  20587,
  20590,
  20594,
  20597,
  20600,
  20604,
  20607,
  20610,
  20614,
  20617,
  20620,
  20624,
  20627,
  20630,
  20634,
  20637,
  20640,
  20644,
  20647,
  20650,
  20654,
  20657,
  20660,
  20664,
  20667,
  20670,
  20673,
  20677,
  20680,
  20683,
  20687,
  20690,
  20693,
  20697,
  20700,
  20703,
  20707,
  20710,
  20713,
  20717,
  20720,
  20723,
  20727,
  20730,
  20733,
  20737,
  20740,
  20743,
  20747,
  20750,
  20753,
  20757,
  20760,
  20763,
  20766,
  20770,
  20773,
  20776,
  20780,
  20783,
  20786,
  20790,
  20793,
  20796,
  20800,
  20803,
  20806,
  20810,
  20813,
  20816,
  20820,
  20823,
  20826,
  20830,
  20833,
  20836,
  20840,
  20843,
  20846,
  20850,
  20853,
  20856,
  20860,
  20863,
  20866,
  20869,
  20873,
  20876,
  20879,
  20883,
  20886,
  20889,
  20893,
  20896,
  20899,
  20903,
  20906,
  20909,
  20913,
  20916,
  20919,
  20923,
  20926,
  20929,
  20933,
  20936,
  20939,
  20943,
  20946,
  20949,
  20953,
  20956,
  20959,
  20962,
  20966,
  20969,
  20972,
  20976,
  20979,
  20982,
  20986,
  20989,
  20992,
  20996,
  20999,
  21002,
  21006,
  21009,
  21012,
  21016,
  21019,
  21022,
  21026,
  21029,
  21032,
  21036,
  21039,
  21042,
  21046,
  21049,
  21052,
  21056,
  21059,
  21062,
  21065,
  21069,
  21072,
  21075,
  21079,
  21082,
  21085,
  21089,
  21092,
  21095,
  21099,
  21102,
  21105,
  21109,
  21112,
  21115,
  21119,
  21122,
  21125,
  21129,
  21132,
  21135,
  21139,
  21142,
  21145,
  21149,
  21152,
  21155,
  21158,
  21162,
  21165,
  21168,
  21172,
  21175,
  21178,
  21182,
  21185,
  21188,
  21192,
  21195,
  21198,
  21202,
  21205,
  21208,
  21212,
  21215,
  21218,
  21222,
  21225,
  21228,
  21232,
  21235,
  21238,
  21242,
  21245,
  21248,
  21252,
  21255,
  21258,
  21261,
  21265,
  21268,
  21271,
  21275,
  21278,
  21281,
  21285,
  21288,
  21291,
  21295,
  21298,
  21301,
  21305,
  21308,
  21311,
  21315,
  21318,
  21321,
  21325,
  21328,
  21331,
  21335,
  21338,
  21341,
  21345,
  21348,
  21351,
  21354,
  21358,
  21361,
  21364,
  21368,
  21371,
  21374,
  21378,
  21381,
  21384,
  21388,
  21391,
  21394,
  21398,
  21401,
  21404,
  21408,
  21411,
  21414,
  21418,
  21421,
  21424,
  21428,
  21431,
  21434,
  21438,
  21441,
  21444,
  21447,
  21451,
  21454,
  21457,
  21461,
  21464,
  21467,
  21471,
  21474,
  21477,
  21481,
  21484,
  21487,
  21491,
  21494,
  21497,
  21501,
  21504,
  21507,
  21511,
  21514,
  21517,
  21521,
  21524,
  21527,
  21531,
  21534,
  21537,
  21541,
  21544,
  21547,
  21550,
  21554,
  21557,
  21560,
  21564,
  21567,
  21570,
  21574,
  21577,
  21580,
  21584,
  21587,
  21590,
  21594,
  21597,
  21600,
  21604,
  21607,
  21610,
  21614,
  21617,
  21620,
  21624,
  21627,
  21630,
  21634,
  21637,
  21640,
  21643,
  21647,
  21650,
  21653,
  21657,
  21660,
  21663,
  21667,
  21670,
  21673,
  21677,
  21680,
  21683,
  21687,
  21690,
  21693,
  21697,
  21700,
  21703,
  21707,
  21710,
  21713,
  21717,
  21720,
  21723,
  21727,
  21730,
  21733,
  21737,
  21740,
  21743,
  21746,
  21750,
  21753,
  21756,
  21760,
  21763,
  21766,
  21770,
  21773,
  21776,
  21780,
  21783,
  21786,
  21790,
  21793,
  21796,
  21800,
  21803,
  21806,
  21810,
  21813,
  21816,
  21820,
  21823,
  21826,
  21830,
  21833,
  21836,
  21839,
  21843,
  21846,
  21849,
  21853,
  21856,
  21859,
  21863,
  21866,
  21869,
  21873,
  21876,
  21879,
  21883,
  21886,
  21889,
  21893,
  21896,
  21899,
  21903,
  21906,
  21909,
  21913,
  21916,
  21919,
  21923,
  21926,
  21929,
  21932,
  21936,
  21939,
  21942,
  21946,
  21949,
  21952,
  21956,
  21959,
  21962,
  21966,
  21969,
  21972,
  21976,
  21979,
  21982,
  21986,
  21989,
  21992,
  21996,
  21999,
  22002,
  22006,
  22009,
  22012,
  22016,
  22019,
  22022,
  22026,
  22029,
  22032,
  22035,
  22039,
  22042,
  22045,
  22049,
  22052,
  22055,
  22059,
  22062,
  22065,
  22069,
  22072,
  22075,
  22079,
  22082,
  22085,
  22089,
  22092,
  22095,
  22099,
  22102,
  22105,
  22109,
  22112,
  22115,
  22119,
  22122,
  22125,
  22128,
  22132,
  22135,
  22138,
  22142,
  22145,
  22148,
  22152,
  22155,
  22158,
  22162,
  22165,
  22168,
  22172,
  22175,
  22178,
  22182,
  22185,
  22188,
  22192,
  22195,
  22198,
  22202,
  22205,
  22208,
  22212,
  22215,
  22218,
  22222,
  22225,
  22228,
  22231,
  22235,
  22238,
  22241,
  22245,
  22248,
  22251,
  22255,
  22258,
  22261,
  22265,
  22268,
  22271,
  22275,
  22278,
  22281,
  22285,
  22288,
  22291,
  22295,
  22298,
  22301,
  22305,
  22308,
  22311,
  22315,
  22318,
  22321,
  22324,
  22328,
  22331,
  22334,
  22338,
  22341,
  22344,
  22348,
  22351,
  22354,
  22358,
  22361,
  22364,
  22368,
  22371,
  22374,
  22378,
  22381,
  22384,
  22388,
  22391,
  22394,
  22398,
  22401,
  22404,
  22408,
  22411,
  22414,
  22417,
  22421,
  22424,
  22427,
  22431,
  22434,
  22437,
  22441,
  22444,
  22447,
  22451,
  22454,
  22457,
  22461,
  22464,
  22467,
  22471,
  22474,
  22477,
  22481,
  22484,
  22487,
  22491,
  22494,
  22497,
  22501,
  22504,
  22507,
  22511,
  22514,
  22517,
  22520,
  22524,
  22527,
  22530,
  22534,
  22537,
  22540,
  22544,
  22547,
  22550,
  22554,
  22557,
  22560,
  22564,
  22567,
  22570,
  22574,
  22577,
  22580,
  22584,
  22587,
  22590,
  22594,
  22597,
  22600,
  22604,
  22607,
  22610,
  22613,
  22617,
  22620,
  22623,
  22627,
  22630,
  22633,
  22637,
  22640,
  22643,
  22647,
  22650,
  22653,
  22657,
  22660,
  22663,
  22667,
  22670,
  22673,
  22677,
  22680,
  22683,
  22687,
  22690,
  22693,
  22697,
  22700,
  22703,
  22707,
  22710,
  22713,
  22716,
  22720,
  22723,
  22726,
  22730,
  22733,
  22736,
  22740,
  22743,
  22746,
  22750,
  22753,
  22756,
  22760,
  22763,
  22766,
  22770,
  22773,
  22776,
  22780,
  22783,
  22786,
  22790,
  22793,
  22796,
  22800,
  22803,
  22806,
  22809,
  22813,
  22816,
  22819,
  22823,
  22826,
  22829,
  22833,
  22836,
  22839,
  22843,
  22846,
  22849,
  22853,
  22856,
  22859,
  22863,
  22866,
  22869,
  22873,
  22876,
  22879,
  22883,
  22886,
  22889,
  22893,
  22896,
  22899,
  22902,
  22906,
  22909,
  22912,
  22916,
  22919,
  22922,
  22926,
  22929,
  22932,
  22936,
  22939,
  22942,
  22946,
  22949,
  22952,
  22956,
  22959,
  22962,
  22966,
  22969,
  22972,
  22976,
  22979,
  22982,
  22986,
  22989,
  22992,
  22996,
  22999,
  23002,
  23005,
  23009,
  23012,
  23015,
  23019,
  23022,
  23025,
  23029,
  23032,
  23035,
  23039,
  23042,
  23045,
  23049,
  23052,
  23055,
  23059,
  23062,
  23065,
  23069,
  23072,
  23075,
  23079,
  23082,
  23085,
  23089,
  23092,
  23095,
  23098,
  23102,
  23105,
  23108,
  23112,
  23115,
  23118,
  23122,
  23125,
  23128,
  23132,
  23135,
  23138,
  23142,
  23145,
  23148,
  23152,
  23155,
  23158,
  23162,
  23165,
  23168,
  23172,
  23175,
  23178,
  23182,
  23185,
  23188,
  23192,
  23195,
  23198,
  23201,
  23205,
  23208,
  23211,
  23215,
  23218,
  23221,
  23225,
  23228,
  23231,
  23235,
  23238,
  23241,
  23245,
  23248,
  23251,
  23255,
  23258,
  23261,
  23265,
  23268,
  23271,
  23275,
  23278,
  23281,
  23285,
  23288,
  23291,
  23294,
  23298,
  23301,
  23304,
  23308,
  23311,
  23314,
  23318,
  23321,
  23324,
  23328,
  23331,
  23334,
  23338,
  23341,
  23344,
  23348,
  23351,
  23354,
  23358,
  23361,
  23364,
  23368,
  23371,
  23374,
  23378,
  23381,
  23384,
  23388,
  23391,
  23394,
  23397,
  23401,
  23404,
  23407,
  23411,
  23414,
  23417,
  23421,
  23424,
  23427,
  23431,
  23434,
  23437,
  23441,
  23444,
  23447,
  23451,
  23454,
  23457,
  23461,
  23464,
  23467,
  23471,
  23474,
  23477,
  23481,
  23484,
  23487,
  23490,
  23494,
  23497,
  23500,
  23504,
  23507,
  23510,
  23514,
  23517,
  23520,
  23524,
  23527,
  23530,
  23534,
  23537,
  23540,
  23544,
  23547,
  23550,
  23554,
  23557,
  23560,
  23564,
  23567,
  23570,
  23574,
  23577,
  23580,
  23583,
  23587,
  23590,
  23593,
  23597,
  23600,
  23603,
  23607,
  23610,
  23613,
  23617,
  23620,
  23623,
  23627,
  23630,
  23633,
  23637,
  23640,
  23643,
  23647,
  23650,
  23653,
  23657,
  23660,
  23663,
  23667,
  23670,
  23673,
  23677,
  23680,
  23683,
  23686,
  23690,
  23693,
  23696,
  23700,
  23703,
  23706,
  23710,
  23713,
  23716,
  23720,
  23723,
  23726,
  23730,
  23733,
  23736,
  23740,
  23743,
  23746,
  23750,
  23753,
  23756,
  23760,
  23763,
  23766,
  23770,
  23773,
  23776,
  23779,
  23783,
  23786,
  23789,
  23793,
  23796,
  23799,
  23803,
  23806,
  23809,
  23813,
  23816,
  23819,
  23823,
  23826,
  23829,
  23833,
  23836,
  23839,
  23843,
  23846,
  23849,
  23853,
  23856,
  23859,
  23863,
  23866,
  23869,
  23873,
  23876,
  23879,
  23882,
  23886,
  23889,
  23892,
  23896,
  23899,
  23902,
  23906,
  23909,
  23912,
  23916,
  23919,
  23922,
  23926,
  23929,
  23932,
  23936,
  23939,
  23942,
  23946,
  23949,
  23952,
  23956,
  23959,
  23962,
  23966,
  23969,
  23972,
  23975,
  23979,
  23982,
  23985,
  23989,
  23992,
  23995,
  23999,
  24002,
  24005,
  24009,
  24012,
  24015,
  24019,
  24022,
  24025,
  24029,
  24032,
  24035,
  24039,
  24042,
  24045,
  24049,
  24052,
  24055,
  24059,
  24062,
  24065,
  24068,
  24072,
  24075,
  24078,
  24082,
  24085,
  24088,
  24092,
  24095,
  24098,
  24102,
  24105,
  24108,
  24112,
  24115,
  24118,
  24122,
  24125,
  24128,
  24132,
  24135,
  24138,
  24142,
  24145,
  24148,
  24152,
  24155,
  24158,
  24162,
  24165,
  24168,
  24171,
  24175,
  24178,
  24181,
  24185,
  24188,
  24191,
  24195,
  24198,
  24201,
  24205,
  24208,
  24211,
  24215,
  24218,
  24221,
  24225,
  24228,
  24231,
  24235,
  24238,
  24241,
  24245,
  24248,
  24251,
  24255,
  24258,
  24261,
  24264,
  24268,
  24271,
  24274,
  24278,
  24281,
  24284,
  24288,
  24291,
  24294,
  24298,
  24301,
  24304,
  24308,
  24311,
  24314,
  24318,
  24321,
  24324,
  24328,
  24331,
  24334,
  24338,
  24341,
  24344,
  24348,
  24351,
  24354,
  24358,
  24361,
  24364,
  24367,
  24371,
  24374,
  24377,
  24381,
  24384,
  24387,
  24391,
  24394,
  24397,
  24401,
  24404,
  24407,
  24411,
  24414,
  24417,
  24421,
  24424,
  24427,
  24431,
  24434,
  24437,
  24441,
  24444,
  24447,
  24451,
  24454,
  24457,
  24460,
  24464,
  24467,
  24470,
  24474,
  24477,
  24480,
  24484,
  24487,
  24490,
  24494,
  24497,
  24500,
  24504,
  24507,
  24510,
  24514,
  24517,
  24520,
  24524,
  24527,
  24530,
  24534,
  24537,
  24540,
  24544,
  24547,
  24550,
  24553,
  24557,
  24560,
  24563,
  24567,
  24570,
  24573,
  24577,
  24580,
  24583,
  24587,
  24590,
  24593,
  24597,
  24600,
  24603,
  24607,
  24610,
  24613,
  24617,
  24620,
  24623,
  24627,
  24630,
  24633,
  24637,
  24640,
  24643,
  24647,
  24650,
  24653,
  24656,
  24660,
  24663,
  24666,
  24670,
  24673,
  24676,
  24680,
  24683,
  24686,
  24690,
  24693,
  24696,
  24700,
  24703,
  24706,
  24710,
  24713,
  24716,
  24720,
  24723,
  24726,
  24730,
  24733,
  24736,
  24740,
  24743,
  24746,
  24749,
  24753,
  24756,
  24759,
  24763,
  24766,
  24769,
  24773,
  24776,
  24779,
  24783,
  24786,
  24789,
  24793,
  24796,
  24799,
  24803,
  24806,
  24809,
  24813,
  24816,
  24819,
  24823,
  24826,
  24829,
  24833,
  24836,
  24839,
  24843,
  24846,
  24849,
  24852,
  24856,
  24859,
  24862,
  24866,
  24869,
  24872,
  24876,
  24879,
  24882,
  24886,
  24889,
  24892,
  24896,
  24899,
  24902,
  24906,
  24909,
  24912,
  24916,
  24919,
  24922,
  24926,
  24929,
  24932,
  24936,
  24939,
  24942,
  24945,
  24949,
  24952,
  24955,
  24959,
  24962,
  24965,
  24969,
  24972,
  24975,
  24979,
  24982,
  24985,
  24989,
  24992,
  24995,
  24999,
  25002,
  25005,
  25009,
  25012,
  25015,
  25019,
  25022,
  25025,
  25029,
  25032,
  25035,
  25038,
  25042,
  25045,
  25048,
  25052,
  25055,
  25058,
  25062,
  25065,
  25068,
  25072,
  25075,
  25078,
  25082,
  25085,
  25088,
  25092,
  25095,
  25098,
  25102,
  25105,
  25108,
  25112,
  25115,
  25118,
  25122,
  25125,
  25128,
  25132,
  25135,
  25138,
  25141,
  25145,
  25148,
  25151,
  25155,
  25158,
  25161,
  25165,
  25168,
  25171,
  25175,
  25178,
  25181,
  25185,
  25188,
  25191,
  25195,
  25198,
  25201,
  25205,
  25208,
  25211,
  25215,
  25218,
  25221,
  25225,
  25228,
  25231,
  25234,
  25238,
  25241,
  25244,
  25248,
  25251,
  25254,
  25258,
  25261,
  25264,
  25268,
  25271,
  25274,
  25278,
  25281,
  25284,
  25288,
  25291,
  25294,
  25298,
  25301,
  25304,
  25308,
  25311,
  25314,
  25318,
  25321,
  25324,
  25328,
  25331,
  25334,
  25337,
  25341,
  25344,
  25347,
  25351,
  25354,
  25357,
  25361,
  25364,
  25367,
  25371,
  25374,
  25377,
  25381,
  25384,
  25387,
  25391,
  25394,
  25397,
  25401,
  25404,
  25407,
  25411,
  25414,
  25417,
  25421,
  25424,
  25427,
  25430,
  25434,
  25437,
  25440,
  25444,
  25447,
  25450,
  25454,
  25457,
  25460,
  25464,
  25467,
  25470,
  25474,
  25477,
  25480,
  25484,
  25487,
  25490,
  25494,
  25497,
  25500,
  25504,
  25507,
  25510,
  25514,
  25517,
  25520,
  25524,
  25527,
  25530,
  25533,
  25537,
  25540,
  25543,
  25547,
  25550,
  25553,
  25557,
  25560,
  25563,
  25567,
  25570,
  25573,
  25577,
  25580,
  25583,
  25587,
  25590,
  25593,
  25597,
  25600,
  25603,
  25607,
  25610,
  25613,
  25617,
  25620,
  25623,
  25626,
  25630,
  25633,
  25636,
  25640,
  25643,
  25646,
  25650,
  25653,
  25656,
  25660,
  25663,
  25666,
  25670,
  25673,
  25676,
  25680,
  25683,
  25686,
  25690,
  25693,
  25696,
  25700,
  25703,
  25706,
  25710,
  25713,
  25716,
  25719,
  25723,
  25726,
  25729,
  25733,
  25736,
  25739,
  25743,
  25746,
  25749,
  25753,
  25756,
  25759,
  25763,
  25766,
  25769,
  25773,
  25776,
  25779,
  25783,
  25786,
  25789,
  25793,
  25796,
  25799,
  25803,
  25806,
  25809,
  25813,
  25816,
  25819,
  25822,
  25826,
  25829,
  25832,
  25836,
  25839,
  25842,
  25846,
  25849,
  25852,
  25856,
  25859,
  25862,
  25866,
  25869,
  25872,
  25876,
  25879,
  25882,
  25886,
  25889,
  25892,
  25896,
  25899,
  25902,
  25906,
  25909,
  25912,
  25915,
  25919,
  25922,
  25925,
  25929,
  25932,
  25935,
  25939,
  25942,
  25945,
  25949,
  25952,
  25955,
  25959,
  25962,
  25965,
  25969,
  25972,
  25975,
  25979,
  25982,
  25985,
  25989,
  25992,
  25995,
  25999,
  26002,
  26005,
  26009,
  26012,
  26015,
  26018,
  26022,
  26025,
  26028,
  26032,
  26035,
  26038,
  26042,
  26045,
  26048,
  26052,
  26055,
  26058,
  26062,
  26065,
  26068,
  26072,
  26075,
  26078,
  26082,
  26085,
  26088,
  26092,
  26095,
  26098,
  26102,
  26105,
  26108,
  26111,
  26115,
  26118,
  26121,
  26125,
  26128,
  26131,
  26135,
  26138,
  26141,
  26145,
  26148,
  26151,
  26155,
  26158,
  26161,
  26165,
  26168,
  26171,
  26175,
  26178,
  26181,
  26185,
  26188,
  26191,
  26195,
  26198,
  26201,
  26204,
  26208,
  26211,
  26214,
  26218,
  26221,
  26224,
  26228,
  26231,
  26234,
  26238,
  26241,
  26244,
  26248,
  26251,
  26254,
  26258,
  26261,
  26264,
  26268,
  26271,
  26274,
  26278,
  26281,
  26284,
  26288,
  26291,
  26294,
  26298,
  26301,
  26304,
  26307,
  26311,
  26314,
  26317,
  26321,
  26324,
  26327,
  26331,
  26334,
  26337,
  26341,
  26344,
  26347,
  26351,
  26354,
  26357,
  26361,
  26364,
  26367,
  26371,
  26374,
  26377,
  26381,
  26384,
  26387,
  26391,
  26394,
  26397,
  26400,
  26404,
  26407,
  26410,
  26414,
  26417,
  26420,
  26424,
  26427,
  26430,
  26434,
  26437,
  26440,
  26444,
  26447,
  26450,
  26454,
  26457,
  26460,
  26464,
  26467,
  26470,
  26474,
  26477,
  26480,
  26484,
  26487,
  26490,
  26494,
  26497,
  26500,
  26503,
  26507,
  26510,
  26513,
  26517,
  26520,
  26523,
  26527,
  26530,
  26533,
  26537,
  26540,
  26543,
  26547,
  26550,
  26553,
  26557,
  26560,
  26563,
  26567,
  26570,
  26573,
  26577,
  26580,
  26583,
  26587,
  26590,
  26593,
  26596,
  26600,
  26603,
  26606,
  26610,
  26613,
  26616,
  26620,
  26623,
  26626,
  26630,
  26633,
  26636,
  26640,
  26643,
  26646,
  26650,
  26653,
  26656,
  26660,
  26663,
  26666,
  26670,
  26673,
  26676,
  26680,
  26683,
  26686,
  26689,
  26693,
  26696,
  26699,
  26703,
  26706,
  26709,
  26713,
  26716,
  26719,
  26723,
  26726,
  26729,
  26733,
  26736,
  26739,
  26743,
  26746,
  26749,
  26753,
  26756,
  26759,
  26763,
  26766,
  26769,
  26773,
  26776,
  26779,
  26783,
  26786,
  26789,
  26792,
  26796,
  26799,
  26802,
  26806,
  26809,
  26812,
  26816,
  26819,
  26822,
  26826,
  26829,
  26832,
  26836,
  26839,
  26842,
  26846,
  26849,
  26852,
  26856,
  26859,
  26862,
  26866,
  26869,
  26872,
  26876,
  26879,
  26882,
  26885,
  26889,
  26892,
  26895,
  26899,
  26902,
  26905,
  26909,
  26912,
  26915,
  26919,
  26922,
  26925,
  26929,
  26932,
  26935,
  26939,
  26942,
  26945,
  26949,
  26952,
  26955,
  26959,
  26962,
  26965,
  26969,
  26972,
  26975,
  26979,
  26982,
  26985,
  26988,
  26992,
  26995,
  26998,
  27002,
  27005,
  27008,
  27012,
  27015,
  27018,
  27022,
  27025,
  27028,
  27032,
  27035,
  27038,
  27042,
  27045,
  27048,
  27052,
  27055,
  27058,
  27062,
  27065,
  27068,
  27072,
  27075,
  27078,
  27081,
  27085,
  27088,
  27091,
  27095,
  27098,
  27101,
  27105,
  27108,
  27111,
  27115,
  27118,
  27121,
  27125,
  27128,
  27131,
  27135,
  27138,
  27141,
  27145,
  27148,
  27151,
  27155,
  27158,
  27161,
  27165,
  27168,
  27171,
  27174,
  27178,
  27181,
  27184,
  27188,
  27191,
  27194,
  27198,
  27201,
  27204,
  27208,
  27211,
  27214,
  27218,
  27221,
  27224,
  27228,
  27231,
  27234,
  27238,
  27241,
  27244,
  27248,
  27251,
  27254,
  27258,
  27261,
  27264,
  27268,
  27271,
  27274,
  27277,
  27281,
  27284,
  27287,
  27291,
  27294,
  27297,
  27301,
  27304,
  27307,
  27311,
  27314,
  27317,
  27321,
  27324,
  27327,
  27331,
  27334,
  27337,
  27341,
  27344,
  27347,
  27351,
  27354,
  27357,
  27361,
  27364,
  27367,
  27370,
  27374,
  27377,
  27380,
  27384,
  27387,
  27390,
  27394,
  27397,
  27400,
  27404,
  27407,
  27410,
  27414,
  27417,
  27420,
  27424,
  27427,
  27430,
  27434,
  27437,
  27440,
  27444,
  27447,
  27450,
  27454,
  27457,
  27460,
  27464,
  27467,
  27470,
  27473,
  27477,
  27480,
  27483,
  27487,
  27490,
  27493,
  27497,
  27500,
  27503,
  27507,
  27510,
  27513,
  27517,
  27520,
  27523,
  27527,
  27530,
  27533,
  27537,
  27540,
  27543,
  27547,
  27550,
  27553,
  27557,
  27560,
  27563,
  27566,
  27570,
  27573,
  27576,
  27580,
  27583,
  27586,
  27590,
  27593,
  27596,
  27600,
  27603,
  27606,
  27610,
  27613,
  27616,
  27620,
  27623,
  27626,
  27630,
  27633,
  27636,
  27640,
  27643,
  27646,
  27650,
  27653,
  27656,
  27660,
  27663,
  27666,
  27669,
  27673,
  27676,
  27679,
  27683,
  27686,
  27689,
  27693,
  27696,
  27699,
  27703,
  27706,
  27709,
  27713,
  27716,
  27719,
  27723,
  27726,
  27729,
  27733,
  27736,
  27739,
  27743,
  27746,
  27749,
  27753,
  27756,
  27759,
  27762,
  27766,
  27769,
  27772,
  27776,
  27779,
  27782,
  27786,
  27789,
  27792,
  27796,
  27799,
  27802,
  27806,
  27809,
  27812,
  27816,
  27819,
  27822,
  27826,
  27829,
  27832,
  27836,
  27839,
  27842,
  27846,
  27849,
  27852,
  27855,
  27859,
  27862,
  27865,
  27869,
  27872,
  27875,
  27879,
  27882,
  27885,
  27889,
  27892,
  27895,
  27899,
  27902,
  27905,
  27909,
  27912,
  27915,
  27919,
  27922,
  27925,
  27929,
  27932,
  27935,
  27939,
  27942,
  27945,
  27949,
  27952,
  27955,
  27958,
  27962,
  27965,
  27968,
  27972,
  27975,
  27978,
  27982,
  27985,
  27988,
  27992,
  27995,
  27998,
  28002,
  28005,
  28008,
  28012,
  28015,
  28018,
  28022,
  28025,
  28028,
  28032,
  28035,
  28038,
  28042,
  28045,
  28048,
  28051,
  28055,
  28058,
  28061,
  28065,
  28068,
  28071,
  28075,
  28078,
  28081,
  28085,
  28088,
  28091,
  28095,
  28098,
  28101,
  28105,
  28108,
  28111,
  28115,
  28118,
  28121,
  28125,
  28128,
  28131,
  28135,
  28138,
  28141,
  28145,
  28148,
  28151,
  28154,
  28158,
  28161,
  28164,
  28168,
  28171,
  28174,
  28178,
  28181,
  28184,
  28188,
  28191,
  28194,
  28198,
  28201,
  28204,
  28208,
  28211,
  28214,
  28218,
  28221,
  28224,
  28228,
  28231,
  28234,
  28238,
  28241,
  28244,
  28247,
  28251,
  28254,
  28257,
  28261,
  28264,
  28267,
  28271,
  28274,
  28277,
  28281,
  28284,
  28287,
  28291,
  28294,
  28297,
  28301,
  28304,
  28307,
  28311,
  28314,
  28317,
  28321,
  28324,
  28327,
  28331,
  28334,
  28337,
  28340,
  28344,
  28347,
  28350,
  28354,
  28357,
  28360,
  28364,
  28367,
  28370,
  28374,
  28377,
  28380,
  28384,
  28387,
  28390,
  28394,
  28397,
  28400,
  28404,
  28407,
  28410,
  28414,
  28417,
  28420,
  28424,
  28427,
  28430,
  28434,
  28437,
  28440,
  28443,
  28447,
  28450,
  28453,
  28457,
  28460,
  28463,
  28467,
  28470,
  28473,
  28477,
  28480,
  28483,
  28487,
  28490,
  28493,
  28497,
  28500,
  28503,
  28507,
  28510,
  28513,
  28517,
  28520,
  28523,
  28527,
  28530,
  28533,
  28536,
  28540,
  28543,
  28546,
  28550,
  28553,
  28556,
  28560,
  28563,
  28566,
  28570,
  28573,
  28576,
  28580,
  28583,
  28586,
  28590,
  28593,
  28596,
  28600,
  28603,
  28606,
  28610,
  28613,
  28616,
  28620,
  28623,
  28626,
  28630,
  28633,
  28636,
  28639,
  28643,
  28646,
  28649,
  28653,
  28656,
  28659,
  28663,
  28666,
  28669,
  28673,
  28676,
  28679,
  28683,
  28686,
  28689,
  28693,
  28696,
  28699,
  28703,
  28706,
  28709,
  28713,
  28716,
  28719,
  28723,
  28726,
  28729,
  28732,
  28736,
  28739,
  28742,
  28746,
  28749,
  28752,
  28756,
  28759,
  28762,
  28766,
  28769,
  28772,
  28776,
  28779,
  28782,
  28786,
  28789,
  28792,
  28796,
  28799,
  28802,
  28806,
  28809,
  28812,
  28816,
  28819,
  28822,
  28825,
  28829,
  28832,
  28835,
  28839,
  28842,
  28845,
  28849,
  28852,
  28855,
  28859,
  28862,
  28865,
  28869,
  28872,
  28875,
  28879,
  28882,
  28885,
  28889,
  28892,
  28895,
  28899,
  28902,
  28905,
  28909,
  28912,
  28915,
  28919,
  28922,
  28925,
  28928,
  28932,
  28935,
  28938,
  28942,
  28945,
  28948,
  28952,
  28955,
  28958,
  28962,
  28965,
  28968,
  28972,
  28975,
  28978,
  28982,
  28985,
  28988,
  28992,
  28995,
  28998,
  29002,
  29005,
  29008,
  29012,
  29015,
  29018,
  29021,
  29025,
  29028,
  29031,
  29035,
  29038,
  29041,
  29045,
  29048,
  29051,
  29055,
  29058,
  29061,
  29065,
  29068,
  29071,
  29075,
  29078,
  29081,
  29085,
  29088,
  29091,
  29095,
  29098,
  29101,
  29105,
  29108,
  29111,
  29115,
  29118,
  29121,
  29124,
  29128,
  29131,
  29134,
  29138,
  29141,
  29144,
  29148,
  29151,
  29154,
  29158,
  29161,
  29164,
  29168,
  29171,
  29174,
  29178,
  29181,
  29184,
  29188,
  29191,
  29194,
  29198,
  29201,
  29204,
  29208,
  29211,
  29214,
  29217,
  29221,
  29224,
  29227,
  29231,
  29234,
  29237,
  29241,
  29244,
  29247,
  29251,
  29254,
  29257,
  29261,
  29264,
  29267,
  29271,
  29274,
  29277,
  29281,
  29284,
  29287,
  29291,
  29294,
  29297,
  29301,
  29304,
  29307,
  29310,
  29314,
  29317,
  29320,
  29324,
  29327,
  29330,
  29334,
  29337,
  29340,
  29344,
  29347,
  29350,
  29354,
  29357,
  29360,
  29364,
  29367,
  29370,
  29374,
  29377,
  29380,
  29384,
  29387,
  29390,
  29394,
  29397,
  29400,
  29404,
  29407,
  29410,
  29413,
  29417,
  29420,
  29423,
  29427,
  29430,
  29433,
  29437,
  29440,
  29443,
  29447,
  29450,
  29453,
  29457,
  29460,
  29463,
  29467,
  29470,
  29473,
  29477,
  29480,
  29483,
  29487,
  29490,
  29493,
  29497,
  29500,
  29503,
  29506,
  29510,
  29513,
  29516,
  29520,
  29523,
  29526,
  29530,
  29533,
  29536,
  29540,
  29543,
  29546,
  29550,
  29553,
  29556,
  29560,
  29563,
  29566,
  29570,
  29573,
  29576,
  29580,
  29583,
  29586,
  29590,
  29593,
  29596,
  29600,
  29603,
  29606,
  29609,
  29613,
  29616,
  29619,
  29623,
  29626,
  29629,
  29633,
  29636,
  29639,
  29643,
  29646,
  29649,
  29653,
  29656,
  29659,
  29663,
  29666,
  29669,
  29673,
  29676,
  29679,
  29683,
  29686,
  29689,
  29693,
  29696,
  29699,
  29702,
  29706,
  29709,
  29712,
  29716,
  29719,
  29722,
  29726,
  29729,
  29732,
  29736,
  29739,
  29742,
  29746,
  29749,
  29752,
  29756,
  29759,
  29762,
  29766,
  29769,
  29772,
  29776,
  29779,
  29782,
  29786,
  29789,
  29792,
  29796,
  29799,
  29802,
  29805,
  29809,
  29812,
  29815,
  29819,
  29822,
  29825,
  29829,
  29832,
  29835,
  29839,
  29842,
  29845,
  29849,
  29852,
  29855,
  29859,
  29862,
  29865,
  29869,
  29872,
  29875,
  29879,
  29882,
  29885,
  29889,
  29892,
  29895,
  29898,
  29902,
  29905,
  29908,
  29912,
  29915,
  29918,
  29922,
  29925,
  29928,
  29932,
  29935,
  29938,
  29942,
  29945,
  29948,
  29952,
  29955,
  29958,
  29962,
  29965,
  29968,
  29972,
  29975,
  29978,
  29982,
  29985,
  29988,
  29991,
  29995,
  29998,
  30001,
  30005,
  30008,
  30011,
  30015,
  30018,
  30021,
  30025,
  30028,
  30031,
  30035,
  30038,
  30041,
  30045,
  30048,
  30051,
  30055,
  30058,
  30061,
  30065,
  30068,
  30071,
  30075,
  30078,
  30081,
  30085,
  30088,
  30091,
  30094,
  30098,
  30101,
  30104,
  30108,
  30111,
  30114,
  30118,
  30121,
  30124,
  30128,
  30131,
  30134,
  30138,
  30141,
  30144,
  30148,
  30151,
  30154,
  30158,
  30161,
  30164,
  30168,
  30171,
  30174,
  30178,
  30181,
  30184,
  30187,
  30191,
  30194,
  30197,
  30201,
  30204,
  30207,
  30211,
  30214,
  30217,
  30221,
  30224,
  30227,
  30231,
  30234,
  30237,
  30241,
  30244,
  30247,
  30251,
  30254,
  30257,
  30261,
  30264,
  30267,
  30271,
  30274,
  30277,
  30281,
  30284,
  30287,
  30290,
  30294,
  30297,
  30300,
  30304,
  30307,
  30310,
  30314,
  30317,
  30320,
  30324,
  30327,
  30330,
  30334,
  30337,
  30340,
  30344,
  30347,
  30350,
  30354,
  30357,
  30360,
  30364,
  30367,
  30370,
  30374,
  30377,
  30380,
  30383,
  30387,
  30390,
  30393,
  30397,
  30400,
  30403,
  30407,
  30410,
  30413,
  30417,
  30420,
  30423,
  30427,
  30430,
  30433,
  30437,
  30440,
  30443,
  30447,
  30450,
  30453,
  30457,
  30460,
  30463,
  30467,
  30470,
  30473,
  30476,
  30480,
  30483,
  30486,
  30490,
  30493,
  30496,
  30500,
  30503,
  30506,
  30510,
  30513,
  30516,
  30520,
  30523,
  30526,
  30530,
  30533,
  30536,
  30540,
  30543,
  30546,
  30550,
  30553,
  30556,
  30560,
  30563,
  30566,
  30570,
  30573,
  30576,
  30579,
  30583,
  30586,
  30589,
  30593,
  30596,
  30599,
  30603,
  30606,
  30609,
  30613,
  30616,
  30619,
  30623,
  30626,
  30629,
  30633,
  30636,
  30639,
  30643,
  30646,
  30649,
  30653,
  30656,
  30659,
  30663,
  30666,
  30669,
  30672,
  30676,
  30679,
  30682,
  30686,
  30689,
  30692,
  30696,
  30699,
  30702,
  30706,
  30709,
  30712,
  30716,
  30719,
  30722,
  30726,
  30729,
  30732,
  30736,
  30739,
  30742,
  30746,
  30749,
  30752,
  30756,
  30759,
  30762,
  30766,
  30769,
  30772,
  30775,
  30779,
  30782,
  30785,
  30789,
  30792,
  30795,
  30799,
  30802,
  30805,
  30809,
  30812,
  30815,
  30819,
  30822,
  30825,
  30829,
  30832,
  30835,
  30839,
  30842,
  30845,
  30849,
  30852,
  30855,
  30859,
  30862,
  30865,
  30868,
  30872,
  30875,
  30878,
  30882,
  30885,
  30888,
  30892,
  30895,
  30898,
  30902,
  30905,
  30908,
  30912,
  30915,
  30918,
  30922,
  30925,
  30928,
  30932,
  30935,
  30938,
  30942,
  30945,
  30948,
  30952,
  30955,
  30958,
  30961,
  30965,
  30968,
  30971,
  30975,
  30978,
  30981,
  30985,
  30988,
  30991,
  30995,
  30998,
  31001,
  31005,
  31008,
  31011,
  31015,
  31018,
  31021,
  31025,
  31028,
  31031,
  31035,
  31038,
  31041,
  31045,
  31048,
  31051,
  31055,
  31058,
  31061,
  31064,
  31068,
  31071,
  31074,
  31078,
  31081,
  31084,
  31088,
  31091,
  31094,
  31098,
  31101,
  31104,
  31108,
  31111,
  31114,
  31118,
  31121,
  31124,
  31128,
  31131,
  31134,
  31138,
  31141,
  31144,
  31148,
  31151,
  31154,
  31157,
  31161,
  31164,
  31167,
  31171,
  31174,
  31177,
  31181,
  31184,
  31187,
  31191,
  31194,
  31197,
  31201,
  31204,
  31207,
  31211,
  31214,
  31217,
  31221,
  31224,
  31227,
  31231,
  31234,
  31237,
  31241,
  31244,
  31247,
  31251,
  31254,
  31257,
  31260,
  31264,
  31267,
  31270,
  31274,
  31277,
  31280,
  31284,
  31287,
  31290,
  31294,
  31297,
  31300,
  31304,
  31307,
  31310,
  31314,
  31317,
  31320,
  31324,
  31327,
  31330,
  31334,
  31337,
  31340,
  31344,
  31347,
  31350,
  31353,
  31357,
  31360,
  31363,
  31367,
  31370,
  31373,
  31377,
  31380,
  31383,
  31387,
  31390,
  31393,
  31397,
  31400,
  31403,
  31407,
  31410,
  31413,
  31417,
  31420,
  31423,
  31427,
  31430,
  31433,
  31437,
  31440,
  31443,
  31446,
  31450,
  31453,
  31456,
  31460,
  31463,
  31466,
  31470,
  31473,
  31476,
  31480,
  31483,
  31486,
  31490,
  31493,
  31496,
  31500,
  31503,
  31506,
  31510,
  31513,
  31516,
  31520,
  31523,
  31526,
  31530,
  31533,
  31536,
  31540,
  31543,
  31546,
  31549,
  31553,
  31556,
  31559,
  31563,
  31566,
  31569,
  31573,
  31576,
  31579,
  31583,
  31586,
  31589,
  31593,
  31596,
  31599,
  31603,
  31606,
  31609,
  31613,
  31616,
  31619,
  31623,
  31626,
  31629,
  31633,
  31636,
  31639,
  31642,
  31646,
  31649,
  31652,
  31656,
  31659,
  31662,
  31666,
  31669,
  31672,
  31676,
  31679,
  31682,
  31686,
  31689,
  31692,
  31696,
  31699,
  31702,
  31706,
  31709,
  31712,
  31716,
  31719,
  31722,
  31726,
  31729,
  31732,
  31736,
  31739,
  31742,
  31745,
  31749,
  31752,
  31755,
  31759,
  31762,
  31765,
  31769,
  31772,
  31775,
  31779,
  31782,
  31785,
  31789,
  31792,
  31795,
  31799,
  31802,
  31805,
  31809,
  31812,
  31815,
  31819,
  31822,
  31825,
  31829,
  31832,
  31835,
  31838,
  31842,
  31845,
  31848,
  31852,
  31855,
  31858,
  31862,
  31865,
  31868,
  31872,
  31875,
  31878,
  31882,
  31885,
  31888,
  31892,
  31895,
  31898,
  31902,
  31905,
  31908,
  31912,
  31915,
  31918,
  31922,
  31925,
  31928,
  31931,
  31935,
  31938,
  31941,
  31945,
  31948,
  31951,
  31955,
  31958,
  31961,
  31965,
  31968,
  31971,
  31975,
  31978,
  31981,
  31985,
  31988,
  31991,
  31995,
  31998,
  32001,
  32005,
  32008,
  32011,
  32015,
  32018,
  32021,
  32025,
  32028,
  32031,
  32034,
  32038,
  32041,
  32044,
  32048,
  32051,
  32054,
  32058,
  32061,
  32064,
  32068,
  32071,
  32074,
  32078,
  32081,
  32084,
  32088,
  32091,
  32094,
  32098,
  32101,
  32104,
  32108,
  32111,
  32114,
  32118,
  32121,
  32124,
  32127,
  32131,
  32134,
  32137,
  32141,
  32144,
  32147,
  32151,
  32154,
  32157,
  32161,
  32164,
  32167,
  32171,
  32174,
  32177,
  32181,
  32184,
  32187,
  32191,
  32194,
  32197,
  32201,
  32204,
  32207,
  32211,
  32214,
  32217,
  32221,
  32224,
  32227,
  32230,
  32234,
  32237,
  32240,
  32244,
  32247,
  32250,
  32254,
  32257,
  32260,
  32264,
  32267,
  32270,
  32274,
  32277,
  32280,
  32284,
  32287,
  32290,
  32294,
  32297,
  32300,
  32304,
  32307,
  32310,
  32314,
  32317,
  32320,
  32323,
  32327,
  32330,
  32333,
  32337,
  32340,
  32343,
  32347,
  32350,
  32353,
  32357,
  32360,
  32363,
  32367,
  32370,
  32373,
  32377,
  32380,
  32383,
  32387,
  32390,
  32393,
  32397,
  32400,
  32403,
  32407,
  32410,
  32413,
  32417,
  32420,
  32423,
  32426,
  32430,
  32433,
  32436,
  32440,
  32443,
  32446,
  32450,
  32453,
  32456,
  32460,
  32463,
  32466,
  32470,
  32473,
  32476,
  32480,
  32483,
  32486,
  32490,
  32493,
  32496,
  32500,
  32503,
  32506,
  32510,
  32513,
  32516,
  32519,
  32523,
  32526,
  32529,
  32533,
  32536,
  32539,
  32543,
  32546,
  32549,
  32553,
  32556,
  32559,
  32563,
  32566,
  32569,
  32573,
  32576,
  32579,
  32583,
  32586,
  32589,
  32593,
  32596,
  32599,
  32603,
  32606,
  32609,
  32612,
  32616,
  32619,
  32622,
  32626,
  32629,
  32632,
  32636,
  32639,
  32642,
  32646,
  32649,
  32652,
  32656,
  32659,
  32662,
  32666,
  32669,
  32672,
  32676,
  32679,
  32682,
  32686,
  32689,
  32692,
  32696,
  32699,
  32702,
  32706,
  32709,
  32712,
  32715,
  32719,
  32722,
  32725,
  32729,
  32732,
  32735,
  32739,
  32742,
  32745,
  32749,
  32752,
  32755,
  32759,
  32762,
  32765,
  32769,
  32772,
  32775,
  32779,
  32782,
  32785,
  32789,
  32792,
  32795,
  32799,
  32802,
  32805,
  32808,
  32812,
  32815,
  32818,
  32822,
  32825,
  32828,
  32832,
  32835,
  32838,
  32842,
  32845,
  32848,
  32852,
  32855,
  32858,
  32862,
  32865,
  32868,
  32872,
  32875,
  32878,
  32882,
};

static const UINT256 multipliers1_binary80[] =
  { {{12415850090107640902ull, 14557465677128539270ull,
      4938398379086257084ull, 5255184001115807319ull}},
{{6296440575779775320ull, 18196832096410674088ull,
  1561311955430433451ull, 6568980001394759149ull}},
{{7870550719724719149ull, 18134354102085954706ull,
  6563325962715429718ull, 8211225001743448936ull}},
{{9530780218255337373ull, 6722285295376333787ull,
  4102078726697143574ull, 5132015626089655585ull}},
{{7301789254391783812ull, 17626228656075193042ull,
  9739284426798817371ull, 6415019532612069481ull}},
{{18350608604844505572ull, 17421099801666603398ull,
  16785791551925909618ull, 8018774415765086851ull}},
{{6857444359600428079ull, 15499873394469015028ull,
  8185276710739999559ull, 5011734009853179282ull}},
{{8571805449500535098ull, 14763155724658880881ull,
  1008223851570223641ull, 6264667512316474103ull}},
{{15326442830303056777ull, 4618886600541437389ull,
  15095337869744943264ull, 7830834390395592628ull}},
{{11884869778153104438ull, 2886804125338398368ull,
  211214131735813732ull, 4894271493997245393ull}},
{{14856087222691380547ull, 3608505156672997960ull,
  4875703683097155069ull, 6117839367496556741ull}},
{{123364954654674068ull, 9122317464268635355ull,
  10706315622298831740ull, 7647299209370695926ull}},
{{16218004161155028957ull, 14924820452022672904ull,
  2079761245509381933ull, 4779562005856684954ull}},
{{1825761127734234580ull, 4820967509746177419ull,
  11823073593741503225ull, 5974452507320856192ull}},
{{16117259464949956936ull, 10637895405610109677ull,
  14778841992176879031ull, 7468065634151070240ull}},
{{12379130174807417037ull, 13566213656147400404ull,
  9236776245110549394ull, 4667541021344418900ull}},
{{15473912718509271297ull, 7734395033329474697ull,
  11545970306388186743ull, 5834426276680523625ull}},
{{5507332842854425409ull, 5056307773234455468ull, 597404827703069717ull,
  7293032845850654532ull}},
{{6884166053568031761ull, 10932070734970457239ull,
  746756034628837146ull, 9116291057313318165ull}},
{{11220132811121101707ull, 11444230227783923678ull,
  2772565530856717168ull, 5697681910820823853ull}},
{{4801793977046601325ull, 14305287784729904598ull,
  8077392931998284364ull, 7122102388526029816ull}},
{{15225614508163027464ull, 17881609730912380747ull,
  10096741164997855455ull, 8902627985657537270ull}},
{{7210166058388198213ull, 18093535109461319823ull,
  1698777209696271755ull, 5564142491035960794ull}},
{{4401021554557859863ull, 18005232868399261875ull,
  11346843548975115502ull, 6955178113794950992ull}},
{{889590924769936924ull, 13283169048644301536ull,
  14183554436218894378ull, 8693972642243688740ull}},
{{555994327981210578ull, 12913666673830076364ull,
  18088093559491584794ull, 5433732901402305462ull}},
{{694992909976513222ull, 6918711305432819647ull,
  13386744912509705185ull, 6792166126752881828ull}},
{{14703799192752805239ull, 13260075150218412462ull,
  16733431140637131481ull, 8490207658441102285ull}},
{{4578188477043115371ull, 1370017941245425933ull,
  12764237472111901128ull, 5306379786525688928ull}},
{{10334421614731282117ull, 1712522426556782416ull,
  15955296840139876410ull, 6632974733157111160ull}},
{{12918027018414102647ull, 11364025070050753828ull,
  1497376976465293896ull, 8291218416446388951ull}},
{{17297138923363589962ull, 7102515668781721142ull,
  7853389637931890541ull, 5182011510278993094ull}},
{{12398051617349711645ull, 13489830604404539332ull,
  593365010560087368ull, 6477514387848741368ull}},
{{15497564521687139556ull, 16862288255505674165ull,
  741706263200109210ull, 8096892984810926710ull}},
{{11991820835268156175ull, 15150616178118434257ull,
  14298624469782231968ull, 5060558115506829193ull}},
{{1154717988803031506ull, 491526148938491206ull, 4038222531945626249ull,
  6325697644383536492ull}},
{{10666769522858565191ull, 5226093704600501911ull,
  5047778164932032811ull, 7907122055479420615ull}},
{{13584259979427685100ull, 960465556161619742ull,
  10072390380723602363ull, 4941951284674637884ull}},
{{7756952937429830567ull, 15035640000484188390ull,
  12590487975904502953ull, 6177439105843297355ull}},
{{472819134932512401ull, 4959491945323071776ull,
  11126423951453240788ull, 7721798882304121694ull}},
{{295511959332820251ull, 12323054502681695668ull,
  2342328951230887588ull, 4826124301440076059ull}},
{{369389949166025313ull, 15403818128352119585ull,
  16762969244320773197ull, 6032655376800095073ull}},
{{5073423454884919546ull, 5419714605157985769ull,
  7118653500118802785ull, 7540819221000118842ull}},
{{14700104705371544476ull, 14916536674292210865ull,
  9060844456001639644ull, 4713012013125074276ull}},
{{4540072826432266883ull, 198926769155711966ull,
  11326055570002049556ull, 5891265016406342845ull}},
{{14898463069895109412ull, 248658461444639957ull, 322511407220398233ull,
  7364081270507928557ull}},
{{4788020782086723053ull, 4922509095233187851ull,
  5014825277452885695ull, 9205101588134910696ull}},
{{686669979590507956ull, 9994097212161824263ull, 3134265798408053559ull,
  5753188492584319185ull}},
{{14693395529770298657ull, 7880935496774892424ull,
  8529518266437454853ull, 7191485615730398981ull}},
{{18366744412212873321ull, 14462855389396003434ull,
  15273583851474206470ull, 8989357019662998726ull}},
{{16090901276060433730ull, 4427598599945114242ull,
  4934303888743991140ull, 5618348137289374204ull}},
{{10890254558220766354ull, 5534498249931392803ull,
  6167879860929988925ull, 7022935171611717755ull}},
{{9001132179348570039ull, 11529808830841628908ull,
  3098163807735098252ull, 8778668964514647194ull}},
{{14849079648947632082ull, 16429502556130793875ull,
  6548038398261824311ull, 5486668102821654496ull}},
{{13949663542757152199ull, 15925192176736104440ull,
  8185047997827280389ull, 6858335128527068120ull}},
{{17437079428446440248ull, 6071432165637966838ull,
  10231309997284100487ull, 8572918910658835150ull}},
{{6286488624351637251ull, 10712174131164811130ull,
  1782882729875174900ull, 5358074319161771969ull}},
{{17081482817294322372ull, 13390217663956013912ull,
  6840289430771356529ull, 6697592898952214961ull}},
{{2905109447908351349ull, 2902714024662853679ull,
  13162047806891583566ull, 8371991123690268701ull}},
{{8733222432583801449ull, 15649254320696447261ull,
  10532122888520933680ull, 5232494452306417938ull}},
{{15528214059157139716ull, 1114823827161007460ull,
  3941781573796391293ull, 6540618065383022423ull}},
{{963523500236873028ull, 6005215802378647230ull, 315540948818101212ull,
  8175772581728778029ull}},
{{14437260242930209355ull, 12976631913341430326ull,
  2503056102225007209ull, 5109857863580486268ull}},
{{8823203266807985885ull, 2385731836394624196ull,
  3128820127781259012ull, 6387322329475607835ull}},
{{11029004083509982357ull, 2982164795493280245ull,
  17746083215008737477ull, 7984152911844509793ull}},
{{9198970561407432925ull, 4169696006396994105ull,
  4173772981739379067ull, 4990095569902818621ull}},
{{16110399220186679060ull, 600433989568854727ull,
  9828902245601611738ull, 6237619462378523276ull}},
{{15526313006805960921ull, 9973914523815844217ull,
  12286127807002014672ull, 7797024327973154095ull}},
{{2786416601612643720ull, 6233696577384902636ull,
  14596358907017341026ull, 4873140204983221309ull}},
{{3483020752015804650ull, 17015492758585904103ull,
  4410390578489512570ull, 6091425256229026637ull}},
{{18188833995301919524ull, 12045993911377604320ull,
  10124674241539278617ull, 7614281570286283296ull}},
{{11368021247063699703ull, 611217166969920844ull,
  6327921400962049136ull, 4758925981428927060ull}},
{{14210026558829624628ull, 764021458712401055ull,
  7909901751202561420ull, 5948657476786158825ull}},
{{13150847180109642881ull, 955026823390501319ull,
  14499063207430589679ull, 7435821845982698531ull}},
{{15136808515209608657ull, 7514420792260145180ull,
  6756071495430424597ull, 4647388653739186582ull}},
{{474266570302459205ull, 14004712008752569380ull,
  17668461406142806554ull, 5809235817173983227ull}},
{{592833212878074006ull, 8282517974085935917ull,
  17473890739251120289ull, 7261544771467479034ull}},
{{5352727534524980412ull, 14964833486034807800ull,
  12618991387209124553ull, 9076930964334348793ull}},
{{3345454709078112758ull, 2435491901130673019ull, 969340589364620990ull,
  5673081852708967996ull}},
{{18016876441629804659ull, 12267736913268117081ull,
  1211675736705776237ull, 7091352315886209995ull}},
{{8686037496755092111ull, 1499613086302982640ull,
  15349652726164384009ull, 8864190394857762493ull}},
{{5428773435471932570ull, 12466473225007833910ull,
  11899375963066433957ull, 5540118996786101558ull}},
{{16009338831194691520ull, 1748033475977628675ull,
  5650847916978266639ull, 6925148745982626948ull}},
{{15399987520565976496ull, 16020099900254199556ull,
  7063559896222833298ull, 8656435932478283685ull}},
{{401620163498959502ull, 14624248456086262627ull,
  6720567944352964763ull, 5410272457798927303ull}},
{{14337083259655863090ull, 13668624551680440379ull,
  3789023912013818050ull, 6762840572248659129ull}},
{{13309668056142440958ull, 7862408652745774666ull,
  9347965908444660467ull, 8453550715310823911ull}},
{{12930228553516413503ull, 2608162398752415214ull,
  12760007720418994648ull, 5283469197069264944ull}},
{{6939413655040741070ull, 3260202998440519018ull,
  15950009650523743310ull, 6604336496336581180ull}},
{{17897639105655702146ull, 13298625784905424580ull,
  1490767989445127521ull, 8255420620420726476ull}},
{{1962652404180038033ull, 1394112087924808507ull,
  10155102030257980509ull, 5159637887762954047ull}},
{{16288373560507211254ull, 6354326128333398537ull,
  8082191519395087732ull, 6449547359703692559ull}},
{{6525408895351850355ull, 7942907660416748172ull,
  5491053380816471761ull, 8061934199629615699ull}},
{{13301752596449682280ull, 16493532333828937367ull,
  1126065353796600898ull, 5038708874768509812ull}},
{{12015504727134714946ull, 11393543380431395901ull,
  1407581692245751123ull, 6298386093460637265ull}},
{{1184322853636229970ull, 9630243207111856973ull,
  6371163133734576808ull, 7872982616825796581ull}},
{{3046044792736337684ull, 6018902004444910608ull,
  6287819967797804457ull, 4920614135516122863ull}},
{{3807555990920422104ull, 12135313523983526164ull,
  3248088941319867667ull, 6150767669395153579ull}},
{{4759444988650527630ull, 10557455886552019801ull,
  17895169231931998296ull, 7688459586743941973ull}},
{{14503868163975049529ull, 6598409929095012375ull,
  13490323779171192887ull, 4805287241714963733ull}},
{{13518149186541424007ull, 3636326392941377565ull,
  3027846668681827397ull, 6006609052143704667ull}},
{{3062628427894616297ull, 9157094009604109861ull,
  17619866391134447958ull, 7508261315179630833ull}},
{{4219985776647829138ull, 1111497737575180759ull,
  4094887466817948118ull, 4692663321987269271ull}},
{{663296202382398518ull, 10612744208823751757ull, 506923315095047243ull,
  5865829152484086589ull}},
{{5440806271405386052ull, 8654244242602301792ull,
  5245340162296196958ull, 7332286440605108236ull}},
{{6801007839256732564ull, 1594433266398101432ull,
  6556675202870246198ull, 9165358050756385295ull}},
{{4250629899535457853ull, 14831578846780977107ull,
  11015451029434985729ull, 5728348781722740809ull}},
{{701601355991934412ull, 4704415503194057672ull,
  18380999805221120066ull, 7160435977153426011ull}},
{{877001694989918015ull, 15103891415847347898ull,
  18364563738099012178ull, 8950544971441782514ull}},
{{5159812077796086663ull, 14051618153331980340ull,
  16089538354739270515ull, 5594090607151114071ull}},
{{6449765097245108329ull, 12952836673237587521ull,
  15500236924996700240ull, 6992613258938892589ull}},
{{12673892389983773315ull, 16191045841546984401ull,
  5540238100963711588ull, 8740766573673615737ull}},
{{1003653716098776466ull, 896031614112089443ull,
  14991863859170789503ull, 5462979108546009835ull}},
{{15089625200405634295ull, 14955097572922275515ull,
  14128143805536098974ull, 6828723885682512294ull}},
{{14250345482079654964ull, 9470499929298068586ull,
  8436807720065347910ull, 8535904857103140368ull}},
{{13518151944727172257ull, 1307376437383904962ull,
  5273004825040842444ull, 5334940535689462730ull}},
{{7674317894054189513ull, 1634220546729881203ull,
  15814628068155828863ull, 6668675669611828412ull}},
{{4981211349140348987ull, 15877833738694515216ull,
  1321541011485234462ull, 8335844587014785516ull}},
{{3113257093212718117ull, 5311960068256684106ull,
  10049335169033047347ull, 5209902866884240947ull}},
{{13114943403370673454ull, 2028264066893467228ull,
  7949982942863921280ull, 6512378583605301184ull}},
{{16393679254213341818ull, 2535330083616834035ull,
  9937478678579901600ull, 8140473229506626480ull}},
{{7940206524669644684ull, 1584581302260521272ull,
  6210924174112438500ull, 5087795768441641550ull}},
{{9925258155837055855ull, 1980726627825651590ull,
  16987027254495323933ull, 6359744710552051937ull}},
{{3183200657941544011ull, 7087594303209452392ull,
  7398726012836991204ull, 7949680888190064922ull}},
{{1989500411213465007ull, 13653118476360683553ull,
  9235889776450507406ull, 4968550555118790576ull}},
{{7098561532444219163ull, 7843026058596078633ull,
  11544862220563134258ull, 6210688193898488220ull}},
{{13484887933982661857ull, 580410536390322483ull,
  14431077775703917823ull, 7763360242373110275ull}},
{{6122211949525469709ull, 7280285612885033408ull,
  6713580600601254687ull, 4852100151483193922ull}},
{{7652764936906837136ull, 4488670997678903856ull,
  17615347787606344167ull, 6065125189353992402ull}},
{{9565956171133546420ull, 999152728671241916ull,
  12795812697653154401ull, 7581406486692490503ull}},
{{15202094643813242321ull, 12153685501487995957ull,
  14914911963674303356ull, 4738379054182806564ull}},
{{5167560249484389189ull, 15192106876859994947ull,
  196895880883327579ull, 5922973817728508206ull}},
{{1847764293428098582ull, 14378447577647605780ull,
  9469491887958935282ull, 7403717272160635257ull}},
{{10378224720247337422ull, 13598215754457141516ull,
  17447647476042804311ull, 4627323295100397035ull}},
{{12972780900309171777ull, 12386083674644038991ull,
  17197873326626117485ull, 5784154118875496294ull}},
{{11604290106959076817ull, 1647546538022885027ull,
  12273969621427871049ull, 7230192648594370368ull}},
{{9893676615271458117ull, 6671119190955994188ull,
  15342462026784838811ull, 9037740810742962960ull}},
{{15406919921399437132ull, 1863606485133802415ull,
  9589038766740524257ull, 5648588006714351850ull}},
{{14646963883321908510ull, 6941194124844640923ull,
  2762926421570879513ull, 7060735008392939813ull}},
{{13697018835724997734ull, 13288178674483189058ull,
  8065344045390987295ull, 8825918760491174766ull}},
{{13172322790755511488ull, 15222640699193075017ull,
  429154009941979155ull, 5516199225306984229ull}},
{{2630345433162225648ull, 14416614855563955868ull,
  5148128530854861848ull, 6895249031633730286ull}},
{{3287931791452782059ull, 18020768569454944835ull,
  15658532700423353118ull, 8619061289542162857ull}},
{{18195858434153846451ull, 6651294337481952617ull,
  2869053910123513843ull, 5386913305963851786ull}},
{{8909764987410144352ull, 3702431903425052868ull,
  12809689424509168112ull, 6733641632454814732ull}},
{{11137206234262680440ull, 4628039879281316085ull,
  16012111780636460140ull, 8417052040568518415ull}},
{{9266596905627869227ull, 12115896961405598361ull,
  16925098890538869443ull, 5260657525355324009ull}},
{{16194932150462224438ull, 10533185183329610047ull,
  7321315557891423092ull, 6575821906694155012ull}},
{{15631979169650392643ull, 13166481479162012559ull,
  9151644447364278865ull, 8219777383367693765ull}},
{{16687516008672577258ull, 1311521896835175993ull,
  8025620788816368243ull, 5137360864604808603ull}},
{{7024336955558557860ull, 15474460426326133704ull,
  5420339967593072399ull, 6421701080756010754ull}},
{{8780421194448197325ull, 14731389514480279226ull,
  15998796996346116307ull, 8027126350945013442ull}},
{{10099449264957511233ull, 6901275437336480564ull,
  14610934141143710596ull, 5016953969340633401ull}},
{{12624311581196889041ull, 8626594296670600705ull,
  4428609621147474533ull, 6271192461675791752ull}},
{{1945331421213947589ull, 15394928889265638786ull,
  5535762026434343166ull, 7838990577094739690ull}},
{{5827518156686105147ull, 5010144537363636337ull,
  8071537284948852383ull, 4899369110684212306ull}},
{{11896083714285019338ull, 1650994653277157517ull,
  866049569331289671ull, 6124211388355265383ull}},
{{1035046587574110460ull, 15898801371878610609ull,
  14917620016946275800ull, 7655264235444081728ull}},
{{12176119163302288798ull, 9936750857424131630ull,
  9323512510591422375ull, 4784540147152551080ull}},
{{5996776917273085189ull, 7809252553352776634ull,
  11654390638239277969ull, 5980675183940688850ull}},
{{16719343183446132294ull, 14373251710118358696ull,
  5344616260944321653ull, 7475843979925861063ull}},
{{10449589489653832684ull, 11289125328037668137ull,
  10257914190731282889ull, 4672402487453663164ull}},
{{17673672880494678759ull, 276348604764921459ull,
  12822392738414103612ull, 5840503109317078955ull}},
{{17480405082190960545ull, 345435755956151824ull,
  11416304904590241611ull, 7300628886646348694ull}},
{{3403762279029149065ull, 14266852750227353493ull,
  5047009093883026205ull, 9125786108307935868ull}},
{{4433194433606912118ull, 11222625978105789885ull,
  12377752720531667186ull, 5703616317692459917ull}},
{{10153179060436028051ull, 4804910435777461548ull,
  1637132845382420271ull, 7129520397115574897ull}},
{{12691473825545035063ull, 1394452026294439031ull,
  6658102075155413243ull, 8911900496394468621ull}},
{{14849700168606728771ull, 17012433580929882058ull,
  6467156806185827228ull, 5569937810246542888ull}},
{{9338753173903635155ull, 2818797902452800957ull,
  8083946007732284036ull, 6962422262808178610ull}},
{{16285127485806931848ull, 3523497378066001196ull,
  881560472810579237ull, 8703027828510223263ull}},
{{954832641774556597ull, 4508028870504944700ull, 7468504323147693879ull,
  5439392392818889539ull}},
{{1193540802218195746ull, 1023350069703792971ull,
  4723944385507229445ull, 6799240491023611924ull}},
{{15326984058054908395ull, 5890873605557129117ull,
  5904930481884036806ull, 8499050613779514905ull}},
{{11885208045498011699ull, 17516854058755369410ull,
  15219796597245992763ull, 5311906633612196815ull}},
{{5633138020017738816ull, 17284381555016823859ull,
  14413059728130103050ull, 6639883292015246019ull}},
{{2429736506594785615ull, 12382104906916254016ull,
  13404638641735240909ull, 8299854115019057524ull}},
{{1518585316621741010ull, 10044658576036352712ull,
  17601271187939301376ull, 5187408821886910952ull}},
{{1898231645777176262ull, 12555823220045440890ull,
  3554844911214575104ull, 6484261027358638691ull}},
{{11596161594076246136ull, 15694779025056801112ull,
  18278614194300382592ull, 8105326284198298363ull}},
{{7247600996297653835ull, 9809236890660500695ull,
  9118290862224045168ull, 5065828927623936477ull}},
{{4447815226944679390ull, 12261546113325625869ull,
  16009549596207444364ull, 6332286159529920596ull}},
{{10171455052108237141ull, 15326932641657032336ull,
  1565192921549753839ull, 7915357699412400746ull}},
{{6357159407567648213ull, 16496861928676727066ull,
  5589931594395984053ull, 4947098562132750466ull}},
{{17169821296314336074ull, 6786019355563745120ull,
  16210786529849755875ull, 6183873202665938082ull}},
{{3015532546683368477ull, 3870838176027293497ull,
  11040111125457419036ull, 7729841503332422603ull}},
{{13413922887745575058ull, 11642645896871834243ull,
  4594226444197192945ull, 4831150939582764127ull}},
{{12155717591254580919ull, 718249315807629092ull,
  1131097036819103278ull, 6038938674478455159ull}},
{{15194646989068226148ull, 10121183681614312173ull,
  15248929351306042809ull, 7548673343098068948ull}},
{{11802497377381335295ull, 17854954847077414868ull,
  307208807711500947ull, 4717920839436293093ull}},
{{14753121721726669118ull, 17707007540419380681ull,
  4995697028066764088ull, 5897401049295366366ull}},
{{4606344096876172686ull, 3687015351814674236ull,
  15467993321938230919ull, 7371751311619207957ull}},
{{5757930121095215857ull, 18443827245050506507ull,
  5499933597140624936ull, 9214689139524009947ull}},
{{1292863316470815959ull, 11527392028156566567ull,
  1131615488999196633ull, 5759180712202506217ull}},
{{15451137200870683660ull, 574181979913544496ull,
  6026205379676383696ull, 7198975890253132771ull}},
{{867177427378802959ull, 717727474891930621ull, 2921070706168091716ull,
  8998719862816415964ull}},
{{2847828901325445802ull, 9671951708662232446ull,
  11049041228209833130ull, 5624199914260259977ull}},
{{12783158163511583060ull, 2866567598973014749ull,
  18422987553689679317ull, 7030249892825324971ull}},
{{2143889649107315113ull, 8194895517143656341ull,
  18417048423684711242ull, 8787812366031656214ull}},
{{3645774039905765898ull, 9733495716642173117ull,
  6898969246375556622ull, 5492382728769785134ull}},
{{9168903568309595276ull, 2943497608947940588ull,
  17847083594824221586ull, 6865478410962231417ull}},
{{11461129460386994095ull, 12902744048039701543ull,
  8473796438248113270ull, 8581848013702789272ull}},
{{14080734940382953166ull, 3452529011597425560ull,
  5296122773905070794ull, 5363655008564243295ull}},
{{17600918675478691457ull, 13539033301351557758ull,
  2008467448953950588ull, 6704568760705304119ull}},
{{12777776307493588513ull, 16923791626689447198ull,
  16345642366474601947ull, 8380710950881630148ull}},
{{3374424173756104917ull, 8271526757467210547ull, 992654442191850409ull,
  5237944344301018843ull}},
{{18053088272477294858ull, 14951094465261401087ull,
  15075876108021976723ull, 6547430430376273553ull}},
{{17954674322169230668ull, 14077182063149363455ull,
  5009787079745307192ull, 8184288037970341942ull}},
{{18139200478996851024ull, 8798238789468352159ull,
  16966174980122980707ull, 5115180023731463713ull}},
{{18062314580318675876ull, 6386112468408052295ull,
  7372660669871562172ull, 6393975029664329642ull}},
{{17966207206970956941ull, 7982640585510065369ull,
  18439197874194228523ull, 7992468787080412052ull}},
{{4311350476715766232ull, 2683307356730096904ull,
  2301126634516617019ull, 4995292991925257533ull}},
{{5389188095894707790ull, 17189192251194784842ull,
  7488094311573159177ull, 6244116239906571916ull}},
{{15959857156723160545ull, 7651432258711317340ull,
  9360117889466448972ull, 7805145299883214895ull}},
{{751538686097199533ull, 14005517198549349146ull,
  12767602708557612463ull, 4878215812427009309ull}},
{{10162795394476275224ull, 12895210479759298528ull,
  2124445330414851867ull, 6097769765533761637ull}},
{{12703494243095344030ull, 11507327081271735256ull,
  7267242681445952738ull, 7622212206917202046ull}},
{{7939683901934590019ull, 11803765444222222439ull,
  18377084731185884173ull, 4763882629323251278ull}},
{{5312918858990849620ull, 919648749995614337ull,
  13747983877127579409ull, 5954853286654064098ull}},
{{11252834592165949928ull, 5761246955921905825ull,
  7961607809554698453ull, 7443566608317580123ull}},
{{115492592462636849ull, 5906622356664885093ull, 2670161871757992581ull,
  4652229130198487577ull}},
{{4756051759005683966ull, 11994963964258494270ull,
  7949388358124878630ull, 5815286412748109471ull}},
{{15168436735611880765ull, 5770332918468342029ull,
  5325049429228710384ull, 7269108015935136839ull}},
{{5125487864232687244ull, 7212916148085427537ull,
  2044625768108500076ull, 9086385019918921049ull}},
{{14732644961213899288ull, 13731444629408168018ull,
  12807106151136282307ull, 5678990637449325655ull}},
{{9192434164662598301ull, 12552619768332822119ull,
  11397196670492964980ull, 7098738296811657069ull}},
{{6878856687400859973ull, 15690774710416027649ull,
  411437782834042513ull, 8873422871014571337ull}},
{{15828500475694007243ull, 2889205166368935424ull,
  11786363660339746331ull, 5545889294384107085ull}},
{{1338881520907957438ull, 17446564513243332993ull,
  897896520142519201ull, 6932361617980133857ull}},
{{6285287919562334701ull, 7973147586272002529ull,
  5734056668605536906ull, 8665452022475167321ull}},
{{15457519995794928948ull, 9594903259847389484ull,
  15113000463946930326ull, 5415907514046979575ull}},
{{875155921034109569ull, 2770257037954461048ull,
  14279564561506275004ull, 6769884392558724469ull}},
{{1093944901292636962ull, 3462821297443076310ull,
  4014397646600680043ull, 8462355490698405587ull}},
{{14518773618590061813ull, 18305164375397780357ull,
  203155519911731074ull, 5288972181686503492ull}},
{{4313408967955413554ull, 13658083432392449639ull,
  253944399889663843ull, 6611215227108129365ull}},
{{780075191516879039ull, 12460918272063174145ull,
  4929116518289467708ull, 8264019033885161706ull}},
{{12016762040766519159ull, 17011445956894259648ull,
  7692383842358305221ull, 5165011896178226066ull}},
{{15020952550958148949ull, 7429249390835660848ull,
  392107766093105719ull, 6456264870222782583ull}},
{{329446614988134570ull, 4674875720117188157ull,
  14325192762898545861ull, 8070331087778478228ull}},
{{2511747143581278059ull, 5227640334286936550ull,
  18176617513666366971ull, 5043956929861548892ull}},
{{12363055966331373381ull, 1922864399431282783ull,
  4274027818373407098ull, 6304946162326936116ull}},
{{10842133939486828822ull, 11626952536143879287ull,
  5342534772966758872ull, 7881182702908670145ull}},
{{13693862739820349870ull, 7266845335089924554ull,
  14868299279172694055ull, 4925739189317918840ull}},
{{7893956387920661530ull, 4471870650435017789ull, 138630025256315953ull,
  6157173986647398551ull}},
{{14479131503328214816ull, 10201524331471160140ull,
  14008345586852558653ull, 7696467483309248188ull}},
{{18272829226434910068ull, 8681795716383169039ull,
  17978588028637624966ull, 4810292177068280117ull}},
{{18229350514616249681ull, 1628872608624185491ull,
  8638176980514867496ull, 6012865221335350147ull}},
{{18175002124842924197ull, 2036090760780231864ull,
  6186035207216196466ull, 7516081526669187684ull}},
{{11359376328026827623ull, 5884242743915032819ull,
  13089644041364898599ull, 4697550954168242302ull}},
{{9587534391606146625ull, 2743617411466403120ull,
  7138683014851347441ull, 5871938692710302878ull}},
{{11984417989507683281ull, 8041207782760391804ull,
  18146725805418960109ull, 7339923365887878597ull}},
{{14980522486884604101ull, 14663195746877877659ull,
  8848349201491536424ull, 9174904207359848247ull}},
{{7056983545089183612ull, 9164497341798673537ull,
  12447747278573292121ull, 5734315129599905154ull}},
{{13432915449788867418ull, 16067307695675729825ull,
  6336312061361839343ull, 7167893911999881443ull}},
{{2956086256953920561ull, 15472448601167274378ull,
  3308704058274911275ull, 8959867389999851804ull}},
{{6459239929023588255ull, 7364437366515852534ull,
  11291312073276595355ull, 5599917118749907377ull}},
{{17297421948134261126ull, 4593860689717427763ull,
  279082036313580482ull, 6999896398437384222ull}},
{{17010091416740438504ull, 14965697899001560512ull,
  9572224582246751410ull, 8749870498046730277ull}},
{{10631307135462774065ull, 13965247205303363224ull,
  8288483373117913583ull, 5468669061279206423ull}},
{{13289133919328467581ull, 12844872988201816126ull,
  5748918197970004075ull, 6835836326599008029ull}},
{{7388045362305808668ull, 11444405216824882254ull,
  11797833765889892998ull, 8544795408248760036ull}},
{{5842333013742514ull, 2541067242088163505ull, 16597018140535958932ull,
  5340497130155475022ull}},
{{4618988934694566046ull, 3176334052610204381ull,
  11522900638815172857ull, 6675621412694343778ull}},
{{10385422186795595462ull, 8582103584190143380ull,
  5180253761664190263ull, 8344526765867929723ull}},
{{15714260903602022972ull, 12281343767759921468ull,
  931815591826424962ull, 5215329228667456077ull}},
{{1196082055792977098ull, 6128307672845126028ull,
  5776455508210419107ull, 6519161535834320096ull}},
{{1495102569741221373ull, 3048698572629019631ull,
  7220569385263023884ull, 8148951919792900120ull}},
{{7851968133729345214ull, 11128808644747913077ull,
  4512855865789389927ull, 5093094949870562575ull}},
{{14426646185589069422ull, 9299324787507503442ull,
  1029383813809349505ull, 6366368687338203219ull}},
{{8809935695131560969ull, 16235842002811767207ull,
  15121787822543850593ull, 7957960859172754023ull}},
{{12423738837098307462ull, 3229872224116272648ull,
  16368646416730988477ull, 4973725536982971264ull}},
{{15529673546372884327ull, 8649026298572728714ull,
  2014063947204183980ull, 6217156921228714081ull}},
{{10188719896111329601ull, 10811282873215910893ull,
  7129265952432617879ull, 7771446151535892601ull}},
{{8673792944283274953ull, 13674580823401026164ull,
  15985006266338855934ull, 4857153844709932875ull}},
{{10842241180354093691ull, 7869853992396506897ull,
  15369571814496182014ull, 6071442305887416094ull}},
{{18164487493870005017ull, 613945453640857813ull,
  9988592731265451710ull, 7589302882359270118ull}},
{{13658647692882447088ull, 14218773963807699845ull,
  1631184438613519414ull, 4743314301474543824ull}},
{{3238251560820895148ull, 8550095417904848999ull,
  2038980548266899268ull, 5929142876843179780ull}},
{{17882872506308282647ull, 10687619272381061248ull,
  2548725685333624085ull, 7411428596053974725ull}},
{{11176795316442676654ull, 8985605054451857232ull,
  3898796562547209005ull, 4632142872533734203ull}},
{{13970994145553345818ull, 15843692336492209444ull,
  261809684756623352ull, 5790178590667167754ull}},
{{17463742681941682272ull, 1357871346905710189ull,
  9550634142800554999ull, 7237723238333959692ull}},
{{7994620297144939128ull, 15532397238914301449ull,
  11938292678500693748ull, 9047154047917449615ull}},
{{16525852731784056715ull, 484376237466662597ull,
  14378961951704015449ull, 5654471279948406009ull}},
{{6822257859447907182ull, 5217156315260716151ull,
  4138644384347855599ull, 7068089099935507512ull}},
{{3916136305882496073ull, 1909759375648507285ull,
  5173305480434819499ull, 8835111374919384390ull}},
{{4753428200390253998ull, 17334500674276174717ull,
  17068373980553925898ull, 5521944609324615243ull}},
{{10553471268915205401ull, 12444753805990442588ull,
  16723781457265019469ull, 6902430761655769054ull}},
{{13191839086144006752ull, 1720884202205889523ull,
  11681354784726498529ull, 8628038452069711318ull}},
{{5939056419626310268ull, 12604767672447150712ull,
  2689160722026673676ull, 5392524032543569574ull}},
{{7423820524532887835ull, 15755959590558938390ull,
  12584822939388117903ull, 6740655040679461967ull}},
{{56403618811333985ull, 15083263469771285084ull,
  11119342655807759475ull, 8425818800849327459ull}},
{{9258624298611859549ull, 7121196659393359225ull,
  4643746150666155720ull, 5266136750530829662ull}},
{{16184966391692212340ull, 8901495824241699031ull,
  15028054725187470458ull, 6582670938163537077ull}},
{{15619521971187877521ull, 1903497743447347981ull,
  4950010351202174361ull, 8228338672704421347ull}},
{{12068044241206117403ull, 12718901135723062248ull,
  787913460287665023ull, 5142711670440263342ull}},
{{15085055301507646754ull, 11286940401226439906ull,
  10208263862214357087ull, 6428389588050329177ull}},
{{9632947090029782634ull, 9496989483105661979ull,
  17372015846195334263ull, 8035486985062911471ull}},
{{3714748922054920194ull, 12853147454582120593ull,
  17775038931513165770ull, 5022179365664319669ull}},
{{9255122170996038147ull, 6843062281372874933ull,
  8383740609109293501ull, 6277724207080399587ull}},
{{16180588732172435587ull, 13165513870143481570ull,
  5867989742959228972ull, 7847155258850499484ull}},
{{14724553976035160146ull, 17451818205694451789ull,
  12890865626204293915ull, 4904472036781562177ull}},
{{4570634414761786471ull, 17203086738690676833ull,
  2278523977473203682ull, 6130590045976952722ull}},
{{10324979036879620992ull, 12280486386508570233ull,
  12071527008696280411ull, 7663237557471190902ull}},
{{17982326944118232880ull, 5369460982354162443ull,
  2933018362007787353ull, 4789523473419494314ull}},
{{17866222661720403196ull, 11323512246370090958ull,
  12889644989364509999ull, 5986904341774367892ull}},
{{13109406290295728187ull, 9542704289535225794ull,
  16112056236705637499ull, 7483630427217959865ull}},
{{12805064949862218021ull, 3658347171745822169ull,
  3152506120299941581ull, 4677269017011224916ull}},
{{2171273132045608814ull, 9184619983109665616ull,
  3940632650374926976ull, 5846586271264031145ull}},
{{2714091415057011018ull, 11480774978887082020ull,
  9537476831396046624ull, 7308232839080038931ull}},
{{3392614268821263772ull, 14350968723608852525ull,
  7310160020817670376ull, 9135291048850048664ull}},
{{4426226927226983810ull, 8969355452255532828ull,
  4568850013011043985ull, 5709556905531280415ull}},
{{5532783659033729762ull, 15823380333746803939ull,
  1099376497836417077ull, 7136946131914100519ull}},
{{2304293555364774299ull, 5944167361901341212ull,
  15209278677577685059ull, 8921182664892625648ull}},
{{10663555508957759745ull, 1409261591974644305ull,
  9505799173486053162ull, 5575739165557891030ull}},
{{17941130404624587585ull, 10984949026823081189ull,
  2658876930002790644ull, 6969673956947363788ull}},
{{8591354950498570769ull, 13731186283528851487ull,
  3323596162503488305ull, 8712092446184204735ull}},
{{12287125871702688587ull, 1664462399564450323ull,
  8994776629205762047ull, 5445057778865127959ull}},
{{10747221321200972829ull, 15915636054737726616ull,
  6631784768079814654ull, 6806322223581409949ull}},
{{13434026651501216037ull, 10671173031567382462ull,
  12901416978527156222ull, 8507902779476762436ull}},
{{3784580638760872119ull, 2057797126302226135ull,
  17286757648434248447ull, 5317439237172976522ull}},
{{119039780023702245ull, 16407304463159946381ull,
  12385075023688034750ull, 6646799046466220653ull}},
{{4760485743457015710ull, 11285758542095157168ull,
  1646285724327879726ull, 8308498808082775817ull}},
{{2975303589660634819ull, 2441913070382085326ull,
  12558143623773394589ull, 5192811755051734885ull}},
{{12942501523930569331ull, 7664077356404994561ull,
  1862621474434579524ull, 6491014693814668607ull}},
{{2343068849631047952ull, 9580096695506243202ull,
  16163334898325388117ull, 8113768367268335758ull}},
{{6076104049446792874ull, 8293403443905095953ull,
  5490398293025979669ull, 5071105229542709849ull}},
{{12206816080235878997ull, 14978440323308757845ull,
  11474683884709862490ull, 6338881536928387311ull}},
{{1423462045012685034ull, 9499678367281171499ull,
  9731668837459940209ull, 7923601921160484139ull}},
{{17030564842628785810ull, 17466514025619201946ull,
  3776450014198768678ull, 4952251200725302587ull}},
{{12064834016431206455ull, 12609770495169226625ull,
  108876499321072944ull, 6190314000906628234ull}},
{{1245984465256844356ull, 15762213118961533282ull,
  9359467661006116988ull, 7737892501133285292ull}},
{{5390426309212915627ull, 628011162496182493ull,
  15073039324983598926ull, 4836182813208303307ull}},
{{11349718904943532437ull, 10008385989975003924ull,
  14229613137802110753ull, 6045228516510379134ull}},
{{14187148631179415547ull, 17122168505896142809ull,
  8563644385397862633ull, 7556535645637973918ull}},
{{1949438866846052861ull, 3783826288544007400ull, 740591722446276242ull,
  4722834778523733699ull}},
{{2436798583557566076ull, 13953154897534785058ull,
  14760797708340009014ull, 5903543473154667123ull}},
{{12269370266301733403ull, 8218071585063705514ull,
  13839311116997623364ull, 7379429341443333904ull}},
{{12280042434865971281ull, 14359666777519591754ull,
  8649569448123514602ull, 4612143338402083690ull}},
{{6126681006727688293ull, 8726211435044713885ull,
  1588589773299617445ull, 5765179173002604613ull}},
{{12270037276836998270ull, 15519450312233280260ull,
  6597423235051909710ull, 7206473966253255766ull}},
{{15337546596046247838ull, 10175940853436824517ull,
  17470151080669662946ull, 9008092457816569707ull}},
{{11891809631742598851ull, 10971649051825403227ull,
  8613001416204845389ull, 5630057786135356067ull}},
{{10253076021250860659ull, 18326247333209141938ull,
  6154565751828668832ull, 7037572232669195084ull}},
{{3592972989708800016ull, 4461065092801875807ull,
  7693207189785836041ull, 8796965290836493855ull}},
{{9163137146209081866ull, 14317380729069642139ull,
  11725783521257229381ull, 5498103306772808659ull}},
{{6842235414333964429ull, 4061667856054888962ull,
  10045543383144148823ull, 6872629133466010824ull}},
{{17776166304772231344ull, 465398801641223298ull,
  12556929228930186029ull, 8590786416832513530ull}},
{{15721789958910032494ull, 2596717260239458513ull,
  12459766786508754172ull, 5369241510520320956ull}},
{{5817179393355376905ull, 3245896575299323142ull,
  15574708483135942715ull, 6711551888150401195ull}},
{{16494846278548996939ull, 17892428774406317639ull,
  14856699585492540489ull, 8389439860188001494ull}},
{{17226807951734204943ull, 4265238956362866668ull,
  4673751222505449902ull, 5243399912617500934ull}},
{{3086765865958204563ull, 14554920732308359144ull,
  15065561064986588185ull, 6554249890771876167ull}},
{{3858457332447755704ull, 4358592860103285218ull,
  14220265312805847328ull, 8192812363464845209ull}},
{{7023221851207235219ull, 2724120537564553261ull,
  1970136792862572724ull, 5120507727165528256ull}},
{{13390713332436431928ull, 3405150671955691576ull,
  2462670991078215905ull, 6400634658956910320ull}},
{{16738391665545539909ull, 8868124358372002374ull,
  3078338738847769881ull, 8000793323696137900ull}},
{{5849808772538574540ull, 17071792770050971244ull,
  11147333748634631983ull, 5000495827310086187ull}},
{{7312260965673218174ull, 16728054944136326151ull,
  9322481167365902075ull, 6250619784137607734ull}},
{{4528640188664134814ull, 16298382661743019785ull,
  2429729422352601786ull, 7813274730172009668ull}},
{{14359615163983554019ull, 14798175182016775269ull,
  10741952925825151924ull, 4883296706357506042ull}},
{{4114460899697278811ull, 50974903811417471ull, 4204069120426664098ull,
  6104120882946882553ull}},
{{531390106194210610ull, 9287090666619047647ull, 9866772418960718026ull,
  7630151103683603191ull}},
{{7249647844012463487ull, 10416117685064292683ull,
  13084261789491530622ull, 4768844439802251994ull}},
{{4450373786588191455ull, 3796775069475590046ull,
  7131955200009637470ull, 5961055549752814993ull}},
{{14786339270090015127ull, 13969340873699263365ull,
  13526630018439434741ull, 7451319437191018741ull}},
{{11547305053019953406ull, 11036681055275733555ull,
  10759986770738340665ull, 4657074648244386713ull}},
{{9822445297847553854ull, 18407537337522054848ull,
  18061669481850313735ull, 5821343310305483391ull}},
{{12278056622309442317ull, 18397735653475180656ull,
  17965400833885504265ull, 7276679137881854239ull}},
{{15347570777886802897ull, 9162111511561812108ull,
  17845065023929492428ull, 9095848922352317799ull}},
{{368859699324476003ull, 14949691731580908376ull,
  18070694667597014623ull, 5684905576470198624ull}},
{{461074624155595003ull, 14075428646048747566ull,
  4141624260786716663ull, 7106131970587748281ull}},
{{9799715317049269562ull, 12982599789133546553ull,
  9788716344410783733ull, 8882664963234685351ull}},
{{17654037119224263236ull, 10419967877422160547ull,
  13035476742897821689ull, 5551665602021678344ull}},
{{17455860380602941141ull, 17636645865205088588ull,
  16294345928622277111ull, 6939582002527097930ull}},
{{3373081402044124810ull, 17434121313078972832ull,
  11144560373923070581ull, 8674477503158872413ull}},
{{2108175876277578007ull, 13202168829888051972ull,
  9271193242915613065ull, 5421548439474295258ull}},
{{2635219845346972508ull, 2667652982077901253ull,
  2365619516789740524ull, 6776935549342869073ull}},
{{7905710825111103539ull, 3334566227597376566ull,
  7568710414414563559ull, 8471169436678586341ull}},
{{329383247267051808ull, 9001632919889442210ull, 7036287018222796176ull,
  5294480897924116463ull}},
{{9635101095938590568ull, 11252041149861802762ull,
  4183672754351107316ull, 6618101122405145579ull}},
{{2820504333068462402ull, 14065051437327253453ull,
  617904924511496241ull, 8272626403006431974ull}},
{{4068658217381482953ull, 1873128120688451552ull,
  14221248633101848863ull, 5170391501879019983ull}},
{{5085822771726853692ull, 16176468206142728152ull,
  13164874772949923174ull, 6462989377348774979ull}},
{{6357278464658567114ull, 10997213220823634382ull,
  11844407447760016064ull, 8078736721685968724ull}},
{{17808357095693768159ull, 6873258263014771488ull,
  16626126691704785848ull, 5049210451053730452ull}},
{{3813702295907658582ull, 8591572828768464361ull,
  2335914290921430694ull, 6311513063817163066ull}},
{{9378813888311961132ull, 1516093999105804643ull,
  12143264900506564176ull, 7889391329771453832ull}},
{{3555915670981281755ull, 947558749441127902ull, 7589540562816602610ull,
  4930869581107158645ull}},
{{13668266625581378002ull, 10407820473656185685ull,
  14098611721948141166ull, 6163586976383948306ull}},
{{3250275226694558791ull, 3786403555215456299ull,
  8399892615580400650ull, 7704483720479935383ull}},
{{18172323081179956908ull, 6978188240437048090ull,
  12167461912378832262ull, 4815302325299959614ull}},
{{13492031814620170327ull, 17946107337401085921ull,
  5985955353618764519ull, 6019127906624949518ull}},
{{3029981712993049197ull, 17820948153323969498ull,
  16705816228878231457ull, 7523909883281186897ull}},
{{6505424589048043652ull, 4220563568186399080ull,
  3523606115407812805ull, 4702443677050741811ull}},
{{8131780736310054565ull, 9887390478660386754ull,
  18239565699541929718ull, 5878054596313427263ull}},
{{941353883532792398ull, 3135866061470707635ull,
  18187771106000024244ull, 7347568245391784079ull}},
{{15011750409698154210ull, 3919832576838384543ull,
  18123027864072642401ull, 9184460306739730099ull}},
{{16299873033702428237ull, 13979110406592460099ull,
  9021049405831707548ull, 5740287691712331312ull}},
{{15763155273700647393ull, 17473888008240575124ull,
  11276311757289634435ull, 7175359614640414140ull}},
{{1257200018416257625ull, 17230673991873331002ull,
  14095389696612043044ull, 8969199518300517675ull}},
{{5397436029937548920ull, 1545799208066056068ull,
  6503775551168832951ull, 5605749698937823547ull}},
{{6746795037421936149ull, 15767307065364733797ull,
  3518033420533653284ull, 7007187123672279434ull}},
{{13045179815204808091ull, 1262389757996365630ull,
  13620913812521842414ull, 8758983904590349292ull}},
{{3541551366075617153ull, 14624051654029892231ull,
  17736443169680927316ull, 5474364940368968307ull}},
{{18261997262876685153ull, 18280064567537365288ull,
  17558867943673771241ull, 6842956175461210384ull}},
{{4380752504886304825ull, 9015022654139542899ull,
  3501840855882662436ull, 8553695219326512981ull}},
{{432127306340246564ull, 14857761195691990120ull,
  4494493544140357974ull, 5346059512079070613ull}},
{{540159132925308205ull, 9348829457760211842ull,
  10229802948602835372ull, 6682574390098838266ull}},
{{9898570953011411064ull, 11686036822200264802ull,
  3563881648898768407ull, 8353217987623547833ull}},
{{10798292864059519819ull, 14221302041516247357ull,
  13756641076630200014ull, 5220761242264717395ull}},
{{18109552098501787678ull, 8553255515040533388ull,
  12584115327360362114ull, 6525951552830896744ull}},
{{4190196049417682981ull, 1468197356945890928ull,
  15730144159200452643ull, 8157439441038620930ull}},
{{2618872530886051863ull, 17058524412587039494ull,
  14443026117927670805ull, 5098399650649138081ull}},
{{12496962700462340637ull, 7488097460451635655ull,
  4218724592127424795ull, 6372999563311422602ull}},
{{11009517357150537892ull, 4748435807137156665ull,
  14496777777014056802ull, 7966249454139278252ull}},
{{18410163394287555943ull, 7579458397888110819ull,
  18283858147488561309ull, 4978905908837048907ull}},
{{18401018224432057024ull, 14086009015787526428ull,
  18243136665933313732ull, 6223632386046311134ull}},
{{4554528706830519664ull, 17607511269734408036ull,
  13580548795561866357ull, 7779540482557888918ull}},
{{12069952478623850598ull, 13310537552797698974ull,
  3876156978798778569ull, 4862212801598680574ull}},
{{5864068561425037440ull, 2803113885714960006ull,
  14068568260353249020ull, 6077766001998350717ull}},
{{16553457738636072608ull, 3503892357143700007ull,
  3750652270159397563ull, 7597207502497938397ull}},
{{17263440114288627236ull, 18330833787710670168ull,
  4650000678063317428ull, 4748254689061211498ull}},
{{3132556069151232429ull, 4466798160928786095ull,
  15035872884433922594ull, 5935318361326514372ull}},
{{17750753141721204248ull, 14806869738015758426ull,
  348097031832851626ull, 7419147951658142966ull}},
{{15705906732003140559ull, 13865979604687236920ull,
  14052618700177695978ull, 4636967469786339353ull}},
{{1185639341294374083ull, 8109102469004270343ull,
  3730715319939956261ull, 5796209337232924192ull}},
{{15317107231900131315ull, 14748064104682725832ull,
  4663394149924945326ull, 7245261671541155240ull}},
{{699639966165612528ull, 9211708093998631483ull, 5829242687406181658ull,
  9056577089426444050ull}},
{{16578176043349365494ull, 10369003577176532580ull,
  8254962698056251440ull, 5660360680891527531ull}},
{{2275975980477155252ull, 12961254471470665726ull,
  5707017354142926396ull, 7075450851114409414ull}},
{{12068342012451219872ull, 16201568089338332157ull,
  16357143729533433803ull, 8844313563893011767ull}},
{{9848556766995706372ull, 7820137046622763646ull,
  17140743858599477983ull, 5527695977433132354ull}},
{{3087323921889857157ull, 5163485289851066654ull,
  12202557786394571671ull, 6909619971791415443ull}},
{{13082526939217097255ull, 1842670593886445413ull,
  10641511214565826685ull, 8637024964739269304ull}},
{{10482422346224379736ull, 3457512130392722335ull,
  6650944509103641678ull, 5398140602962043315ull}},
{{8491341914353086766ull, 13545262199845678727ull,
  3701994617952164193ull, 6747675753702554144ull}},
{{6002491374513970554ull, 3096519694524934697ull,
  4627493272440205242ull, 8434594692128192680ull}},
{{15280772155139701356ull, 6547010827505472089ull,
  2892183295275128276ull, 5271621682580120425ull}},
{{5265907138642462983ull, 8183763534381840112ull,
  8226915137521298249ull, 6589527103225150531ull}},
{{6582383923303078729ull, 14841390436404688044ull,
  5671957903474234907ull, 8236908879031438164ull}},
{{13337361988919200014ull, 6970026013539236075ull,
  12768345726526172625ull, 5148068049394648852ull}},
{{12060016467721612113ull, 13324218535351432998ull,
  15960432158157715781ull, 6435085061743311065ull}},
{{5851648547797239333ull, 2820215113907127536ull,
  6115482142414981015ull, 8043856327179138832ull}},
{{3657280342373274583ull, 8680163473833036566ull,
  3822176339009363134ull, 5027410204486961770ull}},
{{13794972464821369037ull, 1626832305436519899ull,
  14001092460616479726ull, 6284262755608702212ull}},
{{12632029562599323392ull, 11256912418650425682ull,
  17501365575770599657ull, 7855328444510877765ull}},
{{12506704495051965024ull, 118041234015434195ull,
  13244196494070318738ull, 4909580277819298603ull}},
{{11021694600387568376ull, 9370923579374068552ull,
  11943559599160510518ull, 6136975347274123254ull}},
{{13777118250484460470ull, 2490282437362809882ull,
  5706077462095862340ull, 7671219184092654068ull}},
{{13222384924980175698ull, 10779798560206531984ull,
  12789670450664689770ull, 4794511990057908792ull}},
{{16527981156225219623ull, 4251376163403389172ull,
  15987088063330862213ull, 5993139987572385990ull}},
{{2213232371571972912ull, 9925906222681624370ull,
  10760488042308801958ull, 7491424984465482488ull}},
{{5994956250659870974ull, 1592005370748627327ull,
  6725305026443001224ull, 4682140615290926555ull}},
{{2882009294897450814ull, 1990006713435784159ull,
  3794945264626363626ull, 5852675769113658194ull}},
{{17437569673903977229ull, 11710880428649506006ull,
  13967053617637730340ull, 7315844711392072742ull}},
{{12573590055525195728ull, 14638600535811882508ull,
  8235444985192387117ull, 9144805889240090928ull}},
{{17081865821558023138ull, 11454968344096120519ull,
  5147153115745241948ull, 5715503680775056830ull}},
{{16740646258520141019ull, 14318710430120150649ull,
  15657313431536328243ull, 7144379600968821037ull}},
{{7090749767868012561ull, 13286702019222800408ull,
  5736583734138246592ull, 8930474501211026297ull}},
{{4431718604917507851ull, 8304188762014250255ull,
  15114579879904873880ull, 5581546563256891435ull}},
{{927962237719496910ull, 10380235952517812819ull,
  14281538831453704446ull, 6976933204071114294ull}},
{{14995010852431534849ull, 3751922903792490215ull,
  8628551502462354750ull, 8721166505088892868ull}},
{{16289410810410791137ull, 16180009870152470096ull,
  14616216725893747526ull, 5450729065680558042ull}},
{{1915019439303937305ull, 11001640300835811813ull,
  9046898870512408600ull, 6813411332100697553ull}},
{{7005460317557309535ull, 13752050376044764766ull,
  15920309606567898654ull, 8516764165125871941ull}},
{{18213470753755482171ull, 3983345466600590074ull,
  12256036513318630611ull, 5322977603203669963ull}},
{{13543466405339576906ull, 367495814823349689ull,
  10708359623220900360ull, 6653722004004587454ull}},
{{3094274951392307421ull, 459369768529187112ull, 4162077492171349642ull,
  8317152505005734318ull}},
{{1933921844620192138ull, 4898792123758129849ull,
  16436356487889257238ull, 5198220315628583948ull}},
{{7029088324202628076ull, 15346862191552438119ull,
  2098701536152019931ull, 6497775394535729936ull}},
{{4174674386825897191ull, 14571891721013159745ull,
  2623376920190024914ull, 8122219243169662420ull}},
{{14138386537834655505ull, 13719118344060612744ull,
  10862982611973541379ull, 5076387026981039012ull}},
{{17672983172293319381ull, 12537211911648378026ull,
  13578728264966926724ull, 6345483783726298765ull}},
{{12867856928511873418ull, 15671514889560472533ull,
  3138352275926494693ull, 7931854729657873457ull}},
{{10348253589533614838ull, 12100539815188989285ull,
  13490685218522528943ull, 4957409206036170910ull}},
{{17547003005344406452ull, 10513988750558848702ull,
  7639984486298385371ull, 6196761507545213638ull}},
{{12710381719825732257ull, 8530799919771172974ull,
  326608571018205906ull, 7745951884431517048ull}},
{{3332302556463694757ull, 9943435968284371013ull, 204130356886378691ull,
  4841219927769698155ull}},
{{8777064214007006350ull, 7817608941928075862ull,
  14090221001390137076ull, 6051524909712122693ull}},
{{1747958230653982129ull, 9772011177410094828ull,
  3777718196455507633ull, 7564406137140153367ull}},
{{10315845931013514639ull, 17636722031949779027ull,
  9278602900425774126ull, 4727753835712595854ull}},
{{8283121395339505394ull, 12822530503082447976ull,
  2374881588677441850ull, 5909692294640744818ull}},
{{10353901744174381743ull, 6804791091998284162ull,
  12191974022701578121ull, 7387115368300931022ull}},
{{11082874608536376494ull, 15782209478567397361ull,
  3008297745761098421ull, 4616947105188081889ull}},
{{18535205388306905ull, 5892703792927082990ull, 8372058200628760931ull,
  5771183881485102361ull}},
{{9246541043590159439ull, 2754193722731465833ull,
  15076758769213339068ull, 7213979851856377951ull}},
{{16169862322915087202ull, 3442742153414332291ull,
  14234262443089285931ull, 9017474814820472439ull}},
{{7800320942608235550ull, 18292614910379815346ull,
  15813943054571885562ull, 5635921759262795274ull}},
{{527029141405518629ull, 13642396601119993375ull,
  10544056781360081145ull, 7044902199078494093ull}},
{{14493844482039061998ull, 3217937696117828006ull,
  17791756995127489336ull, 8806127748848117616ull}},
{{4446966782847025845ull, 2011211060073642504ull,
  11119848121954680835ull, 5503829843030073510ull}},
{{5558708478558782306ull, 16349071880374216842ull,
  4676438115588575235ull, 6879787303787591888ull}},
{{16171757635053253691ull, 15824653832040383148ull,
  5845547644485719044ull, 8599734129734489860ull}},
{{883976485053507749ull, 667036608170463660ull, 12876839314658350211ull,
  5374833831084056162ull}},
{{1104970606316884686ull, 14668853815495243287ull,
  6872677106468161955ull, 6718542288855070203ull}},
{{15216271313178269569ull, 13724381250941666204ull,
  3979160364657814540ull, 8398177861068837754ull}},
{{286797533881642673ull, 17801110318693317186ull,
  7098661246338521991ull, 5248861163168023596ull}},
{{9581868954206829149ull, 17639701879939258578ull,
  8873326557923152489ull, 6561076453960029495ull}},
{{2753964155903760628ull, 8214569294641909511ull,
  6479972178976552708ull, 8201345567450036869ull}},
{{8638756625080932249ull, 14357477846005969252ull,
  6355825621074039394ull, 5125840979656273043ull}},
{{10798445781351165311ull, 8723475270652685757ull,
  3333096007915161339ull, 6407301224570341304ull}},
{{18109743245116344542ull, 6292658069888469292ull,
  4166370009893951674ull, 8009126530712926630ull}},
{{2095217491342939531ull, 8544597312107681212ull,
  16439039311465883508ull, 5005704081695579143ull}},
{{2619021864178674414ull, 10680746640134601515ull,
  15937113120904966481ull, 6257130102119473929ull}},
{{17108835385505506729ull, 17962619318595639797ull,
  6086333345849044389ull, 7821412627649342412ull}},
{{12998865125154635658ull, 13532480083335968825ull,
  13027330378010428551ull, 4888382892280839007ull}},
{{2413523351161130860ull, 12303914085742573128ull,
  11672476954085647785ull, 6110478615351048759ull}},
{{3016904188951413575ull, 1544834551896052698ull,
  9978910174179671828ull, 7638098269188810949ull}},
{{6497251136522021389ull, 10188893631789808744ull,
  8542661868075988844ull, 4773811418243006843ull}},
{{8121563920652526736ull, 12736117039737260930ull,
  6066641316667598151ull, 5967264272803758554ull}},
{{928582863960882612ull, 11308460281244188259ull,
  16806673682689273497ull, 7459080341004698192ull}},
{{16721265354471409296ull, 150258648136535805ull,
  10504171051680795936ull, 4661925213127936370ull}},
{{7066523637807097908ull, 187823310170669757ull, 3906841777746219112ull,
  5827406516409920463ull}},
{{13444840565686260289ull, 234779137713337196ull, 271866203755385986ull,
  7284258145512400579ull}},
{{16806050707107825362ull, 9516845958996447303ull,
  14174890809976396194ull, 9105322681890500723ull}},
{{17421310719583472707ull, 10559714742800167468ull,
  6553463747021553669ull, 5690826676181562952ull}},
{{3329894325769789268ull, 17811329446927597240ull,
  8191829683776942086ull, 7113533345226953690ull}},
{{4162367907212236585ull, 13040789771804720742ull,
  1016415067866401800ull, 8891916681533692113ull}},
{{16436537997289811578ull, 8150493607377950463ull,
  12164474463484970885ull, 5557447925958557570ull}},
{{15933986478184876568ull, 14799803027649825983ull,
  5982221042501437798ull, 6946809907448196963ull}},
{{15305797079303707806ull, 9276381747707506671ull,
  2866090284699409344ull, 8683512384310246204ull}},
{{16483652202205899235ull, 5797738592317191669ull,
  11014678464791906648ull, 5427195240193903877ull}},
{{6769507197475210331ull, 7247173240396489587ull,
  18380034099417271214ull, 6783994050242379846ull}},
{{3850197978416625010ull, 18282338587350387792ull,
  13751670587416813209ull, 8479992562802974808ull}},
{{2406373736510390632ull, 4508932589452910514ull,
  8594794117135508256ull, 5299995351751859255ull}},
{{12231339207492764097ull, 5636165736816138142ull,
  6131806627991997416ull, 6624994189689824069ull}},
{{6065801972511179314ull, 7045207171020172678ull,
  12276444303417384674ull, 8281242737112280086ull}},
{{17626184288101650783ull, 9014940500314995827ull,
  3061091671208477517ull, 5175776710695175054ull}},
{{17421044341699675575ull, 15880361643821132688ull,
  13049736625865372704ull, 6469720888368968817ull}},
{{3329561353415042852ull, 1403707981066864245ull,
  2477112727049552169ull, 8087151110461211022ull}},
{{4386818855098095735ull, 12406532534235259913ull,
  15383253509688133817ull, 5054469444038256888ull}},
{{10095209587300007573ull, 1673107612511911179ull,
  782322813400615656ull, 6318086805047821111ull}},
{{8007325965697621562ull, 2091384515639888974ull,
  14812961572032933282ull, 7897608506309776388ull}},
{{392892710133625572ull, 5918801340702318513ull, 34728945665807493ull,
  4936005316443610243ull}},
{{5102801906094419869ull, 12010187694305286045ull,
  13878469237364423078ull, 6170006645554512803ull}},
{{10990188401045412740ull, 5789362581026831748ull,
  12736400528278140944ull, 7712508306943141004ull}},
{{16092239787508158771ull, 3618351613141769842ull,
  17183622367028613898ull, 4820317691839463127ull}},
{{10891927697530422655ull, 13746311553281988111ull,
  16867841940358379468ull, 6025397114799328909ull}},
{{9003223603485640415ull, 17182889441602485139ull,
  7249744370165810623ull, 7531746393499161137ull}},
{{3321171742964831308ull, 17656834928642635068ull,
  16060305277422101399ull, 4707341495936975710ull}},
{{4151464678706039134ull, 17459357642375905931ull,
  10852009559922850941ull, 5884176869921219638ull}},
{{577644829955161014ull, 7989138997687718702ull, 4341639913048787869ull,
  7355221087401524548ull}},
{{9945428074298727075ull, 14598109765537036281ull,
  5427049891310984836ull, 9194026359251905685ull}},
{{17745107592505174182ull, 18347190640315423483ull,
  5697749191283059474ull, 5746266474532441053ull}},
{{17569698472204079824ull, 13710616263539503546ull,
  11733872507531212247ull, 7182833093165551316ull}},
{{12738751053400323971ull, 12526584310996991529ull,
  14667340634414015309ull, 8978541366456939145ull}},
{{1044190380734120626ull, 10134958203586813658ull,
  2249558868867677712ull, 5611588354035586966ull}},
{{10528610012772426591ull, 12668697754483517072ull,
  12035320622939372948ull, 7014485442544483707ull}},
{{13160762515965533238ull, 15835872193104396340ull,
  10432464760246828281ull, 8768106803180604634ull}},
{{17448848609333234082ull, 2979891093049165856ull,
  11131976493581655580ull, 5480066751987877896ull}},
{{3364316687956990987ull, 3724863866311457321ull,
  13914970616977069475ull, 6850083439984847370ull}},
{{8817081878373626637ull, 44393814461933747ull, 8170341234366561036ull,
  8562604299981059213ull}},
{{3204833164769822696ull, 9251118170893484400ull,
  7412306280692794599ull, 5351627687488162008ull}},
{{4006041455962278370ull, 6952211695189467596ull,
  9265382850865993249ull, 6689534609360202510ull}},
{{5007551819952847963ull, 13301950637414222399ull,
  2358356526727715753ull, 8361918261700253138ull}},
{{10047248915111611833ull, 1396190120742807143ull,
  6085658847632210250ull, 5226198913562658211ull}},
{{7947375125462126887ull, 10968609687783284737ull,
  2995387541112874908ull, 6532748641953322764ull}},
{{14545904925255046513ull, 13710762109729105921ull,
  3744234426391093635ull, 8165935802441653455ull}},
{{2173661550643322215ull, 6263383309366997249ull,
  9257675544135515378ull, 5103709876526033409ull}},
{{7328762956731540672ull, 17052601173563522369ull,
  16183780448596782126ull, 6379637345657541761ull}},
{{13772639714341813744ull, 12092379430099627153ull,
  6394667505463813946ull, 7974546682071927202ull}},
{{1690370793822551734ull, 12169423162239654875ull,
  8608353209342271620ull, 4984091676294954501ull}},
{{15948021547560353380ull, 15211778952799568593ull,
  15372127530105227429ull, 6230114595368693126ull}},
{{6099968879168278012ull, 5179665635717297030ull,
  9991787375776758479ull, 7787643244210866408ull}},
{{17647538604762337470ull, 10154820049964392499ull,
  6244867109860474049ull, 4867277027631791505ull}},
{{17447737237525533933ull, 17305211080882878528ull,
  12417769905752980465ull, 6084096284539739381ull}},
{{3362927473197365801ull, 7796455795821434449ull,
  1687154326909061870ull, 7605120355674674227ull}},
{{13631044716816823386ull, 261098853961008626ull,
  17195372518814021333ull, 4753200222296671391ull}},
{{7815433859166253424ull, 4938059585878648687ull,
  16882529630090138762ull, 5941500277870839239ull}},
{{5157606305530428876ull, 15395946519203086667ull,
  16491476019185285548ull, 7426875347338549049ull}},
{{917660931742824095ull, 399094537647153359ull, 3389643484349721612ull,
  4641797092086593156ull}},
{{14982134219960693831ull, 498868172058941698ull,
  4237054355437152015ull, 5802246365108241445ull}},
{{9504295738096091481ull, 14458643270355840835ull,
  9908003962723827922ull, 7252807956385301806ull}},
{{7268683654192726447ull, 8849932051090025236ull,
  3161632916550009095ull, 9066009945481627258ull}},
{{13766299320725229838ull, 12448736559572347628ull,
  6587706591271143588ull, 5666256215926017036ull}},
{{17207874150906537297ull, 15560920699465434535ull,
  8234633239088929485ull, 7082820269907521295ull}},
{{16898156670205783717ull, 5616092819049629457ull,
  5681605530433773953ull, 8853525337384401619ull}},
{{3643818891237532967ull, 15039273057974488171ull,
  1245160447307414768ull, 5533453335865251012ull}},
{{18389831669329079921ull, 352347248758558597ull,
  1556450559134268461ull, 6916816669831563765ull}},
{{9152231531379186189ull, 5052120079375586151ull,
  6557249217345223480ull, 8646020837289454706ull}},
{{12637673734753073224ull, 3157575049609741344ull,
  8709966779268152579ull, 5403763023305909191ull}},
{{15797092168441341530ull, 17782026867294340392ull,
  6275772455657802819ull, 6754703779132386489ull}},
{{1299621136842125297ull, 17615847565690537587ull,
  12456401587999641428ull, 8443379723915483111ull}},
{{16953164275022185975ull, 1786532691701810183ull,
  14702780020140857749ull, 5277112327447176944ull}},
{{16579769325350344564ull, 6844851883054650633ull,
  18378475025176072186ull, 6596390409308971180ull}},
{{6889653601405766993ull, 17779436890673089100ull,
  4526349707760538616ull, 8245488011636213976ull}},
{{13529405537733380179ull, 11112148056670680687ull,
  2828968567350336635ull, 5153430007272633735ull}},
{{12300070903739337319ull, 9278499052410962955ull,
  17371268764470084506ull, 6441787509090792168ull}},
{{10763402611246783745ull, 2374751778658927886ull,
  3267341881878054017ull, 8052234386363490211ull}},
{{2115440613601851937ull, 13013434907730299689ull,
  18182989740669641424ull, 5032646491477181381ull}},
{{7255986785429702825ull, 16266793634662874611ull,
  8893679120554888068ull, 6290808114346476727ull}},
{{4458297463359740627ull, 1886747969619041648ull,
  6505412882266222182ull, 7863510142933095909ull}},
{{2786435914599837892ull, 15014275536294064742ull,
  6371726060630082815ull, 4914693839333184943ull}},
{{12706416930104573173ull, 14156158401940193023ull,
  3352971557360215615ull, 6143367299166481179ull}},
{{11271335144203328562ull, 13083511983997853375ull,
  18026272501982433231ull, 7679209123958101473ull}},
{{13962113492768162208ull, 15094724017639740215ull,
  4348891286097938913ull, 4799505702473813421ull}},
{{12840955847532814855ull, 5033346966767511557ull,
  10047800126049811546ull, 5999382128092266776ull}},
{{2216136754133854857ull, 15515055745314165255ull,
  12559750157562264432ull, 7499227660115333470ull}},
{{8302614498974741142ull, 9696909840821353284ull,
  3238157830049027366ull, 4687017287572083419ull}},
{{10378268123718426427ull, 2897765264171915797ull,
  17882755342843447920ull, 5858771609465104273ull}},
{{17584521173075420938ull, 3622206580214894746ull,
  8518386123272146188ull, 7323464511831380342ull}},
{{12757279429489500364ull, 4527758225268618433ull,
  1424610617235406927ull, 9154330639789225428ull}},
{{1055770615789855872ull, 9747377918433968377ull,
  10113753672626905137ull, 5721456649868265892ull}},
{{5931399288164707744ull, 16795908416469848375ull,
  12642192090783631421ull, 7151820812335332365ull}},
{{2802563091778496775ull, 7159827465305146757ull,
  1967682058197375565ull, 8939776015419165457ull}},
{{4057444941575254437ull, 6780735175029410675ull,
  12759016332441829488ull, 5587360009636978410ull}},
{{460120158541680142ull, 8475918968786763344ull, 6725398378697511052ull,
  6984200012046223013ull}},
{{575150198177100177ull, 10594898710983454180ull,
  13018433991799276719ull, 8730250015057778766ull}},
{{9582840910715463419ull, 13539340722005740718ull,
  3524835226447160045ull, 5456406259411111729ull}},
{{2755179101539553466ull, 3089117847225012186ull,
  9017730051486337961ull, 6820507824263889661ull}},
{{12667345913779217640ull, 8473083327458653136ull,
  15883848582785310355ull, 8525634780329862076ull}},
{{7917091196112011025ull, 2989834070447964258ull, 704033327386043164ull,
  5328521737706163798ull}},
{{672991958285237973ull, 3737292588059955323ull,
  10103413696087329763ull, 6660652172132704747ull}},
{{14676298003138711178ull, 59929716647556249ull, 8017581101681774300ull,
  8325815215165880934ull}},
{{2255157224320612631ull, 9260828109759498464ull, 399302170123721033ull,
  5203634509478675584ull}},
{{2818946530400765788ull, 16187721155626760984ull,
  499127712654651291ull, 6504543136848344480ull}},
{{3523683163000957235ull, 15622965426106063326ull,
  623909640818314114ull, 8130678921060430600ull}},
{{16037360032157761984ull, 14376039409743677482ull,
  389943525511446321ull, 5081674325662769125ull}},
{{10823328003342426672ull, 4134991206897433141ull,
  5099115425316695806ull, 6352092907078461406ull}},
{{18140846022605421244ull, 14392111045476567234ull,
  15597266318500645565ull, 7940116133848076757ull}},
{{15949714782555776182ull, 11300912412636548473ull,
  12054134458276597430ull, 4962572583655047973ull}},
{{6102085422912556515ull, 4902768478940909784ull,
  1232610017563583076ull, 6203215729568809967ull}},
{{7627606778640695644ull, 6128460598676137230ull,
  15375820577236642557ull, 7754019661961012458ull}},
{{155568218223046873ull, 6136130883386279721ull,
  14221573879200289502ull, 4846262288725632786ull}},
{{4806146291206196496ull, 16893535641087625459ull,
  8553595312145586069ull, 6057827860907040983ull}},
{{1395996845580357715ull, 7281861496077368112ull,
  6080308121754594683ull, 7572284826133801229ull}},
{{872498028487723572ull, 2245320425834661118ull, 6106035585310315629ull,
  4732678016333625768ull}},
{{10313994572464430273ull, 7418336550720714301ull,
  7632544481637894536ull, 5915847520417032210ull}},
{{17504179234007925746ull, 9272920688400892876ull,
  317308565192592362ull, 7394809400521290263ull}},
{{1716739984400177783ull, 10407261448677945952ull,
  7115846880886452082ull, 4621755875325806414ull}},
{{2145924980500222229ull, 3785704773992656632ull,
  18118180637962840911ull, 5777194844157258017ull}},
{{2682406225625277786ull, 120444949063432886ull, 8812667742171387427ull,
  7221493555196572522ull}},
{{12576379818886373040ull, 13985614241611454819ull,
  1792462640859458475ull, 9026866943995715653ull}},
{{5554394377590289198ull, 6435165891793465310ull,
  3426132159750855499ull, 5641791839997322283ull}},
{{16166365008842637306ull, 3432271346314443733ull,
  18117723254970733086ull, 7052239799996652853ull}},
{{6372898205771132920ull, 13513711219747830475ull,
  8812096013431252645ull, 8815299749995816067ull}},
{{1677218369393264123ull, 10751912521556087999ull,
  3201716999180838951ull, 5509562343747385042ull}},
{{15931581017023743866ull, 8828204633517722094ull,
  13225518285830824497ull, 6886952929684231302ull}},
{{10691104234424904024ull, 15646941810324540522ull,
  7308525820433754813ull, 8608691162105289128ull}},
{{11293626164942952919ull, 12085181640666531778ull,
  4567828637771096758ull, 5380431976315805705ull}},
{{4893660669323915341ull, 5883105013978388915ull,
  10321471815641258852ull, 6725539970394757131ull}},
{{1505389818227506272ull, 7353881267472986144ull,
  8290153751124185661ull, 8406924962993446414ull}},
{{940868636392191420ull, 6902018801384310292ull, 569660076025228134ull,
  5254328101870904009ull}},
{{1176085795490239275ull, 17850895538585163673ull,
  5323761113458923071ull, 6567910127338630011ull}},
{{6081793262790186998ull, 17701933404804066687ull,
  2043015373396265935ull, 8209887659173287514ull}},
{{10718649816884948730ull, 17981237405643623535ull,
  5888570626800054113ull, 5131179786983304696ull}},
{{8786626252678798008ull, 8641488701772365707ull,
  7360713283500067642ull, 6413974733729130870ull}},
{{6371596797421109606ull, 1578488840360681326ull,
  18424263641229860361ull, 8017468417161413587ull}},
{{17817306053670357216ull, 12515770571293895588ull,
  9209321766554968773ull, 5010917760725883492ull}},
{{3824888493378394904ull, 1809655158835205774ull,
  11511652208193710967ull, 6263647200907354365ull}},
{{14004482653577769438ull, 16097127003826170929ull,
  554507204959974996ull, 7829559001134192957ull}},
{{1835272630845024043ull, 837332340536581023ull, 2652410012313678325ull,
  4893474375708870598ull}},
{{16129148843838443765ull, 5658351444098114182ull,
  12538884552246873714ull, 6116842969636088247ull}},
{{10938064017943278899ull, 16296311341977418536ull,
  11061919671881204238ull, 7646053712045110309ull}},
{{6836290011214549312ull, 5573508570308498681ull,
  9219542804139446601ull, 4778783570028193943ull}},
{{13157048532445574544ull, 11578571731313011255ull,
  6912742486746920347ull, 5973479462535242429ull}},
{{11834624647129580276ull, 9861528645713876165ull,
  13252614126861038338ull, 7466849328169053036ull}},
{{9702483413669681624ull, 10775141421998560507ull,
  17506255866142924769ull, 4666780830105658147ull}},
{{7516418248659714126ull, 18080612795925588538ull,
  17271133814251268057ull, 5833476037632072684ull}},
{{172150773969866850ull, 8765707939624821961ull, 3142173194104533456ull,
  7291845047040090856ull}},
{{4826874485889721466ull, 10957134924531027451ull,
  3927716492630666820ull, 9114806308800113570ull}},
{{710953544467381965ull, 16071581364686667965ull,
  7066508826321554666ull, 5696753943000070981ull}},
{{5500377949011615360ull, 10866104669003559148ull,
  13444822051329331237ull, 7120942428750088726ull}},
{{6875472436264519199ull, 18194316854681836839ull,
  7582655527306888238ull, 8901178035937610908ull}},
{{11214699300306406356ull, 6759762015748760120ull,
  13962531741421580957ull, 5563236272461006817ull}},
{{14018374125383007945ull, 13061388538113338054ull,
  3618106621494812484ull, 6954045340576258522ull}},
{{8299595619873984123ull, 16326735672641672568ull,
  13746005313723291413ull, 8692556675720323152ull}},
{{5187247262421240077ull, 12510052804614739307ull,
  8591253321077057133ull, 5432847922325201970ull}},
{{1872373059599162192ull, 1802507950486260422ull,
  1515694614491545609ull, 6791059902906502463ull}},
{{11563838361353728548ull, 6864820956535213431ull,
  15729676323396595723ull, 8488824878633128078ull}},
{{14144928003487162198ull, 1984670088620814442ull,
  5219361683695484423ull, 5305515549145705049ull}},
{{8457787967504176940ull, 16315895666058181765ull,
  11135888123046743432ull, 6631894436432131311ull}},
{{15183920977807609079ull, 1948125508863175590ull,
  9308174135381041387ull, 8289868045540164139ull}},
{{4878264592702367770ull, 17358479507535342408ull,
  3511765825399456914ull, 5181167528462602587ull}},
{{6097830740877959713ull, 12474727347564402202ull,
  18224765337031484855ull, 6476459410578253233ull}},
{{16845660462952225449ull, 10981723166028114848ull,
  8945898616007192357ull, 8095574263222816542ull}},
{{10528537789345140906ull, 9169419987981265732ull,
  979500616577107319ull, 5059733914514260339ull}},
{{13160672236681426132ull, 6850088966549194261ull,
  15059433826003547861ull, 6324667393142825423ull}},
{{2615782240569618953ull, 13174297226613880731ull,
  14212606264077046922ull, 7905834241428531779ull}},
{{17775764964851869510ull, 12845621785061063360ull,
  6577035905834460374ull, 4941146400892832362ull}},
{{3772962132355285271ull, 6833655194471553393ull,
  17444666919147851276ull, 6176433001116040452ull}},
{{9327888683871494493ull, 8542068993089441741ull,
  3359089575225262479ull, 7720541251395050566ull}},
{{8135773436633378010ull, 12256322148321982944ull,
  15934489039797952761ull, 4825338282121906603ull}},
{{10169716795791722513ull, 1485344630120314968ull,
  15306425281320053048ull, 6031672852652383254ull}},
{{12712145994739653141ull, 1856680787650393710ull,
  9909659564795290502ull, 7539591065815479068ull}},
{{3333405228284895309ull, 14995483547563659781ull,
  15416909264851832371ull, 4712244416134674417ull}},
{{8778442553783507040ull, 14132668416027186822ull,
  5436078525782626752ull, 5890305520168343022ull}},
{{1749681155374607992ull, 17665835520033983528ull,
  16018470194083059248ull, 7362881900210428777ull}},
{{2187101444218259990ull, 3635550326332927794ull,
  6188029687321660349ull, 9203602375263035972ull}},
{{5978624421063800398ull, 4578061963171773823ull,
  13090890591430813526ull, 5752251484539397482ull}},
{{2861594507902362593ull, 14945949490819493087ull,
  7140241202433741099ull, 7190314355674246853ull}},
{{17412051190160116954ull, 14070750845096978454ull,
  13536987521469564278ull, 8987892944592808566ull}},
{{6270845975422685192ull, 4182533259758223630ull,
  3848931182491089770ull, 5617433090370505354ull}},
{{17061929506133132298ull, 14451538611552555345ull,
  14034536014968638020ull, 7021791362963131692ull}},
{{7492353827384251661ull, 18064423264440694182ull,
  17543170018710797525ull, 8777239203703914615ull}},
{{71035123687769384ull, 13596107549489127816ull,
  17882010289335330309ull, 5485774502314946634ull}},
{{88793904609711730ull, 3160076381579246058ull, 13129140824814387079ull,
  6857218127893683293ull}},
{{9334364417616915470ull, 17785153532256221284ull,
  2576367975735820136ull, 8571522659867104117ull}},
{{15057349797865347977ull, 11115720957660138302ull,
  3916072994048581537ull, 5357201662416940073ull}},
{{9598315210476909163ull, 59593141793009166ull, 9506777260988114826ull,
  6696502078021175091ull}},
{{2774521976241360646ull, 9297863464096037266ull,
  7271785557807755628ull, 8370627597526468864ull}},
{{6345762253578238308ull, 15034536701914799099ull,
  4544865973629847267ull, 5231642248454043040ull}},
{{3320516798545409981ull, 14181484858966110970ull,
  5681082467037309084ull, 6539552810567553800ull}},
{{13374018035036538284ull, 17726856073707638712ull,
  7101353083796636355ull, 8174441013209442250ull}},
{{8358761271897836428ull, 8773442036853580243ull,
  9050031695800285626ull, 5109025633255901406ull}},
{{5836765571444907630ull, 1743430509212199496ull,
  2089167582895581225ull, 6386282041569876758ull}},
{{7295956964306134538ull, 6790974154942637274ull,
  11834831515474252339ull, 7982852551962345947ull}},
{{9171659121118721990ull, 1938515837625454344ull,
  5090926687957713760ull, 4989282844976466217ull}},
{{11464573901398402488ull, 2423144797031817930ull,
  10975344378374530104ull, 6236603556220582771ull}},
{{5107345339893227302ull, 3028930996289772413ull,
  9107494454540774726ull, 7795754445275728464ull}},
{{5497933846646961016ull, 15728139927963271470ull,
  5692184034087984203ull, 4872346528297330290ull}},
{{16095789345163477078ull, 15048488891526701433ull,
  16338602079464756062ull, 6090433160371662862ull}},
{{6284678626172182635ull, 9587239077553600984ull,
  11199880562476169270ull, 7613041450464578578ull}},
{{3927924141357614147ull, 1380338405043612711ull,
  11611611369974993698ull, 4758150906540361611ull}},
{{298219158269629780ull, 10948795043159291697ull,
  9902828194041354218ull, 5947688633175452014ull}},
{{4984459966264425128ull, 4462621767094338813ull,
  3155163205696916965ull, 7434610791469315018ull}},
{{5421130488128959657ull, 5094981613647655710ull,
  6583663021987961007ull, 4646631744668321886ull}},
{{15999785147015975380ull, 1757040998632181733ull,
  17452950814339727067ull, 5808289680835402357ull}},
{{6164673378487805512ull, 16031359303572390879ull,
  7981130462642495121ull, 7260362101044252947ull}},
{{3094155704682368986ull, 6204141074183324887ull,
  5364727059875730998ull, 9075452626305316184ull}},
{{8851376343067562473ull, 17712646226646741766ull,
  3352954412422331873ull, 5672157891440822615ull}},
{{1840848391979677283ull, 8305749728026263496ull,
  18026251070810078554ull, 7090197364301028268ull}},
{{2301060489974596603ull, 1158815123178053562ull,
  4086069764803046577ull, 8862746705376285336ull}},
{{6049848824661510781ull, 12253474498054753236ull,
  2553793603001904110ull, 5539216690860178335ull}},
{{7562311030826888476ull, 6093471085713665737ull,
  17027300059034543850ull, 6924020863575222918ull}},
{{14064574806960998499ull, 16840210893996857979ull,
  12060753036938404004ull, 8655026079469028648ull}},
{{6484516245136930110ull, 1301759771893260429ull,
  7537970648086502503ull, 5409391299668142905ull}},
{{12717331324848550542ull, 15462257770148739248ull,
  14034149328535516032ull, 6761739124585178631ull}},
{{15896664156060688177ull, 881078138976372444ull,
  12931000642242007137ull, 8452173905731473289ull}},
{{712043060683154303ull, 12079888882928702538ull,
  1164346373760172604ull, 5282608691082170806ull}},
{{10113425862708718686ull, 15099861103660878172ull,
  10678805004054991563ull, 6603260863852713507ull}},
{{12641782328385898358ull, 14263140361148709811ull,
  8736820236641351550ull, 8254076079815891884ull}},
{{5595270946027492522ull, 4302776707290555728ull,
  14683884684755620527ull, 5158797549884932427ull}},
{{6994088682534365652ull, 766784865685806756ull,
  13743169837517137755ull, 6448496937356165534ull}},
{{8742610853167957065ull, 14793539137389422157ull,
  7955590260041646385ull, 8060621171695206918ull}},
{{7769974792443667118ull, 2328432933227306992ull, 360557894098641087ull,
  5037888232309504324ull}},
{{9712468490554583897ull, 16745599221816297452ull,
  450697367623301358ull, 6297360290386880405ull}},
{{12140585613193229871ull, 11708626990415596007ull,
  5175057727956514602ull, 7871700362983600506ull}},
{{14505395035886850526ull, 11929577887437135408ull,
  7846097098400209530ull, 4919812726864750316ull}},
{{18131743794858563157ull, 5688600322441643452ull,
  9807621373000261913ull, 6149765908580937895ull}},
{{4217935669863652330ull, 11722436421479442220ull,
  7647840697822939487ull, 7687207385726172369ull}},
{{11859581830519558515ull, 14244051791065733243ull,
  16309115482207806939ull, 4804504616078857730ull}},
{{10212791269722060239ull, 13193378720404778650ull,
  11163022315904982866ull, 6005630770098572163ull}},
{{3542617050297799491ull, 7268351363651197505ull,
  9342091876453840679ull, 7507038462623215204ull}},
{{13743350702504594442ull, 11460248629923080296ull,
  15062179459638426232ull, 4691899039139509502ull}},
{{17179188378130743052ull, 14325310787403850370ull,
  9604352287693256982ull, 5864873798924386878ull}},
{{12250613435808653007ull, 8683266447400037155ull,
  2782068322761795420ull, 7331092248655483598ull}},
{{10701580776333428355ull, 10854083059250046444ull,
  12700957440307020083ull, 9163865310819354497ull}},
{{15911860022063168530ull, 4477958902817585075ull,
  1020569372550805696ull, 5727415819262096561ull}},
{{15278139009151572758ull, 5597448628521981344ull,
  5887397734115895024ull, 7159269774077620701ull}},
{{650929687729914332ull, 6996810785652476681ull,
  11970933186072256684ull, 8949087217597025876ull}},
{{11936046100899666218ull, 13596378777887573733ull,
  16705205278149936235ull, 5593179510998141172ull}},
{{1084999570842419060ull, 12383787453932079263ull,
  2434762523977868678ull, 6991474388747676466ull}},
{{15191307518835187537ull, 6256362280560323270ull,
  12266825191827111656ull, 8739342985934595582ull}},
{{4882881180844604307ull, 3910226425350202044ull,
  3055079726464556881ull, 5462089366209122239ull}},
{{6103601476055755383ull, 9499469050115140459ull,
  17653907713362859813ull, 6827611707761402798ull}},
{{3017815826642306325ull, 16486022331071313478ull,
  12844012604848798958ull, 8534514634701753498ull}},
{{15721192946933605165ull, 5692077938492183019ull,
  12639193896457887253ull, 5334071646688595936ull}},
{{15039805165239618552ull, 11726783441542616678ull,
  15798992370572359066ull, 6667589558360744920ull}},
{{9576384419694747382ull, 5435107265073495040ull,
  1301996389505897217ull, 8334486947950931151ull}},
{{5985240262309217114ull, 14926157086739404160ull,
  7731276771082267616ull, 5209054342469331969ull}},
{{7481550327886521393ull, 210952284714703584ull,
  14275781982280222425ull, 6511317928086664961ull}},
{{9351937909858151741ull, 4875376374320767384ull,
  4009669422568114319ull, 8139147410108331202ull}},
{{5844961193661344838ull, 9964639261591561471ull,
  7117729407532459353ull, 5086967131317707001ull}},
{{2694515473649293144ull, 17067485095416839743ull,
  13508847777842962095ull, 6358708914147133751ull}},
{{17203202397343780141ull, 16722670350843661774ull,
  12274373703876314715ull, 7948386142683917189ull}},
{{6140315479912474684ull, 8145825960063594657ull,
  9977326574136390649ull, 4967741339177448243ull}},
{{12287080368317981259ull, 14793968468506881225ull,
  7859972199243100407ull, 6209676673971810304ull}},
{{1523792405115312862ull, 13880774567206213628ull,
  9824965249053875509ull, 7762095842464762880ull}},
{{10175742290051846347ull, 10981327113717577469ull,
  6140603280658672193ull, 4851309901540476800ull}},
{{17331363880992195838ull, 18338344910574359740ull,
  7675754100823340241ull, 6064137376925596000ull}},
{{3217460777530693181ull, 9087873082935785964ull,
  9594692626029175302ull, 7580171721156995000ull}},
{{11234285022811459046ull, 1068234658407478323ull,
  5996682891268234564ull, 4737607325723121875ull}},
{{9431170260086935904ull, 1335293323009347904ull,
  2884167595657905301ull, 5922009157153902344ull}},
{{11788962825108669880ull, 6280802672189072784ull,
  3605209494572381626ull, 7402511446442377930ull}},
{{7368101765692918675ull, 8537187688545558394ull,
  6864941952535126420ull, 4626569654026486206ull}},
{{18433499243970924152ull, 10671484610681947992ull,
  17804549477523683833ull, 5783212067533107757ull}},
{{4595129981254103573ull, 17951041781779822895ull,
  8420628791622441079ull, 7229015084416384697ull}},
{{1132226458140241563ull, 17827116208797390715ull,
  15137472007955439253ull, 9036268855520480871ull}},
{{16848542600833508641ull, 13447790639712063148ull,
  16378449032613231389ull, 5647668034700300544ull}},
{{2613934177332334185ull, 2974680244357915224ull,
  2026317217056987621ull, 7059585043375375681ull}},
{{3267417721665417731ull, 8330036323874781934ull,
  7144582539748622430ull, 8824481304219219601ull}},
{{15877194131323049794ull, 594586683994350804ull,
  15994579133411358779ull, 5515300815137012250ull}},
{{1399748590444260626ull, 14578291410275102218ull,
  10769851879909422665ull, 6894126018921265313ull}},
{{10973057774910101591ull, 4387806207561714060ull,
  18074000868314166236ull, 8617657523651581641ull}},
{{16081533146173589302ull, 11965750916580847095ull,
  4378721515055272041ull, 5386035952282238526ull}},
{{15490230414289598724ull, 1122130590443895157ull,
  14696773930673865860ull, 6732544940352798157ull}},
{{5527729962579834693ull, 1402663238054868947ull,
  4535909358060168613ull, 8415681175440997697ull}},
{{1148988217398702731ull, 3182507532997987044ull,
  14364158394856075143ull, 5259800734650623560ull}},
{{1436235271748378414ull, 17813192471529647517ull,
  17955197993570093928ull, 6574750918313279450ull}},
{{6406980108112860921ull, 3819746515702507780ull,
  13220625455107841603ull, 8218438647891599313ull}},
{{13227734604425313884ull, 81498563100373410ull, 1345361881801319146ull,
  5136524154932249571ull}},
{{7311296218676866547ull, 9325245240730242571ull,
  15516760407533812644ull, 6420655193665311963ull}},
{{4527434254918695279ull, 11656556550912803214ull,
  14784264490989877901ull, 8025818992081639954ull}},
{{16664704464606348262ull, 9591190853534195960ull,
  13851851325296061592ull, 5016136870051024971ull}},
{{2384136507048383711ull, 11988988566917744951ull,
  12703128138192689086ull, 6270171087563781214ull}},
{{16815228689092643351ull, 5762863671792405380ull,
  6655538135886085550ull, 7837713859454726518ull}},
{{1286145893828126286ull, 17436847850152417075ull,
  17994769390210967180ull, 4898571162159204073ull}},
{{15442740422567321570ull, 3349315738980969727ull,
  8658403682481545264ull, 6123213952699005092ull}},
{{14691739509781764058ull, 4186644673726212159ull,
  10823004603101931580ull, 7654017440873756365ull}},
{{16099866221254684393ull, 11840024957933658407ull,
  9070220886152401189ull, 4783760900546097728ull}},
{{15513146758140967587ull, 964973142134909297ull,
  11337776107690501487ull, 5979701125682622160ull}},
{{5556375392394045771ull, 15041274482950800334ull,
  14172220134613126858ull, 7474626407103277700ull}},
{{17307792675528442319ull, 14012482570271638112ull,
  18081009620987980094ull, 4671641504439548562ull}},
{{3187996770701001283ull, 8292231175984771833ull,
  13377889989380199310ull, 5839551880549435703ull}},
{{8596681981803639507ull, 1141916933126188983ull,
  12110676468297861234ull, 7299439850686794629ull}},
{{6134166458827161480ull, 10650768203262512037ull,
  1303287530090162830ull, 9124299813358493287ull}},
{{6139697045980669877ull, 2045044108611682119ull,
  7732083733947433625ull, 5702687383349058304ull}},
{{3062935289048449442ull, 7167991154191990553ull,
  9665104667434292031ull, 7128359229186322880ull}},
{{8440355129737949707ull, 4348302924312600287ull,
  12081380834292865039ull, 8910449036482903600ull}},
{{12192750983727300423ull, 9635218355336457035ull,
  7550863021433040649ull, 5569030647801814750ull}},
{{10629252711231737625ull, 16655708962597959198ull,
  215206739936525003ull, 6961288309752268438ull}},
{{4063193852184896223ull, 16207950184820061094ull,
  9492380461775432062ull, 8701610387190335547ull}},
{{16374554212897723851ull, 5518282847085150279ull,
  3626894779395951087ull, 5438506491993959717ull}},
{{15856506747694766910ull, 2286167540429049945ull,
  9145304492672326763ull, 6798133114992449646ull}},
{{5985575379336294926ull, 16692767480818476144ull,
  2208258578985632645ull, 8497666393740562058ull}},
{{3740984612085184329ull, 12738822684725241542ull,
  5991847630293408307ull, 5311041496087851286ull}},
{{13899602801961256219ull, 11311842337479164023ull,
  16713181574721536192ull, 6638801870109814107ull}},
{{12762817484024182369ull, 14139802921848955029ull,
  16279790949974532336ull, 8298502337637267634ull}},
{{10282603936728807933ull, 8837376826155596893ull,
  14786555362161470614ull, 5186563961023292271ull}},
{{17464940939338397820ull, 1823348995839720308ull,
  13871508184274450364ull, 6483204951279115339ull}},
{{3384432100463445659ull, 2279186244799650386ull,
  12727699211915675051ull, 8104006189098894174ull}},
{{6726956081217041441ull, 17565392467495639155ull,
  3343125989019909002ull, 5065003868186808859ull}},
{{3797009083093913897ull, 12733368547514773136ull,
  18013965541557049965ull, 6331254835233511073ull}},
{{4746261353867392371ull, 2081652629111302708ull,
  8682398871664148745ull, 7914068544041888842ull}},
{{12189785383021896040ull, 12830247939263033952ull,
  10038185313217480869ull, 4946292840026180526ull}},
{{15237231728777370050ull, 2202751868796628728ull,
  3324359604667075279ull, 6182866050032725658ull}},
{{599795587262160947ull, 16588497891277949623ull,
  13378821542688619906ull, 7728582562540907072ull}},
{{7292401269679932448ull, 14979497200476106418ull,
  8361763464180387441ull, 4830364101588066920ull}},
{{18338873623954691368ull, 4889313445312969310ull,
  10452204330225484302ull, 6037955126985083650ull}},
{{13700219993088588401ull, 15335013843495987446ull,
  3841883375927079569ull, 7547443908731354563ull}},
{{3950951477252979847ull, 2666854624543910298ull, 95334100740730779ull,
  4717152442957096602ull}},
{{14162061383421000617ull, 17168626335962051584ull,
  9342539662780689281ull, 5896440553696370752ull}},
{{17702576729276250771ull, 7625724864670400768ull,
  11678174578475861602ull, 7370550692120463440ull}},
{{3681476837885761847ull, 308784043983225153ull,
  14597718223094827003ull, 9213188365150579300ull}},
{{13830138069747070915ull, 16333891091985373384ull,
  18346945926289042684ull, 5758242728219112062ull}},
{{17287672587183838643ull, 1970619791272165114ull,
  13710310371006527548ull, 7197803410273890078ull}},
{{12386218697125022496ull, 2463274739090206393ull,
  7914515926903383627ull, 8997254262842362598ull}},
{{823857658062057204ull, 17680447776427236660ull, 334886435887226862ull,
  5623283914276476624ull}},
{{1029822072577571505ull, 12877187683679270017ull,
  418608044859033578ull, 7029104892845595780ull}},
{{5898963609149352285ull, 6873112567744311713ull, 523260056073791973ull,
  8786381116056994725ull}},
{{15216067301786814939ull, 6601538364053888772ull,
  2632880544259813935ull, 5491488197535621703ull}},
{{573340053523967057ull, 3640236936639973062ull,
  17126158735606931131ull, 6864360246919527128ull}},
{{9940047103759734629ull, 18385354226082130039ull,
  2960954345799112297ull, 8580450308649408911ull}},
{{13130058467490915999ull, 4573317363660249418ull,
  8768125493765527042ull, 5362781442905880569ull}},
{{7189201047508869191ull, 14940018741430087581ull,
  15571842885634296706ull, 6703476803632350711ull}},
{{13598187327813474393ull, 9451651389932833668ull,
  14853117588615482979ull, 8379346004540438389ull}},
{{17722239116738197304ull, 3601439109494327090ull,
  11589041502098370814ull, 5237091252837773993ull}},
{{12929426859067970822ull, 13725170923722684671ull,
  651243822340799805ull, 6546364066047217492ull}},
{{11550097555407575623ull, 3321405599371192127ull,
  814054777925999757ull, 8182955082559021865ull}},
{{14136339999770816620ull, 4381721508820689031ull,
  12037999282272219608ull, 5114346926599388665ull}},
{{13058738981286132871ull, 5477151886025861289ull,
  1212441047558110798ull, 6392933658249235832ull}},
{{2488365671325502377ull, 16069811894387102420ull,
  1515551309447638497ull, 7991167072811544790ull}},
{{10778600581433214794ull, 3126103406350857156ull,
  14782277623686937773ull, 4994479420507215493ull}},
{{13473250726791518492ull, 8519315276365959349ull,
  4642788974326508504ull, 6243099275634019367ull}},
{{3006505353207234403ull, 10649144095457449187ull,
  1191800199480747726ull, 7803874094542524209ull}},
{{18019966910250379166ull, 2044029041233517837ull,
  12274090170743937089ull, 4877421309089077630ull}},
{{8689900582530810246ull, 7166722319969285201ull,
  6119240676575145553ull, 6096776636361347038ull}},
{{15474061746590900711ull, 13570088918388994405ull,
  16872422882573707749ull, 7620970795451683797ull}},
{{11977131600833006896ull, 10787148583206815455ull,
  12851107310822261295ull, 4763106747157302373ull}},
{{10359728482613870716ull, 8872249710581131415ull,
  2228826083245662907ull, 5953883433946627967ull}},
{{8337974584839950491ull, 6478626119799026365ull,
  16621090659339242346ull, 7442354292433284958ull}},
{{7517077124738663009ull, 8660827343301779382ull,
  5776495643659638562ull, 4651471432770803099ull}},
{{172974369068552954ull, 1602662142272448420ull, 2608933536147160299ull,
  5814339290963503874ull}},
{{216217961335691192ull, 15838385733122724237ull,
  12484538957038726181ull, 7267924113704379842ull}},
{{4881958470097001894ull, 5962924111121241584ull,
  6382301659443631919ull, 9084905142130474803ull}},
{{3051224043810626184ull, 10644356597091857846ull,
  1683095527938575997ull, 5678065713831546752ull}},
{{13037402091618058538ull, 17917131764792210211ull,
  2103869409923219996ull, 7097582142289433440ull}},
{{11685066596095185268ull, 3949670632280711148ull,
  2629836762404024996ull, 8871977677861791800ull}},
{{16526538659414266601ull, 11691916182030220275ull,
  1643647976502515622ull, 5544986048663619875ull}},
{{16046487305840445347ull, 5391523190682999536ull,
  15889618025910308240ull, 6931232560829524843ull}},
{{1611365058591005067ull, 6739403988353749421ull,
  15250336513960497396ull, 8664040701036906054ull}},
{{3312946170833072119ull, 13435499529575869196ull,
  4919774302797922968ull, 5415025438148066284ull}},
{{4141182713541340149ull, 16794374411969836495ull,
  6149717878497403710ull, 6768781797685082855ull}},
{{564792373499287282ull, 11769595978107519811ull,
  3075461329694366734ull, 8460977247106353569ull}},
{{16493896297932912215ull, 2744311467889811977ull,
  13451378377127448969ull, 5288110779441470980ull}},
{{6782312317133976557ull, 8042075353289652876ull,
  16814222971409311211ull, 6610138474301838725ull}},
{{8477890396417470696ull, 5440908173184678191ull,
  7182720658979475302ull, 8262673092877298407ull}},
{{12216210525402001041ull, 17235625663522587581ull,
  11406729439503253919ull, 5164170683048311504ull}},
{{1435205101470337590ull, 16932846060975846573ull,
  14258411799379067399ull, 6455213353810389380ull}},
{{6405692395265309891ull, 16554371557792420312ull,
  17823014749223834249ull, 8069016692262986725ull}},
{{4003557747040818682ull, 3428953195979180839ull,
  13445227227478590358ull, 5043135432664366703ull}},
{{392761165373635448ull, 13509563531828751857ull,
  12194848015920850043ull, 6303919290830458379ull}},
{{5102637475144432214ull, 12275268396358551917ull,
  10631874001473674650ull, 7879899113538072974ull}},
{{5494991431178964086ull, 12283728766151482852ull,
  2033235232493658752ull, 4924936945961295609ull}},
{{6868739288973705107ull, 15354660957689353565ull,
  7153230059044461344ull, 6156171182451619511ull}},
{{13197610129644519288ull, 746582123402140340ull,
  4329851555378188777ull, 7695213978064524389ull}},
{{17471878367882600363ull, 11995828873194807472ull,
  5012000231325061937ull, 4809508736290327743ull}},
{{3393103886143698838ull, 1159728036211345629ull,
  1653314270728939518ull, 6011885920362909679ull}},
{{8853065876107011451ull, 10673032082118957844ull,
  15901700893693338109ull, 7514857400453637098ull}},
{{14756538209421657965ull, 8976488060538042604ull,
  14550249076985724222ull, 4696785875283523186ull}},
{{18445672761777072457ull, 1997238038817777447ull,
  8964439309377379470ull, 5870982344104403983ull}},
{{18445404933793952667ull, 11719919585376997617ull,
  6593863118294336433ull, 7338727930130504979ull}},
{{9221698111960277121ull, 814841426439083310ull, 3630642879440532638ull,
  9173409912663131224ull}},
{{1151875301547785297ull, 14344333946806590781ull,
  2269151799650332898ull, 5733381195414457015ull}},
{{6051530145362119525ull, 8707045396653462668ull,
  16671497804845079835ull, 7166726494268071268ull}},
{{7564412681702649406ull, 6272120727389440431ull,
  2392628182346798178ull, 8958408117835089086ull}},
{{11645286953705237735ull, 8531761473045788173ull,
  15330450669248912573ull, 5599005073646930678ull}},
{{721550636849383457ull, 15276387859734623121ull,
  9939691299706364908ull, 6998756342058663348ull}},
{{5513624314489117225ull, 648740750958727285ull,
  12424614124632956136ull, 8748445427573329185ull}},
{{5751858205769392218ull, 405462969349204553ull, 847854800254515729ull,
  5467778392233330741ull}},
{{11801508775639128176ull, 5118514730113893595ull,
  5671504518745532565ull, 6834722990291663426ull}},
{{10140199951121522316ull, 11009829431069754898ull,
  16312752685286691514ull, 8543403737864579282ull}},
{{10949310987878339351ull, 11492829412845984715ull,
  14807156446731570100ull, 5339627336165362051ull}},
{{9074952716420536285ull, 14366036766057480894ull,
  13897259539987074721ull, 6674534170206702564ull}},
{{2120318858670894548ull, 4122487902289687406ull,
  17371574424983843402ull, 8343167712758378205ull}},
{{15160257341951472805ull, 7188240957358442532ull,
  13163077024828596078ull, 5214479820473986378ull}},
{{503577603729789390ull, 18208673233552828974ull,
  7230474244180969289ull, 6518099775592482973ull}},
{{9852844041517012545ull, 8925783486658872505ull,
  13649778823653599516ull, 8147624719490603716ull}},
{{17687242572016602601ull, 14801986716016571123ull,
  17754483801638275505ull, 5092265449681627322ull}},
{{17497367196593365347ull, 4667425339738550192ull,
  12969732715193068574ull, 6365331812102034153ull}},
{{3424964922032155068ull, 15057653711527963549ull,
  2377107838709172005ull, 7956664765127542692ull}},
{{4446446085483790870ull, 11716876578918671170ull,
  10709064436048008311ull, 4972915478204714182ull}},
{{14781429643709514395ull, 10034409705220951058ull,
  4162958508205234581ull, 6216144347755892728ull}},
{{9253415017782117185ull, 17154698149953576727ull,
  5203698135256543226ull, 7770180434694865910ull}},
{{12700913413754905097ull, 15333372362148373358ull,
  17087369389817503228ull, 4856362771684291193ull}},
{{6652769730338855563ull, 719971378975915082ull, 7524153681989715324ull,
  6070453464605363992ull}},
{{17539334199778345262ull, 899964223719893852ull,
  9405192102487144155ull, 7588066830756704990ull}},
{{1738711838006689981ull, 16703378704320791322ull,
  1266559045627077192ull, 4742541769222940619ull}},
{{11396761834363138284ull, 2432479306691437536ull,
  15418256862316010203ull, 5928177211528675773ull}},
{{14245952292953922855ull, 16875657188646460632ull,
  5437763022612849041ull, 7410221514410844717ull}},
{{8903720183096201784ull, 3629756715262956039ull,
  5704444898346724603ull, 4631388446506777948ull}},
{{6517964210442864326ull, 18372253949360858761ull,
  7130556122933405753ull, 5789235558133472435ull}},
{{12759141281480968312ull, 9130259381418909739ull,
  4301509135239369288ull, 7236544447666840544ull}},
{{11337240583423822486ull, 11412824226773637174ull,
  5376886419049211610ull, 9045680559583550680ull}},
{{2474089346212501150ull, 11744701160160911138ull,
  3360554011905757256ull, 5653550349739719175ull}},
{{12315983719620402245ull, 14680876450201138922ull,
  18035750570164360282ull, 7066937937174648968ull}},
{{6171607612670726998ull, 9127723525896647845ull,
  4097944138995898737ull, 8833672421468311211ull}},
{{6163097767132898326ull, 17234042249753874663ull,
  255372077658742758ull, 5521045263417694507ull}},
{{3092186190488735004ull, 12319180775337567521ull,
  14154273152355592160ull, 6901306579272118133ull}},
{{8476918756538306658ull, 15398975969171959401ull,
  3857783385162326488ull, 8626633224090147667ull}},
{{16827289268904911422ull, 9624359980732474625ull,
  105271606512760103ull, 5391645765056342292ull}},
{{7199053530848975565ull, 7418763957488205378ull, 131589508140950129ull,
  6739557206320427865ull}},
{{18222188950415995264ull, 13885140965287644626ull,
  4776172903603575565ull, 8424446507900534831ull}},
{{16000554112437384944ull, 10984056112518471843ull,
  9902637092393316584ull, 5265279067437834269ull}},
{{15389006622119343276ull, 13730070140648089804ull,
  16989982383919033634ull, 6581598834297292836ull}},
{{789514203939627479ull, 7939215638955336448ull, 2790733906189240427ull,
  8226998542871616046ull}},
{{493446377462267175ull, 2656166765133391328ull,
  15579266746650438979ull, 5141874089294760028ull}},
{{616807971827833968ull, 17155266511698902872ull,
  1027339359603497107ull, 6427342611618450036ull}},
{{771009964784792460ull, 16832397121196240686ull,
  1284174199504371384ull, 8034178264523062545ull}},
{{14316939283272659000ull, 10520248200747650428ull,
  12331823920758701875ull, 5021361415326914090ull}},
{{17896174104090823750ull, 8538624232507175131ull,
  6191407864093601536ull, 6276701769158642613ull}},
{{17758531611686141783ull, 10673280290633968914ull,
  12350945848544389824ull, 7845877211448303266ull}},
{{15710768275731226518ull, 6670800181646230571ull,
  12331027173767631544ull, 4903673257155189541ull}},
{{15026774326236645244ull, 8338500227057788214ull,
  1578725911927375718ull, 6129591571443986927ull}},
{{9560095870941030747ull, 1199753246967459460ull,
  15808465445191383360ull, 7661989464304983658ull}},
{{15198431956192920025ull, 749845779354662162ull,
  14491976921672002504ull, 4788743415190614786ull}},
{{9774667908386374223ull, 937307224193327703ull, 8891599115235227322ull,
  5985929268988268483ull}},
{{7606648867055579875ull, 10395006067096435437ull,
  6502812875616646248ull, 7482411586235335604ull}},
{{7059998551123431374ull, 6496878791935272148ull,
  13287630084115179713ull, 4676507241397084752ull}},
{{8824998188904289217ull, 12732784508346478089ull,
  16609537605143974641ull, 5845634051746355940ull}},
{{15642933754557749425ull, 2080922580150933899ull,
  2315177932720416686ull, 7307042564682944926ull}},
{{14941981174769798878ull, 11824525262043443182ull,
  12117344452755296665ull, 9133803205853681157ull}},
{{4727052215803736395ull, 472799261136070133ull, 9879183292185754368ull,
  5708627003658550723ull}},
{{10520501288182058397ull, 590999076420087666ull,
  7737293096804805056ull, 7135783754573188404ull}},
{{3927254573372797188ull, 738748845525109583ull, 9671616371006006320ull,
  8919729693216485505ull}},
{{9372063135999080099ull, 461718028453193489ull,
  17573975277947223710ull, 5574831058260303440ull}},
{{16326764938426238028ull, 9800519572421267669ull,
  3520725023724478021ull, 6968538822825379301ull}},
{{6573398117750633822ull, 16862335483953972491ull,
  9012592298082985430ull, 8710673528531724126ull}},
{{1802530814380452187ull, 5927273659043844903ull,
  1021184167874477990ull, 5444170955332327579ull}},
{{16088221573257728946ull, 16632464110659581936ull,
  15111538265125261199ull, 6805213694165409473ull}},
{{1663532892862609566ull, 16178894119897089517ull,
  5054364776124412787ull, 8506517117706761842ull}},
{{3345551067252824931ull, 7805965815721986996ull,
  7770664003505145896ull, 5316573198566726151ull}},
{{4181938834066031164ull, 9757457269652483745ull,
  5101643985954044466ull, 6645716498208407689ull}},
{{9839109561009926858ull, 2973449550210828873ull,
  10988741000869943487ull, 8307145622760509611ull}},
{{17678658521699674047ull, 8775934996522849901ull,
  4562120116330020727ull, 5191966014225318507ull}},
{{8263265096842428846ull, 6358232727226174473ull,
  1090964126985138005ull, 6489957517781648134ull}},
{{14940767389480423962ull, 12559476927460105995ull,
  10587077195586198314ull, 8112446897227060167ull}},
{{7032136609211571024ull, 12461359098089954151ull,
  13534452274882455802ull, 5070279310766912604ull}},
{{4178484743087075876ull, 6353326835757666881ull,
  16918065343603069753ull, 6337849138458640755ull}},
{{9834791947286232749ull, 12553344563124471505ull,
  16535895661076449287ull, 7922311423073300944ull}},
{{17675960013122365228ull, 14763369379593876546ull,
  10334934788172780804ull, 4951444639420813090ull}},
{{12871577979548180727ull, 7467650782794067ull, 3695296448361200198ull,
  6189305799276016363ull}},
{{11477786456007838005ull, 9232706600333268392ull, 7434542024112343ull,
  7736632249095020454ull}},
{{7173616535004898753ull, 12687970652849374601ull,
  13839704644047233926ull, 4835395155684387783ull}},
{{13578706687183511346ull, 6636591279206942443ull,
  12687944786631654504ull, 6044243944605484729ull}},
{{12361697340552001278ull, 8295739099008678054ull,
  2024872928007404418ull, 7555304930756855912ull}},
{{3114374819417612895ull, 9796522955307811688ull,
  1265545580004627761ull, 4722065581723034945ull}},
{{3892968524272016118ull, 16857339712562152514ull,
  6193617993433172605ull, 5902581977153793681ull}},
{{14089582692194795956ull, 7236616585420526930ull,
  12353708510218853661ull, 7378227471442242101ull}},
{{8388606328388719137ull, 13657456750203046567ull,
  1607077582491403364ull, 9222784339302802627ull}},
{{12160407982884031317ull, 17759282505731679912ull,
  17145324553552984766ull, 5764240212064251641ull}},
{{15200509978605039146ull, 12975731095309824082ull,
  7596597636659067246ull, 7205300265080314552ull}},
{{9777265436401523124ull, 6996291832282504295ull,
  9495747045823834058ull, 9006625331350393190ull}},
{{13028319925392033809ull, 8984368413603953088ull,
  1323155885212508382ull, 5629140832093995744ull}},
{{16285399906740042261ull, 2007088480150165552ull,
  1653944856515635478ull, 7036426040117494680ull}},
{{1910005809715501210ull, 11732232637042482749ull,
  2067431070644544347ull, 8795532550146868350ull}},
{{3499596640285882208ull, 5026802388937857766ull,
  15127202474435003929ull, 5497207843841792718ull}},
{{13597867837212128568ull, 10895189004599710111ull,
  9685631056188979103ull, 6871509804802240898ull}},
{{12385648778087772806ull, 9007300237322249735ull,
  2883666783381448071ull, 8589387256002801123ull}},
{{14658559513945939860ull, 12547091675967487940ull,
  17943192804109262708ull, 5368367035001750701ull}},
{{18323199392432424825ull, 15683864594959359925ull,
  8593932949854414673ull, 6710458793752188377ull}},
{{9068941185258367319ull, 5769772688417036195ull,
  15354102205745406246ull, 8388073492190235471ull}},
{{3362245231572785623ull, 17441165985542811334ull,
  16513842906231960759ull, 5242545932618897169ull}},
{{13426178576320757836ull, 17189771463501126263ull,
  6807245577507787237ull, 6553182415773621462ull}},
{{12171037201973559391ull, 7652156274094244117ull,
  17732429008739509855ull, 8191478019717026827ull}},
{{9912741260447168572ull, 11700126698949984429ull,
  8776925121248499707ull, 5119673762323141767ull}},
{{17002612593986348618ull, 10013472355260092632ull,
  6359470383133236730ull, 6399592202903927209ull}},
{{2806521668773384157ull, 3293468407220339983ull,
  12561023997343933817ull, 7999490253629909011ull}},
{{8671605070624446954ull, 13587632800581182249ull,
  5544796989126264683ull, 4999681408518693132ull}},
{{15451192356707946597ull, 12372854982299089907ull,
  6930996236407830854ull, 6249601760648366415ull}},
{{14702304427457545342ull, 6242696691019086576ull,
  4052059277082400664ull, 7812002200810458019ull}},
{{9188940267160965839ull, 3901685431886929110ull, 226694038962806463ull,
  4882501375506536262ull}},
{{2262803297096431490ull, 265420771431273484ull, 9506739585558283887ull,
  6103126719383170327ull}},
{{2828504121370539363ull, 14166834019571255567ull,
  7271738463520466954ull, 7628908399228962909ull}},
{{8685344103497668958ull, 13465957280659422633ull,
  6850679548913985798ull, 4768067749518101818ull}},
{{15468366147799474101ull, 7609074563969502483ull,
  17786721472997258056ull, 5960084686897627272ull}},
{{14723771666321954722ull, 9511343204961878104ull,
  3786657767537020954ull, 7450105858622034091ull}},
{{9202357291451221702ull, 10556275521528561719ull, 60818095496944144ull,
  4656316161638771307ull}},
{{6891260595886639223ull, 13195344401910702149ull,
  13911080674653343892ull, 5820395202048464133ull}},
{{13225761763285686933ull, 16494180502388377686ull,
  3553792788034516153ull, 7275494002560580167ull}},
{{7308830167252332858ull, 6782667572703308396ull,
  18277299040325308904ull, 9094367503200725208ull}},
{{13791390891387483844ull, 4239167232939567747ull,
  11423311900203318065ull, 5683979689500453255ull}},
{{12627552595806966901ull, 9910645059601847588ull,
  9667453856826759677ull, 7104974611875566569ull}},
{{15784440744758708626ull, 16999992342929697389ull,
  16696003339460837500ull, 8881218264844458211ull}},
{{12171118474687886844ull, 1401623177476285060ull,
  8129159077949329486ull, 5550761415527786382ull}},
{{15213898093359858554ull, 10975401008700132133ull,
  938076810581886049ull, 6938451769409732978ull}},
{{5182314561417659481ull, 18330937279302553071ull,
  10395968050082133369ull, 8673064711762166222ull}},
{{10156475628527119032ull, 4539306771923013813ull,
  1885794012873945452ull, 5420665444851353889ull}},
{{17307280554086286694ull, 5674133464903767266ull,
  6968928534519819719ull, 6775831806064192361ull}},
{{12410728655753082559ull, 2480980812702321179ull,
  13322846686577162553ull, 8469789757580240451ull}},
{{5450862400631982647ull, 13079828054007420497ull,
  6020936169897032643ull, 5293618598487650282ull}},
{{11425264019217366213ull, 11738099049081887717ull,
  16749542249226066612ull, 6617023248109562852ull}},
{{446521968739544054ull, 14672623811352359647ull,
  2490183737823031649ull, 8271279060136953566ull}},
{{7196605258103296890ull, 2252860854454142923ull,
  15391422891421558493ull, 5169549412585595978ull}},
{{4384070554201733209ull, 7427762086495066558ull,
  10015906577422172308ull, 6461936765731994973ull}},
{{14703460229606942319ull, 9284702608118833197ull,
  17131569240205103289ull, 8077420957164993716ull}},
{{11495505652718032901ull, 17332154176142740508ull,
  1483858738273413747ull, 5048388098228121073ull}},
{{14369382065897541126ull, 17053506701751037731ull,
  6466509441269155088ull, 6310485122785151341ull}},
{{13350041563944538504ull, 2870139303479245548ull,
  12694822820013831765ull, 7888106403481439176ull}},
{{17567148014320112373ull, 4099680073888222419ull,
  7934264262508644853ull, 4930066502175899485ull}},
{{17347248999472752562ull, 9736286110787665928ull,
  14529516346563193970ull, 6162583127719874356ull}},
{{3237317175631389087ull, 2946985601629806603ull,
  18161895433203992463ull, 7703228909649842945ull}},
{{18164224299265475843ull, 8759395028659710982ull,
  4433655618111413433ull, 4814518068531151841ull}},
{{13481908337227068996ull, 15560929804252026632ull,
  10153755541066654695ull, 6018147585663939801ull}},
{{16852385421533836245ull, 14839476236887645386ull,
  17303880444760706273ull, 7522684482079924751ull}},
{{15144426906886035557ull, 2357143620413696510ull,
  17732454305616523277ull, 4701677801299952969ull}},
{{9707161596752768639ull, 7558115543944508542ull,
  8330509826738490384ull, 5877097251624941212ull}},
{{2910579959086184990ull, 9447644429930635678ull,
  10413137283423112980ull, 7346371564531176515ull}},
{{12861596985712507045ull, 11809555537413294597ull,
  8404735585851503321ull, 9182964455663970644ull}},
{{10344341125284010856ull, 463443183242227267ull,
  14476331778011965384ull, 5739352784789981652ull}},
{{8318740388177625665ull, 579303979052784084ull,
  18095414722514956730ull, 7174190980987477065ull}},
{{10398425485222032082ull, 9947502010670755913ull,
  8784210347861532200ull, 8967738726234346332ull}},
{{18028230974332239811ull, 6217188756669222445ull,
  14713503504268233433ull, 5604836703896466457ull}},
{{8700230662633136052ull, 12383171964263915961ull,
  4556821325053128079ull, 7006045879870583072ull}},
{{15486974346718807969ull, 10867278936902507047ull,
  5696026656316410099ull, 8757557349838228840ull}},
{{16596887994340336837ull, 4486206326350372952ull,
  3560016660197756312ull, 5473473343648893025ull}},
{{2299365919215869430ull, 5607757907937966191ull,
  9061706843674583294ull, 6841841679561116281ull}},
{{16709265454302000499ull, 16233069421777233546ull,
  15938819573020617021ull, 8552302099451395351ull}},
{{15054976927366138216ull, 12451511397824464918ull,
  16879291260778967494ull, 5345188812157122094ull}},
{{9595349122352896962ull, 6341017210425805340ull,
  11875742039118933560ull, 6681486015196402618ull}},
{{11994186402941121202ull, 7926271513032256675ull,
  5621305512043891142ull, 8351857518995503273ull}},
{{5190523492624506800ull, 342233677217772518ull,
  15042530991095901724ull, 5219910949372189545ull}},
{{15711526402635409307ull, 427792096522215647ull,
  4968105683587713443ull, 6524888686715236932ull}},
{{15027721984866873730ull, 14369798175934933271ull,
  6210132104484641803ull, 8156110858394046165ull}},
{{16309855268182877938ull, 6675280850745639342ull,
  6187175574516595079ull, 5097569286496278853ull}},
{{11163947048373821614ull, 3732415045004661274ull,
  12345655486573131753ull, 6371961608120348566ull}},
{{4731561773612501209ull, 9277204824683214497ull,
  6208697321361638883ull, 7964952010150435708ull}},
{{14486441154576283016ull, 3492410006213315108ull,
  13103807862705800110ull, 4978095006344022317ull}},
{{18108051443220353770ull, 13588884544621419693ull,
  2544701773100086425ull, 6222618757930027897ull}},
{{8800006248743278500ull, 3151047625494610905ull,
  7792563234802495936ull, 7778273447412534871ull}},
{{17029218951533018823ull, 1969404765934131815ull,
  11787881049392641816ull, 4861420904632834294ull}},
{{16674837670988885624ull, 2461755957417664769ull,
  5511479274886026462ull, 6076776130791042868ull}},
{{7008489033453943318ull, 12300566983626856770ull,
  6889349093607533077ull, 7595970163488803585ull}},
{{8991991664336102478ull, 9993697373980479433ull,
  15835058229573177933ull, 4747481352180502240ull}},
{{15851675598847516001ull, 17103807735902987195ull,
  1347078713256920800ull, 5934351690225627801ull}},
{{15202908480132007098ull, 2933015596169182378ull,
  6295534409998538905ull, 7417939612782034751ull}},
{{14113503818509892340ull, 13362349793674208746ull,
  10852238033890168671ull, 4636212257988771719ull}},
{{8418507736282589617ull, 12091251223665373029ull,
  8953611523935322935ull, 5795265322485964649ull}},
{{15134820688780624925ull, 10502378011154328382ull,
  15803700423346541573ull, 7244081653107455811ull}},
{{9695153824121005349ull, 17739658532370298382ull,
  15142939510755789062ull, 9055102066384319764ull}},
{{1447785121648240439ull, 6475600564304048585ull, 240965157367592356ull,
  5659438791490199853ull}},
{{6421417420487688453ull, 8094500705380060731ull,
  4912892465136878349ull, 7074298489362749816ull}},
{{3415085757182222662ull, 14729811900152463818ull,
  6141115581421097936ull, 8842873111703437270ull}},
{{6746114616666277068ull, 9206132437595289886ull,
  17673255293670349922ull, 5526795694814648293ull}},
{{17656015307687622143ull, 2284293510139336549ull,
  8256511061805773691ull, 6908494618518310367ull}},
{{8234961079327363966ull, 16690424942956334399ull,
  5708952808829829209ull, 8635618273147887959ull}},
{{12064379702220684335ull, 3513986561706627143ull,
  10485624533159725112ull, 5397261420717429974ull}},
{{10468788609348467515ull, 4392483202133283929ull,
  3883658629594880582ull, 6746576775896787468ull}},
{{17697671780112972297ull, 14713976039521380719ull,
  4854573286993600727ull, 8433220969870984335ull}},
{{17978573890211689542ull, 16113764052341944805ull,
  9951637332012082310ull, 5270763106169365209ull}},
{{8638159307482448215ull, 10918833028572655199ull,
  17051232683442490792ull, 6588453882711706511ull}},
{{6186013115925672365ull, 13648541285715818999ull,
  16702354835875725586ull, 8235567353389633139ull}},
{{10783787225094627084ull, 13142024321999774778ull,
  8133128763208634539ull, 5147229595868520712ull}},
{{4256361994513508047ull, 11815844384072330569ull,
  10166410954010793174ull, 6434036994835650890ull}},
{{9932138511569272963ull, 5546433443235637403ull,
  3484641655658715660ull, 8042546243544563613ull}},
{{3901743560517101650ull, 12689892938877049185ull,
  4483744044000391239ull, 5026591402215352258ull}},
{{9488865469073764966ull, 11250680155168923577ull,
  14828052091855264857ull, 6283239252769190322ull}},
{{16472767854769594112ull, 228292138678990759ull,
  9311693077964305264ull, 7854049065961487903ull}},
{{17213008936872078176ull, 142682586674369224ull,
  12737337201368772646ull, 4908780666225929939ull}},
{{3069517097380546104ull, 9401725270197737339ull,
  11309985483283577903ull, 6135975832782412424ull}},
{{17671954427007846342ull, 7140470569319783769ull,
  14137481854104472379ull, 7669969790978015530ull}},
{{4127442489238822108ull, 2156951096611170904ull,
  13447612177242683141ull, 4793731119361259706ull}},
{{5159303111548527635ull, 7307874889191351534ull,
  7586143184698578118ull, 5992163899201574633ull}},
{{15672500926290435351ull, 18358215648343965225ull,
  14094364999300610551ull, 7490204874001968291ull}},
{{2877784051290440239ull, 18391413807856060122ull,
  6503135115349187642ull, 4681378046251230182ull}},
{{12820602100967826106ull, 13765895222965299344ull,
  17352290931041260361ull, 5851722557814037727ull}},
{{16025752626209782633ull, 3372310973424460468ull,
  17078677645374187548ull, 7314653197267547159ull}},
{{1585446709052676675ull, 4215388716780575586ull,
  16736661038290346531ull, 9143316496584433949ull}},
{{5602590211585310826ull, 328774938774165789ull,
  12766256158145160534ull, 5714572810365271218ull}},
{{11614923782909026436ull, 9634340710322483044ull,
  6734448160826674859ull, 7143216012956589023ull}},
{{14518654728636283045ull, 7431239869475715901ull,
  3806374182605955670ull, 8929020016195736279ull}},
{{11380002214611370855ull, 32838899994934534ull, 9296512891769804150ull,
  5580637510122335174ull}},
{{5001630731409437761ull, 9264420661848443976ull,
  2397269077857479379ull, 6975796887652918968ull}},
{{6252038414261797201ull, 6968839808883167066ull,
  2996586347321849224ull, 8719746109566148710ull}},
{{8519210027341011155ull, 4355524880551979416ull,
  15707924522358319477ull, 5449841318478842943ull}},
{{10649012534176263944ull, 10056092119117362174ull,
  15023219634520511442ull, 6812301648098553679ull}},
{{4087893630865554121ull, 3346743112041926910ull,
  14167338524723251399ull, 8515377060123192099ull}},
{{16389991574573135038ull, 9009243472667286174ull,
  6548743568738338172ull, 5322110662576995062ull}},
{{11264117431361642989ull, 11261554340834107718ull,
  17409301497777698523ull, 6652638328221243827ull}},
{{4856774752347277929ull, 9465256907615246744ull,
  17149940853794735250ull, 8315797910276554784ull}},
{{3035484220217048706ull, 10527471585686917119ull,
  10718713033621709531ull, 5197373693922846740ull}},
{{17629413330553474594ull, 8547653463681258494ull,
  13398391292027136914ull, 6496717117403558425ull}},
{{12813394626337067434ull, 1461194792746797310ull,
  2912931059751757431ull, 8120896396754448032ull}},
{{3396685623033279243ull, 7830775773107830175ull,
  1820581912344848394ull, 5075560247971530020ull}},
{{18080915084073762765ull, 565097679530011910ull,
  2275727390431060493ull, 6344450309964412525ull}},
{{13377771818237427648ull, 5318058117839902792ull,
  7456345256466213520ull, 7930562887455515656ull}},
{{8361107386398392280ull, 3323786323649939245ull,
  4660215785291383450ull, 4956601804659697285ull}},
{{15063070251425378254ull, 13378104941417199864ull,
  10436955750041617216ull, 6195752255824621606ull}},
{{382093740572171202ull, 16722631176771499831ull,
  3822822650697245712ull, 7744690319780777008ull}},
{{7156337615498688857ull, 10451644485482187394ull,
  2389264156685778570ull, 4840431449862985630ull}},
{{18168794056228136880ull, 3841183569997958434ull,
  12209952232711999021ull, 6050539312328732037ull}},
{{13487620533430395291ull, 9413165480924835947ull,
  1427382235607835064ull, 7563174140410915047ull}},
{{6123919824180303105ull, 5883228425578022467ull,
  7809642924895978771ull, 4726983837756821904ull}},
{{3043213761797990977ull, 2742349513545140180ull,
  9762053656119973464ull, 5908729797196027380ull}},
{{3804017202247488722ull, 3427936891931425225ull,
  12202567070149966830ull, 7385912246495034225ull}},
{{13906725797473150211ull, 15977518612739304477ull,
  709075391202647412ull, 4616195154059396391ull}},
{{3548349191559274052ull, 1525154192214578981ull,
  14721402294285472978ull, 5770243942574245488ull}},
{{9047122507876480469ull, 11129814777122999534ull,
  18401752867856841222ull, 7212804928217806860ull}},
{{2085531097990824778ull, 4688896434548973610ull,
  4555447011111499912ull, 9016006160272258576ull}},
{{5915142954671653390ull, 2930560271593108506ull,
  2847154381944687445ull, 5635003850170161610ull}},
{{16617300730194342546ull, 8274886357918773536ull,
  12782315014285635114ull, 7043754812712702012ull}},
{{2324881839033376566ull, 1120235910543691113ull,
  15977893767857043893ull, 8804693515890877515ull}},
{{12982266195464330114ull, 3005990453303500897ull,
  7680340595696958481ull, 5502933447431798447ull}},
{{2392774689048248930ull, 8369174085056764026ull,
  4988739726193810197ull, 6878666809289748059ull}},
{{12214340398165086971ull, 15073153624748342936ull,
  1624238639314874842ull, 8598333511612185074ull}},
{{7633962748853179357ull, 14032407033895102239ull,
  5626835167999184680ull, 5373958444757615671ull}},
{{4930767417639086292ull, 17540508792368877799ull,
  2421857941571592946ull, 6717448055947019589ull}},
{{1551773253621469961ull, 12702263953606321441ull,
  7639008445391879087ull, 8396810069933774486ull}},
{{12499073329581888486ull, 14856443998645032756ull,
  162694259942536525ull, 5248006293708609054ull}},
{{15623841661977360607ull, 4735496943024127233ull,
  9426739861782946465ull, 6560007867135761317ull}},
{{5694744022189537047ull, 10531057197207546946ull,
  16395110845656070985ull, 8200009833919701646ull}},
{{8170901032295848558ull, 18111125794323186601ull,
  5635258260107656461ull, 5125006146199813529ull}},
{{14825312308797198602ull, 8803849187621819539ull,
  11655758843561958481ull, 6406257682749766911ull}},
{{13919954367569110348ull, 15616497502954662328ull,
  9958012536025060197ull, 8007822103437208639ull}},
{{8699971479730693968ull, 12066153948560357907ull,
  13141286862656744479ull, 5004888814648255399ull}},
{{6263278331235979556ull, 10471006417273059480ull,
  11814922559893542695ull, 6256111018310319249ull}},
{{7829097914044974444ull, 8477072003163936446ull, 933595144584764657ull,
  7820138772887899062ull}},
{{281500177850721124ull, 16827385048045930039ull,
  14418555020647641622ull, 4887586733054936913ull}},
{{14186933277595565117ull, 11810859273202636740ull,
  4188135720527388316ull, 6109483416318671142ull}},
{{17733666596994456396ull, 14763574091503295925ull,
  14458541687514011203ull, 7636854270398338927ull}},
{{13389384632335229200ull, 6921390797975866001ull,
  15954117582337338858ull, 4773033918998961829ull}},
{{2901672735136872787ull, 17875110534324608310ull,
  6107588922639509860ull, 5966292398748702287ull}},
{{12850462955775866792ull, 3897144094196208771ull,
  3022800134871999422ull, 7457865498435877859ull}},
{{5725696338146222793ull, 16270773114154794194ull,
  18030151148790857302ull, 4661165936522423661ull}},
{{16380492459537554299ull, 11115094355838716934ull,
  8702630880706407916ull, 5826457420653029577ull}},
{{11252243537567167066ull, 13893867944798396168ull,
  15489974619310397799ull, 7283071775816286971ull}},
{{14065304421958958833ull, 12755648912570607306ull,
  14750782255710609345ull, 9103839719770358714ull}},
{{13402501282151737175ull, 1054751542715547710ull,
  13830924928246518745ull, 5689899824856474196ull}},
{{7529754565834895660ull, 5930125446821822542ull,
  17288656160308148431ull, 7112374781070592745ull}},
{{188821170438843767ull, 2800970790099890274ull, 7775762145103021827ull,
  8890468476338240932ull}},
{{4729699249951665259ull, 17891507808308289085ull,
  14083223377544164449ull, 5556542797711400582ull}},
{{10523810080866969477ull, 8529326705103197644ull,
  8380657185075429754ull, 6945678497139250728ull}},
{{13154762601083711846ull, 1438286344524221247ull,
  10475821481344287193ull, 8682098121424063410ull}},
{{15139255653318401760ull, 12428144011396108039ull,
  11159074444267567399ull, 5426311325890039631ull}},
{{14312383548220614296ull, 10923493995817747145ull,
  9337157036907071345ull, 6782889157362549539ull}},
{{4055421379993604158ull, 18266053513199571836ull,
  7059760277706451277ull, 8478611446703186924ull}},
{{11758010399350778407ull, 13722126454963426349ull,
  13635722210421307856ull, 5299132154189491827ull}},
{{862454943906309296ull, 17152658068704282937ull,
  12432966744599246916ull, 6623915192736864784ull}},
{{5689754698310274524ull, 2994078512170802055ull,
  15541208430749058646ull, 8279893990921080980ull}},
{{10473625714085003434ull, 15706357125388914996ull,
  489883232363385845ull, 5174933744325675613ull}},
{{13092032142606254292ull, 5797888351453980033ull,
  5224040058881620211ull, 6468667180407094516ull}},
{{2529982122975654153ull, 2635674420890087138ull,
  6530050073602025264ull, 8085833975508868145ull}},
{{6192924845287171750ull, 1647296513056304461ull,
  15610496342069735550ull, 5053646234693042590ull}},
{{12352842075036352591ull, 11282492678175156384ull,
  10289748390732393629ull, 6317057793366303238ull}},
{{15441052593795440739ull, 268057792436781768ull,
  3638813451560716229ull, 7896322241707879048ull}},
{{9650657871122150462ull, 2473379129486682557ull,
  2274258407225447643ull, 4935201401067424405ull}},
{{16675008357330075982ull, 16926781967140516908ull,
  7454509027459197457ull, 6169001751334280506ull}},
{{2397016372953043361ull, 7323419403643482424ull, 94764247469221014ull,
  7711252189167850633ull}},
{{1498135233095652101ull, 18412195182559340227ull,
  11588442700736732893ull, 4819532618229906645ull}},
{{15707727096651728838ull, 9180185922917011571ull,
  650495320638752405ull, 6024415772787383307ull}},
{{15022972852387273143ull, 16086918422073652368ull,
  14648177206080604218ull, 7530519715984229133ull}},
{{9389358032742045715ull, 14666010032223420634ull,
  11460953763014071588ull, 4706574822490143208ull}},
{{2513325504072781335ull, 18332512540279275793ull,
  14326192203767589485ull, 5883218528112679010ull}},
{{7753342898518364573ull, 9080582620066931029ull,
  8684368217854711049ull, 7354023160140848763ull}},
{{14303364641575343620ull, 15962414293511051690ull,
  6243774253891000907ull, 9192528950176060954ull}},
{{13551288919411977667ull, 7670665924230713354ull,
  8514044927109263471ull, 5745330593860038096ull}},
{{7715739112410196275ull, 4976646386861003789ull,
  10642556158886579339ull, 7181663242325047620ull}},
{{14256359908940133248ull, 1609121965148866832ull,
  13303195198608224174ull, 8977079052906309525ull}},
{{8910224943087583280ull, 14840759283500205482ull,
  10620340008343834060ull, 5610674408066443453ull}},
{{1914409142004703292ull, 104205030665705237ull,
  17887111028857180480ull, 7013343010083054316ull}},
{{7004697445933267019ull, 130256288332131546ull, 3912144712361923984ull,
  8766678762603817896ull}},
{{8989621922135679791ull, 81410180207582216ull, 2445090445226202490ull,
  5479174226627386185ull}},
{{11237027402669599738ull, 9325134762114253578ull,
  7668049074960141016ull, 6848967783284232731ull}},
{{4822912216482223865ull, 11656418452642816973ull,
  4973375325272788366ull, 8561209729105290914ull}},
{{5320163144515083868ull, 2673575514474372704ull,
  7720045596722880633ull, 5350756080690806821ull}},
{{6650203930643854835ull, 7953655411520353784ull,
  14261743014330988695ull, 6688445100863508526ull}},
{{8312754913304818543ull, 5330383245973054326ull,
  8603806731058960061ull, 8360556376079385658ull}},
{{583785802388123686ull, 5637332537946852906ull, 9989065225339237942ull,
  5225347735049616036ull}},
{{9953104289839930415ull, 16270037709288341940ull,
  12486331531674047427ull, 6531684668812020045ull}},
{{12441380362299913019ull, 15725861118183039521ull,
  1772856359310395572ull, 8164605836015025057ull}},
{{858333698796363781ull, 605291162009623893ull, 12637250270637466993ull,
  5102878647509390660ull}},
{{5684603141922842630ull, 5368299970939417770ull,
  15796562838296833741ull, 6378598309386738325ull}},
{{16329125964258329095ull, 11322060982101660116ull,
  5910645492588878464ull, 7973247886733422907ull}},
{{982331690806679877ull, 7076288113813537573ull, 1388310423654355088ull,
  4983279929208389317ull}},
{{5839600631935737750ull, 8845360142266921966ull,
  6347074047995331764ull, 6229099911510486646ull}},
{{16522872826774447995ull, 11056700177833652457ull,
  17157214596848940513ull, 7786374889388108307ull}},
{{3409266489092948141ull, 18439652657214502546ull,
  8417416113816893868ull, 4866484305867567692ull}},
{{13484955148220960984ull, 4602821747808576566ull,
  10521770142271117336ull, 6083105382334459615ull}},
{{7632821898421425422ull, 5753527184760720708ull,
  8540526659411508766ull, 7603881727918074519ull}},
{{13993885723368166697ull, 17431012545757614154ull,
  12255358189773274834ull, 4752426079948796574ull}},
{{8268985117355432563ull, 12565393645342241885ull,
  6095825700361817735ull, 5940532599935995718ull}},
{{14947917415121678608ull, 11095056038250414452ull,
  16843154162307047977ull, 7425665749919994647ull}},
{{119076347596273322ull, 16880996265427177ull, 17444500379082986842ull,
  4641041093699996654ull}},
{{4760531452922729557ull, 9244473282186559779ull,
  12582253436998957744ull, 5801301367124995818ull}},
{{1338978297726024042ull, 11555591602733199724ull,
  6504444759393921372ull, 7251626708906244773ull}},
{{1673722872157530052ull, 14444489503416499655ull,
  12742241967669789619ull, 9064533386132805966ull}},
{{7963605822739538139ull, 6721962930421618332ull,
  3352215211366230608ull, 5665333366333003729ull}},
{{9954507278424422673ull, 8402453663027022915ull,
  8801955032635176164ull, 7081666707916254661ull}},
{{7831448079603140437ull, 10503067078783778644ull,
  15614129809221358109ull, 8852083384895318326ull}},
{{14118027086606738581ull, 8870259933453555604ull,
  5147145112335960914ull, 5532552115559573954ull}},
{{17647533858258423227ull, 1864452879962168697ull,
  15657303427274726951ull, 6915690144449467442ull}},
{{8224359267540865321ull, 16165624155234874584ull,
  10348257247238632880ull, 8644612680561834303ull}},
{{5140224542213040826ull, 10103515097021796615ull,
  13385189807165227406ull, 5402882925351146439ull}},
{{1813594659338913128ull, 3406021834422469961ull,
  12119801240529146354ull, 6753603656688933049ull}},
{{6878679342601029314ull, 13480899329882863259ull,
  1314693495379269230ull, 8442004570861166312ull}},
{{1993331579911949370ull, 3813876062749401633ull, 821683434612043269ull,
  5276252856788228945ull}},
{{7103350493317324616ull, 9379031096864139945ull,
  5638790311692441990ull, 6595316070985286181ull}},
{{13490874135074043674ull, 2500416834225399123ull,
  11660173908042940392ull, 8244145088731607726ull}},
{{6125953325207583344ull, 1562760521390874452ull,
  2675922674099449841ull, 5152590680457254829ull}},
{{7657441656509479180ull, 6565136670165980969ull,
  7956589361051700205ull, 6440738350571568536ull}},
{{14183488089064236879ull, 12818106856134864115ull,
  9945736701314625256ull, 8050922938214460670ull}},
{{6558837046451454098ull, 8011316785084290072ull,
  1604399419894252881ull, 5031826836384037919ull}},
{{8198546308064317622ull, 14625831999782750494ull,
  15840557330149979813ull, 6289783545480047398ull}},
{{1024810848225621219ull, 4447231944446274406ull,
  10577324625832698959ull, 7862229431850059248ull}},
{{14475564835423176974ull, 9697048992920003359ull,
  6610827891145436849ull, 4913893394906287030ull}},
{{13482770025851583314ull, 16732997259577392103ull,
  17486906900786571869ull, 6142366743632858787ull}},
{{12241776513887091238ull, 7081188519189576417ull,
  17246947607555826933ull, 7677958429541073484ull}},
{{733581293538350168ull, 6731585833707179213ull, 1555970217867616025ull,
  4798724018463170928ull}},
{{5528662635350325614ull, 13026168310561361920ull,
  1944962772334520031ull, 5998405023078963660ull}},
{{6910828294187907017ull, 11671024369774314496ull,
  2431203465418150039ull, 7498006278848704575ull}},
{{4319267683867441886ull, 14211919258750028416ull,
  8437031193527425630ull, 4686253924280440359ull}},
{{5399084604834302357ull, 8541527036582759712ull,
  5934602973481894134ull, 5857817405350550449ull}},
{{6748855756042877946ull, 1453536758873673832ull,
  12029939735279755572ull, 7322271756688188061ull}},
{{8436069695053597433ull, 1816920948592092290ull,
  1202366613817530753ull, 9152839695860235077ull}},
{{9884229577835886300ull, 12664790638938527441ull,
  3057322142849650672ull, 5720524809912646923ull}},
{{16966972990722245779ull, 15830988298673159301ull,
  17656710733844227052ull, 7150656012390808653ull}},
{{7373658183120643511ull, 1341991299631897511ull,
  8235830362023120104ull, 8938320015488510817ull}},
{{11526065392091484051ull, 838744562269935944ull,
  16676609022332919825ull, 5586450009680319260ull}},
{{14407581740114355063ull, 5660116721264807834ull,
  2399017204206598165ull, 6983062512100399076ull}},
{{8786105138288168021ull, 11686831920008397697ull,
  2998771505258247706ull, 8728828140125498845ull}},
{{17020530757498574773ull, 11915955968432636464ull,
  4180075200000098768ull, 5455517587578436778ull}},
{{2828919373163666850ull, 14894944960540795581ull,
  14448466036854899268ull, 6819396984473045972ull}},
{{8147835234881971467ull, 171937126966442860ull,
  18060582546068624086ull, 8524246230591307465ull}},
{{14315769058656007975ull, 13942518759636190499ull,
  4370335063651808197ull, 5327653894119567166ull}},
{{13283025304892622065ull, 3593090394263074412ull,
  14686290866419536055ull, 6659567367649458957ull}},
{{16603781631115777581ull, 18326421048111006727ull,
  4522805527742256356ull, 8324459209561823697ull}},
{{17294892547088442844ull, 2230641118214603396ull,
  14355968500907379983ull, 5202787005976139810ull}},
{{3171871610151001939ull, 16623359453050417958ull,
  8721588589279449170ull, 6503483757470174763ull}},
{{13188211549543528231ull, 11555827279458246639ull,
  6290299718171923559ull, 8129354696837718454ull}},
{{15160161246105787001ull, 14139921077302486005ull,
  17766495379139615936ull, 5080846685523574033ull}},
{{5115143502350070039ull, 17674901346628107507ull,
  8373061168642356208ull, 6351058356904467542ull}},
{{1782243359510199645ull, 3646882609575582768ull,
  1242954423948169453ull, 7938822946130584428ull}},
{{1113902099693874778ull, 4585144640198433182ull,
  10000218551822381716ull, 4961764341331615267ull}},
{{10615749661472119280ull, 5731430800248041477ull,
  7888587171350589241ull, 6202205426664519084ull}},
{{17881373095267537004ull, 11775974518737439750ull,
  9860733964188236551ull, 7752756783330648855ull}},
{{6564172166114822724ull, 14277513101851981700ull,
  13080487755258729700ull, 4845472989581655534ull}},
{{8205215207643528405ull, 17846891377314977125ull,
  7127237657218636317ull, 6056841236977069418ull}},
{{14868205027981798410ull, 8473556166361557694ull,
  18132419108378071205ull, 7571051546221336772ull}},
{{4680942124061236102ull, 7601815613189667511ull,
  2109389905881518695ull, 4731907216388335483ull}},
{{1239491636649157224ull, 4890583498059696485ull,
  16471795437634062081ull, 5914884020485419353ull}},
{{6161050564238834434ull, 10724915391002008510ull,
  6754686241760413889ull, 7393605025606774192ull}},
{{17685714657931435233ull, 18232287165444725078ull,
  4221678901100258680ull, 4621003141004233870ull}},
{{12883771285559518233ull, 4343614883096354732ull,
  14500470663230099159ull, 5776253926255292337ull}},
{{16104714106949397792ull, 817832585443055511ull,
  4290530273755460237ull, 7220317407819115422ull}},
{{15519206615259359335ull, 5633976750231207293ull,
  14586534879049101104ull, 9025396759773894277ull}},
{{12005347143750793537ull, 3521235468894504558ull,
  11422427308619382142ull, 5640872974858683923ull}},
{{5783311892833716113ull, 13624916372972906506ull,
  9666348117346839773ull, 7051091218573354904ull}},
{{16452511902896920949ull, 3196087410933969420ull,
  12082935146683549717ull, 8813864023216693630ull}},
{{1059447902455799785ull, 4303397641047424840ull,
  2940148448249830669ull, 5508665014510433519ull}},
{{1324309878069749732ull, 9990933069736668954ull,
  17510243615594452048ull, 6885831268138041898ull}},
{{10878759384441962972ull, 12488666337170836192ull,
  12664432482638289252ull, 8607289085172552373ull}},
{{6799224615276226858ull, 17028788497586548428ull,
  10221113310862624734ull, 5379555678232845233ull}},
{{8499030769095283572ull, 12062613585128409727ull,
  17388077657005668822ull, 6724444597791056541ull}},
{{6012102442941716561ull, 5854894944555736351ull,
  7900039015974922316ull, 8405555747238820677ull}},
{{10675093054479654707ull, 12882681377202111027ull,
  7243367394198020399ull, 5253472342024262923ull}},
{{8732180299672180479ull, 11491665703075250880ull,
  4442523224320137595ull, 6566840427530328654ull}},
{{10915225374590225599ull, 9752896110416675696ull,
  14776526067254947802ull, 8208550534412910817ull}},
{{6822015859118891000ull, 10707246087437810214ull,
  2317799764393260520ull, 5130344084008069261ull}},
{{17750891860753389558ull, 13384057609297262767ull,
  7508935723918963554ull, 6412930105010086576ull}},
{{17576928807514349043ull, 7506699974766802651ull,
  9386169654898704443ull, 8016162631262608220ull}},
{{8679737495482774200ull, 2385844475015557705ull,
  15089728071166466085ull, 5010101644539130137ull}},
{{15461357887780855654ull, 7593991612196835035ull,
  5027102033675918894ull, 6262627055673912672ull}},
{{14715011341298681663ull, 269117478391267986ull,
  6283877542094898618ull, 7828283819592390840ull}},
{{13808568106739063944ull, 4779884442421930395ull,
  3927423463809311636ull, 4892677387245244275ull}},
{{12649024114996442025ull, 5974855553027412994ull,
  297593311334251641ull, 6115846734056555344ull}},
{{6587908106890776724ull, 12080255459711654147ull,
  371991639167814551ull, 7644808417570694180ull}},
{{1811599557593041500ull, 14467688689960865698ull,
  9455866811334659902ull, 4778005260981683862ull}},
{{11487871483846077683ull, 8861238825596306314ull,
  2596461477313549070ull, 5972506576227104828ull}},
{{5136467317952821296ull, 1853176495140607085ull,
  3245576846641936338ull, 7465633220283881035ull}},
{{5516135082934207262ull, 5769921327890267332ull,
  18169386593647067875ull, 4666020762677425646ull}},
{{6895168853667759078ull, 2600715641435446261ull,
  13488361205204059036ull, 5832525953346782058ull}},
{{13230647085512086751ull, 3250894551794307826ull,
  7637079469650297987ull, 7290657441683477573ull}},
{{7314936820035332631ull, 17898676245025048495ull,
  14158035355490260387ull, 9113321802104346966ull}},
{{11489364540163164750ull, 8880829643926961357ull,
  4237086078754024838ull, 5695826126315216854ull}},
{{526647619921792226ull, 1877665018053925889ull,
  14519729635297306856ull, 7119782657894021067ull}},
{{5269995543329628186ull, 2347081272567407361ull,
  13537976025694245666ull, 8899728322367526334ull}},
{{14822962260649487377ull, 6078611813782017504ull,
  3849548997631515637ull, 5562330201479703959ull}},
{{81958752102307605ull, 12209950785654909785ull, 200250228612006642ull,
  6952912751849629949ull}},
{{4714134458555272410ull, 6039066445213861423ull,
  4861998804192396207ull, 8691140939812037436ull}},
{{9863863064238127112ull, 10691945555899745245ull,
  12262121289475023437ull, 5431963087382523397ull}},
{{16941514848725046794ull, 17976617963302069460ull,
  1492593556561615584ull, 6789953859228154247ull}},
{{2730149487196756876ull, 4024028380418035210ull,
  15700800000984183193ull, 8487442324035192808ull}},
{{6318029447925360952ull, 14044232783829741766ull,
  9813000000615114495ull, 5304651452521995505ull}},
{{17120908846761476998ull, 12943604961359789303ull,
  16877936019196281023ull, 6630814315652494381ull}},
{{16789450040024458343ull, 11567820183272348725ull,
  7262361968713187567ull, 8288517894565617977ull}},
{{12799249284228980417ull, 14147416642186299809ull,
  16068191276514211989ull, 5180323684103511235ull}},
{{2164003550004061809ull, 3849212747450711050ull,
  15473553077215377083ull, 6475404605129389044ull}},
{{11928376474359853069ull, 199829915886000908ull, 895197272809669738ull,
  8094255756411736306ull}},
{{16678607333329683976ull, 4736579715856138471ull,
  5171184313933431490ull, 5058909847757335191ull}},
{{16236573148234717066ull, 15144096681674948897ull,
  1852294373989401458ull, 6323637309696668989ull}},
{{6460658380011232621ull, 9706748815238910314ull,
  6927053985914139727ull, 7904546637120836236ull}},
{{8649597505934408292ull, 12984247037165400802ull,
  13552780778051113137ull, 4940341648200522647ull}},
{{1588624845563234557ull, 2395250741174587291ull,
  12329289954136503518ull, 6175427060250653309ull}},
{{15820839112236206908ull, 12217435463323009921ull,
  1576554387388465685ull, 7719283825313316637ull}},
{{2970495417506547462ull, 9941740173790575153ull,
  3291189501331485005ull, 4824552390820822898ull}},
{{8324805290310572231ull, 17038861235665606845ull,
  13337358913519132064ull, 6030690488526028622ull}},
{{15017692631315603193ull, 2851832470872456940ull,
  7448326605044139273ull, 7538363110657535778ull}},
{{162685857717476188ull, 13311610340363755348ull,
  9266890146579974949ull, 4711476944160959861ull}},
{{203357322146845234ull, 2804454870172530473ull,
  16195298701652356591ull, 5889346180201199826ull}},
{{4865882671110944447ull, 17340626642997826803ull,
  11020751340210669930ull, 7361682725251499783ull}},
{{1470667320461292654ull, 12452411266892507696ull,
  9164253156835949509ull, 9202103406564374729ull}},
{{919167075288307909ull, 10088600051021511262ull,
  17256873269090938203ull, 5751314629102734205ull}},
{{10372330880965160694ull, 7999064045349501173ull,
  7736033531081509042ull, 7189143286378417757ull}},
{{17577099619633838772ull, 775458019832100658ull,
  14281727932279274207ull, 8986429107973022196ull}},
{{15597373280698537137ull, 7402190290036144767ull,
  18149451994529322187ull, 5616518192483138872ull}},
{{14885030582445783517ull, 4641051844117793055ull,
  4240070919452101118ull, 7020647740603923591ull}},
{{13994602209629841492ull, 15024686842002017127ull,
  688402630887738493ull, 8775809675754904489ull}},
{{15664155408659732788ull, 11696272285464954656ull,
  11959466690373306318ull, 5484881047346815305ull}},
{{1133450187115114369ull, 5396968319976417513ull,
  1114275307684469186ull, 6856101309183519132ull}},
{{6028498752321280866ull, 15969582436825297699ull,
  1392844134605586482ull, 8570126636479398915ull}},
{{1461968710987106589ull, 14592675041443198966ull,
  17011428648624349215ull, 5356329147799624321ull}},
{{11050832925588659044ull, 13629157783376610803ull,
  7429227755498272807ull, 6695411434749530402ull}},
{{9201855138558435901ull, 12424761210793375600ull, 63162657518065201ull,
  8369264293436913003ull}},
{{5751159461599022439ull, 847946729104777894ull,
  16180377725444648415ull, 5230790183398070626ull}},
{{16412321363853553856ull, 14894991466663136079ull,
  11002100119951034710ull, 6538487729247588283ull}},
{{15903715686389554416ull, 9395367296474144291ull,
  9140939131511405484ull, 8173109661559485354ull}},
{{7633979294779777558ull, 15095476597151115990ull,
  10324772975622016331ull, 5108193538474678346ull}},
{{319102081619946140ull, 14257659728011507084ull,
  3682594182672744606ull, 6385241923093347933ull}},
{{398877602024932674ull, 8598702623159608047ull, 9214928746768318662ull,
  7981552403866684916ull}},
{{7166827528906664778ull, 762503121047367125ull,
  14982702503584974972ull, 4988470252416678072ull}},
{{13570220429560718876ull, 953128901309208906ull, 281634055771667099ull,
  6235587815520847591ull}},
{{7739403500096122787ull, 15026469181918674845ull,
  14187100624996747585ull, 7794484769401059488ull}},
{{7142970196773770694ull, 2474014211058089922ull,
  8866937890622967241ull, 4871552980875662180ull}},
{{18152084782821989175ull, 7704203782250000306ull,
  11083672363278709051ull, 6089441226094577725ull}},
{{13466733941672710661ull, 5018568709385112479ull, 19532398816222602ull,
  7611801532618222157ull}},
{{15334237741186526019ull, 7748291461793083203ull,
  2318050758473833078ull, 4757375957886388848ull}},
{{14556111158055769620ull, 461992290386578196ull,
  2897563448092291348ull, 5946719947357986060ull}},
{{18195138947569712025ull, 577490362983222745ull,
  3621954310115364185ull, 7433399934197482575ull}},
{{4454432814589988160ull, 11890146522932983976ull,
  9181250471463184471ull, 4645874958873426609ull}},
{{5568041018237485200ull, 10250997135238842066ull,
  16088249107756368493ull, 5807343698591783261ull}},
{{16183423309651632307ull, 17425432437475940486ull,
  6275253329413296904ull, 7259179623239729077ull}},
{{11005907100209764576ull, 3335046473135373992ull,
  12455752680194009035ull, 9073974529049661346ull}},
{{6878691937631102860ull, 18225305110205466409ull,
  12396531443548643550ull, 5671234080656038341ull}},
{{13210050940466266479ull, 13558259350902057203ull,
  1660606249153640726ull, 7089042600820047927ull}},
{{11900877657155445195ull, 7724452151772795696ull,
  15910815866724214620ull, 8861303251025059908ull}},
{{7438048535722153247ull, 14051154631712773118ull,
  720887879847858329ull, 5538314531890662443ull}},
{{74188632797915751ull, 3728885234358802686ull, 14736167905091986624ull,
  6922893164863328053ull}},
{{9316107827852170496ull, 4661106542948503357ull,
  4585151826082819568ull, 8653616456079160067ull}},
{{8128410401621300512ull, 2913191589342814598ull, 559876882088068278ull,
  5408510285049475042ull}},
{{937140965171849832ull, 12864861523533294056ull,
  9923218139464861155ull, 6760637856311843802ull}},
{{1171426206464812290ull, 11469390885989229666ull,
  3180650637476300636ull, 8450797320389804753ull}},
{{5343827397467895585ull, 16391741340598044349ull,
  13517121694491157657ull, 5281748325243627970ull}},
{{11291470265262257386ull, 6654618620465391724ull,
  7673030081259171264ull, 6602185406554534963ull}},
{{14114337831577821732ull, 8318273275581739655ull,
  4979601583146576176ull, 8252731758193168704ull}},
{{15738990172377220439ull, 5198920797238587284ull,
  3112250989466610110ull, 5157957348870730440ull}},
{{1226993641761973932ull, 15722023033403009914ull,
  3890313736833262637ull, 6447446686088413050ull}},
{{10757114089057243223ull, 5817470736471598680ull,
  14086264207896354105ull, 8059308357610516312ull}},
{{6723196305660777015ull, 15165134256363218935ull,
  8803915129935221315ull, 5037067723506572695ull}},
{{3792309363648583364ull, 14344731802026635765ull,
  6393207893991638740ull, 6296334654383215869ull}},
{{9352072722988117109ull, 17930914752533294706ull,
  12603195885916936329ull, 7870418317979019836ull}},
{{10456731470294961097ull, 4289292692692227335ull,
  17100369465552861014ull, 4919011448736887397ull}},
{{8459228319441313468ull, 14584987902720059977ull,
  7540403776658912555ull, 6148764310921109247ull}},
{{15185721417729029738ull, 13619548859972687067ull,
  4813818702396252790ull, 7685955388651386559ull}},
{{7185232876866949635ull, 3900532019055541513ull,
  9926165716638739850ull, 4803722117907116599ull}},
{{13593227114511074947ull, 14099037060674202699ull,
  7796021127371036908ull, 6004652647383895749ull}},
{{12379847874711455780ull, 17623796325842753374ull,
  14356712427641184039ull, 7505815809229869686ull}},
{{3125718903267271959ull, 17932401731292802715ull,
  4361259248848352120ull, 4691134880768668554ull}},
{{17742206684366253660ull, 3968758090406451777ull,
  14674946097915215959ull, 5863918600960835692ull}},
{{8342700300175653363ull, 349261594580676818ull,
  18343682622394019949ull, 7329898251201044615ull}},
{{1205003338364790896ull, 5048263011653233927ull,
  18317917259565137032ull, 9162372814001305769ull}},
{{7670656114119076166ull, 3155164382283271204ull,
  4531169259587128789ull, 5726483008750816106ull}},
{{9588320142648845207ull, 8555641496281476909ull,
  14887333611338686794ull, 7158103760938520132ull}},
{{16597086196738444413ull, 1471179833497070328ull,
  162422940463806877ull, 8947629701173150166ull}},
{{10373178872961527758ull, 3225330405149362907ull,
  13936572393072043010ull, 5592268563233218853ull}},
{{8354787572774521794ull, 13255035043291479442ull,
  3585657436057890050ull, 6990335704041523567ull}},
{{1220112429113376434ull, 7345421767259573495ull,
  18317129850354526275ull, 8737919630051904458ull}},
{{7680099295836942128ull, 2285045595323539482ull,
  16059892174898966826ull, 5461199768782440286ull}},
{{376752082941401851ull, 12079679031009200161ull,
  10851493181768932724ull, 6826499710978050358ull}},
{{5082626122104140218ull, 15099598788761500201ull,
  4340994440356390097ull, 8533124638722562948ull}},
{{14705856372383557396ull, 2519720215334855769ull,
  11936493562077519619ull, 5333202899201601842ull}},
{{4547262410197283033ull, 16984708324450733424ull,
  5697244915742123715ull, 6666503624002002303ull}},
{{5684078012746603792ull, 16619199387136028876ull,
  2509870126250266740ull, 8333129530002502879ull}},
{{12775920794821403178ull, 1163627580105242239ull,
  8486197856547498569ull, 5208205956251564299ull}},
{{11358214975099366068ull, 6066220493558940703ull,
  5996061302256985307ull, 6510257445314455374ull}},
{{9586082700446819681ull, 2971089598521287975ull,
  16718448664676007442ull, 8137821806643069217ull}},
{{12908830715420344157ull, 6468617017503192888ull,
  3531501387781422795ull, 5086138629151918261ull}},
{{16136038394275430196ull, 3474085253451603206ull,
  9026062753154166398ull, 6357673286439897826ull}},
{{10946675955989511937ull, 13565978603669279816ull,
  2059206404587932189ull, 7947091608049872283ull}},
{{6841672472493444961ull, 10784579636506993837ull,
  17427905067363315282ull, 4966932255031170176ull}},
{{13163776609044194105ull, 4257352508778966488ull,
  3338137260494592487ull, 6208665318788962721ull}},
{{16454720761305242631ull, 710004617546320206ull,
  8784357594045628513ull, 7760831648486203401ull}},
{{5672514457388388741ull, 11972967932034919889ull,
  17019438542346987580ull, 4850519780303877125ull}},
{{11702329090162873830ull, 14966209915043649861ull,
  7439240122651570763ull, 6063149725379846407ull}},
{{792853307421428575ull, 14096076375377174423ull,
  4687364134887075550ull, 7578937156724808009ull}},
{{7413062344779474715ull, 4198361716183346110ull,
  14458817630372891979ull, 4736835722953005005ull}},
{{42955894119567586ull, 636266126801794734ull, 4238463982683951262ull,
  5921044653691256257ull}},
{{9277066904504235291ull, 10018704695357019225ull,
  9909765996782326981ull, 7401305817114070321ull}},
{{17327381861383616817ull, 8567533443811830967ull,
  17722818794057424123ull, 4625816135696293950ull}},
{{17047541308302133117ull, 6097730786337400805ull,
  12930151455717004346ull, 5782270169620367438ull}},
{{7474368580095502684ull, 16845535519776526815ull,
  6939317282791479624ull, 7227837712025459298ull}},
{{4731274706691990451ull, 2610175326011106903ull,
  17897518640344125339ull, 9034797140031824122ull}},
{{9874575719323575888ull, 17772260643252799478ull,
  15797635168642466240ull, 5646748212519890076ull}},
{{3119847612299694052ull, 3768581730356447732ull,
  1300299887093531185ull, 7058435265649862596ull}},
{{3899809515374617565ull, 9322413181372947569ull,
  1625374858866913981ull, 8823044082062328245ull}},
{{13966595993177605738ull, 8132351247571786182ull,
  3321702296005515190ull, 5514402551288955153ull}},
{{8234872954617231365ull, 942067022609956920ull, 8763813888434281892ull,
  6893003189111193941ull}},
{{10293591193271539206ull, 1177583778262446150ull,
  15566453378970240269ull, 8616253986388992426ull}},
{{1821808477367324100ull, 3041832870627722796ull,
  14340719380283788072ull, 5385158741493120266ull}},
{{2277260596709155124ull, 3802291088284653495ull,
  8702527188499959282ull, 6731448426866400333ull}},
{{16681633801168607617ull, 13976235897210592676ull,
  15489845004052337006ull, 8414310533583000416ull}},
{{1202649088875603953ull, 4123461417329232519ull,
  9681153127532710629ull, 5258944083489375260ull}},
{{15338369416376668653ull, 9766012790088928552ull,
  12101441409415888286ull, 6573680104361719075ull}},
{{726217696761284200ull, 2984143950756384883ull,
  10515115743342472454ull, 8217100130452148844ull}},
{{16594787124971660289ull, 15700148024504904263ull,
  15795319376443821091ull, 5135687581532593027ull}},
{{16131797887787187458ull, 15013499012203742425ull,
  15132463202127388460ull, 6419609476915741284ull}},
{{6329689304451820610ull, 320129691545126416ull, 468834928949683960ull,
  8024511846144676606ull}},
{{3956055815282387881ull, 200081057215704010ull,
  14128079885875716187ull, 5015319903840422878ull}},
{{14168441805957760660ull, 14085159376801793724ull,
  8436727820489869425ull, 6269149879800528598ull}},
{{17710552257447200824ull, 3771391165720078443ull,
  1322537738757560974ull, 7836437349750660748ull}},
{{8763252151690806563ull, 16192177533857212739ull,
  10049958123578251416ull, 4897773343594162967ull}},
{{6342379171186120300ull, 1793477843611964308ull,
  7950761636045426367ull, 6122216679492703709ull}},
{{7927973963982650375ull, 16076905359797119097ull,
  14550138063484170862ull, 7652770849365879636ull}},
{{16484198773557626245ull, 5436379831445811531ull,
  18317208326532382597ull, 4782981780853674772ull}},
{{15993562448519644902ull, 11407160807734652318ull,
  4449766334455926630ull, 5978727226067093466ull}},
{{10768581023794780319ull, 5035578972813539590ull,
  14785579954924684096ull, 7473409032583866832ull}},
{{2118677121444349796ull, 3147236858008462244ull,
  9240987471827927560ull, 4670880645364916770ull}},
{{2648346401805437244ull, 3934046072510577805ull,
  2327862302930133642ull, 5838600806706145963ull}},
{{7922119020684184459ull, 14140929627492998064ull,
  16744885933944830764ull, 7298251008382682453ull}},
{{9902648775855230574ull, 17676162034366247580ull,
  7096049362148874743ull, 9122813760478353067ull}},
{{15412527521764294917ull, 17965130299119986593ull,
  2129187842129352762ull, 5701758600298970667ull}},
{{5430601346923204934ull, 13233040837045207434ull,
  16496542857943854665ull, 7127198250373713333ull}},
{{16011623720508781975ull, 2706242991024345580ull,
  6785620517147654620ull, 8908997812967141667ull}},
{{783892788463212927ull, 10914773906244991796ull,
  1935169814003590185ull, 5568123633104463542ull}},
{{979865985579016158ull, 18255153401233627649ull,
  11642334304359263539ull, 6960154541380579427ull}},
{{5836518500401158102ull, 18207255733114646657ull,
  9941231862021691520ull, 8700193176725724284ull}},
{{15177039108819193574ull, 11379534833196654160ull,
  15436641950618333008ull, 5437620735453577677ull}},
{{524554812314440351ull, 14224418541495817701ull,
  5460744382990752548ull, 6797025919316972097ull}},
{{5267379533820438343ull, 17780523176869772126ull,
  11437616497165828589ull, 8496282399146215121ull}},
{{17127170263919937677ull, 13418669994757301530ull,
  230981283087561012ull, 5310176499466384451ull}},
{{12185590793045146288ull, 16773337493446626913ull,
  14123784659141614977ull, 6637720624332980563ull}},
{{1396930436024269147ull, 7131613811526119930ull,
  13043044805499630818ull, 8297150780416225704ull}},
{{5484767540942556121ull, 9068944650631212860ull,
  8151903003437269261ull, 5185719237760141065ull}},
{{6855959426178195151ull, 15947866831716403979ull,
  14801564772723974480ull, 6482149047200176331ull}},
{{3958263264295356035ull, 1488089465935953358ull,
  13890269947477580197ull, 8102686309000220414ull}},
{{16308972595466761234ull, 3235898925423664800ull,
  4069732698746099719ull, 5064178943125137759ull}},
{{1939471670623899927ull, 17879931712061744713ull,
  475479855005236744ull, 6330223678906422199ull}},
{{7036025606707262812ull, 3903170566367629275ull,
  14429407874038709643ull, 7912779598633027748ull}},
{{2091672994978345306ull, 133638594766074345ull,
  18241751958128969335ull, 4945487249145642342ull}},
{{7226277262150319536ull, 14002106298739756643ull,
  13578817910806435860ull, 6181859061432052928ull}},
{{4421160559260511516ull, 17502632873424695804ull,
  16973522388508044825ull, 7727323826790066160ull}},
{{11986597386392595506ull, 4021616518249353021ull,
  10608451492817528016ull, 4829577391743791350ull}},
{{1148188677708580670ull, 5027020647811691277ull,
  4037192329167134212ull, 6036971739679739188ull}},
{{6046921865563113741ull, 6283775809764614096ull,
  5046490411458917765ull, 7546214674599673985ull}},
{{3779326165976946088ull, 6233202890316577762ull,
  14683271553230293363ull, 4716384171624796240ull}},
{{13947529744325958418ull, 3179817594468334298ull,
  18354089441537866704ull, 5895480214530995300ull}},
{{8211040143552672215ull, 3974771993085417873ull,
  4495867728212781764ull, 7369350268163744126ull}},
{{14875486197868228173ull, 4968464991356772341ull,
  14843206697120753013ull, 9211687835204680157ull}},
{{11603021882881336560ull, 5411133628811676665ull,
  11582847194914164585ull, 5757304897002925098ull}},
{{668719298319506988ull, 11375603054441983736ull,
  5255186956787929923ull, 7196631121253656373ull}},
{{835899122899383735ull, 9607817799625091766ull,
  11180669714412300308ull, 8995788901567070466ull}},
{{14357495007094278546ull, 15228258161620458161ull,
  11599604589935075596ull, 5622368063479419041ull}},
{{4111810703585684471ull, 588578628316021086ull, 664447682136680784ull,
  7027960079349273802ull}},
{{14363135416336881397ull, 735723285395026357ull,
  10053931639525626788ull, 8784950099186592252ull}},
{{11282802644424244825ull, 9683199090226667281ull,
  15507079311558292550ull, 5490593811991620157ull}},
{{268445250248142319ull, 2880626825928558294ull, 5548791084165701976ull,
  6863242264989525197ull}},
{{9558928599664953707ull, 3600783532410697867ull,
  11547674873634515374ull, 8579052831236906496ull}},
{{3668487365576902115ull, 16085547763038849879ull,
  7217296796021572108ull, 5361908019523066560ull}},
{{18420667262253291356ull, 1660190630089010732ull,
  9021620995026965136ull, 6702385024403833200ull}},
{{4579090004107062578ull, 2075238287611263416ull,
  11277026243783706420ull, 8377981280504791500ull}},
{{2861931252566914112ull, 10520395966611815443ull,
  16271513439219592320ull, 5236238300315494687ull}},
{{17412472120990806351ull, 13150494958264769303ull,
  15727705780597102496ull, 6545297875394368359ull}},
{{17153904132811120035ull, 16438118697830961629ull,
  15047946207318990216ull, 8181622344242960449ull}},
{{13027033092220643974ull, 10273824186144351018ull,
  2487437351933287029ull, 5113513965151850281ull}},
{{7060419328421029160ull, 17453966251107826677ull,
  7720982708343996690ull, 6391892456439812851ull}},
{{13437210178953674353ull, 12594085777030007538ull,
  5039542367002607959ull, 7989865570549766064ull}},
{{13009942380273434375ull, 14788832638284836567ull,
  3149713979376629974ull, 4993665981593603790ull}},
{{11650741956914405065ull, 9262668761001269901ull,
  13160514511075563276ull, 6242082476992004737ull}},
{{728369390860842619ull, 11578335951251587377ull,
  2615585083562290383ull, 7802603096240005922ull}},
{{11984445915356496397ull, 14153988997173323966ull,
  6246426695653819393ull, 4876626935150003701ull}},
{{5757185357340844688ull, 3857428191184491246ull,
  12419719387994662146ull, 6095783668937504626ull}},
{{16419853733530831668ull, 14045157275835389865ull,
  6301277198138551874ull, 7619729586171880783ull}},
{{3344879555815687937ull, 13389909315824506570ull,
  10855827276477676777ull, 4762330991357425489ull}},
{{13404471481624385729ull, 2902328589498469500ull,
  18181470114024483876ull, 5952913739196781861ull}},
{{16755589352030482161ull, 3627910736873086875ull,
  8891779587248441133ull, 7441142173995977327ull}},
{{8166400335805357399ull, 4573287219759373249ull,
  12474891269671357564ull, 4650713858747485829ull}},
{{14819686438184084652ull, 5716609024699216561ull,
  1758556031807033243ull, 5813392323434357287ull}},
{{4689549992447942103ull, 2534075262446632798ull,
  16033253095040955266ull, 7266740404292946608ull}},
{{15085309527414703437ull, 12390966114913066805ull,
  1594822295091642466ull, 9083425505366183261ull}},
{{11734161463847883600ull, 12356039840248054657ull,
  3302606943645970493ull, 5677140940853864538ull}},
{{832643774527690788ull, 1609991745027904610ull,
  13351630716412238925ull, 7096426176067330672ull}},
{{10264176755014389293ull, 6624175699712268666ull,
  16689538395515298656ull, 8870532720084163340ull}},
{{11026796490311381212ull, 4140109812320167916ull,
  1207589460342285852ull, 5544082950052602088ull}},
{{13783495612889226515ull, 5175137265400209895ull,
  1509486825427857315ull, 6930103687565752610ull}},
{{12617683497684145240ull, 1857235563322874465ull,
  11110230568639597452ull, 8662629609457190762ull}},
{{968523158411508919ull, 10384144263931572349ull,
  11555580123827136311ull, 5414143505910744226ull}},
{{5822339966441774053ull, 8368494311487077532ull,
  5221103117929144581ull, 6767679382388430283ull}},
{{7277924958052217566ull, 15072303907786234819ull,
  1914692878984042822ull, 8459599227985537854ull}},
{{2242860089568942027ull, 4808503923939008858ull,
  15031741104647190476ull, 5287249517490961158ull}},
{{12026947148815953341ull, 6010629904923761072ull,
  9566304343954212287ull, 6609061896863701448ull}},
{{15033683936019941677ull, 2901601362727313436ull,
  11957880429942765359ull, 8261327371079626810ull}},
{{172680423157687740ull, 8731029879345652754ull,
  12085361287141616253ull, 5163329606924766756ull}},
{{9439222565801885483ull, 15525473367609453846ull,
  15106701608927020316ull, 6454162008655958445ull}},
{{2575656170397581046ull, 960097635802265692ull, 5048318955876611684ull,
  8067702510819948057ull}},
{{10833157143353263962ull, 9823433059231191865ull,
  14684414393491352062ull, 5042314069262467535ull}},
{{18153132447618967856ull, 3055919287184214023ull,
  13743831973436802174ull, 6302892586578084419ull}},
{{18079729541096321916ull, 13043271145835043337ull,
  12568103948368614813ull, 7878615733222605524ull}},
{{4382301935544119342ull, 10457887475360596038ull,
  17078437004585160066ull, 4924134833264128452ull}},
{{14701249456284924985ull, 3848987307345969239ull,
  2901302182021898467ull, 6155168541580160566ull}},
{{13764875801928768327ull, 199548115755073645ull,
  12849999764382148892ull, 7693960676975200707ull}},
{{10908890385419174157ull, 9348089609201696836ull,
  5725406843525149105ull, 4808725423109500442ull}},
{{13636112981773967696ull, 16296798029929508949ull,
  16380130591261212189ull, 6010906778886875552ull}},
{{3210083171935295907ull, 6535939482129722475ull,
  2028419165366963621ull, 7513633473608594441ull}},
{{18147203046955417606ull, 6390805185544770498ull,
  12796977024422822023ull, 4696020921005371525ull}},
{{13460631771839496200ull, 3376820463503575219ull,
  2161163225246363817ull, 5870026151256714407ull}},
{{12214103696371982345ull, 8832711597806856928ull,
  16536512086840118483ull, 7337532689070893008ull}},
{{15267629620464977932ull, 6429203478831183256ull,
  2223896034840596488ull, 9171915861338616261ull}},
{{9542268512790611207ull, 4018252174269489535ull,
  3695778030989066757ull, 5732447413336635163ull}},
{{7316149622560876105ull, 9634501236264249823ull, 8036520308945542ull,
  7165559266670793954ull}},
{{4533501009773707227ull, 2819754508475536471ull,
  9233417687240957736ull, 8956949083338492442ull}},
{{9750967158749648873ull, 1762346567797210294ull,
  10382572072952986489ull, 5598093177086557776ull}},
{{2965336911582285284ull, 6814619228173900772ull,
  12978215091191233111ull, 6997616471358197220ull}},
{{3706671139477856604ull, 3906588016789988061ull,
  16222768863989041389ull, 8747020589197746525ull}},
{{4622512471387354330ull, 4747460519707436490ull,
  12445073549206844820ull, 5466887868248591578ull}},
{{15001512626088968720ull, 5934325649634295612ull,
  6332969899653780217ull, 6833609835310739473ull}},
{{305146708901659284ull, 12029593080470257420ull,
  12527898392994613175ull, 8542012294138424341ull}},
{{9414088729918312861ull, 14436024702934992743ull,
  10135779504835327186ull, 5338757683836515213ull}},
{{7155924893970503172ull, 8821658841813965121ull,
  17281410399471546887ull, 6673447104795644016ull}},
{{13556592135890516869ull, 6415387533840068497ull,
  3155018925629881993ull, 8341808880994555021ull}},
{{1555341057290491187ull, 15538832254718512571ull,
  4277729837732370197ull, 5213630550621596888ull}},
{{15779234376895277696ull, 5588482263115977001ull,
  5347162297165462747ull, 6517038188276996110ull}},
{{5888984915836933408ull, 2373916810467583348ull,
  15907324908311604242ull, 8146297735346245137ull}},
{{12903987609252859188ull, 6095384024969627496ull,
  3024549040053670795ull, 5091436084591403211ull}},
{{16129984511566073985ull, 3007544012784646466ull,
  17615744355349252206ull, 6364295105739254013ull}},
{{10939108602602816673ull, 12982802052835583891ull,
  8184622388904401545ull, 7955368882174067517ull}},
{{4531099867413066469ull, 1196722255381158076ull,
  7421232002278944918ull, 4972105551358792198ull}},
{{5663874834266333086ull, 10719274856081223403ull, 53167965993905339ull,
  6215131939198490248ull}},
{{2468157524405528453ull, 8787407551674141350ull, 66459957492381674ull,
  7768914923998112810ull}},
{{15377656508035618995ull, 10103815738223726247ull,
  4653223491860126450ull, 4855571827498820506ull}},
{{14610384616617135840ull, 3406397635924882001ull,
  15039901401679933871ull, 6069464784373525632ull}},
{{4427922715489256088ull, 18093055100188266214ull,
  353132678390365722ull, 7586830980466907041ull}},
{{16602509752462948767ull, 15919845456045054287ull,
  11749922970062448336ull, 4741769362791816900ull}},
{{16141451172151298055ull, 1453062746346766243ull,
  14687403712578060421ull, 5927211703489771125ull}},
{{15565127946761734664ull, 6428014451360845708ull,
  4524196585440411814ull, 7409014629362213907ull}},
{{504832929871308357ull, 17852567087382692280ull, 521779856686563431ull,
  4630634143351383692ull}},
{{631041162339135447ull, 17704022840800977446ull, 652224820858204289ull,
  5788292679189229615ull}},
{{10012173489778695116ull, 8294970495719058095ull,
  14650339081354919074ull, 7235365848986537018ull}},
{{7903530843795980991ull, 1145341082794046811ull,
  9089551814838873035ull, 9044207311233171273ull}},
{{2633863768158794168ull, 16856739241242136921ull,
  17210184930342765406ull, 5652629569520732045ull}},
{{7904015728625880613ull, 11847552014697895343ull,
  7677673107646293046ull, 7065786961900915057ull}},
{{5268333642354962863ull, 5586067981517593371ull,
  14208777402985254212ull, 8832233702376143821ull}},
{{986865517258157837ull, 12714664525303271665ull,
  11186328886079477834ull, 5520146063985089888ull}},
{{5845267915000085201ull, 6669958619774313773ull,
  13982911107599347293ull, 6900182579981362360ull}},
{{11918270912177494405ull, 12949134293145280120ull,
  17478638884499184116ull, 8625228224976702950ull}},
{{7448919320110934003ull, 17316580970070575883ull,
  6312463284384602168ull, 5390767640610439344ull}},
{{4699463131711279600ull, 3198982138878668238ull,
  7890579105480752711ull, 6738459550763049180ull}},
{{15097700951493875308ull, 17833785728880499009ull,
  9863223881850940888ull, 8423074438453811475ull}},
{{2518534067042590211ull, 11146116080550311881ull,
  3858671916943144103ull, 5264421524033632172ull}},
{{7759853602230625668ull, 9320959082260501947ull,
  4823339896178930129ull, 6580526905042040215ull}},
{{5088130984360894181ull, 16262884871253015338ull,
  1417488851796274757ull, 8225658631302550269ull}},
{{7791767883652946767ull, 12470146053746828538ull,
  3191773541586365675ull, 5141036644564093918ull}},
{{516337817711407651ull, 10975996548756147769ull,
  13213088963837732902ull, 6426295805705117397ull}},
{{5257108290566647468ull, 4496623649090408903ull,
  2681303149515002416ull, 8032869757131396747ull}},
{{10203221709245236523ull, 2810389780681505564ull,
  17816715532942734174ull, 5020543598207122966ull}},
{{12754027136556545654ull, 12736359262706657763ull,
  13047522379323641909ull, 6275679497758903708ull}},
{{11330847902268294164ull, 2085391023101158492ull,
  16309402974154552387ull, 7844599372198629635ull}},
{{16305151975772459660ull, 17444270453934081721ull,
  7887533849632901289ull, 4902874607624143522ull}},
{{6546381914433410863ull, 7970280012135438440ull, 636045275186350804ull,
  6128593259530179403ull}},
{{8182977393041763579ull, 9962850015169298050ull,
  14630114649265102217ull, 7660741574412724253ull}},
{{9726046889078490141ull, 17755996305549281041ull,
  11449664665004382837ull, 4787963484007952658ull}},
{{16769244629775500580ull, 8359937326654437589ull,
  5088708794400702739ull, 5984954355009940823ull}},
{{7126497731937212013ull, 5838235639890659083ull,
  1749199974573490520ull, 7481192943762426029ull}},
{{2148218073247063557ull, 3648897274931661927ull,
  3399092993322125527ull, 4675745589851516268ull}},
{{16520330646840993158ull, 18396179648946741120ull,
  4248866241652656908ull, 5844681987314395335ull}},
{{2203669234841689831ull, 4548480487473874785ull, 699396783638433232ull,
  7305852484142994169ull}},
{{7366272561979500192ull, 5685600609342343481ull,
  5485931997975429444ull, 9132315605178742711ull}},
{{16133135397305657380ull, 12776872417693740483ull,
  10346236526375725258ull, 5707697253236714194ull}},
{{15554733228204683821ull, 6747718485262399796ull,
  3709423621114880765ull, 7134621566545892743ull}},
{{996672461546303161ull, 13046334125005387650ull, 25093507966213052ull,
  8918276958182365929ull}},
{{5234606306893827380ull, 17377330864983143089ull,
  11544898488547352917ull, 5573923098863978705ull}},
{{11154943902044672128ull, 7886605525946765149ull,
  596065055402027435ull, 6967403873579973382ull}},
{{108621822273676448ull, 5246570889006068533ull, 9968453356107310102ull,
  8709254841974966727ull}},
{{2373731648134741732ull, 17114164860910956545ull,
  13147812375208150669ull, 5443284276234354204ull}},
{{7578850578595815069ull, 7557648020856531969ull,
  16434765469010188337ull, 6804105345292942755ull}},
{{14085249241672156741ull, 14058746044498052865ull,
  15931770817835347517ull, 8505131681616178444ull}},
{{1885751748404016107ull, 11092559287024976993ull,
  733984724292316390ull, 5315707301010111528ull}},
{{6968875703932408038ull, 4642327071926445433ull, 917480905365395488ull,
  6644634126262639410ull}},
{{13322780648342897951ull, 5802908839908056791ull,
  10370223168561520168ull, 8305792657828299262ull}},
{{15244266932855393076ull, 3626818024942535494ull,
  1869703461923562201ull, 5191120411142687039ull}},
{{9831961629214465536ull, 9145208549605557272ull,
  16172187382686616463ull, 6488900513928358798ull}},
{{12289952036518081920ull, 6819824668579558686ull,
  10991862191503494771ull, 8111125642410448498ull}},
{{3069534004396413296ull, 1956547408648530227ull,
  11481599888117072136ull, 5069453526506530311ull}},
{{17671975560777680332ull, 2445684260810662783ull,
  9740313841718952266ull, 6336816908133162889ull}},
{{17478283432544712511ull, 12280477362868104287ull,
  16787078320576078236ull, 7921021135166453611ull}},
{{17841456172981527176ull, 16898670388647340987ull,
  8186080941146354945ull, 4950638209479033507ull}},
{{17690134197799521066ull, 7288279930527012522ull,
  5620915158005555778ull, 6188297761848791884ull}},
{{12889295710394625524ull, 18333721950013541461ull,
  7026143947506944722ull, 7735372202310989855ull}},
{{10361652828210334905ull, 16070262237185851317ull,
  11308868994832922307ull, 4834607626444368659ull}},
{{17563752053690306535ull, 15476141778054926242ull,
  9524400225113764980ull, 6043259533055460824ull}},
{{12731318030258107360ull, 898433148859106187ull,
  11905500281392206226ull, 7554074416319326030ull}},
{{5651230759697623148ull, 5173206736464329271ull,
  2829251657442740987ull, 4721296510199578769ull}},
{{2452352431194641031ull, 1854822402153023685ull,
  8148250590230814138ull, 5901620637749473461ull}},
{{7677126557420689193ull, 11541900039546055414ull,
  14796999256215905576ull, 7377025797186841826ull}},
{{373036159921085683ull, 14427375049432569268ull,
  9272877033415106162ull, 9221282246483552283ull}},
{{9456519636805454360ull, 13628795424322743696ull,
  3489705136670747399ull, 5763301404052220177ull}},
{{11820649546006817950ull, 12424308261976041716ull,
  8973817439265822153ull, 7204126755065275221ull}},
{{14775811932508522437ull, 1695327272187888433ull,
  15828957817509665596ull, 9005158443831594026ull}},
{{2317353430176744667ull, 10282951581972206079ull,
  14504784654370928901ull, 5628224027394746266ull}},
{{16731749843003094546ull, 17465375495892645502ull,
  8907608781108885318ull, 7035280034243432833ull}},
{{11691315266899092375ull, 12608347333011031070ull,
  15746196994813494552ull, 8794100042804291041ull}},
{{2695386023384544830ull, 7880217083131894419ull,
  2923844094117352239ull, 5496312526752681901ull}},
{{17204290584512844750ull, 5238585335487480119ull,
  8266491136074078203ull, 6870390658440852376ull}},
{{16893677212213668033ull, 1936545650931962245ull,
  10333113920092597754ull, 8587988323051065470ull}},
{{12864391266847236473ull, 5822027050259864307ull,
  1846510181630485692ull, 5367492701906915919ull}},
{{11468803065131657687ull, 7277533812824830384ull,
  16143195782320270827ull, 6709365877383644898ull}},
{{14336003831414572109ull, 4485231247603650076ull,
  10955622691045562726ull, 8386707346729556123ull}},
{{18183374431488883376ull, 16638327585034445009ull,
  4541421172689782751ull, 5241692091705972577ull}},
{{8894159984078940508ull, 16186223462865668358ull,
  10288462484289616343ull, 6552115114632465721ull}},
{{1894327943243899827ull, 15621093310154697544ull,
  17472264123789408333ull, 8190143893290582151ull}},
{{1183954964527437392ull, 12069026328060379917ull,
  17837694105009462064ull, 5118839933306613844ull}},
{{6091629724086684644ull, 15086282910075474896ull,
  3850373557552275964ull, 6398549916633267306ull}},
{{7614537155108355805ull, 411109563884792004ull,
  14036338983795120764ull, 7998187395791584132ull}},
{{13982457758797498186ull, 9480315514282770810ull,
  17996083901726726285ull, 4998867122369740082ull}},
{{8254700161642096924ull, 16462080411280851417ull,
  13271732840303632048ull, 6248583902962175103ull}},
{{14930061220480009059ull, 2130856440391512655ull,
  11977980031952152157ull, 7810729878702718879ull}},
{{16248817290441087518ull, 3637628284458389361ull,
  14403766547611176954ull, 4881706174189199299ull}},
{{6475963557769195686ull, 13770407392427762510ull,
  13393022166086583288ull, 6102132717736499124ull}},
{{17318326484066270415ull, 17213009240534703137ull,
  16741277707608229110ull, 7627665897170623905ull}},
{{3906425024900337154ull, 6146444756906801557ull,
  3545769539614061338ull, 4767291185731639941ull}},
{{9494717299552809346ull, 16906427982988277754ull,
  9043897942944964576ull, 5959113982164549926ull}},
{{2645024587586235874ull, 2686290905025795577ull,
  2081500391826429913ull, 7448892477705687408ull}},
{{13182355413309867182ull, 13208146861709591995ull,
  1300937744891518695ull, 4655557798566054630ull}},
{{11866258248209946073ull, 11898497558709602090ull,
  10849544217969174177ull, 5819447248207568287ull}},
{{5609450773407656783ull, 1038063893104838901ull,
  8950244254034079818ull, 7274309060259460359ull}},
{{11623499485186958883ull, 10520951903235824434ull,
  6576119299115211868ull, 9092886325324325449ull}},
{{11876373196669237206ull, 15798966976377166079ull,
  15639289608015477177ull, 5683053953327703405ull}},
{{10233780477409158603ull, 5913650665189293887ull,
  5714053954737182760ull, 7103817441659629257ull}},
{{8180539578334060350ull, 7392063331486617359ull,
  11754253461848866354ull, 8879771802074536571ull}},
{{12030366264099869575ull, 9231725600606523753ull,
  5040565404441847519ull, 5549857376296585357ull}},
{{1202899774842673256ull, 6927970982330766788ull,
  10912392773979697303ull, 6937321720370731696ull}},
{{1503624718553341570ull, 4048277709486070581ull,
  13640490967474621629ull, 8671652150463414620ull}},
{{3245608458309532434ull, 4836016577642488065ull,
  17748678891526414326ull, 5419782594039634137ull}},
{{8668696591314303446ull, 15268392758907885889ull,
  8350790559125854195ull, 6774728242549542672ull}},
{{15447556757570267211ull, 14473804930207469457ull,
  10438488198907317744ull, 8468410303186928340ull}},
{{2737193945840335151ull, 9046128081379668411ull,
  15747427161171849398ull, 5292756439491830212ull}},
{{17256550487582582651ull, 2084288064869809705ull,
  1237539877755260132ull, 6615945549364787766ull}},
{{7735630054196064602ull, 2605360081087262132ull,
  10770296884048850973ull, 8269931936705984707ull}},
{{14058140820727316184ull, 3934193059893232784ull,
  4425592543316837906ull, 5168707460441240442ull}},
{{17572676025909145230ull, 14141113361721316788ull,
  14755362716000823190ull, 6460884325551550552ull}},
{{3519100958676879922ull, 8453019665296870178ull,
  18444203395001028988ull, 8076105406939438190ull}},
{{6811124117600437855ull, 14506509327665319669ull,
  6915941103448255213ull, 5047565879337148869ull}},
{{13125591165427935223ull, 4298078604299485874ull,
  13256612397737706921ull, 6309457349171436086ull}},
{{7183616919930143220ull, 9984284273801745247ull,
  7347393460317357843ull, 7886821686464295108ull}},
{{11407289602597421369ull, 3934334661912396827ull,
  13815492949553124460ull, 4929263554040184442ull}},
{{9647425984819388807ull, 4917918327390496034ull,
  8045994150086629767ull, 6161579442550230553ull}},
{{2835910444169460201ull, 1535711890810732139ull,
  14669178706035675113ull, 7701974303187788191ull}},
{{17913345092101770290ull, 12489034977825177346ull,
  16085765718913378801ull, 4813733939492367619ull}},
{{13168309328272437054ull, 1776235666999307971ull,
  15495521130214335598ull, 6017167424365459524ull}},
{{11848700641913158413ull, 11443666620603910772ull,
  922657339058367881ull, 7521459280456824406ull}},
{{16628809938050499816ull, 234762610236362376ull,
  14411718892193643638ull, 4700912050285515253ull}},
{{2339268348853573154ull, 9516825299650228779ull,
  4179590559959890835ull, 5876140062856894067ull}},
{{16759143491349130155ull, 7284345606135398069ull,
  612802181522475640ull, 7345175078571117584ull}},
{{7113871308904248982ull, 9105432007669247587ull, 766002726903094550ull,
  9181468848213896980ull}},
{{2140326558851461662ull, 1079208986365891838ull,
  9702123741169209902ull, 5738418030133685612ull}},
{{11898780235419102885ull, 10572383269812140605ull,
  12127654676461512377ull, 7173022537667107015ull}},
{{1038417238991714894ull, 17827165105692563661ull,
  10547882327149502567ull, 8966278172083883769ull}},
{{2954853783583515761ull, 18059507218698934144ull,
  18121641500536908864ull, 5603923857552427355ull}},
{{3693567229479394701ull, 4127639949664116064ull,
  18040365857243748177ull, 7004904821940534194ull}},
{{4616959036849243376ull, 9771235955507532984ull,
  13327085284699909413ull, 8756131027425667743ull}},
{{2885599398030777110ull, 8412865481405902067ull,
  15246957330578525239ull, 5472581892141042339ull}},
{{17442057302820635100ull, 5904395833329989679ull,
  14447010644795768645ull, 6840727365176302924ull}},
{{17190885610098405971ull, 11992180810089875003ull,
  18058763305994710806ull, 8550909206470378655ull}},
{{8438460497097809780ull, 2883426987878783973ull,
  18204256093887776110ull, 5344318254043986659ull}},
{{15159761639799650129ull, 12827655771703255774ull,
  18143634098932332233ull, 6680397817554983324ull}},
{{9726330012894786853ull, 2199511659346906006ull,
  4232798549955863676ull, 8350497271943729156ull}},
{{1467270239631853879ull, 10598066823946592062ull,
  11868871130577190605ull, 5219060794964830722ull}},
{{11057459836394593157ull, 17859269548360627981ull,
  5612716876366712448ull, 6523825993706038403ull}},
{{18433510813920629350ull, 3877342861741233360ull,
  2404210077031002657ull, 8154782492132548004ull}},
{{11520944258700393344ull, 13952554334656740610ull,
  10726003334999152468ull, 5096739057582842502ull}},
{{5177808286520715872ull, 17440692918320925763ull,
  4184132131894164777ull, 6370923821978553128ull}},
{{1860574339723506935ull, 7965808092618993492ull,
  5230165164867705972ull, 7963654777473191410ull}},
{{10386230999181967643ull, 14202002094741646740ull,
  7880539246469704136ull, 4977284235920744631ull}},
{{12982788748977459553ull, 17752502618427058425ull,
  5238988039659742266ull, 6221605294900930789ull}},
{{2393427880939660730ull, 12967256236179047224ull,
  11160421068002065737ull, 7777006618626163486ull}},
{{1495892425587287956ull, 1187006119970822659ull,
  2363577149073903182ull, 4860629136641352179ull}},
{{15704923587266273657ull, 10707129686818304131ull,
  16789529491624542689ull, 6075786420801690223ull}},
{{15019468465655454167ull, 17995598126950268068ull,
  16375225846103290457ull, 7594733026002112779ull}},
{{163795754179883047ull, 4329719801702835687ull, 7928673144600862584ull,
  4746708141251320487ull}},
{{14039802748007017520ull, 5412149752128544608ull,
  5299155412323690326ull, 5933385176564150609ull}},
{{17549753435008771900ull, 15988559227015456568ull,
  11235630283832000811ull, 7416731470705188261ull}},
{{10968595896880482438ull, 7687006507670966403ull,
  9328111936608694459ull, 4635457169190742663ull}},
{{9099058852673215143ull, 4997072116161320100ull,
  7048453902333480170ull, 5794321461488428329ull}},
{{11373823565841518929ull, 15469712182056425933ull,
  13422253396344238116ull, 7242901826860535411ull}},
{{382221402019734949ull, 890396153860980801ull, 12166130727002909742ull,
  9053627283575669264ull}},
{{11768103422330804103ull, 14391555651445276712ull,
  7603831704376818588ull, 5658517052234793290ull}},
{{14710129277913505129ull, 17989444564306595890ull,
  281417593616247427ull, 7073146315293491613ull}},
{{9164289560537105603ull, 17875119686955856959ull,
  4963458010447697188ull, 8841432894116864516ull}},
{{12645210002976772858ull, 1948577767492634791ull,
  12325533293384586551ull, 5525895558823040322ull}},
{{11194826485293578169ull, 16270780264647957201ull,
  6183544579875957380ull, 6907369448528800403ull}},
{{158475051334808999ull, 1891731257100394886ull, 3117744706417558822ull,
  8634211810661000504ull}},
{{13934104962366419336ull, 15017390090969910515ull,
  1948590441510974263ull, 5396382381663125315ull}},
{{12805945184530636266ull, 14160051595285000240ull,
  16270796107170881541ull, 6745477977078906643ull}},
{{16007431480663295333ull, 3865006438824086588ull,
  15726809115536214023ull, 8431847471348633304ull}},
{{781272638559783775ull, 9333158051906135974ull, 9829255697210133764ull,
  5269904669592895815ull}},
{{10199962835054505527ull, 11666447564882669967ull,
  7674883603085279301ull, 6587380836991119769ull}},
{{8138267525390744004ull, 748001400821173747ull,
  14205290522283987031ull, 8234226046238899711ull}},
{{2780574194155521051ull, 7385029903154315448ull,
  15795835604068573750ull, 5146391278899312319ull}},
{{3475717742694401313ull, 7915342088118502ull, 15133108486658329284ull,
  6432989098624140399ull}},
{{13568019215222777450ull, 9894177610148127ull, 14304699589895523701ull,
  8041236373280175499ull}},
{{15397541037155317762ull, 2312026870220036531ull,
  6634594234471008361ull, 5025772733300109687ull}},
{{14635240278016759299ull, 7501719606202433568ull,
  3681556774661372547ull, 6282215916625137109ull}},
{{18294050347520949123ull, 4765463489325654056ull,
  9213631986754103588ull, 7852769895781421386ull}},
{{11433781467200593202ull, 12201786717683309593ull,
  10370206010148702646ull, 4907981184863388366ull}},
{{457168778718577791ull, 6028861360249361184ull, 3739385475831102500ull,
  6134976481079235458ull}},
{{571460973398222238ull, 7536076700311701480ull,
  13897603881643653933ull, 7668720601349044322ull}},
{{357163108373888899ull, 7015890946908507377ull,
  13297688444454671612ull, 4792950375843152701ull}},
{{5058139903894749028ull, 8769863683635634221ull,
  2787052500286175803ull, 5991187969803940877ull}},
{{10934360898295824189ull, 6350643586117154872ull,
  8095501643785107658ull, 7488984962254926096ull}},
{{6833975561434890118ull, 8580838259750609699ull,
  5059688527365692286ull, 4680615601409328810ull}},
{{3930783433366224743ull, 1502675787833486316ull,
  15547982696061891166ull, 5850769501761661012ull}},
{{4913479291707780929ull, 11101716771646633703ull,
  988234296367812341ull, 7313461877202076266ull}},
{{1530163096207338257ull, 42087909276128417ull, 10458664907314541235ull,
  9141827346502595332ull}},
{{12485566981198056171ull, 16167206007793437924ull,
  15760037603926364079ull, 5713642091564122082ull}},
{{15606958726497570214ull, 15597321491314409501ull,
  10476674968053179291ull, 7142052614455152603ull}},
{{5673640352839799055ull, 14884965845715623973ull,
  8484157691639086210ull, 8927565768068940754ull}},
{{5851868229738568361ull, 13914789671999652887ull,
  9914284575701816785ull, 5579728605043087971ull}},
{{2703149268745822548ull, 3558429034717402397ull,
  7781169701199883078ull, 6974660756303859964ull}},
{{7990622604359666089ull, 13671408330251528804ull,
  9726462126499853847ull, 8718325945379824955ull}},
{{14217511164579567114ull, 15462159234048287358ull,
  3773195819848714702ull, 5448953715862390597ull}},
{{8548516918869683084ull, 10104327005705583390ull,
  9328180793238281282ull, 6811192144827988246ull}},
{{1462274111732328047ull, 3407036720277203430ull,
  2436853954693075795ull, 8513990181034985308ull}},
{{14748979375114868741ull, 18270299014669109807ull,
  10746405758537948179ull, 5321243863146865817ull}},
{{13824538200466198023ull, 18226187749908999355ull,
  18044693216599823128ull, 6651554828933582271ull}},
{{12668986732155359624ull, 4335990613676697578ull,
  17944180502322391007ull, 8314443536166977839ull}},
{{12529802726024487669ull, 9627523161189017842ull,
  18132641841592576235ull, 5196527210104361149ull}},
{{6438881370675833778ull, 7422717933058884399ull,
  8830744246708556582ull, 6495659012630451437ull}},
{{3436915694917404319ull, 55025379468829691ull, 15650116326813083632ull,
  8119573765788064296ull}},
{{18288973373819235363ull, 34390862168018556ull, 9781322704258177270ull,
  5074733603617540185ull}},
{{4414472643564492588ull, 9266360614564799004ull,
  16838339398750109491ull, 6343417004521925231ull}},
{{5518090804455615735ull, 6971264749778610851ull,
  16436238230010248960ull, 7929271255652406539ull}},
{{1142963743571065883ull, 4357040468611631782ull,
  7966805884542711648ull, 4955794534782754087ull}},
{{10652076716318608161ull, 5446300585764539727ull,
  5346821337251001656ull, 6194743168478442609ull}},
{{8703409876970872297ull, 6807875732205674659ull,
  11295212689991139974ull, 7743428960598053261ull}},
{{3133788163893101234ull, 18089980387910710374ull,
  9365350940458156435ull, 4839643100373783288ull}},
{{13140607241721152350ull, 18000789466461000063ull,
  11706688675572695544ull, 6049553875467229110ull}},
{{11814073033724052534ull, 4054242759366698463ull,
  5409988807611093623ull, 7561942344334036388ull}},
{{14301324673718614690ull, 9451430752245268395ull,
  12604615041611709322ull, 4726213965208772742ull}},
{{13264969823720880458ull, 2590916403451809686ull,
  6532396765159860845ull, 5907767456510965928ull}},
{{7357840242796324765ull, 7850331522742150012ull,
  8165495956449826056ull, 7384709320638707410ull}},
{{13822022188602478786ull, 4906457201713843757ull,
  9715120991208529189ull, 4615443325399192131ull}},
{{3442469680470934770ull, 10744757520569692601ull,
  7532215220583273582ull, 5769304156748990164ull}},
{{8914773119016056367ull, 4207574863857339943ull,
  9415269025729091978ull, 7211630195936237705ull}},
{{6531780380342682555ull, 14482840616676450737ull,
  16380772300588752876ull, 9014537744920297131ull}},
{{15611577783782646357ull, 18275147422277557518ull,
  7932139678654276595ull, 5634086090575185707ull}},
{{10291100192873532138ull, 18232248259419558994ull,
  5303488579890457840ull, 7042607613218982134ull}},
{{3640503204237139364ull, 4343566250564897127ull,
  15852732761717848109ull, 8803259516523727667ull}},
{{9192843530289293959ull, 5020571915816754656ull,
  7602114966859961116ull, 5502037197827329792ull}},
{{11491054412861617448ull, 6275714894770943320ull,
  9502643708574951395ull, 6877546497284162240ull}},
{{14363818016077021810ull, 3232957600036291246ull,
  11878304635718689244ull, 8596933121605202800ull}},
{{4365700241620750728ull, 11243970536877457837ull,
  7423940397324180777ull, 5373083201003251750ull}},
{{10068811320453326313ull, 219905115814658584ull, 56553459800450164ull,
  6716354001254064688ull}},
{{12586014150566657892ull, 274881394768323230ull, 70691824750562705ull,
  8395442501567580860ull}},
{{3254572825676773279ull, 11701015917798671779ull,
  9267554427323877498ull, 5247151563479738037ull}},
{{17903274087378130310ull, 5402897860393563915ull,
  16196129052582234777ull, 6558939454349672546ull}},
{{17767406590795274983ull, 11365308343919342798ull,
  11021789278873017663ull, 8198674317937090683ull}},
{{6492943100819658961ull, 14020846742590671105ull,
  4582775290081942087ull, 5124171448710681677ull}},
{{12727864894451961605ull, 12914372409810950977ull,
  10340155131029815513ull, 6405214310888352096ull}},
{{2074773062782788294ull, 2307907456981525010ull,
  12925193913787269392ull, 8006517888610440120ull}},
{{5908419182666630588ull, 1442442160613453131ull,
  8078246196117043370ull, 5004073680381525075ull}},
{{2773837959905900331ull, 11026424737621592222ull,
  5486121726718916308ull, 6255092100476906344ull}},
{{12690669486737151221ull, 13783030922026990277ull,
  6857652158398645385ull, 7818865125596132930ull}},
{{10237511438424413466ull, 1696865298625787067ull,
  8897718617426541270ull, 4886790703497583081ull}},
{{8185203279603128928ull, 11344453660137009642ull,
  15733834290210564491ull, 6108488379371978851ull}},
{{1008132062649135352ull, 9568881056743874149ull,
  15055606844335817710ull, 7635610474214973564ull}},
{{2935925548369403547ull, 1368864642037533439ull, 186382240855110261ull,
  4772256546384358478ull}},
{{17504964990743918146ull, 6322766820974304702ull,
  9456349837923663634ull, 5965320682980448097ull}},
{{12657834201575121874ull, 17126830563072656686ull,
  16432123315831967446ull, 7456650853725560121ull}},
{{3299460357557063267ull, 6092583083493022525ull,
  3352548044753897798ull, 4660406783578475076ull}},
{{8736011465373716988ull, 16839100891221053964ull,
  4190685055942372247ull, 5825508479473093845ull}},
{{10920014331717146235ull, 16437190095598929551ull,
  9850042338355353213ull, 7281885599341367306ull}},
{{9038331896219044890ull, 6711429564216498227ull,
  3089180886089415709ull, 9102356999176709133ull}},
{{3343114425923209104ull, 6500486486849005344ull,
  4236581063019578770ull, 5688973124485443208ull}},
{{4178893032404011380ull, 17348980145416032488ull,
  5295726328774473462ull, 7111216405606804010ull}},
{{5223616290505014225ull, 12462853144915264802ull,
  15843029947822867636ull, 8889020507008505012ull}},
{{7876446199993021795ull, 17012655252426816309ull,
  678521680534516464ull, 5555637816880315633ull}},
{{14457243768418665147ull, 2819074991823968770ull,
  5459838119095533485ull, 6944547271100394541ull}},
{{8848182673668555626ull, 8135529758207348867ull,
  11436483667296804760ull, 8680684088875493176ull}},
{{3224271161829153314ull, 5084706098879593042ull,
  7147802292060502975ull, 5425427555547183235ull}},
{{13253710989141217451ull, 1744196605172103398ull,
  4323066846648240815ull, 6781784444433979044ull}},
{{7343766699571746006ull, 16015303811747292960ull,
  5403833558310301018ull, 8477230555542473805ull}},
{{4589854187232341254ull, 14621250900769446004ull,
  5683238983157632088ull, 5298269097214046128ull}},
{{5737317734040426567ull, 18276563625961807505ull,
  7104048728947040110ull, 6622836371517557660ull}},
{{11783333185977921113ull, 13622332495597483573ull,
  8880060911183800138ull, 8278545464396947075ull}},
{{9670426250449894648ull, 13125643828175815137ull,
  3244195060276181134ull, 5174090915248091922ull}},
{{16699718831489756213ull, 7183682748364993113ull,
  13278615862200002226ull, 6467613644060114902ull}},
{{7039590484080031555ull, 18202975472311017200ull,
  7374897790895226974ull, 8084517055075143628ull}},
{{4399744052550019722ull, 6765173651766997846ull,
  13832683156164292667ull, 5052823159421964767ull}},
{{14723052102542300460ull, 3844781046281359403ull,
  12679167926777977930ull, 6316028949277455959ull}},
{{13792129109750487671ull, 14029348344706475062ull,
  11237273890045084508ull, 7895036186596819949ull}},
{{4008394675166666891ull, 17991714752296322722ull,
  9329139190491871769ull, 4934397616623012468ull}},
{{14233865380813109421ull, 8654585385088239690ull,
  11661423988114839712ull, 6167997020778765585ull}},
{{8568959689161610968ull, 10818231731360299613ull,
  741721929861385928ull, 7709996275973456982ull}},
{{7661442814939700807ull, 6761394832100187258ull,
  14298634261445529917ull, 4818747672483410613ull}},
{{353431481819850201ull, 13063429558552621977ull,
  4038234771524748684ull, 6023434590604263267ull}},
{{5053475370702200655ull, 16329286948190777471ull,
  436107445978547951ull, 7529293238255329084ull}},
{{10075951134329957266ull, 17123333370260317775ull,
  9495939190591368277ull, 4705808273909580677ull}},
{{7983252899485058678ull, 7569108657543233507ull,
  16481610006666598251ull, 5882260342386975846ull}},
{{5367380105928935443ull, 4849699803501653980ull,
  11378640471478472006ull, 7352825427983719808ull}},
{{6709225132411169304ull, 15285496791231843283ull,
  14223300589348090007ull, 9191031784979649760ull}},
{{1887422698543286863ull, 16470964522160983908ull,
  8889562868342556254ull, 5744394865612281100ull}},
{{2359278373179108579ull, 11365333615846454077ull,
  11111953585428195318ull, 7180493582015351375ull}},
{{7560783984901273627ull, 4983294982953291788ull,
  9278255963357856244ull, 8975616977519189219ull}},
{{13948862027418071825ull, 12337931401200583175ull,
  3493066967884966200ull, 5609760610949493262ull}},
{{12824391515845201878ull, 15422414251500728969ull,
  13589705746710983558ull, 7012200763686866577ull}},
{{2195431339524338635ull, 10054645777521135404ull,
  3152074128106565736ull, 8765250954608583222ull}},
{{10595516624057487455ull, 6284153610950709627ull,
  15805104385348767297ull, 5478281846630364513ull}},
{{8632709761644471415ull, 12466878032115774938ull,
  5921322426403795409ull, 6847852308287955642ull}},
{{1567515165200813460ull, 1748539484862554961ull,
  16625025069859520070ull, 8559815385359944552ull}},
{{12508912024318978173ull, 14927895233321260562ull,
  10390640668662200043ull, 5349884615849965345ull}},
{{6412767993543946908ull, 14048183023224187799ull,
  17599986854255137958ull, 6687355769812456681ull}},
{{3404273973502545731ull, 8336856742175458941ull,
  8164925512536758736ull, 8359194712265570852ull}},
{{4433514242652785034ull, 5210535463859661838ull,
  14326450482190250018ull, 5224496695165981782ull}},
{{14765264840170757100ull, 15736541366679353105ull,
  8684691065883036714ull, 6530620868957477228ull}},
{{4621522994931282663ull, 10447304671494415574ull,
  10855863832353795893ull, 8163276086196846535ull}},
{{16723509927114215377ull, 8835408428897703685ull,
  13702443922862204289ull, 5102047553873029084ull}},
{{7069329353610605509ull, 15655946554549517511ull,
  17128054903577755361ull, 6377559442341286355ull}},
{{4224975673585868982ull, 5734875137904733177ull,
  16798382611044806298ull, 7971949302926607944ull}},
{{14169824842059637874ull, 8195982979617846139ull,
  10498989131903003936ull, 4982468314329129965ull}},
{{13100595034147159438ull, 10244978724522307674ull,
  17735422433306142824ull, 6228085392911412456ull}},
{{7152371755829173489ull, 12806223405652884593ull,
  3722533967923126914ull, 7785106741139265571ull}},
{{15999447393461703191ull, 12615575646960440774ull,
  20740720738260369ull, 4865691713212040982ull}},
{{10775937204972353181ull, 1934411503418387256ull,
  9249297937777601270ull, 6082114641515051227ull}},
{{13469921506215441476ull, 11641386416127759878ull,
  6949936403794613683ull, 7602643301893814034ull}},
{{3807014922957263019ull, 4970023500866155972ull,
  8955396270799021456ull, 4751652063683633771ull}},
{{4758768653696578773ull, 6212529376082694965ull,
  6582559320071388916ull, 5939565079604542214ull}},
{{10560146835548111370ull, 7765661720103368706ull,
  17451571186944011953ull, 7424456349505677767ull}},
{{11211777790644957511ull, 16382753621133075201ull,
  17824761019481089326ull, 4640285218441048604ull}},
{{179664183024033176ull, 11255069989561568194ull,
  3834207200641810042ull, 5800356523051310756ull}},
{{9447952265634817278ull, 4845465450097184434ull,
  4792759000802262553ull, 7250445653814138445ull}},
{{2586568295188745789ull, 10668517831048868447ull,
  10602634769430216095ull, 9063057067267673056ull}},
{{8534134212134047975ull, 13585352672046624635ull,
  6626646730893885059ull, 5664410667042295660ull}},
{{6055981746740172064ull, 12370004821630892890ull,
  8283308413617356324ull, 7080513333802869575ull}},
{{16793349220279990888ull, 15462506027038616112ull,
  5742449498594307501ull, 8850641667253586969ull}},
{{10495843262674994305ull, 11969909276112829022ull,
  15118245982689911948ull, 5531651042033491855ull}},
{{3896432041488967073ull, 14962386595141036278ull,
  14286121459935002031ull, 6914563802541864819ull}},
{{14093912088715984649ull, 14091297225498907443ull,
  13245965806491364635ull, 8643204753177331024ull}},
{{6502852046233796454ull, 6501217756723123200ull,
  8278728629057102897ull, 5402002970735831890ull}},
{{8128565057792245568ull, 12738208214331291904ull,
  1125038749466602813ull, 6752503713419789863ull}},
{{10160706322240306959ull, 2087702212631951168ull,
  15241356492115417229ull, 8440629641774737328ull}},
{{6350441451400191850ull, 3610656892108663432ull,
  9525847807572135768ull, 5275393526109210830ull}},
{{7938051814250239812ull, 4513321115135829290ull,
  2683937722610393902ull, 6594241907636513538ull}},
{{699192730958023957ull, 14865023430774562421ull,
  12578294190117768185ull, 8242802384545641922ull}},
{{2742838466062458925ull, 2373110616593019657ull,
  12473119887250993020ull, 5151751490341026201ull}},
{{8040234101005461560ull, 2966388270741274571ull,
  1756341803781577563ull, 6439689362926282752ull}},
{{5438606607829439046ull, 17543043393708756926ull,
  2195427254726971953ull, 8049611703657853440ull}},
{{17234187185175563116ull, 4046873093426891222ull,
  1372142034204357471ull, 5031007314786158400ull}},
{{12319361944614678087ull, 446905348356226124ull,
  1715177542755446839ull, 6288759143482698000ull}},
{{15399202430768347609ull, 14393689740727446367ull,
  2143971928444308548ull, 7860948929353372500ull}},
{{16542030546871299112ull, 18219428124809429787ull,
  10563354492132468650ull, 4913093080845857812ull}},
{{16065852165161735985ull, 13550913119157011426ull,
  13204193115165585813ull, 6141366351057322265ull}},
{{10858943169597394174ull, 3103583343664100571ull,
  2670183338674818555ull, 7676707938821652832ull}},
{{4480996471784677407ull, 18080640654285920521ull,
  1668864586671761596ull, 4797942461763533020ull}},
{{10212931608158234662ull, 4154056744147849035ull,
  2086080733339701996ull, 5997428077204416275ull}},
{{8154478491770405424ull, 5192570930184811294ull,
  16442658971956791207ull, 7496785096505520343ull}},
{{484863038929115486ull, 10162885859006588915ull,
  17194190885114076360ull, 4685490685315950214ull}},
{{14441136853943558069ull, 12703607323758236143ull,
  12269366569537819642ull, 5856863356644937768ull}},
{{13439735049002059683ull, 6656137117843019371ull,
  15336708211922274553ull, 7321079195806172210ull}},
{{12187982792825186699ull, 12931857415731162118ull,
  9947513228048067383ull, 9151348994757715263ull}},
{{3005803227088353783ull, 14999939912473058180ull,
  13134724795171123970ull, 5719593121723572039ull}},
{{3757254033860442229ull, 9526552853736546917ull,
  11806719975536517059ull, 7149491402154465049ull}},
{{9308253560752940690ull, 7296505048743295742ull, 923341914138482612ull,
  8936864252693081312ull}},
{{1205972457043200027ull, 13783687692319335647ull,
  577088696336551632ull, 5585540157933175820ull}},
{{15342523626586163746ull, 17229609615399169558ull,
  721360870420689540ull, 6981925197416469775ull}},
{{9954782496377928875ull, 3090267945539410332ull,
  14736759143308025638ull, 8727406496770587218ull}},
{{15445111097090981355ull, 15766475521244295169ull,
  13822160482994903927ull, 5454629060481617011ull}},
{{5471330816081562981ull, 15096408383127981058ull,
  12666014585316242005ull, 6818286325602021264ull}},
{{16062535556956729535ull, 5035452423627812610ull,
  15832518231645302507ull, 8522857907002526580ull}},
{{14650770741525343863ull, 841314755553688929ull, 671951857923538259ull,
  5326786191876579113ull}},
{{4478405371624516117ull, 14886701499724274874ull,
  5451625840831810727ull, 6658482739845723891ull}},
{{14821378751385420954ull, 13996690856227955688ull,
  2202846282612375505ull, 8323103424807154864ull}},
{{9263361719615888097ull, 1830402757501390449ull,
  1376778926632734691ull, 5201939640504471790ull}},
{{16190888167947248025ull, 16123061502158901773ull,
  10944345695145694171ull, 6502424550630589737ull}},
{{6403552154651896319ull, 15542140859271239313ull,
  18292118137359505618ull, 8128030688288237171ull}},
{{15531435142725904959ull, 14325524055471912474ull,
  9126730826