; Template Generators
;
; Copyright (C) 2021 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

(include-book "kestrel/std/system/pseudo-event-formp" :dir :system)
(include-book "kestrel/std/util/defmacro-plus" :dir :system)
(include-book "std/strings/decimal" :dir :system)
(include-book "xdoc/defxdoc-plus" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ template-generators
  :parents (event-macros)
  :short "Generators of template ACL2 events."
  :long
  (xdoc::topstring
   (xdoc::p
    "These utilities generate ``schematic'' ACL2 events,
     i.e. events built out of constrained functions
    that are meant to represent classes of events of a certain form.
    An example is a function definition")
   (xdoc::codeblock
    "(defun f (x)"
    "  (if (a x)"
    "      (b x)"
    "    (c x (f (d x)))))")
   (xdoc::p
    "where @('a'), @('b'), @('c'), and @('d') are constrained functions.
     This function definition is a template for unary recursive functions
     with one non-recursive branch followed by one recursive branch.")
   (xdoc::p
    "These utilities provide macros to generate templates of various forms.
     More macros may be added to the current initial selection.
     Existing macros may be renamed to maintain rational and uniform naming
     as more macros are added.")
   (xdoc::p
    "A use for the templates generated by these macros is
     in testing tools that manipulate ACL2 events,
     e.g. " (xdoc::seetopic "apt::apt" "APT transformations") ".
     Assuming that the tools
     operate somewhat ``uniformly'' (in a suitably sense) on ACL2 events,
     testing the tools on a template can be regarded as somewhat equivalent to
     testing the tools on all the instances of the template.
     This uniformity assumption may not be quite true,
     e.g. due to prover heuristics,
     but a template may still ``cover'' a large class of its instances."))
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define template-fn-params ((n natp))
  :returns (params symbol-listp)
  :short "Parameters of a template function."
  :long
  (xdoc::topstring-p
   "Return the list of symbols @('(x1 ... xn)'),
    where @('n') is the argument of this utility.
    These symbols are used as formal parameters of template functions.")
  (template-fn-params-aux 1 n)

  :prepwork
  ((define template-fn-params-aux ((i natp) (n natp))
     :returns (params symbol-listp)
     :parents nil ; avoid separate XDOC topic
     (let ((i (mbe :logic (nfix i) :exec i))
           (n (mbe :logic (nfix n) :exec n)))
       (if (> i n)
           nil
         (cons (add-suffix 'x (str::nat-to-dec-string i))
               (template-fn-params-aux (1+ i) n))))
     :measure (nfix (- (1+ (nfix n)) (nfix i))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define template-fn-nonrecursive ((fn symbolp)
                                  (body symbolp)
                                  (arity natp)
                                  (logic booleanp))
  :returns (fn-def pseudo-event-formp)
  :short "Template of a non-recursive function definition."
  :long
  (xdoc::topstring
   (xdoc::p
    "This template has the form")
   (xdoc::codeblock
    "(defun fn (x1 ... xn) ; n = arity"
    "  (declare (xargs :mode mode))"
    "  (body x1 ... xn))"))
  (b* ((params (template-fn-params arity)))
    `(progn
       (defstub ,body ,(repeat arity '*) => *)
       (defun ,fn ,params
         (declare (xargs :mode ,(if logic :logic :program)))
         (,body ,@params)))))

(defmacro+ gen-template-fn-nonrecursive (fn
                                         body
                                         &key
                                         (arity '1)
                                         (logic 't))
  :short "Generate a schematic non-recursive function definition."
  `(make-event (template-fn-nonrecursive ',fn ',body ',arity ',logic)))
