/*         _____ ___  _   _ _____ 
**    __ _|  ___/ _ \| \ | |_   _|
**   / _` | |_ | | | |  \| | | |  
**  | (_| |  _|| |_| | |\  | | |  
**   \__, |_|   \___/|_| \_| |_|  
**   |___/                        
**  
**  gFONT -- Create GIF image rendered with TeX-available Font
**
**  The gFONT program creates a GIF image for a given ASCII string
**  by the use of an arbitrary TeX-available font (Postscript or
**  METAFONT). The used font is converted from TeX's PK format to
**  gFONT's own GdF format (Gd Font) and rendered into the
**  resulting GIF image by the use of its own enhanced Gd library.
**  The result is intended to be included into HTML pages with an
**  IMG tag.
**
**  ======================================================================
**
**  Copyright (c) 1997 Ralf S. Engelschall, All rights reserved.
**
**  This program is free software; it may be redistributed and/or modified
**  only under the terms of either the Artistic License or the GNU General
**  Public License, which may be found in the ePerl source distribution.
**  Look at the files ARTISTIC and COPYING or run ``eperl -l'' to receive
**  a built-in copy of both license files.
**
**  This program is distributed in the hope that it will be useful, but
**  WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either the
**  Artistic License or the GNU General Public License for more details.
**
**  ======================================================================
**
**  gfont_main.c -- main program
*/

#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#else /* HAVE_STRINGS_H */
#include <string.h>
#endif /* HAVE_STRINGS_H */

#include "gfont_gd.h"
#include "gfont_gdfontl.h"
#include "gfont_getopt.h"

#include "config.h"
#include "config_sc.h"

int fVerbose = 0;
const char program_name[] = "gfont";

void Verbose(char *str, ...)
{
    va_list ap;
    char buf[1024];

    va_start(ap, str);
    if (fVerbose) {
        vsprintf(buf, str, ap);
        fprintf(stderr, "%s", buf);
    }
    va_end(ap);
    return;
}

int hex2int(const char c)
{
    if (c >= 'a' && c <= 'f')
        return (c - 'a' + 10);
    else if (c >= 'A' && c <= 'F')
        return (c - 'A' + 10);
    else
        return (c - '0');
}

void hextriple2gdcolor(const char *str, int *r, int *g, int *b)
{
    *r = (hex2int(str[0]) * 16 + hex2int(str[1]));
    *g = (hex2int(str[2]) * 16 + hex2int(str[3]));
    *b = (hex2int(str[4]) * 16 + hex2int(str[5]));
}

#define ALIGN_L 0
#define ALIGN_C 1
#define ALIGN_R 2
#define ALIGN_T 3
#define ALIGN_M 4
#define ALIGN_B 5

int parsesizespec(char *str, int *xrel, int *xsize, int *xalign,
                              int *yrel, int *ysize, int *yalign)
{
    char spec[128];
    char *xspec;
    char *yspec;
    char *cp;

    strcpy(spec, str);
    if ((cp = strchr(spec, 'x')) == (char *) NULL)
        return 1;
    *cp++ = '\0';
    xspec = spec;
    yspec = cp;

    *xrel = 0;
    if (*xspec == '+') {
        *xrel = 1;
        xspec++;
    }
    *xalign = ALIGN_L;
    if (*(xspec+(strlen(xspec)-1)) == 'L') {
        *xalign = ALIGN_L;
        *(xspec+(strlen(xspec)-1)) = '\0';
    }
    else if (*(xspec+(strlen(xspec)-1)) == 'C') {
        *xalign = ALIGN_C;
        *(xspec+(strlen(xspec)-1)) = '\0';
    }
    else if (*(xspec+(strlen(xspec)-1)) == 'R') {
        *xalign = ALIGN_R;
        *(xspec+(strlen(xspec)-1)) = '\0';
    }
    *xsize = atoi(xspec);
    if (*xsize < 0) {
        *xsize = 0;
        *xrel  = 1;
    }

    *yrel = 0;
    if (*yspec == '+') {
        *yrel = 1;
        yspec++;
    }
    *yalign = ALIGN_T;
    if (*(yspec+(strlen(yspec)-1)) == 'T') {
        *yalign = ALIGN_T;
        *(yspec+(strlen(yspec)-1)) = '\0';
    }
    else if (*(yspec+(strlen(yspec)-1)) == 'M') {
        *yalign = ALIGN_M;
        *(yspec+(strlen(yspec)-1)) = '\0';
    }
    else if (*(yspec+(strlen(yspec)-1)) == 'B') {
        *yalign = ALIGN_B;
        *(yspec+(strlen(yspec)-1)) = '\0';
    }
    *ysize = atoi(yspec);
    if (*ysize < 0) {
        *ysize = 0;
        *yrel  = 1;
    }

    return 0; 
}

int gdImageAntiAlias(gdImage *ip, 
                     int x0, int y0, int x1, int y1, 
                     int antialias, int bgcolor)
{
    int x, y;
    int p;
    int p_u, p_d, p_l, p_r;
    int r_u, r_d, r_l, r_r;
    int g_u, g_d, g_l, g_r;
    int b_u, b_d, b_l, b_r;
    int r_q, g_q, b_q;
    int q;

    if (!gdImageBoundsSafe(ip, x0, y0) || !gdImageBoundsSafe(ip, x1, y1))
        return 1;

    for (x = x0+1; x <= x1-1; x++) {
        for (y = y0+1; y <= y1-1; y++) {
            p = gdImageGetPixel(ip, x, y);
            if (p != bgcolor)
                continue;
            p_l = gdImageGetPixel(ip, x-1, y);
            r_l = ip->red[p_l]; g_l = ip->blue[p_l]; b_l = ip->blue[p_l];
            p_u = gdImageGetPixel(ip, x, y-1);
            r_u = ip->red[p_u]; g_u = ip->blue[p_u]; b_u = ip->blue[p_u];
            p_r = gdImageGetPixel(ip, x+1, y);
            r_r = ip->red[p_r]; g_r = ip->blue[p_r]; b_r = ip->blue[p_r];
            p_d = gdImageGetPixel(ip, x, y+1);
            r_d = ip->red[p_d]; g_d = ip->blue[p_d]; b_d = ip->blue[p_d];

            r_q = (unsigned char)((r_l + r_u + r_r + r_d) / 4);
            g_q = (unsigned char)((g_l + g_u + g_r + g_d) / 4);
            b_q = (unsigned char)((b_l + b_u + b_r + b_d) / 4);

            q = gdImageColorClosest(ip, r_q+64, g_q+64, b_q+64);
            gdImageSetPixel(ip, x, y, q);
        }
    }
    return 0;
}

extern char gFONT_Hello[];

void give_version(void)
{
    fprintf(stderr, "%s\n", gFONT_Hello);
    fprintf(stdout, "\n");
    fprintf(stdout, "Copyright (c) 1997 Ralf S. Engelschall, All rights reserved.\n");
    fprintf(stdout, "\n");
    fprintf(stdout, "This program is distributed in the hope that it will be useful,\n");
    fprintf(stdout, "but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
    fprintf(stdout, "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    fprintf(stdout, "the GNU General Public License for more details.\n");
    fprintf(stdout, "\n");
}


void give_usage(char *name)
{
    fprintf(stderr, "Usage: %s [options] string\n", name);
    fprintf(stderr, "\n");
    fprintf(stderr, "Rendering Parameters:\n");
    fprintf(stderr, "   -b, --background=RRGGBB  set background color (default: `ffffff')\n");
    fprintf(stderr, "   -f, --foreground=RRGGBB  set foreground color (default: `000000')\n");
    fprintf(stderr, "   -F, --fontname=NAME      set specific font (default `CMBoldface-V')\n");
    fprintf(stderr, "   -G, --gdf-path=DIR       add DIR to the path where GdF files are searched\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Output Parameters:\n");
    fprintf(stderr, "   -a, --anti-alias=N       apply anti-alias filter with N colors\n");
    fprintf(stderr, "   -c, --crop               crop image to real edges after rendering\n");
    fprintf(stderr, "   -r, --resize=SPEC        resize the image to a particular result size\n");
    fprintf(stderr, "   -i, --interlace          enable GIF Interlacing feature\n");
    fprintf(stderr, "   -o, --output-file=FILE   set the output file (default `out.gif')\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Giving Feedback:\n");
    fprintf(stderr, "   -v, --verbose            enable verbose processing\n");
    fprintf(stderr, "   -V, --version            display version string\n");
    fprintf(stderr, "   -h, --help               display this page\n");
    fprintf(stderr, "\n");
}

struct option options[] = {
    { "background",  1, NULL, 'b' },
    { "foreground",  1, NULL, 'f' },
    { "fontname",    1, NULL, 'F' },
    { "gdf-path",    1, NULL, 'G' },
    { "anti-alias",  0, NULL, 'a' },
    { "crop",        0, NULL, 'c' },
    { "resize",      0, NULL, 'r' },
    { "output-file", 1, NULL, 'o' },
    { "verbose",     0, NULL, 'v' },
    { "version",     0, NULL, 'V' },
    { "help",        0, NULL, 'h' }
};

int
main(int argc, char *argv[]) 
{
    FILE *fp;
    gdImage *ip;
    gdImage *ipn;
    int cbg, cfg;
    gdFontPtr gdfp;
    int x, y;
    int i, h, w;
    int nX, nY, nW, nH;
    char buffer[20];
    char ca[1024];
    char cmd[1024];
    int antialias = 0;
    int fCrop = 0;
    int fResize = 0;
    int fInterlace = 0;
    char *outputfile = "out.gif";
    int bg_r, bg_g, bg_b;
    int fg_r, fg_g, fg_b;
    char c;
    char *string;
    char *font = "CMBoldface-V";
    int r, g, b;
    int d_r, d_g, d_b;
    char *gdfpath = FONTPATH;
    int bgnondefault = 0;
    char *cp;
    int cont;
    int ss_xrel, ss_xsize, ss_xalign;
    int ss_yrel, ss_ysize, ss_yalign;

    hextriple2gdcolor("ffffff", &bg_r, &bg_g, &bg_b);
    hextriple2gdcolor("000000", &fg_r, &fg_g, &fg_b);

    /*  parse the option arguments */
    opterr = 0;
    while ((c = getopt_long(argc, argv, ":b:f:a:F:G:cir:o:vVh", options, NULL)) != -1) {
        if (optarg == NULL) 
            optarg = "(null)";
        switch (c) {
            case 'b':
                hextriple2gdcolor(optarg, &bg_r, &bg_g, &bg_b);
                bgnondefault = 1;
                break;
            case 'f':
                hextriple2gdcolor(optarg, &fg_r, &fg_g, &fg_b);
                break;
            case 'a':
                antialias = atoi(optarg);
                break;
            case 'F':
                font = strdup(optarg);
                break;
            case 'G':
                sprintf(ca, "%s:%s", optarg, gdfpath);
                gdfpath = strdup(ca);
                break;
            case 'c':
                fCrop = 1;
                break;
            case 'i':
                fInterlace = 1;
                break;
            case 'r':
                if (parsesizespec(optarg, &ss_xrel, &ss_xsize, &ss_xalign, &ss_yrel, &ss_ysize, &ss_yalign)) {
                    fprintf(stderr, "** gFONT::Error: bad size spec: `%s'\n", optarg);
                    exit(1);
                }
                fResize = 1;
                break;
            case 'o':
                outputfile = strdup(optarg);
                break;
            case 'v':
                fVerbose = 1;
                break;
            case 'V':
                give_version();
                exit(0);
                break;
            case 'h':
                give_usage(argv[0]);
                exit(0);
                break;
            default:
				fprintf(stderr, "gFONT: invalid option `%c'\n", optopt);
		        fprintf(stderr, "Try `%s --help' for more information.\n", argv[0]);
                exit(1);
        }
    }

    /* determine source filename and runtime mode */
    if (optind != argc-1) {
		fprintf(stderr, "gFONT: wrong number of arguments\n");
		fprintf(stderr, "Try `%s --help' for more information.\n", argv[0]);
        exit(1);
    }
    string = argv[optind];

    Verbose("Loading font: ");
    if ((cp = getenv("GDFPATH")) != NULL) {
        sprintf(ca, "%s:%s", gdfpath, cp);
        gdfpath = strdup(ca);
    }
    gdFontPathSet(gdfpath);
    gdfp = gdFontLoad(font);
    if (gdfp == (gdFontPtr) NULL) {
        sprintf(cmd, "%s/exec/gfont_mkgdf %s", libdir, font);
        fprintf(stderr, "Running `%s'\n", cmd);
        system(cmd);
        gdfp = gdFontLoad(font);
        if (gdfp == (gdFontPtr) NULL) {
            fprintf(stderr, "Still cannot load font '%s'\n", font);
            exit(1);
        }
    }
    Verbose("%s (%dx%d)\n", gdfp->name, gdfp->w, gdfp->h);

    Verbose("Calculating image size: ");
    w = gdImageStringWidth(gdfp, string) + 2;
    h = gdfp->h + 2;
    Verbose("%d x %d\n", w, h);

    Verbose("Allocating image: ");
    ip = gdImageCreate(w, h);
    cbg = gdImageColorAllocate(ip, bg_r, bg_g, bg_b);  
    cfg = gdImageColorAllocate(ip, fg_r, fg_g, fg_b);         
    if (fInterlace) 
        gdImageInterlace(ip, 1);
    if (!bgnondefault)
       gdImageColorTransparent(ip, cbg);
    Verbose("Done\n");

    Verbose("Drawing image: ");
    gdImageString(ip, gdfp, 1, 1, string, cfg);
    Verbose("Done\n");

    if (antialias > 0) {
        Verbose("Anti-Aliasing image: ");
        r = (bg_r > fg_r ? fg_r : bg_r);
        g = (bg_g > fg_g ? fg_g : bg_g);
        b = (bg_b > fg_b ? fg_b : bg_b);
        d_r = abs(bg_r - fg_r) / (antialias+1);
        d_g = abs(bg_g - fg_g) / (antialias+1);
        d_b = abs(bg_b - fg_b) / (antialias+1);
        for (i = 0; i < antialias; i++) {
            r += d_r;
            g += d_g;
            b += d_b;
            Verbose("r=%d,g=%d,b=%d\n", r+20, g+20, b+20);
            gdImageColorAllocate(ip, r, g, b);         
        }
        gdImageAntiAlias(ip, 1, 1, w-1, h-1, antialias, cbg);
        gdImageAntiAlias(ip, 1, 1, w-1, h-1, antialias, cbg);
        Verbose("Done\n");
    }

    if (fCrop) {
        Verbose("Cropping image to: ");
        nX = 0;
        nW = w;
        cont = 1;
        for (y = 0; y < gdImageSY(ip) && cont; y++) 
            for (x = 0; x < gdImageSX(ip) && cont; x++) 
                if (gdImageGetPixel(ip, x, y) != cbg) 
                    cont = 0;
        y = (y > 0 ? y-1 : y);
        nY = y;
        nH = gdImageSY(ip) - y;
        cont = 1;
        for (y = gdImageSY(ip) - 1; y >= nY && cont; y--) 
            for (x = 0; x < gdImageSX(ip) && cont; x++) 
                if (gdImageGetPixel(ip, x, y) != cbg) 
                    cont = 0;
        y = (y < gdImageSY(ip) ? y+1 : y);
        nH = nH - ((gdImageSY(ip) - 1) - y);
        Verbose("%d x %d, origin: (%d,%d)\n", nW, nH, nX, nY);

        Verbose("Allocating new image: ");
        ipn = gdImageCreate(nW, nH);
        cbg = gdImageColorAllocate(ipn, bg_r, bg_g, bg_b);  
        cfg = gdImageColorAllocate(ipn, fg_r, fg_g, fg_b);         
        if (fInterlace) 
            gdImageInterlace(ipn, 1);
        if (!bgnondefault)
            gdImageColorTransparent(ipn, cbg);
        Verbose("Done\n");

        Verbose("Copying image: ");
        gdImageCopy(ipn, ip, 0, 0, nX, nY, nW, nH);
        Verbose("Done\n");

        Verbose("Freeing old image: ");
        gdImageDestroy(ip);
        ip = ipn;
        Verbose("Done\n");
    }

    if (fResize) {
        Verbose("Adjusting image size: ");
        Verbose("x=%s%d y=%s%d, xalign=%d, yalign=%d\n", 
                (ss_xrel ? "+" : ""), ss_xsize, 
                (ss_yrel ? "+" : ""), ss_ysize,
                ss_xalign,
                ss_yalign
        );

        if (ss_xrel) 
            nW = gdImageSX(ip)+ss_xsize;
        else
            nW = ss_xsize;
        if (ss_yrel) 
            nH = gdImageSY(ip)+ss_ysize;
        else
            nH = ss_ysize;
        if (ss_xalign == ALIGN_L)
            nX = 0;
        else if (ss_xalign == ALIGN_C)
            nX = ((int)((nW/2))-1) - (int)(gdImageSX(ip)/2);
        else if (ss_xalign == ALIGN_R)
            nX = nW-gdImageSX(ip);
        if (ss_yalign == ALIGN_T)
            nY = 0;
        else if (ss_yalign == ALIGN_M)
            nY = ((int)((nH/2))-1) - (int)(gdImageSY(ip)/2);
        else if (ss_yalign == ALIGN_B)
            nY = nH-gdImageSY(ip);
        Verbose("Adjusting parameters: %d x %d, origin: (%d,%d)\n", nW, nH, nX, nY);

        if (nW < gdImageSX(ip) || nH < gdImageSY(ip)) {
            fprintf(stderr, "** gFONT:Error: new image size too small\n");
            exit(1);
        }

        Verbose("Allocating new image: ");
        ipn = gdImageCreate(nW, nH);
        cbg = gdImageColorAllocate(ipn, bg_r, bg_g, bg_b);  
        cfg = gdImageColorAllocate(ipn, fg_r, fg_g, fg_b);         
        if (fInterlace) 
            gdImageInterlace(ipn, 1);
        if (!bgnondefault)
            gdImageColorTransparent(ipn, cbg);
        Verbose("Done\n");

        Verbose("Copying image: ");
        gdImageCopy(ipn, ip, nX, nY, 0, 0, nW, nH);
        Verbose("Done\n");

        Verbose("Freeing old image: ");
        gdImageDestroy(ip);
        ip = ipn;
        Verbose("Done\n");
    }

    Verbose("Exporting image to GIF file: ");
    fp = fopen(outputfile, "w");
    if (fp == (FILE *) NULL) {
       fprintf(stderr, "Unable to open the file %s\n", outputfile);
       exit(1);
    }
    gdImageGif(ip, fp);
    fclose(fp);
    Verbose("Done\n");

    Verbose("Freeing image: ");
    gdImageDestroy(ip);
    Verbose("Done\n");

    exit(0);
}

/*EOF*/
