//
//  OSSP asgui - Accounting system graphical user interface
//  Copyright (c) 2002-2003 The OSSP Project (http://www.ossp.org/)
//  Copyright (c) 2002-2003 Cable & Wireless Deutschland (http://www.cw.com/de/)
//  Copyright (c) 2002-2003 Ralf S. Engelschall <rse@engelschall.com>
//  Copyright (c) 2002-2003 Michael Schloh von Bennewitz <michael@schloh.com>
//
//  This file is part of OSSP asgui, an accounting system graphical user
//  interface which can be found at http://www.ossp.org/pkg/tool/asgui/.
//
//  Permission to use, copy, modify, and distribute this software for
//  any purpose with or without fee is hereby granted, provided that
//  the above copyright notice and this permission notice appear in all
//  copies.
//
//  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
//  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
//  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
//  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
//  SUCH DAMAGE.
//
//  as_reportpanel.cpp: ISO C++ implementation
//

#include <qvariant.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qtextedit.h>
#include <qtoolbutton.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qtextstream.h>
#include <qfiledialog.h>
#include <qfile.h>
#include <qdir.h>

#include "as_reportpanel.h"
#include "as_table.h"
#include "as_pref.h"
#include "as_generic.h"
#include "as_const.h"

// Version information
#define _AS_VERSION_CPP_AS_HEADER_
#include "as_version.cpp"
#undef  _AS_VERSION_CPP_AS_HEADER_


//
// Constructs a Reportpanel as a child of 'pParent', with the 
// name 'kszName' and widget flags set to 'Flags'.
//
// The dialog will by default be modal, unless you set 'bModal' to
// true to construct a modal dialog.
//
Reportpanel::Reportpanel(TiTable *pTable, Preferences *pPreferences,
                         QWidget *pParent, const char *kszName,
                         bool bModal, WFlags Flags)
                         : QDialog(pParent, kszName, bModal, Flags)
{
    // Boilerplate code to initialize the panel
    if (!kszName)
        this->setName("Reportpanel");

    // Make Panel resizeable
    this->setSizeGripEnabled(true);
    this->setSizePolicy(QSizePolicy((QSizePolicy::SizeType)5,
        (QSizePolicy::SizeType)5, 0, 0, sizePolicy().hasHeightForWidth()));

    // Store matrix and prefs members
    m_pReptable = pTable;
//    m_pReptable->sortColumn(TITRAQ_IDXCOLDATE, true, true);
    m_pReprefs  = pPreferences;

    // Build panel using already constructed widgets and layouts
    m_pFormlay = new QVBoxLayout(this, 11, 6, "Formlayout");
    m_pToolay = new QHBoxLayout(0, 0, 6, "Toolbuttonlay"); 
    m_pPushlay = new QHBoxLayout(0, 0, 6, "Pushbuttonlay"); 

    // Groupbox and its text display
    m_pGroupbox = new QGroupBox(this, "Groupbox");
    m_pGroupbox->setFrameShape(QGroupBox::GroupBoxPanel);
    m_pGroupbox->setFrameShadow(QGroupBox::Sunken);
    m_pGroupbox->setColumnLayout(0, Qt::Vertical);
    m_pGroupbox->layout()->setSpacing(6);
    m_pGroupbox->layout()->setMargin(11);
    m_pGrouplay = new QVBoxLayout(m_pGroupbox->layout());
    m_pGrouplay->setAlignment(Qt::AlignTop);
    m_pBrowser = new QTextEdit(m_pGroupbox, "Reportbrowser");
    m_pBrowser->setTextFormat(PlainText);
    m_pBrowser->setReadOnly(true);
    m_pBrowser->setFont(QFont("Courier", 10));
    m_pGrouplay->addWidget(m_pBrowser);
    m_pGrouplay->addLayout(m_pToolay);

    // Tool button suite
    m_pWeeklybutt = new QToolButton(m_pGroupbox, "Weeklybutton");
    m_pMonthlybutt = new QToolButton(m_pGroupbox, "Monthlybutton");
    m_pBothbutt = new QToolButton(m_pGroupbox, "Bothbutton");
    m_pToolay->addWidget(m_pWeeklybutt);
    m_pToolay->addWidget(m_pMonthlybutt);
    m_pToolay->addWidget(m_pBothbutt);
    m_pFormlay->addWidget(m_pGroupbox);

    // Push button suite
    m_pSavebutt = new QPushButton(this, "Savebutton");
    m_pDismissbutt = new QPushButton(this, "Dismissbutton");
    m_pPushlay->addWidget(m_pSavebutt);
    m_pPushlay->addWidget(m_pDismissbutt);
    m_pFormlay->addLayout(m_pPushlay);

    // Connect signals to slots, accept() and reject() are Qt implicit
    connect(m_pWeeklybutt, SIGNAL(clicked(void)), SLOT(reportWeeks(void)));
    connect(m_pMonthlybutt, SIGNAL(clicked(void)), SLOT(reportMonths(void)));
    connect(m_pDismissbutt, SIGNAL(clicked(void)), SLOT(accept(void)));
    connect(m_pSavebutt, SIGNAL(clicked(void)), SLOT(saveReport(void)));
    this->textChange();
    this->resize(QSize(464, 332).expandedTo(minimumSizeHint()));
}

//
// Makes a new weekly report of so many weeks
//
void Reportpanel::reportWeeks(int nWeeks)
{
    m_pBrowser->clear();
    this->writeHeader();
    m_pBrowser->append(this->getWeek());
}

//
// Makes a new monthly report of so many months
//
void Reportpanel::reportMonths(int nMonths)
{
    m_pBrowser->clear();
    this->writeHeader();
    m_pBrowser->append(this->getMonth());
}

//
// Writes a report header to the display window
//
void Reportpanel::writeHeader(void)
{
    QString Header;
    Header = QString("Accounting System ");
    Header += QString(asgui_version.v_short);
    Header += trUtf8("\nLocal report, username '");
    Header += m_pReprefs->getString(TITRAQ_PREFUSER, TITRAQ_DEFUSER);
    Header += QString("'\n");
    Header += trUtf8("Date ");
    Header += QDate::currentDate().toString(Qt::ISODate);
    Header += trUtf8(", Time ");
    Header += QTime::currentTime().toString(Qt::ISODate);
    Header += QString("\n\n");
    Header += QString("Date       Hours Task\n");
    Header += QString("---------- ----- ----\n");
    m_pBrowser->setText(Header);
}

//
// Writes one week of data to the display window
//
QString Reportpanel::getWeek(QDate Dayref)
{
//    m_pBrowser->append(trUtf8("No events logged during week\n"));
    QString Data, Tempstring; // Output string
    int nRows      = m_pReptable->numRows();
    int nIter      = nRows - 1;
    int nFirstrow  = 0;
    int nLastrow   = 0;
    QDate Firstday = Dayref.addDays(Dayref.dayOfWeek() * -1 + 1);   // Do negate
    QDate Lastday  = Firstday.addDays(7);                           // Add week

    // Find data starting the week in question
    while (QDate::fromString(m_pReptable->text(nIter, TITRAQ_IDXDATE), Qt::ISODate) >= Firstday)
        nIter--;
    nFirstrow = ++nIter; // The first row to operate on

    // Find data ending the week in question
    while (nIter < nRows
        && QDate::fromString(m_pReptable->text(nIter, TITRAQ_IDXDATE), Qt::ISODate)
        <  Lastday)
        nIter++;
    nLastrow = nIter - 1; // The last row to operate on

    // Build the long week data string
    for (nIter = nFirstrow; nIter <= nLastrow; nIter++) {
        Tempstring = m_pReptable->text(nIter, TITRAQ_IDXDATE);
        if (!Tempstring.isEmpty())
            Data += Tempstring;
        Tempstring = m_pReptable->text(nIter, TITRAQ_IDXAMOUNT);
        if (!Tempstring.isEmpty())
            Data += QString(TITRAQ_SEPARATORTOK) + Tempstring;
        Tempstring = m_pReptable->text(nIter, TITRAQ_IDXTASK);
        if (!Tempstring.isEmpty())
            Data += QString(TITRAQ_SEPARATORTOK) + Tempstring;
        Data += trUtf8("\n"); // Finish off line
    }

    return Data;
}

//
// Writes one month of data to the display window
//
QString Reportpanel::getMonth(QDate Dayref)
{
    QString Data;
    
    for (int nIter = 0; nIter < 4; nIter++)
        Data += getWeek(Dayref.addDays(nIter * -1));

    return Data;
}

//
// Saves the currently displayed local report
//
void Reportpanel::saveReport(void)
{
    int nResult = 0;        // For checking user's answer
    QFile Report;           // The resulting report file
    QString Filestring;     // The user chosen file name to save
    QString Openas = m_pReprefs->getString(TITRAQ_PREFHOME, TITRAQ_DEFHOME);

    // And then get the name of the selected file to save to
    Filestring = QFileDialog::getSaveFileName(Openas, trUtf8("Text files (*.txt);;All Files (*)"), this, trUtf8("ChooserDialog"), trUtf8("Choose a file to save"), NULL, false);
    if (!Filestring.isEmpty()) {
        if (QFile::exists(Filestring)) {
            nResult = QMessageBox::warning(this, QString(TITRAQ_APPTITLE)
                    + ' ' + asgui_version.v_short, trUtf8(TITRAQ_OVERWRITE),
                      trUtf8("&Yes"), trUtf8("&No"), NULL, 1, 1);
            if (nResult = QMessageBox::Ok) {                // Overwrite a file
                Report.setName(Filestring);                 // User wished name
                Report.open(IO_WriteOnly | IO_Truncate);    // Open report file
                QTextStream Outstream(&Report);             // Convert to stream
                Outstream << m_pBrowser->text();            // Write the data
                Report.close();                             // Finish by closing
            }
        }
        else { // User gave a file name, and there was no preexisting file there
            Report.setName(Filestring);                 // User wished name
            Report.open(IO_WriteOnly | IO_Truncate);    // Open report file
            QTextStream Outstream(&Report);             // Convert to stream
            Outstream << m_pBrowser->text();            // Write the data
            Report.close();                             // Finish by closing
qDebug(m_pBrowser->text().ascii());
        }
    }
    // Otherwise, user did not select a valid file and push okay button
}

//
// Sets the strings of the subwidgets using the current language
//
void Reportpanel::textChange(void)
{
    this->setCaption(tr("AS local report", "Local report using weekly data or monthly data or both."));
    m_pGroupbox->setTitle(tr("Local report", "Comment for Groupbox"));

    // Top level push buttons associated with accept and save slots
    m_pDismissbutt->setText(tr("Dismiss", "Comment for Dismissbutton"));
    QToolTip::add(m_pDismissbutt, tr("Closes the report panel", "Comment for tooltip Dismissbutton"));
    QWhatsThis::add(m_pDismissbutt, tr("The dismiss button dismisses the report panel", "Comment for whatsThis Dsimissbutton"));
    m_pSavebutt->setText(tr("Save", "Comment for Savebutton"));
    QToolTip::add(m_pSavebutt, tr("Saves the report panel text", "Comment for tooltip Savebutton"));
    QWhatsThis::add(m_pSavebutt, tr("The save button saves the report panel text to a file", "Comment for whatsThis Savebutton"));

    // Inner tool buttons for new local report generation
    m_pWeeklybutt->setText(tr("Weekly", "Comment for Weeklybutt"));
    QToolTip::add(m_pWeeklybutt, tr("Generates a weekly report", "Comment for tooltip Weeklybutt"));
    QWhatsThis::add(m_pWeeklybutt, tr("The weekly button generates a new weekly report", "Comment for whatsThis Weeklybutt"));
    m_pMonthlybutt->setText(tr("Monthly", "Comment for Monthlybutt"));
    QToolTip::add(m_pMonthlybutt, tr("Generates a monthly report", "Comment for tooltip Monthlybutt"));
    QWhatsThis::add(m_pMonthlybutt, tr("The monthly button generates a new monthly report", "Comment for whatsThis Monthlybutt"));
    m_pBothbutt->setText(tr("Both", "Comment for Bothbutt"));
    QToolTip::add(m_pBothbutt, tr("Generates a both report", "Comment for tooltip Bothbutt"));
    QWhatsThis::add(m_pBothbutt, tr("The both button generates a new report with both weekly and monthly items mixed in.", "Comment for whatsThis Bothbutt"));
}
