//
//  OSSP asgui - Accounting system graphical user interface
//  Copyright (c) 2002-2003 The OSSP Project (http://www.ossp.org/)
//  Copyright (c) 2002-2003 Cable & Wireless Deutschland (http://www.cw.com/de/)
//  Copyright (c) 2002-2003 Ralf S. Engelschall <rse@engelschall.com>
//  Copyright (c) 2002-2003 Michael Schloh von Bennewitz <michael@schloh.com>
//
//  This file is part of OSSP asgui, an accounting system graphical user
//  interface which can be found at http://www.ossp.org/pkg/tool/asgui/.
//
//  Permission to use, copy, modify, and distribute this software for
//  any purpose with or without fee is hereby granted, provided that
//  the above copyright notice and this permission notice appear in all
//  copies.
//
//  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
//  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
//  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
//  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
//  SUCH DAMAGE.
//
//  tidatops.cpp: ISO C++ implementation
//

#include <qregexp.h>        // Portable regular expressions

#include "as_gui.h"         // Main classes
#include "as_except.h"      // Exception classes
#include "as_tableitem.h"   // For class RtTableItem


//
// Convenience method to load accounting data from a file
//
void Titraqform::loadData(QFile &Fileobj)
{
    if (Fileobj.isOpen()) {             // Check state of file
        Fileobj.flush();                // Begin processing file cleanly
        QTextStream Asentry(&Fileobj);  // Convert data to stream
        this->loadData(Asentry);        // Pass off to do the real work
    }
    else {
        if (!Fileobj.open(IO_ReadOnly)) // Try to open file
            throw Genexcept("Could not read open accounting file.");
        QTextStream Asentry(&Fileobj);  // Convert data to stream
        this->loadData(Asentry);        // Pass off to do the real work
        Fileobj.close();                // Finish fileop by closing
    }
}

//
// Load accounting data from a stream
//
void Titraqform::loadData(QTextStream &Tstream)
{
    bool bValid = true; // Used to warn on invalid accounting data
    int nIter = 0;      // Iterator used in loop and also as a count
    QString Line;       // Used for linewise editing and whitespace eating
    QString Bitbucket;  // Used for null device until we find a better way

    // Strip out extra line feeds at stream start
    while (Line.isEmpty() && !Tstream.atEnd())
        Line = Tstream.readLine();

    // Optimize viewing by repainting cells only once after processing
    m_pMaintable->setUpdatesEnabled(false);

    // Set the table text by linewise reading from the input stream
    // and parsing date, time, account, and other columns out of it
    while (!Line.isEmpty()) {
        QString Date, Start, Finish, Account, Amount, Remark; // Valid fields
        QTextStream Asline(&Line, IO_ReadOnly); // Convert a single line now

        if (nIter % g_knBlocks == 0) // Add blocks of rows to optimize loading
            m_pMaintable->setNumRows(m_pMaintable->numRows() + g_knBlocks);

        Asline.skipWhiteSpace();    // Remove whitespaces
        Asline >> Date;             // Copy the date field
        if (Date != NULL)
            m_pMaintable->setText(nIter, TITRAQ_IDXDATE, Date);
        else
            bValid = false;

        Asline >> Bitbucket;        // Ignore user field

        Asline.skipWhiteSpace();    // Remove whitespaces
        Asline >> Start;            // Copy the date field
        if (Start != NULL)
            m_pMaintable->setText(nIter, TITRAQ_IDXSTART, Start);
        else
            bValid = false;

        Asline.skipWhiteSpace();    // Remove whitespaces
        Asline >> Finish;           // Copy the date field
        if (Start != NULL)
            m_pMaintable->setText(nIter, TITRAQ_IDXFINISH, Finish);
        else
            bValid = false;

        Asline.skipWhiteSpace();    // Remove whitespaces
        Asline >> Account;          // Copy the account field
        if (Account != NULL) {
            QTableItem *pOld = m_pMaintable->item(nIter, TITRAQ_IDXTASK);
            delete(pOld);   // Get rid of the old before going with the new
            m_pMaintable->setItem(nIter, TITRAQ_IDXTASK, new RtTableItem(m_pMaintable, QTableItem::WhenCurrent, Account));
        }
        else
            bValid = false;

        Asline.skipWhiteSpace();    // Remove whitespaces
        Asline >> Amount;           // Copy the amount field
        if (Amount != NULL)
            m_pMaintable->setText(nIter, TITRAQ_IDXAMOUNT, Amount);
        else
            bValid = false;

        Asline.skipWhiteSpace();    // Remove whitespaces
        Remark = Asline.read();     // Copy the remark field
        if (Remark != NULL)
            m_pMaintable->setText(nIter, TITRAQ_IDXREMARK, Remark);

        nIter++;    // The big increment
        Line = "";  // Clear line for next round

        while (Line.isEmpty() && !Tstream.atEnd())  // Strip and get
            Line = Tstream.readLine();              // the new line
    }

    m_pMaintable->setUpdatesEnabled(true);      // Update and repaint
    m_pMaintable->setNumRows(nIter);            // No excess rows
    m_pMaintable->setCurrentCell(nIter - 1, 0); // Move focus to last row

    if (!bValid)
        throw Genexcept("Warning, invalid accounting data.");
}

//
// Convenience method to save accounting data to a file
//
void Titraqform::saveData(QFile &Fileobj)
{
    if (Fileobj.isOpen()) {             // Check state of file
        Fileobj.flush();                // Begin processing file cleanly
        QTextStream Asentry(&Fileobj);  // Convert data to stream
        this->saveData(Asentry);        // Pass off to do the real work
    }
    else {
        if (!Fileobj.open(IO_WriteOnly)) // Try to open file
            throw Genexcept("Could not write open accounting file.");
        QTextStream Asentry(&Fileobj);  // Convert data to stream
        this->saveData(Asentry);        // Pass off to do the real work
        Fileobj.close();                // Finish fileop by closing
    }
}

//
// Save accounting data to a stream
//
void Titraqform::saveData(QTextStream &Tstream)
{
    const int nRows = m_pMaintable->numRows();      // Max rows used in loop
    QString Tempfield;                              // Current field string
    bool bValid = true;                             // Warn on invalid data

    // Linewise save from the main table date, time, account, and others
    for (int nIter = 0; nIter < nRows; nIter++) {
        Tempfield = m_pMaintable->text(nIter, 0);   // Load date field text
        if (Tempfield != NULL)
            Tstream << Tempfield;                   // Save date field text
        else
            bValid = false;

        // Save user field text
        Tstream << trUtf8(" ") << m_pPrefs->getString(TITRAQ_PREFUSER, TITRAQ_DEFUSER);

        Tempfield = m_pMaintable->text(nIter, 1);   // Load start field text
        if (Tempfield != NULL)
            Tstream << trUtf8(" ") << Tempfield;    // Save start field text
//        else
//            bValid = false;

        Tempfield = m_pMaintable->text(nIter, 2);   // Load end field text
        if (Tempfield != NULL)
            Tstream << trUtf8(" ") << Tempfield;    // Save end field text
//        else
//            bValid = false;

        Tempfield = m_pMaintable->text(nIter, 4);   // Load amount field text
        if (Tempfield != NULL)
            Tstream << trUtf8(" ") << Tempfield;    // Save amount field text
        else
            bValid = false;

        Tempfield = m_pMaintable->text(nIter, 3);   // Load acct field text
        if (Tempfield != NULL)
            Tstream << trUtf8(" ") << Tempfield;    // Save acct field text
        else
            bValid = false;

        Tempfield = m_pMaintable->text(nIter, 5);   // Load remark field text
        if (Tempfield != NULL)
            Tstream << trUtf8(" ") << Tempfield;    // Save remark field text
        else
            bValid = false;

        Tstream << endl;                            // Append a newline
    }
}
