/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.database;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.traccar.Context;
import org.traccar.database.DataManager;
import org.traccar.database.GroupTree;
import org.traccar.database.SimpleObjectManager;
import org.traccar.database.UsersManager;
import org.traccar.model.Attribute;
import org.traccar.model.Calendar;
import org.traccar.model.Command;
import org.traccar.model.Device;
import org.traccar.model.Driver;
import org.traccar.model.Geofence;
import org.traccar.model.Group;
import org.traccar.model.Maintenance;
import org.traccar.model.ManagedUser;
import org.traccar.model.Notification;
import org.traccar.model.Permission;
import org.traccar.model.Server;
import org.traccar.model.User;

public class PermissionsManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(PermissionsManager.class);
    private final DataManager dataManager;
    private final UsersManager usersManager;
    private volatile Server server;
    private final Map<Long, Set<Long>> groupPermissions = new HashMap<Long, Set<Long>>();
    private final Map<Long, Set<Long>> devicePermissions = new HashMap<Long, Set<Long>>();
    private final Map<Long, Set<Long>> deviceUsers = new HashMap<Long, Set<Long>>();
    private final Map<Long, Set<Long>> groupDevices = new HashMap<Long, Set<Long>>();

    public PermissionsManager(DataManager dataManager, UsersManager usersManager) {
        this.dataManager = dataManager;
        this.usersManager = usersManager;
        this.refreshServer();
        this.refreshDeviceAndGroupPermissions();
    }

    public User getUser(long userId) {
        return (User)this.usersManager.getById(userId);
    }

    public Set<Long> getGroupPermissions(long userId) {
        if (!this.groupPermissions.containsKey(userId)) {
            this.groupPermissions.put(userId, new HashSet());
        }
        return this.groupPermissions.get(userId);
    }

    public Set<Long> getDevicePermissions(long userId) {
        if (!this.devicePermissions.containsKey(userId)) {
            this.devicePermissions.put(userId, new HashSet());
        }
        return this.devicePermissions.get(userId);
    }

    private Set<Long> getAllDeviceUsers(long deviceId) {
        if (!this.deviceUsers.containsKey(deviceId)) {
            this.deviceUsers.put(deviceId, new HashSet());
        }
        return this.deviceUsers.get(deviceId);
    }

    public Set<Long> getDeviceUsers(long deviceId) {
        Device device = Context.getIdentityManager().getById(deviceId);
        if (device != null && !device.getDisabled()) {
            return this.getAllDeviceUsers(deviceId);
        }
        HashSet<Long> result = new HashSet<Long>();
        for (long userId : this.getAllDeviceUsers(deviceId)) {
            if (!this.getUserAdmin(userId)) continue;
            result.add(userId);
        }
        return result;
    }

    public Set<Long> getGroupDevices(long groupId) {
        if (!this.groupDevices.containsKey(groupId)) {
            this.groupDevices.put(groupId, new HashSet());
        }
        return this.groupDevices.get(groupId);
    }

    public void refreshServer() {
        try {
            this.server = this.dataManager.getServer();
        }
        catch (SQLException error) {
            LOGGER.warn("Refresh server config error", (Throwable)error);
        }
    }

    public final void refreshDeviceAndGroupPermissions() {
        this.groupPermissions.clear();
        this.devicePermissions.clear();
        try {
            GroupTree groupTree = new GroupTree(Context.getGroupsManager().getItems(Context.getGroupsManager().getAllItems()), Context.getDeviceManager().getAllDevices());
            for (Permission groupPermission : this.dataManager.getPermissions(User.class, Group.class)) {
                Set<Long> userGroupPermissions = this.getGroupPermissions(groupPermission.getOwnerId());
                Set<Long> userDevicePermissions = this.getDevicePermissions(groupPermission.getOwnerId());
                userGroupPermissions.add(groupPermission.getPropertyId());
                for (Group group : groupTree.getGroups(groupPermission.getPropertyId())) {
                    userGroupPermissions.add(group.getId());
                }
                for (Device device : groupTree.getDevices(groupPermission.getPropertyId())) {
                    userDevicePermissions.add(device.getId());
                }
            }
            for (Permission devicePermission : this.dataManager.getPermissions(User.class, Device.class)) {
                this.getDevicePermissions(devicePermission.getOwnerId()).add(devicePermission.getPropertyId());
            }
            this.groupDevices.clear();
            Iterator<Object> iterator = Context.getGroupsManager().getAllItems().iterator();
            while (iterator.hasNext()) {
                long groupId = (Long)iterator.next();
                for (Device device : groupTree.getDevices(groupId)) {
                    this.getGroupDevices(groupId).add(device.getId());
                }
            }
        }
        catch (ClassNotFoundException | SQLException error) {
            LOGGER.warn("Refresh device permissions error", (Throwable)error);
        }
        this.deviceUsers.clear();
        for (Map.Entry<Long, Set<Long>> entry : this.devicePermissions.entrySet()) {
            for (long deviceId : entry.getValue()) {
                this.getAllDeviceUsers(deviceId).add(entry.getKey());
            }
        }
    }

    public boolean getUserAdmin(long userId) {
        User user = this.getUser(userId);
        return user != null && user.getAdministrator();
    }

    public void checkAdmin(long userId) throws SecurityException {
        if (!this.getUserAdmin(userId)) {
            throw new SecurityException("Admin access required");
        }
    }

    public boolean getUserManager(long userId) {
        User user = this.getUser(userId);
        return user != null && user.getUserLimit() != 0;
    }

    public void checkManager(long userId) throws SecurityException {
        if (!this.getUserManager(userId)) {
            throw new SecurityException("Manager access required");
        }
    }

    public void checkManager(long userId, long managedUserId) throws SecurityException {
        this.checkManager(userId);
        if (!this.usersManager.getUserItems(userId).contains(managedUserId)) {
            throw new SecurityException("User access denied");
        }
    }

    public void checkUserLimit(long userId) throws SecurityException {
        int userLimit = this.getUser(userId).getUserLimit();
        if (userLimit != -1 && this.usersManager.getUserItems(userId).size() >= userLimit) {
            throw new SecurityException("Manager user limit reached");
        }
    }

    public void checkDeviceLimit(long userId) throws SecurityException {
        int deviceLimit = this.getUser(userId).getDeviceLimit();
        if (deviceLimit != -1) {
            int deviceCount = 0;
            deviceCount = this.getUserManager(userId) ? Context.getDeviceManager().getAllManagedItems(userId).size() : Context.getDeviceManager().getAllUserItems(userId).size();
            if (deviceCount >= deviceLimit) {
                throw new SecurityException("User device limit reached");
            }
        }
    }

    public boolean getUserReadonly(long userId) {
        User user = this.getUser(userId);
        return user != null && user.getReadonly();
    }

    public boolean getUserDeviceReadonly(long userId) {
        User user = this.getUser(userId);
        return user != null && user.getDeviceReadonly();
    }

    public boolean getUserLimitCommands(long userId) {
        User user = this.getUser(userId);
        return user != null && user.getLimitCommands();
    }

    public void checkReadonly(long userId) throws SecurityException {
        if (!this.getUserAdmin(userId) && (this.server.getReadonly() || this.getUserReadonly(userId))) {
            throw new SecurityException("Account is readonly");
        }
    }

    public void checkDeviceReadonly(long userId) throws SecurityException {
        if (!this.getUserAdmin(userId) && (this.server.getDeviceReadonly() || this.getUserDeviceReadonly(userId))) {
            throw new SecurityException("Account is device readonly");
        }
    }

    public void checkLimitCommands(long userId) throws SecurityException {
        if (!this.getUserAdmin(userId) && (this.server.getLimitCommands() || this.getUserLimitCommands(userId))) {
            throw new SecurityException("Account has limit sending commands");
        }
    }

    public void checkUserDeviceCommand(long userId, long deviceId, long commandId) throws SecurityException {
        if (!this.getUserAdmin(userId) && Context.getCommandsManager().checkDeviceCommand(deviceId, commandId)) {
            throw new SecurityException("Command can not be sent to this device");
        }
    }

    public void checkUserEnabled(long userId) throws SecurityException {
        User user = this.getUser(userId);
        if (user == null) {
            throw new SecurityException("Unknown account");
        }
        if (user.getDisabled()) {
            throw new SecurityException("Account is disabled");
        }
        if (user.getExpirationTime() != null && System.currentTimeMillis() > user.getExpirationTime().getTime()) {
            throw new SecurityException("Account has expired");
        }
    }

    public void checkUserUpdate(long userId, User before, User after) throws SecurityException {
        User user;
        if (before.getAdministrator() != after.getAdministrator() || before.getDeviceLimit() != after.getDeviceLimit() || before.getUserLimit() != after.getUserLimit()) {
            this.checkAdmin(userId);
        }
        if ((user = this.getUser(userId)) != null && user.getExpirationTime() != null && (after.getExpirationTime() == null || user.getExpirationTime().compareTo(after.getExpirationTime()) < 0)) {
            this.checkAdmin(userId);
        }
        if (before.getReadonly() != after.getReadonly() || before.getDeviceReadonly() != after.getDeviceReadonly() || before.getDisabled() != after.getDisabled() || before.getLimitCommands() != after.getLimitCommands()) {
            if (userId == after.getId()) {
                this.checkAdmin(userId);
            }
            if (!this.getUserAdmin(userId)) {
                this.checkManager(userId);
            }
        }
    }

    public void checkUser(long userId, long managedUserId) throws SecurityException {
        if (userId != managedUserId && !this.getUserAdmin(userId)) {
            this.checkManager(userId, managedUserId);
        }
    }

    public void checkGroup(long userId, long groupId) throws SecurityException {
        if (!this.getGroupPermissions(userId).contains(groupId) && !this.getUserAdmin(userId)) {
            this.checkManager(userId);
            for (long managedUserId : this.usersManager.getUserItems(userId)) {
                if (!this.getGroupPermissions(managedUserId).contains(groupId)) continue;
                return;
            }
            throw new SecurityException("Group access denied");
        }
    }

    public void checkDevice(long userId, long deviceId) throws SecurityException {
        if (!Context.getDeviceManager().getUserItems(userId).contains(deviceId) && !this.getUserAdmin(userId)) {
            this.checkManager(userId);
            for (long managedUserId : this.usersManager.getUserItems(userId)) {
                if (!Context.getDeviceManager().getUserItems(managedUserId).contains(deviceId)) continue;
                return;
            }
            throw new SecurityException("Device access denied");
        }
    }

    public void checkRegistration(long userId) {
        if (!this.server.getRegistration() && !this.getUserAdmin(userId)) {
            throw new SecurityException("Registration disabled");
        }
    }

    public void checkPermission(Class<?> object, long userId, long objectId) throws SecurityException {
        SimpleObjectManager manager = null;
        if (object.equals(Device.class)) {
            this.checkDevice(userId, objectId);
        } else if (object.equals(Group.class)) {
            this.checkGroup(userId, objectId);
        } else if (object.equals(User.class) || object.equals(ManagedUser.class)) {
            this.checkUser(userId, objectId);
        } else if (object.equals(Geofence.class)) {
            manager = Context.getGeofenceManager();
        } else if (object.equals(Attribute.class)) {
            manager = Context.getAttributesManager();
        } else if (object.equals(Driver.class)) {
            manager = Context.getDriversManager();
        } else if (object.equals(Calendar.class)) {
            manager = Context.getCalendarManager();
        } else if (object.equals(Command.class)) {
            manager = Context.getCommandsManager();
        } else if (object.equals(Maintenance.class)) {
            manager = Context.getMaintenancesManager();
        } else if (object.equals(Notification.class)) {
            manager = Context.getNotificationManager();
        } else {
            throw new IllegalArgumentException("Unknown object type");
        }
        if (manager != null && !manager.checkItemPermission(userId, objectId) && !this.getUserAdmin(userId)) {
            this.checkManager(userId);
            for (long managedUserId : this.usersManager.getManagedItems(userId)) {
                if (!manager.checkItemPermission(managedUserId, objectId)) continue;
                return;
            }
            throw new SecurityException("Type " + object + " access denied");
        }
    }

    public void refreshAllUsersPermissions() {
        if (Context.getGeofenceManager() != null) {
            Context.getGeofenceManager().refreshUserItems();
        }
        Context.getCalendarManager().refreshUserItems();
        Context.getDriversManager().refreshUserItems();
        Context.getAttributesManager().refreshUserItems();
        Context.getCommandsManager().refreshUserItems();
        Context.getMaintenancesManager().refreshUserItems();
        if (Context.getNotificationManager() != null) {
            Context.getNotificationManager().refreshUserItems();
        }
    }

    public void refreshAllExtendedPermissions() {
        if (Context.getGeofenceManager() != null) {
            Context.getGeofenceManager().refreshExtendedPermissions();
        }
        Context.getDriversManager().refreshExtendedPermissions();
        Context.getAttributesManager().refreshExtendedPermissions();
        Context.getCommandsManager().refreshExtendedPermissions();
        Context.getMaintenancesManager().refreshExtendedPermissions();
    }

    public void refreshPermissions(Permission permission) {
        if (permission.getOwnerClass().equals(User.class)) {
            if (permission.getPropertyClass().equals(Device.class) || permission.getPropertyClass().equals(Group.class)) {
                this.refreshDeviceAndGroupPermissions();
                this.refreshAllExtendedPermissions();
            } else if (permission.getPropertyClass().equals(ManagedUser.class)) {
                this.usersManager.refreshUserItems();
            } else if (permission.getPropertyClass().equals(Geofence.class) && Context.getGeofenceManager() != null) {
                Context.getGeofenceManager().refreshUserItems();
            } else if (permission.getPropertyClass().equals(Driver.class)) {
                Context.getDriversManager().refreshUserItems();
            } else if (permission.getPropertyClass().equals(Attribute.class)) {
                Context.getAttributesManager().refreshUserItems();
            } else if (permission.getPropertyClass().equals(Calendar.class)) {
                Context.getCalendarManager().refreshUserItems();
            } else if (permission.getPropertyClass().equals(Command.class)) {
                Context.getCommandsManager().refreshUserItems();
            } else if (permission.getPropertyClass().equals(Maintenance.class)) {
                Context.getMaintenancesManager().refreshUserItems();
            } else if (permission.getPropertyClass().equals(Notification.class) && Context.getNotificationManager() != null) {
                Context.getNotificationManager().refreshUserItems();
            }
        } else if (permission.getOwnerClass().equals(Device.class) || permission.getOwnerClass().equals(Group.class)) {
            if (permission.getPropertyClass().equals(Geofence.class) && Context.getGeofenceManager() != null) {
                Context.getGeofenceManager().refreshExtendedPermissions();
            } else if (permission.getPropertyClass().equals(Driver.class)) {
                Context.getDriversManager().refreshExtendedPermissions();
            } else if (permission.getPropertyClass().equals(Attribute.class)) {
                Context.getAttributesManager().refreshExtendedPermissions();
            } else if (permission.getPropertyClass().equals(Command.class)) {
                Context.getCommandsManager().refreshExtendedPermissions();
            } else if (permission.getPropertyClass().equals(Maintenance.class)) {
                Context.getMaintenancesManager().refreshExtendedPermissions();
            } else if (permission.getPropertyClass().equals(Notification.class) && Context.getNotificationManager() != null) {
                Context.getNotificationManager().refreshExtendedPermissions();
            }
        }
    }

    public Server getServer() {
        return this.server;
    }

    public void updateServer(Server server) throws SQLException {
        this.dataManager.updateObject(server);
        this.server = server;
    }

    public User login(String email, String password) throws SQLException {
        User user = this.dataManager.login(email, password);
        if (user != null) {
            this.checkUserEnabled(user.getId());
            return this.getUser(user.getId());
        }
        return null;
    }

    public Object lookupAttribute(long userId, String key, Object defaultValue) {
        Object serverPreference = this.server.getAttributes().get(key);
        Object userPreference = this.getUser(userId).getAttributes().get(key);
        Object preference = this.server.getForceSettings() ? (serverPreference != null ? serverPreference : userPreference) : (userPreference != null ? userPreference : serverPreference);
        return preference != null ? preference : defaultValue;
    }
}

