Ext.namespace('Zarafa.plugins.spreed');

/**
 * @class Zarafa.plugins.spreed.ABOUT
 * @extends String
 *
 * The copyright string holding the copyright notice for the Zarafa spreed Plugin.
 */
Zarafa.plugins.spreed.ABOUT = ""
	+ "<p>Copyright (C) 2005 - 2015  Zarafa B.V. &lt;info@zarafa.com&gt; and its licensors</p>"

	+ "<p>This program is free software: you can redistribute it and/or modify "
	+ "it under the terms of the GNU Affero General Public License as "
	+ "published by the Free Software Foundation, either version 3 of the "
	+ "License, or (at your option) any later version.</p>"

	+ "<p>This program is distributed in the hope that it will be useful, "
	+ "but WITHOUT ANY WARRANTY; without even the implied warranty of "
	+ "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
	+ "GNU Affero General Public License for more details.</p>"

	+ "<p>You should have received a copy of the GNU Affero General Public License "
	+ "along with this program.  If not, see <a href=\"http://www.gnu.org/licenses/\" target=\"_blank\">http://www.gnu.org/licenses/</a>.</p>"

	+ "<hr />"

	+ "<p>The spreed plugin contains the following third-party components:</p>"

	+ "<h1>timezones.inc</h1>"

	+ "<p>The timezones.inc file distributed with this plugin is Copyright (C) The PHP Group, and covered by the <a href=\"http://www.php.net/license/3_01.txt\" target=\"_blank\">PHP license</a>.</p>"

	+ "<h1>xmlrpc.inc</h1>"

	+ "<p>Copyright (c) 1999,2000,2002 Edd Dumbill &lt;edd@usefulinc.com&gt; All rights reserved.</p>"

	+ "<p>Redistribution and use in source and binary forms, with or without"
	+ "modification, are permitted provided that the following conditions"
	+ "are met:</p>"

	+ "<li>Redistributions of source code must retain the above copyright"
	+ "     notice, this list of conditions and the following disclaimer.</li>"

	+ "<li>Redistributions in binary form must reproduce the above"
	+ "     copyright notice, this list of conditions and the following"
	+ "     disclaimer in the documentation and/or other materials provided"
	+ "     with the distribution.</li>"

	+ "<li>Neither the name of the \"XML-RPC for PHP\" nor the names of its"
	+ "     contributors may be used to endorse or promote products derived"
	+ "     from this software without specific prior written permission.</li>"

	+ "<p>THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "
	+ "\"AS IS\" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT "
	+ "LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS "
	+ "FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE "
	+ "REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, "
	+ "INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES "
	+ "(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR "
	+ "SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) "
	+ "HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, "
	+ "STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) "
	+ "ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED "
	+ "OF THE POSSIBILITY OF SUCH DAMAGE.</p>";
Ext.namespace('Zarafa.plugins.spreed');

/**
 * @class Zarafa.plugins.spreed.SpreedPlugin
 * @extends Zarafa.core.Plugin
 *
 * This class integrates Spreed plugin in existing system.
 * It allows user to setup spreed web meeting settings.
 */
Zarafa.plugins.spreed.SpreedPlugin = Ext.extend(Zarafa.core.Plugin, {

	/**
	 * Contains link to the spreedStore class
	 * initialized once when plugin is created.
	 *
	 * @property
	 * @type Object
	 * @private
	 */
	spreedStore : null,

	/**
	 * Unique id which works instead entryid
	 * for SpreedRecord.
	 *
	 * @property
	 * @type Integer
	 * @private
	 */
	sequenceId : 0,

	/**
	 * Called after constructor.
	 * Registers insertion points in Preview Panel, New Items drop-down and dialogs toolbars.
	 * Initialize SpreedStore property in plugin.
	 * @protected
	 */
	initPlugin : function()
	{
		Zarafa.plugins.spreed.SpreedPlugin.superclass.initPlugin.apply(this, arguments);

		this.spreedStore = new Zarafa.plugins.spreed.data.SpreedStore();

		//After record save the check in url will be returned and put in the record checkin_url field.
		this.spreedStore.on('write', this.onStoreWrite);

		this.registerInsertionPoint('main.maintoolbar.new.item', this.createNewSpreedButton, this);
		this.registerInsertionPoint('previewpanel.toolbar.right', this.createNewSpreedFromPreviewButton, this);
		this.registerInsertionPoint(/context\.mail\.(.*?)\.toolbar\.options/, this.createNewSpreedFromPreviewButton, this);
		this.registerInsertionPoint('context.mail.contextmenu.options', this.createNewSpreedFromMailButton, this);
		this.registerInsertionPoint('context.addressbook.contextmenu.actions', this.createNewSpreedFromAddressbookButton, this);
		this.registerInsertionPoint('context.contact.contextmenu.options',this.createNewSpreedFromAddressbookButton, this);
	},

	/**
	 * New button in main toolbar in Items->New menu. newMenuIndex is equal to 10, to make sure
	 * that spreed meeting button will be the last one always.
	 *
	 * @return {Ext.Button} Button instance
	 * @private
	 */
	createNewSpreedButton : function()
	{
		return  [{
			newMenuIndex : 10,
			xtype : 'menuitem',
			text : _('Spreed Meeting'),
			overflowText : _('Spreed meeting'),
			iconCls : 'icon_spreed_setup_top_menu',
			handler : this.createNewSpreedMeeting,
			scope : this
		}];
	},

	/**
	 * Create a button in ContextMenu from where an existing mail is selected,
	 * and we need to copy all information from the selected mails when the button is
	 * clicked.
	 * @return {Ext.Button} Button instance
	 * @private
	 */
	createNewSpreedFromPreviewButton : function()
	{
		return {
			xtype : 'button',
			tooltip : _('Spreed Meeting'),
			overflowText : _('Spreed meeting'),
			iconCls : 'icon_spreed_setup_mail_context_menu',
			handler : this.createNewSpreedMeetingFromSelection,
			scope : this,
			plugins : ['zarafa.recordcomponentupdaterplugin'],
			update : function(record, contentReset) {
				this.record = record;
			}
		};
	},

	/**
	 * Create a button in ContextMenu from where an existing mail is selected,
	 * and we need to copy all information from the selected mails when the button is
	 * clicked.
	 * @return {Ext.Button} Button instance
	 * @private
	 */
	createNewSpreedFromMailButton : function()
	{
		return {
			xtype : 'zarafa.conditionalitem',
			text : _('Spreed Meeting'),
			iconCls : 'icon_spreed_setup_mail_context_menu',
			handler : this.createNewSpreedMeetingFromSelection,
			scope : this
		};
	},

	/**
	 * Create a button in Contextmenu of the Address Book or Contact. This will convert the
	 * selected Address Book user to a Spreed participant.
	 *
	 * @return {Ext.Button} Button instance
	 * @private
	 */
	createNewSpreedFromAddressbookButton : function()
	{
		return {
			xtype : 'zarafa.conditionalitem',
			text : _('Spreed meeting'),
			iconCls : 'icon_spreed_setup_ab_context_menu',
			handler : this.createNewSpreedMeetingFromAddressbook,
			scope : this,
			beforeShow : this.onSpreedButtonBeforeShow.createDelegate(this)
		};
	},

	/**
	 * Event handler which is fired when the button from {@link #createNewSpreedFromAddressbookButton}
	 * is about to be shown. This will check if the button should be {@link #isSpreedButtonVisible visible}
	 * depending on the records applied to the button.
	 * @param {Ext.Button} item The item which is about to be shown
	 * @param {Ext.data.Record|Array} records The records attached to the button
	 * @private
	 */
	onSpreedButtonBeforeShow : function(item, records)
	{
		var visible = false;

		for (var i = 0, len = records.length; i < len; i++) {
			var record = records[i];

			if (this.isSpreedButtonVisible(record)) {
				visible = true;
				break;
			}
		}

		item.setVisible(visible);
	},

	/**
	 * Check if the given record (which represents a Contact or Distribution list
	 * can be mailed (this requires the record not to be a {@link Ext.data.Record#phantom}
	 * and the Contact should {@link Zarafa.contact.ContactRecord#hasEmailAddress have an email address}.
	 * @param {Zarafa.core.data.MAPIRecord} record The record to check
	 * @return {Boolean} True if we can send an email to this contact/distlist
	 * @private
	 */
	isSpreedButtonVisible : function(record)
	{
		if (record.phantom) {
			return false;
		} else if (record instanceof Zarafa.core.data.IPMRecord) {
			if (record.isMessageClass('IPM.Contact')) {
				if (!record.hasEmailAddress()) {
					return false;
				}
			} else if (record.isMessageClass('IPM.Distlist')) {
				return false;
			}
		} else if (record.get('object_type') !== Zarafa.core.mapi.ObjectType.MAPI_MAILUSER) {
			return false;
		}

		return true;
	},

	/**
	 * Event handler for creating a new {@link Zarafa.plugins.spreed.data.SpreedRecord Spreed Meeting}.
	 * This will create a blank Spreed Meeting, and open the {@link Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel Spreed ContentPanel}.
	 *
	 * @param {Ext.Button} btn The button which was clicked
	 * @private
	 */
	createNewSpreedMeeting : function(btn)
	{
		var record = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, { id : ++this.sequenceId });
		this.spreedStore.add(record);

		this.openSpreedDialog(record);
	},

	/**
	 * Event handler for creating a new {@link Zarafa.plugins.spreed.data.SpreedRecord Spreed Meeting} based
	 * on the selection of one or more {@link Zarafa.mail.MailRecord E-mails}. This will create a new
	 * Spreed Meeting where all the selected E-mails will be {@link #mergeMailToSpreed merged into}.
	 * The spreed meeting will then be opened in the {@link Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel Spreed ContentPanel}.
	 *
	 * @param {Ext.Button} btn The button which was clicked
	 * @private
	 */
	createNewSpreedMeetingFromSelection : function(btn)
	{
		var record = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, { id : ++this.sequenceId });
		this.spreedStore.add(record);
		var tasks = [];
		var mails = [];

		if (btn instanceof Ext.Button) {
			mails = [ btn.record ];
		} else if (btn instanceof Ext.menu.Item) {
			mails = btn.getRecords();
		}

		var partialMerge = mails.length > 1;

		for (var i = 0, len = mails.length; i < len; i++) {
			var mail = mails[i];

			// Check if the mail was already opened, if so we can
			// merge the mail to Spreed directly, otherwise the
			// mail needs to be opened first.
			if (mail.isOpened()) {
				this.mergeMailToSpreed(record, mail, partialMerge);
			} else {
				tasks.push({
					/*
					 * By encapsulating the task function it is possible to get the contact object 
					 * into the scope of the task function. When you add more tasks the contact 
					 * reference changes and without this encapsulation it will change the contact in
					 * all the previously added task functions as well.
					 */
					fn : function() {
						// This mailRecord becomes a private variable, not changable outside.
						var mailRecord = mail;
						return function(panel, spreedRecord, task, callback) {
							var fn = function(store, record) {
								if (record === mailRecord) {
									store.un('open', fn, task);
									this.scope.mergeMailToSpreed(spreedRecord, mailRecord, partialMerge);
									callback();
								}
							};

							mailRecord.getStore().on('open', fn, task);
							mailRecord.open();
						};
					// This triggers the encapsulation and returns the task function
					}(),
					scope : this
				});
			}
		}

		this.openSpreedDialog(record, {
			recordComponentPluginConfig : {
				loadTasks : tasks
			}
		});
	},

	/**
	 * Event handler for creating a new {@link Zarafa.plugins.spreed.data.SpreedRecord Spreed Meeting} based
	 * on the selection of one or more {@link Zarafa.contact.ContactRecord Contacts} or
	 * {@link Zarafa.addressbook.AddressbookRecord AB Users}. This will create a new
	 * Spreed Meeting where all the selected Contacts and users will be {@link #convertAddressbookUserToParticipant converted to participants}.
	 * The spreed meeting will then be opened in the {@link Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel Spreed ContentPanel}.
	 *
	 * @param {Ext.Button} btn The button which was clicked
	 * @private
	 */
	createNewSpreedMeetingFromAddressbook : function(btn)
	{
		var record = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, { id : ++this.sequenceId });
		this.spreedStore.add(record);
		var tasks = [];
		var contacts = btn.getRecords();

		for (var i = 0, len = contacts.length; i < len; i++) {
			var contact = contacts[i];

			// Check if the contact was already opened, if so we can
			// add the contact to Spreed as participant directly, otherwise the
			// mail needs to be opened first.
			if (contact.isOpened()) {
				var participant = this.convertAddressbookUserToParticipant(contact);
				record.getSubStore('recipients').add(participant);
			} else {
				tasks.push({
					/*
					 * By encapsulating the task function it is possible to get the contact object 
					 * into the scope of the task function. When you add more tasks the contact 
					 * reference changes and without this encapsulation it will change the contact in
					 * all the previously added task functions as well.
					 */
					fn : function() {
						// This contactRecord becomes a private variable, not changable outside.
						var contactRecord = contact;
						return function(panel, spreedRecord, task, callback) {
							var fn = function(store, record) {
								if (record === contactRecord) {
									store.un('open', fn, task);
									var participant = this.scope.convertAddressbookUserToParticipant(contact);
									spreedRecord.getSubStore('recipients').add(participant);
									callback();
								}
							};

							contactRecord.getStore().on('open', fn, task);
							contactRecord.open();
						};
					// This triggers the encapsulation and returns the task function
					}(),
					scope : this
				});
			}
		}

		this.openSpreedDialog(record, {
			recordComponentPluginConfig : {
				loadTasks : tasks
			}
		});

	},

	/**
	 * Merge a {@link Zarafa.mail.MailRecord E-mail} into a {@link Zarafa.plugins.spreed.data.SpreedRecord Spreed Meeting}.
	 * This will {@link #mergeSender Merge the sender of the mail}, as well as all {@link #mergeParticipants recipients}
	 * and {@link #mergeAttachments attachments}. Optionally it will also merge the 'subject' and 'Zarafa.mail.MailRecord#getBody body}.
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedRecord} spreed The Spreed meeting to merge to
	 * @param {Zarafa.mail.MailRecord} mail The E-mail to merge from
	 * @param {Boolean} partial True to prevent the subject and body from being merged
	 * @private
	 */
	mergeMailToSpreed : function(spreed, mail, partial)
	{
		// Merge the sender of the original mail
		this.mergeSender(spreed, mail);

		// Merge the Recipients substore
		this.mergeParticipants(spreed.getSubStore('recipients'), mail.getSubStore('recipients'));

		// Merge the Attachments substore
		this.mergeAttachments(spreed.getSubStore('attachments'), mail.getSubStore('attachments'));

		if (partial !== true) {
			spreed.set('subject', mail.get('subject'));
			spreed.set('description', mail.getBody(false));
		}
	},

	/**
	 * Merge the sender of the {@link Zarafa.mail.MailRecord E-mail} into the
	 * {@link Zarafa.plugins.spreed.data.SpreedRecord Spreed Meeting}. This will
	 * check if the 'sent_representing_entryid' propery is there, if so copy the
	 * 'sent_representing' properties, otherwise the 'sender' properties are used.
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedRecord} spreed The Spreed meeting to merge to
	 * @param {Zarafa.mail.MailRecord} mail The E-mail to merge from
	 * @private
	 */
	mergeSender : function(spreed, mail)
	{
		var representingEntryid = mail.get('sent_representing_entryid');
		var data;

		if (!Ext.isEmpty(representingEntryid)) {
			data = {
				entryid : mail.get('sent_representing_entryid'),
				object_type : Zarafa.core.mapi.ObjectType.MAPI_MAILUSER,
				display_name : mail.get('sent_representing_name'),
				display_type : Zarafa.core.mapi.DisplayType.DT_MAILUSER,
				display_type_ex : Zarafa.core.mapi.DisplayType.DT_MAILUSER,
				email_address :  mail.get('sent_representing_email_address'),
				smtp_address :  mail.get('sent_representing_email_address'),
				address_type : mail.get('sent_representing_address_type'),
				recipient_type : Zarafa.core.mapi.RecipientType.MAPI_TO
			};
		} else {
			data = {
				entryid : mail.get('sender_entryid'),
				object_type : Zarafa.core.mapi.ObjectType.MAPI_MAILUSER,
				display_name : mail.get('sender_name'),
				display_type : Zarafa.core.mapi.DisplayType.DT_MAILUSER,
				display_type_ex : Zarafa.core.mapi.DisplayType.DT_MAILUSER,
				email_address :  mail.get('sender_email_address'),
				smtp_address : mail.get('sender_email_address'),
				address_type : mail.get('sender_address_type'),
				recipient_type : Zarafa.core.mapi.RecipientType.MAPI_TO
			};
		}

		var spreedParticipant = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT, data);
		var participantStore = spreed.getSubStore('recipients');

		if (this.isAllowedParticipant(participantStore, spreedParticipant)) {
			participantStore.add(spreedParticipant);
		}
	},

	/**
	 * Convert all {@link Zarafa.core.data.IPMRecipientRecord Recipient Records} from the given
	 * {@link Zarafa.core.data.IPMRecipientStore Recipient Store} to 
	 * {@link Zarafa.plugins.spreed.data.SpreedParticipantRecord Participants}.
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedParticipantStore} participantStore Participants store to which to merge to
	 * @param {Zarafa.core.data.IPMRecipientStore} recipientStore The store from where all recipients are merged from
	 * @private
	 */
	mergeParticipants : function(participantStore, recipientStore)
	{
		recipientStore.each(function(recipient) {
			var spreedParticipant = this.convertRecipientToParticipant(recipient);
			if (this.isAllowedParticipant(participantStore, spreedParticipant)) {
				participantStore.add(spreedParticipant);
			}
		}, this);
	},

	/**
	 * Convert all {@link Zarafa.core.data.IPMAttachmentRecord Attachment Records} from the given
	 * {@link Zarafa.core.data.IPMAttachmentStore Attachment Store} to 
	 * {@link Zarafa.plugins.spreed.data.SpreedAttachmentRecord Spreed Attachments}.
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedAttachmentStore} spreedStore Attachment store to which to merge to
	 * @param {Zarafa.core.data.IPMAttachmentStore} attachmentStore The store from where all attachments are merged from
	 * @private
	 */
	mergeAttachments : function(spreedStore, attachmentStore)
	{
		attachmentStore.each(function(attach) {
			var spreedAttach = this.convertAttachmentToSpreed(attach);
			spreedStore.add(spreedAttach);
		}, this);
	},

	/**
	 * Convert a single {@link Zarafa.core.data.IPMRecipientRecord recipient} to
	 * a {@link Zarafa.plugins.spreed.data.SpreedParticipantRecord Participant}.
	 *
	 * @param {Zarafa.core.data.IPMRecipientRecord} recipient The recipient to convert
	 * @return {Zarafa.plugins.spreed.data.SpreedParticipantRecord} The participant
	 * @private
	 */
	convertRecipientToParticipant : function(recipient)
	{
		var Factory = Zarafa.core.data.RecordFactory;
		var Type = Zarafa.core.data.RecordCustomObjectType;
		var data = Ext.apply({}, recipient.data);

		// Force recipient_type to be MAPI_TO
		data.recipient_type = Zarafa.core.mapi.RecipientType.MAPI_TO;

		return Factory.createRecordObjectByCustomType(Type.ZARAFA_SPREED_PARTICIPANT, data);
	},

	/**
	 * Convert a {@link Zarafa.contact.ContactRecord Contact} or {@link Zarafa.addressbook.AddressbookRecord AB User}
	 * to a {@link Zarafa.plugins.spreed.data.SpreedParticipantRecord Participant}.
	 *
	 * @param {Zarafa.contact.ContactRecord|Zarafa.addressbook.AddressbookRecord} user The user to convert
	 * @return {Zarafa.plugins.spreed.data.SpreedParticipantRecord} The participant
	 * @private
	 */
	convertAddressbookUserToParticipant : function(user)
	{
		var recipients = user.convertToRecipient(Zarafa.core.mapi.RecipientType.MAPI_TO, true);
		var participants = [];

		if (!Ext.isArray(recipients)) {
			recipients = [ recipients ];
		}

		for (var i = 0, len = recipients.length; i < len; i++) {
			participants.push(this.convertRecipientToParticipant(recipients[i]));
		}

		return participants;
	},

	/**
	 * Convert a {@link Zarafa.core.data.IPMAttachmentRecord attachment} to
	 * a {@link Zarafa.plugins.spreed.data.SpreedAttachmentRecord Spreed attachment}.
	 *
	 * @param {Zarafa.core.data.IPMAttachmentRecord} attachment The attachment to convert
	 * @return {Zarafa.plugins.spreed.data.SpreedAttachmentRecord} The Spreed Attachment
	 * @private
	 */
	convertAttachmentToSpreed : function(attachment)
	{
		var Factory = Zarafa.core.data.RecordFactory;
		var Type = Zarafa.core.data.RecordCustomObjectType;
		var data = Ext.apply({}, attachment.data);
		var store_entryid = attachment.store.getParentRecord().get('store_entryid');
		var parent_entryid = attachment.store.getAttachmentParentRecordEntryId();
		var id = attachment.store.getId();

		Ext.apply(data, {
			original_record_entry_id : parent_entryid,
			original_record_store_entry_id : store_entryid,
			original_attachment_store_id : id,
			original_attach_num : attachment.get('attach_num')
		});

		return Factory.createRecordObjectByCustomType(Type.ZARAFA_SPREED_ATTACHMENT, data);
	},

	/**
	 * Check if the given {@link Zarafa.plugins.spreed.data.SpreedParticipantRecord participant}
	 * is allowed to be added to the {@link Zarafa.plugins.spreed.data.SpreedParticipantStore Store}.
	 *
	 * This checks if the participant is not the {@link Zarafa.core.Container#getUser current user},
	 * and is not already present in the participant store
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedParticipantStore} participantStore The store the participant is going
	 * to be added to
	 * @param {Zarafa.plugins.spreed.data.SpreedParticipantRecord} newParticipant The participant which is being checked
	 * @private
	 */
	isAllowedParticipant : function(participantStore, newParticipant)
	{
		var EntryId = Zarafa.core.EntryId;

		if (EntryId.compareEntryIds(container.getUser().getEntryId(), newParticipant.get('entryid'))) {
			return false;
		}

		var index = participantStore.findBy(function(participant) {
			return EntryId.compareEntryIds(participant.get('entryid'), newParticipant.get('entryid'));
		});

		return index < 0;
	},

	/**
	 * Open the SpreedSetupDialog with the custom SpreedRecord
	 * passed as the param. This record contains defualt values
	 * for dialog fields.
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedRecord} record
	 * @param {Object} config Configuration object for the Spreed Dialog
	 * @private
	 */
	openSpreedDialog : function(record, config)
	{
		Zarafa.core.data.UIFactory.openLayerComponent(Zarafa.core.data.SharedComponentType['common.create'], record, config);
	},

	/**
	 * Bid for the type of shared component
	 * and the given record.
	 * This will bid on a common.dialog.create or common.dialog.view for a
	 * record with a message class set to IPM or IPM.Note.
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Ext.data.Record} record Optionally passed record.
	 * @return {Number} The bid for the shared component
	 */
	bidSharedComponent : function(type, record)
	{
		var bid = -1;
		if (Ext.isArray(record)) {
			record = record[0];
		}

		if (record && record.store || record instanceof Zarafa.addressbook.AddressBookRecord) {
			switch (type)
			{
				case Zarafa.core.data.SharedComponentType['common.create']:
				case Zarafa.core.data.SharedComponentType['common.view']:
					if (record instanceof Zarafa.plugins.spreed.data.SpreedRecord) {
						bid = 2;
					} else if (record.store.customObjectType == Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT) {
						bid = 2;
					}
					break;
				case Zarafa.core.data.SharedComponentType['common.contextmenu']:
					if (record.store.customObjectType == Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT) {
						bid = 2;
					}
					break;
			}
		}
		return bid;
	},

	/**
	 * Will return the reference to the shared component.
	 * Based on the type of component requested a component is returned.
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Ext.data.Record} record Optionally passed record.
	 * @return {Ext.Component} Component
	 */
	getSharedComponent : function(type, record)
	{
		var component;
		switch (type)
		{
			case Zarafa.core.data.SharedComponentType['common.create']:
			case Zarafa.core.data.SharedComponentType['common.view']:
				if (record instanceof Zarafa.plugins.spreed.data.SpreedRecord) {
					component = Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel;
				} else if (record.store.customObjectType == Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT) {
					component = Zarafa.plugins.spreed.dialogs.EditSpreedParticipantContentPanel;
				}
				break;
			case Zarafa.core.data.SharedComponentType['common.contextmenu']:
				component = Zarafa.plugins.spreed.dialogs.SpreedParticipantContextMenu;
				break;
		}
		return component;
	},

	/**
	 * Open new window with check in url which comes from server after save record action.
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedStore} store The store which fired the event
	 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
	 * @param {Object} data The 'data' picked-out out of the response for convenience.
	 * @param {Ext.Direct.Transaction} res
	 * @param {Zarafa.plugins.spreed.data.SpreedRecord|Array} records The most recent version of the records
	 * which came from the server.
	 * @private
	 */
	onStoreWrite : function(store, action, result, res, records)
	{
		if (action == Ext.data.Api.actions['create']) {
			var checkin_url;
			if (Ext.isArray(records)) {
				if (records.length > 1) {
					checkin_url = records[0].get('checkin_url');
				}
			} else if (records) {
				checkin_url = records.get('checkin_url');
			}

			if (!Ext.isEmpty(checkin_url)) {
				window.open(checkin_url, '');
			} else {
				Ext.MessageBox.show({
					title   : _('Warning'),
					height  : 300,
					width   : 300,
					msg     : _('Returned url is empty'),
					icon    : Ext.MessageBox.WARNING,
					buttons : Ext.MessageBox.OK
				});
			}
		}
	}
});

Zarafa.onReady(function() {
	container.registerPlugin(new Zarafa.core.PluginMetaData({
		name : 'spreed',
		displayName : _('Spreed Plugin'),
		about : Zarafa.plugins.spreed.ABOUT,
		pluginConstructor : Zarafa.plugins.spreed.SpreedPlugin
	}));
});
Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * @class Zarafa.plugins.spreed.data.SpreedAttachmentRecordFields
 *
 * Array of additional fields for the custom ZARAFA_SPREED_ATTACHMENT object.
 * These fields allow us to track from which IPMRecord we have copied this record
 * and thus will let us to copy the attachment from IPMRecord on server.
 *
 */
Zarafa.plugins.spreed.data.SpreedAttachmentRecordFields = [
	{name : 'original_record_entry_id', type : 'string', defaultValue : ''},
	{name : 'original_record_store_entry_id', type : 'string', defaultValue : ''},
	{name : 'original_attach_num', type : 'int'},
	{name : 'original_attachment_store_id', type : 'string', defaultValue : ''}
];

Zarafa.core.data.RecordCustomObjectType.addProperty('ZARAFA_SPREED_ATTACHMENT');
Zarafa.core.data.RecordFactory.setBaseClassToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_ATTACHMENT, Zarafa.core.data.IPMAttachmentRecord);
Zarafa.core.data.RecordFactory.addFieldToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_ATTACHMENT, Zarafa.plugins.spreed.data.SpreedAttachmentRecordFields);
Zarafa.core.data.RecordFactory.addFieldToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_ATTACHMENT, Zarafa.core.data.IPMAttachmentRecordFields);
Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * @class Zarafa.plugins.spreed.data.SpreedAttachmentStore
 * @extends Zarafa.core.data.IPMAttachmentStore
 *
 * We extends IPMAttachmentStore to allow using our custom record type in AttachmentJsonReader
 * and override IPMAttachmentJsonWriter with our custom JsonWriter which will allow us to
 * send the serialized custom attachment records to server.
 */
Zarafa.plugins.spreed.data.SpreedAttachmentStore = Ext.extend(Zarafa.core.data.IPMAttachmentStore, {

	/**
	 * @constructor
	 * @param config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		var customRecordClass = Zarafa.core.data.RecordFactory.getRecordClassByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_ATTACHMENT);
		Ext.applyIf(config, {
			reader : new Zarafa.core.data.JsonAttachmentReader({dynamicRecord : false}, customRecordClass),
			writer : new Zarafa.plugins.spreed.data.SpreedJsonAttachmentWriter()
		});

		Zarafa.plugins.spreed.data.SpreedAttachmentStore.superclass.constructor.call(this, config);
	},


	/**
	 * This function tries to fill in the parametrs that are used on
	 * server side so server can understand where from comes the attachment -
	 * mailDailog, spreeddialog or smth else
	 * @param {Zarafa.plugins.spreed.data.SpreedAttachmentRecord} attachmentRecord
	 */
	getAttachmentBaseUrl : function(attachmentRecord)
	{
		var url = 'index.php?load=download_attachment';
		var originalRecordEntryId = attachmentRecord.get('original_record_entry_id');
		var originalRecordStoreEntryId = attachmentRecord.get('original_record_store_entry_id');

		var dialogAttachments = null;
		var entryId = null;
		var storeId = null;
		var attachNum = null;

		//Check if user clicked on the currently uploaded from Spreed Dialog attachment
		// or on copied from selected message record.
		if( Ext.isEmpty(originalRecordEntryId) && Ext.isEmpty(originalRecordStoreEntryId) ) {// Original attachment
			entryId = this.getAttachmentParentRecordEntryId();
			storeId = this.getParentRecord().get('store_entryid');
			dialogAttachments = this.getId();
			attachNum = attachmentRecord.get('attach_num');
		} else {
			entryId = originalRecordEntryId;
			storeId = originalRecordStoreEntryId;
			dialogAttachments = attachmentRecord.get('original_attachment_store_id');
			attachNum = attachmentRecord.get('original_attach_num');
		}

		url = Ext.urlAppend(url, 'dialog_attachments=' + dialogAttachments);
		url = Ext.urlAppend(url, 'store='   + storeId);
		url = Ext.urlAppend(url, 'entryid=' + entryId);
		if (attachmentRecord.get('attach_num') != -1)
			url = Ext.urlAppend(url, 'attachNum[]=' + attachNum);
		else
			url = Ext.urlAppend(url, 'attachNum[]=' + attachmentRecord.get('tmpname'));

		return url;
	}

});

Ext.reg('spreed.attachmentstore', Zarafa.plugins.spreed.data.SpreedAttachmentStore);
Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * @class Zarafa.plugins.spreed.data.SpreedJsonAttachmentWriter
 * @extends Zarafa.core.data.JsonAttachmentWriter
 */
Zarafa.plugins.spreed.data.SpreedJsonAttachmentWriter = Ext.extend(Zarafa.core.data.JsonAttachmentWriter, {
	/**
	 * Similar to {@link Zarafa.core.data.JsonAttachmentWriter#toHash}.
	 * Here we serializing only the data of the records in spreed attachment store.
	 * Note that we serialize all the records - not only removed or modified.
	 *
	 * @param {Ext.data.Record} record The record to hash
	 * @return {Object} The hashed object
	 * @override
	 * @private
	 */
	toPropHash : function(record)
	{
		var attachmentStore = record.getAttachmentStore();
		var hash = {};

		if (!Ext.isDefined(attachmentStore))
			return hash;

		// Overwrite previous definition to something we can work with.
		hash.attachments = {};
		hash.attachments.dialog_attachments = attachmentStore.getId();

		var attachmentRecords = attachmentStore.getRange();
		Ext.each(attachmentRecords, function(attach) {
			if (!Ext.isDefined(hash.attachments.add)) {
				hash.attachments.add = [];
			}
			var data = attach.data;
			hash.attachments.add.push(data);
		}, this);

		return hash;
	}
});
Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * @class Zarafa.plugins.spreed.data.SpreedJsonParticipantWriter
 * @extends Zarafa.core.data.JsonRecipientWriter
 * @xtype spreed.spreedparticipantwriter
 */
Zarafa.plugins.spreed.data.SpreedJsonParticipantWriter = Ext.extend(Zarafa.core.data.JsonRecipientWriter, {

	/**
	 * Similar to {@link Ext.data.JsonWriter#toHash}
	 *
	 * Convert recipients into a hash. Recipients exists as
	 * {@link Zarafa.core.data.IPMRecipientRecord IPMRecipientRecord} within
	 * a {@link Zarafa.core.data.IPMRecord IPMRecord} and thus must be serialized
	 * seperately into the hash object.
	 *
	 * @param {Ext.data.Record} record The record to hash
	 * @return {Object} The hashed object
	 * @override
	 * @private
	 */
	toPropHash : function(record)
	{
		var participantsStore = record.getRecipientStore();
		var hash = {};

		if (!Ext.isDefined(participantsStore))
			return hash;

		// Get list of modified (modified and newly added) records
		var participantsRecords = participantsStore.getRange();

		if(participantsRecords.length > 0) {
			hash.recipients = [];

			// Adding the modified records to the add or modified part of the recipients bit
			for (var i = 0; i < participantsRecords.length; i++) {
				var participant = participantsRecords[i];

				// FIXME: serialize?
				var data = participant.data;

				if(participant.isMeetingOrganizer()) {
					// organizer information shouldn't be passed in recipient table at all
					continue;
				}
				hash.recipients.push(data);
			}
		}

		return hash;
	}
});

Ext.reg('spreed.spreedparticipantwriter', Zarafa.plugins.spreed.data.SpreedJsonParticipantWriter);
Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * SpreedLanguages object
 */
Zarafa.plugins.spreed.data.SpreedLanguages =[


	{ value:  'aa', name: _('Afar')},
	{ value:  'af', name: _('Afrikaans')},
	{ value:  'ak', name: _('Akan')},
	{ value:  'sq', name: _('Albanian')},
	{ value:  'am', name: _('Amharic')},
	{ value:  'ar', name: _('Arabic')},
	{ value:  'ab', name: _('Abkhazian')},
	{ value:  'an', name: _('Aragonese')},
	{ value:  'hy', name: _('Armenian')},
	{ value:  'as', name: _('Assamese')},
	{ value:  'av', name: _('Avaric')},
	{ value:  'ae', name: _('Avestan')},
	{ value:  'ay', name: _('Aymara')},
	{ value:  'az', name: _('Azerbaijani')},
	{ value:  'bm', name: _('Bambara')},
	{ value:  'ba', name: _('Bashkir')},
	{ value:  'eu', name: _('Basque')},
	{ value:  'be', name: _('Belarusian')},
	{ value:  'bn', name: _('Bengali')},
	{ value:  'bh', name: _('Bihari')},
	{ value:  'bi', name: _('Bislama')},
	{ value:  'bs', name: _('Bosnian')},
	{ value:  'br', name: _('Breton')},
	{ value:  'bg', name: _('Bulgarian')},
	{ value:  'my', name: _('Burmese')},
	{ value:  'ca', name: _('Catalan; Valencian')},
	{ value:  'km', name: _('Central Khmer')},
	{ value:  'ch', name: _('Chamorro')},
	{ value:  'ce', name: _('Chechen')},
	{ value:  'ny', name: _('Chichewa; Chewa; Nyanja')},
	{ value:  'zh', name: _('Chinese')},
	{ value:  'cu', name: _('Church Slavic; Old Slavonic; Church Slavonic; Old Bulgarian; Old Church Slavonic')},
	{ value:  'cv', name: _('Chuvash')},
	{ value:  'kw', name: _('Cornish')},
	{ value:  'co', name: _('Corsican')},
	{ value:  'cr', name: _('Cree')},
	{ value:  'hr', name: _('Croatian')},
	{ value:  'cs', name: _('Czech')},
	{ value:  'da', name: _('Danish')},
	{ value:  'dv', name: _('Divehi; Dhivehi; Maldivian')},
	{ value:  'nl', name: _('Dutch; Flemish')},
	{ value:  'dz', name: _('Dzongkha')},
	{ value:  'en', name: _('English')},
	{ value:  'eo', name: _('Esperanto')},
	{ value:  'et', name: _('Estonian')},
	{ value:  'ee', name: _('Ewe')},
	{ value:  'fo', name: _('Faroese')},
	{ value:  'fj', name: _('Fijian')},
	{ value:  'fi', name: _('Finnish')},
	{ value:  'fr', name: _('French')},
	{ value:  'ff', name: _('Fulah')},
	{ value:  'gd', name: _('Gaelic; Scottish Gaelic')},
	{ value:  'gl', name: _('Galician')},
	{ value:  'lg', name: _('Ganda')},
	{ value:  'ka', name: _('Georgian')},
	{ value:  'de', name: _('German')},
	{ value:  'el', name: _('Greek, Modern')},
	{ value:  'gn', name: _('Guarani')},
	{ value:  'gu', name: _('Gujarati')},
	{ value:  'ht', name: _('Haitian; Haitian Creole')},
	{ value:  'ha', name: _('Hausa')},
	{ value:  'he', name: _('Hebrew')},
	{ value:  'hz', name: _('Herero')},
	{ value:  'hi', name: _('Hindi')},
	{ value:  'ho', name: _('Hiri Motu')},
	{ value:  'hu', name: _('Hungarian')},
	{ value:  'is', name: _('Icelandic')},
	{ value:  'ig', name: _('Igbo')},
	{ value:  'io', name: _('Ido')},
	{ value:  'ia', name: _('Interlingua (International Auxiliary Language Association)')},
	{ value:  'ie', name: _('Interlingue')},
	{ value:  'id', name: _('Indonesian')},
	{ value:  'iu', name: _('Inuktitut')},
	{ value:  'ik', name: _('Inupiaq')},
	{ value:  'ga', name: _('Irish')},
	{ value:  'it', name: _('Italian')},
	{ value:  'ja', name: _('Japanese')},
	{ value:  'jv', name: _('Javanese')},
	{ value:  'kl', name: _('Kalaallisut; Greenlandic')},
	{ value:  'kn', name: _('Kannada')},
	{ value:  'kr', name: _('Kanuri')},
	{ value:  'ks', name: _('Kashmiri')},
	{ value:  'kk', name: _('Kazakh')},
	{ value:  'ki', name: _('Kikuyu; Gikuyu')},
	{ value:  'rw', name: _('Kinyarwanda')},
	{ value:  'ky', name: _('Kirghiz; Kyrgyz')},
	{ value:  'kv', name: _('Komi')},
	{ value:  'kg', name: _('Kongo')},
	{ value:  'ko', name: _('Korean')},
	{ value:  'kj', name: _('Kuanyama; Kwanyama')},
	{ value:  'ku', name: _('Kurdish')},
	{ value:  'lo', name: _('Lao')},
	{ value:  'la', name: _('Latin')},
	{ value:  'lv', name: _('Latvian')},
	{ value:  'li', name: _('Limburgan; Limburger; Limburgish')},
	{ value:  'ln', name: _('Lingala')},
	{ value:  'lt', name: _('Lithuanian')},
	{ value:  'lu', name: _('Luba-Katanga')},
	{ value:  'lb', name: _('Luxembourgish; Letzeburgesch')},
	{ value:  'mk', name: _('Macedonian')},
	{ value:  'mg', name: _('Malagasy')},
	{ value:  'ms', name: _('Malay')},
	{ value:  'ml', name: _('Malayalam')},
	{ value:  'mt', name: _('Maltese')},
	{ value:  'gv', name: _('Manx')},
	{ value:  'mi', name: _('Maori')},
	{ value:  'mr', name: _('Marathi')},
	{ value:  'mh', name: _('Marshallese')},
	{ value:  'mo', name: _('Moldavian')},
	{ value:  'mn', name: _('Mongolian')},
	{ value:  'na', name: _('Nauru')},
	{ value:  'nv', name: _('Navajo; Navaho')},
	{ value:  'nd', name: _('Ndebele, North; North Ndebele')},
	{ value:  'nr', name: _('Ndebele, South; South Ndebele')},
	{ value:  'ng', name: _('Ndonga')},
	{ value:  'ne', name: _('Nepali')},
	{ value:  'se', name: _('Northern Sami')},
	{ value:  'no', name: _('Norwegian')},
	{ value:  'nn', name: _('Norwegian Nynorsk; Nynorsk, Norwegian')},
	{ value:  'nb', name: _('Norwegian Bokm??l; Bokm??l, Norwegian;')},
	{ value:  'oc', name: _('Occitan;  Proven??al')},
	{ value:  'oj', name: _('Ojibwa')},
	{ value:  'or', name: _('Oriya')},
	{ value:  'om', name: _('Oromo')},
	{ value:  'os', name: _('Ossetian; Ossetic')},
	{ value:  'pi', name: _('Pali')},
	{ value:  'pa', name: _('Panjabi; Punjabi')},
	{ value:  'fa', name: _('Persian')},
	{ value:  'pl', name: _('Polish')},
	{ value:  'pt', name: _('Portuguese')},
	{ value:  'ps', name: _('Pushto')},
	{ value:  'qu', name: _('Quechua')},
	{ value:  'ro', name: _('Romanian')},
	{ value:  'rm', name: _('Romansh')},
	{ value:  'rn', name: _('Rundi')},
	{ value:  'ru', name: _('Russian')},
	{ value:  'sm', name: _('Samoan')},
	{ value:  'sg', name: _('Sango')},
	{ value:  'sa', name: _('Sanskrit')},
	{ value:  'sc', name: _('Sardinian')},
	{ value:  'sr', name: _('Serbian')},
	{ value:  'sn', name: _('Shona')},
	{ value:  'ii', name: _('Sichuan Yi')},
	{ value:  'sd', name: _('Sindhi')},
	{ value:  'si', name: _('Sinhala; Sinhalese')},
	{ value:  'sk', name: _('Slovak')},
	{ value:  'sl', name: _('Slovenian')},
	{ value:  'so', name: _('Somali')},
	{ value:  'st', name: _('Sotho, Southern')},
	{ value:  'es', name: _('Spanish; Castilian')},
	{ value:  'su', name: _('Sundanese')},
	{ value:  'sw', name: _('Swahili')},
	{ value:  'ss', name: _('Swati')},
	{ value:  'sv', name: _('Swedish')},
	{ value:  'tl', name: _('Tagalog')},
	{ value:  'ty', name: _('Tahitian')},
	{ value:  'tg', name: _('Tajik')},
	{ value:  'ta', name: _('Tamil')},
	{ value:  'tt', name: _('Tatar')},
	{ value:  'te', name: _('Telugu')},
	{ value:  'th', name: _('Thai')},
	{ value:  'bo', name: _('Tibetan')},
	{ value:  'ti', name: _('Tigrinya')},
	{ value:  'to', name: _('Tonga (Tonga Islands)')},
	{ value:  'ts', name: _('Tsonga')},
	{ value:  'tn', name: _('Tswana')},
	{ value:  'tr', name: _('Turkish')},
	{ value:  'tk', name: _('Turkmen')},
	{ value:  'tw', name: _('Twi')},
	{ value:  'ug', name: _('Uighur; Uyghur')},
	{ value:  'uk', name: _('Ukrainian')},
	{ value:  'ur', name: _('Urdu')},
	{ value:  'uz', name: _('Uzbek')},
	{ value:  've', name: _('Venda')},
	{ value:  'vi', name: _('Vietnamese')},
	{ value:  'vo', name: _('Volap??k')},
	{ value:  'wa', name: _('Walloon')},
	{ value:  'cy', name: _('Welsh')},
	{ value:  'fy', name: _('Western Frisian')},
	{ value:  'wo', name: _('Wolof')},
	{ value:  'xh', name: _('Xhosa')},
	{ value:  'yi', name: _('Yiddish')},
	{ value:  'yo', name: _('Yoruba')},
	{ value:  'za', name: _('Zhuang; Chuang')},
	{ value:  'zu', name: _('Zulu')}
	];Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * @class Zarafa.plugins.spreed.data.SpreedParticipantRecordFields
 * Array of fields for the {@link Zarafa.plugins.spreed.data.SpreedParticipantRecord} object.
 */
Zarafa.plugins.spreed.data.SpreedParticipantRecordFields =[
		{name: 'isModerator', type: 'boolean'},
		{name: 'timezone', type: 'string'},
		{name: 'language', type: 'string'}
];

// Register a spreed participant record type to be used by the Record Factory
Zarafa.core.data.RecordCustomObjectType.addProperty('ZARAFA_SPREED_PARTICIPANT');
Zarafa.core.data.RecordFactory.setBaseClassToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT, Zarafa.core.data.IPMRecipientRecord);
Zarafa.core.data.RecordFactory.addFieldToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT, Zarafa.core.data.IPMRecipientRecordFields);
Zarafa.core.data.RecordFactory.addFieldToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT, Zarafa.plugins.spreed.data.SpreedParticipantRecordFields);

Zarafa.core.data.RecordFactory.addListenerToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT, 'createphantom', function(record)
{
	var settings = container.getSettingsModel();

	if (Ext.isEmpty(record.get('timezone'))) {
		var timezone = settings.get('zarafa/v1/plugins/spreed/default_timezone');
		record.set('timezone', timezone);
	}

	if (Ext.isEmpty(record.get('language'))) {
		var currentLang = container.getSettingsModel().get('zarafa/v1/main/language');
		record.set('language', currentLang.substr(0, currentLang.indexOf('_')));
	}
});
Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * @class Zarafa.plugins.spreed.data.SpreedParticipantStore
 * @extends Zarafa.core.data.IPMRecipientStore
 */
Zarafa.plugins.spreed.data.SpreedParticipantStore=Ext.extend(Zarafa.core.data.IPMRecipientStore,{

	/**
	 * @constructor
	 * @param config {Object} Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			writer : new Zarafa.plugins.spreed.data.SpreedJsonParticipantWriter(),
			customObjectType : Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT,
			reader: new Zarafa.core.data.JsonRecipientReader({
				id : 'entryid',
				idProperty : 'entryid',
				dynamicRecord : false,
				customObjectType : Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT
			})
		});

		Zarafa.plugins.spreed.data.SpreedParticipantStore.superclass.constructor.call(this, config)
	}

});
Ext.reg('spreed.spreedparticipantstore', Zarafa.plugins.spreed.data.SpreedParticipantStore);
Ext.namespace('Zarafa.plugins.spreed.data');



/**
 * @class Zarafa.plugins.spreed.data.SpreedRecordFields
 * Array of additional fields for the {@link Zarafa.plugins.spreed.data.SpreedRecord} object.
 */
Zarafa.plugins.spreed.data.SpreedRecordFields = [
	{name: 'subject', type: 'string', defaultValue: ''},
	{name: 'start_time', type: 'date', dateFormat: 'timestamp', defaultValue: null},
	{name: 'end_time', type: 'date', dateFormat: 'timestamp', defaultValue: null},
	{name: 'description', type: 'string', defaultValue: ''},
	{name: 'timezone', type: 'string'},
	//will be filled when the record will be created on server
	{name: 'checkin_url', type: 'string', defaultValue: ''}

];

Zarafa.core.data.RecordCustomObjectType.addProperty('ZARAFA_SPREED');
Zarafa.core.data.RecordFactory.addFieldToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, Zarafa.plugins.spreed.data.SpreedRecordFields);

Zarafa.core.data.RecordFactory.setSubStoreToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, 'recipients', Zarafa.plugins.spreed.data.SpreedParticipantStore);
Zarafa.core.data.RecordFactory.setSubStoreToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, 'attachments', Zarafa.plugins.spreed.data.SpreedAttachmentStore);

Zarafa.core.data.RecordFactory.addListenerToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, 'createphantom', function(record)
{
	var settings = container.getSettingsModel();

	// Phantom records must always be marked as opened (they contain the full set of data)
	record.afterOpen();

	if (Ext.isEmpty(record.get('timezone'))) {
		record.set('timezone', settings.get('zarafa/v1/plugins/spreed/default_timezone'));
	}

	// If no appointment date was selected, we just calculate the default
	if (!Ext.isDate(record.get('start_time')) || !Ext.isDate(record.get('end_time'))) {
		var delay = settings.get('zarafa/v1/contexts/calendar/default_zoom_level');

		var startTime = new Date().ceil(Date.MINUTE, delay).fromUTC();
		var duration = settings.get('zarafa/v1/contexts/calendar/default_appointment_period');
		var endTime = startTime.add(Date.MINUTE, duration);

		record.set('start_time', startTime);
		record.set('end_time', endTime);
	}
});

/**
 * @class Zarafa.plugins.spreed.data.SpreedRecord
 * @extends Zarafa.core.data.IPMRecord
 *
 * An extension to the {@link Zarafa.core.data.IPMRecord} specific to Spreed Request/Response Messages.
 */
Zarafa.plugins.spreed.data.SpreedRecord = Ext.extend(Zarafa.core.data.IPMRecord, {

	/**
	 * The base array of ID properties which is copied to the {@link #idProperties}
	 * when the record is being created. Here we overriding the base id properties
	 * of parent because our record have not entryid.
	 *
	 * @property
	 * @type Array
	 * @private
	 * @override
	 */
	baseIdProperties : [ 'id' ],

	/*
	 * Copy the {@link Zarafa.core.data.MAPIRecord Record} to a new instance
	 * @param {String} newId (optional) A new Record id, defaults to the id of the record being copied. See id.
	 * @return {Zarafa.core.data.MAPIRecord} The copy of the record.
	 * @override
	 * @private
	 */
	copy : function(newId)
	{
		var copy = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, this.data, newId || this.id);

		copy.idProperties = this.idProperties.clone();
		copy.phantom = this.phantom;

		return copy.applyData(this);
	},

	/**
	 * Compare this {@link Zarafa.plugins.spreed.data.SpreedRecord record} instance with another one to see
	 * if they are the same Spreed Records from the server (i.e. The id matches).
	 *
	 * @param {Zarafa.core.data.IPMRecord} record The IPMRecord to compare with
	 * @return {Boolean} True if the records are the same.
	 * @override
	 * @private
	 */
	equals : function(record)
	{
		return this.id == record.id;
	}
});

Zarafa.core.data.RecordFactory.setBaseClassToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED, Zarafa.plugins.spreed.data.SpreedRecord);
Ext.namespace('Zarafa.plugins.spreed.data');

/**
 * @class Zarafa.plugins.spreed.data.SpreedStore
 * @extends Zarafa.core.data.MAPIStore
 *
 * This class extends MAPIStore to configure the
 * proxy and reader in custom way that all requests
 * will send to spreedmodule and with custom id param.
 * Instead of defining the records dynamically, reader will
 * create {@link Zarafa.plugins.spreed.data.SpreedRecord SpreedRecord} instance.
 *
 */
Zarafa.plugins.spreed.data.SpreedStore = Ext.extend(Zarafa.core.data.MAPIStore, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 *
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			reader : new Zarafa.core.data.JsonReader({
				id            : 'id',
				idProperty    : 'id',
				dynamicRecord : false
			}, Zarafa.core.data.RecordFactory.getRecordClassByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED)),
			writer : new Zarafa.core.data.JsonWriter(),
			proxy  : new Zarafa.core.data.IPMProxy({
				listModuleName : 'spreedmodule',
				itemModuleName : 'spreedmodule'
			})
		});

		Zarafa.plugins.spreed.data.SpreedStore.superclass.constructor.call(this, config);
	},

	/**
	 * If we need to open records so we use the superclass's mechanism to send them to server and extract data
	 * Else we just create substores and set flag 'opened' on our record to true
	 *
	 * @param {Zarafa.plugins.spreed.data.SpreedRecord} record
	 * @param options
	 */
	open: function(record, options)
	{
		if (record.isNeededToOpen()) {
			Zarafa.plugins.spreed.data.SpreedStore.superclass.open.call(this, record, options);
		} else {
			record.afterOpen();
			this.fireEvent('open', this, record, record);
		}
	}

});

Ext.reg('spreed.spreedstore', Zarafa.plugins.spreed.data.SpreedStore);
Ext.namespace('Zarafa.plugins.spreed.data');

/**
* SpreedTimezones object
*/
Zarafa.plugins.spreed.data.SpreedTimezones = [
{
	value: 'Africa/Abidjan',
	name: _('Africa/Abidjan')+' (GMT 00:00)'
},{
	value: 'Africa/Accra',
	name: _('Africa/Accra')+' (GMT 00:00)'
},{
	value: 'Africa/Addis_Ababa',
	name: _('Africa/Addis Ababa')+' (GMT+03:00)'
},{
	value: 'Africa/Algiers',
	name: _('Africa/Algiers')+' (GMT+01:00)'
},{
	value: 'Africa/Asmera',
	name: _('Africa/Asmera')+' (GMT+03:00)'
},{
	value: 'Africa/Bamako',
	name: _('Africa/Bamako')+' (GMT 00:00)'
},{
	value: 'Africa/Bangui',
	name: _('Africa/Bangui')+' (GMT+01:00)'
},{
	value: 'Africa/Banjul',
	name: _('Africa/Banjul')+' (GMT 00:00)'
},{
	value: 'Africa/Bissau',
	name: _('Africa/Bissau')+' (GMT 00:00)'
},{
	value: 'Africa/Blantyre',
	name: _('Africa/Blantyre')+' (GMT+02:00)'
},{
	value: 'Africa/Brazzaville',
	name: _('Africa/Brazzaville')+' (GMT+01:00)'
},{
	value: 'Africa/Bujumbura',
	name: _('Africa/Bujumbura')+' (GMT+02:00)'
},{
	value: 'Africa/Cairo',
	name: _('Africa/Cairo')+' (GMT+02:00)'
},{
	value: 'Africa/Casablanca',
	name: _('Africa/Casablanca')+' (GMT 00:00)'
},{
	value: 'Africa/Ceuta',
	name: _('Africa/Ceuta')+' (GMT+01:00)'
},{
	value: 'Africa/Conakry',
	name: _('Africa/Conakry')+' (GMT 00:00)'
},{
	value: 'Africa/Dakar',
	name: _('Africa/Dakar')+' (GMT 00:00)'
},{
	value: 'Africa/Dar_es_Salaam',
	name: _('Africa/Dar es Salaam')+' (GMT+03:00)'
},{
	value: 'Africa/Djibouti',
	name: _('Africa/Djibouti')+' (GMT+03:00)'
},{
	value: 'Africa/Douala',
	name: _('Africa/Douala')+' (GMT+03:00)'
},{
	value: 'Africa/El_Aaiun',
	name: _('Africa/El Aaiun')+' (GMT+01:00)'
},{
	value: 'Africa/Freetown',
	name: _('Africa/Freetown')+' (GMT 00:00)'
},{
	value: 'Africa/Gaborone',
	name: _('Africa/Gaborone')+' (GMT+02:00)'
},{
	value: 'Africa/Harare',
	name: _('Africa/Harare')+' (GMT+02:00)'
},{
	value: 'Africa/Johannesburg',
	name: _('Africa/Johannesburg')+' (GMT+02:00)'
},{
	value: 'Africa/Kampala',
	name: _('Africa/Kampala')+' (GMT+03:00)'
},{
	value: 'Africa/Khartoum',
	name: _('Africa/Khartoum')+' (GMT+03:00)'
},{
	value: 'Africa/Kigali',
	name: _('Africa/Kigali')+' (GMT+02:00)'
},{
	value: 'Africa/Kinshasa',
	name: _('Africa/Kinshasa')+' (GMT+01:00)'
},{
	value: 'Africa/Lagos',
	name: _('Africa/Lagos')+' (GMT+01:00)'
},{
	value: 'Africa/Libreville',
	name: _('Africa/Libreville')+' (GMT+01:00)'
},{
	value: 'Africa/Lome',
	name: _('Africa/Lome')+' (GMT 00:00)'
},{
	value: 'Africa/Luanda',
	name: _('Africa/Luanda')+' (GMT+01:00)'
},{
	value: 'Africa/Lubumbashi',
	name: _('Africa/Lubumbashi')+' (GMT+02:00)'
},{
	value: 'Africa/Lusaka',
	name: _('Africa/Lusaka')+' (GMT+02:00)'
},{
	value: 'Africa/Malabo',
	name: _('Africa/Malabo')+' (GMT+01:00)'
},{
	value: 'Africa/Maputo',
	name: _('Africa/Maputo')+' (GMT+02:00)'
},{
	value: 'Africa/Maseru',
	name: _('Africa/Maseru')+' (GMT+02:00)'
},{
	value: 'Africa/Mbabane',
	name: _('Africa/Mbabane')+' (GMT+02:00)'
},{
	value: 'Africa/Mogadishu',
	name: _('Africa/Mogadishu')+' (GMT+03:00)'
},{
	value: 'Africa/Monrovia',
	name: _('Africa/Monrovia')+' (GMT 00:00)'
},{
	value: 'Africa/Nairobi',
	name: _('Africa/Nairobi')+' (GMT+03:00)'
},{
	value: 'Africa/Ndjamena',
	name: _('Africa/Ndjamena')+' (GMT+01:00)'
},{
	value: 'Africa/Niamey',
	name: _('Africa/Niamey')+' (GMT+01:00)'
},{
	value: 'Africa/Nouakchott',
	name: _('Africa/Nouakchott')+' (GMT 00:00)'
},{
	value: 'Africa/Ouagadougou',
	name: _('Africa/Ouagadougou')+' (GMT 00:00)'
},{
	value: 'Africa/Porto-Novo',
	name: _('Africa/Porto-Novo')+' (GMT+01:00)'
},{
	value: 'Africa/Sao_Tome',
	name: _('Africa/Sao Tome')+' (GMT 00:00)'
},{
	value: 'Africa/Tripoli',
	name: _('Africa/Tripoli')+' (GMT+02:00)'
},{
	value: 'Africa/Tunis',
	name: _('Africa/Tunis')+' (GMT+01:00)'
},{
	value: 'Africa/Windhoek',
	name: _('Africa/Windhoek')+' (GMT+01:00)'
},{
	value: 'America/Adak',
	name: _('America/Adak')+' (GMT-09:00)'
},{
	value: 'America/Anchorage',
	name: _('America/Anchorage')+' (GMT-09:00)'
},{
	value: 'America/Anguilla',
	name: _('America/Anguilla')+' (GMT-04:00)'
},{
	value: 'America/Antigua',
	name: _('America/Antigua')+' (GMT-04:00)'
},{
	value: 'America/Araguaina',
	name: _('America/Araguaina')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Buenos_Aires',
	name: _('America/Argentina/Buenos Aires')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Catamarca',
	name: _('America/Argentina/Catamarca')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Cordoba',
	name: _('America/Argentina/Cordoba')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Jujuy',
	name: _('America/Argentina/Jujuy')+' (GMT-03:00)'
},{
	value: 'America/Argentina/La_Rioja',
	name: _('America/Argentina/La Rioja')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Mendoza',
	name: _('America/Argentina/Mendoza')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Rio_Gallegos',
	name: _('America/Argentina/Rio Gallegos')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Salta',
	name: _('America/Argentina/Salta')+' (GMT-03:00)'
},{
	value: 'America/Argentina/San_Juan',
	name: _('America/Argentina/San Juan')+' (GMT-03:00)'
},{
	value: 'America/Argentina/San_Luis',
	name: _('America/Argentina/San Luis')+' (GMT-04:00)'
},{
	value: 'America/Argentina/Tucuman',
	name: _('America/Argentina/Tucuman')+' (GMT-03:00)'
},{
	value: 'America/Argentina/Ushuaia',
	name: _('America/Argentina/Ushuaia')+' (GMT-03:00)'
},{
	value: 'America/Aruba',
	name: _('America/Aruba')+' (GMT-04:00)'
},{
	value: 'America/Asuncion',
	name: _('America/Asuncion')+' (GMT-04:00)'
},{
	value: 'America/Atikokan',
	name: _('America/Atikokan')+' (GMT-05:00)'
},{
	value: 'America/Bahia',
	name: _('America/Bahia')+' (GMT-03:00)'
},{
	value: 'America/Bahia_Banderas',
	name: _('America/Bahia Banderas')+' (GMT-06:00)'
},{
	value: 'America/Barbados',
	name: _('America/Barbados')+' (GMT-04:00)'
},{
	value: 'America/Belem',
	name: _('America/Belem')+' (GMT-03:00)'
},{
	value: 'America/Belize',
	name: _('America/Belize')+' (GMT-06:00)'
},{
	value: 'America/Blanc-Sablon',
	name: _('America/Blanc-Sablon')+' (GMT-04:00)'
},{
	value: 'America/Boa_Vista',
	name: _('America/Boa Vista')+' (GMT-04:00)'
},{
	value: 'America/Bogota',
	name: _('America/Bogota')+' (GMT-05:00)'
},{
	value: 'America/Boise',
	name: _('America/Boise')+' (GMT-07:00)'
},{
	value: 'America/Cambridge_Bay',
	name: _('America/Cambridge Bay')+' (GMT-07:00)'
},{
	value: 'America/Campo_Grande',
	name: _('America/Campo Grande')+' (GMT-04:00)'
},{
	value: 'America/Cancun',
	name: _('America/Cancun')+' (GMT-06:00)'
},{
	value: 'America/Caracas',
	name: _('America/Caracas')+' (GMT-04:00)'
},{
	value: 'America/Cayenne',
	name: _('America/Cayenne')+' (GMT-03:00)'
},{
	value: 'America/Cayman',
	name: _('America/Cayman')+' (GMT-05:00)'
},{
	value: 'America/Chicago',
	name: _('America/Chicago')+' (GMT-06:00)'
},{
	value: 'America/Chihuahua',
	name: _('America/Chihuahua')+' (GMT-07:00)'
},{
	value: 'America/Costa_Rica',
	name: _('America/Costa Rica')+' (GMT-06:00)'
},{
	value: 'America/Cuiaba',
	name: _('America/Cuiaba')+' (GMT-04:00)'
},{
	value: 'America/Curacao',
	name: _('America/Curacao')+' (GMT-04:00)'
},{
	value: 'America/Danmarkshavn',
	name: _('America/Danmarkshavn')+' (GMT 00:00)'
},{
	value: 'America/Dawson',
	name: _('America/Dawson')+' (GMT-08:00)'
},{
	value: 'America/Dawson_Creek',
	name: _('America/Dawson Creek')+' (GMT-07:00)'
},{
	value: 'America/Denver',
	name: _('America/Denver')+' (GMT-05:00)'
},{
	value: 'America/Detroit',
	name: _('America/Detroit')+' (GMT-07:00)'
},{
	value: 'America/Dominica',
	name: _('America/Dominica')+' (GMT-04:00)'
},{
	value: 'America/Edmonton',
	name: _('America/Edmonton')+' (GMT-07:00)'
},{
	value: 'America/Eirunepe',
	name: _('America/Eirunepe')+' (GMT-04:00)'
},{
	value: 'America/El_Salvador',
	name: _('America/El Salvador')+' (GMT-06:00)'
},{
	value: 'America/Fortaleza',
	name: _('America/Fortaleza')+' (GMT-03:00)'
},{
	value: 'America/Glace_Bay',
	name: _('America/Glace Bay')+' (GMT-04:00)'
},{
	value: 'America/Godthab',
	name: _('America/Godthab')+' (GMT-03:00)'
},{
	value: 'America/Goose_Bay',
	name: _('America/Goose Bay')+' (GMT-04:00)'
},{
	value: 'America/Grand_Turk',
	name: _('America/Grand Turk')+' (GMT-05:00)'
},{
	value: 'America/Grenada',
	name: _('America/Grenada')+' (GMT-04:00)'
},{
	value: 'America/Guadeloupe',
	name: _('America/Guadeloupe')+' (GMT-04:00)'
},{
	value: 'America/Guatemala',
	name: _('America/Guatemala')+' (GMT-06:00)'
},{
	value: 'America/Guayaquil',
	name: _('America/Guayaquil')+' (GMT-05:00)'
},{
	value: 'America/Guyana',
	name: _('America/Guyana')+' (GMT-04:00)'
},{
	value: 'America/Halifax',
	name: _('America/Halifax')+' (GMT-04:00)'
},{
	value: 'America/Havana',
	name: _('America/Havana')+' (GMT-05:00)'
},{
	value: 'America/Hermosillo',
	name: _('America/Hermosillo')+' (GMT-07:00)'
},{
	value: 'America/Indiana/Indianapolis',
	name: _('America/Indiana/Indianapolis')+' (GMT-05:00)'
},{
	value: 'America/Indiana/Knox',
	name: _('America/Indiana/Knox')+' (GMT-05:00)'
},{
	value: 'America/Indiana/Marengo',
	name: _('America/Indiana/Marengo')+' (GMT-05:00)'
},{
	value: 'America/Indiana/Petersburg',
	name: _('America/Indiana/Petersburg')+' (GMT-04:00)'
},{
	value: 'America/Indiana/Tell_City',
	name: _('America/Indiana/Tell City')+' (GMT-05:00)'
},{
	value: 'America/Indiana/Vevay',
	name: _('America/Indiana/Vevay')+' (GMT-05:00)'
},{
	value: 'America/Indiana/Vincennes',
	name: _('America/Indiana/Vincennes')+' (GMT-05:00)'
},{
	value: 'America/Indiana/Winamac',
	name: _('America/Indiana/Winamac')+' (GMT-05:00)'
},{
	value: 'America/Inuvik',
	name: _('America/Inuvik')+' (GMT-07:00)'
},{
	value: 'America/Iqaluit',
	name: _('America/Iqaluit')+' (GMT-05:00)'
},{
	value: 'America/Jamaica',
	name: _('America/Jamaica')+' (GMT-05:00)'
},{
	value: 'America/Juneau',
	name: _('America/Juneau')+' (GMT-09:00)'
},{
	value: 'America/Kentucky/Louisville',
	name: _('America/Kentucky/Louisville')+' (GMT-05:00)'
},{
	value: 'America/Kentucky/Monticello',
	name: _('America/Kentucky/Monticello')+' (GMT-05:00)'
},{
	value: 'America/La_Paz',
	name: _('America/La Paz')+' (GMT-04:00)'
},{
	value: 'America/Lima',
	name: _('America/Lima')+' (GMT-05:00)'
},{
	value: 'America/Los_Angeles',
	name: _('America/Los Angeles')+' (GMT-08:00)'
},{
	value: 'America/Maceio',
	name: _('America/Maceio')+' (GMT-03:00)'
},{
	value: 'America/Managua',
	name: _('America/Managua')+' (GMT-06:00)'
},{
	value: 'America/Manaus',
	name: _('America/Manaus')+' (GMT-04:00)'
},{
	value: 'America/Marigot',
	name: _('America/Marigot')+' (GMT-04:00)'
},{
	value: 'America/Martinique',
	name: _('America/Martinique')+' (GMT-04:00)'
},{
	value: 'America/Matamoros',
	name: _('America/Matamoros')+' (GMT-06:00)'
},{
	value: 'America/Mazatlan',
	name: _('America/Mazatlan')+' (GMT-07:00)'
},{
	value: 'America/Menominee',
	name: _('America/Menominee')+' (GMT-06:00)'
},{
	value: 'America/Merida',
	name: _('America/Merida')+' (GMT-06:00)'
},{
	value: 'America/Metlakatla',
	name: _('America/Metlakatla')+' (GMT-08:00)'
},{
	value: 'America/Mexico_City',
	name: _('America/Mexico City')+' (GMT-06:00)'
},{
	value: 'America/Miquelon',
	name: _('America/Miquelon')+' (GMT-03:00)'
},{
	value: 'America/Moncton',
	name: _('America/Moncton')+' (GMT-04:00)'
},{
	value: 'America/Monterrey',
	name: _('America/Monterrey')+' (GMT-06:00)'
},{
	value: 'America/Montevideo',
	name: _('America/Montevideo')+' (GMT-03:00)'
},{
	value: 'America/Montreal',
	name: _('America/Montreal')+' (GMT-05:00)'
},{
	value: 'America/Montserrat',
	name: _('America/Montserrat')+' (GMT-04:00)'
},{
	value: 'America/Nassau',
	name: _('America/Nassau')+' (GMT-05:00)'
},{
	value: 'America/New_York',
	name: _('America/New York')+' (GMT-05:00)'
},{
	value: 'America/Nipigon',
	name: _('America/Nipigon')+' (GMT-05:00)'
},{
	value: 'America/Nome',
	name: _('America/Nome')+' (GMT-09:00)'
},{
	value: 'America/Noronha',
	name: _('America/Noronha')+' (GMT-02:00)'
},{
	value: 'America/North_Dakota/Beulah',
	name: _('America/North Dakota/Beulah')+' (GMT-06:00)'
},{
	value: 'America/North_Dakota/Center',
	name: _('America/North Dakota/Center')+' (GMT-06:00)'
},{
	value: 'America/North_Dakota/New_Salem',
	name: _('America/North Dakota/New Salem')+' (GMT-06:00)'
},{
	value: 'America/Ojinaga',
	name: _('America/Ojinaga')+' (GMT-07:00)'
},{
	value: 'America/Panama',
	name: _('America/Panama')+' (GMT-05:00)'
},{
	value: 'America/Pangnirtung',
	name: _('America/Pangnirtung')+' (GMT-03:00)'
},{
	value: 'America/Paramaribo',
	name: _('America/Paramaribo')+' (GMT-03:00)'
},{
	value: 'America/Phoenix',
	name: _('America/Phoenix')+' (GMT-07:00)'
},{
	value: 'America/Port-au-Prince',
	name: _('America/Port-au-Prince')+' (GMT-05:00)'
},{
	value: 'America/Port_of_Spain',
	name: _('America/Port of Spain')+' (GMT-04:00)'
},{
	value: 'America/Porto_Velho',
	name: _('America/Porto Velho')+' (GMT-04:00)'
},{
	value: 'America/Puerto_Rico',
	name: _('America/Puerto Rico')+' (GMT-04:00)'
},{
	value: 'America/Rainy_River',
	name: _('America/Rainy River')+' (GMT-06:00)'
},{
	value: 'America/Rankin_Inlet',
	name: _('America/Rankin Inlet')+' (GMT-06:00)'
},{
	value: 'America/Recife',
	name: _('America/Recife')+' (GMT-03:00)'
},{
	value: 'America/Regina',
	name: _('America/Regina')+' (GMT-06:00)'
},{
	value: 'America/Resolute',
	name: _('America/Resolute')+' (GMT-05:00)'
},{
	value: 'America/Rio_Branco',
	name: _('America/Rio Branco')+' (GMT-05:00)'
},{
	value: 'America/Santa_Isabel',
	name: _('America/Santa Isabel')+' (GMT-08:00)'
},{
	value: 'America/Santarem',
	name: _('America/Santarem')+' (GMT-03:00)'
},{
	value: 'America/Santiago',
	name: _('America/Santiago')+' (GMT-04:00)'
},{
	value: 'America/Santo_Domingo',
	name: _('America/Santo Domingo')+' (GMT-04:00)'
},{
	value: 'America/Sao_Paulo',
	name: _('America/Sao Paulo')+' (GMT-03:00)'
},{
	value: 'America/Scoresbysund',
	name: _('America/Scoresbysund')+' (GMT-01:00)'
},{
	value: 'America/Shiprock',
	name: _('America/Shiprock')+' (GMT-07:00)'
},{
	value: 'America/Sitka',
	name: _('America/Sitka')+' (GMT-09:00)'
},{
	value: 'America/St_Barthelemy',
	name: _('America/St Barthelemy')+' (GMT-04:00)'
},{
	value: 'America/St_Johns',
	name: _('America/St Johns')+' (GMT-03:30)'
},{
	value: 'America/St_Kitts',
	name: _('America/St Kitts')+' (GMT-04:00)'
},{
	value: 'America/St_Lucia',
	name: _('America/St Lucia')+' (GMT-04:00)'
},{
	value: 'America/St_Thomas',
	name: _('America/St Thomas')+' (GMT-04:00)'
},{
	value: 'America/St_Vincent',
	name: _('America/St Vincent')+' (GMT-04:00)'
},{
	value: 'America/Swift_Current',
	name: _('America/Swift Current')+' (GMT-06:00)'
},{
	value: 'America/Tegucigalpa',
	name: _('America/Tegucigalpa')+' (GMT-06:00)'
},{
	value: 'America/Thule',
	name: _('America/Thule')+' (GMT-04:00)'
},{
	value: 'America/Thunder_Bay',
	name: _('America/Thunder Bay')+' (GMT-05:00)'
},{
	value: 'America/Tijuana',
	name: _('America/Tijuana')+' (GMT-08:00)'
},{
	value: 'America/Toronto',
	name: _('America/Toronto')+' (GMT-04:00)'
},{
	value: 'America/Tortola',
	name: _('America/Tortola')+' (GMT-04:00)'
},{
	value: 'America/Vancouver',
	name: _('America/Vancouver')+' (GMT-08:00)'
},{
	value: 'America/Whitehorse',
	name: _('America/Whitehorse')+' (GMT-08:00)'
},{
	value: 'America/Winnipeg',
	name: _('America/Winnipeg')+' (GMT-06:00)'
},{
	value: 'America/Yakutat',
	name: _('America/Yakutat')+' (GMT-09:00)'
},{
	value: 'America/Yellowknife',
	name: _('America/Yellowknife')+' (GMT-07:00)'
},{
	value: 'Antarctica/Casey',
	name: _('Antarctica/Casey')+' (GMT+08:00)'
},{
	value: 'Antarctica/Davis',
	name: _('Antarctica/Davis')+' (GMT+07:00)'
},{
	value: 'Antarctica/DumontDUrville',
	name: _('Antarctica/DumontDUrville')+' (GMT+10:00)'
},{
	value: 'Antarctica/Macquarie',
	name: _('Antarctica/Macquarie')+' (GMT+10:00)'
},{
	value: 'Antarctica/Mawson',
	name: _('Antarctica/Mawson')+' (GMT+06:00)'
},{
	value: 'Antarctica/McMurdo',
	name: _('Antarctica/McMurdo')+' (GMT+12:00)'
},{
	value: 'Antarctica/Palmer',
	name: _('Antarctica/Palmer')+' (GMT-04:00)'
},{
	value: 'Antarctica/Rothera',
	name: _('Antarctica/Rothera')+' (GMT-04:00)'
},{
	value: 'Antarctica/South_Pole',
	name: _('Antarctica/South Pole')+' (GMT+12:00)'
},{
	value: 'Antarctica/Syowa',
	name: _('Antarctica/Syowa')+' (GMT+03:00)'
},{
	value: 'Antarctica/Vostok',
	name: _('Antarctica/Vostok')+' (GMT+06:00)'
},{
	value: 'Arctic/Longyearbyen',
	name: _('Arctic/Longyearbyen')+' (GMT+01:00)'
},{
	value: 'Asia/Aden',
	name: _('Asia/Aden')+' (GMT+03:00)'
},{
	value: 'Asia/Almaty',
	name: _('Asia/Almaty')+' (GMT+06:00)'
},{
	value: 'Asia/Amman',
	name: _('Asia/Amman')+' (GMT+02:00)'
},{
	value: 'Asia/Anadyr',
	name: _('Asia/Anadyr')+' (GMT+12:00)'
},{
	value: 'Asia/Aqtau',
	name: _('Asia/Aqtau')+' (GMT+04:00)'
},{
	value: 'Asia/Aqtobe',
	name: _('Asia/Aqtobe')+' (GMT+05:00)'
},{
	value: 'Asia/Ashgabat',
	name: _('Asia/Ashgabat')+' (GMT+05:00)'
},{
	value: 'Asia/Baghdad',
	name: _('Asia/Baghdad')+' (GMT+03:00)'
},{
	value: 'Asia/Bahrain',
	name: _('Asia/Bahrain')+' (GMT+03:00)'
},{
	value: 'Asia/Baku',
	name: _('Asia/Baku')+' (GMT+04:00)'
},{
	value: 'Asia/Bangkok',
	name: _('Asia/Bangkok')+' (GMT+07:00)'
},{
	value: 'Asia/Beirut',
	name: _('Asia/Beirut')+' (GMT+02:00)'
},{
	value: 'Asia/Bishkek',
	name: _('Asia/Bishkek')+' (GMT+05:00)'
},{
	value: 'Asia/Brunei',
	name: _('Asia/Brunei')+' (GMT+08:00)'
},{
	value: 'Asia/Choibalsan',
	name: _('Asia/Choibalsan')+' (GMT+08:00)'
},{
	value: 'Asia/Chongqing',
	name: _('Asia/Chongqing')+' (GMT+08:00)'
},{
	value: 'Asia/Colombo',
	name: _('Asia/Colombo')+' (GMT+05:30)'
},{
	value: 'Asia/Damascus',
	name: _('Asia/Damascus')+' (GMT+02:00)'
},{
	value: 'Asia/Dhaka',
	name: _('Asia/Dhaka')+' (GMT+06:00)'
},{
	value: 'Asia/Dili',
	name: _('Asia/Dili')+' (GMT+09:00)'
},{
	value: 'Asia/Dubai',
	name: _('Asia/Dubai')+' (GMT+04:00)'
},{
	value: 'Asia/Dushanbe',
	name: _('Asia/Dushanbe')+' (GMT+05:00)'
},{
	value: 'Asia/Gaza',
	name: _('Asia/Gaza')+' (GMT+02:00)'
},{
	value: 'Asia/Harbin',
	name: _('Asia/Harbin')+' (GMT+08:00)'
},{
	value: 'Asia/Ho_Chi_Minh',
	name: _('Asia/Ho Chi Minh')+' (GMT+07:00)'
},{
	value: 'Asia/Hong_Kong',
	name: _('Asia/Hong Kong')+' (GMT+08:00)'
},{
	value: 'Asia/Hovd',
	name: _('Asia/Hovd')+' (GMT+07:00)'
},{
	value: 'Asia/Irkutsk',
	name: _('Asia/Irkutsk')+' (GMT+08:00)'
},{
	value: 'Asia/Jakarta',
	name: _('Asia/Jakarta')+' (GMT+07:00)'
},{
	value: 'Asia/Jayapura',
	name: _('Asia/Jayapura')+' (GMT+09:00)'
},{
	value: 'Asia/Jerusalem',
	name: _('Asia/Jerusalem')+' (GMT+02:00)'
},{
	value: 'Asia/Kabul',
	name: _('Asia/Kabul')+' (GMT+04:30)'
},{
	value: 'Asia/Kamchatka',
	name: _('Asia/Kamchatka')+' (GMT+12:00)'
},{
	value: 'Asia/Karachi',
	name: _('Asia/Karachi')+' (GMT+05:00)'
},{
	value: 'Asia/Kashgar',
	name: _('Asia/Kashgar')+' (GMT+08:00)'
},{
	value: 'Asia/Kathmandu',
	name: _('Asia/Kathmandu')+' (GMT+05:45)'
},{
	value: 'Asia/Kolkata',
	name: _('Asia/Kolkata')+' (GMT+05:30)'
},{
	value: 'Asia/Krasnoyarsk',
	name: _('Asia/Krasnoyarsk')+' (GMT+07:00)'
},{
	value: 'Asia/Kuala_Lumpur',
	name: _('Asia/Kuala Lumpur')+' (GMT+08:00)'
},{
	value: 'Asia/Kuching',
	name: _('Asia/Kuching')+' (GMT+08:00)'
},{
	value: 'Asia/Kuwait',
	name: _('Asia/Kuwait')+' (GMT+03:00)'
},{
	value: 'Asia/Macau',
	name: _('Asia/Macau')+' (GMT+08:00)'
},{
	value: 'Asia/Magadan',
	name: _('Asia/Magadan')+' (GMT+11:00)'
},{
	value: 'Asia/Makassar',
	name: _('Asia/Makassar')+' (GMT+08:00)'
},{
	value: 'Asia/Manila',
	name: _('Asia/Manila')+' (GMT+08:00)'
},{
	value: 'Asia/Muscat',
	name: _('Asia/Muscat')+' (GMT+04:00)'
},{
	value: 'Asia/Nicosia',
	name: _('Asia/Nicosia')+' (GMT+02:00)'
},{
	value: 'Asia/Novokuznetsk',
	name: _('Asia/Novokuznetsk')+' (GMT+07:00)'
},{
	value: 'Asia/Novosibirsk',
	name: _('Asia/Novosibirsk')+' (GMT+06:00)'
},{
	value: 'Asia/Omsk',
	name: _('Asia/Omsk')+' (GMT+06:00)'
},{
	value: 'Asia/Oral',
	name: _('Asia/Oral')+' (GMT+05:00)'
},{
	value: 'Asia/Phnom_Penh',
	name: _('Asia/Phnom Penh')+' (GMT+07:00)'
},{
	value: 'Asia/Pontianak',
	name: _('Asia/Pontianak')+' (GMT+07:00)'
},{
	value: 'Asia/Pyongyang',
	name: _('Asia/Pyongyang')+' (GMT+09:00)'
},{
	value: 'Asia/Qatar',
	name: _('Asia/Qatar')+' (GMT+03:00)'
},{
	value: 'Asia/Qyzylorda',
	name: _('Asia/Qyzylorda')+' (GMT+06:00)'
},{
	value: 'Asia/Rangoon',
	name: _('Asia/Rangoon')+' (GMT+06:30)'
},{
	value: 'Asia/Riyadh',
	name: _('Asia/Riyadh')+' (GMT+03:00)'
},{
	value: 'Asia/Sakhalin',
	name: _('Asia/Sakhalin')+' (GMT+10:00)'
},{
	value: 'Asia/Samarkand',
	name: _('Asia/Samarkand')+' (GMT+05:00)'
},{
	value: 'Asia/Seoul',
	name: _('Asia/Seoul')+' (GMT+09:00)'
},{
	value: 'Asia/Shanghai',
	name: _('Asia/Shanghai')+' (GMT+08:00)'
},{
	value: 'Asia/Singapore',
	name: _('Asia/Singapore')+' (GMT+08:00)'
},{
	value: 'Asia/Taipei',
	name: _('Asia/Taipei')+' (GMT+08:00)'
},{
	value: 'Asia/Tashkent',
	name: _('Asia/Tashkent')+' (GMT+05:00)'
},{
	value: 'Asia/Tbilisi',
	name: _('Asia/Tbilisi')+' (GMT+03:00)'
},{
	value: 'Asia/Tehran',
	name: _('Asia/Tehran')+' (GMT+03:30)'
},{
	value: 'Asia/Thimphu',
	name: _('Asia/Thimphu')+' (GMT+06:00)'
},{
	value: 'Asia/Tokyo',
	name: _('Asia/Tokyo')+' (GMT+09:00)'
},{
	value: 'Asia/Ulaanbaatar',
	name: _('Asia/Ulaanbaatar')+' (GMT+08:00)'
},{
	value: 'Asia/Urumqi',
	name: _('Asia/Urumqi')+' (GMT+08:00)'
},{
	value: 'Asia/Vientiane',
	name: _('Asia/Vientiane')+' (GMT+07:00)'
},{
	value: 'Asia/Vladivostok',
	name: _('Asia/Vladivostok')+' (GMT+10:00)'
},{
	value: 'Asia/Yakutsk',
	name: _('Asia/Yakutsk')+' (GMT+09:00)'
},{
	value: 'Asia/Yekaterinburg',
	name: _('Asia/Yekaterinburg')+' (GMT+05:00)'
},{
	value: 'Asia/Yerevan',
	name: _('Asia/Yerevan')+' (GMT+04:00)'
},{
	value: 'Atlantic/Azores',
	name: _('Atlantic/Azores')+' (GMT-01:00)'
},{
	value: 'Atlantic/Bermuda',
	name: _('Atlantic/Bermuda')+' (GMT-04:00)'
},{
	value: 'Atlantic/Canary',
	name: _('Atlantic/Canary')+' (GMT 00:00)'
},{
	value: 'Atlantic/Cape_Verde',
	name: _('Atlantic/Cape Verde')+' (GMT-01:00)'
},{
	value: 'Atlantic/Faroe',
	name: _('Atlantic/Faroe')+' (GMT 00:00)'
},{
	value: 'Atlantic/Madeira',
	name: _('Atlantic/Madeira')+' (GMT 00:00)'
},{
	value: 'Atlantic/Reykjavik',
	name: _('Atlantic/Reykjavik')+' (GMT 00:00)'
},{
	value: 'Atlantic/South_Georgia',
	name: _('Atlantic/South Georgia')+' (GMT-02:00)'
},{
	value: 'Atlantic/St_Helena',
	name: _('Atlantic/St Helena')+' (GMT 00:00)'
},{
	value: 'Atlantic/Stanley',
	name: _('Atlantic/Stanley')+' (GMT-04:00)'
},{
	value: 'Australia/Adelaide',
	name: _('Australia/Adelaide')+' (GMT+09:30)'
},{
	value: 'Australia/Brisbane',
	name: _('Australia/Brisbane')+' (GMT+10:00)'
},{
	value: 'Australia/Broken_Hill',
	name: _('Australia/Broken Hill')+' (GMT+09:30)'
},{
	value: 'Australia/Currie',
	name: _('Australia/Currie')+' (GMT+10:00)'
},{
	value: 'Australia/Darwin',
	name: _('Australia/Darwin')+' (GMT+09:30)'
},{
	value: 'Australia/Eucla',
	name: _('Australia/Eucla')+' (GMT+08:45)'
},{
	value: 'Australia/Hobart',
	name: _('Australia/Hobart')+' (GMT+10:00)'
},{
	value: 'Australia/Lindeman',
	name: _('Australia/Lindeman')+' (GMT+10:00)'
},{
	value: 'Australia/Lord_Howe',
	name: _('Australia/Lord Howe')+' (GMT+10:30)'
},{
	value: 'Australia/Melbourne',
	name: _('Australia/Melbourne')+' (GMT+10:00)'
},{
	value: 'Australia/Perth',
	name: _('Australia/Perth')+' (GMT+08:00)'
},{
	value: 'Australia/Sydney',
	name: _('Australia/Sydney')+' (GMT+10:00)'
},{
	value: 'Europe/Amsterdam',
	name: _('Europe/Amsterdam')+' (GMT+01:00)'
},{
	value: 'Europe/Andorra',
	name: _('Europe/Andorra')+' (GMT+01:00)'
},{
	value: 'Europe/Athens',
	name: _('Europe/Athens')+' (GMT+02:00)'
},{
	value: 'Europe/Belgrade',
	name: _('Europe/Belgrade')+' (GMT+02:00)'
},{
	value: 'Europe/Berlin',
	name: _('Europe/Berlin')+' (GMT+01:00)'
},{
	value: 'Europe/Bratislava',
	name: _('Europe/Bratislava')+' (GMT+02:00)'
},{
	value: 'Europe/Brussels',
	name: _('Europe/Brussels')+' (GMT+01:00)'
},{
	value: 'Europe/Bucharest',
	name: _('Europe/Bucharest')+' (GMT+02:00)'
},{
	value: 'Europe/Budapest',
	name: _('Europe/Budapest')+' (GMT+01:00)'
},{
	value: 'Europe/Chisinau',
	name: _('Europe/Chisinau')+' (GMT+02:00)'
},{
	value: 'Europe/Copenhagen',
	name: _('Europe/Copenhagen')+' (GMT+01:00)'
},{
	value: 'Europe/Dublin',
	name: _('Europe/Dublin')+' (GMT 00:00)'
},{
	value: 'Europe/Gibraltar',
	name: _('Europe/Gibraltar')+' (GMT 00:00)'
},{
	value: 'Europe/Guernsey',
	name: _('Europe/Guernsey')+' (GMT 00:00)'
},{
	value: 'Europe/Helsinki',
	name: _('Europe/Helsinki')+' (GMT+02:00)'
},{
	value: 'Europe/Isle_of_Man',
	name: _('Europe/Isle of Man')+' (GMT 00:00)'
},{
	value: 'Europe/Istanbul',
	name: _('Europe/Istanbul')+' (GMT+02:00)'
},{
	value: 'Europe/Jersey',
	name: _('Europe/Jersey')+' (GMT 00:00)'
},{
	value: 'Europe/Kaliningrad',
	name: _('Europe/Kaliningrad')+' (GMT+02:00)'
},{
	value: 'Europe/Kiev',
	name: _('Europe/Kiev')+' (GMT+02:00)'
},{
	value: 'Europe/Lisbon',
	name: _('Europe/Lisbon')+' (GMT 00:00)'
},{
	value: 'Europe/Ljubljana',
	name: _('Europe/Ljubljana')+' (GMT+01:00)'
},{
	value: 'Europe/London',
	name: _('Europe/London')+' (GMT 00:00)'
},{
	value: 'Europe/Luxembourg',
	name: _('Europe/Luxembourg')+' (GMT+01:00)'
},{
	value: 'Europe/Madrid',
	name: _('Europe/Madrid')+' (GMT+01:00)'
},{
	value: 'Europe/Malta',
	name: _('Europe/Malta')+' (GMT+01:00)'
},{
	value: 'Europe/Mariehamn',
	name: _('Europe/Mariehamn')+' (GMT+02:00)'
},{
	value: 'Europe/Minsk',
	name: _('Europe/Minsk')+' (GMT+02:00)'
},{
	value: 'Europe/Monaco',
	name: _('Europe/Monaco')+' (GMT+01:00)'
},{
	value: 'Europe/Moscow',
	name: _('Europe/Moscow')+' (GMT+04:00)'
},{
	value: 'Europe/Oslo',
	name: _('Europe/Oslo')+' (GMT+01:00)'
},{
	value: 'Europe/Paris',
	name: _('Europe/Paris')+' (GMT+01:00)'
},{
	value: 'Europe/Podgorica',
	name: _('Europe/Podgorica')+' (GMT+01:00)'
},{
	value: 'Europe/Prague',
	name: _('Europe/Prague')+' (GMT+01:00)'
},{
	value: 'Europe/Riga',
	name: _('Europe/Riga')+' (GMT+02:00)'
},{
	value: 'Europe/Rome',
	name: _('Europe/Rome')+' (GMT+01:00)'
},{
	value: 'Europe/Samara',
	name: _('Europe/Samara')+' (GMT+04:00)'
},{
	value: 'Europe/San_Marino',
	name: _('Europe/San Marino')+' (GMT+01:00)'
},{
	value: 'Europe/Sarajevo',
	name: _('Europe/Sarajevo')+' (GMT+01:00)'
},{
	value: 'Europe/Simferopol',
	name: _('Europe/Simferopol')+' (GMT+02:00)'
},{
	value: 'Europe/Skopje',
	name: _('Europe/Skopje')+' (GMT+01:00)'
},{
	value: 'Europe/Sofia',
	name: _('Europe/Sofia')+' (GMT+02:00)'
},{
	value: 'Europe/Stockholm',
	name: _('Europe/Stockholm')+' (GMT+01:00)'
},{
	value: 'Europe/Tallinn',
	name: _('Europe/Tallinn')+' (GMT+02:00)'
},{
	value: 'Europe/Tirane',
	name: _('Europe/Tirane')+' (GMT+01:00)'
},{
	value: 'Europe/Uzhgorod',
	name: _('Europe/Uzhgorod')+' (GMT+02:00)'
},{
	value: 'Europe/Vaduz',
	name: _('Europe/Vaduz')+' (GMT+01:00)'
},{
	value: 'Europe/Vatican',
	name: _('Europe/Vatican')+' (GMT+01:00)'
},{
	value: 'Europe/Vienna',
	name: _('Europe/Vienna')+' (GMT+01:00)'
},{
	value: 'Europe/Vilnius',
	name: _('Europe/Vilnius')+' (GMT+02:00)'
},{
	value: 'Europe/Volgograd',
	name: _('Europe/Volgograd')+' (GMT+03:00)'
},{
	value: 'Europe/Warsaw',
	name: _('Europe/Warsaw')+' (GMT+01:00)'
},{
	value: 'Europe/Zagreb',
	name: _('Europe/Zagreb')+' (GMT+01:00)'
},{
	value: 'Europe/Zaporozhye',
	name: _('Europe/Zaporozhye')+' (GMT+02:00)'
},{
	value: 'Europe/Zurich',
	name: _('Europe/Zurich')+' (GMT+01:00)'
},{
	value: 'Indian/Antananarivo',
	name: _('Indian/Antananarivo')+' (GMT+03:00)'
},{
	value: 'Indian/Chagos',
	name: _('Indian/Chagos')+' (GMT+06:00)'
},{
	value: 'Indian/Christmas',
	name: _('Indian/Christmas')+' (GMT+07:00)'
},{
	value: 'Indian/Cocos',
	name: _('Indian/Cocos')+' (GMT+06:30)'
},{
	value: 'Indian/Comoro',
	name: _('Indian/Comoro')+' (GMT+03:00)'
},{
	value: 'Indian/Kerguelen',
	name: _('Indian/Kerguelen')+' (GMT+05:00)'
},{
	value: 'Indian/Mahe',
	name: _('Indian/Mahe')+' (GMT+04:00)'
},{
	value: 'Indian/Maldives',
	name: _('Indian/Maldives')+' (GMT+05:00)'
},{
	value: 'Indian/Mauritius',
	name: _('Indian/Mauritius')+' (GMT+04:00)'
},{
	value: 'Indian/Mayotte',
	name: _('Indian/Mayotte')+' (GMT+03:00)'
},{
	value: 'Indian/Reunion',
	name: _('Indian/Reunion')+' (GMT+04:00)'
},{
	value: 'Pacific/Apia',
	name: _('Pacific/Apia')+' (GMT-11:00)'
},{
	value: 'Pacific/Auckland',
	name: _('Pacific/Auckland')+' (GMT+12:00)'
},{
	value: 'Pacific/Chatham',
	name: _('Pacific/Chatham')+' (GMT+12:45)'
},{
	value: 'Pacific/Chuuk',
	name: _('Pacific/Chuuk')+' (GMT+10:00)'
},{
	value: 'Pacific/Easter',
	name: _('Pacific/Easter')+' (GMT-06:00)'
},{
	value: 'Pacific/Efate',
	name: _('Pacific/Efate')+' (GMT+11:00)'
},{
	value: 'Pacific/Enderbury',
	name: _('Pacific/Enderbury')+' (GMT+13:00)'
},{
	value: 'Pacific/Fakaofo',
	name: _('Pacific/Fakaofo')+' (GMT-10:00)'
},{
	value: 'Pacific/Fiji',
	name: _('Pacific/Fiji')+' (GMT+12:00)'
},{
	value: 'Pacific/Funafuti',
	name: _('Pacific/Funafuti')+' (GMT+12:00)'
},{
	value: 'Pacific/Galapagos',
	name: _('Pacific/Galapagos')+' (GMT-06:00)'
},{
	value: 'Pacific/Gambier',
	name: _('Pacific/Gambier')+' (GMT-09:00)'
},{
	value: 'Pacific/Guadalcanal',
	name: _('Pacific/Guadalcanal')+' (GMT+11:00)'
},{
	value: 'Pacific/Guam',
	name: _('Pacific/Guam')+' (GMT+10:00)'
},{
	value: 'Pacific/Honolulu',
	name: _('Pacific/Honolulu')+' (GMT-10:00)'
},{
	value: 'Pacific/Johnston',
	name: _('Pacific/Johnston')+' (GMT-10:00)'
},{
	value: 'Pacific/Kiritimati',
	name: _('Pacific/Kiritimati')+' (GMT+14:00)'
},{
	value: 'Pacific/Kosrae',
	name: _('Pacific/Kosrae')+' (GMT+11:00)'
},{
	value: 'Pacific/Kwajalein',
	name: _('Pacific/Kwajalein')+' (GMT+12:00)'
},{
	value: 'Pacific/Majuro',
	name: _('Pacific/Majuro')+' (GMT+12:00)'
},{
	value: 'Pacific/Marquesas',
	name: _('Pacific/Marquesas')+' (GMT-09:30)'
},{
	value: 'Pacific/Midway',
	name: _('Pacific/Midway')+' (GMT-11:00)'
},{
	value: 'Pacific/Nauru',
	name: _('Pacific/Nauru')+' (GMT+12:00)'
},{
	value: 'Pacific/Niue',
	name: _('Pacific/Niue')+' (GMT-11:00)'
},{
	value: 'Pacific/Norfolk',
	name: _('Pacific/Norfolk')+' (GMT+11:30)'
},{
	value: 'Pacific/Noumea',
	name: _('Pacific/Noumea')+' (GMT+11:00)'
},{
	value: 'Pacific/Pago_Pago',
	name: _('Pacific/Pago Pago')+' (GMT-11:00)'
},{
	value: 'Pacific/Palau',
	name: _('Pacific/Palau')+' (GMT+09:00)'
},{
	value: 'Pacific/Pitcairn',
	name: _('Pacific/Pitcairn')+' (GMT-08:00)'
},{
	value: 'Pacific/Pohnpei',
	name: _('Pacific/Pohnpei')+' (GMT+11:00)'
},{
	value: 'Pacific/Port_Moresby',
	name: _('Pacific/Port Moresby')+' (GMT+10:00)'
},{
	value: 'Pacific/Rarotonga',
	name: _('Pacific/Rarotonga')+' (GMT-10:00)'
},{
	value: 'Pacific/Saipan',
	name: _('Pacific/Saipan')+' (GMT+10:00)'
},{
	value: 'Pacific/Tahiti',
	name: _('Pacific/Tahiti')+' (GMT-10:00)'
},{
	value: 'Pacific/Tarawa',
	name: _('Pacific/Tarawa')+' (GMT+12:00)'
},{
	value: 'Pacific/Tongatapu',
	name: _('Pacific/Tongatapu')+' (GMT+13:00)'
},{
	value: 'Pacific/Wake',
	name: _('Pacific/Wake')+' (GMT+12:00)'
},{
	value: 'Pacific/Wallis',
	name: _('Pacific/Wallis')+' (GMT+12:00)'
}];Ext.namespace('Zarafa.plugins.spreed.dialogs');

/**
 * @class Zarafa.plugins.spreed.dialogs.EditSpreedParticipantContentPanel
 * @extends Zarafa.core.ui.ContentPanel
 * @xtype spreed.spreededitparticipantcontentpanel
 *
 * This content panel allows editing of a recipient in spreed meeting.
 */
Zarafa.plugins.spreed.dialogs.EditSpreedParticipantContentPanel = Ext.extend(Zarafa.core.ui.ContentPanel, {
	/**
	 * @cfg {Zarafa.plugins.spreed.data.SpreedParticipantRecord} record The participant which
	 * is being edited by this panel.
	 */
	record : undefined,

	/**
	 * The form panel which is loaded inside this panel.
	 * @property
	 * @type Ext.form.FormPanel
	 */
	formPanel : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		var timezonesStore = new Ext.data.JsonStore({
			xtype   : 'jsonstore',
			fields  : ['name', 'value'],
			data    : Zarafa.plugins.spreed.data.SpreedTimezones
		});
		timezonesStore.sort('name', 'ASC');

		var languageStore = new Ext.data.JsonStore({
			xtype : 'jsonstore',
			fields : ['name', 'value'],
			data: Zarafa.plugins.spreed.data.SpreedLanguages,
			sortInfo: {
				field: 'name',
				direction: 'ASC'
			}
		});

		var displayName=config.record.get('display_name');
		var email=config.record.get('smtp_address');
		var moderatorStatus=config.record.get('isModerator');
		var language=config.record.get('language');
		var timezone=config.record.get('timezone');

		Ext.applyIf(config, {
			title  : _('Edit Spreed Participant'),
			layout : 'fit',
			width  : 400,
			height : 180,
			items  : [{
				xtype  : 'form',
				layout : 'form',
				border : false,
				ref    : 'formPanel',
				items  : [
					{
						xtype      : 'textfield',
						fieldLabel : _('Display Name'),
						name       : 'display_name',
						value      : displayName,
						anchor     : '100%'
					},

					{
						xtype      : 'textfield',
						fieldLabel : _('Email Address'),
						name       : 'smtp_address',
						value      : email,
						anchor     : '100%'
					},

					{
						xtype      : 'checkbox',
						name       : 'isModerator',
						fieldLabel : _('Is Moderator'),
						checked    : moderatorStatus,
						value      : moderatorStatus,
						anchor     : '100%'
					},

					{
						xtype           : 'combo',
						name            : 'timezone',
						width           : 150,
						store           : timezonesStore,
						value           : timezone,
						mode            : 'local',
						displayField    : 'name',
						valueField      : 'value',
						triggerAction   : 'all',
						forceSelection  : true,
						editable        : false,
						lazyInit        : false,
						fieldLabel      : _('Timezone'),
						anchor     : '100%'
					},

					{
						xtype		: 'combo',
						name		: 'language',
						value		: language,
						fieldLabel	: _('Language'),
						width		: 200,
						store		: languageStore,
						mode		: 'local',
						triggerAction: 'all',
						displayField: 'name',
						valueField	: 'value',
						lazyInit	: false,
						forceSelection: true,
						editable	: false,
						autoSelect	: true,
						anchor		: '100%'
					}


				],
				buttons: [{
					text: _('Ok'),
					handler: this.onOk,
					scope: this
				},
					{
						text: _('Cancel'),
						handler: this.onCancel,
						scope: this
					}
				]
			}]
		});
		Zarafa.plugins.spreed.dialogs.EditSpreedParticipantContentPanel.superclass.constructor.call(this, config);
		this.mon(this, 'afterlayout', this.onAfterFirstLayout, this, { single: true });
	},

	/**
	 * Event handler which is fired when {@link #afterlayout} has been called for the first time.
	 * This will load the {@link #record} into {@link #formPanel}.
	 * @private
	 */
	onAfterFirstLayout : function()
	{
		this.formPanel.getForm().loadRecord(this.record);
	},

	/**
	 * Event handler which is raised when the user clicks the "Ok" {@link Ext.Button button}
	 * @private
	 */
	onOk : function()
	{
		this.formPanel.getForm().updateRecord(this.record);
		this.close();
	},

	/**
	 * Event handler which is raised when the user clicks the "Cancel" {@link Ext.Button button}
	 * @private
	 */
	onCancel : function()
	{
		this.close();
	}
});

Ext.reg('spreed.spreededitparticipantcontentpanel', Zarafa.plugins.spreed.dialogs.EditSpreedParticipantContentPanel);
Ext.namespace('Zarafa.plugins.spreed.dialogs');
/**
* @class Zarafa.plugins.spreed.dialogs.SpreedParticipantBox
* @extends Zarafa.common.recipientfield.ui.RecipientBox
* @xtype zarafa.spreedparticipantbox
*
* Extension to the  {@link Zarafa.common.recipientfield.ui.RecipientBox}.
* This box offers adding moderator icon for the moderator participant.
*/
Zarafa.plugins.spreed.dialogs.SpreedParticipantBox = Ext.extend(Zarafa.common.recipientfield.ui.RecipientBox, {

	/**
	 * @constructor
	 * @param config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
		});
		Zarafa.plugins.spreed.dialogs.SpreedParticipantBox.superclass.constructor.call(this, config);
	},

	/**
	 * Overriden function to provide custom icon rendering for the given {@link Ext.data.Record}
	 * to the {@link #iconEl} element. The string returned here is the CSS class which will be set on the
	 * {@link #iconEl}.
	 * @param {Ext.data.Record} record The record which is going to be rendered
	 * @return {String} The CSS class which must be applied to the {@link #iconEl}.
	 * @private
	 */
	prepareIcon : function(record)
	{
		if (record.get('isModerator'))
		{
			return 'icon_spreed_mod';
		}
		else
		{
			this.iconEl.removeClass('icon_spreed_mod');
			return Zarafa.plugins.spreed.dialogs.SpreedParticipantBox.superclass.prepareIcon.apply(this, arguments);
		}
	},

	/**
	 * Overriden function to provide custom formatting for the given {@link Ext.data.Record}
	 * to the {@link #update} function. The data object returned here is used by the {@link #textTpl template}
	 * to render the contents of the box.
	 * @param {Ext.data.Record} record The record which is going to be rendered
	 * @return {Object} The data object which can be passed to {@link #textTpl}.
	 * @private
	 */
	prepareData: function(record)
	{
		var prepared = Zarafa.plugins.spreed.dialogs.SpreedParticipantBox.superclass.prepareData.apply(this, arguments);
		prepared.display_name = record.get('display_name');
		prepared.smtp_address = record.get('smtp_address') || record.get('email_address');
		prepared.object_type = record.get('object_type');
		prepared.isModerator = record.get('isModerator');
		return prepared;
	},

	/**
	 * Overriden function checks if the given {Zarafa.core.data.MAPIRecord} record
	 * has an object type {Zarafa.core.mapi.ObjectType.MAPI_DISTLIST} if true
	 * record is marked as invalid.
	 *
	 * @param {Ext.data.Record} record The record to check if it is of distribution list type of record.
	 * @private
	 */
	isValidRecord: function (record)
	{
		if (record.get('object_type')==Zarafa.core.mapi.ObjectType.MAPI_DISTLIST) {
			return false;
		}
		return Zarafa.plugins.spreed.dialogs.SpreedParticipantBox.superclass.isValidRecord.call(this, record);
	}
});
Ext.reg('spreed.spreedparticipantbox',Zarafa.plugins.spreed.dialogs.SpreedParticipantBox);Ext.namespace('Zarafa.plugins.spreed.dialogs');

/**
* @class Zarafa.plugins.spreed.dialogs.SpreedParticipantContextMenu
* @extends Zarafa.common.recipientfield.ui.RecipientContextMenu
* @xtype spreed.spreedrecipientcontextmenu
*/
Zarafa.plugins.spreed.dialogs.SpreedParticipantContextMenu = Ext.extend(Zarafa.common.recipientfield.ui.RecipientContextMenu, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			items: [{
				xtype   : 'zarafa.conditionalitem',
				text    : _('Edit Participant'),
				handler : this.editRecipient,
				scope   : this
			},{
				xtype   : 'zarafa.conditionalitem',
				text    : _('Email Message'),
				handler : this.emailRecipient,
				scope   : this
			},{
				xtype   : 'menuseparator'
			}
			]
		});
		Zarafa.plugins.spreed.dialogs.SpreedParticipantContextMenu.superclass.constructor.call(this, config);
   }
});

Ext.reg('spreed.spreedparticipantcontextmenu', Zarafa.plugins.spreed.dialogs.SpreedParticipantContextMenu);
Ext.namespace('Zarafa.plugins.spreed.dialogs');

/**
 * @class Zarafa.plugins.spreed.dialogs.SpreedParticipantField
 * @extends Zarafa.common.recipientfield.ui.RecipientField
 * @xtype spreed.spreedparticipantfield
 *
 * If the {@link Zarafa.core.plugins.RecordComponentUpdaterPlugin} is installed
 * in the {@link #plugins} array of this component, this component will automatically
 * load the {@link Zarafa.core.data.IPMRecipientStore RecipientStore} into the component.
 * Otherwise the user of this component needs to call {@link #setRecipientStore}.
 */
Zarafa.plugins.spreed.dialogs.SpreedParticipantField = Ext.extend(Zarafa.common.recipientfield.ui.RecipientField, {
	/**
	 * @constructor
	 * @param config{Object} Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};
		Ext.applyIf(config, {
			boxType : 'spreed.spreedparticipantbox'
		});

		Zarafa.plugins.spreed.dialogs.SpreedParticipantField.superclass.constructor.call(this, config);
	},

	/**
	 * Called to handle a selection from the dropdown list. This function needs to
	 * convert the selected record into a record for the {@link #spreedStore}.
	 * @param record {Object} record to handle
	 */
	handleSelection : function(record)
	{
		var participant = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT, {
			object_type    : record.get('object_type'),
			display_name   : record.get('display_name'),
			email_address  : record.get('email_address'),
			smtp_address   : record.get('smtp_address')||record.get('email_address'),
			address_type   : record.get('address_type'),
			isModerator    : record.get('isModerator'),
			recipient_type : record.get('recipientType') || Zarafa.core.mapi.RecipientType.MAPI_TO
		});
		this.boxStore.add(participant);
	},

	/**
	 * Event handler when a Box has been double-clicked.
	 * @param {Zarafa.common.recipientfield.ui.RecipientField} field This field to which the box belongs
	 * @param {Zarafa.common.recipientfield.ui.RecipientBox} box The box for which was double-clicked
	 * @param {Zarafa.core.data.IPMRecipientRecord} record The record which is attached to the box
	 * @private
	 */
	onBoxDblClick : function(field, box, record)
	{
		Zarafa.core.data.UIFactory.openCreateRecord(record, { modal : true });
	}
});
Ext.reg('spreed.spreedparticipantfield', Zarafa.plugins.spreed.dialogs.SpreedParticipantField);
//Ext.namespace('Zarafa.plugins.spreed.dialogs');
//
//Zarafa.plugins.spreed.dialogs.SpreedPlaceHolder=Ext.extend(Ext.Component, {
//
//	/**
//	 * @constructor
//	 * @param {Object} config Configuration object
//	 */
//	constructor : function(config)
//	{
//		config = config || {};
//		Ext.applyIf(config,
//			{
//				cls: 'spreed-possible-participant',
//				html: 'Participant 1',
////				data: 'Participant 1',
//				renderTo: 'my-test-div'
//			}
//
//		);
//
//		Zarafa.plugins.spreed.dialogs.SpreedPlaceHolder.superclass.constructor.call(this, config);
//		this.init()
//	},
//
//	init:function()
//	{
//		console.log(this);
//	}
//
//});
//Ext.reg('spreed.spreedplaceholder', Zarafa.plugins.spreed.dialogs.SpreedPlaceHolder);Ext.namespace('Zarafa.plugins.spreed.dialogs');

/**
 * @class Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel
 * @extends Zarafa.core.ui.ContentPanel
 * @xtype spreed.spreedsetupcontentpanel
 *
 * Content panel with options for creating web meeting with spreed.
 */
Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel = Ext.extend(Zarafa.core.ui.RecordContentPanel, {

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		// Add in some standard configuration data.
		config = config || {};

		Ext.applyIf(config, {
			layout : 'fit',
			title : _('Spreed Meeting'),
			recordComponentPluginConfig : Ext.applyIf(config.recordComponentPluginConfig || {}, {
				allowWrite : true,
				useShadowStore : false,
				enableOpenLoadTask : false
			}),
			closeOnSave : true,
			width : 940,
			height : 480,
			//Add panel
			items : [{
				xtype : 'spreed.spreedsetuppanel',
				tbar : {
					xtype: 'spreed.spreedtoolbar'
				}
			}]
		});

		//Call parent constructor
		Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Called when the {@link #close} event is being fired.
	 * This will clean up the store, by removing current record from it.
	 * Calls parent method after this.
	 */
	doClose : function()
	{
		var store = this.record.getStore();
		store.remove(this.record);

		Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel.superclass.doClose.call(this);
	}

});

Ext.reg('spreed.spreedsetupcontentpanel', Zarafa.plugins.spreed.dialogs.SpreedSetupContentPanel);
Ext.namespace('Zarafa.plugins.spreed.dialogs');

/**
 * @class Zarafa.plugins.spreed.dialogs.SpreedSetupPanel
 * @extends Ext.Panel
 * Main panel of spreed setup dialog.
 *
 * @xtype spreed.spreedsetuppanel
 */
Zarafa.plugins.spreed.dialogs.SpreedSetupPanel = Ext.extend(Ext.form.FormPanel, {

	/**
	 * Number of boxes after adding which
	 * the warning must be shown.
	 *
	 * @property
	 * @type Integer
	 * @private
	 */
	maxParticipants : 2,

	/**
	 * The spreed record with initial date to render in
	 * timeperiod field.
	 * @property
	 * @type Zarafa.plugins.spreed.data.SpreedRecord
	 * @private
	 */
	record:undefined,

	/**
	 * The Spreed warning message used to check if rendred.
	 * @property
	 * @private
	 */
	warning: undefined,
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 *
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');
		this.maxParticipants = container.getSettingsModel().get('zarafa/v1/plugins/spreed/max_participants');
		Ext.apply(config, {
			xtype     : 'spreed.spreedsetuppanel',
			layout    : {
				type  : 'vbox',
				align : 'stretch'
			},
			bodyStyle : 'background-color: inherit;',
			defaults  : {
				border      : true,
				bodyStyle   : 'background-color: inherit; padding: 3px 0px 3px 0px; border-style: none none solid none;'
			},
			items     : [
				this.createParticipantPanel(),
				this.createSubjectPanel(),
				this.createTimePanel(),
				this.createAttachmentsPanel(),
				this.createDescriptionPanel()
			]
		});
		//Call parent constructor
		Zarafa.plugins.spreed.dialogs.SpreedSetupPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Creates the participant panel {@link Ext.Panel panel} with Participant field.
	 * @return {Object} Configuration object for the participant panel
	 * @private
	 */
	createParticipantPanel:function()
	{
		return [{
			xtype      : 'panel',
			layout     : 'form',
			autoHeight : true,
			items      : [{
				xtype       : 'zarafa.compositefield',
				anchor      : '100%',
				hideLabel   : true,
				items       : [{
					xtype       : 'button',
					width       : 100,
					text        : _('Participants')+':',
					handler     : this.addParticipantsFromAddressBook,
					scope       : this
				},{
					xtype       : 'spreed.spreedparticipantfield',
					plugins     : [ 'zarafa.recordcomponentupdaterplugin' ],
					ref	    : '../../spreedparticipantfield',
					flex        : 1
				}]
			},{
				xtype       : 'displayfield',
				anchor      : '100%',
				autoShow    : true,
				ref         : '../spreed_warning',
				cls         : 'spreed-Warning',
				hidden      : true,
				value       : String.format(_('You have too many participants selected. You are allowed to add {0} additional participants to the web meeting.'), this.maxParticipants)
			}]
		}]
	},

	/**
	 * Creates the subject panel {@link Ext.Panel panel} with subject field.
	 * @return {Object} Configuration object for the subject panel
	 * @private
	 */
	createSubjectPanel:function()
	{
		return {
			xtype      : 'panel',
			layout     : 'form',
			autoHeight : false,
			items      : [{
				xtype      : 'textfield',
				fieldLabel : _('Subject'),
				name       : 'subject',
				labelWidth : 100,
				anchor     : '100%',
				listeners  : {
					scope		 : this,
					change		 : this.onFieldChange
				}
			}]
		}
	},

	/**
	 * Creates the Panel with time range fields.
	 * @return configuration object for form panel
	 * @private
	 */
	createTimePanel : function()
	{
		var timezonesStore = new Ext.data.JsonStore({
			xtype   : 'jsonstore',
			fields  : ['name', 'value'],
			data    : Zarafa.plugins.spreed.data.SpreedTimezones,
			sortInfo: {
				field: 'name',
				direction: 'ASC'
			}
		});

		var duration = container.getSettingsModel().get('zarafa/v1/contexts/calendar/default_appointment_period');
		//delay of spreed meeting based on default reminder time from calendar settings
		var delay = container.getSettingsModel().get('zarafa/v1/contexts/calendar/default_zoom_level');
		var startTime = new Date().ceil(Date.MINUTE, delay);

		//we need default value to render correctly start time, when changing the value of timeperiod field
		var defValue = new Zarafa.core.DateRange({
			startDate	: startTime,
			dueDate		: startTime.add(Date.MINUTE, duration)
		});

		var timepanel= {
			xtype       : 'panel',
			layout      : 'hbox',
			items       : [{
				xtype		: 'zarafa.timeperiodfield',
				timeIncrement	: delay,
				defaultValue	: defValue,
				name		: 'timeperiod',
				defaultPeriod	: duration,
				defaultPeriodType   : Date.MINUTE,
				ref	  	: '../toTime',
				listeners 	: {
					scope : this,
					change : this.onTimePeriodFieldChange
				},
				spacerConfig        : {
					width : 5
				},
				startFieldConfig    : {
					name  		: 'start_time',
					width 		: 80,
					minValue	: new Date().clearTime(),
					maxValue	: new Date().clearTime().add(Date.DAY, 1)
				},
				endFieldConfig      : {
					name  		: 'end_time',
					minValue	: new Date().clearTime(),
					maxValue	: new Date().clearTime().add(Date.DAY, 1),
					width : 80
				},
				width: 220
			},{
				xtype           : 'combo',
				name            : 'timezone',
				plugins		: [ 'zarafa.fieldlabeler' ],
				fieldLabel	: _('Timezone'),
				width           : 275,
				store           : timezonesStore,
				value           : container.getSettingsModel().get('zarafa/v1/plugins/spreed/default_timezone'),
				mode            : 'local',
				displayField    : 'name',
				valueField      : 'value',
				triggerAction   : 'all',
				forceSelection  : true,
				editable        : false,
				lazyInit        : false,
				listeners : {
					scope : this,
					select : this.onTimezoneFieldSelect
				}
			}]
		};
		return timepanel;
	},

	/**
	 * Creates the attachment panel {@link Ext.Panel panel} with attachment field.
	 * @return {Object} Configuration object for the attachment panel.
	 * @private
	 */
	createAttachmentsPanel:function()
	{
		return  {
			xtype   : 'panel',
			layout  : 'fit',
			border  : true,
			items   : [{
				xtype     : 'zarafa.compositefield',
				hideLabel : true,
				anchor    : '100%',
				autoHeight: true,
				items     : [{
					xtype     : 'zarafa.attachmentbutton',
					width     : 100,
					text      : _('Attachments') + ':',
					plugins   : [ 'zarafa.recordcomponentupdaterplugin' ]
				},{
					xtype    : 'zarafa.attachmentfield',
					boxtype  : 'zarafa.attachmentbox',
					plugins   : [ 'zarafa.recordcomponentupdaterplugin' ],
					flex     : 1,
					hideLabel: true,
					value    : undefined
				}]
			}]
		}
	},

	/**
	 * Creates the description panel {@link Ext.Panel panel} with description field.
	 * @return {Object} Configuration object for the description panel.
	 * @private
	 */
	createDescriptionPanel:function()
	{
		return {
			xtype   : 'panel',
			flex: 1,
			layout    : 'fit',
			border  : false,
			items   : [{
				xtype      : 'textarea',
				autoScroll : true,
				hideLabel : true,
				name      : 'description',
				listeners : {
					scope : this,
					change : this.onFieldChange
				}
			}]
		}
	},

	/**
	 * Load record into spreed setup panel
	 * and update the ui timeperiod field with
	 * values taken from record.
	 *
	 * @param {Zarafa.core.data.IPMRecord} record The record to load
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update : function(record, contentReset)
	{
		if (record.isOpened()) {
			//warning abbout too many participants
			this.tryShowingWarning(record);
		}

		this.record = record;
		this.getForm().loadRecord(record);

		var newStartTime = record.get('start_time').toUTC();
		var newEndTime = record.get('end_time').toUTC();

		this.toTime.startField.setValue(newStartTime);
		this.toTime.endField.setValue(newEndTime);
	},

	/**
	 * Called before record will be saved to server.
	 * Collect the data from the form fields and update the record
	 * with fields values.
	 *
	 * @param {Zarafa.core.data.IPMRecord} record to update
	 * @private
	 */
	updateRecord : function(record)
	{
		record.beginEdit();

		this.getForm().updateRecord(record);

		//startTime and endTime got from the dialog
		var startTime =this.toTime.getValue().getStartDate();
		var endTime = this.toTime.getValue().getDueDate();

		// Convert to a Date object where the getTime() returns a timestamp that has the same time as displayed to the user
		startTime = startTime.fromUTC();
		endTime = endTime.fromUTC();

		record.set('start_time', startTime);
		record.set('end_time', endTime);

		record.endEdit();
	},

	/**
	 * Event handler which is triggered when one of the Input fields
	 * has been changed by the user. It will
	 * apply the new value to the {@link Zarafa.plugins.spreed.data.SpreedRecord record}.
	 * @param {Ext.form.Field} field The {@link Ext.form.Field field} which was changed.
	 * @param {Mixed} newValue The new value
	 * @param {Mixed} oldValue The old value
	 * @private
	 */
	onFieldChange : function(field, newValue, oldValue)
	{
		var record = this.record;
		record.set(field.getName(), newValue);
	},

	/**
	 * The event hander for timeperiod field.
	 * It will apply the new value to the {@link Zarafa.plugins.spreed.data.SpreedRecord record}.
	 * @param {Ext.form.Field} field The {@link Ext.form.Field field} which was changed.
	 * @param {Mixed} newValue The new value
	 * @param {Mixed} oldValue The old value
	 * @private
	 */
	onTimePeriodFieldChange: function(field, newValue, oldValue)
	{
		var record = this.record;
		record.beginEdit();
		record.set('start_time', newValue.getStartDate().fromUTC());
		record.set('end_time', newValue.getDueDate().fromUTC());
		record.endEdit();
	},

	/**
	 * The event hander for timezone field.
	 * @param {Ext.form.ComboBox} combo - this combobox
	 * @param {Ext.data.Record} record - the data record returned from the underlying store
	 * @param {Number} index - the index of the selected item in the dropdown list
	 * @private
	 */
	onTimezoneFieldSelect:function(combo, record, index)
	{
		var spreedRecord = this.record;
		spreedRecord.set('timezone', record.get(combo.valueField));

		if (spreedRecord.isOpened()) {
			var subStoreRecipients = spreedRecord.getSubStore('recipients');
			subStoreRecipients.each(function(recipient) {
				//this ensures that all recipients which don't have an alternative timezone,
				// are in the set for the same timezone as the meeting request.
				if (Ext.isEmpty(recipient.get('timezone')) || recipient.get('timezone') == spreedRecord.modified.timezone) {
					//setting the general meeting timezone for each recipient
					recipient.set('timezone', spreedRecord.get('timezone'));
				}
			}, this);
		}
	},

	/*
	 * Creates a dialog from shared component addressbook.dialog.recipientselection
	 * to add participants from address book for spreed meeting dialog
	 *
	 * @private
	 */
	addParticipantsFromAddressBook: function()
	{
		// Create a copy of the record, we don't want the changes
		// to be activated until the user presses the Ok button.
		var copy = this.record.copy();
		var store = copy.getSubStore('recipients');

		Zarafa.common.Actions.openABUserMultiSelectionContent({
			callback : function() {
				this.record.applyData(copy);
			},
			scope : this,
			store : store,
			listRestriction :{
				hide_groups : true,
				hide_distlist: true
			},
			convert : function(user, field) {
				if (user.get('object_type') != Zarafa.core.mapi.ObjectType.MAPI_DISTLIST) {
					return Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_SPREED_PARTICIPANT, {
						entryid : user.get('entryid'),
						object_type : user.get('object_type'),
						display_name : user.get('display_name'),
						display_type : user.get('display_type'),
						display_type_ex : user.get('display_type_ex'),
						smtp_address : user.get('smtp_address') || user.get('email_address'),
						address_type : user.get('address_type'),
						recipient_type : field ? field.defaultRecipientType :Zarafa.core.mapi.RecipientType.MAPI_TO
					});
				}
			},
			selectionCfg : [{
				xtype : 'zarafa.recipientfield',
				fieldLabel : _('Participants') + ':',
				height : 50,
				boxStore : store,
				filterRecipientType: Zarafa.core.mapi.RecipientType.MAPI_TO,
				defaultRecipientType: Zarafa.core.mapi.RecipientType.MAPI_TO,
				flex : 1
			}]
		});
	},

	/**
	 * Function to check if we need to show warning about the number of participants.
	 * @param {Zarafa.plugins.spreed.data.SpreedRecord} record - spreedrecord to check
	 * for number of participants
	 * @private
	 */
	tryShowingWarning:function(record)
	{
		if (record.getRecipientStore().getCount() > this.maxParticipants) {
			if (this.spreed_warning.hidden) {
				this.spreed_warning.show();
				this.doLayout();
				this.spreed_warning.getEl().fadeIn({
					block    : true,
					endOpacity : 1,
					easing   : 'easeIn',
					duration : 1
				});
			}
		} else if (!this.spreed_warning.hidden) {
			this.spreed_warning.getEl().fadeOut({
				easing     : 'easeOut',
				endOpacity : 1,
				duration   : 1,
				scope      : this,
				callback   : function() {
					this.spreed_warning.hide();
					this.doLayout();
				}
			});
		}
	}
});

Ext.reg('spreed.spreedsetuppanel', Zarafa.plugins.spreed.dialogs.SpreedSetupPanel);
Ext.namespace('Zarafa.plugins.spreed.dialogs');

/**
 * @class Zarafa.plugins.spreed.dialogs.SpreedToolBar
 * @extends Zarafa.core.ui.ContentPanelToolbar
 * @xtype zarafa.spreedtoolbar
 */
Zarafa.plugins.spreed.dialogs.SpreedToolBar = Ext.extend(Zarafa.core.ui.ContentPanelToolbar, {

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			insertionPointBase: 'plugins.spreed.spreedcontentpanel',
			actionItems : this.createActionButtons()
		});

		Zarafa.plugins.spreed.dialogs.SpreedToolBar.superclass.constructor.call(this, config);
	},

	/**
	 * Create all buttons which should be added by default the the `Actions` {@link Ext.ButtonGroup ButtonGroup}.
	 * This will create {@link Ext.ButtonGroup ButtonGroup} element with Spreed setup button.
	 *
	 * @return {Array} The {@link Ext.ButtonGroup ButtonGroup} elements which should be added
	 * in the Actions section of the {@link Ext.Toolbar Toolbar}.
	 * @private
	 */
	createActionButtons : function()
	{
		return [{
				xtype   : 'button',
				text    : _('Setup Meeting'),
				tooltip : {
					title   : _('Setup Meeting'),
					text    : _('Setup meeting with provided details')
				},
				iconCls : 'icon_spreed_setup_button',
				handler : this.setUpMeeting,
				scope   : this
			}];
	},

	/**
	 * Call the saveRecord on the parent content panel.
	 * Close the panel when the Spreed Meeting is setup ( record is created ) successfully.
	 *
	 * @return void
	 */
	setUpMeeting    : function()
	{
		var currentDialogWindow = this.dialog;
		currentDialogWindow.saveRecord();
	}
});

Ext.reg('spreed.spreedtoolbar', Zarafa.plugins.spreed.dialogs.SpreedToolBar);
