# List Handling
use strict;
use warnings;
our (%gui, %vmc, %signal, %prefs);

sub fill_list_vminfo() {
    my $vhost = &vhost();
    $gui{liststoreInfo}->clear();
    my $gref = &getsel_list_guest();
    &addrow_log("Fetching more details of $$gref{Name}...");
    my $IVRDEServer = IMachine_getVRDEServer($$gref{IMachine});
    my @IStorageController = IMachine_getStorageControllers($$gref{IMachine});
    my $IAudioAdapter = IMachine_getAudioAdapter($$gref{IMachine});
    my @IUSBController = IMachine_getUSBControllers($$gref{IMachine});
    &addrow_info(0, 'GENERAL', 2, 800);
    &addrow_info(0, 'Name:', 1, $$gref{Name});
    &addrow_info(0, 'OS Type:', 1, $$gref{Os});
    &addrow_info(0, 'SYSTEM', 2, 800);
    &addrow_info(0, 'Firmware:', 1, IMachine_getFirmwareType($$gref{IMachine}));
    my $mem = IMachine_getMemorySize($$gref{IMachine});
    $mem = ($mem > 1023) ? sprintf("%.2f GB", $mem / 1024) : "$mem MB";
    &addrow_info(0, 'Base Memory:', 1, $mem);
    &addrow_info(0, 'Processor(s):', 1, IMachine_getCPUCount($$gref{IMachine}));
    my $bootorder = '';

    foreach (1..4) {
        my $bdev = IMachine_getBootOrder($$gref{IMachine}, $_);
       $bootorder .= "$bdev  " if ($bdev ne 'Null');
    }
 
    $bootorder ? &addrow_info(0, 'Boot Order:', 1, $bootorder) : &addrow_info(0, 'Boot Order:', 1, '<No boot devices enabled>');
    my $vtx = '';
    $vtx .= 'VT-x/AMD-V  ' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'Enabled') eq 'true');
    $vtx .= 'VPID  ' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'VPID') eq 'true');
    $vtx .= 'PAE/NX  ' if (IMachine_getCPUProperty($$gref{IMachine}, 'PAE') eq 'true');
    $vtx .= 'Nested Paging  ' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'NestedPaging') eq 'true');
    $vtx ? &addrow_info(0, 'Acceleration:', 1, $vtx) : &addrow_info(0, 'Acceleration:', 1, '<No acceleration enabled>');
    &addrow_info(0, 'DISPLAY', 2, 800);
    &addrow_info(0, 'Video Memory:', 1, IMachine_getVRAMSize($$gref{IMachine}) . ' MB');
    &addrow_info(0, 'Screens: ', 1, IMachine_getMonitorCount($$gref{IMachine}));
    my $vidaccel = '';
    $vidaccel .= '2D Video  ' if (IMachine_getAccelerate2DVideoEnabled($$gref{IMachine}) eq 'true');
    $vidaccel .= '3D  ' if (IMachine_getAccelerate3DEnabled($$gref{IMachine}) eq 'true');
    $vidaccel ? &addrow_info(0, 'Display Acceleration:', 1, $vidaccel) : &addrow_info(0, 'Display Acceleration:', 1, '<No video acceleration enabled>');
    IVRDEServer_getEnabled($IVRDEServer) eq 'true' ? &addrow_info(0, 'Remote Display Port(s):', 1, IVRDEServer_getVRDEProperty($IVRDEServer, 'TCP/Ports'))
                                                   : &addrow_info(0, 'Remote Display Port(s):', 1, '<RDP server disabled>');
    &addrow_info(0, 'STORAGE', 2, 800);

    foreach my $controller (@IStorageController) {
        my $controllername = IStorageController_getName($controller);
        &addrow_info(0, 'Controller:', 1, $controllername);
        my @IMediumAttachment = IMachine_getMediumAttachmentsOfController($$gref{IMachine}, $controllername);
        foreach my $attachment (@IMediumAttachment) {
            if ($$attachment{medium}) {
                IMedium_refreshState($$attachment{medium}); # Needed to bring in current sizes
                # Use the base medium for information purposes
                my $size = &bytesToX(IMedium_getLogicalSize($$attachment{medium}));
                &addrow_info(0, "  Port $$attachment{port}:", 1, IMedium_getName(IMedium_getBase($$attachment{medium})) . " ($$attachment{type}, $size)");
            }
        }
    }

    &addrow_info(0, 'AUDIO', 2, 800);
    IAudioAdapter_getEnabled($IAudioAdapter) eq 'true' ? (&addrow_info(0, 'Host Driver:', 1, IAudioAdapter_getAudioDriver($IAudioAdapter))
                                                       and &addrow_info(0, 'Controller:', 1, IAudioAdapter_getAudioController($IAudioAdapter)))
                                                       : &addrow_info(0, '<Audio Disabled>');
    &addrow_info(0, 'NETWORK', 2, 800);

    foreach (0..($$vhost{maxnet}-1)) {
        my $INetworkAdapter = IMachine_getNetworkAdapter($$gref{IMachine}, $_);

        if (INetworkAdapter_getEnabled($INetworkAdapter) eq 'true') {
            my $attachtype = INetworkAdapter_getAttachmentType($INetworkAdapter);
            my $adapter = INetworkAdapter_getAdapterType($INetworkAdapter) . ' (' . $attachtype;

            if ($attachtype eq 'Bridged') { $adapter .= ', ' . INetworkAdapter_getBridgedInterface($INetworkAdapter); }
            elsif ($attachtype eq 'HostOnly') { $adapter .= ', ' . INetworkAdapter_getHostOnlyInterface($INetworkAdapter); }
            elsif ($attachtype eq 'Internal') { $adapter .= ', ' . INetworkAdapter_getInternalNetwork($INetworkAdapter); }

            $adapter .= ')';
            &addrow_info(0, "Adapter $_:", 1, $adapter);
        }
    }

    &addrow_info(0, 'SERIAL PORTS', 2, 800);

    foreach (0..($$vhost{maxser}-1)) {
        my $ISerialPort = IMachine_getSerialPort($$gref{IMachine}, $_);
        ISerialPort_getEnabled($ISerialPort) eq 'true' ? &addrow_info(0, "Port #$_:", 1, 'Enabled  ' . 
                                                                                         ISerialPort_getHostMode($ISerialPort) . '  ' .
                                                                                         ISerialPort_getPath($ISerialPort))
                                                       : &addrow_info(0, "Port #$_:", 1, 'Disabled');
    }

    &addrow_info(0, 'PARALLEL PORT', 2, 800);
    my $IParallelPort = IMachine_getParallelPort($$gref{IMachine}, 0);
    IParallelPort_getEnabled($IParallelPort) eq 'true' ? &addrow_info(0, 'LPT:', 1, 'Enabled  ' . IParallelPort_getPath($IParallelPort))
                                                       : &addrow_info(0, 'Disabled');

    &addrow_info(0, 'USB', 2, 800);

    if (@IUSBController) {
        foreach my $usbcontroller (@IUSBController) {
            my $usbver = IUSBController_getUSBStandard($usbcontroller);
            my $major = int(IUSBController_getUSBStandard($usbcontroller) / 256);
            my $minor = IUSBController_getUSBStandard($usbcontroller) % 256;
            &addrow_info(0, 'Controller:',
                         1, IUSBController_getName($usbcontroller) . 
                            ' (' . IUSBController_getType($usbcontroller) . ', ' .
                            'USB v' . int($usbver/256) . '.' . $usbver % 256 . ')');
        }

        my $IUSBDeviceFilters = IMachine_getUSBDeviceFilters($$gref{IMachine});
        my @filters = IUSBDeviceFilters_getDeviceFilters($IUSBDeviceFilters);
        my $active = 0;
        foreach (@filters) { $active++ if (IUSBDeviceFilter_getActive($_) eq 'true'); }
        &addrow_info(0, 'Device Filters:', 1,  scalar(@filters) . " ($active active)");
    }
    else { &addrow_info(0, 'None'); }

    my @sf = IMachine_getSharedFolders($$gref{IMachine});
    &addrow_info(0, 'SHARED FOLDERS', 2, 800);
    &addrow_info(0, 'Shared Folders:', 1, scalar(@sf));

    my $sref = &get_session($$gref{IMachine});

    if ($$sref{Lock} eq 'Shared') {
        &addrow_info(0, 'RUNTIME INFORMATION', 2, 800);
        my $IGuest = IConsole_getGuest(ISession_getConsole($$sref{ISession}));
        &addrow_info(0, 'Operating System:', 1, IGuest_getOSTypeId($IGuest));
        my $additionsversion = IGuest_getAdditionsVersion($IGuest);
        if ($additionsversion) { &addrow_info(0, 'Guest Additions:', 1, $additionsversion); }
        else { &addrow_info(0, 'Guest Additions:', 1, 'Not Installed (or have not loaded yet)'); }
    }

    &addrow_log('More details complete.');
    ISession_unlockMachine($$sref{ISession}) if (ISession_getState($$sref{ISession}) eq 'Locked');
}

# Fills the remote file chooser with a list of files. Involves a lot of splicing because
# of the crazy way VirtualBox returns a file list
sub fill_list_remotefiles() {
    &busy_pointer($gui{dialogRemoteFileChooser}, 1);
    my ($location, $filter) = @_;
    my $vhost = &vhost();
    $location = &rcanonpath($location);
    my $IProgress = IVFSExplorer_cd($gui{IVFSExplorer}, $location);
    IProgress_waitForCompletion($IProgress);

    if (&bl(IProgress_getCompleted($IProgress)) and (IProgress_getResultCode($IProgress) == 0)) { # Only update the view if the CD is successful.
        $gui{liststoreRemoteFileChooser}->clear();
        IVFSExplorer_update($gui{IVFSExplorer});
        my @entries = IVFSExplorer_entryList($gui{IVFSExplorer});
        my $chop = (@entries / 4);
        my @filenames = splice @entries, 0, $chop;
        my @types = splice @entries, 0, $chop;
        my @sizes = splice @entries, 0, $chop;
        my @modes = splice @entries, 0, $chop;
        my %files;

        foreach my $ent (0..$#filenames) {
            $files{$filenames[$ent]}{type} = $types[$ent];
            $files{$filenames[$ent]}{size} = $sizes[$ent];
            $files{$filenames[$ent]}{mode} = sprintf "%o", $modes[$ent];
        }

        my $iter = $gui{liststoreRemoteFileChooser}->append();
        $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Parent)', 1, '..', 2, '', 3, '', 4, $gui{img}{ParentIcon});

        foreach my $fname (sort { lc($a) cmp lc($b) } (keys %files)) {

            if ($files{$fname}{type} == 4) { # Always add in directories
                my $iter = $gui{liststoreRemoteFileChooser}->append();
                $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Dir)', 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode}, 4, $gui{img}{DirIcon});
            }
            elsif ($fname =~ m/$filter/i) { # Only add in if it matches the filter
                my $iter = $gui{liststoreRemoteFileChooser}->append();
                $fname =~ m/^.*\.(.*)$/;
                my $ext = $1 ? lc(".$1") : ' ';
                $gui{liststoreRemoteFileChooser}->set($iter, 0, $ext, 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode}, 4, $gui{img}{FileIcon});
            }
        }

        $gui{entryRemoteFileChooserLocation}->set_text(IVFSExplorer_getPath($gui{IVFSExplorer}));
    }
    else {
        IVFSExplorer_cdUp($gui{IVFSExplorer}); # Failed to CD, so the path needs to be set back to the previous one
        $gui{entryRemoteFileChooserLocation}->set_text(IVFSExplorer_getPath($gui{IVFSExplorer}));
        show_err_msg('nodiraccess', '');
    }

    &busy_pointer($gui{dialogRemoteFileChooser}, 0);
}

# Fills a list as returned from reading the remote log file
sub fill_list_log() {
    my ($IMachine) = @_;
    $gui{liststoreLog0}->clear();
    $gui{liststoreLog1}->clear();
    $gui{liststoreLog2}->clear();
    $gui{liststoreLog3}->clear();
    
    for my $lognum (0..3) {
        my $log;
        my $offset = 0;

        if (IMachine_queryLogFilename($IMachine, $lognum)) {
            # Reading logs is limited to a maximum chunk size - normally 32K. The chunks are base64 encoded so we
            # need to read a chunk, decode, calculate next offset. Limit loop to 40 runs (max 1MB retrieval)
            for (1..40) {
                my $rawlog = IMachine_readLog($IMachine, $lognum, $offset, 32768); # Request 32K max. Limit is usually 32K anyway
                last if (!$rawlog); # Terminate loop if we've reached the end or log is empty
                $log .= decode_base64($rawlog); # Rawlog is base64 encoded. Append to log
                $offset = length($log); # Set next offset into log to get the next chunk
            }

            if ($log) {
                my @logarr = split "\n", $log; # Do we need to include windows/mac EOL here?
            
                foreach (0..$#logarr) {
                    $logarr[$_] =~ s/\r//g;
                    my $iter = $gui{'liststoreLog' . $lognum}->append;
                    $gui{'liststoreLog' . $lognum}->set($iter, 0, "$_: ", 1, $logarr[$_]);
                }
            }
            else {
                my $iter = $gui{'liststoreLog' . $lognum}->append;
                $gui{'liststoreLog' . $lognum}->set($iter, 0, '', 1, 'This log file is currently empty');
            }

        }
        else {
            my $iter = $gui{'liststoreLog' . $lognum}->append;
            $gui{'liststoreLog' . $lognum}->set($iter, 0, '', 1, 'This log file does not yet exist.');
        }
    }
}

# Fills a list of basic information about the remote server
sub fill_list_serverinfo() {
    $gui{liststoreInfo}->clear();
    my $vhost = &vhost();
    &addrow_info(0, 'URL:', 1, $endpoint);
    &addrow_info(0, 'VirtualBox Version:', 1, $$vhost{vbver});
    $$vhost{vrdeextpack} ? &addrow_info(0, 'Extension Pack:', 1, $$vhost{vrdeextpack}) : &addrow_info(0, 'Extension Pack:', 1, '<None>');
    &addrow_info(0, 'Build Revision:', 1, $$vhost{buildrev});
    &addrow_info(0, 'Package Type:', 1, $$vhost{pkgtype});
    &addrow_info(0, 'Global Settings File:', 1, $$vhost{settingsfile});
    &addrow_info(0, 'Machine Folder:', 1, $$vhost{machinedir});
    &addrow_info(0, 'Server Logical CPUs:', 1, $$vhost{maxhostcpuon});
    &addrow_info(0, 'Server CPU Type:', 1, $$vhost{cpudesc});
    &addrow_info(0, 'Server CPU Speed:', 1, "$$vhost{cpuspeed} Mhz (approx)");
    &addrow_info(0, 'VT-x/AMD-V Support:', 1, "$$vhost{vtx}");
    &addrow_info(0, 'VT-x/AMD-V Exclusive:', 1, $$vhost{hwexclusive});
    &addrow_info(0, 'PAE Support:', 1, "$$vhost{pae}");
    &addrow_info(0, 'Server Memory Size:', 1, "$$vhost{memsize} MB");
    &addrow_info(0, 'Server OS:', 1, $$vhost{os});
    &addrow_info(0, 'Server OS Version:', 1, $$vhost{osver});
    &addrow_info(0, 'Default Audio:', 1, $$vhost{defaudio});
    &addrow_info(0, 'Min Guest RAM:', 1, "$$vhost{minguestram} MB");
    &addrow_info(0, 'Max Guest RAM:', 1, &bytesToX($$vhost{maxguestram} * 1048576));
    &addrow_info(0, 'Min Guest Video RAM:', 1, "$$vhost{minguestvram} MB");
    &addrow_info(0, 'Max Guest Video RAM:', 1, "$$vhost{maxguestvram} MB");
    &addrow_info(0, 'Max Guest CPUs:', 1, $$vhost{maxguestcpu});
    &addrow_info(0, 'Max Guest Monitors:', 1, $$vhost{maxmonitors});
    &addrow_info(0, 'Max HD Image Size:', 1, &bytesToX($$vhost{maxhdsize}));
    &addrow_info(0, 'Guest Additions ISO:', 1, $$vhost{additionsiso});
    &addrow_info(0, 'Autostart DB:', 1, $$vhost{autostartdb});
}

# Populate the shared folder list for editing guest settings
sub fill_list_editshared() {
    my ($IMachine) = @_;
    $gui{buttonEditSharedRemove}->set_sensitive(0);
    $gui{buttonEditSharedEdit}->set_sensitive(0);
    $gui{liststoreEditShared}->clear();
    my @ISharedFolder = IMachine_getSharedFolders($IMachine);

    foreach my $share (@ISharedFolder) {
        my $access = 'Full';
        my $automount = 'No';
        my $tooltip = "$$share{name}: $$share{hostPath}";
        $access = 'Read-Only' if ($$share{writable} eq 'false');
        $automount = 'Yes' if ($$share{autoMount} eq 'true');
        $tooltip = $$share{lastAccessError} if ($$share{lastAccessError});
        $tooltip = 'Share is not accessible' if ($$share{accessible} eq 'false');
        my $iter = $gui{liststoreEditShared}->append;
        if ($$share{accessible} eq 'false') { $gui{liststoreEditShared}->set($iter, 0, $$share{name}, 1, $$share{hostPath}, 2, $access, 3, $automount, 4, $gui{img}{Error}, 5, $tooltip); }
        else { $gui{liststoreEditShared}->set($iter, 0, $$share{name}, 1, $$share{hostPath}, 2, $access, 3, $automount, 5, $tooltip); }
    }
}

# Populates the guest's storage list
sub fill_list_editstorage() {
    my ($IMachine) = @_;
    &busy_pointer($gui{dialogEdit}, 1);
    &sens_editstor_nosel();
    $gui{treestoreEditStor}->clear();
    my @IStorageController = IMachine_getStorageControllers($IMachine);

    foreach my $controller (@IStorageController) {
        Gtk2->main_iteration() while Gtk2->events_pending();
        my %ctr_attr = (name  => 1,
                        bus   => 1);
        &get_icontroller_attrs(\%ctr_attr, $controller); # Fill hash with attributes
        my $iter = $gui{treestoreEditStor}->append(undef);

        $gui{treestoreEditStor}->set($iter, 0,  $ctr_attr{name},                 # Display Name
                                            1,  $ctr_attr{bus} . ' Controller',  # Display Type
                                            2,  $ctr_attr{bus} . ' Controller',  # Tooltip
                                            3,  1,                               # Is it a controller
                                            4,  $ctr_attr{bus},                  # Controller BUS
                                            5,  $ctr_attr{name},                 # Controller's Name
                                            7,  $controller,                     # IStorageController object
                                            12, $gui{img}{ctr}{$ctr_attr{bus}});

        my @IMediumAttachment = IMachine_getMediumAttachmentsOfController($IMachine, $ctr_attr{name});

        foreach my $attach (@IMediumAttachment) {
            my $citer = $gui{treestoreEditStor}->append($iter);
            my %medium_attr = (refresh  => 1,
                               size     => 1,
                               logsize  => 1,
                               location => 1);
            &get_imedium_attrs(\%medium_attr, $$attach{medium});

            if ($$attach{medium}) { # Is it a medium or empty drive
                my $baseIMedium = IMedium_getBase($$attach{medium});
                my $mediumname = ($$attach{medium} eq $baseIMedium) ? IMedium_getName($baseIMedium) : "(*) " . IMedium_getName($baseIMedium); #Tests for snapshots
                $mediumname = '<Server Drive> ' . $medium_attr{location} if (&bl(IMedium_getHostDrive($$attach{medium})));
                $gui{treestoreEditStor}->set($citer, 0,  $mediumname,                            # Display Name
                                                     1,  $$attach{type},                         # Display Type
                                                     2,  "$medium_attr{location}\nPhysical Size: " .
                                                         &bytesToX($medium_attr{size}) . "\nLogical Size: " .
                                                         &bytesToX($medium_attr{logsize}),       # ToolTip
                                                     3,  0,                                      # Is it a controller
                                                     4,  $ctr_attr{bus},                         # The bus the medium is on
                                                     5,  $ctr_attr{name},                        # The name of the controller it is on
                                                     6,  $$attach{medium},                       # IMedium Object
                                                     7,  $controller,                            # IStorageController it is on
                                                     8,  $$attach{type},                         # Medium Type
                                                     9,  $$attach{device},                       # Device number
                                                     10, $$attach{port},                         # Port Number
                                                     11, $medium_attr{location},                 # Location
                                                     12, $gui{img}{$$attach{type}});

            }
            else {
                $gui{treestoreEditStor}->set($citer, 0,  '<Empty Drive>',               # Display Name
                                                     1,  $$attach{type},                # Display Typee
                                                     2,  "Empty Drive",  # Tooltip
                                                     3,  0,                             # Is it a controller
                                                     4,  $ctr_attr{bus},                # The bus the medium is on
                                                     5,  $ctr_attr{name},               # The name of the controller it is on
                                                     7,  $controller,                   # IStorageController it is on
                                                     8,  $$attach{type},                # Medium Type
                                                     9,  $$attach{device},              # Device number
                                                     10, $$attach{port},                # Port Number
                                                     12, $gui{img}{$$attach{type}});
            }
        }
    }

    $gui{treeviewEditStor}->expand_all();
    &busy_pointer($gui{dialogEdit}, 0);
}

# VBPrefs NAT List Handling
{
    my %selected = (INATNetwork => '');

    sub getsel_list_vbprefsnat() { return \%selected; }

    sub fill_list_vbprefsnat() {
        &busy_pointer($gui{dialogVBPrefs}, 1);
        $gui{buttonVBPrefsDelNAT}->set_sensitive(0);
        $gui{buttonVBPrefsEditNAT}->set_sensitive(0);
        $gui{liststoreVBPrefsNAT}->clear();
        my @INATNetwork = IVirtualBox_getNATNetworks($gui{websn});

        foreach my $nat (@INATNetwork) {
            my $iter = $gui{liststoreVBPrefsNAT}->append();
            $gui{liststoreVBPrefsNAT}->set($iter, 0, &bl(INATNetwork_getEnabled($nat)),
                                                  1, INATNetwork_getNetworkName($nat),
                                                  2, $nat);

            if ($nat eq $selected{INATNetwork}) {
                $gui{treeviewVBPrefsNAT}->get_selection()->select_iter($iter);
                &onsel_list_vbprefsnat();
            }
        }

        &busy_pointer($gui{dialogVBPrefs}, 0);
    }

    sub onsel_list_vbprefsnat() {
        my $model = $gui{treeviewVBPrefsNAT}->get_model();
        my $iter = $gui{treeviewVBPrefsNAT}->get_selection->get_selected() ? $gui{treeviewVBPrefsNAT}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach('Enabled', 'Name', 'INATNetwork');
        $gui{buttonVBPrefsDelNAT}->set_sensitive(1);
        $gui{buttonVBPrefsEditNAT}->set_sensitive(1);
    }
}

# VBPrefs HON List Handling
{
    my %selected = (Uuid => '');

    sub getsel_list_vbprefshon() { return \%selected; }

    sub fill_list_vbprefshon() {
        &busy_pointer($gui{dialogVBPrefs}, 1);
        $gui{buttonVBPrefsDelHON}->set_sensitive(0);
        $gui{buttonVBPrefsEditHON}->set_sensitive(0);
        $gui{liststoreVBPrefsHON}->clear();
        my $IHost = IVirtualBox_getHost($gui{websn});
        my @IHostNetworkInterface = IHost_findHostNetworkInterfacesOfType($IHost, 'HostOnly');

        foreach my $if (@IHostNetworkInterface) {
            my $iter = $gui{liststoreVBPrefsHON}->append();
            my $uuid = IHostNetworkInterface_getId($if);
            $gui{liststoreVBPrefsHON}->set($iter, 0, IHostNetworkInterface_getName($if),
                                                  1, $if,
                                                  2, $uuid);

            if ($uuid eq $selected{Uuid}) {
                $gui{treeviewVBPrefsHON}->get_selection()->select_iter($iter);
                &onsel_list_vbprefshon();
            }
        }

        &busy_pointer($gui{dialogVBPrefs}, 0);
    }

    sub onsel_list_vbprefshon() {
        my $model = $gui{treeviewVBPrefsHON}->get_model();
        my $iter =  $gui{treeviewVBPrefsHON}->get_selection->get_selected() ? $gui{treeviewVBPrefsHON}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach ('Name', 'IHostNetworkInterface', 'Uuid');
        $gui{buttonVBPrefsDelHON}->set_sensitive(1);
        $gui{buttonVBPrefsEditHON}->set_sensitive(1);
    }
}

# Adds a message to message log and scrolls to bottom
sub addrow_log() {
    my ($msg) = @_;
    my $iter = $gui{liststoreLog}->append;
    my ($sec,$min,$hour,$mday,$mon,$year) = localtime();
    $mon += 1;
    $year += 1900;
    $msg = sprintf("%d-%02d-%02d %02d:%02d:%02d    %s", $year, $mon, $mday, $hour, $min, $sec, $msg);
    $gui{liststoreLog}->set($iter, 0, $msg);
    $gui{treeviewLog}->scroll_to_cell($gui{treeviewLog}->get_model->get_path($iter));
}

sub addrow_info() {
    my $iter = $gui{liststoreInfo}->append;
    $gui{liststoreInfo}->set($iter, @_);
    return $iter;
}

sub addrow_ec() {
    my $iter = $gui{liststoreEvalConfig}->append;
    $gui{liststoreEvalConfig}->set($iter, @_);
    return $iter;
}

sub getsel_combo() {
    my ($widget, $col) = @_;
    my $model = $widget->get_model();
    my $iter = $widget->get_active_iter();

    if (defined($col)) { return $model->get($iter, $col); }
    else { return $model->get($iter); }
}

# Handles single and multiple selections
sub getsel_list_remotefiles() {
    my @filearray;
    my $model = $gui{treeviewRemoteFileChooser}->get_model();
    my @selections = $gui{treeviewRemoteFileChooser}->get_selection->get_selected_rows();

    foreach my $select (@selections) {
        my $iter = $model->get_iter($select);
        next if (!$iter);
        my @row = $model->get($iter);

        push @filearray, {Type     => $row[0],
                          FileName => $row[1],
                          Size     => $row[2],
                          Mode     => $row[3]};
    }

    return \@filearray;
}

sub getsel_list_editshared() {
    my $model = $gui{treeviewEditShared}->get_model();
    my ($path) = $gui{treeviewEditShared}->get_cursor();
    my $iter = $gui{treeviewEditShared}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Folder', 'Access', 'Mount', 'Accessible', 'Tooltip');
    return \%hash;
}

sub getsel_list_editstorage() {
    my $model = $gui{treeviewEditStor}->get_model();
    my ($path) = $gui{treeviewEditStor}->get_cursor();
    my $iter = $gui{treeviewEditStor}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('DisplayName', 'DisplayType', 'Tooltip', 'IsController', 'Bus', 'ControllerName', 'IMedium', 'IStorageController', 'MediumType', 'Device', 'Port', 'Location', 'Icon');
    return \%hash;
}

# USB Filter List Handling
{
    my %selected = (IUSBDeviceFilter => '');

    sub getsel_list_usbfilters() { return \%selected; }

    sub onsel_list_usbfilters() {
        my $model = $gui{treeviewEditUSBFilters}->get_model();
        my $iter = $gui{treeviewEditUSBFilters}->get_selection->get_selected() ? $gui{treeviewEditUSBFilters}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach ('Enabled', 'IUSBDeviceFilter', 'Name', 'Position');
        $gui{buttonEditUSBEdit}->set_sensitive(1);
        $gui{buttonEditUSBRemove}->set_sensitive(1);
        $gui{buttonEditUSBUp}->set_sensitive(1);
        $gui{buttonEditUSBDown}->set_sensitive(1);
    }

    sub fill_list_usbfilters() {
        &busy_pointer($gui{dialogEdit}, 1);
        my ($IMachine) = @_;
        $gui{buttonEditUSBEdit}->set_sensitive(0);
        $gui{buttonEditUSBRemove}->set_sensitive(0);
        $gui{buttonEditUSBUp}->set_sensitive(0);
        $gui{buttonEditUSBDown}->set_sensitive(0);
        $gui{liststoreEditUSBFilter}->clear();
        my $IUSBDeviceFilters = IMachine_getUSBDeviceFilters($IMachine);
        my @filters = IUSBDeviceFilters_getDeviceFilters($IUSBDeviceFilters);
        my $pos = 0;

        foreach my $filter (@filters) {
            my $iter = $gui{liststoreEditUSBFilter}->append();
            $gui{liststoreEditUSBFilter}->set($iter,
                                        0, &bl(IUSBDeviceFilter_getActive($filter)),
                                        1, $filter,
                                        2, IUSBDeviceFilter_getName($filter),
                                        3, $pos);

            if ($filter eq $selected{IUSBDeviceFilter}) {
                $gui{treeviewEditUSBFilters}->get_selection()->select_iter($iter);
                &onsel_list_usbfilters();
            }

            $pos++;
        }
        &busy_pointer($gui{dialogEdit}, 0);
    }
}

sub onsel_list_remotefiles() {
    my $filearrayref = &getsel_list_remotefiles();

    # We only care about the first file selected, and only if it's a directory
    my $fileref = ${$filearrayref}[0];

    if ($$fileref{FileName} eq '..') { &cdup_remotefilechooser(); }
    elsif ($$fileref{Type} eq '(Dir)') {
        my $path = IVFSExplorer_getPath($gui{IVFSExplorer});
        IVFSExplorer_cd($gui{IVFSExplorer}, &rcatdir($path, $$fileref{FileName}));
        &fill_list_remotefiles(IVFSExplorer_getPath($gui{IVFSExplorer}), $gui{entryRemoteFileChooserFilter}->get_text());
    }
}

sub onsel_list_remotefiles_single() {
    my $filearrayref = &getsel_list_remotefiles();

    # We only care about the first file selected, and only if it's a file
    my $fileref = ${$filearrayref}[0];
    if ($$fileref{FileName} ne '..' and $$fileref{Type} ne '(Dir)') { $gui{entryRemoteFileChooserFile}->set_text($$fileref{FileName}); }
}

# Activates when selecting an item in the edit storage list, could be reduced a little
# as a lot of the options are the same for each controller but this gives flexibility
# to expand
sub onsel_list_editstorage() {
    my $storref = &getsel_list_editstorage();

    if ($$storref{IsController}) {
        &sens_editstor_allctr();
        $gui{comboboxEditStorCtrType}->signal_handler_block($signal{stortype});
        $gui{entryEditStorCtrName}->set_text($$storref{ControllerName});
        $gui{checkbuttonEditStorCache}->set_active(&bl(IStorageController_getUseHostIOCache($$storref{IStorageController})));
        my $variant = IStorageController_getControllerType($$storref{IStorageController});

        if ($$storref{Bus} eq 'IDE') {
            &sens_editstor_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorIDECtrType});
        }
        elsif ($$storref{Bus} eq 'SATA') {
            &sens_editstor_sata_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorSATACtrType});
            $gui{adjustmentEditStorPortCount}->set_value(IStorageController_getPortCount($$storref{IStorageController}));
        }
        elsif ($$storref{Bus} eq 'Floppy') {
            &sens_editstor_floppy_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorFloppyCtrType});
        }
        elsif ($$storref{Bus} eq 'SAS') {
            &sens_editstor_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorSASCtrType});
        }
        else { # Defaults to SCSI
            &sens_editstor_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorSCSICtrType});
        }

        &combobox_set_active_text($gui{comboboxEditStorCtrType}, $variant);
        $gui{comboboxEditStorCtrType}->signal_handler_unblock($signal{stortype});
    }
    else {
        &sens_editstor_allmedia();

        if ($$storref{MediumType} eq 'DVD') {
            my $attach = IMachine_getMediumAttachment($vmc{IMachine}, $$storref{ControllerName}, $$storref{Port}, $$storref{Device});
            $gui{checkbuttonEditStorLive}->set_active(&bl($$attach{temporaryEject}));
            &sens_editstor_dvd();
        }
        elsif ($$storref{MediumType} eq 'Floppy') { &sens_editstor_floppy(); }
        else { # Default to HD
            my $attach = IMachine_getMediumAttachment($vmc{IMachine}, $$storref{ControllerName}, $$storref{Port}, $$storref{Device});
            $gui{checkbuttonEditStorSSD}->set_active(&bl($$attach{nonRotational}));
            &sens_editstor_hd();
        }

        # We also need to setup the port comboboxEditStorDevPort
        if ($$storref{Bus} eq 'SATA') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortSATA});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Port});
        }
        elsif ($$storref{Bus} eq 'IDE') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortIDE});
            if ($$storref{Device} == 0 and $$storref{Port} == 0) { $gui{comboboxEditStorDevPort}->set_active(0); }
            elsif ($$storref{Device} == 1 and $$storref{Port} == 0) { $gui{comboboxEditStorDevPort}->set_active(1); }
            elsif ($$storref{Device} == 0 and $$storref{Port} == 1) { $gui{comboboxEditStorDevPort}->set_active(2); }
            elsif ($$storref{Device} == 1 and $$storref{Port} == 1) { $gui{comboboxEditStorDevPort}->set_active(3); }
        }
        elsif ($$storref{Bus} eq 'SAS') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortSAS});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Port});
        }
        elsif ($$storref{Bus} eq 'SCSI') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortSCSI});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Port});
        }
        elsif ($$storref{Bus} eq 'Floppy') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortFloppy});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Device});
        }
    }
}

# VMM Floppy List Handling
{
    my %selected = (IMedium => '');

    # Return the selected entry in the VMM floppy disk list
    sub getsel_list_vmmfloppy() { return \%selected; }

    # Fill the floppy media list in the VMM
    sub fill_list_vmmfloppy() {
        &busy_pointer($gui{dialogVMM}, 1);
        &clr_list_vmm($gui{treestoreVMMFloppy});
        my $IMediumRef = &get_all_media('Floppy');

        foreach (sort { lc($$IMediumRef{$a}) cmp lc($$IMediumRef{$b}) } (keys %$IMediumRef)) {
            my %mattr = (name       => 1,
                         logsize    => 1,
                         refresh    => 1,
                         accesserr  => 1,
                         location   => 1,
                         type       => 1); # medium attributes to get

            &get_imedium_attrs(\%mattr, $_);
            my $iter = $gui{treestoreVMMFloppy}->append(undef);

            if ($mattr{refresh} eq 'Inaccessible') {
                $gui{treestoreVMMFloppy}->set($iter, 0, $mattr{name},
                                                     1, $_,
                                                     2, 0,
                                                     3, $gui{img}{Error},
                                                     4, $mattr{accesserr}, # Tooltip can be access error
                                                     5, $mattr{location},
                                                     6, $mattr{type});
            }
            else {
                $gui{treestoreVMMFloppy}->set($iter, 0, $mattr{name},
                                                     1, $_,
                                                     2, &bytesToX($mattr{logsize}),
                                                     4, $mattr{location}, # Tooltip can be location
                                                     5, $mattr{location},
                                                     6, $mattr{type});
            }

            if ($_ eq $selected{IMedium}) {
                $gui{treeviewVMMFloppy}->get_selection()->select_iter($iter);
                &onsel_list_vmmfloppy();
            }
        }

        &busy_pointer($gui{dialogVMM}, 0);
    }

    sub onsel_list_vmmfloppy() {
        my $model = $gui{treeviewVMMFloppy}->get_model();
        my $iter = $gui{treeviewVMMFloppy}->get_selection->get_selected() ? $gui{treeviewVMMFloppy}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach ('Name', 'IMedium', 'Size', 'Accessible', 'Tooltip', 'Location', 'Type');
        $gui{toolbuttonVMMCopy}->set_sensitive(0);
        $gui{toolbuttonVMMModify}->set_sensitive(0);

        my $gnames;
        my @mids = IMedium_getMachineIds($selected{IMedium});

        foreach my $id (@mids) {
            my $snames;
            my $IMachine = IVirtualBox_findMachine($gui{websn}, $id);
            my $mname = IMachine_getName($IMachine);
            my @sids = IMedium_getSnapshotIds($selected{IMedium}, $id);

            foreach my $snapid (@sids) {
                next if ($snapid eq $id);
                my $ISnapshot = IMachine_findSnapshot($IMachine, $snapid);
                my $sname = ISnapshot_getName($ISnapshot);
                $snames .= "$sname, ";
            }

            if ($snames) {
                $snames =~ s/, $//; # Remove any trailing comma
                $gnames .= "$mname ($snames). ";
            }
            else { $gnames .= "$mname, "; }
        }

        if ($gnames) {
            $gui{toolbuttonVMMRemove}->set_sensitive(0);
            $gui{toolbuttonVMMRelease}->set_sensitive(1);
            $gnames =~ s/, $//; # Remove any trailing comma
        }
        else {
            $gnames = '<Not Attached>';
            $gui{toolbuttonVMMRemove}->set_sensitive(1);
            $gui{toolbuttonVMMRelease}->set_sensitive(0);
        }

        $gui{labelVMMTypeField}->set_text("$selected{Type}, " . uc(IMedium_getFormat($selected{IMedium})) . ', ' . IMedium_getVariant($selected{IMedium}));
        $gui{labelVMMAttachedToField}->set_text($gnames);
        $gui{labelVMMLocationField}->set_text($selected{Location});
    }
}

# VMM DVD List Handling
{
    my %selected = (IMedium => '');

    sub getsel_list_vmmdvd() { return \%selected; }

    # Fill the DVD media list in the VMM
    sub fill_list_vmmdvd() {
        &busy_pointer($gui{dialogVMM}, 1);
        &clr_list_vmm($gui{treestoreVMMDVD});
        my $IMediumRef = &get_all_media('DVD');

        foreach (sort { lc($$IMediumRef{$a}) cmp lc($$IMediumRef{$b}) } (keys %$IMediumRef)) {
            my %mattr = (name       => 1,
                         logsize    => 1,
                         refresh    => 1,
                         accesserr  => 1,
                         location   => 1,
                         type       => 1); # medium attributes to get

            &get_imedium_attrs(\%mattr, $_);
            my $iter = $gui{treestoreVMMDVD}->append(undef);

            if ($mattr{refresh} eq 'Inaccessible') {
                $gui{treestoreVMMDVD}->set($iter, 0, $mattr{name},
                                                  1, $_,
                                                  2, 0,
                                                  3, $gui{img}{Error},
                                                  4, $mattr{accesserr}, # Tooltip can be access error
                                                  5, $mattr{location},
                                                  6, $mattr{type});
            }
            else {
                $gui{treestoreVMMDVD}->set($iter, 0, $mattr{name},
                                                  1, $_,
                                                  2, &bytesToX($mattr{logsize}),
                                                  4, $mattr{location}, # Tooltip can be location
                                                  5, $mattr{location},
                                                  6, $mattr{type});
            }

            if ($_ eq $selected{IMedium}) {
                $gui{treeviewVMMDVD}->get_selection()->select_iter($iter);
                &onsel_list_vmmdvd();
            }
        }

        &busy_pointer($gui{dialogVMM}, 0);
    }

    sub onsel_list_vmmdvd() {
        my $model = $gui{treeviewVMMDVD}->get_model();
        my $iter = $gui{treeviewVMMDVD}->get_selection->get_selected() ? $gui{treeviewVMMDVD}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach ('Name', 'IMedium', 'Size', 'Accessible', 'Tooltip', 'Location', 'Type');
        $gui{toolbuttonVMMCopy}->set_sensitive(0);
        $gui{toolbuttonVMMModify}->set_sensitive(0);

        my $gnames;
        my @mids = IMedium_getMachineIds($selected{IMedium});

        foreach my $id (@mids) {
            my $snames;
            my $IMachine = IVirtualBox_findMachine($gui{websn}, $id);
            my $mname = IMachine_getName($IMachine);
            my @sids = IMedium_getSnapshotIds($selected{IMedium}, $id);

            foreach my $snapid (@sids) {
                next if ($snapid eq $id);
                my $ISnapshot = IMachine_findSnapshot($IMachine, $snapid);
                my $sname = ISnapshot_getName($ISnapshot);
                $snames .= "$sname, ";
            }

            if ($snames) {
                $snames =~ s/, $//; # Remove any trailing comma
                $gnames .= "$mname ($snames). ";
            }
            else { $gnames .= "$mname, "; }
        }

        if ($gnames) {
            $gui{toolbuttonVMMRemove}->set_sensitive(0);
            $gui{toolbuttonVMMRelease}->set_sensitive(1);
            $gnames =~ s/, $//; # Remove any trailing comma
        }
        else {
            $gnames = '<Not Attached>';
            $gui{toolbuttonVMMRemove}->set_sensitive(1);
            $gui{toolbuttonVMMRelease}->set_sensitive(0);
        }

        $gui{labelVMMTypeField}->set_text("$selected{Type}, " . uc(IMedium_getFormat($selected{IMedium})) . ', ' . IMedium_getVariant($selected{IMedium}));
        $gui{labelVMMAttachedToField}->set_text($gnames);
        $gui{labelVMMLocationField}->set_text($selected{Location});
    }
}

# Port Forwarding List Handling
{
    my %selected = (Name => '');

    sub getsel_list_pf { return \%selected; }

    sub fill_list_pf {
        my ($INATNetwork) = @_;
        &busy_pointer($gui{dialogNATDetails}, 1);
        &clr_list_pf();
        my @rules = INATNetwork_getPortForwardRules4($INATNetwork);
        foreach my $rule (@rules) {
            my ($rname, $rproto, $rhip, $rhport, $rgip, $rgport) = split ':', $rule;
            $rhip =~ s/[^0-9,.]//g; # Strip everything but these chars
            $rgip =~ s/[^0-9,.]//g; # Strip everything but these chars
            my $iter = $gui{liststorePFRulesIPv4}->append;
            $gui{liststorePFRulesIPv4}->set($iter, 0, $rname, 1, uc($rproto), 2, $rhip, 3, $rhport, 4, $rgip, 5, $rgport, 6, $INATNetwork);

            if ($rname eq $selected{Name}) {
                $gui{treeviewPFRulesIPv4}->get_selection()->select_iter($iter);
                &onsel_list_pf();
            }
        }
        &busy_pointer($gui{dialogNATDetails}, 0);
    }

    sub onsel_list_pf {
        my $model = $gui{treeviewPFRulesIPv4}->get_model();
        my $iter = $gui{treeviewPFRulesIPv4}->get_selection->get_selected() ? $gui{treeviewPFRulesIPv4}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach ('Name', 'Protocol', 'HostIP', 'HostPort', 'GuestIP', 'GuestPort', 'INATNetwork');
        $gui{buttonPFRulesRemove}->set_sensitive(1);
    }

    sub clr_list_pf {
        $gui{liststorePFRulesIPv4}->clear();
        $gui{buttonPFRulesRemove}->set_sensitive(0);
    }

}


# VMM HD List Handling
{
    my %selected = (IMedium => '');

    sub getsel_list_vmmhd() { return \%selected; }

    # Fill the hard disk media list in the VMM
    sub fill_list_vmmhd() {
        &busy_pointer($gui{dialogVMM}, 1);
        &clr_list_vmm($gui{treestoreVMMHD});
        my $IMediumRef = &get_all_media('HardDisk');

        foreach (sort { lc($$IMediumRef{$a}) cmp lc($$IMediumRef{$b}) } (keys %$IMediumRef)) {
            &recurse_hd_snapshot($gui{treestoreVMMHD}, $_, undef);
        }

        &busy_pointer($gui{dialogVMM}, 0);
    }

    sub onsel_list_vmmhd() {
        my $model = $gui{treeviewVMMHD}->get_model();
        my $iter = $gui{treeviewVMMHD}->get_selection->get_selected() ? $gui{treeviewVMMHD}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach ('Name', 'IMedium', 'Asize', 'Vsize', 'Accessible', 'Tooltip', 'Location', 'Type');
        $gui{toolbuttonVMMCopy}->set_sensitive(1);
        $gui{toolbuttonVMMModify}->set_sensitive(1);

        my $gnames;
        my @mids = IMedium_getMachineIds($selected{IMedium});

        foreach my $id (@mids) {
            my $snames;
            my $IMachine = IVirtualBox_findMachine($gui{websn}, $id);
            my $mname = IMachine_getName($IMachine);
            my @sids = IMedium_getSnapshotIds($selected{IMedium}, $id);

            foreach my $snapid (@sids) {
                next if ($snapid eq $id);
                my $ISnapshot = IMachine_findSnapshot($IMachine, $snapid);
                my $sname = ISnapshot_getName($ISnapshot);
                $snames .= "$sname, ";
            }

            if ($snames) {
                $snames =~ s/, $//; # Remove any trailing comma
                $gnames .= "$mname ($snames). ";
            }
            else { $gnames .= "$mname, "; }
        }

        if ($gnames) {
            $gui{toolbuttonVMMRemove}->set_sensitive(0);
            $gui{toolbuttonVMMRelease}->set_sensitive(1);
            $gnames =~ s/, $//; # Remove any trailing comma
        }
        else {
            $gnames = '<Not Attached>';
            $gui{toolbuttonVMMRemove}->set_sensitive(1);
            $gui{toolbuttonVMMRelease}->set_sensitive(0);
        }

        # Don't allow remove/release if it has sub-snapshots
        if (IMedium_getChildren($selected{IMedium})) {
            $gui{toolbuttonVMMRemove}->set_sensitive(0);
            $gui{toolbuttonVMMRelease}->set_sensitive(0);
        }

        $gui{labelVMMTypeField}->set_text("$selected{Type}, " . uc(IMedium_getFormat($selected{IMedium})) . ', ' . IMedium_getVariant($selected{IMedium}));
        $gui{labelVMMAttachedToField}->set_text($gnames);
        $gui{labelVMMLocationField}->set_text($selected{Location});
    }

    # Recurses through the media for populating the VMM media lists, including
    # identifying snapshots
    sub recurse_hd_snapshot() {
        my ($treestore, $IMedium, $iter) = @_;
        my %mattr = (name       => 1,
                     size       => 1,
                     logsize    => 1,
                     refresh    => 1,
                     accesserr  => 1,
                     children   => 1,
                     location   => 1,
                     type       => 1); # medium attributes to get

        &get_imedium_attrs(\%mattr, $IMedium);
        my $citer = $treestore->append($iter);

        if ($mattr{refresh} eq 'Inaccessible') {
            $treestore->set($citer, 0, $mattr{name},
                                    1, $IMedium,
                                    2, 0,
                                    3, 0,
                                    4, $gui{img}{Error},
                                    5, $mattr{accesserr},
                                    6, $mattr{location},
                                    7, $mattr{type});
        }
        else {
            $treestore->set($citer, 0, $mattr{name},
                                    1, $IMedium,
                                    2, &bytesToX($mattr{size}),
                                    3, &bytesToX($mattr{logsize}),
                                    5, $mattr{location}, # Tooltip can be location
                                    6, $mattr{location},
                                    7, $mattr{type});
        }
        
        if (($IMedium eq $selected{IMedium})) {
            $gui{treeviewVMMHD}->expand_all() if (IMedium_getParent($IMedium)); # If item is a snapshot, we need to expand the list in order for selection to work
            $gui{treeviewVMMHD}->get_selection()->select_iter($citer);
            &onsel_list_vmmhd();
        }

        &recurse_hd_snapshot($treestore, $_, $citer) foreach (@{$mattr{children}});
    }
}

sub clr_list_vmm() {
    my ($treestore) = @_;
    &vmm_sens_unselected(); # Do whenever list is cleared
    $treestore->clear();
}

# Snapshot List Handling
{
    my %selected = (ISnapshot => '');

    sub getsel_list_snapshots() { return \%selected; }

    # Set sensitivity when a snapshot is selected
    sub onsel_list_snapshots() {
        my $model = $gui{treeviewSnapshots}->get_model();
        my $iter = $gui{treeviewSnapshots}->get_selection->get_selected() ? $gui{treeviewSnapshots}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        $selected{$_} = shift @row foreach ('Name', 'Date', 'ISnapshot', 'Icon');
        if ($selected{ISnapshot}) {
            $gui{buttonRestoreSnapshot}->set_sensitive(1);
            $gui{buttonDeleteSnapshot}->set_sensitive(1);
            $gui{buttonDetailsSnapshot}->set_sensitive(1);
            $gui{buttonCloneSnapshot}->set_sensitive(1);
            $gui{buttonTakeSnapshot}->set_sensitive(0);
        }
        else {
            $gui{buttonRestoreSnapshot}->set_sensitive(0);
            $gui{buttonDeleteSnapshot}->set_sensitive(0);
            $gui{buttonDetailsSnapshot}->set_sensitive(0);
            $gui{buttonCloneSnapshot}->set_sensitive(0);
            $gui{buttonTakeSnapshot}->set_sensitive(1);
        }
    }

    sub fill_list_snapshots() {
        my $gref = &getsel_list_guest();
        &addrow_log("Fetching snapshots for $$gref{Name}...");
        &busy_pointer($gui{windowMain}, 1);
        &clr_list_snapshots();

        if (IMachine_getSnapshotCount($$gref{IMachine}) > 0) {
            my $ISnapshot_current = IMachine_getCurrentSnapshot($$gref{IMachine});
            my $ISnapshot = IMachine_findSnapshot($$gref{IMachine}, undef); # get first snapshot
            &recurse_snapshot($ISnapshot, undef, $ISnapshot_current);
            $gui{treeviewSnapshots}->expand_all();
        }

        &busy_pointer($gui{windowMain}, 0);
        &addrow_log('Snapshot fetch complete.');
    }

    # Clear snapshot list and set sensitivity
    sub clr_list_snapshots() {
        $gui{buttonRestoreSnapshot}->set_sensitive(0);
        $gui{buttonDeleteSnapshot}->set_sensitive(0);
        $gui{buttonDetailsSnapshot}->set_sensitive(0);
        $gui{buttonCloneSnapshot}->set_sensitive(0);
        $gui{treestoreSnapshots}->clear();
    }

    sub recurse_snapshot() {
        my ($ISnapshot, $iter, $ISnapshot_current) = @_;
        my $citer = $gui{treestoreSnapshots}->append($iter);
        my $snapname = ISnapshot_getName($ISnapshot);
        my $date = scalar(localtime((ISnapshot_getTimeStamp($ISnapshot))/1000)); # VBox returns msecs so / 1000
        $gui{treestoreSnapshots}->set($citer,
                                      0, $snapname,
                                      1, $date,
                                      2, $ISnapshot,
                                      3, &bl(ISnapshot_getOnline($ISnapshot)) ? $gui{img}{SnapshotOnline} : $gui{img}{SnapshotOffline});

        if ($ISnapshot eq $ISnapshot_current) {
            my $curiter = $gui{treestoreSnapshots}->append($citer);
            $gui{treestoreSnapshots}->set($curiter, 0, '[Current State]', 1, '', 2, '', 3, $gui{img}{SnapshotCurrent});
        }
        
        my @snapshots = ISnapshot_getChildren($ISnapshot);
        if (@snapshots > 0) { &recurse_snapshot($_, $citer, $ISnapshot_current) foreach (@snapshots); }
    }
}

# Guest List Handling
{
    my %selected = (Uuid => 'None'); # Initialize this element as it may be tested before the hash is fully initialized

    sub makesel_list_guest() { $selected{Uuid} = $_[0]; }

    sub getsel_list_guest() { return \%selected; }

    sub onsel_list_guest() {
        my $model = $gui{treeviewGuest}->get_model();
        my $iter = $gui{treeviewGuest}->get_selection->get_selected() ? $gui{treeviewGuest}->get_selection->get_selected() : $model->get_iter_first();
        my @row = $model->get($iter);
        
        # If there's no IMachine, it's a group so don't waste anymore time
        if (!$row[2]) { 
            &sens_unselected();
            &clr_summarydetails();
            return;
        }

        $selected{$_} = shift @row foreach ('Name', 'Os', 'IMachine', 'Status', 'Osid', 'Uuid', 'Icon', 'Prettyname', 'Statusicon');
        &fill_summarydetails();
        &sens_unselected();
        &fill_list_snapshots();
        my $status = IMachine_getState($selected{IMachine});

        if ($status eq 'Running' | $status eq 'Starting') {
            my @IMediumAttachment = IMachine_getMediumAttachments($selected{IMachine});
            my @IUSBController = IMachine_getUSBControllers($selected{IMachine});
            $gui{menuitemAction}->set_sensitive(1);
            $gui{menuitemStop}->set_sensitive(1);
            $gui{menuitemPause}->set_sensitive(1);
            $gui{menuitemReset}->set_sensitive(1);
            $gui{menuitemKeyboard}->set_sensitive(1);
            $gui{menuitemDisplay}->set_sensitive(1);
            $gui{menuitemLogs}->set_sensitive(1);
            $gui{toolbuttonStop}->set_sensitive(1);
            $gui{toolbuttonReset}->set_sensitive(1);
            $gui{toolbuttonRemoteDisplay}->set_sensitive(1);
            $gui{buttonShowDetails}->set_sensitive(1);
            $gui{toolbuttonSettings}->set_sensitive(1);         # Online editing
            $gui{buttonRefreshSnapshot}->set_sensitive(1);
            $gui{buttonTakeSnapshot}->set_sensitive(1);
            $gui{menuitemScreenshot}->set_sensitive(1);
            $gui{menuitemUSB}->set_sensitive(1) if $IUSBController[0];
            $gui{menuitemHotPlugCPU}->set_sensitive(1) if (&bl(IMachine_getCPUHotPlugEnabled($selected{IMachine})));

            foreach my $attach (@IMediumAttachment) {
                $gui{menuitemDVD}->set_sensitive(1) if ($$attach{type} eq 'DVD');
                $gui{menuitemFloppy}->set_sensitive(1) if ($$attach{type} eq 'Floppy');
            }

        }
        elsif ($status eq 'Saved') {
            $gui{menuitemAction}->set_sensitive(1);
            $gui{menuitemStart}->set_sensitive(1);
            $gui{menuitemClone}->set_sensitive(1);
            $gui{menuitemLogs}->set_sensitive(1);
            $gui{menuitemDiscard}->set_sensitive(1);
            $gui{menuitemSetGroup}->set_sensitive(1);
            $gui{menuitemUngroup}->set_sensitive(1);
            $gui{toolbuttonStart}->set_sensitive(1);
            $gui{toolbuttonDiscard}->set_sensitive(1);
            $gui{buttonShowDetails}->set_sensitive(1);
            $gui{buttonRefreshSnapshot}->set_sensitive(1);
            $gui{buttonTakeSnapshot}->set_sensitive(1);
        }
        elsif ($status eq 'Paused') {
            $gui{menuitemAction}->set_sensitive(1);
            $gui{menuitemResume}->set_sensitive(1);
            $gui{menuitemRemoteDisplay}->set_sensitive(1);
            $gui{menuitemLogs}->set_sensitive(1);
            $gui{toolbuttonRemoteDisplay}->set_sensitive(1);
            $gui{buttonShowDetails}->set_sensitive(1);
        }
        elsif ($status eq 'PoweredOff' | $status eq 'Aborted') {
            $gui{menuitemExportAppliance}->set_sensitive(1);
            $gui{menuitemAction}->set_sensitive(1);
            $gui{menuitemStart}->set_sensitive(1);
            $gui{menuitemSettings}->set_sensitive(1);
            $gui{menuitemClone}->set_sensitive(1);
            $gui{menuitemRemove}->set_sensitive(1);
            $gui{menuitemSetGroup}->set_sensitive(1);
            $gui{menuitemUngroup}->set_sensitive(1);
            $gui{menuitemLogs}->set_sensitive(1);
            $gui{toolbuttonStart}->set_sensitive(1);
            $gui{toolbuttonSettings}->set_sensitive(1);
            $gui{buttonShowDetails}->set_sensitive(1);
            $gui{buttonRefreshSnapshot}->set_sensitive(1);
            $gui{buttonTakeSnapshot}->set_sensitive(1);
            $gui{menuitemHotPlugCPU}->set_sensitive(1) if (&bl(IMachine_getCPUHotPlugEnabled($selected{IMachine})));
        }
        else { &sens_unselected(); }
    }

    sub add_guest_group() {
        my ($node, $name, $piter) = @_;
        
        if (defined($$node{$name})) { return $$node{$name}{node}, $$node{$name}{iter}; }
        else {
            my $citer = $gui{treestoreGuest}->append($piter);
            $gui{treestoreGuest}->set($citer, 0, $name,
                                              6, $gui{img}{VMGroup},
                                              7, $name);

            $$node{$name}{iter} = $citer;
            $$node{$name}{node} = {};
            return $$node{$name}{node}, $citer;
        }
    }

    # Populates the list of available guests
    sub fill_list_guest() {
        my $osver = &osver();
        my %grouptree;
        &addrow_log("Fetching guest list from $endpoint...");
        &busy_pointer($gui{windowMain}, 1);
        &clr_list_guest();
        my %guestlist;
        my @IMachine = IVirtualBox_getMachines($gui{websn});
        my $selection;
        my $inaccessible = 0;

        # Preprocess groups first, leads to a neater layout as groups will all be added to the treeview
        # before the guests. Add iter to guestlist for use later to save us needing to look it up
        foreach my $machine (@IMachine) {
            my $node = \%grouptree; # Reset the tree to the start for each new guest

            if (&bl(IMachine_getAccessible($machine))) {
                $guestlist{$machine}{name} = IMachine_getName($machine);
                my ($group) = IMachine_getGroups($machine); # We only care about the first group returned
                $group =~ s/^\///; # Leading / is optional so always remove or simplicity
                my @components = split('/', $group);
                my $piter = undef;
                ($node, $piter) = &add_guest_group($node, $_, $piter) foreach (@components); 
                $guestlist{$machine}{iter} = $piter;
            }
            else { $inaccessible = 1; }
        }

        foreach my $machine (sort { lc($guestlist{$a}{name}) cmp lc($guestlist{$b}{name}) } (keys %guestlist)) {
            my $ISnapshot = IMachine_getCurrentSnapshot($machine);
            my $osid = IMachine_getOSTypeId($machine);
            my $uuid = IMachine_getId($machine);
            my $prettyname = $guestlist{$machine}{name};
            my $status = IMachine_getState($machine);
            if ($ISnapshot) { $prettyname .=  ' (' . ISnapshot_getName($ISnapshot) . ")\n$status"; }
            else { $prettyname .=  "\n$status"; }

            my $iter = $gui{treestoreGuest}->append($guestlist{$machine}{iter});
            $gui{treestoreGuest}->set($iter,
                                      0, $guestlist{$machine}{name},
                                      1, $$osver{$osid}{description},
                                      2, $machine,
                                      3, $status,
                                      4, $osid,
                                      5, $uuid,
                                      6, (-e "$gui{THUMBDIR}/$uuid.png") ? Gtk2::Gdk::Pixbuf->new_from_file("$gui{THUMBDIR}/$uuid.png") : $$osver{$osid}{icon},
                                      7, $prettyname,
                                      8, $gui{img}{$status});

            $gui{treeviewGuest}->expand_all() if ($prefs{GUESTLISTEXPAND});
            $selection = $iter if ($uuid eq $selected{Uuid});
        }

        if ($selection) {
                $gui{treeviewGuest}->get_selection()->select_iter($selection);
                &onsel_list_guest();
        }

        &busy_pointer($gui{windowMain}, 0);
        &addrow_log('Guest list complete.');
        &addrow_log('WARNING: You have one or more guests that are inaccessible and have been excluded from ' .
                    'the guest list. You must use vboxmanage or VirtualBox on the server to fix these issues') if ($inaccessible);

        &update_server_membar();
    }

    sub clr_list_guest() {
        &sens_unselected();
        $gui{treestoreGuest}->clear();
        &clr_summarydetails();
        &clr_list_snapshots();
    }
}

1;
