/**
 * <copyright>
 *
 * Service Data Objects
 * Version 1.0
 * Licensed Materials - Property of BEA and IBM
 *
 * (c) Copyright BEA Systems, Inc. and International Business Machines Corp 2003.  All rights reserved.
 *
 * </copyright>
 * 
 * $Id: ChangeSummary.java,v 1.1 2004/03/26 15:24:15 marcelop Exp $
 */
package commonj.sdo;

import java.util.List;

/**
 * A change summary is used to record changes to the objects in a {@link DataGraph data graph},
 * allowing applications to efficiently and incrementally update back-end storage when required.
 */
public interface ChangeSummary
{
  /**
   * Clears the list of {@link #getChangedDataObjects changes} and turns change logging on.
   * @see #endLogging
   * @see #isLogging
   */
  void beginLogging();

  /**
   * Turns change logging off.
   * @see #beginLogging
   * @see #isLogging
   */
  void endLogging();

  /**
   * Indicates whether change logging is on (<code>true</code>) or off (<code>false</code>).
   * @return <code>true</code> if change logging is on.
   * @see #beginLogging
   * @see #endLogging
   */
  boolean isLogging();

  /**
   * Returns the {@link DataGraph data graph} associated with this change log.
   * @return the data graph.
   * @see DataGraph#getChangeSummary
   */
  DataGraph getDataGraph();

  /**
   * Returns a list consisting of all the {@link DataObject data objects} that have been changed while {@link #isLogging logging}.
   * <p>
   * The {@link #isCreated new} and modified objects in the list are references to objects that
   * are {@link DataObject#getDataGraph contained} in the {@link #getDataGraph data graph} associated with this log. 
   * The {@link #isDeleted deleted} objects in the list are references to copies of the objects 
   * as they appeared at the time that event logging was first enabled; 
   * if the deleted objects have references to other objects, 
   * the references will also refer to copies of the target objects.
   * @return a list of changed data objects.
   * @see #isCreated(DataObject)
   * @see #isDeleted(DataObject)
   */
  List /*DataObject*/ getChangedDataObjects();

  /**
   * Returns whether or not the specified data object was created while {@link #isLogging logging}.
   * Any object that was added to the {@link #getDataGraph data graph}
   * but was not {@link DataObject#getDataGraph contained} in the data graph when logging began, 
   * will be considered created.
   * @param dataObject the data object in question.
   * @return <code>true</code> if the specified data object was created.
   * @see #getChangedDataObjects
   */
  boolean isCreated(DataObject dataObject);

  /**
   * Returns whether or not the specified data object was deleted while {@link #isLogging logging}.
   * Any object that is not {@link DataObject#getDataGraph contained} by the {@link #getDataGraph data graph} will be considered deleted.
   * @param dataObject the data object in question.
   * @return <code>true</code> if the specified data object was deleted.
   * @see #getChangedDataObjects
   */
  boolean isDeleted(DataObject dataObject);

  /**
   * A setting encapsulates a {@link Property property} and a corresponding single value of the property's {@link Property#getType type}.
   */
  public interface Setting
  {
    /**
     * Returns the property of the setting.
     * @return the setting property.
     */
    Property getProperty();

    /**
     * Returns the value of the setting.
     * @return the setting value.
     */
    Object getValue();

    /**
     * Returns whether or not the property is set.
     * @return <code>true</code> if the property is set.
     */
    boolean isSet();
  }

  /**
   * Returns a list of {@link ChangeSummary.Setting settings} 
   * that represent the property values of the given <code>dataObject</code>
   * at the point when logging {@link #beginLogging() began}.
   * <p>In the case of a {@link #isDeleted(DataObject) deleted} object, 
   * the list will include settings for all the properties.
   * @param dataObject the object in question.
   * @return a list of settings.
   * @see #getChangedDataObjects
   */
  List /*ChangeSummary.Setting*/ getOldValues(DataObject dataObject); 
}
