# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.core.utils import defer
from elisa.core.utils import caching

from elisa.plugins.base import utils
from elisa.plugins.base.models.media import RawDataModel

from twisted.trial.unittest import TestCase

import os


class _ItemMock(object):
    pass


class _ApplicationMock(object):
    pass


class _ResourceManagerMock(object):

    def get(self, uri):
        model = RawDataModel()
        model.data = str(uri)
        model.size = len(model.data)
        return model, defer.succeed(model)


class TestUtils(TestCase):

    def setUp(self):
        self._application_orig = common.application
        common.set_application(_ApplicationMock())
        common.application.resource_manager = _ResourceManagerMock()
        self._cache_path = os.path.abspath(self.mktemp())
        if not os.path.exists(self._cache_path):
            os.mkdir(self._cache_path)
        self._get_cache_path_orig = caching.get_cache_path
        caching.get_cache_path = self._get_cache_path

    def tearDown(self):
        caching.get_cache_path = self._get_cache_path_orig
        common.set_application(self._application_orig)

    def _get_cache_path(self, cache_dir):
        return self._cache_path

    def test_get_and_cache_image_local(self):
        tempfile = os.path.abspath(self.mktemp())
        open(tempfile, 'wb').close()
        uri = MediaUri('file://%s' % tempfile)
        dfr = utils.get_and_cache_image(uri)
        dfr.addCallback(self.failUnlessEqual, tempfile)
        return dfr

    def test_get_and_cache_image_distant(self):
        uri = MediaUri('http://example.com/image.jpg')
        dfr = utils.get_and_cache_image(uri)
        expected = os.path.join(self._cache_path,
                                '442719ec3cb80e9790fffe4cef1385ad.jpg')
        dfr.addCallback(self.failUnlessEqual, expected)
        return dfr

    def _check_thumbnail_path(self, value, item, expected):
        self.failUnlessEqual(value, expected)
        self.failUnless(hasattr(item, 'thumbnail_path'))
        self.failUnlessEqual(item.thumbnail_path, expected)

    def test_get_and_cache_thumbnail_already_cached(self):
        item = _ItemMock()
        thumbnail_path = u'path_to_file'
        item.thumbnail_path = thumbnail_path
        thumbnail_uri = None # unused here
        dfr = utils.get_and_cache_thumbnail(item, thumbnail_uri)
        dfr.addCallback(self._check_thumbnail_path, item, thumbnail_path)
        return dfr

    def test_get_and_cache_thumbnail(self):
        item = _ItemMock()
        thumbnail_uri = MediaUri('http://example.com/image.jpg')
        thumbnail_path = os.path.join(self._cache_path,
                                      '442719ec3cb80e9790fffe4cef1385ad.jpg')
        dfr = utils.get_and_cache_thumbnail(item, thumbnail_uri)
        dfr.addCallback(self._check_thumbnail_path, item, thumbnail_path)
        return dfr
