# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

from elisa.core import common
from elisa.core.components.input_provider import PushInputProvider
from elisa.core.input_event import *
from elisa.plugins.pigment.message import PigmentFrontendLoadedMessage

import gobject
import win32api

WM_APPCOMMAND = 0x319

# defined in winuser.h
def GET_APPCOMMAND_LPARAM(lparam):
    return win32api.HIWORD(lparam) & ~0xF000

APPCOMMAND_BROWSER_BACKWARD=1
APPCOMMAND_BROWSER_STOP=4
APPCOMMAND_VOLUME_MUTE=8
APPCOMMAND_VOLUME_DOWN=9
APPCOMMAND_VOLUME_UP=10
APPCOMMAND_MEDIA_NEXTTRACK=11
APPCOMMAND_MEDIA_PREVIOUSTRACK=12
APPCOMMAND_MEDIA_STOP=13
APPCOMMAND_MEDIA_PLAY_PAUSE=14
APPCOMMAND_CLOSE=31
APPCOMMAND_MEDIA_PLAY=46
APPCOMMAND_MEDIA_PAUSE=47
APPCOMMAND_MEDIA_FAST_FORWARD=49
APPCOMMAND_MEDIA_REWIND=50

class MultimediaKeyboardInput(PushInputProvider):
    """
    This class implements win32 support for multimedia keyboards
    """
    convert_table = {
        APPCOMMAND_BROWSER_BACKWARD  : EventValue.KEY_MENU,
        APPCOMMAND_MEDIA_PLAY : EventValue.KEY_PLAY,
        APPCOMMAND_MEDIA_PAUSE : EventValue.KEY_PAUSE,
        APPCOMMAND_MEDIA_STOP : EventValue.KEY_STOP,
        APPCOMMAND_MEDIA_PLAY_PAUSE: EventValue.KEY_PLAY_PAUSE,
        APPCOMMAND_MEDIA_FAST_FORWARD : EventValue.KEY_SEEK_FORWARD,
        APPCOMMAND_MEDIA_REWIND : EventValue.KEY_SEEK_BACKWARD,
        APPCOMMAND_MEDIA_NEXTTRACK : EventValue.KEY_NEXT,
        APPCOMMAND_MEDIA_PREVIOUSTRACK : EventValue.KEY_PREVIOUS,
        APPCOMMAND_VOLUME_MUTE  : EventValue.KEY_MUTE,
        APPCOMMAND_VOLUME_DOWN : EventValue.KEY_VOL_DOWN,
        APPCOMMAND_VOLUME_UP : EventValue.KEY_VOL_UP,
        APPCOMMAND_CLOSE : EventValue.KEY_EXIT,
        APPCOMMAND_BROWSER_STOP: EventValue.KEY_EXIT,
        }

    def initialize(self):
        self._signal_handler = None
        dfr = super(MultimediaKeyboardInput, self).initialize()
        bus = common.application.bus
        bus.register(self._frontend_loaded, PigmentFrontendLoadedMessage)
        return dfr

    def _frontend_loaded(self, msg, frontend):
        frontend.windows_msg_handler.add_wndproc(WM_APPCOMMAND,
                                                 self._got_win32_message)

    def _got_win32_message(self, frontend, event):
        return self.create_input_event(event)

    def unbind(self):
        if self._signal_handler:
            gobject.disconnect(self._signal_handler)
            self._signal_handler = None

    def create_input_event(self, event):
        self.info("Callback create_input_event called")
        code = GET_APPCOMMAND_LPARAM(event.lparam)
        if code in self.convert_table:
            self.info("Supported key code received: %d", code)
            evt = InputEvent(EventSource.KEYBOARD, EventType.OTHER,
                             self.convert_table[code])
            self.input_manager.process_event(evt)
            return True
        else:
            self.info('Unsupported key code received: %d', code)
            return False
