# Copyright 2000 by Jeffrey Chang, Brad Chapman.  All rights reserved.
# Copyright 2006, 2007 by Peter Cock.  All rights reserved.
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.

"""Code to work with GenBank formatted files.

Classes:
Iterator              Iterate through a file of GenBank entries
Dictionary            Access a GenBank file using a dictionary interface.
ErrorFeatureParser    Catch errors caused during parsing.
FeatureParser         Parse GenBank data in Seq and SeqFeature objects.
RecordParser          Parse GenBank data into a Record object.
NCBIDictionary        Access GenBank using a dictionary interface.

_BaseGenBankConsumer  A base class for GenBank consumer that implements
                      some helpful functions that are in common between
                      consumers.
_FeatureConsumer      Create SeqFeature objects from info generated by
                      the Scanner
_RecordConsumer       Create a GenBank record object from Scanner info.
_PrintingConsumer     A debugging consumer.

ParserFailureError    Exception indicating a failure in the parser (ie.
                      scanner or consumer)
LocationParserError   Exception indiciating a problem with the spark based
                      location parser.

Functions:
index_file            Get a GenBank file ready to be used as a Dictionary.
search_for            Do a query against GenBank.
download_many         Download many GenBank records.

"""
import cStringIO

# other Biopython stuff
from Bio.ParserSupport import AbstractConsumer
from utils import FeatureValueCleaner

#There used to be a (GenBank only) class _Scanner in
#this file.  Now use a more generic system which we import:
from Scanner import GenBankScanner

#These are used by the index_file function and Dictionary class:
#Moved the import inside the deprecated functions
#from Bio import Mindy
#from Bio.Mindy import SimpleSeqRecord

#These are used for downloading files from GenBank
from Bio import EUtils
from Bio.EUtils import DBIds, DBIdsClient

#Constants used to parse GenBank header lines
GENBANK_INDENT = 12
GENBANK_SPACER = " " * GENBANK_INDENT

#Constants for parsing GenBank feature lines
FEATURE_KEY_INDENT = 5
FEATURE_QUALIFIER_INDENT = 21
FEATURE_KEY_SPACER = " " * FEATURE_KEY_INDENT
FEATURE_QUALIFIER_SPACER = " " * FEATURE_QUALIFIER_INDENT
        
class Dictionary:
    """Access a GenBank file using a dictionary-like interface.
    """
    def __init__(self, indexname, parser = None):
        """Initialize and open up a GenBank dictionary. DEPRECATED

        Each entry is a full GenBank record (i.e. from the LOCUS line
        to the // at  the end of the sequence).

        Most GenBank files have only one such "entry", in which case
        using this dictionary class is rather unnecessary.

        Arguments:
        o indexname - The name of the index for the file. This should have been
        created using the index_file function.
        o parser - An optional argument specifying a parser object that
        the records should be run through before returning the output. If
        parser is None then the unprocessed contents of the file will be
        returned.
        """

        import warnings
        warnings.warn("Bio.GenBank.index_file Bio.GenBank.Dictionary are deprecated." \
                      + " We hope an in memory dictionary, for example using the" \
                      + " Bio.SeqIO.to_dict() function, will be suitable for" \
                      + " most users.  Please get in touch on the mailing lists if" \
                      + " this (or its removal) causes any problems for you.",
                      DeprecationWarning)

        from Bio import Mindy
        self._index = Mindy.open(indexname)
        self._parser = parser
    
    def __len__(self):
        return len(self.keys())

    def __getitem__(self, key):
        # first try to retrieve by the base id
        try:
            seqs = self._index.lookup(id = key)
        # if we can't do that, we have to try and fetch by alias
        except KeyError:
            seqs = self._index.lookup(aliases = key)

        if len(seqs) == 1:
            seq = seqs[0]
        else:
            raise KeyError("Multiple sequences found for %s" % key)

        if self._parser:
            handle = cStringIO.StringIO(seq.text)
            return self._parser.parse(handle)
        else:
            return seq.text

    def keys(self):
        primary_key_retriever = self._index['id']
        return primary_key_retriever.keys()
        
class Iterator:
    """Iterator interface to move over a file of GenBank entries one at a time.
    """
    def __init__(self, handle, parser = None):
        """Initialize the iterator.

        Arguments:
        o handle - A handle with GenBank entries to iterate through.
        o parser - An optional parser to pass the entries through before
        returning them. If None, then the raw entry will be returned.
        """
        self.handle = handle
        self._parser = parser

    def next(self):
        """Return the next GenBank record from the handle.

        Will return None if we ran out of records.
        """
        if self._parser is None :
            lines = []
            while True :
                line = self.handle.readline()
                if not line : return None #Premature end of file?
                lines.append(line)
                if line.rstrip() == "//" : break
            return "".join(lines)
        try :
            return self._parser.parse(self.handle)
        except StopIteration :
            return None

    def __iter__(self):
        return iter(self.next, None)

class ParserFailureError(Exception):
    """Failure caused by some kind of problem in the parser.
    """
    pass

class LocationParserError(Exception):
    """Could not Properly parse out a location from a GenBank file.
    """
    pass
                                                          
class FeatureParser:
    """Parse GenBank files into Seq + Feature objects.
    """
    def __init__(self, debug_level = 0, use_fuzziness = 1, 
                 feature_cleaner = FeatureValueCleaner()):
        """Initialize a GenBank parser and Feature consumer.

        Arguments:
        o debug_level - An optional argument that species the amount of
        debugging information the parser should spit out. By default we have
        no debugging info (the fastest way to do things), but if you want
        you can set this as high as two and see exactly where a parse fails.
        o use_fuzziness - Specify whether or not to use fuzzy representations.
        The default is 1 (use fuzziness).
        o feature_cleaner - A class which will be used to clean out the
        values of features. This class must implement the function 
        clean_value. GenBank.utils has a "standard" cleaner class, which
        is used by default.
        """
        self._scanner = GenBankScanner(debug_level)
        self.use_fuzziness = use_fuzziness
        self._cleaner = feature_cleaner

    def parse(self, handle):
        """Parse the specified handle.
        """
        self._consumer = _FeatureConsumer(self.use_fuzziness, 
                                          self._cleaner)
        self._scanner.feed(handle, self._consumer)
        return self._consumer.data

class RecordParser:
    """Parse GenBank files into Record objects
    """
    def __init__(self, debug_level = 0):
        """Initialize the parser.

        Arguments:
        o debug_level - An optional argument that species the amount of
        debugging information the parser should spit out. By default we have
        no debugging info (the fastest way to do things), but if you want
        you can set this as high as two and see exactly where a parse fails.
        """
        self._scanner = GenBankScanner(debug_level)

    def parse(self, handle):
        """Parse the specified handle into a GenBank record.
        """
        self._consumer = _RecordConsumer()
        self._scanner.feed(handle, self._consumer)
        return self._consumer.data

class _BaseGenBankConsumer(AbstractConsumer):
    """Abstract GenBank consumer providing useful general functions.

    This just helps to eliminate some duplication in things that most
    GenBank consumers want to do.
    """
    # Special keys in GenBank records that we should remove spaces from
    # For instance, \translation keys have values which are proteins and
    # should have spaces and newlines removed from them. This class
    # attribute gives us more control over specific formatting problems.
    remove_space_keys = ["translation"]

    def __init__(self):
        pass

    def _split_keywords(self, keyword_string):
        """Split a string of keywords into a nice clean list.
        """
        # process the keywords into a python list
        if keyword_string == "" or keyword_string == "." :
            keywords = ""
        elif keyword_string[-1] == '.':
            keywords = keyword_string[:-1]
        else:
            keywords = keyword_string
        keyword_list = keywords.split(';')
        clean_keyword_list = [x.strip() for x in keyword_list]
        return clean_keyword_list

    def _split_accessions(self, accession_string):
        """Split a string of accession numbers into a list.
        """
        # first replace all line feeds with spaces
        # Also, EMBL style accessions are split with ';'
        accession = accession_string.replace("\n", " ").replace(";"," ")

        return [x.strip() for x in accession.split(' ')]

    def _split_taxonomy(self, taxonomy_string):
        """Split a string with taxonomy info into a list.
        """
        if not taxonomy_string or taxonomy_string=="." :
            #Missing data, no taxonomy
            return []
        
        if taxonomy_string[-1] == '.':
            tax_info = taxonomy_string[:-1]
        else:
            tax_info = taxonomy_string
        tax_list = tax_info.split(';')
        new_tax_list = []
        for tax_item in tax_list:
            new_items = tax_item.split("\n")
            new_tax_list.extend(new_items)
        while '' in new_tax_list:
            new_tax_list.remove('')
        clean_tax_list = [x.strip() for x in new_tax_list]

        return clean_tax_list

    def _clean_location(self, location_string):
        """Clean whitespace out of a location string.

        The location parser isn't a fan of whitespace, so we clean it out
        before feeding it into the parser.
        """
        import string
        location_line = location_string
        for ws in string.whitespace:
            location_line = location_line.replace(ws, '')

        return location_line

    def _remove_newlines(self, text):
        """Remove any newlines in the passed text, returning the new string.
        """
        # get rid of newlines in the qualifier value
        newlines = ["\n", "\r"]
        for ws in newlines:
            text = text.replace(ws, "")

        return text

    def _normalize_spaces(self, text):
        """Replace multiple spaces in the passed text with single spaces.
        """
        # get rid of excessive spaces
        text_parts = text.split(" ")
        text_parts = filter(None, text_parts)
        return ' '.join(text_parts)

    def _remove_spaces(self, text):
        """Remove all spaces from the passed text.
        """
        return text.replace(" ", "")

    def _convert_to_python_numbers(self, start, end):
        """Convert a start and end range to python notation.

        In GenBank, starts and ends are defined in "biological" coordinates,
        where 1 is the first base and [i, j] means to include both i and j.

        In python, 0 is the first base and [i, j] means to include i, but
        not j. 

        So, to convert "biological" to python coordinates, we need to 
        subtract 1 from the start, and leave the end and things should
        be converted happily.
        """
        new_start = start - 1
        new_end = end

        return new_start, new_end

class _FeatureConsumer(_BaseGenBankConsumer):
    """Create a SeqRecord object with Features to return.

    Attributes:
    o use_fuzziness - specify whether or not to parse with fuzziness in
    feature locations.
    o feature_cleaner - a class that will be used to provide specialized
    cleaning-up of feature values.
    """
    def __init__(self, use_fuzziness, feature_cleaner = None):
        from Bio.SeqRecord import SeqRecord
        _BaseGenBankConsumer.__init__(self)
        self.data = SeqRecord(None, id = None)
        self.data.id = None
        self.data.description = ""

        self._use_fuzziness = use_fuzziness
        self._feature_cleaner = feature_cleaner

        self._seq_type = ''
        self._seq_data = []
        self._current_ref = None
        self._cur_feature = None
        self._cur_qualifier_key = None
        self._cur_qualifier_value = None

    def locus(self, locus_name):
        """Set the locus name is set as the name of the Sequence.
        """
        self.data.name = locus_name

    def size(self, content):
        pass

    def residue_type(self, type):
        """Record the sequence type so we can choose an appropriate alphabet.
        """
        self._seq_type = type

    def data_file_division(self, division):
        self.data.annotations['data_file_division'] = division

    def date(self, submit_date):
        self.data.annotations['date'] = submit_date 

    def definition(self, definition):
        """Set the definition as the description of the sequence.
        """
        if self.data.description :
            #Append to any existing description
            #e.g. EMBL files with two DE lines.
            self.data.description += " " + definition
        else :
            self.data.description = definition

    def accession(self, acc_num):
        """Set the accession number as the id of the sequence.

        If we have multiple accession numbers, the first one passed is
        used.
        """
        new_acc_nums = self._split_accessions(acc_num)

        #Also record them ALL in the annotations
        try :
            #On the off chance there was more than one accession line:
            self.data.annotations['accessions'].extend(new_acc_nums)
        except KeyError :
            self.data.annotations['accessions'] = new_acc_nums

        # if we haven't set the id information yet, add the first acc num
        if self.data.id is None:
            if len(new_acc_nums) > 0:
                #self.data.id = new_acc_nums[0]
                #Use the FIRST accession as the ID, not the first on this line!
                self.data.id = self.data.annotations['accessions'][0]
                

    def nid(self, content):
        self.data.annotations['nid'] = content

    def pid(self, content):
        self.data.annotations['pid'] = content

    def version(self, version_id):
        #Want to use the versioned accension as the record.id
        #This comes from the VERSION line in GenBank files, or the
        #obsolete SV line in EMBL.  For the new EMBL files we need
        #both the version suffix from the ID line and the accession
        #from the AC line.
        if version_id.count(".")==1 and version_id.split(".")[1].isdigit() :
            self.accession(version_id.split(".")[0])
            self.version_suffix(version_id.split(".")[1])
        else :
            #For backwards compatibility...
            self.data.id = version_id

    def version_suffix(self, version):
        """Set the version to overwrite the id.

        Since the verison provides the same information as the accession
        number, plus some extra info, we set this as the id if we have
        a version.
        """
        #e.g. GenBank line:
        #VERSION     U49845.1  GI:1293613
        #or the obsolete EMBL line:
        #SV   U49845.1
        #Scanner calls consumer.version("U49845.1")
        #which then calls consumer.version_suffix(1)
        #
        #e.g. EMBL new line:
        #ID   X56734; SV 1; linear; mRNA; STD; PLN; 1859 BP.
        #Scanner calls consumer.version_suffix(1)
        assert version.isdigit()
        self.data.annotations['sequence_version'] = int(version)

    def db_source(self, content):
        self.data.annotations['db_source'] = content.rstrip()

    def gi(self, content):
        self.data.annotations['gi'] = content

    def keywords(self, content):
        self.data.annotations['keywords'] = self._split_keywords(content)

    def segment(self, content):
        self.data.annotations['segment'] = content

    def source(self, content):
        if content[-1] == '.':
            source_info = content[:-1]
        else:
            source_info = content
        self.data.annotations['source'] = source_info

    def organism(self, content):
        self.data.annotations['organism'] = content

    def taxonomy(self, content):
        self.data.annotations['taxonomy'] = self._split_taxonomy(content)
        
    def reference_num(self, content):
        """Signal the beginning of a new reference object.
        """
        from Bio.SeqFeature import Reference
        # if we have a current reference that hasn't been added to
        # the list of references, add it.
        if self._current_ref is not None:
            self.data.annotations['references'].append(self._current_ref)
        else:
            self.data.annotations['references'] = []

        self._current_ref = Reference()

    def reference_bases(self, content):
        """Attempt to determine the sequence region the reference entails.

        Possible types of information we may have to deal with:
        
        (bases 1 to 86436)
        (sites)
        (bases 1 to 105654; 110423 to 111122)
        1  (residues 1 to 182)
        """
        # first remove the parentheses or other junk
        ref_base_info = content[1:-1]

        all_locations = []
        # parse if we've got 'bases' and 'to'
        if ref_base_info.find('bases') != -1 and \
            ref_base_info.find('to') != -1:
            # get rid of the beginning 'bases'
            ref_base_info = ref_base_info[5:]
            locations = self._split_reference_locations(ref_base_info)
            all_locations.extend(locations)
        elif (ref_base_info.find("residues") >= 0 and
              ref_base_info.find("to") >= 0):
            residues_start = ref_base_info.find("residues")
            # get only the information after "residues"
            ref_base_info = ref_base_info[(residues_start + len("residues ")):]
            locations = self._split_reference_locations(ref_base_info)
            all_locations.extend(locations)

        # make sure if we are not finding information then we have
        # the string 'sites' or the string 'bases'
        elif (ref_base_info == 'sites' or
              ref_base_info.strip() == 'bases'):
            pass
        # otherwise raise an error
        else:
            raise ValueError("Could not parse base info %s in record %s" %
                             (ref_base_info, self.data.id))

        self._current_ref.location = all_locations

    def _split_reference_locations(self, location_string):
        """Get reference locations out of a string of reference information
        
        The passed string should be of the form:

            1 to 20; 20 to 100

        This splits the information out and returns a list of location objects
        based on the reference locations.
        """
        from Bio import SeqFeature
        # split possibly multiple locations using the ';'
        all_base_info = location_string.split(';')

        new_locations = []
        for base_info in all_base_info:
            start, end = base_info.split('to')
            new_start, new_end = \
              self._convert_to_python_numbers(int(start.strip()),
                                              int(end.strip()))
            this_location = SeqFeature.FeatureLocation(new_start, new_end)
            new_locations.append(this_location)
        return new_locations

    def authors(self, content):
        self._current_ref.authors = content

    def consrtm(self, content):
        self._current_ref.consrtm = content

    def title(self, content):
        self._current_ref.title = content

    def journal(self, content):
        self._current_ref.journal = content

    def medline_id(self, content):
        self._current_ref.medline_id = content

    def pubmed_id(self, content):
        self._current_ref.pubmed_id = content

    def remark(self, content):
        self._current_ref.comment = content

    def comment(self, content):
        try :
            self.data.annotations['comment'] += "\n" + "\n".join(content)
        except KeyError :
            self.data.annotations['comment'] = "\n".join(content)

    def features_line(self, content):
        """Get ready for the feature table when we reach the FEATURE line.
        """
        self.start_feature_table()

    def start_feature_table(self):
        """Indicate we've got to the start of the feature table.
        """
        # make sure we've added on our last reference object
        if self._current_ref is not None:
            self.data.annotations['references'].append(self._current_ref)
            self._current_ref = None

    def _add_feature(self):
        """Utility function to add a feature to the SeqRecord.

        This does all of the appropriate checking to make sure we haven't
        left any info behind, and that we are only adding info if it
        exists.
        """
        if self._cur_feature:
            # if we have a left over qualifier, add it to the qualifiers
            # on the current feature
            self._add_qualifier()

            self._cur_qualifier_key = ''
            self._cur_qualifier_value = ''
            self.data.features.append(self._cur_feature)
            
    def feature_key(self, content):
        from Bio import SeqFeature
        # if we already have a feature, add it on
        self._add_feature()

        # start a new feature
        self._cur_feature = SeqFeature.SeqFeature()
        self._cur_feature.type = content

        # assume positive strand to start with if we have DNA or cDNA
        # (labelled as mRNA). The complement in the location will 
        # change this later if something is on the reverse strand
        if self._seq_type.find("DNA") >= 0 or self._seq_type.find("mRNA") >= 0:
            self._cur_feature.strand = 1

    def location(self, content):
        """Parse out location information from the location string.

        This uses Andrew's nice spark based parser to do the parsing
        for us, and translates the results of the parse into appropriate
        Location objects.
        """
        from Bio.GenBank import LocationParser
        # --- first preprocess the location for the spark parser
        
        # we need to clean up newlines and other whitespace inside
        # the location before feeding it to the parser.
        # locations should have no whitespace whatsoever based on the
        # grammer
        location_line = self._clean_location(content)

        # Older records have junk like replace(266,"c") in the
        # location line. Newer records just replace this with
        # the number 266 and have the information in a more reasonable
        # place. So we'll just grab out the number and feed this to the
        # parser. We shouldn't really be losing any info this way.
        if location_line.find('replace') != -1:
            comma_pos = location_line.find(',')
            location_line = location_line[8:comma_pos]
        
        # feed everything into the scanner and parser
        try:
            parse_info = \
                       LocationParser.parse(LocationParser.scan(location_line))
        # spark raises SystemExit errors when parsing fails
        except SystemExit:
            raise LocationParserError(location_line)

        # print "parse_info:", repr(parse_info)
        
        # add the parser information the current feature
        self._set_location_info(parse_info, self._cur_feature)

    def _set_function(self, function, cur_feature):
        """Set the location information based on a function.

        This handles all of the location functions like 'join', 'complement'
        and 'order'.

        Arguments:
        o function - A LocationParser.Function object specifying the
        function we are acting on.
        o cur_feature - The feature to add information to.
        """
        from Bio import SeqFeature
        from Bio.GenBank import LocationParser
        assert isinstance(function, LocationParser.Function), \
               "Expected a Function object, got %s" % function
        
        if function.name == "complement":
            # mark the current feature as being on the opposite strand
            cur_feature.strand = -1
            # recursively deal with whatever is left inside the complement
            for inner_info in function.args:
                self._set_location_info(inner_info, cur_feature)
        # deal with functions that have multipe internal segments that
        # are connected somehow.
        # join and order are current documented functions.
        # one-of is something I ran across in old files. Treating it
        # as a sub sequence feature seems appropriate to me.
        # bond is some piece of junk I found in RefSeq files. I have
        # no idea how to interpret it, so I jam it in here
        elif (function.name == "join" or function.name == "order" or
              function.name == "one-of" or function.name == "bond"):
            self._set_ordering_info(function, cur_feature)
        elif (function.name == "gap"):
            assert len(function.args) == 1, \
              "Unexpected number of arguments in gap %s" % function.args
            # make the cur information location a gap object
            position = self._get_position(function.args[0].local_location)
            cur_feature.location = SeqFeature.PositionGap(position)
        else:
            raise ValueError("Unexpected function name: %s" % function.name)

    def _set_ordering_info(self, function, cur_feature):
        """Parse a join or order and all of the information in it.

        This deals with functions that order a bunch of locations,
        specifically 'join' and 'order'. The inner locations are
        added as subfeatures of the top level feature
        """
        from Bio import SeqFeature
        # for each inner element, create a sub SeqFeature within the
        # current feature, then get the information for this feature
        for inner_element in function.args:
            new_sub_feature = SeqFeature.SeqFeature()
            # inherit the type from the parent
            new_sub_feature.type = cur_feature.type 
            # add the join or order info to the location_operator
            cur_feature.location_operator = function.name
            new_sub_feature.location_operator = function.name
            # inherit references and strand from the parent feature
            new_sub_feature.ref = cur_feature.ref
            new_sub_feature.ref_db = cur_feature.ref_db
            new_sub_feature.strand = cur_feature.strand

            # set the information for the inner element
            self._set_location_info(inner_element, new_sub_feature)

            # now add the feature to the sub_features
            cur_feature.sub_features.append(new_sub_feature)

        # set the location of the top -- this should be a combination of
        # the start position of the first sub_feature and the end position
        # of the last sub_feature

        # these positions are already converted to python coordinates 
        # (when the sub_features were added) so they don't need to
        # be converted again
        feature_start = cur_feature.sub_features[0].location.start
        feature_end = cur_feature.sub_features[-1].location.end
        cur_feature.location = SeqFeature.FeatureLocation(feature_start,
                                                          feature_end)

    def _set_location_info(self, parse_info, cur_feature):
        """Set the location information for a feature from the parse info.

        Arguments:
        o parse_info - The classes generated by the LocationParser.
        o cur_feature - The feature to add the information to.
        """
        from Bio.GenBank import LocationParser
        # base case -- we are out of information
        if parse_info is None:
            return
        # parse a location -- this is another base_case -- we assume
        # we have no information after a single location
        elif isinstance(parse_info, LocationParser.AbsoluteLocation):
            self._set_location(parse_info, cur_feature)
            return
        # parse any of the functions (join, complement, etc)
        elif isinstance(parse_info, LocationParser.Function):
            self._set_function(parse_info, cur_feature)
        # otherwise we are stuck and should raise an error
        else:
            raise ValueError("Could not parse location info: %s"
                             % parse_info)

    def _set_location(self, location, cur_feature):
        """Set the location information for a feature.

        Arguments:
        o location - An AbsoluteLocation object specifying the info
        about the location.
        o cur_feature - The feature to add the information to.
        """
        # check to see if we have a cross reference to another accession
        # ie. U05344.1:514..741
        if location.path is not None:
            cur_feature.ref = location.path.accession
            cur_feature.ref_db = location.path.database
        # now get the actual location information
        cur_feature.location = self._get_location(location.local_location)

    def _get_location(self, range_info):
        """Return a (possibly fuzzy) location from a Range object.

        Arguments:
        o range_info - A location range (ie. something like 67..100). This
        may also be a single position (ie 27).

        This returns a FeatureLocation object.
        If parser.use_fuzziness is set at one, the positions for the
        end points will possibly be fuzzy.
        """
        from Bio import SeqFeature
        from Bio.GenBank import LocationParser
        # check if we just have a single base
        if not(isinstance(range_info, LocationParser.Range)):
            pos = self._get_position(range_info)
            # move the single position back one to be consistent with how
            # python indexes numbers (starting at 0)
            pos.position = pos.position  - 1
            return SeqFeature.FeatureLocation(pos, pos)
        # otherwise we need to get both sides of the range
        else:
            # get *Position objects for the start and end
            start_pos = self._get_position(range_info.low)
            end_pos = self._get_position(range_info.high)

            start_pos.position, end_pos.position = \
              self._convert_to_python_numbers(start_pos.position,
                                              end_pos.position)

            return SeqFeature.FeatureLocation(start_pos, end_pos)

    def _get_position(self, position):
        """Return a (possibly fuzzy) position for a single coordinate.

        Arguments:
        o position - This is a LocationParser.* object that specifies
        a single coordinate. We will examine the object to determine
        the fuzziness of the position.

        This is used with _get_location to parse out a location of any
        end_point of arbitrary fuzziness.
        """
        from Bio import SeqFeature
        from Bio.GenBank import LocationParser
        # case 1 -- just a normal number
        if (isinstance(position, LocationParser.Integer)):
            final_pos = SeqFeature.ExactPosition(position.val) 
        # case 2 -- we've got a > sign
        elif isinstance(position, LocationParser.LowBound):
            final_pos = SeqFeature.AfterPosition(position.base.val)
        # case 3 -- we've got a < sign
        elif isinstance(position, LocationParser.HighBound):
            final_pos = SeqFeature.BeforePosition(position.base.val)
        # case 4 -- we've got 100^101
        elif isinstance(position, LocationParser.Between):
            final_pos = SeqFeature.BetweenPosition(position.low.val,
                                                 position.high.val)
        # case 5 -- we've got (100.101)
        elif isinstance(position, LocationParser.TwoBound):
            final_pos = SeqFeature.WithinPosition(position.low.val,
                                                position.high.val)
        # case 6 -- we've got a one-of(100, 110) location
        elif isinstance(position, LocationParser.Function) and \
                        position.name == "one-of":
            # first convert all of the arguments to positions
            position_choices = []
            for arg in position.args:
                # we only handle AbsoluteLocations with no path
                # right now. Not sure if other cases will pop up
                assert isinstance(arg, LocationParser.AbsoluteLocation), \
                  "Unhandled Location type %r" % arg
                assert arg.path is None, "Unhandled path in location"
                position = self._get_position(arg.local_location)
                position_choices.append(position)
            final_pos = SeqFeature.OneOfPosition(position_choices)
        # if it is none of these cases we've got a problem!
        else:
            raise ValueError("Unexpected LocationParser object %r" %
                             position)

        # if we are using fuzziness return what we've got
        if self._use_fuzziness:
            return final_pos
        # otherwise return an ExactPosition equivalent
        else:
            return SeqFeature.ExactPosition(final_pos.location)

    def _add_qualifier(self):
        """Add a qualifier to the current feature without loss of info.

        If there are multiple qualifier keys with the same name we
        would lose some info in the dictionary, so we append a unique
        number to the end of the name in case of conflicts.
        """
        # if we've got a key from before, add it to the dictionary of
        # qualifiers
        if self._cur_qualifier_key:
            key = self._cur_qualifier_key
            value = "".join(self._cur_qualifier_value)
            if self._feature_cleaner is not None:
                value = self._feature_cleaner.clean_value(key, value)
            # if the qualifier name exists, append the value
            if key in self._cur_feature.qualifiers:
                self._cur_feature.qualifiers[key].append(value)
            # otherwise start a new list of the key with its values
            else:
                self._cur_feature.qualifiers[key] = [value]

    def feature_qualifier_name(self, content_list):
        """When we get a qualifier key, use it as a dictionary key.
        
        We receive a list of keys, since you can have valueless keys such as
        /pseudo which would be passed in with the next key (since no other
        tags separate them in the file)
        """
        for content in content_list:
            # add a qualifier if we've got one
            self._add_qualifier()

            # remove the / and = from the qualifier if they're present
            qual_key = content.replace('/', '')
            qual_key = qual_key.replace('=', '')
            qual_key = qual_key.strip()
            
            self._cur_qualifier_key = qual_key
            self._cur_qualifier_value = []
        
    def feature_qualifier_description(self, content):
        # get rid of the quotes surrounding the qualifier if we've got 'em
        qual_value = content.replace('"', '')
        
        self._cur_qualifier_value.append(qual_value)

    def contig_location(self, content):
        """Deal with a location of CONTIG information.
        """
        from Bio import SeqFeature
        # add a last feature if is hasn't been added,
        # so that we don't overwrite it
        self._add_feature()

        # make a feature to add the information to
        self._cur_feature = SeqFeature.SeqFeature()
        self._cur_feature.type = "contig"
        
        # now set the location on the feature using the standard
        # location handler
        self.location(content)
        # add the contig information to the annotations and get rid
        # of the feature to prevent it from being added to the feature table
        self.data.annotations["contig"] = self._cur_feature
        self._cur_feature = None

    def origin_name(self, content):
        pass

    def base_count(self, content):
        pass

    def base_number(self, content):
        pass

    def sequence(self, content):
        """Add up sequence information as we get it.

        To try and make things speedier, this puts all of the strings
        into a list of strings, and then uses string.join later to put
        them together. Supposedly, this is a big time savings
        """
        new_seq = content.replace(' ', '')
        new_seq = new_seq.upper()

        self._seq_data.append(new_seq)

    def record_end(self, content):
        """Clean up when we've finished the record.
        """
        from Bio import Alphabet
        from Bio.Alphabet import IUPAC
        from Bio.Seq import Seq

        #Try and append the version number to the accession for the full id
        if self.data.id.count('.') == 0 :
            try :
                self.data.id+='.%i' % self.data.annotations['sequence_version']
            except KeyError :
                pass
        
        # add the last feature in the table which hasn't been added yet
        self._add_feature()

        # add the sequence information
        # first, determine the alphabet
        # we default to an generic alphabet if we don't have a
        # seq type or have strange sequence information.
        seq_alphabet = Alphabet.generic_alphabet

        if self._seq_type:
            # mRNA is really also DNA, since it is actually cDNA
            if self._seq_type.find('DNA') != -1 or \
               self._seq_type.find('mRNA') != -1:
                seq_alphabet = IUPAC.ambiguous_dna
            # are there every really RNA sequences in GenBank?
            elif self._seq_type.find('RNA') != -1:
                seq_alphabet = IUPAC.ambiguous_rna
            elif self._seq_type.find('PROTEIN') != -1 :
                seq_alphabet = IUPAC.protein  # or extended protein?
            # work around ugly GenBank records which have circular or
            # linear but no indication of sequence type
            elif self._seq_type in ["circular", "linear"]:
                pass
            # we have a bug if we get here
            else:
                raise ValueError("Could not determine alphabet for seq_type %s"
                                 % self._seq_type)

        # now set the sequence
        sequence = "".join(self._seq_data)
        self.data.seq = Seq(sequence, seq_alphabet)

class _RecordConsumer(_BaseGenBankConsumer):
    """Create a GenBank Record object from scanner generated information.
    """
    def __init__(self):
        _BaseGenBankConsumer.__init__(self)
        import Record
        self.data = Record.Record()

        self._seq_data = []
        self._cur_reference = None
        self._cur_feature = None
        self._cur_qualifier = None
        
    def locus(self, content):
        self.data.locus = content

    def size(self, content):
        self.data.size = content

    def residue_type(self, content):
        self.data.residue_type = content

    def data_file_division(self, content):
        self.data.data_file_division = content

    def date(self, content):
        self.data.date = content

    def definition(self, content):
        self.data.definition = content

    def accession(self, content):
        new_accessions = self._split_accessions(content)
        self.data.accession.extend(new_accessions)

    def nid(self, content):
        self.data.nid = content

    def pid(self, content):
        self.data.pid = content

    def version(self, content):
        self.data.version = content

    def db_source(self, content):
        self.data.db_source = content.rstrip()

    def gi(self, content):
        self.data.gi = content

    def keywords(self, content):
        self.data.keywords = self._split_keywords(content)

    def segment(self, content):
        self.data.segment = content

    def source(self, content):
        self.data.source = content

    def organism(self, content):
        self.data.organism = content

    def taxonomy(self, content):
        self.data.taxonomy = self._split_taxonomy(content)

    def reference_num(self, content):
        """Grab the reference number and signal the start of a new reference.
        """
        # check if we have a reference to add
        if self._cur_reference is not None:
            self.data.references.append(self._cur_reference)

        self._cur_reference = Record.Reference()
        self._cur_reference.number = content

    def reference_bases(self, content):
        self._cur_reference.bases = content

    def authors(self, content):
        self._cur_reference.authors = content

    def consrtm(self, content):
        self._cur_reference.consrtm = content

    def title(self, content):
        self._cur_reference.title = content

    def journal(self, content):
        self._cur_reference.journal = content

    def medline_id(self, content):
        self._cur_reference.medline_id = content
        
    def pubmed_id(self, content):
        self._cur_reference.pubmed_id = content

    def remark(self, content):
        self._cur_reference.remark = content
        
    def comment(self, content):
        self.data.comment += "\n".join(content)

    def primary_ref_line(self,content):
        """Data for the PRIMARY line"""
        self.data.primary.append(content)

    def primary(self,content):
        pass
    
    def features_line(self, content):
        """Get ready for the feature table when we reach the FEATURE line.
        """
        self.start_feature_table()

    def start_feature_table(self):
        """Signal the start of the feature table.
        """
        # we need to add on the last reference
        if self._cur_reference is not None:
            self.data.references.append(self._cur_reference)

    def feature_key(self, content):
        """Grab the key of the feature and signal the start of a new feature.
        """
        # first add on feature information if we've got any
        self._add_feature()

        self._cur_feature = Record.Feature()
        self._cur_feature.key = content

    def _add_feature(self):
        """Utility function to add a feature to the Record.

        This does all of the appropriate checking to make sure we haven't
        left any info behind, and that we are only adding info if it
        exists.
        """
        if self._cur_feature is not None:
            # if we have a left over qualifier, add it to the qualifiers
            # on the current feature
            if self._cur_qualifier is not None:
                self._cur_feature.qualifiers.append(self._cur_qualifier)

            self._cur_qualifier = None
            self.data.features.append(self._cur_feature)

    def location(self, content):
        self._cur_feature.location = self._clean_location(content)

    def feature_qualifier_name(self, content_list):
        """Deal with qualifier names
        
        We receive a list of keys, since you can have valueless keys such as
        /pseudo which would be passed in with the next key (since no other
        tags separate them in the file)
        """
        for content in content_list:
            # the record parser keeps the /s -- add them if we don't have 'em
            if content.find("/") != 0:
                content = "/%s" % content
            # add on a qualifier if we've got one
            if self._cur_qualifier is not None:
                self._cur_feature.qualifiers.append(self._cur_qualifier)

            self._cur_qualifier = Record.Qualifier()
            self._cur_qualifier.key = content

    def feature_qualifier_description(self, content):
        # if we have info then the qualifier key should have a ='s
        if self._cur_qualifier.key.find("=") == -1:
            self._cur_qualifier.key = "%s=" % self._cur_qualifier.key
        cur_content = self._remove_newlines(content)
        # remove all spaces from the value if it is a type where spaces
        # are not important
        for remove_space_key in self.__class__.remove_space_keys:
            if self._cur_qualifier.key.find(remove_space_key) >= 0:
                cur_content = self._remove_spaces(cur_content)
        self._cur_qualifier.value = self._normalize_spaces(cur_content)

    def base_count(self, content):
        self.data.base_counts = content

    def origin_name(self, content):
        self.data.origin = content

    def contig_location(self, content):
        """Signal that we have contig information to add to the record.
        """
        self.data.contig = self._clean_location(content) 

    def sequence(self, content):
        """Add sequence information to a list of sequence strings.

        This removes spaces in the data and uppercases the sequence, and
        then adds it to a list of sequences. Later on we'll join this
        list together to make the final sequence. This is faster than
        adding on the new string every time.
        """
        new_seq = content.replace(' ', '')
        self._seq_data.append(new_seq.upper())

    def record_end(self, content):
        """Signal the end of the record and do any necessary clean-up.
        """
        # add together all of the sequence parts to create the
        # final sequence string
        self.data.sequence = "".join(self._seq_data)
        # add on the last feature
        self._add_feature()

def _strip_and_combine(line_list):
    """Combine multiple lines of content separated by spaces.

    This function is used by the EventGenerator callback function to
    combine multiple lines of information. The lines are first
    stripped to remove whitepsace, and then combined so they are separated
    by a space. This is a simple minded way to combine lines, but should
    work for most cases.
    """
    # first strip out extra whitespace
    stripped_line_list = [x.strip() for x in line_list]

    # now combine everything with spaces
    return ' '.join(stripped_line_list)

def index_file(filename, indexname, rec2key = None, use_berkeley = 0):
    """Index a GenBank file to prepare it for use as a dictionary. DEPRECATED

    Arguments:
    filename - The name of the GenBank file to be indexed.
    indexname - The name of the index to create
    rec2key - A reference to a function object which, when called with a 
    SeqRecord object, will return a key to be used for the record. If no 
    function is specified then the records will be indexed by the 'id'
    attribute of the SeqRecord (the versioned GenBank id).
    use_berkeley - specifies whether to use the BerkeleyDB indexer, which 
    uses the bsddb3 wrappers around the embedded database Berkeley DB. By
    default, the standard flat file (non-Berkeley) indexes are used.
    """

    import warnings
    warnings.warn("Bio.GenBank.index_file Bio.GenBank.Dictionary are deprecated." \
                  + " We hope an in memory dictionary, for example using the" \
                  + " Bio.SeqIO.to_dict() function, will be suitable for" \
                  + " most users.  Please get in touch on the mailing lists if" \
                  + " this (or its removal) causes any problems for you.",
                  DeprecationWarning)

    from Bio.Mindy import SimpleSeqRecord
    if rec2key:
        indexer = SimpleSeqRecord.FunctionIndexer(rec2key)
    else:
        indexer = SimpleSeqRecord.SimpleIndexer()

    if use_berkeley:
        SimpleSeqRecord.create_berkeleydb([filename], indexname, indexer)
    else:
        SimpleSeqRecord.create_flatdb([filename], indexname, indexer)

class NCBIDictionary:
    """Access GenBank using a read-only dictionary interface.
    """
    VALID_DATABASES = ['nucleotide', 'protein', 'genome']
    VALID_FORMATS = ['genbank', 'fasta']
    def __init__(self, database, format, parser = None):
        """Initialize an NCBI dictionary to retrieve sequences.

        Create a new Dictionary to access GenBank.  Valid values for
        database are 'nucleotide' and 'protein'.
        Valid values for format are 'genbank' (for nucleotide genbank and
        protein genpept) and 'fasta'.
        dely and retmax are old options kept only for compatibility -- do not
        bother to set them.
        parser is an optional parser object
        to change the results into another form.  If unspecified, then
        the raw contents of the file will be returned.
        """
        from Bio import db
        self.parser = parser
        if database not in self.__class__.VALID_DATABASES:
            raise ValueError("Invalid database %s, should be one of %s" %
                    (database, self.__class__.VALID_DATABASES))
        if format not in self.__class__.VALID_FORMATS:
            raise ValueError("Invalid format %s, should be one of %s" %
                    (format, self.__class__.VALID_FORMATS))

        if format == 'fasta':
            self.db = db["fasta-sequence-eutils"]
        elif format == 'genbank':
            if database == 'nucleotide':
                self.db = db["nucleotide-genbank-eutils"]
            elif database == 'protein':
                self.db = db["protein-genbank-eutils"]
            elif database == 'genome':
                self.db = db["genome-genbank-eutils"]

    def __len__(self):
        raise NotImplementedError, "GenBank contains lots of entries"
    def clear(self):
        raise NotImplementedError, "This is a read-only dictionary"
    def __setitem__(self, key, item):
        raise NotImplementedError, "This is a read-only dictionary"
    def update(self):
        raise NotImplementedError, "This is a read-only dictionary"
    def copy(self):
        raise NotImplementedError, "You don't need to do this..."
    def keys(self):
        raise NotImplementedError, "You don't really want to do this..."
    def items(self):
        raise NotImplementedError, "You don't really want to do this..."
    def values(self):
        raise NotImplementedError, "You don't really want to do this..."
    
    def has_key(self, id):
        """S.has_key(id) -> bool"""
        try:
            self[id]
        except KeyError:
            return 0
        return 1

    def get(self, id, failobj=None):
        try:
            return self[id]
        except KeyError:
            return failobj
        raise "How did I get here?"

    def __getitem__(self, id):
        """Return the GenBank entry specified by the GenBank ID.
        
        Raises a KeyError if there's an error.
        """
        handle = self.db[id]
        # Parse the record if a parser was passed in.
        if self.parser is not None:
            return self.parser.parse(handle)
        return handle.read()

def search_for(search, database='nucleotide',
               reldate=None, mindate=None, maxdate=None,
               start_id = 0, max_ids = 50000000):
    """search_for(search[, reldate][, mindate][, maxdate]
    [, batchsize][, delay][, callback_fn][, start_id][, max_ids]) -> ids

    Search GenBank and return a list of the GenBank identifiers (gi's)
    that match the criteria.  search is the search string used to
    search the database.  Valid values for database are
    'nucleotide', 'protein', 'popset' and 'genome'.  reldate is
    the number of dates prior to the current date to restrict the
    search.  mindate and maxdate are the dates to restrict the search,
    e.g. 2002/01/01.  start_id is the number to begin retrieval on.
    max_ids specifies the maximum number of id's to retrieve.
    
    batchsize, delay and callback_fn are old parameters for
    compatibility -- do not set them.
    """
    # deal with dates
    date_restrict = None
    if reldate:
        date_restrict = EUtils.WithinNDays(reldate)
    elif mindate:
        date_restrict = EUtils.DateRange(mindate, maxdate)

    eutils_client = DBIdsClient.DBIdsClient()
    db_ids = eutils_client.search(search, database, daterange = date_restrict,
            retstart = start_id, retmax = max_ids)
    ids = []
    for db_id in db_ids:
        ids.append(db_id.dbids.ids[0])
    return ids

def download_many(ids, database = 'nucleotide'):
    """download_many(ids, database) -> handle of results

    Download many records from GenBank.  ids is a list of gis or
    accessions.  

    callback_fn, broken_fn, delay, faildelay, batchsize, parser are old
    parameter for compatibility. They should not be used.
    """
    db_ids = DBIds(database, ids)
    if database in ['nucleotide']:
        format = 'gb'
    elif database in ['protein']:
        format = 'gp'
    else:
        raise ValueError("Unexpected database: %s" % database)

    eutils_client = DBIdsClient.from_dbids(db_ids)
    result_handle = eutils_client.efetch(retmode = "text", rettype = format)
    return cStringIO.StringIO(result_handle.read())

