require 'common'
require 'net/ssh/authentication/methods/publickey'
require 'authentication/methods/common'

module Authentication; module Methods

  class TestPublickey < Test::Unit::TestCase
    include Common

    def test_authenticate_should_return_false_when_no_key_manager_has_been_set
      assert_equal false, subject(:key_manager => nil).authenticate("ssh-connection", "jamis")
    end

    def test_authenticate_should_return_false_when_key_manager_has_no_keys
      assert_equal false, subject(:keys => []).authenticate("ssh-connection", "jamis")
    end

    def test_authenticate_should_return_false_if_no_keys_can_authenticate
      transport.expect do |t, packet|
        assert_equal USERAUTH_REQUEST, packet.type
        assert verify_userauth_request_packet(packet, keys.first, false)
        t.return(USERAUTH_FAILURE, :string, "hostbased,password")

        t.expect do |t, packet|
          assert_equal USERAUTH_REQUEST, packet.type
          assert verify_userauth_request_packet(packet, keys.last, false)
          t.return(USERAUTH_FAILURE, :string, "hostbased,password")
        end
      end

      assert_equal false, subject.authenticate("ssh-connection", "jamis")
    end

    def test_authenticate_should_return_false_if_signature_exchange_fails
      key_manager.expects(:sign).with(&signature_parameters(keys.first)).returns("sig-one")
      key_manager.expects(:sign).with(&signature_parameters(keys.last)).returns("sig-two")

      transport.expect do |t, packet|
        assert_equal USERAUTH_REQUEST, packet.type
        assert verify_userauth_request_packet(packet, keys.first, false)
        t.return(USERAUTH_PK_OK, :string, keys.first.ssh_type, :string, Net::SSH::Buffer.from(:key, keys.first))

        t.expect do |t,packet|
          assert_equal USERAUTH_REQUEST, packet.type
          assert verify_userauth_request_packet(packet, keys.first, true)
          assert_equal "sig-one", packet.read_string
          t.return(USERAUTH_FAILURE, :string, "hostbased,password")

          t.expect do |t, packet|
            assert_equal USERAUTH_REQUEST, packet.type
            assert verify_userauth_request_packet(packet, keys.last, false)
            t.return(USERAUTH_PK_OK, :string, keys.last.ssh_type, :string, Net::SSH::Buffer.from(:key, keys.last))

            t.expect do |t,packet|
              assert_equal USERAUTH_REQUEST, packet.type
              assert verify_userauth_request_packet(packet, keys.last, true)
              assert_equal "sig-two", packet.read_string
              t.return(USERAUTH_FAILURE, :string, "hostbased,password")
            end
          end
        end
      end

      assert !subject.authenticate("ssh-connection", "jamis")
    end

    def test_authenticate_should_return_true_if_any_key_can_authenticate
      key_manager.expects(:sign).with(&signature_parameters(keys.first)).returns("sig-one")

      transport.expect do |t, packet|
        assert_equal USERAUTH_REQUEST, packet.type
        assert verify_userauth_request_packet(packet, keys.first, false)
        t.return(USERAUTH_PK_OK, :string, keys.first.ssh_type, :string, Net::SSH::Buffer.from(:key, keys.first))

        t.expect do |t,packet|
          assert_equal USERAUTH_REQUEST, packet.type
          assert verify_userauth_request_packet(packet, keys.first, true)
          assert_equal "sig-one", packet.read_string
          t.return(USERAUTH_SUCCESS)
        end
      end

      assert subject.authenticate("ssh-connection", "jamis")
    end

    private

      def signature_parameters(key)
        Proc.new do |given_key, data|
          next false unless given_key.to_blob == key.to_blob
          buffer = Net::SSH::Buffer.new(data)
          buffer.read_string == "abcxyz123"      && # session-id
          buffer.read_byte   == USERAUTH_REQUEST && # type
          verify_userauth_request_packet(buffer, key, true)
        end
      end

      def verify_userauth_request_packet(packet, key, has_sig)
        packet.read_string == "jamis"          && # user-name
        packet.read_string == "ssh-connection" && # next service
        packet.read_string == "publickey"      && # auth-method
        packet.read_bool   == has_sig          && # whether a signature is appended
        packet.read_string == key.ssh_type     && # ssh key type
        packet.read_buffer.read_key.to_blob == key.to_blob # key
      end

      @@keys = nil
      def keys
        @@keys ||= [OpenSSL::PKey::RSA.new(32), OpenSSL::PKey::DSA.new(32)]
      end

      def key_manager(options={})
        @key_manager ||= stub("key_manager", :identities => options[:keys] || keys)
      end

      def subject(options={})
        options[:key_manager] = key_manager(options) unless options.key?(:key_manager)
        @subject ||= Net::SSH::Authentication::Methods::Publickey.new(session(options), options)
      end

  end

end; end