/*******************************************************************************
 * Copyright (c) 2001, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.connection.internal.ui.wizards;

import java.net.MalformedURLException;
import java.net.URL;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.wst.rdb.connection.internal.ui.util.resource.ResourceLoader;
import org.eclipse.wst.rdb.core.internal.ui.RDBCoreUIPlugin;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogProvider;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;


/**
 * Wizard allows user to select an existing connection to associate the resource
 * to or to create an new connection and append it to the server view
 * @author Quy V. On, quyon@us.ibm.com
 */
public class ConnectionSelectionWizard extends Wizard {
	
    public static final String SKIP = "skip";  
    public static final String CONNECTION_SELECTION_EXISTINGPAGE_NAME =
        "ConnectionSelectionExistingWizardPage";
    public static final String CONNECTION_SELECTION_JDBCPAGE_NAME = 
        "ConnectionSelectionJDBCWizardPage";
    public static final String CONNECTION_SELECTION_FILTER_NAME =
        "ConnectionSelectionFilterWizardPage";   
    
    private static final ResourceLoader resourceLoader = ResourceLoader.INSTANCE;
    protected ConnectionSelectionExistingWizardPage existingPage;
    protected ConnectionSelectionJDBCWizardPage jdbcPage;
    protected ConnectionSelectionFilterWizardPage filterPage;
    protected List myStartingConInfoNames;
    protected boolean actionCanceled;
    
    private ConnectionInfo connectionInfo;
    
    /**
     * Creates an instance of ConnectionSelectionWizard
     */
    public ConnectionSelectionWizard() {
    	super();
        setWindowTitle(resourceLoader.queryString("CUI_CONN_SELECT_EXT_PAGE_TITLE")); //$NON-NLS-1$

          URL coreUIPluginURL = RDBCoreUIPlugin.getInstallURL();
          String imagePath = "icons" + java.io.File.separator + "connection_wiz.gif"; //$NON-NLS-1$ $NON-NLS-2$
          try {
              ImageDescriptor connWizardImageDescriptor = ImageDescriptor.createFromURL(new URL(coreUIPluginURL, imagePath ));
              setDefaultPageImageDescriptor( connWizardImageDescriptor );
          }
          catch (MalformedURLException e) {
              // ignore
          }

        IDialogSettings workbenchSettings = IDEWorkbenchPlugin.getDefault().getDialogSettings();
        IDialogSettings section = workbenchSettings.getSection("connection.ui.connectionselection");
        if (section == null)
            section = workbenchSettings.addNewSection("connection.ui.connectionselection");
        setDialogSettings(section);
    }
    
    /**
     * Adds pages to wizard
     * Overrides addPages in Wizard
     */
    public void addPages() {
    	existingPage = new ConnectionSelectionExistingWizardPage(
                ConnectionSelectionWizard.CONNECTION_SELECTION_EXISTINGPAGE_NAME);
        addPage(existingPage);
        jdbcPage = new ConnectionSelectionJDBCWizardPage(
                ConnectionSelectionWizard.CONNECTION_SELECTION_JDBCPAGE_NAME);
        addPage(jdbcPage);
        filterPage = new ConnectionSelectionFilterWizardPage(
                ConnectionSelectionWizard.CONNECTION_SELECTION_FILTER_NAME);
        addPage(filterPage);
        
        // init the list of starting connection info names (so we don't remove an 
        // existing connection when canceling the wizard).
        myStartingConInfoNames = getStartingConInfoNames();
    }
    
    /**
     * Gets a list of connectionInfo names
     * @return a list of starting connection Info names
     */
    public List getStartingConInfoNames() {
        ConnectionInfo[] conInfos = RDBCorePlugin.getDefault().getConnectionManager().getAllNamedConnectionInfo();
        List list = new ArrayList();
        for (int i = 0; i < conInfos.length; i++) {
            list.add(conInfos[i].getName());
        }
        return list;
    }
    
    /**
     * Remove the newly created connection
     */
    public void removeConnection() {
        if (connectionInfo != null) {
            removeConnection(connectionInfo.getName());
        }
    }
    
    /**
     * Removes a connection from the server explorer
     * @param aConName the name of the connection
     */
    public void removeConnection(String aConName) {
        if (aConName != null & !myStartingConInfoNames.contains(aConName)) {
            RDBCorePlugin.getDefault().getConnectionManager().removeConnectionInfo(aConName);
        }
    }
    
    /**
     * Sets the connection Info object
     * @param info the connection info object
     */
    public void setConInfo(ConnectionInfo info) {
    	connectionInfo = info;
    }
    
    /**
     * Creates a new connection
     * @return true if connection successfully created, false if not
     */
    public boolean createConnection() {
    	if (connectionInfo != null) {
        	removeConnection(connectionInfo.getName());
            connectionInfo = null;
        }
        jdbcPage.internalSaveWidgetValues();
        jdbcPage.performTestConnection(false);
        boolean rtnCode = jdbcPage.isFinalConnection();
        if (rtnCode) {
        	connectionInfo = jdbcPage.getConnection();
            try {
            	addConnection(connectionInfo);
            }
            catch (Exception ex) {
            	return false;
            }
        }
        return rtnCode;        
    }
    
    /**
     * Reestablish a connection to a database
     * @param info the connectionInfo object to reestablish connection
     * @exception throws Exception if any error occurs during reconnect     
     */
    public void reconnect(ConnectionInfo info) throws Exception {
    	try {
    		addConnection(info);
        }
        catch (Exception ex) {
        	throw ex;
        }
       	    
    }
    
    /**
     * Adds a connection to the server view
     * @param info the ConnectionInfo object to be added
     * @return the connection info with properties added
     */
    public ConnectionInfo addConnection(ConnectionInfo info) throws Exception {
    	Connection connection = null;
        try {
        	connection = info.connect();
        	info.setSharedConnection(connection);
        	info.saveConnectionInfo();
        	setDatabase(connection, info, info.getDatabaseName());
        }
        catch (Exception e) {
        	removeConnection(info.getName());
        	connection = null; 
            throw e;            
        }
        return info;
    }
    
    /**
     * Sets the database of a connectionInfo object
     * @param connection the JDBC connection
     * @param info the connectionInfo to set the Database object with
     * @param databaseName the name of the database
     */
    public void setDatabase(Connection connection, ConnectionInfo info, String databaseName) {
        Database database = getCatalogDatabase(connection, info);
        database.setName(databaseName);
     }
    
    private Database getCatalogDatabase(Connection connection, ConnectionInfo info) {
    	ICatalogProvider catalogProvider = info.getDatabaseDefinition().getDatabaseCatalogProvider();
        Database database = catalogProvider.getCatalogDatabase(connection);
        RDBCorePlugin.getDefault().getConnectionManager().setConnectionInfo(database, info);
        setSharedInformation(info, database);
        return database;
    }
    
    private void setSharedInformation(ConnectionInfo info, Database database) {
    	if (info.getSharedDatabase() != null) {
    		info.removeSharedDatabase();
    	}
        info.setSharedDatabase(database);
    }
    
    /**
     * Gets the connection info object
     * @return the connection info
     */
    public ConnectionInfo getConInfo() {
    	return connectionInfo;
    }
    
    /**
     * Gets whether or not the user cancels this wizard
     * @return true if user clicks on cancel, false if user click on finish
     */
    public boolean isActionCanceled() {
    	return actionCanceled;
    }
    
    public boolean canFinish() {        
        if (existingPage.isExistingConnectionSelected()) {            
        	existingPage.refresh();
            return existingPage.isConnected();
        }
            
      	return super.canFinish();
    }
    
    /**
     * Called when user clicks on Finish button
     * @return true if allowed to continue, false if not
     */
    public boolean performFinish() {
    	actionCanceled = false;
        if (connectionInfo != null && filterPage != null && 
                filterPage.getControl().getData(ConnectionSelectionWizard.SKIP) == Boolean.FALSE) {
            filterPage.applyFilter();

            // The following is to get the filter to "take" on the schema folder.
            Database db = connectionInfo.getSharedDatabase();
            ((ICatalogObject) db).refresh();
        }
        else if (existingPage.isNewConnectionSelected() && 
                filterPage.getControl().getData(ConnectionSelectionWizard.SKIP) == Boolean.TRUE) {
        	createConnection();
        }
        return (connectionInfo == null) ? false : true;        
    }
    
    /**
     * @see org.eclipse.jface.wizard.IWizard#performCancel()
     */
    public boolean performCancel() {
        actionCanceled = true;
        if (connectionInfo != null) {
            removeConnection();
            connectionInfo = null;
        }
        return (super.performCancel());
    }
}