/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Vector;

import org.eclipse.core.runtime.IStatus;

/**
 *  The OutputViewAPI provides a set of methods for use of the DB Output view.
 *  The OutputViewAPI is implemented as a singleton, hence its methods must be
 *  accessed by first getting an instance via <b> OutputViewAPI.getInstance()</b>.
 *  The DB Output view can be used to display the action against an object and it's
 *  respective status, messages, result set in a table and it's input and output
 *  parameters if applicable.
 *  <p>
 *  The action against an object and it's status is saved in an action list (left
 *  table in the DB Output view) and the messages, paramaters and result sets can viewed
 *  in the right side display pages.  All output for an object is logged, hence when new
 *  action/object pairs are displayed in the DB Output view, they are added to the action
 *  list, thus allowing any previous ones to be selected for viewing.  The entries
 *  in the action list can be explicitly deleted by the user and programatically.
 *  <p>
 *  Messages for a given action/object pair can be appended to, replaced, cleared and
 *  the status can be updated.
 *  See specific methods for detailed usage.
 *  <p>
 *  <b>Usage:</b>
 *  <p>
 *  An entry in the action list with it's status, action and object name is represented
 *  by an OutputItem object. To use the Output View, you must first instantiate
 *  an OuputItem object (see <b>OutputItem</b> for details).
 *  The OutputItem is used as input to all methods against the Output View to provide
 *  the action list item for which a messages is to be displayed, replaced or cleared,
 *  for status changes, and to display parameters and results when applicable for
 *  actions against database objects.  When an OutputItem is passed as input, it is added
 *  as a new entry to the action list if it doesn't already exist.   If it already exists,
 *  the information is added to the already existing OutputItem.
 *
 * <p>
 * <b>Usage example:</b>
 * <pre>
 * import com.ibm.etools.outputview.OutputViewAPI;
 * import com.ibm.etools.outputview.OutputItem;
 *
 * protected OutputItem sp1;
 * protected ResultSet resultSet;
 * protected Vector paramaters;
 *
 * sp1 = new OutputItem(OutputItem.STATUS_IN_PROGRESS, OutputItem.ACTION_BUILD, "SProc1");
 * OutputViewAPI.getInstance().showMessage(sp1,"Sproc1 build message");
 *
 * OutputViewAPI.getInstance().showMessage(sp1,"Another message for Sproc1 build. It gets appended to the first");
 *
 * OutputViewAPI.getInstance().updateStatus(sp1,OutputItem.STATUS_SUCCESS);  // the status changes to success
 *
 * OutputViewAPI.getInstance().replaceMessage(sp1,"The new msg that replaces the old");
 *
 * resultSet =  .... a JDBC ResultSet..
 * paramaters =  ... a Vector of String[4] elements containing name, input value, output value, binaryTypeFlag
 *
 * OutputViewAPI.getInstance().showResults(sp1,resultSet,paramaters);
 *
 * OutputViewAPI.getInstance().deleteOutputItem(sp1);     // now its gone
 *</pre>
 *
 *  @see       OutputItem 
 */
public class OutputViewAPI
{
   /** Code for the messages tab. */
   public static final int MESSAGE_PAGE =	30;
   /** Code for the parameters tab. */
   public static final int PARAMATERS_PAGE =	31;
   /** Code for the results tab. */
   public static final int RESULTS_PAGE =	32;

   /** Reference to the view. */
   protected static OutputView outputView;

   /** Singleton that handles all inputs. */
   protected static OutputViewAPI myself;

   /** Constructs an API. Protected. Use getInstance(). */
   protected OutputViewAPI()
   {
      outputView = null;
   }

   /**
    *  Gets the singleton instance of OutputViewAPI.
    *  @return   The instance of OutputViewAPI.
    */
   public synchronized static OutputViewAPI getInstance()
   {
      if (myself == null)
      {
         myself = new OutputViewAPI();
      }
      return myself;
   }

   /**
    *     Displays the msg for the given OutputItem in the Messages Page
    *     of the Output View. If prior messages exist for this OutputItem,
    *     then the new msg is appended to the previous ones.  The OutputItem
    *     is added to the action list unless it already exists.
    *     <p>
    *     @param outputItem  the OutputItem for which to show the msg
    *     @param msg the String containing the message to be displayed
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showMessage(OutputItem outputItem, String msg, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.showMessage(outputItem, msg, async);
   }

   /**
    *     Replaces the msg for the given OutputItem in the Messages Page
    *     of the Output View. If the OutputItem is not found in the action
    *     list of the Output View, nothing occurs. To show a new message or append
    *     addditional messages, use showMessage(...).  This is a synchronous update.
    *     <p>
    *     @param outputItem  the OutputItem for which to replace its message
    *     @param msg the String containing the message to replace the previous one
    */
   public void replaceMessage(OutputItem outputItem, String msg)
   {
      replaceMessage (outputItem, msg, false);
   }

   /**
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void replaceMessage(OutputItem outputItem, String msg, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.replaceMessage(outputItem,msg, async);
   }

   /**
    *     Deletes the OutputItem from the action list and its corresponding
    *     content in the Output View pane area. If the OutputItem does not exist,
    *     nothing occurs. This is a synchronous update.
    *     <p>
    *     @param outputItem  the OutputItem to delete
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void deleteOutputItem(OutputItem outputItem, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         outputView.getModel().removeOutputItem(outputItem);
         outputView.refreshWithDefault();

      }
   }

   public void deleteOutputItem(OutputItem outputItem) {
      deleteOutputItem (outputItem, false);
   }

   /**
    *     Displays the parameters for the given OutputItem in the Parameters
    *       tab of the OutputView.  The OutputItem is added to the action list
    *       unless it already exists.  A Message should be associated with each
    *       OutputItem. This is a synchronous update.
    *     <p>
    *     @param parms a Vector of String[4] (an array of  4strings, first one
    *                  for parameter name, the second for the input parameter
    *                  value, the third string in the array for the output
    *                  parameter value and the fourth one a flag indicating if
    *                  the parameter is a binary type.  If the parameter is INOUT,
    *                  then both the input and output entries are filled in).
    */
   public void showParameters(OutputItem outputItem, Vector parms) {
      showParameters(outputItem, parms, false);
   }

   /**
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showParameters(OutputItem outputItem, Vector parms, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.showParameters(outputItem,parms); //this is handled asyn already.
   }

    /**
    *     Displays the result set in the Output View and the parameters passed in
    *     for the designated OutputItem. The Results page of the Output View is displayed.
    *     If the OutputItem does not yet exist in the output view list,
    *     the OutputItem is first added to the list. This is an asynchronous update, i.e.
    *     it is delayed to the event dispath thread via the Display syncExec method.
    *     <p>
    *     @param outputItem   The OutputItem for which to show the results
    *     @param rs           A ResultSet object to display in the output view
    *     @param parameters   A Vector of String[4] elements, where each string array
    *                         represents one parameter.
    *                         The array of 4 strings consist of: first entry for
    *                         the parameter name, the second for the input parameter
    *                         value, the third string in the array for the output
    *                         parameter value and the fourth a flag indicating if the
    *                         parameter is a binary type .  If the parameter is INOUT, then both
    *                         the input and output entries are filled in.
    *
    */
   public void showOutputContent(OutputItem outputItem, ResultSet rs,
            Vector parameters, int showWhichTab, boolean async) throws SQLException{
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         outputView.getModel().addResultAndParameters(outputItem, rs, parameters);
         switch (showWhichTab) {
         case MESSAGE_PAGE    : outputView.showMessage(outputItem, async);
         case PARAMATERS_PAGE : outputView.showParameters(outputItem, async);
         case RESULTS_PAGE    : outputView.showResult(outputItem, async);
         default          : outputView.showMessage(outputItem, async);

         }	

      }
   }

   /**
    *     Displays the result sets contained in the ProcedureCallCache, and any messages associated
    *     with the given OutputItem.  The Results page of the Output View is displayed.
    *     If the OutputItem does not yet exist in the output view list,
    *     the OutputItem is first added to the list. This is an asynchronous update, i.e.
    *     it is delayed to the event dispath thread via the Display syncExec method.
    *     <p>
    *     @param outputItem  the OutputItem for which to show the results for the
    * 						 stored procedure.
    *     @param proc a ProcedureCallCache object to display.  By default the result's
    *                   page with the results is displayed.
    */
   public void showOutputContent(OutputItem outputItem, ProcedureCallCache proc) {
      showOutputContent(outputItem, proc, false);
   }

   /**
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showOutputContent(OutputItem outputItem, ProcedureCallCache proc, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         try{

            outputView.getModel().addProcedureCall(outputItem, proc);
            outputView.showResult(outputItem); //showResult is already handled asynchronizelly.
         }
         catch (SQLException ex){
            OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                    "OutputViewAPI:showOutputContent(OutputItem, ProcedureCallCache, boolean)",
                    ex);   //$NON-NLS-1$         
         }
      }
   }

   /**
    *     Displays the result sets contained in the ProcedureCallCache, the parameters passed in
    *     and any messages associated with the given OutputItem.
    *     The Results page of the Output View is displayed.
    *     If the OutputItem does not yet exist in the output view list,
    *     the OutputItem is first added to the list. This is an asynchronous update, i.e.
    *     it is delayed to the event dispath thread via the Display syncExec method.
    *     <p>
    *     @param outputItem   The OutputItem for which to show the results and parameters
    *                         for the stored procedure.
    *     @param proc         a ProcedureCallCache object to display.  By default the result's
    *                         page with the results is displayed.
    *     @param parameters   A Vector of String[4] elements, where each string array
    *                         represents one parameter.
    *                         The array of 4 strings consist of: first entry for
    *                         the parameter name, the second for the input parameter
    *                         value, the third string in the array for the output
    *                         parameter value and the fourth a flag indicating if the
    *                         parameter is a binary type .  If the parameter is INOUT, then both
    *                         the input and output entries are filled in.
    */
   public void showOutputContent(OutputItem outputItem, ProcedureCallCache proc, Vector parameters) {
      showOutputContent(outputItem, proc, parameters, false);
   }

   /**
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showOutputContent(OutputItem outputItem, ProcedureCallCache proc, Vector parameters, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         try {

            outputView.getModel().addProcedureCallAndParameters(outputItem, proc, parameters);
            outputView.showResult(outputItem);
         }
         catch (SQLException ex){
            OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                    "OutputViewAPI:showOutputContent(OutputItem, ProcedureCallCache, Vector, boolean)",
                    ex); //$NON-NLS-1$        
         }
      }
   }

   /**
    *     Displays the result set in the Output View for the designated OutputItem.
    *     If the OutputItem does not yet exist in the output view list,
    *     the OutputItem is first added to the list. This is an asynchronous update, i.e.
    *     it is delayed to the event dispath thread via the Display syncExec method.
    *     <p>
    *     @param outputItem  the OutputItem for which to show the results
    *     @param rs          a ResultSet object to display in the output view
    */
   public void showOutputContent(OutputItem outputItem, ResultSet rs) throws SQLException {
      showOutputContent(outputItem, rs, false);   	
   }

   /**
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showOutputContent(OutputItem outputItem, ResultSet rs, boolean async) throws SQLException
   {
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         outputView.getModel().addResult(outputItem, rs);
         outputView.showResult(outputItem);
      }
   }

   /**
    *     Displays the result set in the Output View and the parameters passed in
    *     for the designated OutputItem. The Results page of the Output View is displayed.
    *     If the OutputItem does not yet exist in the output view list,
    *     the OutputItem is first added to the list. This is a synchronous update, i.e.
    *     it is not delayed to the event dispath thread via the Display syncExec method.
    *     <p>
    *     @param outputItem   The OutputItem for which to show the results
    *     @param rs           A ResultSet object to display in the output view
    *     @param parameters   A Vector of String[4] elements, where each string array
    *                         represents one parameter.
    *                         The array of 4 strings consist of: first entry for
    *                         the parameter name, the second for the input parameter
    *                         value, the third string in the array for the output
    *                         parameter value and the fourth a flag indicating if the
    *                         parameter is a binary type .  If the parameter is INOUT, then both
    *                         the input and output entries are filled in.
    */
   public void showOutputContent(OutputItem outputItem, ResultSet rs, Vector parameters) throws SQLException{
      showOutputContent(outputItem, rs, parameters, false);
   }

   /**
    * Displays the result set in the Output View and the parameters passed in
    *     for the designated OutputItem. The Results page of the Output View is displayed.
    *     If the OutputItem does not yet exist in the output view list,
    *     the OutputItem is first added to the list. This is an asynchronous update, i.e.
    *     it is delayed to the event dispath thread via the Display syncExec method.
    * @param outputItem   The OutputItem for which to show the results
    * @param rs           A ResultSet object to display in the output view
    * @param parameters   A Vector of String[4] elements, where each string array
    *                     represents one parameter.
    * @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   public void showOutputContent(OutputItem outputItem, ResultSet rs, Vector parameters, boolean async) throws SQLException
   {
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         outputView.getModel().addResultAndParameters(outputItem, rs, parameters);
         outputView.showResult(outputItem);
      }
   }

   /**
    * Displays the specified contributed content set in the Output View for the designated OutputItem.
    *     If the OutputItem does not yet exist in the output view list,
    *     the OutputItem is first added to the list. This is an asynchronous update, i.e.
    *     it is delayed to the event dispath thread via the Display syncExec method.
    * @param outputItem   The OutputItem for which to show the content
    * @param id The id attribute of the additionalTab element for the contribution to org.eclipse.wst.rdb.outputview.additionalOutputViewTabs, used to identify the proper contribution
    * @param contents The content to set for the contributed tab; the caller must ensure that it can be properly processed for the given id
    * @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   public void showOutputContent(OutputItem outputItem, String id, Object contents, boolean async) {
	   outputView =  getOutputView(async);
	   if (outputView != null)
	   {
		   // The OutputView class holds contributed model classes, via a hashtable keyed off of the id.
		   // These classes must implement  the IAdditionalOutputViewTab interface.
		   // Basically, this allows the existing model to be embellished to hold
		   // arbitrary contents for an arbitrary tab
		   outputView.getModel().addContents(outputItem, id, contents);

		   // now that we've set the contents in the mode, show it in the output view
		   outputView.showContents (outputItem, id, async);                 
	   }
   } 
      
   /**
    *     Changes the status for the designated OutputItem in the Output View to the new
    *     status.  The Message page remains showing.  This is an asynchronous update, i.e.
    *     it is delayed to the event dispath thread via the Display syncExec method.
    *     <p>
    *     @param outputItem  The OutputItem for which the status is changing.
    *     @param statusCode  The new OutputItem Status code (such as OutputItem.STATUS_SUCCESS ).
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   public void updateStatus(OutputItem outputItem, int statusCode, boolean async) //this one should be async
   {
      outputItem.setStatus(statusCode);
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.refreshActionTable(outputItem, async);
   }

   /**
    * Changes the status for the designated OutputItem synchronously.
    * @param outputItem  The OutputItem for which the status is changing.
    * @param statusCode  The new OutputItem Status code (such as OutputItem.STATUS_SUCCESS ).
    */
   public void updateStatus(OutputItem outputItem, int statusCode) {
      updateStatus(outputItem, statusCode, false);
   }
   /**
    *     Given a uniqueName and an actionCode, locates the OutputItem
    *       in the action list if it exists.  NULL is returned if an OutputItem
    *       was not found.
    *     <p>
    *     @param uniqueName  The String making up the unique name.
    *     @param action      The int constant (defined in OutputItem)
    *                        representing the action such as
    *                        OutputItem.ACTION_BUILD.
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    *     returns OutputItem
    */
   public OutputItem findOutputItem(String uniqueName, int action, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         return outputView.getModel().findOutputItem(uniqueName,action);
      else
         return null;
   }

   /**
    * Locates the output item for a unique name and action code synchronously.
    * @param uniqueName  The String making up the unique name.
    * @param action      The int constant (defined in OutputItem)
    *                        representing the action such as
    *                        OutputItem.ACTION_BUILD.
    */
   public OutputItem findOutputItem(String uniqueName, int action)
   {
      return findOutputItem(uniqueName, action, false);
   }

   /**
    *     Given a uniqueName and an actionCode, locates the OutputItem
    *       in the action list if it exists.  NULL is returned if an OutputItem
    *       was not found.
    *     <p>
    *     @param uniqueName  The String making up the unique name.
    *     @param action      The display string for the action.
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    *     returns OutputItem
    */
   public OutputItem findOutputItem(String uniqueName, String action, boolean async)
   {
      int actionCode = OutputItem.getActionCode(action);
      if (actionCode < 0)
         return null;
      else
         return findOutputItem(uniqueName, actionCode, async);
   }

   /**
    * Locates the output item for a unique name and action code synchronously.
    * @param uniqueName  The String making up the unique name.
    * @param action      The display string for the action.
    */
   public OutputItem findOutputItem(String uniqueName, String action)
   {
      return findOutputItem(uniqueName, action, false);
   }

   /**
    *     Return the row count of a Result Set previously passed in.
    *      If a Result Set does not exists, -1 is returned.
    *     <p>
    *     @param OutputItem  the OutputItem
    *     returns the row count as an integer of an existing Result Set,
    *     -1 if a Result Set does not exist or an Output View does not exist.
    */
   public int getRowCount(OutputItem item, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         return outputView.getModel().getResultSetRowCount(item);
      }
      else
         return -1;
   }

   /**
    * Gets the row count synchronously.
    * @param OutputItem  the OutputItem
    */
   public int getRowCount(OutputItem item)
   {
      return getRowCount(item, false);
   }

   /**
    *     Resets the content associated with the passed OuputItem.
    *       Used when an OutputItem is re-used for a repeat action on the same
    *       object.
    *     <p>
    *     @param OutputItem  The OutputItem to be re-set.
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    *     returns the re-set OutputItem
    */
   public OutputItem resetOutputItem(OutputItem item, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
      {
         return outputView.getModel().resetItem(item);
      }
      else
         return null;
   }

   /**
    * Reset the content associated with the passed OuputItem synchronously.
    * @param OutputItem  the OutputItem
    */
   public OutputItem resetOutputItem(OutputItem item)
   {
      return resetOutputItem(item, false);
   }

   /**
    *     Clears (deletes) the messages, parameters, and results for the designated
    *     OutputItem in the pane area of the Output View. If the OutputItem does not exist,
    *     nothing occurs.  This is a synchronous update.
    *     <p>
    *     @param outputItem  The OutputItem for which to clear the messages,results, and parameters.
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   public void clearOutput(OutputItem outputItem, boolean async)
   {
      clearMessages(outputItem, async);
      clearResults(outputItem, async);
      clearParameters(outputItem, async);
   }

   /**
    * Clears (deletes) the messages, parameters, and results synchronously.
    * @param outputItem  The OutputItem for which to clear the messages,results, and parameters,
    */
   public void clearOutput(OutputItem outputItem) {
      clearOutput(outputItem, false);
   }
   /**
    *     Clears (deletes) the results for the designated OutputItem in the
    *     Resutls page of the Output View. If the OutputItem does not exist,
    *     nothing occurs. This is a synchronous update.
    *     <p>
    *     @param outputItem  The OutputItem for which to clear the results
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   public void clearResults(OutputItem outputItem, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.clearResults();
   }

   /**
    * Clears (deletes) the results synchronously.
    * @param outputItem  The OutputItem for which to clear the messages,results, and parameters,
    */
   public void clearResults(OutputItem outputItem) {
      clearResults(outputItem, false);
   }

   /**
    *     Clears (deletes) the parameters for the designated OutputItem in the
    *     Parameters page of the Output View. If the OutputItem does not exist,
    *     nothing occurs. This is a synchronous update.
    *     <p>
    *     @param outputItem  The OutputItem for which to clear the parameters
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   public void clearParameters(OutputItem outputItem, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.clearParameters();
   }

   /**
    * Clears parameters synchronously.
    * @param outputItem  The OutputItem for which to clear the messages,results, and parameters,
    */
   public void clearParameters(OutputItem outputItem) {
      clearParameters(outputItem, false);
   }

   /**
    *     Clears (deletes) the messages for the designated OutputItem in the
    *     Messages page of the Output View. If the OutputItem does not exist,
    *     nothing occurs. This is a synchronous update.
    *     <p>
    *     @param outputItem  the OutputItem for which to clear the messages
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void clearMessages(OutputItem outputItem, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.clearMessages(outputItem);
   }

   public void clearMessages(OutputItem outputItem)
   {
      clearMessages (outputItem, false);
   }

   /**
    *     Adds the OutputItem to the Output View without a message.
    *     The outputItem is expected to be passed in again to display
    *     messages and results or parameters if appropriate.
    *     If this call is not followed with this OutputItem for a message
    *     display and a result or parameter display, the message area
    *     will remain blank.
    *     <p>
    *     @param outputItem  the OutputItem to add to the Output View
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void addOutputItem(OutputItem outputItem, boolean async)
   {
      outputView =  getOutputView(async); //add async
      if (outputView != null)
      {
         outputView.addOutputItem(outputItem, async, false);
      }
   }

   /**
    * Shows the page with the results of the currently selected OutputItem in the action list
    * synchronously.
    */
   public void showResultsPage()
   {
      showResultsPage(false);
   }

   /**
    * Shows the page with the results of the currently selected OutputItem in the action list.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showResultsPage(boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.showResultsPage(async);
   }

   /**
    * Shows the page with the results of the currently selected OutputItem in the action list
    * synchronously.
    */
   public void showMessagePage() {
      showMessagePage(false);
   }

   /**
    * Shows the page with the results of the currently selected OutputItem in the action list.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showMessagePage(boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.showMessagePage(async);
   }

   /**
    * Shows the page with the results of the currently selected OutputItem in the aciton list
    * synchronously.
    */
   public void showParametersPage() {
      showParametersPage(false);
   }

   /**
    * Shows the page with the results of the currently selected OutputItem in the aciton list.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void showParametersPage(boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.showParametersPage(async);
   }

   /**
    *     Removes one entry from the output view.
    *     @param obj The OutputItem in the action list.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void removeOutputItem(OutputItem item, boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.getModel().removeOutputItem(item);
   }
   /**
    * Removes one entry from the output view synchronously.
    * @param obj The OutputItem in the action list.
    */
   public void removeOutputItem(OutputItem item) {
      removeOutputItem(item, false);
   }

   /**
    *     Removes all the entries in the output view.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   public void removeAllFromOutput(boolean async)
   {
      outputView =  getOutputView(async);
      if (outputView != null)
         outputView.getModel().removeAll();
   }

   /**
    * Removes all the entries in the output view synchronously.
    */
   public void removeAllFromOutput() {
      removeAllFromOutput(false);
   }

   /**
    * Gets the view synchronously.
    * @return The OutputView.
    */
   protected OutputView getOutputView()
   {
      return getOutputView(false);
   }

   /**
    * Gets the view.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    * @return The OutputView.
    */
   protected OutputView getOutputView(boolean async)
   {
      return OutputUtil.getOutputView(async);
   }
}
