-module(edoc_parser).
-export([parse/1, parse_and_scan/1, format_error/1]).
-file("edoc_parser.yrl", 181).

%% ========================== -*-Erlang-*- =============================
%% EDoc function specification parser, generated from the file
%% "edoc_parser.yrl" by the Yecc parser generator.
%%
%% Copyright (C) 2002-2005 Richard Carlsson
%%
%% This library is free software; you can redistribute it and/or modify
%% it under the terms of the GNU Lesser General Public License as
%% published by the Free Software Foundation; either version 2 of the
%% License, or (at your option) any later version.
%%
%% This library is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
%% Lesser General Public License for more details.
%%
%% You should have received a copy of the GNU Lesser General Public
%% License along with this library; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
%% USA
%% ====================================================================

-export([parse_spec/2, parse_typedef/2, parse_throws/2, parse_ref/2,
	 parse_see/2]).

-include("edoc_types.hrl").

%% Multiple entry point hack:

start_spec(Ts, L) -> run_parser(Ts, L, start_spec).

start_typedef(Ts, L) -> run_parser(Ts, L, start_typedef).

start_throws(Ts, L) -> run_parser(Ts, L, start_throws).

start_ref(Ts, L) -> run_parser(Ts, L, start_ref).

%% Error reporting fix

run_parser(Ts, L, Start) ->
    case parse([{Start,L} | Ts]) of
	{error, {999999,?MODULE,_}} ->
	    What = case Start of
		       start_spec -> "specification";
		       start_typedef -> "type definition";
		       start_throws -> "exception declaration";
		       start_ref -> "reference"
		   end,
	    {error, {L,?MODULE,["unexpected end of ", What]}};
	Other -> Other
    end.

%% Utility functions:

tok_val(T) -> element(3, T).

qname([A]) ->
    A;    % avoid unnecessary call to packages:concat/1.
qname(List) ->
    list_to_atom(packages:concat(lists:reverse(List))).

union(Ts) ->
    case Ts of
	[T] -> T;
	_ -> #t_union{types = lists:reverse(Ts)}
    end.

ann(T, A) -> ?set_t_ann(T, A).

%% ---------------------------------------------------------------------

%% @doc EDoc type specification parsing. Parses the content of
%% <a href="overview-summary.html#ftag-spec">`@spec'</a> declarations.

parse_spec(S, L) ->
    case edoc_scanner:string(S, L) of
	{ok, Ts, _} ->
	    case start_spec(Ts, L) of
		{ok, Spec} ->
		    Spec;
		{error, E} ->
		    throw_error(E, L)
	    end;
	{error, E, _} ->
	    throw_error(E, L)
    end.

%% ---------------------------------------------------------------------

%% @doc EDoc type definition parsing. Parses the content of
%% <a href="overview-summary.html#gtag-type">`@type'</a> declarations.

parse_typedef(S, L) ->
    {S1, S2} = edoc_lib:split_at_stop(S),
    N = edoc_lib:count($\n, S1),
    L1 = L + N,
    Text = edoc_lib:strip_space(S2),
    {parse_typedef_1(S1, L), edoc_wiki:parse_xml(Text, L1)}.

parse_typedef_1(S, L) ->
    case edoc_scanner:string(S, L) of
	{ok, Ts, _} ->
	    case start_typedef(Ts, L) of
		{ok, T} ->
		    T;
		{error, E} ->
		    throw_error({parse_typedef, E}, L)
	    end;
	{error, E, _} ->
	    throw_error({parse_typedef, E}, L)
    end.

%% ---------------------------------------------------------------------

%% @doc Parses a <a
%% href="overview-summary.html#References">reference</a> to a module,
%% package, function, type, or application

parse_ref(S, L) ->
    case edoc_scanner:string(S, L) of
	{ok, Ts, _} ->
	    case start_ref(Ts, L) of
		{ok, T} ->
		    T;
		{error, E} ->
		    throw_error({parse_ref, E}, L)
	    end;
	{error, E, _} ->
	    throw_error({parse_ref, E}, L)
    end.

%% ---------------------------------------------------------------------

%% @doc Parses the content of
%% <a href="overview-summary.html#ftag-see">`@see'</a> references.
parse_see(S, L) ->
    {S1, S2} = edoc_lib:split_at_stop(S),
    N = edoc_lib:count($\n, S1),
    L1 = L + N,
    Text = edoc_lib:strip_space(S2),
    {parse_ref(S1, L), edoc_wiki:parse_xml(Text, L1)}.

%% ---------------------------------------------------------------------

%% @doc EDoc exception specification parsing. Parses the content of
%% <a href="overview-summary.html#ftag-throws">`@throws'</a> declarations.

parse_throws(S, L) ->
    case edoc_scanner:string(S, L) of
	{ok, Ts, _} ->
	    case start_throws(Ts, L) of
		{ok, Spec} ->
		    Spec;
		{error, E} ->
		    throw_error({parse_throws, E}, L)
	    end;
	{error, E, _} ->
	    throw_error({parse_throws, E}, L)
    end.

%% ---------------------------------------------------------------------

throw_error({L, M, D}, _L0) ->
    throw({error,L,{format_error,M,D}});
throw_error({parse_spec, E}, L) ->
    throw_error({"specification", E}, L);
throw_error({parse_typedef, E}, L) ->
    throw_error({"type definition", E}, L);
throw_error({parse_ref, E}, L) ->
    throw_error({"reference", E}, L);
throw_error({parse_throws, E}, L) ->
    throw_error({"throws-declaration", E}, L);
throw_error({Where, E}, L) when is_list(Where) ->
    throw({error,L,{"unknown error parsing ~s: ~P.",[Where,E,15]}});
throw_error(E, L) ->
    %% Just in case.
    throw({error,L,{"unknown parse error: ~P.",[E,15]}}).

-file("/usr/obj/ports/erlang-11b.3/otp_src_R11B-3/bootstrap/lib/parsetools/include/yeccpre.hrl", 0).
%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id $
%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The parser generator will insert appropriate declarations before this line.%

parse(Tokens) ->
    yeccpars0(Tokens, false).

parse_and_scan({F, A}) -> % Fun or {M, F}
    yeccpars0([], {F, A});
parse_and_scan({M, F, A}) ->
    yeccpars0([], {{M, F}, A}).

format_error(Message) ->
    case io_lib:deep_char_list(Message) of
	true ->
	    Message;
	_ ->
	    io_lib:write(Message)
    end.

% To be used in grammar files to throw an error message to the parser
% toplevel. Doesn't have to be exported!
-compile({nowarn_unused_function,{return_error,2}}).
return_error(Line, Message) ->
    throw({error, {Line, ?MODULE, Message}}).

yeccpars0(Tokens, MFA) ->
    try yeccpars1(Tokens, MFA, 0, [], [])
    catch 
        throw: {error, {_Line, ?MODULE, _M}} = Error -> 
                   Error % probably from return_error/1
    end.

% Don't change yeccpars1/6 too much, it is called recursively by yeccpars2/8!
yeccpars1([Token | Tokens], Tokenizer, State, States, Vstack) ->
    yeccpars2(State, element(1, Token), States, Vstack, Token, Tokens,
	      Tokenizer);
yeccpars1([], {F, A}, State, States, Vstack) ->
    case apply(F, A) of
        {ok, Tokens, _Endline} ->
	    yeccpars1(Tokens, {F, A}, State, States, Vstack);
        {eof, _Endline} ->
            yeccpars1([], false, State, States, Vstack);
        {error, Descriptor, _Endline} ->
            {error, Descriptor}
    end;
yeccpars1([], false, State, States, Vstack) ->
    yeccpars2(State, '$end', States, Vstack, {'$end', 999999}, [], false).

% For internal use only.
yeccerror(Token) ->
    {error,
     {element(2, Token), ?MODULE,
      ["syntax error before: ", yecctoken2string(Token)]}}.

yecctoken2string({atom, _, A}) -> io_lib:write(A);
yecctoken2string({integer,_,N}) -> io_lib:write(N);
yecctoken2string({float,_,F}) -> io_lib:write(F);
yecctoken2string({char,_,C}) -> io_lib:write_char(C);
yecctoken2string({var,_,V}) -> io_lib:format('~s', [V]);
yecctoken2string({string,_,S}) -> io_lib:write_string(S);
yecctoken2string({reserved_symbol, _, A}) -> io_lib:format('~w', [A]);
yecctoken2string({_Cat, _, Val}) -> io_lib:format('~w', [Val]);
yecctoken2string({'dot', _}) -> io_lib:format('~w', ['.']);
yecctoken2string({'$end', _}) ->
    [];
yecctoken2string({Other, _}) when is_atom(Other) ->
    io_lib:format('~w', [Other]);
yecctoken2string(Other) ->
    io_lib:write(Other).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



-file("./edoc_parser.erl", 276).

yeccpars2(0, start_ref, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 2, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, start_spec, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 3, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, start_throws, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 4, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, start_typedef, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 5, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(1, '$end', _, __Stack, _, _, _) ->
 {ok, hd(__Stack)};
yeccpars2(1, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(2, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 105, [2 | __Ss], [__T | __Stack]);
yeccpars2(2, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 106, [2 | __Ss], [__T | __Stack]);
yeccpars2(2, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(3, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [3 | __Ss], [__T | __Stack]);
yeccpars2(3, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 95, [3 | __Ss], [__T | __Stack]);
yeccpars2(3, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(4, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 42, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(5, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [5 | __Ss], [__T | __Stack]);
yeccpars2(5, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(6, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_6_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(start, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(7, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 9, [7 | __Ss], [__T | __Stack]);
yeccpars2(7, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(8, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 17, [8 | __Ss], [__T | __Stack]);
yeccpars2(8, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_8_(__Stack),
 yeccpars2(16, __Cat, [8 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(9, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 11, [9 | __Ss], [__T | __Stack]);
yeccpars2(9, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 12, [9 | __Ss], [__T | __Stack]);
yeccpars2(9, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(10, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [10 | __Ss], [__T | __Stack]);
yeccpars2(10, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 14, [10 | __Ss], [__T | __Stack]);
yeccpars2(10, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(11, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_11_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(var_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(12, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_12_(__Stack),
 yeccpars2(yeccgoto(vars, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(13, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_13_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(var_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(14, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 15, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(15, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_15_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(vars, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(16, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 79, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 80, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_16_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(typedef, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(17, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [17 | __Ss], [__T | __Stack]);
yeccpars2(17, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(18, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_18_(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(19, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 86, [19 | __Ss], [__T | __Stack]);
yeccpars2(19, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(20, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_20_(__Stack),
 yeccpars2(77, __Cat, [20 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(21, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [21 | __Ss], [__T | __Stack]);
yeccpars2(21, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [21 | __Ss], [__T | __Stack]);
yeccpars2(21, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(22, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_22_(__Stack),
 yeccpars2(yeccgoto(utype, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(23, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_23_(__Stack),
 yeccpars2(yeccgoto(ptypes, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(24, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(25, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(26, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(27, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [27 | __Ss], [__T | __Stack]);
yeccpars2(27, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(28, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(29, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [29 | __Ss], [__T | __Stack]);
yeccpars2(29, '$end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_29_$end'(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), '$end', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_29_)'(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), ')', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_29_+'(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), '+', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_29_,'(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), ',', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_29_]'(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), ']', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_29_atom(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), atom, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_29_var(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), var, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_29_|'(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), '|', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_29_}'(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), '}', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_29_(__Stack),
 yeccpars2(yeccgoto(qname, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(30, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_30_(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(31, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_31_(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(32, '::', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 40, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_32_(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(33, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 36, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(34, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 37, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 38, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(35, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_35_(__Stack),
 yeccpars2(yeccgoto(utypes, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(36, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_36_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(utype_tuple, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(37, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(38, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_38_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utype_tuple, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(39, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_39_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utypes, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(40, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 42, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [40 | __Ss], [__T | __Stack]);
yeccpars2(40, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(41, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [41 | __Ss], [__T | __Stack]);
yeccpars2(41, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [41 | __Ss], [__T | __Stack]);
yeccpars2(41, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_41_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(42, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_42_(__Stack),
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(43, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 42, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(44, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 42, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(45, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_45_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(ptypes, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(46, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_46_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(ptypes, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(47, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_47_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(48, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(49, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_49_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(50, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_50_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(51, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [51 | __Ss], [__T | __Stack]);
yeccpars2(51, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(52, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(53, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [53 | __Ss], [__T | __Stack]);
yeccpars2(53, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [53 | __Ss], [__T | __Stack]);
yeccpars2(53, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(54, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_54_(__Stack),
 yeccpars2(yeccgoto(qname, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(55, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(56, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [56 | __Ss], [__T | __Stack]);
yeccpars2(56, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(57, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(58, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_58_(__Stack),
 __Nss = lists:nthtail(6, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(59, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_59_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(qname, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(60, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [60 | __Ss], [__T | __Stack]);
yeccpars2(60, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 37, [60 | __Ss], [__T | __Stack]);
yeccpars2(60, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(61, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_61_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(utype_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(62, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_62_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utype_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(63, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(64, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 68, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(65, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 72, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(66, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_66_(__Stack),
 yeccpars2(yeccgoto(fields, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(67, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 69, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(68, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_68_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(69, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(70, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_70_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(field, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(71, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [71 | __Ss], [__T | __Stack]);
yeccpars2(71, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(72, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_72_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(73, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_73_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fields, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(74, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 75, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(75, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(76, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_76_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(77, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 79, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 80, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_77_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(typedef, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(78, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_78_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(defs, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(79, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 9, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(80, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 81, [80 | __Ss], [__T | __Stack]);
yeccpars2(80, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(81, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [81 | __Ss], [__T | __Stack]);
yeccpars2(81, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(82, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_82_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(def, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(83, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 84, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(84, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(85, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_85_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(def, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(86, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '//', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(87, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_87_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fun_type, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(88, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_88_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(start, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(89, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_89_(__Stack),
 yeccpars2(yeccgoto(etype, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(90, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_90_(__Stack),
 yeccpars2(91, __Cat, [90 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(91, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 79, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 80, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_91_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(throws, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(92, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_92_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(start, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(93, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [93 | __Ss], [__T | __Stack]);
yeccpars2(93, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(94, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_94_(__Stack),
 yeccpars2(96, __Cat, [94 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(95, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_95_(__Stack),
 yeccpars2(yeccgoto(function_name, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(96, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 79, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 80, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_96_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(spec, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(97, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_97_(__Stack),
 yeccpars2(98, __Cat, [97 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(98, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 79, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 80, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_98_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(spec, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(99, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_99_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(start, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(100, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 115, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 116, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_100_(__Stack),
 yeccpars2(yeccgoto(mref, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(101, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(ref, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(102, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(ref, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(103, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(ref, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(104, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(ref, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(105, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 111, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(106, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 107, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 108, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_106_(__Stack),
 yeccpars2(yeccgoto(qname, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(107, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 110, [107 | __Ss], [__T | __Stack]);
yeccpars2(107, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(108, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 109, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(109, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_109_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(lref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(110, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_110_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(lref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(111, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 112, [111 | __Ss], [__T | __Stack]);
yeccpars2(111, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_111_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(aref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(112, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(113, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_113_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(aref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(114, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_114_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(aref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(115, '*', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 122, [115 | __Ss], [__T | __Stack]);
yeccpars2(115, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [115 | __Ss], [__T | __Stack]);
yeccpars2(115, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(116, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 117, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(117, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 118, [117 | __Ss], [__T | __Stack]);
yeccpars2(117, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 119, [117 | __Ss], [__T | __Stack]);
yeccpars2(117, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(118, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 121, [118 | __Ss], [__T | __Stack]);
yeccpars2(118, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(119, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 120, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(120, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_120_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(mref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(121, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_121_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(mref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(122, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_122_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(pref, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(__Other, _, _, _, _, _, _) ->
 erlang:error({yecc_bug,"1.1",{missing_state_in_action_table, __Other}}).

yeccgoto(aref, 2) ->
 104;
yeccgoto(def, 16) ->
 78;
yeccgoto(def, 77) ->
 78;
yeccgoto(def, 91) ->
 78;
yeccgoto(def, 96) ->
 78;
yeccgoto(def, 98) ->
 78;
yeccgoto(defs, 8) ->
 16;
yeccgoto(defs, 20) ->
 77;
yeccgoto(defs, 90) ->
 91;
yeccgoto(defs, 94) ->
 96;
yeccgoto(defs, 97) ->
 98;
yeccgoto(etype, 4) ->
 90;
yeccgoto(field, 64) ->
 66;
yeccgoto(field, 71) ->
 73;
yeccgoto(fields, 64) ->
 65;
yeccgoto(fun_type, 3) ->
 94;
yeccgoto(fun_type, 4) ->
 24;
yeccgoto(fun_type, 17) ->
 24;
yeccgoto(fun_type, 26) ->
 24;
yeccgoto(fun_type, 28) ->
 24;
yeccgoto(fun_type, 33) ->
 24;
yeccgoto(fun_type, 37) ->
 24;
yeccgoto(fun_type, 40) ->
 24;
yeccgoto(fun_type, 43) ->
 24;
yeccgoto(fun_type, 44) ->
 24;
yeccgoto(fun_type, 69) ->
 24;
yeccgoto(fun_type, 81) ->
 24;
yeccgoto(fun_type, 84) ->
 24;
yeccgoto(fun_type, 86) ->
 24;
yeccgoto(fun_type, 93) ->
 97;
yeccgoto(function_name, 3) ->
 93;
yeccgoto(lref, 2) ->
 103;
yeccgoto(mref, 2) ->
 102;
yeccgoto(mref, 112) ->
 114;
yeccgoto(pref, 2) ->
 101;
yeccgoto(pref, 112) ->
 113;
yeccgoto(ptype, 4) ->
 23;
yeccgoto(ptype, 17) ->
 23;
yeccgoto(ptype, 26) ->
 23;
yeccgoto(ptype, 28) ->
 23;
yeccgoto(ptype, 33) ->
 23;
yeccgoto(ptype, 37) ->
 23;
yeccgoto(ptype, 40) ->
 23;
yeccgoto(ptype, 43) ->
 46;
yeccgoto(ptype, 44) ->
 45;
yeccgoto(ptype, 69) ->
 23;
yeccgoto(ptype, 81) ->
 23;
yeccgoto(ptype, 84) ->
 23;
yeccgoto(ptype, 86) ->
 23;
yeccgoto(ptypes, 4) ->
 89;
yeccgoto(ptypes, 17) ->
 22;
yeccgoto(ptypes, 26) ->
 22;
yeccgoto(ptypes, 28) ->
 22;
yeccgoto(ptypes, 33) ->
 22;
yeccgoto(ptypes, 37) ->
 22;
yeccgoto(ptypes, 40) ->
 41;
yeccgoto(ptypes, 69) ->
 22;
yeccgoto(ptypes, 81) ->
 22;
yeccgoto(ptypes, 84) ->
 22;
yeccgoto(ptypes, 86) ->
 22;
yeccgoto(qname, 2) ->
 100;
yeccgoto(qname, 4) ->
 21;
yeccgoto(qname, 17) ->
 21;
yeccgoto(qname, 26) ->
 21;
yeccgoto(qname, 28) ->
 21;
yeccgoto(qname, 33) ->
 21;
yeccgoto(qname, 37) ->
 21;
yeccgoto(qname, 40) ->
 21;
yeccgoto(qname, 43) ->
 21;
yeccgoto(qname, 44) ->
 21;
yeccgoto(qname, 52) ->
 53;
yeccgoto(qname, 69) ->
 21;
yeccgoto(qname, 81) ->
 21;
yeccgoto(qname, 84) ->
 21;
yeccgoto(qname, 86) ->
 21;
yeccgoto(qname, 112) ->
 100;
yeccgoto(ref, 2) ->
 99;
yeccgoto(spec, 3) ->
 92;
yeccgoto(start, 0) ->
 1;
yeccgoto(throws, 4) ->
 88;
yeccgoto(typedef, 5) ->
 6;
yeccgoto(utype, 17) ->
 20;
yeccgoto(utype, 26) ->
 35;
yeccgoto(utype, 28) ->
 48;
yeccgoto(utype, 33) ->
 35;
yeccgoto(utype, 37) ->
 39;
yeccgoto(utype, 69) ->
 70;
yeccgoto(utype, 81) ->
 82;
yeccgoto(utype, 84) ->
 85;
yeccgoto(utype, 86) ->
 87;
yeccgoto(utype_list, 3) ->
 19;
yeccgoto(utype_list, 4) ->
 19;
yeccgoto(utype_list, 17) ->
 19;
yeccgoto(utype_list, 26) ->
 19;
yeccgoto(utype_list, 28) ->
 19;
yeccgoto(utype_list, 29) ->
 47;
yeccgoto(utype_list, 33) ->
 19;
yeccgoto(utype_list, 37) ->
 19;
yeccgoto(utype_list, 40) ->
 19;
yeccgoto(utype_list, 43) ->
 19;
yeccgoto(utype_list, 44) ->
 19;
yeccgoto(utype_list, 57) ->
 58;
yeccgoto(utype_list, 69) ->
 19;
yeccgoto(utype_list, 75) ->
 76;
yeccgoto(utype_list, 81) ->
 19;
yeccgoto(utype_list, 84) ->
 19;
yeccgoto(utype_list, 86) ->
 19;
yeccgoto(utype_list, 93) ->
 19;
yeccgoto(utype_tuple, 4) ->
 18;
yeccgoto(utype_tuple, 17) ->
 18;
yeccgoto(utype_tuple, 26) ->
 18;
yeccgoto(utype_tuple, 28) ->
 18;
yeccgoto(utype_tuple, 33) ->
 18;
yeccgoto(utype_tuple, 37) ->
 18;
yeccgoto(utype_tuple, 40) ->
 18;
yeccgoto(utype_tuple, 43) ->
 18;
yeccgoto(utype_tuple, 44) ->
 18;
yeccgoto(utype_tuple, 69) ->
 18;
yeccgoto(utype_tuple, 81) ->
 18;
yeccgoto(utype_tuple, 84) ->
 18;
yeccgoto(utype_tuple, 86) ->
 18;
yeccgoto(utypes, 26) ->
 60;
yeccgoto(utypes, 33) ->
 34;
yeccgoto(var_list, 7) ->
 8;
yeccgoto(var_list, 79) ->
 83;
yeccgoto(vars, 9) ->
 10;
yeccgoto(__Symbol, __State) ->
 erlang:error({yecc_bug,"1.1",{__Symbol, __State, missing_in_goto_table}}).

-compile({inline,{yeccpars2_6_,1}}).
-file("edoc_parser.yrl", 43).
yeccpars2_6_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_8_,1}}).
-file("edoc_parser.yrl", 113).
yeccpars2_8_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_11_,1}}).
-file("edoc_parser.yrl", 124).
yeccpars2_11_([__2,__1 | __Stack]) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_12_,1}}).
-file("edoc_parser.yrl", 128).
yeccpars2_12_([__1 | __Stack]) ->
 [begin
   [ # t_var { name = tok_val ( __1 ) } ]
  end | __Stack].

-compile({inline,{yeccpars2_13_,1}}).
-file("edoc_parser.yrl", 125).
yeccpars2_13_([__3,__2,__1 | __Stack]) ->
 [begin
   lists : reverse ( __2 )
  end | __Stack].

-compile({inline,{yeccpars2_15_,1}}).
-file("edoc_parser.yrl", 129).
yeccpars2_15_([__3,__2,__1 | __Stack]) ->
 [begin
   [ # t_var { name = tok_val ( __3 ) } | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_16_,1}}).
-file("edoc_parser.yrl", 132).
yeccpars2_16_([__3,__2,__1 | __Stack]) ->
 [begin
   # t_typedef { name = # t_name { name = tok_val ( __1 ) } ,
    args = __2 ,
    defs = lists : reverse ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_18_,1}}).
-file("edoc_parser.yrl", 84).
yeccpars2_18_([__1 | __Stack]) ->
 [begin
   # t_tuple { types = __1 }
  end | __Stack].

-compile({inline,{yeccpars2_20_,1}}).
-file("edoc_parser.yrl", 113).
yeccpars2_20_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_22_,1}}).
-file("edoc_parser.yrl", 72).
yeccpars2_22_([__1 | __Stack]) ->
 [begin
   union ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_23_,1}}).
-file("edoc_parser.yrl", 75).
yeccpars2_23_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{'yeccpars2_29_$end',1}}).
-file("edoc_parser.yrl", 80).
'yeccpars2_29_$end'([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{'yeccpars2_29_)',1}}).
-file("edoc_parser.yrl", 80).
'yeccpars2_29_)'([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{'yeccpars2_29_+',1}}).
-file("edoc_parser.yrl", 80).
'yeccpars2_29_+'([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{'yeccpars2_29_,',1}}).
-file("edoc_parser.yrl", 80).
'yeccpars2_29_,'([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{'yeccpars2_29_]',1}}).
-file("edoc_parser.yrl", 80).
'yeccpars2_29_]'([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_29_atom,1}}).
-file("edoc_parser.yrl", 80).
yeccpars2_29_atom([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_29_var,1}}).
-file("edoc_parser.yrl", 80).
yeccpars2_29_var([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{'yeccpars2_29_|',1}}).
-file("edoc_parser.yrl", 80).
'yeccpars2_29_|'([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{'yeccpars2_29_}',1}}).
-file("edoc_parser.yrl", 80).
'yeccpars2_29_}'([__1 | __Stack]) ->
 [begin
   # t_atom { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_29_,1}}).
-file("edoc_parser.yrl", 47).
yeccpars2_29_([__1 | __Stack]) ->
 [begin
   [ tok_val ( __1 ) ]
  end | __Stack].

-compile({inline,{yeccpars2_30_,1}}).
-file("edoc_parser.yrl", 82).
yeccpars2_30_([__1 | __Stack]) ->
 [begin
   # t_float { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_31_,1}}).
-file("edoc_parser.yrl", 81).
yeccpars2_31_([__1 | __Stack]) ->
 [begin
   # t_integer { val = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_32_,1}}).
-file("edoc_parser.yrl", 79).
yeccpars2_32_([__1 | __Stack]) ->
 [begin
   # t_var { name = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_35_,1}}).
-file("edoc_parser.yrl", 68).
yeccpars2_35_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_36_,1}}).
-file("edoc_parser.yrl", 64).
yeccpars2_36_([__2,__1 | __Stack]) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_38_,1}}).
-file("edoc_parser.yrl", 65).
yeccpars2_38_([__3,__2,__1 | __Stack]) ->
 [begin
   lists : reverse ( __2 )
  end | __Stack].

-compile({inline,{yeccpars2_39_,1}}).
-file("edoc_parser.yrl", 69).
yeccpars2_39_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_41_,1}}).
-file("edoc_parser.yrl", 71).
yeccpars2_41_([__3,__2,__1 | __Stack]) ->
 [begin
   ann ( union ( __3 ) , tok_val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_42_,1}}).
-file("edoc_parser.yrl", 79).
yeccpars2_42_([__1 | __Stack]) ->
 [begin
   # t_var { name = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_45_,1}}).
-file("edoc_parser.yrl", 77).
yeccpars2_45_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_46_,1}}).
-file("edoc_parser.yrl", 76).
yeccpars2_46_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_47_,1}}).
-file("edoc_parser.yrl", 93).
yeccpars2_47_([__2,__1 | __Stack]) ->
 [begin
   # t_type { name = # t_name { name = tok_val ( __1 ) } ,
    args = __2 }
  end | __Stack].

-compile({inline,{yeccpars2_49_,1}}).
-file("edoc_parser.yrl", 85).
yeccpars2_49_([__2,__1 | __Stack]) ->
 [begin
   # t_nil { }
  end | __Stack].

-compile({inline,{yeccpars2_50_,1}}).
-file("edoc_parser.yrl", 86).
yeccpars2_50_([__3,__2,__1 | __Stack]) ->
 [begin
   # t_list { type = __2 }
  end | __Stack].

-compile({inline,{yeccpars2_54_,1}}).
-file("edoc_parser.yrl", 47).
yeccpars2_54_([__1 | __Stack]) ->
 [begin
   [ tok_val ( __1 ) ]
  end | __Stack].

-compile({inline,{yeccpars2_58_,1}}).
-file("edoc_parser.yrl", 100).
yeccpars2_58_([__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   # t_type { name = # t_name { app = tok_val ( __2 ) ,
    module = qname ( __4 ) ,
    name = tok_val ( __6 ) } ,
    args = __7 }
  end | __Stack].

-compile({inline,{yeccpars2_59_,1}}).
-file("edoc_parser.yrl", 48).
yeccpars2_59_([__3,__2,__1 | __Stack]) ->
 [begin
   [ tok_val ( __3 ) | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_61_,1}}).
-file("edoc_parser.yrl", 61).
yeccpars2_61_([__2,__1 | __Stack]) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_62_,1}}).
-file("edoc_parser.yrl", 62).
yeccpars2_62_([__3,__2,__1 | __Stack]) ->
 [begin
   lists : reverse ( __2 )
  end | __Stack].

-compile({inline,{yeccpars2_66_,1}}).
-file("edoc_parser.yrl", 106).
yeccpars2_66_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_68_,1}}).
-file("edoc_parser.yrl", 88).
yeccpars2_68_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   # t_record { name = # t_atom { val = tok_val ( __2 ) } }
  end | __Stack].

-compile({inline,{yeccpars2_70_,1}}).
-file("edoc_parser.yrl", 110).
yeccpars2_70_([__3,__2,__1 | __Stack]) ->
 [begin
   # t_field { name = # t_atom { val = tok_val ( __1 ) } , type = __3 }
  end | __Stack].

-compile({inline,{yeccpars2_72_,1}}).
-file("edoc_parser.yrl", 90).
yeccpars2_72_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   # t_record { name = # t_atom { val = tok_val ( __2 ) } ,
    fields = lists : reverse ( __4 ) }
  end | __Stack].

-compile({inline,{yeccpars2_73_,1}}).
-file("edoc_parser.yrl", 107).
yeccpars2_73_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_76_,1}}).
-file("edoc_parser.yrl", 96).
yeccpars2_76_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   # t_type { name = # t_name { module = qname ( __1 ) ,
    name = tok_val ( __3 ) } ,
    args = __4 }
  end | __Stack].

-compile({inline,{yeccpars2_77_,1}}).
-file("edoc_parser.yrl", 136).
yeccpars2_77_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   # t_typedef { name = # t_name { name = tok_val ( __1 ) } ,
    args = __2 ,
    type = __4 ,
    defs = lists : reverse ( __5 ) }
  end | __Stack].

-compile({inline,{yeccpars2_78_,1}}).
-file("edoc_parser.yrl", 114).
yeccpars2_78_([__2,__1 | __Stack]) ->
 [begin
   [ __2 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_82_,1}}).
-file("edoc_parser.yrl", 117).
yeccpars2_82_([__3,__2,__1 | __Stack]) ->
 [begin
   # t_def { name = # t_var { name = tok_val ( __1 ) } ,
    type = __3 }
  end | __Stack].

-compile({inline,{yeccpars2_85_,1}}).
-file("edoc_parser.yrl", 120).
yeccpars2_85_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   # t_def { name = # t_type { name = # t_name { name = tok_val ( __1 ) } ,
    args = __2 } ,
    type = __4 }
  end | __Stack].

-compile({inline,{yeccpars2_87_,1}}).
-file("edoc_parser.yrl", 58).
yeccpars2_87_([__3,__2,__1 | __Stack]) ->
 [begin
   # t_fun { args = __1 , range = __3 }
  end | __Stack].

-compile({inline,{yeccpars2_88_,1}}).
-file("edoc_parser.yrl", 42).
yeccpars2_88_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_89_,1}}).
-file("edoc_parser.yrl", 172).
yeccpars2_89_([__1 | __Stack]) ->
 [begin
   union ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_90_,1}}).
-file("edoc_parser.yrl", 113).
yeccpars2_90_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_91_,1}}).
-file("edoc_parser.yrl", 175).
yeccpars2_91_([__2,__1 | __Stack]) ->
 [begin
   # t_throws { type = __1 ,
    defs = lists : reverse ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_92_,1}}).
-file("edoc_parser.yrl", 41).
yeccpars2_92_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_94_,1}}).
-file("edoc_parser.yrl", 113).
yeccpars2_94_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_95_,1}}).
-file("edoc_parser.yrl", 55).
yeccpars2_95_([__1 | __Stack]) ->
 [begin
   # t_name { name = tok_val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_96_,1}}).
-file("edoc_parser.yrl", 51).
yeccpars2_96_([__2,__1 | __Stack]) ->
 [begin
   # t_spec { type = __1 , defs = lists : reverse ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_97_,1}}).
-file("edoc_parser.yrl", 113).
yeccpars2_97_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_98_,1}}).
-file("edoc_parser.yrl", 53).
yeccpars2_98_([__3,__2,__1 | __Stack]) ->
 [begin
   # t_spec { name = __1 , type = __2 , defs = lists : reverse ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_99_,1}}).
-file("edoc_parser.yrl", 44).
yeccpars2_99_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_100_,1}}).
-file("edoc_parser.yrl", 160).
yeccpars2_100_([__1 | __Stack]) ->
 [begin
   edoc_refs : module ( qname ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_106_,1}}).
-file("edoc_parser.yrl", 47).
yeccpars2_106_([__1 | __Stack]) ->
 [begin
   [ tok_val ( __1 ) ]
  end | __Stack].

-compile({inline,{yeccpars2_109_,1}}).
-file("edoc_parser.yrl", 166).
yeccpars2_109_([__3,__2,__1 | __Stack]) ->
 [begin
   edoc_refs : function ( tok_val ( __1 ) , tok_val ( __3 ) )
  end | __Stack].

-compile({inline,{yeccpars2_110_,1}}).
-file("edoc_parser.yrl", 168).
yeccpars2_110_([__3,__2,__1 | __Stack]) ->
 [begin
   edoc_refs : type ( tok_val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_111_,1}}).
-file("edoc_parser.yrl", 149).
yeccpars2_111_([__2,__1 | __Stack]) ->
 [begin
   edoc_refs : app ( tok_val ( __2 ) )
  end | __Stack].

-compile({inline,{yeccpars2_113_,1}}).
-file("edoc_parser.yrl", 153).
yeccpars2_113_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   edoc_refs : app ( tok_val ( __2 ) , __4 )
  end | __Stack].

-compile({inline,{yeccpars2_114_,1}}).
-file("edoc_parser.yrl", 151).
yeccpars2_114_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   edoc_refs : app ( tok_val ( __2 ) , __4 )
  end | __Stack].

-compile({inline,{yeccpars2_120_,1}}).
-file("edoc_parser.yrl", 156).
yeccpars2_120_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   edoc_refs : function ( qname ( __1 ) , tok_val ( __3 ) , tok_val ( __5 ) )
  end | __Stack].

-compile({inline,{yeccpars2_121_,1}}).
-file("edoc_parser.yrl", 158).
yeccpars2_121_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   edoc_refs : type ( qname ( __1 ) , tok_val ( __3 ) )
  end | __Stack].

-compile({inline,{yeccpars2_122_,1}}).
-file("edoc_parser.yrl", 163).
yeccpars2_122_([__3,__2,__1 | __Stack]) ->
 [begin
   edoc_refs : package ( qname ( __1 ) )
  end | __Stack].


-file("edoc_parser.yrl", 360).
