### Name: connections
### Title: Functions to Manipulate Connections
### Aliases: connections connection file pipe fifo gzfile unz bzfile url
###   socketConnection open open.connection isOpen isIncomplete close
###   close.connection flush flush.connection print.connection
###   summary.connection
### Keywords: file connection

### ** Examples

zz <- file("ex.data", "w")  # open an output file connection
cat("TITLE extra line", "2 3 5 7", "", "11 13 17", file = zz, sep = "\n")
cat("One more line\n", file = zz)
close(zz)
readLines("ex.data")
unlink("ex.data")

zz <- gzfile("ex.gz", "w")  # compressed file
cat("TITLE extra line", "2 3 5 7", "", "11 13 17", file = zz, sep = "\n")
close(zz)
readLines(gzfile("ex.gz"))
unlink("ex.gz")

zz <- bzfile("ex.bz2", "w")  # bzip2-ed file
cat("TITLE extra line", "2 3 5 7", "", "11 13 17", file = zz, sep = "\n")
close(zz)
print(readLines(bzfile("ex.bz2")))
unlink("ex.bz2")

## An example of a file open for reading and writing
Tfile <- file("test1", "w+")
c(isOpen(Tfile, "r"), isOpen(Tfile, "w")) # both TRUE
cat("abc\ndef\n", file=Tfile)
readLines(Tfile)
seek(Tfile, 0, rw="r") # reset to beginning
readLines(Tfile)
cat("ghi\n", file=Tfile)
readLines(Tfile)
close(Tfile)
unlink("test1")

## We can do the same thing with an anonymous file.
Tfile <- file()
cat("abc\ndef\n", file=Tfile)
readLines(Tfile)
close(Tfile)

if(capabilities("fifo")) {
  zz <- fifo("foo", "w+")
  writeLines("abc", zz)
  print(readLines(zz))
  close(zz)
  unlink("foo")
}

## Not run: ## Unix examples of use of pipes
##D 
##D # read listing of current directory
##D readLines(pipe("ls -1"))
##D 
##D # remove trailing commas. Suppose
##D % cat data2
##D 450, 390, 467, 654,  30, 542, 334, 432, 421,
##D 357, 497, 493, 550, 549, 467, 575, 578, 342,
##D 446, 547, 534, 495, 979, 479
##D # Then read this by
##D scan(pipe("sed -e s/,$// data2"), sep=",")
##D 
##D # convert decimal point to comma in output
##D # both R strings and (probably) the shell need \ doubled
##D zz <- pipe(paste("sed s/\\\\./,/ >", "outfile"), "w")
##D cat(format(round(rnorm(100), 4)), sep = "\n", file = zz)
##D close(zz)
##D file.show("outfile", delete.file=TRUE)
## End(Not run)

## Not run: ## example for Unix machine running a finger daemon
##D 
##D con <- socketConnection(port = 79, blocking = TRUE)
##D writeLines(paste(system("whoami", intern=TRUE), "\r", sep=""), con)
##D gsub(" *$", "", readLines(con))
##D close(con)
## End(Not run)

## Not run: ## two R processes communicating via non-blocking sockets
##D # R process 1
##D con1 <- socketConnection(port = 6011, server=TRUE)
##D writeLines(LETTERS, con1)
##D close(con1)
##D 
##D # R process 2
##D con2 <- socketConnection(Sys.info()["nodename"], port = 6011)
##D # as non-blocking, may need to loop for input
##D readLines(con2)
##D while(isIncomplete(con2)) {Sys.sleep(1); readLines(con2)}
##D close(con2)
## End(Not run)

## Not run: 
##D  ## examples of use of encodings
##D cat(x, file = file("foo", "w", encoding="UTF-8"))
##D # read a 'Windows Unicode' file including names
##D A <- read.table(file("students", encoding="UCS-2LE"))
## End(Not run)


