"""
Movie module (imdb package).

This module provides the Movie class, used to store information about
a given movie.

Copyright 2004 Davide Alberani <davide.alberani@erlug.linux.it>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""

import types
from copy import deepcopy
from utils import analyze_title, build_title
from Person import Person


class Movie:
    """A Movie.
    
    A Movie object emulates (most of) the dictionary interface.

    Every information about a movie can be accessed as:
        movieObject['information']
    to get a list of the kind of information stored in a
    Movie object, use the keys() method; some useful aliases
    are defined (as "plot summary" for the "plot" key); see the
    keys_alias dictionary.
    """

    # Every information set available for a Movie object.
    all_info = ('main', # Information from the "combined details" page
                        # of a movie: title, director, cast, companies,
                        # akas, genres, votes, rating, runtime, etc. etc.
                'plot', # the plot.
                'awards',   # awards won and nominations.
                'taglines', # promotional taglines.
                'keywords', # plot keywords.
                'alternate versions',
                'crazy credits',
                'goofs',
                'quotes',
                'release dates',
                'vote details', # number of votes for rating [1-10],
                                # demographic breakdowns and top 250 rank.
                'official sites',
                'trivia')
    # The default sets of information retrieved.
    default_info = ('main', 'plot')

    # Aliases for some not-so-intuitive keys.
    keys_alias = {
                'user rating':  'rating',
                'plot summary': 'plot',
                'plot summaries': 'plot',
                'directed by':  'director',
                'writting credits': 'writer',
                'produced by':  'producer',
                'original music by':    'composer',
                'original music':    'composer',
                'non-original music by':    'composer',
                'non-original music':    'composer',
                'music':    'composer',
                'cinematography by':    'cinematographer',
                'cinematography':   'cinematographer',
                'film editing by':  'editor',
                'film editing': 'editor',
                'editing':  'editor',
                'actors':   'cast',
                'actresses':    'cast',
                'casting by':   'casting director',
                'casting':  'casting director',
                'art direction by': 'art director',
                'art direction': 'art director',
                'set decoration by':    'set decorator',
                'set decoration':   'set decorator',
                'costume design by':    'costume designer',
                'costume design':    'costume designer',
                'makeup department':    'make up',
                'makeup':    'make up',
                'make-up':    'make up',
                'production management':    'production manager',
                'second unit director or assistant director':
                                                'assistant director',
                'second unit director':   'assistant director',
                'sound department': 'sound crew',
                'special effects by':   'special effects',
                'visual effects by':    'visual effects',
                'stunts':   'stunt performer',
                'other crew':   'crewmembers',
                'misc crew':   'crewmembers',
                'other companies':  'miscellaneous companies',
                'aka':  'akas',
                'also known as':    'akas',
                'country':  'countries',
                'runtime':  'runtimes',
                'lang': 'languages',
                'language': 'languages',
                'certificate':  'certificates',
                'certifications':   'certificates',
                'certification':    'certificates'}

    def __init__(self, movieID=None, title='', myTitle='',
                    myID=None, movieData={}, currentRole='',
                    accessSystem=None):
        """Initialize a Movie object.
        
        *movieID* -- the unique identifier for the movie.
        *title* -- the title of the Movie, if not in the movieData dictionary.
        *myTitle* -- your personal title for the movie.
        *myID* -- your personal identifier for the movie.
        *movieData* -- a dictionary used to initialize the object.
        *currentRole* -- a string representing the current role or duty
                        of the person in this movie.
        *accessSystem* -- a string representing the data access system used.
        """
        self.reset()
        self.accessSystem = accessSystem
        self.set_data(movieData, override=1)
        if title and not movieData.has_key('title'):
            self.set_title(title)
        self.movieID = movieID
        self.myTitle = myTitle
        self.myID = myID
        self.currentRole = currentRole

    def get_current_info(self):
        """Return the current set of information retrieved."""
        return self.current_info

    def set_current_info(self, ci):
        """Set the current set of information retrieved."""
        self.current_info = ci

    def add_to_current_info(self, val):
        """Add a set of information to the current list."""
        if val not in self.current_info:
            self.current_info.append(val)

    def has_current_info(self, val):
        """Return true if the given set of information is in the list."""
        return val in self.current_info

    def reset(self):
        """Reset the Movie object."""
        self.__movie_data = {}
        self.current_info = []
        self.movieID = None
        self.myTitle = ''
        self.myID = None
        self.currentRole = ''

    def set_title(self, title):
        """Set the title of the movie."""
        d_title = analyze_title(title)
        self.__movie_data.update(d_title)

    def set_data(self, md, override=0):
        """Set the movie data to the given dictionary; if 'override' is
        set, the previous data is removed, otherwise the two dictionary
        are merged.
        """
        # XXX: uh.  Not sure this the best place/way to do it.
        md = deepcopy(md)
        if not override:
            self.__movie_data.update(md)
        else:
            self.__movie_data = md
    
    def clear(self):
        """Reset the dictionary."""
        self.__movie_data.clear()

    def has_key(self, key):
        """Return true if a given section is defined."""
        try:
            self.__getitem__(key)
        except KeyError:
            return 0
        return 1

    def keys(self):
        """Return a list of valid keys."""
        l = self.__movie_data.keys()
        if 'title' in l:
            l += ['long imdb title']
        return l

    def items(self):
        """Return the items in the dictionary."""
        # XXX: what about 'long imdb title'?
        return self.__movie_data.items()

    def values(self):
        """Return the values in the dictionary."""
        # XXX: what about 'long imdb title'?
        return self.__movie_data.values()

    def append_item(self, key, item):
        """The item is appended to the list identified by
        the given key.
        """
        # TODO: this and the two other methods below are here only
        #       for _future_ usage, when it will make sense to modify
        #       a Movie object; right now they're incomplete and should
        #       not be used.
        if not self.__movie_data.has_key(key):
            self.__movie_data[key] = []
        self.__movie_data[key].append(item)

    def set_item(self, key, item):
        """Directly store the item with the given key."""
        self.__movie_data[key] = item

    def __setitem__(self, key, item):
        """Directly store the item with the given key."""
        self.__movie_data[key] = item
    
    # TODO: should methods like __delitem__() be defined?

    def get(self, key, default=None):
        """Return the given section, or default if it's not found."""
        try:
            return self.__getitem__(key)
        except KeyError:
            return default

    def __getitem__(self, key):
        """Return the value for a given key, checking key aliases;
        a KeyError exception is raised if the key is not found.
        """
        if key == 'long imdb title' and self.__movie_data.has_key('title'):
            return build_title(self.__movie_data)
        if key in self.keys_alias.keys():
            key = self.keys_alias[key]
        return self.__movie_data[key]

    def __nonzero__(self):
        """The Movie is "false" if the self.__movie_data is empty."""
        # XXX: check the title and the movieID?
        if self.__movie_data:
            return 1
        return 0

    def __cmp__(self, other):
        """Compare two Movie objects."""
        # XXX: check the title and the movieID?
        if not isinstance(other, self.__class__):
            return -1
        if self.__movie_data == other.__movie_data:
            return 0
        return 1

    def isSameTitle(self, other):
        """Return true if this and the compared object have the same
        long imdb title and/or movieID.
        """
        if not isinstance(other, self.__class__):
            return 0
        if self.__movie_data.has_key('title') and \
                build_title(self.__movie_data) == \
                build_title(other.__movie_data):
            return 1
        if self.accessSystem == other.accessSystem and self.movieID \
                and self.movieID == other.movieID:
            return 1
        return 0

    def __contains__(self, item):
        """Return true if the given Person object is listed in this Movie."""
        if not isinstance(item, Person):
            return 0
        for i in self.__movie_data.values():
            if type(i) in (types.ListType, types.TupleType):
                for j in i:
                    if isinstance(j, Person) and item.isSamePerson(j):
                        return 1
        return 0

    def copy(self):
        """Return a deep copy of the object itself."""
        return deepcopy(self)

    def __str__(self):
        """Simply print the short title."""
        return self.get('title', '')

    def summary(self):
        """Return a string with a pretty-printed summary for the movie."""
        if not self:
            return ''
        s = 'Movie\n=====\n'
        title = self.get('title')
        if title:
            s += 'Title: %s\n' % title
        genres = self.get('genres')
        if genres:
            s += 'Genres: '
            for gen in genres:
                s += gen + ', '
            s = s[:-2] + '.\n'
        director = self.get('director')
        if director:
            s += 'Director: '
            for name in director:
                s += str(name)
                if name.currentRole:
                    s += ' (%s)' % name.currentRole
                s += ', '
            s = s[:-2] + '.\n'
        writer = self.get('writer')
        if writer:
            s += 'Writer: '
            for name in writer:
                s += str(name)
                if name.currentRole:
                    s += ' (%s)' % name.currentRole
                s += ', '
            s = s[:-2] + '.\n'
        cast = self.get('cast')
        if cast:
            cast = cast[:5]
            s += 'Cast: '
            for name in cast:
                s += str(name)
                if name.currentRole:
                    s += ' (%s)' % name.currentRole
                s += ', '
            s = s[:-2] + '.\n'
        runtime = self.get('runtimes')
        if runtime:
            s += 'Runtime: '
            for r in runtime:
                s += r + ', '
            s = s[:-2] + '.\n'
        countries = self.get('countries')
        if countries:
            s += 'Country: '
            for c in countries:
                s += c + ', '
            s = s[:-2] + '.\n'
        lang = self.get('languages')
        if lang:
            s += 'Language: '
            for l in lang:
                s += l + ', '
            s = s[:-2] + '.\n'
        rating = self.get('rating')
        if rating:
            s += 'Rating: %s\n' % rating
        nr_votes = self.get('votes')
        if nr_votes:
            s += 'Votes: %s\n' % nr_votes
        plot = self.get('plot')
        if plot:
            plot = plot[0]
            i = plot.find('::')
            if i != -1:
                plot = plot[i+2:]
            s += 'Plot: %s' % plot
        return s


