%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% Copyright (C) 1990, Ellemtel Telecommunications Systems Laboratories
%% File    : io.erl
%% Author  : Robert Virding
%% Purpose : Standard i/o server interface functions.
%% Revision: $Id: io.erl,v 4.3.1.5 1996/04/24 18:02:53 rv Exp $

-module(io).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-export([put_chars/1,put_chars/2,nl/0,nl/1,
	 get_chars/2,get_chars/3,get_line/1,get_line/2]).
-export([write/1,write/2,read/1,read/2]).
-export([fwrite/1,fwrite/2,fwrite/3,fread/2,fread/3,
	 format/1,format/2,format/3]).
-export([scan_erl_exprs/1,scan_erl_exprs/2,scan_erl_exprs/3,
	 scan_erl_form/1,scan_erl_form/2,scan_erl_form/3,
	 parse_erl_exprs/1,parse_erl_exprs/2,parse_erl_exprs/3,
	 parse_erl_form/1,parse_erl_form/2,parse_erl_form/3]).
-export([request/1,request/2,requests/1,requests/2]).

%% The following exports are here for backwards compatibility.
-export([parse_exprs/2]).
-export([scan_erl_seq/1,scan_erl_seq/2,scan_erl_seq/3,
	 parse_erl_seq/1,parse_erl_seq/2, parse_erl_seq/3]).
	 
%%  These calls are here for backwards compatibility (BC sucks!).

scan_erl_seq(P)          -> scan_erl_exprs(P).
scan_erl_seq(I, P)       -> scan_erl_exprs(I, P).
scan_erl_seq(I, P, Pos)  -> scan_erl_exprs(I, P, Pos).

parse_erl_seq(P)          -> parse_erl_exprs(P).
parse_erl_seq(I, P)       -> parse_erl_exprs(I, P).
parse_erl_seq(I, P, Pos)  -> parse_erl_exprs(I, P, Pos).

parse_exprs(I, P) -> parse_erl_exprs(I, P).

%%
%% User interface.
%%

%% Writing and reading characters.

put_chars(Chars) ->
    put_chars(default_output(), Chars).

put_chars(Io, Chars) ->
    request(Io, {put_chars,Chars}).

nl() ->
    nl(default_output()).

nl(Io) ->
    request(Io, {put_chars,io_lib:nl()}).

get_chars(Prompt, N) ->
    get_chars(default_input(), Prompt, N).

get_chars(Io, Prompt, N) ->
    request(Io, {get_chars,Prompt,N}).

get_line(Prompt) ->
    get_line(default_input(), Prompt).

get_line(Io, Prompt) ->
    request(Io, {get_line,Prompt}).

%% Writing and reading Erlang terms.

write(Term) ->
    write(default_output(), Term).

write(Io, Term) ->
    request(Io, {write,Term}).

read(Prompt) ->
    read(default_input(), Prompt).

read(Io, Prompt) ->
    case request(Io, {get_until,Prompt,erl_scan,tokens,[1]}) of
	{ok,Toks,EndLine} ->
	    erl_parse:parse_term(Toks);
	{error,E,EndLine} ->
	    {error,E};
	{eof,EndLine} ->
	    eof;
	Other ->
	    Other
    end.

%% Formatted writing and reading.

fwrite(Format) ->
    fwrite(Format, []).

fwrite(Format, Args) ->
    fwrite(default_output(), Format, Args).

fwrite(Io, Format, Args) ->
    request(Io, {fwrite,Format,Args}).

fread(Prompt, Format) ->
    fread(default_input(), Prompt, Format).

fread(Io, Prompt, Format) ->
    request(Io, {fread,Prompt,Format}).

format(Format) ->
    format(Format, []).

format(Format, Args) ->
    format(default_output(), Format, Args).

format(Io, Format, Args) ->
    request(Io, {format,Format,Args}).

%% Scanning Erlang code.

scan_erl_exprs(Prompt) ->
    scan_erl_exprs(default_input(), Prompt, 1).

scan_erl_exprs(Io, Prompt) ->
    scan_erl_exprs(Io, Prompt, 1).

scan_erl_exprs(Io, Prompt, Pos0) ->
    request(Io, {get_until,Prompt,erl_scan,tokens,[Pos0]}).

scan_erl_form(Prompt) ->
    scan_erl_form(default_input(), Prompt, 1).

scan_erl_form(Io, Prompt) ->
    scan_erl_form(Io, Prompt, 1).

scan_erl_form(Io, Prompt, Pos0) ->
    request(Io, {get_until,Prompt,erl_scan,tokens,[Pos0]}).

%% Parsing Erlang code.

parse_erl_exprs(Prompt) ->
    parse_erl_exprs(default_input(), Prompt, 1).

parse_erl_exprs(Io, Prompt) ->
    parse_erl_exprs(Io, Prompt, 1).

parse_erl_exprs(Io, Prompt, Pos0) ->
    case request(Io, {get_until,Prompt,erl_scan,tokens,[Pos0]}) of
	{ok,Toks,EndPos} ->
	    case erl_parse:parse_exprs(Toks) of
		{ok,Exprs} -> {ok,Exprs,EndPos};
		{error,E} -> {error,E,EndPos}
	    end;
	Other ->
	    Other
    end.

parse_erl_form(Prompt) ->
    parse_erl_form(default_input(), Prompt, 1).

parse_erl_form(Io, Prompt) ->
    parse_erl_form(Io, Prompt, 1).

parse_erl_form(Io, Prompt, Pos0) ->
    case request(Io, {get_until,Prompt,erl_scan,tokens,[Pos0]}) of
	{ok,Toks,EndPos} ->
	    case erl_parse:parse_form(Toks) of
		{ok,Exprs} -> {ok,Exprs,EndPos};
		{error,E} -> {error,E,EndPos}
	    end;
	Other ->
	    Other
    end.

%% Miscellaneous functions.

request(Request) ->
    request(default_output(), Request).

request(Io, Request) ->
    case io_request(Io, io_request(Request)) of
	{error, E} ->
	    {error, E};
	_ ->
	    wait_io_reply(Io)
    end.

requests(Requests) ->				%Requests as atomic action
    requests(default_output(), Requests).

requests(Io, Requests) ->			%Requests as atomic action
    case io_request(Io, {requests,io_requests(Requests)}) of
	{error, E} ->
	    {error, E};
	_ ->
	    wait_io_reply(Io)
    end.

default_input() ->
    group_leader().

default_output() ->
    group_leader().

io_request(standard_io, Request) ->
    group_leader() ! {io_request,self(),standard_io,Request};
io_request(Io, Request) ->
    case catch Io ! {io_request,self(),Io,Request} of
	{'EXIT', R} -> {error, arguments};
	X -> X
    end.

wait_io_reply(From) ->
    wait_io_reply(From, status_p(From)).

wait_io_reply(From, undefined) ->
    {error, terminated};
wait_io_reply(From, _) ->
    receive
	{io_reply,From,Reply} ->
	    Reply;
	{'EXIT',From,Reason} ->			%In case we are trapping exits
	    {error,terminated}
    end.

status_p(P) when pid(P) , node(P) == node()  ->
    process_info(P, status);
status_p(undefined) ->
    undefined;
status_p(standard_io) ->
    status_p(group_leader());
status_p(N) when atom(N) -> 
    status_p(whereis(N));
status_p(_) -> {status, remote}.

    
%% io_requests(Requests)
%%  Transform requests into correct i/o server messages. Only handle the
%%  one we KNOW must be changed, others, including incorrect ones, are
%%  passed straight through. Perform a flatten on the request list.

io_requests(Rs) ->
    io_requests(Rs, [], []).

io_requests([{requests,Rs1}|Rs], Cont, Tail) ->
    io_requests(Rs1, [Rs|Cont], Tail);
io_requests([R|Rs], Cont, Tail) ->
    [io_request(R)|io_requests(Rs, Cont, Tail)];
io_requests([], [Rs|Cont], Tail) ->
    io_requests(Rs, Cont, Tail);
io_requests([], [], Tail) ->
    [].

io_request({write,Term}) ->
    {put_chars,io_lib,write,[Term]};
io_request({format,Format,Args}) ->
    {put_chars,io_lib,format,[Format,Args]};
io_request({fwrite,Format,Args}) ->
    {put_chars,io_lib,fwrite,[Format,Args]};
io_request(nl) ->
    {put_chars,io_lib:nl()};
io_request({get_chars,Prompt,N}) ->
    {get_until,Prompt,io_lib,collect_chars,[N]};
io_request({get_line,Prompt}) ->
    {get_until,Prompt,io_lib,collect_line,[]};
io_request({fread,Prompt,Format}) ->
    {get_until,Prompt,io_lib,fread,[Format]};
io_request(R) ->				%Pass this straight through
    R.
