/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.model.xml.web.converters;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInsight.lookup.LookupElementBuilder;
import com.intellij.javaee.model.xml.web.HttpStatusCode;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.ResolvingConverter;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.EnumSet;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * @author Yann C&eacute;bron
 * @since 14
 */
public class HttpStatusCodeConverter extends ResolvingConverter<HttpStatusCode> {

  private static final Pattern HTTP_STATUS_CODE = Pattern.compile("\\d{3}");

  private final Set<HttpStatusCode> myAllowedStatusCodes;
  private final boolean myCustomCodeAllowed;


  @SuppressWarnings("unused")
  public HttpStatusCodeConverter() {
    this(EnumSet.allOf(HttpStatusCode.class), true);
  }

  public HttpStatusCodeConverter(Set<HttpStatusCode> allowedStatusCodes, boolean customCodeAllowed) {
    myAllowedStatusCodes = allowedStatusCodes;
    myAllowedStatusCodes.remove(HttpStatusCode.CUSTOM);

    myCustomCodeAllowed = customCodeAllowed;
  }

  @Override
  public final HttpStatusCode fromString(final String s, final ConvertContext context) {
    return findHttpStatusCode(s);
  }

  @Nullable
  private HttpStatusCode findHttpStatusCode(@Nullable final String s) {
    if (s == null) {
      return null;
    }

    final HttpStatusCode value = ContainerUtil.find(myAllowedStatusCodes, code -> code.getValue().equals(s));

    if (value != null) {
      return value;
    }

    if (!myCustomCodeAllowed) {
      return null;
    }

    // allow custom 3-digit (error) codes
    return HTTP_STATUS_CODE.matcher(s).matches() ? HttpStatusCode.CUSTOM : null;
  }

  @Override
  public final String toString(final HttpStatusCode statusCode, final ConvertContext context) {
    return statusCode == null ? null : statusCode.getValue();
  }

  @NotNull
  @Override
  public Collection<HttpStatusCode> getVariants(ConvertContext context) {
    return myAllowedStatusCodes;
  }

  @Override
  public String getErrorMessage(@Nullable final String s, final ConvertContext context) {
    if (s == null || !HTTP_STATUS_CODE.matcher(s).matches()) {
      return "Invalid HTTP status code '" + s + "'";
    }

    return "HTTP status code '" + s + "' not allowed here";
  }

  @Nullable
  @Override
  public LookupElement createLookupElement(HttpStatusCode statusCode) {
    return LookupElementBuilder.create(statusCode.getValue())
      .withStrikeoutness(statusCode.isDeprecated())
      .withTailText(" (" + statusCode.getReason() + ")", true);
  }
}
