/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.facet.ProjectFacetManager;
import com.intellij.ide.fileTemplates.FileTemplate;
import com.intellij.ide.fileTemplates.FileTemplateUtil;
import com.intellij.jam.JamService;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.libraries.JarVersionDetectionUtil;
import com.intellij.openapi.util.io.FileUtilRt;
import com.intellij.openapi.util.text.CharFilter;
import com.intellij.psi.*;
import com.intellij.psi.search.FilenameIndex;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.PackageScope;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PsiModificationTracker;
import com.intellij.psi.util.PsiUtil;
import com.intellij.spring.SpringLibraryUtil;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.constants.SpringConstants;
import com.intellij.spring.facet.SpringFacet;
import com.intellij.spring.facet.SpringSchemaVersion;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.jam.stereotype.SpringComponent;
import com.intellij.spring.model.jam.stereotype.SpringConfiguration;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class SpringCommonUtils {

  public static final String SPRING_DELIMITERS = ",; ";

  public static final CharFilter ourFilter = ch -> SPRING_DELIMITERS.indexOf(ch) >= 0;

  private SpringCommonUtils() {
  }

  public static List<String> tokenize(@NotNull String str) {
    final List<String> list = new SmartList<>();
    StringTokenizer st = new StringTokenizer(str, SPRING_DELIMITERS);
    while (st.hasMoreTokens()) {
      final String token = st.nextToken().trim();
      if (!token.isEmpty()) {
        list.add(token);
      }
    }
    return list;
  }

  @NotNull
  public static List<PsiType> resolveGenerics(PsiClassType classType) {
    final PsiClassType.ClassResolveResult resolveResult = classType.resolveGenerics();
    final PsiClass psiClass = resolveResult.getElement();
    if (psiClass != null) {
      final PsiSubstitutor substitutor = resolveResult.getSubstitutor();
      List<PsiType> generics = new SmartList<>();
      for (PsiTypeParameter typeParameter : psiClass.getTypeParameters()) {
        generics.add(substitutor.substitute(typeParameter));
      }
      return generics;
    }
    return Collections.emptyList();
  }

  /**
   * @param project Current project.
   * @return Whether given project contains at least one Spring facet.
   * @since 15.1
   */
  public static boolean hasSpringFacets(@NotNull final Project project) {
    return ProjectFacetManager.getInstance(project).hasFacets(SpringFacet.FACET_TYPE_ID);
  }

  /**
   * @return Whether given module has Spring facet.
   * @since 2016.3
   */
  public static boolean hasSpringFacet(@Nullable Module module) {
    return module != null && SpringFacet.getInstance(module) != null;
  }

  /**
   * Returns whether the given class has {@value SpringAnnotationsConstants#JAVA_SPRING_CONFIGURATION} annotation, <em>excluding</em> meta-annotations.
   *
   * @see #isConfigurationOrMeta(PsiClass)
   */
  public static boolean isConfiguration(@NotNull PsiClass psiClass) {
    return isSpringBeanCandidateClass(psiClass) &&
           AnnotationUtil.isAnnotated(psiClass, SpringAnnotationsConstants.JAVA_SPRING_CONFIGURATION, 0);
  }

  /**
   * Returns whether the given class is <em>(meta-)</em>annotated with {@value SpringAnnotationsConstants#JAVA_SPRING_CONFIGURATION}.
   *
   * @param psiClass Class to check.
   * @return {@code true} if class annotated.
   * @since 15
   */
  public static boolean isConfigurationOrMeta(@NotNull PsiClass psiClass) {
    if (!isSpringBeanCandidateClass(psiClass)) return false;
    return JamService.getJamService(psiClass.getProject()).getJamElement(SpringConfiguration.JAM_KEY, psiClass) != null;
  }

  /**
   * Returns whether the given class is <em>(meta-)</em>annotated with {@value SpringAnnotationsConstants#COMPONENT}.
   *
   * @param psiClass Class to check.
   * @return {@code true} if class annotated.
   * @since 2017.3
   */
  public static boolean isComponentOrMeta(@NotNull PsiClass psiClass) {
    if (!isSpringBeanCandidateClass(psiClass)) return false;
    return JamService.getJamService(psiClass.getProject())
             .getJamElement(psiClass, SpringComponent.META) != null;
  }

  /**
   * Returns whether the given class is <em>(meta-)</em>annotated with {@value SpringAnnotationsConstants#COMPONENT, SpringAnnotationsConstants#SERVICE, , SpringAnnotationsConstants#REPOSITORY}.
   *
   * @param psiClass Class to check.
   * @return {@code true} if class annotated.
   * @since 2017.2
   */
  public static boolean isStereotypeComponentOrMeta(@NotNull PsiClass psiClass) {
    if (!isSpringBeanCandidateClass(psiClass)) return false;
    return JamService.getJamService(psiClass.getProject())
             .getJamElement(JamPsiMemberSpringBean.PSI_MEMBER_SPRING_BEAN_JAM_KEY, psiClass) != null;
  }

  /**
   * Returns whether the given PsiClass (or its inheritors) could possibly be mapped as Spring Bean.
   *
   * @param psiClass PsiClass to check.
   * @return {@code true} if yes.
   * @since 14
   */
  @Contract("null->false")
  public static boolean isSpringBeanCandidateClass(@Nullable PsiClass psiClass) {
    if (psiClass == null ||
        psiClass instanceof PsiTypeParameter ||
        psiClass.hasModifierProperty(PsiModifier.PRIVATE) ||
        psiClass.isAnnotationType() ||
        psiClass.getQualifiedName() == null ||
        PsiUtil.isLocalOrAnonymousClass(psiClass)) {
      return false;
    }
    return true;
  }

  public static PsiElement createSpringXmlConfigFile(String newName, PsiDirectory directory) throws Exception {
    final Module module = ModuleUtilCore.findModuleForPsiElement(directory);
    final FileTemplate template = getSpringXmlTemplate(module);
    @NonNls final String fileName = FileUtilRt.getExtension(newName).length() == 0 ? newName + ".xml" : newName;
    return FileTemplateUtil.createFromTemplate(template, fileName, null, directory);
  }

  public static FileTemplate getSpringXmlTemplate(final Module... modules) {
    for (Module module : modules) {
      final String version = JarVersionDetectionUtil.detectJarVersion(SpringConstants.SPRING_VERSION_CLASS, module);
      if (version != null) {
        return version.startsWith("1") ?
               SpringSchemaVersion.SPRING_1_DTD.getTemplate(module.getProject()) :
               SpringSchemaVersion.SPRING_SCHEMA.getTemplate(module.getProject());
      }
    }
    return SpringSchemaVersion.SPRING_SCHEMA.getTemplate(modules[0].getProject());
  }

  @Contract("null -> false")
  public static boolean isSpringConfigured(@Nullable Module module) {
    if (module != null) {
      if (hasSpringFacet(module)) return true;
      for (Module dependent : ModuleUtilCore.getAllDependentModules(module)) {
        if (hasSpringFacet(dependent)) return true;
      }
      Set<Module> dependencies = ContainerUtil.newHashSet();
      ModuleUtilCore.getDependencies(module, dependencies);
      for (Module dependency : dependencies) {
        if (hasSpringFacet(dependency)) return true;
      }
    }

    return false;
  }

  /**
   * Returns true if the given class is a bean candidate, Spring library is present in project and
   * Spring facet in current/dependent module(s) (or at least one in Project for PsiClass located in JAR) exists.
   *
   * @param psiClass Class to check.
   * @return true if all conditions apply
   * @since 14.1
   */
  @Contract("null->false")
  public static boolean isSpringBeanCandidateClassInSpringProject(@Nullable PsiClass psiClass) {
    if (psiClass == null) {
      return false;
    }

    if (!hasSpringFacets(psiClass.getProject())) {
      return false;
    }

    if (!SpringLibraryUtil.hasSpringLibrary(psiClass.getProject())) {
      return false;
    }

    if (!isSpringBeanCandidateClass(psiClass)) {
      return false;
    }

    final Module module = ModuleUtilCore.findModuleForPsiElement(psiClass);
    if (isSpringConfigured(module) || SpringModelUtils.getInstance().hasAutoConfiguredModels(module)) {
      return true;
    }

    // located in JAR
    return module == null;
  }

  /**
   * Intersects given scope with {@code META-INF} package scope.
   *
   * @param project Project to search in.
   * @param scope   Scope to search in.
   * @return {@code null} if {@code META-INF} package does not exist, adjusted scope otherwise.
   * @since 2018.2.5
   */
  @Nullable
  public static GlobalSearchScope getConfigFilesScope(Project project, GlobalSearchScope scope) {
    final PsiPackage metaInfPackage = JavaPsiFacade.getInstance(project).findPackage("META-INF");
    if (metaInfPackage == null) {
      return null;
    }
    final GlobalSearchScope packageScope = PackageScope.packageScope(metaInfPackage, false);
    return scope.intersectWith(packageScope);
  }

  /**
   * Finds all configuration files in {@code META-INF} package with given name in module runtime scope.
   *
   * @param module      Module to search.
   * @param withTests   Include tests scope.
   * @param filename    Config file name.
   * @param psiFileType Expected PsiFile type.
   * @param <T>         Type.
   * @return List of matching files.
   * @since 14.1
   */
  public static <T extends PsiFile> List<T> findConfigFilesInMetaInf(Module module,
                                                                     boolean withTests,
                                                                     String filename,
                                                                     Class<T> psiFileType) {
    final GlobalSearchScope moduleScope = GlobalSearchScope.moduleRuntimeScope(module, withTests);
    return findConfigFilesInMetaInf(module.getProject(), moduleScope, filename, psiFileType);
  }

  /**
   * Finds all configuration files with given name in {@code META-INF} package within search scope.
   *
   * @param <T>         Type.
   * @param project     Project to search in.
   * @param scope       Search scope.
   * @param filename    Config file name
   * @param psiFileType Expected PsiFile type.
   * @return List of matching files.
   * @since 2017.1
   */
  @NotNull
  public static <T extends PsiFile> List<T> findConfigFilesInMetaInf(Project project,
                                                                     GlobalSearchScope scope,
                                                                     String filename,
                                                                     Class<T> psiFileType) {
    GlobalSearchScope searchScope = getConfigFilesScope(project, scope);
    if (searchScope == null) {
      return Collections.emptyList();
    }

    final PsiFile[] configFiles = FilenameIndex.getFilesByName(project, filename, searchScope);
    if (configFiles.length == 0) {
      return Collections.emptyList();
    }

    return ContainerUtil.findAll(configFiles, psiFileType);
  }

  /**
   * Returns the <em>library class</em> resolved in runtime production scope of given module, caching the result.
   *
   * @param module    Module.
   * @param className FQN of class located in libraries to find.
   * @return {@code null} if module is {@code null} or class not found.
   * @since 15
   */
  public static PsiClass findLibraryClass(@Nullable final Module module, @NotNull final String className) {
    if (module == null || module.isDisposed()) {
      return null;
    }

    final Project project = module.getProject();
    final Map<String, PsiClass> cache =
      CachedValuesManager.getManager(project).getCachedValue(module, () -> {
        final Map<String, PsiClass> map = ConcurrentFactoryMap.createMap(key -> {
                                                                           final GlobalSearchScope searchScope = GlobalSearchScope.moduleRuntimeScope(module, false);
                                                                           return JavaPsiFacade.getInstance(project).findClass(key, searchScope);
                                                                         }
        );
        return Result.createSingleDependency(map, PsiModificationTracker.JAVA_STRUCTURE_MODIFICATION_COUNT);
      });
    return cache.get(className);
  }
}
