// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.reflect.*;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.aliasFor.SpringAliasFor;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.converters.PackageJamConverter;
import com.intellij.spring.model.jam.utils.filters.SpringContextFilter;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public class CustomJamComponentScan extends SpringComponentScan {

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = COMPONENT_SCAN_META_KEY.subKey("SpringJamComponentScan");

  public static final SemKey<CustomJamComponentScan> JAM_KEY = SpringComponentScan.COMPONENT_SCAN_JAM_KEY.subKey("CustomJamComponentScan");
  public static final SemKey<JamMemberMeta<PsiClass, CustomJamComponentScan>> META_KEY =
    SpringJamComponentScan.META.getMetaKey().subKey("CustomJamComponentScan");

  private final JamAnnotationMeta myMeta;
  private final AnnotationChildLink myAnnotationChildLink;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;

  public CustomJamComponentScan(@NotNull String anno, @NotNull PsiClass psiClassAnchor) {
    super(psiClassAnchor);
    myAnnotationChildLink = new AnnotationChildLink(anno);
    myPsiAnnotation = myAnnotationChildLink.createChildRef(psiClassAnchor);
    myMeta = new JamAnnotationMeta(anno);
  }

  @Override
  @NotNull
  public JamAnnotationMeta getAnnotationMeta() {
    return myMeta;
  }

  @Override
  @NotNull
  public PsiElementRef<PsiAnnotation> getAnnotationRef() {
    return myPsiAnnotation;
  }

  @NotNull
  @Override
  public List<JamStringAttributeMeta.Collection<Collection<PsiPackage>>> getPackageJamAttributes() {
    List<JamStringAttributeMeta.Collection<Collection<PsiPackage>>> list = ContainerUtil.newSmartList();

    ContainerUtil.addIfNotNull(list, getPackagesAttrMeta(BASE_PACKAGES_ATTR_NAME));
    ContainerUtil.addIfNotNull(list, getPackagesAttrMeta(VALUE_ATTR_NAME));

    return list;
  }

  @Override
  @Nullable
  protected JamClassAttributeMeta.Collection getBasePackageClassMeta() {
    SpringAliasFor aliasFor = getAliasAttribute(BASE_PACKAGE_CLASSES_ATTR_NAME);
    if (aliasFor != null) {
      return JamAttributeMeta.classCollection(aliasFor.getMethodName());
    }
    return null;
  }

  @Nullable
  private JamStringAttributeMeta.Collection<Collection<PsiPackage>> getPackagesAttrMeta(@NotNull String name) {
    SpringAliasFor aliasFor = getAliasAttribute(name);
    if (aliasFor != null) {
      return new JamStringAttributeMeta.Collection<>(aliasFor.getMethodName(), new PackageJamConverter());
    }
    return null;
  }

  @Override
  public boolean useDefaultFilters() {
    return true;
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Exclude> getExcludeContextFilters() {
    return Collections.emptySet(); // todo
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Include> getIncludeContextFilters() {
    return Collections.emptySet();  //todo
  }

  protected SpringAliasFor getAliasAttribute(@NotNull String attrName) {
    return SpringAliasForUtils.findAliasFor(getPsiElement(), getAnnotationQualifiedName(),
                                            SpringAnnotationsConstants.COMPONENT_SCAN, attrName);
  }

  protected String getAnnotationQualifiedName() {
    return myAnnotationChildLink.getAnnotationQualifiedName();
  }
}
