// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.converters;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.psi.PsiClassType;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.PsiType;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.SpringManager;
import com.intellij.spring.contexts.model.SpringModel;
import com.intellij.spring.contexts.model.XmlSpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.converters.fixes.bean.SpringBeanResolveQuickFixManager;
import com.intellij.spring.model.utils.SpringBeanUtils;
import com.intellij.spring.model.values.PlaceholderUtils;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.spring.model.xml.beans.TypeHolder;
import com.intellij.spring.model.xml.beans.TypeHolderUtil;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ReflectionUtil;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.*;
import com.intellij.util.xml.impl.GenericDomValueReference;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public class SpringBeanResolveConverter extends ResolvingConverter<SpringBeanPointer> implements CustomReferenceConverter {

  @Override
  public boolean canResolveTo(Class<? extends PsiElement> elementClass) {
    return ReflectionUtil.isAssignable(SpringBeanPointer.class, elementClass);
  }

  @Nullable
  protected CommonSpringModel getSpringModel(final ConvertContext context) {
    return SpringManager.getInstance(context.getFile().getProject()).getSpringModelByFile(context.getFile());
  }

  /**
   * Used to filter variants by bean class.
   *
   * @param context conversion context.
   * @return empty list if no requirement applied.
   */
  @NotNull
  public List<PsiClassType> getRequiredClasses(ConvertContext context) {
    return SpringConverterUtil.getRequiredBeanTypeClasses(context);
  }

  @Override
  @Nullable
  public SpringBeanPointer fromString(final @Nullable String s, final ConvertContext context) {
    if (s == null) return null;

    CommonSpringModel springModel = getSpringModel(context);
    if (springModel == null) return null;

    return SpringBeanUtils.getInstance().findBean(springModel, s);
  }

  @Override
  public String toString(final @Nullable SpringBeanPointer springBeanPointer, final ConvertContext context) {
    return springBeanPointer == null ? null : springBeanPointer.getName();
  }

  @Override
  public LookupElement createLookupElement(SpringBeanPointer springBeanPointer) {
    return SpringConverterUtil.createCompletionVariant(springBeanPointer);
  }

  @Override
  public PsiElement getPsiElement(@Nullable SpringBeanPointer resolvedValue) {
    if (resolvedValue == null || !resolvedValue.isValid()) return null;

    return resolvedValue.getPsiElement();
  }

  @Override
  public String getErrorMessage(final String s, final ConvertContext context) {
    return SpringApiBundle.message("model.bean.error.message", s);
  }

  @NotNull
  @Override
  public PsiReference[] createReferences(GenericDomValue value, PsiElement element, ConvertContext context) {
    if (isPlaceholder(context)) {
      return ArrayUtil.append(PlaceholderUtils.getInstance().createPlaceholderPropertiesReferences(value),
                              new GenericDomValueReference(value));
    }
    return PsiReference.EMPTY_ARRAY;
  }

  @Override
  public LocalQuickFix[] getQuickFixes(final ConvertContext context) {
    final GenericDomValue element = (GenericDomValue)context.getInvocationElement();
    return SpringBeanResolveQuickFixManager.getInstance().getQuickFixes(context,
                                                                        element.getParentOfType(Beans.class, false),
                                                                        null,
                                                                        getRequiredClasses(context));
  }

  @Override
  @NotNull
  public Collection<SpringBeanPointer> getVariants(final ConvertContext context) {
    if (isPlaceholder(context)) {
      return Collections.emptySet();
    }

    return getVariants(context, false, false, getRequiredClasses(context), getSpringModel(context));
  }

  protected static boolean isPlaceholder(ConvertContext context) {
    final DomElement element = context.getInvocationElement();
    if (element instanceof GenericDomValue &&
        PlaceholderUtils.getInstance().isRawTextPlaceholder((GenericDomValue)element)) {
      return true;
    }
    return false;
  }

  protected static Collection<SpringBeanPointer> getVariants(ConvertContext context,
                                                             boolean parentBeans,
                                                             boolean allowAbstracts,
                                                             List<PsiClassType> requiredClasses,
                                                             CommonSpringModel model) {
    if (model == null) return Collections.emptyList();

    final List<SpringBeanPointer> variants = new ArrayList<>();
    final CommonSpringBean currentBean = SpringConverterUtil.getCurrentBeanCustomAware(context);

    // remove java.lang.Object (e.g. "fallback type" via TypeHolder.getRequiredTypes)
    if (!requiredClasses.isEmpty()) {
      final PsiClassType object = PsiType.getJavaLangObject(context.getPsiManager(), GlobalSearchScope.allScope(context.getProject()));
      if (requiredClasses.contains(object)) {
        requiredClasses = new ArrayList<>(requiredClasses); // guard against immutable
        requiredClasses.remove(object);
      }
    }

    Collection<SpringBeanPointer> pointers = getModelVariants(parentBeans, model, requiredClasses, currentBean);
    SpringConverterUtil.processBeans(model, variants, pointers, allowAbstracts, currentBean);
    return variants;
  }

  private static Collection<SpringBeanPointer> getModelVariants(boolean parentBeans,
                                                                CommonSpringModel model,
                                                                @NotNull List<PsiClassType> requiredClasses,
                                                                CommonSpringBean currentBean) {
    if (parentBeans) {
      if (model instanceof XmlSpringModel) {
        final Collection<SpringBeanPointer> allBeans = new ArrayList<>();
        for (SpringModel springModel : ((XmlSpringModel)model).getDependencies()) {
          if (requiredClasses.isEmpty()) {
            allBeans.addAll(springModel.getAllCommonBeans());
          }
          else {
            allBeans.addAll(SpringConverterUtil.getSmartVariants(currentBean, requiredClasses, model));
          }
        }
        return allBeans;
      }
      return Collections.emptySet();
    }

    return requiredClasses.isEmpty()
           ? model.getAllCommonBeans()
           : SpringConverterUtil.getSmartVariants(currentBean, requiredClasses, model);
  }

  @NotNull
  protected static List<PsiClassType> getValueClasses(ConvertContext context) {
    final TypeHolder valueHolder = context.getInvocationElement().getParentOfType(TypeHolder.class, false);
    if (valueHolder == null) {
      return Collections.emptyList(); // invalid XML
    }
    return ContainerUtil.findAll(TypeHolderUtil.getRequiredTypes(valueHolder), PsiClassType.class);
  }


  public static class PropertyBean extends SpringBeanResolveConverter {

    @Override
    @NotNull
    public List<PsiClassType> getRequiredClasses(ConvertContext context) {
      return getValueClasses(context);
    }
  }

  public static class Parent extends SpringBeanResolveConverter {
    @NotNull
    @Override
    public Collection<SpringBeanPointer> getVariants(ConvertContext context) {
      return getVariants(context, false, true, getRequiredClasses(context), getSpringModel(context));
    }
  }
}
