/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.aliasFor;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamConverter;
import com.intellij.jam.JamService;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.semantic.SemKey;
import com.intellij.spring.model.jam.SpringSemContributorUtil;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.Consumer;
import com.intellij.util.NotNullFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

import static com.intellij.codeInsight.AnnotationUtil.CHECK_HIERARCHY;

/**
 * @since 16
 */
public class SpringAliasForUtils {

  @Nullable
  public static SpringAliasFor findAliasFor(@NotNull PsiElement context,
                                            @Nullable String toSearchInAnnotation,
                                            @NotNull String aliasedClassName,
                                            @NotNull String attrName) {
    return getAliasFor(context.getProject(), context.getResolveScope(), toSearchInAnnotation, aliasedClassName, attrName);
  }

  @Nullable
  public static SpringAliasFor findAliasFor(@NotNull Project project,
                                            @Nullable String toSearchInAnnotation,
                                            @NotNull String aliasedClassName,
                                            @NotNull String attrName) {
    return getAliasFor(project, GlobalSearchScope.allScope(project), toSearchInAnnotation, aliasedClassName, attrName);
  }

  protected static SpringAliasFor getAliasFor(@NotNull Project project,
                                              @NotNull GlobalSearchScope scope,
                                              @Nullable String toSearchInAnnotation,
                                              @NotNull String aliasedClassName, @NotNull String attrName) {
    if (toSearchInAnnotation == null) return null;

    final JavaPsiFacade javaPsiFacade = JavaPsiFacade.getInstance(project);
    PsiClass toSearchInClass = javaPsiFacade.findClass(toSearchInAnnotation, scope);
    if (toSearchInClass == null || !toSearchInClass.isAnnotationType()) {
      return null;
    }

    PsiClass annotationClass = javaPsiFacade.findClass(aliasedClassName, scope);
    if (annotationClass == null || !annotationClass.isAnnotationType()) {
      return null;
    }

    for (SpringAliasFor aliasFor : getAliasForAttributes(toSearchInClass)) {
      if (attrName.equals(aliasFor.getAttributeName()) && annotationClass.equals(aliasFor.getAnnotationClass())) {
        return aliasFor;
      }
    }
    return null;
  }

  @NotNull
  private static List<SpringAliasFor> getAliasForAttributes(@NotNull PsiClass psiClass) {
    return CachedValuesManager.getCachedValue(psiClass, () -> {
      final List<SpringAliasFor> aliasForList = JamService.getJamService(psiClass.getProject())
        .getAnnotatedMembersList(psiClass, SpringAliasFor.SEM_KEY,
                                 false, true, false, false);
      return CachedValueProvider.Result.create(aliasForList, psiClass);
    });
  }

  @Nullable
  public static PsiAnnotation findDefiningMetaAnnotation(@Nullable PsiElement context,
                                                         @NotNull String customAnnotationFqn,
                                                         @NotNull String baseMetaAnnotationFqn) {
    if (context == null) return null;
    final Module module = ModuleUtilCore.findModuleForPsiElement(context);
    if (module == null) return null;

    PsiClass customAnnoClass = JavaPsiFacade.getInstance(module.getProject())
      .findClass(customAnnotationFqn, context.getResolveScope());

    return findDefiningMetaAnnotation(customAnnoClass, baseMetaAnnotationFqn,
                                      JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(baseMetaAnnotationFqn));
  }

  @Nullable
  public static PsiAnnotation findDefiningMetaAnnotation(@Nullable PsiClass customAnnoClass,
                                                         @NotNull String baseMetaAnnotationFqn,
                                                         @NotNull final Collection<? extends PsiClass> allMetaAnnotations) {
    if (customAnnoClass == null || !customAnnoClass.isAnnotationType()) {
      return null;
    }

    final PsiAnnotation psiAnnotation = AnnotationUtil.findAnnotation(customAnnoClass, true, baseMetaAnnotationFqn);
    if (psiAnnotation != null) {
      return psiAnnotation;
    }

    for (PsiClass customMetaAnnoClass : allMetaAnnotations) {
      final String qualifiedName = customMetaAnnoClass.getQualifiedName();
      if (qualifiedName != null && AnnotationUtil.isAnnotated(customAnnoClass, qualifiedName, CHECK_HIERARCHY)) {
        return findDefiningMetaAnnotation(customMetaAnnoClass, baseMetaAnnotationFqn, allMetaAnnotations);
      }
    }
    return null;
  }

  /**
   * Builds JamAttributeMeta supporting converters for parent's meta String attributes.
   *
   * @param annoMetaKey SemKey.
   * @param parentMetas Parent metas to search for attributes.
   * @return Function.
   * @see SpringSemContributorUtil#createFunction(SemKey, Class, com.intellij.util.Function, com.intellij.util.Function, Consumer, NotNullFunction)
   * @since 2017.1
   */
  public static NotNullFunction<Pair<String, Project>, JamAnnotationMeta> getAnnotationMetaProducer(@NotNull SemKey<JamAnnotationMeta> annoMetaKey,
                                                                                                    @NotNull JamMemberMeta<?, ?>... parentMetas) {
    return anno -> new JamAnnotationMeta(anno.first, null, annoMetaKey) {

      @Nullable
      @Override
      public JamAttributeMeta<?> findAttribute(@Nullable @NonNls String name) {
        JamAttributeMeta<?> attribute = super.findAttribute(name);
        if (attribute != null) return attribute;

        if (name == null) name = "value";
        return getAliasedAttributeMeta(name);
      }

      @Nullable
      private JamAttributeMeta<?> getAliasedAttributeMeta(@NotNull String name) {
        for (JamMemberMeta<?, ?> parentMeta : parentMetas) {
          for (JamAnnotationMeta annotationMeta : parentMeta.getAnnotations()) {
            List<JamAttributeMeta<?>> parentAnnotationAttributes = getRegisteredAttributes(annotationMeta);
            for (JamAttributeMeta<?> attributeMeta : parentAnnotationAttributes) {
              if (attributeMeta instanceof JamStringAttributeMeta) {
                final JamStringAttributeMeta<?, ?> meta = (JamStringAttributeMeta)attributeMeta;
                final JamConverter<?> converter = meta.getConverter();

                SpringAliasFor aliasFor = findAliasFor(anno.second,
                                                       anno.first,
                                                       annotationMeta.getAnnoName(),
                                                       meta.getAttributeLink().getAttributeName());
                if (aliasFor != null) {
                  String aliasForMethodName = aliasFor.getMethodName();
                  if (name.equals(aliasForMethodName)) {
                    if (attributeMeta instanceof JamStringAttributeMeta.Single) {
                      return JamAttributeMeta.singleString(aliasForMethodName, converter);
                    }
                    if (attributeMeta instanceof JamStringAttributeMeta.Collection) {
                      return JamAttributeMeta.collectionString(aliasForMethodName, converter);
                    }
                  }
                }
              }
            }
          }
        }
        return null;
      }

      private List<JamAttributeMeta<?>> getRegisteredAttributes(@NotNull JamAnnotationMeta annotationMeta) {
        List<JamAttributeMeta<?>> attributeMetas = ContainerUtil.newSmartList();
        attributeMetas.addAll(annotationMeta.getAttributes());
        JamAnnotationArchetype archetype = annotationMeta.getArchetype();
        if (archetype != null) {
          attributeMetas.addAll(archetype.getAttributes());
        }
        return attributeMetas;
      }
    };
  }
}
