// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.aliasFor;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.reflect.*;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * @see SpringAliasForUtils
 * @since 16
 */
public abstract class SpringAliasFor implements JamElement {

  public static final SemKey<SpringAliasFor> SEM_KEY = SemKey.createKey("SpringAliasFor");

  private static final JamClassAttributeMeta.Single ALIAS_FOR_CLASS_ATTR_META = JamAttributeMeta.singleClass("annotation");
  private static final JamStringAttributeMeta.Single<PsiMethod> ALIAS_FOR_ATTR_META =
    JamAttributeMeta.singleString("attribute", new AliasForAttributePsiMethodJamConverter());

  private static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.ALIAS_FOR)
    .addAttribute(ALIAS_FOR_CLASS_ATTR_META)
    .addAttribute(ALIAS_FOR_ATTR_META);

  public static final JamMethodMeta<SpringAliasFor> METHOD_META =
    new JamMethodMeta<>(null, SpringAliasFor.class, SEM_KEY).addAnnotation(ANNO_META);

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final PsiMethod myPsiMethod;

  @SuppressWarnings("unused")
  public SpringAliasFor(@NotNull PsiMethod psiMethod) {
    myPsiMethod = psiMethod;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiMethod);
  }

  @SuppressWarnings("unused")
  public SpringAliasFor(PsiAnnotation annotation) {
    myPsiAnnotation = PsiElementRef.real(annotation);
    myPsiMethod = PsiTreeUtil.getParentOfType(annotation, PsiMethod.class, true);
  }

  @NotNull
  public PsiMethod getPsiElement() {
    return myPsiMethod;
  }

  /**
   * @return Name of annotated method.
   * @since 2016.3
   */
  @NotNull
  public String getMethodName() {
    return myPsiMethod.getName();
  }

  @JamPsiValidity
  public abstract boolean isPsiValid();

  @Nullable
  private static PsiClass getAliasForAnnotationClass(@NotNull PsiMethod method) {
    PsiClass forAnnotationClass = ANNO_META.getAttribute(method, ALIAS_FOR_CLASS_ATTR_META).getValue();
    return forAnnotationClass == null ? method.getContainingClass() : forAnnotationClass;
  }

  @Nullable
  public PsiClass getAnnotationClass() {
    return getAliasForAnnotationClass(getPsiElement());
  }

  /**
   * @return Returns configured {@code attribute} value or {@link #getMethodName()} when not defined.
   */
  public String getAttributeName() {
    final JamStringAttributeElement<PsiMethod> attribute = ALIAS_FOR_ATTR_META.getJam(myPsiAnnotation);
    return attribute.getPsiLiteral() != null ? attribute.getStringValue() : getMethodName();
  }

  @Nullable
  public PsiAnnotation getAnnotation() {
    return ANNO_META.getAnnotation(getPsiElement());
  }


  private static class AliasForAttributePsiMethodJamConverter extends JamSimpleReferenceConverter<PsiMethod> {

    @Override
    public PsiMethod fromString(@Nullable String s, JamStringAttributeElement<PsiMethod> context) {
      for (PsiMethod psiMethod : getAliasForMethods(context)) {
        if (psiMethod.getName().equals(s)) {
          return psiMethod;
        }
      }
      return null;
    }

    private static PsiMethod[] getAliasForMethods(@NotNull JamStringAttributeElement<PsiMethod> context) {
      PsiMethod psiMethod = getIdentifyingMethod(context);
      if (psiMethod != null) {
        PsiClass aliasForAnnotationClass = getAliasForAnnotationClass(psiMethod);
        if (aliasForAnnotationClass != null) {
          return aliasForAnnotationClass.getAllMethods();
        }
      }
      return PsiMethod.EMPTY_ARRAY;
    }

    private static PsiMethod getIdentifyingMethod(@NotNull JamStringAttributeElement<PsiMethod> context) {
      PsiAnnotationMemberValue contextPsiElement = context.getPsiElement();
      if (contextPsiElement != null) {
        return PsiTreeUtil.getParentOfType(contextPsiElement, PsiMethod.class);
      }
      return null;
    }

    @Override
    public Collection<PsiMethod> getVariants(JamStringAttributeElement<PsiMethod> context) {
      List<PsiMethod> methods = new ArrayList<>();
      PsiMethod identifyingMethod = getIdentifyingMethod(context);
      if (identifyingMethod != null) {
        PsiType returnType = identifyingMethod.getReturnType();
        for (PsiMethod method : getAliasForMethods(context)) {
          final PsiClass containingClass = method.getContainingClass();
          if (containingClass != null && containingClass.isAnnotationType()) {
            PsiType methodReturnType = method.getReturnType();
            if (returnType == null || methodReturnType == null || returnType.isAssignableFrom(methodReturnType)) {
              methods.add(method);
            }
          }
        }
      }
      return methods;
    }
  }
}
