// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.facet.beans;

import com.intellij.openapi.util.Key;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class CustomSetting {

  private final String myName;
  private final String myDescription;

  public String getName() {
    return myName;
  }

  public String getDescription() {
    return myDescription;
  }

  public abstract void setStringValue(@NotNull String value);

  @Nullable
  public abstract String getStringValue();

  public abstract boolean isModified();

  public abstract void apply();

  public abstract void reset();

  protected CustomSetting(String name, String description) {
    myName = name;
    myDescription = description;
  }

  protected CustomSetting(Key<? extends CustomSetting> key, String description) {
    myName = key.toString();
    myDescription = description;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    CustomSetting setting = (CustomSetting)o;

    if (!myName.equals(setting.myName)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    return myName.hashCode();
  }

  /**
   * @since 2017.2
   */
  public static class STRING extends CustomSetting {

    private String myValue;
    private final String myDefaultValue;
    private boolean myModified;

    public STRING(Key<CustomSetting.STRING> key, String description, String defaultValue) {
      super(key, description);
      myDefaultValue = defaultValue;
    }

    public void clear() {
      myValue = null;
    }

    @NotNull
    public String getDefaultValue() {
      return myDefaultValue;
    }

    @Override
    public void setStringValue(@NotNull String value) {
      if (!value.equals(myValue)) {
        myValue = value;
        myModified = true;
      }
    }

    @Nullable
    @Override
    public String getStringValue() {
      return myValue;
    }

    @Override
    public boolean isModified() {
      return myModified;
    }

    @Override
    public void apply() {
      myModified = false;
    }

    @Override
    public void reset() {
      if (myModified) myValue = null;
      myModified = false;
    }
  }

  public static class BOOLEAN extends CustomSetting {
    private Boolean myValue = null;
    private final boolean myDefaultValue;
    private boolean myModified;

    public BOOLEAN(Key<CustomSetting.BOOLEAN> key, String description, boolean defaultValue) {
      super(key, description);
      myDefaultValue = defaultValue;
    }

    public void setBooleanValue(boolean value) {
      boolean currentValue = getBooleanValue();
      if (value != currentValue) {
        myModified = true;
        myValue = value;
      }
    }

    public boolean getBooleanValue() {
      return (myModified || myValue == null) ? myDefaultValue : myValue.booleanValue();
    }

    @Nullable
    public Boolean getValue() {
      return myValue;
    }

    public boolean getDefaultValue() {
      return myDefaultValue;
    }

    @Override
    public void setStringValue(@NotNull String value) {
      setBooleanValue(Boolean.valueOf(value));
    }

    @Nullable
    @Override
    public String getStringValue() {
      return myValue == null ? null : Boolean.toString(myValue);
    }

    @Override
    public void apply() {
      myModified = false;
    }

    @Override
    public void reset() {
      if (myModified) myValue = null;
      myModified = false;
    }

    @Override
    public boolean isModified() {
      return myModified;
    }
  }
}
