/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiType;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.CommonProcessors;
import com.intellij.util.Processor;
import com.intellij.util.Processors;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public abstract class AbstractSimpleSpringModel extends CachedLocalModel {

  private final SpringCachingProcessor<SpringModelSearchParameters.BeanClass> myBeanTypeCachingProcessor =
    new LocalBeansCachingProcessor<SpringModelSearchParameters.BeanClass>() {
      @Override
      protected void doProcessBeans(@NotNull SpringModelSearchParameters.BeanClass parameters,
                                    Processor<SpringBeanPointer> collectProcessor) {
        processLocalBeans(parameters, collectProcessor);
      }
    };

  private final SpringCachingProcessor<SpringModelSearchParameters.BeanName> myBeanNameCachingProcessor =
    new LocalBeansCachingProcessor<SpringModelSearchParameters.BeanName>() {
      @Override
      protected void doProcessBeans(@NotNull SpringModelSearchParameters.BeanName params, Processor<SpringBeanPointer> processor) {
        for (SpringBeanPointer beanPointer : getLocalBeans()) {
          if (matchesName(params, beanPointer)) {
            if (!processor.process(beanPointer)) return;
          }
        }
      }
    };

  private final Map<SpringQualifier, List<SpringBeanPointer>> myLocalBeansByQualifier =
    ConcurrentFactoryMap.createMap(key -> computeLocalBeansByQualifier(key));


  private abstract static class LocalBeansCachingProcessor<InParams extends SpringModelSearchParameters>
    extends SpringCachingProcessor<InParams> {
    @NotNull
    @Override
    protected Collection<SpringBeanPointer> findPointers(@NotNull InParams parameters) {
      final Collection<SpringBeanPointer> results = new SmartList<>();
      Processor<SpringBeanPointer> collectProcessor = Processors.cancelableCollectProcessor(results);

      doProcessBeans(parameters, collectProcessor);

      return results.isEmpty() ? Collections.emptyList() : results;
    }

    @Nullable
    @Override
    protected SpringBeanPointer findFirstPointer(@NotNull InParams parameters) {
      CommonProcessors.FindFirstProcessor<SpringBeanPointer> firstProcessor = new CommonProcessors.FindFirstProcessor<>();
      doProcessBeans(parameters, firstProcessor);

      return firstProcessor.getFoundValue();
    }

    protected abstract void doProcessBeans(@NotNull InParams parameters, Processor<SpringBeanPointer> collectProcessor);
  }


  @NotNull
  public Set<String> getProfiles() {
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    return Collections.emptySet();
  }

  @Override
  public boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params,
                                @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;
    if(!processLocalBeansByClass(params, processor)) return false;

    return super.processByClass(params, processor);
  }

  private void processLocalBeans(@NotNull SpringModelSearchParameters.BeanClass params,
                                 @NotNull Processor<SpringBeanPointer> processor) {
    final PsiType searchType = params.getSearchType();
    if (params.isEffectiveBeanTypes()) {
      for (SpringBeanPointer beanPointer : getLocalBeans()) {
        for (PsiType effectiveBeanType : beanPointer.getEffectiveBeanTypes()) {
          if (!processLocalBeanClass(processor, searchType, beanPointer, effectiveBeanType)) return;
        }
      }
    }
    else {
      for (SpringBeanPointer beanPointer : getLocalBeans()) {
        if (!processLocalBeanClass(processor, searchType, beanPointer, beanPointer.getSpringBean().getBeanType())) return;
      }
    }
  }

  private static boolean processLocalBeanClass(@NotNull Processor<? super SpringBeanPointer> processor,
                                               @NotNull PsiType searchType,
                                               SpringBeanPointer beanPointer,
                                               @Nullable PsiType beanType) {
    if (beanType != null && searchType.isAssignableFrom(beanType)) {
      return processor.process(beanPointer);
    }
    return true;
  }

  @Override
  public boolean processByName(@NotNull SpringModelSearchParameters.BeanName params,
                               @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;
    if (!processLocalBeansByName(params, processor)) return false;

    return super.processByName(params, processor);
  }

  @NotNull
  public Boolean processLocalBeansByClass(@NotNull SpringModelSearchParameters.BeanClass params,
                                          @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;
    return myBeanTypeCachingProcessor.process(params, processor, getActiveProfiles());
  }

  @NotNull
  public Boolean processLocalBeansByName(@NotNull SpringModelSearchParameters.BeanName params,
                                         @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;
    return myBeanNameCachingProcessor.process(params, processor, getActiveProfiles());
  }

  @Override
  public final boolean processAllBeans(@NotNull Processor<SpringBeanPointer> processor) {
    for (SpringBeanPointer pointer : getLocalBeans()) {
      if(!processor.process(pointer)) return false;
    }
    return super.processAllBeans(processor);
  }

  private static boolean matchesName(SpringModelSearchParameters.BeanName params, SpringBeanPointer pointer) {
    final String paramsBeanName = params.getBeanName();
    if (paramsBeanName.equals(pointer.getName())) return true;

    for (String aliasName : pointer.getAliases()) {
      if (paramsBeanName.equals(aliasName)) return true;
    }

    return false;
  }

  @NotNull
  protected Set<CommonSpringModel> getPackageScanModels(@NotNull Set<? extends LocalModel> localModels,
                                                        @NotNull Module module,
                                                        @NotNull Set<String> activeProfiles) {
    Set<CommonSpringModel> models = ContainerUtil.newLinkedHashSet();

    for (LocalModel model : localModels) {
      final List<? extends SpringBeansPackagesScan> scans = model.getPackagesScans();
      for (SpringBeansPackagesScan scan : scans) {
        models.add(new SpringComponentScanModel<>(module, scan, activeProfiles));
      }
    }

    return models;
  }

  @NotNull
  public Set<String> getAllBeanNames(@NotNull SpringBeanPointer beanPointer) {
    String beanName = beanPointer.getName();
    if (StringUtil.isEmptyOrSpaces(beanName)) return Collections.emptySet();

    Set<String> names = ContainerUtil.newHashSet(beanName);
    for (String aliasName : beanPointer.getAliases()) {
      if (!StringUtil.isEmptyOrSpaces(aliasName)) names.add(aliasName);
    }
    return names;
  }

  @NotNull
  public List<SpringBeanPointer> findQualified(@NotNull final SpringQualifier qualifier) {
    return myLocalBeansByQualifier.get(qualifier);
  }

  private List<SpringBeanPointer> computeLocalBeansByQualifier(final SpringQualifier springQualifier) {
    final List<SpringBeanPointer> beans = new SmartList<>();
    final Collection<SpringBeanPointer> pointers = getLocalBeans();
    for (SpringBeanPointer beanPointer : pointers) {
      if (!beanPointer.isValid()) continue;
      final CommonSpringBean bean = beanPointer.getSpringBean();
      for (SpringQualifier qualifier : bean.getSpringQualifiers()) {
        if (qualifier.compareQualifiers(springQualifier, getModule())) {
          beans.add(beanPointer);
        }
      }
    }
    return beans.isEmpty() ? Collections.emptyList() : beans;
  }
}
