/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.jam.model.common;

import com.intellij.jam.JamElement;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiManager;
import com.intellij.psi.xml.XmlTag;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public interface CommonModelElement {
  boolean isValid();

  XmlTag getXmlTag();

  PsiManager getPsiManager();

  @Nullable
  Module getModule();

  @Nullable
  PsiElement getIdentifyingPsiElement();

  @Nullable
  PsiFile getContainingFile();

  abstract class PsiBase implements CommonModelElement {
    @NotNull
    public abstract PsiElement getPsiElement();

    @Override
    public boolean isValid() {
      return getPsiElement().isValid();
    }

    @Override
    @Nullable
    public XmlTag getXmlTag() {
      return null;
    }

    @Override
    public PsiManager getPsiManager() {
      return getPsiElement().getManager();
    }

    @Override
    public Module getModule() {
      return isValid() ? ModuleUtilCore.findModuleForPsiElement(getPsiElement()) : null;
    }

    @Override
    public PsiElement getIdentifyingPsiElement() {
      return getPsiElement();
    }

    @Override
    public PsiFile getContainingFile() {
      return getPsiElement().getContainingFile();
    }
  }

  abstract class ModuleBase implements CommonModelElement {

    @Override
    @NotNull
    public abstract Module getModule();

    @Override
    public boolean isValid() {
      return !getModule().isDisposed();
    }

    @Override
    @Nullable
    public XmlTag getXmlTag() {
      return null;
    }

    @Override
    public PsiManager getPsiManager() {
      return PsiManager.getInstance(getModule().getProject());
    }

    @Override
    public PsiElement getIdentifyingPsiElement() {
      return null;
    }

    @Override
    public PsiFile getContainingFile() {
      return null;
    }
  }

  interface CommonModelNewJamElement extends CommonModelElement, JamElement {

  }
}
