package com.intellij.psi.css;

import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.css.descriptor.CssContextType;
import com.intellij.psi.css.descriptor.CssFunctionDescriptor;
import com.intellij.psi.css.descriptor.CssPseudoSelectorDescriptor;
import com.intellij.psi.css.descriptor.value.CssValueDescriptor;
import com.intellij.psi.css.descriptor.value.CssValueValidator;
import com.intellij.psi.css.descriptor.value.CssValueValidatorStub;
import com.intellij.psi.css.resolve.CssStyleReferenceStub;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.awt.*;
import java.util.Collection;
import java.util.Collections;

/**
 * @author Eugene.Kudelevsky
 */
public abstract class CssElementDescriptorProvider {
  public static final ExtensionPointName<CssElementDescriptorProvider> EP_NAME = ExtensionPointName.create("com.intellij.css.elementDescriptorProvider");
  private static final CssValueValidatorStub VALUE_VALIDATOR_STUB = new CssValueValidatorStub();

  public abstract boolean isMyContext(@Nullable PsiElement context);

  public boolean shouldAskOtherProviders(@Nullable PsiElement context) {
    return true;
  }
  
  /**
   * Find descriptor of css property. If several specification contains property with
   * given name, then descriptor from the latest specification will be returned.
   * 
   * @deprecated use {@link this#findPropertyDescriptors(String, PsiElement)}
   */
  @Deprecated
  @Nullable
  public CssPropertyDescriptor getPropertyDescriptor(@NotNull String propertyName, @Nullable PsiElement context) {
    return ContainerUtil.getFirstItem(findPropertyDescriptors(propertyName, context));
  }

  /**
   * Find pseudo selector descriptors with given name.
   */
  @NotNull
  public Collection<? extends CssPseudoSelectorDescriptor> findPseudoSelectorDescriptors(@NotNull String name, @Nullable PsiElement context) {
    return Collections.emptyList();
  }

  /**
   * Find named value descriptor with given name.
   */
  @NotNull
  public Collection<? extends CssValueDescriptor> getNamedValueDescriptors(@NotNull String name, @Nullable CssValueDescriptor parent) {
    return Collections.emptyList();
  }
  
  /**
   * Find function descriptors with given name.
   */
  @NotNull
  public Collection<? extends CssPropertyDescriptor> findPropertyDescriptors(@NotNull String propertyName, @Nullable PsiElement context) {
    return Collections.emptyList();
  }

  /**
   * Find function descriptors with given name.
   */
  @NotNull
  public Collection<? extends CssFunctionDescriptor> findFunctionDescriptors(@NotNull String functionName, @Nullable PsiElement context) {
    return Collections.emptyList();
  }
  
  /**
   * Find media feature descriptors with given name.
   */
  @NotNull
  public Collection<? extends CssMediaFeatureDescriptor> findMediaFeatureDescriptors(@NotNull String mediaFeatureName, @Nullable PsiElement context) {
    return Collections.emptyList();
  }

  /**
   * Returns true if given string is possible simple selector name in given context.
   */
  public boolean isPossibleSelector(@NotNull String selector, @NotNull PsiElement context) {
    return ArrayUtil.contains(selector, getSimpleSelectors(context));
  }

  /**
   * Retrieves list of pseudo-classes and pseudo-elements descriptors, that allowed in given context.
   * It's used in completion providers.
   * 
   * NOTE: the method could be invoke in DumbMode so you have to check DumbService before using any index
   *
   * @param context Given context. If {@code null} then all possible variants will be returned.
   */
  @NotNull
  public Collection<? extends CssPseudoSelectorDescriptor> getAllPseudoSelectorDescriptors(@Nullable PsiElement context) {
    return Collections.emptyList();
  }

  /**
   * Retrieves list of property descriptors, that allowed in given context.
   * Nullable context means that context is simple ruleset.
   * It's used in completion providers.
   * 
   * NOTE: the method could be invoke in DumbMode so you have to check DumbService before using any index
   */
  @NotNull
  public Collection<? extends CssPropertyDescriptor> getAllPropertyDescriptors(@Nullable PsiElement context) {
    return Collections.emptyList();
  }
  
  /**
   * Retrieves list of media features descriptors, that allowed in given context.
   * Nullable context means that context is simple ruleset.
   * It's used in completion providers.
   * 
   * NOTE: the method could be invoke in DumbMode so you have to check DumbService before using any index
   */
  @NotNull
  public Collection<? extends CssMediaFeatureDescriptor> getAllMediaFeatureDescriptors(@Nullable PsiElement context) {
    return Collections.emptyList();
  }


  /**
   * Retrieves list of simple selectors names (e.g. html-tags), that allowed in given context.
   */
  @NotNull
  public String[] getSimpleSelectors(@NotNull PsiElement context) {
    return ArrayUtil.EMPTY_STRING_ARRAY;
  }

  @NotNull
  public abstract PsiElement[] getDeclarationsForSimpleSelector(@NotNull CssSimpleSelector selector);

  /**
   * If true then for context of current provider will be enabled extra inspections,
   * e.g. CssNegativeValueInspection, W3C validator inspection etc.
   */
  public boolean providesClassicCss() {
    return true;
  }

  /**
   * Retrieves documentation element for selector.
   * Introduced for custom selectors in Flex.
   */
  @Nullable
  public PsiElement getDocumentationElementForSelector(@NotNull String selectorName, @Nullable PsiElement context) {
    return null;
  }

  /**
   * Retrieves documentation text for selector.
   * Introduced for custom selectors in Flex.
   */
  @Nullable
  public String generateDocForSelector(@NotNull String selectorName, @Nullable PsiElement context) {
    return null;
  }

  public PsiReference getStyleReference(PsiElement element, int start, int end, boolean caseSensitive) {
    return new CssStyleReferenceStub(element, TextRange.create(start, end));
  }

  @NotNull
  public CssValueValidator getValueValidator() {
    return VALUE_VALIDATOR_STUB;
  }

  @Nullable
  public Color getColorByValue(@NotNull String value) {
    return null;
  }

  public boolean isColorTerm(@NotNull CssTerm term) {
    return false;
  }

  /**
   * Introduced for custom properties in Flex.
   */
  @NotNull
  public LocalQuickFix[] getQuickFixesForUnknownProperty(@NotNull String propertyName, @NotNull PsiElement context, boolean isOnTheFly) {
    return LocalQuickFix.EMPTY_ARRAY;
  }

  /**
   * Introduced for custom selectors in Flex.
   */
  @NotNull
  public LocalQuickFix[] getQuickFixesForUnknownSimpleSelector(@NotNull String selectorName,
                                                               @NotNull PsiElement context,
                                                               boolean isOnTheFly) {
    return LocalQuickFix.EMPTY_ARRAY;
  }

  public boolean isColorTermsSupported() {
    return true;
  }

  public CssContextType getCssContextType(@Nullable PsiElement context) {
    return CssContextType.ANY;
  }
}
