// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model.autoconfigure.jam;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamService;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.module.Module;
import com.intellij.psi.*;
import com.intellij.semantic.SemKey;
import com.intellij.spring.model.jam.SpringSemContributorUtil;
import com.intellij.spring.spi.SpringSpiClassesListJamConverter;
import com.intellij.util.Function;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import static com.intellij.spring.boot.model.autoconfigure.SpringBootAutoconfigureClassesConstants.ENABLE_AUTO_CONFIGURATION;

public class EnableAutoConfiguration implements JamElement {

  private static final JamClassAttributeMeta.Collection EXCLUDE =
    new JamClassAttributeMeta.Collection("exclude");

  private static final JamStringAttributeMeta.Collection<PsiClass> EXCLUDE_NAME =
    JamAttributeMeta.collectionString("excludeName",
                                      new SpringSpiClassesListJamConverter(ENABLE_AUTO_CONFIGURATION));

  static final JamAnnotationArchetype EXCLUDE_ARCHETYPE =
    new JamAnnotationArchetype().addAttribute(EXCLUDE).addAttribute(EXCLUDE_NAME);

  private static final JamAnnotationMeta ANNOTATION_META =
    new JamAnnotationMeta(ENABLE_AUTO_CONFIGURATION,
                          EXCLUDE_ARCHETYPE);


  public static final SemKey<EnableAutoConfiguration> JAM_KEY =
    JamService.JAM_ELEMENT_KEY.subKey("EnableAutoConfigSemKey");

  public static final JamClassMeta<EnableAutoConfiguration> META =
    new JamClassMeta<>(null, EnableAutoConfiguration.class, JAM_KEY)
      .addAnnotation(ANNOTATION_META);

  public static final SemKey<JamMemberMeta<PsiClass, EnableAutoConfiguration>> META_KEY =
    JamService.MEMBER_META_KEY.subKey("EnableAutoConfigurationMeta");

  private final String myAnno;
  private final PsiAnchor myPsiClassAnchor;
  private final PsiElementRef<PsiAnnotation> myAnnotationRef;

  private static final Map<String, JamAnnotationMeta> ourJamAnnotationMetaMap =
    ConcurrentFactoryMap.createMap(key -> new JamAnnotationMeta(key, EXCLUDE_ARCHETYPE));

  @SuppressWarnings("unused")
  public EnableAutoConfiguration(@NotNull PsiClass psiClass) {
    this(ENABLE_AUTO_CONFIGURATION, psiClass);
  }

  public EnableAutoConfiguration(@Nullable String anno, @NotNull PsiClass psiClass) {
    myAnno = anno;
    myPsiClassAnchor = PsiAnchor.create(psiClass);
    JamAnnotationMeta annotationMeta = ourJamAnnotationMetaMap.get(anno);
    myAnnotationRef = annotationMeta.getAnnotationRef(psiClass);
  }

  public static Function<Module, Collection<String>> getAnnotations() {
    return SpringSemContributorUtil.getCustomMetaAnnotations(ENABLE_AUTO_CONFIGURATION);
  }

  @NotNull
  public String getAnnotationFqn() {
    return myAnno;
  }

  public boolean isValid() {
    return myPsiClassAnchor.retrieve() != null;
  }

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myAnnotationRef.getPsiElement();
  }

  @NotNull
  @JamPsiConnector
  public PsiClass getPsiElement() {
    final PsiElement psiElement = myPsiClassAnchor.retrieve();
    assert psiElement != null;
    return (PsiClass)psiElement;
  }

  public List<PsiClass> getExcludes() {
    List<PsiClass> allExcludes = new SmartList<>();

    ContainerUtil.addAll(allExcludes, ContainerUtil.mapNotNull(EXCLUDE.getJam(myAnnotationRef),
                                                               element -> element.getValue()));
    ContainerUtil.addAll(allExcludes, ContainerUtil.mapNotNull(EXCLUDE_NAME.getJam(myAnnotationRef),
                                                               element -> element.getValue()));
    return allExcludes;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    EnableAutoConfiguration that = (EnableAutoConfiguration)o;

    if (!Objects.equals(myAnno, that.myAnno)) return false;
    if (!myPsiClassAnchor.equals(that.myPsiClassAnchor)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = myAnno != null ? myAnno.hashCode() : 0;
    result = 31 * result + myPsiClassAnchor.hashCode();
    return result;
  }
}
