// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model

import com.intellij.openapi.module.Module
import com.intellij.openapi.util.text.StringUtil
import com.intellij.psi.PsiElement
import com.intellij.psi.PsiFile
import com.intellij.spring.boot.application.metadata.MetaConfigKeyReference
import com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKey
import com.intellij.util.containers.ContainerUtil
import java.lang.IllegalStateException

/**
 * @param configFile        Configuration file from this contributor.
 * @param configKey         Config key to search in file.
 * @param checkRelaxedNames Whether to check for relaxed names.
 * @param activeProfiles    Active profiles, {@code null} to check all files.
 * @since 2018.1
 */
data class ConfigurationValueSearchParams(val module: Module,
                                          val configFile: PsiFile,
                                          val checkRelaxedNames: Boolean = true,
                                          val activeProfiles: Set<String>?,
                                          val configKey: SpringBootApplicationMetaConfigKey) {

  companion object {
    private val PROCESS_ALL_VALUES = setOf("PROCESS_ALL_VALUES")
  }

  /**
   * Process all values, regardless of profiles/filenames/structure. Only useful to collect all occurrences across all files.
   */
  constructor(module: Module, configFile: PsiFile, configKey: SpringBootApplicationMetaConfigKey) :
    this(module, configFile, true, PROCESS_ALL_VALUES, configKey)

  fun isProcessAllProfiles(): Boolean {
    return activeProfiles == PROCESS_ALL_VALUES
  }

}


open class ConfigurationValueResult(val keyElement: PsiElement,
                                    val valueElement: PsiElement?, val valueText: String?,
                                    val params: ConfigurationValueSearchParams) {

  /**
   * Returns corresponding meta config key reference.
   *
   * @return Reference.
   * @since 2018.1
   */
  open fun getMetaConfigKeyReference(): MetaConfigKeyReference {
    val metaConfigKeyReference = ContainerUtil.findInstance(keyElement.references, MetaConfigKeyReference::class.java)
    if (metaConfigKeyReference == null) {
      val refText = StringUtil.join(keyElement.references, { param -> param!!.javaClass.name!! }, "|")
      throw IllegalStateException("${keyElement.javaClass} - '${keyElement.text}' - ${refText}}")
    }
    return metaConfigKeyReference
  }

}
