// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata;

import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.psi.PsiElement;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

public abstract class SpringBootApplicationMetaConfigKeyManager {

  public static SpringBootApplicationMetaConfigKeyManager getInstance() {
    return ServiceManager.getService(SpringBootApplicationMetaConfigKeyManager.class);
  }

  /**
   * @see #getAllMetaConfigKeys(Module)
   */
  public List<SpringBootApplicationMetaConfigKey> getAllMetaConfigKeys(@Nullable PsiElement psiElement) {
    if (psiElement == null) {
      return Collections.emptyList();
    }
    return getAllMetaConfigKeys(ModuleUtilCore.findModuleForPsiElement(psiElement));
  }

  /**
   * Returns all defined configuration keys.
   *
   * @param module Current module (search scope).
   * @return Configuration keys.
   */
  public abstract List<SpringBootApplicationMetaConfigKey> getAllMetaConfigKeys(@Nullable Module module);

  /**
   * Finds configuration key via given config key text.
   * <p/>
   * If no exact textual match was found, resolving will be performed via {@link ConfigKeyNameBinder}.
   * For exact search use {@link #findCanonicalApplicationMetaConfigKey(Module, String)}.
   *
   * @param module    Current module (search scope).
   * @param configKey Configuration key text.
   * @return Key or {@code null} if no such key was found.
   */
  @Nullable
  @Contract("_, null -> null; null, !null -> null")
  public abstract SpringBootApplicationMetaConfigKey findApplicationMetaConfigKey(@Nullable Module module,
                                                                                  @Nullable String configKey);

  @NotNull
  public abstract List<SpringBootApplicationMetaConfigKey> findAllApplicationMetaConfigKeys(PsiElement psiElement,
                                                                                            @Nullable String configKey);

  /**
   * Finds configuration key via exact canonical name.
   *
   * @param module    Current module (search scope).
   * @param configKey Configuration key in canonical notation.
   * @return Key or {@code null} if no such key was found.
   * @since 2018.3
   */
  @Nullable
  public abstract SpringBootApplicationMetaConfigKey findCanonicalApplicationMetaConfigKey(@Nullable Module module,
                                                                                           @Nullable String configKey);

  /**
   * Retrieves binder according to Spring Boot version for given module.
   *
   * @param module Current module.
   * @return Binder instance.
   */
  @NotNull
  public abstract ConfigKeyNameBinder getConfigKeyNameBinder(@NotNull Module module);


  public interface ConfigKeyNameBinder {

    boolean bindsTo(@NotNull SpringBootApplicationMetaConfigKey key, @NotNull String configKeyText);

    boolean matchesPrefix(@NotNull SpringBootApplicationMetaConfigKey key, @NotNull String prefixText);
  }
}
