// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKey;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

import static com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKey.AccessType;
import static com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKey.ItemHint;

/**
 * Gets references provided by SB 1.3 metadata and/or manual metadata.
 */
public abstract class SpringBootHintReferencesProvider {

  /**
   * Map/Index/List key with POJO property path.
   */
  public static final Condition<SpringBootApplicationMetaConfigKey> MAP_OR_INDEXED_WITHOUT_KEY_HINTS_CONDITION = configKey ->
    configKey.getKeyItemHint() == ItemHint.NONE &&
    configKey.isAccessType(AccessType.ENUM_MAP, AccessType.MAP, AccessType.INDEXED);

  public static SpringBootHintReferencesProvider getInstance() {
    return ServiceManager.getService(SpringBootHintReferencesProvider.class);
  }

  /**
   * Available in ProcessingContext.
   *
   * @since 2018.3
   */
  public static final Key<SpringBootApplicationMetaConfigKey> HINT_REFERENCES_CONFIG_KEY = Key.create("HINT_REFERENCES_CONFIG_KEY");

  /**
   * Gets additional key references (via configured hints).
   *
   * @param module        Current module.
   * @param configKey     Current config key.
   * @param keyPsiElement Current key element.
   * @param context       Context, {@link #HINT_REFERENCES_CONFIG_KEY} is made available.
   * @return References.
   */
  @NotNull
  public abstract PsiReference[] getKeyReferences(Module module,
                                                  SpringBootApplicationMetaConfigKey configKey,
                                                  PsiElement keyPsiElement,
                                                  ProcessingContext context);

  /**
   * Gets value references (hints, by type, key POJO path, explicit/fallback config).
   *
   * @param module          Current module.
   * @param configKey       Current config key.
   * @param keyPsiElement   (Optional) The corresponding key element to determine value provider by POJO property (if applicable).
   * @param valuePsiElement Current value element.
   * @param valueTextRanges Text range(s) for value text(s).
   * @param context         Context, {@link #HINT_REFERENCES_CONFIG_KEY} is made available.
   * @return References.
   */
  @NotNull
  public abstract PsiReference[] getValueReferences(Module module,
                                                    SpringBootApplicationMetaConfigKey configKey,
                                                    @Nullable PsiElement keyPsiElement,
                                                    PsiElement valuePsiElement,
                                                    List<TextRange> valueTextRanges,
                                                    ProcessingContext context);
}
