// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.lang.annotation.AnnotationHolder;
import com.intellij.lang.annotation.Annotator;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.editor.DefaultLanguageHighlighterColors;
import com.intellij.openapi.editor.colors.TextAttributesKey;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PsiPackageReference;
import com.intellij.spring.boot.application.config.hints.HintReferenceBase;
import com.intellij.spring.spi.SpringSpiClassesListJamConverter;
import com.intellij.ui.SimpleTextAttributes;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.SmartHashSet;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

/**
 * @since 2017.3
 */
public abstract class SpringBootConfigFileAnnotatorBase implements Annotator {

  protected abstract TextAttributesKey getPlaceholderTextAttributesKey();

  /**
   * Highlights references in value using language default colors.
   *
   * @param element Value element.
   * @param holder  Holder.
   */
  protected void annotateValue(@NotNull PsiElement element, @NotNull AnnotationHolder holder) {
    final int elementOffset = element.getNode().getStartOffset();
    final PsiReference[] references = element.getReferences();

    boolean highlightOnlyPlaceholders =
      ContainerUtil.findInstance(references, SpringBootPlaceholderReference.class) != null;

    Set<Integer> annotatedOffsets = new SmartHashSet<>();

    for (PsiReference reference : references) {
      TextAttributesKey key = null;
      if (highlightOnlyPlaceholders) {
        if (reference instanceof SpringBootPlaceholderReference) {
          key = getPlaceholderTextAttributesKey();
        }
      }
      else {
        if (reference instanceof JavaClassReference ||
            reference instanceof SpringSpiClassesListJamConverter.SpringSpiClassReference ||
            reference instanceof PsiPackageReference) {
          if (reference.resolve() != null) {   // FQN references are injected by default in .properties
            key = DefaultLanguageHighlighterColors.CLASS_REFERENCE;
          }
        }
        else if (reference instanceof HintReferenceBase) {
          key = ((HintReferenceBase)reference).getTextAttributesKey();
        }
      }

      if (key != null) {
        TextRange highlightTextRange = reference.getRangeInElement().shiftRight(elementOffset);
        if (!annotatedOffsets.add(highlightTextRange.getStartOffset())) continue;

        doAnnotate(holder, highlightTextRange, key);
      }
    }
  }

  private static final boolean DEBUG_MODE = ApplicationManager.getApplication().isUnitTestMode();

  protected static void doAnnotate(AnnotationHolder holder, TextRange range, TextAttributesKey key) {
    if (range.isEmpty()) return;

    String message = DEBUG_MODE ? key.getExternalName() : null;
    holder.createInfoAnnotation(range, message).setTextAttributes(key);
  }

  protected static void doAnnotateEnforced(AnnotationHolder holder, TextRange range, SimpleTextAttributes key, String debugMessage) {
    if (range.isEmpty()) return;

    String message = DEBUG_MODE ? debugMessage : null;
    holder.createInfoAnnotation(range, message).setEnforcedTextAttributes(key.toTextAttributes());
  }
}

