use std::{
    fmt::{Debug, Display},
    ops::Deref,
    str::FromStr,
    sync::Arc,
};

use serde::{Deserialize, Serialize};
use strum_macros::{AsRefStr, EnumIter, EnumString};

use crate::tui::themes::{
    aardvark_blue::AardvarkBlue, abernathy::Abernathy, adventure::Adventure,
    adventure_time::AdventureTime, adwaita::Adwaita, adwaita_dark::AdwaitaDark,
    afterglow::Afterglow, alabaster::Alabaster, alien_blood::AlienBlood, andromeda::Andromeda,
    apple_classic::AppleClassic, apple_system_colors::AppleSystemColors,
    apple_system_colors_light::AppleSystemColorsLight, arcoiris::Arcoiris, ardoise::Ardoise,
    argonaut::Argonaut, arthur::Arthur, atelier_sulphurpool::AtelierSulphurpool, atom::Atom,
    atom_one_dark::AtomOneDark, atom_one_light::AtomOneLight, aura::Aura, aurora::Aurora, ayu::Ayu,
    ayu_light::AyuLight, ayu_mirage::AyuMirage, banana_blueberry::BananaBlueberry, batman::Batman,
    belafonte_day::BelafonteDay, belafonte_night::BelafonteNight,
    birds_of_paradise::BirdsOfParadise, black_metal::BlackMetal,
    black_metal_bathory_::BlackMetalBathory, black_metal_burzum_::BlackMetalBurzum,
    black_metal_dark_funeral_::BlackMetalDarkFuneral, black_metal_gorgoroth_::BlackMetalGorgoroth,
    black_metal_immortal_::BlackMetalImmortal, black_metal_khold_::BlackMetalKhold,
    black_metal_marduk_::BlackMetalMarduk, black_metal_mayhem_::BlackMetalMayhem,
    black_metal_nile_::BlackMetalNile, black_metal_venom_::BlackMetalVenom, blazer::Blazer,
    blue_berry_pie::BlueBerryPie, blue_dolphin::BlueDolphin, blue_matrix::BlueMatrix,
    bluloco_dark::BlulocoDark, bluloco_light::BlulocoLight, borland::Borland, breadog::Breadog,
    breeze::Breeze, bright_lights::BrightLights, broadcast::Broadcast, brogrammer::Brogrammer,
    builtin_dark::BuiltinDark, builtin_light::BuiltinLight, builtin_pastel_dark::BuiltinPastelDark,
    builtin_solarized_dark::BuiltinSolarizedDark, builtin_solarized_light::BuiltinSolarizedLight,
    builtin_tango_dark::BuiltinTangoDark, builtin_tango_light::BuiltinTangoLight,
    calamity::Calamity, catppuccin::Catppuccin, catppuccin_frappe::CatppuccinFrappe,
    catppuccin_latte::CatppuccinLatte, catppuccin_macchiato::CatppuccinMacchiato,
    catppuccin_mocha::CatppuccinMocha, cga::Cga, chakra::Chakra, chalk::Chalk,
    chalkboard::Chalkboard, challenger_deep::ChallengerDeep, chester::Chester, ciapre::Ciapre,
    citruszest::Citruszest, clrs::Clrs, cobalt_neon::CobaltNeon, cobalt_next::CobaltNext,
    cobalt_next_dark::CobaltNextDark, cobalt_next_minimal::CobaltNextMinimal,
    coffee_theme::CoffeeTheme, crayon_pony_fish::CrayonPonyFish, cursor_dark::CursorDark,
    custom::Custom, cutie_pro::CutiePro, cyberdyne::Cyberdyne, cyberpunk::Cyberpunk,
    cyberpunk_scarlet_protocol::CyberpunkScarletProtocol, dark_modern::DarkModern,
    dark_pastel::DarkPastel, darkermatrix::Darkermatrix, darkmatrix::Darkmatrix,
    darkside::Darkside, dayfox::Dayfox, deep::Deep, desert::Desert, detuned::Detuned,
    dimidium::Dimidium, dimmed_monokai::DimmedMonokai, django::Django,
    django_reborn_again::DjangoRebornAgain, django_smooth::DjangoSmooth, doom_one::DoomOne,
    doom_peacock::DoomPeacock, dot_gov::DotGov, dracula::Dracula, duckbones::Duckbones,
    duotone_dark::DuotoneDark, earthsong::Earthsong, electron_highlighter::ElectronHighlighter,
    elegant::Elegant, elemental::Elemental, elementary::Elementary, embark::Embark,
    embers_dark::EmbersDark, encom::Encom, espresso::Espresso, espresso_libre::EspressoLibre,
    everblush::Everblush, everforest_dark_hard::EverforestDarkHard,
    everforest_light_med::EverforestLightMed, fahrenheit::Fahrenheit, fairyfloss::Fairyfloss,
    farmhouse_dark::FarmhouseDark, farmhouse_light::FarmhouseLight, fideloper::Fideloper,
    firefly_traditional::FireflyTraditional, firefox_dev::FirefoxDev, firewatch::Firewatch,
    fish_tank::FishTank, flat::Flat, flatland::Flatland, flexoki_dark::FlexokiDark,
    flexoki_light::FlexokiLight, floraverse::Floraverse, forest_blue::ForestBlue, framer::Framer,
    front_end_delight::FrontEndDelight, fun_forrest::FunForrest, galaxy::Galaxy, galizur::Galizur,
    ghostty_default_style_dark::GhosttyDefaultStyleDark, git_hub::Github, git_hub_dark::GithubDark,
    git_hub_dark_colorblind::GithubDarkColorblind, git_hub_dark_default::GithubDarkDefault,
    git_hub_dark_dimmed::GithubDarkDimmed, git_hub_dark_high_contrast::GithubDarkHighContrast,
    git_hub_light_colorblind::GithubLightColorblind, git_hub_light_default::GithubLightDefault,
    git_hub_light_high_contrast::GithubLightHighContrast, git_lab_dark::GitlabDark,
    git_lab_dark_grey::GitlabDarkGrey, git_lab_light::GitlabLight, glacier::Glacier, grape::Grape,
    grass::Grass, grey_green::GreyGreen, gruber_darker::GruberDarker, gruvbox_dark::GruvboxDark,
    gruvbox_dark_hard::GruvboxDarkHard, gruvbox_light::GruvboxLight,
    gruvbox_light_hard::GruvboxLightHard, gruvbox_material::GruvboxMaterial, guezwhoz::Guezwhoz,
    hacktober::Hacktober, hardcore::Hardcore, harper::Harper, havn_daggry::HavnDaggry,
    havn_skumring::HavnSkumring, heeler::Heeler, highway::Highway, hipster_green::HipsterGreen,
    hivacruz::Hivacruz, homebrew::Homebrew, hopscotch::Hopscotch, horizon::Horizon,
    horizon_bright::HorizonBright, hurtado::Hurtado, hybrid::Hybrid, ic_green_ppl::IcGreenPpl,
    ic_orange_ppl::IcOrangePpl, iceberg_dark::IcebergDark, iceberg_light::IcebergLight, idea::Idea,
    idle_toes::IdleToes, ir_black::IrBlack, irix_console::IrixConsole, irix_terminal::IrixTerminal,
    jackie_brown::JackieBrown, japanesque::Japanesque, jellybeans::Jellybeans,
    jet_brains_darcula::JetbrainsDarcula, jubi::Jubi, kanagawa_dragon::KanagawaDragon,
    kanagawa_wave::KanagawaWave, kanagawabones::Kanagawabones, kibble::Kibble,
    kitty_default::KittyDefault, kitty_low_contrast::KittyLowContrast, kolorit::Kolorit,
    konsolas::Konsolas, kurokula::Kurokula, lab_fox::LabFox, laser::Laser,
    later_this_evening::LaterThisEvening, lavandula::Lavandula, light_owl::LightOwl,
    liquid_carbon::LiquidCarbon, liquid_carbon_transparent::LiquidCarbonTransparent,
    liquid_carbon_transparent_inverse::LiquidCarbonTransparentInverse, lovelace::Lovelace,
    man_page::ManPage, mariana::Mariana, material::Material, material_dark::MaterialDark,
    material_darker::MaterialDarker, material_design_colors::MaterialDesignColors,
    material_ocean::MaterialOcean, mathias::Mathias, matrix::Matrix, medallion::Medallion,
    melange_dark::MelangeDark, melange_light::MelangeLight, mellifluous::Mellifluous,
    mellow::Mellow, miasma::Miasma, midnight_in_mojave::MidnightInMojave, mirage::Mirage,
    misterioso::Misterioso, molokai::Molokai, mona_lisa::MonaLisa, monokai::Monokai,
    monokai_classic::MonokaiClassic, monokai_pro::MonokaiPro, monokai_pro_light::MonokaiProLight,
    monokai_pro_light_sun::MonokaiProLightSun, monokai_pro_machine::MonokaiProMachine,
    monokai_pro_octagon::MonokaiProOctagon, monokai_pro_ristretto::MonokaiProRistretto,
    monokai_pro_spectrum::MonokaiProSpectrum, monokai_remastered::MonokaiRemastered,
    monokai_soda::MonokaiSoda, monokai_vivid::MonokaiVivid, moonfly::Moonfly,
    neobones_dark::NeobonesDark, neobones_light::NeobonesLight, neon::Neon, neopolitan::Neopolitan,
    neutron::Neutron, night_owl::NightOwl, night_owlish_light::NightOwlishLight,
    nightfox::Nightfox, niji::Niji, nocturnal_winter::NocturnalWinter, nord::Nord,
    nord_light::NordLight, nord_wave::NordWave, novel::Novel, nvim_dark::NvimDark,
    nvim_light::NvimLight, obsidian::Obsidian, ocean::Ocean, oceanic_material::OceanicMaterial,
    oceanic_next::OceanicNext, ollie::Ollie, one_double_dark::OneDoubleDark,
    one_double_light::OneDoubleLight, one_half_dark::OneHalfDark, one_half_light::OneHalfLight,
    operator_mono_dark::OperatorMonoDark, overnight_slumber::OvernightSlumber,
    oxocarbon::Oxocarbon, pale_night_hc::PaleNightHc, pandora::Pandora, paraiso_dark::ParaisoDark,
    paul_millr::PaulMillr, pencil_dark::PencilDark, pencil_light::PencilLight,
    peppermint::Peppermint, phala_green_dark::PhalaGreenDark, piatto_light::PiattoLight,
    pnevma::Pnevma, popping_and_locking::PoppingAndLocking, powershell::Powershell,
    primary::Primary, pro::Pro, pro_light::ProLight, purple_rain::PurpleRain,
    purplepeter::Purplepeter, rapture::Rapture, raycast_dark::RaycastDark,
    raycast_light::RaycastLight, rebecca::Rebecca, red_alert::RedAlert, red_planet::RedPlanet,
    red_sands::RedSands, relaxed::Relaxed, retro::Retro, retro_legends::RetroLegends,
    rippedcasts::Rippedcasts, rose_pine::RosePine, rose_pine_dawn::RosePineDawn,
    rose_pine_moon::RosePineMoon, royal::Royal, ryuuko::Ryuuko, sakura::Sakura,
    scarlet_protocol::ScarletProtocol, sea_shells::SeaShells, seafoam_pastel::SeafoamPastel,
    selenized_dark::SelenizedDark, selenized_light::SelenizedLight,
    seoulbones_dark::SeoulbonesDark, seoulbones_light::SeoulbonesLight, seti::Seti,
    shades_of_purple::ShadesOfPurple, shaman::Shaman, slate::Slate, sleepy_hollow::SleepyHollow,
    smyck::Smyck, snazzy::Snazzy, snazzy_soft::SnazzySoft, soft_server::SoftServer,
    solarized_darcula::SolarizedDarcula,
    solarized_dark_higher_contrast::SolarizedDarkHigherContrast,
    solarized_dark_patched::SolarizedDarkPatched, solarized_osaka_night::SolarizedOsakaNight,
    sonokai::Sonokai, spacedust::Spacedust, spacegray::Spacegray,
    spacegray_bright::SpacegrayBright, spacegray_eighties::SpacegrayEighties,
    spacegray_eighties_dull::SpacegrayEightiesDull, spiderman::Spiderman, spring::Spring,
    square::Square, squirrelsong_dark::SquirrelsongDark, srcery::Srcery, starlight::Starlight,
    styler::Styler, sublette::Sublette, subliminal::Subliminal, sugarplum::Sugarplum,
    sundried::Sundried, symfonic::Symfonic, synthwave::Synthwave, synthwave_alpha::SynthwaveAlpha,
    synthwave_everything::SynthwaveEverything, tango_adapted::TangoAdapted,
    tango_half_adapted::TangoHalfAdapted, tearout::Tearout, teerb::Teerb, terafox::Terafox,
    terminal::Terminal, terminal_basic::TerminalBasic, terminal_basic_dark::TerminalBasicDark,
    thayer_bright::ThayerBright, the_hulk::TheHulk, tinacious_design_dark::TinaciousDesignDark,
    tinacious_design_light::TinaciousDesignLight, tokyo_night::TokyoNight,
    tokyo_night_day::TokyoNightDay, tokyo_night_moon::TokyoNightMoon,
    tokyo_night_night::TokyoNightNight, tokyo_night_storm::TokyoNightStorm, tomorrow::Tomorrow,
    tomorrow_night::TomorrowNight, tomorrow_night_blue::TomorrowNightBlue,
    tomorrow_night_bright::TomorrowNightBright, tomorrow_night_burns::TomorrowNightBurns,
    tomorrow_night_eighties::TomorrowNightEighties, toy_chest::ToyChest, treehouse::Treehouse,
    twilight::Twilight, ubuntu::Ubuntu, ultra_dark::UltraDark, ultra_violent::UltraViolent,
    under_the_sea::UnderTheSea, unikitty::Unikitty, urple::Urple, vague::Vague, vaughn::Vaughn,
    vercel::Vercel, vesper::Vesper, vibrant_ink::VibrantInk, vimbones::Vimbones,
    violet_dark::VioletDark, violet_light::VioletLight, violite::Violite, warm_neon::WarmNeon,
    wez::Wez, whimsy::Whimsy, wild_cherry::WildCherry, wilmersdorf::Wilmersdorf, wombat::Wombat,
    wryan::Wryan, xcode_dark::XcodeDark, xcode_dark_hc::XcodeDarkHc, xcode_light::XcodeLight,
    xcode_light_hc::XcodeLightHc, xcode_wwdc::XcodeWwdc, zenbones::Zenbones,
    zenbones_dark::ZenbonesDark, zenbones_light::ZenbonesLight, zenburn::Zenburn,
    zenburned::Zenburned, zenwritten_dark::ZenwrittenDark, zenwritten_light::ZenwrittenLight,
};

#[derive(Debug, Clone)]
pub struct LoadedTheme {
    app_theme: Theme,
    styler: Arc<dyn Styler + Send + Sync>,
}

impl LoadedTheme {
    pub fn app_theme(&self) -> &Theme {
        &self.app_theme
    }

    fn new<S: Styler + Send + Sync + 'static>(app_theme: Theme, theme: S) -> Self {
        LoadedTheme {
            app_theme,
            styler: Arc::new(theme),
        }
    }
}

impl Default for LoadedTheme {
    fn default() -> Self {
        Self::new(Theme::Monokai, Monokai)
    }
}

impl From<Theme> for LoadedTheme {
    fn from(value: Theme) -> Self {
        match value {
            Theme::MonokaiProLightSun => LoadedTheme::new(value, MonokaiProLightSun),
            Theme::BlackMetalImmortal => LoadedTheme::new(value, BlackMetalImmortal),
            Theme::Starlight => LoadedTheme::new(value, Starlight),
            Theme::AyuLight => LoadedTheme::new(value, AyuLight),
            Theme::BuiltinTangoLight => LoadedTheme::new(value, BuiltinTangoLight),
            Theme::NvimLight => LoadedTheme::new(value, NvimLight),
            Theme::Twilight => LoadedTheme::new(value, Twilight),
            Theme::Hybrid => LoadedTheme::new(value, Hybrid),
            Theme::Lovelace => LoadedTheme::new(value, Lovelace),
            Theme::Spring => LoadedTheme::new(value, Spring),
            Theme::ZenwrittenDark => LoadedTheme::new(value, ZenwrittenDark),
            Theme::Neopolitan => LoadedTheme::new(value, Neopolitan),
            Theme::Ciapre => LoadedTheme::new(value, Ciapre),
            Theme::Sublette => LoadedTheme::new(value, Sublette),
            Theme::ManPage => LoadedTheme::new(value, ManPage),
            Theme::Purplepeter => LoadedTheme::new(value, Purplepeter),
            Theme::Breadog => LoadedTheme::new(value, Breadog),
            Theme::LightOwl => LoadedTheme::new(value, LightOwl),
            Theme::Galizur => LoadedTheme::new(value, Galizur),
            Theme::RosePine => LoadedTheme::new(value, RosePine),
            Theme::Molokai => LoadedTheme::new(value, Molokai),
            Theme::Shaman => LoadedTheme::new(value, Shaman),
            Theme::Darkermatrix => LoadedTheme::new(value, Darkermatrix),
            Theme::JackieBrown => LoadedTheme::new(value, JackieBrown),
            Theme::Espresso => LoadedTheme::new(value, Espresso),
            Theme::Peppermint => LoadedTheme::new(value, Peppermint),
            Theme::UltraViolent => LoadedTheme::new(value, UltraViolent),
            Theme::Primary => LoadedTheme::new(value, Primary),
            Theme::NvimDark => LoadedTheme::new(value, NvimDark),
            Theme::RetroLegends => LoadedTheme::new(value, RetroLegends),
            Theme::IcebergDark => LoadedTheme::new(value, IcebergDark),
            Theme::MonokaiRemastered => LoadedTheme::new(value, MonokaiRemastered),
            Theme::DotGov => LoadedTheme::new(value, DotGov),
            Theme::OceanicNext => LoadedTheme::new(value, OceanicNext),
            Theme::Material => LoadedTheme::new(value, Material),
            Theme::BlackMetalMarduk => LoadedTheme::new(value, BlackMetalMarduk),
            Theme::GreyGreen => LoadedTheme::new(value, GreyGreen),
            Theme::Broadcast => LoadedTheme::new(value, Broadcast),
            Theme::Matrix => LoadedTheme::new(value, Matrix),
            Theme::OneDoubleLight => LoadedTheme::new(value, OneDoubleLight),
            Theme::ElectronHighlighter => LoadedTheme::new(value, ElectronHighlighter),
            Theme::Ubuntu => LoadedTheme::new(value, Ubuntu),
            Theme::Clrs => LoadedTheme::new(value, Clrs),
            Theme::TomorrowNightBlue => LoadedTheme::new(value, TomorrowNightBlue),
            Theme::Django => LoadedTheme::new(value, Django),
            Theme::Brogrammer => LoadedTheme::new(value, Brogrammer),
            Theme::Mellow => LoadedTheme::new(value, Mellow),
            Theme::JetbrainsDarcula => LoadedTheme::new(value, JetbrainsDarcula),
            Theme::MonokaiVivid => LoadedTheme::new(value, MonokaiVivid),
            Theme::Fideloper => LoadedTheme::new(value, Fideloper),
            Theme::FarmhouseDark => LoadedTheme::new(value, FarmhouseDark),
            Theme::Royal => LoadedTheme::new(value, Royal),
            Theme::GruvboxLight => LoadedTheme::new(value, GruvboxLight),
            Theme::GruvboxDark => LoadedTheme::new(value, GruvboxDark),
            Theme::Smyck => LoadedTheme::new(value, Smyck),
            Theme::Pnevma => LoadedTheme::new(value, Pnevma),
            Theme::BlackMetalNile => LoadedTheme::new(value, BlackMetalNile),
            Theme::Elementary => LoadedTheme::new(value, Elementary),
            Theme::SquirrelsongDark => LoadedTheme::new(value, SquirrelsongDark),
            Theme::CatppuccinMocha => LoadedTheme::new(value, CatppuccinMocha),
            Theme::BlackMetalBathory => LoadedTheme::new(value, BlackMetalBathory),
            Theme::LiquidCarbonTransparentInverse => {
                LoadedTheme::new(value, LiquidCarbonTransparentInverse)
            }
            Theme::FunForrest => LoadedTheme::new(value, FunForrest),
            Theme::Arcoiris => LoadedTheme::new(value, Arcoiris),
            Theme::Desert => LoadedTheme::new(value, Desert),
            Theme::DoomOne => LoadedTheme::new(value, DoomOne),
            Theme::SpacegrayEighties => LoadedTheme::new(value, SpacegrayEighties),
            Theme::WarmNeon => LoadedTheme::new(value, WarmNeon),
            Theme::Dimidium => LoadedTheme::new(value, Dimidium),
            Theme::GitlabDarkGrey => LoadedTheme::new(value, GitlabDarkGrey),
            Theme::Cyberpunk => LoadedTheme::new(value, Cyberpunk),
            Theme::Tearout => LoadedTheme::new(value, Tearout),
            Theme::Aura => LoadedTheme::new(value, Aura),
            Theme::Grape => LoadedTheme::new(value, Grape),
            Theme::SelenizedLight => LoadedTheme::new(value, SelenizedLight),
            Theme::Blazer => LoadedTheme::new(value, Blazer),
            Theme::KanagawaDragon => LoadedTheme::new(value, KanagawaDragon),
            Theme::Hurtado => LoadedTheme::new(value, Hurtado),
            Theme::MaterialOcean => LoadedTheme::new(value, MaterialOcean),
            Theme::XcodeDarkHc => LoadedTheme::new(value, XcodeDarkHc),
            Theme::Citruszest => LoadedTheme::new(value, Citruszest),
            Theme::CrayonPonyFish => LoadedTheme::new(value, CrayonPonyFish),
            Theme::Relaxed => LoadedTheme::new(value, Relaxed),
            Theme::Rapture => LoadedTheme::new(value, Rapture),
            Theme::BlackMetalGorgoroth => LoadedTheme::new(value, BlackMetalGorgoroth),
            Theme::Cyberdyne => LoadedTheme::new(value, Cyberdyne),
            Theme::GitlabLight => LoadedTheme::new(value, GitlabLight),
            Theme::IcebergLight => LoadedTheme::new(value, IcebergLight),
            Theme::Floraverse => LoadedTheme::new(value, Floraverse),
            Theme::AtomOneDark => LoadedTheme::new(value, AtomOneDark),
            Theme::ShadesOfPurple => LoadedTheme::new(value, ShadesOfPurple),
            Theme::Wez => LoadedTheme::new(value, Wez),
            Theme::AyuMirage => LoadedTheme::new(value, AyuMirage),
            Theme::Obsidian => LoadedTheme::new(value, Obsidian),
            Theme::Kolorit => LoadedTheme::new(value, Kolorit),
            Theme::Hivacruz => LoadedTheme::new(value, Hivacruz),
            Theme::AdwaitaDark => LoadedTheme::new(value, AdwaitaDark),
            Theme::PencilLight => LoadedTheme::new(value, PencilLight),
            Theme::BlulocoDark => LoadedTheme::new(value, BlulocoDark),
            Theme::BuiltinTangoDark => LoadedTheme::new(value, BuiltinTangoDark),
            Theme::BelafonteDay => LoadedTheme::new(value, BelafonteDay),
            Theme::BlueDolphin => LoadedTheme::new(value, BlueDolphin),
            Theme::MonokaiPro => LoadedTheme::new(value, MonokaiPro),
            Theme::TheHulk => LoadedTheme::new(value, TheHulk),
            Theme::Synthwave => LoadedTheme::new(value, Synthwave),
            Theme::Konsolas => LoadedTheme::new(value, Konsolas),
            Theme::Dayfox => LoadedTheme::new(value, Dayfox),
            Theme::SolarizedDarcula => LoadedTheme::new(value, SolarizedDarcula),
            Theme::NordWave => LoadedTheme::new(value, NordWave),
            Theme::Urple => LoadedTheme::new(value, Urple),
            Theme::XcodeLightHc => LoadedTheme::new(value, XcodeLightHc),
            Theme::TokyoNightNight => LoadedTheme::new(value, TokyoNightNight),
            Theme::SynthwaveAlpha => LoadedTheme::new(value, SynthwaveAlpha),
            Theme::EspressoLibre => LoadedTheme::new(value, EspressoLibre),
            Theme::GithubDarkDefault => LoadedTheme::new(value, GithubDarkDefault),
            Theme::Breeze => LoadedTheme::new(value, Breeze),
            Theme::Fahrenheit => LoadedTheme::new(value, Fahrenheit),
            Theme::CatppuccinLatte => LoadedTheme::new(value, CatppuccinLatte),
            Theme::Miasma => LoadedTheme::new(value, Miasma),
            Theme::EverforestDarkHard => LoadedTheme::new(value, EverforestDarkHard),
            Theme::Seti => LoadedTheme::new(value, Seti),
            Theme::BlackMetalDarkFuneral => LoadedTheme::new(value, BlackMetalDarkFuneral),
            Theme::Zenburned => LoadedTheme::new(value, Zenburned),
            Theme::CobaltNextDark => LoadedTheme::new(value, CobaltNextDark),
            Theme::DjangoRebornAgain => LoadedTheme::new(value, DjangoRebornAgain),
            Theme::Treehouse => LoadedTheme::new(value, Treehouse),
            Theme::Mariana => LoadedTheme::new(value, Mariana),
            Theme::Retro => LoadedTheme::new(value, Retro),
            Theme::CursorDark => LoadedTheme::new(value, CursorDark),
            Theme::Hardcore => LoadedTheme::new(value, Hardcore),
            Theme::Neon => LoadedTheme::new(value, Neon),
            Theme::GitlabDark => LoadedTheme::new(value, GitlabDark),
            Theme::UnderTheSea => LoadedTheme::new(value, UnderTheSea),
            Theme::VioletLight => LoadedTheme::new(value, VioletLight),
            Theme::Ayu => LoadedTheme::new(value, Ayu),
            Theme::HavnSkumring => LoadedTheme::new(value, HavnSkumring),
            Theme::Afterglow => LoadedTheme::new(value, Afterglow),
            Theme::DoomPeacock => LoadedTheme::new(value, DoomPeacock),
            Theme::AppleClassic => LoadedTheme::new(value, AppleClassic),
            Theme::VibrantInk => LoadedTheme::new(value, VibrantInk),
            Theme::Arthur => LoadedTheme::new(value, Arthur),
            Theme::TinaciousDesignLight => LoadedTheme::new(value, TinaciousDesignLight),
            Theme::WildCherry => LoadedTheme::new(value, WildCherry),
            Theme::HipsterGreen => LoadedTheme::new(value, HipsterGreen),
            Theme::Mathias => LoadedTheme::new(value, Mathias),
            Theme::NeobonesDark => LoadedTheme::new(value, NeobonesDark),
            Theme::SpacegrayEightiesDull => LoadedTheme::new(value, SpacegrayEightiesDull),
            Theme::NightOwlishLight => LoadedTheme::new(value, NightOwlishLight),
            Theme::Chester => LoadedTheme::new(value, Chester),
            Theme::Pandora => LoadedTheme::new(value, Pandora),
            Theme::GhosttyDefaultStyleDark => LoadedTheme::new(value, GhosttyDefaultStyleDark),
            Theme::OneDoubleDark => LoadedTheme::new(value, OneDoubleDark),
            Theme::ZenbonesLight => LoadedTheme::new(value, ZenbonesLight),
            Theme::Niji => LoadedTheme::new(value, Niji),
            Theme::ProLight => LoadedTheme::new(value, ProLight),
            Theme::SpacegrayBright => LoadedTheme::new(value, SpacegrayBright),
            Theme::SeaShells => LoadedTheme::new(value, SeaShells),
            Theme::Vague => LoadedTheme::new(value, Vague),
            Theme::TokyoNightMoon => LoadedTheme::new(value, TokyoNightMoon),
            Theme::XcodeLight => LoadedTheme::new(value, XcodeLight),
            Theme::AardvarkBlue => LoadedTheme::new(value, AardvarkBlue),
            Theme::ZenwrittenLight => LoadedTheme::new(value, ZenwrittenLight),
            Theme::CobaltNeon => LoadedTheme::new(value, CobaltNeon),
            Theme::DuotoneDark => LoadedTheme::new(value, DuotoneDark),
            Theme::Cga => LoadedTheme::new(value, Cga),
            Theme::Idea => LoadedTheme::new(value, Idea),
            Theme::SynthwaveEverything => LoadedTheme::new(value, SynthwaveEverything),
            Theme::BlackMetal => LoadedTheme::new(value, BlackMetal),
            Theme::HavnDaggry => LoadedTheme::new(value, HavnDaggry),
            Theme::Github => LoadedTheme::new(value, Github),
            Theme::SeafoamPastel => LoadedTheme::new(value, SeafoamPastel),
            Theme::Chalkboard => LoadedTheme::new(value, Chalkboard),
            Theme::Elemental => LoadedTheme::new(value, Elemental),
            Theme::Novel => LoadedTheme::new(value, Novel),
            Theme::Snazzy => LoadedTheme::new(value, Snazzy),
            Theme::MonokaiClassic => LoadedTheme::new(value, MonokaiClassic),
            Theme::Zenburn => LoadedTheme::new(value, Zenburn),
            Theme::GithubLightDefault => LoadedTheme::new(value, GithubLightDefault),
            Theme::Laser => LoadedTheme::new(value, Laser),
            Theme::BirdsOfParadise => LoadedTheme::new(value, BirdsOfParadise),
            Theme::Mellifluous => LoadedTheme::new(value, Mellifluous),
            Theme::Fairyfloss => LoadedTheme::new(value, Fairyfloss),
            Theme::NightOwl => LoadedTheme::new(value, NightOwl),
            Theme::OneHalfDark => LoadedTheme::new(value, OneHalfDark),
            Theme::FireflyTraditional => LoadedTheme::new(value, FireflyTraditional),
            Theme::Terafox => LoadedTheme::new(value, Terafox),
            Theme::RosePineDawn => LoadedTheme::new(value, RosePineDawn),
            Theme::Sundried => LoadedTheme::new(value, Sundried),
            Theme::TokyoNightStorm => LoadedTheme::new(value, TokyoNightStorm),
            Theme::Nord => LoadedTheme::new(value, Nord),
            Theme::PiattoLight => LoadedTheme::new(value, PiattoLight),
            Theme::Abernathy => LoadedTheme::new(value, Abernathy),
            Theme::TokyoNightDay => LoadedTheme::new(value, TokyoNightDay),
            Theme::Powershell => LoadedTheme::new(value, Powershell),
            Theme::Jellybeans => LoadedTheme::new(value, Jellybeans),
            Theme::BlueMatrix => LoadedTheme::new(value, BlueMatrix),
            Theme::Sonokai => LoadedTheme::new(value, Sonokai),
            Theme::TomorrowNightBurns => LoadedTheme::new(value, TomorrowNightBurns),
            Theme::Sakura => LoadedTheme::new(value, Sakura),
            Theme::RaycastDark => LoadedTheme::new(value, RaycastDark),
            Theme::KanagawaWave => LoadedTheme::new(value, KanagawaWave),
            Theme::CutiePro => LoadedTheme::new(value, CutiePro),
            Theme::Horizon => LoadedTheme::new(value, Horizon),
            Theme::Spiderman => LoadedTheme::new(value, Spiderman),
            Theme::BlackMetalMayhem => LoadedTheme::new(value, BlackMetalMayhem),
            Theme::KittyDefault => LoadedTheme::new(value, KittyDefault),
            Theme::IdleToes => LoadedTheme::new(value, IdleToes),
            Theme::TerminalBasicDark => LoadedTheme::new(value, TerminalBasicDark),
            Theme::Framer => LoadedTheme::new(value, Framer),
            Theme::Moonfly => LoadedTheme::new(value, Moonfly),
            Theme::MonokaiSoda => LoadedTheme::new(value, MonokaiSoda),
            Theme::Slate => LoadedTheme::new(value, Slate),
            Theme::Firewatch => LoadedTheme::new(value, Firewatch),
            Theme::Spacegray => LoadedTheme::new(value, Spacegray),
            Theme::Symfonic => LoadedTheme::new(value, Symfonic),
            Theme::Spacedust => LoadedTheme::new(value, Spacedust),
            Theme::Teerb => LoadedTheme::new(value, Teerb),
            Theme::Rippedcasts => LoadedTheme::new(value, Rippedcasts),
            Theme::TomorrowNightBright => LoadedTheme::new(value, TomorrowNightBright),
            Theme::Chalk => LoadedTheme::new(value, Chalk),
            Theme::Heeler => LoadedTheme::new(value, Heeler),
            Theme::Alabaster => LoadedTheme::new(value, Alabaster),
            Theme::Galaxy => LoadedTheme::new(value, Galaxy),
            Theme::Adventure => LoadedTheme::new(value, Adventure),
            Theme::GithubLightHighContrast => LoadedTheme::new(value, GithubLightHighContrast),
            Theme::Unikitty => LoadedTheme::new(value, Unikitty),
            Theme::Lavandula => LoadedTheme::new(value, Lavandula),
            Theme::TinaciousDesignDark => LoadedTheme::new(value, TinaciousDesignDark),
            Theme::Flatland => LoadedTheme::new(value, Flatland),
            Theme::GithubLightColorblind => LoadedTheme::new(value, GithubLightColorblind),
            Theme::ChallengerDeep => LoadedTheme::new(value, ChallengerDeep),
            Theme::Mirage => LoadedTheme::new(value, Mirage),
            Theme::Deep => LoadedTheme::new(value, Deep),
            Theme::Vesper => LoadedTheme::new(value, Vesper),
            Theme::Aurora => LoadedTheme::new(value, Aurora),
            Theme::Jubi => LoadedTheme::new(value, Jubi),
            Theme::Calamity => LoadedTheme::new(value, Calamity),
            Theme::Neutron => LoadedTheme::new(value, Neutron),
            Theme::RedAlert => LoadedTheme::new(value, RedAlert),
            Theme::MaterialDark => LoadedTheme::new(value, MaterialDark),
            Theme::Everblush => LoadedTheme::new(value, Everblush),
            Theme::Kurokula => LoadedTheme::new(value, Kurokula),
            Theme::SolarizedOsakaNight => LoadedTheme::new(value, SolarizedOsakaNight),
            Theme::PaleNightHc => LoadedTheme::new(value, PaleNightHc),
            Theme::Ardoise => LoadedTheme::new(value, Ardoise),
            Theme::MonokaiProSpectrum => LoadedTheme::new(value, MonokaiProSpectrum),
            Theme::GruberDarker => LoadedTheme::new(value, GruberDarker),
            Theme::GithubDark => LoadedTheme::new(value, GithubDark),
            Theme::SolarizedDarkPatched => LoadedTheme::new(value, SolarizedDarkPatched),
            Theme::MelangeLight => LoadedTheme::new(value, MelangeLight),
            Theme::Hacktober => LoadedTheme::new(value, Hacktober),
            Theme::IcGreenPpl => LoadedTheme::new(value, IcGreenPpl),
            Theme::OperatorMonoDark => LoadedTheme::new(value, OperatorMonoDark),
            Theme::TomorrowNight => LoadedTheme::new(value, TomorrowNight),
            Theme::Homebrew => LoadedTheme::new(value, Homebrew),
            Theme::Wryan => LoadedTheme::new(value, Wryan),
            Theme::GithubDarkColorblind => LoadedTheme::new(value, GithubDarkColorblind),
            Theme::Borland => LoadedTheme::new(value, Borland),
            Theme::RedSands => LoadedTheme::new(value, RedSands),
            Theme::Rebecca => LoadedTheme::new(value, Rebecca),
            Theme::TerminalBasic => LoadedTheme::new(value, TerminalBasic),
            Theme::Dracula => LoadedTheme::new(value, Dracula),
            Theme::Nightfox => LoadedTheme::new(value, Nightfox),
            Theme::BuiltinLight => LoadedTheme::new(value, BuiltinLight),
            Theme::BuiltinSolarizedLight => LoadedTheme::new(value, BuiltinSolarizedLight),
            Theme::Glacier => LoadedTheme::new(value, Glacier),
            Theme::CatppuccinFrappe => LoadedTheme::new(value, CatppuccinFrappe),
            Theme::ForestBlue => LoadedTheme::new(value, ForestBlue),
            Theme::BuiltinDark => LoadedTheme::new(value, BuiltinDark),
            Theme::Whimsy => LoadedTheme::new(value, Whimsy),
            Theme::XcodeDark => LoadedTheme::new(value, XcodeDark),
            Theme::GruvboxDarkHard => LoadedTheme::new(value, GruvboxDarkHard),
            Theme::Ryuuko => LoadedTheme::new(value, Ryuuko),
            Theme::DjangoSmooth => LoadedTheme::new(value, DjangoSmooth),
            Theme::VioletDark => LoadedTheme::new(value, VioletDark),
            Theme::PurpleRain => LoadedTheme::new(value, PurpleRain),
            Theme::Square => LoadedTheme::new(value, Square),
            Theme::SeoulbonesDark => LoadedTheme::new(value, SeoulbonesDark),
            Theme::DarkModern => LoadedTheme::new(value, DarkModern),
            Theme::MonokaiProRistretto => LoadedTheme::new(value, MonokaiProRistretto),
            Theme::IrixConsole => LoadedTheme::new(value, IrixConsole),
            Theme::MelangeDark => LoadedTheme::new(value, MelangeDark),
            Theme::Earthsong => LoadedTheme::new(value, Earthsong),
            Theme::LiquidCarbon => LoadedTheme::new(value, LiquidCarbon),
            Theme::GruvboxLightHard => LoadedTheme::new(value, GruvboxLightHard),
            Theme::AdventureTime => LoadedTheme::new(value, AdventureTime),
            Theme::PhalaGreenDark => LoadedTheme::new(value, PhalaGreenDark),
            Theme::SoftServer => LoadedTheme::new(value, SoftServer),
            Theme::PencilDark => LoadedTheme::new(value, PencilDark),
            Theme::FrontEndDelight => LoadedTheme::new(value, FrontEndDelight),
            Theme::Grass => LoadedTheme::new(value, Grass),
            Theme::Ollie => LoadedTheme::new(value, Ollie),
            Theme::MidnightInMojave => LoadedTheme::new(value, MidnightInMojave),
            Theme::Highway => LoadedTheme::new(value, Highway),
            Theme::OvernightSlumber => LoadedTheme::new(value, OvernightSlumber),
            Theme::BelafonteNight => LoadedTheme::new(value, BelafonteNight),
            Theme::TangoAdapted => LoadedTheme::new(value, TangoAdapted),
            Theme::BlueBerryPie => LoadedTheme::new(value, BlueBerryPie),
            Theme::Tomorrow => LoadedTheme::new(value, Tomorrow),
            Theme::ScarletProtocol => LoadedTheme::new(value, ScarletProtocol),
            Theme::KittyLowContrast => LoadedTheme::new(value, KittyLowContrast),
            Theme::OceanicMaterial => LoadedTheme::new(value, OceanicMaterial),
            Theme::Guezwhoz => LoadedTheme::new(value, Guezwhoz),
            Theme::XcodeWwdc => LoadedTheme::new(value, XcodeWwdc),
            Theme::ToyChest => LoadedTheme::new(value, ToyChest),
            Theme::PoppingAndLocking => LoadedTheme::new(value, PoppingAndLocking),
            Theme::NeobonesLight => LoadedTheme::new(value, NeobonesLight),
            Theme::AtelierSulphurpool => LoadedTheme::new(value, AtelierSulphurpool),
            Theme::CobaltNext => LoadedTheme::new(value, CobaltNext),
            Theme::Atom => LoadedTheme::new(value, Atom),
            Theme::ZenbonesDark => LoadedTheme::new(value, ZenbonesDark),
            Theme::Zenbones => LoadedTheme::new(value, Zenbones),
            Theme::BrightLights => LoadedTheme::new(value, BrightLights),
            Theme::GithubDarkHighContrast => LoadedTheme::new(value, GithubDarkHighContrast),
            Theme::PaulMillr => LoadedTheme::new(value, PaulMillr),
            Theme::NocturnalWinter => LoadedTheme::new(value, NocturnalWinter),
            Theme::LabFox => LoadedTheme::new(value, LabFox),
            Theme::GithubDarkDimmed => LoadedTheme::new(value, GithubDarkDimmed),
            Theme::Detuned => LoadedTheme::new(value, Detuned),
            Theme::Misterioso => LoadedTheme::new(value, Misterioso),
            Theme::BlackMetalKhold => LoadedTheme::new(value, BlackMetalKhold),
            Theme::CoffeeTheme => LoadedTheme::new(value, CoffeeTheme),
            Theme::Japanesque => LoadedTheme::new(value, Japanesque),
            Theme::BlackMetalBurzum => LoadedTheme::new(value, BlackMetalBurzum),
            Theme::LiquidCarbonTransparent => LoadedTheme::new(value, LiquidCarbonTransparent),
            Theme::Darkmatrix => LoadedTheme::new(value, Darkmatrix),
            Theme::Vimbones => LoadedTheme::new(value, Vimbones),
            Theme::SolarizedDarkHigherContrast => {
                LoadedTheme::new(value, SolarizedDarkHigherContrast)
            }
            Theme::DimmedMonokai => LoadedTheme::new(value, DimmedMonokai),
            Theme::IrBlack => LoadedTheme::new(value, IrBlack),
            Theme::Pro => LoadedTheme::new(value, Pro),
            Theme::MonaLisa => LoadedTheme::new(value, MonaLisa),
            Theme::IcOrangePpl => LoadedTheme::new(value, IcOrangePpl),
            Theme::Wilmersdorf => LoadedTheme::new(value, Wilmersdorf),
            Theme::RaycastLight => LoadedTheme::new(value, RaycastLight),
            Theme::CobaltNextMinimal => LoadedTheme::new(value, CobaltNextMinimal),
            Theme::CatppuccinMacchiato => LoadedTheme::new(value, CatppuccinMacchiato),
            Theme::Oxocarbon => LoadedTheme::new(value, Oxocarbon),
            Theme::FishTank => LoadedTheme::new(value, FishTank),
            Theme::BuiltinSolarizedDark => LoadedTheme::new(value, BuiltinSolarizedDark),
            Theme::Andromeda => LoadedTheme::new(value, Andromeda),
            Theme::SnazzySoft => LoadedTheme::new(value, SnazzySoft),
            Theme::Vercel => LoadedTheme::new(value, Vercel),
            Theme::NordLight => LoadedTheme::new(value, NordLight),
            Theme::BuiltinPastelDark => LoadedTheme::new(value, BuiltinPastelDark),
            Theme::TomorrowNightEighties => LoadedTheme::new(value, TomorrowNightEighties),
            Theme::OneHalfLight => LoadedTheme::new(value, OneHalfLight),
            Theme::BlackMetalVenom => LoadedTheme::new(value, BlackMetalVenom),
            Theme::HorizonBright => LoadedTheme::new(value, HorizonBright),
            Theme::TangoHalfAdapted => LoadedTheme::new(value, TangoHalfAdapted),
            Theme::MaterialDarker => LoadedTheme::new(value, MaterialDarker),
            Theme::EmbersDark => LoadedTheme::new(value, EmbersDark),
            Theme::Violite => LoadedTheme::new(value, Violite),
            Theme::MonokaiProMachine => LoadedTheme::new(value, MonokaiProMachine),
            Theme::Encom => LoadedTheme::new(value, Encom),
            Theme::Embark => LoadedTheme::new(value, Embark),
            Theme::Ocean => LoadedTheme::new(value, Ocean),
            Theme::Batman => LoadedTheme::new(value, Batman),
            Theme::RedPlanet => LoadedTheme::new(value, RedPlanet),
            Theme::AppleSystemColors => LoadedTheme::new(value, AppleSystemColors),
            Theme::AppleSystemColorsLight => LoadedTheme::new(value, AppleSystemColorsLight),
            Theme::MonokaiProOctagon => LoadedTheme::new(value, MonokaiProOctagon),
            Theme::ThayerBright => LoadedTheme::new(value, ThayerBright),
            Theme::DarkPastel => LoadedTheme::new(value, DarkPastel),
            Theme::BlulocoLight => LoadedTheme::new(value, BlulocoLight),
            Theme::EverforestLightMed => LoadedTheme::new(value, EverforestLightMed),
            Theme::LaterThisEvening => LoadedTheme::new(value, LaterThisEvening),
            Theme::FlexokiLight => LoadedTheme::new(value, FlexokiLight),
            Theme::Darkside => LoadedTheme::new(value, Darkside),
            Theme::Harper => LoadedTheme::new(value, Harper),
            Theme::AtomOneLight => LoadedTheme::new(value, AtomOneLight),
            Theme::MaterialDesignColors => LoadedTheme::new(value, MaterialDesignColors),
            Theme::Adwaita => LoadedTheme::new(value, Adwaita),
            Theme::FirefoxDev => LoadedTheme::new(value, FirefoxDev),
            Theme::Kibble => LoadedTheme::new(value, Kibble),
            Theme::Duckbones => LoadedTheme::new(value, Duckbones),
            Theme::Hopscotch => LoadedTheme::new(value, Hopscotch),
            Theme::FarmhouseLight => LoadedTheme::new(value, FarmhouseLight),
            Theme::Srcery => LoadedTheme::new(value, Srcery),
            Theme::Flat => LoadedTheme::new(value, Flat),
            Theme::Subliminal => LoadedTheme::new(value, Subliminal),
            Theme::Kanagawabones => LoadedTheme::new(value, Kanagawabones),
            Theme::AlienBlood => LoadedTheme::new(value, AlienBlood),
            Theme::SelenizedDark => LoadedTheme::new(value, SelenizedDark),
            Theme::FlexokiDark => LoadedTheme::new(value, FlexokiDark),
            Theme::UltraDark => LoadedTheme::new(value, UltraDark),
            Theme::BananaBlueberry => LoadedTheme::new(value, BananaBlueberry),
            Theme::Elegant => LoadedTheme::new(value, Elegant),
            Theme::CyberpunkScarletProtocol => LoadedTheme::new(value, CyberpunkScarletProtocol),
            Theme::RosePineMoon => LoadedTheme::new(value, RosePineMoon),
            Theme::Wombat => LoadedTheme::new(value, Wombat),
            Theme::GruvboxMaterial => LoadedTheme::new(value, GruvboxMaterial),
            Theme::Sugarplum => LoadedTheme::new(value, Sugarplum),
            Theme::Medallion => LoadedTheme::new(value, Medallion),
            Theme::SeoulbonesLight => LoadedTheme::new(value, SeoulbonesLight),
            Theme::IrixTerminal => LoadedTheme::new(value, IrixTerminal),
            Theme::Vaughn => LoadedTheme::new(value, Vaughn),
            Theme::MonokaiProLight => LoadedTheme::new(value, MonokaiProLight),
            Theme::ParaisoDark => LoadedTheme::new(value, ParaisoDark),
            Theme::SleepyHollow => LoadedTheme::new(value, SleepyHollow),
            Theme::Monokai => LoadedTheme::new(value, Monokai),
            Theme::Argonaut => LoadedTheme::new(value, Argonaut),
            Theme::Catppuccin => LoadedTheme::new(value, Catppuccin),
            Theme::TokyoNight => LoadedTheme::new(value, TokyoNight),
            Theme::Terminal => LoadedTheme::new(value, Terminal),
            Theme::Chakra => LoadedTheme::new(value, Chakra),
            Theme::Custom => LoadedTheme::new(value, Custom::create_or_load()),
        }
    }
}

impl Deref for LoadedTheme {
    type Target = Arc<dyn Styler + Send + Sync>;

    fn deref(&self) -> &Self::Target {
        &self.styler
    }
}

impl Serialize for LoadedTheme {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(self.app_theme.id())
    }
}

impl<'de> Deserialize<'de> for LoadedTheme {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let theme_str = String::deserialize(deserializer)?;
        let app_theme =
            Theme::from_str(&theme_str).map_err(|e| serde::de::Error::custom(e.to_string()))?;
        Ok(LoadedTheme::from(app_theme))
    }
}

#[derive(Debug, Clone, Copy, EnumIter, PartialEq, Eq, AsRefStr, EnumString)]
pub enum Theme {
    AardvarkBlue,
    Abernathy,
    Adventure,
    AdventureTime,
    Adwaita,
    AdwaitaDark,
    Afterglow,
    Alabaster,
    AlienBlood,
    Andromeda,
    AppleClassic,
    AppleSystemColors,
    AppleSystemColorsLight,
    Arcoiris,
    Ardoise,
    Argonaut,
    Arthur,
    AtelierSulphurpool,
    Atom,
    AtomOneDark,
    AtomOneLight,
    Aura,
    Aurora,
    Ayu,
    AyuLight,
    AyuMirage,
    BananaBlueberry,
    Batman,
    BelafonteDay,
    BelafonteNight,
    BirdsOfParadise,
    BlackMetal,
    BlackMetalBathory,
    BlackMetalBurzum,
    BlackMetalDarkFuneral,
    BlackMetalGorgoroth,
    BlackMetalImmortal,
    BlackMetalKhold,
    BlackMetalMarduk,
    BlackMetalMayhem,
    BlackMetalNile,
    BlackMetalVenom,
    Blazer,
    BlueBerryPie,
    BlueDolphin,
    BlueMatrix,
    BlulocoDark,
    BlulocoLight,
    Borland,
    Breadog,
    Breeze,
    BrightLights,
    Broadcast,
    Brogrammer,
    BuiltinDark,
    BuiltinLight,
    BuiltinPastelDark,
    BuiltinSolarizedDark,
    BuiltinSolarizedLight,
    BuiltinTangoDark,
    BuiltinTangoLight,
    Calamity,
    Catppuccin,
    CatppuccinFrappe,
    CatppuccinLatte,
    CatppuccinMacchiato,
    CatppuccinMocha,
    Cga,
    Chakra,
    Chalk,
    Chalkboard,
    ChallengerDeep,
    Chester,
    Ciapre,
    Citruszest,
    Clrs,
    CobaltNeon,
    CobaltNext,
    CobaltNextDark,
    CobaltNextMinimal,
    CoffeeTheme,
    CrayonPonyFish,
    CursorDark,
    CutiePro,
    Cyberdyne,
    Cyberpunk,
    CyberpunkScarletProtocol,
    DarkModern,
    DarkPastel,
    Darkermatrix,
    Darkmatrix,
    Darkside,
    Dayfox,
    Deep,
    Desert,
    Detuned,
    Dimidium,
    DimmedMonokai,
    Django,
    DjangoRebornAgain,
    DjangoSmooth,
    DoomOne,
    DoomPeacock,
    DotGov,
    Dracula,
    Duckbones,
    DuotoneDark,
    Earthsong,
    ElectronHighlighter,
    Elegant,
    Elemental,
    Elementary,
    Embark,
    EmbersDark,
    Encom,
    Espresso,
    EspressoLibre,
    Everblush,
    EverforestDarkHard,
    EverforestLightMed,
    Fahrenheit,
    Fairyfloss,
    FarmhouseDark,
    FarmhouseLight,
    Fideloper,
    FireflyTraditional,
    FirefoxDev,
    Firewatch,
    FishTank,
    Flat,
    Flatland,
    FlexokiDark,
    FlexokiLight,
    Floraverse,
    ForestBlue,
    Framer,
    FrontEndDelight,
    FunForrest,
    Galaxy,
    Galizur,
    GhosttyDefaultStyleDark,
    Github,
    GithubDark,
    GithubDarkColorblind,
    GithubDarkDefault,
    GithubDarkDimmed,
    GithubDarkHighContrast,
    GithubLightColorblind,
    GithubLightDefault,
    GithubLightHighContrast,
    GitlabDark,
    GitlabDarkGrey,
    GitlabLight,
    Glacier,
    Grape,
    Grass,
    GreyGreen,
    GruberDarker,
    GruvboxDark,
    GruvboxDarkHard,
    GruvboxLight,
    GruvboxLightHard,
    GruvboxMaterial,
    Guezwhoz,
    Hacktober,
    Hardcore,
    Harper,
    HavnDaggry,
    HavnSkumring,
    Heeler,
    Highway,
    HipsterGreen,
    Hivacruz,
    Homebrew,
    Hopscotch,
    Horizon,
    HorizonBright,
    Hurtado,
    Hybrid,
    IcGreenPpl,
    IcOrangePpl,
    IcebergDark,
    IcebergLight,
    Idea,
    IdleToes,
    IrBlack,
    IrixConsole,
    IrixTerminal,
    JackieBrown,
    Japanesque,
    Jellybeans,
    JetbrainsDarcula,
    Jubi,
    KanagawaDragon,
    KanagawaWave,
    Kanagawabones,
    Kibble,
    KittyDefault,
    KittyLowContrast,
    Kolorit,
    Konsolas,
    Kurokula,
    LabFox,
    Laser,
    LaterThisEvening,
    Lavandula,
    LightOwl,
    LiquidCarbon,
    LiquidCarbonTransparent,
    LiquidCarbonTransparentInverse,
    Lovelace,
    ManPage,
    Mariana,
    Material,
    MaterialDark,
    MaterialDarker,
    MaterialDesignColors,
    MaterialOcean,
    Mathias,
    Matrix,
    Medallion,
    MelangeDark,
    MelangeLight,
    Mellifluous,
    Mellow,
    Miasma,
    MidnightInMojave,
    Mirage,
    Misterioso,
    Molokai,
    MonaLisa,
    Monokai,
    MonokaiClassic,
    MonokaiPro,
    MonokaiProLight,
    MonokaiProLightSun,
    MonokaiProMachine,
    MonokaiProOctagon,
    MonokaiProRistretto,
    MonokaiProSpectrum,
    MonokaiRemastered,
    MonokaiSoda,
    MonokaiVivid,
    Moonfly,
    NeobonesDark,
    NeobonesLight,
    Neon,
    Neopolitan,
    Neutron,
    NightOwl,
    NightOwlishLight,
    Nightfox,
    Niji,
    NocturnalWinter,
    Nord,
    NordLight,
    NordWave,
    Novel,
    NvimDark,
    NvimLight,
    Obsidian,
    Ocean,
    OceanicMaterial,
    OceanicNext,
    Ollie,
    OneDoubleDark,
    OneDoubleLight,
    OneHalfDark,
    OneHalfLight,
    OperatorMonoDark,
    OvernightSlumber,
    Oxocarbon,
    PaleNightHc,
    Pandora,
    ParaisoDark,
    PaulMillr,
    PencilDark,
    PencilLight,
    Peppermint,
    PhalaGreenDark,
    PiattoLight,
    Pnevma,
    PoppingAndLocking,
    Powershell,
    Primary,
    Pro,
    ProLight,
    PurpleRain,
    Purplepeter,
    Rapture,
    RaycastDark,
    RaycastLight,
    Rebecca,
    RedAlert,
    RedPlanet,
    RedSands,
    Relaxed,
    Retro,
    RetroLegends,
    Rippedcasts,
    RosePine,
    RosePineDawn,
    RosePineMoon,
    Royal,
    Ryuuko,
    Sakura,
    ScarletProtocol,
    SeaShells,
    SeafoamPastel,
    SelenizedDark,
    SelenizedLight,
    SeoulbonesDark,
    SeoulbonesLight,
    Seti,
    ShadesOfPurple,
    Shaman,
    Slate,
    SleepyHollow,
    Smyck,
    Snazzy,
    SnazzySoft,
    SoftServer,
    SolarizedDarcula,
    SolarizedDarkHigherContrast,
    SolarizedDarkPatched,
    SolarizedOsakaNight,
    Sonokai,
    Spacedust,
    Spacegray,
    SpacegrayBright,
    SpacegrayEighties,
    SpacegrayEightiesDull,
    Spiderman,
    Spring,
    Square,
    SquirrelsongDark,
    Srcery,
    Starlight,
    Sublette,
    Subliminal,
    Sugarplum,
    Sundried,
    Symfonic,
    Synthwave,
    SynthwaveAlpha,
    SynthwaveEverything,
    TangoAdapted,
    TangoHalfAdapted,
    Tearout,
    Teerb,
    Terafox,
    Terminal,
    TerminalBasic,
    TerminalBasicDark,
    ThayerBright,
    TheHulk,
    TinaciousDesignDark,
    TinaciousDesignLight,
    TokyoNight,
    TokyoNightDay,
    TokyoNightMoon,
    TokyoNightNight,
    TokyoNightStorm,
    Tomorrow,
    TomorrowNight,
    TomorrowNightBlue,
    TomorrowNightBright,
    TomorrowNightBurns,
    TomorrowNightEighties,
    ToyChest,
    Treehouse,
    Twilight,
    Ubuntu,
    UltraDark,
    UltraViolent,
    UnderTheSea,
    Unikitty,
    Urple,
    Vague,
    Vaughn,
    Vercel,
    Vesper,
    VibrantInk,
    Vimbones,
    VioletDark,
    VioletLight,
    Violite,
    WarmNeon,
    Wez,
    Whimsy,
    WildCherry,
    Wilmersdorf,
    Wombat,
    Wryan,
    XcodeDark,
    XcodeDarkHc,
    XcodeLight,
    XcodeLightHc,
    XcodeWwdc,
    Zenbones,
    ZenbonesDark,
    ZenbonesLight,
    Zenburn,
    Zenburned,
    ZenwrittenDark,
    ZenwrittenLight,
    Custom,
}

impl Theme {
    pub fn id(&self) -> &str {
        self.as_ref()
    }

    pub fn title(&self) -> &str {
        self.as_ref()
    }
}

impl Display for Theme {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.title())
    }
}
