# Event System

The event system handles all input and internal communication in reovim.

## Overview

```
lib/core/src/
├── event_bus/          # Type-erased event system (NEW)
│   ├── mod.rs          # Event trait, DynEvent, exports
│   └── bus.rs          # EventBus implementation
│
├── event/              # Legacy event system
│   ├── mod.rs          # Trait definitions, exports
│   ├── input.rs        # InputEventBroker
│   ├── key/
│   │   └── mod.rs      # KeyEventBroker
│   ├── handler/
│   │   ├── mod.rs      # TerminateHandler
│   │   ├── command/    # CommandHandler
│   │   │   ├── mod.rs
│   │   │   ├── dispatcher.rs
│   │   │   └── count_parser.rs
│   │   └── completion.rs # CompletionHandler
│   └── inner/
│       └── mod.rs      # InnerEvent enum
```

## Event Types

### Event Bus (Type-Erased Events)

The event bus provides a type-erased event system for plugin communication. Unlike `InnerEvent`, the event bus allows plugins to define their own event types without modifying core enums.

**Location:** `lib/core/src/event_bus/`

```rust
// Event trait - implemented by all plugin events
pub trait Event: Send + Sync + 'static {
    fn priority(&self) -> u32 { 100 }  // Lower = higher priority
}

// EventBus - type-erased event dispatch
pub struct EventBus {
    handlers: RwLock<HashMap<TypeId, Vec<EventHandler>>>,
}

impl EventBus {
    // Subscribe to events of type E
    pub fn subscribe<E: Event, F>(&self, priority: u32, handler: F)
    where
        F: Fn(&E, &EventContext) -> EventResult + Send + Sync + 'static;

    // Emit an event to all subscribers
    pub fn emit<E: Event>(&self, event: E);
}
```

**Event Result:**
```rust
pub enum EventResult {
    Handled,      // Event was handled, stop propagation
    Continue,     // Event was handled, continue propagation
    NotHandled,   // Event was not handled
}
```

**Example - Leap Events:**
```rust
// Define event
#[derive(Debug, Clone)]
pub struct LeapStartEvent {
    pub direction: LeapDirection,
    pub operator: Option<OperatorType>,
    pub count: Option<usize>,
}

impl Event for LeapStartEvent {
    fn priority(&self) -> u32 { 50 }  // High priority for mode changes
}

// Subscribe in plugin
bus.subscribe::<LeapStartEvent, _>(100, |event, _ctx| {
    tracing::trace!(direction = ?event.direction, "Leap started");
    EventResult::Handled
});

// Emit from runtime
event_bus.emit(LeapStartEvent {
    direction: LeapDirection::Forward,
    operator: None,
    count: None,
});
```

**Leap Events (via Event Bus):**
| Event | Description |
|-------|-------------|
| `LeapStartEvent` | Leap mode activated |
| `LeapFirstCharEvent` | First character entered |
| `LeapSecondCharEvent` | Second character entered |
| `LeapSelectLabelEvent` | Label selected for jump |
| `LeapCancelEvent` | Leap mode cancelled |
| `LeapJumpEvent` | Jump completed (from, to, direction) |
| `LeapMatchesFoundEvent` | Matches found (count, pattern) |

**Core Events (via Event Bus):**

Core events are defined in `lib/core/src/event_bus/core_events.rs` and allow plugins to request runtime actions.

| Event | Description |
|-------|-------------|
| `RequestSetRegister` | Set register content (unnamed or named) |

**Option Events (via Event Bus):**

Option events are defined in `lib/core/src/option/events.rs` for the extensible settings system.

| Event | Priority | Description |
|-------|----------|-------------|
| `RegisterSettingSection` | 40 | Plugin registers a settings section with UI metadata |
| `RegisterOption` | 50 | Plugin registers an option specification |
| `OptionChanged` | 100 | Option value was modified |
| `QueryOption` | 100 | Request to query an option value |
| `ResetOption` | 100 | Request to reset option to default |

**RegisterSettingSection:**

Emitted by plugins to register a settings section for the settings menu:

```rust
use reovim_core::option::RegisterSettingSection;

bus.emit(RegisterSettingSection::new("my_plugin", "My Plugin")
    .with_description("Plugin-specific settings")
    .with_order(100));  // Core sections use 0-50
```

Fields:
- `id: Cow<str>` - Section identifier (must match `OptionSpec.section`)
- `display_name: Cow<str>` - Display name for the section header
- `description: Option<Cow<str>>` - Optional description
- `order: u32` - Display order (lower = earlier)

**OptionChanged:**

Emitted when any option value changes, allowing plugins to react:

```rust
use reovim_core::option::{OptionChanged, ChangeSource};

bus.subscribe::<OptionChanged, _>(100, move |event, ctx| {
    if event.name == "plugin.treesitter.highlight_timeout_ms" {
        if let Some(timeout) = event.as_int() {
            // Update internal timeout setting
        }
        ctx.request_render();
    }
    EventResult::Continue
});
```

Fields:
- `name: String` - Full option name (e.g., `"plugin.treesitter.timeout"`)
- `old_value: OptionValue` - Previous value
- `new_value: OptionValue` - New value
- `source: ChangeSource` - How the change was triggered

`ChangeSource` variants:
- `UserCommand` - User typed a `:set` command
- `ProfileLoad` - Loaded from a profile file
- `Plugin` - Changed programmatically by a plugin
- `SettingsMenu` - Changed via the settings menu UI
- `Rpc` - Changed via RPC (server mode)
- `Default` - Default value applied during initialization

**RequestSetRegister:**

Allows plugins to set register content, enabling features like copy-to-clipboard:

```rust
use reovim_core::event_bus::core_events::RequestSetRegister;

// Copy path to both unnamed register (for 'p' paste) and system clipboard
ctx.emit(RequestSetRegister {
    register: None,      // Unnamed register - used by 'p' paste
    text: path.clone(),
});
ctx.emit(RequestSetRegister {
    register: Some('+'), // System clipboard register
    text: path,
});
```

Register names:
- `None` - Unnamed register (default for yank/paste)
- `Some('+')` - System clipboard
- `Some('a')` to `Some('z')` - Named registers

## Event Design Patterns

When designing events for your plugin, choose the appropriate pattern based on your needs.

### Unified Command-Event Types (Recommended)

**Modern approach:** Use `declare_event_command!` macro to create a single type that serves as both command and event.

```rust
use reovim_core::declare_event_command;

// Single unified type - both command AND event
declare_event_command! {
    ExplorerRefresh,
    id: "explorer_refresh",
    description: "Refresh explorer view",
}

// Register as command
ctx.register_command(ExplorerRefresh);

// Subscribe as event (same type!)
bus.subscribe::<ExplorerRefresh, _>(100, |_event, ctx| {
    ctx.state.with_mut::<ExplorerState, _, _>(|state| {
        state.refresh();
        EventResult::Handled
    }).unwrap_or(EventResult::NotHandled)
});
```

**When to use:**
- ✅ Handler has access to all needed state via `PluginStateRegistry`
- ✅ No parameters required from command
- ✅ Simple trigger/notification
- ✅ **Preferred for most plugin commands**

**Benefits:**
- 50% fewer types (one instead of two)
- ~20 lines of boilerplate eliminated per command
- Minimal memory overhead (zero-sized type)
- Implements `Copy` - cheap to clone
- Simple to construct with `Default`

### Legacy: Separate Event Types (Deprecated)

**Old approach:** Define standalone event types (no longer recommended).

```rust
// DON'T DO THIS - use declare_event_command! instead
#[derive(Debug, Clone, Copy, Default)]
pub struct ExplorerRefreshEvent;

impl Event for ExplorerRefreshEvent {}

// Requires separate command type too (20+ lines of boilerplate)
```

### Unified Types with Data (Counted Commands)

Use `declare_counted_event_command!` for commands that use repeat counts:

```rust
use reovim_core::declare_counted_event_command;

// Single type with count - both command AND event
declare_counted_event_command! {
    ExplorerCursorDown,
    id: "explorer_cursor_down",
    description: "Move cursor down",
}

// Register (Default provides count=1)
ctx.register_command(ExplorerCursorDown::new(1));

// Subscribe - access event.count directly
bus.subscribe::<ExplorerCursorDown, _>(100, |event, ctx| {
    for _ in 0..event.count {
        // Move down
    }
    EventResult::Handled
});
```

**When to use:**
- ✅ Command receives repeat count (e.g., `5j` for "down 5 times")
- ✅ Handler needs count parameter
- ✅ **Preferred over separate Command/Event types**

### Custom Data Fields

For events with custom data that don't fit the standard macros:

```rust
use reovim_core::event_bus::Event;

#[derive(Debug, Clone, Copy)]
pub struct ExplorerInputChar {
    pub c: char,
}

impl ExplorerInputChar {
    pub const fn new(c: char) -> Self {
        Self { c }
    }
}

impl Event for ExplorerInputChar {
    fn priority(&self) -> u32 { 100 }
}

// Still use same type for both command and event
bus.subscribe::<ExplorerInputChar, _>(100, |event, ctx| {
    // Access event.c
    EventResult::Handled
});
```

**When to use:**
- Command receives custom parameters (char, string, complex data)
- Handler needs specific context not available in state
- Information passed from external source

**Benefits:**
- Explicit context passing
- Type-safe parameters
- Self-documenting through fields
- Still uses single type for both command and event

### Decision Guidelines

**Use `declare_event_command!` when:**
- ✅ State is in `PluginStateRegistry`
- ✅ No parameters needed
- ✅ Simple trigger action
- ✅ **This is the default choice for most commands**

**Use `declare_counted_event_command!` when:**
- ✅ Need count from command execution (e.g., `5j`)
- ✅ Command accepts repeat count parameter

**Use custom implementation when:**
- ✅ Passing custom data between plugins
- ✅ External context required (char, string, complex data)
- ✅ Handler can't access needed information from state

### Common Patterns (Modern)

**Pattern 1: Simple Toggle**
```rust
// Unified type - state knows whether it's on/off
declare_event_command! {
    Toggle,
    id: "toggle",
    description: "Toggle feature",
}
```

**Pattern 2: Counted Action**
```rust
// Unified with count - count affects behavior
declare_counted_event_command! {
    Repeat,
    id: "repeat",
    description: "Repeat action",
}
```

**Pattern 3: Custom Data**
```rust
// Custom implementation for specific data
#[derive(Debug, Clone)]
pub struct MoveInDirection {
    pub direction: Direction,
    pub count: usize,
}

impl Event for MoveInDirection {
    fn priority(&self) -> u32 { 100 }
}
```

**Pattern 4: State Change Notification**
```rust
// Unified type - handler queries state
declare_event_command! {
    StateChanged,
    id: "state_changed",
    description: "State has changed",
}
```

### InnerEvent (Legacy)

Internal events passed to the runtime via mpsc channel.

> **Note:** Features are being migrated to the Event Bus. New plugins should use `Event` trait and `EventBus` instead of adding variants to `InnerEvent`.

```rust
pub enum InnerEvent {
    // Core events
    BufferEvent(BufferEvent),
    CommandEvent(CommandEvent),
    ModeChangeEvent(ModeState),
    PendingKeysEvent(String),
    WindowEvent(WindowEvent),
    HighlightEvent(HighlightEvent),

    // Feature events
    CompletionEvent(CompletionEvent),
    ExplorerEvent(ExplorerEvent),
    TelescopeEvent(TelescopeEvent),
    LeapEvent(LeapEvent),
    TreesitterEvent(TreesitterEvent),
    OperatorMotionEvent(OperatorMotionAction),

    // Text input events (direct dispatch to components)
    TextInputEvent(TextInputEvent),
    VisualTextObjectEvent(VisualTextObjectAction),

    // UI events
    WhichKeyShow { prefix: String, bindings: Vec<WhichKeyBinding> },
    WhichKeyHide,

    // System
    RenderSignal,
    KillSignal,
}
```

### BufferEvent

Buffer management operations:

```rust
pub enum BufferEvent {
    SetContent { buffer_id: usize, content: String },
    LoadFile { path: String },
    Create,
    Close { buffer_id: usize },
    Switch { buffer_id: usize },
}
```

### CommandEvent

Command execution request:

```rust
pub struct CommandEvent {
    pub command: CommandRef,
    pub context: CommandContext,
}

pub struct CommandContext {
    pub buffer_id: usize,
    pub window_id: usize,
    pub count: Option<usize>,
}
```

### CompletionEvent

Text completion operations:

```rust
pub enum CompletionEvent {
    Trigger { buffer_id: usize },
    Update { items: Vec<CompletionItem>, prefix: String, start_col: usize, start_row: usize },
    SelectNext,
    SelectPrev,
    Confirm,
    Dismiss,
    UpdateFilter { new_prefix: String },
}
```

### TelescopeEvent

Fuzzy finder operations:

```rust
pub enum TelescopeEvent {
    Open { picker: String },
    UpdateQuery { query: String },
    UpdateItems { items: Vec<TelescopeItem> },
    SelectNext,
    SelectPrev,
    PageDown,
    PageUp,
    Confirm,
    Close,
    UpdatePreview { content: String },
}
```

### LeapEvent

Leap motion operations:

```rust
pub enum LeapEvent {
    Start { direction: LeapDirection, operator: Option<OperatorType>, count: Option<usize> },
    FirstChar { char: char },
    SecondChar { char: char },
    SelectLabel { label: char },
    Cancel,
}
```

### TextInputEvent

Character input event routed to the currently focused component. Uses a two-tier dispatch system: built-in components (Editor, CommandLine) are handled directly via fast path for synchronous Runtime access, while plugin components implement `UIComponent` trait methods.

```rust
pub enum TextInputEvent {
    InsertChar(char),       // Character to insert
    DeleteCharBackward,     // Backspace/delete
}
```

**Direct Dispatch (Runtime Fast Path):**
```rust
fn handle_interactor_input(&mut self, event: TextInputEvent) {
    let interactor_id = self.mode_state.interactor_id;

    // Fast path: Built-in components with direct Runtime access
    match interactor_id {
        ComponentId::EDITOR => {
            match event {
                TextInputEvent::InsertChar(c) => {
                    handle_editor_input(self, Some(c), false, false);
                }
                TextInputEvent::DeleteCharBackward => {
                    handle_editor_input(self, None, true, false);
                }
            }
            self.request_render();
            return;
        }
        ComponentId::COMMAND_LINE => {
            // Similar direct dispatch
        }
        _ => {
            // Plugin component path via UIComponent trait
            component.handle_insert_char(c, &mode_state, &plugin_state)
        }
    }
}
```

**Design Rationale:**
- **Built-in components** (Editor, CommandLine) need direct Runtime access (buffers, command_line state)
- **Plugin components** use `PluginStateRegistry` with interior mutability via `UIComponent` trait
- **Synchronous execution** for built-ins eliminates async bounce and race conditions
- **Clear separation** between core and plugin input handling

**Emitted by:**
- `CommandHandler` when processing single printable characters in insert/command mode
- `CommandHandler` when processing Backspace key in input-accepting modes
- Single printable characters → `TextInputEvent::InsertChar(c)`
- Backspace key → `TextInputEvent::DeleteCharBackward`

### TreesitterEvent

Treesitter parsing and highlight events:

```rust
pub enum TreesitterEvent {
    /// Request reparse for buffer
    Reparse { buffer_id: usize },
}
```

### ExplorerEvent

File explorer operations:

```rust
pub enum ExplorerEvent {
    Toggle,
    Focus,
    Unfocus,
    CursorUp,
    CursorDown,
    ToggleNode,
    OpenNode,
    // ... more variants
}
```

### WindowEvent

Window management:

```rust
pub enum WindowEvent {
    ToggleExplorer,
    FocusExplorer,
    FocusEditor,
}
```

### KeyEvent

Terminal key events from crossterm, broadcast to handlers.

## Components

### InputEventBroker

Reads terminal events asynchronously:

```rust
impl InputEventBroker {
    pub async fn subscribe(mut self, mut key_broker: KeyEventBroker) {
        let mut reader = EventStream::new();
        loop {
            tokio::select! {
                event = reader.next() => {
                    // Filter for key Press events
                    // Route to KeyEventBroker
                }
            }
        }
    }
}
```

**Responsibilities:**
- Read crossterm EventStream
- Filter key Press events (ignore Release/Repeat)
- Forward to KeyEventBroker

### KeyEventBroker

Broadcasts key events to subscribed handlers:

```rust
pub struct KeyEventBroker {
    tx: broadcast::Sender<KeyEvent>,
}

impl KeyEventBroker {
    pub fn new() -> Self {
        let (tx, _) = broadcast::channel(255);
        Self { tx }
    }

    pub fn enlist<T: Subscribe<KeyEvent>>(&self, handler: &mut T) {
        handler.subscribe(self.tx.subscribe());
    }

    pub fn handle(&self, event: KeyEvent) {
        let _ = self.tx.send(event);
    }
}
```

### Subscribe Trait

Handlers implement this to receive events:

```rust
pub trait Subscribe<T> {
    fn subscribe(&mut self, rx: broadcast::Receiver<T>);
}
```

## Event Handlers

### CommandHandler

Translates key events to commands:

```rust
pub struct CommandHandler {
    keymap: KeyMap,
    rx: Option<broadcast::Receiver<KeyEvent>>,
    tx: mpsc::Sender<InnerEvent>,
    pending_keys: String,
    pending_count: Option<usize>,
    local_mode: ModeState,
    mode_rx: watch::Receiver<ModeState>,
    command_registry: Arc<CommandRegistry>,
}
```

**Key Translation Process:**
1. Receive KeyEvent from broadcast
2. Update pending_keys with key representation
3. Look up in mode-specific keymap
4. If command found:
   - Create CommandContext with count
   - Send CommandEvent to runtime
   - Clear pending state
5. If partial match: wait for more keys, show which-key
6. If no match: handle based on mode
   - Insert: send InsertChar
   - Command: send CommandLineChar
   - Telescope Insert: send TelescopeInsertChar
   - Normal/Visual: ignore

### CompletionHandler

Handles async completion item fetching:

```rust
pub struct CompletionHandler {
    rx: mpsc::Receiver<CompletionRequest>,
    tx: mpsc::Sender<InnerEvent>,
    engine: Arc<CompletionEngine>,
}
```

**Process:**
1. Receive trigger request
2. Fetch completion items asynchronously
3. Send CompletionEvent::Update with results

### TerminateHandler

Handles Ctrl+C for graceful exit:

```rust
impl TerminateHandler {
    pub async fn subscribe(mut self) {
        while let Ok(event) = self.rx.recv().await {
            if event.code == KeyCode::Char('c')
               && event.modifiers == KeyModifiers::CONTROL {
                let _ = self.tx.send(InnerEvent::KillSignal).await;
            }
        }
    }
}
```

## Event Flow

### Complete Flow Diagram

```
1. TERMINAL INPUT
   crossterm EventStream (async)
        │
        ▼
2. INPUT BROKER
   InputEventBroker::subscribe()
   - Filters Key Press events
   - Calls KeyEventBroker::handle()
        │
        ▼
3. KEY BROADCAST
   KeyEventBroker (tokio broadcast, buffer: 255)
        │
        ├────────────────────┬────────────────────┐
        ▼                    ▼                    ▼
4. HANDLERS
   CommandHandler        TerminateHandler    CompletionHandler
   - Keys → Commands     - Ctrl+C → Kill    - Async fetching
   - Track pending_keys
   - Watch mode changes
        │                    │                    │
        ▼                    ▼                    ▼
   CommandEvent          KillSignal        CompletionEvent
   TelescopeEvent
   LeapEvent
   ModeChangeEvent
        │                    │                    │
        └─────────┬──────────┴────────────────────┘
                  ▼
5. RUNTIME EVENT LOOP
   Runtime::rx.recv().await

   match event {
       CommandEvent => execute command
       ModeChangeEvent => update mode, broadcast
       CompletionEvent => update completion state
       TelescopeEvent => update telescope state
       LeapEvent => handle leap motion
       TreesitterEvent => update highlights
       ExplorerEvent => handle explorer
       OperatorMotionEvent => execute operator+motion
       WhichKeyShow/Hide => update which-key panel
       RenderSignal => render()
       KillSignal => exit
   }
        │
        ▼
6. RENDERING
   Screen::render()
```

### Example: "5j" Command

```
Step 1: User presses "5"
┌─────────────────────────────────────────────┐
│ KeyEvent { code: Char('5'), ... }           │
│     │                                       │
│     ▼                                       │
│ CommandHandler                              │
│     pending_keys = "5"                      │
│     pending_count = Some(5)                 │
│     (no command yet, wait for more)         │
└─────────────────────────────────────────────┘

Step 2: User presses "j"
┌─────────────────────────────────────────────┐
│ KeyEvent { code: Char('j'), ... }           │
│     │                                       │
│     ▼                                       │
│ CommandHandler                              │
│     pending_keys = "5j"                     │
│     lookup("j") → CommandRef for cursor_down│
│     clear pending_keys                      │
│     │                                       │
│     ▼                                       │
│ Send CommandEvent {                         │
│     command: ById(CURSOR_DOWN),             │
│     context: { count: 5, ... }              │
│ }                                           │
└─────────────────────────────────────────────┘

Step 3: Runtime processes
┌─────────────────────────────────────────────┐
│ Runtime receives CommandEvent               │
│     │                                       │
│     ▼                                       │
│ Resolve CommandRef → Arc<CursorDownCommand> │
│ Create ExecutionContext                     │
│ cmd.execute(&mut ctx)                       │
│     Move cursor down by 5 lines             │
│     │                                       │
│     ▼                                       │
│ Returns CommandResult::NeedsRender          │
│     │                                       │
│     ▼                                       │
│ Screen::render()                            │
└─────────────────────────────────────────────┘
```

### Example: Leap Motion "sab"

```
Step 1: User presses "s"
┌─────────────────────────────────────────────┐
│ CommandHandler                              │
│     lookup("s") → LeapForwardCommand        │
│     │                                       │
│     ▼                                       │
│ Execute → DeferToRuntime(Leap(Start))       │
│     │                                       │
│     ▼                                       │
│ Runtime: set_mode(leap(Forward))            │
│ LeapState: WaitingFirstChar                 │
└─────────────────────────────────────────────┘

Step 2: User presses "a" (first char)
┌─────────────────────────────────────────────┐
│ CommandHandler (in Leap mode)               │
│     Send LeapEvent::FirstChar { char: 'a' } │
│     │                                       │
│     ▼                                       │
│ Runtime: Find all "a?" matches              │
│ LeapState: WaitingSecondChar                │
│ Render targets with labels                  │
└─────────────────────────────────────────────┘

Step 3: User presses "b" (second char)
┌─────────────────────────────────────────────┐
│ LeapEvent::SecondChar { char: 'b' }         │
│     │                                       │
│     ▼                                       │
│ Runtime: Find "ab" matches                  │
│ If single match: jump directly              │
│ If multiple: show labels for selection      │
│ set_mode(normal())                          │
│ Render                                      │
└─────────────────────────────────────────────┘
```

## Key Bindings

The keymap uses a trie structure for multi-key sequences:

```rust
pub struct KeyMapInner {
    pub command: Option<CommandRef>,
    pub next: HashMap<String, Self>,
}

pub struct KeyMap {
    pub normal: HashMap<String, KeyMapInner>,
    pub insert: HashMap<String, KeyMapInner>,
    pub visual: HashMap<String, KeyMapInner>,
    pub command: HashMap<String, KeyMapInner>,
    pub explorer: HashMap<String, KeyMapInner>,
    pub explorer_input: HashMap<String, KeyMapInner>,
    pub operator_pending: HashMap<String, KeyMapInner>,
    pub telescope_normal: HashMap<String, KeyMapInner>,
    pub telescope_insert: HashMap<String, KeyMapInner>,
    pub leap: HashMap<String, KeyMapInner>,
}
```

### Mode-Specific Keymaps

| Mode | Keymap | Purpose |
|------|--------|---------|
| Normal | `normal` | Standard editing commands |
| Insert | `insert` | Text input, Escape, completion |
| Visual | `visual` | Selection extension, operations |
| Command | `command` | Ex-command input |
| Explorer | `explorer` | File browser navigation |
| Explorer Input | `explorer_input` | File creation/rename input |
| Operator Pending | `operator_pending` | Motion after d/y/c |
| Telescope Normal | `telescope_normal` | Navigation with j/k |
| Telescope Insert | `telescope_insert` | Query typing |
| Leap | `leap` | Leap motion key handling |

### Default Bindings

**Normal Mode:**
| Key | Command |
|-----|---------|
| h/j/k/l | Cursor movement |
| 0/$ | Line start/end |
| w/b | Word forward/backward |
| gg/G | Document start/end |
| i/a/A | Insert modes |
| o/O | Open line |
| v/Ctrl-v | Visual modes |
| : | Command mode |
| x | Delete char |
| p/P | Paste after/before |
| u/Ctrl-r | Undo/redo |
| d/y/c | Operators |
| s/S | Leap forward/backward |
| Ctrl-o/Ctrl-i | Jump list |
| Space e | Toggle explorer |
| Space ff/fb/fg/fr | Telescope pickers |

**Insert Mode:**
| Key | Command |
|-----|---------|
| Escape | Normal mode |
| Backspace | Delete backward |
| Enter | Newline |
| Alt-Space | Trigger completion |
| Ctrl-n/Ctrl-p | Next/prev completion |
| Tab | Confirm completion |
| Ctrl-e | Dismiss completion |
| (any char) | Insert character |

**Visual Mode:**
| Key | Command |
|-----|---------|
| Escape | Normal mode |
| h/j/k/l | Extend selection |
| d | Delete selection |
| y | Yank selection |

**Command Mode:**
| Key | Command |
|-----|---------|
| Escape | Normal mode |
| Enter | Execute command |
| Backspace | Delete char |
| (any char) | Append to command |

## Mode-Specific Behavior

The CommandHandler adjusts behavior based on current mode:

```rust
if mode.is_insert() {
    // Unmapped keys become InsertChar(c)
} else if mode.is_command() {
    // Unmapped keys become CommandLineChar(c)
} else if mode.is_telescope_focus() && mode.is_insert() {
    // Unmapped keys become TelescopeInsertChar(c)
} else if mode.is_leap() {
    // Keys go to LeapEvent handling
} else if mode.is_normal() || mode.is_visual() {
    // Unmapped keys are ignored
}
```

## Mode Change Broadcasting

Mode changes are broadcast via `watch` channel:

```rust
// In Runtime
pub fn set_mode(&mut self, mode: ModeState) {
    self.mode_state = mode.clone();
    let _ = self.mode_tx.send(mode);
}

// In CommandHandler
loop {
    tokio::select! {
        Ok(key) = self.rx.recv() => { /* handle key */ }
        Ok(()) = self.mode_rx.changed() => {
            self.local_mode = self.mode_rx.borrow().clone();
        }
    }
}
```
