/*
 * Copyright (C) 2025 isaki@NetBSD.org
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

//
// 自前の uint128
//

#pragma once

#include "header.h"

// テスト用。
//#define FORCE_MY_UINT128

#if defined(HAVE___INT128) && !defined(FORCE_MY_UINT128)
using uint128 = unsigned __int128;
#else
#define UINT128_IS_CLASS 1

class uint128
{
	friend bool operator==(const uint128&, const uint128&);

 public:
	uint128() { }
	/*noexplicit*/ uint128(uint64 lo_) {
		hi = 0;
		lo = lo_;
	}
	uint128(uint64 hi_, uint64 lo_) {
		hi = hi_;
		lo = lo_;
	}

	// 代入
	uint128& operator=(const uint64& other) {
		hi = 0;
		lo = other;
		return *this;
	}
	uint128& operator=(const uint128& other) {
		hi = other.hi;
		lo = other.lo;
		return *this;
	}

	// キャスト演算子
	explicit operator uint64() const noexcept {
		return lo;
	}
	explicit operator uint32() const noexcept {
		return (uint32)lo;
	}

	// ビット反転
	uint128 operator~() const {
		return uint128(~hi, ~lo);
	}

	// 複合演算子
	uint128 operator<<=(const int& n) {
		if (__predict_false(n <= 0 || n >= 128)) {
			// 0 なら何もしない。
			// 負またはビット数以上のシフトは未定義だが、
			// 現状手元の __int128 は変化しないようだ。
		} else if (n >= 64) {
			hi = lo << (n - 64);
			lo = 0;
		} else {
			uint64 new_hi = (hi << n) | (lo >> (64 - n));
			uint64 new_lo = lo << n;
			hi = new_hi;
			lo = new_lo;
		}
		return *this;
	}
	uint128 operator>>=(const int& n) {
		if (__predict_false(n <= 0 || n >= 128)) {
			// 0 なら何もしない。
			// 負またはビット数以上のシフトは未定義だが、
			// 現状手元の __int128 は変化しないようだ。
		} else if (n >= 64) {
			// 順序注意。
			lo = hi >> (n - 64);
			hi = 0;
		} else {
			uint64 new_hi = hi >> n;
			uint64 new_lo = (hi << (64 - n)) | (lo >> n);
			hi = new_hi;
			lo = new_lo;
		}
		return *this;
	}
	uint128 operator&=(const uint128& other) {
		hi &= other.hi;
		lo &= other.lo;
		return *this;
	}
	uint128 operator|=(const uint128& other) {
		hi |= other.hi;
		lo |= other.lo;
		return *this;
	}
	uint128 operator^=(const uint128& other) {
		hi ^= other.hi;
		lo ^= other.lo;
		return *this;
	}

	// デバッグ用
	uint64 GetH() const { return hi; }
	uint64 GetL() const { return lo; }
	std::string to_str() const {
		char buf[48];
		snprintf(buf, sizeof(buf), "%08x'%08x'%08x'%08x",
			(uint32)(hi >> 32),
			(uint32)hi,
			(uint32)(lo >> 32),
			(uint32)lo);
		return std::string(buf);
	}

 private:
	uint64 hi {};
	uint64 lo {};
};

// 比較
inline bool operator==(const uint128& lhs, const uint128& rhs) {
	return (lhs.lo == rhs.lo) && (lhs.hi == rhs.hi);
}
inline bool operator!=(const uint128& lhs, const uint128& rhs) {
	return !(lhs == rhs);
}

// シフト
inline uint128 operator<<(const uint128& v, std::size_t n) {
	uint128 tmp(v);
	tmp <<= n;
	return tmp;
}
inline uint128 operator>>(const uint128& v, std::size_t n) {
	uint128 tmp(v);
	tmp >>= n;
	return tmp;
}

// ビット演算
inline uint128 operator&(const uint128& lhs, const uint128& rhs) {
	uint128 tmp(lhs);
	tmp &= rhs;
	return tmp;
}
inline uint128 operator|(const uint128& lhs, const uint128& rhs) {
	uint128 tmp(lhs);
	tmp |= rhs;
	return tmp;
}
inline uint128 operator^(const uint128& lhs, const uint128& rhs) {
	uint128 tmp(lhs);
	tmp ^= rhs;
	return tmp;
}

#endif
