//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// メモリストリーム
//

#pragma once

#include "header.h"

// ホストのメモリ領域に対するストリームクラスみたいなもの。
class MemoryStream
{
 public:
	explicit MemoryStream(void *start_) {
		Init(start_);
	}

	// 開始ポインタを設定する
	// 現在のポインタ位置は開始ポインタ位置になる
	void Init(void *start_) {
		start = (uint8 *)start_;
		ptr = start;
	}

	// 開始ポインタを取得する
	void *GetStartPtr() const
	{
		return (void *)start;
	}

	// バイトオフセットを設定する
	void SetOffset(uint32 ofs)
	{
		ptr = start + ofs;
	}

	// 現在のバイトオフセットを取得する
	uint32 GetOffset() const
	{
		return (uint32)(ptr - start);
	}

	// 1バイト読み込んでポインタを進める
	uint32 Read1() {
		uint32 data = *ptr++;
		return data;
	}

	// 1バイト書き込んでポインタを進める
	void Write1(uint32 data) {
		*ptr++ = data;
	}

	// 文字列を書き込んでポインタを進める。
	// 書き込む文字列も終端の '\0' まで書く。
	void WriteString(const char *p) {
		while (*p) {
			Write1(*p++);
		}
		Write1('\0');
	}

 protected:
	uint8 *ptr {};
	uint8 *start {};
};

// データの受け渡しはホストエンディアンで、
// アクセスはビッグエンディアンになる。境界への整列は不要。
class MemoryStreamBE : public MemoryStream
{
	using inherited = MemoryStream;
 public:
	explicit MemoryStreamBE(void *start_)
		: inherited(start_)
	{
	}

	// BE で2バイト読み込んでポインタを進める
	uint32 Read2() {
		uint32 data;
		data  = Read1() << 8;
		data |= Read1();
		return data;
	}
	// BE で4バイト読み込んでポインタを進める
	uint32 Read4() {
		uint32 data;
		data  = Read2() << 16;
		data |= Read2();
		return data;
	}

	// BE で2バイト書き込んでポインタを進める
	void Write2(uint32 data) {
		Write1(data >> 8);
		Write1(data & 0xff);
	}
	// BE で4バイト書き込んでポインタを進める
	void Write4(uint32 data) {
		Write2(data >> 16);
		Write2(data & 0xffff);
	}
};

// データの受け渡しはホストエンディアンで、
// アクセスはリトルエンディアンになる。境界への整列は不要。
class MemoryStreamLE : public MemoryStream
{
	using inherited = MemoryStream;
 public:
	explicit MemoryStreamLE(void *start_)
		: inherited(start_)
	{
	}

	// LE で2バイト読み込んでポインタを進める
	uint32 Read2() {
		uint32 data;
		data  = Read1();
		data |= Read1() << 8;
		return data;
	}
	// LE で4バイト読み込んでポインタを進める
	uint32 Read4() {
		uint32 data;
		data  = Read2();
		data |= Read2() << 16;
		return data;
	}

	// LE で2バイト書き込んでポインタを進める。
	void Write2(uint32 data) {
		Write1(data & 0xff);
		Write1(data >> 8);
	}
	// LE で4バイト書き込んでポインタを進める。
	void Write4(uint32 data) {
		Write2(data & 0xffff);
		Write2(data >> 16);
	}
	// LE で8バイト書き込んでポインタを進める。
	void Write8(uint64 data) {
		Write4((uint32)data);
		Write4(data >> 32);
	}
};
