/*	$NetBSD: adiantum.c,v 1.7 2021/10/17 14:45:45 jmcneill Exp $	*/

/*-
 * Copyright (c) 2020 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * The Adiantum wide-block cipher, from
 *
 *	Paul Crowley and Eric Biggers, `Adiantum: length-preserving
 *	encryption for entry-level processors', IACR Transactions on
 *	Symmetric Cryptology 2018(4), pp. 39--61.
 *
 *	https://doi.org/10.13154/tosc.v2018.i4.39-61
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(1, "$NetBSD: adiantum.c,v 1.7 2021/10/17 14:45:45 jmcneill Exp $");

#include <sys/types.h>
#include <sys/endian.h>

#ifdef _KERNEL

#include <sys/module.h>
#include <sys/systm.h>

#include <lib/libkern/libkern.h>

#include <crypto/adiantum/adiantum.h>
#include <crypto/aes/aes.h>
#include <crypto/chacha/chacha.h>

#else  /* !defined(_KERNEL) */

#include <sys/cdefs.h>

#include <assert.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include <openssl/aes.h>

struct aesenc {
	AES_KEY enckey;
};

struct aesdec {
	AES_KEY deckey;
};

#define	AES_256_NROUNDS	14
#define	aes_setenckey256(E, K)	AES_set_encrypt_key((K), 256, &(E)->enckey)
#define	aes_setdeckey256(D, K)	AES_set_decrypt_key((K), 256, &(D)->deckey)
#define	aes_enc(E, P, C, NR)	AES_encrypt(P, C, &(E)->enckey)
#define	aes_dec(D, C, P, NR)	AES_decrypt(C, P, &(D)->deckey)

#include "adiantum.h"

#define	CTASSERT	__CTASSERT
#define	KASSERT		assert
#define	MIN(x,y)	((x) < (y) ? (x) : (y))

static void
hexdump(int (*prf)(const char *, ...) __printflike(1,2), const char *prefix,
    const void *buf, size_t len)
{
	const uint8_t *p = buf;
	size_t i;

	(*prf)("%s (%zu bytes)\n", prefix, len);
	for (i = 0; i < len; i++) {
		if (i % 16 == 8)
			(*prf)("  ");
		else
			(*prf)(" ");
		(*prf)("%02hhx", p[i]);
		if ((i + 1) % 16 == 0)
			(*prf)("\n");
	}
	if (i % 16)
		(*prf)("\n");
}

#endif	/* _KERNEL */

/* Arithmetic modulo 2^128, represented by 16-digit strings in radix 2^8.  */

/* s := a + b (mod 2^128) */
static inline void
add128(uint8_t s[restrict static 16],
    const uint8_t a[static 16], const uint8_t b[static 16])
{
	unsigned i, c;

	c = 0;
	for (i = 0; i < 16; i++) {
		c = a[i] + b[i] + c;
		s[i] = c & 0xff;
		c >>= 8;
	}
}

/* s := a - b (mod 2^128) */
static inline void
sub128(uint8_t d[restrict static 16],
    const uint8_t a[static 16], const uint8_t b[static 16])
{
	unsigned i, c;

	c = 0;
	for (i = 0; i < 16; i++) {
		c = a[i] - b[i] - c;
		d[i] = c & 0xff;
		c = 1 & (c >> 8);
	}
}

static int
addsub128_selftest(void)
{
	static const uint8_t zero[16] = {
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
	};
	static const uint8_t one[16] = {
		0x01,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
	};
	static const uint8_t negativeone[16] = {
		0xff,0xff,0xff,0xff, 0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff, 0xff,0xff,0xff,0xff,
	};
	static const uint8_t a[16] = {
		0x03,0x80,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
	};
	static const uint8_t b[16] = {
		0x01,0x82,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
	};
	static const uint8_t c[16] = {
		0x02,0xfe,0xff,0xff, 0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff, 0xff,0xff,0xff,0xff,
	};
	uint8_t r[16];
	int result = 0;

	sub128(r, zero, one);
	if (memcmp(r, negativeone, 16)) {
		hexdump(printf, "sub128 1", r, sizeof r);
		result = -1;
	}

	sub128(r, a, b);
	if (memcmp(r, c, 16)) {
		hexdump(printf, "sub128 2", r, sizeof r);
		result = -1;
	}

	return result;
}

/* Poly1305 */

struct poly1305 {
	uint32_t r[5];		/* evaluation point */
	uint32_t h[5];		/* value */
};

static void
poly1305_init(struct poly1305 *P, const uint8_t key[static 16])
{

	/* clamp */
	P->r[0] = (le32dec(key +  0) >> 0) & 0x03ffffff;
	P->r[1] = (le32dec(key +  3) >> 2) & 0x03ffff03;
	P->r[2] = (le32dec(key +  6) >> 4) & 0x03ffc0ff;
	P->r[3] = (le32dec(key +  9) >> 6) & 0x03f03fff;
	P->r[4] = (le32dec(key + 12) >> 8) & 0x000fffff;

	/* initialize polynomial evaluation */
	P->h[0] = P->h[1] = P->h[2] = P->h[3] = P->h[4] = 0;
}

static void
poly1305_update_blocks(struct poly1305 *P, const uint8_t *m, size_t mlen)
{
	uint32_t r0 = P->r[0];
	uint32_t r1 = P->r[1];
	uint32_t r2 = P->r[2];
	uint32_t r3 = P->r[3];
	uint32_t r4 = P->r[4];
	uint32_t h0 = P->h[0];
	uint32_t h1 = P->h[1];
	uint32_t h2 = P->h[2];
	uint32_t h3 = P->h[3];
	uint32_t h4 = P->h[4];
	uint32_t m0, m1, m2, m3, m4; /* 26-bit message chunks */
	uint64_t k0, k1, k2, k3, k4; /* 64-bit extension of h */
	uint64_t p0, p1, p2, p3, p4; /* columns of product */
	uint32_t c;		     /* carry */

	while (mlen) {
		if (__predict_false(mlen < 16)) {
			/* Handle padding for uneven last block.  */
			uint8_t buf[16];
			unsigned i;

			for (i = 0; i < mlen; i++)
				buf[i] = m[i];
			buf[i++] = 1;
			for (; i < 16; i++)
				buf[i] = 0;
			m0 = le32dec(buf +  0) >> 0;
			m1 = le32dec(buf +  3) >> 2;
			m2 = le32dec(buf +  6) >> 4;
			m3 = le32dec(buf +  9) >> 6;
			m4 = le32dec(buf + 12) >> 8;
			mlen = 0;

			explicit_memset(buf, 0, sizeof buf);
		} else {
			m0 = le32dec(m +  0) >> 0;
			m1 = le32dec(m +  3) >> 2;
			m2 = le32dec(m +  6) >> 4;
			m3 = le32dec(m +  9) >> 6;
			m4 = le32dec(m + 12) >> 8;
			m4 |= 1u << 24;
			m += 16;
			mlen -= 16;
		}

		/* k := h + m, extended to 64 bits */
		k0 = h0 + (m0 & 0x03ffffff);
		k1 = h1 + (m1 & 0x03ffffff);
		k2 = h2 + (m2 & 0x03ffffff);
		k3 = h3 + m3;
		k4 = h4 + m4;

		/* p := k * r = (h + m)*r mod 2^130 - 5 */
		p0 = r0*k0 + 5*r4*k1 + 5*r3*k2 + 5*r2*k3 + 5*r1*k4;
		p1 = r1*k0 +   r0*k1 + 5*r4*k2 + 5*r3*k3 + 5*r2*k4;
		p2 = r2*k0 +   r1*k1 +   r0*k2 + 5*r4*k3 + 5*r3*k4;
		p3 = r3*k0 +   r2*k1 +   r1*k2 +   r0*k3 + 5*r4*k4;
		p4 = r4*k0 +   r3*k1 +   r2*k2 +   r1*k3 +   r0*k4;

		/* propagate carries and update h */
		p0 += 0; c = p0 >> 26; h0 = p0 & 0x03ffffff;
		p1 += c; c = p1 >> 26; h1 = p1 & 0x03ffffff;
		p2 += c; c = p2 >> 26; h2 = p2 & 0x03ffffff;
		p3 += c; c = p3 >> 26; h3 = p3 & 0x03ffffff;
		p4 += c; c = p4 >> 26; h4 = p4 & 0x03ffffff;

		/* reduce 2^130 = 5 */
		h0 += c*5; c = h0 >> 26; h0 &= 0x03ffffff;
		h1 += c;
	}

	/* update hash values */
	P->h[0] = h0;
	P->h[1] = h1;
	P->h[2] = h2;
	P->h[3] = h3;
	P->h[4] = h4;
}

static void
poly1305_final(uint8_t h[static 16], struct poly1305 *P)
{
	uint32_t h0 = P->h[0];
	uint32_t h1 = P->h[1];
	uint32_t h2 = P->h[2];
	uint32_t h3 = P->h[3];
	uint32_t h4 = P->h[4];
	uint32_t s0, s1, s2, s3, s4; /* h - (2^130 - 5) */
	uint32_t m;		     /* mask */
	uint32_t c;

	/* propagate carries */
	h1 += 0; c = h1 >> 26; h1 &= 0x03ffffff;
	h2 += c; c = h2 >> 26; h2 &= 0x03ffffff;
	h3 += c; c = h3 >> 26; h3 &= 0x03ffffff;
	h4 += c; c = h4 >> 26; h4 &= 0x03ffffff;

	/* reduce 2^130 = 5 */
	h0 += c*5; c = h0 >> 26; h0 &= 0x03ffffff;
	h1 += c;

	/* s := h - (2^130 - 5) */
	c = 5;
	s0 = h0 + c; c = s0 >> 26; s0 &= 0x03ffffff;
	s1 = h1 + c; c = s1 >> 26; s1 &= 0x03ffffff;
	s2 = h2 + c; c = s2 >> 26; s2 &= 0x03ffffff;
	s3 = h3 + c; c = s3 >> 26; s3 &= 0x03ffffff;
	s4 = h4 + c;
	s4 -= 0x04000000;

	/* m := -1 if h < 2^130 - 5 else 0 */
	m = -(s4 >> 31);

	/* conditional subtract */
	h0 = (m & h0) | (~m & s0);
	h1 = (m & h1) | (~m & s1);
	h2 = (m & h2) | (~m & s2);
	h3 = (m & h3) | (~m & s3);
	h4 = (m & h4) | (~m & s4);

	/* reduce modulo 2^128 */
	le32enc(h +  0, ((h1 << 26) | (h0 >>  0)) & 0xffffffff);
	le32enc(h +  4, ((h2 << 20) | (h1 >>  6)) & 0xffffffff);
	le32enc(h +  8, ((h3 << 14) | (h2 >> 12)) & 0xffffffff);
	le32enc(h + 12, ((h4 <<  8) | (h3 >> 18)) & 0xffffffff);
}

static void
poly1305(uint8_t h[static 16], const uint8_t *m, size_t mlen,
    const uint8_t k[static 16])
{
	struct poly1305 P;

	poly1305_init(&P, k);
	poly1305_update_blocks(&P, m, mlen);
	poly1305_final(h, &P);
}

static int
poly1305_selftest(void)
{
	/* https://tools.ietf.org/html/rfc7539#section-2.5.2 */
	static const uint8_t r[16] = {
		0x85,0xd6,0xbe,0x78, 0x57,0x55,0x6d,0x33,
		0x7f,0x44,0x52,0xfe, 0x42,0xd5,0x06,0xa8,
	};
	static const uint8_t s[16] = {
		0x01,0x03,0x80,0x8a, 0xfb,0x0d,0xb2,0xfd,
		0x4a,0xbf,0xf6,0xaf, 0x41,0x49,0xf5,0x1b,
	};
	static const uint8_t m[] = {
		0x43,0x72,0x79,0x70, 0x74,0x6f,0x67,0x72,
		0x61,0x70,0x68,0x69, 0x63,0x20,0x46,0x6f,
		0x72,0x75,0x6d,0x20, 0x52,0x65,0x73,0x65,
		0x61,0x72,0x63,0x68, 0x20,0x47,0x72,0x6f,
		0x75,0x70,
	};
	static const uint8_t expected[16] = {
		0xa8,0x06,0x1d,0xc1, 0x30,0x51,0x36,0xc6,
		0xc2,0x2b,0x8b,0xaf, 0x0c,0x01,0x27,0xa9,
	};
	uint8_t h[16], t[16];
	int result = 0;

	poly1305(h, m, sizeof m, r);
	add128(t, h, s);
	if (memcmp(t, expected, 16)) {
		hexdump(printf, "poly1305 h", h, sizeof h);
		hexdump(printf, "poly1305 t", t, sizeof t);
		result = -1;
	}

	return result;
}

/* NHPoly1305 */

static void
nh(uint8_t h[static 32], const uint8_t *m, size_t mlen,
    const uint32_t k[static 268 /* u/w + 2s(r - 1) */])
{
	enum {
	    s = 2,	 /* stride */
	    r = 4,	 /* rounds */
	    w = 32,	 /* word size */
	    u = 8192	 /* unit count (bits per msg unit) */
	};
	uint64_t h0 = 0, h1 = 0, h2 = 0, h3 = 0;
	unsigned i;

	CTASSERT(r*w/8 == 16);
	CTASSERT(u/w + 2*s*(r - 1) == 268);

	KASSERT(mlen <= u/8);
	KASSERT(mlen % 16 == 0);

	for (i = 0; i < mlen/16; i++) {
		uint32_t m0 = le32dec(m + 16*i + 4*0);
		uint32_t m1 = le32dec(m + 16*i + 4*1);
		uint32_t m2 = le32dec(m + 16*i + 4*2);
		uint32_t m3 = le32dec(m + 16*i + 4*3);

		uint32_t k00 = k[4*i + 4*0 + 0];
		uint32_t k01 = k[4*i + 4*0 + 1];
		uint32_t k02 = k[4*i + 4*0 + 2];
		uint32_t k03 = k[4*i + 4*0 + 3];
		uint32_t k10 = k[4*i + 4*1 + 0];
		uint32_t k11 = k[4*i + 4*1 + 1];
		uint32_t k12 = k[4*i + 4*1 + 2];
		uint32_t k13 = k[4*i + 4*1 + 3];
		uint32_t k20 = k[4*i + 4*2 + 0];
		uint32_t k21 = k[4*i + 4*2 + 1];
		uint32_t k22 = k[4*i + 4*2 + 2];
		uint32_t k23 = k[4*i + 4*2 + 3];
		uint32_t k30 = k[4*i + 4*3 + 0];
		uint32_t k31 = k[4*i + 4*3 + 1];
		uint32_t k32 = k[4*i + 4*3 + 2];
		uint32_t k33 = k[4*i + 4*3 + 3];

		CTASSERT(s == 2);
		h0 += (uint64_t)(m0 + k00) * (m2 + k02);
		h1 += (uint64_t)(m0 + k10) * (m2 + k12);
		h2 += (uint64_t)(m0 + k20) * (m2 + k22);
		h3 += (uint64_t)(m0 + k30) * (m2 + k32);
		h0 += (uint64_t)(m1 + k01) * (m3 + k03);
		h1 += (uint64_t)(m1 + k11) * (m3 + k13);
		h2 += (uint64_t)(m1 + k21) * (m3 + k23);
		h3 += (uint64_t)(m1 + k31) * (m3 + k33);
	}

	le64enc(h + 8*0, h0);
	le64enc(h + 8*1, h1);
	le64enc(h + 8*2, h2);
	le64enc(h + 8*3, h3);
}

static void
nhpoly1305(uint8_t h[static 16], const uint8_t *m, size_t mlen,
    const uint8_t pk[static 16],
    const uint32_t nhk[static 268 /* u/w + 2s(r - 1) */])
{
	struct poly1305 P;
	uint8_t h0[32];

	/*
	 * In principle NHPoly1305 is defined on uneven message
	 * lengths, but that's a pain in the patootie.
	 */
	KASSERT(mlen % 16 == 0);

	poly1305_init(&P, pk);
	for (; mlen; m += MIN(mlen, 1024), mlen -= MIN(mlen, 1024)) {
		nh(h0, m, MIN(mlen, 1024), nhk);
		poly1305_update_blocks(&P, h0, 32);
	}
	poly1305_final(h, &P);
}

/* https://github.com/google/adiantum/blob/68971e9c6684121b2203b4b05a22768b84051b58/test_vectors/ours/NH/NH.json */
static int
nh_selftest(void)
{
	static const struct {
		uint8_t k[1072];
		unsigned mlen;
		uint8_t m[1024];
		uint8_t h[32];
	} C[] = {
		[0] = {		/* 16-byte message */
			.k = {
				0x22,0x5b,0x80,0xc8, 0x18,0x05,0x37,0x09,
				0x76,0x14,0x4b,0x67, 0xc4,0x50,0x7f,0x2b,
				0x2c,0xff,0x56,0xc5, 0xd5,0x66,0x45,0x68,
				0x35,0xe6,0xd2,0x9a, 0xe5,0xd0,0xc1,0xfb,
				0xac,0x59,0x81,0x1a, 0x60,0xb0,0x3d,0x81,
				0x4b,0xa3,0x5b,0xa9, 0xcc,0xb3,0xfe,0x2d,
				0xc2,0x4d,0xd9,0x26, 0xad,0x36,0xcf,0x8c,
				0x05,0x11,0x3b,0x8a, 0x99,0x15,0x81,0xc8,
				0x23,0xf5,0x5a,0x94, 0x10,0x2f,0x92,0x80,
				0x38,0xc5,0xb2,0x63, 0x80,0xd5,0xdc,0xa3,
				0x6c,0x2f,0xaa,0x03, 0x96,0x4a,0x75,0x33,
				0x4c,0xa8,0x60,0x05, 0x96,0xbf,0xe5,0x7a,
				0xc8,0x4f,0x5c,0x22, 0xf9,0x92,0x74,0x4a,
				0x75,0x5f,0xa2,0x2a, 0x8d,0x3f,0xe2,0x43,
				0xfd,0xd9,0x04,0x8c, 0x8e,0xea,0x84,0xcc,
				0x4d,0x3f,0x94,0x96, 0xed,0x1a,0x51,0xbb,
				0x2f,0xc4,0x63,0x28, 0x31,0x0b,0xda,0x92,
				0x1e,0x4d,0xe2,0x1d, 0x82,0xb5,0x65,0xb4,
				0x75,0x69,0xd7,0x6f, 0x29,0xe4,0xbe,0x7e,
				0xcc,0xbd,0x95,0xbd, 0x7a,0x62,0xea,0xfa,
				0x33,0x34,0x80,0x58, 0xbf,0xfa,0x00,0x7e,
				0xa7,0xb4,0xc9,0x32, 0x7c,0xc7,0x8f,0x8a,
				0x28,0x27,0xdd,0xeb, 0xb9,0x1c,0x01,0xad,
				0xec,0xf4,0x30,0x5e, 0xce,0x3b,0xaa,0x22,
				0x60,0xbd,0x84,0xd9, 0x9e,0xaf,0xe8,0x4c,
				0x44,0xb6,0x84,0x2d, 0x5c,0xe6,0x26,0xee,
				0x8a,0xa2,0x0d,0xe3, 0x97,0xed,0xf5,0x47,
				0xdb,0x50,0x72,0x4a, 0x5e,0x9a,0x8d,0x10,
				0xc2,0x25,0xdd,0x5b, 0xd0,0x39,0xc4,0x5b,
				0x2a,0x79,0x81,0xb7, 0x5c,0xda,0xed,0x77,
				0x17,0x53,0xb5,0x8b, 0x1e,0x5f,0xf3,0x48,
				0x30,0xac,0x97,0x7d, 0x29,0xe3,0xc9,0x18,
				0xe1,0x2b,0x31,0xa0, 0x08,0xe9,0x15,0x59,
				0x29,0xdb,0x84,0x2a, 0x33,0x98,0x8a,0xd4,
				0xc3,0xfc,0xf7,0xca, 0x65,0x02,0x4d,0x9f,
				0xe2,0xb1,0x5e,0xa6, 0x6a,0x01,0xf9,0xcf,
				0x7e,0xa6,0x09,0xd9, 0x16,0x90,0x14,0x5f,
				0x3a,0xf8,0xd8,0x34, 0x38,0xd6,0x1f,0x89,
				0x0c,0x81,0xc2,0x68, 0xc4,0x65,0x78,0xf3,
				0xfe,0x27,0x48,0x70, 0x38,0x43,0x48,0x5a,
				0xc1,0x24,0xc5,0x6f, 0x65,0x63,0x1b,0xb0,
				0x5b,0xb4,0x07,0x1e, 0x69,0x08,0x8f,0xfc,
				0x93,0x29,0x04,0x16, 0x6a,0x8b,0xb3,0x3d,
				0x0f,0xba,0x5f,0x46, 0xff,0xfe,0x77,0xa1,
				0xb9,0xdc,0x29,0x66, 0x9a,0xd1,0x08,0xdd,
				0x32,0xe3,0x21,0x7b, 0xcc,0x2e,0x5c,0xf7,
				0x79,0x68,0xd4,0xc1, 0x8b,0x3c,0x5d,0x0e,
				0xd4,0x26,0xa6,0x19, 0x92,0x45,0xf7,0x19,
				0x0e,0xa2,0x17,0xd8, 0x1c,0x7f,0x8d,0xd6,
				0x68,0x37,0x6c,0xbf, 0xb1,0x8a,0x5e,0x36,
				0x4b,0xc0,0xca,0x21, 0x02,0x24,0x69,0x9b,
				0x2b,0x19,0x0a,0x1b, 0xe3,0x17,0x30,0x57,
				0xf6,0xfc,0xd6,0x66, 0x36,0x30,0xc2,0x11,
				0x08,0x8d,0xc5,0x84, 0x67,0xa0,0x89,0xc3,
				0x74,0x48,0x15,0xca, 0x6e,0x0c,0x6d,0x78,
				0x66,0x15,0x73,0x85, 0xf9,0x8b,0xba,0xb2,
				0x09,0xda,0x79,0xe6, 0x00,0x08,0x2a,0xda,
				0x6b,0xd7,0xd1,0xa7, 0x8b,0x5f,0x11,0x87,
				0x96,0x1b,0x23,0xb0, 0x6c,0x55,0xb6,0x86,
				0xfb,0xff,0xe3,0x69, 0xac,0x43,0xcd,0x8f,
				0x8a,0xe7,0x1c,0x3c, 0xa0,0x6a,0xd5,0x63,
				0x80,0x66,0xd8,0x7f, 0xb5,0xb8,0x96,0xd4,
				0xe2,0x20,0x40,0x53, 0x6d,0x0d,0x8b,0x6d,
				0xd5,0x5d,0x51,0xfb, 0x4d,0x80,0x82,0x01,
				0x14,0x97,0x96,0x9b, 0x13,0xb8,0x1d,0x76,
				0x7a,0xa1,0xca,0x19, 0x90,0xec,0x7b,0xe0,
				0x8e,0xa8,0xb4,0xf2, 0x33,0x67,0x0e,0x10,
				0xb1,0xa2,0x82,0xea, 0x81,0x82,0xa2,0xc6,
				0x78,0x51,0xa6,0xd3, 0x25,0xe4,0x9c,0xf2,
				0x6b,0xa8,0xec,0xfb, 0xd4,0x1d,0x5b,0xa4,
				0x79,0x66,0x62,0xb8, 0x2b,0x6f,0x9e,0x0f,
				0xcc,0xcb,0x9e,0x92, 0x6f,0x06,0xdb,0xf0,
				0x97,0xce,0x3f,0x90, 0xa2,0x1f,0xbe,0x3b,
				0x7b,0x10,0xf0,0x23, 0x30,0x0c,0xc5,0x0c,
				0x6c,0x78,0xfc,0xa8, 0x71,0x62,0xcf,0x98,
				0xa2,0xb1,0x44,0xb5, 0xc6,0x3b,0x5c,0x63,
				0x83,0x1d,0x35,0xf2, 0xc7,0x42,0x67,0x5d,
				0xc1,0x26,0x36,0xc8, 0x6e,0x1d,0xf6,0xd5,
				0x52,0x35,0xa4,0x9e, 0xce,0x4c,0x3b,0x92,
				0x20,0x86,0xb7,0x89, 0x63,0x73,0x1a,0x8b,
				0xa6,0x35,0xfe,0xb9, 0xdf,0x5e,0x0e,0x53,
				0x0b,0xf2,0xb3,0x4d, 0x34,0x1d,0x66,0x33,
				0x1f,0x08,0xf5,0xf5, 0x0a,0xab,0x76,0x19,
				0xde,0x82,0x2f,0xcf, 0x11,0xa6,0xcb,0xb3,
				0x17,0xec,0x8d,0xaf, 0xcb,0xf0,0x92,0x1e,
				0xb8,0xa3,0x04,0x0a, 0xac,0x2c,0xae,0xc5,
				0x0b,0xc4,0x4e,0xef, 0x0a,0xe2,0xda,0xe9,
				0xd7,0x75,0x2d,0x95, 0xc7,0x1b,0xf3,0x0b,
				0x43,0x19,0x16,0xd7, 0xc6,0x90,0x2d,0x6b,
				0xe1,0xb2,0xce,0xbe, 0xd0,0x7d,0x15,0x99,
				0x24,0x37,0xbc,0xb6, 0x8c,0x89,0x7a,0x8c,
				0xcb,0xa7,0xf7,0x0b, 0x5f,0xd4,0x96,0x8d,
				0xf5,0x80,0xa3,0xce, 0xf5,0x9e,0xed,0x60,
				0x00,0x92,0xa5,0x67, 0xc9,0x21,0x79,0x0b,
				0xfb,0xe2,0x57,0x0e, 0xdf,0xb6,0x16,0x90,
				0xd3,0x75,0xf6,0xb0, 0xa3,0x4e,0x43,0x9a,
				0xb7,0xf4,0x73,0xd8, 0x34,0x46,0xc6,0xbe,
				0x80,0xec,0x4a,0xc0, 0x7f,0x9e,0xb6,0xb0,
				0x58,0xc2,0xae,0xa1, 0xf3,0x60,0x04,0x62,
				0x11,0xea,0x0f,0x90, 0xa9,0xea,0x6f,0x0c,
				0x4c,0xcf,0xe8,0xd0, 0xea,0xbf,0xdb,0xf2,
				0x53,0x0c,0x09,0x4d, 0xd4,0xed,0xf3,0x22,
				0x10,0x99,0xc6,0x4f, 0xcf,0xcf,0x96,0xc9,
				0xd9,0x6b,0x08,0x3b, 0xf0,0x62,0x2d,0xac,
				0x55,0x38,0xd5,0x5c, 0x57,0xad,0x51,0xc3,
				0xf5,0xd2,0x37,0x45, 0xb3,0x3f,0x6d,0xaf,
				0x10,0x62,0x57,0xb9, 0x58,0x40,0xb3,0x3c,
				0x6a,0x98,0x97,0x1a, 0x9c,0xeb,0x66,0xf1,
				0xa5,0x93,0x0b,0xe7, 0x8b,0x29,0x0f,0xff,
				0x2c,0xd0,0x90,0xf2, 0x67,0xa0,0x69,0xcd,
				0xd3,0x59,0xad,0xad, 0xf1,0x1f,0xd7,0xad,
				0x24,0x74,0x29,0xcd, 0x06,0xd5,0x42,0x90,
				0xf9,0x96,0x4a,0xd9, 0xa0,0x37,0xe4,0x64,
				0x8e,0x13,0x2a,0x2a, 0xe7,0xc2,0x1e,0xf6,
				0xb2,0xd3,0xdc,0x9f, 0x33,0x32,0x0c,0x50,
				0x88,0x37,0x8b,0x9b, 0xfe,0x6f,0xfd,0x05,
				0x96,0x26,0x6c,0x96, 0x73,0x73,0xe1,0x09,
				0x28,0xf3,0x7f,0xa6, 0x59,0xc5,0x2e,0xf4,
				0xd3,0xd5,0xda,0x6b, 0xca,0x42,0x05,0xe5,
				0xed,0x13,0xe2,0x4e, 0xcd,0xd5,0xd0,0xfb,
				0x6e,0xf7,0x8a,0x3e, 0x91,0x9d,0x6b,0xc5,
				0x33,0x05,0x07,0x86, 0xb2,0x26,0x41,0x6e,
				0xf8,0x38,0x38,0x7a, 0xf0,0x6c,0x27,0x5a,
				0x01,0xd8,0x03,0xe5, 0x91,0x33,0xaa,0x20,
				0xcd,0xa7,0x4f,0x18, 0xa0,0x91,0x28,0x74,
				0xc0,0x58,0x27,0x0f, 0x9b,0xa8,0x85,0xb0,
				0xe0,0xfd,0x5b,0xdb, 0x5b,0xb8,0x86,0x79,
				0x94,0x6d,0xde,0x26, 0x64,0x2d,0x6c,0xb9,
				0xba,0xc7,0xf0,0xd7, 0xaa,0x68,0x68,0xd0,
				0x40,0x71,0xdb,0x94, 0x54,0x62,0xa5,0x7f,
				0x98,0xea,0xe3,0x4c, 0xe4,0x44,0x9a,0x03,
				0xf9,0x1c,0x20,0x36, 0xeb,0x0d,0xa4,0x41,
				0x24,0x06,0xcb,0x94, 0x86,0x35,0x22,0x62,
				0x80,0x19,0x16,0xba, 0x2c,0x10,0x38,0x96,
			},
			.mlen = 16,
			.m = {
				0xd3,0x82,0xe7,0x04, 0x35,0xcc,0xf7,0xa4,
				0xf9,0xb2,0xc5,0xed, 0x5a,0xd9,0x58,0xeb,
			},
			.h = {
				0x41,0xd9,0xad,0x54, 0x5a,0x0d,0xcc,0x53,
				0x48,0xf6,0x4c,0x75, 0x43,0x5d,0xdd,0x77,
				0xda,0xca,0x7d,0xec, 0x91,0x3b,0x53,0x16,
				0x5c,0x4b,0x58,0xdc, 0x70,0x0a,0x7b,0x37,
			},
		},
		[1] = {		/* 1008-byte message */
			.k = {
				0xd9,0x94,0x65,0xda, 0xc2,0x60,0xdd,0xa9,
				0x39,0xe5,0x37,0x11, 0xf6,0x74,0xa5,0x95,
				0x36,0x07,0x24,0x99, 0x64,0x6b,0xda,0xe2,
				0xd5,0xd1,0xd2,0xd9, 0x25,0xd5,0xcc,0x48,
				0xf8,0xa5,0x9e,0xff, 0x84,0x5a,0xd1,0x6f,
				0xb7,0x6a,0x4d,0xd2, 0xc8,0x13,0x3d,0xde,
				0x17,0xed,0x64,0xf1, 0x2b,0xcc,0xdd,0x65,
				0x11,0x16,0xf2,0xaf, 0x34,0xd2,0xc5,0x31,
				0xaa,0x69,0x33,0x0a, 0x0b,0xc1,0xb4,0x6d,
				0xaa,0xcd,0x43,0xc4, 0x0b,0xef,0xf9,0x7d,
				0x97,0x3c,0xa7,0x22, 0xda,0xa6,0x6a,0xf0,
				0xad,0xe3,0x6f,0xde, 0xfb,0x33,0xf3,0xd8,
				0x96,0x5f,0xca,0xda, 0x18,0x63,0x03,0xd0,
				0x8f,0xb6,0xc4,0x62, 0x9d,0x50,0x6c,0x8f,
				0x85,0xdd,0x6d,0x52, 0x2d,0x45,0x01,0x36,
				0x57,0x9f,0x51,0xf0, 0x70,0xe0,0xb2,0x99,
				0x3a,0x11,0x68,0xbd, 0xe5,0xfa,0x7c,0x59,
				0x12,0x5a,0xbc,0xd9, 0xd6,0x9a,0x09,0xe6,
				0xa2,0x80,0x1f,0xd6, 0x47,0x20,0x82,0x4e,
				0xac,0xb5,0x6d,0xde, 0x5b,0xff,0x9c,0xd4,
				0x2a,0xae,0x27,0x7c, 0x0f,0x5a,0x5d,0x35,
				0x2d,0xff,0x07,0xf9, 0x79,0x6a,0xf9,0x3e,
				0xd9,0x22,0x62,0x30, 0x40,0xce,0xe1,0xf4,
				0x46,0x0a,0x24,0xca, 0x7a,0x3e,0xa1,0x92,
				0x1a,0x29,0xa0,0xbf, 0x23,0x95,0x99,0x31,
				0xe3,0x51,0x25,0x3d, 0xaf,0x1e,0xfc,0xb3,
				0x65,0xa2,0x10,0x37, 0xe6,0xa7,0x20,0xa0,
				0xe3,0x6a,0xd4,0x81, 0x2c,0x8d,0xa0,0x87,
				0xec,0xae,0x9f,0x44, 0x10,0xda,0x2e,0x17,
				0xba,0xb2,0xa5,0x5c, 0x89,0xc6,0xfa,0x70,
				0x7e,0xc2,0xe3,0xb6, 0xa0,0x98,0x9c,0xb8,
				0x14,0x33,0x27,0x3a, 0x6e,0x4d,0x94,0x72,
				0x4b,0xc8,0xac,0x24, 0x2f,0x85,0xd9,0xa4,
				0xda,0x22,0x95,0xc5, 0xb3,0xfc,0xbe,0xd2,
				0x96,0x57,0x91,0xf9, 0xfd,0x18,0x9c,0x56,
				0x70,0x15,0x5f,0xe7, 0x40,0x45,0x28,0xb3,
				0x2b,0x56,0x44,0xca, 0x6a,0x2b,0x0e,0x25,
				0x66,0x3e,0x32,0x04, 0xe2,0xb7,0x91,0xc8,
				0xd2,0x02,0x79,0x0f, 0x7e,0xa9,0xb3,0x86,
				0xb2,0x76,0x74,0x18, 0x57,0x16,0x63,0x06,
				0x6e,0x16,0xfa,0xef, 0x52,0x3c,0x5e,0x0d,
				0x33,0x55,0xd2,0x8d, 0x57,0x4d,0xfe,0x54,
				0x65,0x7a,0x54,0x52, 0xf0,0x7b,0x2c,0xf8,
				0xd5,0x43,0xba,0x92, 0xa5,0x2e,0xbe,0x1a,
				0xce,0x25,0x4f,0x34, 0x31,0xe7,0xa3,0xff,
				0x90,0xf6,0xbc,0x0c, 0xbc,0x98,0xdf,0x4a,
				0xc3,0xeb,0xb6,0x27, 0x68,0xa9,0xb5,0x33,
				0xbc,0x13,0xe8,0x13, 0x7c,0x6b,0xec,0x31,
				0xd9,0x79,0x2a,0xa7, 0xe4,0x02,0x4f,0x02,
				0xd4,0x5c,0x57,0x4f, 0xa4,0xbc,0xa3,0xe1,
				0x7e,0x36,0x8a,0xde, 0x11,0x55,0xec,0xb3,
				0x8b,0x65,0x06,0x02, 0x9a,0x68,0x06,0x64,
				0x63,0xc7,0x9a,0x67, 0xdc,0x70,0xbf,0xb5,
				0xf8,0x49,0x2a,0xe1, 0x59,0x4c,0xe4,0x1e,
				0xb5,0x56,0xa5,0xad, 0x24,0x82,0x8c,0xd0,
				0x66,0xe4,0x72,0x79, 0x02,0x5d,0x0d,0xf9,
				0x19,0x44,0xe3,0x86, 0x1a,0xda,0xda,0xf0,
				0x2d,0x47,0xc0,0x07, 0x47,0x0b,0xf8,0x06,
				0xf6,0x45,0x8a,0x7f, 0xb9,0xf9,0x33,0x2e,
				0xc2,0xf1,0xf1,0x81, 0x41,0x99,0xcd,0xf6,
				0xb1,0x71,0x1b,0xfa, 0x21,0x53,0x7c,0xa1,
				0xeb,0x2a,0x38,0x5b, 0x9b,0xfe,0x96,0xa5,
				0xe3,0x78,0x77,0x47, 0x98,0x0f,0x7d,0xef,
				0xf6,0x05,0x37,0x88, 0x79,0x0c,0x21,0x8d,
				0x87,0x1f,0xae,0xce, 0x83,0xaf,0xa3,0xd6,
				0x6e,0xc5,0x3c,0x47, 0xc6,0xd6,0x4a,0xdc,
				0x7c,0xcc,0xdc,0x11, 0x7c,0x7d,0x0f,0x03,
				0xc1,0x80,0x75,0x2a, 0x64,0x76,0xf0,0x08,
				0x0c,0x11,0x4b,0xe4, 0x05,0x41,0x78,0x0f,
				0x86,0xa0,0xd6,0x61, 0xb0,0xfb,0x15,0x3d,
				0x3c,0xc3,0xd5,0x1b, 0x72,0x0e,0x79,0x53,
				0x07,0xd2,0x2c,0x6e, 0x83,0xbd,0x72,0x88,
				0x41,0x07,0x4b,0xd2, 0xe9,0xcc,0x2a,0x9d,
				0x5b,0x82,0x0d,0x02, 0x29,0x6e,0xf3,0xbc,
				0x34,0x31,0x62,0x8d, 0x83,0xc1,0x7e,0x94,
				0x21,0xd5,0xfd,0xa6, 0x6a,0x2b,0xe8,0x86,
				0x05,0x48,0x97,0x41, 0xad,0xca,0xef,0x79,
				0x5e,0xd8,0x51,0xc4, 0xae,0xf7,0xfa,0xac,
				0x3d,0x74,0x2e,0xf4, 0x41,0x3b,0x19,0xc2,
				0x04,0xf3,0x40,0xfe, 0x77,0x7c,0x6a,0x4c,
				0x8e,0x24,0x84,0xe0, 0x70,0xe4,0xb2,0x19,
				0x6c,0x0c,0x85,0x9e, 0xe1,0xad,0xa4,0x73,
				0x90,0xdd,0xbf,0x7d, 0x1b,0x6f,0x8b,0x4d,
				0x3b,0xec,0xd7,0xb0, 0xd9,0x90,0xf1,0xf5,
				0xb9,0x32,0xe3,0x79, 0x15,0x08,0x3e,0x71,
				0xed,0x91,0xc4,0x5c, 0x18,0xe8,0x16,0x52,
				0xae,0x9d,0xf3,0x09, 0xac,0x57,0x11,0xf8,
				0x16,0x55,0xd0,0x28, 0x60,0xc1,0x7e,0x6d,
				0x87,0xc1,0x7a,0xe8, 0x5d,0xc5,0x12,0x68,
				0x6d,0x63,0x39,0x27, 0x49,0xb8,0x0c,0x78,
				0x92,0xea,0x6f,0x52, 0xeb,0x43,0xc2,0x0b,
				0xd8,0x28,0x77,0xe5, 0x43,0x5f,0xb8,0xa6,
				0x32,0xb7,0xaa,0x01, 0x1e,0xa6,0xde,0xe4,
				0x9b,0x0f,0xb6,0x49, 0xcc,0x6f,0x2c,0x04,
				0x41,0xcb,0xd8,0x80, 0xd1,0x15,0x5e,0x57,
				0x1e,0x4a,0x77,0xbf, 0xc4,0xcb,0x09,0x7c,
				0x6e,0x81,0xb8,0x64, 0x51,0x6a,0xf2,0x71,
				0x06,0xf6,0x00,0xac, 0x79,0x2c,0x83,0x7a,
				0x6c,0xa4,0x85,0x89, 0x69,0x06,0x26,0x72,
				0xe1,0x00,0x66,0xc0, 0xc5,0x8e,0xc8,0x51,
				0x6e,0x25,0xdd,0xc9, 0x54,0x98,0x45,0x64,
				0xaa,0x51,0x18,0x1b, 0xe4,0xbe,0x1b,0xee,
				0x13,0xd6,0x34,0x50, 0x4c,0xcf,0x3c,0x31,
				0x9b,0xd2,0x6f,0x07, 0x79,0xf4,0x63,0x3f,
				0x09,0x01,0x64,0xf1, 0xc1,0xf1,0xae,0xa9,
				0x0c,0x60,0xc9,0x62, 0x84,0xf6,0xe8,0x15,
				0x55,0xdf,0xdd,0x71, 0x95,0xa9,0x0f,0x65,
				0x97,0x40,0x79,0x86, 0x95,0xd9,0x57,0x23,
				0x2f,0x61,0x51,0xb5, 0x16,0x18,0x62,0xd2,
				0x1a,0xd9,0x8b,0x88, 0x84,0xa9,0x9b,0x47,
				0xd7,0x22,0x68,0xe9, 0x9c,0x69,0x68,0x74,
				0x13,0x95,0xd3,0x99, 0x33,0xdb,0x30,0x96,
				0xbf,0x01,0xc6,0x68, 0xbd,0x19,0x32,0xc1,
				0xf8,0xa9,0x7f,0x2b, 0xc5,0x69,0x2f,0xa2,
				0xce,0x5a,0x46,0x43, 0x8d,0x36,0x9c,0xfa,
				0x5c,0x7f,0x03,0xe0, 0x80,0xaa,0xc7,0x9e,
				0x3b,0xa3,0x27,0x6b, 0x2e,0xc6,0x59,0x0a,
				0xf6,0x36,0x37,0xa6, 0xc0,0xd1,0xa1,0xa1,
				0x7e,0xc1,0xf8,0x5b, 0x0f,0x9b,0xdd,0x6d,
				0x9f,0x54,0x16,0x6b, 0x6e,0x53,0xfd,0xe8,
				0x72,0xd0,0x3e,0x46, 0xce,0xaf,0x94,0x36,
				0x85,0xa8,0xae,0x4c, 0x8d,0xb5,0xc2,0x1b,
				0x5d,0x29,0x46,0x40, 0x87,0x50,0x59,0xdd,
				0x04,0xbe,0xba,0x8f, 0x0b,0x9b,0xd2,0x50,
				0x67,0x19,0x83,0x80, 0x87,0x5c,0x58,0x86,
				0x20,0x39,0xbf,0xdf, 0xd2,0xc8,0xbb,0xe8,
				0xc8,0xd8,0xe8,0x8d, 0xcc,0x97,0xe0,0xc9,
				0x6c,0x2f,0x47,0xb6, 0x75,0x8f,0x0d,0x37,
				0x5a,0x83,0xb0,0xce, 0x59,0xc2,0x0b,0x84,
				0xa2,0x54,0xe5,0x38, 0x59,0x29,0x0f,0xa8,
				0x26,0x2d,0x11,0xa9, 0x89,0x0e,0x0b,0x75,
				0xe0,0xbc,0xf0,0xf8, 0x92,0x1f,0x29,0x71,
				0x91,0xc4,0x63,0xcc, 0xf8,0x52,0xb5,0xd4,
				0xb8,0x94,0x6a,0x30, 0x90,0xf7,0x44,0xbe,
			},
			.mlen = 1008,
			.m = {
				0x05,0xe3,0x6f,0x44, 0xa4,0x40,0x35,0xf6,
				0xeb,0x86,0xa9,0x6d, 0xed,0x16,0xdb,0xb6,
				0x5b,0x59,0xda,0x30, 0x54,0x6c,0x59,0x35,
				0x42,0x59,0x56,0x45, 0x9a,0x85,0x20,0x73,
				0xcf,0x21,0xf5,0x98, 0x58,0x07,0x0e,0x7f,
				0x44,0x1f,0xf1,0x53, 0x92,0xc7,0x81,0x53,
				0x5e,0x97,0x8a,0x23, 0x1d,0xe8,0xad,0xca,
				0x19,0x55,0x96,0x9d, 0x9b,0xfd,0x0a,0x0a,
				0xad,0xa8,0x0f,0x76, 0xe2,0x6a,0x8f,0x33,
				0x36,0xbf,0xcb,0x7a, 0xfd,0x61,0xc6,0xfb,
				0x75,0xea,0xd4,0x09, 0x5e,0x70,0xfb,0x32,
				0x54,0xe3,0x47,0x48, 0xd4,0x8c,0xa9,0x7c,
				0x72,0xdb,0xdb,0xf7, 0x09,0x6d,0x58,0xa6,
				0x42,0xb5,0x74,0x8c, 0x98,0x66,0x83,0x7a,
				0x6d,0xeb,0x91,0xfb, 0x22,0x1c,0x78,0x3d,
				0x22,0xa6,0xf8,0xb0, 0xd1,0x9f,0xc8,0x69,
				0x8a,0xba,0xd3,0x78, 0x21,0xb0,0x7b,0x9f,
				0xb8,0xed,0xe0,0x65, 0xff,0xa0,0x8b,0x4c,
				0x17,0x9e,0xf7,0x3e, 0xa2,0x5f,0x82,0x77,
				0xce,0x2a,0xda,0x41, 0x76,0x07,0x68,0xa4,
				0xa1,0xbb,0xe0,0x1d, 0x7b,0xab,0x9c,0x03,
				0x90,0x2c,0xd2,0x93, 0x46,0x43,0x3a,0x44,
				0x29,0xe8,0xb5,0x7a, 0x23,0xbb,0xe9,0xaf,
				0x2b,0x17,0x88,0x8f, 0x7a,0x81,0x7a,0x25,
				0x3b,0xc7,0x1e,0x6e, 0xde,0x3e,0x54,0xbc,
				0xc6,0xff,0x07,0xdc, 0xe6,0x29,0x02,0x4c,
				0x95,0x57,0x0e,0x44, 0xc4,0x9c,0xc7,0x45,
				0x01,0xd7,0x17,0xfd, 0x0f,0x1a,0x83,0x74,
				0xa0,0xd5,0xb3,0x1a, 0xc0,0x97,0xdc,0xc3,
				0x0f,0x3d,0x5d,0x8c, 0x02,0x58,0xc6,0x4d,
				0x43,0x10,0xae,0xc9, 0x94,0xe2,0x9b,0xcd,
				0xf9,0xcc,0xfe,0xbd, 0x9c,0x69,0xd0,0xec,
				0xf8,0x67,0xde,0x98, 0xe5,0x50,0x5e,0x93,
				0x6a,0x5b,0x31,0x2a, 0x62,0xee,0x03,0xbe,
				0x76,0x9c,0x1d,0x13, 0x16,0x13,0xcf,0x63,
				0x30,0x18,0x7d,0x1e, 0x55,0x94,0xf5,0x29,
				0xb4,0x91,0xb4,0x76, 0x1c,0x31,0x9e,0xe5,
				0x1b,0x0a,0xee,0x89, 0xb4,0xd9,0x45,0x19,
				0xd7,0x47,0x2c,0x01, 0x20,0xe6,0x1d,0x7c,
				0xb3,0x5e,0x1b,0x2a, 0x8c,0x3d,0x4d,0x1a,
				0x6b,0x35,0x84,0x41, 0x6a,0xe4,0x32,0x8f,
				0x9a,0x0d,0xbf,0x90, 0xff,0xcf,0x4c,0xfb,
				0x9b,0x07,0x81,0x94, 0xcf,0x8e,0x1a,0x8a,
				0xfc,0xbd,0x91,0xfe, 0xc3,0xe1,0x18,0xc7,
				0x1f,0x0d,0x8e,0x1c, 0x2e,0xfc,0x02,0xe8,
				0x39,0xbf,0x05,0x90, 0x58,0x94,0xee,0xe7,
				0x15,0x31,0x5d,0x9f, 0x68,0x36,0x64,0x32,
				0x25,0x49,0xdd,0x3e, 0xc8,0xb6,0x83,0x5e,
				0x09,0x90,0xcd,0x48, 0xaf,0x9e,0xfe,0xd6,
				0x79,0x8e,0x69,0x4b, 0x94,0xd5,0xf4,0x84,
				0x7b,0xce,0xea,0x2f, 0x9b,0x79,0x7a,0x7c,
				0x22,0x28,0x4d,0xa1, 0x38,0x1a,0x66,0x24,
				0x79,0xa3,0xfa,0xfa, 0x8d,0x98,0x7c,0x54,
				0x71,0x54,0xef,0x37, 0xa6,0xf1,0x97,0x54,
				0xad,0xe7,0x67,0xa0, 0xf3,0x33,0xcf,0x4f,
				0x4e,0xa3,0x47,0xee, 0x31,0xd3,0x98,0xf9,
				0x7f,0x9f,0x44,0x18, 0x2f,0x13,0x1b,0x44,
				0x57,0xcd,0x15,0x5b, 0xde,0x8f,0x1a,0x3c,
				0xb5,0x1e,0xa7,0x2d, 0x4d,0xbe,0x85,0x08,
				0x78,0xeb,0xe2,0x35, 0x3a,0xbe,0x55,0x6b,
				0xc3,0xe1,0x0f,0x77, 0x43,0x41,0x11,0x5a,
				0x61,0xc9,0x3b,0xbc, 0xad,0x88,0x9e,0xba,
				0xc6,0xd2,0xdc,0x87, 0xd9,0x54,0xcc,0x86,
				0x46,0xe6,0xa5,0x29, 0x2c,0x08,0x49,0x53,
				0x2c,0xe3,0x0e,0x60, 0xc5,0x48,0xca,0x62,
				0x3f,0xf6,0x93,0xc1, 0xba,0x8d,0x36,0x49,
				0xe7,0x0f,0x9c,0x49, 0x7d,0xee,0x2a,0x22,
				0xc3,0xe5,0x11,0x21, 0xfa,0xc7,0xeb,0x79,
				0xcc,0x4d,0x75,0x4e, 0x66,0x33,0xf5,0x09,
				0xa3,0xb9,0x60,0xa5, 0xd6,0xbd,0x38,0x75,
				0x0c,0x2f,0x5f,0x1f, 0xea,0xa5,0x9d,0x45,
				0x3c,0xe4,0x41,0xb8, 0xf6,0x4e,0x15,0x87,
				0x0b,0x7f,0x42,0x4e, 0x51,0x3d,0xc4,0x9a,
				0xb2,0xca,0x37,0x16, 0x0f,0xed,0x9e,0x0b,
				0x93,0x86,0x12,0x93, 0x36,0x5e,0x39,0xc4,
				0xf0,0xf4,0x48,0xdb, 0xeb,0x18,0x5e,0x50,
				0x71,0x30,0x83,0xe5, 0x0f,0xb1,0x73,0xa7,
				0xc6,0xf0,0xca,0x29, 0x0e,0xc4,0x07,0x5b,
				0x8b,0x09,0x68,0x68, 0x10,0x32,0x92,0x62,
				0x6a,0x6c,0x56,0x8b, 0x01,0x46,0x9a,0x20,
				0x89,0xe0,0x93,0x85, 0x8c,0x53,0x87,0xf6,
				0x02,0xd3,0x8d,0x72, 0x31,0x35,0xa1,0x34,
				0x63,0x70,0x61,0x80, 0x06,0xf1,0x54,0xb3,
				0x5d,0xdf,0xad,0x9c, 0x7e,0x3a,0xc2,0x8f,
				0x76,0x8b,0x4c,0x74, 0x2c,0x8c,0x6f,0x0a,
				0x60,0x13,0xa8,0xce, 0x4c,0x49,0x70,0x90,
				0x59,0x57,0xf5,0x7b, 0x03,0x94,0x37,0x87,
				0xfa,0xfe,0xeb,0xe7, 0x2d,0x01,0x45,0x69,
				0xb4,0x10,0x80,0x6d, 0x13,0x26,0xe3,0x9b,
				0x49,0x2a,0x0b,0xb1, 0x36,0xf9,0x62,0x63,
				0x33,0x2a,0xee,0x51, 0x5e,0x35,0xa4,0x2e,
				0x34,0xa1,0x77,0xac, 0x27,0x99,0x03,0xc6,
				0xe2,0x83,0x11,0x72, 0x77,0x30,0x8b,0xb7,
				0xde,0x1a,0xa1,0x4b, 0xa9,0x9c,0x07,0x02,
				0xf2,0xdc,0x06,0x45, 0xf2,0xab,0x31,0x46,
				0x50,0x25,0x34,0x54, 0xa8,0x06,0x88,0x6c,
				0xfc,0x88,0xb5,0xae, 0x30,0xbd,0xe1,0xe7,
				0xfe,0x51,0x46,0x05, 0x9a,0x29,0xd9,0x93,
				0x99,0x60,0x69,0x4a, 0x5c,0xb2,0x29,0x6b,
				0xa1,0xbb,0x9d,0xe4, 0x9b,0x7d,0x4a,0xe5,
				0x37,0xcb,0x16,0x6f, 0x44,0x93,0xe4,0x71,
				0x34,0x7b,0x54,0xec, 0x5b,0x2b,0xe0,0xf7,
				0x32,0xed,0x77,0xa6, 0xb3,0x7c,0x8d,0x1a,
				0xc0,0x57,0xbe,0x2b, 0x6d,0x7f,0xd7,0x35,
				0xe6,0x93,0xed,0x90, 0x26,0xfe,0x41,0xf3,
				0x58,0x55,0x03,0xb7, 0xb2,0x94,0xe2,0x0c,
				0x34,0xc3,0x06,0xc6, 0x9e,0x4b,0x17,0xc7,
				0xb9,0x58,0x23,0x58, 0xd3,0x73,0x18,0x5e,
				0xcf,0x28,0xac,0x90, 0xa0,0xba,0x35,0x90,
				0x96,0xb3,0xc7,0x6c, 0xe1,0x07,0xdf,0x5d,
				0xaa,0x2c,0xa6,0x6b, 0x82,0x2d,0x71,0x66,
				0xb7,0x76,0x37,0xdb, 0x39,0x7f,0x22,0x8f,
				0x38,0x70,0xd4,0xeb, 0xf8,0xf0,0x73,0xed,
				0xb6,0x67,0x75,0xaf, 0xd7,0x5d,0x01,0x01,
				0xc4,0xd6,0x7c,0xbc, 0xc3,0xe6,0xad,0x9a,
				0x9c,0x6a,0x43,0x9b, 0xfb,0x34,0x55,0x47,
				0xcd,0xeb,0x4e,0x2c, 0x29,0x6f,0xb0,0xeb,
				0xb5,0x08,0xdb,0x6b, 0x40,0x26,0x51,0x54,
				0x5a,0x97,0x64,0x74, 0x95,0xe6,0xae,0x8a,
				0x4c,0xe9,0x44,0x47, 0x85,0xd6,0xcf,0xe0,
				0x11,0x65,0x45,0xb3, 0xe1,0xfc,0x6a,0x01,
				0x38,0x40,0x8a,0x71, 0xc5,0xd6,0x64,0xa8,
				0x36,0x95,0x44,0x9c, 0x10,0x41,0xa3,0x71,
				0xb4,0x70,0x02,0xdf, 0xf9,0xad,0x2b,0xec,
				0x75,0xf7,0x09,0x6c, 0x5d,0x2a,0xd0,0x0b,
				0x2e,0xb3,0xf0,0xd3, 0xce,0xdb,0x26,0x80,
			},
			.h = {
				0x2d,0xb3,0x7e,0x73, 0xde,0x6a,0x9e,0xa9,
				0x54,0x9a,0x0f,0xb3, 0x0b,0xcc,0xc9,0xde,
				0x7a,0x4e,0x4a,0x71, 0x07,0x33,0xee,0x06,
				0x5c,0x9a,0xa1,0x30, 0x5e,0x39,0x4e,0x10,
			},
		},
		[2] = {		/* 1024-byte message */
			.k = {
				0x4c,0xe4,0x3c,0x6e, 0xa0,0xe3,0x0e,0x64,
				0x35,0x44,0x3e,0x0b, 0x4d,0x29,0xbe,0x04,
				0xa7,0xaa,0x88,0xe0, 0xe0,0x07,0x7d,0xa8,
				0x2b,0x87,0x7d,0x08, 0xa6,0x59,0xd0,0xa5,
				0x03,0xae,0x9b,0xee, 0xd4,0x11,0x39,0x7d,
				0x9e,0x1d,0x89,0xe3, 0xc6,0x92,0x36,0x07,
				0xa4,0x43,0xad,0x2f, 0xd5,0x71,0x84,0x2d,
				0xc0,0x37,0xed,0x62, 0x4e,0x2b,0x8c,0xd5,
				0x1d,0xf7,0x00,0xbb, 0x3d,0x5e,0xcc,0xc5,
				0x6d,0xdd,0x17,0xf2, 0x89,0x25,0x30,0x16,
				0x04,0xd7,0x1f,0x84, 0x7d,0x61,0xa0,0x7a,
				0x49,0x88,0x44,0x46, 0xc6,0x05,0xd1,0xc9,
				0xa0,0x2a,0x86,0xdd, 0xd3,0x80,0x40,0xa4,
				0x28,0xb3,0xa4,0x3b, 0x71,0x0a,0x7f,0x2d,
				0x3b,0xcd,0xe6,0xac, 0x59,0xda,0x43,0x56,
				0x6e,0x9a,0x3f,0x1e, 0x82,0xcf,0xb3,0xa0,
				0xa1,0x46,0xcf,0x2e, 0x32,0x05,0xcd,0x68,
				0xbb,0x51,0x71,0x8a, 0x16,0x75,0xbe,0x49,
				0x7e,0xb3,0x63,0x30, 0x95,0x34,0xe6,0x85,
				0x7e,0x9a,0xdd,0xe6, 0x43,0xd6,0x59,0xf8,
				0x6a,0xb8,0x8f,0x5f, 0x5d,0xd9,0x55,0x41,
				0x12,0xf9,0x98,0xc6, 0x93,0x7c,0x3f,0x46,
				0xab,0x7c,0x8b,0x28, 0xde,0x9a,0xb1,0xf0,
				0x6c,0x43,0x2a,0xb3, 0x70,0xc5,0x9d,0xc0,
				0x26,0xcf,0xad,0x9c, 0x87,0x9b,0x3f,0x7c,
				0x24,0xac,0xe7,0xd4, 0xe8,0x14,0xe3,0x3e,
				0xf6,0x8a,0x97,0x87, 0x63,0x2c,0x88,0xdc,
				0xc5,0x23,0x68,0x6e, 0x94,0xe1,0x09,0xc4,
				0x44,0xda,0x8f,0xa7, 0x9f,0xc4,0x52,0xa4,
				0x18,0x1d,0x3c,0x08, 0xca,0x0a,0x3e,0xb4,
				0xbf,0xbe,0xc6,0x47, 0xe2,0x89,0x2b,0x07,
				0x71,0xd9,0xc8,0x6a, 0x06,0xd5,0xd0,0x47,
				0x4e,0x07,0x4f,0x6b, 0xdb,0xdf,0x3d,0xf0,
				0x7c,0x5f,0x49,0x70, 0x17,0x4f,0x9f,0x33,
				0x7e,0x4b,0x72,0x3b, 0x8c,0x68,0x22,0xf9,
				0xd2,0xad,0xe4,0xe4, 0xb2,0x61,0x9d,0xb8,
				0xc2,0x5c,0xf0,0x3b, 0x08,0xb2,0x75,0x30,
				0x3a,0xd0,0x7d,0xf9, 0xb2,0x00,0x40,0x56,
				0x79,0xe2,0x0d,0x31, 0x72,0xe2,0xc2,0xd1,
				0x2e,0x27,0xe7,0xc8, 0x96,0x1a,0xc6,0x7e,
				0xb8,0xc1,0x93,0xfb, 0x1d,0xbc,0xed,0x97,
				0x2f,0x2f,0xea,0xa1, 0x40,0x49,0xf6,0x1d,
				0xab,0x54,0x46,0x2e, 0x73,0xf2,0x74,0xf1,
				0x6d,0x5c,0xe6,0xa0, 0xd4,0x73,0x1c,0xbc,
				0x07,0x81,0xf5,0x94, 0xe6,0x18,0xdc,0x42,
				0x68,0xb9,0xeb,0xfb, 0xa3,0x76,0x8c,0x83,
				0x98,0xe9,0x96,0xa6, 0xa6,0x5e,0x0e,0xd1,
				0xfc,0xb7,0x8e,0x8b, 0x9e,0xa4,0x00,0x76,
				0x0e,0x35,0x92,0x5e, 0x05,0xa1,0x92,0xc4,
				0x0c,0xd1,0xec,0x8c, 0x04,0x8e,0x65,0x56,
				0x43,0xae,0x16,0x18, 0x2e,0x3e,0xfe,0x47,
				0x92,0xe1,0x76,0x1b, 0xb6,0xcc,0x0b,0x82,
				0xe1,0x8c,0x7b,0x43, 0xe4,0x90,0xed,0x28,
				0x0b,0xe6,0x05,0xea, 0x4a,0xc0,0xf1,0x12,
				0x54,0x09,0x93,0xda, 0xfc,0xf4,0x86,0xff,
				0x4c,0xaa,0x7d,0xbe, 0xd0,0x4a,0xa6,0x9d,
				0x6b,0x27,0x8f,0xb1, 0xb5,0x3a,0x9b,0xce,
				0xe2,0x5c,0x29,0x35, 0xd6,0xe7,0xf3,0xa4,
				0x5e,0x70,0xf6,0xc6, 0xde,0x63,0x86,0xf7,
				0xc9,0xab,0x42,0xb9, 0xe7,0x5d,0x1c,0x68,
				0x73,0xa3,0xed,0xb0, 0xa0,0xb6,0x18,0x15,
				0xe6,0x57,0x4c,0x21, 0xf7,0xf3,0xc6,0x32,
				0x4d,0x07,0x4a,0x14, 0xde,0xb2,0xc7,0xca,
				0xf0,0x78,0xc4,0x85, 0xe3,0xdc,0xfb,0x35,
				0x7c,0x6b,0xc0,0xb8, 0xcd,0x7a,0x22,0xfc,
				0xe4,0xe8,0xe2,0x98, 0x6c,0x8e,0xdf,0x37,
				0x8e,0x0f,0x25,0x23, 0xdd,0xea,0x40,0x6f,
				0xb3,0x07,0x7e,0x7a, 0x6b,0xa1,0xa1,0xcf,
				0x24,0xd9,0xad,0x72, 0x7a,0x45,0x49,0xca,
				0xfe,0xc7,0x2e,0x6d, 0xaa,0xc1,0x08,0x2c,
				0xe6,0xde,0xde,0x73, 0x01,0x9c,0xdc,0x65,
				0x3a,0xdf,0xc6,0x15, 0x37,0x62,0x0b,0x2c,
				0x9a,0x36,0xed,0x37, 0xd9,0xfc,0xa9,0xb3,
				0x32,0xc3,0xde,0x26, 0xe7,0xf0,0x3f,0x02,
				0xed,0x35,0x74,0xea, 0xdd,0x32,0xe9,0x96,
				0x75,0x66,0xb8,0xf0, 0x75,0x98,0x8f,0x3a,
				0xd0,0xc2,0xa1,0x98, 0x5f,0xf9,0x32,0x31,
				0x00,0x18,0x7d,0xc5, 0x9d,0x15,0x5b,0xdc,
				0x13,0x37,0x69,0xfc, 0x95,0x7a,0x62,0x0e,
				0x8a,0x86,0xed,0x18, 0x78,0x3c,0x49,0xf4,
				0x18,0x73,0xcd,0x2e, 0x7b,0xa3,0x40,0xd7,
				0x01,0xf6,0xc7,0x2a, 0xc5,0xce,0x13,0x09,
				0xb1,0xe5,0x25,0x17, 0xdf,0x9d,0x7e,0x0b,
				0x50,0x46,0x62,0x78, 0xb5,0x25,0xb2,0xd9,
				0x65,0xfa,0x5b,0xf7, 0xfe,0xc6,0xe0,0x7b,
				0x7b,0x4e,0x14,0x2e, 0x0d,0x3a,0xd0,0xe0,
				0xa0,0xd2,0xeb,0x4d, 0x87,0x11,0x42,0x28,
				0x02,0x7e,0xa8,0x56, 0x5b,0x53,0xbd,0x76,
				0x47,0x8f,0x5f,0x8b, 0xc7,0xd9,0x72,0xf7,
				0x11,0xbb,0x94,0xdb, 0x0d,0x07,0xb7,0x0a,
				0xcc,0x41,0x00,0xcd, 0xd0,0x50,0x25,0x31,
				0xc9,0x47,0x6b,0xdd, 0x3f,0x70,0x24,0x3e,
				0xde,0x02,0x62,0x6c, 0xb4,0x44,0x92,0x8e,
				0x98,0x9c,0x0e,0x30, 0x2f,0x80,0xb9,0x5e,
				0x75,0x90,0xa6,0x02, 0xf0,0xed,0xb0,0x8b,
				0x44,0xa3,0x59,0x2d, 0xc3,0x08,0xe5,0xd9,
				0x89,0x6a,0x71,0x44, 0x04,0xc4,0xb2,0x61,
				0x5b,0xf5,0x46,0x44, 0xdc,0x36,0x2e,0xfd,
				0x41,0xf5,0xa1,0x3a, 0xb3,0x93,0x74,0x7d,
				0x54,0x5e,0x64,0xdc, 0xbc,0xd7,0x07,0x48,
				0x3e,0x73,0x81,0x22, 0x9c,0x5a,0xf6,0xde,
				0x94,0x42,0xe1,0x6c, 0x92,0xe7,0x6d,0xa0,
				0x5e,0xc3,0xd6,0xe9, 0x84,0xd9,0xba,0x57,
				0xef,0x85,0x6a,0x9b, 0xe6,0x9a,0x2b,0xf8,
				0x8d,0xfe,0x9d,0xad, 0x70,0x26,0x05,0x14,
				0x45,0x07,0xcb,0x72, 0xd4,0x8b,0x14,0x44,
				0x74,0x40,0x9c,0x29, 0x8b,0xba,0x40,0x09,
				0x52,0xfc,0xc5,0x40, 0xb1,0x25,0x69,0xaa,
				0x8f,0x12,0xc4,0xc6, 0x2b,0x3f,0x73,0x9d,
				0xff,0x52,0xd4,0xac, 0x77,0x43,0xdc,0xd2,
				0x06,0x9a,0x1b,0xfc, 0x0c,0x8f,0x6b,0x59,
				0xa5,0xd4,0xde,0x06, 0x16,0x34,0xef,0x75,
				0x22,0x54,0x9c,0x53, 0x38,0x0b,0x57,0xc7,
				0xaa,0x78,0x2d,0x3a, 0x9b,0xdd,0xed,0xb5,
				0x0b,0xb0,0x08,0x5f, 0x57,0xdb,0xfc,0xbe,
				0x44,0xfd,0x71,0x5f, 0x71,0x14,0xd5,0x14,
				0x70,0xb6,0xee,0xd0, 0xf3,0x37,0x6f,0x57,
				0x55,0x3c,0x7c,0x23, 0x6f,0xbe,0x83,0x5c,
				0xb5,0x64,0xfd,0x6d, 0x7c,0xe4,0x05,0x2b,
				0xdb,0xc4,0xf5,0xa0, 0xd3,0xa6,0x15,0x48,
				0xc2,0x50,0xf8,0xf7, 0xc2,0xab,0xb5,0x6a,
				0x0d,0x1a,0xb5,0x30, 0x33,0xf8,0x12,0x2d,
				0xfb,0xa6,0x2e,0xe5, 0xbe,0x40,0xba,0x48,
				0xef,0x05,0xc8,0x37, 0x3a,0x36,0xad,0x99,
				0x77,0x87,0x84,0xac, 0xd8,0xcb,0x7a,0x88,
				0x3e,0x2d,0x8b,0xbe, 0x9a,0x35,0x88,0x26,
				0xe9,0x20,0xd4,0x66, 0x80,0x8b,0xf8,0x54,
				0xba,0xcd,0xa8,0x47, 0x35,0x1b,0xc4,0x09,
				0x6d,0xff,0x0e,0x60, 0x7c,0xf3,0x68,0xbf,
				0xe3,0xe9,0x73,0x07, 0x84,0xf0,0x08,0x45,
				0x97,0x65,0x94,0xd1, 0x35,0x4e,0x67,0x0c,
				0xe3,0xb7,0x61,0x7b, 0x09,0x22,0xed,0x18,
				0xee,0x0b,0x54,0xc0, 0xab,0x8b,0xaa,0x71,
				0x4c,0x40,0xbf,0xf7, 0xe0,0x7e,0x08,0xaa,
			},
			.mlen = 1024,
			.m = {
				0x1d,0xea,0xe5,0x2b, 0x4c,0x22,0x4d,0xf3,
				0x15,0x53,0xcb,0x41, 0xf5,0xcf,0x0b,0x7b,
				0xc9,0x80,0xc0,0x95, 0xd2,0x7b,0x08,0x4b,
				0x3d,0xcd,0xd8,0x3b, 0x2f,0x18,0xd4,0x70,
				0x38,0xb2,0xa7,0x2f, 0x7f,0xba,0xd8,0xed,
				0xbc,0x8f,0xac,0xe4, 0xe2,0x11,0x2d,0x6d,
				0xe6,0xa4,0x36,0x90, 0xc2,0x7f,0xdf,0xe3,
				0xdc,0x50,0xdb,0x6c, 0x56,0xcf,0x7d,0xd6,
				0xd0,0xcb,0xd6,0x9b, 0x01,0xbb,0xef,0x1c,
				0x0a,0x6c,0x92,0x23, 0xeb,0x77,0xf9,0xd1,
				0x25,0xdc,0x94,0x30, 0x30,0xa4,0x96,0x3e,
				0xdf,0x52,0x4c,0xe7, 0xdf,0x27,0x9f,0x73,
				0x78,0x0c,0x8c,0x7f, 0x9d,0xae,0x79,0x5d,
				0x91,0x5e,0x4b,0x02, 0xa9,0x31,0x9c,0xff,
				0x46,0x73,0xec,0x0d, 0x5a,0xb8,0xeb,0x48,
				0x19,0x9c,0x44,0xe0, 0xc8,0x81,0x96,0x4c,
				0x47,0x0c,0xe7,0x1d, 0x2a,0x9c,0xd5,0xe0,
				0xe7,0xd6,0xa0,0x88, 0xf0,0xf6,0xda,0xa7,
				0x6a,0xdd,0xfd,0x4f, 0x00,0x6e,0x25,0x7d,
				0xb9,0x81,0x19,0x2f, 0x4e,0xcc,0x8d,0x6e,
				0xa6,0x92,0xcf,0xd8, 0x6e,0x78,0x0a,0xf6,
				0x8a,0x43,0xeb,0x60, 0x0c,0x8b,0x93,0x50,
				0x88,0xd1,0x67,0x05, 0x0c,0xdc,0x43,0x85,
				0x50,0x91,0x63,0xa4, 0x32,0x14,0x66,0x84,
				0xdb,0x04,0x9f,0x77, 0x95,0x60,0x19,0xc6,
				0x98,0x60,0x62,0xe4, 0xc6,0xee,0x70,0x76,
				0xb0,0x59,0x80,0x59, 0x46,0xae,0x99,0x26,
				0x62,0x4a,0xf0,0x45, 0x8f,0xf0,0x70,0x5b,
				0x52,0xfc,0xee,0x4d, 0x30,0x47,0xc8,0xae,
				0xe2,0xbc,0x2c,0x73, 0x78,0x67,0xf1,0x00,
				0xb4,0xda,0x01,0xad, 0x3b,0xc4,0x5c,0x6c,
				0x65,0xca,0x84,0x22, 0x95,0x32,0x95,0x20,
				0x4d,0xdc,0x96,0x2e, 0x61,0xe4,0xc8,0xec,
				0x2d,0xbf,0xc1,0x5d, 0x70,0xf9,0x75,0xf2,
				0xad,0x0a,0xc9,0xd7, 0x0a,0x81,0x3c,0xa1,
				0x13,0xec,0x63,0xd4, 0xd0,0x67,0xf4,0xcc,
				0x6e,0xb8,0x52,0x08, 0x46,0xc9,0x2a,0x92,
				0x59,0xd9,0x14,0x17, 0xde,0x2f,0xc7,0x36,
				0xd5,0xd5,0xfc,0x8a, 0x63,0xd5,0x5f,0xe3,
				0xdd,0x55,0x00,0x8e, 0x5e,0xc9,0xed,0x04,
				0x1d,0xeb,0xae,0xc5, 0xd0,0xf9,0x73,0x28,
				0xf3,0x81,0xd5,0xb4, 0x60,0xb2,0x42,0x81,
				0x68,0xf3,0xb9,0x73, 0x07,0x2e,0x34,0x8e,
				0x47,0x12,0xae,0x7c, 0xa8,0xc2,0xce,0xad,
				0x0f,0x6e,0x44,0xa5, 0x35,0x5e,0x61,0x6b,
				0xfc,0x67,0x9c,0x82, 0xa1,0xd2,0xff,0xfe,
				0x60,0x7c,0x40,0x02, 0x24,0x9e,0x8b,0x90,
				0xa0,0x89,0xd9,0x83, 0x04,0xd8,0xef,0x9c,
				0x96,0x28,0x77,0x3e, 0xe3,0xb0,0xf8,0x3d,
				0xfb,0x91,0x8f,0x6f, 0x83,0x58,0x1e,0x4b,
				0x64,0xc7,0xf6,0xe0, 0x85,0x03,0xe3,0xf9,
				0x6b,0xc9,0x9e,0x9d, 0x57,0x25,0xe4,0x69,
				0x08,0x59,0x28,0x4a, 0x52,0x9c,0x49,0x19,
				0x24,0x49,0xba,0xb1, 0x82,0xd4,0xcf,0xd0,
				0x1e,0x1d,0xc2,0x02, 0x42,0x4e,0xdf,0xf7,
				0x2b,0x3d,0x99,0xf6, 0x99,0xa4,0x3a,0xe1,
				0x9d,0x68,0xc8,0x08, 0xec,0xec,0x1c,0xa8,
				0x41,0x4a,0x27,0x84, 0xe9,0x0d,0x95,0x54,
				0x1a,0xca,0x5f,0x5d, 0x5a,0x96,0xb9,0x5b,
				0x6e,0xbc,0x39,0x7f, 0x7a,0x20,0xc5,0xb2,
				0x60,0x0c,0xa3,0x78, 0xc3,0x2b,0x87,0xcc,
				0xea,0xb0,0x4d,0x27, 0xfb,0x6c,0x58,0x51,
				0xce,0x90,0xca,0xd6, 0x86,0x91,0x4d,0x2c,
				0x8c,0x82,0xf0,0xc9, 0x9a,0x0a,0x73,0xb3,
				0xcb,0xa9,0xd4,0x26, 0x4d,0x74,0xbe,0x0e,
				0x4a,0x6e,0x10,0xeb, 0x4e,0xba,0x4e,0xba,
				0x0d,0x26,0x69,0x87, 0x5e,0x08,0x2b,0x43,
				0xbe,0x97,0x4e,0x2a, 0x63,0xbc,0x52,0xb7,
				0xda,0x23,0x23,0x11, 0xfa,0xcf,0x89,0xac,
				0x90,0x5f,0x60,0x7a, 0x50,0xb7,0xbe,0x79,
				0x0b,0x2c,0xf0,0x27, 0xf0,0xfb,0xaf,0x64,
				0xc8,0x57,0x7c,0xeb, 0x1c,0xf7,0x36,0xec,
				0x09,0x97,0x66,0x31, 0x54,0xe4,0x00,0xcf,
				0x68,0x24,0x77,0x1a, 0xbc,0x27,0x3a,0xad,
				0x8a,0x01,0x7e,0x45, 0xe7,0xe4,0xa4,0xeb,
				0x38,0x62,0x9d,0x90, 0xea,0x00,0x9c,0x03,
				0x5e,0xb2,0x7d,0xd8, 0x2f,0xe9,0xc9,0x3c,
				0x1a,0x5c,0x21,0x1a, 0x59,0x45,0x62,0x47,
				0x93,0x1b,0xdc,0xd8, 0x3e,0x07,0x8b,0x75,
				0xd0,0x6d,0xcc,0x8d, 0xec,0x79,0xa8,0x9a,
				0x51,0xa5,0x50,0x18, 0xae,0x44,0x93,0x75,
				0xc1,0xc8,0x1e,0x10, 0x59,0x1e,0x0b,0xb3,
				0x06,0x30,0xa8,0x66, 0x8d,0x8e,0xd6,0x4d,
				0x0d,0x8a,0xb4,0x28, 0xdc,0xfb,0x5d,0x59,
				0xe0,0x92,0x77,0x38, 0xfa,0xad,0x46,0x46,
				0x25,0x15,0x4c,0xca, 0x09,0x2b,0x31,0xe9,
				0x36,0xe8,0xc2,0x67, 0x34,0x4d,0x5e,0xa0,
				0x8f,0x9a,0xe8,0x7f, 0xf2,0x2a,0x92,0x78,
				0xde,0x09,0x75,0xe7, 0xe5,0x50,0x0a,0x2e,
				0x88,0x63,0xc0,0x8f, 0xa8,0x73,0x0f,0xe5,
				0x1e,0x9d,0xdb,0xce, 0x53,0xe0,0x42,0x94,
				0x7b,0x5c,0xa1,0x5e, 0x1e,0x8f,0x0a,0x6e,
				0x8b,0x1a,0xad,0x93, 0x70,0x86,0xf1,0x69,
				0x70,0x93,0x24,0xe3, 0x83,0x2f,0xa8,0x04,
				0xba,0x27,0x0a,0x2e, 0x03,0xeb,0x69,0xd9,
				0x56,0x0e,0xc4,0x10, 0x55,0x31,0x2c,0x3f,
				0xd1,0xb2,0x94,0x0f, 0x28,0x15,0x3c,0x02,
				0x15,0x5e,0xec,0x26, 0x9c,0xc3,0xfc,0xa7,
				0x5c,0xb0,0xfa,0xc0, 0x02,0xf9,0x01,0x3f,
				0x01,0x73,0x24,0x22, 0x50,0x28,0x2a,0xca,
				0xb1,0xf2,0x03,0x00, 0x2f,0xc6,0x6f,0x28,
				0x4f,0x4b,0x4f,0x1a, 0x9a,0xb8,0x16,0x93,
				0x31,0x60,0x7c,0x3d, 0x35,0xc8,0xd6,0x90,
				0xde,0x8c,0x89,0x39, 0xbd,0x21,0x11,0x05,
				0xe8,0xc4,0x04,0x3b, 0x65,0xa5,0x15,0xcf,
				0xcf,0x15,0x14,0xf6, 0xe7,0x2e,0x3c,0x47,
				0x59,0x0b,0xaa,0xc0, 0xd4,0xab,0x04,0x14,
				0x9c,0xd7,0xe2,0x43, 0xc7,0x87,0x09,0x03,
				0x27,0xd2,0x0a,0xff, 0x8d,0xd5,0x80,0x34,
				0x93,0xa2,0x2c,0xb1, 0x4e,0x16,0x2d,0x82,
				0x51,0x5c,0x3c,0xe5, 0x75,0x51,0x7b,0xb4,
				0xd8,0x1e,0x59,0x98, 0x0f,0x75,0xed,0x02,
				0x1c,0x13,0xf6,0x02, 0xda,0xf9,0x47,0xf7,
				0x45,0x25,0x0f,0x58, 0x22,0x5d,0xef,0xf0,
				0x1b,0xdb,0xae,0xaf, 0xbe,0xc6,0xe1,0xcd,
				0x70,0x46,0x6e,0x03, 0x9a,0x20,0x77,0x00,
				0x3c,0x32,0xb5,0x8f, 0x04,0xb6,0x6f,0xa2,
				0x31,0xc9,0x7c,0xf9, 0x84,0x67,0x87,0xfb,
				0x7b,0x13,0xb0,0x4d, 0x35,0xfd,0x37,0x5b,
				0xf4,0x25,0xf0,0x02, 0x74,0xa0,0x69,0xd4,
				0x53,0x61,0x4b,0x54, 0x68,0x94,0x0e,0x08,
				0x25,0x82,0x90,0xfc, 0x25,0xb6,0x63,0xe2,
				0x07,0x9f,0x42,0xf1, 0xbb,0x33,0xea,0xab,
				0x92,0x54,0x2b,0x9f, 0x88,0xc0,0x31,0x2b,
				0xfd,0x36,0x50,0x80, 0xfc,0x1a,0xff,0xab,
				0xe8,0xc4,0x7f,0xb6, 0x98,0xb9,0x2e,0x17,
				0xca,0x28,0x3d,0xdf, 0x0f,0x07,0x43,0x20,
				0xf0,0x07,0xea,0xe5, 0xcd,0x4e,0x81,0x34,
			},
			.h = {
				0x9d,0x22,0x88,0xfd, 0x41,0x43,0x88,0x45,
				0x34,0xfe,0x85,0xc4, 0xb9,0xff,0xe1,0x55,
				0x40,0x1d,0x25,0x37, 0xd1,0xf8,0xfc,0x2b,
				0x3a,0xf5,0x3b,0x69, 0xbf,0xa6,0x9d,0xed,
			},
		},
	};
	static uint32_t k[268];
	uint8_t h[32];
	unsigned i, j;
	int result = 0;

	for (i = 0; i < __arraycount(C); i++) {
		for (j = 0; j < 268; j++)
			k[j] = le32dec(C[i].k + 4*j);
		nh(h, C[i].m, C[i].mlen, k);
		if (memcmp(h, C[i].h, 32)) {
			char prefix[10];
			snprintf(prefix, sizeof prefix, "nh %u", i);
			hexdump(printf, prefix, h, 32);
			result = -1;
		}
	}

	return result;
}

/* https://github.com/google/adiantum/blob/a5ad5134ab11b10a3ee982c52385953fac88fedc/test_vectors/ours/NHPoly1305/NHPoly1305.json */
static int
nhpoly1305_selftest(void)
{
	static const struct {
		uint8_t k[1088];
		unsigned mlen;
		uint8_t m[1024];
		uint8_t h[16];
	} C[] = {
		[0] = {		/* 0-byte message */
			.k = {
				/* Poly1305 key */
				0xd2,0x5d,0x4c,0xdd, 0x8d,0x2b,0x7f,0x7a,
				0xd9,0xbe,0x71,0xec, 0xd1,0x83,0x52,0xe3,

				/* NH key */
				0xe1,0xad,0xd7,0x5c, 0x0a,0x75,0x9d,0xec,
				0x1d,0x13,0x7e,0x5d, 0x71,0x07,0xc9,0xe4,
				0x57,0x2d,0x44,0x68, 0xcf,0xd8,0xd6,0xc5,
				0x39,0x69,0x7d,0x32, 0x75,0x51,0x4f,0x7e,
				0xb2,0x4c,0xc6,0x90, 0x51,0x6e,0xd9,0xd6,
				0xa5,0x8b,0x2d,0xf1, 0x94,0xf9,0xf7,0x5e,
				0x2c,0x84,0x7b,0x41, 0x0f,0x88,0x50,0x89,
				0x30,0xd9,0xa1,0x38, 0x46,0x6c,0xc0,0x4f,
				0xe8,0xdf,0xdc,0x66, 0xab,0x24,0x43,0x41,
				0x91,0x55,0x29,0x65, 0x86,0x28,0x5e,0x45,
				0xd5,0x2d,0xb7,0x80, 0x08,0x9a,0xc3,0xd4,
				0x9a,0x77,0x0a,0xd4, 0xef,0x3e,0xe6,0x3f,
				0x6f,0x2f,0x9b,0x3a, 0x7d,0x12,0x1e,0x80,
				0x6c,0x44,0xa2,0x25, 0xe1,0xf6,0x60,0xe9,
				0x0d,0xaf,0xc5,0x3c, 0xa5,0x79,0xae,0x64,
				0xbc,0xa0,0x39,0xa3, 0x4d,0x10,0xe5,0x4d,
				0xd5,0xe7,0x89,0x7a, 0x13,0xee,0x06,0x78,
				0xdc,0xa4,0xdc,0x14, 0x27,0xe6,0x49,0x38,
				0xd0,0xe0,0x45,0x25, 0x36,0xc5,0xf4,0x79,
				0x2e,0x9a,0x98,0x04, 0xe4,0x2b,0x46,0x52,
				0x7c,0x33,0xca,0xe2, 0x56,0x51,0x50,0xe2,
				0xa5,0x9a,0xae,0x18, 0x6a,0x13,0xf8,0xd2,
				0x21,0x31,0x66,0x02, 0xe2,0xda,0x8d,0x7e,
				0x41,0x19,0xb2,0x61, 0xee,0x48,0x8f,0xf1,
				0x65,0x24,0x2e,0x1e, 0x68,0xce,0x05,0xd9,
				0x2a,0xcf,0xa5,0x3a, 0x57,0xdd,0x35,0x91,
				0x93,0x01,0xca,0x95, 0xfc,0x2b,0x36,0x04,
				0xe6,0x96,0x97,0x28, 0xf6,0x31,0xfe,0xa3,
				0x9d,0xf6,0x6a,0x1e, 0x80,0x8d,0xdc,0xec,
				0xaf,0x66,0x11,0x13, 0x02,0x88,0xd5,0x27,
				0x33,0xb4,0x1a,0xcd, 0xa3,0xf6,0xde,0x31,
				0x8e,0xc0,0x0e,0x6c, 0xd8,0x5a,0x97,0x5e,
				0xdd,0xfd,0x60,0x69, 0x38,0x46,0x3f,0x90,
				0x5e,0x97,0xd3,0x32, 0x76,0xc7,0x82,0x49,
				0xfe,0xba,0x06,0x5f, 0x2f,0xa2,0xfd,0xff,
				0x80,0x05,0x40,0xe4, 0x33,0x03,0xfb,0x10,
				0xc0,0xde,0x65,0x8c, 0xc9,0x8d,0x3a,0x9d,
				0xb5,0x7b,0x36,0x4b, 0xb5,0x0c,0xcf,0x00,
				0x9c,0x87,0xe4,0x49, 0xad,0x90,0xda,0x4a,
				0xdd,0xbd,0xff,0xe2, 0x32,0x57,0xd6,0x78,
				0x36,0x39,0x6c,0xd3, 0x5b,0x9b,0x88,0x59,
				0x2d,0xf0,0x46,0xe4, 0x13,0x0e,0x2b,0x35,
				0x0d,0x0f,0x73,0x8a, 0x4f,0x26,0x84,0x75,
				0x88,0x3c,0xc5,0x58, 0x66,0x18,0x1a,0xb4,
				0x64,0x51,0x34,0x27, 0x1b,0xa4,0x11,0xc9,
				0x6d,0x91,0x8a,0xfa, 0x32,0x60,0x9d,0xd7,
				0x87,0xe5,0xaa,0x43, 0x72,0xf8,0xda,0xd1,
				0x48,0x44,0x13,0x61, 0xdc,0x8c,0x76,0x17,
				0x0c,0x85,0x4e,0xf3, 0xdd,0xa2,0x42,0xd2,
				0x74,0xc1,0x30,0x1b, 0xeb,0x35,0x31,0x29,
				0x5b,0xd7,0x4c,0x94, 0x46,0x35,0xa1,0x23,
				0x50,0xf2,0xa2,0x8e, 0x7e,0x4f,0x23,0x4f,
				0x51,0xff,0xe2,0xc9, 0xa3,0x7d,0x56,0x8b,
				0x41,0xf2,0xd0,0xc5, 0x57,0x7e,0x59,0xac,
				0xbb,0x65,0xf3,0xfe, 0xf7,0x17,0xef,0x63,
				0x7c,0x6f,0x23,0xdd, 0x22,0x8e,0xed,0x84,
				0x0e,0x3b,0x09,0xb3, 0xf3,0xf4,0x8f,0xcd,
				0x37,0xa8,0xe1,0xa7, 0x30,0xdb,0xb1,0xa2,
				0x9c,0xa2,0xdf,0x34, 0x17,0x3e,0x68,0x44,
				0xd0,0xde,0x03,0x50, 0xd1,0x48,0x6b,0x20,
				0xe2,0x63,0x45,0xa5, 0xea,0x87,0xc2,0x42,
				0x95,0x03,0x49,0x05, 0xed,0xe0,0x90,0x29,
				0x1a,0xb8,0xcf,0x9b, 0x43,0xcf,0x29,0x7a,
				0x63,0x17,0x41,0x9f, 0xe0,0xc9,0x10,0xfd,
				0x2c,0x56,0x8c,0x08, 0x55,0xb4,0xa9,0x27,
				0x0f,0x23,0xb1,0x05, 0x6a,0x12,0x46,0xc7,
				0xe1,0xfe,0x28,0x93, 0x93,0xd7,0x2f,0xdc,
				0x98,0x30,0xdb,0x75, 0x8a,0xbe,0x97,0x7a,
				0x02,0xfb,0x8c,0xba, 0xbe,0x25,0x09,0xbe,
				0xce,0xcb,0xa2,0xef, 0x79,0x4d,0x0e,0x9d,
				0x1b,0x9d,0xb6,0x39, 0x34,0x38,0xfa,0x07,
				0xec,0xe8,0xfc,0x32, 0x85,0x1d,0xf7,0x85,
				0x63,0xc3,0x3c,0xc0, 0x02,0x75,0xd7,0x3f,
				0xb2,0x68,0x60,0x66, 0x65,0x81,0xc6,0xb1,
				0x42,0x65,0x4b,0x4b, 0x28,0xd7,0xc7,0xaa,
				0x9b,0xd2,0xdc,0x1b, 0x01,0xe0,0x26,0x39,
				0x01,0xc1,0x52,0x14, 0xd1,0x3f,0xb7,0xe6,
				0x61,0x41,0xc7,0x93, 0xd2,0xa2,0x67,0xc6,
				0xf7,0x11,0xb5,0xf5, 0xea,0xdd,0x19,0xfb,
				0x4d,0x21,0x12,0xd6, 0x7d,0xf1,0x10,0xb0,
				0x89,0x07,0xc7,0x5a, 0x52,0x73,0x70,0x2f,
				0x32,0xef,0x65,0x2b, 0x12,0xb2,0xf0,0xf5,
				0x20,0xe0,0x90,0x59, 0x7e,0x64,0xf1,0x4c,
				0x41,0xb3,0xa5,0x91, 0x08,0xe6,0x5e,0x5f,
				0x05,0x56,0x76,0xb4, 0xb0,0xcd,0x70,0x53,
				0x10,0x48,0x9c,0xff, 0xc2,0x69,0x55,0x24,
				0x87,0xef,0x84,0xea, 0xfb,0xa7,0xbf,0xa0,
				0x91,0x04,0xad,0x4f, 0x8b,0x57,0x54,0x4b,
				0xb6,0xe9,0xd1,0xac, 0x37,0x2f,0x1d,0x2e,
				0xab,0xa5,0xa4,0xe8, 0xff,0xfb,0xd9,0x39,
				0x2f,0xb7,0xac,0xd1, 0xfe,0x0b,0x9a,0x80,
				0x0f,0xb6,0xf4,0x36, 0x39,0x90,0x51,0xe3,
				0x0a,0x2f,0xb6,0x45, 0x76,0x89,0xcd,0x61,
				0xfe,0x48,0x5f,0x75, 0x1d,0x13,0x00,0x62,
				0x80,0x24,0x47,0xe7, 0xbc,0x37,0xd7,0xe3,
				0x15,0xe8,0x68,0x22, 0xaf,0x80,0x6f,0x4b,
				0xa8,0x9f,0x01,0x10, 0x48,0x14,0xc3,0x02,
				0x52,0xd2,0xc7,0x75, 0x9b,0x52,0x6d,0x30,
				0xac,0x13,0x85,0xc8, 0xf7,0xa3,0x58,0x4b,
				0x49,0xf7,0x1c,0x45, 0x55,0x8c,0x39,0x9a,
				0x99,0x6d,0x97,0x27, 0x27,0xe6,0xab,0xdd,
				0x2c,0x42,0x1b,0x35, 0xdd,0x9d,0x73,0xbb,
				0x6c,0xf3,0x64,0xf1, 0xfb,0xb9,0xf7,0xe6,
				0x4a,0x3c,0xc0,0x92, 0xc0,0x2e,0xb7,0x1a,
				0xbe,0xab,0xb3,0x5a, 0xe5,0xea,0xb1,0x48,
				0x58,0x13,0x53,0x90, 0xfd,0xc3,0x8e,0x54,
				0xf9,0x18,0x16,0x73, 0xe8,0xcb,0x6d,0x39,
				0x0e,0xd7,0xe0,0xfe, 0xb6,0x9f,0x43,0x97,
				0xe8,0xd0,0x85,0x56, 0x83,0x3e,0x98,0x68,
				0x7f,0xbd,0x95,0xa8, 0x9a,0x61,0x21,0x8f,
				0x06,0x98,0x34,0xa6, 0xc8,0xd6,0x1d,0xf3,
				0x3d,0x43,0xa4,0x9a, 0x8c,0xe5,0xd3,0x5a,
				0x32,0xa2,0x04,0x22, 0xa4,0x19,0x1a,0x46,
				0x42,0x7e,0x4d,0xe5, 0xe0,0xe6,0x0e,0xca,
				0xd5,0x58,0x9d,0x2c, 0xaf,0xda,0x33,0x5c,
				0xb0,0x79,0x9e,0xc9, 0xfc,0xca,0xf0,0x2f,
				0xa8,0xb2,0x77,0xeb, 0x7a,0xa2,0xdd,0x37,
				0x35,0x83,0x07,0xd6, 0x02,0x1a,0xb6,0x6c,
				0x24,0xe2,0x59,0x08, 0x0e,0xfd,0x3e,0x46,
				0xec,0x40,0x93,0xf4, 0x00,0x26,0x4f,0x2a,
				0xff,0x47,0x2f,0xeb, 0x02,0x92,0x26,0x5b,
				0x53,0x17,0xc2,0x8d, 0x2a,0xc7,0xa3,0x1b,
				0xcd,0xbc,0xa7,0xe8, 0xd1,0x76,0xe3,0x80,
				0x21,0xca,0x5d,0x3b, 0xe4,0x9c,0x8f,0xa9,
				0x5b,0x7f,0x29,0x7f, 0x7c,0xd8,0xed,0x6d,
				0x8c,0xb2,0x86,0x85, 0xe7,0x77,0xf2,0x85,
				0xab,0x38,0xa9,0x9d, 0xc1,0x4e,0xc5,0x64,
				0x33,0x73,0x8b,0x59, 0x03,0xad,0x05,0xdf,
				0x25,0x98,0x31,0xde, 0xef,0x13,0xf1,0x9b,
				0x3c,0x91,0x9d,0x7b, 0xb1,0xfa,0xe6,0xbf,
				0x5b,0xed,0xa5,0x55, 0xe6,0xea,0x6c,0x74,
				0xf4,0xb9,0xe4,0x45, 0x64,0x72,0x81,0xc2,
				0x4c,0x28,0xd4,0xcd, 0xac,0xe2,0xde,0xf9,
				0xeb,0x5c,0xeb,0x61, 0x60,0x5a,0xe5,0x28,
			},
			.mlen = 0,
			.h = {0},
		},
		[1] = {		/* 16-byte message */
			.k = {
				/* Poly1305 key */
				0x29,0x21,0x43,0xcb, 0xcb,0x13,0x07,0xde,
				0xbf,0x48,0xdf,0x8a, 0x7f,0xa2,0x84,0xde,

				/* NH key */
				0x72,0x23,0x9d,0xf5, 0xf0,0x07,0xf2,0x4c,
				0x20,0x3a,0x93,0xb9, 0xcd,0x5d,0xfe,0xcb,
				0x99,0x2c,0x2b,0x58, 0xc6,0x50,0x5f,0x94,
				0x56,0xc3,0x7c,0x0d, 0x02,0x3f,0xb8,0x5e,
				0x7b,0xc0,0x6c,0x51, 0x34,0x76,0xc0,0x0e,
				0xc6,0x22,0xc8,0x9e, 0x92,0xa0,0x21,0xc9,
				0x85,0x5c,0x7c,0xf8, 0xe2,0x64,0x47,0xc9,
				0xe4,0xa2,0x57,0x93, 0xf8,0xa2,0x69,0xcd,
				0x62,0x98,0x99,0xf4, 0xd7,0x7b,0x14,0xb1,
				0xd8,0x05,0xff,0x04, 0x15,0xc9,0xe1,0x6e,
				0x9b,0xe6,0x50,0x6b, 0x0b,0x3f,0x22,0x1f,
				0x08,0xde,0x0c,0x5b, 0x08,0x7e,0xc6,0x2f,
				0x6c,0xed,0xd6,0xb2, 0x15,0xa4,0xb3,0xf9,
				0xa7,0x46,0x38,0x2a, 0xea,0x69,0xa5,0xde,
				0x02,0xc3,0x96,0x89, 0x4d,0x55,0x3b,0xed,
				0x3d,0x3a,0x85,0x77, 0xbf,0x97,0x45,0x5c,
				0x9e,0x02,0x69,0xe2, 0x1b,0x68,0xbe,0x96,
				0xfb,0x64,0x6f,0x0f, 0xf6,0x06,0x40,0x67,
				0xfa,0x04,0xe3,0x55, 0xfa,0xbe,0xa4,0x60,
				0xef,0x21,0x66,0x97, 0xe6,0x9d,0x5c,0x1f,
				0x62,0x37,0xaa,0x31, 0xde,0xe4,0x9c,0x28,
				0x95,0xe0,0x22,0x86, 0xf4,0x4d,0xf3,0x07,
				0xfd,0x5f,0x3a,0x54, 0x2c,0x51,0x80,0x71,
				0xba,0x78,0x69,0x5b, 0x65,0xab,0x1f,0x81,
				0xed,0x3b,0xff,0x34, 0xa3,0xfb,0xbc,0x73,
				0x66,0x7d,0x13,0x7f, 0xdf,0x6e,0xe2,0xe2,
				0xeb,0x4f,0x6c,0xda, 0x7d,0x33,0x57,0xd0,
				0xd3,0x7c,0x95,0x4f, 0x33,0x58,0x21,0xc7,
				0xc0,0xe5,0x6f,0x42, 0x26,0xc6,0x1f,0x5e,
				0x85,0x1b,0x98,0x9a, 0xa2,0x1e,0x55,0x77,
				0x23,0xdf,0x81,0x5e, 0x79,0x55,0x05,0xfc,
				0xfb,0xda,0xee,0xba, 0x5a,0xba,0xf7,0x77,
				0x7f,0x0e,0xd3,0xe1, 0x37,0xfe,0x8d,0x2b,
				0xd5,0x3f,0xfb,0xd0, 0xc0,0x3c,0x0b,0x3f,
				0xcf,0x3c,0x14,0xcf, 0xfb,0x46,0x72,0x4c,
				0x1f,0x39,0xe2,0xda, 0x03,0x71,0x6d,0x23,
				0xef,0x93,0xcd,0x39, 0xd9,0x37,0x80,0x4d,
				0x65,0x61,0xd1,0x2c, 0x03,0xa9,0x47,0x72,
				0x4d,0x1e,0x0e,0x16, 0x33,0x0f,0x21,0x17,
				0xec,0x92,0xea,0x6f, 0x37,0x22,0xa4,0xd8,
				0x03,0x33,0x9e,0xd8, 0x03,0x69,0x9a,0xe8,
				0xb2,0x57,0xaf,0x78, 0x99,0x05,0x12,0xab,
				0x48,0x90,0x80,0xf0, 0x12,0x9b,0x20,0x64,
				0x7a,0x1d,0x47,0x5f, 0xba,0x3c,0xf9,0xc3,
				0x0a,0x0d,0x8d,0xa1, 0xf9,0x1b,0x82,0x13,
				0x3e,0x0d,0xec,0x0a, 0x83,0xc0,0x65,0xe1,
				0xe9,0x95,0xff,0x97, 0xd6,0xf2,0xe4,0xd5,
				0x86,0xc0,0x1f,0x29, 0x27,0x63,0xd7,0xde,
				0xb7,0x0a,0x07,0x99, 0x04,0x2d,0xa3,0x89,
				0xa2,0x43,0xcf,0xf3, 0xe1,0x43,0xac,0x4a,
				0x06,0x97,0xd0,0x05, 0x4f,0x87,0xfa,0xf9,
				0x9b,0xbf,0x52,0x70, 0xbd,0xbc,0x6c,0xf3,
				0x03,0x13,0x60,0x41, 0x28,0x09,0xec,0xcc,
				0xb1,0x1a,0xec,0xd6, 0xfb,0x6f,0x2a,0x89,
				0x5d,0x0b,0x53,0x9c, 0x59,0xc1,0x84,0x21,
				0x33,0x51,0x47,0x19, 0x31,0x9c,0xd4,0x0a,
				0x4d,0x04,0xec,0x50, 0x90,0x61,0xbd,0xbc,
				0x7e,0xc8,0xd9,0x6c, 0x98,0x1d,0x45,0x41,
				0x17,0x5e,0x97,0x1c, 0xc5,0xa8,0xe8,0xea,
				0x46,0x58,0x53,0xf7, 0x17,0xd5,0xad,0x11,
				0xc8,0x54,0xf5,0x7a, 0x33,0x90,0xf5,0x19,
				0xba,0x36,0xb4,0xfc, 0x52,0xa5,0x72,0x3d,
				0x14,0xbb,0x55,0xa7, 0xe9,0xe3,0x12,0xf7,
				0x1c,0x30,0xa2,0x82, 0x03,0xbf,0x53,0x91,
				0x2e,0x60,0x41,0x9f, 0x5b,0x69,0x39,0xf6,
				0x4d,0xc8,0xf8,0x46, 0x7a,0x7f,0xa4,0x98,
				0x36,0xff,0x06,0xcb, 0xca,0xe7,0x33,0xf2,
				0xc0,0x4a,0xf4,0x3c, 0x14,0x44,0x5f,0x6b,
				0x75,0xef,0x02,0x36, 0x75,0x08,0x14,0xfd,
				0x10,0x8e,0xa5,0x58, 0xd0,0x30,0x46,0x49,
				0xaf,0x3a,0xf8,0x40, 0x3d,0x35,0xdb,0x84,
				0x11,0x2e,0x97,0x6a, 0xb7,0x87,0x7f,0xad,
				0xf1,0xfa,0xa5,0x63, 0x60,0xd8,0x5e,0xbf,
				0x41,0x78,0x49,0xcf, 0x77,0xbb,0x56,0xbb,
				0x7d,0x01,0x67,0x05, 0x22,0xc8,0x8f,0x41,
				0xba,0x81,0xd2,0xca, 0x2c,0x38,0xac,0x76,
				0x06,0xc1,0x1a,0xc2, 0xce,0xac,0x90,0x67,
				0x57,0x3e,0x20,0x12, 0x5b,0xd9,0x97,0x58,
				0x65,0x05,0xb7,0x04, 0x61,0x7e,0xd8,0x3a,
				0xbf,0x55,0x3b,0x13, 0xe9,0x34,0x5a,0x37,
				0x36,0xcb,0x94,0x45, 0xc5,0x32,0xb3,0xa0,
				0x0c,0x3e,0x49,0xc5, 0xd3,0xed,0xa7,0xf0,
				0x1c,0x69,0xcc,0xea, 0xcc,0x83,0xc9,0x16,
				0x95,0x72,0x4b,0xf4, 0x89,0xd5,0xb9,0x10,
				0xf6,0x2d,0x60,0x15, 0xea,0x3c,0x06,0x66,
				0x9f,0x82,0xad,0x17, 0xce,0xd2,0xa4,0x48,
				0x7c,0x65,0xd9,0xf8, 0x02,0x4d,0x9b,0x4c,
				0x89,0x06,0x3a,0x34, 0x85,0x48,0x89,0x86,
				0xf9,0x24,0xa9,0x54, 0x72,0xdb,0x44,0x95,
				0xc7,0x44,0x1c,0x19, 0x11,0x4c,0x04,0xdc,
				0x13,0xb9,0x67,0xc8, 0xc3,0x3a,0x6a,0x50,
				0xfa,0xd1,0xfb,0xe1, 0x88,0xb6,0xf1,0xa3,
				0xc5,0x3b,0xdc,0x38, 0x45,0x16,0x26,0x02,
				0x3b,0xb8,0x8f,0x8b, 0x58,0x7d,0x23,0x04,
				0x50,0x6b,0x81,0x9f, 0xae,0x66,0xac,0x6f,
				0xcf,0x2a,0x9d,0xf1, 0xfd,0x1d,0x57,0x07,
				0xbe,0x58,0xeb,0x77, 0x0c,0xe3,0xc2,0x19,
				0x14,0x74,0x1b,0x51, 0x1c,0x4f,0x41,0xf3,
				0x32,0x89,0xb3,0xe7, 0xde,0x62,0xf6,0x5f,
				0xc7,0x6a,0x4a,0x2a, 0x5b,0x0f,0x5f,0x87,
				0x9c,0x08,0xb9,0x02, 0x88,0xc8,0x29,0xb7,
				0x94,0x52,0xfa,0x52, 0xfe,0xaa,0x50,0x10,
				0xba,0x48,0x75,0x5e, 0x11,0x1b,0xe6,0x39,
				0xd7,0x82,0x2c,0x87, 0xf1,0x1e,0xa4,0x38,
				0x72,0x3e,0x51,0xe7, 0xd8,0x3e,0x5b,0x7b,
				0x31,0x16,0x89,0xba, 0xd6,0xad,0x18,0x5e,
				0xba,0xf8,0x12,0xb3, 0xf4,0x6c,0x47,0x30,
				0xc0,0x38,0x58,0xb3, 0x10,0x8d,0x58,0x5d,
				0xb4,0xfb,0x19,0x7e, 0x41,0xc3,0x66,0xb8,
				0xd6,0x72,0x84,0xe1, 0x1a,0xc2,0x71,0x4c,
				0x0d,0x4a,0x21,0x7a, 0xab,0xa2,0xc0,0x36,
				0x15,0xc5,0xe9,0x46, 0xd7,0x29,0x17,0x76,
				0x5e,0x47,0x36,0x7f, 0x72,0x05,0xa7,0xcc,
				0x36,0x63,0xf9,0x47, 0x7d,0xe6,0x07,0x3c,
				0x8b,0x79,0x1d,0x96, 0x61,0x8d,0x90,0x65,
				0x7c,0xf5,0xeb,0x4e, 0x6e,0x09,0x59,0x6d,
				0x62,0x50,0x1b,0x0f, 0xe0,0xdc,0x78,0xf2,
				0x5b,0x83,0x1a,0xa1, 0x11,0x75,0xfd,0x18,
				0xd7,0xe2,0x8d,0x65, 0x14,0x21,0xce,0xbe,
				0xb5,0x87,0xe3,0x0a, 0xda,0x24,0x0a,0x64,
				0xa9,0x9f,0x03,0x8d, 0x46,0x5d,0x24,0x1a,
				0x8a,0x0c,0x42,0x01, 0xca,0xb1,0x5f,0x7c,
				0xa5,0xac,0x32,0x4a, 0xb8,0x07,0x91,0x18,
				0x6f,0xb0,0x71,0x3c, 0xc9,0xb1,0xa8,0xf8,
				0x5f,0x69,0xa5,0xa1, 0xca,0x9e,0x7a,0xaa,
				0xac,0xe9,0xc7,0x47, 0x41,0x75,0x25,0xc3,
				0x73,0xe2,0x0b,0xdd, 0x6d,0x52,0x71,0xbe,
				0xc5,0xdc,0xb4,0xe7, 0x01,0x26,0x53,0x77,
				0x86,0x90,0x85,0x68, 0x6b,0x7b,0x03,0x53,
				0xda,0x52,0x52,0x51, 0x68,0xc8,0xf3,0xec,
				0x6c,0xd5,0x03,0x7a, 0xa3,0x0e,0xb4,0x02,
				0x5f,0x1a,0xab,0xee, 0xca,0x67,0x29,0x7b,
				0xbd,0x96,0x59,0xb3, 0x8b,0x32,0x7a,0x92,
				0x9f,0xd8,0x25,0x2b, 0xdf,0xc0,0x4c,0xda,
			},
			.mlen = 16,
			.m = {
				0xbc,0xda,0x81,0xa8, 0x78,0x79,0x1c,0xbf,
				0x77,0x53,0xba,0x4c, 0x30,0x5b,0xb8,0x33,
			},
			.h = {
				0x04,0xbf,0x7f,0x6a, 0xce,0x72,0xea,0x6a,
				0x79,0xdb,0xb0,0xc9, 0x60,0xf6,0x12,0xcc,
			},
		},
		[2] = {		/* 1024-byte message */
			.k = {
				0x65,0x4d,0xe3,0xf8, 0xd2,0x4c,0xac,0x28,
				0x68,0xf5,0xb3,0x81, 0x71,0x4b,0xa1,0xfa,
				0x04,0x0e,0xd3,0x81, 0x36,0xbe,0x0c,0x81,
				0x5e,0xaf,0xbc,0x3a, 0xa4,0xc0,0x8e,0x8b,
				0x55,0x63,0xd3,0x52, 0x97,0x88,0xd6,0x19,
				0xbc,0x96,0xdf,0x49, 0xff,0x04,0x63,0xf5,
				0x0c,0x11,0x13,0xaa, 0x9e,0x1f,0x5a,0xf7,
				0xdd,0xbd,0x37,0x80, 0xc3,0xd0,0xbe,0xa7,
				0x05,0xc8,0x3c,0x98, 0x1e,0x05,0x3c,0x84,
				0x39,0x61,0xc4,0xed, 0xed,0x71,0x1b,0xc4,
				0x74,0x45,0x2c,0xa1, 0x56,0x70,0x97,0xfd,
				0x44,0x18,0x07,0x7d, 0xca,0x60,0x1f,0x73,
				0x3b,0x6d,0x21,0xcb, 0x61,0x87,0x70,0x25,
				0x46,0x21,0xf1,0x1f, 0x21,0x91,0x31,0x2d,
				0x5d,0xcc,0xb7,0xd1, 0x84,0x3e,0x3d,0xdb,
				0x03,0x53,0x2a,0x82, 0xa6,0x9a,0x95,0xbc,
				0x1a,0x1e,0x0a,0x5e, 0x07,0x43,0xab,0x43,
				0xaf,0x92,0x82,0x06, 0x91,0x04,0x09,0xf4,
				0x17,0x0a,0x9a,0x2c, 0x54,0xdb,0xb8,0xf4,
				0xd0,0xf0,0x10,0x66, 0x24,0x8d,0xcd,0xda,
				0xfe,0x0e,0x45,0x9d, 0x6f,0xc4,0x4e,0xf4,
				0x96,0xaf,0x13,0xdc, 0xa9,0xd4,0x8c,0xc4,
				0xc8,0x57,0x39,0x3c, 0xc2,0xd3,0x0a,0x76,
				0x4a,0x1f,0x75,0x83, 0x44,0xc7,0xd1,0x39,
				0xd8,0xb5,0x41,0xba, 0x73,0x87,0xfa,0x96,
				0xc7,0x18,0x53,0xfb, 0x9b,0xda,0xa0,0x97,
				0x1d,0xee,0x60,0x85, 0x9e,0x14,0xc3,0xce,
				0xc4,0x05,0x29,0x3b, 0x95,0x30,0xa3,0xd1,
				0x9f,0x82,0x6a,0x04, 0xf5,0xa7,0x75,0x57,
				0x82,0x04,0xfe,0x71, 0x51,0x71,0xb1,0x49,
				0x50,0xf8,0xe0,0x96, 0xf1,0xfa,0xa8,0x88,
				0x3f,0xa0,0x86,0x20, 0xd4,0x60,0x79,0x59,
				0x17,0x2d,0xd1,0x09, 0xf4,0xec,0x05,0x57,
				0xcf,0x62,0x7e,0x0e, 0x7e,0x60,0x78,0xe6,
				0x08,0x60,0x29,0xd8, 0xd5,0x08,0x1a,0x24,
				0xc4,0x6c,0x24,0xe7, 0x92,0x08,0x3d,0x8a,
				0x98,0x7a,0xcf,0x99, 0x0a,0x65,0x0e,0xdc,
				0x8c,0x8a,0xbe,0x92, 0x82,0x91,0xcc,0x62,
				0x30,0xb6,0xf4,0x3f, 0xc6,0x8a,0x7f,0x12,
				0x4a,0x8a,0x49,0xfa, 0x3f,0x5c,0xd4,0x5a,
				0xa6,0x82,0xa3,0xe6, 0xaa,0x34,0x76,0xb2,
				0xab,0x0a,0x30,0xef, 0x6c,0x77,0x58,0x3f,
				0x05,0x6b,0xcc,0x5c, 0xae,0xdc,0xd7,0xb9,
				0x51,0x7e,0x8d,0x32, 0x5b,0x24,0x25,0xbe,
				0x2b,0x24,0x01,0xcf, 0x80,0xda,0x16,0xd8,
				0x90,0x72,0x2c,0xad, 0x34,0x8d,0x0c,0x74,
				0x02,0xcb,0xfd,0xcf, 0x6e,0xef,0x97,0xb5,
				0x4c,0xf2,0x68,0xca, 0xde,0x43,0x9e,0x8a,
				0xc5,0x5f,0x31,0x7f, 0x14,0x71,0x38,0xec,
				0xbd,0x98,0xe5,0x71, 0xc4,0xb5,0xdb,0xef,
				0x59,0xd2,0xca,0xc0, 0xc1,0x86,0x75,0x01,
				0xd4,0x15,0x0d,0x6f, 0xa4,0xf7,0x7b,0x37,
				0x47,0xda,0x18,0x93, 0x63,0xda,0xbe,0x9e,
				0x07,0xfb,0xb2,0x83, 0xd5,0xc4,0x34,0x55,
				0xee,0x73,0xa1,0x42, 0x96,0xf9,0x66,0x41,
				0xa4,0xcc,0xd2,0x93, 0x6e,0xe1,0x0a,0xbb,
				0xd2,0xdd,0x18,0x23, 0xe6,0x6b,0x98,0x0b,
				0x8a,0x83,0x59,0x2c, 0xc3,0xa6,0x59,0x5b,
				0x01,0x22,0x59,0xf7, 0xdc,0xb0,0x87,0x7e,
				0xdb,0x7d,0xf4,0x71, 0x41,0xab,0xbd,0xee,
				0x79,0xbe,0x3c,0x01, 0x76,0x0b,0x2d,0x0a,
				0x42,0xc9,0x77,0x8c, 0xbb,0x54,0x95,0x60,
				0x43,0x2e,0xe0,0x17, 0x52,0xbd,0x90,0xc9,
				0xc2,0x2c,0xdd,0x90, 0x24,0x22,0x76,0x40,
				0x5c,0xb9,0x41,0xc9, 0xa1,0xd5,0xbd,0xe3,
				0x44,0xe0,0xa4,0xab, 0xcc,0xb8,0xe2,0x32,
				0x02,0x15,0x04,0x1f, 0x8c,0xec,0x5d,0x14,
				0xac,0x18,0xaa,0xef, 0x6e,0x33,0x19,0x6e,
				0xde,0xfe,0x19,0xdb, 0xeb,0x61,0xca,0x18,
				0xad,0xd8,0x3d,0xbf, 0x09,0x11,0xc7,0xa5,
				0x86,0x0b,0x0f,0xe5, 0x3e,0xde,0xe8,0xd9,
				0x0a,0x69,0x9e,0x4c, 0x20,0xff,0xf9,0xc5,
				0xfa,0xf8,0xf3,0x7f, 0xa5,0x01,0x4b,0x5e,
				0x0f,0xf0,0x3b,0x68, 0xf0,0x46,0x8c,0x2a,
				0x7a,0xc1,0x8f,0xa0, 0xfe,0x6a,0x5b,0x44,
				0x70,0x5c,0xcc,0x92, 0x2c,0x6f,0x0f,0xbd,
				0x25,0x3e,0xb7,0x8e, 0x73,0x58,0xda,0xc9,
				0xa5,0xaa,0x9e,0xf3, 0x9b,0xfd,0x37,0x3e,
				0xe2,0x88,0xa4,0x7b, 0xc8,0x5c,0xa8,0x93,
				0x0e,0xe7,0x9a,0x9c, 0x2e,0x95,0x18,0x9f,
				0xc8,0x45,0x0c,0x88, 0x9e,0x53,0x4f,0x3a,
				0x76,0xc1,0x35,0xfa, 0x17,0xd8,0xac,0xa0,
				0x0c,0x2d,0x47,0x2e, 0x4f,0x69,0x9b,0xf7,
				0xd0,0xb6,0x96,0x0c, 0x19,0xb3,0x08,0x01,
				0x65,0x7a,0x1f,0xc7, 0x31,0x86,0xdb,0xc8,
				0xc1,0x99,0x8f,0xf8, 0x08,0x4a,0x9d,0x23,
				0x22,0xa8,0xcf,0x27, 0x01,0x01,0x88,0x93,
				0x9c,0x86,0x45,0xbd, 0xe0,0x51,0xca,0x52,
				0x84,0xba,0xfe,0x03, 0xf7,0xda,0xc5,0xce,
				0x3e,0x77,0x75,0x86, 0xaf,0x84,0xc8,0x05,
				0x44,0x01,0x0f,0x02, 0xf3,0x58,0xb0,0x06,
				0x5a,0xd7,0x12,0x30, 0x8d,0xdf,0x1f,0x1f,
				0x0a,0xe6,0xd2,0xea, 0xf6,0x3a,0x7a,0x99,
				0x63,0xe8,0xd2,0xc1, 0x4a,0x45,0x8b,0x40,
				0x4d,0x0a,0xa9,0x76, 0x92,0xb3,0xda,0x87,
				0x36,0x33,0xf0,0x78, 0xc3,0x2f,0x5f,0x02,
				0x1a,0x6a,0x2c,0x32, 0xcd,0x76,0xbf,0xbd,
				0x5a,0x26,0x20,0x28, 0x8c,0x8c,0xbc,0x52,
				0x3d,0x0a,0xc9,0xcb, 0xab,0xa4,0x21,0xb0,
				0x54,0x40,0x81,0x44, 0xc7,0xd6,0x1c,0x11,
				0x44,0xc6,0x02,0x92, 0x14,0x5a,0xbf,0x1a,
				0x09,0x8a,0x18,0xad, 0xcd,0x64,0x3d,0x53,
				0x4a,0xb6,0xa5,0x1b, 0x57,0x0e,0xef,0xe0,
				0x8c,0x44,0x5f,0x7d, 0xbd,0x6c,0xfd,0x60,
				0xae,0x02,0x24,0xb6, 0x99,0xdd,0x8c,0xaf,
				0x59,0x39,0x75,0x3c, 0xd1,0x54,0x7b,0x86,
				0xcc,0x99,0xd9,0x28, 0x0c,0xb0,0x94,0x62,
				0xf9,0x51,0xd1,0x19, 0x96,0x2d,0x66,0xf5,
				0x55,0xcf,0x9e,0x59, 0xe2,0x6b,0x2c,0x08,
				0xc0,0x54,0x48,0x24, 0x45,0xc3,0x8c,0x73,
				0xea,0x27,0x6e,0x66, 0x7d,0x1d,0x0e,0x6e,
				0x13,0xe8,0x56,0x65, 0x3a,0xb0,0x81,0x5c,
				0xf0,0xe8,0xd8,0x00, 0x6b,0xcd,0x8f,0xad,
				0xdd,0x53,0xf3,0xa4, 0x6c,0x43,0xd6,0x31,
				0xaf,0xd2,0x76,0x1e, 0x91,0x12,0xdb,0x3c,
				0x8c,0xc2,0x81,0xf0, 0x49,0xdb,0xe2,0x6b,
				0x76,0x62,0x0a,0x04, 0xe4,0xaa,0x8a,0x7c,
				0x08,0x0b,0x5d,0xd0, 0xee,0x1d,0xfb,0xc4,
				0x02,0x75,0x42,0xd6, 0xba,0xa7,0x22,0xa8,
				0x47,0x29,0xb7,0x85, 0x6d,0x93,0x3a,0xdb,
				0x00,0x53,0x0b,0xa2, 0xeb,0xf8,0xfe,0x01,
				0x6f,0x8a,0x31,0xd6, 0x17,0x05,0x6f,0x67,
				0x88,0x95,0x32,0xfe, 0x4f,0xa6,0x4b,0xf8,
				0x03,0xe4,0xcd,0x9a, 0x18,0xe8,0x4e,0x2d,
				0xf7,0x97,0x9a,0x0c, 0x7d,0x9f,0x7e,0x44,
				0x69,0x51,0xe0,0x32, 0x6b,0x62,0x86,0x8f,
				0xa6,0x8e,0x0b,0x21, 0x96,0xe5,0xaf,0x77,
				0xc0,0x83,0xdf,0xa5, 0x0e,0xd0,0xa1,0x04,
				0xaf,0xc1,0x10,0xcb, 0x5a,0x40,0xe4,0xe3,
				0x38,0x7e,0x07,0xe8, 0x4d,0xfa,0xed,0xc5,
				0xf0,0x37,0xdf,0xbb, 0x8a,0xcf,0x3d,0xdc,
				0x61,0xd2,0xc6,0x2b, 0xff,0x07,0xc9,0x2f,
				0x0c,0x2d,0x5c,0x07, 0xa8,0x35,0x6a,0xfc,
				0xae,0x09,0x03,0x45, 0x74,0x51,0x4d,0xc4,
				0xb8,0x23,0x87,0x4a, 0x99,0x27,0x20,0x87,
				0x62,0x44,0x0a,0x4a, 0xce,0x78,0x47,0x22,
			},
			.mlen = 1024,
			.m = {
				0x8e,0xb0,0x4c,0xde, 0x9c,0x4a,0x04,0x5a,
				0xf6,0xa9,0x7f,0x45, 0x25,0xa5,0x7b,0x3a,
				0xbc,0x4d,0x73,0x39, 0x81,0xb5,0xbd,0x3d,
				0x21,0x6f,0xd7,0x37, 0x50,0x3c,0x7b,0x28,
				0xd1,0x03,0x3a,0x17, 0xed,0x7b,0x7c,0x2a,
				0x16,0xbc,0xdf,0x19, 0x89,0x52,0x71,0x31,
				0xb6,0xc0,0xfd,0xb5, 0xd3,0xba,0x96,0x99,
				0xb6,0x34,0x0b,0xd0, 0x99,0x93,0xfc,0x1a,
				0x01,0x3c,0x85,0xc6, 0x9b,0x78,0x5c,0x8b,
				0xfe,0xae,0xd2,0xbf, 0xb2,0x6f,0xf9,0xed,
				0xc8,0x25,0x17,0xfe, 0x10,0x3b,0x7d,0xda,
				0xf4,0x8d,0x35,0x4b, 0x7c,0x7b,0x82,0xe7,
				0xc2,0xb3,0xee,0x60, 0x4a,0x03,0x86,0xc9,
				0x4e,0xb5,0xc4,0xbe, 0xd2,0xbd,0x66,0xf1,
				0x13,0xf1,0x09,0xab, 0x5d,0xca,0x63,0x1f,
				0xfc,0xfb,0x57,0x2a, 0xfc,0xca,0x66,0xd8,
				0x77,0x84,0x38,0x23, 0x1d,0xac,0xd3,0xb3,
				0x7a,0xad,0x4c,0x70, 0xfa,0x9c,0xc9,0x61,
				0xa6,0x1b,0xba,0x33, 0x4b,0x4e,0x33,0xec,
				0xa0,0xa1,0x64,0x39, 0x40,0x05,0x1c,0xc2,
				0x3f,0x49,0x9d,0xae, 0xf2,0xc5,0xf2,0xc5,
				0xfe,0xe8,0xf4,0xc2, 0xf9,0x96,0x2d,0x28,
				0x92,0x30,0x44,0xbc, 0xd2,0x7f,0xe1,0x6e,
				0x62,0x02,0x8f,0x3d, 0x1c,0x80,0xda,0x0e,
				0x6a,0x90,0x7e,0x75, 0xff,0xec,0x3e,0xc4,
				0xcd,0x16,0x34,0x3b, 0x05,0x6d,0x4d,0x20,
				0x1c,0x7b,0xf5,0x57, 0x4f,0xfa,0x3d,0xac,
				0xd0,0x13,0x55,0xe8, 0xb3,0xe1,0x1b,0x78,
				0x30,0xe6,0x9f,0x84, 0xd4,0x69,0xd1,0x08,
				0x12,0x77,0xa7,0x4a, 0xbd,0xc0,0xf2,0xd2,
				0x78,0xdd,0xa3,0x81, 0x12,0xcb,0x6c,0x14,
				0x90,0x61,0xe2,0x84, 0xc6,0x2b,0x16,0xcc,
				0x40,0x99,0x50,0x88, 0x01,0x09,0x64,0x4f,
				0x0a,0x80,0xbe,0x61, 0xae,0x46,0xc9,0x0a,
				0x5d,0xe0,0xfb,0x72, 0x7a,0x1a,0xdd,0x61,
				0x63,0x20,0x05,0xa0, 0x4a,0xf0,0x60,0x69,
				0x7f,0x92,0xbc,0xbf, 0x4e,0x39,0x4d,0xdd,
				0x74,0xd1,0xb7,0xc0, 0x5a,0x34,0xb7,0xae,
				0x76,0x65,0x2e,0xbc, 0x36,0xb9,0x04,0x95,
				0x42,0xe9,0x6f,0xca, 0x78,0xb3,0x72,0x07,
				0xa3,0xba,0x02,0x94, 0x67,0x4c,0xb1,0xd7,
				0xe9,0x30,0x0d,0xf0, 0x3b,0xb8,0x10,0x6d,
				0xea,0x2b,0x21,0xbf, 0x74,0x59,0x82,0x97,
				0x85,0xaa,0xf1,0xd7, 0x54,0x39,0xeb,0x05,
				0xbd,0xf3,0x40,0xa0, 0x97,0xe6,0x74,0xfe,
				0xb4,0x82,0x5b,0xb1, 0x36,0xcb,0xe8,0x0d,
				0xce,0x14,0xd9,0xdf, 0xf1,0x94,0x22,0xcd,
				0xd6,0x00,0xba,0x04, 0x4c,0x05,0x0c,0xc0,
				0xd1,0x5a,0xeb,0x52, 0xd5,0xa8,0x8e,0xc8,
				0x97,0xa1,0xaa,0xc1, 0xea,0xc1,0xbe,0x7c,
				0x36,0xb3,0x36,0xa0, 0xc6,0x76,0x66,0xc5,
				0xe2,0xaf,0xd6,0x5c, 0xe2,0xdb,0x2c,0xb3,
				0x6c,0xb9,0x99,0x7f, 0xff,0x9f,0x03,0x24,
				0xe1,0x51,0x44,0x66, 0xd8,0x0c,0x5d,0x7f,
				0x5c,0x85,0x22,0x2a, 0xcf,0x6d,0x79,0x28,
				0xab,0x98,0x01,0x72, 0xfe,0x80,0x87,0x5f,
				0x46,0xba,0xef,0x81, 0x24,0xee,0xbf,0xb0,
				0x24,0x74,0xa3,0x65, 0x97,0x12,0xc4,0xaf,
				0x8b,0xa0,0x39,0xda, 0x8a,0x7e,0x74,0x6e,
				0x1b,0x42,0xb4,0x44, 0x37,0xfc,0x59,0xfd,
				0x86,0xed,0xfb,0x8c, 0x66,0x33,0xda,0x63,
				0x75,0xeb,0xe1,0xa4, 0x85,0x4f,0x50,0x8f,
				0x83,0x66,0x0d,0xd3, 0x37,0xfa,0xe6,0x9c,
				0x4f,0x30,0x87,0x35, 0x18,0xe3,0x0b,0xb7,
				0x6e,0x64,0x54,0xcd, 0x70,0xb3,0xde,0x54,
				0xb7,0x1d,0xe6,0x4c, 0x4d,0x55,0x12,0x12,
				0xaf,0x5f,0x7f,0x5e, 0xee,0x9d,0xe8,0x8e,
				0x32,0x9d,0x4e,0x75, 0xeb,0xc6,0xdd,0xaa,
				0x48,0x82,0xa4,0x3f, 0x3c,0xd7,0xd3,0xa8,
				0x63,0x9e,0x64,0xfe, 0xe3,0x97,0x00,0x62,
				0xe5,0x40,0x5d,0xc3, 0xad,0x72,0xe1,0x28,
				0x18,0x50,0xb7,0x75, 0xef,0xcd,0x23,0xbf,
				0x3f,0xc0,0x51,0x36, 0xf8,0x41,0xc3,0x08,
				0xcb,0xf1,0x8d,0x38, 0x34,0xbd,0x48,0x45,
				0x75,0xed,0xbc,0x65, 0x7b,0xb5,0x0c,0x9b,
				0xd7,0x67,0x7d,0x27, 0xb4,0xc4,0x80,0xd7,
				0xa9,0xb9,0xc7,0x4a, 0x97,0xaa,0xda,0xc8,
				0x3c,0x74,0xcf,0x36, 0x8f,0xe4,0x41,0xe3,
				0xd4,0xd3,0x26,0xa7, 0xf3,0x23,0x9d,0x8f,
				0x6c,0x20,0x05,0x32, 0x3e,0xe0,0xc3,0xc8,
				0x56,0x3f,0xa7,0x09, 0xb7,0xfb,0xc7,0xf7,
				0xbe,0x2a,0xdd,0x0f, 0x06,0x7b,0x0d,0xdd,
				0xb0,0xb4,0x86,0x17, 0xfd,0xb9,0x04,0xe5,
				0xc0,0x64,0x5d,0xad, 0x2a,0x36,0x38,0xdb,
				0x24,0xaf,0x5b,0xff, 0xca,0xf9,0x41,0xe8,
				0xf9,0x2f,0x1e,0x5e, 0xf9,0xf5,0xd5,0xf2,
				0xb2,0x88,0xca,0xc9, 0xa1,0x31,0xe2,0xe8,
				0x10,0x95,0x65,0xbf, 0xf1,0x11,0x61,0x7a,
				0x30,0x1a,0x54,0x90, 0xea,0xd2,0x30,0xf6,
				0xa5,0xad,0x60,0xf9, 0x4d,0x84,0x21,0x1b,
				0xe4,0x42,0x22,0xc8, 0x12,0x4b,0xb0,0x58,
				0x3e,0x9c,0x2d,0x32, 0x95,0x0a,0x8e,0xb0,
				0x0a,0x7e,0x77,0x2f, 0xe8,0x97,0x31,0x6a,
				0xf5,0x59,0xb4,0x26, 0xe6,0x37,0x12,0xc9,
				0xcb,0xa0,0x58,0x33, 0x6f,0xd5,0x55,0x55,
				0x3c,0xa1,0x33,0xb1, 0x0b,0x7e,0x2e,0xb4,
				0x43,0x2a,0x84,0x39, 0xf0,0x9c,0xf4,0x69,
				0x4f,0x1e,0x79,0xa6, 0x15,0x1b,0x87,0xbb,
				0xdb,0x9b,0xe0,0xf1, 0x0b,0xba,0xe3,0x6e,
				0xcc,0x2f,0x49,0x19, 0x22,0x29,0xfc,0x71,
				0xbb,0x77,0x38,0x18, 0x61,0xaf,0x85,0x76,
				0xeb,0xd1,0x09,0xcc, 0x86,0x04,0x20,0x9a,
				0x66,0x53,0x2f,0x44, 0x8b,0xc6,0xa3,0xd2,
				0x5f,0xc7,0x79,0x82, 0x66,0xa8,0x6e,0x75,
				0x7d,0x94,0xd1,0x86, 0x75,0x0f,0xa5,0x4f,
				0x3c,0x7a,0x33,0xce, 0xd1,0x6e,0x9d,0x7b,
				0x1f,0x91,0x37,0xb8, 0x37,0x80,0xfb,0xe0,
				0x52,0x26,0xd0,0x9a, 0xd4,0x48,0x02,0x41,
				0x05,0xe3,0x5a,0x94, 0xf1,0x65,0x61,0x19,
				0xb8,0x88,0x4e,0x2b, 0xea,0xba,0x8b,0x58,
				0x8b,0x42,0x01,0x00, 0xa8,0xfe,0x00,0x5c,
				0xfe,0x1c,0xee,0x31, 0x15,0x69,0xfa,0xb3,
				0x9b,0x5f,0x22,0x8e, 0x0d,0x2c,0xe3,0xa5,
				0x21,0xb9,0x99,0x8a, 0x8e,0x94,0x5a,0xef,
				0x13,0x3e,0x99,0x96, 0x79,0x6e,0xd5,0x42,
				0x36,0x03,0xa9,0xe2, 0xca,0x65,0x4e,0x8a,
				0x8a,0x30,0xd2,0x7d, 0x74,0xe7,0xf0,0xaa,
				0x23,0x26,0xdd,0xcb, 0x82,0x39,0xfc,0x9d,
				0x51,0x76,0x21,0x80, 0xa2,0xbe,0x93,0x03,
				0x47,0xb0,0xc1,0xb6, 0xdc,0x63,0xfd,0x9f,
				0xca,0x9d,0xa5,0xca, 0x27,0x85,0xe2,0xd8,
				0x15,0x5b,0x7e,0x14, 0x7a,0xc4,0x89,0xcc,
				0x74,0x14,0x4b,0x46, 0xd2,0xce,0xac,0x39,
				0x6b,0x6a,0x5a,0xa4, 0x0e,0xe3,0x7b,0x15,
				0x94,0x4b,0x0f,0x74, 0xcb,0x0c,0x7f,0xa9,
				0xbe,0x09,0x39,0xa3, 0xdd,0x56,0x5c,0xc7,
				0x99,0x56,0x65,0x39, 0xf4,0x0b,0x7d,0x87,
				0xec,0xaa,0xe3,0x4d, 0x22,0x65,0x39,0x4e,
			},
			.h = {
				0x64,0x3a,0xbc,0xc3, 0x3f,0x74,0x40,0x51,
				0x6e,0x56,0x01,0x1a, 0x51,0xec,0x36,0xde,
			},
		},
	};
	const uint8_t *pk;
	const uint8_t *nhk;
	static uint32_t nhk32[268];
	uint8_t h[16];
	unsigned i, j;
	int result = 0;

	for (i = 0; i < __arraycount(C); i++) {
		pk = C[i].k;
		nhk = C[i].k + 16;
		for (j = 0; j < 268; j++)
			nhk32[j] = le32dec(nhk + 4*j);
		nhpoly1305(h, C[i].m, C[i].mlen, pk, nhk32);
		if (memcmp(h, C[i].h, 16)) {
			char prefix[16];
			snprintf(prefix, sizeof prefix, "nhpoly1305 %u", i);
			hexdump(printf, prefix, h, 32);
			result = -1;
		}
	}

	return result;
}

void
adiantum_init(struct adiantum *A, const uint8_t key[static 32])
{
	uint8_t nonce[24] = {1};
	unsigned i;

	memcpy(A->ks, key, 32);

	/* Relies on ordering of struct members.  */
	memset(A->kk, 0, 32 + 16 + 16 + 1072);
	xchacha_stream_xor(A->kk, A->kk, 32 + 16 + 16 + 1072, 0, nonce, A->ks,
	    12);

	/* Put the NH key words into host byte order.  */
	for (i = 0; i < __arraycount(A->kn); i++)
		A->kn[i] = le32toh(A->kn[i]);

	/* Expand the AES key.  */
	aes_setenckey256(&A->kk_enc, A->kk);
	aes_setdeckey256(&A->kk_dec, A->kk);
}

static void
adiantum_hash(uint8_t h[static 16], const void *l, size_t llen,
    const void *t, size_t tlen,
    const uint8_t kt[static 16],
    const uint8_t kl[static 16],
    const uint32_t kn[static 268])
{
	struct poly1305 P;
	uint8_t llenbuf[16];
	uint8_t ht[16];
	uint8_t hl[16];

	KASSERT(llen % 16 == 0);

	memset(llenbuf, 0, sizeof llenbuf);
	le64enc(llenbuf, 8*llen);

	/* Compute H_T := Poly1305_{K_T}(le128(|l|) || tweak).  */
	poly1305_init(&P, kt);
	poly1305_update_blocks(&P, llenbuf, 16);
	poly1305_update_blocks(&P, t, tlen);
	poly1305_final(ht, &P);

	/* Compute H_L := Poly1305_{K_L}(NH(pad_128(l))).  */
	nhpoly1305(hl, l, llen, kl, kn);

	/* Compute H := H_T + H_L (mod 2^128).  */
	add128(h, ht, hl);
}

void
adiantum_enc(void *c, const void *p, size_t len, const void *t, size_t tlen,
    const struct adiantum *A)
{
	size_t Rlen = 16;
	size_t Llen = len - Rlen;
	uint8_t *c8 = c;
	uint8_t *cL = c8;
	uint8_t *cR = c8 + Llen;
	const uint8_t *p8 = p;
	const uint8_t *pL = p8;
	const uint8_t *pR = p8 + Llen;
	uint8_t h[16];
	uint8_t buf[16] __aligned(16);
	uint8_t nonce[24];

	KASSERT(len % 16 == 0);

	adiantum_hash(h, pL, Llen, t, tlen, A->kt, A->kl, A->kn);
	add128(buf, pR, h);	/* buf := P_M */
	aes_enc(&A->kk_enc, buf, buf, AES_256_NROUNDS); /* buf := C_M */

	memcpy(nonce, buf, 16);
	le64enc(nonce + 16, 1);
	xchacha_stream_xor(cL, pL, Llen, 0, nonce, A->ks, 12);

	adiantum_hash(h, cL, Llen, t, tlen, A->kt, A->kl, A->kn);
	sub128(cR, buf, h);

	explicit_memset(h, 0, sizeof h);
	explicit_memset(buf, 0, sizeof buf);
}

void
adiantum_dec(void *p, const void *c, size_t len, const void *t, size_t tlen,
    const struct adiantum *A)
{
	size_t Rlen = 16;
	size_t Llen = len - Rlen;
	const uint8_t *c8 = c;
	const uint8_t *cL = c8;
	const uint8_t *cR = c8 + Llen;
	uint8_t *p8 = p;
	uint8_t *pL = p8;
	uint8_t *pR = p8 + Llen;
	uint8_t h[16];
	uint8_t buf[16] __aligned(16);
	uint8_t nonce[24];

	KASSERT(len % 16 == 0);

	adiantum_hash(h, cL, Llen, t, tlen, A->kt, A->kl, A->kn);
	add128(buf, cR, h);	/* buf := C_M */

	memcpy(nonce, buf, 16);
	le64enc(nonce + 16, 1);
	xchacha_stream_xor(pL, cL, Llen, 0, nonce, A->ks, 12);

	aes_dec(&A->kk_dec, buf, buf, AES_256_NROUNDS); /* buf := P_M */
	adiantum_hash(h, pL, Llen, t, tlen, A->kt, A->kl, A->kn);
	sub128(pR, buf, h);

	explicit_memset(h, 0, sizeof h);
	explicit_memset(buf, 0, sizeof buf);
}

#ifdef _KERNEL

MODULE(MODULE_CLASS_MISC, adiantum, "aes,chacha");

static int
adiantum_modcmd(modcmd_t cmd, void *opaque)
{

	switch (cmd) {
	case MODULE_CMD_INIT: {
		int result = 0;
		result |= addsub128_selftest();
		result |= poly1305_selftest();
		result |= nh_selftest();
		result |= nhpoly1305_selftest();
		result |= adiantum_selftest();
		if (result)
			panic("adiantum self-test failed");
		aprint_debug("adiantum: self-test passed\n");
		return 0;
	}
	case MODULE_CMD_FINI:
		return 0;
	default:
		return ENOTTY;
	}
}

#else  /* !defined(_KERNEL) */

#include <err.h>
#include <stdio.h>
#include <unistd.h>

static int
read_block(int fd, void *buf, size_t len)
{
	char *p = buf;
	size_t n = len;
	ssize_t nread;

	for (;;) {
		if ((nread = read(fd, p, n)) == -1)
			err(1, "read");
		if (nread == 0) {
			if (n < len)
				errx(1, "partial block");
			return -1; /* eof */
		}
		if ((size_t)nread >= n)
			break;
		p += (size_t)nread;
		n -= (size_t)nread;
	}

	return 0;
}

static void
write_block(int fd, const void *buf, size_t len)
{
	const char *p = buf;
	size_t n = len;
	ssize_t nwrit;

	for (;;) {
		if ((nwrit = write(fd, p, n)) == -1)
			err(1, "write");
		if ((size_t)nwrit >= n)
			break;
		p += (size_t)nwrit;
		n -= (size_t)nwrit;
	}
}

#define	SECSIZE	512

static void
process(void)
{
	static const uint8_t k[32] = {0};
	static uint8_t buf[65536];
	static struct adiantum C;
	uint8_t blkno[16] = {0};
	unsigned i;

	adiantum_init(&C, k);
	while (read_block(STDIN_FILENO, buf, sizeof buf) == 0) {
		for (i = 0; i < sizeof buf; i += SECSIZE) {
			adiantum_enc(buf + i, buf + i, SECSIZE, blkno, 16, &C);
			le64enc(blkno, 1 + le32dec(blkno));
		}
		write_block(STDOUT_FILENO, buf, sizeof buf);
		if (le64dec(blkno) == 1024*1024*1024/SECSIZE)
			return;
	}
}

int
main(void)
{
	int result = 0;

	result |= addsub128_selftest();
	result |= poly1305_selftest();
	result |= nh_selftest();
	result |= nhpoly1305_selftest();
	result |= adiantum_selftest();
	if (result)
		return result;

	process();
	return 0;
}

#endif	/* _KERNEL */
