/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Duration,
  TalerCorebankApiClient,
  TalerMerchantApi,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  HarnessExchangeBankAccount,
  MerchantService,
  getTestHarnessPaytoForLabel,
  setupDb,
} from "../harness/harness.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runMultiExchangeTest(t: GlobalTestState) {
  // Set up test environment
  const dbDefault = await setupDb(t);

  const dbExchangeTwo = await setupDb(t, {
    nameSuffix: "exchange2",
  });

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: dbDefault.connStr,
    httpPort: 8082,
  });

  const exchangeOne = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: dbDefault.connStr,
  });

  const exchangeTwo = ExchangeService.create(t, {
    name: "testexchange-2",
    currency: "TESTKUDOS",
    httpPort: 8281,
    database: dbExchangeTwo.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: dbDefault.connStr,
  });

  let exchangeOneBankAccount: HarnessExchangeBankAccount = {
    wireGatewayApiBaseUrl: new URL(
      "accounts/myexchange/taler-wire-gateway/",
      bank.corebankApiBaseUrl,
    ).href,
    wireGatewayAuth: {
      username: "myexchange",
      password: "password",
    },
    accountPaytoUri: getTestHarnessPaytoForLabel("myexchange"),
  };

  let exchangeTwoBankAccount: HarnessExchangeBankAccount = {
    wireGatewayAuth: {
      username: "myexchange2",
      password: "password",
    },
    wireGatewayApiBaseUrl: new URL(
      "accounts/myexchange2/taler-wire-gateway/",
      bank.corebankApiBaseUrl,
    ).href,
    accountPaytoUri: getTestHarnessPaytoForLabel("myexchange2"),
  };

  bank.setSuggestedExchange(
    exchangeOne,
    exchangeOneBankAccount.accountPaytoUri,
  );

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: exchangeOneBankAccount.wireGatewayAuth.username,
    username: exchangeOneBankAccount.wireGatewayAuth.username,
    password: exchangeOneBankAccount.wireGatewayAuth.password,
    is_taler_exchange: true,
    payto_uri: exchangeOneBankAccount.accountPaytoUri,
  });
  await exchangeOne.addBankAccount("1", exchangeOneBankAccount);

  await bankClient.registerAccountExtended({
    name: exchangeTwoBankAccount.wireGatewayAuth.username,
    username: exchangeTwoBankAccount.wireGatewayAuth.username,
    password: exchangeTwoBankAccount.wireGatewayAuth.password,
    is_taler_exchange: true,
    payto_uri: exchangeTwoBankAccount.accountPaytoUri,
  });
  await exchangeTwo.addBankAccount("1", exchangeTwoBankAccount);

  // Set up the first exchange

  exchangeOne.addOfferedCoins(defaultCoinConfig);
  await exchangeOne.start();
  await exchangeOne.pingUntilAvailable();

  // Set up the second exchange

  exchangeTwo.addOfferedCoins(defaultCoinConfig);
  await exchangeTwo.start();
  await exchangeTwo.pingUntilAvailable();

  // Start and configure merchant

  merchant.addExchange(exchangeOne);
  merchant.addExchange(exchangeTwo);

  await merchant.start();
  await merchant.pingUntilAvailable();

  const { accessToken: adminAccessToken } = await merchant.addInstanceWithWireAccount({
    id: "admin",
    name: "Default Instance",
    paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [getTestHarnessPaytoForLabel("minst1")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  }, {adminAccessToken});

  const { walletClient, walletService } = await createWalletDaemonWithClient(
    t,
    { name: "wallet" },
  );

  console.log("setup done!");

  // Withdraw digital cash into the wallet.

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange: exchangeOne,
    amount: "TESTKUDOS:6",
  });

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange: exchangeTwo,
    amount: "TESTKUDOS:6",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const order: TalerMerchantApi.Order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:10",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  console.log("making test payment");

  await makeTestPaymentV2(t, { walletClient, merchant, order, merchantAdminAccessToken: adminAccessToken });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runMultiExchangeTest.suites = ["wallet"];
