/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.message.CreateDelegationTokenRequestData;
import org.apache.kafka.common.message.CreateDelegationTokenResponseData;
import org.apache.kafka.common.message.ExpireDelegationTokenRequestData;
import org.apache.kafka.common.message.ExpireDelegationTokenResponseData;
import org.apache.kafka.common.message.RenewDelegationTokenRequestData;
import org.apache.kafka.common.message.RenewDelegationTokenResponseData;
import org.apache.kafka.common.metadata.DelegationTokenRecord;
import org.apache.kafka.common.metadata.RemoveDelegationTokenRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.security.token.delegation.TokenInformation;
import org.apache.kafka.common.security.token.delegation.internals.DelegationTokenCache;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.ControllerRequestContext;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.metadata.DelegationTokenData;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.common.MetadataVersion;
import org.slf4j.Logger;

public class DelegationTokenControlManager {
    private Time time = Time.SYSTEM;
    private final Logger log;
    private final DelegationTokenCache tokenCache;
    private final String tokenSecretKeyString;
    private final long tokenDefaultMaxLifetimeMs;
    private final long tokenDefaultRenewLifetimeMs;

    private DelegationTokenControlManager(LogContext logContext, DelegationTokenCache tokenCache, String tokenSecretKeyString, long tokenDefaultMaxLifetimeMs, long tokenDefaultRenewLifetimeMs) {
        this.log = logContext.logger(DelegationTokenControlManager.class);
        this.tokenCache = tokenCache;
        this.tokenSecretKeyString = tokenSecretKeyString;
        this.tokenDefaultMaxLifetimeMs = tokenDefaultMaxLifetimeMs;
        this.tokenDefaultRenewLifetimeMs = tokenDefaultRenewLifetimeMs;
    }

    private static byte[] toBytes(String str) {
        return str.getBytes(StandardCharsets.UTF_8);
    }

    private byte[] createHmac(String tokenId) throws Exception {
        Mac mac = Mac.getInstance("HmacSHA512");
        SecretKeySpec secretKey = new SecretKeySpec(DelegationTokenControlManager.toBytes(this.tokenSecretKeyString), mac.getAlgorithm());
        mac.init(secretKey);
        return mac.doFinal(DelegationTokenControlManager.toBytes(tokenId));
    }

    private TokenInformation getToken(byte[] hmac) {
        String base64Pwd = Base64.getEncoder().encodeToString(hmac);
        return this.tokenCache.tokenForHmac(base64Pwd);
    }

    private boolean allowedToRenew(TokenInformation tokenInfo, KafkaPrincipal renewer) {
        return tokenInfo.owner().equals((Object)renewer) || tokenInfo.renewers().contains(renewer);
    }

    public boolean isEnabled() {
        return this.tokenSecretKeyString != null;
    }

    public ControllerResult<CreateDelegationTokenResponseData> createDelegationToken(ControllerRequestContext context, CreateDelegationTokenRequestData requestData, MetadataVersion metadataVersion) {
        byte[] hmac;
        long now = this.time.milliseconds();
        KafkaPrincipal owner = context.principal();
        if (requestData.ownerPrincipalName() != null && !requestData.ownerPrincipalName().isEmpty()) {
            owner = new KafkaPrincipal(requestData.ownerPrincipalType(), requestData.ownerPrincipalName());
        }
        CreateDelegationTokenResponseData responseData = new CreateDelegationTokenResponseData().setPrincipalName(owner.getName()).setPrincipalType(owner.getPrincipalType()).setTokenRequesterPrincipalName(context.principal().getName()).setTokenRequesterPrincipalType(context.principal().getPrincipalType());
        ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
        if (!this.isEnabled()) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_AUTH_DISABLED.code()));
        }
        if (!metadataVersion.isDelegationTokenSupported()) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.UNSUPPORTED_VERSION.code()));
        }
        long maxLifeTime = this.tokenDefaultMaxLifetimeMs;
        if (requestData.maxLifetimeMs() > 0L) {
            maxLifeTime = Math.min(maxLifeTime, requestData.maxLifetimeMs());
        }
        long maxTimestamp = now + maxLifeTime;
        long expiryTimestamp = Math.min(maxTimestamp, now + this.tokenDefaultRenewLifetimeMs);
        String tokenId = Uuid.randomUuid().toString();
        ArrayList<KafkaPrincipal> renewers = new ArrayList<KafkaPrincipal>();
        for (CreateDelegationTokenRequestData.CreatableRenewers renewer : requestData.renewers()) {
            if (renewer.principalType().equals("User")) {
                renewers.add(new KafkaPrincipal(renewer.principalType(), renewer.principalName()));
                continue;
            }
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.INVALID_PRINCIPAL_TYPE.code()));
        }
        try {
            hmac = this.createHmac(tokenId);
        }
        catch (Throwable e) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(ApiError.fromThrowable((Throwable)e).error().code()));
        }
        TokenInformation newTokenInformation = new TokenInformation(tokenId, owner, context.principal(), renewers, now, maxTimestamp, expiryTimestamp);
        DelegationTokenData newDelegationTokenData = new DelegationTokenData(newTokenInformation);
        responseData.setErrorCode(Errors.NONE.code()).setIssueTimestampMs(now).setExpiryTimestampMs(expiryTimestamp).setMaxTimestampMs(maxTimestamp).setTokenId(tokenId).setHmac(hmac);
        records.add(new ApiMessageAndVersion((ApiMessage)newDelegationTokenData.toRecord(), 0));
        return ControllerResult.atomicOf(records, responseData);
    }

    public ControllerResult<RenewDelegationTokenResponseData> renewDelegationToken(ControllerRequestContext context, RenewDelegationTokenRequestData requestData, MetadataVersion metadataVersion) {
        long now = this.time.milliseconds();
        ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
        RenewDelegationTokenResponseData responseData = new RenewDelegationTokenResponseData();
        if (!this.isEnabled()) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_AUTH_DISABLED.code()));
        }
        if (!metadataVersion.isDelegationTokenSupported()) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.UNSUPPORTED_VERSION.code()));
        }
        TokenInformation myTokenInformation = this.getToken(requestData.hmac());
        if (myTokenInformation == null) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_NOT_FOUND.code()));
        }
        if (myTokenInformation.maxTimestamp() < now || myTokenInformation.expiryTimestamp() < now) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_EXPIRED.code()));
        }
        if (!this.allowedToRenew(myTokenInformation, context.principal())) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_OWNER_MISMATCH.code()));
        }
        long renewLifeTime = this.tokenDefaultRenewLifetimeMs;
        if (requestData.renewPeriodMs() > 0L) {
            renewLifeTime = Math.min(renewLifeTime, requestData.renewPeriodMs());
        }
        long renewTimeStamp = now + renewLifeTime;
        long expiryTimestamp = Math.min(myTokenInformation.maxTimestamp(), renewTimeStamp);
        DelegationTokenData newDelegationTokenData = new DelegationTokenData(myTokenInformation);
        responseData.setErrorCode(Errors.NONE.code()).setExpiryTimestampMs(expiryTimestamp);
        records.add(new ApiMessageAndVersion((ApiMessage)newDelegationTokenData.toRecord().setExpirationTimestamp(expiryTimestamp), 0));
        return ControllerResult.atomicOf(records, responseData);
    }

    public ControllerResult<ExpireDelegationTokenResponseData> expireDelegationToken(ControllerRequestContext context, ExpireDelegationTokenRequestData requestData, MetadataVersion metadataVersion) {
        long now = this.time.milliseconds();
        ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
        ExpireDelegationTokenResponseData responseData = new ExpireDelegationTokenResponseData();
        if (!this.isEnabled()) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_AUTH_DISABLED.code()));
        }
        if (!metadataVersion.isDelegationTokenSupported()) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.UNSUPPORTED_VERSION.code()));
        }
        TokenInformation myTokenInformation = this.getToken(requestData.hmac());
        if (myTokenInformation == null) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_NOT_FOUND.code()));
        }
        if (!this.allowedToRenew(myTokenInformation, context.principal())) {
            return ControllerResult.atomicOf(records, responseData.setErrorCode(Errors.DELEGATION_TOKEN_OWNER_MISMATCH.code()));
        }
        if (requestData.expiryTimePeriodMs() < 0L) {
            responseData.setErrorCode(Errors.NONE.code()).setExpiryTimestampMs(now);
            records.add(new ApiMessageAndVersion((ApiMessage)new RemoveDelegationTokenRecord().setTokenId(myTokenInformation.tokenId()), 0));
        } else if (myTokenInformation.maxTimestamp() < now || myTokenInformation.expiryTimestamp() < now) {
            responseData.setErrorCode(Errors.DELEGATION_TOKEN_EXPIRED.code());
        } else {
            long expiryTimestamp = Math.min(myTokenInformation.maxTimestamp(), now + requestData.expiryTimePeriodMs());
            responseData.setErrorCode(Errors.NONE.code()).setExpiryTimestampMs(expiryTimestamp);
            DelegationTokenData newDelegationTokenData = new DelegationTokenData(myTokenInformation);
            records.add(new ApiMessageAndVersion((ApiMessage)newDelegationTokenData.toRecord().setExpirationTimestamp(expiryTimestamp), 0));
        }
        return ControllerResult.atomicOf(records, responseData);
    }

    public List<ApiMessageAndVersion> sweepExpiredDelegationTokens() {
        long now = this.time.milliseconds();
        ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
        for (TokenInformation oldTokenInformation : this.tokenCache.tokens()) {
            if (oldTokenInformation.maxTimestamp() >= now && oldTokenInformation.expiryTimestamp() >= now) continue;
            this.log.info("Delegation token expired for token: {} for owner: {}", (Object)oldTokenInformation.tokenId(), (Object)oldTokenInformation.ownerAsString());
            records.add(new ApiMessageAndVersion((ApiMessage)new RemoveDelegationTokenRecord().setTokenId(oldTokenInformation.tokenId()), 0));
        }
        return records;
    }

    public void replay(DelegationTokenRecord record) {
        this.log.info("Replayed DelegationTokenRecord for {}.", (Object)record.tokenId());
    }

    public void replay(RemoveDelegationTokenRecord record) {
        this.log.info("Replayed RemoveDelegationTokenRecord for {}.", (Object)record.tokenId());
    }

    static class Builder {
        private LogContext logContext = null;
        private DelegationTokenCache tokenCache = null;
        private String tokenSecretKeyString = null;
        private long tokenDefaultMaxLifetimeMs = 0L;
        private long tokenDefaultRenewLifetimeMs = 0L;

        Builder() {
        }

        Builder setLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder setTokenCache(DelegationTokenCache tokenCache) {
            this.tokenCache = tokenCache;
            return this;
        }

        Builder setDelegationTokenSecretKey(String tokenSecretKeyString) {
            this.tokenSecretKeyString = tokenSecretKeyString;
            return this;
        }

        Builder setDelegationTokenMaxLifeMs(long tokenDefaultMaxLifetimeMs) {
            this.tokenDefaultMaxLifetimeMs = tokenDefaultMaxLifetimeMs;
            return this;
        }

        Builder setDelegationTokenExpiryTimeMs(long tokenDefaultRenewLifetimeMs) {
            this.tokenDefaultRenewLifetimeMs = tokenDefaultRenewLifetimeMs;
            return this;
        }

        DelegationTokenControlManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            return new DelegationTokenControlManager(this.logContext, this.tokenCache, this.tokenSecretKeyString, this.tokenDefaultMaxLifetimeMs, this.tokenDefaultRenewLifetimeMs);
        }
    }
}

