/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.computer.core.output.hg.task;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import org.apache.hugegraph.computer.core.config.ComputerOptions;
import org.apache.hugegraph.computer.core.config.Config;
import org.apache.hugegraph.computer.core.output.hg.exceptions.WriteBackException;
import org.apache.hugegraph.computer.core.output.hg.metrics.LoadSummary;
import org.apache.hugegraph.computer.core.output.hg.metrics.Printer;
import org.apache.hugegraph.computer.core.output.hg.task.BatchInsertTask;
import org.apache.hugegraph.computer.core.output.hg.task.SingleInsertTask;
import org.apache.hugegraph.driver.HugeClient;
import org.apache.hugegraph.driver.HugeClientBuilder;
import org.apache.hugegraph.structure.graph.Vertex;
import org.apache.hugegraph.util.ExecutorUtil;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public final class TaskManager {
    private static final Logger LOG = Log.logger(TaskManager.class);
    public static final String BATCH_WORKER = "batch-worker-%d";
    public static final String SINGLE_WORKER = "single-worker-%d";
    private HugeClient client;
    private Config config;
    private final Semaphore batchSemaphore;
    private final Semaphore singleSemaphore;
    private final ExecutorService batchService;
    private final ExecutorService singleService;
    private LoadSummary loadSummary;

    public TaskManager(Config config) {
        this.config = config;
        String url = (String)config.get(ComputerOptions.HUGEGRAPH_URL);
        String graph = (String)config.get(ComputerOptions.HUGEGRAPH_GRAPH_NAME);
        this.client = new HugeClientBuilder(url, graph).build();
        this.batchSemaphore = new Semaphore(this.batchSemaphoreNum());
        this.singleSemaphore = new Semaphore(this.singleSemaphoreNum());
        this.batchService = ExecutorUtil.newFixedThreadPool((Integer)config.get(ComputerOptions.OUTPUT_BATCH_THREADS), BATCH_WORKER);
        this.singleService = ExecutorUtil.newFixedThreadPool((Integer)config.get(ComputerOptions.OUTPUT_SINGLE_THREADS), SINGLE_WORKER);
        this.loadSummary = new LoadSummary();
        this.loadSummary.startTimer();
    }

    public HugeClient client() {
        return this.client;
    }

    private int batchSemaphoreNum() {
        return 1 + (Integer)this.config.get(ComputerOptions.OUTPUT_BATCH_THREADS);
    }

    private int singleSemaphoreNum() {
        return 2 * (Integer)this.config.get(ComputerOptions.OUTPUT_SINGLE_THREADS);
    }

    public void waitFinished() {
        LOG.info("Waiting for the insert tasks finished");
        try {
            this.batchSemaphore.acquire(this.batchSemaphoreNum());
            LOG.info("The batch-mode tasks stopped");
        }
        catch (InterruptedException e) {
            LOG.error("Interrupted while waiting batch-mode tasks");
        }
        finally {
            this.batchSemaphore.release(this.batchSemaphoreNum());
        }
        try {
            this.singleSemaphore.acquire(this.singleSemaphoreNum());
            LOG.info("The single-mode tasks stopped");
        }
        catch (InterruptedException e) {
            LOG.error("Interrupted while waiting single-mode tasks");
        }
        finally {
            this.singleSemaphore.release(this.singleSemaphoreNum());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        long timeout = ((Integer)this.config.get(ComputerOptions.OUTPUT_THREAD_POOL_SHUTDOWN_TIMEOUT)).intValue();
        try {
            this.batchService.shutdown();
            this.batchService.awaitTermination(timeout, TimeUnit.SECONDS);
            LOG.info("The batch-mode tasks service executor shutdown");
        }
        catch (InterruptedException e) {
            LOG.error("The batch-mode tasks are interrupted");
        }
        finally {
            if (!this.batchService.isTerminated()) {
                LOG.error("The unfinished batch-mode tasks will be cancelled");
            }
            this.batchService.shutdownNow();
        }
        try {
            this.singleService.shutdown();
            this.singleService.awaitTermination(timeout, TimeUnit.SECONDS);
            LOG.info("The single-mode tasks service executor shutdown");
        }
        catch (InterruptedException e) {
            LOG.error("The single-mode tasks are interrupted");
        }
        finally {
            if (!this.singleService.isTerminated()) {
                LOG.error("The unfinished single-mode tasks will be cancelled");
            }
            this.singleService.shutdownNow();
        }
        this.loadSummary.stopTimer();
        Printer.printSummary(this.loadSummary);
        this.client.close();
    }

    public void submitBatch(List<Vertex> batch) {
        try {
            this.batchSemaphore.acquire();
        }
        catch (InterruptedException e2) {
            throw new WriteBackException("Interrupted while waiting to submit batch", e2);
        }
        BatchInsertTask task = new BatchInsertTask(this.config, this.client, batch, this.loadSummary);
        ((CompletableFuture)CompletableFuture.runAsync(task, this.batchService).exceptionally(e -> {
            LOG.warn("Batch insert error, try single insert", (Throwable)e);
            this.submitInSingle(batch);
            return null;
        })).whenComplete((r, e) -> this.batchSemaphore.release());
    }

    private void submitInSingle(List<Vertex> batch) {
        try {
            this.singleSemaphore.acquire();
        }
        catch (InterruptedException e2) {
            throw new WriteBackException("Interrupted while waiting to submit single", e2);
        }
        SingleInsertTask task = new SingleInsertTask(this.config, this.client, batch, this.loadSummary);
        CompletableFuture.runAsync(task, this.singleService).whenComplete((r, e) -> this.singleSemaphore.release());
    }
}

