/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.netty;

import java.io.IOException;
import java.lang.reflect.Field;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.SocketException;
import java.net.SocketOption;
import javax.annotation.Nullable;
import jdk.net.ExtendedSocketOptions;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.runtime.io.network.netty.NettyBufferPool;
import org.apache.flink.runtime.io.network.netty.NettyConfig;
import org.apache.flink.runtime.io.network.netty.NettyProtocol;
import org.apache.flink.runtime.io.network.netty.NettyServer;
import org.apache.flink.runtime.io.network.netty.SSLHandlerFactory;
import org.apache.flink.shaded.netty4.io.netty.bootstrap.Bootstrap;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelException;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelFuture;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelHandler;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelInitializer;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelOption;
import org.apache.flink.shaded.netty4.io.netty.channel.EventLoopGroup;
import org.apache.flink.shaded.netty4.io.netty.channel.epoll.Epoll;
import org.apache.flink.shaded.netty4.io.netty.channel.epoll.EpollChannelOption;
import org.apache.flink.shaded.netty4.io.netty.channel.epoll.EpollEventLoopGroup;
import org.apache.flink.shaded.netty4.io.netty.channel.epoll.EpollSocketChannel;
import org.apache.flink.shaded.netty4.io.netty.channel.nio.NioEventLoopGroup;
import org.apache.flink.shaded.netty4.io.netty.channel.socket.SocketChannel;
import org.apache.flink.shaded.netty4.io.netty.channel.socket.nio.NioChannelOption;
import org.apache.flink.shaded.netty4.io.netty.channel.socket.nio.NioSocketChannel;
import org.apache.flink.shaded.netty4.io.netty.handler.ssl.SslHandler;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class NettyClient {
    private static final Logger LOG = LoggerFactory.getLogger(NettyClient.class);
    @VisibleForTesting
    static final String NIO_TCP_KEEPIDLE_KEY = "TCP_KEEPIDLE";
    @VisibleForTesting
    static final String NIO_TCP_KEEPINTERVAL_KEY = "TCP_KEEPINTERVAL";
    @VisibleForTesting
    static final String NIO_TCP_KEEPCOUNT_KEY = "TCP_KEEPCOUNT";
    private final NettyConfig config;
    private NettyProtocol protocol;
    private Bootstrap bootstrap;
    @Nullable
    private SSLHandlerFactory clientSSLFactory;

    NettyClient(NettyConfig config) {
        this.config = config;
    }

    void init(NettyProtocol protocol, NettyBufferPool nettyBufferPool) throws IOException {
        Preconditions.checkState((this.bootstrap == null ? 1 : 0) != 0, (Object)"Netty client has already been initialized.");
        this.protocol = protocol;
        long start = System.nanoTime();
        this.bootstrap = new Bootstrap();
        switch (this.config.getTransportType()) {
            case NIO: {
                this.initNioBootstrap();
                break;
            }
            case EPOLL: {
                this.initEpollBootstrap();
                break;
            }
            case AUTO: {
                if (Epoll.isAvailable()) {
                    this.initEpollBootstrap();
                    LOG.info("Transport type 'auto': using EPOLL.");
                    break;
                }
                this.initNioBootstrap();
                LOG.info("Transport type 'auto': using NIO.");
            }
        }
        this.bootstrap.option(ChannelOption.TCP_NODELAY, (Object)true);
        this.bootstrap.option(ChannelOption.SO_KEEPALIVE, (Object)true);
        this.bootstrap.option(ChannelOption.CONNECT_TIMEOUT_MILLIS, (Object)(this.config.getClientConnectTimeoutSeconds() * 1000));
        this.bootstrap.option(ChannelOption.ALLOCATOR, (Object)nettyBufferPool);
        int receiveAndSendBufferSize = this.config.getSendAndReceiveBufferSize();
        if (receiveAndSendBufferSize > 0) {
            this.bootstrap.option(ChannelOption.SO_SNDBUF, (Object)receiveAndSendBufferSize);
            this.bootstrap.option(ChannelOption.SO_RCVBUF, (Object)receiveAndSendBufferSize);
        }
        try {
            this.clientSSLFactory = this.config.createClientSSLEngineFactory();
        }
        catch (Exception e) {
            throw new IOException("Failed to initialize SSL Context for the Netty client", e);
        }
        long duration = (System.nanoTime() - start) / 1000000L;
        LOG.info("Successful initialization (took {} ms).", (Object)duration);
    }

    NettyConfig getConfig() {
        return this.config;
    }

    Bootstrap getBootstrap() {
        return this.bootstrap;
    }

    void shutdown() {
        long start = System.nanoTime();
        if (this.bootstrap != null) {
            if (this.bootstrap.config().group() != null) {
                this.bootstrap.config().group().shutdownGracefully();
            }
            this.bootstrap = null;
        }
        long duration = (System.nanoTime() - start) / 1000000L;
        LOG.info("Successful shutdown (took {} ms).", (Object)duration);
    }

    private void initNioBootstrap() {
        String name = "Flink Netty Client (" + this.config.getServerPortRange() + ")";
        NioEventLoopGroup nioGroup = new NioEventLoopGroup(this.config.getClientNumThreads(), NettyServer.getNamedThreadFactory(name));
        ((Bootstrap)this.bootstrap.group((EventLoopGroup)nioGroup)).channel(NioSocketChannel.class);
        this.config.getTcpKeepIdleInSeconds().ifPresent(idle -> this.setNioKeepaliveOptions(NIO_TCP_KEEPIDLE_KEY, (int)idle));
        this.config.getTcpKeepInternalInSeconds().ifPresent(interval -> this.setNioKeepaliveOptions(NIO_TCP_KEEPINTERVAL_KEY, (int)interval));
        this.config.getTcpKeepCount().ifPresent(count -> this.setNioKeepaliveOptions(NIO_TCP_KEEPCOUNT_KEY, (int)count));
    }

    private void setNioKeepaliveOptions(String option, int value) {
        try {
            Field field = ExtendedSocketOptions.class.getField(option);
            this.bootstrap.option(NioChannelOption.of((SocketOption)((SocketOption)field.get(null))), (Object)value);
        }
        catch (IllegalAccessException | NoSuchFieldException e) {
            LOG.error("Ignore keepalive option {}, this may be due to using netty transport type of nio and an older version of jdk 8, refer to https://bugs.openjdk.org/browse/JDK-8194298", (Object)option, (Object)e);
        }
    }

    private void initEpollBootstrap() {
        String name = "Flink Netty Client (" + this.config.getServerPortRange() + ")";
        EpollEventLoopGroup epollGroup = new EpollEventLoopGroup(this.config.getClientNumThreads(), NettyServer.getNamedThreadFactory(name));
        ((Bootstrap)this.bootstrap.group((EventLoopGroup)epollGroup)).channel(EpollSocketChannel.class);
        this.config.getTcpKeepIdleInSeconds().ifPresent(idle -> {
            Bootstrap cfr_ignored_0 = (Bootstrap)this.bootstrap.option(EpollChannelOption.TCP_KEEPIDLE, idle);
        });
        this.config.getTcpKeepInternalInSeconds().ifPresent(interval -> {
            Bootstrap cfr_ignored_0 = (Bootstrap)this.bootstrap.option(EpollChannelOption.TCP_KEEPINTVL, interval);
        });
        this.config.getTcpKeepCount().ifPresent(count -> {
            Bootstrap cfr_ignored_0 = (Bootstrap)this.bootstrap.option(EpollChannelOption.TCP_KEEPCNT, count);
        });
    }

    ChannelFuture connect(final InetSocketAddress serverSocketAddress) {
        Preconditions.checkState((this.bootstrap != null ? 1 : 0) != 0, (Object)"Client has not been initialized yet.");
        this.bootstrap.handler((ChannelHandler)new ChannelInitializer<SocketChannel>(){

            public void initChannel(SocketChannel channel) throws Exception {
                if (NettyClient.this.clientSSLFactory != null) {
                    SslHandler sslHandler = NettyClient.this.clientSSLFactory.createNettySSLHandler(channel.alloc(), serverSocketAddress.getAddress().getCanonicalHostName(), serverSocketAddress.getPort());
                    channel.pipeline().addLast("ssl", (ChannelHandler)sslHandler);
                }
                channel.pipeline().addLast(NettyClient.this.protocol.getClientChannelHandlers());
            }
        });
        try {
            return this.bootstrap.connect((SocketAddress)serverSocketAddress);
        }
        catch (ChannelException e) {
            if (e.getCause() instanceof SocketException && e.getCause().getMessage().equals("Too many open files") || e.getCause() instanceof ChannelException && e.getCause().getCause() instanceof SocketException && e.getCause().getCause().getMessage().equals("Too many open files")) {
                throw new ChannelException("The operating system does not offer enough file handles to open the network connection. Please increase the number of available file handles.", e.getCause());
            }
            throw e;
        }
    }
}

