/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Date;
import net.snowflake.client.core.Constants;
import net.snowflake.client.core.FileUtil;
import net.snowflake.client.core.ObjectMapperFactory;
import net.snowflake.client.jdbc.SnowflakeUtil;
import net.snowflake.client.jdbc.internal.fasterxml.jackson.databind.JsonNode;
import net.snowflake.client.jdbc.internal.fasterxml.jackson.databind.ObjectMapper;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

class FileCacheManager {
    private static final SFLogger logger = SFLoggerFactory.getLogger(FileCacheManager.class);
    private static final ObjectMapper OBJECT_MAPPER = ObjectMapperFactory.getObjectMapper();
    private static final Charset DEFAULT_FILE_ENCODING = StandardCharsets.UTF_8;
    private String cacheDirectorySystemProperty;
    private String cacheDirectoryEnvironmentVariable;
    private String baseCacheFileName;
    private long cacheExpirationInMilliseconds;
    private long cacheFileLockExpirationInMilliseconds;
    private File cacheFile;
    private File cacheLockFile;
    private File cacheDir;

    private FileCacheManager() {
    }

    static FileCacheManager builder() {
        return new FileCacheManager();
    }

    FileCacheManager setCacheDirectorySystemProperty(String cacheDirectorySystemProperty) {
        this.cacheDirectorySystemProperty = cacheDirectorySystemProperty;
        return this;
    }

    FileCacheManager setCacheDirectoryEnvironmentVariable(String cacheDirectoryEnvironmentVariable) {
        this.cacheDirectoryEnvironmentVariable = cacheDirectoryEnvironmentVariable;
        return this;
    }

    FileCacheManager setBaseCacheFileName(String baseCacheFileName) {
        this.baseCacheFileName = baseCacheFileName;
        return this;
    }

    FileCacheManager setCacheExpirationInSeconds(long cacheExpirationInSeconds) {
        this.cacheExpirationInMilliseconds = cacheExpirationInSeconds * 1000L;
        return this;
    }

    FileCacheManager setCacheFileLockExpirationInSeconds(long cacheFileLockExpirationInSeconds) {
        this.cacheFileLockExpirationInMilliseconds = cacheFileLockExpirationInSeconds * 1000L;
        return this;
    }

    void overrideCacheFile(File newCacheFile) {
        this.cacheFile = newCacheFile;
        this.cacheDir = newCacheFile.getParentFile();
        this.baseCacheFileName = newCacheFile.getName();
        FileUtil.logFileUsage(this.cacheFile, "Override cache file", true);
    }

    FileCacheManager build() {
        String cacheDirPath;
        String string = cacheDirPath = this.cacheDirectorySystemProperty != null ? SnowflakeUtil.systemGetProperty(this.cacheDirectorySystemProperty) : null;
        if (cacheDirPath == null) {
            try {
                cacheDirPath = this.cacheDirectoryEnvironmentVariable != null ? SnowflakeUtil.systemGetEnv(this.cacheDirectoryEnvironmentVariable) : null;
            }
            catch (Throwable ex) {
                logger.debug("Cannot get environment variable for cache directory, skip using cache", false);
                return this;
            }
        }
        if (cacheDirPath != null) {
            this.cacheDir = new File(cacheDirPath);
        } else {
            String homeDir = SnowflakeUtil.systemGetProperty("user.home");
            if (homeDir == null) {
                homeDir = SnowflakeUtil.systemGetProperty("java.io.tmpdir");
            } else {
                File homeFile = new File(homeDir);
                if (!homeFile.canWrite()) {
                    logger.debug("Home directory not writeable, using tmpdir", false);
                    homeDir = SnowflakeUtil.systemGetProperty("java.io.tmpdir");
                }
            }
            if (homeDir == null) {
                return this;
            }
            this.cacheDir = Constants.getOS() == Constants.OS.WINDOWS ? new File(new File(new File(new File(homeDir, "AppData"), "Local"), "Snowflake"), "Caches") : (Constants.getOS() == Constants.OS.MAC ? new File(new File(new File(homeDir, "Library"), "Caches"), "Snowflake") : new File(new File(homeDir, ".cache"), "snowflake"));
        }
        if (!this.cacheDir.mkdirs() && !this.cacheDir.exists()) {
            logger.debug("Cannot create the cache directory {}. Giving up.", this.cacheDir.getAbsolutePath());
            return this;
        }
        logger.debug("Verified Directory {}", this.cacheDir.getAbsolutePath());
        File cacheFileTmp = new File(this.cacheDir, this.baseCacheFileName).getAbsoluteFile();
        try {
            if (cacheFileTmp.createNewFile()) {
                logger.debug("Successfully created a cache file {}", cacheFileTmp);
            } else {
                logger.debug("Cache file already exists {}", cacheFileTmp);
            }
            FileUtil.logFileUsage(cacheFileTmp, "Cache file creation", false);
            this.cacheFile = cacheFileTmp.getCanonicalFile();
            this.cacheLockFile = new File(this.cacheFile.getParentFile(), this.baseCacheFileName + ".lck");
        }
        catch (IOException | SecurityException ex) {
            logger.info("Failed to touch the cache file. Ignored. {}", cacheFileTmp.getAbsoluteFile());
        }
        return this;
    }

    JsonNode readCacheFile() {
        JsonNode jsonNode;
        if (this.cacheFile == null || !this.checkCacheLockFile()) {
            return null;
        }
        if (!this.cacheFile.exists()) {
            logger.debug("Cache file doesn't exists. File: {}", this.cacheFile);
            return null;
        }
        InputStreamReader reader = new InputStreamReader((InputStream)new FileInputStream(this.cacheFile), DEFAULT_FILE_ENCODING);
        try {
            FileUtil.logFileUsage(this.cacheFile, "Read cache", false);
            jsonNode = OBJECT_MAPPER.readTree(reader);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((Reader)reader).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ex) {
                logger.debug("Failed to read the cache file. No worry. File: {}, Err: {}", this.cacheFile, ex);
                return null;
            }
        }
        ((Reader)reader).close();
        return jsonNode;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void writeCacheFile(JsonNode input) {
        logger.debug("Writing cache file. File: {}", this.cacheFile);
        if (this.cacheFile == null || !this.tryLockCacheFile()) {
            logger.debug("No cache file exists or failed to lock the file. Skipping writing the cache", false);
            return;
        }
        try {
            if (input == null) {
                return;
            }
            try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(this.cacheFile), DEFAULT_FILE_ENCODING);){
                FileUtil.logFileUsage(this.cacheFile, "Write to cache", false);
                writer.write(input.toString());
            }
        }
        catch (IOException ex) {
            logger.debug("Failed to write the cache file. File: {}", this.cacheFile);
        }
        finally {
            if (!this.unlockCacheFile()) {
                logger.debug("Failed to unlock cache file", false);
            }
        }
    }

    void deleteCacheFile() {
        logger.debug("Deleting cache file. File: {}, lock file: {}", this.cacheFile, this.cacheLockFile);
        if (this.cacheFile == null) {
            return;
        }
        this.unlockCacheFile();
        if (!this.cacheFile.delete()) {
            logger.debug("Failed to delete the file: {}", this.cacheFile);
        }
    }

    private boolean tryLockCacheFile() {
        boolean locked = false;
        for (int cnt = 0; cnt < 100 && !(locked = this.lockCacheFile()); ++cnt) {
            try {
                Thread.sleep(100L);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        if (!locked) {
            logger.debug("Failed to lock the cache file.", false);
        }
        return locked;
    }

    private boolean lockCacheFile() {
        return this.cacheLockFile.mkdirs();
    }

    private boolean unlockCacheFile() {
        return this.cacheLockFile.delete();
    }

    private boolean checkCacheLockFile() {
        long currentTime = new Date().getTime();
        long cacheFileTs = FileCacheManager.fileCreationTime(this.cacheFile);
        if (!this.cacheLockFile.exists() && cacheFileTs > 0L && currentTime - this.cacheExpirationInMilliseconds <= cacheFileTs) {
            logger.debug("No cache file lock directory exists and cache file is up to date.", false);
            return true;
        }
        long lockFileTs = FileCacheManager.fileCreationTime(this.cacheLockFile);
        if (lockFileTs < 0L) {
            return false;
        }
        if (lockFileTs < currentTime - this.cacheFileLockExpirationInMilliseconds) {
            if (!this.cacheLockFile.delete()) {
                logger.debug("Failed to delete the directory. Dir: {}", this.cacheLockFile);
                return false;
            }
            logger.debug("Deleted the cache lock directory, because it was old.", false);
            return currentTime - this.cacheExpirationInMilliseconds <= cacheFileTs;
        }
        logger.debug("Failed to lock the file. Ignored.", false);
        return false;
    }

    private static long fileCreationTime(File targetFile) {
        if (!targetFile.exists()) {
            logger.debug("File not exists. File: {}", targetFile);
            return -1L;
        }
        try {
            Path cacheFileLockPath = Paths.get(targetFile.getAbsolutePath(), new String[0]);
            BasicFileAttributes attr = Files.readAttributes(cacheFileLockPath, BasicFileAttributes.class, new LinkOption[0]);
            return attr.creationTime().toMillis();
        }
        catch (IOException ex) {
            logger.debug("Failed to get creation time. File/Dir: {}, Err: {}", targetFile, ex);
            return -1L;
        }
    }

    String getCacheFilePath() {
        return this.cacheFile.getAbsolutePath();
    }
}

