/*  ZA2 Recorder for Linux
    Copyright (C) 1998,1999 Peter Wahl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Send bug reports to:
    Peter Wahl
    wahl@uni-bonn.de
*/

#include <linux/soundcard.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <curses.h>
#include <stdio.h>

#define BUFSIZE 30000

struct wav_header
{
  unsigned int magic,tot_len,unkn1,unkn2,head_size,unkn3,srate,srate2,attrib,unkn4,wav_size;
} wave_head;

void usage(char *n)
{
  printf("This is a recorder for raw sound files for Linux.\n");
  printf("Usage:\n  %s [-t time] [-h] [-r] file\n",n);
  printf(" Description of the parameters:\n"
	 "  -t time : recording time in second. It should be an integer. Default: 10s\n"
	 "  -k      : wait for key before recording. Default: don't wait for anything\n"
	 "  -r      : resample to 44100 kHz\n"
	 "  -R      : force raw mode. Default: Wav\n"
	 "  -h      : displays this help text and exits immediately.\n"
	 " file     : filename of raw file to be recorded\n");
  exit(0);
}

void writeheader(int fh)
{
  lseek(fh,SEEK_SET,sizeof(struct wav_header));
}

void setheader(int fh,int srate,int length)
{
  length-=sizeof(struct wav_header);
  wave_head.magic=0x46464952;
  wave_head.tot_len=36+length;
  wave_head.unkn1=0x45564157;
  wave_head.unkn2=0x20746d66;
  wave_head.head_size=0x10;
  wave_head.unkn3=0x20001;
  wave_head.srate=srate;
  wave_head.srate2=srate*4+2;
  wave_head.attrib=0x100004;
  wave_head.unkn4=0x61746164;
  wave_head.wav_size=length;
  lseek(fh,SEEK_SET,0);
  write(fh,&wave_head,sizeof(struct wav_header));
}

int main(int argc,char *argv[])
{
  int f,g,a,i=0,t=10,ch,key=0,r=0,w=1,sf=0;
  double rec=0.0;
  char buf[BUFSIZE];
  printf("record v1.0\n"
	 "(c) 1999 Peter Wahl\n");
  if(argc>1)
    {
      while((ch=getopt(argc,argv,"hrkt:"))!=EOF)
	switch(ch)
	  {
	  case 't' : t=atoi(optarg);
	    break;
	  case 'r' : r=1;
	    break;
	  case 'R' : w=0;
	  case 'k' : key=1;
	    break;
	  case 'h' : usage(argv[0]);
	  }
      f=open("/dev/dsp",O_RDONLY);
      g=creat(argv[optind],0644);
      if(r)
	{
	  printf("Resampling to 44.1 kHz ...\n");
	  sf=a=44100; ioctl(f,SNDCTL_DSP_SPEED,&a);
	  if(a!=44100) printf("Resampling not possible !\n");
	}
      else ioctl(f,SNDCTL_DSP_SPEED,&sf);
      if(w) writeheader(g);
      a=AFMT_S16_LE; ioctl(f,SNDCTL_DSP_SETFMT,&a);
      a=2; ioctl(f,SNDCTL_DSP_CHANNELS,&a);
      if(key) { printf("Press any key to start recording ...\n"); getchar(); }  
      printf("Recording "); fflush(stdout); 
      while((int)rec<t)
	{
	  i++;
	  if(read(f,buf,BUFSIZE)==-1)
	    {
	      printf("\n%s: Error reading from soundcard !\n",argv[0]);
	      return 1;
	    }
	  if(write(g,buf,BUFSIZE)==-1)
	    {
	      printf("\n%s: Error writing to %s !\n",argv[0],argv[optind]);
	      return 1;
	    }
	  if(!(i%10))
	    {
	      printf("."); fflush(stdout);
	    }
	  rec+=BUFSIZE/(4.0*(double) sf);
	}
      printf("\n");
      close(f); 
      if(w) setheader(g,sf,i*BUFSIZE); 
      close(g);
    }
  else
    {
      printf("%s: Not enough arguments !\nUse %s -h for help.\n",argv[0],argv[0]);
      return(1);
    }
}

